%
%  write a Dakota .in input file.
%
%  []=dakota_in_write(method,dmeth,dvar,dresp,params,filei,varargin)
%
%  where the required input is:
%    method        (character, dakota method name)
%    dmeth         (dakota_method, method class object)
%    dvar          (structure array, variable class objects)
%    dresp         (structure array, response class objects)
%    params        (structure array, method-independent parameters)
%    filei         (character, name of .in file)
%
%  the method, dmeth, and filei will be prompted if empty.
%  params may be empty, in which case defaults will be used.
%
%  the optional varargin are not yet used.
%
%  this function writes a dakota .in input file to be used
%  by dakota.  this file is independent of the particular
%  analysis package.
%
%  this data would typically be generated by a matlab script
%  for a specific model, using the method, variable, and
%  response class objects.  this function may be called by
%  dakota_in_data.
%
%  "Copyright 2009, by the California Institute of Technology.
%  ALL RIGHTS RESERVED. United States Government Sponsorship
%  acknowledged. Any commercial use must be negotiated with
%  the Office of Technology Transfer at the California Institute
%  of Technology.  (NTR 47078)
%
%  This software may be subject to U.S. export control laws.
%  By accepting this  software, the user agrees to comply with
%  all applicable U.S. export laws and regulations. User has the
%  responsibility to obtain export licenses, or other export
%  authority as may be required before exporting such information
%  to foreign countries or providing access to foreign persons."
%
function []=dakota_in_write(method,dmeth,dvar,dresp,params,filei,varargin)

if ~nargin
    help dakota_in_write
    return
end

%  process the input parameters

if ~exist('method','var') || isempty(method)
    method=input('Method?  ','s');
end

if ~exist('dmeth' ,'var') || isempty(dmeth)
    dmeth=dakota_method(method);
end

if ~exist('filei' ,'var') || isempty(filei)
    filei=input('Dakota input file to write?  ','s');
end
[pathstr,name,ext,versn] = fileparts(filei);
if isempty(ext)
% fileparts only considers '.in' to be the extension, not '.qmu.in'
    ext='.qmu.in';
end
filei2=fullfile(pathstr,[name ext versn]);

display(sprintf('Opening Dakota input file ''%s''.',filei2));
fidi=fopen(sprintf('%s',filei2),'w');
if (fidi < 0)
    error('''%s'' could not be opened.',filei2);
end

if ~exist('params','var')
    params=struct();
end
params=dakota_in_params(params);

%  write the strategy section

strategy_write(fidi,params);

%  write the method section

method_write(fidi,dmeth,dresp,params);
param_write(fidi,'\t  ','output','   ','\n',params);

%  write the model section

model_write(fidi);

%  write the variables section

variables_write(fidi,dmeth,dvar);

%  write the interface section

interface_write(fidi,params);

%  write the responses section

responses_write(fidi,dmeth,dresp,params);

fclose(fidi);
display('End of file successfully written.');

end

%%  function to write the strategy section of the file

function []=strategy_write(fidi,params)

display('Writing strategy section of Dakota input file.');

fprintf(fidi,'strategy,\n');
fprintf(fidi,'\tsingle_method\n');
param_write(fidi,'\t  ','graphics','','\n',params);
param_write(fidi,'\t  ','tabular_graphics_data','','\n',params);
param_write(fidi,'\t  ','tabular_graphics_file',' ''','''\n',params);
fprintf(fidi,'\n');

end

%%  function to write the method section of the file

function []=method_write(fidi,dmeth,dresp,params)

display('Writing method section of Dakota input file.');

fprintf(fidi,'method,\n');
fprintf(fidi,'\t%s\n',dmeth.method);

dmeth_params_write(dmeth,fidi);

%  write response levels

if strcmp(dmeth.type,'nond') && isfield(dresp,'rf')
    param_write(fidi,'\t  ','distribution',' ','\n',params);
    [respl,probl,rell,grell]=prop_levels(dresp.rf);
    if ~isempty(respl)
        rlev_write(fidi,'response_levels',respl);
        param_write(fidi,'\t  ','compute',' ','\n',params);
    end 
    if ~isempty(probl)
        rlev_write(fidi,'probability_levels',probl);
    end
    if ~isempty(rell)
        rlev_write(fidi,'reliability_levels',rell);
    end
    if ~isempty(grell)
        rlev_write(fidi,'gen_reliability_levels',grell);
    end
end
fprintf(fidi,'\n');

end

%%  function to write response levels

function []=rlev_write(fidi,ltype,levels)

fprintf(fidi,'\t  num_%s =',ltype);
for i=1:length(levels)
    fprintf(fidi,' %d',length(levels{i}));
end
fprintf(fidi,'\n');

fprintf(fidi,'\t  %s =\n',ltype);

for i=1:length(levels)
    if ~isempty(levels{i})
        vector_write(fidi,sprintf('\t    '),levels{i},8,76);
    end
end

end

%%  function to write the model section of the file

function []=model_write(fidi)

display('Writing model section of Dakota input file.');

fprintf(fidi,'model,\n');
fprintf(fidi,'\t%s\n\n','single');

end

%%  function to write the variables section of the file

function []=variables_write(fidi,dmeth,dvar)

display('Writing variables section of Dakota input file.');

fprintf(fidi,'variables,\n');

%  variables vary by method

vsets_write(fidi,dvar,dmeth.variables);

%  linear constraints vary by method

lcsets_write(fidi,dvar,dmeth.lcspec);

fprintf(fidi,'\n');

end

%%  function to write variable sets

function []=vsets_write(fidi,dvar,variables)

for i=1:length(variables)
    switch(variables{i})
        case 'cdv'
            cstring='continuous_design';
        case 'nuv'
            cstring='normal_uncertain';
        case 'csv'
            cstring='continuous_state';
        otherwise
            warning('vsets_write:unrec_var',...
                'Unrecognized variable ''%s''.',variables{i});
    end

    if isfield(dvar,variables{i})
        vlist_write(fidi,cstring,variables{i},dvar.(variables{i}));
    end
end

end

%%  function to write variable list

function []=vlist_write(fidi,cstring,cstring2,dvar)

%  put variables into lists for writing

pinitpt=prop_initpt(dvar);
plower =prop_lower (dvar);
pupper =prop_upper (dvar);
pmean  =prop_mean  (dvar);
pstddev=prop_stddev(dvar);
pinitst=prop_initst(dvar);
pstype =prop_stype (dvar);
pscale =prop_scale (dvar);
pdesc  =prop_desc  (dvar);

%  write variables
%  (using Dakota 4.1 syntax for backward compatability)

disp(sprintf('  Writing %d %s variables.',length(dvar),class(dvar)));

fprintf(fidi,'\t%s = %d\n',cstring,length(dvar));
if ~isempty(pinitpt)
    fprintf(fidi,'\t  %s_initial_point =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pinitpt,6,76);
end
if ~isempty(plower)
    fprintf(fidi,'\t  %s_lower_bounds =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),plower ,6,76);
end
if ~isempty(pupper)
    fprintf(fidi,'\t  %s_upper_bounds =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pupper ,6,76);
end
if ~isempty(pmean)
    fprintf(fidi,'\t  %s_means =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pmean  ,6,76);
end
if ~isempty(pstddev)
    fprintf(fidi,'\t  %s_std_deviations =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pstddev,6,76);
end
if ~isempty(pinitst)
    fprintf(fidi,'\t  %s_initial_state =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pinitst,6,76);
end
if ~isempty(pstype)
    fprintf(fidi,'\t  %s_scale_types =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pstype ,6,76);
end
if ~isempty(pscale)
    fprintf(fidi,'\t  %s_scales =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pscale ,6,76);
end
if ~isempty(pdesc)
    fprintf(fidi,'\t  %s_descriptors =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pdesc  ,6,76);
end

end

%%  function to write linear constraint sets

function []=lcsets_write(fidi,dvar,lcspec)

for i=1:length(lcspec)
    switch(lcspec{i})
        case 'lic'
            cstring ='linear_inequality_constraints';
            cstring2='linear_inequality';
        case 'lec'
            cstring ='linear_equality_constraints';
            cstring2='linear_equality';
        otherwise
            warning('lcspec_write:unrec_lcspec',...
                'Unrecognized linear constraint ''%s''.',lcspec{i});
    end
    
    if isfield(dvar,lcspec{i})
        lclist_write(fidi,cstring,cstring2,dvar.(lcspec{i}));
    end
end

end

%%  function to write linear constraint list

function []=lclist_write(fidi,cstring,cstring2,dvar)

%  put linear constraints into lists for writing

pmatrix=prop_matrix(dvar);
plower =prop_lower (dvar);
pupper =prop_upper (dvar);
ptarget=prop_target(dvar);
pstype =prop_stype (dvar);
pscale =prop_scale (dvar);

%  write linear constraints

disp(sprintf('  Writing %d %s linear constraints.',...
    length(dvar),class(dvar)));

if ~isempty(pmatrix)
    fprintf(fidi,'\t  %s_matrix =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pmatrix,6,76);
end
if ~isempty(plower)
    fprintf(fidi,'\t  %s_lower_bounds =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),plower ,6,76);
end
if ~isempty(pupper)
    fprintf(fidi,'\t  %s_upper_bounds =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pupper ,6,76);
end
if ~isempty(ptarget)
    fprintf(fidi,'\t  %s_targets =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),ptarget,6,76);
end
if ~isempty(pstype)
    fprintf(fidi,'\t  %s_scale_types =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pstype ,6,76);
end
if ~isempty(pscale)
    fprintf(fidi,'\t  %s_scales =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pscale ,6,76);
end

end

%%  function to write the interface section of the file

function []=interface_write(fidi,params)

display('Writing interface section of Dakota input file.');

fprintf(fidi,'interface,\n');

if     ~params.system && ~params.fork && ~params.direct
    params.fork=true;
elseif (params.system+params.fork+params.direct > 1)
    error('Too many interfaces selected.')
end

if     params.system || params.fork
    param_write(fidi,'\t','asynchronous','','\n',params);
    param_write(fidi,'\t  ','evaluation_concurrency',' = ','\n',params);
    param_write(fidi,'\t','system','','\n',params);
    param_write(fidi,'\t','fork','','\n',params);
    param_write(fidi,'\t  ','analysis_driver',' = ''','''\n',params);
    if ~isempty(params.input_filter)
        param_write(fidi,'\t  ','input_filter','    = ''','''\n',params);
    end
    if ~isempty(params.output_filter)
        param_write(fidi,'\t  ','output_filter','   = ''','''\n',params);
    end
    param_write(fidi,'\t  ','failure_capture','   ','\n',params);
    param_write(fidi,'\t  ','parameters_file',' = ''','''\n',params);
    param_write(fidi,'\t  ','results_file','    = ''','''\n',params);
    param_write(fidi,'\t  ','file_tag', '','\n',params);
    param_write(fidi,'\t  ','file_save','','\n',params);
elseif params.direct
%  Error: asynchronous capability not yet supported in direct interfaces.
    param_write(fidi,'\t','direct','','\n',params);
    param_write(fidi,'\t  ','analysis_driver','     = ''','''\n',params);
    [pathstr,name,ext,versn] = fileparts(params.analysis_components);
    if isempty(ext)
        ext='.m';
    end
    params.analysis_components=fullfile(pathstr,[name ext versn]);
    param_write(fidi,'\t  ','analysis_components',' = ''','''\n',params);
    if ~isempty(params.input_filter)
        param_write(fidi,'\t  ','input_filter','    = ''','''\n',params);
    end
    if ~isempty(params.output_filter)
        param_write(fidi,'\t  ','output_filter','   = ''','''\n',params);
    end
    param_write(fidi,'\t  ','failure_capture','   ','\n',params);
    param_write(fidi,'\t  ','deactivate','   ','\n',params);
end

fprintf(fidi,'\n');

end

%%  function to write the responses section of the file

function []=responses_write(fidi,dmeth,dresp,params)

display('Writing responses section of Dakota input file.');

fprintf(fidi,'responses,\n');

%  functions, gradients, and hessians vary by method

rsets_write(fidi,dresp,dmeth.responses);
ghspec_write(fidi,params,dmeth.ghspec);

fprintf(fidi,'\n');

end

%%  function to write response sets

function []=rsets_write(fidi,dresp,responses)

rdesc={};

for i=1:length(responses)
    switch(responses{i})
        case 'of'
            cstring ='objective_functions';
            cstring2='objective_function';
        case 'lst'
            cstring ='least_squares_terms';
            cstring2='least_squares_term';
        case 'nic'
            cstring ='nonlinear_inequality_constraints';
            cstring2='nonlinear_inequality';
        case 'nec'
            cstring ='nonlinear_equality_constraints';
            cstring2='nonlinear_equality';
        case 'rf'
            cstring ='response_functions';
            cstring2='response_function';
        otherwise
            warning('rsets_write:unrec_resp',...
                'Unrecognized response ''%s''.',responses{i});
    end
    
    if isfield(dresp,responses{i})
        [rdesc]=rlist_write(fidi,cstring,cstring2,dresp.(responses{i}),rdesc);
    end
end

%  write response descriptors

if ~isempty(rdesc)
    fprintf(fidi,'\tresponse_descriptors =\n');
    vector_write(fidi,sprintf('\t  '),rdesc,6,76);
end

end

%%  function to write response list

function [rdesc]=rlist_write(fidi,cstring,cstring2,dresp,rdesc)

%  put responses into lists for writing

pstype =prop_stype (dresp);
pscale =prop_scale (dresp);
pweight=prop_weight(dresp);
plower =prop_lower (dresp);
pupper =prop_upper (dresp);
ptarget=prop_target(dresp);

%  accumulate descriptors into list for subsequent writing

rdesc(end+1:end+numel(dresp))=prop_desc  (dresp);

%  write responses

disp(sprintf('  Writing %d %s responses.',length(dresp),class(dresp)));

fprintf(fidi,'\tnum_%s = %d\n',cstring,length(dresp));
if ~isempty(pstype)
    fprintf(fidi,'\t  %s_scale_types =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pstype ,6,76);
end
if ~isempty(pscale)
    fprintf(fidi,'\t  %s_scales =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pscale ,6,76);
end
if ~isempty(pweight)
    switch cstring2
        case 'objective_function'
            fprintf(fidi,'\t  %s_weights =\n','multi_objective');
            vector_write(fidi,sprintf('\t    '),pweight,6,76);
        case 'least_squares_term'
            fprintf(fidi,'\t  %s_weights =\n','least_squares');
            vector_write(fidi,sprintf('\t    '),pweight,6,76);
    end
end
if ~isempty(plower)
    fprintf(fidi,'\t  %s_lower_bounds =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),plower ,6,76);
end
if ~isempty(pupper)
    fprintf(fidi,'\t  %s_upper_bounds =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),pupper ,6,76);
end
if ~isempty(ptarget)
    fprintf(fidi,'\t  %s_targets =\n',cstring2);
    vector_write(fidi,sprintf('\t    '),ptarget,6,76);
end

end

%%  function to write gradient and hessian specifications

function []=ghspec_write(fidi,params,ghspec)

%  gradients

if find_string(ghspec,'grad')
    if     params.numerical_gradients
        param_write(fidi,'\t','numerical_gradients','','\n',params);
        param_write(fidi,'\t  ','method_source',' ','\n',params);
        param_write(fidi,'\t  ','interval_type',' ','\n',params);
        param_write(fidi,'\t  ','fd_gradient_step_size',' = ','\n',params);
    elseif params.analytic_gradients
        param_write(fidi,'\t','analytic_gradients','','\n',params);
%     elseif params.mixed_gradients
    end
else
    fprintf(fidi,'\tno_gradients\n');
end

%  hessians (no implemented methods use them yet)

if find_string(ghspec,'hess')
    error('Hessians needed by method but not provided.');
else
    fprintf(fidi,'\tno_hessians\n');
end

end

%%  function to write a parameter

function []=param_write(fidi,sbeg,pname,smid,send,params)

if ~isfield(params,pname)
    warning('param_write:param_not_found',...
        'Parameter ''%s'' not found in structure.',pname);
    return
end

if islogical(params.(pname)) && ~params.(pname)
    return
end

if     islogical(params.(pname))
    fprintf(fidi,[sbeg '%s' send],pname);
elseif ischar   (params.(pname))
    fprintf(fidi,[sbeg '%s' smid '%s' send],pname,params.(pname));
elseif isnumeric(params.(pname))
    fprintf(fidi,[sbeg '%s' smid '%g' send],pname,params.(pname));
end

end

%%  function to write a vector on multiple lines

function []=vector_write(fidi,sbeg,vec,nmax,cmax)

if ~exist('nmax','var') || isempty(nmax)
    nmax=Inf;
end
if ~exist('cmax','var') || isempty(cmax)
    cmax=Inf;
end

%  set up first iteration

svec =[];
nitem=nmax;
lsvec=cmax;

%  transpose vector from column-wise to row-wise

vec=vec';

%  assemble each line, flushing when necessary

for i=1:numel(vec)
    if isnumeric(vec(i))
        sitem=sprintf('%g'    ,vec(i));
    else
        sitem=sprintf('''%s''',char(vec(i)));
    end
    nitem=nitem+1;
    lsvec=lsvec+1+length(sitem);
    
    if (nitem <= nmax) && (lsvec <= cmax)
        svec=[svec ' ' sitem];
    else
        if ~isempty(svec)
            fprintf(fidi,'%s\n',svec);
        end
        svec=[sbeg sitem];
        nitem=1;
        lsvec=length(svec);
    end
end

%  flush buffer at end, if necessary

if ~isempty(svec)
    fprintf(fidi,'%s\n',svec);
end

end
