%
%  define the data to write the dakota .in and .m files.
%
%  []=dakota_in_data(dmeth,variables,responses,dparams,filei,package,varargin)
%
%  where the required input is:
%    dmeth         (dakota_method, method class object)
%    variables     (structure array, variable class objects)
%    responses     (structure array, response class objects)
%    dparams       (structure array, method-independent parameters)
%    filei         (character, name of .in and .m files)
%    package       (character, analysis package)
%
%  params may be empty, in which case defaults will be used.
%
%  the optional varargin are passed directly through to the
%  QmuUpdateFunctions brancher to be used by the analysis
%  package.  for example, this could be model information.
%
%  this function defines the data to write the dakota .in and
%  .m files.  it is necessary for multiple reasons.  first,
%  it collects the parameters and applies some defaults that
%  are unique to the environment.  second, some analysis package
%  variables and/or responses may be treated differently by
%  dakota.  for example, an analysis package variable may be
%  defined as an array, so the QmuSetupDesign brancher will
%  create dakota variables for each element of the array.
%  finally it calls the functions to write the .in and .m files.
%  this function is independent of the particular analysis
%  package.
%
%  this data would typically be generated by a matlab script
%  for a specific model, using the method, variable, and
%  response class objects.
%
%  "Copyright 2009, by the California Institute of Technology.
%  ALL RIGHTS RESERVED. United States Government Sponsorship
%  acknowledged. Any commercial use must be negotiated with
%  the Office of Technology Transfer at the California Institute
%  of Technology.  (NTR 47078)
%
%  This software may be subject to U.S. export control laws.
%  By accepting this  software, the user agrees to comply with
%  all applicable U.S. export laws and regulations. User has the
%  responsibility to obtain export licenses, or other export
%  authority as may be required before exporting such information
%  to foreign countries or providing access to foreign persons."
%
function []=dakota_in_data(dmeth,variables,responses,dparams,filei,package,varargin)

if ~nargin
    help dakota_in_data
    return
end

%%  parameters

%  get default set of parameters

params=dakota_in_params(struct());

%  merge specified parameters into default set, whether or not
%  they already exist

fnames=fieldnames(dparams);

for i=1:numel(fnames)
    if ~isfield(params,fnames{i})
        warning('dakota_in_data:unknown_param',...
            'No parameter ''%s'' in default parameter set.',...
            fnames{i});
    end
    params.(fnames{i})=dparams.(fnames{i});
end

if params.direct && ...
   isempty(params.analysis_driver)
    params.analysis_driver='matlab';
end

if strcmpi(params.analysis_driver,'matlab') && ...
   isempty(params.analysis_components)
    [pathstr,name,ext,versn] = fileparts(filei);
    params.analysis_components=fullfile(pathstr,[name '.m' versn]);
end

%  merge method parameters, though they shouldn't be in dparams

% dmeth=dmeth_params_merge(dmeth,dparams)


%%  variables

fnames=fieldnames(variables);

for i=1:length(fnames)
    
%  for linear constraints, just copy

    if isa(variables.(fnames{i}),'linear_inequality_constraint') || ...
       isa(variables.(fnames{i}),'linear_equality_constraint'  )
        dvar.(fnames{i})=variables.(fnames{i});

%  for variables, call the setup function

    else
        fhandle=str2func([class(variables.(fnames{i})) '.empty']);
        dvar.(fnames{i})=fhandle();
        for j=1:length(variables.(fnames{i}))
            %call setupdesign
            dvar.(fnames{i})=QmuSetupDesign(dvar.(fnames{i}),variables.(fnames{i})(j),params,varargin{:}); 
        end
    end
end

%%  responses

fnames=fieldnames(responses);

for i=1:length(fnames)
%     fhandle=str2func([class(responses.(fnames{i})) '.empty']);
%     dresp.(fnames{i})=fhandle();
%     for j=1:length(responses.(fnames{i}))
%         dresp.(fnames{i})(j)=responses.(fnames{i})(j);
%     end

%  currently all response types can just be copied

    dresp.(fnames{i})=responses.(fnames{i});
end

%%  write files

%Write in file
dakota_in_write(dmeth.method,dmeth,dvar,dresp,params,filei,varargin{:});

%Write m file
%dakota_m_write(dmeth.method,dmeth,dvar,dresp,params,filei,package,varargin{:});

end
