/*!\file Beam.c
 * \brief: implementation of the Beam object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "./Beam.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../DataSet/DataSet.h"
#include "../include/typedefs.h"


Beam::Beam(){
	return;
}

Beam::~Beam(){
	return;
}
		
Beam::Beam(int beam_id, int beam_mid, int beam_mparid, int beam_g[2], double beam_h[2], double beam_s[2],double beam_b[2],double beam_k[2],bool beam_onbed){

	int i;
	
	id=beam_id;
	mid=beam_mid;
	mparid=beam_mparid;
	for(i=0;i<2;i++){
		node_ids[i]=beam_g[i];
		node_offsets[i]=UNDEF;
		nodes[i]=NULL;
		h[i]=beam_h[i];
		s[i]=beam_s[i];
		b[i]=beam_b[i];
		k[i]=beam_k[i];
	}
	matice=NULL;
	matice_offset=UNDEF;
	matpar=NULL;
	matpar_offset=UNDEF;
	onbed=beam_onbed;
	
	return;
}

#undef __FUNCT__
#define __FUNCT__ "Beam::Echo"
void Beam::Echo(void){

	printf("Beam:\n");
	printf("   id: %i\n",id);
	printf("   mid: %i\n",mid);
	printf("   mparid: %i\n",mparid);
	printf("   node_ids=[%i,%i]\n",node_ids[0],node_ids[1]);
	printf("   node_offsets=[%i,%i]\n",node_offsets[0],node_offsets[1]);
	printf("   matice_offset=%i\n",matice_offset);
	printf("   matpar_offset=%i\n",matpar_offset);
	printf("   h=[%g,%g]\n",h[0],h[1]);
	printf("   s=[%g,%g]\n",s[0],s[1]);
	printf("   b=[%g,%g]\n",b[0],b[1]);
	printf("   k=[%g,%g]\n",k[0],k[1]);
	printf("   onbed=%i\n",onbed);
	printf("   nodes: \n");
	if(nodes[0])nodes[0]->Echo();
	if(nodes[1])nodes[1]->Echo();
	if(matice)matice->Echo();
	if(matpar)matpar->Echo();

	return;
}

#undef __FUNCT__
#define __FUNCT__ "Beam::DeepEcho"
void Beam::DeepEcho(void){

	printf("Beam:\n");
	printf("   id: %i\n",id);
	printf("   mid: %i\n",mid);
	printf("   mparid: %i\n",mparid);
	printf("   node_ids=[%i,%i]\n",node_ids[0],node_ids[1]);
	printf("   node_offsets=[%i,%i]\n",node_offsets[0],node_offsets[1]);
	printf("   matice_offset=%i\n",matice_offset);
	printf("   matpar_offset=%i\n",matpar_offset);
	printf("   h=[%g,%g]\n",h[0],h[1]);
	printf("   s=[%g,%g]\n",s[0],s[1]);
	printf("   b=[%g,%g]\n",b[0],b[1]);
	printf("   k=[%g,%g]\n",k[0],k[1]);
	printf("   onbed=%i\n",onbed);
	printf("   nodes: \n");
	if(nodes[0])nodes[0]->Echo();
	if(nodes[1])nodes[1]->Echo();
	if(matice)matice->Echo();
	if(matpar)matpar->Echo();

	return;
}
void  Beam::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Beam: */
	enum_type=BeamEnum();
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Beam data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&mid,sizeof(mid));marshalled_dataset+=sizeof(mid);
	memcpy(marshalled_dataset,&mparid,sizeof(mparid));marshalled_dataset+=sizeof(mparid);
	memcpy(marshalled_dataset,&node_ids,sizeof(node_ids));marshalled_dataset+=sizeof(node_ids);
	memcpy(marshalled_dataset,&nodes,sizeof(nodes));marshalled_dataset+=sizeof(nodes);
	memcpy(marshalled_dataset,&node_offsets,sizeof(node_offsets));marshalled_dataset+=sizeof(node_offsets);
	memcpy(marshalled_dataset,&matice,sizeof(matice));marshalled_dataset+=sizeof(matice);
	memcpy(marshalled_dataset,&matice_offset,sizeof(matice_offset));marshalled_dataset+=sizeof(matice_offset);
	memcpy(marshalled_dataset,&matpar,sizeof(matpar));marshalled_dataset+=sizeof(matpar);
	memcpy(marshalled_dataset,&matpar_offset,sizeof(matpar_offset));marshalled_dataset+=sizeof(matpar_offset);
	memcpy(marshalled_dataset,&h,sizeof(h));marshalled_dataset+=sizeof(h);
	memcpy(marshalled_dataset,&s,sizeof(s));marshalled_dataset+=sizeof(s);
	memcpy(marshalled_dataset,&b,sizeof(b));marshalled_dataset+=sizeof(b);
	memcpy(marshalled_dataset,&k,sizeof(k));marshalled_dataset+=sizeof(k);
	memcpy(marshalled_dataset,&onbed,sizeof(onbed));marshalled_dataset+=sizeof(onbed);
	
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
		
int   Beam::MarshallSize(){
	return sizeof(id)
		+sizeof(mid)
		+sizeof(mparid)
		+sizeof(node_ids)
		+sizeof(nodes)
		+sizeof(node_offsets)
		+sizeof(matice)
		+sizeof(matice_offset)
		+sizeof(matpar)
		+sizeof(matpar_offset)
		+sizeof(h)
		+sizeof(s)
		+sizeof(b)
		+sizeof(k)
		+sizeof(onbed)
		+sizeof(int); //sizeof(int) for enum type
}

char* Beam::GetName(void){
	return "beam";
}

void  Beam::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&mid,marshalled_dataset,sizeof(mid));marshalled_dataset+=sizeof(mid);
	memcpy(&mparid,marshalled_dataset,sizeof(mparid));marshalled_dataset+=sizeof(mparid);
	memcpy(&node_ids,marshalled_dataset,sizeof(node_ids));marshalled_dataset+=sizeof(node_ids);
	memcpy(&nodes,marshalled_dataset,sizeof(nodes));marshalled_dataset+=sizeof(nodes);
	memcpy(&node_offsets,marshalled_dataset,sizeof(node_offsets));marshalled_dataset+=sizeof(node_offsets);
	memcpy(&matice,marshalled_dataset,sizeof(matice));marshalled_dataset+=sizeof(matice);
	memcpy(&matice_offset,marshalled_dataset,sizeof(matice_offset));marshalled_dataset+=sizeof(matice_offset);
	memcpy(&matpar,marshalled_dataset,sizeof(matpar));marshalled_dataset+=sizeof(matpar);
	memcpy(&matpar_offset,marshalled_dataset,sizeof(matpar_offset));marshalled_dataset+=sizeof(matpar_offset);
	memcpy(&h,marshalled_dataset,sizeof(h));marshalled_dataset+=sizeof(h);
	memcpy(&s,marshalled_dataset,sizeof(s));marshalled_dataset+=sizeof(s);
	memcpy(&b,marshalled_dataset,sizeof(b));marshalled_dataset+=sizeof(b);
	memcpy(&k,marshalled_dataset,sizeof(k));marshalled_dataset+=sizeof(k);
	memcpy(&onbed,marshalled_dataset,sizeof(onbed));marshalled_dataset+=sizeof(onbed);

	/*nodes and materials are not pointing to correct objects anymore:*/
	for(i=0;i<2;i++)nodes[i]=NULL;
	matice=NULL;
	matpar=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
int Beam::Enum(void){

	return BeamEnum();

}
int    Beam::GetId(void){ return id; }

int    Beam::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}


#undef __FUNCT__ 
#define __FUNCT__ "Beam::Configure"
void  Beam::Configure(void* ploadsin,void* pnodesin,void* pmaterialsin){

	int i;
	
	DataSet* loadsin=NULL;
	DataSet* nodesin=NULL;
	DataSet* materialsin=NULL;

	/*Recover pointers :*/
	loadsin=(DataSet*)ploadsin;
	nodesin=(DataSet*)pnodesin;
	materialsin=(DataSet*)pmaterialsin;

	/*Link this element with its nodes, ie find pointers to the nodes in the nodes dataset.: */
	ResolvePointers((Object**)nodes,node_ids,node_offsets,2,nodesin);
	
	/*Same for materials: */
	ResolvePointers((Object**)&matice,&mid,&matice_offset,1,materialsin);
	ResolvePointers((Object**)&matpar,&mparid,&matpar_offset,1,materialsin);

}

#undef __FUNCT__ 
#define __FUNCT__ "Beam::CreateKMatrix"

void  Beam::CreateKMatrix(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticAnalysisEnum()) {
	
		if (sub_analysis_type==HutterAnalysisEnum()) {

			CreateKMatrixDiagnosticHutter( Kgg,inputs,analysis_type,sub_analysis_type);
		}
		else 
			throw ErrorException(__FUNCT__,exprintf("%s%i%s\n","sub_analysis_type: ",sub_analysis_type," not supported yet"));
	}
	else{
		throw ErrorException(__FUNCT__,exprintf("%s%i%s\n","analysis: ",analysis_type," not supported yet"));
	}

}


#undef __FUNCT__ 
#define __FUNCT__ "Beam::CreateKMatrixDiagnosticHutter"

void  Beam::CreateKMatrixDiagnosticHutter(Mat Kgg,void* vinputs,int analysis_type,int sub_analysis_type){
	
	
	const int numgrids=2;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	double    Ke_gg[numdofs][numdofs]={{0,0,0,0},{0,0,0,0},{0,0,0,0},{0,0,0,0}};
	int       numberofdofspernode;
	
	GetDofList(&doflist[0],&numberofdofspernode);

	if (onbed){
		Ke_gg[0][0]=1;
		Ke_gg[1][1]=1;
		Ke_gg[2][0]=-1;
		Ke_gg[2][2]=1;
		Ke_gg[3][1]=-1;
		Ke_gg[3][3]=1;
	}
	else{
		Ke_gg[2][0]=-1;
		Ke_gg[2][2]=1;
		Ke_gg[3][1]=-1;
		Ke_gg[3][3]=1;
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdofs,doflist,numdofs,doflist,(const double*)Ke_gg,ADD_VALUES);

}


#undef __FUNCT__ 
#define __FUNCT__ "Beam::CreatePVector"
void  Beam::CreatePVector(Vec pg,void* inputs,int analysis_type,int sub_analysis_type){
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticAnalysisEnum()) {
		if (sub_analysis_type==HutterAnalysisEnum()) {
			CreatePVectorDiagnosticHutter( pg,inputs,analysis_type,sub_analysis_type);
		}
		else
			throw ErrorException(__FUNCT__,exprintf("%s%i%s"," analysis ",analysis_type," not supported yet"));
	}
	else{
		throw ErrorException(__FUNCT__,exprintf("%s%i%s"," analysis ",analysis_type," not supported yet"));
	}

}


#undef __FUNCT__ 
#define __FUNCT__ "Beam::CreatePVectorDiagnosticHutter"

void Beam::CreatePVectorDiagnosticHutter( Vec pg, void* vinputs, int analysis_type,int sub_analysis_type){

	int i,j,k;
	
	const int numgrids=2;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	double    pe_g[4]={0,0,0,0};
	double    pe_g_gaussian[4]={0,0,0,0};
	int       found=0;
	int       dofs[1]={0};
	double    xyz_list[numgrids][3];
	double    z_list[numgrids];
	double    constant_part;
	int       numberofdofspernode;

	/*gaussian points: */
	int      num_gauss;
	double*  segment_gauss_coord=NULL;
	double   gauss_coord;
	double*  gauss_weights=NULL;
	double   gauss_weight;
	int      ig;
	double   l1l2[2];
	double   slope[2];
	double   slope2;

	double   z_g;
	double   rho_ice,gravity,n,B;
	double   Jdet;
	double   ub,vb;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover doflist: */
	GetDofList(&doflist[0],&numberofdofspernode);

	/*recover parameters: */
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	n=matice->GetN();
	B=matice->GetB();

	//recover extra inputs
	found=inputs->Recover("surfaceslopex",&slope[0],1,dofs,1,(void**)&nodes[0]); //only recover from fist node, second node will have the same slope
	if(!found)throw ErrorException(__FUNCT__," surfaceslopex missing from inputs!");
	
	found=inputs->Recover("surfaceslopey",&slope[1],1,dofs,1,(void**)&nodes[0]);//only recover from fist node, second node will have the same slope
	if(!found)throw ErrorException(__FUNCT__," surfaceslopey missing from inputs!");

	//Get all element grid data:
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
	for(i=0;i<numgrids;i++)z_list[i]=xyz_list[i][2];
	
	//compute slope2 slopex and slopy
	slope2=pow(slope[0],2)+pow(slope[1],2);

	//%compute constant_part
	constant_part=-2*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2));

	//Get gaussian points and weights. order 2 since we have a product of 2 nodal functions
	num_gauss=3;
	GaussSegment(&segment_gauss_coord, &gauss_weights, num_gauss);

	//Start  looping on the number of gaussian points:
	for(ig=0;ig<num_gauss;ig++){

		//Pick up the gaussian point and its weight:
		gauss_weight=gauss_weights[ig];
		gauss_coord=segment_gauss_coord[ig];

		//compute constant_part
		GetParameterValue(&z_g, &z_list[0],gauss_coord);

		//Get Jacobian determinant:
		GetJacobianDeterminant(&Jdet, &z_list[0],gauss_coord);
		
		for(j=0;j<NDOF2;j++){
			pe_g_gaussian[NDOF2+j]=constant_part*pow((s[0]-z_g)/B,n)*slope[j]*Jdet*gauss_weight;
		}
		
		//add pe_gaussian vector to pe:
		for(j=0;j<numdofs;j++){
			pe_g[j]+=pe_g_gaussian[j];
		}
	} //for(ig=0;ig<num_gauss;ig++)

	//Deal with lower surface
	if (onbed){

		//compute ub
		constant_part=-1.58*pow((double)10.0,-(double)10.0)*rho_ice*gravity*h[0];
		ub=constant_part*slope[0];
		vb=constant_part*slope[1];

		//Add to pe: 
		pe_g[0]+=ub;
		pe_g[1]+=vb;
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdofs,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&segment_gauss_coord);
	xfree((void**)&gauss_weights);
}


#undef __FUNCT__ 
#define __FUNCT__ "Beam::UpdateFromInputs"
void  Beam::UpdateFromInputs(void* vinputs){

	int     dofs[1]={0};
	double  temperature_list[2];
	double  temperature_average;
	double  B_list[2];
	double  B_average;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*Update internal data if inputs holds new values: */
	inputs->Recover("thickness",&h[0],1,dofs,2,(void**)nodes);
	inputs->Recover("surface",&s[0],1,dofs,2,(void**)nodes);
	inputs->Recover("bed",&b[0],1,dofs,2,(void**)nodes);
	inputs->Recover("drag",&k[0],1,dofs,2,(void**)nodes);
	
	//Update material if necessary
	if(inputs->Recover("temperature_average",&temperature_list[0],1,dofs,2,(void**)nodes)){
		temperature_average=(temperature_list[0]+temperature_list[1])/2.0;
		B_average=Paterson(temperature_average);
		matice->SetB(B_average);
	}
	if(inputs->Recover("temperature",&temperature_list[0],1,dofs,2,(void**)nodes)){
		temperature_average=(temperature_list[0]+temperature_list[1])/2.0;
		B_average=Paterson(temperature_average);
		matice->SetB(B_average);
	}
	
	if(inputs->Recover("B",&B_list[0],1,dofs,2,(void**)nodes)){
		B_average=(B_list[0]+B_list[1])/2.0;
		matice->SetB(B_average);
	}

}
		
void  Beam::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;
	
	for(i=0;i<2;i++){
		nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
		for(j=0;j<numberofdofspernode;j++){
			doflist[i*numberofdofspernode+j]=doflist_per_node[j];
		}
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}

void  Beam::GetDofList1(int* doflist){

	int i;
	for(i=0;i<2;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}


void* Beam::GetMatPar(){
	return matpar;
}

		
void  Beam::GetNodes(void** vpnodes){
	int i;
	Node** pnodes=(Node**)vpnodes;

	for(i=0;i<3;i++){
		pnodes[i]=nodes[i];
	}
}
		

Object* Beam::copy() {
	
	return new Beam(*this); 

}


#undef __FUNCT__ 
#define __FUNCT__ "Beam::NodeConfiguration"
void  Beam::NodeConfiguration(int* beam_node_ids,Node* beam_nodes[2],int* beam_node_offsets){

	int i;
	for(i=0;i<2;i++){
		node_ids[i]=beam_node_ids[i];
		nodes[i]=beam_nodes[i];
		node_offsets[i]=beam_node_offsets[i];
	}

}
#undef __FUNCT__ 
#define __FUNCT__ "Beam::MaticeConfiguration"
void  Beam::MaticeConfiguration(Matice* beam_matice,int beam_matice_offset){
	matice=beam_matice;
	matice_offset=beam_matice_offset;
}

#undef __FUNCT__ 
#define __FUNCT__ "Beam::MatparConfiguration"
void  Beam::MatparConfiguration(Matpar* beam_matpar,int beam_matpar_offset){

	matpar=beam_matpar;
	matpar_offset=beam_matpar_offset;

}



#undef __FUNCT__ 
#define __FUNCT__ "Beam::GetShelf"
int   Beam::GetShelf(){
	throw ErrorException(__FUNCT__," not supported yet!");
}
#undef __FUNCT__ 
#define __FUNCT__ "Beam::GetOnBed"
int   Beam::GetOnBed(){
	throw ErrorException(__FUNCT__," not supported yet!");
}


#undef __FUNCT__ 
#define __FUNCT__ "Beam::GetBedList"
void  Beam::GetBedList(double*){
	throw ErrorException(__FUNCT__," not supported yet!");
}

#undef __FUNCT__ 
#define __FUNCT__ "Beam::Du"
void  Beam::Du(_p_Vec*,void*,int,int){
	throw ErrorException(__FUNCT__," not supported yet!");
}
#undef __FUNCT__ 
#define __FUNCT__ "Beam::Gradj"
void  Beam::Gradj(_p_Vec*, void*, int, int,char*){
	throw ErrorException(__FUNCT__," not supported yet!");
}
#undef __FUNCT__ 
#define __FUNCT__ "Beam::GradjDrag"
void  Beam::GradjDrag(_p_Vec*, void*, int,int ){
	throw ErrorException(__FUNCT__," not supported yet!");
}
#undef __FUNCT__ 
#define __FUNCT__ "Beam::GradjB"
void  Beam::GradjB(_p_Vec*, void*, int, int){
	throw ErrorException(__FUNCT__," not supported yet!");
}
#undef __FUNCT__ 
#define __FUNCT__ "Beam::Misfit"
double Beam::Misfit(void*,int,int ){
	throw ErrorException(__FUNCT__," not supported yet!");
}

#undef __FUNCT__ 
#define __FUNCT__ "Beam::GetThicknessList"
void  Beam::GetThicknessList(double* thickness_list){
	throw ErrorException(__FUNCT__," not supported yet!");
}

#undef __FUNCT__ 
#define __FUNCT__ "Beam::GetParameterValue"
void Beam::GetParameterValue(double* pvalue, double* value_list,double gauss_coord){

	double l1l2[2];
	
	GetNodalFunctions(&l1l2[0],gauss_coord);

	*pvalue=l1l2[0]*value_list[0]+l1l2[1]*value_list[1];
}


#undef __FUNCT__ 
#define __FUNCT__ "Beam::GetNodalFunctions"
void Beam::GetNodalFunctions(double* l1l2, double gauss_coord){
	
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	/*First nodal function: */
	l1l2[0]=.5*gauss_coord+.5;

	/*Second nodal function: */
	l1l2[1]=-.5*gauss_coord+.5;
}

#undef __FUNCT__ 
#define __FUNCT__ "Beam::GetJacobianDeterminant"
void Beam::GetJacobianDeterminant(double* pJdet,double* z_list, double gauss_coord){


	double Jdet;

	Jdet=1.0/2.0*(z_list[1]-z_list[0]);

	if(Jdet<0){
		throw ErrorException(__FUNCT__," negative jacobian determinant!");
	}
	
	*pJdet=Jdet;
}

#undef __FUNCT__ 
#define __FUNCT__ "Beam::ComputePressure"
void  Beam::ComputePressure(Vec p_g){

	int i;
	const int numgrids=2;
	int doflist[numgrids];
	double pressure[numgrids];
	double rho_ice,g;
	double xyz_list[numgrids][3];

	/*Get node data: */
	GetElementNodeData( &xyz_list[0][0], nodes, numgrids);
        
	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	for(i=0;i<numgrids;i++){
		pressure[i]=rho_ice*g*(s[i]-xyz_list[i][2]);
	}
	
	/*plug local pressure values into global pressure vector: */
	VecSetValues(p_g,numgrids,doflist,(const double*)pressure,INSERT_VALUES);

}
#undef __FUNCT__ 
#define __FUNCT__ "Beam::MassFlux"
double Beam::MassFlux( double* segment,double* ug){
	throw ErrorException(__FUNCT__," not supported yet!");
}
