/*!\file: CreateParametersQmu.cpp
 * \brief general driver for creating parameters dataset
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "CreateParametersQmu"

#include "../../DataSet/DataSet.h"
#include "../../toolkits/toolkits.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../objects/objects.h"
#include "../../shared/shared.h"
#include "../../include/macros.h"
#include "../../MeshPartitionx/MeshPartitionx.h"
#include "../IoModel.h"

void CreateParametersQmu(DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle){
	
	int i,j,k;
	
	DataSet* parameters = NULL;
	Param*   param = NULL;
	int      count;
	int      second_count;
	
	int*     epart=NULL;
	int*     part=NULL;
	double*  dpart=NULL;
	int      elements_width;

	char**  responsedescriptors=NULL;
	char**  variabledescriptors=NULL;
	char*   descriptor=NULL;
	char*   tag=NULL;
	double* dakota_parameter=NULL;

	//qmu files
	char* qmuinname=NULL;
	char* qmuerrname=NULL;
	char* qmuoutname=NULL;
	extern int my_rank;
				
	
	/*parameters for mass flux: */
	double* qmu_mass_flux_segments=NULL;
	double* my_qmu_mass_flux_segments=NULL;
	int num_qmu_mass_flux_segments=0;
	int my_num_qmu_mass_flux_segments=0;

	#ifdef _SERIAL_
		mxArray* pfield=NULL;
		mxArray* pfield2=NULL;
	#endif
	
	/*recover parameters : */
	parameters=*pparameters;

	count=parameters->Size();

	//qmu analysis?
	count++;
	param= new Param(count,"qmu_analysis",INTEGER);
	param->SetInteger(iomodel->qmu_analysis);
	parameters->AddObject(param);


	if(iomodel->qmu_analysis){

		//name of qmu input, error and output files
		qmuinname=(char*)xmalloc((strlen(iomodel->name)+strlen(".qmu.in")+1)*sizeof(char));
		sprintf(qmuinname,"%s%s",iomodel->name,".qmu.in");
		
		count++;
		param= new Param(count,"qmuinname",STRING);
		param->SetString(qmuinname);
		parameters->AddObject(param);

		qmuoutname=(char*)xmalloc((strlen(iomodel->name)+strlen(".qmu.out")+1)*sizeof(char));
		sprintf(qmuoutname,"%s%s",iomodel->name,".qmu.out");
		
		count++;
		param= new Param(count,"qmuoutname",STRING);
		param->SetString(qmuoutname);
		parameters->AddObject(param);

		qmuerrname=(char*)xmalloc((strlen(iomodel->name)+strlen(".qmu.err")+1)*sizeof(char));
		sprintf(qmuerrname,"%s%s",iomodel->name,".qmu.err");
		
		count++;
		param= new Param(count,"qmuerrname",STRING);
		param->SetString(qmuerrname);
		parameters->AddObject(param);

		//npart
		count++;
		param= new Param(count,"qmu_npart",INTEGER);
		param->SetInteger(iomodel->qmu_npart);
		parameters->AddObject(param);

		/*Deal with variables for qmu iomodeling: */
		variabledescriptors=(char**)xmalloc(iomodel->numberofvariables*sizeof(char*));


		/*Fetch descriptors: logic varies if we are running parallel or serial. In parallel, qmumarshall 
		 * took care of marshalling all the variable descriptors, so it's easy. In serial mode, 
		 * the variables are in md.variables(md.ivar), as a strucuture: */

		#ifdef _SERIAL_
		pfield=mxGetField(iomodel_handle,0,"variabledescriptors");
		for(i=0;i<iomodel->numberofvariables;i++){
			pfield2=mxGetCell(pfield,i);
			FetchData((void**)&descriptor,NULL,NULL,pfield2,"String",NULL);
			variabledescriptors[i]=descriptor;
		}
		#else
		tag=(char*)xmalloc((strlen("variabledescriptori")+1)*sizeof(char));
		for(i=0;i<iomodel->numberofvariables;i++){
			sprintf(tag,"%s%i","variabledescriptor",i);
			IoModelFetchData((void**)&descriptor,NULL,NULL,iomodel_handle,tag,"String",NULL);
			variabledescriptors[i]=descriptor;
		}
		#endif


		/*Ok, we have all the variable descriptors. Build a parameter with it: */
		count++;
		param= new Param(count,"variabledescriptors",STRINGARRAY);
		param->SetStringArray(variabledescriptors,iomodel->numberofvariables);
		parameters->AddObject(param);


		/*Deal with responses and partition for qmu iomodeling: */
		responsedescriptors=(char**)xmalloc(iomodel->numberofresponses*sizeof(char*));

		/*Fetch descriptors: */
		#ifdef _SERIAL_
		pfield=mxGetField(iomodel_handle,0,"responsedescriptors");
		for(i=0;i<iomodel->numberofresponses;i++){
			pfield2=mxGetCell(pfield,i);
			FetchData((void**)&descriptor,NULL,NULL,pfield2,"String",NULL);
			responsedescriptors[i]=descriptor;
		}
		#else
		xfree((void**)&tag);
		tag=(char*)xmalloc((strlen("responsedescriptori")+1)*sizeof(char));

		for(i=0;i<iomodel->numberofresponses;i++){
			sprintf(tag,"%s%i","responsedescriptor",i);
			IoModelFetchData((void**)&descriptor,NULL,NULL,iomodel_handle,tag,"String",NULL);
			responsedescriptors[i]=descriptor;
		}
		#endif


		/*Ok, we have all the response descriptors. Build a parameter with it: */
		count++;
		param= new Param(count,"responsedescriptors",STRINGARRAY);
		param->SetStringArray(responsedescriptors,iomodel->numberofresponses);
		parameters->AddObject(param);

		#ifdef _ISSM_DEBUG_
			for(i=0;i<iomodel->numberofvariables;i++){
				_printf_("variable descriptor %s\n",variabledescriptors[i]);
			}
			
			for(i=0;i<iomodel->numberofresponses;i++){
				_printf_("response descriptor %s\n",responsedescriptors[i]);
			}
		#endif


		/*partition grids in iomodel->qmu_npart parts, unless a partition is already present: */
		IoModelFetchData((void**)&dpart,NULL,NULL,iomodel_handle,"part","Matrix","Mat");

		if(!dpart){

			if(strcmp(iomodel->meshtype,"2d")==0){
				IoModelFetchData((void**)&iomodel->elements,NULL,NULL,iomodel_handle,"elements","Matrix","Mat");
				elements_width=3; //tria elements
			}
			else{
				IoModelFetchData((void**)&iomodel->elements2d,NULL,NULL,iomodel_handle,"elements2d","Matrix","Mat");
				elements_width=6; //penta elements
			}

			MeshPartitionx(&epart, &part,iomodel->numberofelements,iomodel->numberofnodes,iomodel->elements, iomodel->numberofelements2d,iomodel->numberofnodes2d,iomodel->elements2d,iomodel->numlayers,elements_width, iomodel->meshtype,iomodel->qmu_npart);

			dpart=(double*)xmalloc(iomodel->numberofnodes*sizeof(double));
			for(i=0;i<iomodel->numberofnodes;i++)dpart[i]=part[i];
		}

		count++;
		param= new Param(count,"qmu_part",DOUBLEVEC);
		param->SetDoubleVec(dpart,iomodel->numberofnodes,1);
		parameters->AddObject(param);


		/*Ok, now if any of the variables input from Dakota are distributed, we are going to need the parameters: */
		for(i=0;i<iomodel->numberofvariables;i++){

			descriptor=variabledescriptors[i];

			if ((strcmp(descriptor,"thickness")==0) ||
				(strcmp(descriptor,"drag")     ==0)
				){

				//Fetch data: 
				IoModelFetchData((void**)&dakota_parameter,NULL,NULL,iomodel_handle,descriptor,"Matrix","Mat");

				//Add parameter
				count++;
				param= new Param(count,descriptor,DOUBLEVEC);
				param->SetDoubleVec(dakota_parameter,iomodel->numberofnodes,1);
				parameters->AddObject(param);

				//free data
				xfree((void**)&dakota_parameter);

			}
		}


		/*Deal with data needed for some responses: */
		for(i=0;i<iomodel->numberofresponses;i++){
			char* descriptor=responsedescriptors[i];
			if (strcmp(descriptor,"mass_flux")==0){

				/*We need the qmu_mass_flux_segments to be able to compute the mass flux: */
				IoModelFetchData((void**)&qmu_mass_flux_segments,&num_qmu_mass_flux_segments,NULL,iomodel_handle,"qmu_mass_flux_segments","Matrix","Mat");

				#ifdef _PARALLEL_

					/*Only if partitioning exist do we care about the segments: */
					if(iomodel->epart){

						/*Use the element partitioning vector from the iomodel to down select qmu_mass_flux_segments to only segments that are relevant 
						 * to this cpu: */
						my_num_qmu_mass_flux_segments=0;
						for(j=0;j<num_qmu_mass_flux_segments;j++){
							if (  iomodel->epart[(int)(*(qmu_mass_flux_segments+5*j+4))-1]   == my_rank)my_num_qmu_mass_flux_segments++;
						}

					
						if(my_num_qmu_mass_flux_segments){
							my_qmu_mass_flux_segments=(double*)xcalloc(5*my_num_qmu_mass_flux_segments,sizeof(double));
							second_count=0;
							for(j=0;j<num_qmu_mass_flux_segments;j++){
								if (iomodel->epart[(int)*(qmu_mass_flux_segments+5*j+4)-1]==my_rank){
									for(k=0;k<5;k++)*(my_qmu_mass_flux_segments+5*second_count+k)=*(qmu_mass_flux_segments+5*j+k);
									second_count++;
								}
							}
						}

						count++;
						param= new Param(count,"qmu_mass_flux_segments",DOUBLEMAT);
						param->SetDoubleMat(my_qmu_mass_flux_segments,my_num_qmu_mass_flux_segments,5);
						parameters->AddObject(param);

					}
					
				#else

					count++;
					param= new Param(count,"qmu_mass_flux_segments",DOUBLEMAT);
					param->SetDoubleMat(qmu_mass_flux_segments,num_qmu_mass_flux_segments,5);
					parameters->AddObject(param);

				#endif

				xfree((void**)&qmu_mass_flux_segments);
				xfree((void**)&my_qmu_mass_flux_segments);
			}
		}

		/*Free data: */
		xfree((void**)&tag);
		for(i=0;i<iomodel->numberofresponses;i++){
			char* descriptor=responsedescriptors[i];
			xfree((void**)&descriptor);
		}
		xfree((void**)&responsedescriptors);

		for(i=0;i<iomodel->numberofvariables;i++){
			char* descriptor=variabledescriptors[i];
			xfree((void**)&descriptor);
		}
		xfree((void**)&variabledescriptors);

		xfree((void**)&iomodel->elements);
		xfree((void**)&iomodel->elements2d);
		xfree((void**)&epart);
		xfree((void**)&part);
		xfree((void**)&dpart);
		xfree((void**)&qmuinname);
		xfree((void**)&qmuerrname);
		xfree((void**)&qmuoutname);
	}

	/*Assign output pointer: */
	*pparameters=parameters;
}
