/*This is where we have all our webgl relevant functionality for the plotting routines: */
//{{{ Canvas Initialization
function initCanvas(options) {
	//Initialize open Gl for each canvas, if needed: 
	var canvas=document.getElementById(options.getfieldvalue('canvasid'));
	if (!canvas.initialized) {
		canvas.gl = initWebGL(canvas,options);
		canvas.nodes = [];
		if (canvas.drawHandler)	window.cancelAnimationFrame(canvas.drawHandler);
		draw(canvas,options);
		canvas.initialized = true;
	}
	return canvas;
}
function initWebGL(canvas,options) { //{{{
	var gl;
	try {
		if (!canvas.gl) gl = GL.create({canvas:canvas});
		else gl = canvas.gl;
	}
	catch(e) {
		console.log(e);
		return;
	}

	// Enable depth testing
	gl.enable(gl.DEPTH_TEST);
	// Near things obscure far things
	gl.depthFunc(gl.LEQUAL);
	// Enable color blending/overlay
	gl.enable(gl.BLEND);

	// Load shaders and store them in gl object
	gl.shaders = loadShaders(gl);
	
	// Add context state variables
	//TODO:Group variables in objects for organization and naming
	canvas.gl = gl;
	canvas.zoomBounds = options.getfieldvalue('zoomlim',[0.001,100.0]);
	canvas.zoom = clamp(options.getfieldvalue('zoom',1.0), canvas.zoomBounds[0], canvas.zoomBounds[1]);
	canvas.zoomLast = canvas.zoom;
	canvas.cameraMatrix = mat4.create();
	canvas.translation = options.getfieldvalue('origin',[0,0,0.0]);
	canvas.viewPanning = options.getfieldvalue('viewpanning','off') == 'on';
	canvas.view = options.getfieldvalue('view',[0,90]); //0 azimuth - up is north, 90 elevation - looking straight down
	canvas.rotation = canvas.view;
	canvas.rotationAzimuthBounds = options.getfieldvalue('azlim',[0,360]);
	canvas.rotationElevationBounds = options.getfieldvalue('ellim',[-180,180]);
	canvas.controlSensitivity = options.getfieldvalue('controlsensitivity',1);
	canvas.twod = options.getfieldvalue('2d','off') == 'on';
	canvas.moviePlay = true;
	canvas.movieReverse = false;
	canvas.movieIncrement = true;
	canvas.moviefps = options.getfieldvalue('moviefps',5);
	var backgroundcolor = new RGBColor(options.getfieldvalue('backgroundcolor','lightcyan'));
	if (backgroundcolor.ok) canvas.backgroundcolor = [backgroundcolor.r/255.0, backgroundcolor.g/255.0, backgroundcolor.b/255.0, 1.0];
	else throw Error(sprintf("s%s%s\n","initWebGL error message: cound not find out background color for curent canvas ",canvas));
	
	// Add event listeners for canvas
	var displayview = options.getfieldvalue('displayview','off') == 'on';
	var displayzoom = options.getfieldvalue('displayzoom','off') == 'on';
	var mc = new Hammer.Manager(canvas);
	
    mc.add(new Hammer.Pan({threshold:0, pointers:0}));
    mc.add(new Hammer.Pinch({threshold:0})).recognizeWith(mc.get('pan'));
    mc.on("panstart panmove", function (ev) {onPan(ev,canvas,displayview);});
    mc.on("pinchstart pinchmove", function (ev) {onPinch(ev,canvas,displayview);});
	//mc.on("mousewheel DOMMouseScroll", function (ev) {onZoom(ev,canvas,options);});
	
	canvas.addEventListener("mousewheel", function (ev) {onZoom(ev,canvas,displayzoom)}, false);
	canvas.addEventListener("DOMMouseScroll", function (ev) {onZoom(ev,canvas,displayzoom)}, false);
	
	return gl;
} //}}}
function initTexture(gl,imageSource) { //{{{
	return GL.Texture.fromURL(imageSource, {minFilter:gl.LINEAR_MIPMAP_LINEAR, magFilter:gl.LINEAR});
} //}}}
function Node(gl,options) { //{{{
	//Returns a Node object that contains default display states for webgl object
	return {buffers:[],
		shader:gl.shaders["colored"],
		draw:null,
		hideOcean:false,
		level:0,
		useIndexBuffer:true,
		alpha:1.0,
		lineWidth:1.0,
		disableDepthTest:false, 
		enableCullFace:false,
		cullFace:gl.FRONT,
		drawMode:gl.TRIANGLES,
		texture:null,
		translation:vec3.create(),
		rotation:vec3.fromValues(-90, 0, 0),
		scale:vec3.fromValues(1, 1, 1),
		modelMatrix:mat4.create(),
		shaderName:"colored",
		drawOrder:0,
		maskEnabled:false,
		maskHeight:150.0,
		maskColor:vec4.fromValues(0.0, 0.0, 1.0, 1.0),
		enabled:true,
	};
} //}}}
function recalculateModelMatrix(node) { //{{{
	//TODO: move to 0,0,0, rotate,move back to normal space, then apply transform
	var modelMatrix = mat4.create();

	var scaleMatrix = mat4.create();
	mat4.scale(scaleMatrix, scaleMatrix, node["scale"]);
	mat4.multiply(modelMatrix, scaleMatrix, modelMatrix);

	var translationMatrix = mat4.create();
	mat4.translate(translationMatrix, translationMatrix, node["translation"]); //relative translation
	mat4.multiply(modelMatrix, translationMatrix, modelMatrix);
	
	var zRotationMatrix = mat4.create();	
	mat4.rotate(zRotationMatrix, zRotationMatrix, radians(node["rotation"][2]), [0.0, 0.0, 1.0]);
	mat4.multiply(modelMatrix, zRotationMatrix, modelMatrix);
	var yRotationMatrix = mat4.create();	
	mat4.rotate(yRotationMatrix, yRotationMatrix, radians(node["rotation"][1]), [0.0, 1.0, 0.0]);
	mat4.multiply(modelMatrix, yRotationMatrix, modelMatrix);
	var xRotationMatrix = mat4.create();	
	mat4.rotate(xRotationMatrix, xRotationMatrix, radians(node["rotation"][0]), [1.0, 0.0, 0.0]);
	mat4.multiply(modelMatrix, xRotationMatrix, modelMatrix);

	return modelMatrix;
} //}}}
function radians (degrees) { //{{{
  return degrees * Math.PI / 180;
} //}}}
function degrees (radians) { //{{{
  return radians * 180 / Math.PI;
} //}}}
function clamp(value, min, max) { //{{{
	return Math.max(min, Math.min(value, max));
} //}}}
function recover(canvasid,name,defaultvalue) { //{{{
	var canvas  = document.getElementById(canvasid);
	if (canvas && canvas.hasOwnProperty(name)) {
		return canvas[name];
	}
	return defaultvalue;
} //}}}
//}}}
//{{{ Shader Loading
function loadShaders(gl) { //{{{
	var shaderNames = ["colored", "unlit_textured"];
	shaders = {};
	shaders["colored"] = {loaded:false, vsh:{}, fsh:{}};
			//basic phong shader
	shaders["colored"] = new Shader('\
		precision highp float;\
		attribute vec3 a_vertex;\
		attribute vec4 a_color;\
		uniform mat4 u_mvp;\
		uniform float u_alpha;\
		varying vec4 v_color;\
		void main() {\
			gl_PointSize = 3.0;\
			gl_Position = u_mvp * vec4(a_vertex.xyz, 1.0);\
			v_color = vec4(a_color.xyz, u_alpha);\
		}\
		', '\
		precision mediump float;\
		varying vec4 v_color;\
		void main() {\
			gl_FragColor = v_color;\
		}\
	');
			
	shaders["unlit_textured"] = new Shader('\
		precision highp float;\
		attribute vec3 a_vertex;\
		attribute vec2 a_coord;\
		uniform mat4 u_mvp;\
		varying vec2 v_coord;\
		varying float v_z;\
		void main() {\
			gl_PointSize = 3.0;\
			gl_Position = u_mvp * vec4(a_vertex.xyz, 1.0);\
			v_coord = a_coord;\
			v_z = a_vertex.z;\
		}\
		', '\
		precision mediump float;\
		varying vec2 v_coord;\
		varying float v_z;\
		uniform sampler2D u_texture;\
		uniform float u_alpha;\
		uniform bool u_maskEnabled;\
		uniform float u_maskHeight;\
		uniform vec4 u_maskColor;\
		void main() {\
			if (u_maskEnabled && (v_z < u_maskHeight)) {\
				gl_FragColor = vec4(u_maskColor.rgb, u_alpha);\
			}\
			else {\
				gl_FragColor = vec4(texture2D(u_texture, v_coord).rgb, u_alpha);\
			}\
		}\
	');
	/*
	shaders["phong"] = new Shader('\
		precision highp float;\
		attribute vec3 a_vertex;\
		attribute vec3 a_normal;\
		attribute vec2 a_coord;\
		varying vec3 v_normal;\
		varying vec2 v_coord;\
		uniform mat4 u_mvp;\
		uniform mat4 u_model;\
		void main() {\
			v_coord = a_coord;\
			v_normal = (u_model * vec4(a_normal,0.0)).xyz;\
			gl_Position = u_mvp * vec4(a_vertex,1.0);\
		}\
		', '\
		precision highp float;\
		varying vec3 v_normal;\
		varying vec2 v_coord;\
		uniform vec3 u_lightvector;\
		uniform vec4 u_color;\
		uniform sampler2D u_texture;\
		void main() {\
		  vec3 N = normalize(v_normal);\
		  vec4 color = u_color * texture2D( u_texture, v_coord);\
		  gl_FragColor = color * max(0.0, dot(u_lightvector,N));\
		}\
	');
	*/
	return shaders;
} //}}}
//{{{ Interface Functions
function onPan(ev,canvas,displaylog) { //{{{
	ev.preventDefault();
	if (ev.type == 'panstart') {
		canvas.lastDeltaX = 0;
		canvas.lastDeltaY = 0;
	}
	if (ev.srcEvent.shiftKey || ev.pointers.length == 2) {
		if (!canvas.viewPanning) return;
		var deltaX = (canvas.lastDeltaX - ev.deltaX) / canvas.clientWidth / canvas.zoom * -2 * canvas.controlSensitivity;
		var deltaY = (canvas.lastDeltaY - ev.deltaY) / canvas.clientHeight / canvas.zoom * -2 * canvas.controlSensitivity;
		
		if (canvas.twod) {
			canvas.translation[0] += Math.cos(radians(canvas.rotation[0])) * deltaX - Math.sin(radians(0)) * deltaY;
			canvas.translation[1] += Math.sin(radians(canvas.rotation[0])) * deltaX + Math.cos(radians(0)) * deltaY;
		}
		else {
			canvas.translation[0] += Math.cos(radians(canvas.rotation[0])) * deltaX - Math.sin(radians(canvas.rotation[0])) * deltaY;
			canvas.translation[1] += Math.sin(radians(canvas.rotation[0])) * deltaX + Math.cos(radians(canvas.rotation[0])) * deltaY;
		}
	}
	else {
		canvas.rotation[0] += degrees((canvas.lastDeltaX - ev.deltaX) / canvas.clientWidth * -2 * canvas.controlSensitivity);
		canvas.rotation[1] += degrees((canvas.lastDeltaY - ev.deltaY) / canvas.clientHeight * -2 * canvas.controlSensitivity);
		
		if (canvas.rotation[0] > 360) {canvas.rotation[0] -= 360};
		if (canvas.rotation[0] < -360) {canvas.rotation[0] += 360};
		if (canvas.rotation[1] > 180) {canvas.rotation[1] -= 360};
		if (canvas.rotation[1] < -180) {canvas.rotation[1] += 360};
		
		canvas.rotation[0] = clamp(canvas.rotation[0], canvas.rotationAzimuthBounds[0], canvas.rotationAzimuthBounds[1]);
		canvas.rotation[1] = clamp(canvas.rotation[1], canvas.rotationElevationBounds[0], canvas.rotationElevationBounds[1])
	}
	canvas.lastDeltaX = ev.deltaX;
	canvas.lastDeltaY = ev.deltaY;

	if (displaylog) console.log(canvas.rotation);
} //}}}
function onPinch(ev,canvas,displaylog) { //{{{
	ev.preventDefault();
	if (ev.type == 'pinchstart') {
		canvas.zoomLast = canvas.zoom;
	}
	else {
		canvas.zoom = clamp(ev.scale * canvas.zoomLast, canvas.zoomBounds[0], canvas.zoomBounds[1]);
		if (displaylog) console.log(canvas.zoom);
	}
} //}}}
function onZoom(ev,canvas,displaylog) { //{{{
	ev.preventDefault();
	var delta = clamp(clamp(ev.scale || ev.wheelDelta || -ev.detail, -1, 1) * canvas.controlSensitivity * canvas.zoom / 20, -1.0, 1.0);
	canvas.zoom = clamp(canvas.zoom + delta, canvas.zoomBounds[0], canvas.zoomBounds[1]);
	
	if (displaylog) console.log(canvas.zoom);
} //}}}
//}}}
//{{{ Drawing Functions
function updateCameraMatrix(canvas) { //{{{
    //Update view matrix and multiply with projection matrix to get the view-projection (camera) matrix.
	var vMatrix = mat4.create();
	var pMatrix = mat4.create();
	var translateMatrix = mat4.create();
	var rotationMatrix = mat4.create();
	var azimuthRotationMatrix = mat4.create();
	var elevationRotationMatrix = mat4.create();
	var aspectRatio = canvas.clientWidth / canvas.clientHeight;

	if (canvas.twod) {
		mat4.ortho(pMatrix, -aspectRatio/canvas.zoom, aspectRatio/canvas.zoom, -1/canvas.zoom, 1/canvas.zoom, -1.0, 10000.0);
	}
	else {
		mat4.perspective(pMatrix, 60 * Math.PI / 180, aspectRatio, 0.001, 10000.0);
	}
	
	//Apply worldspace translation
	mat4.translate(translateMatrix, translateMatrix, [canvas.translation[0], canvas.translation[2], canvas.translation[1]]);
	mat4.multiply(vMatrix, translateMatrix, vMatrix);
	
	//Calculate rotation around camera focal point about worldspace origin
	if (canvas.twod) {
		mat4.rotate(azimuthRotationMatrix, azimuthRotationMatrix, radians(0), [0, 1, 0]);
		mat4.rotate(elevationRotationMatrix, elevationRotationMatrix, radians(90), [1, 0, 0]);
		mat4.multiply(rotationMatrix, elevationRotationMatrix, azimuthRotationMatrix);
	}
	else {
		mat4.rotate(azimuthRotationMatrix, azimuthRotationMatrix, radians(canvas.rotation[0]), [0, 1, 0]);
		mat4.rotate(elevationRotationMatrix, elevationRotationMatrix, radians(canvas.rotation[1]), [1, 0, 0]);
		mat4.multiply(rotationMatrix, elevationRotationMatrix, azimuthRotationMatrix);
	}
	
	//Apply rotation and scaling transform
	mat4.multiply(vMatrix, rotationMatrix, vMatrix);
	
	//Apply screenspace translation
	mat4.identity(translateMatrix);
	mat4.translate(translateMatrix, translateMatrix, [0.0, 0.0, -1/canvas.zoom]);
	mat4.multiply(vMatrix, translateMatrix, vMatrix);

	//Apply projection matrix to get camera matrix
	mat4.multiply(canvas.cameraMatrix, pMatrix, vMatrix);
}//}}}
function drawSceneGraphNode(canvas,node) { //{{{
	if (!node["enabled"]) return;
	
	var mvpMatrix = mat4.create();
	mat4.multiply(mvpMatrix, canvas.cameraMatrix, node["modelMatrix"]);
	
	if (node["texture"]) node["texture"].bind(0);
	if (node["disableDepthTest"]) gl.disable(gl.DEPTH_TEST);
	
	gl.lineWidth(node["lineWidth"]);
	gl.blendFunc(gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA);

	node["shader"].uniforms({
		u_mvp: mvpMatrix,
		u_texture: 0,
		u_alpha: node["alpha"],
		u_maskEnabled: node["maskEnabled"],
		u_maskHeight: node["maskHeight"],
		u_maskColor: node["maskColor"]
	})
	if (node["useIndexBuffer"] == true) node["shader"].draw(node["mesh"], node["drawMode"], "indices");
	else node["shader"].draw(node["mesh"], node["drawMode"]);

	gl.enable(gl.DEPTH_TEST);
} //}}}
function draw(canvas,options) { //{{{
	// Ensure canvas and gl viewport sizes are the same
	var displayWidth  = canvas.clientWidth;
	var displayHeight = canvas.clientHeight;
	if (canvas.width  != displayWidth || canvas.height != displayHeight) {
		canvas.width  = displayWidth;
		canvas.height = displayHeight;
		canvas.gl.viewport(0, 0, canvas.width, canvas.height);
	}
	
	if (canvas.textcanvas) canvas.textcanvas.draw(canvas);

	var nodes = canvas.nodes;
	if (nodes.length < 1) {
		canvas.drawHandler = window.requestAnimationFrame(function(time) {draw(canvas,options)});
		return;
	}
	for (var node in nodes) {
		if (nodes[node]["texture"] && nodes[node]["texture"]["ready"] == false) {
			canvas.drawHandler = window.requestAnimationFrame(function(time) {draw(canvas,options)});
			return;
		}
	}
	
	var gl = canvas.gl;
	gl.clearColor(canvas.backgroundcolor[0], canvas.backgroundcolor[1], canvas.backgroundcolor[2], canvas.backgroundcolor[3]);
	gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
	
	
	
	updateCameraMatrix(canvas);
	
	var drawPassNumber = 2;
	for (var i = drawPassNumber - 1; i >= 0; i--) {
		for (var node in nodes) {
			if (nodes[node]["drawOrder"] == i) drawSceneGraphNode(canvas,nodes[node]);
		}
	}
	canvas.drawHandler = window.requestAnimationFrame(function(time) {draw(canvas,options)});
} //}}}
//}}}
