function ArrayMax(array){ //{{{
	return Math.max.apply(null,array);
} //}}}
function ArrayMax2D(array){ //{{{
	
	var max=0;

	for (var i=0;i<array.length;i++){
		var subarray=array[i];
		max=Math.max(max,ArrayMax(subarray));
	}

	return max;
} //}}}
function ArrayMin(array){ //{{{
	return Math.min.apply(null,array);
} //}}}
function ArraySum(array){ //{{{
	var sum=0;
	for(var i=0;i<array.length;i++)sum+=array[i];
	return sum;
} //}}}
function ArrayXPY(){ //{{{
    if (arguments.length<2)throw Error("ArrayXPY error message: sum has to be for at least two arrays!");

	//check internal consistency of arrays provided!: 
	var firstarray=arguments[0];
	var firstsize=firstarray.length;
	
	for(var a=1;a<arguments.length;a++){
		var array=arguments[a];
		if(array.length!=firstsize)throw Error("ArrayXPY error message: arrays provided as arguments are not of the same length!");
	}

	//do the sum:
	var sum=NewArrayFill(firstsize,0);
	for(var a=0;a<arguments.length;a++){
		var array=arguments[a];
		for(var i=0;i<array.length;i++){
			sum[i]+=array[i];
		}
	}
	return sum;

} //}}}
function ArrayOr(){ //{{{
    if (arguments.length<2)throw Error("ArrayOr error message: sum has to be for at least two arrays!");

	//check internal consistency of arrays provided!: 
	var firstarray=arguments[0];
	var firstsize=firstarray.length;
	
	for(var a=1;a<arguments.length;a++){
		var array=arguments[a];
		if(array.length!=firstsize)throw Error("ArrayOr error message: arrays provided as arguments are not of the same length!");
	}

	//do the or:
	var or=NewArrayFill(firstsize,0);
	for(var a=0;a<arguments.length;a++){
		var array=arguments[a];
		for(var i=0;i<array.length;i++){
			or[i] = or[i] | array[i];
		}
	}
	return or;

} //}}}
function ArrayMin2D(array){ //{{{
	
	var min=ArrayMax2D(array);

	for (var i=0;i<array.length;i++){
		var subarray=array[i];
		min=Math.min(min,ArrayMin(subarray));
	}

	return min;
} //}}}
function ListToMatrix(list, elementsPerSubArray) { //{{{
	var matrix = [], i, k;

	for (i = 0, k = -1; i < list.length; i++) {
		if (i % elementsPerSubArray === 0) {
			k++;
			matrix[k] = [];
		}

		matrix[k].push(list[i]);
	}

	return matrix;
} //}}}
function MatrixToList(matrixin) { //{{{

	var matrix=matrixin;

	if (!IsArray(matrix[0])) return matrix;
	else{
		var width = matrix[0].length;
		var length = matrix.length;
		var list= new Array(width*length);

		for(var i=0;i<length;i++){
			for(var j=0;j<width;j++){
				list[i*width+j]=matrix[i][j];
			}
		}
		return list;
	}
} //}}}
function IsArray(object) { //{{{

	var type=Object.prototype.toString.call( object );
	if( type === '[object Array]' ) return 1;
	if( type === '[object Float64Array]' ) return 1;
	if( type === '[object Float32Array]' ) return 1;
	if( type === '[object Int32Array]' ) return 1;
	if( type === '[object Int16Array]' ) return 1;
	if( type === '[object Uint32Array]' ) return 1;
	if( type === '[object Uint16Array]' ) return 1;
	if( type === '[object Uint8Array]' ) return 1;
	return 0;

} //}}}
function ArrayNot(array) { //{{{

	var notarray=array;
	for (var i=0;i<array.length;i++)notarray[i]=-array[i];
	return notarray;
} //}}}
function ArrayCopy(array) { //{{{

	var copy=[];
	for(var i=0;i<array.length;i++)copy[i]=array[i];
	return copy;
} //}}}
function ArrayPow(array,coefficient) { //{{{

	var powarray=array;
	for (var i=0;i<array.length;i++)powarray[i]=Math.pow(array[i],coefficient);
	return powarray;
} //}}}
function ArraySqrt(array) { //{{{

	var sqrtarray=array;
	for (var i=0;i<array.length;i++)sqrtarray[i]=Math.sqrt(array[i]);
	return sqrtarray;
} //}}}
function ArrayScale(array,alpha) { //{{{

	for (var i=0;i<array.length;i++)array[i]=array[i]*alpha;

} //}}}
function ArrayMag(array1,array2) { //{{{

	var arraymag=NewArrayFill(array1.length,0);
	for (var i=0;i<array1.length;i++)arraymag[i]=Math.sqrt(Math.pow(array1[i],2)+Math.pow(array2[i],2));
	return arraymag;
} //}}}
function ArrayAnyNaN(array) { //{{{

    if(IsArray(array[0])){
        for(var i=0;i<array.length;i++){
            for(var j=0;j<array[0].length;j++){
                if (isNaN(array[i][j])) return 1;
            }
        }
    }
    else{
        for(var i=0;i<array.length;i++){
            if (isNaN(array[i])) return 1;
        }
    }
    return 0;
} //}}}
function ArrayUnique(arr) { //{{{

	return arr.reverse().filter(function (e, i, arr) {
		    return arr.indexOf(e, i+1) === -1;
	}).reverse();
} //}}}
function ArraySort(array) { //{{{

	return array.sort(function(a, b) {
		return a - b;
	});

} //}}}
function ArrayAnyEqual(array,value) { //{{{
	
	if(!isNaN(value)){
		for(var i=0;i<array.length;i++){
			if (array[i]==value)return 1;
		}
	}
	else{
		for(var i=0;i<array.length;i++){
			if (isNaN(array[i]))return 1;
		}
	}
	return 0;
} //}}}
function ArrayAnyBelowOrEqual(array,value) { //{{{

	for(var i=0;i<array.length;i++){
		if (array[i]<=value)return 1;
	}
	return 0;
} //}}}
function ArrayAnyBelowStrict(array,value) { //{{{

	for(var i=0;i<array.length;i++){
		if (array[i]<value)return 1;
	}
	return 0;
} //}}}
function ArrayAnyAboveOrEqual(array,value) { //{{{

	for(var i=0;i<array.length;i++){
		if (array[i]>=value)return 1;
	}
	return 0;
} //}}}
function ArrayAnyAboveStrict(array,value) { //{{{

	for(var i=0;i<array.length;i++){
		if (array[i]>value)return 1;
	}
	return 0;
} //}}}
function ArrayAnd(array1,array2) { //{{{

	var array=array1;
	for (var i=0;i<array1.length;i++)array[i]=array1[i] & array2[i];
	return array;
} //}}}
function ArrayIsMember(array1,array2) { //{{{

	var array=NewArrayFill(array1.length,0);
	for (var i=0;i<array1.length;i++){
		for(var j=0;j<array2.length;j++){
			if (array1[i] == array2[j]){
				array[i]=1;
				break;
			}
		}
	}
	return array;
} //}}}
function NewArrayFill(size,value) { //{{{

	return new Array(size).fill(value);
} //}}}
function NewArrayFillIncrement(size,start,increment) { //{{{

	var array=new Array(size); 

	for(var i=0;i<size;i++){
		array[i]=start+i*increment;
	}

	return array;
} //}}}
function ArrayFind(array,value) { //{{{
	
	//find number of indices
	var count=0;
	for (var i=0;i<array.length;i++)if(array[i]==value)count++;

	//allocate:
	var indices= NewArrayFill(count,0);

	//fill in:
	count=0;
	for (var i=0;i<array.length;i++){
		if(array[i]==value){
			indices[count]=i;
			count++;
		}
	}
	return indices;
} //}}}
function ArrayFindNot(array,value) { //{{{
	
	//find number of indices
	var count=0;
	for (var i=0;i<array.length;i++)if(array[i]!=value)count++;

	//allocate:
	var indices= NewArrayFill(count,0);

	//fill in:
	count=0;
	for (var i=0;i<array.length;i++){
		if(array[i]!=value){
			indices[count]=i;
			count++;
		}
	}
	return indices;
} //}}}
function Create2DArray(rows,cols) { //{{{
	var arr = [];

	for (var i=0;i<rows;i++) {
		arr[i] = new Array(cols);
	}

	return arr;
} //}}}
function MapIsEmpty(map) { //{{{
	for (var key in map){
		if(map.hasOwnProperty(key)){
			return false;
		}
	}
	return true;
} //}}}
function clone(obj) {//{{{
	
	var copy;

	// Handle the 3 simple types, and null or undefined
	if (null == obj || "object" != typeof obj) return obj;

	// Handle Date
	if (obj instanceof Date) {
		copy = new Date();
		copy.setTime(obj.getTime());
		return copy;
	}

	// Handle Array
	if (obj instanceof Array) {
		copy = [];
		for (var i = 0, len = obj.length; i < len; i++) {
			copy[i] = clone(obj[i]);
		}
		return copy;
	}

	// Handle Object
	if (obj instanceof Object) {
		copy = {};
		for (var attr in obj) {
			if (obj.hasOwnProperty(attr)) copy[attr] = clone(obj[attr]);
		}
		return copy;
	}

	throw new Error("Unable to copy obj! Its type isn't supported.");
} //}}}
function FloatFix(pointer,size) {//{{{

	var buffer=new Float64Array(size);
	for(var i=0;i<size;i++)buffer[i]=pointer[i];
	return buffer;


} //}}}
function NullFix(pointer,value) {//{{{

	if(pointer==null)return value;
	else{
		//check that the pointer values are not null: 
		if(IsArray(pointer)){
			if(IsArray(pointer[0])){
				for(var i=0;i<pointer.length;i++){
					for(var j=0;j<pointer[0].length;j++){
						if(pointer[i][j]==null)pointer[i][j]=value;
					}
				}	
			}
			else{
				for(var i=0;i<pointer.length;i++){
					if(pointer[i]==null)pointer[i]=value;
				}
			}
		}
		return pointer;
	}

} //}}}
function SetIceShelfBC(md) { 
//SETICESHELFBC - Create the boundary conditions for stressbalance and thermal models for a  Ice Shelf with Ice Front
//
//   Neumann BC are used on the ice front (an ANRGUS contour around the ice front
//   must be given in input)
//   Dirichlet BC are used elsewhere for stressbalance
//
//   Usage:
//      md=SetIceShelfBC(md,varargin)
//
//   Example:
//      SetIceShelfBC(md);
//      SetIceShelfBC(md,'Front.exp');
//
//   See also: SETICESHEETBC, SETMARINEICESHEETBC

	//node on Dirichlet (boundary and ~icefront)
	if (arguments.length==2){
		icefront=arguments[1];
		nodeinsideicefront=ContourToMesh(md.mesh.elements,md.mesh.x,md.mesh.y,icefront,'node',2);
		nodeonicefront=ArrayAnd(md.mesh.vertexonboundary,nodeinsideicefront);
	}
	else if(arguments.length==1){
		nodeonicefront=NewArrayFill(md.mesh.numberofvertices,0);
	}
	else{
		throw Error('SetIceShelfBC usage error');
	}

	md.stressbalance.spcvx=NewArrayFill(md.mesh.numberofvertices,NaN); 
	md.stressbalance.spcvy=NewArrayFill(md.mesh.numberofvertices,NaN);
	md.stressbalance.spcvz=NewArrayFill(md.mesh.numberofvertices,NaN);
	md.stressbalance.referential=Create2DArray(md.mesh.numberofvertices,6);
	for(var i=0;i<md.mesh.numberofvertices;i++)for(var j=0;j<6;j++)md.stressbalance.referential[i][j]=NaN;
	md.stressbalance.loadingforce=NewArrayFill(md.mesh.numberofvertices,0);

	//Ice front position: 
	pos=ArrayFind(nodeonicefront,1);
	for(var i=0;i<pos.length;i++)md.mask.ice_levelset[pos[i]]=0;

	//First find segments that are not completely on the front
	if (md.mesh.elementtype() === 'Penta'){
		numbernodesfront=4;
	}
	else if (md.mesh.elementtype() === 'Tria'){
		numbernodesfront=2;
	}
	else{
		throw Error('mesh type not supported yet');
	}
	var obs=false;
	if((md.inversion.vx_obs.length == md.mesh.numberofvertices) & (md.inversion.vy_obs.length==md.mesh.numberofvertices))obs=true;

	if(obs==true){
		console.log('      boundary conditions for stressbalance model: setting spc as observed velocities');
	}
	else{
		console.log('      boundary conditions for stressbalance model: setting spc as zero');
	}
	for(var i=0;i<md.mesh.segments.length;i++){
		var sum=0;
		for (var j=0;j<numbernodesfront;j++) sum+=md.mask.ice_levelset[md.mesh.segments[i][j]-1];
		if(sum!=0){
			for (var j=0;j<numbernodesfront;j++){
				if(obs==false){
					md.stressbalance.spcvx[md.mesh.segments[i][j]-1]=0;
					md.stressbalance.spcvy[md.mesh.segments[i][j]-1]=0;
				}
				else{
					md.stressbalance.spcvx[md.mesh.segments[i][j]-1]=md.inversion.vx_obs[md.mesh.segments[i][j]-1];
					md.stressbalance.spcvy[md.mesh.segments[i][j]-1]=md.inversion.vy_obs[md.mesh.segments[i][j]-1];
				}
				md.stressbalance.spcvz[md.mesh.segments[i][j]-1]=0;

			}
		}
	}

	//Initialize surface and basal forcings
	md.smb.initialize(md);
	md.basalforcings.initialize(md);

	//Deal with other boundary conditions
	if (isNaN(md.balancethickness.thickening_rate)){
		md.balancethickness.thickening_rate=NewArrayFill(md.mesh.numberofvertices,0);
		console.log('      no balancethickness.thickening_rate specified: values set as zero');
	}
		
	md.masstransport.spcthickness=NewArrayFill(md.mesh.numberofvertices,NaN);
	md.balancethickness.spcthickness=NewArrayFill(md.mesh.numberofvertices,NaN);
	md.damage.spcdamage=NewArrayFill(md.mesh.numberofvertices,NaN);

	if (md.initialization.temperature.length==md.mesh.numberofvertices){
		md.thermal.spctemperature=NewArrayFill(md.mesh.numberofvertices,NaN);
		if ('vertexonsurface' in md.mesh){
			pos=ArrayFind(md.mesh.vertexonsurface,1);
			for(var i=0;i<pos.length;i++)md.thermal.spctemperature[i]=md.initialization.temperature[i]; //impose observed temperature on surface
		}
		if (md.basalforcings.geothermalflux.length != md.mesh.numberofvertices){
			md.basalforcings.geothermalflux=NewArrayFill(md.mesh.numberofvertices,0);
		}
	}
	else{
		console.log('      no thermal boundary conditions created: no observed temperature found');
	}
}
//AUTODIFF class definition
//
//   Usage:
//      autodiff=new autodiff();

function autodiff (){
	//methods
	this.setdefaultparameters = function(){// {{{

		this.obufsize     = 524288;
		this.lbufsize     = 524288;
		this.cbufsize     = 524288;
		this.tbufsize     = 524288;
		this.gcTriggerRatio=2.0;
		this.gcTriggerMaxSize=65536;

	}// }}}
	this.disp= function(){// {{{

		console.log(sprintf('   automatic differentiation parameters:'));
		fielddisplay(this,'isautodiff','indicates if the automatic differentiation is activated');
		fielddisplay(this,'dependents','list of dependent variables');
		fielddisplay(this,'independents','list of independent variables');
		fielddisplay(this,'driver',"ADOLC driver ('fos_forward' or 'fov_forward')");
		fielddisplay(this,'obufsize','Number of operations per buffer (==OBUFSIZE in usrparms.h)');
		fielddisplay(this,'lbufsize','Number of locations per buffer (==LBUFSIZE in usrparms.h)');
		fielddisplay(this,'cbufsize','Number of values per buffer (==CBUFSIZE in usrparms.h)');
		fielddisplay(this,'tbufsize','Number of taylors per buffer (<=TBUFSIZE in usrparms.h)');
		fielddisplay(this,'gcTriggerRatio','free location block sorting/consolidation triggered if the ratio between allocated and used locations exceeds gcTriggerRatio');
		fielddisplay(this,'gcTriggerMaxSize','free location block sorting/consolidation triggered if the allocated locations exceed gcTriggerMaxSize');

	}// }}}
	this.classname= function(){// {{{
		return "autodiff";
	}// }}}
		this.checkconsistency = function(md,solution,analyses){ //{{{

			//Early return 
			if (!this.isautodiff) return; 

			//Driver value:
			checkfield(md,'fieldname','autodiff.driver','values',['fos_forward','fov_forward','fov_forward_all','fos_reverse','fov_reverse','fov_reverse_all']);
			
			//buffer values: 
			checkfield(md,'fieldname','autodiff.obufsize','>=',16);
			checkfield(md,'fieldname','autodiff.lbufsize','>=',16);
			checkfield(md,'fieldname','autodiff.cbufsize','>=',16);
			checkfield(md,'fieldname','autodiff.tbufsize','>=',16);
			checkfield(md,'fieldname','autodiff.gcTriggerRatio','>=',0);
			checkfield(md,'fieldname','autodiff.gcTriggerMaxSize','>=',65536);

			//go through our dependents and independents and check consistency: 
			for (var i=0;i<this.dependents.length;i++){
				dep=this.dependents[i];
				dep.checkconsistency(md,solution,analyses);
			}
			for (var i=0;i<this.independents.length;i++){
				indep=this.independents[i];
				indep.checkconsistency(md,i,solution,analyses,this.driver);
			}
		} // }}}
		this.marshall=function(md,fid) { //{{{

			WriteData(fid,'object',this,'fieldname','isautodiff','format','Boolean');
			WriteData(fid,'object',this,'fieldname','driver','format','String');

			//early return
			if (!this.isautodiff){
				WriteData(fid,'data',false,'enum',AutodiffMassFluxSegmentsPresentEnum(),'format','Boolean');
				WriteData(fid,'data',false,'enum',AutodiffKeepEnum(),'format','Boolean');
				return;
			}

			//buffer sizes {{{
			WriteData(fid,'object',this,'fieldname','obufsize','format','Double');
			WriteData(fid,'object',this,'fieldname','lbufsize','format','Double');
			WriteData(fid,'object',this,'fieldname','cbufsize','format','Double');
			WriteData(fid,'object',this,'fieldname','tbufsize','format','Double');
			WriteData(fid,'object',this,'fieldname','gcTriggerRatio','format','Double');
			WriteData(fid,'object',this,'fieldname','gcTriggerMaxSize','format','Double');
			//}}}
			//process dependent variables {{{
			num_dependent_objects=this.dependents.length;
			WriteData(fid,'data',num_dependent_objects,'enum',AutodiffNumDependentObjectsEnum(),'format','Integer');

			if(num_dependent_objects){
				var names=[];
				types=NewArrayFill(num_dependent_objects,0);
				indices=NewArrayFill(num_dependent_objects,0);

				for (var i=0;i<num_dependent_objects;i++){
					dep=this.dependents[i];

					names.push(dep.name);
					types[i]=dep.typetoscalar();
					indices[i]=dep.index;
				}
				WriteData(fid,'data',names,'enum',AutodiffDependentObjectNamesEnum(),'format','StringArray');
				WriteData(fid,'data',types,'enum',AutodiffDependentObjectTypesEnum(),'format','IntMat','mattype',3);
				WriteData(fid,'data',indices,'enum',AutodiffDependentObjectIndicesEnum(),'format','IntMat','mattype',3);
			}
			//}}}
			//process independent variables {{{
			num_independent_objects=this.independents.length;
			WriteData(fid,'data',num_independent_objects,'enum',AutodiffNumIndependentObjectsEnum(),'format','Integer');

			if(num_independent_objects){
				names=NewArrayFill(num_independent_objects,0);
				types=NewArrayFill(num_independent_objects,0);

				for (var i=0;i<num_independent_objects;i++){
					indep=this.independents[i];

					names[i]=StringToEnum(indep.name);
					types[i]=indep.typetoscalar();
				}
				WriteData(fid,'data',names,'enum',AutodiffIndependentObjectNamesEnum(),'format','IntMat','mattype',3);
				WriteData(fid,'data',types,'enum',AutodiffIndependentObjectTypesEnum(),'format','IntMat','mattype',3);
			}
			//}}}
			//if driver is fos_forward, build index:  {{{
			if (this.driver == 'fos_forward'){
				var index=0;

				for (var i=0;i<num_independent_objects;i++){
					indep=this.independents[i];
					if (!(isNaN(indep.fos_forward_index))){
						index=index+indep.fos_forward_index;
						break;
					}
					else{
						if (indep.type=='scalar') index=index+1;
						else index=index+indep.nods;
					}
				}
				index=index-1; //get c-index numbering going
				WriteData(fid,'data',index,'enum',AutodiffFosForwardIndexEnum(),'format','Integer');
			}
			//}}}
			//if driver is fos_reverse, build index:  {{{
			if (this.driver  == 'fos_reverse'){
				var index=0;

				for (var i=0;i<num_dependent_objects;i++){
					dep=this.dependents[i];
					if (!(isNaN(dep.fos_reverse_index))){
						index=index+dep.fos_reverse_index;
						break;
					}
					else{
						if (dep.type =='scalar') index=index+1;
						else index=index+dep.nods;
					}
				}
				index=index-1; //get c-index numbering going
				WriteData(fid,'data',index,'enum',AutodiffFosReverseIndexEnum(),'format','Integer');
			}
			//}}}
			//if driver is fov_forward, build indices:  {{{
			if (this.driver == 'fov_forward'){
				var indices=0;

				for (var i=0;i<num_independent_objects;i++){
					indep=this.independents[i];
					if (!indep.fos_forward_index.length){
						indices=indices+indep.fov_forward_indices;
						break;
					}
					else{
						if (indep.type =='scalar') indices=indices+1;
						else indices=indices+indep.nods;
					}
				}
				indices=indices-1; //get c-indices numbering going
				WriteData(fid,'data',indices,'enum',AutodiffFovForwardIndicesEnum(),'format','IntMat','mattype',3);
			}
			//}}}
			//deal with mass fluxes:  {{{
			mass_flux_segments=[];
			for (var i=0;i<num_dependent_objects;i++){
				dep=this.dependents[i];
				if (dep.name =='MassFlux'){
					mass_flux_segments.push(dep.segments);
				}
			}
			if (mass_flux_segments.length){
				WriteData(fid,'data',mass_flux_segments,'enum',MassFluxSegmentsEnum(),'format','MatArray');
				flag=true;
			}
			else flag=false;
			WriteData(fid,'data',flag,'enum',AutodiffMassFluxSegmentsPresentEnum(),'format','Boolean');
			//}}}
			//deal with trace keep on: {{{
			keep=false;

			//From ADOLC userdoc: 
			// The optional integer argument keep of trace on determines whether the numerical values of all active variables are 
			// recorded in a buffered temporary array or file called the taylor stack. This option takes effect if keep = 1 and 
			// prepares the scene for an immediately following gradient evaluation by a call to a routine implementing the reverse 
			// mode as described in the Section 4 and Section 5. 
			//

			if (this.driver.length<=3) keep=false; //there is no "_reverse" string within the driver string: 
			else{
				if (this.driver.splice(4) == '_reverse') keep=true;
				else keep=false;
			}
			WriteData(fid,'data',keep,'enum',AutodiffKeepEnum(),'format','Boolean');
			//}}}
		}//}}}
		this.fix=function() { //{{{
			this.obufsize=NullFix(this.obufsize,NaN);
			this.lbufsize=NullFix(this.lbufsize,NaN);
			this.cbufsize=NullFix(this.cbufsize,NaN);
			this.tbufsize=NullFix(this.tbufsize,NaN);
			this.gcTriggerRatio=NullFix(this.gcTriggerRatio,NaN);
			this.gcTriggerMaxSize=NullFix(this.gcTriggerMaxSize,NaN);
		}//}}}
	//properties 
	// {{{
	this.isautodiff   = false;
	this.dependents   = [];
	this.independents = [];
	this.driver       = 'fos_forward';
	this.obufsize     = NaN;
	this.lbufsize     = NaN;
	this.cbufsize     = NaN;
	this.tbufsize     = NaN;
	this.gcTriggerRatio = NaN;
	this.gcTriggerMaxSize = NaN;

	this.setdefaultparameters();
	//}}}
}
//BALANCETHICKNESS class definition
//
//   Usage:
//      balancethickness=new balancethickness();

function balancethickness (){
	//methods
	this.setdefaultparameters = function(){// {{{

		//Type of stabilization used
		this.stabilization=1;

	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   balance thickness solution parameters:'));

		fielddisplay(this,'spcthickness','thickness constraints (NaN means no constraint) [m]');
		fielddisplay(this,'thickening_rate','ice thickening rate used in the mass conservation (dh/dt) [m/yr]');
		fielddisplay(this,'stabilization',"0: None, 1: SU, 2: SSA's artificial diffusivity, 3:DG");

	}// }}}
	this.classname= function(){// {{{
		return "balancethickness";

	}// }}}
		this.checkconsistency = function(md,solution,analyses){ // {{{
			//Early return
			if (solution!=BalancethicknessSolutionEnum())return;

			checkfield(md,'fieldname','balancethickness.spcthickness');
			checkfield(md,'fieldname','balancethickness.thickening_rate','size',[md.mesh.numberofvertices ,1],'NaN',1,'Inf',1);
			checkfield(md,'fieldname','balancethickness.stabilization','size',[1, 1],'values',[0, 1, 2 ,3]);
			//checkfield(md,'fieldname','balancethickness.omega','size',[md.mesh.numberofvertices ,1],'NaN',1,'Inf',1,'>=',0);
		} //}}}
		this.marshall=function(md,fid) { //{{{

			var yts=365.0*24.0*3600.0;

			WriteData(fid,'object',this,'fieldname','spcthickness','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'fieldname','thickening_rate','format','DoubleMat','mattype',1,'scale',1/yts);
			WriteData(fid,'object',this,'fieldname','stabilization','format','Integer');
			WriteData(fid,'object',this,'fieldname','omega','format','DoubleMat','mattype',1);

		}//}}}
		this.fix=function() { //{{{
			this.spcthickness=NullFix(this.spcthickness,NaN);
			this.thicknening_rate=NullFix(this.thicknening_rate,NaN);
			this.omega=NullFix(this.omega,NaN);
		}//}}}
	//properties 
	// {{{
	this.spcthickness      = NaN;
	this.thickening_rate   = NaN;
	this.stabilization     = 0;
	this.omega             = NaN;
	this.setdefaultparameters();
	//}}}
}
//BASAL FORCINGS class definition
//
//   Usage:
//      basalforcings=basalforcings();

function basalforcings(){
	//methods
	this.setdefaultparameters = function() {//{{{

	} // }}}
	this.disp = function(){ // {{{
		console.log(sprintf('   basal forcings parameters:'));

		fielddisplay(this,'groundedice_melting_rate','basal melting rate (positive if melting) [m/yr]');
		fielddisplay(this,'floatingice_melting_rate','basal melting rate (positive if melting) [m/yr]');
		fielddisplay(this,'geothermalflux','geothermal heat flux [W/m^2]');

	} // }}}
	this.classname = function(){ // {{{
		return "basalforcings";
	} // }}}
		this.initialize = function (md){ // {{{

			if (isNaN(this.groundedice_melting_rate)){
				this.groundedice_melting_rate=NewArrayFill(md.mesh.numberofvertices,0);
				console.log('      no basalforcings.groundedice_melting_rate specified: values set as zero');
			}

			if (isNaN(this.floatingice_melting_rate)){
				this.floatingice_melting_rate=NewArrayFill(md.mesh.numberofvertices,0);
				console.log('      no basalforcings.floatingice_melting_rate specified: values set as zero');
			}

		} // }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{

			if(ArrayAnyEqual(ArrayIsMember(MasstransportAnalysisEnum(),analyses),1)){
				if (!(solution==TransientSolutionEnum() & md.trans.ismasstransport==0)){
					checkfield(md,'fieldname','basalforcings.groundedice_melting_rate','NaN',1,'Inf',1,'timeseries',1);
					checkfield(md,'fieldname','basalforcings.floatingice_melting_rate','NaN',1,'Inf',1,'timeseries',1);
				}
			}

			if(ArrayAnyEqual(ArrayIsMember(BalancethicknessAnalysisEnum(),analyses),1)){
				checkfield(md,'fieldname','basalforcings.groundedice_melting_rate','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
				checkfield(md,'fieldname','basalforcings.floatingice_melting_rate','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			}
			if(ArrayAnyEqual(ArrayIsMember(ThermalAnalysisEnum(),analyses),1)){
				if (!(solution==TransientSolutionEnum() & md.trans.isthermal==0)){
					checkfield(md,'fieldname','basalforcings.groundedice_melting_rate','NaN',1,'Inf',1,'timeseries',1);
					checkfield(md,'fieldname','basalforcings.floatingice_melting_rate','NaN',1,'Inf',1,'timeseries',1);
					checkfield(md,'fieldname','basalforcings.geothermalflux','NaN',1,'Inf',1,'timeseries',1,'>=',0);
				}
			}
		} // }}}
		this.marshall=function(md,fid) { //{{{

			var yts=365.0*24.0*3600.0;

			WriteData(fid,'enum',BasalforcingsEnum(),'data',FloatingMeltRateEnum(),'format','Integer');
			WriteData(fid,'object',this,'fieldname','groundedice_melting_rate','format','DoubleMat','mattype',1,'scale',1./yts,'timeserieslength',md.mesh.numberofvertices+1)
			WriteData(fid,'object',this,'fieldname','floatingice_melting_rate','format','DoubleMat','mattype',1,'scale',1./yts,'timeserieslength',md.mesh.numberofvertices+1)
			WriteData(fid,'object',this,'fieldname','geothermalflux','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties
	//{{{
	this.groundedice_melting_rate  = NaN;
	this.floatingice_melting_rate  = NaN;
	this.geothermalflux            = NaN;
	this.setdefaultparameters();
	//}}}
}
//CALVING class definition
//
//   Usage:
//      calving=new calving();

function calving (){
	//methods
	this.setdefaultparameters = function(){// {{{

	}// }}}
	this.disp= function(){// {{{

		console.log(sprintf('   Calving parameters:'));
		fielddisplay(this,'calvingrate','calving rate at given location [m/a]');
		fielddisplay(this,'meltingrate','melting rate at given location [m/a]');

	}// }}}
	this.classname= function(){// {{{
		return "calving";
	}// }}}
	this.checkconsistency = function(md,solution,analyses) { // {{{
		//Early return
		if (solution!=TransientSolutionEnum() | md.trans.ismovingfront==0) return;

		checkfield(md,'fieldname','calving.calvingrate(1:md.mesh.numberofvertices,:)','>=',0,'timeseries',1,'NaN',1,'Inf',1);
		checkfield(md,'fieldname','calving.meltingrate(1:md.mesh.numberofvertices,:)','>=',0,'timeseries',1,'NaN',1,'Inf',1);
	} //}}}
		this.marshall=function(md,fid) { //{{{
			var yts=365.0*24.0*3600.0;
			WriteData(fid,'enum',CalvingLawEnum(),'data',DefaultCalvingEnum(),'format','Integer');
			WriteData(fid,'object',this,'fieldname','calvingrate','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1,'scale',1./yts);
			WriteData(fid,'object',this,'fieldname','meltingrate','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1,'scale',1./yts);
		}//}}}
		this.fix=function() { //{{{
			this.calvingrate=NullFix(this.calvingrate,NaN);
			this.meltingrate=NullFix(this.meltingrate,NaN);
		}//}}}
	//properties 
	// {{{

	this.calvingrate   = NaN;
	this.meltingrate   = NaN;

	this.setdefaultparameters();
	//}}}
}
//GENERIC class definition
//
//   Usage:
//      generic=new generic();

function generic (){
	//properties 
	// {{{
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(0,args.length));

	this.url=options.getfieldvalue('url','');
	this.np=options.getfieldvalue('np',3);
	this.codepath=options.getfieldvalue('codepath','issmdir/bin');
	this.executionpath=options.getfieldvalue('executionpath','issmdir/execution');
	//}}}
	//methods
	this.disp= function(){// {{{
		console.log(sprintf('   generic class echo:'));
		console.log(sprintf('    url: "%s"',this.url));
		console.log(sprintf('    np: %i',this.np));
		console.log(sprintf('    codepath: "%s"',this.codepath));
		console.log(sprintf('    executionpath: "%s"',this.executionpath));
	}// }}}
	this.classname= function(){// {{{
		return "generic";
	}// }}}
	this.checkconsistency = function (md,solution,analyses) { //{{{
		if (cluster.np<1){
			md.checkmessage('number of processors should be at least 1');
		}
		if (isNaN(cluster.np)){
			md.checkmessage('number of processors should not be NaN!');
		}
	} //}}}
	this.BuildQueueScript = function (cluster,dirname,modelname,solution,io_gather,isvalgrind,isgprof,isdakota) { // {{{

			//write queuing script 
			//what is the executable being called? 
			executable='issm.exe';

			fid=fopen(modelname+'.queue','w');
			fprintf(fid,'#!%s\n',cluster.shell);
			fprintf(fid,'mpiexec -np %i %s/%s %s %s %s 2> %s.errlog >%s.outlog ',cluster.np,cluster.codepath,executable,EnumToString(solution),cluster.executionpath+'/'+dirname,modelname,modelname,modelname);					
			fclose(fid);
	} //}}}
	this.UploadAndRun = function (md,callbackfunction,fid,toolkitsstring,solutionstring,name,runtimename) { //{{{

		var oReq = new XMLHttpRequest();
		oReq.open("POST", this.url, true);
		oReq.responseType = 'application/octet-stream';
		oReq.position = 0
	
		//TODO: dynamic progress update with package lengths and parsing
		oReq.onprogress = function (oEvent) {
			var newResponse = oReq.responseText;
			if (newResponse.length == 13) { //"Starting" length as encoded in base64 length
				$(".run-button").html("COMPUTING...").prop("disabled", true);
				console.log("base64 repsonse: " + newResponse + " string response: " + window.atob(oReq.responseText));
			}
		};
		
		oReq.onload = function (oEvent) {
			//get context to this.str2ab to avoid duplciation
			function str2ab(str) {
				var buf = new Uint8Array(str.length);
				for (var i=0, strLen=str.length; i < strLen; i++) {
					buf[i] = str.charCodeAt(i);
				}
				return buf;
			} //}}}
			var buffer2 = str2ab(window.atob(oReq.responseText.slice(13)));			
			var returnBuffer = new Uint8Array(buffer2);
			var returnBuffer_size=returnBuffer.byteLength;
			try {
				md.results= parseresultsfrombuffer(returnBuffer,returnBuffer_size);
			}
			catch (e) {
				console.log(e);
			}
			callbackfunction();
		};
		
		var npbuffer = this.str2ab(md.cluster.np.toString());
		var nplength = new Uint32Array(1);
		nplength[0] = npbuffer.byteLength;

		var runtimenamebuffer = this.str2ab(runtimename);
		var runtimenamelength = new Uint32Array(1);
		runtimenamelength[0] = runtimenamebuffer.byteLength;
		
		var namebuffer = this.str2ab(name);
		var namelength = new Uint32Array(1);
		namelength[0] = namebuffer.byteLength;
		
		var toolkitsbuffer = this.str2ab(toolkitsstring);
		var toolkitslength = new Uint32Array(1);
		toolkitslength[0] = toolkitsbuffer.byteLength;
		
		var solutionbuffer = this.str2ab(solutionstring);
		var solutionlength = new Uint32Array(1);
		solutionlength[0] = solutionbuffer.byteLength;
		
		var binbuffer = new Uint8Array(fid.rawbuffer());
		//var binbuffer = new Uint16Array(fid.rawbuffer()); seems that 16 array bytes length could be incompatible.


		var binlength = new Uint32Array(1);
		binlength[0] = binbuffer.byteLength;
		
		var data = new Blob([nplength,npbuffer,runtimenamelength,runtimenamebuffer,namelength,namebuffer,toolkitslength,toolkitsbuffer,solutionlength,solutionbuffer,binlength,binbuffer]);
		//if (typeof(download) != "undefined") download(fid.rawbuffer());
		oReq.send(data);
		
		return;

	} //}}}
	this.ab2str = function(buf) { //{{{
		return String.fromCharCode.apply(null, new Uint16Array(buf));
	}
	this.str2ab = function(str) {
		var buf = new Uint8Array(str.length);
		for (var i=0, strLen=str.length; i < strLen; i++) {
			buf[i] = str.charCodeAt(i);
		}
		return buf;
	} //}}}
}
//LOCAL cluster class definition
//
//   Usage:
//      local=new local();

function local (){
	//methods
	this.setdefaultparameters = function(){// {{{
	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   local cluster class echo: []'));
	}// }}}
	this.classname= function(){// {{{
		return "local";
	}// }}}
		this.checkconsistency = function (md,solution,analyses) { //{{{
		} //}}}
	//properties 
	// {{{
	this.setdefaultparameters();
	//}}}
}
//CONSTANTS class definition
//
//   Usage:
//      constants=constants();

function constants() {
	//methods 
		this.setdefaultparameters = function (){ //{{{

			//acceleration due to gravity (m/s^2)
			this.g=9.81;

			//Earth's rotation speed 
			this.omega = 7.292*1e-5;

			//converstion from year to seconds
			this.yts=365*24*3600;

			//the reference temperature for enthalpy model (cf Aschwanden)
			this.referencetemperature=223.15;
		}// }}}
		this.disp = function () { //{{{
			console.log(sprintf("   Constants parameters:")); 
			
			fielddisplay(this,'g','gravitational acceleration [m/s^2]');
			fielddisplay(this,'omega','angular velocity of Earth [rad/s]');
			fielddisplay(this,'yts','number of seconds in a year [s/yr]');
			fielddisplay(this,'referencetemperature','reference temperature used in the enthalpy model [K]');

		} //}}}
		this.classname = function () { //{{{
			return "constants";

		} //}}}
		this.checkconsistency = function(md,solution,analyses) {//% {{{

			checkfield(md,'fieldname','constants.g','>=',0,'size',[1,1]); //We allow 0 for validation tests
			checkfield(md,'fieldname','constants.omega','>=',0,'size',[1,1]);
			checkfield(md,'fieldname','constants.yts','>',0,'size',[1,1]);
			checkfield(md,'fieldname','constants.referencetemperature','size',[1,1]);

		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'fieldname','g','format','Double');
			WriteData(fid,'object',this,'fieldname','yts','format','Double');
			WriteData(fid,'object',this,'fieldname','referencetemperature','format','Double');
		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{
		this.g                    = 0.;
		this.omega                = 0.;
		this.yts                  = 0.;
		this.referencetemperature = 0.;
		this.setdefaultparameters();
		//}}}
}
//DAMAGE class definition
//
//   Usage:
//      damage=new damage();

function damage (){
	//methods
	this.setdefaultparameters = function(){// {{{
		
		//damage parameters: 
		this.isdamage=0;
		this.D=0;
		this.law=0;

		this.max_damage=1-1e-5; //if damage reaches 1, solve becomes singular, as viscosity becomes nil

		//Type of stabilization used
		this.stabilization=4;

		//Maximum number of iterations
		this.maxiter=100;

		//finite element interpolation
		this.elementinterp='P1';

		//damage evolution parameters 
		this.stress_threshold=1.3e5;
		this.kappa=2.8;
		this.healing=0;
		this.c1=0;
		this.c2=0;
		this.c3=0;
		this.c4=0;
		this.equiv_stress=0;

		//output default:
		this.requested_outputs=['default'];

	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   Damage:\n'));

		fielddisplay(this,'isdamage','is damage mechanics being used? {true,false}');
		if (this.isdamage){
			fielddisplay(this,'law',"damage law ['0: analytical','1: pralong']");
			fielddisplay(this,'D','damage tensor (scalar)');
			fielddisplay(this,'spcdamage','damage constraints (NaN means no constraint)');
			fielddisplay(this,'max_damage','maximum possible damage (0<=max_damage<1)');

			fielddisplay(this,'stabilization','0: no, 1: artificial_diffusivity, 2: SUPG (not working), 4: flux corrected transport');
			fielddisplay(this,'maxiter','maximum number of non linear iterations');
			fielddisplay(this,'elementinterp',"interpolation scheme for finite elements {'P1','P2'}");
			fielddisplay(this,'stress_threshold','stress threshold for damage initiation [Pa]');
			fielddisplay(this,'kappa','ductility parameter for stress softening and damage');
			fielddisplay(this,'c1','damage parameter 1');
			fielddisplay(this,'c2','damage parameter 2');
			fielddisplay(this,'c3','damage parameter 3');
			fielddisplay(this,'c4','damage parameter 4');
			fielddisplay(this,'healing','damage healing parameter');
			fielddisplay(this,'equiv_stress','0: von Mises, 1: max principal');
			fielddisplay(this,'requested_outputs','additional outputs requested');
		}
	}// }}}
	this.classname= function(){// {{{
		return "damage";
	}// }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{
			
			checkfield(md,'fieldname','damage.isdamage','values',[1,0]);
			if (this.isdamage){
				checkfield(md,'fieldname','damage.law','numel',[1],'values',[0,1,2]);
				checkfield(md,'fieldname','damage.D','>=',0,'<=',self.max_damage,'size',[md.mesh.numberofvertices ,1]);
				checkfield(md,'fieldname','damage.spcdamage','Inf',1,'timeseries',1);
				checkfield(md,'fieldname','damage.max_damage','<',1,'>=',0);
				checkfield(md,'fieldname','damage.stabilization','numel',[1],'values',[0, 1, 2, 4]);
				checkfield(md,'fieldname','damage.maxiter','>=0',0);
				checkfield(md,'fieldname','damage.elementinterp','values',['P1','P2']);
				checkfield(md,'fieldname','damage.stress_threshold','>=',0);
				checkfield(md,'fieldname','damage.kappa','>',1);
				checkfield(md,'fieldname','damage.healing','>=',0);
				checkfield(md,'fieldname','damage.c1','>=',0);
				checkfield(md,'fieldname','damage.c2','>=',0);
				checkfield(md,'fieldname','damage.c3','>=',0);
				checkfield(md,'fieldname','damage.c4','>=',0);
				checkfield(md,'fieldname','damage.equiv_stress','numel',[1],'values',[0, 1]);
				checkfield(md,'fieldname','damage.requested_outputs','stringrow',1);
			}
			else if (this.law!=0){
				if (solution==DamageEvolutionSolutionEnum()){
					throw Error('Invalid evolution law (md.damage.law) for a damage solution');
				}
			}
		} //}}}
		this.marshall=function(md,fid) { //{{{
		
			WriteData(fid,'object',this,'fieldname','isdamage','format','Boolean');
			if (this.isdamage){
				WriteData(fid,'object',this,'fieldname','law','format','Integer');
				WriteData(fid,'object',this,'fieldname','D','format','DoubleMat','mattype',1);
				WriteData(fid,'object',this,'fieldname','spcdamage','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
				WriteData(fid,'object',this,'fieldname','max_damage','format','Double');

				WriteData(fid,'object',this,'fieldname','stabilization','format','Integer');
				WriteData(fid,'object',this,'fieldname','maxiter','format','Integer');
				WriteData(fid,'enum',DamageElementinterpEnum(),'data',StringToEnum(this.elementinterp),'format','Integer');
				WriteData(fid,'object',this,'fieldname','stress_threshold','format','Double');
				WriteData(fid,'object',this,'fieldname','kappa','format','Double');
				WriteData(fid,'object',this,'fieldname','c1','format','Double');
				WriteData(fid,'object',this,'fieldname','c2','format','Double');
				WriteData(fid,'object',this,'fieldname','c3','format','Double');
				WriteData(fid,'object',this,'fieldname','c4','format','Double');
				WriteData(fid,'object',this,'fieldname','healing','format','Double');
				WriteData(fid,'object',this,'fieldname','equiv_stress','format','Integer');
			}

			//process requested outputs
			var outputs = this.requested_outputs;
			for (var i=0;i<outputs.length;i++){
				if (outputs[i] == 'default') {
					outputs.splice(i,1);
					outputs.push(this.defaultoutputs(md));
				}
			}
			if (this.isdamage){
				WriteData(fid,'data',outputs,'enum',DamageEvolutionRequestedOutputsEnum(),'format','StringArray');
			}

		}//}}}
		this.fix=function() { //{{{
		}//}}}
		this.defaultoutputs = function(md){ //{{{

			if (md.mesh.domaintype() == '2Dhorizontal') return 'DamageDbar';
			else return 'DamageD';

		}//}}}
	//properties 
	// {{{
	this.isdamage            = 0;
	this.D                   = NaN;
	this.law                 = 0;
	this.spcdamage           = NaN; 
	this.max_damage          = 0;

	//numerical
	this.stabilization       = 0;
	this.maxiter             = 0;
	this.elementinterp       = '';

	//general parameters for evolution law: 
	this.stress_threshold    = 0;
	this.kappa               = 0;
	this.c1                  = 0;
	this.c2                  = 0;
	this.c3                  = 0;
	this.c4                  = 0;
	this.healing             = 0;
	this.equiv_stress		  = 0;
	this.requested_outputs   = [];

	this.setdefaultparameters();
	//}}}
}
//DEBUG class definition
//
//   Usage:
//      debug=new debug();

function debug (){
	//methods
	this.setdefaultparameters = function(){// {{{
	}// }}}
	this.classname= function(){// {{{
		return "debug";
	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   debug parameters:'));
		console.log(sprintf('   debug parameters:'));

		fielddisplay(this,'valgrind','use Valgrind to debug (0 or 1)');
		fielddisplay(this,'gprof','use gnu-profiler to find out where the time is spent');
		fielddisplay(this,'profiling','enables profiling (memory, flops, time)');

	}// }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'fieldname','profiling','format','Boolean');
		}//}}}
		this.fix=function() { //{{{
		}//}}}

	//properties 
	// {{{
	this.valgrind = false;
	this.gprof    = false;
	this.profiling = false;
	this.setdefaultparameters();
	//}}}
}
//FLAIM class definition
//
//   Usage:
//      flaim=new flaim();

function flaim (){
	//methods
	this.setdefaultparameters = function(){// {{{
	}// }}}
	this.disp= function(){// {{{

		console.log(sprintf('   FLAIM - Flight Line Adaptation using Ice sheet Modeling:')); 

		console.log(sprintf('\n      Input:'));
		fielddisplay(this,'targets'            ,'name of kml output targets file ');
		fielddisplay(this,'tracks'             ,'name of kml input tracks file ');
		fielddisplay(this,'flightreqs'         ,'structure of kml flight requirements (not used yet)');
		fielddisplay(this,'criterion'          ,'element or nodal criterion for flight path evaluation (metric)');

		console.log(sprintf('\n      Arguments:'));
		fielddisplay(this,'gridsatequator'     ,'number of grids at equator (determines resolution)');
		fielddisplay(this,'usevalueordering'   ,'flag to consider target values for flight path evaluation');
		fielddisplay(this,'split_antimeridian' ,'flag to split polygons on the antimeridian');

		console.log(sprintf('\n      Optimization:'));
		fielddisplay(this,'path_optimize'     ,'optimize? (default false)');
		fielddisplay(this,'opt_ndir'     ,['number of directions to test when moving a point.  If this value = 1, a random direction is tested.',
				'A value > 1 results in directions equally spaced from [0, 2*PI] being tested.',
				'For example, 4 would result in directions [0, PI/2, PI, 3PI/2].']);
		fielddisplay(this,'opt_dist'     ,'specifies the distance in km (default 25) to move a randomly selected path point on each iteration');
		fielddisplay(this,'opt_niter'     ,['number of iterations (default 30,000) to run for flightplan optimization',
				'i.e. the number of times to randomly select a point and move it.']);

		console.log(sprintf('\n      Output:'));
		fielddisplay(this,'solution'           ,'name of kml solution file');
		fielddisplay(this,'quality'            ,'quality of kml solution');

	}// }}}
	this.checkconsistency = function(md,solution,analyses) { // {{{

		//Early return
		if (solution!=FlaimSolutionEnum()) return;

		checkfield(md,'fieldname','flaim.tracks','file',1);
		if (isNaN(md.flaim.criterion) | md.flaim.criterion.length==0){
			checkfield(md,'fieldname','flaim.targets','file',1);
		}
		else{
			checkfield(md,'fieldname','flaim.criterion','numel',[md.mesh.numberofvertices, md.mesh.numberofelements]);
		}
	} //}}}
	//properties 
	// {{{

	this.targets            = '';
	this.tracks             = '';
	this.flightreqs         = {};
	this.criterion          = NaN;
	this.gridsatequator     = 200000;
	this.usevalueordering   = true;
	this.split_antimeridian = true;
	this.solution           = '';
	this.quality            = 0;
	this.path_optimize      = false;
	this.opt_ndir           = 1;
	this.opt_dist           = 25;
	this.opt_niter          = 30000;

	this.setdefaultparameters();
	//}}}
}
//FLOWEQUATION class definition
//
//   Usage:
//      flowequation=new flowequation();

function flowequation (){
	//methods
	this.setdefaultparameters = function(){// {{{
		//P1 for SSA
		this.fe_SSA= 'P1';

		//P1 for HO
		this.fe_HO= 'P1';

		//MINI condensed element for FS by default
		this.fe_FS = 'MINIcondensed';
	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   flow equation parameters:'));

		fielddisplay(this,'isSIA','is the Shallow Ice Approximation (SIA) used ?');
		fielddisplay(this,'isSSA','is the Shelfy-Stream Approximation (SSA) used ?');
		fielddisplay(this,'isL1L2','is the L1L2 approximation used ?');
		fielddisplay(this,'isHO','is the Higher-Order (HO) approximation used ?');
		fielddisplay(this,'isFS','are the Full-FS (FS) equations used ?');
		fielddisplay(this,'fe_SSA',"Finite Element for SSA  'P1', 'P1bubble' 'P1bubblecondensed' 'P2'");
		fielddisplay(this,'fe_HO', "Finite Element for HO   'P1' 'P1bubble' 'P1bubblecondensed' 'P1xP2' 'P2xP1' 'P2'");
		fielddisplay(this,'fe_FS', "Finite Element for FS   'P1P1' (debugging only) 'P1P1GLS' 'MINIcondensed' 'MINI' 'TaylorHood' 'XTaylorHood'");
		fielddisplay(this,'vertex_equation','flow equation for each vertex');
		fielddisplay(this,'element_equation','flow equation for each element');
		fielddisplay(this,'borderSSA',"vertices on SSA's border (for tiling)");
		fielddisplay(this,'borderHO',"vertices on HO's border (for tiling)");
		fielddisplay(this,'borderFS',"vertices on FS' border (for tiling)");

	}// }}}
	this.classname= function(){// {{{
		return "flowequation";

	}// }}}
		this.checkconsistency = function(md,solution,analyses) {//{{{

			//Early return
			if ( ((!ArrayAnyEqual(ArrayIsMember(StressbalanceAnalysisEnum(),analyses),1)) & (!ArrayAnyEqual(ArrayIsMember(StressbalanceSIAAnalysisEnum(),analyses),1))) | 
					(solution==TransientSolutionEnum() & md.trans.isstressbalance==0)
			   ) return ;

			checkfield(md,'fieldname','flowequation.isSIA','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','flowequation.isSSA','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','flowequation.isL1L2','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','flowequation.isHO','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','flowequation.isFS','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','flowequation.fe_SSA','values',['P1','P1bubble','P1bubblecondensed','P2','P2bubble']);
			checkfield(md,'fieldname','flowequation.fe_HO' ,'values',['P1','P1bubble','P1bubblecondensed','P1xP2','P2xP1','P2','P2bubble','P1xP3','P2xP4']);
			checkfield(md,'fieldname','flowequation.fe_FS' ,'values',['P1P1','P1P1GLS','MINIcondensed','MINI','TaylorHood','LATaylorHood','XTaylorHood','OneLayerP4z','CrouzeixRaviart','LACrouzeixRaviart']);
			checkfield(md,'fieldname','flowequation.augmented_lagrangian_r','numel',[1],'>=',0.);
			checkfield(md,'fieldname','flowequation.augmented_lagrangian_rlambda','numel',[1],'>=',0.);
			checkfield(md,'fieldname','flowequation.augmented_lagrangian_rhop','numel',[1],'>=',0.);
			checkfield(md,'fieldname','flowequation.augmented_lagrangian_rholambda','numel',[1],'>=',0.);
			checkfield(md,'fieldname','flowequation.XTH_theta','numel',[1],'>=',0.,'<',0.5);
			checkfield(md,'fieldname','flowequation.borderSSA','size',[md.mesh.numberofvertices, 1],'values',[0, 1]);
			checkfield(md,'fieldname','flowequation.borderHO','size',[md.mesh.numberofvertices, 1],'values',[0, 1]);
			checkfield(md,'fieldname','flowequation.borderFS','size',[md.mesh.numberofvertices, 1],'values',[0, 1]);
			if (md.mesh.domaintype() == '2Dhorizontal'){
				checkfield(md,'fieldname','flowequation.vertex_equation','size',[md.mesh.numberofvertices, 1],'values',[1,2]);
				checkfield(md,'fieldname','flowequation.element_equation','size',[md.mesh.numberofelements, 1],'values',[1,2]);
			}
			else if (md.mesh.domaintype() == '3Dsurface'){
				checkfield(md,'fieldname','flowequation.vertex_equation','size',[md.mesh.numberofvertices, 1],'values',[1,2]);
				checkfield(md,'fieldname','flowequation.element_equation','size',[md.mesh.numberofelements, 1],'values',[1,2]);
			}
			else if (md.mesh.domaintype() =='2Dvertical'){
				checkfield(md,'fieldname','flowequation.vertex_equation','size',[md.mesh.numberofvertices, 1],'values',[2,4,5]);
				checkfield(md,'fieldname','flowequation.element_equation','size',[md.mesh.numberofelements, 1],'values',[2,4,5]);
			}
			else if (md.mesh.domaintype() =='3D'){
				checkfield(md,'fieldname','flowequation.vertex_equation','size',[md.mesh.numberofvertices, 1],'values',[0,1,2,3,4,5,6,7,8]);
				checkfield(md,'fieldname','flowequation.element_equation','size',[md.mesh.numberofelements, 1],'values',[0,1,2,3,4,5,6,7,8]);
			}
			else throw Error('Case not supported yet');
			
			if (!(this.isSIA | this.isSSA | this.isL1L2 | this.isHO | this.isFS)){
				md = checkmessage(md,['no element types set for this model']);
			}
			if(ArrayAnyEqual(ArrayIsMember(StressbalanceSIAAnalysisEnum(), analyses),1)){
				if (ArrayAnyEqual(this.element_equation,1)){
					if(this.vertex_equation & ArrayAnyBelowStrict(md.mask.groundedice_levelset)){
						console.log(sprintf("\n !!! Warning: SIA's model is not consistent on ice shelves !!!\n"));
					}
				}
			}
		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'fieldname','isSIA','format','Boolean');
			WriteData(fid,'object',this,'fieldname','isSSA','format','Boolean');
			WriteData(fid,'object',this,'fieldname','isL1L2','format','Boolean');
			WriteData(fid,'object',this,'fieldname','isHO','format','Boolean');
			WriteData(fid,'object',this,'fieldname','isFS','format','Boolean');
			WriteData(fid,'enum',FlowequationFeSSAEnum(),'data',StringToEnum(this.fe_SSA),'format','Integer');
			WriteData(fid,'enum',FlowequationFeHOEnum() ,'data',StringToEnum(this.fe_HO) ,'format','Integer');
			WriteData(fid,'enum',FlowequationFeFSEnum() ,'data',StringToEnum(this.fe_FS) ,'format','Integer');
			WriteData(fid,'enum',AugmentedLagrangianREnum(),'data',this.augmented_lagrangian_r ,'format','Double');
			WriteData(fid,'enum',AugmentedLagrangianRhopEnum(),'data',this.augmented_lagrangian_rhop ,'format','Double');
			WriteData(fid,'enum',AugmentedLagrangianRlambdaEnum(),'data',this.augmented_lagrangian_rlambda ,'format','Double');
			WriteData(fid,'enum',AugmentedLagrangianRholambdaEnum(),'data',this.augmented_lagrangian_rholambda ,'format','Double');
			WriteData(fid,'enum',AugmentedLagrangianThetaEnum() ,'data',this.XTH_theta ,'format','Double');
			WriteData(fid,'object',this,'fieldname','borderSSA','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'fieldname','borderHO','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'fieldname','borderFS','format','DoubleMat','mattype',1);

			//convert approximations to enums
			data=this.vertex_equation;
			
			//some optimization: 
			var noneap=NoneApproximationEnum();
			var siaap=SIAApproximationEnum();
			var ssaap=SSAApproximationEnum();
			var l1l2ap=L1L2ApproximationEnum();
			var hoap=HOApproximationEnum();
			var fsap=FSApproximationEnum();
			var ssahoap=SSAHOApproximationEnum();
			var hofsaap=HOFSApproximationEnum();
			var ssafsap=SSAFSApproximationEnum();

			for(var i=0;i<data.length;i++){
				if(data[i]==0)data[i]=nonap;
				if(data[i]==1) data[i]=siaap;
				if(data[i]==2) data[i]=ssaap;
				if(data[i]==3) data[i]=l1l2ap;
				if(data[i]==4) data[i]=hoap;
				if(data[i]==5) data[i]=fsap;
				if(data[i]==6) data[i]=ssahoap;
				if(data[i]==7) data[i]=hofsaap;
				if(data[i]==8) data[i]=ssafsap;
			}
			WriteData(fid,'data',data,'enum',FlowequationVertexEquationEnum(),'format','DoubleMat','mattype',1);

			data=this.element_equation;
			for(var i=0;i<data.length;i++){
				if(data[i]==0) data[i]=nonap;
				if(data[i]==1) data[i]=siaap;
				if(data[i]==2) data[i]=ssaap;
				if(data[i]==3) data[i]=l1l2ap;
				if(data[i]==4) data[i]=hoap;
				if(data[i]==5) data[i]=fsap;
				if(data[i]==6) data[i]=ssahoap;
				if(data[i]==7) data[i]=ssafsap;
				if(data[i]==8) data[i]=hofsaap;
			}
			WriteData(fid,'data',data,'enum',FlowequationElementEquationEnum(),'format','DoubleMat','mattype',2);

		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{
	this.isSIA                          = 0;
	this.isSSA                          = 0;
	this.isL1L2                         = 0;
	this.isHO                           = 0;
	this.isFS                           = 0;
	this.fe_SSA                         = '';
	this.fe_HO                          = '';
	this.fe_FS                          = '';
	this.augmented_lagrangian_r         = 1.;
	this.augmented_lagrangian_rhop      = 1.;
	this.augmented_lagrangian_rlambda   = 1.;
	this.augmented_lagrangian_rholambda = 1.;
	this.XTH_theta                      = 0.;
	this.vertex_equation                = NaN;
	this.element_equation               = NaN;
	this.borderSSA                      = NaN;
	this.borderHO                       = NaN;
	this.borderFS                       = NaN;
	this.setdefaultparameters();
	//}}}
}
//FRICTION class definition
//
//   Usage:
//      friction=friction();

function friction (){
	//methods
	this.setdefaultparameters = function(){ // {{{

	} // }}}
	this.disp= function (){// {{{
		console.log(sprintf('Basal shear stress parameters: Sigma_b = coefficient^2 * Neff ^r * |u_b|^(s-1) * u_b\n(effective stress Neff=rho_ice*g*thickness+rho_water*g*bed, r=q/p and s=1/p)'));
		fielddisplay(this,'coefficient','friction coefficient [SI]');
		fielddisplay(this,'p','p exponent');
		fielddisplay(this,'q','q exponent');
	} // }}}
	this.classname= function (){// {{{
		return "friction";
	} // }}}
		this.checkconsistency = function(md,solution,analyses){ //{{{

			//Early return
			if ((!ArrayAnyEqual(ArrayIsMember(StressbalanceAnalysisEnum(),analyses),1)) & (!ArrayAnyEqual(ArrayIsMember(StressbalanceAnalysisEnum(),analyses),1))){
				return; 
			}
			md = checkfield(md,'fieldname','friction.coefficient','timeseries',1,'NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','friction.q','NaN',1,'Inf',1,'size',[md.mesh.numberofelements ,1]);
			md = checkfield(md,'fieldname','friction.p','NaN',1,'Inf',1,'size',[md.mesh.numberofelements ,1]);

		} // }}}
		this.marshall=function(md,fid) { //{{{
			var yts=365.0*24.0*3600.0;

			WriteData(fid,'enum',FrictionLawEnum(),'data',1,'format','Integer');
			WriteData(fid,'object',this,'fieldname','coefficient','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
			//WriteData(fid,'object',this,'fieldname','coefficient','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'fieldname','p','format','DoubleMat','mattype',2);
			WriteData(fid,'object',this,'fieldname','q','format','DoubleMat','mattype',2);
			

		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	//{{{
	this.coefficient = NaN;
	this.p           = NaN;
	this.q           = NaN;
	this.setdefaultparameters();
	//}}}
}
//GEOMETRY class definition
//
//   Usage:
//      geometry=geometry();

function geometry(){
	//methods 
		this.setdefaultparameters = function (){ //{{{
		}// }}}
		this.disp = function () { //{{{
			console.log(sprintf("   Geometry parameters:"));

			fielddisplay(this,'surface','ice upper surface elevation [m]');
			fielddisplay(this,'thickness','ice thickness [m]');
			fielddisplay(this,'base','ice base elevation [m]');
			fielddisplay(this,'bed','bed elevation [m]');
		} //}}}
		this.classname = function () { //{{{
			return 'geometry';
		} //}}}
		this.checkconsistency = function(md,solution,analyses) { //{{{

			if ((solution==TransientSolutionEnum() & md.trans.isgia) | (solution==GiaSolutionEnum())){
				checkfield(md,'fieldname','geometry.thickness','timeseries',1,'NaN',1,'Inf',1,'>=',0);
			}
			else{
				checkfield(md,'fieldname','geometry.surface'  ,'NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
				checkfield(md,'fieldname','geometry.base'      ,'NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
				checkfield(md,'fieldname','geometry.thickness','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1],'>',0);
				for(var i=0;i<md.mesh.numberofvertices;i++){
					if (Math.abs(md.geometry.thickness.thickness-md.geometry.surface+md.geometry.base)>Math.pow(10,9)){
						md = checkmessage(md,'equality thickness=surface-base violated');
						break;
					}
				}
				if (solution==TransientSolutionEnum() & md.trans.isgroundingline){
					checkfield(md,'fieldname','geometry.bed','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
				}
			}
		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'data',this.surface,'format','DoubleMat','mattype',1,'enum',SurfaceEnum());
			WriteData(fid,'data',this.thickness,'format','DoubleMat','mattype',1,'enum',ThicknessEnum(),'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'data',this.base,'format','DoubleMat','mattype',1,'enum',BaseEnum());
			WriteData(fid,'data',this.bed,'format','DoubleMat','mattype',1,'enum',BedEnum());
			WriteData(fid,'object',this,'fieldname','hydrostatic_ratio','format','DoubleMat','mattype',1);
		}//}}}
		this.fix=function() { //{{{
			this.hydrostatic_ratio=NullFix(this.hydrostatic_ratio,NaN);
		}//}}}
	//properties 
	// {{{
		this.surface           = NaN;
		this.thickness         = NaN;
		this.base              = NaN;
		this.bed               = NaN;
		this.hydrostatic_ratio = NaN;
		this.setdefaultparameters();
		//}}}
}
//GIA class definition
//
//   Usage:
//      gia=new gia();

function gia (){
	//methods
	this.setdefaultparameters = function(){// {{{

		this.cross_section_shape=1; //square as default (see iedge in GiaDeflectionCorex)
	
	}// }}}
	this.disp= function(){// {{{

		console.log(sprintf('   gia parameters:'));

		fielddisplay(this,'mantle_viscosity','mantle viscosity[Pa s]');
		fielddisplay(this,'lithosphere_thickness','lithosphere thickness (km)');
		fielddisplay(this,'cross_section_shape','1: square-edged (default). 2: elliptical.  See iedge in GiaDeflectionCore');

	}// }}}
	this.classname= function(){// {{{
		return "gia";
	}// }}}
	this.checkconsistency = function(md,solution,analyses) { // {{{

		if(!ArrayAnyEqual(ArrayIsMember(GiaAnalysisEnum(),analyses),1))return;

		checkfield(md,'fieldname','gia.mantle_viscosity','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices,1],'>',0);
		checkfield(md,'fieldname','gia.lithosphere_thickness','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices,1],'>',0);
		checkfield(md,'fieldname','gia.cross_section_shape','numel',[1],'values',[1,2]);

		//be sure that if we are running a masstransport ice flow model coupled with gia, that thickness forcings 
		//are not provided into the future.
		if (solution==TransientSolutionEnum() & md.trans.ismasstransport & md.trans.isgia){
			//figure out if thickness is a transient forcing: 
			if (md.geometry.thickness.length == (md.mesh.numberofvertices+1)){
				//recover the furthest time "in time": 
				t=md.geometry.thickness[0].length;
				if(md.geometry.thickness[md.geometry.thickness.length-1][t-1]!=md.timestepping.start_time){
					md.checkmessage('if ismasstransport is on, transient thickness forcing for the gia model should not be provided in the future. Synchronize your start_time to correspond to the most recent transient thickness forcing timestep');
				}
			}
		}
	} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'fieldname','mantle_viscosity','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'fieldname','lithosphere_thickness','format','DoubleMat','mattype',1,'scale',Math.pow(10,3)); //from km to m
			WriteData(fid,'object',this,'fieldname','cross_section_shape','format','Integer');
		}//}}}
		this.fix=function() { //{{{
			this.mantle_viscosity=NullFix(this.mantle_viscosity,NaN);
			this.lithosphere_thickness=NullFix(this.lithosphere_thickness,NaN);
		}//}}}
	//properties 
	// {{{

	this.mantle_viscosity              = NaN;
	this.lithosphere_thickness         = NaN;
	this.cross_section_shape           = 0;

	this.setdefaultparameters();
	//}}}
}
//GROUNDINGLINE class definition
//
//   Usage:
//      groundingline=new groundingline();

function groundingline (){
	//methods
	this.setdefaultparameters = function(){// {{{
		//Type of migration
		this.migration='None';

	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   grounding line migration parameters:'));
		fielddisplay(this,'migration',"type of grounding line migration: 'SoftMigration','AggressiveMigration','SubelementMigration','SubelementMigration2' or 'None'");


	}// }}}
	this.classname= function(){// {{{
		return "groundingline";
	}// }}}
		this.checkconsistency = function(md,solution,analyses) {// {{{

			checkfield(md,'fieldname','groundingline.migration','values',['None', 'AggressiveMigration', 'SoftMigration', 'SubelementMigration', 'SubelementMigration2', 'Contact', 'None', 'GroundingOnly']);

			if (this.migration !='None'){
				if (isNaN(md.geometry.bed)){
					md.checkmessage('requesting grounding line migration, but bathymetry is absent!');
				}
				for (var i=0;i<md.mesh.numberofvertices;i++){
					if(md.mask.groundedice_levelset[i]>0){
						md.checkmessage('base not equal to bed on grounded ice!');
						break;
					}
					if(md.geometry.bed[i] - md.geometry.base[i] > Math.pow(10,-9)){
						md = checkmessage(md,'bed superior to base on floating ice!');
						break;
					}
				}
			}
		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'data',StringToEnum(this.migration),'enum',GroundinglineMigrationEnum(),'format','Integer');
		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{
	this.migration    = '';
	this.setdefaultparameters();
	//}}}
}
//HYDROLOGYSHREVE class definition
//
//   Usage:
//      hydrologyshreve=new hydrologyshreve();

function hydrologyshreve (){
	//methods
	this.setdefaultparameters = function(){// {{{

		//Type of stabilization to use 0:nothing 1:artificial_diffusivity
		self.stabilization=1;

	}// }}}
		this.disp= function(){// {{{

		console.log(sprintf('   hydrologyshreve solution parameters:'));
		fielddisplay(this,'spcwatercolumn','water thickness constraints (NaN means no constraint) [m]');
		fielddisplay(this,'stabilization','artificial diffusivity (default is 1). can be more than 1 to increase diffusivity.');

	}// }}}
		this.classname= function(){// {{{
			return "hydrologyshreve";

		}// }}}
	this.type= function(){// {{{

		return "hydrologyshreve";
	}// }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{

			//Early return
			if(!ArrayAnyEqual(ArrayIsMember(HydrologyShreveAnalysisEnum(),analyses),1)) return;

			checkfield(md,'fieldname','hydrology.spcwatercolumn','Inf',1,'timeseries',1);
			checkfield(md,'fieldname','hydrology.stabilization','>=',0);

		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'enum',HydrologyModelEnum(),'data',HydrologyshreveEnum(),'format','Integer');
			WriteData(fid,'object',this,'fieldname','spcwatercolumn','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',this,'fieldname','stabilization','format','Double');
		}//}}}
		this.fix=function() { //{{{
			this.spcwatercolumn=NullFix(this.spcwatercolumn,NaN);
		}//}}}
	//properties 
	// {{{
	this.spcwatercolumn = NaN;
	this.stabilization  = 0;
	this.setdefaultparameters();
	//}}}
}
//INITIALIZATION class definition
//
//   Usage:
//      initialization=new initialization();

function initialization (){
	//methods
	this.setdefaultparameters = function(){// {{{
	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   initial field values:'));

		fielddisplay(this,'vx','x component of velocity [m/yr]');
		fielddisplay(this,'vy','y component of velocity [m/yr]');
		fielddisplay(this,'vz','z component of velocity [m/yr]');
		fielddisplay(this,'vel','velocity norm [m/yr]');
		fielddisplay(this,'pressure','pressure field [Pa]');
		fielddisplay(this,'temperature','temperature [K]');
		fielddisplay(this,'waterfraction','fraction of water in the ice');
		fielddisplay(this,'sediment_head','sediment water head of subglacial system [m]');
		fielddisplay(this,'epl_head','epl water head of subglacial system [m]');
		fielddisplay(this,'epl_thickness','epl layer thickness [m]');
		fielddisplay(this,'watercolumn','thickness of subglacial water [m]');

	}// }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{
			if(ArrayAnyEqual(ArrayIsMember(StressbalanceAnalysisEnum(),analyses),1)){
				if (!(isNaN(md.initialization.vx) | isNaN(md.initialization.vy))){
					checkfield(md,'fieldname','initialization.vx','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
					checkfield(md,'fieldname','initialization.vy','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
				}
			}
			if(ArrayAnyEqual(ArrayIsMember(MasstransportAnalysisEnum(),analyses),1)){
				checkfield(md,'fieldname','initialization.vx','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
				checkfield(md,'fieldname','initialization.vy','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			}
			if(ArrayAnyEqual(ArrayIsMember(BalancethicknessSolutionEnum(),analyses),1) & (solution==BalancethicknessSolutionEnum())){
				checkfield(md,'fieldname','initialization.vx','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
				checkfield(md,'fieldname','initialization.vy','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
				//Triangle with zero velocity
				for(var i=0;i<md.mesh.numberofelements;i++){
					var sum=0;
					for(var j=0;j<md.mesh.elements[0].length;j++){
						if  ((md.initialization.vx[md.mesh.elements[i][j]-1]==0) & (md.initialization.vy[md.mesh.elements[i][j]-1]==0)) sum+=1;
					}
					if (sum==md.mesh.elements[0].length){
						md.checkmessage('at least one triangle has all its vertices with a zero velocity');
					}
				}
			}
			if(ArrayAnyEqual(ArrayIsMember(ThermalAnalysisEnum(),analyses),1)){
				checkfield(md,'fieldname','initialization.vx','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
				checkfield(md,'fieldname','initialization.vy','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
				if (md.mesh.dimension() == 3){
					checkfield(md,'fieldname','initialization.vz','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices ,1]);
				}
				checkfield(md,'fieldname','initialization.pressure','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices ,1]);
				checkfield(md,'fieldname','initialization.temperature','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices ,1]);
			}
			if( ArrayAnyEqual(ArrayIsMember(EnthalpyAnalysisEnum(),analyses),1) & md.thermal.isenthalpy){
				checkfield(md,'fieldname','initialization.waterfraction','>=',0,'size',[md.mesh.numberofvertices, 1]);
				checkfield(md,'fieldname','initialization.watercolumn'  ,'>=',0,'size',[md.mesh.numberofvertices, 1]);
			}
			if(ArrayAnyEqual(ArrayIsMember(HydrologyShreveAnalysisEnum(),analyses),1)){
				if (md.hydrology.type() == 'hydrologyshreve'){
					checkfield(md,'fieldname','initialization.watercolumn','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices ,1]);
				}
			}
			if(ArrayAnyEqual(ArrayIsMember(HydrologyDCInefficientAnalysisEnum(),analyses),1)){
				if (md.hydrology.type() == 'hydrologydc'){
					checkfield(md,'fieldname','initialization.sediment_head','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
				}
			}
			if(ArrayAnyEqual(ArrayIsMember(HydrologyDCEfficientAnalysisEnum(),analyses),1)){
				if (md.hydrology.type() == 'hydrologydc'){
					if (md.hydrology.isefficientlayer==1){
						checkfield(md,'fieldname','initialization.epl_head','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices ,1]);
						checkfield(md,'fieldname','initialization.epl_thickness','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices ,1]);
					}
				}
			}
		} //}}}
		this.marshall=function(md,fid) { //{{{

			var yts=365.0*24.0*3600.0;

			WriteData(fid,'data',this.vx,'format','DoubleMat','mattype',1,'enum',VxEnum(),'scale',1./yts);
			WriteData(fid,'data',this.vy,'format','DoubleMat','mattype',1,'enum',VyEnum(),'scale',1./yts);
			WriteData(fid,'data',this.vz,'format','DoubleMat','mattype',1,'enum',VzEnum(),'scale',1./yts);
			WriteData(fid,'data',this.pressure,'format','DoubleMat','mattype',1,'enum',PressureEnum());
			WriteData(fid,'data',this.temperature,'format','DoubleMat','mattype',1,'enum',TemperatureEnum());
			WriteData(fid,'data',this.waterfraction,'format','DoubleMat','mattype',1,'enum',WaterfractionEnum());
			WriteData(fid,'data',this.sediment_head,'format','DoubleMat','mattype',1,'enum',SedimentHeadEnum());
			WriteData(fid,'data',this.epl_head,'format','DoubleMat','mattype',1,'enum',EplHeadEnum());
			WriteData(fid,'data',this.epl_thickness,'format','DoubleMat','mattype',1,'enum',HydrologydcEplThicknessEnum());
			WriteData(fid,'data',this.watercolumn,'format','DoubleMat','mattype',1,'enum',WatercolumnEnum());

			if (md.thermal.isenthalpy){
				tpmp=NewArrayFill(md.mesh.numberofvertices,0);
				for (var i=0;i<md.mesh.numberofvertices;i++) tpmp[i]= md.materials.meltingpoint - md.materials.beta*md.initialization.pressure[i];
				enthalpy=NewArrayFill(md.mesh.numberofvertices,0); 
				for (var i=0;i<md.mesh.numberofvertices;i++)enthalpy[i] = md.materials.heatcapacity*(md.initialization.temperature[i]-md.constants.referencetemperature);
				
				for (var i=0;i<md.mesh.numberofvertices;i++)if(md.initialization.temperature[i]>=tpmp[i]){
					enthalpy[i] = md.materials.heatcapacity*(tpmp[i] - md.constants.referencetemperature) + md.materials.latentheat*md.initialization.waterfraction[i];
				}
				WriteData(fid,'data',enthalpy,'format','DoubleMat','mattype',1,'enum',EnthalpyEnum());
			}
		}//}}}
		this.fix=function(md) { //{{{
			this.vx=FloatFix(this.vx,md.mesh.numberofvertices); 
			this.vy=FloatFix(this.vx,md.mesh.numberofvertices); 
			this.vy=FloatFix(this.vx,md.mesh.numberofvertices); 
			this.waterfraction=NullFix(this.waterfraction,NaN);
			this.sediment_head=NullFix(this.sediment_head,NaN);
			this.epl_head=NullFix(this.epl_head,NaN);
			this.epl_thickness=NullFix(this.epl_thickness,NaN);
			this.watercolumn=NullFix(this.watercolumn,NaN);
		}//}}}
	//properties 
	// {{{
	this.vx            = NaN;
	this.vy            = NaN;
	this.vz            = NaN;
	this.vel           = NaN;
	this.pressure      = NaN;
	this.temperature   = NaN;
	this.waterfraction = NaN;
	this.sediment_head = NaN;
	this.epl_head      = NaN;
	this.epl_thickness = NaN;
	this.watercolumn   = NaN;
	this.setdefaultparameters();

	//}}}
}
//INVERSION class definition
//
//   Usage:
//      inversion=new inversion();

function inversion (){
	//methods
	this.setdefaultparameters = function(){// {{{

		//default is incomplete adjoint for now
		this.incomplete_adjoint=1;

		//parameter to be inferred by control methods (only
		//drag and B are supported yet)
		this.control_parameters=['FrictionCoefficient'];

		//number of steps in the control methods
		this.nsteps=20;

		//maximum number of iteration in the optimization algorithm for
		//each step
		this.maxiter_per_step=20*NewArrayFill(this.nsteps,1);

		//the inversed parameter is updated as follows:
		//new_par=old_par + gradient_scaling(n)*C*gradient with C in [0 1];
		//usually the gradient_scaling must be of the order of magnitude of the 
		//inversed parameter (10^8 for B, 50 for drag) and can be decreased
		//after the first iterations
		this.gradient_scaling=NewArrayFill(this.nsteps,50);

		//several responses can be used:
		this.cost_functions=101;

		//step_threshold is used to speed up control method. When
		//misfit(1)/misfit(0) < this.step_threshold, we go directly to
		//the next step
		this.step_threshold=NewArrayFill(this.nsteps,.7); //30 per cent decrement.

		//cost_function_threshold is a criteria to stop the control methods.
		//if J[n]-J[n-1]/J[n] < criteria, the control run stops
		//NaN if not applied
		this.cost_function_threshold=NaN; //not activated

	}// }}}
	this.disp= function(){// {{{

		console.log(sprintf('   inversion parameters:'));
		fielddisplay(this,'iscontrol','is inversion activated?');
		fielddisplay(this,'incomplete_adjoint','1: linear viscosity, 0: non-linear viscosity');
		fielddisplay(this,'control_parameters',"ex: {'FrictionCoefficient'}, or {'MaterialsRheologyBbar'}");
		fielddisplay(this,'nsteps','number of optimization searches');
		fielddisplay(this,'cost_functions','indicate the type of response for each optimization step');
		fielddisplay(this,'cost_functions_coefficients','cost_functions_coefficients applied to the misfit of each vertex and for each control_parameter');
		fielddisplay(this,'cost_function_threshold','misfit convergence criterion. Default is 1%, NaN if not applied');
		fielddisplay(this,'maxiter_per_step','maximum iterations during each optimization step');
		fielddisplay(this,'gradient_scaling','scaling factor on gradient direction during optimization, for each optimization step');
		fielddisplay(this,'step_threshold','decrease threshold for misfit, default is 30%');
		fielddisplay(this,'min_parameters','absolute minimum acceptable value of the inversed parameter on each vertex');
		fielddisplay(this,'max_parameters','absolute maximum acceptable value of the inversed parameter on each vertex');
		fielddisplay(this,'vx_obs','observed velocity x component [m/yr]');
		fielddisplay(this,'vy_obs','observed velocity y component [m/yr]');
		fielddisplay(this,'vel_obs','observed velocity magnitude [m/yr]');
		fielddisplay(this,'thickness_obs','observed thickness [m]');
		fielddisplay(this,'surface_obs','observed surface elevation [m]');
		console.log('Available cost functions:');
		console.log('   101: SurfaceAbsVelMisfit');
		console.log('   102: SurfaceRelVelMisfit');
		console.log('   103: SurfaceLogVelMisfit');
		console.log('   104: SurfaceLogVxVyMisfit');
		console.log('   105: SurfaceAverageVelMisfit');
		console.log('   201: ThicknessAbsMisfit');
		console.log('   501: DragCoefficientAbsGradient');
		console.log('   502: RheologyBbarAbsGradient');
		console.log('   503: ThicknessAbsGradient');

	}// }}}
	this.classname= function(){// {{{
		return "inversion";
	}// }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{

			//Early return
			if (!this.iscontrol) return;

			num_controls=md.inversion.control_parameters.length;
			num_costfunc=md.inversion.control_parameters[0].length;

			checkfield(md,'fieldname','inversion.iscontrol','values',[0, 1]);
			checkfield(md,'fieldname','inversion.incomplete_adjoint','values',[0 ,1]);
			checkfield(md,'fieldname','inversion.control_parameters','cell',1,'values',supportedcontrols());
			checkfield(md,'fieldname','inversion.nsteps','numel',1,'>=',0);
			checkfield(md,'fieldname','inversion.maxiter_per_step','size',[md.inversion.nsteps, 1],'>=',0);
			checkfield(md,'fieldname','inversion.step_threshold','size',[md.inversion.nsteps, 1]);
			checkfield(md,'fieldname','inversion.cost_functions','size',[1, num_costfunc],'values',supportedcostfunctions());
			checkfield(md,'fieldname','inversion.cost_functions_coefficients','size',[md.mesh.numberofvertices, num_costfunc],'>=',0);
			checkfield(md,'fieldname','inversion.gradient_scaling','size',[md.inversion.nsteps, num_controls]);
			checkfield(md,'fieldname','inversion.min_parameters','size',[md.mesh.numberofvertices , num_controls]);
			checkfield(md,'fieldname','inversion.max_parameters','size',[md.mesh.numberofvertices ,num_controls]);

			//Only SSA, HO and FS are supported right now
			if (solution==StressbalanceSolutionEnum()){
				if (!(md.flowequation.isSSA | md.flowequation.isHO | md.flowequation.isFS | md.flowequation.isL1L2)){
					md.checkmessage('inversion can only be performed for SSA, HO or FS ice flow models');
				}
			}

			if (solution==BalancethicknessSolutionEnum()){
				checkfield(md,'fieldname','inversion.thickness_obs','size',[md.mesh.numberofvertices ,1],'NaN',1,'Inf',1);
			}
			else if (solution==BalancethicknessSoftSolutionEnum()){
				checkfield(md,'fieldname','inversion.thickness_obs','size',[md.mesh.numberofvertices, 1],'NaN',1,'Inf',1);
			}
			else{
				checkfield(md,'fieldname','inversion.vx_obs','size',[md.mesh.numberofvertices ,1],'NaN',1,'Inf',1);
				checkfield(md,'fieldname','inversion.vy_obs','size',[md.mesh.numberofvertices ,1],'NaN',1,'Inf',1);
			}
		} // }}}
		this.marshall=function(md,fid) { //{{{

			var yts=365.0*24.0*3600.0;

			WriteData(fid,'enum',InversionTypeEnum(),'data',0,'format','Integer');
			WriteData(fid,'object',this,'fieldname','iscontrol','format','Boolean');
			WriteData(fid,'object',this,'fieldname','incomplete_adjoint','format','Boolean');
			if (!this.iscontrol) return;
			WriteData(fid,'object',this,'fieldname','nsteps','format','Integer');
			WriteData(fid,'object',this,'fieldname','maxiter_per_step','format','IntMat','mattype',3);
			WriteData(fid,'object',this,'fieldname','cost_functions_coefficients','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'fieldname','gradient_scaling','format','DoubleMat','mattype',3);
			WriteData(fid,'object',this,'fieldname','cost_function_threshold','format','Double');
			WriteData(fid,'object',this,'fieldname','min_parameters','format','DoubleMat','mattype',3);
			WriteData(fid,'object',this,'fieldname','max_parameters','format','DoubleMat','mattype',3);
			WriteData(fid,'object',this,'fieldname','step_threshold','format','DoubleMat','mattype',3);
			WriteData(fid,'object',this,'fieldname','vx_obs','format','DoubleMat','mattype',1,'scale',1./yts);
			WriteData(fid,'object',this,'fieldname','vy_obs','format','DoubleMat','mattype',1,'scale',1./yts);
			WriteData(fid,'object',this,'fieldname','vz_obs','format','DoubleMat','mattype',1,'scale',1./yts);
			if(this.thickness_obs.length==md.mesh.numberofelements) mattype=2;
			else mattype=1;
			WriteData(fid,'object',this,'class','inversion','fieldname','thickness_obs','format','DoubleMat','mattype',mattype);
			WriteData(fid,'object',this,'class','inversion','fieldname','surface_obs','format','DoubleMat','mattype',mattype);


			//process control parameters
			num_control_parameters=this.control_parameters.length;
			data=NewArrayFill(num_control_parameters,0);
			for (var i=0;i<num_control_parameters;i++){
				data[i]=StringToEnum(this.control_parameters[i]);
			}
			WriteData(fid,'data',data,'enum',InversionControlParametersEnum(),'format','DoubleMat','mattype',3);
			WriteData(fid,'data',num_control_parameters,'enum',InversionNumControlParametersEnum(),'format','Integer');

			//process cost functions
			num_cost_functions=this.cost_functions[0].length;
			data=marshallcostfunctions(this.cost_functions);
			WriteData(fid,'data',data,'enum',InversionCostFunctionsEnum(),'format','DoubleMat','mattype',3);
			WriteData(fid,'data',num_cost_functions,'enum',InversionNumCostFunctionsEnum(),'format','Integer');
		}//}}}
		this.fix=function() { //{{{
			this.control_parameters=NullFix(this.control_parameters,NaN);
			this.maxiter_per_step=NullFix(this.maxiter_per_step,NaN);
			this.cost_functions=NullFix(this.cost_functions,NaN);
			this.cost_functions_coefficients=NullFix(this.cost_functions_coefficients,NaN);
			this.cost_function_threshold=NullFix(this.cost_function_threshold,NaN);
			this.gradient_scaling=NullFix(this.gradient_scaling,NaN);
			this.min_parameters=NullFix(this.min_parameters,NaN);
			this.max_parameters=NullFix(this.max_parameters,NaN);
			this.step_threshold=NullFix(this.step_threshold,NaN);
			this.vx_obs=NullFix(this.vx_obs,NaN);
			this.vy_obs=NullFix(this.vy_obs,NaN);
			this.vz_obs=NullFix(this.vz_obs,NaN);
			this.vel_obs=NullFix(this.vel_obs,NaN);
			this.thickness_obs=NullFix(this.thickness_obs,NaN);
			this.surface_obs=NullFix(this.surface_obs,NaN);
		}//}}}
	//properties 
	// {{{

	this.iscontrol                   = 0;
	this.incomplete_adjoint          = 0;
	this.control_parameters          = NaN;
	this.nsteps                      = 0;
	this.maxiter_per_step            = NaN;
	this.cost_functions              = NaN;
	this.cost_functions_coefficients = NaN;
	this.gradient_scaling            = NaN;
	this.cost_function_threshold     = 0;
	this.min_parameters              = NaN;
	this.max_parameters              = NaN;
	this.step_threshold              = NaN;
	this.vx_obs                      = NaN;
	this.vy_obs                      = NaN;
	this.vz_obs                      = NaN;
	this.vel_obs                     = NaN;
	this.thickness_obs               = NaN;
	this.surface_obs                 = NaN;

	this.setdefaultparameters();
	//}}}
}
//MASK class definition
//
//   Usage:
//      mask= new mask();

function mask () {
	//properties 
	// {{{
		this.groundedice_levelset                           = NaN;
		this.ice_levelset                           = NaN;
		//}}}
	//methods 
		this.setdefaultparameters = function (){ //{{{
		} // }}}
		this.disp = function () { //{{{
			console.log(sprintf("   mask:")); 

			fielddisplay(this,"groundedice_levelset","is ice grounded ? grounded ice if > 0, grounding line position if = 0, floating ice if < 0");
			fielddisplay(this,"ice_levelset","presence of ice if < 0, icefront position if = 0, no ice if > 0");
		} //}}}
		this.classname = function () { //{{{
			return "mask";
		} //}}}
		this.checkconsistency = function(md,solution,analyses){ //{{{

			checkfield(md,'fieldname','mask.groundedice_levelset','size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mask.ice_levelset'        ,'size',[md.mesh.numberofvertices, 1]);
			var isice=NewArrayFill(md.mesh.numberofvertices,0); 
			for(var i=0;i<md.mesh.numberofvertices;i++)if(md.mask.ice_levelset[i]<=0)isice[i]=1;
			if (ArraySum(isice)==0){
				console.log('no ice present in the domain');
			}
			if (ArrayMax(md.mask.ice_levelset)<0){
				console.log('no ice front provided');
			}
				
			var icefront=NewArrayFill(md.mesh.numberofelements,0);
			for(var i=0;i<md.mesh.numberofelements;i++){
				for(var j=0;j<md.mesh.elements[0].length;j++){
					icefront[i]+=(md.mask.ice_levelset[md.mesh.elements[i][j]-1]==0);
				}
			}
			if ((ArrayMax(icefront)==3 & (md.mesh.elementtype() == 'Tria')) | (ArrayMax(icefront)==6 & md.mesh.elementtype() == 'Penta')){
				if (md.mesh.elementtype()=='Tria'){
					var pos=ArrayFindEqual(icefront,3); numberemptyelements=pos.length;
				}
				else if (md.mesh.elementtype() == 'Penta'){
					var pos=ArrayFindEqual(icefront,6); numberemptyelements=pos.length;
				}
				throw Error(sprintf(" %i have all nodes on ice front, change md.mask.ice_levelset to fix it",numberemptyelements));
			}
		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'fieldname','groundedice_levelset','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'fieldname','ice_levelset','format','DoubleMat','mattype',1);
		}//}}}
		this.fix=function() { //{{{
		}//}}}

}
//MASKPSL class definition
//
//   Usage:
//      maskpsl= new maskpsl();

function maskpsl () {
	//properties 
	// {{{
		this.groundedice_levelset                           = NaN;
		this.ice_levelset                           = NaN;
		this.land_levelset                           = NaN;
		this.ocean_levelset                           = NaN;
		//}}}
	//methods 
		this.setdefaultparameters = function (){ //{{{
		} // }}}
		this.disp = function () { //{{{
			console.log(sprintf("   mask:")); 

			fielddisplay(this,"groundedice_levelset","is ice grounded ? grounded ice if > 0, grounding line position if = 0, floating ice if < 0");
			fielddisplay(this,"ice_levelset","presence of ice if < 0, icefront position if = 0, no ice if > 0");
			fielddisplay(this,"ocean_levelset","is the vertex on the ocean? yes if = 1, no if = 0");
			fielddisplay(this,"land_levelset","is the vertex on land? yes if = 1, no if = 0");
		} //}}}
		this.classname = function () { //{{{
			return "maskpsl";
		} //}}}
		this.checkconsistency = function(md,solution,analyses){ //{{{

			checkfield(md,'fieldname','mask.groundedice_levelset','size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mask.ice_levelset'        ,'size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mask.ocean_levelset'        ,'size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mask.land_levelset'        ,'size',[md.mesh.numberofvertices, 1]);
			
			var isice=NewArrayFill(md.mesh.numberofvertices,0); 
			for(var i=0;i<md.mesh.numberofvertices;i++)if(md.mask.ice_levelset[i]<=0)isice[i]=1;
			if (ArraySum(isice)==0){
				console.log('no ice present in the domain');
			}
			if (ArrayMax(md.mask.ice_levelset)<0){
				console.log('no ice front provided');
			}
				
			var icefront=NewArrayFill(md.mesh.numberofelements,0);
			for(var i=0;i<md.mesh.numberofelements;i++){
				for(var j=0;j<md.mesh.elements[0].length;j++){
					icefront[i]+=(md.mask.ice_levelset[md.mesh.elements[i][j]-1]==0);
				}
			}
			if ((ArrayMax(icefront)==3 & (md.mesh.elementtype() == 'Tria')) | (ArrayMax(icefront)==6 & md.mesh.elementtype() == 'Penta')){
				if (md.mesh.elementtype()=='Tria'){
					var pos=ArrayFindEqual(icefront,3); numberemptyelements=pos.length;
				}
				else if (md.mesh.elementtype() == 'Penta'){
					var pos=ArrayFindEqual(icefront,6); numberemptyelements=pos.length;
				}
				throw Error(sprintf(" %i have all nodes on ice front, change md.mask.ice_levelset to fix it",numberemptyelements));
			}
		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'class','mask','fieldname','groundedice_levelset','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mask','fieldname','ice_levelset','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mask','fieldname','ocean_levelset','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mask','fieldname','land_levelset','format','DoubleMat','mattype',1);
		}//}}}
		this.fix=function() { //{{{
		}//}}}

}
//MASSTRANSPORT class definition
//
//   Usage:
//      masstransport=new masstransport();

function masstransport (){
	//methods
	this.setdefaultparameters = function(){// {{{

		//Type of stabilization to use 0:nothing 1:artificial_diffusivity 3:Discontinuous Galerkin
		this.stabilization=1;

		//Factor applied to compute the penalties kappa=max(stiffness matrix)*10^penalty_factor
		this.penalty_factor=3;

		//Minimum ice thickness that can be used
		this.min_thickness=1;

		//Hydrostatic adjustment
		this.hydrostatic_adjustment='Absolute';

		//default output
		this.requested_outputs=['default'];

	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   Masstransport solution parameters:'));
		fielddisplay(this,'spcthickness','thickness constraints (NaN means no constraint) [m]');
		fielddisplay(this,'isfreesurface','do we use free surfaces (FS only) are mass conservation');
		fielddisplay(this,'min_thickness','minimum ice thickness allowed [m]');
		fielddisplay(this,'hydrostatic_adjustment',"adjustment of ice shelves surface and bed elevations: 'Incremental' or 'Absolute' ");
		fielddisplay(this,'stabilization','0: no, 1: artificial_diffusivity, 2: streamline upwinding, 3: discontinuous Galerkin, 4: Flux Correction Transport');

		console.log(sprintf('\n      %s','Penalty options:'));
		fielddisplay(this,'penalty_factor','offset used by penalties: penalty = Kmax*10^offset');
		fielddisplay(this,'vertex_pairing','pairs of vertices that are penalized');
		fielddisplay(this,'requested_outputs','additional outputs requested');

	}// }}}
	this.classname= function(){// {{{
		return "masstransport";
	}// }}}
		this.checkconsistency = function (md,solution,analyses){  // {{{

			//Early return
			if(!ArrayAnyEqual(ArrayIsMember(HydrologyShreveAnalysisEnum(),analyses),1) | (solution==TransientSolutionEnum() & md.trans.ismasstransport==0)) return; 

			checkfield(md,'fieldname','masstransport.spcthickness','Inf',1,'timeseries',1);
			checkfield(md,'fieldname','masstransport.isfreesurface','values',[0 ,1]);
			checkfield(md,'fieldname','masstransport.hydrostatic_adjustment','values',['Absolute', 'Incremental']);
			checkfield(md,'fieldname','masstransport.stabilization','values',[0,1,2,3,4]);
			checkfield(md,'fieldname','masstransport.min_thickness','>',0);
			checkfield(md,'fieldname','masstransport.requested_outputs','stringrow',1);

		} // }}}
		this.marshall=function(md,fid) { //{{{

			var yts=365.*24.*3600.;

			WriteData(fid,'object',this,'fieldname','spcthickness','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',this,'fieldname','isfreesurface','format','Boolean');
			WriteData(fid,'object',this,'fieldname','min_thickness','format','Double');
			WriteData(fid,'data',StringToEnum(this.hydrostatic_adjustment),'format','Integer','enum',MasstransportHydrostaticAdjustmentEnum());
			WriteData(fid,'object',this,'fieldname','stabilization','format','Integer');
			WriteData(fid,'object',this,'fieldname','vertex_pairing','format','DoubleMat','mattype',3);
			WriteData(fid,'object',this,'fieldname','penalty_factor','format','Double');

			//process requested outputs
			var outputs = this.requested_outputs;
			for (var i=0;i<outputs.length;i++){
				if (outputs[i] == 'default') {
					outputs.splice(i,1);
					var newoutputs=this.defaultoutputs(md);
					for (var j=0;j<newoutputs.length;j++) outputs.push(newoutputs[j]);
				}
			}
			WriteData(fid,'data',outputs,'enum',MasstransportRequestedOutputsEnum(),'format','StringArray');
		}//}}}
		this.defaultoutputs = function(md) { //{{{
			return ['Thickness','Surface','Base'];
		}//}}}
		this.fix=function() { //{{{
			this.spcthickness=NullFix(this.spcthickness,NaN);
			this.vertex_pairing=NullFix(this.vertex_pairing,NaN);
		}//}}}
	//properties 
	// {{{

	this.spcthickness           = NaN;
	this.isfreesurface          = 0;
	this.min_thickness          = 0;
	this.hydrostatic_adjustment = 0;
	this.stabilization          = 0;
	this.vertex_pairing         = NaN;
	this.penalty_factor         = 0;
	this.requested_outputs      = [];

	this.setdefaultparameters();
	//}}}
}
//MATICE class definition
//
//   Usage:
//      matice=matice();

function matice(){
	//methods
		this.setdefaultparameters = function(){ // {{{

			//ice density (kg/m^3)
			this.rho_ice=917.;

			//ocean water density (kg/m^3)
			this.rho_water=1023.;

			//fresh water density (kg/m^3)
			this.rho_freshwater=1000.;

			//water viscosity (N.s/m^2)
			this.mu_water=0.001787;  

			//ice heat capacity cp (J/kg/K)
			this.heatcapacity=2093.;

			//ice latent heat of fusion L (J/kg)
			this.latentheat=3.34*10^5;

			//ice thermal conductivity (W/m/K)
			this.thermalconductivity=2.4;
			
			//wet ice thermal conductivity (W/m/K)
			this.temperateiceconductivity=.24;

			//the melting point of ice at 1 atmosphere of pressure in K
			this.meltingpoint=273.15;

			//rate of change of melting point with pressure (K/Pa)
			this.beta=9.8*10^-8;

			//mixed layer (ice-water interface) heat capacity (J/kg/K)
			this.mixed_layer_capacity=3974.;

			//thermal exchange velocity (ice-water interface) (m/s)
			this.thermal_exchange_velocity=1.00*10^-4;

			//Rheology law: what is the temperature dependence of B with T
			//available: none, paterson and arrhenius
			this.rheology_law='Paterson';

			// GIA:
			this.lithosphere_shear_modulus  = 6.7*Math.pow(10,10);  // (Pa)
			this.lithosphere_density        = 3.32;       // (g/cm^-3)
			this.mantle_shear_modulus       = 1.45*Math.pow(10,11); // (Pa)
			this.mantle_density             = 3.34;       // (g/cm^-3)
			
			//SLR
			this.earth_density= 5512;  // average density of the Earth, (kg/m^3)


		} //}}}
		this.disp = function() {// {{{
			console.log(sprintf('   Materials:'));

			fielddisplay(this,'rho_ice','ice density [kg/m^3]');
			fielddisplay(this,'rho_water','ocean water density [kg/m^3]');
			fielddisplay(this,'rho_freshwater','fresh water density [kg/m^3]');
			fielddisplay(this,'mu_water','water viscosity [N s/m^2]');
			fielddisplay(this,'heatcapacity','heat capacity [J/kg/K]');
			fielddisplay(this,'thermalconductivity','ice thermal conductivity [W/m/K]');
			fielddisplay(this,'temperateiceconductivity','temperate ice thermal conductivity [W/m/K]');
			fielddisplay(this,'meltingpoint','melting point of ice at 1atm in K');
			fielddisplay(this,'latentheat','latent heat of fusion [J/kg]');
			fielddisplay(this,'beta','rate of change of melting point with pressure [K/Pa]');
			fielddisplay(this,'mixed_layer_capacity','mixed layer capacity [W/kg/K]');
			fielddisplay(this,'thermal_exchange_velocity','thermal exchange velocity [m/s]');
			fielddisplay(this,'rheology_B','flow law parameter [Pa/s^(1/n)]');
			fielddisplay(this,'rheology_n',"Glen's flow law exponent");
			fielddisplay(this,'rheology_law',"law for the temperature dependance of the rheology: 'None', 'Cuffey', 'Paterson', 'Arrhenius' or 'LliboutryDuval'");
			fielddisplay(this,'lithosphere_shear_modulus','Lithosphere shear modulus [Pa]');
			fielddisplay(this,'lithosphere_density','Lithosphere density [g/cm^-3]');
			fielddisplay(this,'mantle_shear_modulus','Mantle shear modulus [Pa]');
			fielddisplay(this,'mantle_density','Mantle density [g/cm^-3]');
			fielddisplay(this,'earth_density','Mantle density [kg/m^-3]');

		} // }}}
		this.classname = function() {// {{{
			return "matice";
		} // }}}
		this.checkconsistency = function(md,solution,analyses) { // {{{
			checkfield(md,'fieldname','materials.rho_ice','>',0);
			checkfield(md,'fieldname','materials.rho_water','>',0);
			checkfield(md,'fieldname','materials.rho_freshwater','>',0);
			checkfield(md,'fieldname','materials.mu_water','>',0);
			checkfield(md,'fieldname','materials.rheology_B','>',0,'timeseries',1,'NaN',1,'Inf',1);
			checkfield(md,'fieldname','materials.rheology_n','>',0,'size',[md.mesh.numberofelements,1]);
			checkfield(md,'fieldname','materials.rheology_law','values',['None','Cuffey','Paterson','Arrhenius','LliboutryDuval']);

			if(ArrayAnyEqual(ArrayIsMember(GiaAnalysisEnum(),analyses),1)){
				checkfield(md,'fieldname','materials.lithosphere_shear_modulus','>',0,'numel',1);
				checkfield(md,'fieldname','materials.lithosphere_density','>',0,'numel',1);
				checkfield(md,'fieldname','materials.mantle_shear_modulus','>',0,'numel',1);
				checkfield(md,'fieldname','materials.mantle_density','>',0,'numel',1);
			}
			if (ArrayAnyEqual(ArrayIsMember(SealevelriseAnalysisEnum(),analyses),1)){
				checkfield(md,'fieldname','materials.earth_density','>',0,'numel',1);
			}


		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'enum',MaterialsEnum(),'data',MaticeEnum(),'format','Integer');
			WriteData(fid,'object',this,'class','materials','fieldname','rho_ice','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','rho_water','enum',MaterialsRhoSeawaterEnum(),'format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','rho_freshwater','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','mu_water','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','heatcapacity','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','latentheat','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','thermalconductivity','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','temperateiceconductivity','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','meltingpoint','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','beta','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','mixed_layer_capacity','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','thermal_exchange_velocity','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','rheology_B','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',this,'class','materials','fieldname','rheology_n','format','DoubleMat','mattype',2);
			WriteData(fid,'data',StringToEnum(this.rheology_law),'enum',MaterialsRheologyLawEnum(),'format','Integer');
			WriteData(fid,'object',this,'class','materials','fieldname','lithosphere_shear_modulus','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','lithosphere_density','format','Double','scale',10^3);
			WriteData(fid,'object',this,'class','materials','fieldname','mantle_shear_modulus','format','Double');
			WriteData(fid,'object',this,'class','materials','fieldname','mantle_density','format','Double','scale',10^3);
			WriteData(fid,'object',this,'class','materials','fieldname','earth_density','format','Double');

		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{
		this.rho_ice                    = 0.;
		this.rho_water                  = 0.;
		this.rho_freshwater             = 0.;
		this.mu_water                   = 0.;
		this.heatcapacity               = 0.;
		this.latentheat                 = 0.;
		this.thermalconductivity        = 0.;
		this.temperateiceconductivity   = 0.;
		this.meltingpoint               = 0.;
		this.beta                       = 0.;
		this.mixed_layer_capacity       = 0.;
		this.thermal_exchange_velocity  = 0.;
		this.rheology_B   = NaN;
		this.rheology_n   = NaN;
		this.rheology_law = '';

		//gia: 
		this.lithosphere_shear_modulus  = 0.;
		this.lithosphere_density        = 0.;
		this.mantle_shear_modulus       = 0.;
		this.mantle_density             = 0.;

		//SLR
		this.earth_density= 5512;  // average density of the Earth, (kg/m^3)

		this.setdefaultparameters();
		//}}}
}
//MESH2D class definition
//
//   Usage:
//      mesh2d= new mesh2d();

function mesh2d () {
	//methods 
		this.setdefaultparameters = function (){ //{{{

			//the connectivity is the averaged number of nodes linked to a
			//given node through an edge. This connectivity is used to initially
			//allocate memory to the stiffness matrix. A value of 16 seems to
			//give a good memory/time ration. This value can be checked in
			//trunk/test/Miscellaneous/runme.m
			this.average_vertex_connectivity=25;
		}
		// }}}
		this.disp = function () { //{{{
			console.log(sprintf("   2D tria Mesh (horizontal):")); 

			console.log(sprintf("\n      Elements and vertices:"));
			fielddisplay(this,"numberofelements","number of elements");
			fielddisplay(this,"numberofvertices","number of vertices");
			fielddisplay(this,"elements","vertex indices of the mesh elements");
			fielddisplay(this,"x","vertices x coordinate [m]");
			fielddisplay(this,"y","vertices y coordinate [m]");
			fielddisplay(this,"edges","edges of the 2d mesh (vertex1 vertex2 element1 element2)");
			fielddisplay(this,"numberofedges","number of edges of the 2d mesh");

			console.log(sprintf("\n      Properties:"));
			fielddisplay(this,"vertexonboundary","vertices on the boundary of the domain flag list");
			fielddisplay(this,"segments","edges on domain boundary (vertex1 vertex2 element)");
			fielddisplay(this,"segmentmarkers","number associated to each segment");
			fielddisplay(this,"vertexconnectivity","list of vertices connected to vertex_i");
			fielddisplay(this,"elementconnectivity","list of vertices connected to element_i");
			fielddisplay(this,"average_vertex_connectivity","average number of vertices connected to one vertex");

			console.log(sprintf("\n      Extracted model:"));
			fielddisplay(this,"extractedvertices","vertices extracted from the model");
			fielddisplay(this,"extractedelements","elements extracted from the model");

			console.log(sprintf("\n      Projection:"));
			fielddisplay(this,"lat","vertices latitude [degrees]");
			fielddisplay(this,"long","vertices longitude [degrees]");
			fielddisplay(this,"epsg","EPSG code (ex: 3413 for UPS Greenland, 3031 for UPS Antarctica)");
		} //}}}
		this.classname = function () { //{{{
			return "mesh2d";
		} //}}}
		this.domaintype=function (){ // {{{
			return '2Dhorizontal';
		} // }}}
		this.dimension = function () { //{{{
			return 2;
		} //}}}
		this.elementtype = function() {//{{{
			return 'Tria';
		} // }}}
		this.checkconsistency = function(md,solution,analyses){ //{{{

			checkfield(md,'fieldname','mesh.x','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mesh.y','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mesh.elements','NaN',1,'Inf',1,'>',0,'values',NewArrayFillIncrement(md.mesh.numberofvertices,1,1));
			checkfield(md,'fieldname','mesh.elements','size',[md.mesh.numberofelements, 3]);
			if(ArrayAnyEqual(ArrayIsMember(NewArrayFillIncrement(md.mesh.numberofvertices,1,1),ArraySort(ArrayUnique(MatrixToList(md.mesh.elements)))),0)){
				md.checkmessage('orphan nodes have been found. Check the mesh outline');
			}
			checkfield(md,'fieldname','mesh.numberofelements','>',0);
			checkfield(md,'fieldname','mesh.numberofvertices','>',0);
			checkfield(md,'fieldname','mesh.average_vertex_connectivity','>=',9,'message',"'mesh.average_vertex_connectivity' should be at least 9 in 2d");
			checkfield(md,'fieldname','mesh.segments','NaN',1,'Inf',1,'>',0,'size',[NaN, 3]);

			switch(solution){
			case ThermalSolutionEnum():
				checkmessage(md,'thermal not supported for 2d mesh');
				break;
			default:
				break
			}
		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'enum',DomainTypeEnum(),'data',StringToEnum('Domain' + this.domaintype()),'format','Integer');
			WriteData(fid,'enum',DomainDimensionEnum(),'data',this.dimension(),'format','Integer');
			WriteData(fid,'enum',MeshElementtypeEnum(),'data',StringToEnum(this.elementtype()),'format','Integer');
			WriteData(fid,'object',this,'class','mesh','fieldname','x','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mesh','fieldname','y','format','DoubleMat','mattype',1);
			WriteData(fid,'enum',MeshZEnum(),'data',NewArrayFill(this.numberofvertices,0),'format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mesh','fieldname','elements','format','DoubleMat','mattype',2);
			WriteData(fid,'object',this,'class','mesh','fieldname','numberofelements','format','Integer');
			WriteData(fid,'object',this,'class','mesh','fieldname','numberofvertices','format','Integer');
			WriteData(fid,'object',this,'class','mesh','fieldname','average_vertex_connectivity','format','Integer');
			WriteData(fid,'object',this,'class','mesh','fieldname','vertexonboundary','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mesh','fieldname','segments','format','DoubleMat','mattype',3);
		}//}}}
		this.fix=function() { //{{{
			//Transform objects into Float64Arrays:
			this.x=FloatFix(this.x,this.numberofvertices); 
			this.y=FloatFix(this.y,this.numberofvertices); 
			this.edges=NullFix(this.edges,NaN);
			this.vertexonboundary=FloatFix(this.vertexonboundary,this.numberofvertices); 
			this.segmentmarkers=FloatFix(this.segmentmarkers,this.segments.length);
			this.extractedvertices=NullFix(this.extractedvertices,NaN);
			this.extractedelements=NullFix(this.extractedelements,NaN);
			this.lat=NullFix(this.lat,NaN);
			this.long=NullFix(this.long,NaN);
		}//}}}

	//properties 
	// {{{
		this.x                           = NaN;
		this.y                           = NaN;
		this.elements                    = NaN;
		this.numberofelements            = 0;
		this.numberofvertices            = 0;
		this.numberofedges               = 0;

		this.lat                         = NaN;
		this.long                        = NaN;
		this.epsg                        = 0;

		this.vertexonboundary            = NaN;

		this.edges                       = NaN;
		this.segments                    = NaN;
		this.segmentmarkers              = NaN;
		this.vertexconnectivity          = NaN;
		this.elementconnectivity         = NaN;
		this.average_vertex_connectivity = 0;

		this.extractedvertices           = NaN;
		this.extractedelements           = NaN;

		this.setdefaultparameters();
		//}}}
}
//MESH3DSURFACE class definition
//
//   Usage:
//      mesh3dsurface= new mesh3dsurface();

function mesh3dsurface () {
	//methods 
		this.setdefaultparameters = function (){ //{{{

			//the connectivity is the averaged number of nodes linked to a
			//given node through an edge. This connectivity is used to initially
			//allocate memory to the stiffness matrix. A value of 16 seems to
			//give a good memory/time ration. This value can be checked in
			//trunk/test/Miscellaneous/runme.m
			this.average_vertex_connectivity=25;
		}
		// }}}
		this.disp = function () { //{{{
			console.log(sprintf("   2D tria Mesh (3D surface):")); 

			console.log(sprintf("\n      Elements and vertices:"));
			fielddisplay(this,"numberofelements","number of elements");
			fielddisplay(this,"numberofvertices","number of vertices");
			fielddisplay(this,"elements","vertex indices of the mesh elements");
			fielddisplay(this,"x","vertices x coordinate [m]");
			fielddisplay(this,"y","vertices y coordinate [m]");
			fielddisplay(this,"z","vertices z coordinate [m]");
			fielddisplay(this,"edges","edges of the 2d mesh (vertex1 vertex2 element1 element2)");
			fielddisplay(this,"numberofedges","number of edges of the 2d mesh");

			console.log(sprintf("\n      Properties:"));
			fielddisplay(this,"vertexonboundary","vertices on the boundary of the domain flag list");
			fielddisplay(this,"segments","edges on domain boundary (vertex1 vertex2 element)");
			fielddisplay(this,"segmentmarkers","number associated to each segment");
			fielddisplay(this,"vertexconnectivity","list of vertices connected to vertex_i");
			fielddisplay(this,"elementconnectivity","list of vertices connected to element_i");
			fielddisplay(this,"average_vertex_connectivity","average number of vertices connected to one vertex");

			console.log(sprintf("\n      Extracted model:"));
			fielddisplay(this,"extractedvertices","vertices extracted from the model");
			fielddisplay(this,"extractedelements","elements extracted from the model");

			console.log(sprintf("\n      Projection:"));
			fielddisplay(this,"lat","vertices latitude [degrees]");
			fielddisplay(this,"long","vertices longitude [degrees]");
			fielddisplay(this,"r","vertices radius [m]");
		} //}}}
		this.classname = function () { //{{{
			return "mesh3dsurface";
		} //}}}
		this.domaintype=function (){ // {{{
			return '3Dsurface';
		} // }}}
		this.dimension = function () { //{{{
			return 2;
		} //}}}
		this.elementtype = function() {//{{{
			return 'Tria';
		} // }}}
		this.checkconsistency = function(md,solution,analyses){ //{{{

			checkfield(md,'fieldname','mesh.x','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mesh.y','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mesh.z','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mesh.lat','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mesh.long','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mesh.r','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			checkfield(md,'fieldname','mesh.elements','NaN',1,'Inf',1,'>',0,'values',NewArrayFillIncrement(md.mesh.numberofvertices,1,1));
			checkfield(md,'fieldname','mesh.elements','size',[md.mesh.numberofelements, 3]);
			if(ArrayAnyEqual(ArrayIsMember(NewArrayFillIncrement(md.mesh.numberofvertices,1,1),ArraySort(ArrayUnique(MatrixToList(md.mesh.elements)))),0)){
				md.checkmessage('orphan nodes have been found. Check the mesh outline');
			}
			checkfield(md,'fieldname','mesh.numberofelements','>',0);
			checkfield(md,'fieldname','mesh.numberofvertices','>',0);
			checkfield(md,'fieldname','mesh.average_vertex_connectivity','>=',9,'message',"'mesh.average_vertex_connectivity' should be at least 9 in 2d");
			checkfield(md,'fieldname','mesh.segments','NaN',1,'Inf',1,'>',0,'size',[NaN, 3]);

		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'enum',DomainTypeEnum(),'data',StringToEnum('Domain' + this.domaintype()),'format','Integer');
			WriteData(fid,'enum',DomainDimensionEnum(),'data',this.dimension(),'format','Integer');
			WriteData(fid,'enum',MeshElementtypeEnum(),'data',StringToEnum(this.elementtype()),'format','Integer');
			WriteData(fid,'object',this,'class','mesh','fieldname','x','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mesh','fieldname','y','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mesh','fieldname','z','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mesh','fieldname','lat','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mesh','fieldname','long','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mesh','fieldname','r','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mesh','fieldname','elements','format','DoubleMat','mattype',2);
			WriteData(fid,'object',this,'class','mesh','fieldname','numberofelements','format','Integer');
			WriteData(fid,'object',this,'class','mesh','fieldname','numberofvertices','format','Integer');
			WriteData(fid,'object',this,'class','mesh','fieldname','average_vertex_connectivity','format','Integer');
			WriteData(fid,'object',this,'class','mesh','fieldname','vertexonboundary','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','mesh','fieldname','segments','format','DoubleMat','mattype',3);
		}//}}}
		this.fix=function() { //{{{
			//Transform objects into Float64Arrays:
			this.x=FloatFix(this.x,this.numberofvertices); 
			this.y=FloatFix(this.y,this.numberofvertices); 
			this.z=FloatFix(this.y,this.numberofvertices); 
			this.r=FloatFix(this.y,this.numberofvertices); 
			this.edges=NullFix(this.edges,NaN);
			this.vertexonboundary=FloatFix(this.vertexonboundary,this.numberofvertices); 
			this.segmentmarkers=FloatFix(this.segmentmarkers,this.segments.length);
			this.extractedvertices=NullFix(this.extractedvertices,NaN);
			this.extractedelements=NullFix(this.extractedelements,NaN);
			this.lat=NullFix(this.lat,NaN);
			this.long=NullFix(this.long,NaN);
		}//}}}

	//properties 
	// {{{
		this.x                           = NaN;
		this.y                           = NaN;
		this.z                           = NaN;
		this.elements                    = NaN;
		this.numberofelements            = 0;
		this.numberofvertices            = 0;
		this.numberofedges               = 0;

		this.lat                         = NaN;
		this.long                        = NaN;
		this.r                           = NaN;

		this.vertexonboundary            = NaN;
		this.edges                       = NaN;
		this.segments                    = NaN;
		this.segmentmarkers              = NaN;
		this.vertexconnectivity          = NaN;
		this.elementconnectivity         = NaN;
		this.average_vertex_connectivity = 0;

		this.extractedvertices           = NaN;
		this.extractedelements           = NaN;

		this.setdefaultparameters();
		//}}}
}
//MISCELLANEOUS class definition
//
//   Usage:
//      miscellaneous=new miscellaneous();

function miscellaneous (){
	//methods
	this.setdefaultparameters = function(){// {{{
	}// }}}
	this.disp= function(){// {{{

		console.log(sprintf('   miscellaneous parameters:'));

		fielddisplay(this,'notes','notes in a cell of strings');
		fielddisplay(this,'name','model name');
		fielddisplay(this,'dummy','empty field to store some data');

	}// }}}
	this.classname= function(){// {{{
		return "miscellaneous";
	}// }}}
		this.checkconsistency= function(md,solution,analyses) {// {{{

			checkfield(md,'fieldname','miscellaneous.name','empty',1);
		}// }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'fieldname','name','format','String');
		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{
	this.notes = '';
	this.name  = '';
	this.dummy = [];

	this.setdefaultparameters();
	//}}}
}
//MODEL class definition
//
//   Usage:
//      md = new model()

function model () {
	//methods
		this.disp = function() { //{{{
			console.log(sprintf("class model echo: "));
			console.log(sprintf("%19s: %-22s -- %s","mesh"            ,"[1x1 " + typeof(this.mesh) + "]","mesh properties"));
			console.log(sprintf("%19s: %-22s -- %s","mask"            ,"[1x1 " + typeof(this.mask) + "]","defines grounded and floating elements"));
			console.log(sprintf("%19s: %-22s -- %s","geometry"        ,"[1x1 " + typeof(this.geometry) + "]","surface elevation, bedrock topography, ice thickness,..."));
			console.log(sprintf("%19s: %-22s -- %s","constants"       ,"[1x1 " + typeof(this.constants) + "]","physical constants"));
			console.log(sprintf("%19s: %-22s -- %s","smb"             ,"[1x1 " + typeof(this.smb) + "]","surface mass balance"));
			console.log(sprintf("%19s: %-22s -- %s","basalforcings"   ,"[1x1 " + typeof(this.basalforcings) + "]","bed forcings"));
			console.log(sprintf("%19s: %-22s -- %s","materials"       ,"[1x1 " + typeof(this.materials) + "]","material properties"));
			console.log(sprintf("%19s: %-22s -- %s","damage"          ,"[1x1 " + typeof(this.damage) + "]","parameters for damage evolution solution"));
			console.log(sprintf("%19s: %-22s -- %s","friction"        ,"[1x1 " + typeof(this.friction) + "]","basal friction/drag properties"));
			console.log(sprintf("%19s: %-22s -- %s","flowequation"    ,"[1x1 " + typeof(this.flowequation) + "]","flow equations"));
			console.log(sprintf("%19s: %-22s -- %s","timestepping"    ,"[1x1 " + typeof(this.timestepping) + "]","time stepping for trans models"));
			console.log(sprintf("%19s: %-22s -- %s","initialization"  ,"[1x1 " + typeof(this.initialization) + "]","initial guess/state"));
			console.log(sprintf("%19s: %-22s -- %s","rifts"           ,"[1x1 " + typeof(this.rifts) + "]","rifts properties"));
			console.log(sprintf("%19s: %-22s -- %s","slr"             ,"[1x1 " + typeof(this.slr) + "]","slr forcings"));
			console.log(sprintf("%19s: %-22s -- %s","debug"           ,"[1x1 " + typeof(this.debug) + "]","debugging tools (valgrind, gprof)"));
			console.log(sprintf("%19s: %-22s -- %s","verbose"         ,"[1x1 " + typeof(this.verbose) + "]","verbosity level in solve"));
			console.log(sprintf("%19s: %-22s -- %s","settings"        ,"[1x1 " + typeof(this.settings) + "]","settings properties"));
			console.log(sprintf("%19s: %-22s -- %s","toolkits"        ,"[1x1 " + typeof(this.toolkits) + "]","PETSc options for each solution"));
			console.log(sprintf("%19s: %-22s -- %s","cluster"         ,"[1x1 " + typeof(this.cluster) + "]","cluster parameters (number of cpus...)"));
			console.log(sprintf("%19s: %-22s -- %s","balancethickness","[1x1 " + typeof(this.balancethickness) + "]","parameters for balancethickness solution"));
			console.log(sprintf("%19s: %-22s -- %s","stressbalance"   ,"[1x1 " + typeof(this.stressbalance) + "]","parameters for stressbalance solution"));
			console.log(sprintf("%19s: %-22s -- %s","groundingline"   ,"[1x1 " + typeof(this.groundingline) + "]","parameters for groundingline solution"));
			console.log(sprintf("%19s: %-22s -- %s","hydrology"       ,"[1x1 " + typeof(this.hydrology) + "]","parameters for hydrology solution"));
			console.log(sprintf("%19s: %-22s -- %s","masstransport"   ,"[1x1 " + typeof(this.masstransport) + "]","parameters for masstransport solution"));
			console.log(sprintf("%19s: %-22s -- %s","thermal"         ,"[1x1 " + typeof(this.thermal) + "]","parameters for thermal solution"));
			console.log(sprintf("%19s: %-22s -- %s","steadystate"     ,"[1x1 " + typeof(this.steadystate) + "]","parameters for steadystate solution"));
			console.log(sprintf("%19s: %-22s -- %s","trans"       ,"[1x1 " + typeof(this.trans) + "]","parameters for trans solution"));
			console.log(sprintf("%19s: %-22s -- %s","levelset"        ,"[1x1 " + typeof(this.levelset) + "]","parameters for moving boundaries (level-set method)"));
			console.log(sprintf("%19s: %-22s -- %s","calving"         ,"[1x1 " + typeof(this.calving) + "]","parameters for calving"));
			console.log(sprintf("%19s: %-22s -- %s","gia"             ,"[1x1 " + typeof(this.gia) + "]","parameters for gia solution"));
			console.log(sprintf("%19s: %-22s -- %s","autodiff"        ,"[1x1 " + typeof(this.autodiff) + "]","automatic differentiation parameters"));
			console.log(sprintf("%19s: %-22s -- %s","flaim"           ,"[1x1 " + typeof(this.flaim) + "]","flaim parameters"));
			console.log(sprintf("%19s: %-22s -- %s","inversion"       ,"[1x1 " + typeof(this.inversion) + "]","parameters for inverse methods"));
			console.log(sprintf("%19s: %-22s -- %s","qmu"             ,"[1x1 " + typeof(this.qmu) + "]","dakota properties"));
			console.log(sprintf("%19s: %-22s -- %s","outputdefinition","[1x1 " + typeof(this.outputdefinition) + "]","output definition"));
			console.log(sprintf("%19s: %-22s -- %s","results"         ,"[1x1 " + typeof(this.results) + "]","model results"));
			console.log(sprintf("%19s: %-22s -- %s","radaroverlay"    ,"[1x1 " + typeof(this.radaroverlay) + "]","radar image for plot overlay"));
			console.log(sprintf("%19s: %-22s -- %s","miscellaneous"   ,"[1x1 " + typeof(this.miscellaneous) + "]","miscellaneous fields"));
		} //}}}
		this.setdefaultparameters = function () { // {{{

			//initialize subclasses
			this.mesh             = new mesh2d();
			this.mask             = new mask();
			this.constants        = new constants();
			this.geometry         = new geometry();
			this.initialization   = new initialization();
			this.smb              = new SMBforcing();
			this.basalforcings    = new basalforcings();
			this.friction         = new friction();
			this.rifts            = new rifts();
			this.slr              = new slr();
			this.timestepping     = new timestepping();
			this.groundingline    = new groundingline();
			this.materials        = new matice();
			this.damage           = new damage();
			this.flowequation     = new flowequation();
			this.debug            = new debug();
			this.verbose          = new verbose();
			this.settings         = new settings();
			this.toolkits         = new toolkits();
			this.cluster          = new local();
			this.balancethickness = new balancethickness();
			this.stressbalance    = new stressbalance();
			this.hydrology        = new hydrologyshreve();
			this.masstransport    = new masstransport();
			this.thermal          = new thermal();
			this.steadystate      = new steadystate();
			this.trans            = new trans();
			this.levelset		  = new levelset();
			this.calving          = new calving();
			this.gia              = new gia();
			this.autodiff         = new autodiff();
			this.flaim            = new flaim();
			this.inversion        = new inversion();
			this.qmu              = new qmu();
			this.radaroverlay     = new radaroverlay();
			this.results          = {};
			this.outputdefinition = new outputdefinition();
			this.miscellaneous    = new miscellaneous();
			this.priv             = new priv();
		} //}}}
		this.checkmessage = function(string){ //{{{
			console.log('model not consistent: ' + string);
			md.private.isconsistent=false;
		} //}}}
		this.fix = function(){ //{{{

			for (var field in this){

				//Some properties do not need to be fixed
				if (field == 'results' | field =='radaroverlay' | field == 'toolkits' | field =='cluster' | field == 'flaim' | field == 'priv') continue;

				//Check that current field is a class
				if(typeof this[field] == 'function'){
					continue;
				}

				//Fix current object
				this[field].fix(this);
			}

		} //}}}
	//properties
	// {{{
		//Careful here: no other class should be used as default value this is a bug of matlab
		this.mesh             = 0;
		this.mask             = 0;

		this.geometry         = 0;
		this.constants        = 0;
		this.smb              = 0;
		this.basalforcings    = 0;
		this.materials        = 0;
		this.damage           = 0;
		this.friction         = 0;
		this.flowequation     = 0;
		this.timestepping     = 0;
		this.initialization   = 0;
		this.rifts            = 0;
		this.slr              = 0;

		this.debug            = 0;
		this.verbose          = 0;
		this.settings         = 0;
		this.toolkits         = 0;
		this.cluster          = 0;

		this.balancethickness = 0;
		this.stressbalance    = 0;
		this.groundingline    = 0;
		this.hydrology        = 0;
		this.masstransport    = 0;
		this.thermal          = 0;
		this.steadystate      = 0;
		this.trans            = 0;
		this.levelset         = 0;
		this.calving          = 0;
		this.gia              = 0;

		this.autodiff         = 0;
		this.flaim            = 0;
		this.inversion        = 0;
		this.qmu              = 0;

		this.results          = 0;
		this.outputdefinition = 0;
		this.radaroverlay     = 0;
		this.miscellaneous    = 0;
		this.priv             = 0;

		//set default values for fields
		this.setdefaultparameters();

		//}}}
}
//OUTPUTDEFINITION class definition
//
//   Usage:
//      outputdefinition=new outputdefinition();

function outputdefinition (){
	//methods
	this.setdefaultparameters = function(){// {{{
		this.definitions=[];
	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   outputdefinition:'));
		fielddisplay(this,'definitions','list of potential outputs that can be requested, but which need additional data to be defined');


	}// }}}
	this.classname= function(){// {{{
		return "outputdefinition";
	}// }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{

			checkfield(md,'fieldname','outputdefinition.definitions','cell',1);

			for (var i=0;i<this.definitions.length;i++){
				this.definitions[i].checkconsistency(md,solution,analyses);
			}

		} // }}}
		this.marshall=function(md,fid) { //{{{

		var enums=NewArrayFill(this.definitions.length,0);
		for (var i=0;i<this.definitions.length;i++){
			this.definitions[i].marshall(md,fid);
			classdefinition=this.definitions[i].classname();
			classdefinition=classdefinition.charAt(0).tuUpperCase() + classdefinition.slice(1); //so it matches our enums definitions.
			enums[i]=StringToEnum(classdefinition);
		}
		enums=ArrayUnique(enums);
		if (enums.length==0){ enums=NaN; }
		WriteData(fid,'data',enums,'enum',OutputdefinitionListEnum(),'format','DoubleMat','mattype',1);
		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{
	this.definitions                 = [];
	this.setdefaultparameters();
	//}}}
}
//PAIROPTIONS class definition
//
//   Usage:
//      pairoptions=pairoptions();
//      pairoptions=pairoptions('module',true,'solver',false);

function pairoptions(args) { 


	//methods
	this.constructor = function (args) {// {{{

		//initialize list
		if (args.length==0){
			this.list=[];
		}
		else{

			//check length of input
			if (args.length % 2){
				throw Error('pairoptions error message: invalid parameter/value pair arguments') 
			}
			numoptions = args.length/2;

			//Allocate memory
			this.list= Create2DArray(numoptions,3);

			//go through args and build list of obj
			for (var i=0;i<numoptions;i++){
				if (typeof args[2*i] === 'string'){
					this.list[i][0]=args[2*i];
					this.list[i][1]=args[2*i+1];
					this.list[i][2]=false;  //used?
				}
				else{
					//option is not a string, ignore it
					console.log(sprintf('%s%i%s\n','WARNING: option number ',i,' is not a string, it will be ignored'));
					this.list[i][0]=[];
					this.list[i][1]=[];
					this.list[i][2]=[];
					continue
				}
			}
		}
	}// }}}
	this.addfield = function (field, value){ // {{{
		if (typeof field === 'string'){
			this.list.push([field,value,false]);
		}
	}// }}}
	this.numoptions = function (){ // {{{
		return this.list.length;
	}// }}}
	this.addfielddefault = function (field,value){ // {{{
		//ADDFIELDDEFAULT - add a field to an options list if it does not exist
		if (typeof field === 'string'){
			if (!this.exist(field)){
				this.list.push([field,value,true]); //true is a default so user will not be notified if not used
			}
		}
	} // }}}
	this.AssignObjectFields = function(object){ // {{{
		//ASSIGNOBJECTFIELDS - assign object fields from options
		for (var i=0;i<list.length;i++){
			fieldname=list[i][0];
			fieldvalue=list[i][1];
			if (fieldname in object){
				obj2[fieldname]=fieldvalue;
			}
			else{
				console.log(sprintf("%s'%s'%s%s\n",'WARNING: ',fieldname, 'is not a property of ',typeof object));
			}
		}
	} // }}}
	this.changefieldvalue = function(field,newvalue){ // {{{
		//CHANGEOPTIONVALUE - change the value of an option in an option list

		var found=0;
		for (var i=0;i<this.list.length;i++){
			if (this.list[i][0] === field){
				found=1;
			}
		}

		if (found==0){
			this.list.push([field,newvalue,true]); // do not notify user if unused
		}
		else{
			for (var i=0;i<this.list.length;i++){
				if (this.list[i][0] === field){
					this.list[i][1] = newvalue;
				}
			}
		}
	} // }}}
	this.deleteduplicates = function(warn){ // {{{
		//DELETEDUPLICATES - delete duplicates in an option list

		//track the first occurrence of each option
		var indices=NewArrayFill(this.list.length,0);
		for (var i=0;i<this.list.length;i++){
			if(indices[i]==0){
				for(var j=i+1;j<this.list.length;j++){
					if (this.list[i][0] === this.list[j][0])indices[j]=1;
				}
			}
		}
		sumindices=ArraySum(indices);

		//remove duplicates from the options list
		newlist=Create2DArray(sumindices,3);
		var count=0;
		for (var i=0;i<this.list.length;i++){
			if (indices[i]==1) if (warn) console.log(sprintf("%s%s%s\n",'WARNING: option ', this.list[i,0],' appeared more than once. Only its first occurrence will be kept'));
			else{
				newlist[count]=this.list[i];
				count++;
			}
		}
	} // }}}
	this.displayunused = function (){ // {{{
		//DISPLAYUNUSED - display unused options

		for (var i=0;i<this.list.length;i++){
			if (!(this.list[i][2])){
				console.log(sprintf("%s%s%s\n",'WARNING: option ',this.list[i][0],' was not used'));
			}
		}
	}// }}}
	this.disp = function (){ //{{{
		if (this.list.length){
			console.log(sprintf('   pairoptions: (%i)\n',this.list.length));
			for (var i=0;i<this.list.length;i++){
				if (typeof this.list[i][1] === 'string'){
					console.log(sprintf("     field: '%s' value(string): ''%s''",this.list[i][0],this.list[i][1]));
				}
				else if( typeof this.list[i][1] === 'number'){
					console.log(sprintf("     field: '%s' value(number): %g",this.list[i][0],this.list[i][1]));
				}
				else if( IsArray(this.list[i][1])){
					console.log(sprintf("     field: '%s' value(array): [%i]",this.list[i][0],this.list[i][1].length));
				}
			}
		}
		else{
			console.log(sprintf('   list: empty'));
		}
	}// }}}
	this.exist = function (field) { //{{{

		//EXIST - check if the option exists
		//some argument checking: 
		if (!(typeof field === 'string')){
			throw Error('exist error message: field should be a string');
		}

		//Recover option
		var bool=0;
		for (var i=0;i<this.list.length;i++){
			if (this.list[i][0] === field){
				bool=1;
				this.list[i][2]=1; //It is a default so user will not be notified if not used
				break;
			}
		}
		return bool;
	} // }}}
	this.fieldoccurrences = function(field){ // {{{

		//FIELDOCCURRENCES - get number of occurrence of a field
		var num=0;

		//check input 
		if (!(typeof field === 'string')){
			throw Error('exist error message: field should be a string');
		}

		//count number of occurrences:
		for (var i=0;i<this.list.length;i++) if (this.list[i][0] === field)num++;

		return num;

	} // }}}
	this.getfieldvalue = function(field){ // {{{
		//GETOPTION - get the value of an option
		//
		//   Usage:
		//      value=pairoptions.getfieldvalue(field,varargin)
		//
		//   Find an option value from a field. A default option
		//   can be given in input if the field does not exist
		//
		//   Examples:
		//      value=pairoptions.getfieldvalue('caxis');
		//      value=pairoptions.getfieldvalue('caxis',[0 2]);

		//some argument checking: 
		if(!(arguments.length==1 | arguments.length==2)){
			error('pairoptions usage error: getfieldvalue bad usage');
		}

		if (!(typeof field === 'string')){
			throw Error('pairoptions error message: field should be a string');
		}

		//Recover option
		for(var i=0;i<this.list.length;i++){
			if (this.list[i][0] === field){
				this.list[i][2]=1; //option used
				return value=this.list[i][1];
			}
		}

		//The option has not been found, output default if provided
		if (arguments.length==2){
			return arguments[1];
		}
		else{
			throw Error(sprintf("%s%s%s\n",'error message: field ',field,' has not been provided by user (and no default value has been specified)'));
		}
	} // }}}
	this.removefield = function(field,warn){// {{{

		//REMOVEFIELD - delete a field in an option list
		//
		//   Usage:
		//      options.removefield(field,warn)
		//
		//   if warn==1 display an info message to warn user that
		//   some of his options have been removed.

		//check if field exists
		if (this.exist(field)){

			var indices;
			var count;

			//find where the field is located
			indices=NewArrayFill(this.list.length,1);
			for (var i=0;i<this.list.length;i++)if(this.list[i][1] === field)indices[i]=0;
			sumindices=ArraySum(indices);

			//remove duplicates from the options list
			newlist=Create2DArray(sumindices,3);

			count=0;
			for (var i=0;i<this.list.length;i++){
				if(!(this.list[i][1] === field)){
					newlist[count]=this.list[i];
					count++;
				}
			}
			this.list=newlist;

			//warn user if requested
			if (warn){
				console.log(sprintf("%s%s%s\n",'removefield info: option ',field,' has been removed from the list of options.'));
			}
		}
	} // }}}
	this.marshall = function(fid,firstindex){// {{{

		throw Error('pairoptions marshall error: not implemented yet!');
	} // }}}

	//properties 
	this.list         = [];
	this.constructor(args);
}
//PLOTOPTIONS class definition
//
//   Usage:
//      plotoptions = plotoptions(varargin)

function plotoptions(args) {
	//methods
	this.disp = function (){ // {{{
		console.log(sprintf('\nplotoptions = \n'));
		console.log(sprintf('   figurenumber: %i',this.figurenumber));
		console.log(sprintf('   numberofplots: %i',this.numberofplots));
		if (this.list.length){
			for (var i=0;i<this.list.length;i++){
				console.log(sprintf('\n   options of plot number %i',i+1));
				this.list[i].disp();
			}
		}
		else{
			console.log(sprintf('   list: empty'));
		}
	}
	//}}}
	this.constructor = function (args){ // {{{

		//check length of input
		if (args.length % 2){
			for (i=0;i<args.length;i+=2){
				if (!(typeof args[i] === 'string')){
					console.log('Last valid option: ' + args[i-2]);
					break;
				}
			}
			throw Error('plotoptions error message: invalid parameter/value pair arguments');
		}

		//go through varargin and build list (like pairoptions)
		var rawoptions=new pairoptions(args);
		numoptions=rawoptions.numoptions();

		var counter=0;
		for (i=0;i<numoptions;i++){
			if(typeof args[2*i] === 'string')counter++;
		}
		rawlist=Create2DArray(counter,2);
		var counter=0;
		for (i=0;i<numoptions;i++){
			optionname=args[2*i];
			optionval=args[2*i+1];
			if(typeof optionname === 'string'){
				rawlist[counter][0]=optionname;
				rawlist[counter][1]=optionval;
				counter++;
			}
			else{
				//option is not a string, ignore it
				console.log(sprintf("%s%i%s\n",'WARNING: option number ',i,' is not a string, it will be ignored'));
				rawlist[counter]=[];
				continue
			}
		}
		
			
		//get number of data to be plotted
		numberofplots=rawoptions.fieldoccurrences('data');
		this.numberofplots=numberofplots;

		//figure out wether alloptions flog is on
		if (rawoptions.getfieldvalue('alloptions','off') === 'on') allflag=1;
		else allflag=0;

		//initialize list
		var list=new Array(numberofplots);
		for (i=0;i<numberofplots;i++){
			list[i]=new pairoptions([]);
		}
				
		//process plot options
		for(var i=0;i<rawlist.length;i++){

			//If alloptions flag has is on, apply to all plots
			if (allflag & !(rawlist[i][0] === 'data') & (rawlist[i][0].indexOf('#') == -1)){
				for(var j=0;j<numberofplots;j++){
					list[j].addfield(rawlist[i][0],rawlist[i][1]);
				}
			}
			else if (rawlist[i][0].indexOf('#') != -1){ //option contains '#'

				//get suplot(s) associated
				string=rawlist[i][0].split('#');
				plotnums=string[1];
				field=string[0];

				//divide plotnums if there is a comma ','
				plotnums=plotnums.split(',');

				//loop over plotnums
				for (k=0;k<plotnums.length;k++){
					plotnum=plotnums[k];

					//Empty
					if (plotnum === '') continue;

					else if (plotnum === 'all'){ //pound all
						for(var j=0;j<numberofplots;j++){
							list[j].addfield(field,rawlist[i][1]);
						}
					}
					else if (plotnum.indexOf('-')!=-1){  //pound i-j
						nums=plotnum.split('-');
						if (nums.length!=2) continue;
						if ((nums[0] == '') | (nums[1] === '')){
							throw Error(sprintf("%s%s\n",'the option #i-j is not set properly for ',field));
						}
						for (j=(Number(nums[0])-1);j<(Number(nums[1])); j++){
							list[j].addfield(field,rawlist[i][1]);
						}
					}
					else{ //pound i
						//assign to subplot
						if (Number(plotnum)>numberofplots){
							throw Error(sprintf("%s%s%s%i%s\n",'plotoptions error message: ',field,' cannot be assigned (',plotnum,' exceeds maximum number of plot)'));
						}
						list[Number(plotnum)-1].addfield(field,rawlist[i][1]);
					}
				}
			}
			else{ //assign option field to corresponding subplot

				
				//go through all subplot and assign to the first one free
				var inc=0;
				
				while (inc<numberofplots){
					
					if (!list[inc].exist(rawlist[i][0])){
						list[inc].addfield(rawlist[i][0],rawlist[i][1]);
						break
					}
					else inc++;
				}

				if (inc>numberofplots-1){
					console.log(sprintf("%s%s%s\n",'plot info message: too many ',rawlist[i][0],' options'));
				}
			}
		}

		//check that there is no duplicates
		for (var i=0;i<numberofplots;i++) list[i].deleteduplicates();

		//allocate canvasid automatically
		for (var i=0;i<numberofplots;i++) {
			if (!list[i].exist('canvasid')) {
				list[i].addfield('canvasid',i);
			}
		}

		//Get figure number (should be in options for subplot 1)
		this.figurenumber=list[0].getfieldvalue('figure',1);
		list[0].removefield('figure',0);

		//asign output
		this.list=list;

	} //}}}
	//properties
	// {{{
	this.numberofplots = 0;
	this.figurenumber  = 1;
	this.list          = [];
	this.constructor(args);
	//}}}
}
//PRIV class definition
//
//   Usage:
//      priv =new priv();

function priv (){
	//methods
	this.setdefaultparameters = function(){// {{{
	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   private parameters: do not change'));

		fielddisplay(this,'isconsistent','is model this consistent');
		fielddisplay(this,'runtimename','name of the run launched');
		fielddisplay(this,'bamg','structure with mesh properties constructed if bamg is used to mesh the domain');
		fielddisplay(this,'solution','type of solution launched');
	}// }}}
	this.checkconsistency = function(md,solution,analyses){ // {{{

	}// % }}}
	//properties 
	// {{{
	this.isconsistent = true;
	this.runtimename  = '';
	this.bamg         = {};
	this.solution     = '';

	this.setdefaultparameters();
	//}}}
}
//QMU class definition
//
//   Usage:
//      qmu=new qmu();

function qmu (){
	//methods
	this.setdefaultparameters = function(){// {{{
	}// }}}
	this.disp= function(){// {{{

		console.log(sprintf('   qmu parameters:'));

		/*fielddisplay(this,'isdakota','is qmu analysis activated?');
		for (var i=0;i<this.variables.length;i++){
			console.log(sprintf('         variables%s:  (arrays of each variable class)',...
						string_dim(this.variables,i)));
		}
		fnames=fieldnames(this.variables(i));
		maxlen=0;
		for j=1:numel(fnames)
			maxlen=max(maxlen,length(fnames{j}));
		end

			for j=1:numel(fnames)
				console.log(sprintf(['            %-' num2str(maxlen+1) 's:    [%ix%i]    ''%s'''],...
							fnames{j},size(this.variables.(fnames{j})),class(this.variables.(fnames{j}))));
		end
			end
			for i=1:numel(this.responses)
				console.log(sprintf('         responses%s:  (arrays of each response class)',...
							string_dim(this.responses,i)));
		fnames=fieldnames(this.responses(i));
		maxlen=0;
		for j=1:numel(fnames)
			maxlen=max(maxlen,length(fnames{j}));
		end

			for j=1:numel(fnames)
				console.log(sprintf(['            %-' num2str(maxlen+1) 's:    [%ix%i]    ''%s'''],...
							fnames{j},size(this.responses.(fnames{j})),class(this.responses.(fnames{j}))));
		end
			end
			fielddisplay(this,'numberofresponses','number of responses') 
			for i=1:numel(this.method);
		if strcmp(class(this.method(i)),'dakota_method')
			console.log(sprintf('            method%s :    ''%s''',...
						string_dim(this.method,i),this.method(i).method));
		end
			end
			for i=1:numel(this.params)
				console.log(sprintf('         params%s:  (array of method-independent parameters)',...
							string_dim(this.params,i)));
		fnames=fieldnames(this.params(i));
		maxlen=0;
		for j=1:numel(fnames)
			maxlen=max(maxlen,length(fnames{j}));
		end

			for j=1:numel(fnames)
				console.log(sprintf(['            %-' num2str(maxlen+1) 's: %s'],...
							fnames{j},any2str(this.params(i).(fnames{j}))));
		end
			end
			for i=1:numel(this.results)
				console.log(sprintf('         results%s:  (information from dakota files)',...
							string_dim(this.results,i)));
		fnames=fieldnames(this.results(i));
		maxlen=0;
		for j=1:numel(fnames)
			maxlen=max(maxlen,length(fnames{j}));
		end

			for j=1:numel(fnames)
				console.log(sprintf(['            %-' num2str(maxlen+1) 's:    [%ix%i]    ''%s'''],...
							fnames{j},size(this.results.(fnames{j})),class(this.results.(fnames{j}))));
		end
			end
			fielddisplay(this,'partition','user provided mesh partitioning, defaults to metis if not specified') 
			fielddisplay(this,'numberofpartitions','number of partitions for semi-discrete qmu') 
			fielddisplay(this,'variabledescriptors','');
		fielddisplay(this,'responsedescriptors','');
		fielddisplay(this,'method','array of dakota_method class');
		fielddisplay(this,'mass_flux_profile_directory','directory for mass flux profiles');
		fielddisplay(this,'mass_flux_profiles','list of mass_flux profiles');
		fielddisplay(this,'mass_flux_segments','');
		fielddisplay(this,'adjacency','');
		fielddisplay(this,'vertex_weight','weight applied to each mesh vertex');
		*/

	}// }}}
	this.classname= function(){// {{{
		return "qmu";
	}// }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{

			///Early return
			if (!md.qmu.isdakota) return;
			else md.checkmessage('qmu runs not supported yet!');

		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'fieldname','isdakota','format','Boolean');
			if (!this.isdakota){
				WriteData(fid,'data',0,'enum',QmuMassFluxSegmentsPresentEnum(),'format','Boolean');
			}
			else{
				WriteData(fid,'object',this,'fieldname','partition','format','DoubleMat','mattype',2);
				WriteData(fid,'object',this,'fieldname','numberofpartitions','format','Integer');
				WriteData(fid,'object',this,'fieldname','numberofresponses','format','Integer');
				WriteData(fid,'object',this,'fieldname','variabledescriptors','format','StringArray');
				WriteData(fid,'object',this,'fieldname','responsedescriptors','format','StringArray');
				if (this.mass_flux_segments.length){
					WriteData(fid,'data',this.mass_flux_segments,'enum',MassFluxSegmentsEnum(),'format','MatArray');
					flag=true; 
				}
				else flag=false; 
				WriteData(fid,'data',flag,'enum',QmuMassFluxSegmentsPresentEnum(),'format','Boolean');
			}
		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{

	this.isdakota                    = 0;
	this.variables                   = []
	this.responses                   = [];
	this.method                      = []
	this.params                      = []
	this.results                     = []
	this.partition                   = NaN;
	this.numberofpartitions          = 0;
	this.numberofresponses           = 0;
	this.variabledescriptors         = []
	this.responsedescriptors         = []
	this.mass_flux_profile_directory = NaN;
	this.mass_flux_profiles          = NaN;
	this.mass_flux_segments          = []
	this.adjacency                   = NaN;
	this.vertex_weight               = NaN;

	this.setdefaultparameters();
	//}}}
}
//RADAROVERLAY class definition
//
//   Usage:
//      radaroverlay=new radaroverlay();

function radaroverlay (){
	//methods
	this.setdefaultparameters = function(){// {{{
	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   radaroverlay parameters:'));

		fielddisplay(this,'xlim','corresponding x boundaries[m]');
		fielddisplay(this,'ylim','corresponding y boundaries [m]');
		fielddisplay(this,'outerindex','outer triangulation between mesh and bounding box');
		fielddisplay(this,'outerx','outer triangulation x coordinate between mesh and bounding box');
		fielddisplay(this,'outery','outer triangulation y coordinate between mesh and bounding box');

	}// }}}
	//properties 
	// {{{
	this.xlim   = NaN;
	this.ylim   = NaN;
	this.outerindex   = NaN;
	this.outerx   = NaN;
	this.outery   = NaN;
	this.setdefaultparameters();
	//}}}
}
//RIFTS class definition
//
//   Usage:
//      rifts=new rifts();

function rifts (){
	//methods
	this.setdefaultparameters = function(){// {{{
	}// }}}
	this.classname= function(){// {{{
		return "rifts";
	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   rifts class echo:'));
		fielddisplay(this,'riftstruct','structure containing all rift information (vertices coordinates, segments, type of melange, ...)');
		fielddisplay(this,'riftproperties','');
	}// }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{
			var numrifts;
			if (isNaN(this.riftstruct) | this.riftstruct.length==0){
				numrifts=0;
			}
			else{
				numrifts=this.riftstruct.length;
			}
			if (numrifts){
				if (!(md.mesh.domaintype() == '2Dhorizontal')){
					md.checkmessage('models with rifts are only supported in 2d for now!');
				}
				if (!IsArray(this.riftstruct)){
					md.checkmessage('rifts.riftstruct should be a structure!');
				}
				for(var i=0;i<md.mesh.segmentmarkers.length;i++){
					if (md.mesh.segmentmarkers[i]>=2){
						//We have segments with rift markers, but no rift structure!
						md.checkmessage(['model should be processed for rifts (run meshprocessrifts)!']);
						break;
					}
				}
				for (var i=0;i<numrifts;i++){
					md = checkfield(md,'fieldname',sprintf('rifts.riftstruct[%i].fill',i),'values',[WaterEnum(), AirEnum(), IceEnum(), MelangeEnum()]);
				}
			}
			else{
				if (!isNaN(this.riftstruct)) md.checkmessage('riftstruct should be NaN since numrifts is 0!');
			}
		} //}}}
		this.marshall=function(md,fid) { //{{{

			var numrifts;
			//Process rift info
			if ((this.riftstruct.length==0) | isNaN(this.riftstruct)){
				numrifts=0;
			}
			else{
				numrifts=this.riftstruct.length;
			}
			var numpairs=0;
			for (var i=0;i<numrifts;i++){
				numpairs=numpairs+this.riftstruct[i].penaltypairs.length;
			}

			// 2 for nodes + 2 for elements+ 2 for  normals + 1 for length + 1 for fill + 1 for friction + 1 for fraction + 1 for fractionincrement + 1 for state.
			data=Create2DArray(numpairs,12);
			var count=0;
			for (var i=0;i<numrifts;i++){
				numpairsforthisrift=this.riftstruct[i].penaltypairs.length;
				for(var j=0;j<numpairsforthisrift;j++){
					for(var k=0;k<7;k++)data[count+j][k]=this.riftstruct[i].penaltypairs;
					data[count+j][7]=this.riftstruct[i].fill;
					data[count+j][8]=this.riftstruct[i].friction;
					data[count+j][9]=this.riftstruct[i].fraction;
					data[count+j][10]=this.riftstruct[i].fractionincrement;
					data[count+j][11]=this.riftstruct[i].state;
					count+=numpairsforthisrift;
				}
			}
			WriteData(fid,'data',numrifts,'enum',RiftsNumriftsEnum(),'format','Integer');
			WriteData(fid,'data',data,'enum',RiftsRiftstructEnum(),'format','DoubleMat','mattype',3);
		}//}}}
		this.fix=function() { //{{{
			this.riftstruct=NullFix(this.riftstruct,NaN);
			this.riftproperties=NullFix(this.riftproperties,NaN);
		}//}}}
	//properties 
	// {{{
	this.riftstruct     = NaN;
	this.riftproperties = NaN;
	this.setdefaultparameters();
	//}}}
}
//SLR class definition
//
//   Usage:
//      slr=slr();

function slr(){
	//methods
		this.setdefaultparameters = function (){ //{{{
		
		//Convergence criterion: absolute, relative and residual
		this.reltol=NaN; //default
		this.abstol=0.001; //1 mm of sea level rise

		//maximum of non-linear iterations.
		this.maxiter=10;

		//computational flags: 
		this.rigid=1;
		this.elastic=1;
		this.rotation=1;
		
		//tidal love numbers: 
		self.tide_love_h=0.6149; //degree 2
		self.tide_love_k=0.3055; //degree 2

		//numerical discretization accuracy
		this.degacc=.01;
		
		//output default:
		this.requested_outputs=['default'];

		//transitions should be a cell array of vectors: 
		this.transitions=[];
		
		}// }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{

			//Early return
			if(ArrayAnyEqual(ArrayIsMember(SealevelriseAnalysisEnum(),analyses),0))return;
			
			md = checkfield(md,'fieldname','slr.deltathickness','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			md = checkfield(md,'fieldname','slr.sealevel','NaN',1,'Inf',1,'size',[md.mesh.numberofvertices, 1]);
			md = checkfield(md,'fieldname','slr.love_h','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','slr.love_k','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','slr.tide_love_h','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','slr.tide_love_k','NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','slr.reltol','size',[1, 1]);
			md = checkfield(md,'fieldname','slr.abstol','size',[1, 1]);
			md = checkfield(md,'fieldname','slr.maxiter','size',[1, 1],'>=',1);
			md = checkfield(md,'fieldname','slr.degacc','size',[1, 1],'>=',1e-10);
			md = checkfield(md,'fieldname','slr.requested_outputs','stringrow',1);
			
			//check that love numbers are provided at the same level of accuracy: 
			if (this.love_h.length != this.love_k.length){
				throw Error('slr error message: love numbers should be provided at the same level of accuracy');
			}

		} // }}}
		this.defaultoutputs = function(md){ // {{{
			return ['Sealevel'];
		}//}}}
	this.classname= function(){// {{{
		return "slr";
	}// }}}
		this.disp= function(){// {{{
			
		console.log(sprintf('   Sealevelrise solution parameters:'));

		fielddisplay(this,'deltathickness','thickness change (main loading of the slr solution core [m]');
		fielddisplay(this,'sealevel','current sea level (prior to computation) [m]');
		fielddisplay(this,'reltol','sea level rise relative convergence criterion, (default, NaN: not applied)');
		fielddisplay(this,'abstol','sea level rise absolute convergence criterion, NaN: not applied');
		fielddisplay(this,'maxiter','maximum number of nonlinear iterations');
		fielddisplay(this,'love_h','love load number for radial displacement');
		fielddisplay(this,'love_k','love load number for gravitational potential perturbation');
		fielddisplay(this,'tide_love_h','tidal love number (degree 2)');
		fielddisplay(this,'tide_love_k','tidal love number (degree 2)');
		fielddisplay(this,'rigid','rigid earth graviational potential perturbation');
		fielddisplay(this,'elastic','elastic earth graviational potential perturbation');
		fielddisplay(this,'rotation','rotational earth potential perturbation');
		fielddisplay(this,'degacc',"accuracy (default .01 deg) for numerical discretization of the Green's functions");
		fielddisplay(this,'transitions','indices into parts of the mesh that will be icecaps');
		fielddisplay(this,'requested_outputs','additional outputs requested');
		} //}}}
			this.marshall=function(md,fid) { //{{{

			WriteData(fid,'object',this,'class','sealevelrise','fieldname','deltathickness','format','DoubleMat','mattype',1);
			WriteData(fid,'data',this.sealevel,'mattype',1,'format','DoubleMat','enum',SealevelEnum(),'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','reltol','format','Double');
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','abstol','format','Double');
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','maxiter','format','Integer');
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','love_h','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','love_k','format','DoubleMat','mattype',1);
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','tide_love_h','format','Double');
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','tide_love_k','format','Double');
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','rigid','format','Boolean');
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','elastic','format','Boolean');
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','rotation','format','Boolean');
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','degacc','format','Double');
			WriteData(fid,'object',this,'class','sealevelrise','fieldname','transitions','format','MatArray');
			
			//process requested outputs
			var outputs = this.requested_outputs;
			for (var i=0;i<outputs.length;i++){
				if (outputs[i] == 'default') {
					outputs.splice(i,1);
					var newoutputs=this.defaultoutputs(md);
					for (var j=0;j<newoutputs.length;j++) outputs.push(newoutputs[j]);
				}
			}
			WriteData(fid,'data',outputs,'enum',SealevelriseRequestedOutputsEnum(),'format','StringArray');
		}//}}}
		this.fix=function() { //{{{
			this.deltathickness=NullFix(this.deltathickness,NaN);
			this.sealevel=NullFix(this.sealevel,NaN);
			this.maxiter=NullFix(this.maxiter,NaN);
			this.reltol=NullFix(this.reltol,NaN);
			this.abstol=NullFix(this.abstol,NaN);
			this.love_h=NullFix(this.love_h,NaN);
			this.love_k=NullFix(this.love_k,NaN);
			this.tide_love_h=NullFix(this.tide_love_h,NaN);
			this.tide_love_k=NullFix(this.tide_love_k,NaN);
			this.rigid=NullFix(this.rigid,NaN);
			this.elastic=NullFix(this.elastic,NaN);
			this.rotation=NullFix(this.rotation,NaN);
			this.degacc=NullFix(this.degacc,NaN);
		}//}}}
	//properties
	//{{{
	this.deltathickness = NaN;
	this.sealevel       = NaN; 
	this.maxiter        = 0;
	this.reltol         = 0;
	this.abstol         = 0;
	this.love_h         = 0; //provided by PREM model
	this.love_k         = 0; //idam
	this.tide_love_h    = 0; 
	this.tide_love_k    = 0; 
	this.rigid          = 0;
	this.elastic        = 0;
	this.rotation       = 0;
	this.degacc         = 0;
	this.requested_outputs = [];
	this.transitions    = [];
	this.setdefaultparameters();
	//}}}
}
//SETTINGS class definition
//
//   Usage:
//      settings=new settings();

function settings (){
	//methods
	this.setdefaultparameters = function(){// {{{
		//are we short in memory ? (0 faster but requires more memory)
		this.lowmem=0;

		//i/o:
		this.io_gather=1;

		//results frequency by default every step
		this.output_frequency=1;

		//checkpoints frequency, by default never: 
		this.recording_frequency=0;

		//this option can be activated to load automatically the results
		//onto the model after a parallel run by waiting for the lock file
		//N minutes that is generated once the solution has converged
		//0 to deactivate
		this.waitonlock=Infinity;

		//upload options: 
		upload_port         = 0;

	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   settings class echo:'));
		
		fielddisplay(this,'results_on_nodes','results are output for all the nodes of each element');
		fielddisplay(this,'io_gather','I/O gathering strategy for result outputs (default 1)');
		fielddisplay(this,'lowmem','is the memory limited ? (0 or 1)');
		fielddisplay(this,'output_frequency','frequency at which results are saved in all solutions with multiple time_steps');
		fielddisplay(this,'recording_frequency','frequency at which the runs are being recorded, allowing for a restart');
		fielddisplay(this,'waitonlock','maximum number of minutes to wait for batch results (NaN to deactivate)');
		fielddisplay(this,'upload_server','server hostname where model should be uploaded');
		fielddisplay(this,'upload_path','path on server where model should be uploaded');
		fielddisplay(this,'upload_login','server login');
		fielddisplay(this,'upload_port','port login (default is 0)');
		fielddisplay(this,'upload_filename','unique id generated when uploading the file to server');


	}// }}}
	this.classname= function(){// {{{
		return "settings";

	}// }}}
		this.checkconsistency = function(md,solution,analyses) { // {{{

			checkfield(md,'fieldname','settings.results_on_nodes','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','settings.io_gather','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','settings.lowmem','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','settings.output_frequency','numel',[1],'>=',1);
			checkfield(md,'fieldname','settings.recording_frequency','numel',[1],'>=',0);
			checkfield(md,'fieldname','settings.waitonlock','numel',[1]);
		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'fieldname','results_on_nodes','format','Boolean');
			WriteData(fid,'object',this,'fieldname','io_gather','format','Boolean');
			WriteData(fid,'object',this,'fieldname','lowmem','format','Boolean');
			WriteData(fid,'object',this,'fieldname','output_frequency','format','Integer');
			WriteData(fid,'object',this,'fieldname','recording_frequency','format','Integer');
			if (this.waitonlock>0) WriteData(fid,'enum',SettingsWaitonlockEnum(),'data',true,'format','Boolean');
			else WriteData(fid,'enum',SettingsWaitonlockEnum(),'data',false,'format','Boolean');
		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{
	this.results_on_nodes    = 0;
	this.io_gather           = 0;
	this.lowmem              = 0;
	this.output_frequency    = 0;
	this.recording_frequency   = 0;
	this.waitonlock          = 0;
	this.upload_server       = '';
	this.upload_path         = '';
	this.upload_login        = '';
	this.upload_port         = 0;
	this.upload_filename     = '';
	this.setdefaultparameters();
	//}}}
}
//SMBforcing Class definition
//
//   Usage:
//      SMB=SMBforcing();

function SMBforcing(){
	//methods
	this.setdefaultparameters = function(){// {{{
		this.requested_outputs=['default'];
	} // }}}
	this.disp = function(){ // {{{
		console.log(sprintf('   surface forcings parameters:'));
		fielddisplay(this,'mass_balance','surface mass balance [m/yr ice eq]');
		fielddisplay(this,'requested_outputs','additional outputs requested');
	} // }}}
	this.defaultoutputs = function(){ // {{{
		return '';
	}//}}}
this.classname = function(){ // {{{
	return "SMBforcing";
} // }}}
		this.initialize = function(md) {// {{{

			if (isNaN(self.mass_balance)){
				self.mass_balance=NewArrayFill(md.mesh.numberofvertices,0);
				console.log('      no smb.mass_balance specified: values set as zero');
			}

		} // }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{

			if(ArrayAnyEqual(ArrayIsMember(MasstransportAnalysisEnum(),analyses),1)){
				checkfield(md,'fieldname','smb.mass_balance','timeseries',1,'NaN',1,'Inf',1);
			}
			if(ArrayAnyEqual(ArrayIsMember(BalancethicknessAnalysisEnum(),analyses),1)){
				checkfield(md,'fieldname','smb.mass_balance','size',[md.mesh.numberofvertices,1],'NaN',1,'Inf',1);
			}
			checkfield(md,'fieldname','smb.requested_outputs','stringrow',1);

		} // }}}
		this.marshall=function(md,fid) { //{{{

			var yts=365.0*24.0*3600.0;

			WriteData(fid,'enum',SmbEnum(),'data',SMBforcingEnum(),'format','Integer');
			WriteData(fid,'object',this,'class','smb','fieldname','mass_balance','format','DoubleMat','mattype',1,'scale',1./yts,'timeserieslength',md.mesh.numberofvertices+1);
			
			//process requested outputs
			var outputs = this.requested_outputs.slice();
			for (var i=0;i<outputs.length;i++){
				if (outputs[i] == 'default') {
					outputs.splice(i,1);
					var newoutputs=this.defaultoutputs(md);
					for (var j=0;j<newoutputs.length;j++) outputs.push(newoutputs[j]);
				}
			}
			WriteData(fid,'data',outputs,'enum',SmbRequestedOutputsEnum(),'format','StringArray');

		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	this.mass_balance = NaN;
	this.requested_outputs      = [];
	this.setdefaultparameters();
}
//STEADYSTATE class definition
//
//   Usage:
//      steadystate=new steadystate();

function steadystate (){
	//methods
	this.setdefaultparameters = function(){// {{{

		//maximum of steady state iterations
		this.maxiter=100;

		//Relative tolerance for the steadystate convertgence
		this.reltol=0.01;

		//default output
		this.requested_outputs=['default'];


	}// }}}
	this.disp= function(){// {{{

		console.log(sprintf('   steadystate solution parameters:'));

		fielddisplay(this,'reltol','relative tolerance criterion');
		fielddisplay(this,'maxiter','maximum number of iterations');
		fielddisplay(this,'requested_outputs','additional requested outputs');

	}// }}}
	this.classname= function(){// {{{
		return "steadystate";

	}// }}}
	this.checkconsistency = function(md,solution,analyses) {// {{{

		//Early return
		if (solution!=SteadystateSolutionEnum()) return;

		if (md.timestepping.time_step!=0){
			md.checkmessage('for a steadystate computation, timestepping.time_step must be zero.');
		}
		checkfield(md,'fieldname','steadystate.requested_outputs','stringrow',1);

		if (isNaN(md.stressbalance.reltol)){
			md.checkmessage('for a steadystate computation, stressbalance.reltol (relative convergence criterion) must be defined!');
		}
	} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'fieldname','reltol','format','Double');
			WriteData(fid,'object',this,'fieldname','maxiter','format','Integer');

			//process requested outputs
			var outputs = this.requested_outputs;
			for (var i=0;i<outputs.length;i++){
				if (outputs[i] == 'default') {
					outputs.splice(i,1);
					var newoutputs=this.defaultoutputs(md);
					for (var j=0;j<newoutputs.length;j++) outputs.push(newoutputs[j]);
				}
			}
			WriteData(fid,'data',outputs,'enum',SteadystateRequestedOutputsEnum(),'format','StringArray');
		}//}}}
		this.defaultoutputs = function(md) { //{{{

			var list=[];

			for (var i=0;i<md.stressbalance.defaultoutputs(md).length;i++)list.push(md.stressbalance.defaultoutputs(md)[i]);
			for (var i=0;i<md.thermal.defaultoutputs(md).length;i++)list.push(md.thermal.defaultoutputs(md)[i]);

			return list;

		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{

	this.reltol            = 0;
	this.maxiter           = 0;
	this.requested_outputs = [];

	this.setdefaultparameters();
	//}}}
}
//STRESSBALANCE class definition
//
//   Usage:
//      stressbalance=new stressbalance();

function stressbalance (){
	//methods
	this.setdefaultparameters = function(){// {{{

		//maximum of non-linear iterations.
		this.maxiter=100;

		//Convergence criterion: absolute, relative and residual
		this.restol=Math.pow(10,-4); 
		this.reltol=0.01;
		this.abstol=10;

		this.FSreconditioning=Math.pow(10,13);
		this.shelf_dampening=0;

		//Penalty factor applied kappa=max(stiffness matrix)*10^penalty_factor
		this.penalty_factor=3;

		//coefficient to update the viscosity between each iteration of
		//a stressbalance according to the following formula
		//viscosity(n)=viscosity(n)+viscosity_overshoot(viscosity(n)-viscosity(n-1))
		this.viscosity_overshoot=0;

		//Stop the iterations of rift if below a threshold
		this.rift_penalty_threshold=0;

		//in some solutions, it might be needed to stop a run when only
		//a few constraints remain unstable. For thermal computation, this
		//parameter is often used.
		this.rift_penalty_lock=10;

		//output default:
		this.requested_outputs=['default'];

	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   StressBalance solution parameters:'));

		console.log(sprintf('\n      %s','Convergence criteria:'));
		fielddisplay(this,'restol','mechanical equilibrium residual convergence criterion');
		fielddisplay(this,'reltol','velocity relative convergence criterion, NaN: not applied');
		fielddisplay(this,'abstol','velocity absolute convergence criterion, NaN: not applied');
		fielddisplay(this,'isnewton',"0: Picard's fixed point, 1: Newton's method, 2: hybrid");
		fielddisplay(this,'maxiter','maximum number of nonlinear iterations');
		fielddisplay(this,'viscosity_overshoot','over-shooting constant new=new+C*(new-old)');

		console.log(sprintf('\n      %s','boundary conditions:'));
		fielddisplay(this,'spcvx','x-axis velocity constraint (NaN means no constraint) [m/yr]');
		fielddisplay(this,'spcvy','y-axis velocity constraint (NaN means no constraint) [m/yr]');
		fielddisplay(this,'spcvz','z-axis velocity constraint (NaN means no constraint) [m/yr]');

		console.log(sprintf('\n      %s','Rift options:'));
		fielddisplay(this,'rift_penalty_threshold','threshold for instability of mechanical constraints');
		fielddisplay(this,'rift_penalty_lock','number of iterations before rift penalties are locked');

		console.log(sprintf('\n      %s','Penalty options:'));
		fielddisplay(this,'penalty_factor','offset used by penalties: penalty = Kmax*10^offset');
		fielddisplay(this,'vertex_pairing','pairs of vertices that are penalized');

		console.log(sprintf('\n      %s','Other:'));
		fielddisplay(this,'shelf_dampening','use dampening for floating ice ? Only for FS model');
		fielddisplay(this,'FSreconditioning','multiplier for incompressibility equation. Only for FS model');
		fielddisplay(this,'referential','local referential');
		fielddisplay(this,'loadingforce','loading force applied on each point [N/m^3]');
		fielddisplay(this,'requested_outputs','additional outputs requested');



	}// }}}
	this.classname= function(){// {{{
		return "stressbalance";
	}// }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{

			//Early return
			if(ArrayAnyEqual(ArrayIsMember(StressbalanceAnalysisEnum(),analyses),0))return;

			checkfield(md,'fieldname','stressbalance.spcvx','Inf',1,'timeseries',1);
			checkfield(md,'fieldname','stressbalance.spcvy','Inf',1,'timeseries',1);
			checkfield(md,'fieldname','stressbalance.spcvz','Inf',1,'timeseries',1);
			checkfield(md,'fieldname','stressbalance.restol','size',[1, 1],'>',0,'NaN',1,'Inf',1);
			checkfield(md,'fieldname','stressbalance.reltol','size',[1, 1]);
			checkfield(md,'fieldname','stressbalance.abstol','size',[1, 1]);
			checkfield(md,'fieldname','stressbalance.isnewton','numel',[1],'values',[0, 1, 2]);
			checkfield(md,'fieldname','stressbalance.FSreconditioning','size',[1, 1],'NaN',1,'Inf',1);
			checkfield(md,'fieldname','stressbalance.viscosity_overshoot','size',[1, 1],'NaN',1,'Inf',1);
			checkfield(md,'fieldname','stressbalance.maxiter','size',[1, 1],'>=',1);
			checkfield(md,'fieldname','stressbalance.referential','size',[md.mesh.numberofvertices, 6]);
			checkfield(md,'fieldname','stressbalance.loadingforce','size',[md.mesh.numberofvertices, 3]);
			checkfield(md,'fieldname','stressbalance.requested_outputs','stringrow',1);

			//singular solution
			if(!ArrayAnyNaN(md.stressbalance.spcvx) | !ArrayAnyNaN(md.stressbalance.spcvy) |  !ArrayAnyAboveStrict(md.mask.groundedice_levelset,0)){
				md = checkmessage(md,'model is not well posed (singular). You need at least one node with fixed velocity!');
				console.log(sprintf('\n !!! Warning: no spc applied, model might not be well posed if no basal friction is applied, check for solution crash\n'));
			}
			//CHECK THAT EACH LINES CONTAINS ONLY NAN VALUES OR NO NAN VALUES
			for(var i=0;i<md.stressbalance.referential.length;i++){
				var sum=0;
				for(j=0;j<md.stressbalance.referential[0].length;j++)sum+=md.stressbalance.referential[i][j];
				if (sum!=0 & sum!=6){
					md = checkmessage(md,'Each line of stressbalance.referential should contain either only NaN values or no NaN values');
					break;
				}
			}
			//CHECK THAT THE TWO VECTORS PROVIDED ARE ORTHOGONAL
			for(var i=0;i<md.stressbalance.referential.length;i++){
				var sum=0;
				for(j=0;j<md.stressbalance.referential[0].length;j++)sum+=md.stressbalance.referential[i][j];
				if(sum==0){
					var dot=0;
					for(j=0;j<3;j++)dot+=md.stressbalance.referential[i][j]*md.stressbalance.referential[i][j+3];
					dot=Math.abs(dot);
					if(dot>Math.pow(10,-18)){
						md.checkmessage('Vectors in stressbalance.referential (columns 1 to 3 and 4 to 6) must be orthogonal');
						break;
					}
				}
			}
			//CHECK THAT NO rotation specified for FS Grounded ice at base
			if (md.mesh.domaintype() == '3D' & md.flowequation.isFS){
				for(var i=0;i<md.mask.groundedice_levelset.length;i++){
					if(md.mask.groundedice_levelset[i]>0 & md.mesh.vertexonbase[i]){
						if(!ArrayIsNan(md.stressbalance.referential[i])){
							md.checkmessage('no referential should be specified for basal vertices of grounded ice');
							break;
						}
					}
				}
				checkfield(md,'fieldname','stressbalance.FSreconditioning','>',0);
			}
		} // }}}
		this.marshall=function(md,fid) { //{{{

			WriteData(fid,'object',this,'class','stressbalance','fieldname','vertex_pairing','format','DoubleMat','mattype',3);

			var yts=365.0*24.0*3600.0;

			WriteData(fid,'object',this,'class','stressbalance','fieldname','spcvx','format','DoubleMat','mattype',1,'scale',1./yts,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',this,'class','stressbalance','fieldname','spcvy','format','DoubleMat','mattype',1,'scale',1./yts,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',this,'class','stressbalance','fieldname','spcvz','format','DoubleMat','mattype',1,'scale',1./yts,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',this,'class','stressbalance','fieldname','restol','format','Double');
			WriteData(fid,'object',this,'class','stressbalance','fieldname','reltol','format','Double');
			WriteData(fid,'object',this,'class','stressbalance','fieldname','abstol','format','Double','scale',1./yts);
			WriteData(fid,'object',this,'class','stressbalance','fieldname','isnewton','format','Integer');
			WriteData(fid,'object',this,'class','stressbalance','fieldname','FSreconditioning','format','Double');
			WriteData(fid,'object',this,'class','stressbalance','fieldname','viscosity_overshoot','format','Double');
			WriteData(fid,'object',this,'class','stressbalance','fieldname','maxiter','format','Integer');
			WriteData(fid,'object',this,'class','stressbalance','fieldname','shelf_dampening','format','Integer');
			WriteData(fid,'object',this,'class','stressbalance','fieldname','penalty_factor','format','Double');
			WriteData(fid,'object',this,'class','stressbalance','fieldname','rift_penalty_lock','format','Integer');
			WriteData(fid,'object',this,'class','stressbalance','fieldname','rift_penalty_threshold','format','Integer');
			WriteData(fid,'object',this,'class','stressbalance','fieldname','referential','format','DoubleMat','mattype',1);

			var lx=NewArrayFill(this.loadingforce.length,0); for(var i=0;i<lx.length;i++)lx[i]=this.loadingforce[i][0];
			var ly=NewArrayFill(this.loadingforce.length,0); for(var i=0;i<lx.length;i++)ly[i]=this.loadingforce[i][1];
			var lz=NewArrayFill(this.loadingforce.length,0); for(var i=0;i<lx.length;i++)lz[i]=this.loadingforce[i][2];

			WriteData(fid,'data',lx,'format','DoubleMat','mattype',1,'enum',LoadingforceXEnum());
			WriteData(fid,'data',ly,'format','DoubleMat','mattype',1,'enum',LoadingforceYEnum());
			WriteData(fid,'data',lz,'format','DoubleMat','mattype',1,'enum',LoadingforceZEnum());

			//process requested outputs
			var outputs = this.requested_outputs;
			for (var i=0;i<outputs.length;i++){
				if (outputs[i] == 'default') {
					outputs.splice(i,1);
					var newoutputs=this.defaultoutputs(md);
					for (var j=0;j<newoutputs.length;j++) outputs.push(newoutputs[j]);
				}
			}
			WriteData(fid,'data',outputs,'enum',StressbalanceRequestedOutputsEnum(),'format','StringArray');
		}//}}}
		this.defaultoutputs = function(md){ // {{{

			var list;
			if (md.mesh.dimension() == 3) list = ['Vx','Vy','Vz','Vel','Pressure'];
			else if (md.mesh.dimension()==2) list = ['Vx','Vy','Vel','Pressure'];
			else throw Error('mesh type not supported yet');
			return list;

		}//}}}
		this.fix=function() { //{{{
			this.abstol=NullFix(this.abstol,NaN);
			this.rift_penalty_lock=NullFix(this.rift_penalty_lock,NaN);
			this.referential=NullFix(this.referential,NaN);
			this.loadingforce=NullFix(this.loadingforce,NaN);
			this.spcvx=NullFix(this.spcvx,NaN);
			this.spcvy=NullFix(this.spcvy,NaN);
			this.spcvz=NullFix(this.spcvz,NaN);
			if(this.vertex_pairing=[])this.vertex_pairing=NaN;
		}//}}}
	//properties 
	// {{{
	this.spcvx                  = NaN;
	this.spcvy                  = NaN;
	this.spcvz                  = NaN;
	this.restol                 = 0;
	this.reltol                 = 0;
	this.abstol                 = 0;
	this.isnewton               = 0;
	this.FSreconditioning       = 0;
	this.viscosity_overshoot    = 0;
	this.maxiter                = 0;
	this.shelf_dampening        = 0;
	this.vertex_pairing         = NaN;
	this.penalty_factor         = NaN;
	this.rift_penalty_lock      = NaN;
	this.rift_penalty_threshold = 0;
	this.referential            = NaN;
	this.loadingforce           = NaN;
	this.requested_outputs      = []

	this.setdefaultparameters();
	//}}}
}
//TEMPLATE class definition
//
//   Usage:
//      template=new template();

function template (){
	//methods
	this.setdefaultparameters = function(){// {{{
	}// }}}
	this.disp= function(){// {{{
		console.log(sprintf('   template class echo:'));

	}// }}}
	//properties 
	// {{{
	this.setdefaultparameters();
	//}}}
}
//THERMAL class definition
//
//   Usage:
//      thermal=new thermal();

function thermal (){
	//methods
	this.setdefaultparameters = function(){// {{{

		//Number of unstable constraints acceptable
		this.penalty_threshold=0;

		//Type of stabilization used
		this.stabilization=1;

		//Relative tolerance for the enthalpy convergence
		this.reltol=0.01;

		//Maximum number of iterations
		this.maxiter=100;

		//factor used to compute the values of the penalties: kappa=max(stiffness matrix)*10^penalty_factor
		this.penalty_factor=3;

		//Should we use cold ice (default) or enthalpy formulation
		this.isenthalpy=0;

		//will basal boundary conditions be set dynamically
		this.isdynamicbasalspc=0;

		//default output
		this.requested_outputs=['default'];

	}// }}}
	this.disp= function(){// {{{

		console.log(sprintf('   Thermal solution parameters:'));

		fielddisplay(this,'spctemperature','temperature constraints (NaN means no constraint) [K]');
		fielddisplay(this,'stabilization','0: no, 1: artificial_diffusivity, 2: SUPG');
		fielddisplay(this,'reltol','relative tolerance convergence criterion for enthalpy');
		fielddisplay(this,'maxiter','maximum number of non linear iterations');
		fielddisplay(this,'penalty_lock','stabilize unstable thermal constraints that keep zigzagging after n iteration (default is 0, no stabilization)');
		fielddisplay(this,'penalty_threshold','threshold to declare convergence of thermal solution (default is 0)');
		fielddisplay(this,'penalty_factor','scaling exponent (default is 3)');
		fielddisplay(this,'isenthalpy','use an enthalpy formulation to include temperate ice (default is 0)');
		fielddisplay(this,'isdynamicbasalspc','enable dynamic setting of basal forcing. required for enthalpy formulation (default is 0)');
		fielddisplay(this,'requested_outputs','additional outputs requested');

	}// }}}
	this.classname= function(){// {{{
		return "thermal";
	}// }}}
	this.checkconsistency = function(md,solution,analyses){ // {{{

		//Early return
		if(!ArrayAnyEqual(ArrayIsMember(ThermalAnalysisEnum(),analyses),1) & !ArrayAnyEqual(ArrayIsMember(EnthalpyAnalysisEnum(),analyses),1)  | (solution == TransientSolutionEnum() & md.trans.isthermal==0)) return;

		checkfield(md,'fieldname','thermal.stabilization','numel',[1],'values',[0 ,1, 2]);
		checkfield(md,'fieldname','thermal.spctemperature','Inf',1,'timeseries',1);
		if(ArrayAnyEqual(ArrayIsMember(EnthalpyAnalysisEnum(),analyses),1) & md.thermal.isenthalpy & md.mesh.dimension() == 3){
			
			for(var i=0;i<md.mesh.numberofvertices;i++){
				for(var j=0;j<md.thermal.spctemperature[0].length;j++){
					if (!isNaN(md.thermal.spctemperature[i][j])){
						var rep=md.geometry.surface[i]-md.mesh.z[i];
						if (md.thermal.spctemperature[i][j] <= md.materials.melting-md.materials.beta*md.materials.rho_ice*md.constants.g*rep+Math.pow(10,-5)){

							md.checkmessage('spctemperature should be less or equal than the adjusted melting point');
							break;
						}
					}
				}
			}
			checkfield(md,'fieldname','thermal.isenthalpy','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','thermal.isdynamicbasalspc','numel', [1],'values',[0, 1]);
			if(md.thermal.isenthalpy){
				if (isNan(md.stressbalance.reltol)){
					md.checkmessage('for a steadystate computation, thermal.reltol (relative convergence criterion) must be defined!');
				}
			}
			checkfield(md,'fieldname','thermal.reltol','>',0.,'message','reltol must be larger than zero');
		}
		checkfield(md,'fieldname','thermal.requested_outputs','stringrow',1);
	} // }}} 
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'fieldname','spctemperature','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',this,'fieldname','penalty_threshold','format','Integer');
			WriteData(fid,'object',this,'fieldname','stabilization','format','Integer');
			WriteData(fid,'object',this,'fieldname','reltol','format','Double');
			WriteData(fid,'object',this,'fieldname','maxiter','format','Integer');
			WriteData(fid,'object',this,'fieldname','penalty_lock','format','Integer');
			WriteData(fid,'object',this,'fieldname','penalty_factor','format','Double');
			WriteData(fid,'object',this,'fieldname','isenthalpy','format','Boolean');
			WriteData(fid,'object',this,'fieldname','isdynamicbasalspc','format','Boolean');

			//process requested outputs
			var outputs = this.requested_outputs;
			for (var i=0;i<outputs.length;i++){
				if (outputs[i] == 'default') {
					outputs.splice(i,1);
					var newoutputs=this.defaultoutputs(md);
					for (var j=0;j<newoutputs.length;j++) outputs.push(newoutputs[j]);
				}
			}
			WriteData(fid,'data',outputs,'enum',ThermalRequestedOutputsEnum(),'format','StringArray');
        	}//}}}
		this.defaultoutputs = function(md) { //{{{

			if (this.isenthalpy) return ['Enthalpy','Temperature','Waterfraction','Watercolumn','BasalforcingsGroundediceMeltingRate'];
			else return ['Temperature','BasalforcingsGroundediceMeltingRate'];
		}//}}}
		this.fix=function() { //{{{
			this.spctemperature=NullFix(this.spctemperature,NaN);
		}//}}}
	//properties 
	// {{{

	this.spctemperature    = NaN;
	this.penalty_threshold = 0;
	this.stabilization     = 0;
	this.reltol				= 0;
	this.maxiter           = 0;
	this.penalty_lock      = 0;
	this.penalty_factor    = 0;
	this.isenthalpy        = 0;
	this.isdynamicbasalspc = 0;
	this.requested_outputs = [];

	this.setdefaultparameters();
	//}}}
}
//TIMESTEPPING class definition
//
//   Usage:
//      timestepping=new timestepping();

function timestepping (){
	//methods
	this.setdefaultparameters = function(){// {{{
		//time between 2 time steps
		this.time_step=1./2.;

		//final time
		this.final_time=10.*this.time_step;

		//time adaptation? 
		this.time_adapt=0;
		this.cfl_coefficient=0.5;

		//should we interpolate forcings between timesteps?
		this.interp_forcings=1;
	}// }}}
	this.disp= function(){// {{{

		var unit;
		console.log(sprintf('   timestepping parameters:'));
		unit = 'yr';
		fielddisplay(this,'start_time','simulation starting time ['+ unit + ']');
		fielddisplay(this,'final_time','final time to stop the simulation ['+ unit + ']');
		fielddisplay(this,'time_step','length of time steps [' +unit+ ']');
		fielddisplay(this,'time_adapt','use cfl condition to define time step ? (0 or 1) ');
		fielddisplay(this,'cfl_coefficient','coefficient applied to cfl condition');
		fielddisplay(this,'interp_forcings','interpolate in time between requested forcing values ? (0 or 1)');

	}// }}}
	this.classname= function(){// {{{
		return "timestepping";

	}// }}}
		this.checkconsistency = function(md,solution,analyses) { //{{{
			
			checkfield(md,'fieldname','timestepping.start_time','numel',[1],'NaN',1,'Inf',1);
			checkfield(md,'fieldname','timestepping.final_time','numel',[1],'NaN',1,'Inf',1);
			checkfield(md,'fieldname','timestepping.time_step','numel',[1],'>=',0,'NaN',1,'Inf',1);
			checkfield(md,'fieldname','timestepping.time_adapt','numel',[1],'values',[0,1]);
			checkfield(md,'fieldname','timestepping.cfl_coefficient','numel',[1],'>',0,'<=',1);
			checkfield(md,'fieldname','timestepping.interp_forcings','numel',[1],'values',[0,1]);
			if (this.final_time-this.start_time<0){
				md.checkmessage('timestepping.final_time should be larger than timestepping.start_time');
			}
		} // }}}
		this.marshall=function(md,fid) { //{{{

			var scale;
			scale = 365.0*24.0*3600.0;
			
			WriteData(fid,'object',this,'fieldname','start_time','format','Double','scale',scale);
			WriteData(fid,'object',this,'fieldname','final_time','format','Double','scale',scale);
			WriteData(fid,'object',this,'fieldname','time_step','format','Double','scale',scale);
			WriteData(fid,'object',this,'fieldname','time_adapt','format','Boolean');
			WriteData(fid,'object',this,'fieldname','cfl_coefficient','format','Double');
			WriteData(fid,'object',this,'fieldname','interp_forcings','format','Boolean');

		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{
	this.start_time      = 0.;
	this.final_time      = 0.;
	this.time_step       = 0.;
	this.time_adapt      = 0;
	this.cfl_coefficient = 0.;
	this.interp_forcings = 1;

	this.setdefaultparameters();
	//}}}
}
//TOOLKITS class definition
//
//   Usage:
//      toolkits=new toolkits();

function toolkits (){
	//methods
	this.setdefaultparameters = function(){// {{{

		//default toolkits: 
		if (IssmConfig('_HAVE_PETSC_')){
			//MUMPS is the default toolkits
			if (IssmConfig('_HAVE_MUMPS_')){
				this.DefaultAnalysis           = mumpsoptions();
			}
			else{
				this.DefaultAnalysis           = iluasmoptions(); 
			}
		}
		else{
			if (IssmConfig('_HAVE_MUMPS_')){
				this.DefaultAnalysis           = issmmumpssolver(); 
			}
			else if (IssmConfig('_HAVE_GSL_')){
				this.DefaultAnalysis           = issmgslsolver(); 
			}
			else{
				console.warn('toolkits setdefaultparameters message: need at least Mumps or Gsl to define an issm solver type, no default solver assigned');
			}
		}
	}// }}}
	this.disp = function(){// {{{
		console.log(sprintf('List of toolkits options per analysis:\n'));
		for(var prop in this){
			if(typeof this[prop] == 'object'){
				console.log(prop+ ':',this[prop]);
			}
		}
	}// }}}
	this.checkconsistency = function (md,solution,analyses) { // {{{
		for(var prop in this){
			if(typeof this[prop] == 'object'){
				if (this[prop] == ''){
					md.checkmessage(sprintf("md.toolkits.%s is empty",prop));
				}
			}
		}
	} // }}}
		 this.ToolkitsFile = function(filename) { //{{{
		 //TOOLKITSFILE - build toolkits file (in string format)
		 //
		 //   Build a Petsc compatible options string, from the toolkits model field  + return options string. 
		 //   This file string will also be used when the toolkit used is 'issm' instead of 'petsc'
		 //
		 //   Usage:     var toolkitsstring = toolkits.ToolkitsFile();

			 var string = '';

			 //write header
			 string += sprintf('%s%s%s\n','\%Toolkits options file: ',filename,' written from Javascript toolkits array');

			 //start writing options
			 for (var analysis in this){
				 var options;
				 
				 if(typeof this[analysis] == 'object') options=this[analysis]; else continue;

				 //first write analysis:
				 string += sprintf('\n+%s\n',analysis); //append a + to recognize it's an analysis enum

				 //now, write options
			
				 for(var optionname in options){
					 var optionvalue=options[optionname];

					 if (optionvalue.length==0){
						 //this option has only one argument
						 string+=sprintf('-%s\n',optionname);
					 }
					 else{
						 //option with value. value can be string or scalar
						 if (typeof optionvalue == 'number'){
							 string+=sprintf('-%s %g\n',optionname,optionvalue);
						 }
						 else if (typeof optionvalue == 'string'){
							 string+=sprintf('-%s %s\n',optionname,optionvalue);
						 }
						 else throw Error(sprintf("ToolkitsFile error: option '%s' is not well formatted",optionname));
					 }
				 }
			 }
			 return string;
		 } //}}}
	//properties 
	// {{{
	this.DefaultAnalysis           = [];
	//The other properties are dynamic
	this.setdefaultparameters();
	//}}}
}
//LEVELSET class definition
//
//   Usage:
//      levelset=new levelset();

function levelset (){
	//methods
	this.setdefaultparameters = function(){// {{{

		//stabilization = 2 by default
		this.stabilization		= 2;
		this.reinit_frequency	= NaN;
	
	}// }}}
	this.disp= function(){// {{{

		console.log(sprintf('   Level-set parameters:'));
		fielddisplay(this,'stabilization','0: no, 1: artificial_diffusivity, 2: streamline upwinding');
		fielddisplay(this,'spclevelset','Levelset constraints (NaN means no constraint)');
		fielddisplay(this,'reinit_frequency','Amount of time steps after which the levelset function in re-initialized (NaN: no re-initialization).');

	}// }}}
	this.classname= function(){// {{{
		return "levelset";
	}// }}}
	this.checkconsistency = function(md,solution,analyses) { // {{{
		//Early return
		if (solution!=TransientSolutionEnum() | md.trans.ismovingfront==0) return;

		checkfield(md,'fieldname','levelset.spclevelset','Inf',1,'timeseries',1);
		checkfield(md,'fieldname','levelset.stabilization','values',[0,1,2]);
	} //}}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'class','levelset','fieldname','stabilization','format','Integer');
			WriteData(fid,'enum',SpclevelsetEnum(),'data',this.spclevelset,'format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',this,'class','levelset','fieldname','reinit_frequency','format','Integer');

		}//}}}
		this.fix=function() { //{{{
			this.spclevelset=NullFix(this.spclevelset,NaN);
		}//}}}
	//properties 
	// {{{

	this.stabilization		= 0;
	this.spclevelset			= NaN;
	this.reinit_frequency	= NaN;

	this.setdefaultparameters();
	//}}}
}
//TRANS class definition
//
//   Usage:
//      trans =new trans();

function trans (){
	//methods
	this.setdefaultparameters = function(){// {{{

		//full analysis: Stressbalance, Masstransport and Thermal but no groundingline migration for now
		this.issmb           = 1;
		this.ismasstransport = 1;
		this.isstressbalance = 1;
		this.isthermal       = 1;
		this.isgroundingline = 0;
		this.isgia           = 0;
		this.isdamageevolution = 0;
		this.ismovingfront   = 0;
		this.ishydrology     = 0;
		this.isslr           = 0;
		this.iscoupler       = 0;

		//default output
		this.requested_outputs=['default'];

	}// }}}
	this.disp= function(){// {{{

		console.log(sprintf('   transient solution parameters:'));

		fielddisplay(this,'issmb','indicates whether a surface mass balance solution is used in the transient');
		fielddisplay(this,'ismasstransport','indicates whether a masstransport solution is used in the transient');
		fielddisplay(this,'isstressbalance','indicates whether a stressbalance solution is used in the transient');
		fielddisplay(this,'isthermal','indicates whether a thermal solution is used in the transient');
		fielddisplay(this,'isgroundingline','indicates whether a groundingline migration is used in the transient');
		fielddisplay(this,'isgia','indicates whether a postglacial rebound model is used in the transient');
		fielddisplay(this,'isdamageevolution','indicates whether damage evolution is used in the transient');
		fielddisplay(this,'ismovingfront','indicates whether a moving front capability is used in the transient');
		fielddisplay(this,'ishydrology','indicates whether an hydrology model is used');
		fielddisplay(this,'isslr','indicates whether a sea-level rise model is used');
		fielddisplay(this,'iscoupler','indicates whether different models are being run with need for coupling');
		fielddisplay(this,'requested_outputs','list of additional outputs requested');


	}// }}}
	this.classname= function(){// {{{
		return "transient";
	}// }}}
		this.checkconsistency = function(md,solution,analyses) { // {{{

			//Early return
			if (solution!=TransientSolutionEnum()) return;

			checkfield(md,'fieldname','trans.issmb','numel',[1],'values',[0 ,1]);
			checkfield(md,'fieldname','trans.ismasstransport','numel',[1],'values',[0 ,1]);
			checkfield(md,'fieldname','trans.isstressbalance','numel',[1],'values',[0 ,1]);
			checkfield(md,'fieldname','trans.isthermal','numel',[1],'values',[0 ,1]);
			checkfield(md,'fieldname','trans.isgroundingline','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','trans.isgia','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','trans.isdamageevolution','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','trans.ismovingfront','numel',[1],'values',[0, 1]);
			checkfield(md,'fieldname','trans.ishydrology','numel',[1],'values',[0 ,1]);
			checkfield(md,'fieldname','trans.iscoupler','numel',[1],'values',[0 ,1]);
			checkfield(md,'fieldname','trans.isslr','numel',[1],'values',[0 ,1]);
			checkfield(md,'fieldname','trans.requested_outputs','stringrow',1);
		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'object',this,'fieldname','issmb','format','Boolean');
			WriteData(fid,'object',this,'fieldname','ismasstransport','format','Boolean');
			WriteData(fid,'object',this,'fieldname','isstressbalance','format','Boolean');
			WriteData(fid,'object',this,'fieldname','isthermal','format','Boolean');
			WriteData(fid,'object',this,'fieldname','isgroundingline','format','Boolean');
			WriteData(fid,'object',this,'fieldname','isgia','format','Boolean');
			WriteData(fid,'object',this,'fieldname','isdamageevolution','format','Boolean');
			WriteData(fid,'object',this,'fieldname','ishydrology','format','Boolean');
			WriteData(fid,'object',this,'fieldname','ismovingfront','format','Boolean');
			WriteData(fid,'object',this,'fieldname','isslr','format','Boolean');
			WriteData(fid,'object',this,'fieldname','iscoupler','format','Boolean');

			//process requested outputs
			var outputs = this.requested_outputs;
			for (var i=0;i<outputs.length;i++){
				if (outputs[i] == 'default') {
					outputs.splice(i,1);
					var newoutputs=this.defaultoutputs(md);
					for (var j=0;j<newoutputs.length;j++) outputs.push(newoutputs[j]);
				}
			}			
			WriteData(fid,'data',outputs,'enum',TransientRequestedOutputsEnum(),'format','StringArray');
		}//}}}
		this.defaultoutputs = function(md) { //{{{
			if(this.issmb)return ['SmbMassBalance'];
			else return [];
		}//}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{

	this.issmb             = 0;
	this.ismasstransport   = 0;
	this.isstressbalance   = 0;
	this.isthermal         = 0;
	this.isgroundingline   = 0;
	this.isgia             = 0;
	this.isdamageevolution = 0;
	this.ismovingfront     = 0;
	this.ishydrology       = 0;
	this.isslr             = 0;
	this.iscoupler         = 0;
	this.requested_outputs = [];

	this.setdefaultparameters();
	//}}}
}
//VERBOSE class definition
//
//   Available verbosity levels:
//      mprocessor  : model processing 
//      module      : modules
//      solution    : solution sequence
//      solver      : solver info (extensive)
//      convergence : convergence criteria
//      control     : control method
//      qmu         : sensitivity analysis
//      autodiff    : AD analysis
//      smb         : smb analysis
//
//   Usage:
//      verbose=verbose();
//      verbose=verbose(3);
//      verbose=verbose('all');
//      verbose=verbose('001100');
//      verbose=verbose('module',true,'solver',false);

//WARNING: some parts of this file are Synchronized with src/c/shared/Numerics/Verbosity.h
//         Do not modify these sections. See src/c/shared/Numerics/README for more info

function verbose (){
	//methods
	this.setdefaultparameters = function(){// {{{
	}// }}}
	this.disp= function(){// {{{
		//BEGINDISP
		console.log(sprintf('verbose class echo:'));
		console.log(sprintf('   %s : %i','mprocessor',this.mprocessor));
		console.log(sprintf('   %s : %i','module',this.module));
		console.log(sprintf('   %s : %i','solution',this.solution));
		console.log(sprintf('   %s : %i','solver',this.solver));
		console.log(sprintf('   %s : %i','convergence',this.convergence));
		console.log(sprintf('   %s : %i','control',this.control));
		console.log(sprintf('   %s : %i','qmu',this.qmu));
		console.log(sprintf('   %s : %i','autodiff',this.autodiff));
		console.log(sprintf('   %s : %i','smb',this.smb));
		//ENDDISP
	}// }}}
		this.checkconsistency = function(md,solution,analyses){ // {{{

		} // }}}
		this.marshall=function(md,fid) { //{{{
			WriteData(fid,'data',this.VerboseToBinary(),'enum',VerboseEnum(),'format','Integer');
		}//}}}
		this.VerboseToBinary = function () { //{{{

			//BEGINVERB2BIN
			var binary=0;
			if (this.mprocessor) binary=binary|1; 
			if (this.module) binary=binary|2; 
			if (this.solution) binary=binary|4; 
			if (this.solver) binary=binary|8; 
			if (this.convergence) binary=binary|16; 
			if (this.control) binary=binary|32; 
			if (this.qmu) binary=binary|64; 
			if (this.autodiff) binary=binary|128; 
			if (this.smb) binary=binary|256; 
			//ENDVERB2BIN
			return binary;

		} //}}}
		this.fix=function() { //{{{
		}//}}}
	//properties 
	// {{{
	//BEGINFIELDS
	this.mprocessor=false;
	this.module=false;
	this.solution=false;
	this.solver=false;
	this.convergence=false;
	this.control=false;
	this.qmu=false;
	this.autodiff=false;
	this.smb=false;
	//ENDFIELDS
	// }}}
	this.setdefaultparameters();
	//}}}
}
function checkfield(md){
//CHECKFIELD - check field consistency
//
//   Used to check model consistency.
//   Requires: 
//     'field' or 'fieldname' option. If 'fieldname' is provided, it will retrieve it from the model md. (md.(fieldname)) 
//             If 'field' is provided, it will assume the argument following 'field' is a numeric array.
//   Available options:
//      - NaN: 1 if check that there is no NaN
//      - Inf: 1 if check that there is no Inf
//      - size: [lines cols], NaN for non checked dimensions
//      - >:  greater than provided value
//      - >=: greater or equal to provided value
//      - <:  smallerthan provided value
//      - <=: smaller or equal to provided value
//      - < vec:  smallerthan provided values on each vertex
//      - timeseries: 1 if check time series consistency (size and time)
//      - values: array of strings or vector of acceptable values
//      - numel: list of acceptable number of elements
//      - array: 1 if check that is array
//      - empty: 1 if check that non empty
//      - message: overloaded error message
//
//   Usage:
//      checkfield(md,fieldname,options);

	//get options
	var args = Array.prototype.slice.call(arguments);
	var  options = new pairoptions(args.slice(1,args.length));
	
	//get field: 
	if (options.exist('field')){
		field=options.getfieldvalue('field'); 
		fieldname=options.getfieldvalue('fieldname','no fieldname'); 
	}
	else{
		fieldname=options.getfieldvalue('fieldname'); 
		eval("field=md." + fieldname + ";");
	}

	//check empty
	if (options.exist('empty')){
		if (field.length == 0){
			md.checkmessage(options.getfieldvalue('message','field ' + "'" + fieldname + "'" + 'is empty'));
		}
	}

	//Check size
	if (options.exist('size')){
		fieldsize=options.getfieldvalue('size');
		if (isNaN(fieldsize[0])){
			if (field[0].length !=fieldsize[1]){
				md.checkmessage(options.getfieldvalue('message', sprintf("field '%s' should have %i columns",fieldname,fieldsize[1])));
			}
		}
		else if (isNaN(fieldsize[1])){
			if (field.length!= fieldsize[0]){
				md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have %i lines",fieldname,fieldsize[0])));
			}
		}
		else{
			if (IsArray(field)){
				if ((field.length!=fieldsize[0])){
					md.checkmessage(options.getfieldvalue('message', sprintf("field '%s' should be %ix%i",fieldname,fieldsize[0],fieldsize[1])));
				}
			}
			if (IsArray(field[0])){
				if(field[0].length!=fieldsize[1]){
					md.checkmessage(options.getfieldvalue('message', sprintf("field '%s' should be %ix%i",fieldname,fieldsize[0],fieldsize[1])));
				}
			}
		}
	}

	//Check numel
	if (options.exist('numel')){
		fieldnumel=options.getfieldvalue('numel');
		if (!ArrayIsMember(fieldnumel,[field.length])){
			if (fieldnumel.length==1){
				md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' size should be %i",fieldname,fieldnumel[0])));
			}
			else if (fieldnumel.length==2){
				md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' size should be %i or %i",fieldname,fieldnumel[0],fieldnumel[1])));
			}
			else{
				var string=''; 
				for (var i=0;i<fieldnumel.length;i++)string=sprintf("%s or %i",string,fieldnumel[i]);
				md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' size should be %s",fieldname,string)));
			}
		}
	}

	//check NaN
	if (options.getfieldvalue('NaN',0)){
		field2=MatrixToList(field);
		if (ArrayAnyEqual(field2,NaN)){
			md.checkmessage(options.getfieldvalue('message',sprintf("NaN values found in field %s",field)));
		}
	}

	//check Inf
	if (options.getfieldvalue('Inf',0)){
		field2=MatrixToList(field);
		if (ArrayAnyEqual(field2,Infinity)){
			md.checkmessage(options.getfieldvalue('message',sprintf("Inf values found in field %s",field)));
		}
	}

	//check arry
	if (options.getfieldvalue('array',0)){
		if (!IsArray(field)){
			md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should be an array!",fieldname)));
		}
	}

	//check values
	if (options.exist('values')){
		fieldvalues=options.getfieldvalue('values');
		if (typeof fieldvalues[0]== 'string'){
			if (typeof field == 'string'){
				if(ArrayAnyEqual(ArrayIsMember([field],fieldvalues),0)){
					if (fieldvalues.length==1){
						md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' value should be %s",fieldname,fieldvalues[0])));
					}
					else if (fieldvalues.length==2){
						md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' values should be %s or %s",fieldname,fieldvalues[0],fieldvalues[1])));
					}
					else{
						var string=''; 
						for (var i=0;i<fieldvalues.length;i++)string=sprintf("%s or %s",string,fieldvalues[i]);
						md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have values in %s",fieldname,string)));
					}
				}
			}
			else{
				var string=''; for (var i=0;i<fieldvalues.length;i++)string=sprintf("%s or %s",string,fieldvalues[i]);
				md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have values in %s",fieldname,string)));
			}
		}
		else{
			if (typeof field == 'number') field2=MatrixToList([field]);
			else field2=MatrixToList(field);
			if (typeof field2[0] == 'number'){
				if(ArrayAnyEqual(ArrayIsMember(field2,fieldvalues),0)){
					var string=''; for (var i=0;i<fieldvalues.length;i++)string=sprintf("%s or %g",string,fieldvalues[i]);
					md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have values in %s",fieldname,string)));
				}
			}
			else{
				var string=''; for (var i=0;i<fieldvalues.length;i++)string=sprintf("%s or %g",string,fieldvalues[i]);
				md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should be a number in %s",fieldname,string)));
			}
		}
	}
	
	//check greater
	if (options.exist('>=')){
		lowerbound=options.getfieldvalue('>=');
		field2=MatrixToList(field);
		if (options.getfieldvalue('timeseries',0)) field2=MatrixToList(ArrayCopy(field).splice(-1,1));

		if (ArrayAnyBelowStrict(field2,lowerbound)){
			md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have values above %g",fieldname,lowerbound)));
		}
	}
	if (options.exist('>')){
		lowerbound=options.getfieldvalue('>');
		field2=MatrixToList(field);
		if (options.getfieldvalue('timeseries',0)) field2=MatrixToList(ArrayCopy(field).splice(-1,1));

		if (ArrayAnyBelowOrEqual(field2,lowerbound)){
			md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have values above %g",fieldname,lowerbound)));
		}
	}
	
	//check smaller
	if (options.exist('<=')){
		upperbound=options.getfieldvalue('<=');
		field2=MatrixToList(field);
		if (options.getfieldvalue('timeseries',0)) field2=MatrixToList(ArrayCopy(field).splice(-1,1));

		if (ArrayAnyAboveOrEqual(field2,upperbound)){
			md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have values below %g",fieldname,upperbound)));
		}
	}
	
	
	if (options.exist('<')){
		upperbound=options.getfieldvalue('<');
		field2=MatrixToList(field);
		if (options.getfieldvalue('timeseries',0)) field2=MatrixToList(ArrayCopy(field).splice(-1,1));
		if (ArrayAnyAboveStrict(field2,upperbound)){
			md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have values below %g",fieldname,upperbound)));
		}
	}

	//Check row of stringrow
	if (options.getfieldvalue('stringrow',0)){
		if (IsArray(field[0])){
			md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have only one row",field)));
		}
		if (!IsArray(field)){
			md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should be an array of string",fieldname)));
		}
		else{
			for(var i=0;i<field.length;i++){
				if (!(typeof field[i] == 'string')){
					md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' values should be a cell of strings",fieldname)));
				}
			}
		}
	}

	//check file
	if (options.getfieldvalue('file',0)){
		/*if ~exist(field,'file')
			md.checkmessage(['file provided in ''' fieldname ''': ''' field ''' does not exist']);
		end*/
		throw Error("checkfield error message: file checking on javascript not supported yet!");
	}

	//Check forcings (size and times)
	if (options.getfieldvalue('timeseries',0)){
		if (field.length==md.mesh.numberofvertices | field.length==md.mesh.numberofelements){
			if (IsArray(field[0])){
				md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have only one column as there are md.mesh.numberofvertices lines",fieldname)));
			}
		}
		else if ((field.length==md.mesh.numberofvertices+1) | (field.length==md.mesh.numberofelements+1)){
			var times=field[field.length-1]; var sorted_times=ArraySort(times);
			for(var i=0;i<times.length;i++){
				if(times[i] !=sorted_times[i]){
					md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' columns should be sorted chronologically",fieldname)));
					break;
				}
			}
			var timesm=ArrayCopy(times).splice(0,-1); var timesp=ArrayCopy(times).shift();
			for(var i=0;i<timesm.length;i++){
				if(timesm[i]==timesp[i]){
					md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' columns must not contain duplicate timesteps",fieldname)));
					break;
				}
			}
		}
		else{
			md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have md.mesh.numberofvertices or md.mesh.numberofvertices+1 lines",fieldname)));
		}
	}

	//Check single value forcings (size and times)
	if (options.getfieldvalue('singletimeseries',0)){
		if (field.length==2){
			var times=field[1]; var sorted_times=ArraySort(times);
			for(var i=0;i<times.length;i++){
				if(times[i] !=sorted_times[i]){
					md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' columns should be sorted chronologically",fieldname)));
					break;
				}
			}
			var timesm=ArrayCopy(times).splice(0,-1); var timesp=ArrayCopy(times).shift();
			for(var i=0;i<timesm.length;i++){
				if(timesm[i]==timesp[i]){
					md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' columns must not contain duplicate timesteps",fieldname)));
					break;
				}
			}
		}
		else{
			md.checkmessage(options.getfieldvalue('message',sprintf("field '%s' should have 2 lines",fieldname)));
		}
	}
}
function ismodelselfconsistent(md){
//ISMODELSELFCONSISTENT - check that model forms a closed form solvable problem.
//
//   Usage:
//      ismodelselfconsistent(md),

	//initialize consistency as true
	md.priv.isconsistent=true;

	//Get solution and associated analyses
	solution=md.priv.solution;
	if(typeof solution != 'number')throw Error('ismodelselfconsistent: did not provide solution type in the private class!');
	
	var analyses = AnalysisConfiguration(solution);

	//Go through a model field, check that it is a class, and call checkconsistency
	for(field in md){

		//Some properties do not need to be checked
		if (field == 'results' | field == 'debug' | field == 'radaroverlay'){
			continue;
		}

		//Check that current field is a class
		if(typeof md[field] == 'function'){
			continue;
		}

		//Check consistency of the class
		md[field].checkconsistency(md,solution,analyses);
	}

	//error message if mode is not consistent
	if (md.priv.isconsistent==false){
		throw Error('Model not consistent, see messages above');
	}
}

function AnalysisConfiguration(solutiontype){ // {{{
	//ANALYSISCONFIGURATION - return type of analyses, number of analyses 
	//
	//   Usage:
	//      [analyses]=AnalysisConfiguration(solutiontype);

	var analyses=[];
	switch (solutiontype){

		case StressbalanceSolutionEnum():
			analyses=[StressbalanceAnalysisEnum(),StressbalanceVerticalAnalysisEnum(),StressbalanceSIAAnalysisEnum(),L2ProjectionBaseAnalysisEnum()];
			break;

		case SteadystateSolutionEnum():
			analyses=[StressbalanceAnalysisEnum(),StressbalanceVerticalAnalysisEnum(),StressbalanceSIAAnalysisEnum(),L2ProjectionBaseAnalysisEnum(),ThermalAnalysisEnum(),MeltingAnalysisEnum()];
			break;

		case ThermalSolutionEnum():
			analyses=[EnthalpyAnalysisEnum(),ThermalAnalysisEnum(),MeltingAnalysisEnum()];
			break;

		case MasstransportSolutionEnum():
			analyses=[MasstransportAnalysisEnum()];
			break;

		case BalancethicknessSolutionEnum():
			analyses=[BalancethicknessAnalysisEnum()];
			break;

		case Balancethickness2SolutionEnum():
			analyses=[Balancethickness2AnalysisEnum()];
			break;

		case BalancethicknessSoftSolutionEnum():
			analyses=[BalancethicknessAnalysisEnum()];
			break;

		case BalancevelocitySolutionEnum():
			analyses=[BalancevelocityAnalysisEnum()];
			break;

		case SurfaceSlopeSolutionEnum():
			analyses=[L2ProjectionBaseAnalysisEnum()];
			break;

		case BedSlopeSolutionEnum():
			analyses=[L2ProjectionBaseAnalysisEnum()];
			break;

		case GiaSolutionEnum():
			analyses=[GiaAnalysisEnum()];
			break;

		case TransientSolutionEnum():
			analyses=[StressbalanceAnalysisEnum(),StressbalanceVerticalAnalysisEnum(),StressbalanceSIAAnalysisEnum(),L2ProjectionBaseAnalysisEnum(),ThermalAnalysisEnum(),MeltingAnalysisEnum(),EnthalpyAnalysisEnum(),MasstransportAnalysisEnum(),HydrologySommersAnalysisEnum];
			break;

		case FlaimSolutionEnum():
			analyses=[FlaimAnalysisEnum()];
			break;
	
		case SealevelriseSolutionEnum():
			analyses=[SealevelriseAnalysisEnum()];
			break;

		case HydrologySolutionEnum():
			analyses=[L2ProjectionBaseAnalysisEnum(),HydrologyShreveAnalysisEnum(),HydrologyDCInefficientAnalysisEnum(),HydrologyDCEfficientAnalysisEnum()];
			break;

		case DamageEvolutionSolutionEnum():
			analyses=[DamageEvolutionAnalysisEnum()];
			break;

		default:
			throw Error(sprintf("%s%s%s\n",' solution type: ',EnumToString(solutiontype),' not supported yet!'));
	}
	return analyses;
} // }}}
//WARNING: DO NOT MODIFY THIS FILE
//this file has been automatically generated by src/c/shared/Enum/Synchronize.sh
//Please read src/c/shared/Enum/README for more information
function FemModelEnum(){ return 0;}
function FemModelCommEnum(){ return 1;}
function WorldCommEnum(){ return 2;}
function IcecapToEarthCommEnum(){ return 3;}
function NumModelsEnum(){ return 4;}
function ModelIdEnum(){ return 5;}
function EarthIdEnum(){ return 6;}
function AutodiffIsautodiffEnum(){ return 7;}
function AutodiffNumDependentsEnum(){ return 8;}
function AutodiffNumDependentObjectsEnum(){ return 9;}
function AutodiffDependentObjectNamesEnum(){ return 10;}
function AutodiffDependentObjectTypesEnum(){ return 11;}
function AutodiffDependentObjectIndicesEnum(){ return 12;}
function AutodiffDependentObjectsEnum(){ return 13;}
function AutodiffNumIndependentsEnum(){ return 14;}
function AutodiffNumIndependentObjectsEnum(){ return 15;}
function AutodiffIndependentObjectNamesEnum(){ return 16;}
function AutodiffIndependentObjectTypesEnum(){ return 17;}
function AutodiffIndependentObjectsEnum(){ return 18;}
function AutodiffJacobianEnum(){ return 19;}
function AutodiffXpEnum(){ return 20;}
function AutodiffDriverEnum(){ return 21;}
function AutodiffFosForwardIndexEnum(){ return 22;}
function AutodiffFovForwardIndicesEnum(){ return 23;}
function AutodiffFosReverseIndexEnum(){ return 24;}
function AutodiffMassFluxSegmentsPresentEnum(){ return 25;}
function AutodiffKeepEnum(){ return 26;}
function AutodiffObufsizeEnum(){ return 27;}
function AutodiffLbufsizeEnum(){ return 28;}
function AutodiffCbufsizeEnum(){ return 29;}
function AutodiffTbufsizeEnum(){ return 30;}
function AutodiffGcTriggerRatioEnum(){ return 31;}
function AutodiffGcTriggerMaxSizeEnum(){ return 32;}
function BalancethicknessSpcthicknessEnum(){ return 33;}
function BalancethicknessStabilizationEnum(){ return 34;}
function BalancethicknessThickeningRateEnum(){ return 35;}
function BasalforcingsEnum(){ return 36;}
function BasalforcingsGeothermalfluxEnum(){ return 37;}
function BasalforcingsGroundediceMeltingRateEnum(){ return 38;}
function BasalforcingsFloatingiceMeltingRateEnum(){ return 39;}
function BasalforcingsDeepwaterMeltingRateEnum(){ return 40;}
function BasalforcingsDeepwaterElevationEnum(){ return 41;}
function BasalforcingsUpperwaterElevationEnum(){ return 42;}
function BasalforcingsMeltrateFactorEnum(){ return 43;}
function BasalforcingsThresholdThicknessEnum(){ return 44;}
function BasalforcingsUpperdepthMeltEnum(){ return 45;}
function BasalforcingsMantleconductivityEnum(){ return 46;}
function BasalforcingsNusseltEnum(){ return 47;}
function BasalforcingsDtbgEnum(){ return 48;}
function BasalforcingsPlumeradiusEnum(){ return 49;}
function BasalforcingsTopplumedepthEnum(){ return 50;}
function BasalforcingsBottomplumedepthEnum(){ return 51;}
function BasalforcingsPlumexEnum(){ return 52;}
function BasalforcingsPlumeyEnum(){ return 53;}
function BasalforcingsCrustthicknessEnum(){ return 54;}
function BasalforcingsUppercrustthicknessEnum(){ return 55;}
function BasalforcingsUppercrustheatEnum(){ return 56;}
function BasalforcingsLowercrustheatEnum(){ return 57;}
function FloatingMeltRateEnum(){ return 58;}
function LinearFloatingMeltRateEnum(){ return 59;}
function MismipFloatingMeltRateEnum(){ return 60;}
function MantlePlumeGeothermalFluxEnum(){ return 61;}
function BedEnum(){ return 62;}
function BaseEnum(){ return 63;}
function ConstantsGEnum(){ return 64;}
function ConstantsReferencetemperatureEnum(){ return 65;}
function ConstantsYtsEnum(){ return 66;}
function DependentObjectEnum(){ return 67;}
function StressbalanceAbstolEnum(){ return 68;}
function StressbalanceIsnewtonEnum(){ return 69;}
function StressbalanceMaxiterEnum(){ return 70;}
function StressbalancePenaltyFactorEnum(){ return 71;}
function StressbalanceReferentialEnum(){ return 72;}
function StressbalanceReltolEnum(){ return 73;}
function StressbalanceNumRequestedOutputsEnum(){ return 74;}
function StressbalanceRequestedOutputsEnum(){ return 75;}
function StressbalanceRestolEnum(){ return 76;}
function StressbalanceRiftPenaltyLockEnum(){ return 77;}
function StressbalanceRiftPenaltyThresholdEnum(){ return 78;}
function StressbalanceShelfDampeningEnum(){ return 79;}
function StressbalanceSpcvxEnum(){ return 80;}
function StressbalanceSpcvyEnum(){ return 81;}
function StressbalanceSpcvzEnum(){ return 82;}
function StressbalanceFSreconditioningEnum(){ return 83;}
function StressbalanceVertexPairingEnum(){ return 84;}
function StressbalanceViscosityOvershootEnum(){ return 85;}
function LoadingforceXEnum(){ return 86;}
function LoadingforceYEnum(){ return 87;}
function LoadingforceZEnum(){ return 88;}
function FlowequationBorderSSAEnum(){ return 89;}
function FlowequationBorderHOEnum(){ return 90;}
function FlowequationBorderFSEnum(){ return 91;}
function FlowequationElementEquationEnum(){ return 92;}
function FlowequationIsSIAEnum(){ return 93;}
function FlowequationIsSSAEnum(){ return 94;}
function FlowequationIsL1L2Enum(){ return 95;}
function FlowequationIsHOEnum(){ return 96;}
function FlowequationIsFSEnum(){ return 97;}
function FlowequationFeSSAEnum(){ return 98;}
function FlowequationFeHOEnum(){ return 99;}
function FlowequationFeFSEnum(){ return 100;}
function FlowequationVertexEquationEnum(){ return 101;}
function FrictionAsEnum(){ return 102;}
function FrictionCoefficientEnum(){ return 103;}
function FrictionCoefficientcoulombEnum(){ return 104;}
function FrictionPEnum(){ return 105;}
function FrictionQEnum(){ return 106;}
function FrictionMEnum(){ return 107;}
function FrictionCEnum(){ return 108;}
function FrictionLawEnum(){ return 109;}
function FrictionGammaEnum(){ return 110;}
function FrictionFEnum(){ return 111;}
function FrictionWaterLayerEnum(){ return 112;}
function FrictionEffectivePressureEnum(){ return 113;}
function FrictionCouplingEnum(){ return 114;}
function GeometryHydrostaticRatioEnum(){ return 115;}
function HydrologyModelEnum(){ return 116;}
function HydrologyshreveEnum(){ return 117;}
function HydrologyshreveSpcwatercolumnEnum(){ return 118;}
function HydrologyshreveStabilizationEnum(){ return 119;}
function HydrologydcEnum(){ return 120;}
function SedimentHeadEnum(){ return 121;}
function SedimentHeadOldEnum(){ return 122;}
function SedimentHeadResidualEnum(){ return 123;}
function EffectivePressureEnum(){ return 124;}
function EplHeadEnum(){ return 125;}
function EplHeadOldEnum(){ return 126;}
function EplHeadSlopeXEnum(){ return 127;}
function EplHeadSlopeYEnum(){ return 128;}
function EplZigZagCounterEnum(){ return 129;}
function HydrologydcMaxIterEnum(){ return 130;}
function HydrologydcRelTolEnum(){ return 131;}
function HydrologydcSpcsedimentHeadEnum(){ return 132;}
function HydrologydcSedimentCompressibilityEnum(){ return 133;}
function HydrologydcSedimentPorosityEnum(){ return 134;}
function HydrologydcSedimentThicknessEnum(){ return 135;}
function HydrologydcSedimentTransmitivityEnum(){ return 136;}
function HydrologydcWaterCompressibilityEnum(){ return 137;}
function HydrologydcSpceplHeadEnum(){ return 138;}
function HydrologydcMaskEplactiveNodeEnum(){ return 139;}
function HydrologydcMaskEplactiveEltEnum(){ return 140;}
function HydrologydcEplCompressibilityEnum(){ return 141;}
function HydrologydcEplPorosityEnum(){ return 142;}
function HydrologydcEplInitialThicknessEnum(){ return 143;}
function HydrologydcEplColapseThicknessEnum(){ return 144;}
function HydrologydcEplMaxThicknessEnum(){ return 145;}
function HydrologydcEplThicknessEnum(){ return 146;}
function HydrologydcEplThicknessOldEnum(){ return 147;}
function HydrologydcEplThickCompEnum(){ return 148;}
function HydrologydcEplConductivityEnum(){ return 149;}
function HydrologydcIsefficientlayerEnum(){ return 150;}
function HydrologydcSedimentlimitFlagEnum(){ return 151;}
function HydrologydcSedimentlimitEnum(){ return 152;}
function HydrologydcTransferFlagEnum(){ return 153;}
function HydrologydcLeakageFactorEnum(){ return 154;}
function HydrologydcPenaltyFactorEnum(){ return 155;}
function HydrologydcPenaltyLockEnum(){ return 156;}
function HydrologydcEplflipLockEnum(){ return 157;}
function HydrologydcBasalMoulinInputEnum(){ return 158;}
function HydrologyLayerEnum(){ return 159;}
function HydrologySedimentEnum(){ return 160;}
function HydrologyEfficientEnum(){ return 161;}
function HydrologySedimentKmaxEnum(){ return 162;}
function HydrologysommersEnum(){ return 163;}
function HydrologyHeadEnum(){ return 164;}
function HydrologyGapHeightEnum(){ return 165;}
function HydrologyBumpSpacingEnum(){ return 166;}
function HydrologyBumpHeightEnum(){ return 167;}
function HydrologyEnglacialInputEnum(){ return 168;}
function HydrologyMoulinInputEnum(){ return 169;}
function HydrologyReynoldsEnum(){ return 170;}
function HydrologyNeumannfluxEnum(){ return 171;}
function HydrologySpcheadEnum(){ return 172;}
function HydrologyConductivityEnum(){ return 173;}
function IndependentObjectEnum(){ return 174;}
function InversionControlParametersEnum(){ return 175;}
function InversionControlScalingFactorsEnum(){ return 176;}
function InversionCostFunctionThresholdEnum(){ return 177;}
function InversionCostFunctionsCoefficientsEnum(){ return 178;}
function InversionCostFunctionsEnum(){ return 179;}
function InversionGradientScalingEnum(){ return 180;}
function InversionIscontrolEnum(){ return 181;}
function InversionTypeEnum(){ return 182;}
function InversionIncompleteAdjointEnum(){ return 183;}
function InversionMaxParametersEnum(){ return 184;}
function InversionMaxiterPerStepEnum(){ return 185;}
function InversionMaxiterEnum(){ return 186;}
function InversionMaxstepsEnum(){ return 187;}
function InversionFatolEnum(){ return 188;}
function InversionFrtolEnum(){ return 189;}
function InversionGatolEnum(){ return 190;}
function InversionGrtolEnum(){ return 191;}
function InversionGttolEnum(){ return 192;}
function InversionAlgorithmEnum(){ return 193;}
function InversionMinParametersEnum(){ return 194;}
function InversionNstepsEnum(){ return 195;}
function InversionDxminEnum(){ return 196;}
function InversionNumControlParametersEnum(){ return 197;}
function InversionNumCostFunctionsEnum(){ return 198;}
function InversionStepThresholdEnum(){ return 199;}
function InversionThicknessObsEnum(){ return 200;}
function InversionSurfaceObsEnum(){ return 201;}
function InversionVxObsEnum(){ return 202;}
function InversionVyObsEnum(){ return 203;}
function InversionVzObsEnum(){ return 204;}
function MaskIceLevelsetEnum(){ return 205;}
function MaskOceanLevelsetEnum(){ return 206;}
function MaskLandLevelsetEnum(){ return 207;}
function MaterialsBetaEnum(){ return 208;}
function MaterialsHeatcapacityEnum(){ return 209;}
function MaterialsLatentheatEnum(){ return 210;}
function MaterialsMeltingpointEnum(){ return 211;}
function MaterialsMixedLayerCapacityEnum(){ return 212;}
function MaterialsRheologyBEnum(){ return 213;}
function MaterialsRheologyBbarEnum(){ return 214;}
function MaterialsRheologyLawEnum(){ return 215;}
function MaterialsRheologyNEnum(){ return 216;}
function DamageIsdamageEnum(){ return 217;}
function DamageDEnum(){ return 218;}
function DamageFEnum(){ return 219;}
function DamageDbarEnum(){ return 220;}
function DamageLawEnum(){ return 221;}
function DamageC1Enum(){ return 222;}
function DamageC2Enum(){ return 223;}
function DamageC3Enum(){ return 224;}
function DamageC4Enum(){ return 225;}
function DamageElementinterpEnum(){ return 226;}
function DamageHealingEnum(){ return 227;}
function DamageStressThresholdEnum(){ return 228;}
function DamageKappaEnum(){ return 229;}
function DamageStabilizationEnum(){ return 230;}
function DamageMaxiterEnum(){ return 231;}
function DamageSpcdamageEnum(){ return 232;}
function DamageMaxDamageEnum(){ return 233;}
function DamageEquivStressEnum(){ return 234;}
function DamageEvolutionNumRequestedOutputsEnum(){ return 235;}
function DamageEvolutionRequestedOutputsEnum(){ return 236;}
function DamageEnum(){ return 237;}
function NewDamageEnum(){ return 238;}
function StressIntensityFactorEnum(){ return 239;}
function CalvingLawEnum(){ return 240;}
function CalvingCalvingrateEnum(){ return 241;}
function CalvingMeltingrateEnum(){ return 242;}
function CalvingLevermannEnum(){ return 243;}
function CalvingDevEnum(){ return 244;}
function DefaultCalvingEnum(){ return 245;}
function CalvingRequestedOutputsEnum(){ return 246;}
function CalvinglevermannCoeffEnum(){ return 247;}
function CalvinglevermannMeltingrateEnum(){ return 248;}
function CalvingdevCoeffEnum(){ return 249;}
function CalvingratexEnum(){ return 250;}
function CalvingrateyEnum(){ return 251;}
function CalvingratexAverageEnum(){ return 252;}
function CalvingrateyAverageEnum(){ return 253;}
function StrainRateparallelEnum(){ return 254;}
function StrainRateperpendicularEnum(){ return 255;}
function StrainRateeffectiveEnum(){ return 256;}
function MaterialsRhoIceEnum(){ return 257;}
function MaterialsRhoSeawaterEnum(){ return 258;}
function MaterialsRhoFreshwaterEnum(){ return 259;}
function MaterialsMuWaterEnum(){ return 260;}
function MaterialsThermalExchangeVelocityEnum(){ return 261;}
function MaterialsThermalconductivityEnum(){ return 262;}
function MaterialsTemperateiceconductivityEnum(){ return 263;}
function MaterialsLithosphereShearModulusEnum(){ return 264;}
function MaterialsLithosphereDensityEnum(){ return 265;}
function MaterialsMantleShearModulusEnum(){ return 266;}
function MaterialsMantleDensityEnum(){ return 267;}
function MaterialsEarthDensityEnum(){ return 268;}
function MeshAverageVertexConnectivityEnum(){ return 269;}
function MeshElements2dEnum(){ return 270;}
function MeshElementsEnum(){ return 271;}
function MeshLowerelementsEnum(){ return 272;}
function MeshNumberofelements2dEnum(){ return 273;}
function MeshNumberofelementsEnum(){ return 274;}
function MeshNumberoflayersEnum(){ return 275;}
function MeshNumberofvertices2dEnum(){ return 276;}
function MeshNumberofverticesEnum(){ return 277;}
function MeshUpperelementsEnum(){ return 278;}
function MeshVertexonbaseEnum(){ return 279;}
function MeshVertexonsurfaceEnum(){ return 280;}
function MeshVertexonboundaryEnum(){ return 281;}
function MeshXEnum(){ return 282;}
function MeshYEnum(){ return 283;}
function MeshZEnum(){ return 284;}
function MeshLatEnum(){ return 285;}
function MeshLongEnum(){ return 286;}
function MeshREnum(){ return 287;}
function MeshElementtypeEnum(){ return 288;}
function MeshSegmentsEnum(){ return 289;}
function DomainTypeEnum(){ return 290;}
function DomainDimensionEnum(){ return 291;}
function Domain2DhorizontalEnum(){ return 292;}
function Domain2DverticalEnum(){ return 293;}
function Domain3DEnum(){ return 294;}
function Domain3DsurfaceEnum(){ return 295;}
function MiscellaneousNameEnum(){ return 296;}
function MasstransportHydrostaticAdjustmentEnum(){ return 297;}
function MasstransportIsfreesurfaceEnum(){ return 298;}
function MasstransportMinThicknessEnum(){ return 299;}
function MasstransportPenaltyFactorEnum(){ return 300;}
function MasstransportSpcthicknessEnum(){ return 301;}
function MasstransportStabilizationEnum(){ return 302;}
function MasstransportVertexPairingEnum(){ return 303;}
function MasstransportNumRequestedOutputsEnum(){ return 304;}
function MasstransportRequestedOutputsEnum(){ return 305;}
function QmuIsdakotaEnum(){ return 306;}
function MassFluxSegmentsEnum(){ return 307;}
function MassFluxSegmentsPresentEnum(){ return 308;}
function QmuMassFluxSegmentsPresentEnum(){ return 309;}
function QmuNumberofpartitionsEnum(){ return 310;}
function QmuNumberofresponsesEnum(){ return 311;}
function QmuPartitionEnum(){ return 312;}
function QmuResponsedescriptorsEnum(){ return 313;}
function QmuVariabledescriptorsEnum(){ return 314;}
function RiftsNumriftsEnum(){ return 315;}
function RiftsRiftstructEnum(){ return 316;}
function SettingsResultsOnNodesEnum(){ return 317;}
function SettingsIoGatherEnum(){ return 318;}
function SettingsLowmemEnum(){ return 319;}
function SettingsOutputFrequencyEnum(){ return 320;}
function SettingsRecordingFrequencyEnum(){ return 321;}
function SettingsWaitonlockEnum(){ return 322;}
function DebugProfilingEnum(){ return 323;}
function ProfilingCurrentMemEnum(){ return 324;}
function ProfilingCurrentFlopsEnum(){ return 325;}
function ProfilingSolutionTimeEnum(){ return 326;}
function SteadystateMaxiterEnum(){ return 327;}
function SteadystateNumRequestedOutputsEnum(){ return 328;}
function SteadystateReltolEnum(){ return 329;}
function SteadystateRequestedOutputsEnum(){ return 330;}
function SurfaceEnum(){ return 331;}
function ThermalIsenthalpyEnum(){ return 332;}
function ThermalIsdynamicbasalspcEnum(){ return 333;}
function ThermalReltolEnum(){ return 334;}
function ThermalMaxiterEnum(){ return 335;}
function ThermalPenaltyFactorEnum(){ return 336;}
function ThermalPenaltyLockEnum(){ return 337;}
function ThermalPenaltyThresholdEnum(){ return 338;}
function ThermalSpctemperatureEnum(){ return 339;}
function ThermalStabilizationEnum(){ return 340;}
function ThermalNumRequestedOutputsEnum(){ return 341;}
function ThermalRequestedOutputsEnum(){ return 342;}
function GiaMantleViscosityEnum(){ return 343;}
function GiaLithosphereThicknessEnum(){ return 344;}
function ThicknessEnum(){ return 345;}
function TimesteppingStartTimeEnum(){ return 346;}
function TimesteppingFinalTimeEnum(){ return 347;}
function TimesteppingCflCoefficientEnum(){ return 348;}
function TimesteppingTimeAdaptEnum(){ return 349;}
function TimesteppingTimeStepEnum(){ return 350;}
function TimesteppingInterpForcingsEnum(){ return 351;}
function TransientIssmbEnum(){ return 352;}
function TransientIscouplerEnum(){ return 353;}
function TransientIsstressbalanceEnum(){ return 354;}
function TransientIsgroundinglineEnum(){ return 355;}
function TransientIsmasstransportEnum(){ return 356;}
function TransientIsthermalEnum(){ return 357;}
function TransientIsgiaEnum(){ return 358;}
function TransientIsdamageevolutionEnum(){ return 359;}
function TransientIshydrologyEnum(){ return 360;}
function TransientIsmovingfrontEnum(){ return 361;}
function TransientIsslrEnum(){ return 362;}
function TransientNumRequestedOutputsEnum(){ return 363;}
function TransientRequestedOutputsEnum(){ return 364;}
function PotentialEnum(){ return 365;}
function BalancethicknessSpcpotentialEnum(){ return 366;}
function BalancethicknessApparentMassbalanceEnum(){ return 367;}
function Balancethickness2MisfitEnum(){ return 368;}
function BalancethicknessDiffusionCoefficientEnum(){ return 369;}
function BalancethicknessCmuEnum(){ return 370;}
function BalancethicknessOmegaEnum(){ return 371;}
function BalancethicknessD0Enum(){ return 372;}
function SmbEnum(){ return 373;}
function SmbAnalysisEnum(){ return 374;}
function SmbSolutionEnum(){ return 375;}
function SmbNumRequestedOutputsEnum(){ return 376;}
function SmbRequestedOutputsEnum(){ return 377;}
function SmbIsInitializedEnum(){ return 378;}
function SMBforcingEnum(){ return 379;}
function SmbMassBalanceEnum(){ return 380;}
function SMBgembEnum(){ return 381;}
function SmbInitDensityScalingEnum(){ return 382;}
function SmbTaEnum(){ return 383;}
function SmbVEnum(){ return 384;}
function SmbDswrfEnum(){ return 385;}
function SmbDlwrfEnum(){ return 386;}
function SmbPEnum(){ return 387;}
function SmbSwfEnum(){ return 388;}
function SmbEAirEnum(){ return 389;}
function SmbPAirEnum(){ return 390;}
function SmbTmeanEnum(){ return 391;}
function SmbCEnum(){ return 392;}
function SmbTzEnum(){ return 393;}
function SmbVzEnum(){ return 394;}
function SmbDtEnum(){ return 395;}
function SmbDzEnum(){ return 396;}
function SmbAIdxEnum(){ return 397;}
function SmbSwIdxEnum(){ return 398;}
function SmbDenIdxEnum(){ return 399;}
function SmbZTopEnum(){ return 400;}
function SmbDzTopEnum(){ return 401;}
function SmbDzMinEnum(){ return 402;}
function SmbZYEnum(){ return 403;}
function SmbZMaxEnum(){ return 404;}
function SmbZMinEnum(){ return 405;}
function SmbOutputFreqEnum(){ return 406;}
function SmbASnowEnum(){ return 407;}
function SmbAIceEnum(){ return 408;}
function SmbCldFracEnum(){ return 409;}
function SmbT0wetEnum(){ return 410;}
function SmbT0dryEnum(){ return 411;}
function SmbKEnum(){ return 412;}
function SmbDEnum(){ return 413;}
function SmbReEnum(){ return 414;}
function SmbGdnEnum(){ return 415;}
function SmbGspEnum(){ return 416;}
function SmbECEnum(){ return 417;}
function SmbCondensationEnum(){ return 418;}
function SmbWEnum(){ return 419;}
function SmbAEnum(){ return 420;}
function SmbTEnum(){ return 421;}
function SmbIsgraingrowthEnum(){ return 422;}
function SmbIsalbedoEnum(){ return 423;}
function SmbIsshortwaveEnum(){ return 424;}
function SmbIsthermalEnum(){ return 425;}
function SmbIsaccumulationEnum(){ return 426;}
function SmbIsmeltEnum(){ return 427;}
function SmbIsdensificationEnum(){ return 428;}
function SmbIsturbulentfluxEnum(){ return 429;}
function SMBpddEnum(){ return 430;}
function SmbDelta18oEnum(){ return 431;}
function SmbDelta18oSurfaceEnum(){ return 432;}
function SmbIsdelta18oEnum(){ return 433;}
function SmbIsmungsmEnum(){ return 434;}
function SmbIsd18opdEnum(){ return 435;}
function SmbPrecipitationsPresentdayEnum(){ return 436;}
function SmbPrecipitationsLgmEnum(){ return 437;}
function SmbTemperaturesPresentdayEnum(){ return 438;}
function SmbTemperaturesLgmEnum(){ return 439;}
function SmbPrecipitationEnum(){ return 440;}
function SmbDesfacEnum(){ return 441;}
function SmbS0pEnum(){ return 442;}
function SmbS0tEnum(){ return 443;}
function SmbRlapsEnum(){ return 444;}
function SmbRlapslgmEnum(){ return 445;}
function SmbPfacEnum(){ return 446;}
function SmbTdiffEnum(){ return 447;}
function SmbSealevEnum(){ return 448;}
function SMBd18opddEnum(){ return 449;}
function SmbDpermilEnum(){ return 450;}
function SMBgradientsEnum(){ return 451;}
function SmbMonthlytemperaturesEnum(){ return 452;}
function SmbHrefEnum(){ return 453;}
function SmbSmbrefEnum(){ return 454;}
function SmbBPosEnum(){ return 455;}
function SmbBNegEnum(){ return 456;}
function SMBhenningEnum(){ return 457;}
function SMBcomponentsEnum(){ return 458;}
function SmbAccumulationEnum(){ return 459;}
function SmbEvaporationEnum(){ return 460;}
function SmbRunoffEnum(){ return 461;}
function SMBmeltcomponentsEnum(){ return 462;}
function SmbMeltEnum(){ return 463;}
function SmbRefreezeEnum(){ return 464;}
function SMBgcmEnum(){ return 465;}
function SmbIspddEnum(){ return 466;}
function SmbIssmbgradientsEnum(){ return 467;}
function SolutionTypeEnum(){ return 468;}
function AnalysisTypeEnum(){ return 469;}
function ConfigurationTypeEnum(){ return 470;}
function AdjointBalancethicknessAnalysisEnum(){ return 471;}
function AdjointBalancethickness2AnalysisEnum(){ return 472;}
function AdjointHorizAnalysisEnum(){ return 473;}
function AnalysisCounterEnum(){ return 474;}
function DefaultAnalysisEnum(){ return 475;}
function BalancethicknessAnalysisEnum(){ return 476;}
function BalancethicknessSolutionEnum(){ return 477;}
function Balancethickness2AnalysisEnum(){ return 478;}
function Balancethickness2SolutionEnum(){ return 479;}
function BalancethicknessSoftAnalysisEnum(){ return 480;}
function BalancethicknessSoftSolutionEnum(){ return 481;}
function BalancevelocityAnalysisEnum(){ return 482;}
function BalancevelocitySolutionEnum(){ return 483;}
function L2ProjectionEPLAnalysisEnum(){ return 484;}
function L2ProjectionBaseAnalysisEnum(){ return 485;}
function BedSlopeSolutionEnum(){ return 486;}
function DamageEvolutionSolutionEnum(){ return 487;}
function DamageEvolutionAnalysisEnum(){ return 488;}
function StressbalanceAnalysisEnum(){ return 489;}
function StressbalanceSIAAnalysisEnum(){ return 490;}
function StressbalanceSolutionEnum(){ return 491;}
function StressbalanceVerticalAnalysisEnum(){ return 492;}
function EnthalpyAnalysisEnum(){ return 493;}
function FlaimAnalysisEnum(){ return 494;}
function FlaimSolutionEnum(){ return 495;}
function HydrologyShreveAnalysisEnum(){ return 496;}
function HydrologyDCInefficientAnalysisEnum(){ return 497;}
function HydrologyDCEfficientAnalysisEnum(){ return 498;}
function HydrologySommersAnalysisEnum(){ return 499;}
function HydrologySolutionEnum(){ return 500;}
function MeltingAnalysisEnum(){ return 501;}
function MasstransportAnalysisEnum(){ return 502;}
function MasstransportSolutionEnum(){ return 503;}
function FreeSurfaceBaseAnalysisEnum(){ return 504;}
function FreeSurfaceTopAnalysisEnum(){ return 505;}
function SurfaceNormalVelocityEnum(){ return 506;}
function ExtrudeFromBaseAnalysisEnum(){ return 507;}
function ExtrudeFromTopAnalysisEnum(){ return 508;}
function DepthAverageAnalysisEnum(){ return 509;}
function SteadystateSolutionEnum(){ return 510;}
function SurfaceSlopeSolutionEnum(){ return 511;}
function SmoothAnalysisEnum(){ return 512;}
function ThermalAnalysisEnum(){ return 513;}
function ThermalSolutionEnum(){ return 514;}
function TransientSolutionEnum(){ return 515;}
function UzawaPressureAnalysisEnum(){ return 516;}
function GiaSolutionEnum(){ return 517;}
function GiaAnalysisEnum(){ return 518;}
function MeshdeformationSolutionEnum(){ return 519;}
function MeshdeformationAnalysisEnum(){ return 520;}
function LevelsetAnalysisEnum(){ return 521;}
function LevelsetStabilizationEnum(){ return 522;}
function ExtrapolationAnalysisEnum(){ return 523;}
function LsfReinitializationAnalysisEnum(){ return 524;}
function ApproximationEnum(){ return 525;}
function NoneApproximationEnum(){ return 526;}
function SIAApproximationEnum(){ return 527;}
function SSAApproximationEnum(){ return 528;}
function SSAHOApproximationEnum(){ return 529;}
function SSAFSApproximationEnum(){ return 530;}
function L1L2ApproximationEnum(){ return 531;}
function HOApproximationEnum(){ return 532;}
function HOFSApproximationEnum(){ return 533;}
function FSApproximationEnum(){ return 534;}
function FSvelocityEnum(){ return 535;}
function FSpressureEnum(){ return 536;}
function DataSetEnum(){ return 537;}
function ConstraintsEnum(){ return 538;}
function LoadsEnum(){ return 539;}
function MaterialsEnum(){ return 540;}
function NodesEnum(){ return 541;}
function ContoursEnum(){ return 542;}
function ParametersEnum(){ return 543;}
function VerticesEnum(){ return 544;}
function ResultsEnum(){ return 545;}
function GenericParamEnum(){ return 546;}
function AdolcParamEnum(){ return 547;}
function BoolInputEnum(){ return 548;}
function BoolParamEnum(){ return 549;}
function ContourEnum(){ return 550;}
function ControlInputEnum(){ return 551;}
function DatasetInputEnum(){ return 552;}
function DoubleInputEnum(){ return 553;}
function DoubleArrayInputEnum(){ return 554;}
function DataSetParamEnum(){ return 555;}
function DoubleMatArrayParamEnum(){ return 556;}
function DoubleMatParamEnum(){ return 557;}
function DoubleParamEnum(){ return 558;}
function DoubleVecParamEnum(){ return 559;}
function ElementEnum(){ return 560;}
function ElementHookEnum(){ return 561;}
function HookEnum(){ return 562;}
function ExternalResultEnum(){ return 563;}
function FileParamEnum(){ return 564;}
function InputEnum(){ return 565;}
function IntInputEnum(){ return 566;}
function InputToExtrudeEnum(){ return 567;}
function InputToL2ProjectEnum(){ return 568;}
function InputToDepthaverageEnum(){ return 569;}
function InputToSmoothEnum(){ return 570;}
function SmoothThicknessMultiplierEnum(){ return 571;}
function IntParamEnum(){ return 572;}
function IntVecParamEnum(){ return 573;}
function TransientParamEnum(){ return 574;}
function MaticeEnum(){ return 575;}
function MatdamageiceEnum(){ return 576;}
function MatparEnum(){ return 577;}
function NodeEnum(){ return 578;}
function NumericalfluxEnum(){ return 579;}
function NumericalfluxTypeEnum(){ return 580;}
function NeumannfluxEnum(){ return 581;}
function ParamEnum(){ return 582;}
function MoulinEnum(){ return 583;}
function PengridEnum(){ return 584;}
function PenpairEnum(){ return 585;}
function ProfilerEnum(){ return 586;}
function MatrixParamEnum(){ return 587;}
function MassconEnum(){ return 588;}
function MassconNameEnum(){ return 589;}
function MassconDefinitionenumEnum(){ return 590;}
function MassconLevelsetEnum(){ return 591;}
function MassconaxpbyEnum(){ return 592;}
function MassconaxpbyNameEnum(){ return 593;}
function MassconaxpbyDefinitionenumEnum(){ return 594;}
function MassconaxpbyNamexEnum(){ return 595;}
function MassconaxpbyNameyEnum(){ return 596;}
function MassconaxpbyAlphaEnum(){ return 597;}
function MassconaxpbyBetaEnum(){ return 598;}
function NodeSIdEnum(){ return 599;}
function VectorParamEnum(){ return 600;}
function RiftfrontEnum(){ return 601;}
function RiftfrontTypeEnum(){ return 602;}
function SegmentEnum(){ return 603;}
function SegmentRiftfrontEnum(){ return 604;}
function SpcDynamicEnum(){ return 605;}
function SpcStaticEnum(){ return 606;}
function SpcTransientEnum(){ return 607;}
function StringArrayParamEnum(){ return 608;}
function StringParamEnum(){ return 609;}
function SegEnum(){ return 610;}
function SegInputEnum(){ return 611;}
function TriaEnum(){ return 612;}
function TriaInputEnum(){ return 613;}
function TetraEnum(){ return 614;}
function TetraInputEnum(){ return 615;}
function PentaEnum(){ return 616;}
function PentaInputEnum(){ return 617;}
function VertexEnum(){ return 618;}
function VertexPIdEnum(){ return 619;}
function VertexSIdEnum(){ return 620;}
function AirEnum(){ return 621;}
function IceEnum(){ return 622;}
function MelangeEnum(){ return 623;}
function WaterEnum(){ return 624;}
function ClosedEnum(){ return 625;}
function FreeEnum(){ return 626;}
function OpenEnum(){ return 627;}
function AdjointpEnum(){ return 628;}
function AdjointxEnum(){ return 629;}
function AdjointyEnum(){ return 630;}
function AdjointzEnum(){ return 631;}
function BalancethicknessMisfitEnum(){ return 632;}
function BedSlopeXEnum(){ return 633;}
function BedSlopeYEnum(){ return 634;}
function BoundaryEnum(){ return 635;}
function ConvergedEnum(){ return 636;}
function FillEnum(){ return 637;}
function FractionIncrementEnum(){ return 638;}
function FrictionEnum(){ return 639;}
function InternalEnum(){ return 640;}
function MassFluxEnum(){ return 641;}
function MeltingOffsetEnum(){ return 642;}
function MisfitEnum(){ return 643;}
function PressureEnum(){ return 644;}
function PressurePicardEnum(){ return 645;}
function AndroidFrictionCoefficientEnum(){ return 646;}
function ResetPenaltiesEnum(){ return 647;}
function SegmentOnIceShelfEnum(){ return 648;}
function SurfaceAbsVelMisfitEnum(){ return 649;}
function SurfaceAreaEnum(){ return 650;}
function SurfaceAverageVelMisfitEnum(){ return 651;}
function SurfaceLogVelMisfitEnum(){ return 652;}
function SurfaceLogVxVyMisfitEnum(){ return 653;}
function SurfaceRelVelMisfitEnum(){ return 654;}
function SurfaceSlopeXEnum(){ return 655;}
function SurfaceSlopeYEnum(){ return 656;}
function TemperatureEnum(){ return 657;}
function TemperaturePicardEnum(){ return 658;}
function TemperaturePDDEnum(){ return 659;}
function ThicknessAbsMisfitEnum(){ return 660;}
function SurfaceAbsMisfitEnum(){ return 661;}
function VelEnum(){ return 662;}
function VelocityEnum(){ return 663;}
function VxAverageEnum(){ return 664;}
function VxEnum(){ return 665;}
function VxPicardEnum(){ return 666;}
function VyAverageEnum(){ return 667;}
function VyEnum(){ return 668;}
function VyPicardEnum(){ return 669;}
function VzEnum(){ return 670;}
function VzSSAEnum(){ return 671;}
function VzHOEnum(){ return 672;}
function VzPicardEnum(){ return 673;}
function VzFSEnum(){ return 674;}
function VxMeshEnum(){ return 675;}
function VyMeshEnum(){ return 676;}
function VzMeshEnum(){ return 677;}
function EnthalpyEnum(){ return 678;}
function EnthalpyPicardEnum(){ return 679;}
function ThicknessAbsGradientEnum(){ return 680;}
function ThicknessAlongGradientEnum(){ return 681;}
function ThicknessAcrossGradientEnum(){ return 682;}
function ThicknessPositiveEnum(){ return 683;}
function IntMatParamEnum(){ return 684;}
function RheologyBbarAbsGradientEnum(){ return 685;}
function RheologyBAbsGradientEnum(){ return 686;}
function DragCoefficientAbsGradientEnum(){ return 687;}
function TransientInputEnum(){ return 688;}
function WaterfractionEnum(){ return 689;}
function WatercolumnEnum(){ return 690;}
function BasalFrictionEnum(){ return 691;}
function ViscousHeatingEnum(){ return 692;}
function HydrologyWaterVxEnum(){ return 693;}
function HydrologyWaterVyEnum(){ return 694;}
function DrivingStressXEnum(){ return 695;}
function DrivingStressYEnum(){ return 696;}
function SigmaNNEnum(){ return 697;}
function StressTensorEnum(){ return 698;}
function StressTensorxxEnum(){ return 699;}
function StressTensorxyEnum(){ return 700;}
function StressTensorxzEnum(){ return 701;}
function StressTensoryyEnum(){ return 702;}
function StressTensoryzEnum(){ return 703;}
function StressTensorzzEnum(){ return 704;}
function StressMaxPrincipalEnum(){ return 705;}
function DeviatoricStressEnum(){ return 706;}
function DeviatoricStressxxEnum(){ return 707;}
function DeviatoricStressxyEnum(){ return 708;}
function DeviatoricStressxzEnum(){ return 709;}
function DeviatoricStressyyEnum(){ return 710;}
function DeviatoricStressyzEnum(){ return 711;}
function DeviatoricStresszzEnum(){ return 712;}
function DeviatoricStresseffectiveEnum(){ return 713;}
function StrainRateEnum(){ return 714;}
function StrainRatexxEnum(){ return 715;}
function StrainRatexyEnum(){ return 716;}
function StrainRatexzEnum(){ return 717;}
function StrainRateyyEnum(){ return 718;}
function StrainRateyzEnum(){ return 719;}
function StrainRatezzEnum(){ return 720;}
function DivergenceEnum(){ return 721;}
function MaxDivergenceEnum(){ return 722;}
function GiaCrossSectionShapeEnum(){ return 723;}
function GiadWdtEnum(){ return 724;}
function GiaWEnum(){ return 725;}
function P0Enum(){ return 726;}
function P0ArrayEnum(){ return 727;}
function P1Enum(){ return 728;}
function P1DGEnum(){ return 729;}
function P1bubbleEnum(){ return 730;}
function P1bubblecondensedEnum(){ return 731;}
function P2Enum(){ return 732;}
function P2bubbleEnum(){ return 733;}
function P2bubblecondensedEnum(){ return 734;}
function P2xP1Enum(){ return 735;}
function P1xP2Enum(){ return 736;}
function P1xP3Enum(){ return 737;}
function P2xP4Enum(){ return 738;}
function P1P1Enum(){ return 739;}
function P1P1GLSEnum(){ return 740;}
function MINIEnum(){ return 741;}
function MINIcondensedEnum(){ return 742;}
function TaylorHoodEnum(){ return 743;}
function LATaylorHoodEnum(){ return 744;}
function XTaylorHoodEnum(){ return 745;}
function OneLayerP4zEnum(){ return 746;}
function CrouzeixRaviartEnum(){ return 747;}
function LACrouzeixRaviartEnum(){ return 748;}
function SaveResultsEnum(){ return 749;}
function BoolExternalResultEnum(){ return 750;}
function DoubleExternalResultEnum(){ return 751;}
function DoubleMatExternalResultEnum(){ return 752;}
function IntExternalResultEnum(){ return 753;}
function JEnum(){ return 754;}
function StringExternalResultEnum(){ return 755;}
function StepEnum(){ return 756;}
function TimeEnum(){ return 757;}
function WaterColumnOldEnum(){ return 758;}
function OutputdefinitionEnum(){ return 759;}
function Outputdefinition1Enum(){ return 760;}
function Outputdefinition2Enum(){ return 761;}
function Outputdefinition3Enum(){ return 762;}
function Outputdefinition4Enum(){ return 763;}
function Outputdefinition5Enum(){ return 764;}
function Outputdefinition6Enum(){ return 765;}
function Outputdefinition7Enum(){ return 766;}
function Outputdefinition8Enum(){ return 767;}
function Outputdefinition9Enum(){ return 768;}
function Outputdefinition10Enum(){ return 769;}
function Outputdefinition11Enum(){ return 770;}
function Outputdefinition12Enum(){ return 771;}
function Outputdefinition13Enum(){ return 772;}
function Outputdefinition14Enum(){ return 773;}
function Outputdefinition15Enum(){ return 774;}
function Outputdefinition16Enum(){ return 775;}
function Outputdefinition17Enum(){ return 776;}
function Outputdefinition18Enum(){ return 777;}
function Outputdefinition19Enum(){ return 778;}
function Outputdefinition20Enum(){ return 779;}
function Outputdefinition21Enum(){ return 780;}
function Outputdefinition22Enum(){ return 781;}
function Outputdefinition23Enum(){ return 782;}
function Outputdefinition24Enum(){ return 783;}
function Outputdefinition25Enum(){ return 784;}
function Outputdefinition26Enum(){ return 785;}
function Outputdefinition27Enum(){ return 786;}
function Outputdefinition28Enum(){ return 787;}
function Outputdefinition29Enum(){ return 788;}
function Outputdefinition30Enum(){ return 789;}
function Outputdefinition31Enum(){ return 790;}
function Outputdefinition32Enum(){ return 791;}
function Outputdefinition33Enum(){ return 792;}
function Outputdefinition34Enum(){ return 793;}
function Outputdefinition35Enum(){ return 794;}
function Outputdefinition36Enum(){ return 795;}
function Outputdefinition37Enum(){ return 796;}
function Outputdefinition38Enum(){ return 797;}
function Outputdefinition39Enum(){ return 798;}
function Outputdefinition40Enum(){ return 799;}
function Outputdefinition41Enum(){ return 800;}
function Outputdefinition42Enum(){ return 801;}
function Outputdefinition43Enum(){ return 802;}
function Outputdefinition44Enum(){ return 803;}
function Outputdefinition45Enum(){ return 804;}
function Outputdefinition46Enum(){ return 805;}
function Outputdefinition47Enum(){ return 806;}
function Outputdefinition48Enum(){ return 807;}
function Outputdefinition49Enum(){ return 808;}
function Outputdefinition50Enum(){ return 809;}
function Outputdefinition51Enum(){ return 810;}
function Outputdefinition52Enum(){ return 811;}
function Outputdefinition53Enum(){ return 812;}
function Outputdefinition54Enum(){ return 813;}
function Outputdefinition55Enum(){ return 814;}
function Outputdefinition56Enum(){ return 815;}
function Outputdefinition57Enum(){ return 816;}
function Outputdefinition58Enum(){ return 817;}
function Outputdefinition59Enum(){ return 818;}
function Outputdefinition60Enum(){ return 819;}
function Outputdefinition61Enum(){ return 820;}
function Outputdefinition62Enum(){ return 821;}
function Outputdefinition63Enum(){ return 822;}
function Outputdefinition64Enum(){ return 823;}
function Outputdefinition65Enum(){ return 824;}
function Outputdefinition66Enum(){ return 825;}
function Outputdefinition67Enum(){ return 826;}
function Outputdefinition68Enum(){ return 827;}
function Outputdefinition69Enum(){ return 828;}
function Outputdefinition70Enum(){ return 829;}
function Outputdefinition71Enum(){ return 830;}
function Outputdefinition72Enum(){ return 831;}
function Outputdefinition73Enum(){ return 832;}
function Outputdefinition74Enum(){ return 833;}
function Outputdefinition75Enum(){ return 834;}
function Outputdefinition76Enum(){ return 835;}
function Outputdefinition77Enum(){ return 836;}
function Outputdefinition78Enum(){ return 837;}
function Outputdefinition79Enum(){ return 838;}
function Outputdefinition80Enum(){ return 839;}
function Outputdefinition81Enum(){ return 840;}
function Outputdefinition82Enum(){ return 841;}
function Outputdefinition83Enum(){ return 842;}
function Outputdefinition84Enum(){ return 843;}
function Outputdefinition85Enum(){ return 844;}
function Outputdefinition86Enum(){ return 845;}
function Outputdefinition87Enum(){ return 846;}
function Outputdefinition88Enum(){ return 847;}
function Outputdefinition89Enum(){ return 848;}
function Outputdefinition90Enum(){ return 849;}
function Outputdefinition91Enum(){ return 850;}
function Outputdefinition92Enum(){ return 851;}
function Outputdefinition93Enum(){ return 852;}
function Outputdefinition94Enum(){ return 853;}
function Outputdefinition95Enum(){ return 854;}
function Outputdefinition96Enum(){ return 855;}
function Outputdefinition97Enum(){ return 856;}
function Outputdefinition98Enum(){ return 857;}
function Outputdefinition99Enum(){ return 858;}
function Outputdefinition100Enum(){ return 859;}
function OutputdefinitionListEnum(){ return 860;}
function MassfluxatgateEnum(){ return 861;}
function MassfluxatgateNameEnum(){ return 862;}
function MassfluxatgateDefinitionenumEnum(){ return 863;}
function MassfluxatgateSegmentsEnum(){ return 864;}
function NodalvalueEnum(){ return 865;}
function NodalvalueNameEnum(){ return 866;}
function NodalvalueDefinitionenumEnum(){ return 867;}
function NodalvalueModelEnum(){ return 868;}
function NodalvalueNodeEnum(){ return 869;}
function MisfitNameEnum(){ return 870;}
function MisfitDefinitionenumEnum(){ return 871;}
function MisfitModelEnum(){ return 872;}
function MisfitObservationEnum(){ return 873;}
function MisfitObservationEnum(){ return 874;}
function MisfitLocalEnum(){ return 875;}
function MisfitTimeinterpolationEnum(){ return 876;}
function MisfitWeightsEnum(){ return 877;}
function MisfitWeightsEnum(){ return 878;}
function SurfaceObservationEnum(){ return 879;}
function WeightsSurfaceObservationEnum(){ return 880;}
function VxObsEnum(){ return 881;}
function WeightsVxObsEnum(){ return 882;}
function VyObsEnum(){ return 883;}
function WeightsVyObsEnum(){ return 884;}
function MinVelEnum(){ return 885;}
function MaxVelEnum(){ return 886;}
function MinVxEnum(){ return 887;}
function MaxVxEnum(){ return 888;}
function MaxAbsVxEnum(){ return 889;}
function MinVyEnum(){ return 890;}
function MaxVyEnum(){ return 891;}
function MaxAbsVyEnum(){ return 892;}
function MinVzEnum(){ return 893;}
function MaxVzEnum(){ return 894;}
function MaxAbsVzEnum(){ return 895;}
function FloatingAreaEnum(){ return 896;}
function GroundedAreaEnum(){ return 897;}
function IceMassEnum(){ return 898;}
function IceVolumeEnum(){ return 899;}
function IceVolumeAboveFloatationEnum(){ return 900;}
function TotalFloatingBmbEnum(){ return 901;}
function TotalGroundedBmbEnum(){ return 902;}
function TotalSmbEnum(){ return 903;}
function AbsoluteEnum(){ return 904;}
function IncrementalEnum(){ return 905;}
function AugmentedLagrangianREnum(){ return 906;}
function AugmentedLagrangianRhopEnum(){ return 907;}
function AugmentedLagrangianRlambdaEnum(){ return 908;}
function AugmentedLagrangianRholambdaEnum(){ return 909;}
function AugmentedLagrangianThetaEnum(){ return 910;}
function NoneEnum(){ return 911;}
function AggressiveMigrationEnum(){ return 912;}
function SoftMigrationEnum(){ return 913;}
function SubelementMigrationEnum(){ return 914;}
function SubelementMigration2Enum(){ return 915;}
function ContactEnum(){ return 916;}
function GroundingOnlyEnum(){ return 917;}
function MaskGroundediceLevelsetEnum(){ return 918;}
function GaussSegEnum(){ return 919;}
function GaussTriaEnum(){ return 920;}
function GaussTetraEnum(){ return 921;}
function GaussPentaEnum(){ return 922;}
function FSSolverEnum(){ return 923;}
function AdjointEnum(){ return 924;}
function ColinearEnum(){ return 925;}
function ControlSteadyEnum(){ return 926;}
function FsetEnum(){ return 927;}
function Gradient1Enum(){ return 928;}
function Gradient2Enum(){ return 929;}
function Gradient3Enum(){ return 930;}
function GradientEnum(){ return 931;}
function GroundinglineMigrationEnum(){ return 932;}
function GsetEnum(){ return 933;}
function IndexEnum(){ return 934;}
function IndexedEnum(){ return 935;}
function IntersectEnum(){ return 936;}
function NodalEnum(){ return 937;}
function OldGradientEnum(){ return 938;}
function OutputBufferPointerEnum(){ return 939;}
function OutputBufferSizePointerEnum(){ return 940;}
function OutputFilePointerEnum(){ return 941;}
function ToolkitsFileNameEnum(){ return 942;}
function RootPathEnum(){ return 943;}
function OutputFileNameEnum(){ return 944;}
function InputFileNameEnum(){ return 945;}
function LockFileNameEnum(){ return 946;}
function RestartFileNameEnum(){ return 947;}
function ToolkitsOptionsAnalysesEnum(){ return 948;}
function ToolkitsOptionsStringsEnum(){ return 949;}
function QmuErrNameEnum(){ return 950;}
function QmuInNameEnum(){ return 951;}
function QmuOutNameEnum(){ return 952;}
function RegularEnum(){ return 953;}
function ScaledEnum(){ return 954;}
function SeparateEnum(){ return 955;}
function SsetEnum(){ return 956;}
function VerboseEnum(){ return 957;}
function TriangleInterpEnum(){ return 958;}
function BilinearInterpEnum(){ return 959;}
function NearestInterpEnum(){ return 960;}
function XYEnum(){ return 961;}
function XYZEnum(){ return 962;}
function DenseEnum(){ return 963;}
function MpiDenseEnum(){ return 964;}
function MpiSparseEnum(){ return 965;}
function SeqEnum(){ return 966;}
function MpiEnum(){ return 967;}
function MumpsEnum(){ return 968;}
function GslEnum(){ return 969;}
function OptionEnum(){ return 970;}
function GenericOptionEnum(){ return 971;}
function OptionCellEnum(){ return 972;}
function OptionStructEnum(){ return 973;}
function CuffeyEnum(){ return 974;}
function PatersonEnum(){ return 975;}
function ArrheniusEnum(){ return 976;}
function LliboutryDuvalEnum(){ return 977;}
function SpclevelsetEnum(){ return 978;}
function ExtrapolationVariableEnum(){ return 979;}
function IceMaskNodeActivationEnum(){ return 980;}
function LevelsetfunctionSlopeXEnum(){ return 981;}
function LevelsetfunctionSlopeYEnum(){ return 982;}
function LevelsetfunctionPicardEnum(){ return 983;}
function LevelsetReinitFrequencyEnum(){ return 984;}
function SealevelriseSolutionEnum(){ return 985;}
function SealevelriseAnalysisEnum(){ return 986;}
function SealevelEnum(){ return 987;}
function SealevelEustaticEnum(){ return 988;}
function SealevelriseDeltathicknessEnum(){ return 989;}
function SealevelriseMaxiterEnum(){ return 990;}
function SealevelriseReltolEnum(){ return 991;}
function SealevelriseAbstolEnum(){ return 992;}
function SealevelriseLoveHEnum(){ return 993;}
function SealevelriseLoveKEnum(){ return 994;}
function SealevelriseTideLoveHEnum(){ return 995;}
function SealevelriseTideLoveKEnum(){ return 996;}
function SealevelriseRigidEnum(){ return 997;}
function SealevelriseElasticEnum(){ return 998;}
function SealevelriseRotationEnum(){ return 999;}
function SealevelriseGElasticEnum(){ return 1000;}
function SealevelriseDegaccEnum(){ return 1001;}
function SealevelriseTransitionsEnum(){ return 1002;}
function SealevelriseRequestedOutputsEnum(){ return 1003;}
function SealevelriseNumRequestedOutputsEnum(){ return 1004;}
function MaximumNumberOfDefinitionsEnum(){ return 1005;}
function FlagElements(md,region){
//FLAGELEMENTS - flag the elements in an region
//
//   The region can be given as a string, or as a javascript array
//
//   Usage: 
//      flag=FlagElements(md,region);
//
//   Example:
//      flag=FlagElements(md,'all');
//      flag=FlagElements(md,'');
//      flag=FlagElements(md,domain);

	//variables
	var flag;
	
	if (typeof region == 'string'){
		if (region === ''){
			flag=NewArrayFill(md.mesh.numberofelements,0);
		}
		else if (region === 'all'){
			flag=NewArrayFill(md.mesh.numberofelements,1);
		}
		else{
			flag=ContourToMesh(md.mesh.elements,md.mesh.x,md.mesh.y,region,'element',1);
		}
	}
	else if(IsArray(region)){
		if (region.length==md.mesh.numberofelements){
			flag=region;
		}
		else if (region.length==md.mesh.numberofvertices){
			var flag=NewArrayFill(md.mesh.numberofelements,0);
			for (var i=0;i<md.mesh.numberofelements;i++)
				var sum=0;
				for(var j=0;j<md.mesh.elements[0].length;j++){
					sum += region[md.mesh.element[i][j]-1];
				}
				if (sum==md.mesh.elements[0].length)flag[i]=1;
		}
		else{
			throw Error('Flaglist for region must be of same size as number of elements in model');
		}
	}
	else{
		throw Error('Invalid region option');
	}
	return flag;
}
function  marshallcostfunctions(cost_functions){
	for(var i=0;i<cost_functions.length;i++){
		if(cost_functions[i]==101) data[i]=SurfaceAbsVelMisfitEnum();
		if(cost_functions[i]==102) data[i]=SurfaceRelVelMisfitEnum();
		if(cost_functions[i]==103) data[i]=SurfaceLogVelMisfitEnum();
		if(cost_functions[i]==104) data[i]=SurfaceLogVxVyMisfitEnum();
		if(cost_functions[i]==105) data[i]=SurfaceAverageVelMisfitEnum();
		if(cost_functions[i]==201) data[i]=ThicknessAbsMisfitEnum();
		if(cost_functions[i]==501) data[i]=DragCoefficientAbsGradientEnum();
		if(cost_functions[i]==502) data[i]=RheologyBbarAbsGradientEnum();
		if(cost_functions[i]==503) data[i]=ThicknessAbsGradientEnum();
		if(cost_functions[i]==504) data[i]=ThicknessAlongGradientEnum();
		if(cost_functions[i]==505) data[i]=ThicknessAcrossGradientEnum();
		if(cost_functions[i]==506) data[i]=BalancethicknessMisfitEnum();
		if(cost_functions[i]==507) data[i]=RheologyBAbsGradientEnum();
		if(cost_functions[i]==601) data[i]=SurfaceAbsMisfitEnum();
	}
	return data;
}
//FILEPTR class definition
////
//// Usage: 
//// var fid = new fileptr(); 
//
//

function fileptr() {
	//properties
	this.increment=NaN;
	this.buffer =NaN;
	this.view =NaN;
	this.ptr =NaN;
	this.buffersize =NaN;
	this.mode ='';
	this.options = new pairoptions(Array.prototype.slice.call(arguments));
	
	//methods
		this.disp = function () { //{{{
			console.log(sprintf("   fileptr:")); 

			console.log(sprintf("       buffer: ArrayBuffer{ byteLength: %i }\n",this.buffer.byteLength));
			console.log(sprintf("       ptr: %i\n",this.ptr));
			console.log(sprintf("       increment: %i\n",this.increment));
			console.log(sprintf("       mode: %s\n",this.mode));

		} //}}}
		this.setdefaultparameters = function (options) { //{{{
	
			this.mode=options.getfieldvalue('mode');
			this.ptr=0;
			this.increment=0;
			this.buffersize=0;
			if (this.mode=='w'){
				this.increment=options.getfieldvalue('increment',800000); //80000 bytes,  10000 doubles.
				this.buffer=new ArrayBuffer(this.increment);
				this.view=new DataView(this.buffer);
			}
			else if(this.mode == 'r'){
				
				/*recover buffer and its size: */
				var bufferin= options.getfieldvalue('buffer');
				this.buffersize= options.getfieldvalue('buffersize');
				
				/*crete a typed array buffer: */
				this.buffer=new ArrayBuffer(this.buffersize);
				this.view=new DataView(this.buffer); 
				for(var i=0;i<this.buffersize;i++) this.view.setUint8(i,bufferin[i]);
			}

		} //}}}
		this.fwrite = function (value,format) { //{{{

			
			if(format == 'int'){
				if(this.ptr+4>=this.buffer.byteLength)this.resize();
				this.view.setUint32(this.ptr,value,true); this.ptr+=4;
			}
			else if(format == 'char'){
				if(this.ptr+value.length>=this.buffer.byteLength)this.resize();
				for(var i=0;i<value.length;i++){
					this.view.setUint8(this.ptr,value.charCodeAt(i),true); 
					this.ptr+=1;
				}
			}
			else if(format == 'double'){
				if(this.ptr+8>=this.buffer.byteLength)this.resize();
				if (!IsArray(value)){
					this.view.setFloat64(this.ptr,value,true);
					this.ptr+=8;
				}
				else{
					if (!IsArray(value[0])){
						if(this.ptr+value.length*8>=this.buffer.byteLength){
							this.resize();
							if(this.ptr+value.length*8>=this.buffer.byteLength)throw Error('fileptr.fwrite error: need to increase increment size!');
						}
						for(var i=0;i<value.length;i++){
							this.view.setFloat64(this.ptr,value[i],true);
							this.ptr+=8;
						}
					}
					else{
						if(this.ptr+value.length*value[0].length*8>=this.buffer.byteLength)this.resize();
						for(var i=0;i<value.length;i++){
							for(var j=0;j<value[0].length;j++){
								this.view.setFloat64(this.ptr,value[i][j],true);
								this.ptr+=8;
							}
						}
					}
				}
			}
			else throw Error('fileptr.fwrite error message: wrong type of format');
		} //}}}
		this.fread = function (size,format) { //{{{
			
			var value;

			if(this.ptr==this.buffersize)return -1;
			if(format == 'int'){
				if(size==1){
					value=this.view.getInt32(this.ptr,true); 
					this.ptr+=4;
				}
				else{
					value = new Int32Array(size);
					for(var i=0;i<size;i++){
						value[i]=this.view.getInt32(this.ptr,true); 
						this.ptr+=4;
					}
				}
			}
			else if(format == 'char'){
				value = ''; 
				for(var i=0;i<(size-1);i++){
					value+= String.fromCharCode(this.view.getUint8(this.ptr,true));
					this.ptr+=1;
				}
				this.ptr+=1; //pass over the '\0';

			}
			else if(format == 'double'){
				if(size==1){
					value=this.view.getFloat64(this.ptr,true);
					this.ptr+=8;
				}
				else{ 
					value = new Float64Array(size);
					for(var i=0;i<size;i++){
						value[i]=this.view.getFloat64(this.ptr,true);
						this.ptr+=8;
					}
				}
			}
			else throw Error('fileptr.fwrite error message: wrong type of format');
			
			return value;
		} //}}}
		this.rawbuffer = function () { //{{{
			return this.buffer.slice(0,this.ptr);
		} //}}}
		this.resize = function () { //{{{
			var  newbuffer = new ArrayBuffer(this.buffer.byteLength+this.increment);
			new Uint8Array(newbuffer).set(new Uint8Array(this.buffer));
			this.buffer=newbuffer;
			this.view=new DataView(this.buffer);
		} //}}}
	//set defaults
	this.setdefaultparameters(this.options);
}
function savemodel(md) {

	var string=encodeURI(JSONfn.stringify(md));

	var url='data:text/json:charset=utf8,' + encodeURIComponent(string);
	window.open(url, '_blank');
	window.focus();
}
function loadmodel(modelstring) {

	var md=JSONfn.parse(decodeURI(modelstring));
	md.fix();
	return md;
}
function saveAsFile(content) {

	var url='data:text/json:charset=utf8,' + encodeURIComponent(content);
	window.open(url, '_blank');
	window.focus();
}
function paterson(temperature){
//PATERSON - figure out the rigidity of ice for a given temperature
//
//   rigidity (in s^(1/3)Pa) is the flow law paramter in the flow law sigma=B*e(1/3) (Paterson, p97). 
//   temperature is in Kelvin degrees
//
//   Usage:
//      var rigidity=paterson(temperature)

	//variables:
	var T=[];

	if (ArrayAnyBelowStrict(temperature,0)){
		throw Error('input temperature should be in Kelvin (positive)');
	}
	
	T=temperature;
	for(var i=0;i<temperature.length;i++)T[i]=temperature[i]-273.15;

	//The routine below is equivalent to:

	// n=3; T=temperature-273;
	// //From paterson,
	// Temp=[0;-2;-5;-10;-15;-20;-25;-30;-35;-40;-45;-50];
	// A=[6.8*10^-15;2.4*10^-15;1.6*10^-15;4.9*10^-16;2.9*10^-16;1.7*10^-16;9.4*
	// 10^-17;5.1*10^-17;2.7*10^-17;1.4*10^-17;7.3*10^-18;3.6*10^-18];;//s-1(kPa-3)
	// //Convert into rigidity B
	// B=A.^(-1/n)*10^3; //s^(1/3)Pa
	// //Now, do a cubic fit between Temp and B: 
	// fittedmodel=fit(Temp,B,'cubicspline');
	// rigidity=fittedmodel(temperature);

	var rigidity=NewArrayFill(T.length,0);
	
	for (var i=0;i<T.length;i++){
		
		if(T[i]<=-45)              rigidity[i]=Math.pow(10,8)*(-0.000292866376675*Math.pow(T[i]+50,3)+ 0.011672640664130*Math.pow(T[i]+50,2)  -0.325004442485481*(T[i]+50)+  6.524779401948101);
		if(-45<=T[i] & T[i]<-40)   rigidity[i]=Math.pow(10,8)*(-0.000292866376675*Math.pow(T[i]+45,3)+ 0.007279645014004*Math.pow(T[i]+45,2)  -0.230243014094813*(T[i]+45)+  5.154964909039554);
		if(-40<=T[i] & T[i]<-35)   rigidity[i]=Math.pow(10,8)*(0.000072737147457*Math.pow(T[i]+40,3)+  0.002886649363879*Math.pow(T[i]+40,2)  -0.179411542205399*(T[i]+40)+  4.149132666831214);
		if(-35<=T[i] & T[i]<-30)   rigidity[i]=Math.pow(10,8)*(-0.000086144770023*Math.pow(T[i]+35,3)+ 0.003977706575736*Math.pow(T[i]+35,2)  -0.145089762507325*(T[i]+35)+  3.333333333333331);
		if(-30<=T[i] & T[i]<-25)   rigidity[i]=Math.pow(10,8)*(-0.000043984685769*Math.pow(T[i]+30,3)+ 0.002685535025386*Math.pow(T[i]+30,2)  -0.111773554501713*(T[i]+30)+  2.696559088937191);
		if(-25<=T[i] & T[i]<-20)   rigidity[i]=Math.pow(10,8)*(-0.000029799523463*Math.pow(T[i]+25,3)+ 0.002025764738854*Math.pow(T[i]+25,2)  -0.088217055680511*(T[i]+25)+  2.199331606342181);
		if(-20<=T[i] & T[i]<-15)   rigidity[i]=Math.pow(10,8)*(0.000136920904777*Math.pow(T[i]+20,3)+  0.001578771886910*Math.pow(T[i]+20,2)  -0.070194372551690*(T[i]+20)+  1.805165505978111);
		if(-15<=T[i] & T[i]<-10)   rigidity[i]=Math.pow(10,8)*(-0.000899763781026*Math.pow(T[i]+15,3)+ 0.003632585458564*Math.pow(T[i]+15,2)  -0.044137585824322*(T[i]+15)+  1.510778053489523);
		if(-10<=T[i] & T[i]<-5)    rigidity[i]=Math.pow(10,8)*(0.001676964325070*Math.pow(T[i]+10,3)-  0.009863871256831*Math.pow(T[i]+10,2)  -0.075294014815659*(T[i]+10)+  1.268434288203714);
		if(-5<=T[i] & T[i]<-2)     rigidity[i]=Math.pow(10,8)*(-0.003748937622487*Math.pow(T[i]+5,3)+0.015290593619213*Math.pow(T[i]+5,2)  -0.048160403003748*(T[i]+5)+  0.854987973338348);
		if(-2<=T[i])              rigidity[i]=Math.pow(10,8)*(-0.003748937622488*Math.pow(T[i]+2,3)-0.018449844983174*Math.pow(T[i]+2,2)  -0.057638157095631*(T[i]+2)+  0.746900791092860);

		//Now make sure that rigidity is positive
		if(rigidity[i]<0)          rigidity[i]=Math.pow(10,6);
	}
	return rigidity;
}
function triangle(md){
//TRIANGLE - create model mesh using the triangle package
//
//   This routine creates a model mesh using TriMesh and a domain outline, to within a certain resolution
//   where md is a @model object, domainname is the name of an Argus domain outline file, 
//   and resolution is a characteristic length for the mesh (same unit as the domain outline
//   unit). Riftname is an optional argument (Argus domain outline) describing rifts.
//
//   Usage:
//      triangle(md,domain,resolution)
//   or triangle(md,domain,riftname, resolution)
//
//   Examples:
//      triangle(md,domain,1000);
//      triangle(md,domain, rifts, 1500);

	if (!(arguments.length==3 | arguments.length==4)){
		console.log('triangle usage error.');
	}
	
	var md=arguments[0];
	var domain=arguments[1];

	if (arguments.length==3){
		var resolution=arguments[2];
		var rifts=[];
	}
	if (arguments.length==4){
		var rifts=arguments[2];
		var resolution=arguments[3];
	}

	//Figure out a characteristic area. Resolution is a node oriented concept (ex a 1000m  resolution node would 
	//be made of 1000*1000 area squares). 
	var area=Math.pow(resolution,2);

	//Call mesher: 
	var return_array=TriMesh(md, domain, rifts, area); 

	//Plug into md:
	md.mesh.elements=return_array[0];
	md.mesh.x=return_array[1];
	md.mesh.y=return_array[2];
	md.mesh.segments=return_array[3];
	md.mesh.segmentmarkers=return_array[4];
	
	//Fill in rest of fields:
	md.mesh.numberofelements=md.mesh.elements.length;
	md.mesh.numberofvertices=md.mesh.x.length;
	md.mesh.vertexonboundary=new Float64Array(md.mesh.numberofvertices); 

	for (i=0;i<md.mesh.segments.length;i++) for(var j=0;j<2;j++) md.mesh.vertexonboundary[md.mesh.segments[i][j]-1]=1;

	//Now, build the connectivity tables for this mesh.
	md.mesh.vertexconnectivity=NodeConnectivity(md.mesh.elements,md.mesh.numberofvertices);
	md.mesh.elementconnectivity=ElementConnectivity(md.mesh.elements,md.mesh.vertexconnectivity);	

}
var colorbars={};
colorbars["jet"]=[[0.000000,0.000000,0.750000],
[0.000000,0.000000,1.000000],
[0.000000,0.250000,1.000000],
[0.000000,0.500000,1.000000],
[0.000000,0.750000,1.000000],
[0.000000,1.000000,1.000000],
[0.250000,1.000000,0.750000],
[0.500000,1.000000,0.500000],
[0.750000,1.000000,0.250000],
[1.000000,1.000000,0.000000],
[1.000000,0.750000,0.000000],
[1.000000,0.500000,0.000000],
[1.000000,0.250000,0.000000],
[1.000000,0.000000,0.000000],
[0.750000,0.000000,0.000000],
[0.500000,0.000000,0.000000]];
colorbars["hsv"]=[[1.000000,0.000000,0.000000],
[1.000000,0.375000,0.000000],
[1.000000,0.750000,0.000000],
[0.875000,1.000000,0.000000],
[0.500000,1.000000,0.000000],
[0.125000,1.000000,0.000000],
[0.000000,1.000000,0.250000],
[0.000000,1.000000,0.625000],
[0.000000,1.000000,1.000000],
[0.000000,0.625000,1.000000],
[0.000000,0.250000,1.000000],
[0.125000,0.000000,1.000000],
[0.500000,0.000000,1.000000],
[0.875000,0.000000,1.000000],
[1.000000,0.000000,0.750000],
[1.000000,0.000000,0.375000]];
colorbars["hot"]=[[0.166667,0.000000,0.000000],
[0.333333,0.000000,0.000000],
[0.500000,0.000000,0.000000],
[0.666667,0.000000,0.000000],
[0.833333,0.000000,0.000000],
[1.000000,0.000000,0.000000],
[1.000000,0.166667,0.000000],
[1.000000,0.333333,0.000000],
[1.000000,0.500000,0.000000],
[1.000000,0.666667,0.000000],
[1.000000,0.833333,0.000000],
[1.000000,1.000000,0.000000],
[1.000000,1.000000,0.250000],
[1.000000,1.000000,0.500000],
[1.000000,1.000000,0.750000],
[1.000000,1.000000,1.000000]];
colorbars["gray"]=[[0.000000,0.000000,0.000000],
[0.066667,0.066667,0.066667],
[0.133333,0.133333,0.133333],
[0.200000,0.200000,0.200000],
[0.266667,0.266667,0.266667],
[0.333333,0.333333,0.333333],
[0.400000,0.400000,0.400000],
[0.466667,0.466667,0.466667],
[0.533333,0.533333,0.533333],
[0.600000,0.600000,0.600000],
[0.666667,0.666667,0.666667],
[0.733333,0.733333,0.733333],
[0.800000,0.800000,0.800000],
[0.866667,0.866667,0.866667],
[0.933333,0.933333,0.933333],
[1.000000,1.000000,1.000000]];
colorbars["bone"]=[[0.000000,0.000000,0.020833],
[0.058333,0.058333,0.100000],
[0.116667,0.116667,0.179167],
[0.175000,0.175000,0.258333],
[0.233333,0.233333,0.337500],
[0.291667,0.291667,0.416667],
[0.350000,0.370833,0.475000],
[0.408333,0.450000,0.533333],
[0.466667,0.529167,0.591667],
[0.525000,0.608333,0.650000],
[0.583333,0.687500,0.708333],
[0.641667,0.766667,0.766667],
[0.731250,0.825000,0.825000],
[0.820833,0.883333,0.883333],
[0.910417,0.941667,0.941667],
[1.000000,1.000000,1.000000]];
colorbars["copper"]=[[0.000000,0.000000,0.000000],
[0.083333,0.052080,0.033167],
[0.166667,0.104160,0.066333],
[0.250000,0.156240,0.099500],
[0.333333,0.208320,0.132667],
[0.416667,0.260400,0.165833],
[0.500000,0.312480,0.199000],
[0.583333,0.364560,0.232167],
[0.666667,0.416640,0.265333],
[0.750000,0.468720,0.298500],
[0.833333,0.520800,0.331667],
[0.916667,0.572880,0.364833],
[1.000000,0.624960,0.398000],
[1.000000,0.677040,0.431167],
[1.000000,0.729120,0.464333],
[1.000000,0.781200,0.497500]];
colorbars["pink"]=[[0.235702,0.000000,0.000000],
[0.394405,0.210819,0.210819],
[0.505525,0.298142,0.298142],
[0.596285,0.365148,0.365148],
[0.674949,0.421637,0.421637],
[0.745356,0.471405,0.471405],
[0.774597,0.567646,0.516398],
[0.802773,0.649786,0.557773],
[0.829993,0.722649,0.596285],
[0.856349,0.788811,0.632456],
[0.881917,0.849837,0.666667],
[0.906765,0.906765,0.699206],
[0.930949,0.930949,0.785281],
[0.954521,0.954521,0.862812],
[0.977525,0.977525,0.933928],
[1.000000,1.000000,1.000000]];
colorbars["white"]=[[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000],
[1.000000,1.000000,1.000000]];
colorbars["flag"]=[[1.000000,0.000000,0.000000],
[1.000000,1.000000,1.000000],
[0.000000,0.000000,1.000000],
[0.000000,0.000000,0.000000],
[1.000000,0.000000,0.000000],
[1.000000,1.000000,1.000000],
[0.000000,0.000000,1.000000],
[0.000000,0.000000,0.000000],
[1.000000,0.000000,0.000000],
[1.000000,1.000000,1.000000],
[0.000000,0.000000,1.000000],
[0.000000,0.000000,0.000000],
[1.000000,0.000000,0.000000],
[1.000000,1.000000,1.000000],
[0.000000,0.000000,1.000000],
[0.000000,0.000000,0.000000]];
colorbars["lines"]=[[0.000000,0.000000,1.000000],
[0.000000,0.500000,0.000000],
[1.000000,0.000000,0.000000],
[0.000000,0.750000,0.750000],
[0.750000,0.000000,0.750000],
[0.750000,0.750000,0.000000],
[0.250000,0.250000,0.250000],
[0.000000,0.000000,1.000000],
[0.000000,0.500000,0.000000],
[1.000000,0.000000,0.000000],
[0.000000,0.750000,0.750000],
[0.750000,0.000000,0.750000],
[0.750000,0.750000,0.000000],
[0.250000,0.250000,0.250000],
[0.000000,0.000000,1.000000],
[0.000000,0.500000,0.000000]];
colorbars["colorcube"]=[[1.000000,1.000000,0.000000],
[0.000000,1.000000,1.000000],
[1.000000,0.000000,1.000000],
[0.333333,0.000000,0.000000],
[0.666667,0.000000,0.000000],
[1.000000,0.000000,0.000000],
[0.000000,0.333333,0.000000],
[0.000000,0.666667,0.000000],
[0.000000,1.000000,0.000000],
[0.000000,0.000000,0.333333],
[0.000000,0.000000,0.666667],
[0.000000,0.000000,1.000000],
[0.000000,0.000000,0.000000],
[0.333333,0.333333,0.333333],
[0.666667,0.666667,0.666667],
[1.000000,1.000000,1.000000]];
colorbars["vga"]=[[1.000000,1.000000,1.000000],
[0.750000,0.750000,0.750000],
[1.000000,0.000000,0.000000],
[1.000000,1.000000,0.000000],
[0.000000,1.000000,0.000000],
[0.000000,1.000000,1.000000],
[0.000000,0.000000,1.000000],
[1.000000,0.000000,1.000000],
[0.000000,0.000000,0.000000],
[0.500000,0.500000,0.500000],
[0.500000,0.000000,0.000000],
[0.500000,0.500000,0.000000],
[0.000000,0.500000,0.000000],
[0.000000,0.500000,0.500000],
[0.000000,0.000000,0.500000],
[0.500000,0.000000,0.500000]];
colorbars["prism"]=[[1.000000,0.000000,0.000000],
[1.000000,0.500000,0.000000],
[1.000000,1.000000,0.000000],
[0.000000,1.000000,0.000000],
[0.000000,0.000000,1.000000],
[0.666667,0.000000,1.000000],
[1.000000,0.000000,0.000000],
[1.000000,0.500000,0.000000],
[1.000000,1.000000,0.000000],
[0.000000,1.000000,0.000000],
[0.000000,0.000000,1.000000],
[0.666667,0.000000,1.000000],
[1.000000,0.000000,0.000000],
[1.000000,0.500000,0.000000],
[1.000000,1.000000,0.000000],
[0.000000,1.000000,0.000000]];
colorbars["cool"]=[[0.000000,1.000000,1.000000],
[0.066667,0.933333,1.000000],
[0.133333,0.866667,1.000000],
[0.200000,0.800000,1.000000],
[0.266667,0.733333,1.000000],
[0.333333,0.666667,1.000000],
[0.400000,0.600000,1.000000],
[0.466667,0.533333,1.000000],
[0.533333,0.466667,1.000000],
[0.600000,0.400000,1.000000],
[0.666667,0.333333,1.000000],
[0.733333,0.266667,1.000000],
[0.800000,0.200000,1.000000],
[0.866667,0.133333,1.000000],
[0.933333,0.066667,1.000000],
[1.000000,0.000000,1.000000]];
colorbars["autumn"]=[[1.000000,0.000000,0.000000],
[1.000000,0.066667,0.000000],
[1.000000,0.133333,0.000000],
[1.000000,0.200000,0.000000],
[1.000000,0.266667,0.000000],
[1.000000,0.333333,0.000000],
[1.000000,0.400000,0.000000],
[1.000000,0.466667,0.000000],
[1.000000,0.533333,0.000000],
[1.000000,0.600000,0.000000],
[1.000000,0.666667,0.000000],
[1.000000,0.733333,0.000000],
[1.000000,0.800000,0.000000],
[1.000000,0.866667,0.000000],
[1.000000,0.933333,0.000000],
[1.000000,1.000000,0.000000]];
colorbars["spring"]=[[1.000000,0.000000,1.000000],
[1.000000,0.066667,0.933333],
[1.000000,0.133333,0.866667],
[1.000000,0.200000,0.800000],
[1.000000,0.266667,0.733333],
[1.000000,0.333333,0.666667],
[1.000000,0.400000,0.600000],
[1.000000,0.466667,0.533333],
[1.000000,0.533333,0.466667],
[1.000000,0.600000,0.400000],
[1.000000,0.666667,0.333333],
[1.000000,0.733333,0.266667],
[1.000000,0.800000,0.200000],
[1.000000,0.866667,0.133333],
[1.000000,0.933333,0.066667],
[1.000000,1.000000,0.000000]];
colorbars["winter"]=[[0.000000,0.000000,1.000000],
[0.000000,0.066667,0.966667],
[0.000000,0.133333,0.933333],
[0.000000,0.200000,0.900000],
[0.000000,0.266667,0.866667],
[0.000000,0.333333,0.833333],
[0.000000,0.400000,0.800000],
[0.000000,0.466667,0.766667],
[0.000000,0.533333,0.733333],
[0.000000,0.600000,0.700000],
[0.000000,0.666667,0.666667],
[0.000000,0.733333,0.633333],
[0.000000,0.800000,0.600000],
[0.000000,0.866667,0.566667],
[0.000000,0.933333,0.533333],
[0.000000,1.000000,0.500000]];
colorbars["summer"]=[[0.000000,0.500000,0.400000],
[0.066667,0.533333,0.400000],
[0.133333,0.566667,0.400000],
[0.200000,0.600000,0.400000],
[0.266667,0.633333,0.400000],
[0.333333,0.666667,0.400000],
[0.400000,0.700000,0.400000],
[0.466667,0.733333,0.400000],
[0.533333,0.766667,0.400000],
[0.600000,0.800000,0.400000],
[0.666667,0.833333,0.400000],
[0.733333,0.866667,0.400000],
[0.800000,0.900000,0.400000],
[0.866667,0.933333,0.400000],
[0.933333,0.966667,0.400000],
[1.000000,1.000000,0.400000]];
function fielddisplay(md,name,comment){
//FIELDDISPLAY - display model field
//
//   Usage:
//      fielddisplay(md,name,comment)

	//get field
	field=md[name];

	//disp corresponding line as a function of field type (offset set as 9 spaces)
	parsedisplay('         ',name,field,comment);
}

function parsedisplay(offset,name,field,comment) { //{{{

	//string
	if (typeof(field) == "string"){

		if (field.length > 30){
			displayunit(offset,name,"not displayed",comment);
		}
		else{
			displayunit(offset,name,"'"+field+"'",comment);
		}
	}
	//numeric
	else if (typeof(field) == "number"){
		
		displayunit(offset,name,sprintf("%g",field),comment);

	}
	//logical
	else if (typeof(field) == "boolean") {

		if (field){
			displayunit(offset,name,"true",comment);
		}
		else{
			displayunit(offset,name,"false",comment);
		}

	}
	//object
	else if (typeof(field) == "object"){

		if(field.length == 0) displayunit(offset,name,sprintf("(%i)",field.length),comment);
		else if ((field[0].length==0) | (typeof field[0].length =='undefined')){
			displayunit(offset,name,sprintf("(%i)",field.length),comment);
		}
		else{
			displayunit(offset,name,sprintf("(%i,%i)",field.length,field[0].length),comment);
		}

	}
	else{
		displayunit(offset,name,"not displayed",comment);
	}
} //}}}

function displayunit(offset,name,characterization,comment){ // {{{

	//take care of name
	if (name.length>23){
		name=name.slice(0,21) + "...";
	}

	//take care of characterization
	if ( characterization == "\" \"" || characterization == "NaN" ){
	
		characterization="N/A";
	}
	if (characterization.length>15){
		characterization=characterization.slice(0,13) + "...";
	}

	//print
	if (comment.length==0){
		console.log(sprintf("%s%-23s: %-15s",offset,name,characterization));
	}
	else{
		if (typeof(comment) == "string"){
			//console.log(sprintf("%s%-23s: %-15s -- %s",offset,name,characterization,comment));
			console.log(sprintf("%s%s: %-15s -- %s",offset,name,characterization,comment));
		}
		else{
			throw Error("fielddisplay error message: format for comment not supported yet");
		}
	}
} //}}}
/**
 * A class to parse color values
 * @author Stoyan Stefanov <sstoo@gmail.com>
 * @link   http://www.phpied.com/rgb-color-parser-in-javascript/
 * @license Use it if you like it
 */
function RGBColor(color_string)
{
    this.ok = false;

    // strip any leading #
    if (color_string.charAt(0) == '#') { // remove # if any
        color_string = color_string.substr(1,6);
    }

    color_string = color_string.replace(/ /g,'');
    color_string = color_string.toLowerCase();

    // before getting into regexps, try simple matches
    // and overwrite the input
    var simple_colors = {
        aliceblue: 'f0f8ff',
        antiquewhite: 'faebd7',
        aqua: '00ffff',
        aquamarine: '7fffd4',
        azure: 'f0ffff',
        beige: 'f5f5dc',
        bisque: 'ffe4c4',
        black: '000000',
        blanchedalmond: 'ffebcd',
        blue: '0000ff',
        blueviolet: '8a2be2',
        brown: 'a52a2a',
        burlywood: 'deb887',
        cadetblue: '5f9ea0',
        chartreuse: '7fff00',
        chocolate: 'd2691e',
        coral: 'ff7f50',
        cornflowerblue: '6495ed',
        cornsilk: 'fff8dc',
        crimson: 'dc143c',
        cyan: '00ffff',
        darkblue: '00008b',
        darkcyan: '008b8b',
        darkgoldenrod: 'b8860b',
        darkgray: 'a9a9a9',
        darkgreen: '006400',
        darkkhaki: 'bdb76b',
        darkmagenta: '8b008b',
        darkolivegreen: '556b2f',
        darkorange: 'ff8c00',
        darkorchid: '9932cc',
        darkred: '8b0000',
        darksalmon: 'e9967a',
        darkseagreen: '8fbc8f',
        darkslateblue: '483d8b',
        darkslategray: '2f4f4f',
        darkturquoise: '00ced1',
        darkviolet: '9400d3',
        deeppink: 'ff1493',
        deepskyblue: '00bfff',
        dimgray: '696969',
        dodgerblue: '1e90ff',
        feldspar: 'd19275',
        firebrick: 'b22222',
        floralwhite: 'fffaf0',
        forestgreen: '228b22',
        fuchsia: 'ff00ff',
        gainsboro: 'dcdcdc',
        ghostwhite: 'f8f8ff',
        gold: 'ffd700',
        goldenrod: 'daa520',
        gray: '808080',
        green: '008000',
        greenyellow: 'adff2f',
        honeydew: 'f0fff0',
        hotpink: 'ff69b4',
        indianred : 'cd5c5c',
        indigo : '4b0082',
        ivory: 'fffff0',
        khaki: 'f0e68c',
        lavender: 'e6e6fa',
        lavenderblush: 'fff0f5',
        lawngreen: '7cfc00',
        lemonchiffon: 'fffacd',
        lightblue: 'add8e6',
        lightcoral: 'f08080',
        lightcyan: 'e0ffff',
        lightgoldenrodyellow: 'fafad2',
        lightgrey: 'd3d3d3',
        lightgreen: '90ee90',
        lightpink: 'ffb6c1',
        lightsalmon: 'ffa07a',
        lightseagreen: '20b2aa',
        lightskyblue: '87cefa',
        lightslateblue: '8470ff',
        lightslategray: '778899',
        lightsteelblue: 'b0c4de',
        lightyellow: 'ffffe0',
        lime: '00ff00',
        limegreen: '32cd32',
        linen: 'faf0e6',
        magenta: 'ff00ff',
        maroon: '800000',
        mediumaquamarine: '66cdaa',
        mediumblue: '0000cd',
        mediumorchid: 'ba55d3',
        mediumpurple: '9370d8',
        mediumseagreen: '3cb371',
        mediumslateblue: '7b68ee',
        mediumspringgreen: '00fa9a',
        mediumturquoise: '48d1cc',
        mediumvioletred: 'c71585',
        midnightblue: '191970',
        mintcream: 'f5fffa',
        mistyrose: 'ffe4e1',
        moccasin: 'ffe4b5',
        navajowhite: 'ffdead',
        navy: '000080',
        oldlace: 'fdf5e6',
        olive: '808000',
        olivedrab: '6b8e23',
        orange: 'ffa500',
        orangered: 'ff4500',
        orchid: 'da70d6',
        palegoldenrod: 'eee8aa',
        palegreen: '98fb98',
        paleturquoise: 'afeeee',
        palevioletred: 'd87093',
        papayawhip: 'ffefd5',
        peachpuff: 'ffdab9',
        peru: 'cd853f',
        pink: 'ffc0cb',
        plum: 'dda0dd',
        powderblue: 'b0e0e6',
        purple: '800080',
        red: 'ff0000',
        rosybrown: 'bc8f8f',
        royalblue: '4169e1',
        saddlebrown: '8b4513',
        salmon: 'fa8072',
        sandybrown: 'f4a460',
        seagreen: '2e8b57',
        seashell: 'fff5ee',
        sienna: 'a0522d',
        silver: 'c0c0c0',
        skyblue: '87ceeb',
        slateblue: '6a5acd',
        slategray: '708090',
        snow: 'fffafa',
        springgreen: '00ff7f',
        steelblue: '4682b4',
        tan: 'd2b48c',
        teal: '008080',
        thistle: 'd8bfd8',
        tomato: 'ff6347',
        turquoise: '40e0d0',
        violet: 'ee82ee',
        violetred: 'd02090',
        wheat: 'f5deb3',
        white: 'ffffff',
        whitesmoke: 'f5f5f5',
        yellow: 'ffff00',
        yellowgreen: '9acd32'
    };
    for (var key in simple_colors) {
        if (color_string == key) {
            color_string = simple_colors[key];
        }
    }
    // emd of simple type-in colors

    // array of color definition objects
    var color_defs = [
        {
            re: /^rgb\((\d{1,3}),\s*(\d{1,3}),\s*(\d{1,3})\)$/,
            example: ['rgb(123, 234, 45)', 'rgb(255,234,245)'],
            process: function (bits){
                return [
                    parseInt(bits[1]),
                    parseInt(bits[2]),
                    parseInt(bits[3])
                ];
            }
        },
        {
            re: /^(\w{2})(\w{2})(\w{2})$/,
            example: ['#00ff00', '336699'],
            process: function (bits){
                return [
                    parseInt(bits[1], 16),
                    parseInt(bits[2], 16),
                    parseInt(bits[3], 16)
                ];
            }
        },
        {
            re: /^(\w{1})(\w{1})(\w{1})$/,
            example: ['#fb0', 'f0f'],
            process: function (bits){
                return [
                    parseInt(bits[1] + bits[1], 16),
                    parseInt(bits[2] + bits[2], 16),
                    parseInt(bits[3] + bits[3], 16)
                ];
            }
        }
    ];

    // search through the definitions to find a match
    for (var i = 0; i < color_defs.length; i++) {
        var re = color_defs[i].re;
        var processor = color_defs[i].process;
        var bits = re.exec(color_string);
        if (bits) {
            channels = processor(bits);
            this.r = channels[0];
            this.g = channels[1];
            this.b = channels[2];
            this.ok = true;
        }

    }

    // validate/cleanup values
    this.r = (this.r < 0 || isNaN(this.r)) ? 0 : ((this.r > 255) ? 255 : this.r);
    this.g = (this.g < 0 || isNaN(this.g)) ? 0 : ((this.g > 255) ? 255 : this.g);
    this.b = (this.b < 0 || isNaN(this.b)) ? 0 : ((this.b > 255) ? 255 : this.b);

    // some getters
    this.toRGB = function () {
        return 'rgb(' + this.r + ', ' + this.g + ', ' + this.b + ')';
    }
    this.toHex = function () {
        var r = this.r.toString(16);
        var g = this.g.toString(16);
        var b = this.b.toString(16);
        if (r.length == 1) r = '0' + r;
        if (g.length == 1) g = '0' + g;
        if (b.length == 1) b = '0' + b;
        return '#' + r + g + b;
    }

    // help
    this.getHelpXML = function () {

        var examples = new Array();
        // add regexps
        for (var i = 0; i < color_defs.length; i++) {
            var example = color_defs[i].example;
            for (var j = 0; j < example.length; j++) {
                examples[examples.length] = example[j];
            }
        }
        // add type-in colors
        for (var sc in simple_colors) {
            examples[examples.length] = sc;
        }

        var xml = document.createElement('ul');
        xml.setAttribute('id', 'rgbcolor-examples');
        for (var i = 0; i < examples.length; i++) {
            try {
                var list_item = document.createElement('li');
                var list_color = new RGBColor(examples[i]);
                var example_div = document.createElement('div');
                example_div.style.cssText =
                        'margin: 3px; '
                        + 'border: 1px solid black; '
                        + 'background:' + list_color.toHex() + '; '
                        + 'color:' + list_color.toHex()
                ;
                example_div.appendChild(document.createTextNode('test'));
                var list_item_value = document.createTextNode(
                    ' ' + examples[i] + ' -> ' + list_color.toRGB() + ' -> ' + list_color.toHex()
                );
                list_item.appendChild(example_div);
                list_item.appendChild(list_item_value);
                xml.appendChild(list_item);

            } catch(e){}
        }
        return xml;

    }

}

function setflowequation(md){
//SETFLOWEQUATION - associate a solution type to each element
//
//   This routine works like plotmodel: it works with an even number of inputs
//   'SIA','SSA','L1L2','HO','FS' and 'fill' are the possible options
//   that must be followed by the corresponding exp file or flags list
//   It can either be a domain file (argus type, .exp extension), or an array of element flags. 
//   If user wants every element outside the domain to be 
//   setflowequationd, add '~' to the name of the domain file (ex: '~HO.exp');
//   an empty string '' will be considered as an empty domain
//   a string 'all' will be considered as the entire domain
//   You can specify the type of coupling, 'penalties' or 'tiling', to use with the input 'coupling'
//   NB: L1L2 cannot currently be coupled to any other ice flow model
//
//   Usage:
//      setflowequation(md,varargin)
//
//   Example:
//      setflowequation(md,'HO',HO,'fill','SIA','coupling','tiling');

	//some checks on list of arguments
	if(arguments.length<3) throw Error('setflowequation error message');

	//Process options
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(1,args.length));
	options.deleteduplicates(1);

	//Find_out what kind of coupling to use
	coupling_method=options.getfieldvalue('coupling','tiling');
	if ((coupling_method != 'tiling') & !(coupling_method != 'penalties')){
		throw error('coupling type can only be: tiling or penalties');
	}

	//recover elements distribution
	SIAflag  = FlagElements(md,options.getfieldvalue('SIA',''));
	SSAflag  = FlagElements(md,options.getfieldvalue('SSA',''));
	HOflag   = FlagElements(md,options.getfieldvalue('HO',''));
	L1L2flag = FlagElements(md,options.getfieldvalue('L1L2',''));
	FSflag   = FlagElements(md,options.getfieldvalue('FS',''));
	filltype = options.getfieldvalue('fill','none');
	options.displayunused();

	//Flag the elements that have not been flagged as filltype
	if (filltype === 'SIA'){
		for(var i=0;i<md.mesh.numberofelements;i++)if(!(SSAflag[i] | HOflag[i]))SIAflag[i]=1;
	}
	else if (filltype === 'SSA'){
		for(var i=0;i<md.mesh.numberofelements;i++)if(!(SIAflag[i] | HOflag[i] | FSflag[i]))SSAflag[i]=1;
	}
	else if (filltype === 'HO'){
		for(var i=0;i<md.mesh.numberofelements;i++)if(!(SIAflag[i] | SSAflag[i] | FSflag[i]))HOflag[i]=1;
	}

	//check that each element has at least one flag
	for(var i=0;i<md.mesh.numberofelements;i++)if((SIAflag[i] + SSAflag[i] + HOflag[i] + L1L2flag[i] + FSflag[i])==0)
	throw Error("elements type not assigned, supported models are 'SIA','SSA','HO' and 'FS'");

	//check that each element has only one flag
	if (ArrayAnyAboveStrict(ArrayXPY(SIAflag,SSAflag,HOflag,L1L2flag),1)){
		console.log('setflowequation warning message: some elements have several types, higher order type is used for them')

		for(var i=0;i<md.mesh.numberofelements;i++){
			if(SIAflag[i] & SSAflag[i])SIAflag[i]=0;
			if(SIAflag[i] & HOflag[i])SIAflag[i]=0;
			if(SSAflag[i] & HOflag[i])SSAflag[i]=0;
		}
	}

	//check that L1L2 is not coupled to any other model for now
	if (ArrayAnyEqual(L1L2flag,1) & ArrayAnyEqual(ArrayOr(SIAflag,SSAflag,HOflag,FSflag),1)) throw Error('L1L2 cannot be coupled to any other model');

	//Check that no HO or FS for 2d mesh
	if (md.mesh.domaintype() == '2Dhorizontal'){
		for(var i=0;i<FSflag.length;i++){
			if(FSflag[i] | HOflag[i]) throw Error('FS and HO elements not allowed in 2d mesh, extrude it first')
		}
	}

	//FS can only be used alone for now:
	if (ArrayAnyEqual(FSflag,1) & ArrayAnyEqual(SIAflag,1)) throw Error('FS cannot be used with any other model for now, put FS everywhere')

	//Initialize node fields
	nodeonSIA=NewArrayFill(md.mesh.numberofvertices,0);
	pos=ArrayFind(SIAflag,1);
	for(var i=0;i<pos.length;i++) for(var j=0;j<md.mesh.elements[0].length;j++) nodeonSIA[md.mesh.elements[pos[i]][j]-1]=1;
	
	nodeonSSA=NewArrayFill(md.mesh.numberofvertices,0);
	pos=ArrayFind(SSAflag,1);
	for(var i=0;i<pos.length;i++) for(var j=0;j<md.mesh.elements[0].length;j++) nodeonSSA[md.mesh.elements[pos[i]][j]-1]=1;
	
	nodeonHO=NewArrayFill(md.mesh.numberofvertices,0);
	pos=ArrayFind(HOflag,1);
	for(var i=0;i<pos.length;i++) for(var j=0;j<md.mesh.elements[0].length;j++) nodeonHO[md.mesh.elements[pos[i]][j]-1]=1;
	
	nodeonL1L2=NewArrayFill(md.mesh.numberofvertices,0);
	pos=ArrayFind(L1L2flag,1);
	for(var i=0;i<pos.length;i++) for(var j=0;j<md.mesh.elements[0].length;j++) nodeonL1L2[md.mesh.elements[pos[i]][j]-1]=1;

	nodeonFS=NewArrayFill(md.mesh.numberofvertices,0);
	noneflag=NewArrayFill(md.mesh.numberofvertices,0);
	
	
	//First modify FSflag to get rid of elements contrained everywhere (spc + border with HO or SSA)
	if (ArrayAnyEqual(FSflag,1)){
		throw Error("FS elements not supported yet!");
		/*fullspcnodes=double((~isnan(md.stressbalance.spcvx)+~isnan(md.stressbalance.spcvy)+~isnan(md.stressbalance.spcvz))==3 | (nodeonHO & nodeonFS));         //find all the nodes on the boundary of the domain without icefront
		fullspcelems=double(sum(fullspcnodes(md.mesh.elements),2)==6);         //find all the nodes on the boundary of the domain without icefront
		FSflag(find(fullspcelems))=0;
		nodeonFS(md.mesh.elements(find(FSflag),:))=1;*/
	}

	//Then complete with NoneApproximation or the other model used if there is no FS
	if (ArrayAnyEqual(FSflag,1)){
		throw Error("FS elements not supported yet!");
		/*if any(HOflag), //fill with HO
			HOflag(~FSflag)=1;
			nodeonHO(md.mesh.elements(find(HOflag),:))=1;
		elseif any(SSAflag), //fill with SSA
			SSAflag(~FSflag)=1;
			nodeonSSA(md.mesh.elements(find(SSAflag),:))=1;
		else //fill with none 
			noneflag(find(~FSflag))=1;
		end*/
	}

	//Now take care of the coupling between SSA and HO
	md.stressbalance.vertex_pairing=[];
	nodeonSSAHO=NewArrayFill(md.mesh.numberofvertices,0);
	nodeonHOFS=NewArrayFill(md.mesh.numberofvertices,0);
	nodeonSSAFS=NewArrayFill(md.mesh.numberofvertices,0);
	SSAHOflag=NewArrayFill(md.mesh.numberofelements,0);
	SSAFSflag=NewArrayFill(md.mesh.numberofelements,0);
	HOFSflag=NewArrayFill(md.mesh.numberofelements,0);

	/*if strcmpi(coupling_method,'penalties'),
		//Create the border nodes between HO and SSA and extrude them
		numnodes2d=md.mesh.numberofvertices2d;
		numlayers=md.mesh.numberoflayers;
		bordernodes2d=find(nodeonHO(1:numnodes2d) & nodeonSSA(1:numnodes2d)); //Nodes connected to two different types of elements

		//initialize and fill in penalties structure
		if ~isnan(bordernodes2d),
			penalties=[];
			for	i=1:numlayers-1,
				penalties=[penalties; [bordernodes2d bordernodes2d+md.mesh.numberofvertices2d*(i)]];
			end
			md.stressbalance.vertex_pairing=penalties;
		end
	elseif strcmpi(coupling_method,'tiling'),
		if any(SSAflag) & any(HOflag), //coupling SSA HO
			//Find node at the border
			nodeonSSAHO(find(nodeonSSA & nodeonHO))=1;
			//SSA elements in contact with this layer become SSAHO elements
			matrixelements=ismember(md.mesh.elements,find(nodeonSSAHO));
			commonelements=sum(matrixelements,2)~=0;
			commonelements(find(HOflag))=0; //only one layer: the elements previously in SSA
			SSAflag(find(commonelements))=0; //these elements are now SSAHOelements
			SSAHOflag(find(commonelements))=1;
			nodeonSSA(:)=0;
			nodeonSSA(md.mesh.elements(find(SSAflag),:))=1;

			//rule out elements that don't touch the 2 boundaries
			pos=find(SSAHOflag);
			elist=zeros(length(pos),1);
			elist = elist + any(sum(nodeonSSA(md.mesh.elements(pos,:)),2),2);
			elist = elist - any(sum(nodeonHO(md.mesh.elements(pos,:))  ,2),2);
			pos1=find(elist==1);
			SSAflag(pos(pos1))=1;
			SSAHOflag(pos(pos1))=0;
			pos2=find(elist==-1);
			HOflag(pos(pos2))=1;
			SSAHOflag(pos(pos2))=0;

			//Recompute nodes associated to these elements
			nodeonSSA(:)=0;
			nodeonSSA(md.mesh.elements(find(SSAflag),:))=1;
			nodeonHO(:)=0;
			nodeonHO(md.mesh.elements(find(HOflag),:))=1;
			nodeonSSAHO(:)=0;
			nodeonSSAHO(md.mesh.elements(find(SSAHOflag),:))=1;

		elseif any(HOflag) & any(FSflag), //coupling HO FS
			//Find node at the border
			nodeonHOFS(find(nodeonHO & nodeonFS))=1;
			//FS elements in contact with this layer become HOFS elements
			matrixelements=ismember(md.mesh.elements,find(nodeonHOFS));
			commonelements=sum(matrixelements,2)~=0;
			commonelements(find(HOflag))=0; //only one layer: the elements previously in SSA
			FSflag(find(commonelements))=0; //these elements are now SSAHOelements
			HOFSflag(find(commonelements))=1;
			nodeonFS=zeros(md.mesh.numberofvertices,1);
			nodeonFS(md.mesh.elements(find(FSflag),:))=1;

			//rule out elements that don't touch the 2 boundaries
			pos=find(HOFSflag);
			elist=zeros(length(pos),1);
			elist = elist + any(sum(nodeonFS(md.mesh.elements(pos,:)),2),2);
			elist = elist - any(sum(nodeonHO(md.mesh.elements(pos,:)),2),2);
			pos1=find(elist==1);
			FSflag(pos(pos1))=1;
			HOFSflag(pos(pos1))=0;
			pos2=find(elist==-1);
			HOflag(pos(pos2))=1;
			HOFSflag(pos(pos2))=0;

			//Recompute nodes associated to these elements
			nodeonFS(:)=0;
			nodeonFS(md.mesh.elements(find(FSflag),:))=1;
			nodeonHO(:)=0;
			nodeonHO(md.mesh.elements(find(HOflag),:))=1;
			nodeonHOFS(:)=0;
			nodeonHOFS(md.mesh.elements(find(HOFSflag),:))=1;

		elseif any(FSflag) & any(SSAflag),
			//Find node at the border
			nodeonSSAFS(find(nodeonSSA & nodeonFS))=1;
			//FS elements in contact with this layer become SSAFS elements
			matrixelements=ismember(md.mesh.elements,find(nodeonSSAFS));
			commonelements=sum(matrixelements,2)~=0;
			commonelements(find(SSAflag))=0; //only one layer: the elements previously in SSA
			FSflag(find(commonelements))=0; //these elements are now SSASSAelements
			SSAFSflag(find(commonelements))=1;
			nodeonFS=zeros(md.mesh.numberofvertices,1);
			nodeonFS(md.mesh.elements(find(FSflag),:))=1;

			//rule out elements that don't touch the 2 boundaries
			pos=find(SSAFSflag);
			elist=zeros(length(pos),1);
			elist = elist + any(sum(nodeonSSA(md.mesh.elements(pos,:)),2),2);
			elist = elist - any(sum(nodeonFS(md.mesh.elements(pos,:))  ,2),2);
			pos1=find(elist==1);
			SSAflag(pos(pos1))=1;
			SSAFSflag(pos(pos1))=0;
			pos2=find(elist==-1);
			FSflag(pos(pos2))=1;
			SSAFSflag(pos(pos2))=0;

			//Recompute nodes associated to these elements
			nodeonSSA(:)=0;
			nodeonSSA(md.mesh.elements(find(SSAflag),:))=1;
			nodeonFS(:)=0;
			nodeonFS(md.mesh.elements(find(FSflag),:))=1;
			nodeonSSAFS(:)=0;
			nodeonSSAFS(md.mesh.elements(find(SSAFSflag),:))=1;

		elseif any(FSflag) & any(SIAflag),
			error('type of coupling not supported yet');
		end
	end*/

	//Create element equations
	md.flowequation.element_equation=NewArrayFill(md.mesh.numberofelements,0);
	pos=ArrayFind(noneflag,1);for(var i=0;i<pos.length;i++)md.flowequation.element_equation[pos[i]]=0;
	pos=ArrayFind(SIAflag,1);for(var i=0;i<pos.length;i++)md.flowequation.element_equation[pos[i]]=1;
	pos=ArrayFind(SSAflag,1);for(var i=0;i<pos.length;i++)md.flowequation.element_equation[pos[i]]=2;
	pos=ArrayFind(L1L2flag,1);for(var i=0;i<pos.length;i++)md.flowequation.element_equation[pos[i]]=3;
	pos=ArrayFind(HOflag,1);for(var i=0;i<pos.length;i++)md.flowequation.element_equation[pos[i]]=4;
	pos=ArrayFind(FSflag,1);for(var i=0;i<pos.length;i++)md.flowequation.element_equation[pos[i]]=5;
	pos=ArrayFind(SSAHOflag,1);for(var i=0;i<pos.length;i++)md.flowequation.element_equation[pos[i]]=6;
	pos=ArrayFind(SSAFSflag,1);for(var i=0;i<pos.length;i++)md.flowequation.element_equation[pos[i]]=7;
	pos=ArrayFind(HOFSflag,1);for(var i=0;i<pos.length;i++)md.flowequation.element_equation[pos[i]]=8;


	//border
	md.flowequation.borderHO=nodeonHO;
	md.flowequation.borderSSA=nodeonSSA;
	md.flowequation.borderFS=nodeonFS;
	

	//Create vertices_type
	md.flowequation.vertex_equation=NewArrayFill(md.mesh.numberofvertices,0);

	pos=ArrayFind(nodeonSSA,1);for(var i=0;i<pos.length;i++)md.flowequation.vertex_equation[pos[i]]=2;
	pos=ArrayFind(nodeonL1L2,1);for(var i=0;i<pos.length;i++)md.flowequation.vertex_equation[pos[i]]=3;
	pos=ArrayFind(nodeonHO,1);for(var i=0;i<pos.length;i++)md.flowequation.vertex_equation[pos[i]]=4;
	pos=ArrayFind(nodeonFS,1);for(var i=0;i<pos.length;i++)md.flowequation.vertex_equation[pos[i]]=5;
	//DO SIA LAST! Otherwise spcs might not be set up correctly (SIA should have priority)
	pos=ArrayFind(nodeonSIA,1);for(var i=0;i<pos.length;i++)md.flowequation.vertex_equation[pos[i]]=1;
	if (ArrayAnyEqual(FSflag,1)){
		pos=ArrayFind(nodeonFS==0);
		if(ArrayAnyEqual(HOflag,0) & ArrayAnyEqual(SSA,0)){
			for(var i=0;i<pos.length;i++)md.flowequation.vertex_equation[pos[i]]=0;
		}
	}

	pos=ArrayFind(nodeonSSAHO,1);for(var i=0;i<pos.length;i++)md.flowequation.vertex_equation[pos[i]]=6;
	pos=ArrayFind(nodeonHOFS,1);for(var i=0;i<pos.length;i++)md.flowequation.vertex_equation[pos[i]]=7;
	pos=ArrayFind(nodeonSSAFS,1);for(var i=0;i<pos.length;i++)md.flowequation.vertex_equation[pos[i]]=8;

	//figure out solution types
	md.flowequation.isSIA  = ArrayAnyEqual(md.flowequation.element_equation,1);
	md.flowequation.isSSA  = ArrayAnyEqual(md.flowequation.element_equation,2);
	md.flowequation.isL1L2 = ArrayAnyEqual(md.flowequation.element_equation,3);
	md.flowequation.isHO   = ArrayAnyEqual(md.flowequation.element_equation,4);
	md.flowequation.isFS   = ArrayAnyEqual(md.flowequation.element_equation,5);
	return

	//Check that tiling can work:
	/*if any(md.flowequation.borderSSA) & any(md.flowequation.borderHO) & any(md.flowequation.borderHO + md.flowequation.borderSSA ~=1),
		error('error coupling domain too irregular');
	end
	if any(md.flowequation.borderSSA) & any(md.flowequation.borderFS) & any(md.flowequation.borderFS + md.flowequation.borderSSA ~=1),
		error('error coupling domain too irregular');
	end
	if any(md.flowequation.borderFS) & any(md.flowequation.borderHO) & any(md.flowequation.borderHO + md.flowequation.borderFS~=1),
		error('error coupling domain too irregular');
	end*/
}
function setmask(md,floatingice,groundedice){
//SETMASK - establish boundaries between grounded and floating ice.
//
//   By default, ice is considered grounded. The contour floatingice defines nodes 
//   for which ice is floating. The contour groundedice defines nodes inside a floatingice, 
//   that are grounded (ie: ice rises, islands, etc ...)
//   All inputs are either strings or actually javascript arrays (included in the html file)
//   For example: 
//
//	   floatingice[0]['x']=[0,0,0,1];
//	   floatingice[0]['y']=[0,1,1,1];
//	   floatingice[1]['x']=[0,0.5,0.5,.5];
//	   floatingice[1]['y']=[0,.5,.5,.5];
//
//
//   Usage:
//      md=setmask(md,floatingice,groundedice)
//
//   Examples:
//      md=setmask(md,'all','');
//      md=setmask(md,iceshelves,islands);

	//variables: 
	var  icedomain=[];
	
	//some checks on list of arguments
	if (!((arguments.length==3) | (arguments.length==5))){
		throw Error('mask error message: wrong usage.');
	}

	if(arguments.length>3){
		if (arguments[3]=='icedomain'){
			icedomain=arguments[4];
		}
		else{
			throw Error('mask error message: wrong field specified. Only icedomain allowed for now.');
		}
		if (IsArray(icedomain)){
			throw Error('setmask error message: icedomain should be an array!');
		}
	}
	
	//Get assigned fields
	var x=md.mesh.x;
	var y=md.mesh.y;
	var elements=md.mesh.elements;

	//Assign elementonfloatingice, elementongroundedice, vertexongroundedice and vertexonfloatingice. 
	//Only change at your own peril! This is synchronized heavily with the GroundingLineMigration module. 
	elementonfloatingice=FlagElements(md,floatingice);
	elementongroundedice=FlagElements(md,groundedice);

	//Because groundedice nodes and elements can be included into an floatingice, we need to update. Remember, all the previous 
	//arrays come from domain outlines that can intersect one another: 
	elementonfloatingice=ArrayAnd(elementonfloatingice,ArrayNot(elementongroundedice));
	elementongroundedice=ArrayNot(elementonfloatingice);

	//the order here is important. we choose vertexongroundedice as default on the grounding line.
	vertexonfloatingice=NewArrayFill(md.mesh.numberofvertices,0);
	vertexongroundedice=NewArrayFill(md.mesh.numberofvertices,0);
	pos=ArrayFind(elementongroundedice,1); for (var i=0;i<pos.length;i++)for(var j=0;j<3;j++) vertexongroundedice[md.mesh.elements[i,j]-1]=1;
	pos=ArrayFind(vertexongroundedice,0); for (var i=0;i<pos.length;i++)vertexonfloatingice[i]=1;

	//level sets
	groundedice_levelset=vertexongroundedice;
	pos=ArrayFind(vertexongroundedice,0);for(var i=0;i<pos.length;i++) groundedice_levelset[i]=-1;
	md.mask.groundedice_levelset=groundedice_levelset;

	if(arguments.length>3){
		md.mask.ice_levelset = NewArrayFill(md.mesh.numberofvertices,1.0);
		//use contourtomesh to set ice values inside ice domain
		//[vertexinsideicedomain,elementinsideicedomain]=ContourToMesh(elements,x,y,icedomain,'node',1);
		pos=ArrayFind(vertexinsideicedomain,1.0);for(var i=0;i<pos.length;i++) md.mask.ice_levelset[pos]=-1;
	}
	else{
		md.mask.ice_levelset = NewArrayFill(md.mesh.numberofvertices,-1);
	}
}
function slider(){
	
	//Convert arguments to options
	var args = Array.prototype.slice.call(arguments);
	var  options = new pairoptions(args.slice());

	//Recover option values:
	var value = options.getfieldvalue('value',0);
	var callback = options.getfieldvalue('callback',function(){});
	var name = options.getfieldvalue('name','');
	var min = options.getfieldvalue('min',0.6*value);
	var max = options.getfieldvalue('max',1.4*value);
	var width = options.getfieldvalue('width','auto');
	var height = options.getfieldvalue('height',32);
	var message = options.getfieldvalue('message','');
	var startmessage = options.getfieldvalue('startmessage',message);
	var middlemessage = options.getfieldvalue('middlemessage',message);
	var endmessage = options.getfieldvalue('endmessage',message);
	var color = options.getfieldvalue('color','#bbbbbb');
	var precision = options.getfieldvalue('precision',3);
	var step = options.getfieldvalue('step',1);
	var slidersdiv = options.getfieldvalue('slidersdiv','slidersdiv');

	$('#'+slidersdiv).css({width:width});
	$('<div class="'+name+'-slider"></div>').appendTo('#'+slidersdiv);
	$('<div class="info'+name+'">'+startmessage[0]+value.toString()+startmessage[1]+'</div>').appendTo('#'+slidersdiv);
	var info=$('.info'+name);
	$('.'+name+'-slider').slider({
		range:'min',
		value:value,
		min:min,
		max:max,
		step:step,
		slide:function(event,ui){
			moveSlide(event,ui,info,middlemessage,precision);
		},
		stop:function(event,ui){
			stopSlide(event,ui,info,callback,middlemessage,endmessage,precision);
		}
	});
	$('.'+name+'-slider.ui-slider').css({
		width:'auto',
		height:height,
		background:color,
		margin:'8px'
	});
	$('.'+name+'-slider .ui-slider-handle').css({
		background:color,
		height:parseInt(height)+8
	});
	$('.'+name+'-slider .ui-slider-range').css({
		background:color
	});
}

function moveSlide(event,ui,info,middlemessage,precision){
	var val=ui.value;
	info.text(middlemessage[0]+val.toPrecision(precision).toString()+middlemessage[1]);
}

function stopSlide(event,ui,info,functionvalue,middlemessage,endmessage,precision){
	var val=ui.value;
	info.text(middlemessage[0]+val.toPrecision(precision).toString()+middlemessage[1]);
	functionvalue(val);
	info.text(endmessage[0]+val.toPrecision(precision).toString()+endmessage[1]);
}

function progress(){
	//Convert arguments to options
	var args = Array.prototype.slice.call(arguments);
	var  options = new pairoptions(args.slice());
	
	var value = options.getfieldvalue('value',0);
	var name = options.getfieldvalue('name','hma');
	var min = options.getfieldvalue('min',0.6*value);
	var max = options.getfieldvalue('max',1.4*value);
	var width = options.getfieldvalue('width','auto');
	var height = options.getfieldvalue('height',32);
	var color = options.getfieldvalue('color','#bbbbbb');
	var progressdiv = options.getfieldvalue('progressdiv','progressdiv');
	
	var canvas = $('#'+name)[0];
	var progressbar = $('#'+name+'-progressbar');
	var playbutton = $('#'+name+'-playbutton');
	var reversebutton = $('#'+name+'-reversebutton');
	var timelabel = $('#'+name+'-timelabel');

	
	playbutton.click(function() {
		canvas.moviePlay = !canvas.moviePlay;
		if (canvas.moviePlay) {
			$("#playButton").html("&#10074&#10074");
		}
		else {
			$("#playButton").html("&#9654");
		}
	});
	reversebutton.click(function() {
		canvas.movieReverse = !canvas.movieReverse;
		if (canvas.movieReverse) {
			reversebutton.html("&#9664&#9664");
		}
		else {
			reversebutton.html("&#9654&#9654");
		}
	});
	canvas.timeLabel = timelabel;
	
	$('#'+progressdiv).css({width:width});
	$('<div class="'+name+'-progressbar"></div>').prependTo('#'+progressdiv);
	$('.'+name+'-progressbar').slider({
		range:'min',
		value:0,
		min:0,
		max:1,
		step:1,
		start:function(event,ui){
			startProgress(event,ui,canvas);
		},
		slide:function(event,ui){
			moveProgress(event,ui,canvas);
		},
		stop:function(event,ui){
			stopProgress(event,ui,canvas);
		}
	});
	$('.'+name+'-progressbar.ui-slider').css({
		width:'auto',
		height:height,
		background:color,
		margin:'8px'
	});
	$('.'+name+'-progressbar .ui-slider-handle').css({
		background:color,
		height:parseInt(height)+8
	});
	$('.'+name+'-progressbar .ui-slider-range').css({
		background:'red'
	});
	
	canvas.progressBar = $('.'+name+'-progressbar');
}

function startProgress(event,ui,canvas){
	canvas.movieFrame = ui.value;
	canvas.movieIncrement = false;
}

function moveProgress(event,ui,canvas){
	canvas.movieFrame = ui.value;
}

function stopProgress(event,ui,canvas){
	canvas.movieFrame = ui.value;
	canvas.movieIncrement = true;
}function tooltip(){
	
	//Convert arguments to options
	var args = Array.prototype.slice.call(arguments);
	var  options = new pairoptions(args.slice());

	//Recover option values:
	var value = options.getfieldvalue('value','');
	var tooltipdiv = options.getfieldvalue('tooltipdiv','tooltipdiv');

	$('<div class="container-div">' + value + '</div>').appendTo('#'+tooltipdiv);	
	$("#"+tooltipdiv).hover(function() {
		var tooltip = $("> div", this).show();
		var pos = tooltip.offset();
		tooltip.hide();
		var right = pos.left + tooltip.width();
		var pageWidth = $(document).width();
		if (pos.left < 0) {
			tooltip.css("marginLeft", "+=" + (-pos.left) + "px");
		}
		else if (right > pageWidth) {
			tooltip.css("marginLeft", "-=" + (right - pageWidth));
		}
		tooltip.fadeIn();
		}, function() {
		$("> div", this).fadeOut(function() {$(this).css("marginLeft", "");});
	});
}
function applyoptions(md,data,datatype,options,canvas,gl,node){
	//APPLYOPTIONS - apply the options to current plot
	//
	//   Usage:
	//      applyoptions(md,data,options)
	//
	//   See also: PLOTMODEL, PARSE_OPTIONS
	
	//colorbar {{{
	if (options.exist('colorbar')) {
		if (options.getfieldvalue('colorbar')==1) {
			//Handle movie data {{{
			if (datatype == 5) {
				data = data[0];
			} //}}}
			//Variable options initialization {{{		
			var caxis = options.getfieldvalue('caxis',[ArrayMin(data),ArrayMax(data)]);
			var canvassize = options.getfieldvalue('canvassize',480);
			var cheight;
			var cwidth;
			var cdivisions = cheight/4;
			var color;
			var ccontext,ccanvas,ccanvasid,ccanvashtml;
			var colorbarorientiation = options.getfieldvalue('colorbarorientation','vertical');
			var cheightoffset = options.getfieldvalue('colorbarfontsize',18);
			var cwidthoffset = options.getfieldvalue('colorbarfontsize',18)*3;
			var y;
			var x;
			//}}}
			//Set colorbar lables {{{
			var labels = [];
			var cdivisions = options.getfieldvalue('colorbarnticks',6);
			var caxisdelta = caxis[1] - caxis[0];
			var clabelitem;
			var precision=options.getfieldvalue('colorbarprecision',3);
			
			if (options.getfieldvalue('log','off')!='off') {
				for (var i = cdivisions; i >= 0; i--) {
					var scale = (Math.log10(caxis[1])-Math.log10(caxis[0]))/Math.log10(options.getfieldvalue('log',10));
					labels[i] = (Math.pow(options.getfieldvalue('log',10),Math.log10(caxis[0])/Math.log10(options.getfieldvalue('log',10))+scale*(cdivisions-i)/cdivisions)).toPrecision(precision);
				}
			} else {
				for (var i = cdivisions; i >= 0; i--) {
					labels[i] = (caxisdelta*(cdivisions-i)/cdivisions+caxis[0]).toPrecision(precision);
				}
			} //}}}
			//Handle orientation {{{
			if (colorbarorientiation=='vertical') { //{{{
				cheight = options.getfieldvalue('colorbarheight',0.95)*canvassize;
				cwidth = options.getfieldvalue('colorbarwidth',canvassize/20);
				ccanvasid = options.getfieldvalue('colorbarcanvasid',options.getfieldvalue('canvasid')+'_colorbar');
				ccanvashtml = document.getElementById(ccanvasid);
				if (ccanvashtml==null) {
					ccanvas = $('<canvas id="'+ccanvasid+'" width="'+String(cwidth+cheightoffset*4)+'" height="'+String(cheight+cheightoffset*2)+'"></canvas>').insertAfter('#'+options.getfieldvalue('canvasid'));
					ccanvas.css({'position':'relative','top':((canvassize-cheight-cheightoffset)/-2).toFixed(2)+'px'});
					ccontext = ccanvas[0].getContext('2d');
				}
				else{
					ccanvas = $('#'+ccanvasid);
					if (options.exist('colorbarcanvasid')&&!ccanvashtml.hasOwnProperty('init')){
						//store actual colobar size
						cwidth = ccanvashtml.clientWidth;
						cheight = ccanvashtml.clientHeight;
						ccanvas.attr({width:cwidth+cheightoffset*4,height:cheight+cheightoffset*2}).css({width:cwidth+cheightoffset*4,height:cheight+cheightoffset*2});
						//get html object instead of jqurey object to modify height/width to accomodate labels
						ccanvashtml.cwidth = cwidth;
						ccanvashtml.cheight = cheight;
						ccanvashtml.init = true;
					}
					else {
						cwidth = ccanvashtml.cwidth;
						cheight = ccanvashtml.cheight;
					}
					ccontext = ccanvas[0].getContext('2d');
					/*erase existing colorbar:*/
					ccontext.clearRect(0,0, ccanvashtml.width, ccanvashtml.height);
					ccontext.beginPath();
				}
				var cgradient = ccontext.createLinearGradient(0,cheightoffset/2,0,cheight);
				
				var cmap = options.getfieldvalue('cmap','jet');
				var colorbar = colorbars[cmap];
				for (var i = 0; i < colorbar.length; i++) {
					color = colorbar[colorbar.length-i-1];
					color = [Math.round(color[0]*255),Math.round(color[1]*255),Math.round(color[2]*255)];	
					cgradient.addColorStop(i/colorbar.length,'rgba('+color.toString()+',1.0)');
				}
				ccontext.fillStyle=cgradient;
				ccontext.fillRect(0,cheightoffset*3/2,cwidth,cheight);
				
				//Draw colorbar border
				ccontext.beginPath();
				ccontext.lineWidth='1';
				ccontext.strokeStyle='black';
				ccontext.rect(0,cheightoffset*3/2,cwidth,cheight);
				ccontext.stroke();
				
				for (var i = 0; i <= cdivisions; i++) {
					y = i/cdivisions*cheight+cheightoffset/2;
					x = 0.2*cwidth;
					ccontext.beginPath();
					ccontext.moveTo(0,y+cheightoffset);
					ccontext.lineTo(x,y+cheightoffset);
					ccontext.moveTo(cwidth-x,y+cheightoffset);
					ccontext.lineTo(cwidth,y+cheightoffset);
					ccontext.stroke();
					ccontext.font=String(options.getfieldvalue('colorbarfontsize',18))+'px "Lato",Helvetica,Arial,sans-serif';
					ccontext.fillStyle='black';
					ccontext.textAlign='left';
					ccontext.fillText(labels[i],cwidth+x,y+cheightoffset*3/2);
				}
				if (options.exist('colorbartitle')) {
					ccontext.font=String(options.getfieldvalue('colorbarfontsize',18))+'px "Lato",Helvetica,Arial,sans-serif';
					ccontext.fillStyle='black';
					ccontext.textAlign='left';
					ccontext.textBaseline='bottom';
					ccontext.fillText(options.getfieldvalue('colorbartitle'),0,cheightoffset);
				}
			} //}}}
			else { //colorbarorientiation=='horizontal' {{{
				cheight = options.getfieldvalue('colorbarheight',canvassize/20);
				cwidth = options.getfieldvalue('colorbarwidth',0.95)*canvassize;
				ccanvasid = options.getfieldvalue('colorbarcanvasid',options.getfieldvalue('canvasid')+'_colorbar');
				ccanvashtml = document.getElementById(ccanvasid);
				if (ccanvashtml==null) {
					ccanvas = $('<canvas id="'+ccanvasid+'" width="'+String(cwidth+cheightoffset*4)+'" height="'+String(cheight+cheightoffset*5/2)+'"></canvas>').insertAfter('#'+options.getfieldvalue('canvasid'));
					ccanvas.css({'position':'relative','top':((canvassize-cheight-cheightoffset*2)/-2).toFixed(2)+'px'});
					ccontext = ccanvas[0].getContext('2d');
				}
				else{
					ccanvas = $('#'+ccanvasid);
					if (options.exist('colorbarcanvasid')&&!ccanvashtml.hasOwnProperty('init')){
						//store actual colobar size
						cwidth = ccanvashtml.clientWidth;
						cheight = ccanvashtml.clientHeight;
						//get html object instead of jqurey object to modify height/width to accomodate labels
						ccanvashtml.width = cwidth+cheightoffset*6;
						ccanvashtml.height = cheight+cheightoffset*5/2;
						ccanvashtml.cwidth = cwidth;
						ccanvashtml.cheight = cheight;
						ccanvashtml.init = true;
					}
					else {
						cwidth = ccanvashtml.cwidth;
						cheight = ccanvashtml.cheight;
					}
					ccontext = ccanvas[0].getContext('2d');
					/*erase existing colorbar:*/
					ccontext.clearRect(0,0, ccanvashtml.width, ccanvashtml.height);
					ccontext.beginPath();
				}
				var cgradient = ccontext.createLinearGradient(cwidth+cwidthoffset,0,cwidthoffset,0);
				
				var cmap = options.getfieldvalue('cmap','jet');
				var colorbar = colorbars[cmap];
				for (var i = 0; i < colorbar.length; i++) {
					color = colorbar[colorbar.length-i-1];
					color = [Math.round(color[0]*255),Math.round(color[1]*255),Math.round(color[2]*255)];	
					cgradient.addColorStop(i/colorbar.length,'rgba('+color.toString()+',1.0)');
				}
				ccontext.fillStyle=cgradient;
				ccontext.fillRect(cwidthoffset,cheightoffset,cwidth,cheight);
				
				//Draw colorbar border
				ccontext.beginPath();
				ccontext.lineWidth='1';
				ccontext.strokeStyle='black';
				ccontext.rect(cwidthoffset,cheightoffset,cwidth,cheight);
				ccontext.stroke();
				
				for (var i = 0; i <= cdivisions; i++) {
					y = 0.2*cheight;
					x = i/cdivisions*cwidth;
					ccontext.beginPath();
					ccontext.moveTo(x+cwidthoffset,cheightoffset);
					ccontext.lineTo(x+cwidthoffset,y+cheightoffset);
					ccontext.moveTo(x+cwidthoffset,cheight-y+cheightoffset);
					ccontext.lineTo(x+cwidthoffset,cheight+cheightoffset);
					ccontext.stroke();
					ccontext.font=String(options.getfieldvalue('colorbarfontsize',18))+'px "Lato",Helvetica,Arial,sans-serif';
					ccontext.fillStyle='black';
					ccontext.textAlign='center';
					ccontext.fillText(labels[cdivisions-i],x+cwidthoffset,cheight+cheightoffset*2);
				}
				if (options.exist('colorbartitle')) {
					ccontext.font=String(options.getfieldvalue('colorbarfontsize',18))+'px "Lato",Helvetica,Arial,sans-serif';
					ccontext.fillStyle='black';
					ccontext.textAlign='center';
					ccontext.textBaseline='bottom';
					ccontext.fillText(options.getfieldvalue('colorbartitle'),cwidth/2+cwidthoffset,cheightoffset);
				}
			} //}}}
			//}}}
		} 
	}
	//}}}
	//texture canvas //{{{
	var tcontext,tcanvas,tcanvasid,tcanvashtml,tURL,tgradient;
	tcanvasid = 'texturecanvas';
	var tcanvashtml = document.getElementById(tcanvasid);
	if (tcanvashtml == null) {
		tcanvas = $('<canvas id="texturecanvas" width="256" height="256"></canvas>').insertAfter('#'+String(options.getfieldvalue('canvasid')));
		tcanvas.css({'display':'none'});
		tcanvashtml = document.getElementById(tcanvasid);
	}
	else {
		tcanvas = $('#' + tcanvasid);
	}
	tcontext = tcanvashtml.getContext('2d');
	tgradient = tcontext.createLinearGradient(0,0,0,256);
		
	var cmap = options.getfieldvalue('cmap','jet');
	var colorbar = colorbars[cmap];
	for (var i = 0; i < colorbar.length; i++) {
		color = colorbar[colorbar.length-i-1];
		color = [Math.round(color[0]*255),Math.round(color[1]*255),Math.round(color[2]*255)];	
		tgradient.addColorStop(i/colorbar.length,'rgba('+color.toString()+',1.0)');
	}
	
	tcontext.fillStyle = tgradient;
	tcontext.fillRect(0,0,256,256);
	tURL = tcanvashtml.toDataURL();
	node['texture'] = initTexture(gl,tURL);
	//}}}
	//expdisp contours {{{
	if (options.exist('expdisp')) {
		canvas.nodes['expdisp'] = Node(gl,options);
		var node = canvas.nodes['expdisp'];
		
		//declare variables:  {{{
		var vertices = [];
		var indices = [];
		var colors = [];
		var rgbcolor = [];
		var xmin,xmax;
		var ymin,ymax;
		var zmin,zmax;
		var scale;
		
		//Process data and model
		var x = options.getfieldvalue('expdisp')['x'];
		var y = options.getfieldvalue('expdisp')['y'];
		var z = Array.apply(null, Array(x.length)).map(Number.prototype.valueOf,0);
		
		if (options.getfieldvalue('expdisp')['z']) {
			z = options.getfieldvalue('expdisp')['z'];
		}
		//}}}

		//Compute coordinates and data range: //{{{
		var modelxlim = [ArrayMin(x),ArrayMax(x)];
		var modelylim = [ArrayMin(y),ArrayMax(y)];
		var modelzlim = [ArrayMin(z),ArrayMax(z)];
		var xlim = options.getfieldvalue('xlim',modelxlim);
		var ylim = options.getfieldvalue('ylim',modelylim);
		var zlim = options.getfieldvalue('zlim',modelzlim);
		xmin = xlim[0];
		xmax = xlim[1];
		ymin = ylim[0];
		ymax = ylim[1];
		zmin = zlim[0];
		zmax = zlim[1];
		//}}}

		//Compute scaling: //{{{
		var scale = 1 / (xmax - xmin);
		node['shaderName'] = 'colored';
		node['shader'] = gl['shaders'][node['shaderName']]['program'];
		node['scale'] = [scale, scale, scale*options.getfieldvalue('heightscale',1)];
		node['translation'] = [(xmin + xmax) / (-2 / scale), (ymin + ymax) / (-2 / scale), (zmin + zmax) / (-2 / scale)];
		node['modelMatrix'] = recalculateModelMatrix(node);
		node['drawMode'] = gl.LINE_LOOP;
		node['drawOrder'] = 0;
		node['useIndexBuffer'] = false;
		node['disableDepthTest'] = true;
		//}}}

		//some defaults:
		colors.itemSize = 4;

		//retrieve some options
		var linewidth=options.getfieldvalue('linewidth',1);
		var edgecolor=options.getfieldvalue('edgecolor','black'); //RGBCOLOR?

		vertices.itemSize = 3;
		for(var i = 0; i < x.length; i++){
			vertices[vertices.length] = x[i];
			vertices[vertices.length] = y[i];
			vertices[vertices.length] = z[i];

			//edgecolor
			rgbcolor = [0.0, 0.0, 0.0];
			colors[colors.length] = rgbcolor[0];
			colors[colors.length] = rgbcolor[1];
			colors[colors.length] = rgbcolor[2];
			colors[colors.length] = 1.0;
		}

		//Initalize buffers:
		node['arrays'] = [vertices, colors];
		node['buffers'] = initBuffers(gl, node['arrays']);
	} //}}}
	//cloud of points {{{
	if (options.exist('cloud')) {
		canvas.nodes['cloud'] = Node(gl,options);
		var node = canvas.nodes['cloud'];

		//declare variables:  {{{
		var vertices = [];
		var indices = [];
		var colors = [];
		var rgbcolor = [];
		var xmin,xmax;
		var ymin,ymax;
		var zmin,zmax;
		var scale;
		
		//Process data and model
		var x = options.getfieldvalue('cloud')['x'];
		var y = options.getfieldvalue('cloud')['y'];
		var z = Array.apply(null, Array(x.length)).map(Number.prototype.valueOf,0);
		
		if (options.getfieldvalue('cloud')['z']) {
			z = options.getfieldvalue('cloud')['z'];
		}
		//}}}

		//Compute coordinates and data range: //{{{
		var modelxlim = [ArrayMin(x),ArrayMax(x)];
		var modelylim = [ArrayMin(y),ArrayMax(y)];
		var modelzlim = [ArrayMin(z),ArrayMax(z)];
		var xlim = options.getfieldvalue('xlim',modelxlim);
		var ylim = options.getfieldvalue('ylim',modelylim);
		var zlim = options.getfieldvalue('zlim',modelzlim);
		xmin = xlim[0];
		xmax = xlim[1];
		ymin = ylim[0];
		ymax = ylim[1];
		zmin = zlim[0];
		zmax = zlim[1];
		//}}}

		//Compute scaling: //{{{
		var scale = 1 / (xmax - xmin);
		node['shaderName'] = 'colored';
		node['shader'] = gl['shaders'][node['shaderName']]['program'];
		node['scale'] = [scale, scale, scale*options.getfieldvalue('heightscale',1)];
		node['translation'] = [(xmin + xmax) / (-2 / scale), (ymin + ymax) / (-2 / scale), (zmin + zmax) / (-2 / scale)];
		node['modelMatrix'] = recalculateModelMatrix(node);
		node['drawMode'] = gl.POINTS;
		node['drawOrder'] = 0;
		node['useIndexBuffer'] = false;
		node['disableDepthTest'] = true;
		//}}}

		//some defaults:
		colors.itemSize = 4;

		//retrieve some options
		var linewidth=options.getfieldvalue('linewidth',1);
		var edgecolor=options.getfieldvalue('edgecolor','black'); //RGBCOLOR?

		vertices.itemSize = 3;
		for(var i = 0; i < x.length; i++){
			vertices[vertices.length] = x[i];
			vertices[vertices.length] = y[i];
			vertices[vertices.length] = z[i];

			//edgecolor
			rgbcolor = [0.0, 0.0, 0.0];
			colors[colors.length] = rgbcolor[0];
			colors[colors.length] = rgbcolor[1];
			colors[colors.length] = rgbcolor[2];
			colors[colors.length] = 1.0;
		}

		//Initalize buffers:
		node['arrays'] = [vertices, colors];
		node['buffers'] = initBuffers(gl, node['arrays']);
	} //}}}
	
	//text display //{{{
	if (options.exist('textlabels')) {
		var textcanvas,textcanvasid;	
		textcanvasid = options.getfieldvalue('textcanvasid',options.getfieldvalue('canvasid')+'-text');
		textcanvas = $('#'+textcanvasid);
		textcanvas.textlabels = options.getfieldvalue('textlabels',[]);
		
		//setup drawing function for text canvas draw calls
		textcanvas.draw = function(canvas) {
			var textcontext,textlabels,textlabel,textcanvaswidth,textcanvasheight,textcoordinates;	
			var textposition = vec3.create();
			var mvpMatrix = mat4.create();
			
			//ensure correct canvas coordinate scaling
			textcanvaswidth = this[0].clientWidth;
			textcanvasheight = this[0].clientHeight;
			this[0].width  = textcanvaswidth;
			this[0].height = textcanvasheight;
			
			textcontext = this[0].getContext('2d');
			textlabels = options.getfieldvalue('textlabels',[]);
			textcontext.clearRect(0, 0, textcanvaswidth, textcanvasheight);
			
			//worldspace to screenspace transformation for text
			for (text in textlabels) {
				textlabel = textlabels[text];
				mat4.multiply(mvpMatrix, canvas.cameraMatrix, canvas.nodes['overlay']['modelMatrix']);
				textposition = vec3.transformMat4(textposition, textlabel['pos'], mvpMatrix);
				if (textposition[2] > 1) { //clip coordinates with z > 1
					continue;
				}
				textcoordinates = [(textposition[0]+1.0)/2.0*textcanvaswidth, (-textposition[1]+1.0)/2.0*textcanvasheight]; //NDC to screenspace
				textcontext.font = String(options.getfieldvalue('colorbarfontsize',18))+'px "Lato",Helvetica,Arial,sans-serif';
				textcontext.fillStyle = 'black';
				textcontext.strokeStyle = 'black';
				textcontext.textAlign = 'center';
				textcontext.textBaseline = 'middle';
				textcontext.fillText(textlabel['text'], textcoordinates[0], textcoordinates[1]);
				textcontext.strokeText(textlabel['text'], textcoordinates[0], textcoordinates[1]);
			}
		}
		canvas.textcanvas = textcanvas;
	} //}}}
}function checkplotoptions(md,options){
	//PARSE_OPTIONS - build a structure that holds all plot options
	//
	//   Usage:
	//      checkplotoptions(md,options);
	//
	//   See also: PLOTMODEL

	//units
	if (options.exist('unit')){
		if (options.getfieldvalue('unit') === 'km'){
			options.changefieldvalue('unit',10^-3);
		}
		if (options.getfieldvalue('unit') === '100km'){
			options.changefieldvalue('unit',10^-5);
		}
	}

	//density
	if (options.exist('density')){
		density=options.getfieldvalue('density');
		options.changefieldvalue('density',Math.abs(Math.ceil(density)));
	}

	//Show section
	if (options.exist('showsection')){
		if (options.getfieldvalue('showsection') === 'on'){
			options.changefieldvalue('showsection',4);
		}
	}

	//smooth values
	if (options.exist('smooth')){
		if (options.getfieldvalue('smooth') === 'on'){
			options.changefieldvalue('smooth',0);
		}
	}

	//contouronly values
	if (options.exist('contouronly')){
		if (options.getfieldvalue('contouronly') === 'on'){
			options.changefieldvalue('contouronly',1);
		}
	}

	//Colorbar;
	if (options.exist('colorbar')){
		if (options.getfieldvalue('colorbar') === 'on'){
			options.changefieldvalue('colorbar',1);
		}
		else if (options.getfieldvalue('colorbar') === 'off'){
			options.changefieldvalue('colorbar',0);
		}
	}

	//text
	if (options.exist('text')){

		//1: textvalue
		textvalues=options.getfieldvalue('text');

		//ischar if only one expstyle -> create a cell
		if (typeof textvalues === 'string'){
			textvalues=[textvalues];
			numtext=1;
		}
		else if (IsArray(textvalues)){
			numtext=textvalues.length;
		}
		else throw Error("plot error message: ''text'' option should be either a string or a cell");

		//2: textweight
		if (options.exist('textweight')){

			textweightvalues=options.getfieldvalue('textweight');

			//ischar if only one textweight -> create a cell
			if (typeof textweightvalues === 'string'){
				textweightvalues=[textweightvalues];
			}
			else if (!IsArray(textweightvalues)){
				throw Error("plot error message: ''textweight'' option should be either a string or a cell");
			}
		}
		else{
			textweightvalues=['n'];
		}
		if (textweightvalues.length==1){
			var value=textweightvalues[0];
			for (var i=0;i<numtext-1;i++)textweightvalues.push(value);
		}

		//3: textsize
		if (options.exist('textsize')){
			textsizevalues=options.getfieldvalue('textsize');
		}
		//ischar if only one textsize -> create a cell
		if (typeof textsizevalues === 'number'){
			textsizevalues=[textsizevalues];
		}
		else if (!IsArray(textsizevalues)){
			throw Error("plot error message: ''textsize'' option should be either a number or a cell");
		}
		else{
			textsizevalues=[14];
		}
		if (textsizevalues.length==1){
			var value=textsizevalues[0];
			for (var i=0;i<numtext-1;i++)textsizevalues.push(value);
		}
			
		//4: textcolor
		if (options.exist('textcolor')){
			textcolorvalues=options.getfieldvalue('textcolor');
		}
		if (typeof textcolorvalues === 'string'){ //ischar if only one textcolor -> create a cell
			textcolorvalues=[textcolorvalues];
		}
		else if (!IsArray(textcolorvalues)){
			throw Error("plot error message: ''textcolor'' option should be either a string or a cell");
		}
		else textcolorvalues=['k'];

		if (textcolorvalues.length==1){
			var value=textcolorvalues[0];
			for (var i=0;i<numtext-1;i++)textcolorvalues.push(value);
		}
		
		//5: textposition
		if (options.exist('textposition')){
			textpositionvalues=options.getfieldvalue('textposition');
		}
		//ischar if only one textposition -> create a cell
		if (typeof textpositionvalues === 'number'){
			textpositionvalues=[textpositionvalues];
		}
		else if(!IsArray(textpositionvalues)){
			throw Error("plot error message: ''textposition'' option should be either a string or a cell");
		}
		else throw Error("plot error message: ''textposition'' option is missing");
			
		//6: textrotation
		if (options.exist('textrotation')){
			textrotationvalues=options.getfieldvalue('textrotation');
		}
		//ischar if only one textsize -> create a cell
		if (typeof textrotationvalues === 'number'){
			textrotationvalues=[textrotationvalues];
		}
		else if (!IsArray(textrotationvalues)){
			throw Error("plot error message: ''textrotation'' option should be either a number or a cell");
		}
		else textrotationvalues=[0];
		
		if (textrotationvalues.length==1){
			var value=textrotationvalues[0];
			for (var i=0;i<numtext-1;i++)textrotationvalues.push(value);
		}
			
		options.changefieldvalue('text',textvalues);
		options.changefieldvalue('textsize',textsizevalues);
		options.changefieldvalue('textweight',textweightvalues);
		options.changefieldvalue('textcolor',textcolorvalues);
		options.changefieldvalue('textposition',textpositionvalues);
		options.changefieldvalue('textrotation',textrotationvalues);
	}

	//expdisp
	expdispvaluesarray=[];
	expstylevaluesarray=[];
	expstylevalues=[];
	if (options.exist('expstyle')){
		expstylevalues=options.getfieldvalue('expstyle');
		//ischar if only one expstyle -> create a cell
		if (typeof expstylevalues === 'string'){
			expstylevalues=[expstylevalues];
		}
		options.changefieldvalue('expdisp',expdispvaluesarray);
	}
		
	if (options.exist('expdisp')){
		expdispvalues=options.getfieldvalue('expdisp');
	
		//ischar if only one expstyle -> create a cell
		if (typeof expdispvalues === 'string'){
			expdispvalues=[expdispvalues];
		}
		for (var i=0; i< expdispvalues.length;i++){
			expdispvaluesarray.push(expdispvalues[i]);
			if (expstylevalues.length>i){
				expstylevaluesarray.push(expstylevalues[i]);
			}
			else{
				expstylevaluesarray.push('g-');
			}
		}
		options.changefieldvalue('expstyle',expstylevaluesarray);
	}

	//latlonnumbering
	if (options.exist('latlonclick')){
		if (options.getfieldvalue('latlonclick') === 'on'){
			options.changefieldvalue('latlonclick',1);
		}
	}

	//north arrow
	if (options.exist('northarrow')){
	   if (options.getfieldvalue('northarrow') === 'on'){
		   
		   //default values
		   Lx=ArrayMax(md.mesh.y)-ArrayMin(md.mesh.y);
		   Ly=ArrayMax(md.mesh.y)-ArrayMin(md.mesh.y);
		  
		   //default values
		   options.changefieldvalue('northarrow',[ArrayMin(md.mesh.x)+1/6*Lx,ArrayMin(md.mesh.y)+5/6*Ly,1/15*Ly,0.25,1/250*Ly]);
	   }
	}

	//scale ruler
	if (options.exist('scaleruler')){
	   if (options.getfieldvalue('scaleruler') === 'on'){
		   //default values
		   Lx=ArrayMax(md.mesh.x)-ArrayMin(md.mesh.x);
		   Ly=ArrayMax(md.mesh.y)-ArrayMin(md.mesh.y);
		   
		   //default values
		   options.changefieldvalue('scaleruler',[ArrayMin(md.mesh.x)+6/8*Lx, ArrayMin(md.mesh.y)+1/10*Ly, 10^(Mat.ceil(Mat.log10(Lx)))/5, Math.floor(Lx/100), 5]);
	   }
	}
}
function plot_manager(md,options,subplotwidth,nlines,ncols,i){
//PLOT__MANAGER - distribute the plots, called by plotmodel
//
//   Usage:
//      plot_manager(md,options,subplotwidth,i);
//
//   See also: PLOTMODEL, PLOT_UNIT
			
	//parse options and get a structure of options. 
	checkplotoptions(md,options);
	//get data to be displayed
	var data = options.getfieldvalue('data');
	var canvas = initCanvas(options);
	var gl = canvas.gl;
	//TODO: each plot_ should add their node to the canvas.node array

	//figure out if this is a special plot
	if (typeof data === 'string'){

		switch(data){

			case 'boundaries':
				plot_boundaries(md,options,subplotwidth,i);
				return;
			case 'BC':
				plot_BC(md,options,subplotwidth,i,data);
				return;
			case 'edges':
				plot_edges(md,options,subplotwidth,i,data);
				return;
			case 'elementnumbering':
				plot_elementnumbering(md,options,subplotwidth,i);
				return;
			case 'highlightelements':
				plot_highlightelements(md,options,subplotwidth,i);
				return;
			case 'qmumean':
				plot_qmumean(md,options,nlines,ncols,i);
				return;
			case 'qmustddev':
				plot_qmustddev(md,options,nlines,ncols,i);
				return;
			case 'qmuhistnorm':
				plot_qmuhistnorm(md,options,nlines,ncols,i);
				return;
			case 'qmu_mass_flux_segments':
				plot_qmu_mass_flux_segments(md,options,nlines,ncols,i);
				return;
			case 'part_hist':
				plot_parthist(md,options,nlines,ncols,i);
				return;
			case 'part_hist_n':
				plot_parthistn(md,options,nlines,ncols,i);
				return;
			case 'part_hist_w':
				plot_parthistw(md,options,nlines,ncols,i);
				return;
			case 'elements_type':
				plot_elementstype(md,options,subplotwidth,i);
				return;
			case 'vertexnumbering':
				plot_vertexnumbering(md,options,subplotwidth,i);
				return;
			case 'highlightvertices':
				plot_highlightvertices(md,options,subplotwidth,i);
				return;
			case 'basal_drag':
				plot_basaldrag(md,options,subplotwidth,i,data);
				return;
			case 'basal_dragx':
				plot_basaldrag(md,options,subplotwidth,i,data);
				return;
			case 'basal_dragy':
				plot_basaldrag(md,options,subplotwidth,i,data);
				return;
			case 'driving_stress':
				plot_drivingstress(md,options,subplotwidth,i);
				return;
			case 'mesh':
				plot_mesh(md,options,canvas);
				return;
			case 'none':
				if (!(options.exist('overlay'))){
					plot_none(md,options,nlines,ncols,i);
				}
				return;
			case 'penalties':
				plot_penalties(md,options,subplotwidth,i);
				return;
			case 'partition':
				plot_partition(md,options,nlines,ncols,i);
				return;
			case 'referential':
				plot_referential(md,options,nlines,ncols,i);
				return;
			case 'riftvel':
				plot_riftvel(md,options,nlines,ncols,i);
				return;
			case 'riftnumbering':
				plot_riftnumbering(md,options,nlines,ncols,i);
				return;
			case 'rifts':
				plot_rifts(md,options,nlines,ncols,i);
				return;
			case 'riftrelvel':
				plot_riftrelvel(md,options,nlines,ncols,i);
				return;
			case 'riftpenetration':
				plot_riftpenetration(md,options,nlines,ncols,i);
				return;
			case 'riftfraction':
				plot_riftfraction(md,options,nlines,ncols,i);
				return;
			case 'sarpwr':
				plot_sarpwr(md,options,subplotwidth,i);
				return;
			case 'time_dependant':
				plot_vstime(md,options,nlines,ncols,i);
				return;
			case 'icefront':
				plot_icefront(md,options,subplotwidth,i,data);
				return;
			case 'segments':
				plot_segments(md,options,subplotwidth,i,data);
				return;
			case 'quiver':
				plot_quiver(md,options,canvas);
				return;
			case 'strainrate_tensor':
			case 'strainrate':
			case 'strainrate_principal':
			case 'strainrate_principalaxis1':
			case 'strainrate_principalaxis2':
			case 'strainrate_principalaxis3':
			case 'stress_tensor':
			case 'stress':
			case 'stress_principal':
			case 'stress_principalaxis1':
			case 'stress_principalaxis2':
			case 'stress_principalaxis3':
			case 'deviatoricstress_tensor':
			case 'deviatoricstress':
			case 'deviatoricstress_principal':
			case 'deviatoricstress_principalaxis1':
			case 'deviatoricstress_principalaxis2':
			case 'deviatoricstress_principalaxis3':
				plot_tensor(md,options,subplotwidth,i,data);
				return;
			case 'thermaltransient_results':
				plot_thermaltransient_results(md,options,subplotwidth,i);
				return;
			case 'transient_movie':
				plot_transient_movie(md,options,subplotwidth,i);
				return;
			case 'transient_results':
				plot_transient_results(md,options,subplotwidth,i);
				return;
			case 'transient_field':
				plot_transient_field(md,options,subplotwidth,i);
				return;
			default:
				if (data in md){
					data=md[data];
				}
				else{
					error('plot error message: data provided not supported yet. Type plotdoc for help');
				}
		}
	}

	//Figure out if this is a semi-transparent plot.
	if (options.getfieldvalue('overlay','off')=='on'){
		plot_overlay(md,data,options,canvas);
	}

	//Figure out if this is a semi-transparent plot.
	if (options.exist('googlemaps')){
		plot_googlemaps(md,data,options,nlines,ncols,i);
		return;
	}

	//Figure out if this is a semi-transparent plot.
	if (options.exist('gridded')){
		plot_gridded(md,data,options,nlines,ncols,i);
		return;
	}

	//Figure out if this is a Section plot
	if (options.exist('sectionvalue')){
		plot_section(md,data,options,nlines,ncols,i);
		return;
	}

	//Figure out if this is a Profile plot
	if (options.exist('profile')){
		plot_profile(md,data,options,nlines,ncols,i);
		return;
	}
	
	var	dataresults = processdata(md,data,options);
	var	data2 = dataresults[0]; 
	var	datatype = dataresults[1];
	
	//plot unit
	plot_unit(md,data2,datatype,options,canvas);

	applyoptions(md,data2,datatype,options,canvas,gl,canvas.nodes[canvas.nodes.length-1]);
}
function plot_mesh(md,options,canvas) {
	//   Usage:
	//      plot_mesh(md,options,canvas);
	//
	//   See also: PLOTMODEL

	//declare variables:  {{{
	var vertices = [];
	var indices = [];
	var colors = [];
	var nanindices = {};
	var xmin,xmax;
	var ymin,ymax;
	var zmin,zmax;
	var scale,matrixscale,vertexscale;
	
	//Process data and model
	var meshresults = processmesh(md,[],options);
	var x = meshresults[0]; 
	var y = meshresults[1]; 
	var z = meshresults[2]; 
	var elements = meshresults[3]; 
	var is2d = meshresults[4]; 
	var isplanet = meshresults[5];
		
	//Compue scaling through matrices for 2d meshes and vertices for 3d meshes
	if (!md.geometry.surface) {
		md.geometry.surface=NewArrayFill(md.mesh.x.length,0);
	}
	if (md.mesh.classname() == 'mesh3dsurface') {
		matrixscale = 1;
		vertexscale = options.getfieldvalue('heightscale',1);
	}
	else {
		if (md.geometry.surface) {
			z=md.geometry.surface;
		}	
		matrixscale = options.getfieldvalue('heightscale',1);
		vertexscale = 0;
	}
	//}}}

	//Compute coordinates and data range:
	var modelxlim = [ArrayMin(x),ArrayMax(x)];
	var modelylim = [ArrayMin(y),ArrayMax(y)];
	var modelzlim = [ArrayMin(z),ArrayMax(z)];
	var xlim = options.getfieldvalue('xlim',modelxlim);
	var ylim = options.getfieldvalue('ylim',modelylim);
	var zlim = options.getfieldvalue('zlim',modelzlim);
	xmin = xlim[0];
	xmax = xlim[1];
	ymin = ylim[0];
	ymax = ylim[1];
	zmin = zlim[0];
	zmax = zlim[1];

	//Compute gl variables:
	var gl = canvas.gl;
	var node = Node(gl,options);
	canvas.nodes[canvas.nodes.length] = node;
	scale = 1 / (xmax - xmin);
	node["shaderName"] = "colored";
	node["shader"] = gl["shaders"][node["shaderName"]]["program"];
	node["scale"] = [scale, scale, scale * matrixscale];
	node["translation"] = [(xmin + xmax) / (-2 / scale), (ymin + ymax) / (-2 / scale), (zmin + zmax) / (2 / scale)];
	node["modelMatrix"] = recalculateModelMatrix(node);
	node["drawMode"] = gl.LINES;
	node["drawOrder"] = 0;
	node["maskEnabled"] = options.getfieldvalue('innermask','off') == 'on';
	node["maskHeight"] = options.getfieldvalue('innermaskheight',150.0)*options.getfieldvalue('heightscale',1);
	node["maskColor"] = options.getfieldvalue('innermaskcolor',[0.0,0.0,1.0,1.0]);
				
	//some defaults:
	colors.itemSize = 4;

	//retrieve some options
	var linewidth=options.getfieldvalue('linewidth',1);
	var edgecolor=new RGBColor(options.getfieldvalue('edgecolor','black'));
	if (edgecolor.ok) edgecolor = [edgecolor.r/255.0, edgecolor.g/255.0, edgecolor.b/255.0, 1.0];
	else throw Error(sprintf("s%s%s\n","initWebGL error message: cound not find out edgecolor color for curent canvas ",canvas));

	//node plot {{{
	if (elements[0].length==6){ //prisms
	}
	else if (elements[0].length==4){ //tetras
	}
	else{ //2D triangular elements
		vertices.itemSize = 3;
		var xyz = vec3.create();
		var direction = vec3.create();
		var vertex = vec3.create();
		var magnitude;
		var color = edgecolor;
		for(var i = 0; i < x.length; i++){
			//Check for NaN values and remove from indices array as necessary, but preserve vertex array spacing
			if (isNaN(x[i]) || isNaN(y[i]) || isNaN(z[i])) {
				nanindices[i] = i;
				vertices[vertices.length] = vertex[0];
				vertices[vertices.length] = vertex[1];
				vertices[vertices.length] = vertex[2];
				
				colors[colors.length] = color[0];
				colors[colors.length] = color[1];
				colors[colors.length] = color[2];
				colors[colors.length] = color[3];
				continue;
			}
			//Scale vertices
			xyz = vec3.fromValues(x[i], y[i], z[i]);
			magnitude = vec3.length(xyz) + md.geometry.surface[i] * vertexscale;
			vec3.normalize(direction, xyz);
			vec3.scale(vertex, direction, magnitude);
			vertices[vertices.length] = vertex[0];
			vertices[vertices.length] = vertex[1];
			vertices[vertices.length] = vertex[2];

			colors[colors.length] = color[0];
			colors[colors.length] = color[1];
			colors[colors.length] = color[2];
			colors[colors.length] = color[3];
		}
		
		//linearize the elements array: 
		var element;
		for(var i = 0; i < elements.length; i++){
			element = [elements[i][0] - 1, elements[i][1] - 1, elements[i][2] - 1];
			if (element[0] in nanindices || element[1] in nanindices || element[2] in nanindices) continue;
			indices[indices.length] = element[0];
			indices[indices.length] = element[1];
			indices[indices.length] = element[1];
			indices[indices.length] = element[2];
			indices[indices.length] = element[2];
			indices[indices.length] = element[0];
		}
		indices.itemSize = 1;
	}
	//}}}

	/*Initalize buffers: */
	node["arrays"] = [vertices, colors, indices];
	node["buffers"] = initBuffers(gl, node["arrays"]);
}
function plot_overlay(md,data,options,canvas){
	//PLOT_OVERLAY - Function for plotting a georeferenced image.  
	//This function is called from within the plotmodel code.
	//
	//   Usage:
	//      plot_overlay(md,data,options,canvas);
	//
	//   See also: PLOTMODEL, PLOT_MANAGER

	//declare variables:  {{{
	var vertices = [];
	var indices = [];
	var texcoords = [];
	var nanindices = {};
	var xmin,xmax;
	var ymin,ymax;
	var zmin,zmax;
	var scale,matrixscale,vertexscale;

	//Process data and model
	var meshresults = processmesh(md,data,options);
	var x = meshresults[0]; 
	var y = meshresults[1]; 
	var z = meshresults[2]; 
	var elements = meshresults[3]; 
	var is2d = meshresults[4]; 
	var isplanet = meshresults[5];

	//Compue scaling through matrices for 2d meshes and vertices for 3d meshes
	if (!md.geometry.surface) {
		md.geometry.surface=NewArrayFill(md.mesh.x.length,0);
	}
	if (md.mesh.classname() == 'mesh3dsurface') {
		matrixscale = 1;
		vertexscale = options.getfieldvalue('heightscale',1);
	}
	else {
		if (md.geometry.surface) {
			z=md.geometry.surface;
		}	
		matrixscale = options.getfieldvalue('heightscale',1);
		vertexscale = 0;
	}
	//}}}
	
	//Compute coordinates and data range:
	var modelxlim = [ArrayMin(x),ArrayMax(x)];
	var modelylim = [ArrayMin(y),ArrayMax(y)];
	var modelzlim = [ArrayMin(z),ArrayMax(z)];
	var xlim = options.getfieldvalue('xlim',modelxlim);
	var ylim = options.getfieldvalue('ylim',modelylim);
	var zlim = options.getfieldvalue('zlim',modelzlim);
	xmin = xlim[0];
	xmax = xlim[1];
	ymin = ylim[0];
	ymax = ylim[1];
	zmin = zlim[0];
	zmax = zlim[1];
		
	//Compute gl variables:
	var gl = canvas.gl;
	var node = Node(gl,options);
	canvas.nodes[canvas.nodes.length] = node;
	scale = 1 / (xmax - xmin);
	node["shaderName"] = "unlit_textured";
	node["shader"] = gl["shaders"][node["shaderName"]]["program"];
	node["scale"] = [scale, scale, scale * matrixscale];
	node["translation"] = [(xmin + xmax) / (-2 / scale), (ymin + ymax) / (-2 / scale), (zmin + zmax) / (2 / scale)];
	node["modelMatrix"] = recalculateModelMatrix(node);
	node["texture"] = initTexture(gl,options.getfieldvalue('image'));
	node["alpha"] = options.getfieldvalue('outeralpha',1.0);
	node["drawOrder"] = 1;
	node["maskEnabled"] = options.getfieldvalue('outermask','off') == 'on';
	node["maskHeight"] = options.getfieldvalue('outermaskheight',150.0);
	node["maskColor"] = options.getfieldvalue('outermaskcolor',[0.0,0.0,1.0,1.0]);
				
	//some defaults:
	texcoords.itemSize = 2;
	vertices.itemSize = 3;
	
	//Handle outer radaroverlay
	if (md.radaroverlay.outerx) {
		var newelements = [];
		for (var i = 0; i < md.radaroverlay.outerindex.length; i++) {
			newelements[newelements.length] = [md.radaroverlay.outerindex[i][0] + x.length, md.radaroverlay.outerindex[i][1] + y.length, md.radaroverlay.outerindex[i][2] + z.length];
		}
		x = [].concat(x, md.radaroverlay.outerx);
		y = [].concat(y, md.radaroverlay.outery);
		z = [].concat(z, md.radaroverlay.outerheight);
		elements = [].concat(elements, newelements);
		
		//Reclaculate bounds based on otuer radaroverlay
		modelxlim = [ArrayMin(x),ArrayMax(x)];
		modelylim = [ArrayMin(y),ArrayMax(y)];
		modelzlim = [ArrayMin(z),ArrayMax(z)];
		xmin = xlim[0];
		xmax = xlim[1];
		ymin = ylim[0];
		ymax = ylim[1];
		zmin = zlim[0];
		zmax = zlim[1];
		
		node["translation"] = [node["translation"][0], node["translation"][1], (zmax) / (-1 / scale)];
	}
	
	var xrange = modelxlim[1] - modelxlim[0];
	var yrange = modelylim[1] - modelylim[0];
	
	var xyz = vec3.create();
	var direction = vec3.create();
	var vertex = vec3.create();
	var magnitude;

	//generate mesh:
	for(var i = 0; i < x.length; i++){
		//Check for NaN values and remove from indices array as necessary, but preserve vertex array spacing
		if (isNaN(x[i]) || isNaN(y[i]) || isNaN(z[i])) {
			nanindices[i] = i;
			vertices[vertices.length] = vertex[0];
			vertices[vertices.length] = vertex[1];
			vertices[vertices.length] = vertex[2];
			
			texcoords[texcoords.length] = 0.0;
			texcoords[texcoords.length] = 0.0;
			continue;
		}

		if (md.mesh.classname() == 'mesh3dsurface') {
			//Scale vertices
			xyz = vec3.fromValues(x[i], y[i], z[i]);
			magnitude = vec3.length(xyz) + md.geometry.surface[i] * vertexscale;
			vec3.normalize(direction, xyz);
			vec3.scale(vertex, direction, magnitude);
			vertices[vertices.length] = vertex[0];
			vertices[vertices.length] = vertex[1];
			vertices[vertices.length] = vertex[2];
			
			texcoords[texcoords.length] = degrees(Math.atan2(vertex[1], vertex[0])) / 360 + 0.5;
			texcoords[texcoords.length] = degrees(Math.asin(vertex[2] / magnitude)) / 180 + 0.5;
		}
		else {
			//Scale vertices
			xyz = vec3.fromValues(x[i], y[i], z[i]);
			magnitude = vec3.length(xyz);
			vec3.normalize(direction, xyz);
			vec3.scale(vertex, direction, magnitude);
			vertices[vertices.length] = vertex[0];
			vertices[vertices.length] = vertex[1];
			vertices[vertices.length] = vertex[2];
			
			texcoords[texcoords.length] = (x[i] - modelxlim[0]) / xrange;
			texcoords[texcoords.length] = (y[i] - modelylim[0]) / yrange;
		}
	}

	//linearize the elements array:
	var element;
	for(var i = 0; i < elements.length; i++){
		element = [elements[i][0] - 1, elements[i][1] - 1, elements[i][2] - 1];
		if (element[0] in nanindices || element[1] in nanindices || element[2] in nanindices) continue;
		indices[indices.length] = element[0];
		indices[indices.length] = element[1];
		indices[indices.length] = element[2];
	}
	indices.itemSize = 1;
	
	/*Initalize buffers: */
	node["arrays"] = [vertices, texcoords, indices];
	node["buffers"] = initBuffers(gl,node["arrays"]);	
}
function plot_unit(md,data,datatype,options,canvas) {
	//PLOT_UNIT - unit plot, display data
	//
	//   Usage:
	//      plot_unit(md,data,options,canvas);
	//
	//   See also: PLOTMODEL, PLOT_MANAGER

	//declare variables:  {{{
	var vertices = [];
	var indices = [];
	var texcoords = [];
	var nanindices = {};
	var xmin,xmax;
	var ymin,ymax;
	var zmin,zmax;
	var datamin,datamax,datadelta;
	var scale,matrixscale,vertexscale;

	//Process data and model
	var meshresults = processmesh(md,data,options);
	var x = meshresults[0]; 
	var y = meshresults[1]; 
	var z = meshresults[2]; 
	var elements = meshresults[3];
	var is2d = meshresults[4]; 
	var isplanet = meshresults[5];

	//Compue scaling through matrices for 2d meshes and vertices for 3d meshes
	if (!md.geometry.surface) {
		md.geometry.surface=NewArrayFill(md.mesh.x.length,0);
	}
	if (md.mesh.classname() == 'mesh3dsurface') {
		matrixscale = 1;
		vertexscale = options.getfieldvalue('heightscale',1);
	}
	else {
		if (md.geometry.surface) {
			z=md.geometry.surface;
		}	
		matrixscale = options.getfieldvalue('heightscale',1);
		vertexscale = 0;
	}
	//}}}

	//Compute coordinates and data range:
	var modelxlim = [ArrayMin(x),ArrayMax(x)];
	var modelylim = [ArrayMin(y),ArrayMax(y)];
	var modelzlim = [ArrayMin(z),ArrayMax(z)];
	var xlim = options.getfieldvalue('xlim',modelxlim);
	var ylim = options.getfieldvalue('ylim',modelylim);
	var zlim = options.getfieldvalue('zlim',modelzlim);
	xmin = xlim[0];
	xmax = xlim[1];
	ymin = ylim[0];
	ymax = ylim[1];
	zmin = zlim[0];
	zmax = zlim[1];
	var caxis;
	
	//Compute gl variables:
	var gl = canvas.gl;
	var node = Node(gl,options);
	canvas.nodes[canvas.nodes.length] = node;
	scale = 1 / (xmax - xmin);
	node["shaderName"] = "unlit_textured";
	node["shader"] = gl["shaders"][node["shaderName"]]["program"];
	node["scale"] = [scale, scale, scale * matrixscale];
	node["translation"] = [(xmin + xmax) / (-2 / scale), (ymin + ymax) / (-2 / scale), (zmin + zmax) / (2 / scale)];
	node["modelMatrix"] = recalculateModelMatrix(node);
	node["alpha"] = options.getfieldvalue('alpha',1.0);
	node["drawOrder"] = 0;
	node["maskEnabled"] = options.getfieldvalue('innermask','off') == 'on';
	node["maskHeight"] = options.getfieldvalue('innermaskheight',150.0);
	node["maskColor"] = options.getfieldvalue('innermaskcolor',[0.0,0.0,1.0,1.0]);
	node["enabled"] = options.getfieldvalue('nodata','off') == 'off';
	
	switch(datatype){
		//element plot {{{
		case 1:
			pos=ArrayFindNot(data,NaN); //needed for element on water
			if (elements[0].length==6){ //prisms
			}
			else if (elements[0].length==4){ //tetras
			}
			else{ //2D triangular elements
			}
			break;
		//}}}
		//node plot {{{
		case 2:
			if (elements[0].length==6){ //prisms
			}
			else if (elements[0].length==4){ //tetras
			}
			else{ //triangular elements	
				caxis = options.getfieldvalue('caxis',[ArrayMin(data),ArrayMax(data)]);
				if (options.getfieldvalue('log','off')!='off') caxis = [Math.log10(caxis[0])/Math.log10(options.getfieldvalue('log',10)),Math.log10(caxis[1])/Math.log10(options.getfieldvalue('log',10))];
				datamin = caxis[0];
				datamax = caxis[1];
				datadelta = datamax - datamin;

				vertices.itemSize = 3;
				texcoords.itemSize = 2;
				
				var xyz = vec3.create();
				var direction = vec3.create();
				var vertex = vec3.create();
				var magnitude;

				for(var i = 0; i < x.length; i++){
					//Check for NaN values and remove from indices array as necessary, but preserve vertex array spacing
					if (isNaN(x[i]) || isNaN(y[i]) || isNaN(z[i]) || isNaN(data[i])) {
						nanindices[i] = i;
						vertices[vertices.length] = vertex[0];
						vertices[vertices.length] = vertex[1];
						vertices[vertices.length] = vertex[2];
						
						texcoords[texcoords.length] = 0.0;
						texcoords[texcoords.length] = 0.0;
						continue;
					}
					//Scale vertices
					xyz = vec3.fromValues(x[i], y[i], z[i]);
					magnitude = vec3.length(xyz) + md.geometry.surface[i] * vertexscale;
					vec3.normalize(direction, xyz);
					vec3.scale(vertex, direction, magnitude);
					vertices[vertices.length] = vertex[0];
					vertices[vertices.length] = vertex[1];
					vertices[vertices.length] = vertex[2];

					texcoords[texcoords.length] = 0.5;
					texcoords[texcoords.length] = (data[i] - datamin) / datadelta;
				}

				//linearize the elements array: 
				var element;
				for(var i = 0; i < elements.length; i++){
					element = [elements[i][0] - 1, elements[i][1] - 1, elements[i][2] - 1];
					if (element[0] in nanindices || element[1] in nanindices || element[2] in nanindices) continue;
					indices[indices.length] = element[0];
					indices[indices.length] = element[1];
					indices[indices.length] = element[2];
				}
				indices.itemSize = 1;

			}
		
			//Initalize buffers
			node["arrays"] = [vertices, texcoords, indices];
			node["buffers"] = initBuffers(gl,node["arrays"]);
			break;
		//}}}
		//quiver plot {{{
		case 3:
			if (is2d){
				//plot_quiver(x,y,data(:,1),data(:,2),options);
			}
			else{
				//plot_quiver3(x,y,z,data(:,1),data(:,2),data(:,3),options);
			}
			break;
		//}}}
		//node transient plot {{{
		case 5:
			if (elements[0].length==6){ //prisms
			}
			else if (elements[0].length==4){//tetras
			}
			else{ //triangular elements
				vertices.itemSize = 3;
				
				var xyz = vec3.create();
				var direction = vec3.create();
				var vertex = vec3.create();
				var magnitude;
				var timestamps = data[data.length-1];
				for(var i = 0; i < x.length; i++){
					//Check for NaN values and remove from indices array as necessary, but preserve vertex array spacing
					if (isNaN(x[i]) || isNaN(y[i]) || isNaN(z[i]) || isNaN(data[i][0])) {
						nanindices[i] = i;
						vertices[vertices.length] = vertex[0];
						vertices[vertices.length] = vertex[1];
						vertices[vertices.length] = vertex[2];
						
						texcoords[texcoords.length] = 0.0;
						texcoords[texcoords.length] = 0.0;
						continue;
					}
					//Scale vertices
					xyz = vec3.fromValues(x[i], y[i], z[i]);
					magnitude = vec3.length(xyz) + md.geometry.surface[i] * vertexscale;
					vec3.normalize(direction, xyz);
					vec3.scale(vertex, direction, magnitude);
					vertices[vertices.length] = vertex[0];
					vertices[vertices.length] = vertex[1];
					vertices[vertices.length] = vertex[2];
				}	
				//Transpose data to obtain column addressable data matrix
				data = data[0].map(function(col, i) { 
					return data.map(function(row) { 
						return row[i]
					})
				});
				//Prevent evaluation of datasubarray min/max if caxis exists
				if (options.exist('caxis')) {
					caxis = options.getfieldvalue('caxis');
				}
				else {
					caxis = [ArrayMin(data[0]),ArrayMax(data[0].slice(0,-1))];
				}
				if (options.getfieldvalue('log','off')!='off') caxis = [Math.log10(caxis[0])/Math.log10(options.getfieldvalue('log',10)),Math.log10(caxis[1])/Math.log10(options.getfieldvalue('log',10))];
				
				for(var i = 0; i < data.length; i++){					
					datamin = caxis[0];
					datamax = caxis[1];
					datadelta = datamax - datamin;

					//Precalculate arrays for each datasubarray, trimming off timestamp value by using x.length instead of data[i].length
					texcoords[i] = [];
					texcoords[i].itemSize = 2;
					for(var j = 0; j < x.length; j++){
						texcoords[i][texcoords[i].length] = 0.5;
						texcoords[i][texcoords[i].length] = (data[i][j] - datamin) / datadelta;
					}
				}
				
				//linearize the elements array:
				var element;
				for(var i = 0; i < elements.length; i++){
					element = [elements[i][0] - 1, elements[i][1] - 1, elements[i][2] - 1];
					if (element[0] in nanindices || element[1] in nanindices || element[2] in nanindices) continue;
					indices[indices.length] = element[0];
					indices[indices.length] = element[1];
					indices[indices.length] = element[2];
				}
				indices.itemSize = 1;
				
				//Initialize movie loop
				node["movieInterval"] = 1000 / canvas.moviefps;
				node["movieTimestamps"] = timestamps;
				node["movieLength"] = timestamps.length;
				node["movieFrame"] = 0;
			
				if (canvas["movieHandler"])	clearInterval(canvas["movieHandler"]);
				canvas["movieHandler"] = setInterval(function () {
						if (canvas["moviePlay"]) {
							node["movieFrame"] = canvas["movieFrame"];
							if (canvas["movieIncrement"]) {
								if (canvas["movieReverse"]) {
									node["movieFrame"] = (((node["movieFrame"] - 1) % node["movieLength"]) + node["movieLength"]) % node["movieLength"]; //Handle negative modulus
								}
								else {
									node["movieFrame"] = (((node["movieFrame"] + 1) % node["movieLength"]) + node["movieLength"]) % node["movieLength"]; //Handle negative modulus
								}
							}
							if (canvas["timeLabel"]) canvas["timeLabel"].html(String(node["movieTimestamps"][node["movieFrame"]]) + " " + options.getfieldvalue("movietimeunit","yr"));
							if (canvas["progressBar"]) canvas["progressBar"].slider("value", node["movieFrame"]);
							node["buffers"] = initBuffers(gl,[node["arrays"][0],node["arrays"][1][node["movieFrame"]],node["arrays"][2]]);
							canvas["movieFrame"] = node["movieFrame"];
						}
					}, node["movieInterval"]);
				if (canvas["progressBar"]) {
					canvas["movieFrame"] = 0;
					canvas["progressBar"].slider("value", 0);
					canvas["progressBar"].slider("option", {max: node["movieLength"]-1});
				}
			}
			
			//Initalize buffers
			node["arrays"] = [vertices, texcoords, indices];
			node["buffers"] = initBuffers(gl,[node["arrays"][0],node["arrays"][1][0],node["arrays"][2]]);
			break;
		//}}}
		default:
			throw Error(sprintf("%s%i%s\n",'case ',datatype,' not supported'));
	}
}
function plot_quiver(md,options,canvas) {
	//PLOT_QUIVER - quiver plot with colors
	//
	//   Usage:
	//      plot_quiver(md,options,canvas)
	//

	//declare variables:  {{{
	var vertices = [];
	var indices = [];
	var colors = [];
	var rgbcolor = [];
	var xmin,xmax;
	var ymin,ymax;
	var zmin,zmax;
	var scale,matrixscale,vertexscale;
	
	//Process data and model
	var meshresults = processmesh(md,[],options);
	var x = meshresults[0]; 
	var y = meshresults[1]; 
	var z = meshresults[2]; 
	var elements = meshresults[3]; 
	var is2d = meshresults[4]; 
	var isplanet = meshresults[5];
	var vx = md.initialization.vx; 
	var vy = md.initialization.vy; 
		
	//Compue scaling through matrices for 2d meshes and vertices for 3d meshes
	if (!md.geometry.surface) {
		md.geometry.surface=NewArrayFill(md.mesh.x.length,0);
	}
	if (md.mesh.classname() == 'mesh3dsurface') {
		matrixscale = 1;
		vertexscale = options.getfieldvalue('heightscale',1);
	}
	else {
		if (md.geometry.surface) {
			z=md.geometry.surface;
		}	
		matrixscale = options.getfieldvalue('heightscale',1);
		vertexscale = 0;
	}
	//}}}

	//Compute coordinates and data range:
	var modelxlim = [ArrayMin(x),ArrayMax(x)];
	var modelylim = [ArrayMin(y),ArrayMax(y)];
	var modelzlim = [ArrayMin(z),ArrayMax(z)];
	var xlim = options.getfieldvalue('xlim',modelxlim);
	var ylim = options.getfieldvalue('ylim',modelylim);
	var zlim = options.getfieldvalue('zlim',modelzlim);
	xmin = xlim[0];
	xmax = xlim[1];
	ymin = ylim[0];
	ymax = ylim[1];
	zmin = zlim[0];
	zmax = zlim[1];

	//Compute gl variables:
	var gl = canvas.gl;
	var node = Node(gl,options);
	canvas.nodes[canvas.nodes.length] = node;
	scale = 1 / (xmax - xmin);
	node["shaderName"] = "colored";
	node["shader"] = gl["shaders"][node["shaderName"]]["program"];
	node["scale"] = [scale, scale, scale * matrixscale];
	node["translation"] = [(xmin + xmax) / (-2 / scale), (ymin + ymax) / (-2 / scale), (zmin + zmax) / (2 / scale)];
	node["modelMatrix"] = recalculateModelMatrix(node);
	node["drawMode"] = gl.LINES;
	node["useIndexBuffer"] = false;
	node["drawOrder"] = 0;
	node["maskEnabled"] = options.getfieldvalue('innermask','off') == 'on';
	node["maskHeight"] = options.getfieldvalue('innermaskheight',150.0)*options.getfieldvalue('heightscale',1);
	node["maskColor"] = options.getfieldvalue('innermaskcolor',[0.0,0.0,1.0,1.0]);
				
	//some defaults:
	colors.itemSize = 4;

	//retrieve some options
	var linewidth=options.getfieldvalue('linewidth',1);
	
	var edgecolor=new RGBColor(options.getfieldvalue('edgecolor','black'));
	if (edgecolor.ok) edgecolor = [edgecolor.r/255.0, edgecolor.g/255.0, edgecolor.b/255.0, 1.0];
	else throw Error(sprintf("s%s%s\n","initWebGL error message: cound not find out edgecolor color for curent canvas ",canvas));

	//node plot {{{
	if (elements[0].length==6){ //prisms
	}
	else if (elements[0].length==4){ //tetras
	}
	else{ //2D triangular elements
		vertices.itemSize = 3;
		var xyz = vec3.create();
		var direction = vec3.create();
		var vertex = vec3.create();
		var magnitude;
		var color = edgecolor;
		for(var i = 0; i < x.length; i++){
			//Check for NaN values and remove from indices array as necessary, but preserve vertex array spacing
			if (isNaN(x[i]) || isNaN(y[i]) || isNaN(z[i])) continue;
			//Scale vertices
			xyz1 = vec3.fromValues(x[i], y[i], z[i]);
			magnitude = vec3.length(xyz1) + md.geometry.surface[i] * vertexscale;
			vec3.normalize(direction, xyz1);
			vec3.scale(vertex, direction, magnitude);
			vertices[vertices.length] = vertex[0];
			vertices[vertices.length] = vertex[1];
			vertices[vertices.length] = vertex[2];
			
			xyz2 = vec3.fromValues(vx[i], vy[i], 0.0);
			magnitude = 1000.0; //mesh resolution
			vec3.normalize(direction, xyz2);
			vec3.scale(direction, direction, magnitude);
			vec3.add(vertex, direction, vertex);
			vertices[vertices.length] = vertex[0];
			vertices[vertices.length] = vertex[1];
			vertices[vertices.length] = vertex[2];
			
			colors[colors.length] = color[0];
			colors[colors.length] = color[1];
			colors[colors.length] = color[2];
			colors[colors.length] = color[3];
			colors[colors.length] = color[0];
			colors[colors.length] = color[1];
			colors[colors.length] = color[2];
			colors[colors.length] = color[3];
		}
	}
	//}}}

	/*Initalize buffers: */
	node["arrays"] = [vertices, colors];
	node["buffers"] = initBuffers(gl, node["arrays"]);
}
function plotmodel(md){

	//Convert arguments to array: 
	var args = Array.prototype.slice.call(arguments);

	//First process options
	var  options = new plotoptions(args.slice(1,args.length));

	
	//get number of subplots
	subplotwidth=Math.ceil(Math.sqrt(options.numberofplots)); 
	
	//Get figure number and number of plots
	numberofplots=options.numberofplots;

	//if nlines and ncols specified, then bypass.
	var nlines,ncols;
	if (options.list[0].exist('nlines')){
		nlines=options.list[0].getfieldvalue('nlines');
	}
	else {
		nlines=Math.ceil(numberofplots/subplotwidth);
	}
	if (options.list[0].exist('ncols')){
		ncols=options.list[0].getfieldvalue('ncols');
	}
	else {
		ncols=subplotwidth;
	}
	
	//check that nlines and ncols were given at the same time!
	if ((options.list[0].exist('ncols') & !options.list[0].exist('nlines')) | (options.list[0].exist('nlines') & !options.list[0].exist('ncols'))) throw Error('plotmodel error message: nlines and ncols  need to be specified together, or not at all');

	//go through subplots
	if (numberofplots){
		//Reinitialize all canvases
		for (var i=0;i<numberofplots;i++){
			document.getElementById(options.list[i].getfieldvalue('canvasid')).initialized = false;
		}
		//Go through all data plottable and close window if an error occurs
		for (var i=0;i<numberofplots;i++){
			plot_manager(options.list[i].getfieldvalue('model',md),options.list[i],subplotwidth,nlines,ncols,i);

			//List all unused options
			options.list[i].displayunused();
		}
	}
}
function processdata(md,data,options){
	//PROCESSDATA - process data to be plotted
	//
	//   datatype = 1 -> elements
	//   datatype = 2 -> nodes
	//   datatype = 3 -> node quivers
	//   datatype = 4 -> patch
	//   datatype = 5 -> nodes transient
	//
	//   Usage:
	//      var array = processdata(md,data,options);
	//      var data = array[0]; 
	//      var datatype = array[1];
	//
	//   See also: PLOTMODEL, PROCESSMESH

	//variables: 
	var datatype;
	
	//check format
	if ( data.length ==0 | data === [] | typeof data === 'number' | ArrayAnyNaN(data) ){
		throw Error('plotmodel error message: data provided is empty');
	}

	//Process NaN if any (do not know before mask is applied)
	if (options.exist('nan')){
		var valuefornan=options.getfieldvalue('nan',0);
		for (var i=0;i<data.length;i++)if(IsNaN(data[i]))data[i]=valuefornan;
	}

	//special case for mesh 2dvertical
	if (md.mesh.domaintype() === '2Dvertical'){
		return processdata(md.mesh,md,data,options);
	}

	//needed later on
	if ('numberofvertices2d' in md.mesh){
		numberofvertices2d=md.mesh.numberofvertices2d; 
		numberofelements2d=md.mesh.numberofelements2d; 
	}
	else {
		numberofvertices2d=NaN;
		numberofelements2d=NaN;
	}

	//initialize datatype
	datatype=0;

	//get datasize
	datasize=data.length;

	//transpose data if necessary
	if (data[0].length > data.length){
		throw Error('processdata error message: you need to tranpose your data!');
	}


	//check length
	if (datasize != md.mesh.numberofvertices & datasize !=md.mesh.numberofelements & datasize!=md.mesh.numberofvertices*6 & 
			((md.mesh.domaintype() === '3D') & !(datasize==numberofelements2d | datasize==numberofvertices2d))){
		throw Error('plotmodel error message: data not supported yet');
	}


	//quiver?
	if (data[0].length>1){
		datatype=3;

		//check number of columns, add zeros if necessary,
		if (md.mesh.dimension()==3){
			if (data[0].length==2){
				data=[data, NewArrayFill(data.length,1)];
			}
			else if (data[0].length!=3){
				throw Error('plotmodel error message: data provided should have 2 or 3 columns for quiver plot, and 1 for regular plot');
			}
		}
	}

	//treat the case datasize(1)=6*nodes
	if (datasize==6*md.mesh.numberofvertices){
		//keep the only norm of data
		data1=new Array(md.mesh.numberofvertices);
		data2=new Array(md.mesh.numberofvertices);
		data=new Array(md.mesh.numberofvertices);
		for(var i=0;i<md.mesh.numberofvertices;i++){
			data1[i]=data[6*i+0];
			data2[i]=data[6*i+1];
			data[i]=Math.sqrt(pow(data1[i],2),pow(data2[i],2));
		}
		datasize=md.mesh.numberofvertices;
		//---> go to node data
	}

	//treat the case datasize(1)=nodes2d
	if (md.mesh.dimension()==3 & datasize==numberofvertices2d){
		data=project3d(md,'vector',data,'type','node');
		datasize=md.mesh.numberofvertices;
		//---> go to node data
	}

	//treat the case datasize=nodes2d
	if (md.mesh.dimension()==3 & datasize==numberofelements2d){
		data=project3d(md,'vector',data,'type','element');
		datasize=md.mesh.numberofelements;
		//---> go to node data
	}

	//smoothing?
	if (options.exist('smooth')){
		data=averaging(md,data,options.getfieldvalue('smooth'));
		datasize=md.mesh.numberofvertices;
		//---> go to node data
	}

	//element data
	if (datasize==md.mesh.numberofelements & data[0].length==1){

		//Initialize datatype if non patch
		if(datatype!=4 & datatype!=5){
			datatype=1;
		}

		//Mask?
		if(options.exist('mask')){
			flags=options.getfieldvalue('mask');
			if(flags.length==md.mesh.numberofvertices){
				for(var i=0;i<md.mesh.numberofelements;i++){
					var nanify=0;
					for(var j=0;j<md.mesh.elements[0].length;j++){
						if (flags[md.mesh.elements[i][j]-1]==0)nanify=1;
					}
					if(nanify) for(var j=0;j<md.mesh.elements[0].length;j++)data[md.mesh.elements[i][j]-1]=NaN;
				}
			}
			else if (flags.length==md.mesh.numberofelements){
				for(var i=0;i<md.mesh.numberofelements;i++)if (flags[i]==0)data[i]=NaN;
			}
			else{
				console.log('plotmodel warning: mask length not supported yet (supported length are md.mesh.numberofvertices and md.mesh.numberofelements)');
			}
		}

		//log?
		if (options.getfieldvalue('log','off')!='off'){
			var bounds=options.getfieldvalue('caxis',[ArrayMin(data),ArrayMax(data)]);
			for(var i=0;i<md.mesh.numberofelements;i++)if(data[i]<bounds[0])data[i]=bounds[0];
			for(var i=0;i<md.mesh.numberofelements;i++)if(data[i]<=0){
				throw Error("Log option cannot be applied on negative values. Use caxis option (Rignot''s settings: [1.5 max(data)])");
			}
			for(var i=0;i<md.mesh.numberofelements;i++){
				if(!IsNaN(data[i])){
					data[i]=Math.log10(data[i])/Math.log10(options.getfieldvalue('log',10));
				}
			}
		}
	}

	//node data
	if (datasize==md.mesh.numberofvertices){
		datatype=2;

		//Mask?
		if (options.exist('mask')){
			flags=options.getfieldvalue('mask');
			if (flags.length==md.mesh.numberofvertices){
				for(var i=0;i<md.mesh.numberofvertices;i++){
					if(flags[i]==0)data[i]=NaN;
				}
			}
			else if( length(flags)==md.mesh.numberofelements){
				for(var i=0;i<md.mesh.numberofelements;i++){
					if(flags[i]==0){
						for(var j=0;j<md.mesh.elements[0].length;j++){
							data[md.mesh.elements[i][j]-1]=NaN;
						}
					}
				}
			}
			else{
				console.log("plotmodel warning: mask length not supported yet (supported length are md.mesh.numberofvertices and md.mesh.numberofelements");
			}
		}

		//log?
		if (options.getfieldvalue('log','off')!='off'){
			var bounds=options.getfieldvalue('caxis',[ArrayMin(data),ArrayMax(data)]);
			for(var i=0;i<md.mesh.numberofvertices;i++)if(data[i]<bounds[0])data[i]=bounds[0];
			for(var i=0;i<md.mesh.numberofvertices;i++)if(data[i]>bounds[1])data[i]=bounds[1];
			for(var i=0;i<md.mesh.numberofvertices;i++)if(data[i]<=0){
				throw Error("Log option cannot be applied on negative values. Use caxis option (Rignot''s settings: [1.5 max(data)])");
			}
			for(var i=0;i<md.mesh.numberofvertices;i++){
			   data[i]=Math.log10(data[i])/Math.log10(options.getfieldvalue('log',10));
			}
		}
	}
	
	//node transient data
    if (datasize==md.mesh.numberofvertices+1){
        datatype=5;
		
		//log?	
		if (options.getfieldvalue('log','off')!='off'){
			var bounds=options.getfieldvalue('caxis',[ArrayMin(data),ArrayMax(data)]);
			for(var i=0;i<md.mesh.numberofvertices;i++) {
				for(var j=0;j<data[i].length;j++) {
					if(data[i][j]<bounds[0])data[i][j]=bounds[0];
				}
			}
			for(var i=0;i<md.mesh.numberofvertices;i++) {
				for(var j=0;j<data[i].length;j++) {
					if(data[i][j]>bounds[1])data[i][j]=bounds[1];
				}
			}
			for(var i=0;i<md.mesh.numberofvertices;i++) {
				for(var j=0;j<data[i].length;j++) {
					if(data[i][j]<=0) {
						throw Error("Log option cannot be applied on negative values. Use caxis option (Rignot''s settings: [1.5 max(data)])");
					}
				}
			}
			for(var i=0;i<md.mesh.numberofvertices;i++){
				for(var j=0;j<data[i].length;j++) {
					data[i][j]=Math.log10(data[i][j])/Math.log10(options.getfieldvalue('log',10));
				}
			}
		}
    }
	

	//layer projection? 
	if (options.getfieldvalue('layer',0)>=1){
		data=project2d(md,data,options.getfieldvalue('layer')); //project onto 2d mesh
	}

	//control arrow density if quiverplot: not done yet since conversion of matlab to javascript.
	/*if (datatype==3 & options.exist('density')){
		databak=data;
		data=NewArrayFill(datasize,NaN);

		density=options.getfieldvalue('density');
		data(1:density:end,:)=databak(1:density:end,:);
		clear databak
	}*/

	/*if (datatype==3){ //not done yet
		//Mask?
		if (options.exist('mask')){
			flags=options.getfieldvalue('mask');
			pos=find(~flags);
			if (flags.length==md.mesh.numberofvertices){
			   data(pos,:)=NaN;
			}
			else if (flags.length==md.mesh.numberofelements){
				data(md.mesh.elements(pos,:),:)=NaN;
			}
			else{
				console.log("plotmodel warning: mask length not supported yet (supported length are md.mesh.numberofvertices and md.mesh.numberofelements");
			}
		}
	}*/

	//OK, if datatype=0 error out
	if (datatype==0){
	   throw Error('data provided not recognized or not supported');
	}

	return [data,datatype];
}
function processmesh(md,data,options){
//PROCESSMESH - process mesh to be plotted
//
//   Usage:
//      var meshresults=processmesh(md,data,options)
//      var x=meshresults[0]; 
//      var y=meshresults[1]; 
//      var z=meshresults[2]; 
//      var elements=meshresults[3]; 
//      var is2d=meshresults[4]; 
//      var isplanet=meshresults[5]; 
//
//   See also: PLOTMODEL, PROCESSDATA

	var x,y,z,elements,is2d,isplanet;

	if (md.mesh.numberofvertices==0){
		throw Error('plot error message: mesh is empty');
	}


	if (md.mesh.numberofvertices==md.mesh.numberofelements){
		throw Error(['plot error message: the number of elements is the same as the number of nodes...']);
	}

	if (options.getfieldvalue('coord','xy') !== 'latlon'){
		x=md.mesh.x.slice();
		if ('x2d' in md.mesh) x2d=md.mesh.x2d.slice();
		y=md.mesh.y.slice();
		if ('y2d' in md.mesh) y2d=md.mesh.y2d.slice();
	}
	else{
		x=md.mesh.long.slice();
		y=md.mesh.lat.slice();
	}

	if ('z' in md.mesh){
		z=md.mesh.z.slice();
	}
	else{
		z=NewArrayFill(x.length,0);
	}
	z=options.getfieldvalue('z',z);
	if (typeof z === 'string'){
		z=md[z];
	}
	
	//TODO: Make deep copy of elements array to prevent unwanted modification of model (slice creates deep copies for primitive types, shallow copies for obejcts)
	if ('elements2d' in md.mesh) elements2d=md.mesh.elements2d.slice();
	elements=md.mesh.elements.slice();

	//is it a 2d plot?
	if (md.mesh.dimension()==2){
		is2d=1;
	}
	else{
		if (options.getfieldvalue('layer',0)>=1){
			is2d=1;
		}
		else{
			is2d=0;
		}
	}

	//layer projection? 
	if (options.getfieldvalue('layer',0)>=1){
		if (options.getfieldvalue('coord','xy') === 'latlon'){
			throw Error('processmesh error message: cannot work with 3D meshes for now');
		}
		
		//we modify the mesh temporarily to a 2d mesh from which the 3d mesh was extruded. 
		x=x2d;
		y=y2d;
		z=NewArrayFill(x2d.length,0);
		elements=elements2d;
	}

	//units
	if (options.exist('unit')){
		unit=options.getfieldvalue('unit');
		x=x*unit;
		y=y*unit;
		z=z*unit;
	}

	//for now, always isplanet = 0, as we don't have the isa capability: 
	//if isa(md,'planet'),
	//	isplanet=1;
	//else
	isplanet=0;
	//end

	return  [x,y,z,elements,is2d,isplanet];
}
function gauge(){
	//Convert arguments to options
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args);

	gaugecanvas=options.getfieldvalue('gaugecanvas','gauge');
	gaugetext=options.getfieldvalue('gaugetext','gauge-text');
	gaugediv=options.getfieldvalue('gaugediv','slr-controldiv');
	gaugevalue=options.getfieldvalue('value',0);
	min=options.getfieldvalue('min',0);
	max=options.getfieldvalue('max',22);
	colors=options.getfieldvalue('colors',[[0.0, "#000000" ], [0.50, "#880000"], [1.0, "#ff0000"]]);
	label=options.getfieldvalue('label','');
	
	$('<canvas id="'+gaugecanvas+'" width=80px height=50%></canvas>').appendTo('#'+gaugediv);
	$('<div id="'+gaugetext+'" class="gaugetext"></div>').appendTo('#'+gaugediv);
	
	var opts = {
		lines: 12, // The number of lines to draw
		angle: 0, // The length of each line
		lineWidth: 0.38, // The line thickness
		pointer: {
			length: 0, // The radius of the inner circle
			strokeWidth: 0, // The rotation offset
			color: '#000000' // Fill color
		},
		limitMax: 'false',   // If true, the pointer will not go past the end of the gauge
		colorStart: '#6F6EA0',   // Colors
		colorStop: '#C0C0DB',    // just experiment with them
		strokeColor: '#EEEEEE',   // to see which ones work best for you
		percentColors: colors,
		generateGradient: true
	};
	var target = document.getElementById(gaugecanvas)
	var gauge = new Gauge(target).setOptions(opts);
	gauge.setTextField(document.getElementById(gaugetext));
	gauge.label = label;
	gauge.textField.render = function(gauge) {
      return this.el.innerHTML = gauge.displayedValue.toFixed(0) + ' ' + gauge.label;
    };
	gauge.minValue = min;
	gauge.maxValue = max;
	gauge.animationSpeed = 1;
	gauge.set(1); // set once to initialize?
	gauge.set(gaugevalue); // set actual value
	
	return gauge;
}
/*This is where we have all our webgl relevant functionality for the plotting routines: */
//{{{ Canvas Initialization
function initCanvas(options) {
	//Initialize open Gl for each canvas, if needed: 
	var canvas=document.getElementById(options.getfieldvalue('canvasid'));
	if (!canvas) {
		canvas = $('<div><canvas id="'+options.getfieldvalue('canvasid')+'" width="'+options.getfieldvalue('canvassize',480)+'" height="'+options.getfieldvalue('canvassize',480)+'"></canvas></div>)')
		canvas.css({'height':String(options.getfieldvalue('canvassize',480)+'px')});
		canvas.appendTo('body');
	}
	if (!canvas.initialized) {
		canvas.gl = initWebGL(canvas,options);
		canvas.nodes = [];
		if (canvas.drawHandler)	window.cancelAnimationFrame(canvas.drawHandler);
		draw(canvas,options);
		canvas.initialized = true;
	}
	return canvas;
}
function initWebGL(canvas,options) { //{{{
	var gl;

	try {
		// Try to grab the standard context. If it fails, fallback to experimental.
		gl = canvas.getContext("webgl") || canvas.getContext("experimental-webgl");
	}
	catch(e) {}

	// If we don't have a GL context, give up now
	if (!gl) {
		alert("Unable to initialize WebGL. Your browser may not support it.");
	}
		
	// Enable depth testing
	gl.enable(gl.DEPTH_TEST);
	// Near things obscure far things
	gl.depthFunc(gl.LEQUAL);
	// Clear the color as well as the depth buffer.
	gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
	// Enable color blending/overlay
	gl.enable(gl.BLEND);

	// Allocate arrays equal to maximium number of attributes used by any one shader
	gl.enableVertexAttribArray(0);
	gl.enableVertexAttribArray(1);

	// Load shaders and store them in gl object
	gl.shaders = loadShaders(gl);
	
	// Add context state variables
	//TODO:Group variables in objects for organization and naming
	canvas.gl = gl;
	canvas.zoomBounds = options.getfieldvalue('zoombounds',[0.001,100.0]);
	canvas.zoomFactor = clamp(options.getfieldvalue('zoomfactor',1.0), canvas.zoomBounds[0], canvas.zoomBounds[1]);
	canvas.zoomLast = canvas.zoomFactor;
	canvas.cameraMatrix = mat4.create();
	canvas.translation = options.getfieldvalue('centeroffset',[0,0,0.0]);
	canvas.rotationAzimuthBounds = options.getfieldvalue('azimuthbounds',[0,360]);
	canvas.rotationElevationBounds = options.getfieldvalue('elevationbounds',[-180,180]);
	canvas.rotationDefault = options.getfieldvalue('view',[0,90]); //0 azimuth - up is north, 90 elevation - looking straight down
	canvas.rotation = canvas.rotationDefault;
	canvas.controlsensitivity = options.getfieldvalue('controlsensitivity',1);
	canvas.twod = options.getfieldvalue('2d','off') == 'on';
	canvas.moviePlay = true;
	canvas.movieReverse = false;
	canvas.movieIncrement = true;
	canvas.moviefps = options.getfieldvalue('moviefps',5);

	if (canvas.twod) {
		canvas.rotationAzimuthBounds = [0,0];
		canvas.rotationElevationBounds = [90,90];
		canvas.rotationDefault = [0,90];
		canvas.rotation = canvas.rotationDefault;
	}
	
	// Add event listeners for canvas
	var displayview = options.getfieldvalue('displayview','off') == 'on';
	var displayzoom = options.getfieldvalue('displayzoom','off') == 'on';
	var mc = new Hammer.Manager(canvas);
	
    mc.add(new Hammer.Pan({threshold:0, pointers:0}));
    mc.add(new Hammer.Pinch({threshold:0})).recognizeWith(mc.get('pan'));
    mc.on("panstart panmove", function (ev) {onPan(ev,canvas,displayview);});
    mc.on("pinchstart pinchmove", function (ev) {onPinch(ev,canvas,displayview);});
	//mc.on("mousewheel DOMMouseScroll", function (ev) {onZoom(ev,canvas,options);});
	
	canvas.addEventListener("mousewheel", function (ev) {onZoom(ev,canvas,displayzoom)}, false);
	canvas.addEventListener("DOMMouseScroll", function (ev) {onZoom(ev,canvas,displayzoom)}, false);
	
	return gl;
} //}}}
function initBuffers(gl,arrays) { //{{{	
	var bufferArray = [];
	for (var i = 0; i < arrays.length; i++) {
		bufferArray[i] = gl.createBuffer();	
		bufferArray[i].itemSize = arrays[i].itemSize;
		bufferArray[i].numItems = arrays[i].length/bufferArray[i].itemSize;
		
		if (bufferArray[i].itemSize > 1) {
			gl.bindBuffer(gl.ARRAY_BUFFER, bufferArray[i]);
			gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(arrays[i]), gl.STATIC_DRAW);
		}
		else {
			//TODO: identify index buffers uniquely (by name)
			gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, bufferArray[i]);
			gl.bufferData(gl.ELEMENT_ARRAY_BUFFER, new Uint16Array(arrays[i]), gl.STATIC_DRAW);
		}
	}	
	return bufferArray;
} //}}}
function initTexture(gl,imageSource) { //{{{
	var texture = gl.createTexture();
	texture.image = new Image();
	texture.isLoaded = false;
	texture.image.onload = function () {
		handleLoadedTexture(gl,texture);
	}
	texture.image.src = imageSource;
	return texture;
} //}}}
function handleLoadedTexture(gl,texture) { //{{{
	gl.activeTexture(gl.TEXTURE0);
	gl.bindTexture(gl.TEXTURE_2D, texture);
	gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, true);
	gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, texture.image);
	gl.generateMipmap(gl.TEXTURE_2D);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.NEAREST_MIPMAP_LINEAR);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
	gl.bindTexture(gl.TEXTURE_2D, null);
	texture.isLoaded = true;
} //}}}
function Node(gl,options) { //{{{
	//Returns a Node object that contains default display states for webgl object
	return {buffers:[],
		shader:gl.shaders["colored"]["program"],
		draw:null,
		hideOcean:false,
		level:0,
		useIndexBuffer:true,
		alpha:1.0,
		disableDepthTest:false, 
		enableCullFace:false,
		cullFace:gl.FRONT,
		drawMode:gl.TRIANGLES,
		texture:null,
		translation:vec3.create(),
		rotation:vec3.fromValues(-90, 0, 0),
		scale:vec3.fromValues(1, 1, 1),
		modelMatrix:mat4.create(),
		shaderName:"colored",
		drawOrder:0,
		maskEnabled:false,
		maskHeight:150.0,
		maskColor:vec4.fromValues(0.0, 0.0, 1.0, 1.0),
		enabled:true,
	};
} //}}}
function recalculateModelMatrix(node) { //{{{
	//TODO: move to 0,0,0, rotate,move back to normal space, then apply transform
	var modelMatrix = mat4.create();

	var scaleMatrix = mat4.create();
	mat4.scale(scaleMatrix, scaleMatrix, node["scale"]);
	mat4.multiply(modelMatrix, scaleMatrix, modelMatrix);

	var translationMatrix = mat4.create();
	mat4.translate(translationMatrix, translationMatrix, node["translation"]); //relative translation
	mat4.multiply(modelMatrix, translationMatrix, modelMatrix);
	
	var zRotationMatrix = mat4.create();	
	mat4.rotate(zRotationMatrix, zRotationMatrix, radians(node["rotation"][2]), [0.0, 0.0, 1.0]);
	mat4.multiply(modelMatrix, zRotationMatrix, modelMatrix);
	var yRotationMatrix = mat4.create();	
	mat4.rotate(yRotationMatrix, yRotationMatrix, radians(node["rotation"][1]), [0.0, 1.0, 0.0]);
	mat4.multiply(modelMatrix, yRotationMatrix, modelMatrix);
	var xRotationMatrix = mat4.create();	
	mat4.rotate(xRotationMatrix, xRotationMatrix, radians(node["rotation"][0]), [1.0, 0.0, 0.0]);
	mat4.multiply(modelMatrix, xRotationMatrix, modelMatrix);

	return modelMatrix;
} //}}}
function radians (degrees) { //{{{
  return degrees * Math.PI / 180;
} //}}}
function degrees (radians) { //{{{
  return radians * 180 / Math.PI;
} //}}}
function clamp(value, min, max) { //{{{
	return Math.max(min, Math.min(value, max));
} //}}}
function recoverview(canvasid,defaultview) { //{{{
	var canvas  = document.getElementById(canvasid);
	if (canvas && canvas.hasOwnProperty("rotation")) {
		return canvas.rotation;
	}
	return defaultview;
} //}}}
function recovercenteroffset(canvasid,defaultcenter) { //{{{
	var canvas  = document.getElementById(canvasid);
	if (canvas && canvas.hasOwnProperty("translation")) {
		return canvas.translation;
	}
	return defaultcenter;
} //}}}
//}}}
//{{{ Shader Loading
function loadShaders(gl) { //{{{
	var shaderNames = ["colored", "unlit_textured"];
	shaders = {};
	shaders["colored"] = {loaded:false, vsh:{}, fsh:{}};
	shaders["colored"]["vsh"]["string"] = 
		['attribute vec3 aVertexPosition;',
		'attribute vec4 aVertexColor;',
		'',
		'uniform mat4 uMVPMatrix;',
		'uniform float uAlpha;',
		'',
		'varying vec4 vColor;',
		'',
		'void main(void) {',
		'	gl_PointSize = 3.0;',
		'	gl_Position = uMVPMatrix * vec4(aVertexPosition.xyz, 1.0);',
		'	vColor = vec4(aVertexColor.xyz, uAlpha);',
		'}'].join('\n');
	shaders["colored"]["fsh"]["string"] =
		['precision mediump float;',
		'',
		'varying vec4 vColor;',
		'',
		'void main(void) {',
		'	gl_FragColor = vColor;',
		'}'].join('\n');
	shaders["unlit_textured"] = {loaded:false, vsh:{}, fsh:{}};
	shaders["unlit_textured"]["vsh"]["string"] = 
		['attribute vec3 aVertexPosition;',
		'attribute vec2 aTextureCoord;',
		'',
		'uniform mat4 uMVPMatrix;',
		'',
		'varying vec2 vTextureCoord;',
		'varying float vZCoord;',
		'',
		'void main(void) {',
		'	gl_PointSize = 3.0;',
		'	gl_Position = uMVPMatrix * vec4(aVertexPosition.xyz, 1.0);',
		'	vTextureCoord = aTextureCoord;',
		'	vZCoord = aVertexPosition.z;',
		'}'].join('\n');
	shaders["unlit_textured"]["fsh"]["string"] =
		['precision mediump float;',
		'',
		'varying vec2 vTextureCoord;',
		'varying float vZCoord;',
		'',
		'uniform sampler2D uColorSampler;',
		'uniform float uAlpha;',
		'uniform bool uMaskEnabled;',
		'uniform float uMaskHeight;',
		'uniform vec4 uMaskColor;',
		'',
		'void main(void) {',
		'	if (uMaskEnabled && (vZCoord < uMaskHeight)) {',
		'		gl_FragColor = vec4(uMaskColor.xyz, uAlpha);',
		'	}',
		'	else {',
		'		gl_FragColor = vec4(texture2D(uColorSampler, vec2(vTextureCoord.s, vTextureCoord.t)).rgb, uAlpha);',
		'	}',
		'}'].join('\n');
	shaderNames.forEach(function(shaderName){
		shaders[shaderName]["vsh"]["shader"] = getShaderByString(gl, shaders[shaderName]["vsh"]["string"], "vsh");
		shaders[shaderName]["fsh"]["shader"] = getShaderByString(gl, shaders[shaderName]["fsh"]["string"], "fsh");

		shaders[shaderName]["program"] = gl.createProgram();
		gl.attachShader(shaders[shaderName]["program"], shaders[shaderName]["vsh"]["shader"]);
		gl.attachShader(shaders[shaderName]["program"], shaders[shaderName]["fsh"]["shader"]);
		gl.linkProgram(shaders[shaderName]["program"]);

		if (!gl.getProgramParameter(shaders[shaderName]["program"], gl.LINK_STATUS)) {
			alert("Could not initialise shaders");
		}

		var vshStringArray = shaders[shaderName]["vsh"]["string"].split("\n");
		var fshStringArray = shaders[shaderName]["fsh"]["string"].split("\n");
		var line = "";
		var property = "";
		for (var i = 0; i < vshStringArray.length; i++) {
			line = vshStringArray[i];
			if (line.search("attribute") != -1) {
				property = nameFromLine(line);
				shaders[shaderName]["program"][property] = gl.getAttribLocation(shaders[shaderName]["program"], property);
			}
			else if (line.search("uniform") != -1) {
				property = nameFromLine(line);
				shaders[shaderName]["program"][property] = gl.getUniformLocation(shaders[shaderName]["program"], property);
			}
			else if (line.search("void main") != -1) {
				break;
			}
		}
		for (var i = 0; i < fshStringArray.length; i++) {
			line = fshStringArray[i];
			if (line.search("uniform") != -1) {
				property = nameFromLine(line);
				shaders[shaderName]["program"][property] = gl.getUniformLocation(shaders[shaderName]["program"], property);
			}
			else if (line.search("void main") != -1) {
				break;
			}
		}
		shaders[shaderName]["loaded"] = true;
	});
	return shaders;
} //}}}
function getShaderByString(gl,str,type) { //{{{
	var shader;
	if (type == "fsh") {
		shader = gl.createShader(gl.FRAGMENT_SHADER);
	}
	else if (type == "vsh") {
		shader = gl.createShader(gl.VERTEX_SHADER);
	}
	else {
		return null;
	}
	
	gl.shaderSource(shader, str);
	gl.compileShader(shader);

	if (!gl.getShaderParameter(shader, gl.COMPILE_STATUS)) {	
		alert(gl.getShaderInfoLog(shader));
		return null;
	}

	return shader;
} //}}}
function nameFromLine(line) { //{{{
	//returns lowerCamelCase property name from shader line
	var fullName = line.split(" ")[2];
	return fullName.slice(0, fullName.search(";"));
} //}}}
//}}}
//{{{ Interface Functions
function onPan(ev,canvas,displaylog) { //{{{
	ev.preventDefault();
	if (ev.type == 'panstart') {
		canvas.lastDeltaX = 0;
		canvas.lastDeltaY = 0;
	}
	if (ev.srcEvent.shiftKey || ev.pointers.length == 2) {
		var deltaX = (canvas.lastDeltaX - ev.deltaX) / canvas.clientWidth / canvas.zoomFactor * -2 * canvas.controlsensitivity;
		var deltaY = (canvas.lastDeltaY - ev.deltaY) / canvas.clientHeight / canvas.zoomFactor * -2 * canvas.controlsensitivity;
		
		canvas.translation[0] += Math.cos(radians(canvas.rotation[0])) * deltaX - Math.sin(radians(canvas.rotation[0])) * deltaY;
		canvas.translation[1] += Math.sin(radians(canvas.rotation[0])) * deltaX + Math.cos(radians(canvas.rotation[0])) * deltaY;
	}
	else {
		canvas.rotation[0] += degrees((canvas.lastDeltaX - ev.deltaX) / canvas.clientWidth / canvas.zoomFactor * -2 * canvas.controlsensitivity);
		canvas.rotation[1] += degrees((canvas.lastDeltaY - ev.deltaY) / canvas.clientHeight / canvas.zoomFactor * -2 * canvas.controlsensitivity);
		
		if (canvas.rotation[0] > 360) {canvas.rotation[0] -= 360};
		if (canvas.rotation[0] < -360) {canvas.rotation[0] += 360};
		if (canvas.rotation[1] > 180) {canvas.rotation[1] -= 360};
		if (canvas.rotation[1] < -180) {canvas.rotation[1] += 360};
		
		canvas.rotation[0] = clamp(canvas.rotation[0], canvas.rotationAzimuthBounds[0], canvas.rotationAzimuthBounds[1]);
		canvas.rotation[1] = clamp(canvas.rotation[1], canvas.rotationElevationBounds[0], canvas.rotationElevationBounds[1])
	}
	canvas.lastDeltaX = ev.deltaX;
	canvas.lastDeltaY = ev.deltaY;
	
	if (displaylog) console.log(canvas.rotation);
} //}}}
function onPinch(ev,canvas,displaylog) { //{{{
	ev.preventDefault();
	if (ev.type == 'pinchstart') {
		canvas.zoomLast = canvas.zoomFactor;
	}
	else {
		canvas.zoomFactor = clamp(ev.scale * canvas.zoomLast, canvas.zoomBounds[0], canvas.zoomBounds[1]);
		if (displaylog) console.log(canvas.zoomFactor);
	}
} //}}}
function onZoom(ev,canvas,displaylog) { //{{{
	ev.preventDefault();
	var delta = clamp(clamp(ev.scale || ev.wheelDelta || -ev.detail, -1, 1) * canvas.controlsensitivity * canvas.zoomFactor / 20, -1.0, 1.0);
	canvas.zoomFactor = clamp(canvas.zoomFactor + delta, canvas.zoomBounds[0], canvas.zoomBounds[1]);
	
	if (displaylog) console.log(canvas.zoomFactor);
} //}}}
//}}}
//{{{ Drawing Functions
function updateCameraMatrix(canvas) { //{{{
    //Update view matrix and multiply with projection matrix to get the view-projection (camera) matrix.
	var vMatrix = mat4.create();
	var pMatrix = mat4.create();
	var translateMatrix = mat4.create();
	var rotationMatrix = mat4.create();
	var azimuthRotationMatrix = mat4.create();
	var elevationRotationMatrix = mat4.create();

	if (canvas.twod) {
		mat4.ortho(pMatrix, -1/canvas.zoomFactor, 1/canvas.zoomFactor, -1/canvas.zoomFactor, 1/canvas.zoomFactor, -1.0, 10000.0);
	}
	else {
		mat4.perspective(pMatrix, 60 * Math.PI / 180, canvas.clientWidth / canvas.clientHeight, 0.001, 10000.0);
	}
	
	//Apply worldspace translation
	mat4.translate(translateMatrix, translateMatrix, [canvas.translation[0], canvas.translation[2], canvas.translation[1]]);
	mat4.multiply(vMatrix, translateMatrix, vMatrix);
	
	//Calculate rotation around camera focal point about worldspace origin
	mat4.rotate(azimuthRotationMatrix, azimuthRotationMatrix, radians(canvas.rotation[0]), [0, 1, 0]);
	mat4.rotate(elevationRotationMatrix, elevationRotationMatrix, radians(canvas.rotation[1]), [1, 0, 0]);
	mat4.multiply(rotationMatrix, elevationRotationMatrix, azimuthRotationMatrix);
	
	//Apply rotation and scaling transform
	mat4.multiply(vMatrix, rotationMatrix, vMatrix);
	
	//Apply screenspace translation
	mat4.identity(translateMatrix);
	mat4.translate(translateMatrix, translateMatrix, [0.0, 0.0, -1/canvas.zoomFactor]);
	mat4.multiply(vMatrix, translateMatrix, vMatrix);

	//Apply projection matrix to get camera matrix
	mat4.multiply(canvas.cameraMatrix, pMatrix, vMatrix);
}//}}}
function drawSceneGraphNode(canvas,node) { //{{{
	if (!node["enabled"]) {
		return;
	}
	if (node["texture"]) {
		if (!node["texture"]["isLoaded"]) {
			return;
		}
	}
	var gl = canvas.gl;
	bindAttributes(gl, node["shader"], node["buffers"]);
	var mvpMatrix = mat4.create();
	mat4.multiply(mvpMatrix, canvas.cameraMatrix, node["modelMatrix"]);
	gl.uniformMatrix4fv(node["shader"]["uMVPMatrix"], false, mvpMatrix);
	gl.uniform1f(node["shader"]["uAlpha"], node["alpha"]);
	gl.uniform1i(node["shader"]["uMaskEnabled"], node["maskEnabled"]);
	gl.uniform1f(node["shader"]["uMaskHeight"], node["maskHeight"]);
	gl.uniform4fv(node["shader"]["uMaskColor"], node["maskColor"]);
	if (node["texture"]) {
		gl.activeTexture(gl.TEXTURE0);
		gl.bindTexture(gl.TEXTURE_2D, node["texture"]);
		gl.uniform1i(node["shader"]["uColorSampler"], 0);	
	}
	if (node["disableDepthTest"]) {
		gl.disable(gl.DEPTH_TEST);
	}
	gl.blendFunc (gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA);
	if  (node["useIndexBuffer"] == true) {
		gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, node["buffers"][node["buffers"].length - 1]);
		gl.drawElements(node["drawMode"], node["buffers"][node["buffers"].length - 1].numItems, gl.UNSIGNED_SHORT, 0);
	}
	else {
		gl.drawArrays(node["drawMode"], 0, node["buffers"][0].numItems);
	}	
	gl.enable(gl.DEPTH_TEST);
} //}}}
function bindAttributes(gl,shaderProgram,bufferArray) { //{{{
	gl.useProgram(shaderProgram);
	var arrayNumber = 0;
	for (var propertyName in shaderProgram) {
		if (propertyName[0] == "a") {
			if (bufferArray[arrayNumber].itemSize > 1) {
				gl.bindBuffer(gl.ARRAY_BUFFER, bufferArray[arrayNumber]);
				gl.vertexAttribPointer(shaderProgram[propertyName], bufferArray[arrayNumber].itemSize, gl.FLOAT, false, 0, 0);
				arrayNumber++;
			}
		}
	}
} //}}}
function draw(canvas,options) { //{{{
	if (!canvas.nodes) {
		canvas.drawHandler = window.requestAnimationFrame(function(time) {draw(canvas,options)});
		return;
	}
	// Ensure canvas and gl viewport sizes are the same
	var displayWidth  = canvas.clientWidth;
	var displayHeight = canvas.clientHeight;
	if (canvas.width  != displayWidth || canvas.height != displayHeight) {
		canvas.width  = displayWidth;
		canvas.height = displayHeight;
		canvas.gl.viewport(0, 0, canvas.width, canvas.height);
	}
	
	if (canvas.textcanvas) {
		canvas.textcanvas.draw(canvas);
	}

	// Set clear color to black, fully opaque
	var gl = canvas.gl;
	var backgroundcolor = new RGBColor(options.getfieldvalue('backgroundcolor','lightcyan'));
	if (backgroundcolor.ok) gl.clearColor(backgroundcolor.r/255.0, backgroundcolor.g/255.0, backgroundcolor.b/255.0, 1.0);
	else throw Error(sprintf("s%s%s\n","initWebGL error message: cound not find out background color for curent canvas ",canvas));
		
	// Skip drawing of new frame if any texture is not yet loaded
	var nodes = canvas.nodes;
	for (var node in nodes) {
		if (nodes[node]["texture"] && !nodes[node]["texture"]["isLoaded"]) {
			canvas.drawHandler = window.requestAnimationFrame(function(time) {draw(canvas,options)});
			return;
		}
	}
	// Else, clear the color as well as the depth buffer for new frame
	gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
	
	updateCameraMatrix(canvas);
	
	var drawPassNumber = 2;
	for (var i = drawPassNumber - 1; i >= 0; i--) {
		for (var node in nodes) {
			if (nodes[node]["drawOrder"] == i) {
				drawSceneGraphNode(canvas,nodes[node]);
			}
		}
	}
	canvas.drawHandler = window.requestAnimationFrame(function(time) {draw(canvas,options)});
} //}}}
//}}}
function sprintf() {
  //  discuss at: http://phpjs.org/functions/sprintf/
  // original by: Ash Searle (http://hexmen.com/blog/)
  // improved by: Michael White (http://getsprink.com)
  // improved by: Jack
  // improved by: Kevin van Zonneveld (http://kevin.vanzonneveld.net)
  // improved by: Kevin van Zonneveld (http://kevin.vanzonneveld.net)
  // improved by: Kevin van Zonneveld (http://kevin.vanzonneveld.net)
  // improved by: Dj
  // improved by: Allidylls
  //    input by: Paulo Freitas
  //    input by: Brett Zamir (http://brett-zamir.me)
  //   example 1: sprintf("%01.2f", 123.1);
  //   returns 1: 123.10
  //   example 2: sprintf("[%10s]", 'monkey');
  //   returns 2: '[    monkey]'
  //   example 3: sprintf("[%'#10s]", 'monkey');
  //   returns 3: '[####monkey]'
  //   example 4: sprintf("%d", 123456789012345);
  //   returns 4: '123456789012345'
  //   example 5: sprintf('%-03s', 'E');
  //   returns 5: 'E00'

  var regex = /%%|%(\d+\$)?([\-+\'#0 ]*)(\*\d+\$|\*|\d+)?(?:\.(\*\d+\$|\*|\d+))?([scboxXuideEfFgG])/g;
  var a = arguments;
  var i = 0;
  var format = a[i++];

  // pad()
  var pad = function(str, len, chr, leftJustify) {
    if (!chr) {
      chr = ' ';
    }
    var padding = (str.length >= len) ? '' : new Array(1 + len - str.length >>> 0)
      .join(chr);
    return leftJustify ? str + padding : padding + str;
  };

  // justify()
  var justify = function(value, prefix, leftJustify, minWidth, zeroPad, customPadChar) {
    var diff = minWidth - value.length;
    if (diff > 0) {
      if (leftJustify || !zeroPad) {
        value = pad(value, minWidth, customPadChar, leftJustify);
      } else {
        value = value.slice(0, prefix.length) + pad('', diff, '0', true) + value.slice(prefix.length);
      }
    }
    return value;
  };

  // formatBaseX()
  var formatBaseX = function(value, base, prefix, leftJustify, minWidth, precision, zeroPad) {
    // Note: casts negative numbers to positive ones
    var number = value >>> 0;
    prefix = (prefix && number && {
      '2'  : '0b',
      '8'  : '0',
      '16' : '0x'
    }[base]) || '';
    value = prefix + pad(number.toString(base), precision || 0, '0', false);
    return justify(value, prefix, leftJustify, minWidth, zeroPad);
  };

  // formatString()
  var formatString = function(value, leftJustify, minWidth, precision, zeroPad, customPadChar) {
    if (precision !== null && precision !== undefined) {
      value = value.slice(0, precision);
    }
    return justify(value, '', leftJustify, minWidth, zeroPad, customPadChar);
  };

  // doFormat()
  var doFormat = function(substring, valueIndex, flags, minWidth, precision, type) {
    var number, prefix, method, textTransform, value;

    if (substring === '%%') {
      return '%';
    }

    // parse flags
    var leftJustify = false;
    var positivePrefix = '';
    var zeroPad = false;
    var prefixBaseX = false;
    var customPadChar = ' ';
    var flagsl = flags.length;
    var j;
    for (j = 0; flags && j < flagsl; j++) {
      switch (flags.charAt(j)) {
      case ' ':
        positivePrefix = ' ';
        break;
      case '+':
        positivePrefix = '+';
        break;
      case '-':
        leftJustify = true;
        break;
      case "'":
        customPadChar = flags.charAt(j + 1);
        break;
      case '0':
        zeroPad = true;
        customPadChar = '0';
        break;
      case '#':
        prefixBaseX = true;
        break;
      }
    }

    // parameters may be null, undefined, empty-string or real valued
    // we want to ignore null, undefined and empty-string values
    if (!minWidth) {
      minWidth = 0;
    } else if (minWidth === '*') {
      minWidth = +a[i++];
    } else if (minWidth.charAt(0) === '*') {
      minWidth = +a[minWidth.slice(1, -1)];
    } else {
      minWidth = +minWidth;
    }

    // Note: undocumented perl feature:
    if (minWidth < 0) {
      minWidth = -minWidth;
      leftJustify = true;
    }

    if (!isFinite(minWidth)) {
      throw new Error('sprintf: (minimum-)width must be finite');
    }

    if (!precision) {
      precision = 'fFeE'.indexOf(type) > -1 ? 6 : (type === 'd') ? 0 : undefined;
    } else if (precision === '*') {
      precision = +a[i++];
    } else if (precision.charAt(0) === '*') {
      precision = +a[precision.slice(1, -1)];
    } else {
      precision = +precision;
    }

    // grab value using valueIndex if required?
    value = valueIndex ? a[valueIndex.slice(0, -1)] : a[i++];

    switch (type) {
    case 's':
      return formatString(String(value), leftJustify, minWidth, precision, zeroPad, customPadChar);
    case 'c':
      return formatString(String.fromCharCode(+value), leftJustify, minWidth, precision, zeroPad);
    case 'b':
      return formatBaseX(value, 2, prefixBaseX, leftJustify, minWidth, precision, zeroPad);
    case 'o':
      return formatBaseX(value, 8, prefixBaseX, leftJustify, minWidth, precision, zeroPad);
    case 'x':
      return formatBaseX(value, 16, prefixBaseX, leftJustify, minWidth, precision, zeroPad);
    case 'X':
      return formatBaseX(value, 16, prefixBaseX, leftJustify, minWidth, precision, zeroPad)
        .toUpperCase();
    case 'u':
      return formatBaseX(value, 10, prefixBaseX, leftJustify, minWidth, precision, zeroPad);
    case 'i':
    case 'd':
      number = +value || 0;
      // Plain Math.round doesn't just truncate
      number = Math.round(number - number % 1);
      prefix = number < 0 ? '-' : positivePrefix;
      value = prefix + pad(String(Math.abs(number)), precision, '0', false);
      return justify(value, prefix, leftJustify, minWidth, zeroPad);
    case 'e':
    case 'E':
    case 'f': // Should handle locales (as per setlocale)
    case 'F':
    case 'g':
    case 'G':
      number = +value;
      prefix = number < 0 ? '-' : positivePrefix;
      method = ['toExponential', 'toFixed', 'toPrecision']['efg'.indexOf(type.toLowerCase())];
      textTransform = ['toString', 'toUpperCase']['eEfFgG'.indexOf(type) % 2];
      value = prefix + Math.abs(number)[method](precision);
      return justify(value, prefix, leftJustify, minWidth, zeroPad)[textTransform]();
    default:
      return substring;
    }
  };

  return format.replace(regex, doFormat);
}
function loadresultsfrombuffer(md,buffer,buffersize){
//LOADRESULTSFROMBUFFER - load results of solution sequence from memory buffer 
//
//   Usage:
//      loadresultsfrombuffer(md,buffer,buffersize);

	//check number of inputs/outputs
	if (arguments.length!=3) throw Error('loadresultsfrombuffer error message: wrong usage!');

	if (!md.qmu.isdakota){

		//initialize md.results if not a structure yet
		if (MapIsEmpty(md.results)) md.results={};

		//load results: 
		structure=parseresultsfrombuffer(buffer,buffersize);

		//load structure onto results: 
		solutiontype=structure[0].SolutionType;
		md.results[solutiontype]=structure;

		return md;

	}
	else throw Error('loadresultsfrombuffer error message: qmu results not supported yet!');
}
function marshall(md){
//MARSHALL - outputs a typed array buffer to be send to the issm module.
//
//   The routine creates a compatible binary stream from @model md
//   This binary stream will be used for single cpu runs using the issm module.
//
//   Usage:
//      fid=marshall(md)

	if (md.verbose.solution){
		console.log('marshalling file ' + md.miscellaneous.name + '.bin');
	}

	//open file for binary writing
	var fid=new fileptr('mode','w');

	//First, write MaximumNumberOfEnum to make sure that the Enums are synchronized
	WriteData(fid,'enum',MaximumNumberOfDefinitionsEnum(),'data',true,'format','Boolean');

	//Go through all model fields: check that it is a class and call checkconsistency
	for (field in md){

		//Some properties do not need to be marshalled
		if (field == 'results' | field =='radaroverlay' | field == 'toolkits' | field =='cluster' | field == 'flaim' | field == 'priv') continue;
		
		//Check that current field is a class
		if(typeof md[field] == 'function'){
			continue;
		}

		//Marshall current object
		md[field].marshall(md,fid);
	}

	//Last, write MaximumNumberOfEnum+1 to make sure that the binary file is not corrupt
	WriteData(fid,'enum',MaximumNumberOfDefinitionsEnum()+1,'data',true,'format','Boolean');
	return fid;
}
function parseresultsfrombuffer(buffer,buffersize){ //{{{
//PARSERESULTSFROMDISK - ...
//
//   Usage:
//      results=parseresultsfrombuffer(buffer,buffersize);

	//Open file
	results=[];
	
	var fid = new fileptr('mode','r','buffer',buffer,'buffersize',buffersize);

	//Read fields until the end of the file.
	result  = ReadData(fid);

	if (MapIsEmpty(result))throw Error('no results found in binary memory buffer');
	
	var counter = 0;
	var check_nomoresteps=0;
	var step    = result['step'];
	var index;

	while (!MapIsEmpty(result)){

		if (check_nomoresteps){
			//check that the new result does not add a step, which would be an error: 
			if (result['step']>=1)throw Error('parsing results for a steady-state core, which incorporates transient results!');
		}

		//Check step, increase counter if this is a new step
		if(step!=result['step'] & result['step']>1){
			counter = counter + 1;
			step    = result['step'];
		}

		//Add result
		if(result['step']==0){
			//if we have a step = 0, this is a steady state solutoin, don't expect more steps. 
			index = 0;
			check_nomoresteps=1;
		}
		else if(result['step']==1){
			index = 0;
		}
		else index = counter;

		if(index>results.length-1)results.push({});
		for(var i=results.length-1;i<index-1;i++)results[i]={};
		results[index][result['fieldname']]=result['field'];
		
		//Get time and step
		if(result['time']!=-9999){
			results[index]['step']=result['step'];
		}
		if(result['time']!=-9999){
			results[index]['time']=result['time'];
		}

		//read next result
		result  = ReadData(fid);
	}
	return results;
} // }}}
function ReadData(fid){ //{{{
//READDATA - ...
//
//   Usage:
//      field=ReadData(fid)

	//read field
	var length=fid.fread(1,'int');
	
	var result={};

	if (length!==-1){
		fieldname=fid.fread(length,'char');
		time=fid.fread(1,'double');
		step=fid.fread(1,'int');
		type=fid.fread(1,'int');
		M=fid.fread(1,'int');

		if (type==1) field=fid.fread(M,'double');
		else if (type==2) field=fid.fread(M,'char');
		else if (type==3) {
			N=fid.fread(1,'int');
			field=fid.fread(N*M,'double');
		}
		else throw Error(sprintf("%s%i",'ReadData error message: cannot read data of type ',type));

		//Process units here FIXME: this should not be done here!
		var yts=365.0*24.0*3600.0;
		if (fieldname == 'BalancethicknessThickeningRate') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'HydrologyWaterVx') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'HydrologyWaterVy') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'Vx') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'Vy') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'Vz') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'Vel') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'BasalforcingsGroundediceMeltingRate') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'BasalforcingsFloatingiceMeltingRate') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'TotalSmb') for (var i=0;i<field.length;i++)field[i]= field[i]/Math.pow(10,12)*yts; //(GigaTon/year)
		else if (fieldname == 'SmbMassBalance') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'SmbPrecipitation') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'SmbRunoff') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'SmbCondensation') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'SmbAccumulation') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'SmbMelt') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;
		else if (fieldname == 'CalvingCalvingrate') for (var i=0;i<field.length;i++)field[i]= field[i]*yts;

		result['fieldname']=fieldname;
		result['time']=time;
		if (result['time']!=-9999) result['time']=time/yts;
		result['step']=step;
		result['field']=field;
	}
	return result;

} // }}}
function solve(md,solutionenum){
//SOLVE - apply solution sequence for this model
//
//   Usage:
//      solve(md,solutionenum,varargin)
//      where varargin is a lit of paired arguments of string OR enums
//
//   solution types available comprise:
//		 - StressbalanceSolutionEnum
//		 - MasstransportSolutionEnum
//		 - ThermalSolutionEnum
//		 - SteadystateSolutionEnum
//		 - TransientSolutionEnum
//		 - BalancethicknessSolutionEnum
//		 - BedSlopeSolutionEnum
//		 - SurfaceSlopeSolutionEnum
//		 - HydrologySolutionEnum
//		 - FlaimSolutionEnum
//
//  extra options:
//      - loadonly    : does not solve. only load results
//      - runtimename : true or false (default is true), makes name unique
//      - checkconsistency : 'yes' or 'no' (default is 'yes'), ensures checks on consistency of model
//      - restart: 'directory name (relative to the execution directory) where the restart file is located.
//      - callback: callback function to be called upon receiving the results from the server, or local computations. 
//
//   Examples:
//      md=solve(md,StressbalanceSolutionEnum);

	solutionstring=EnumToString(solutionenum);
	
	//recover and process solve options
	if (solutionstring.slice(-8) !== 'Solution'){
		throw Error(sprintf("%s%s%s\n",'solutionenum ',solutionstring,' not supported!'));
	}
	
	//Process options
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(2,args.length));
	options.addfield('solutionenum',solutionenum);

	//recover some fields
	md.priv.solution=solutionenum;
	cluster=md.cluster;

	//check model consistency
	if (options.getfieldvalue('checkconsistency','yes') == 'yes'){
		if (md.verbose.solution){
			console.log('checking model consistency');
		}
		if (solutionenum == FlaimSolutionEnum()){
			md.priv.isconsistent=true;
			md.mesh.checkconsistency(md,solutionenum);
			md.flaim.checkconsistency(md,solutionenum);
			if (md.priv.isconsistent==false){
				throw error('solve error message: model not consistent, see messages above');
			}
		}
		else{
			ismodelselfconsistent(md);
		}
	}

	//If we are restarting, actually use the provided runtime name:
	restart=options.getfieldvalue('restart','');

	//First, build a runtime name that is unique
	if (restart==1 ){
		//Leave the runtimename as is
		}
	else{
		if (!(restart == '')){
			md.priv.runtimename=restart;
		}
		else if (options.getfieldvalue('runtimename',true)){
			c=new Date().getTime();
			md.priv.runtimename=sprintf('%s-%g',md.miscellaneous.name,c);
		}
		else{
			md.priv.runtimename=md.miscellaneous.name;
		}
	}

	//if running qmu analysis, some preprocessing of dakota files using models
	//fields needs to be carried out. 
	if (md.qmu.isdakota){
		throw Error("solve error message: qmu runs not supported yet!");
		//md.preqmu(options);
	}


	//Do we load results only?
	if (options.getfieldvalue('loadonly',false)){
		loadresultsfromcluster(md);
		return;
	}

	//Marshall into a binary array (fid) all the fields of model.
	var fid = marshall(md);                                          // bin file
	
	//deal with toolkits options: 
	toolkitsstring= md.toolkits.ToolkitsFile(md.miscellaneous.name + '.toolkits'); // toolkits file

	//callback function: 
	function callbackfunction(){}; //default, do nothing if no callback requested.
	if (options.getfieldvalue('callback',false)){
		callbackfunction=options.getfieldvalue('callback');
	}

	if (cluster.classname() == 'local'){  //{{{

		/*We are running locally on the machine, using the issm module:*/
		console.log('running issm locally');
		
		//Call issm:
		var outputs = issm(fid, toolkitsstring, solutionstring, md.miscellaneous.name); 
		
		//Recover output arguments: 
		var outputbuffer = outputs[0]; var outputbuffersize = outputs[1];

		//Load results: 
		md = loadresultsfrombuffer(md,outputbuffer,outputbuffersize); 
		
		//Call back? 
		callbackfunction();

		return md;

	} //}}}
	else { //{{{

		/*We are running somewhere else on a computational server. Send the buffer to that server and retrieve output: */
		cluster.UploadAndRun(md,callbackfunction,fid,toolkitsstring,solutionstring,md.miscellaneous.name,md.priv.runtimename);

		return md;

	} //}}}
}
function WriteData(fid){
//WRITEDATA - write model field into binary buffer 
//
//   Usage:
//      WriteData(fid,varargin);

	//process options
	var args = Array.prototype.slice.call(arguments);
	var options = new pairoptions(args.slice(1,args.length));
	var enumm;
	var dataini;
	var data;

	//Get data properties
	if (options.exist('object')){
		//This is a object field, construct enum and data
		obj       = options.getfieldvalue('object');
		fieldname = options.getfieldvalue('fieldname');
		if (options.exist('enum')){
			enumm = options.getfieldvalue('enum');
		}
		else{
			classname = options.getfieldvalue('class',obj.classname());
			enumm = BuildEnum(classname  + '_' + fieldname);
		}
		dataini  = obj[fieldname];
	}
	else{
		//No processing required
		dataini = options.getfieldvalue('data');
		enumm = options.getfieldvalue('enum');
	}
	if (IsArray(dataini)){
	   data=dataini.slice(0);
	}
	else{
		data=dataini;
	}

	format  = options.getfieldvalue('format');
	mattype = options.getfieldvalue('mattype',0);    //only required for matrices
	timeserieslength = options.getfieldvalue('timeserieslength',-1);

	//Scale data if necesarry
	if (options.exist('scale')){
		scale = options.getfieldvalue('scale');
		if (data.length==timeserieslength){
			if (IsArray(data[0])){
				for(var i=0;i<data.length-1;i++){
					for(var j=0;j<data[0].length;j++){
						data[i][j]=scale*data[i][j];
					}
				}
			}
			else{
				for(var i=0;i<data.length-1;i++){
					data[i]=scale*data[i];
				}
			}
		}
		else{
			if (!IsArray(data)) data=data*scale;
			else ArrayScale(data,scale);
		}
	}

	if(data.length == timeserieslength){
		var yts=365.0*24.0*3600.0;
		if (IsArray(data[0])){
			for(var j=0;j<data[0].length;j++)data[timeserieslength-1][j]=data[timeserieslength-1][j]*yts;
		}
		else data[timeserieslength-1]=data[timeserieslength-1]*yts;
	}

	//Step 1: write the enum to identify this record uniquely
	fid.fwrite(enumm,'int'); 

	//Step 2: write the data itself.
	if (format == 'Boolean'){// {{{
		if(IsArray(data)) throw  Error(sprintf("field '%s' cannot be marshalled as it has more than one element!",EnumToString(enumm)));

		//first write length of record
		fid.fwrite(4+4,'int');  //1 bool (disguised as an int)+code

		//write data code: 
		fid.fwrite(FormatToCode(format),'int'); 

		//now write integer
		fid.fwrite(data,'int');  //send an int, not easy to send a bool
	} // }}}
	else if (format == 'Integer'){ // {{{
		if(IsArray(data)) throw  Error(sprintf("field '%s' cannot be marshalled as it has more than one element!",EnumToString(enumm)));

		//first write length of record
		fid.fwrite(4+4,'int');  //1 integer + code

		//write data code: 
		fid.fwrite(FormatToCode(format),'int'); 

		//now write integer
		fid.fwrite(data,'int'); 
	} // }}}
	else if (format == 'Double'){ // {{{
		if(IsArray(data)) throw  Error(sprintf("field '%s' cannot be marshalled as it has more than one element!",EnumToString(enumm)));

		//first write length of record
		fid.fwrite(8+4,'int');  //1 double+code

		//write data code: 
		fid.fwrite(FormatToCode(format),'int'); 

		//now write double
		fid.fwrite(data,'double'); 
	} // }}}
	else if (format == 'String'){ // {{{
		//first write length of record
		fid.fwrite(data.length+4+4,'int');  //string + string size + code

		//write data code: 
		fid.fwrite(FormatToCode(format),'int'); 

		//now write string
		fid.fwrite(data.length,'int'); 
		fid.fwrite(data,'char'); 
	} // }}}
	else if (format == 'BooleanMat'){ // {{{

		//Get size
		var s=[data.length,1];
		if(IsArray(data[0]))s[1]=data[0].length;

		//if matrix = NaN, then do not write anything
		if (s[0]==1 & s[1]==1 & isNaN(data)){
			s[0]=0; s[1]=0;
		}
		if (s[0]==1 & s[1]==1 & isNaN(data[0])){
			s[0]=0; s[1]=0;
		}

		//first write length of record
		fid.fwrite(4+4+8*s[0]*s[1]+4+4,'int');  //2 integers (32 bits) + the double matrix + code + matrix type

		//write data code and matrix type: 
		fid.fwrite(FormatToCode(format),'int'); 
		fid.fwrite(mattype,'int');

		//now write matrix
		fid.fwrite(s[0],'int'); 
		fid.fwrite(s[1],'int'); 
		if (s[0]*s[1]) fid.fwrite(MatrixToList(data),'double'); //get to the "c" convention, hence the transpose
	} // }}}
	else if (format == 'IntMat'){ // {{{

		//Get size
		var s=[data.length,1];
		if(IsArray(data[0]))s[1]=data[0].length;

		//if matrix = NaN, then do not write anything
		if (s[0]==1 & s[1]==1 & isNaN(data)){
			s[0]=0; s[1]=0;
		}
		if (s[0]==1 & s[1]==1 & isNaN(data[0])){
			s[0]=0; s[1]=0;
		}

		//first write length of record
		fid.fwrite(4+4+8*s[0]*s[1]+4+4,'int');  //2 integers (32 bits) + the double matrix + code + matrix type

		//write data code and matrix type: 
		fid.fwrite(FormatToCode(format),'int'); 
		fid.fwrite(mattype,'int');

		//now write matrix
		fid.fwrite(s[0],'int'); 
		fid.fwrite(s[1],'int'); 
		if (s[0]*s[1]) fid.fwrite(MatrixToList(data),'double'); //get to the "c" convention, hence the transpose

	} // }}}
	else if (format == 'DoubleMat'){ // {{{

		//Get size
		var s=[data.length,1];
		if(IsArray(data[0]))s[1]=data[0].length;
		if(typeof data == 'number'){s[0]=1; s[1]=1}

		//if matrix = NaN, then do not write anything
		if (s[0]==1 & s[1]==1 & isNaN(data)){
			s[0]=0; s[1]=0;
		}
		if (s[0]==1 & s[1]==1 & isNaN(data[0])){
			s[0]=0; s[1]=0;
		}

		//first write length of record
		var recordlength=4+4+8*s[0]*s[1]+4+4; //2 integers (32 bits) + the double matrix + code + matrix type
		if (recordlength>Math.pow(2,31)) throw Error(sprintf("field '%s' cannot be marshalled because it is larger than 2^31 bytes!",EnumToString(enumm)));
		fid.fwrite(recordlength,'int');

		//write data code and matrix type: 
		fid.fwrite(FormatToCode(format),'int'); 
		fid.fwrite(mattype,'int');

		//now write matrix
		fid.fwrite(s[0],'int'); 
		fid.fwrite(s[1],'int'); 
		if (s[0]*s[1]) fid.fwrite(MatrixToList(data),'double'); //get to the "c" convention, hence the transpose
	} // }}}
	else if (format == 'MatArray'){ // {{{

		numrecords=data.length;

		//first get length of record
		recordlength=4+4; //number of records + code
		for (var i=0;i<numrecords;i++){
			matrix=data[i];
			var s=[matrix.length,1];
			if(IsArray(matrix[0]))s[1]=matrix[0].length;

			recordlength=recordlength+4*2+ //row and col of matrix
				s[0]*s[1]*8; //matrix of doubles
		}

		//write length of record
		fid.fwrite(recordlength,'int'); 

		//write data code: 
		fid.fwrite(FormatToCode(format),'int'); 

		//write data, first number of records
		fid.fwrite(numrecords,'int'); 

		//write each matrix: 
		for (var i=0;i<numrecords;i++){
			matrix=data[i];
			var s=[matrix.length,1];
			if(IsArray(matrix[0]))s[1]=matrix[0].length;

			fid.fwrite(s[0],'int'); 
			fid.fwrite(s[1],'int'); 
			fid.fwrite(MatrixToList(matrix),'double');
		}
	} // }}}
	else if (format == 'StringArray'){ // {{{

		//first get length of string array: 
		num=data.length;
		if ((typeof data[0] == 'numeric') & num==1 & isNaN(data[0])){
			num = 0;
		}

		//now get length of record: 
		recordlength=4+4; //for length of array + code
		for (var i=0;i<num;i++){
			string=data[i];
			recordlength=recordlength+4+string.length; //for each string
		}

		//write length of record
		fid.fwrite(recordlength,'int'); 

		//write data code: 
		fid.fwrite(FormatToCode(format),'int'); 

		//now write length of string array
		fid.fwrite(num,'int'); 

		//now write the strings
		for (var i=0;i<num;i++){
			string=data[i];
			fid.fwrite(string.length,'int'); 
			fid.fwrite(string,'char'); 
		}
	} // }}}
	else { 
		throw Error(sprintf("WriteData error message: data type: %s not supported yet! ('%s')",
					format.toString(),EnumToString(enumm)));
	}
}

function BuildEnum(string){ // {{{
	//BUILDENUM - build enum out of string
	//
	//   Usage:
	//      enumm=BuildEnum(string)

	var index;
	while( string.indexOf('_') !=-1){
		index=string.indexOf('_');
		string=string.slice(0,index)+string[index+1].toUpperCase() + string.slice(index+2);
	}

	//take first letter of string and make it uppercase: 
	string = string.charAt(0).toUpperCase() + string.slice(1);

	//Get Enum
	enumm=StringToEnum(string); 

	return enumm;
} // }}}
function FormatToCode(format){ // {{{
	//This routine takes the format string, and hardcodes it into an integer, which 
	//is passed along the record, in order to identify the nature of the dataset being 
	//sent.
	if  (format == 'Boolean') code=1;
	else if (format == 'Integer') code=2;
	else if (format == 'Double') code=3;
	else if (format == 'String') code=4;
	else if (format == 'BooleanMat') code=5;
	else if (format == 'IntMat') code=6;
	else if (format == 'DoubleMat') code=7;
	else if (format == 'MatArray') code=8;
	else if (format == 'StringArray') code=9;
	else throw Error('FormatToCode error message: data type not supported yet!');
	return code;
}// }}}
function issmgslsolver(){
//ISSMSOLVER - 
//
//   Usage:
//      options=issmsolver(varargin);

	//Retrieve options provided in varargin. First convert arguments to array:
	var args = Array.prototype.slice.call(arguments);

	//Then process options
	var  options = new pairoptions(args.slice(1,args.length));

	//default issmoptions options
	var issmoptions={};
	issmoptions['toolkit']='issm';
	issmoptions['mat_type']=options.getfieldvalue('mat_type','dense');
	issmoptions['vec_type']=options.getfieldvalue('vec_type','seq');
	issmoptions['solver_type']=options.getfieldvalue('solver_type','gsl');

	return issmoptions;
}
function TriMesh(md,domain,rifts, area){
/*TriMesh 
	   usage: var array = TriMesh(domain,rifts,area);
	      where: array is made of [index,x,y,segments,segmentmarkers]
		  and index,x,y defines a triangulation, segments is an array made 
	      of exterior segments to the mesh domain outline, segmentmarkers is an array 
		  flagging each segment, domain a js array defining the domain outline  (sames for 
		  rifts) and area is the maximum area desired for any element of the resulting mesh.

		  Ok, for now, we are not dealing with rifts. Also, the domain is made of only one 
		  profile, this to avoid passing a double** pointer to js. 
*/

	//Dynamic allocations: {{{
	//Retrieve domain arrays, and allocate on Module heap: 
	
	//input
	var dx=new Float64Array(domain['x']); var nx=dx.length * dx.BYTES_PER_ELEMENT;
	var dxPtr= Module._malloc(nx); var domainxHeap = new Uint8Array(Module.HEAPU8.buffer,dxPtr,nx);
	domainxHeap.set(new Uint8Array(dx.buffer)); var domainx=domainxHeap.byteOffset;

	var dy=new Float64Array(domain['y']); var ny=dy.length * dy.BYTES_PER_ELEMENT;
	var dyPtr = Module._malloc(ny); var domainyHeap = new Uint8Array(Module.HEAPU8.buffer,dyPtr,ny);
	domainyHeap.set(new Uint8Array(dy.buffer)); var domainy=domainyHeap.byteOffset;
	
	//output
	var nel,indexlinear,index,nods,x,y;
	var pnel= Module._malloc(4); 
	var pindex= Module._malloc(4); 
	var pnods= Module._malloc(4); 
	var px= Module._malloc(4); 
	var py= Module._malloc(4); 
	var psegments= Module._malloc(4); 
	var psegmentmarkers= Module._malloc(4); 
	var pnsegs= Module._malloc(4); 
	//}}}

	//Declare TriMesh module: 
	TriMeshModule = Module.cwrap('TriMeshModule','number',['number','number','number','number','number','number','number','number','number','number','number','number']);
	
	//Call TriMesh module: 
	TriMeshModule(pindex,px,py,pnel,pnods,psegments,psegmentmarkers,pnsegs, domainx,domainy,dx.length,area);
	
	/*Dynamic copying from heap: {{{*/
	//recover mesh: 
	nel = Module.getValue(pnel, 'i32');
	var indexptr = Module.getValue(pindex,'i32');
	indexlinear = Module.HEAPF64.slice(indexptr /8, indexptr/8 + nel*3);
	index = ListToMatrix(indexlinear,3);

	nods = Module.getValue(pnods, 'i32');
	var xptr = Module.getValue(px,'i32');
	var yptr = Module.getValue(py,'i32');
	x = Module.HEAPF64.slice(xptr /8, xptr/8 + nods);
	y = Module.HEAPF64.slice(yptr /8, yptr/8 + nods);
	
	nsegs = Module.getValue(pnsegs, 'i32');
	var segmentsptr = Module.getValue(psegments,'i32');
	segmentslinear = Module.HEAPF64.slice(segmentsptr /8, segmentsptr/8 + nsegs*3);
	segments = ListToMatrix(segmentslinear,3);
	
	var segmentmarkersptr = Module.getValue(psegmentmarkers,'i32');
	segmentmarkers = Module.HEAPF64.slice(segmentmarkersptr /8, segmentmarkersptr/8 + nsegs);
	/*}}}*/

	var return_array=[index,x,y,segments,segmentmarkers];

	/*Free ressources: */
	Module._free(pindex); 
	Module._free(indexlinear); 
	Module._free(px); 
	Module._free(x); 
	Module._free(py); 
	Module._free(y); 
	Module._free(pnel); 
	Module._free(pnods); 
	Module._free(psegments); 
	Module._free(psegmentmarkers); 
	Module._free(pnsegs); 

	return return_array;
}
function NodeConnectivity(elementsin,nods){
/*NodeConnectivity 
	   usage: var md.mesh.vertexconnectivity = NodeConnectivity(md.mesh.elements,md.mesh.numberofvertices);
*/

	//Dynamic allocations: {{{
	//Retrieve elements and allocate on Module heap: 
	
	//input
	var delements=new Int32Array(MatrixToList(elementsin)); var nelements=delements.length * delements.BYTES_PER_ELEMENT;
	var delementsPtr= Module._malloc(nelements); var elementsHeap = new Uint8Array(Module.HEAPU8.buffer,delementsPtr,nelements);
	elementsHeap.set(new Uint8Array(delements.buffer)); var elements=elementsHeap.byteOffset;

	//output
	var width,connectivitylinear,connectivity;
	var pwidth= Module._malloc(4); 
	var pnods= Module._malloc(4); 
	var pconnectivity= Module._malloc(4); 
	var nels=elementsin.length;
	//}}}

	//Declare NodeConnectivity module: 
	NodeConnectivityModule = Module.cwrap('NodeConnectivityModule','number',['number','number','number','number']);
	
	//Call NodeConnectivity module: 
	NodeConnectivityModule(pconnectivity,pnods,pwidth,elements,nels,nods);
	
	/*Dynamic copying from heap: {{{*/
	//recover mesh: 
	width = Module.getValue(pwidth, 'i32');
	var connectivityptr = Module.getValue(pconnectivity,'i32');
	connectivitylinear = Module.HEAPF64.slice(connectivityptr /8, connectivityptr/8 + nods*width);
	connectivity = ListToMatrix(connectivitylinear,width);
	/*}}}*/

	/*Free ressources: */
	Module._free(pconnectivity); 
	Module._free(connectivitylinear); 
	Module._free(pwidth); 
	Module._free(pnods); 

	return connectivity;
}
function ContourToMesh(indexin,xin,yin,contour,interptype,edgevalue){
/* CONTOURTOMESH - Flag the elements or nodes inside a contour;
	
	      Usage: ;
	         [in_nod,in_elem]=ContourToMesh(index,x,y,contourname,interptype,edgevalue);
	
	         index,x,y: mesh triangulation
	         contourname: name of .exp file containing the contours
	         interptype: string definining type of interpolation ('element', or 'node')
	         edgevalue: integer (0, 1 or 2) defining the value associated to the nodes on the edges of the polygons.
	         in_nod: vector of flags (0 or 1), of size nods if interptype is set to 'node' or 'element and node',
	            or of size 0 otherwise.
	         in_elem: vector of flags (0 or 1), of size nel if interptype is set to 'element' or 'element and node', 
	            or of size 0 otherwise.
	
	      Example: 
	         in_nod=ContourToMesh(md.elements,md.x,md.y,'Contour.exp','node',1)
	         in_elements=ContourToMesh(md.elements,md.x,md.y,'Contour.exp','element',0)
	         return_values=ContourToMesh(md.elements,md.x,md.y,'Contour.exp','element and node',0); in_nodes=return_values[0]; in_elements=return_values[1];
*/

	//Dynamic allocations: {{{
	//Retrieve elements and allocate on Module heap: 
	
	//input
	
	var dindex=new Int32Array(MatrixToList(indexin)); var nindex=dindex.length * dindex.BYTES_PER_ELEMENT;
	var dindexPtr= Module._malloc(nindex); var indexHeap = new Uint8Array(Module.HEAPU8.buffer,dindexPtr,nindex);
	indexHeap.set(new Uint8Array(dindex.buffer)); var index=indexHeap.byteOffset;

	var dx=new Float64Array(xin); var nx=dx.length * dx.BYTES_PER_ELEMENT;
	var dxPtr= Module._malloc(nx); var dxHeap = new Uint8Array(Module.HEAPU8.buffer,dxPtr,nx);
	dxHeap.set(new Uint8Array(dx.buffer)); var x=dxHeap.byteOffset;
	
	var dy=new Float64Array(yin); var ny=dy.length * dy.BYTES_PER_ELEMENT;
	var dyPtr= Module._malloc(nx); var dyHeap = new Uint8Array(Module.HEAPU8.buffer,dyPtr,ny);
	dyHeap.set(new Uint8Array(dy.buffer)); var y=dyHeap.byteOffset;
	
	var dcontourx=new Float64Array(contour['x']); var nx=dcontourx.length * dcontourx.BYTES_PER_ELEMENT;
	var dcontourxPtr= Module._malloc(nx); var contourxHeap = new Uint8Array(Module.HEAPU8.buffer,dcontourxPtr,nx);
	contourxHeap.set(new Uint8Array(dcontourx.buffer)); var contourx=contourxHeap.byteOffset;

	var dcontoury=new Float64Array(contour['y']); var ny=dcontoury.length * dcontoury.BYTES_PER_ELEMENT;
	var dcontouryPtr = Module._malloc(ny); var contouryHeap = new Uint8Array(Module.HEAPU8.buffer,dcontouryPtr,ny);
	contouryHeap.set(new Uint8Array(dcontoury.buffer)); var contoury=contouryHeap.byteOffset;
	
	nel=indexin.length;
	nods=xin.length;
	contour_nods=dcontourx.length;

	//output
	var in_nod;
	var pin_nod= Module._malloc(4); 
	var in_nel;
	var pin_nel= Module._malloc(4); 
	//}}}

	//Declare ContourToMesh module: 
	ContourToMeshModule = Module.cwrap('ContourToMeshModule','number',['number','number','number','number','number','number','number','string','number','number','number']);
	
	//Call ContourToMesh module: 
	ContourToMeshModule(pin_nod,pin_nel,index,x,y,contourx,contoury,interptype,nel, nods, contour_nods, edgevalue);

	/*Dynamic copying from heap: {{{*/
	if(interptype == 'node'){
		var in_nodptr = Module.getValue(pin_nod,'i32');
		in_nod = Module.HEAPF64.slice(in_nodptr /8, in_nodptr/8 + nods);
	}
	else if (interptype == 'element'){
		var in_nelptr = Module.getValue(pin_nel,'i32');
		in_nel = Module.HEAPF64.slice(in_nelptr /8, in_nelptr/8 + nel);
	}
	else if (interptype == 'element and node'){
		var in_nodptr = Module.getValue(pin_nod,'i32');
		in_nod = Module.HEAPF64.slice(in_nodptr /8, in_nodptr/8 + nods);
		var in_nelptr = Module.getValue(pin_nel,'i32');
		in_nel = Module.HEAPF64.slice(in_nelptr /8, in_nelptr/8 + nel);
	}
	else throw Error('ContourToMeshModule error message: wrong interpolation type!');
	/*}}}*/

	/*Free ressources: */
	Module._free(pin_nod); 
	Module._free(pin_nel); 
	
	if(interptype == 'node'){
		return in_nod;
	}
	else if (interptype == 'element'){
		return in_nel;
	}
	else if (interptype == 'element and node'){
		return [in_nod,in_nel];
	}
}
function ElementConnectivity(elementsin,nodeconnectivityin){
/*ElementConnectivity 
	   usage: var md.mesh.elementconnectivity= ElementConnectivity(md.mesh.elements,md.mesh.vertexconnectivity);
*/

	//Dynamic allocations: {{{
	//Retrieve elements and allocate on Module heap: 
	
	//input
	nel=elementsin.length;
	nods=nodeconnectivityin.length;
	width=nodeconnectivityin[0].length;
	
	var delements=new Int32Array(MatrixToList(elementsin)); var nelements=delements.length * delements.BYTES_PER_ELEMENT;
	var delementsPtr= Module._malloc(nelements); var elementsHeap = new Uint8Array(Module.HEAPU8.buffer,delementsPtr,nelements);
	elementsHeap.set(new Uint8Array(delements.buffer)); var elements=elementsHeap.byteOffset;
	
	var dnodeconnectivity=new Int32Array(MatrixToList(nodeconnectivityin)); var nnodeconnectivity=dnodeconnectivity.length * dnodeconnectivity.BYTES_PER_ELEMENT;
	var dnodeconnectivityPtr= Module._malloc(nnodeconnectivity); var nodeconnectivityHeap = new Uint8Array(Module.HEAPU8.buffer,dnodeconnectivityPtr,nnodeconnectivity);
	nodeconnectivityHeap.set(new Uint8Array(dnodeconnectivity.buffer)); var nodeconnectivity=nodeconnectivityHeap.byteOffset;

	//output
	var connectivitylinear,connectivity;
	var pconnectivity= Module._malloc(4); 
	//}}}

	//Declare ElementConnectivity module: 
	ElementConnectivityModule = Module.cwrap('ElementConnectivityModule','number',['number','number','number','number','number','number']);
	
	//Call ElementConnectivity module: 
	ElementConnectivityModule(pconnectivity,elements, nodeconnectivity, nel, nods, width);
	
	/*Dynamic copying from heap: {{{*/
	//recover mesh: 
	var connectivityptr = Module.getValue(pconnectivity,'i32');
	connectivitylinear = Module.HEAPF64.slice(connectivityptr /8, connectivityptr/8 + nel*3);
	connectivity = ListToMatrix(connectivitylinear,3);
	/*}}}*/

	/*Free ressources: */
	Module._free(pconnectivity); 
	Module._free(connectivitylinear); 

	return connectivity;
}
function InterpFromMeshToMesh2d(indexin,xin,yin,datain,x_interpin,y_interpin){

/* INTERFROMMESHTOMESH2D - interpolation from a 2d triangular mesh onto a list of point
  
  This function interpolates a field defined on a Delaunay triangulation onto a list of points.

  Usage:
	  var data_interp=InterpFromMeshToMesh2d(index,x,y,data,x_interp,y_interp); or
	  var data_interp=InterpFromMeshToMesh2d(index,x,y,data,x_interp,y_interp,default_value);

	  index             : index of the mesh where data is defined
	  x,y               : coordinates of the nodes where data is defined
	  data              : matrix holding the data to be interpolated onto the mesh. (one column per field)
	  x_interp,y_interp : coordinates of the points onto which we interpolate.
	  default_value     : default value if point is outsite of triangulation (instead of linear interpolation)
	  data_interp       : vector of mesh interpolated data.

*/

	/*Figure out default_value: */
	if (arguments.length==7)default_value=arguments[6];
	else default_value=0;
	
	//Dynamic allocations: {{{
	//Retrieve elements and allocate on Module heap: 
	
	//input
	
	var dindex=new Int32Array(MatrixToList(indexin)); var nindex=dindex.length * dindex.BYTES_PER_ELEMENT;
	var dindexPtr= Module._malloc(nindex); var indexHeap = new Uint8Array(Module.HEAPU8.buffer,dindexPtr,nindex);
	indexHeap.set(new Uint8Array(dindex.buffer)); var index=indexHeap.byteOffset;

	var dx=new Float64Array(xin); var nx=dx.length * dx.BYTES_PER_ELEMENT;
	var dxPtr= Module._malloc(nx); var dxHeap = new Uint8Array(Module.HEAPU8.buffer,dxPtr,nx);
	dxHeap.set(new Uint8Array(dx.buffer)); var x=dxHeap.byteOffset;
	
	var dy=new Float64Array(yin); var ny=dy.length * dy.BYTES_PER_ELEMENT;
	var dyPtr= Module._malloc(nx); var dyHeap = new Uint8Array(Module.HEAPU8.buffer,dyPtr,ny);
	dyHeap.set(new Uint8Array(dy.buffer)); var y=dyHeap.byteOffset;
	
	var ddata=new Float64Array(datain); var ndata=ddata.length * ddata.BYTES_PER_ELEMENT;
	var ddataPtr= Module._malloc(ndata); var ddataHeap = new Uint8Array(Module.HEAPU8.buffer,ddataPtr,ndata);
	ddataHeap.set(new Uint8Array(ddata.buffer)); var data=ddataHeap.byteOffset;
	
	var dx_interp=new Float64Array(x_interpin); var nx_interp=dx_interp.length * dx_interp.BYTES_PER_ELEMENT;
	var dx_interpPtr= Module._malloc(nx_interp); var dx_interpHeap = new Uint8Array(Module.HEAPU8.buffer,dx_interpPtr,nx_interp);
	dx_interpHeap.set(new Uint8Array(dx_interp.buffer)); var x_interp=dx_interpHeap.byteOffset;
	
	var dy_interp=new Float64Array(y_interpin); var ny_interp=dy_interp.length * dy_interp.BYTES_PER_ELEMENT;
	var dy_interpPtr= Module._malloc(ny_interp); var dy_interpHeap = new Uint8Array(Module.HEAPU8.buffer,dy_interpPtr,ny_interp);
	dy_interpHeap.set(new Uint8Array(dy_interp.buffer)); var y_interp=dy_interpHeap.byteOffset;
	
	nel=indexin.length;
	nods=xin.length;
	nods_interp=x_interpin.length;

	//output
	var data_interp;
	var pdata_interp= Module._malloc(4); 
	//}}}

	//Declare InterpFromMeshToMesh2d module: 
	InterpFromMeshToMesh2dModule = Module.cwrap('InterpFromMeshToMesh2dModule','number',['number','number','number','number','number','number','number','number','number','number','number']);
	
	//Call InterpFromMeshToMesh2d module: 
	InterpFromMeshToMesh2dModule(pdata_interp,index,x,y,data,x_interp,y_interp,nel,nods,nods_interp,default_value);
	
	/*Dynamic copying from heap: {{{*/
	//recover mesh: 
	var data_interpptr = Module.getValue(pdata_interp,'i32');
	data_interp = Module.HEAPF64.slice(data_interpptr /8, data_interpptr/8 + nods_interp);
	/*}}}*/

	/*Free ressources: */
	Module._free(pdata_interp); 

	return data_interp;
}
function IssmConfig(string){
/*IssmConfig 
	   usage: var config = IssmConfig('_HAVE_PETSC_');
*/

	//output
	var pvalue= Module._malloc(4); 
	var psvalue= Module._malloc(4); 

	//Declare IssmConfig module: 
	IssmConfigModule = Module.cwrap('IssmConfigModule','number',['number','string','string']);
	
	//Call IssmConfig module: 
	IssmConfigModule(pvalue, psvalue, string);
	
	/*Dynamic copying from heap: {{{*/
	var value = Module.getValue(pvalue, 'double');
	/*}}}*/

	/*Free ressources: */
	Module._free(pvalue); 

	return value;
}
function EnumToString(enumin){
/*EnumToString 
	   usage: var string = EnumToString(enumin);
*/

	var pstring= Module._malloc(4); 

	//Declare EnumToString module: 
	EnumToStringModule = Module.cwrap('EnumToStringModule','number',['number','number']);
	
	//Call EnumToString module: 
	EnumToStringModule(pstring, enumin);
	
	string_ptr= Module.getValue(pstring,'i32');

	/*Free ressources: */
	Module._free(pstring); 
	
	return UTF8ToString(string_ptr);
}
function StringToEnum(string){
/*StringToEnum 
	   usage: var enum = StringToEnum(string);
*/

	var penum= Module._malloc(4); 

	//Declare StringToEnum module: 
	StringToEnumModule = Module.cwrap('StringToEnumModule','number',['number','string']);
	
	//Call StringToEnum module: 
	StringToEnumModule(penum, string);
	
	enumout= Module.getValue(penum,'double');

	/*Free ressources: */
	Module._free(penum); 
	
	return enumout;
}
function issm(fid,toolkitstring,solutionstring,modelname){
/*issm 
	   usage: var output = issm(fid,toolkitstring);
	      where: fid is a pointer to a memory buffer created by marshall, toolkitstring is a toolkits 
		  string created by ToolkitsToFile and 
		  output is a binary buffer to be read by loadresultsfromcluster.
*/
	
	/*variables: */
	var poutput,output,poutputsize,outputsize;
	var dbinaryPtr,binHeap,binary,binarybuffer,nb

	/*recover input buffer: */
	binarybuffer=fid.rawbuffer(); //binarybuffer is now an arraybuffer
	nb = fid.ptr; //size of array buffer in bytes.

	/*dyanmically allocate the raw buffer onto the Module heap: */
	dbinaryPtr= Module._malloc(nb); binHeap = new Uint8Array(Module.HEAPU8.buffer,dbinaryPtr,nb);
	binHeap.set(new Uint8Array(binarybuffer)); binary=binHeap.byteOffset;

	/*allocate output pointers: */
	poutputsize = Module._malloc(4); 
	poutput = Module._malloc(4); 

	//Declare TriMesh module: 
	issmmodule= Module.cwrap('IssmModule','number',['number','number','number','number','string','string','string']);
	
	//Call issm:
	issmmodule(poutput, poutputsize,binary, nb, toolkitstring,solutionstring,modelname);

	//recover outputs from pointers: 
	var outputsize = Module.getValue(poutputsize,'i32');
	
	var outputptr = Module.getValue(poutput,'i32');
	output = Module.HEAP8.slice(outputptr, outputptr + outputsize);
	
	return [output,outputsize];
}
var Module;if(!Module)Module=(typeof Module!=="undefined"?Module:null)||{};var moduleOverrides={};for(var key in Module){if(Module.hasOwnProperty(key)){moduleOverrides[key]=Module[key]}}var ENVIRONMENT_IS_WEB=typeof window==="object";var ENVIRONMENT_IS_WORKER=typeof importScripts==="function";var ENVIRONMENT_IS_NODE=typeof process==="object"&&typeof require==="function"&&!ENVIRONMENT_IS_WEB&&!ENVIRONMENT_IS_WORKER;var ENVIRONMENT_IS_SHELL=!ENVIRONMENT_IS_WEB&&!ENVIRONMENT_IS_NODE&&!ENVIRONMENT_IS_WORKER;if(ENVIRONMENT_IS_NODE){if(!Module["print"])Module["print"]=function print(x){process["stdout"].write(x+"\n")};if(!Module["printErr"])Module["printErr"]=function printErr(x){process["stderr"].write(x+"\n")};var nodeFS=require("fs");var nodePath=require("path");Module["read"]=function read(filename,binary){filename=nodePath["normalize"](filename);var ret=nodeFS["readFileSync"](filename);if(!ret&&filename!=nodePath["resolve"](filename)){filename=path.join(__dirname,"..","src",filename);ret=nodeFS["readFileSync"](filename)}if(ret&&!binary)ret=ret.toString();return ret};Module["readBinary"]=function readBinary(filename){var ret=Module["read"](filename,true);if(!ret.buffer){ret=new Uint8Array(ret)}assert(ret.buffer);return ret};Module["load"]=function load(f){globalEval(read(f))};if(!Module["thisProgram"]){if(process["argv"].length>1){Module["thisProgram"]=process["argv"][1].replace(/\\/g,"/")}else{Module["thisProgram"]="unknown-program"}}Module["arguments"]=process["argv"].slice(2);if(typeof module!=="undefined"){module["exports"]=Module}process["on"]("uncaughtException",(function(ex){if(!(ex instanceof ExitStatus)){throw ex}}));Module["inspect"]=(function(){return"[Emscripten Module object]"})}else if(ENVIRONMENT_IS_SHELL){if(!Module["print"])Module["print"]=print;if(typeof printErr!="undefined")Module["printErr"]=printErr;if(typeof read!="undefined"){Module["read"]=read}else{Module["read"]=function read(){throw"no read() available (jsc?)"}}Module["readBinary"]=function readBinary(f){if(typeof readbuffer==="function"){return new Uint8Array(readbuffer(f))}var data=read(f,"binary");assert(typeof data==="object");return data};if(typeof scriptArgs!="undefined"){Module["arguments"]=scriptArgs}else if(typeof arguments!="undefined"){Module["arguments"]=arguments}}else if(ENVIRONMENT_IS_WEB||ENVIRONMENT_IS_WORKER){Module["read"]=function read(url){var xhr=new XMLHttpRequest;xhr.open("GET",url,false);xhr.send(null);return xhr.responseText};if(typeof arguments!="undefined"){Module["arguments"]=arguments}if(typeof console!=="undefined"){if(!Module["print"])Module["print"]=function print(x){console.log(x)};if(!Module["printErr"])Module["printErr"]=function printErr(x){console.log(x)}}else{var TRY_USE_DUMP=false;if(!Module["print"])Module["print"]=TRY_USE_DUMP&&typeof dump!=="undefined"?(function(x){dump(x)}):(function(x){})}if(ENVIRONMENT_IS_WORKER){Module["load"]=importScripts}if(typeof Module["setWindowTitle"]==="undefined"){Module["setWindowTitle"]=(function(title){document.title=title})}}else{throw"Unknown runtime environment. Where are we?"}function globalEval(x){eval.call(null,x)}if(!Module["load"]&&Module["read"]){Module["load"]=function load(f){globalEval(Module["read"](f))}}if(!Module["print"]){Module["print"]=(function(){})}if(!Module["printErr"]){Module["printErr"]=Module["print"]}if(!Module["arguments"]){Module["arguments"]=[]}if(!Module["thisProgram"]){Module["thisProgram"]="./this.program"}Module.print=Module["print"];Module.printErr=Module["printErr"];Module["preRun"]=[];Module["postRun"]=[];for(var key in moduleOverrides){if(moduleOverrides.hasOwnProperty(key)){Module[key]=moduleOverrides[key]}}var Runtime={setTempRet0:(function(value){tempRet0=value}),getTempRet0:(function(){return tempRet0}),stackSave:(function(){return STACKTOP}),stackRestore:(function(stackTop){STACKTOP=stackTop}),getNativeTypeSize:(function(type){switch(type){case"i1":case"i8":return 1;case"i16":return 2;case"i32":return 4;case"i64":return 8;case"float":return 4;case"double":return 8;default:{if(type[type.length-1]==="*"){return Runtime.QUANTUM_SIZE}else if(type[0]==="i"){var bits=parseInt(type.substr(1));assert(bits%8===0);return bits/8}else{return 0}}}}),getNativeFieldSize:(function(type){return Math.max(Runtime.getNativeTypeSize(type),Runtime.QUANTUM_SIZE)}),STACK_ALIGN:16,prepVararg:(function(ptr,type){if(type==="double"||type==="i64"){if(ptr&7){assert((ptr&7)===4);ptr+=4}}else{assert((ptr&3)===0)}return ptr}),getAlignSize:(function(type,size,vararg){if(!vararg&&(type=="i64"||type=="double"))return 8;if(!type)return Math.min(size,8);return Math.min(size||(type?Runtime.getNativeFieldSize(type):0),Runtime.QUANTUM_SIZE)}),dynCall:(function(sig,ptr,args){if(args&&args.length){if(!args.splice)args=Array.prototype.slice.call(args);args.splice(0,0,ptr);return Module["dynCall_"+sig].apply(null,args)}else{return Module["dynCall_"+sig].call(null,ptr)}}),functionPointers:[],addFunction:(function(func){for(var i=0;i<Runtime.functionPointers.length;i++){if(!Runtime.functionPointers[i]){Runtime.functionPointers[i]=func;return 2*(1+i)}}throw"Finished up all reserved function pointers. Use a higher value for RESERVED_FUNCTION_POINTERS."}),removeFunction:(function(index){Runtime.functionPointers[(index-2)/2]=null}),warnOnce:(function(text){if(!Runtime.warnOnce.shown)Runtime.warnOnce.shown={};if(!Runtime.warnOnce.shown[text]){Runtime.warnOnce.shown[text]=1;Module.printErr(text)}}),funcWrappers:{},getFuncWrapper:(function(func,sig){assert(sig);if(!Runtime.funcWrappers[sig]){Runtime.funcWrappers[sig]={}}var sigCache=Runtime.funcWrappers[sig];if(!sigCache[func]){sigCache[func]=function dynCall_wrapper(){return Runtime.dynCall(sig,func,arguments)}}return sigCache[func]}),getCompilerSetting:(function(name){throw"You must build with -s RETAIN_COMPILER_SETTINGS=1 for Runtime.getCompilerSetting or emscripten_get_compiler_setting to work"}),stackAlloc:(function(size){var ret=STACKTOP;STACKTOP=STACKTOP+size|0;STACKTOP=STACKTOP+15&-16;return ret}),staticAlloc:(function(size){var ret=STATICTOP;STATICTOP=STATICTOP+size|0;STATICTOP=STATICTOP+15&-16;return ret}),dynamicAlloc:(function(size){var ret=DYNAMICTOP;DYNAMICTOP=DYNAMICTOP+size|0;DYNAMICTOP=DYNAMICTOP+15&-16;if(DYNAMICTOP>=TOTAL_MEMORY){var success=enlargeMemory();if(!success){DYNAMICTOP=ret;return 0}}return ret}),alignMemory:(function(size,quantum){var ret=size=Math.ceil(size/(quantum?quantum:16))*(quantum?quantum:16);return ret}),makeBigInt:(function(low,high,unsigned){var ret=unsigned?+(low>>>0)+ +(high>>>0)*+4294967296:+(low>>>0)+ +(high|0)*+4294967296;return ret}),GLOBAL_BASE:8,QUANTUM_SIZE:4,__dummy__:0};Module["Runtime"]=Runtime;var __THREW__=0;var ABORT=false;var EXITSTATUS=0;var undef=0;var tempValue,tempInt,tempBigInt,tempInt2,tempBigInt2,tempPair,tempBigIntI,tempBigIntR,tempBigIntS,tempBigIntP,tempBigIntD,tempDouble,tempFloat;var tempI64,tempI64b;var tempRet0,tempRet1,tempRet2,tempRet3,tempRet4,tempRet5,tempRet6,tempRet7,tempRet8,tempRet9;function assert(condition,text){if(!condition){abort("Assertion failed: "+text)}}var globalScope=this;function getCFunc(ident){var func=Module["_"+ident];if(!func){try{func=eval("_"+ident)}catch(e){}}assert(func,"Cannot call unknown function "+ident+" (perhaps LLVM optimizations or closure removed it?)");return func}var cwrap,ccall;((function(){var JSfuncs={"stackSave":(function(){Runtime.stackSave()}),"stackRestore":(function(){Runtime.stackRestore()}),"arrayToC":(function(arr){var ret=Runtime.stackAlloc(arr.length);writeArrayToMemory(arr,ret);return ret}),"stringToC":(function(str){var ret=0;if(str!==null&&str!==undefined&&str!==0){ret=Runtime.stackAlloc((str.length<<2)+1);writeStringToMemory(str,ret)}return ret})};var toC={"string":JSfuncs["stringToC"],"array":JSfuncs["arrayToC"]};ccall=function ccallFunc(ident,returnType,argTypes,args,opts){var func=getCFunc(ident);var cArgs=[];var stack=0;if(args){for(var i=0;i<args.length;i++){var converter=toC[argTypes[i]];if(converter){if(stack===0)stack=Runtime.stackSave();cArgs[i]=converter(args[i])}else{cArgs[i]=args[i]}}}var ret=func.apply(null,cArgs);if(returnType==="string")ret=Pointer_stringify(ret);if(stack!==0){if(opts&&opts.async){EmterpreterAsync.asyncFinalizers.push((function(){Runtime.stackRestore(stack)}));return}Runtime.stackRestore(stack)}return ret};var sourceRegex=/^function\s*\(([^)]*)\)\s*{\s*([^*]*?)[\s;]*(?:return\s*(.*?)[;\s]*)?}$/;function parseJSFunc(jsfunc){var parsed=jsfunc.toString().match(sourceRegex).slice(1);return{arguments:parsed[0],body:parsed[1],returnValue:parsed[2]}}var JSsource={};for(var fun in JSfuncs){if(JSfuncs.hasOwnProperty(fun)){JSsource[fun]=parseJSFunc(JSfuncs[fun])}}cwrap=function cwrap(ident,returnType,argTypes){argTypes=argTypes||[];var cfunc=getCFunc(ident);var numericArgs=argTypes.every((function(type){return type==="number"}));var numericRet=returnType!=="string";if(numericRet&&numericArgs){return cfunc}var argNames=argTypes.map((function(x,i){return"$"+i}));var funcstr="(function("+argNames.join(",")+") {";var nargs=argTypes.length;if(!numericArgs){funcstr+="var stack = "+JSsource["stackSave"].body+";";for(var i=0;i<nargs;i++){var arg=argNames[i],type=argTypes[i];if(type==="number")continue;var convertCode=JSsource[type+"ToC"];funcstr+="var "+convertCode.arguments+" = "+arg+";";funcstr+=convertCode.body+";";funcstr+=arg+"="+convertCode.returnValue+";"}}var cfuncname=parseJSFunc((function(){return cfunc})).returnValue;funcstr+="var ret = "+cfuncname+"("+argNames.join(",")+");";if(!numericRet){var strgfy=parseJSFunc((function(){return Pointer_stringify})).returnValue;funcstr+="ret = "+strgfy+"(ret);"}if(!numericArgs){funcstr+=JSsource["stackRestore"].body.replace("()","(stack)")+";"}funcstr+="return ret})";return eval(funcstr)}}))();Module["ccall"]=ccall;Module["cwrap"]=cwrap;function setValue(ptr,value,type,noSafe){type=type||"i8";if(type.charAt(type.length-1)==="*")type="i32";switch(type){case"i1":HEAP8[ptr>>0]=value;break;case"i8":HEAP8[ptr>>0]=value;break;case"i16":HEAP16[ptr>>1]=value;break;case"i32":HEAP32[ptr>>2]=value;break;case"i64":tempI64=[value>>>0,(tempDouble=value,+Math_abs(tempDouble)>=+1?tempDouble>+0?(Math_min(+Math_floor(tempDouble/+4294967296),+4294967295)|0)>>>0:~~+Math_ceil((tempDouble- +(~~tempDouble>>>0))/+4294967296)>>>0:0)],HEAP32[ptr>>2]=tempI64[0],HEAP32[ptr+4>>2]=tempI64[1];break;case"float":HEAPF32[ptr>>2]=value;break;case"double":HEAPF64[ptr>>3]=value;break;default:abort("invalid type for setValue: "+type)}}Module["setValue"]=setValue;function getValue(ptr,type,noSafe){type=type||"i8";if(type.charAt(type.length-1)==="*")type="i32";switch(type){case"i1":return HEAP8[ptr>>0];case"i8":return HEAP8[ptr>>0];case"i16":return HEAP16[ptr>>1];case"i32":return HEAP32[ptr>>2];case"i64":return HEAP32[ptr>>2];case"float":return HEAPF32[ptr>>2];case"double":return HEAPF64[ptr>>3];default:abort("invalid type for setValue: "+type)}return null}Module["getValue"]=getValue;var ALLOC_NORMAL=0;var ALLOC_STACK=1;var ALLOC_STATIC=2;var ALLOC_DYNAMIC=3;var ALLOC_NONE=4;Module["ALLOC_NORMAL"]=ALLOC_NORMAL;Module["ALLOC_STACK"]=ALLOC_STACK;Module["ALLOC_STATIC"]=ALLOC_STATIC;Module["ALLOC_DYNAMIC"]=ALLOC_DYNAMIC;Module["ALLOC_NONE"]=ALLOC_NONE;function allocate(slab,types,allocator,ptr){var zeroinit,size;if(typeof slab==="number"){zeroinit=true;size=slab}else{zeroinit=false;size=slab.length}var singleType=typeof types==="string"?types:null;var ret;if(allocator==ALLOC_NONE){ret=ptr}else{ret=[_malloc,Runtime.stackAlloc,Runtime.staticAlloc,Runtime.dynamicAlloc][allocator===undefined?ALLOC_STATIC:allocator](Math.max(size,singleType?1:types.length))}if(zeroinit){var ptr=ret,stop;assert((ret&3)==0);stop=ret+(size&~3);for(;ptr<stop;ptr+=4){HEAP32[ptr>>2]=0}stop=ret+size;while(ptr<stop){HEAP8[ptr++>>0]=0}return ret}if(singleType==="i8"){if(slab.subarray||slab.slice){HEAPU8.set(slab,ret)}else{HEAPU8.set(new Uint8Array(slab),ret)}return ret}var i=0,type,typeSize,previousType;while(i<size){var curr=slab[i];if(typeof curr==="function"){curr=Runtime.getFunctionIndex(curr)}type=singleType||types[i];if(type===0){i++;continue}if(type=="i64")type="i32";setValue(ret+i,curr,type);if(previousType!==type){typeSize=Runtime.getNativeTypeSize(type);previousType=type}i+=typeSize}return ret}Module["allocate"]=allocate;function getMemory(size){if(!staticSealed)return Runtime.staticAlloc(size);if(typeof _sbrk!=="undefined"&&!_sbrk.called||!runtimeInitialized)return Runtime.dynamicAlloc(size);return _malloc(size)}Module["getMemory"]=getMemory;function Pointer_stringify(ptr,length){if(length===0||!ptr)return"";var hasUtf=0;var t;var i=0;while(1){t=HEAPU8[ptr+i>>0];hasUtf|=t;if(t==0&&!length)break;i++;if(length&&i==length)break}if(!length)length=i;var ret="";if(hasUtf<128){var MAX_CHUNK=1024;var curr;while(length>0){curr=String.fromCharCode.apply(String,HEAPU8.subarray(ptr,ptr+Math.min(length,MAX_CHUNK)));ret=ret?ret+curr:curr;ptr+=MAX_CHUNK;length-=MAX_CHUNK}return ret}return Module["UTF8ToString"](ptr)}Module["Pointer_stringify"]=Pointer_stringify;function AsciiToString(ptr){var str="";while(1){var ch=HEAP8[ptr++>>0];if(!ch)return str;str+=String.fromCharCode(ch)}}Module["AsciiToString"]=AsciiToString;function stringToAscii(str,outPtr){return writeAsciiToMemory(str,outPtr,false)}Module["stringToAscii"]=stringToAscii;function UTF8ArrayToString(u8Array,idx){var u0,u1,u2,u3,u4,u5;var str="";while(1){u0=u8Array[idx++];if(!u0)return str;if(!(u0&128)){str+=String.fromCharCode(u0);continue}u1=u8Array[idx++]&63;if((u0&224)==192){str+=String.fromCharCode((u0&31)<<6|u1);continue}u2=u8Array[idx++]&63;if((u0&240)==224){u0=(u0&15)<<12|u1<<6|u2}else{u3=u8Array[idx++]&63;if((u0&248)==240){u0=(u0&7)<<18|u1<<12|u2<<6|u3}else{u4=u8Array[idx++]&63;if((u0&252)==248){u0=(u0&3)<<24|u1<<18|u2<<12|u3<<6|u4}else{u5=u8Array[idx++]&63;u0=(u0&1)<<30|u1<<24|u2<<18|u3<<12|u4<<6|u5}}}if(u0<65536){str+=String.fromCharCode(u0)}else{var ch=u0-65536;str+=String.fromCharCode(55296|ch>>10,56320|ch&1023)}}}Module["UTF8ArrayToString"]=UTF8ArrayToString;function UTF8ToString(ptr){return UTF8ArrayToString(HEAPU8,ptr)}Module["UTF8ToString"]=UTF8ToString;function stringToUTF8Array(str,outU8Array,outIdx,maxBytesToWrite){if(!(maxBytesToWrite>0))return 0;var startIdx=outIdx;var endIdx=outIdx+maxBytesToWrite-1;for(var i=0;i<str.length;++i){var u=str.charCodeAt(i);if(u>=55296&&u<=57343)u=65536+((u&1023)<<10)|str.charCodeAt(++i)&1023;if(u<=127){if(outIdx>=endIdx)break;outU8Array[outIdx++]=u}else if(u<=2047){if(outIdx+1>=endIdx)break;outU8Array[outIdx++]=192|u>>6;outU8Array[outIdx++]=128|u&63}else if(u<=65535){if(outIdx+2>=endIdx)break;outU8Array[outIdx++]=224|u>>12;outU8Array[outIdx++]=128|u>>6&63;outU8Array[outIdx++]=128|u&63}else if(u<=2097151){if(outIdx+3>=endIdx)break;outU8Array[outIdx++]=240|u>>18;outU8Array[outIdx++]=128|u>>12&63;outU8Array[outIdx++]=128|u>>6&63;outU8Array[outIdx++]=128|u&63}else if(u<=67108863){if(outIdx+4>=endIdx)break;outU8Array[outIdx++]=248|u>>24;outU8Array[outIdx++]=128|u>>18&63;outU8Array[outIdx++]=128|u>>12&63;outU8Array[outIdx++]=128|u>>6&63;outU8Array[outIdx++]=128|u&63}else{if(outIdx+5>=endIdx)break;outU8Array[outIdx++]=252|u>>30;outU8Array[outIdx++]=128|u>>24&63;outU8Array[outIdx++]=128|u>>18&63;outU8Array[outIdx++]=128|u>>12&63;outU8Array[outIdx++]=128|u>>6&63;outU8Array[outIdx++]=128|u&63}}outU8Array[outIdx]=0;return outIdx-startIdx}Module["stringToUTF8Array"]=stringToUTF8Array;function stringToUTF8(str,outPtr,maxBytesToWrite){return stringToUTF8Array(str,HEAPU8,outPtr,maxBytesToWrite)}Module["stringToUTF8"]=stringToUTF8;function lengthBytesUTF8(str){var len=0;for(var i=0;i<str.length;++i){var u=str.charCodeAt(i);if(u>=55296&&u<=57343)u=65536+((u&1023)<<10)|str.charCodeAt(++i)&1023;if(u<=127){++len}else if(u<=2047){len+=2}else if(u<=65535){len+=3}else if(u<=2097151){len+=4}else if(u<=67108863){len+=5}else{len+=6}}return len}Module["lengthBytesUTF8"]=lengthBytesUTF8;function UTF16ToString(ptr){var i=0;var str="";while(1){var codeUnit=HEAP16[ptr+i*2>>1];if(codeUnit==0)return str;++i;str+=String.fromCharCode(codeUnit)}}Module["UTF16ToString"]=UTF16ToString;function stringToUTF16(str,outPtr,maxBytesToWrite){if(maxBytesToWrite===undefined){maxBytesToWrite=2147483647}if(maxBytesToWrite<2)return 0;maxBytesToWrite-=2;var startPtr=outPtr;var numCharsToWrite=maxBytesToWrite<str.length*2?maxBytesToWrite/2:str.length;for(var i=0;i<numCharsToWrite;++i){var codeUnit=str.charCodeAt(i);HEAP16[outPtr>>1]=codeUnit;outPtr+=2}HEAP16[outPtr>>1]=0;return outPtr-startPtr}Module["stringToUTF16"]=stringToUTF16;function lengthBytesUTF16(str){return str.length*2}Module["lengthBytesUTF16"]=lengthBytesUTF16;function UTF32ToString(ptr){var i=0;var str="";while(1){var utf32=HEAP32[ptr+i*4>>2];if(utf32==0)return str;++i;if(utf32>=65536){var ch=utf32-65536;str+=String.fromCharCode(55296|ch>>10,56320|ch&1023)}else{str+=String.fromCharCode(utf32)}}}Module["UTF32ToString"]=UTF32ToString;function stringToUTF32(str,outPtr,maxBytesToWrite){if(maxBytesToWrite===undefined){maxBytesToWrite=2147483647}if(maxBytesToWrite<4)return 0;var startPtr=outPtr;var endPtr=startPtr+maxBytesToWrite-4;for(var i=0;i<str.length;++i){var codeUnit=str.charCodeAt(i);if(codeUnit>=55296&&codeUnit<=57343){var trailSurrogate=str.charCodeAt(++i);codeUnit=65536+((codeUnit&1023)<<10)|trailSurrogate&1023}HEAP32[outPtr>>2]=codeUnit;outPtr+=4;if(outPtr+4>endPtr)break}HEAP32[outPtr>>2]=0;return outPtr-startPtr}Module["stringToUTF32"]=stringToUTF32;function lengthBytesUTF32(str){var len=0;for(var i=0;i<str.length;++i){var codeUnit=str.charCodeAt(i);if(codeUnit>=55296&&codeUnit<=57343)++i;len+=4}return len}Module["lengthBytesUTF32"]=lengthBytesUTF32;function demangle(func){var hasLibcxxabi=!!Module["___cxa_demangle"];if(hasLibcxxabi){try{var buf=_malloc(func.length);writeStringToMemory(func.substr(1),buf);var status=_malloc(4);var ret=Module["___cxa_demangle"](buf,0,0,status);if(getValue(status,"i32")===0&&ret){return Pointer_stringify(ret)}}catch(e){}finally{if(buf)_free(buf);if(status)_free(status);if(ret)_free(ret)}}var i=3;var basicTypes={"v":"void","b":"bool","c":"char","s":"short","i":"int","l":"long","f":"float","d":"double","w":"wchar_t","a":"signed char","h":"unsigned char","t":"unsigned short","j":"unsigned int","m":"unsigned long","x":"long long","y":"unsigned long long","z":"..."};var subs=[];var first=true;function dump(x){if(x)Module.print(x);Module.print(func);var pre="";for(var a=0;a<i;a++)pre+=" ";Module.print(pre+"^")}function parseNested(){i++;if(func[i]==="K")i++;var parts=[];while(func[i]!=="E"){if(func[i]==="S"){i++;var next=func.indexOf("_",i);var num=func.substring(i,next)||0;parts.push(subs[num]||"?");i=next+1;continue}if(func[i]==="C"){parts.push(parts[parts.length-1]);i+=2;continue}var size=parseInt(func.substr(i));var pre=size.toString().length;if(!size||!pre){i--;break}var curr=func.substr(i+pre,size);parts.push(curr);subs.push(curr);i+=pre+size}i++;return parts}function parse(rawList,limit,allowVoid){limit=limit||Infinity;var ret="",list=[];function flushList(){return"("+list.join(", ")+")"}var name;if(func[i]==="N"){name=parseNested().join("::");limit--;if(limit===0)return rawList?[name]:name}else{if(func[i]==="K"||first&&func[i]==="L")i++;var size=parseInt(func.substr(i));if(size){var pre=size.toString().length;name=func.substr(i+pre,size);i+=pre+size}}first=false;if(func[i]==="I"){i++;var iList=parse(true);var iRet=parse(true,1,true);ret+=iRet[0]+" "+name+"<"+iList.join(", ")+">"}else{ret=name}paramLoop:while(i<func.length&&limit-->0){var c=func[i++];if(c in basicTypes){list.push(basicTypes[c])}else{switch(c){case"P":list.push(parse(true,1,true)[0]+"*");break;case"R":list.push(parse(true,1,true)[0]+"&");break;case"L":{i++;var end=func.indexOf("E",i);var size=end-i;list.push(func.substr(i,size));i+=size+2;break};case"A":{var size=parseInt(func.substr(i));i+=size.toString().length;if(func[i]!=="_")throw"?";i++;list.push(parse(true,1,true)[0]+" ["+size+"]");break};case"E":break paramLoop;default:ret+="?"+c;break paramLoop}}}if(!allowVoid&&list.length===1&&list[0]==="void")list=[];if(rawList){if(ret){list.push(ret+"?")}return list}else{return ret+flushList()}}var parsed=func;try{if(func=="Object._main"||func=="_main"){return"main()"}if(typeof func==="number")func=Pointer_stringify(func);if(func[0]!=="_")return func;if(func[1]!=="_")return func;if(func[2]!=="Z")return func;switch(func[3]){case"n":return"operator new()";case"d":return"operator delete()"}parsed=parse()}catch(e){parsed+="?"}if(parsed.indexOf("?")>=0&&!hasLibcxxabi){Runtime.warnOnce("warning: a problem occurred in builtin C++ name demangling; build with  -s DEMANGLE_SUPPORT=1  to link in libcxxabi demangling")}return parsed}function demangleAll(text){return text.replace(/__Z[\w\d_]+/g,(function(x){var y=demangle(x);return x===y?x:x+" ["+y+"]"}))}function jsStackTrace(){var err=new Error;if(!err.stack){try{throw new Error(0)}catch(e){err=e}if(!err.stack){return"(no stack trace available)"}}return err.stack.toString()}function stackTrace(){return demangleAll(jsStackTrace())}Module["stackTrace"]=stackTrace;var PAGE_SIZE=4096;function alignMemoryPage(x){if(x%4096>0){x+=4096-x%4096}return x}var HEAP;var HEAP8,HEAPU8,HEAP16,HEAPU16,HEAP32,HEAPU32,HEAPF32,HEAPF64;var STATIC_BASE=0,STATICTOP=0,staticSealed=false;var STACK_BASE=0,STACKTOP=0,STACK_MAX=0;var DYNAMIC_BASE=0,DYNAMICTOP=0;function enlargeMemory(){var OLD_TOTAL_MEMORY=TOTAL_MEMORY;var LIMIT=Math.pow(2,31);if(DYNAMICTOP>=LIMIT)return false;while(TOTAL_MEMORY<=DYNAMICTOP){if(TOTAL_MEMORY<LIMIT/2){TOTAL_MEMORY=alignMemoryPage(2*TOTAL_MEMORY)}else{var last=TOTAL_MEMORY;TOTAL_MEMORY=alignMemoryPage((3*TOTAL_MEMORY+LIMIT)/4);if(TOTAL_MEMORY<=last)return false}}TOTAL_MEMORY=Math.max(TOTAL_MEMORY,16*1024*1024);if(TOTAL_MEMORY>=LIMIT)return false;try{if(ArrayBuffer.transfer){buffer=ArrayBuffer.transfer(buffer,TOTAL_MEMORY)}else{var oldHEAP8=HEAP8;buffer=new ArrayBuffer(TOTAL_MEMORY)}}catch(e){return false}var success=_emscripten_replace_memory(buffer);if(!success)return false;Module["buffer"]=buffer;Module["HEAP8"]=HEAP8=new Int8Array(buffer);Module["HEAP16"]=HEAP16=new Int16Array(buffer);Module["HEAP32"]=HEAP32=new Int32Array(buffer);Module["HEAPU8"]=HEAPU8=new Uint8Array(buffer);Module["HEAPU16"]=HEAPU16=new Uint16Array(buffer);Module["HEAPU32"]=HEAPU32=new Uint32Array(buffer);Module["HEAPF32"]=HEAPF32=new Float32Array(buffer);Module["HEAPF64"]=HEAPF64=new Float64Array(buffer);if(!ArrayBuffer.transfer){HEAP8.set(oldHEAP8)}return true}var byteLength;try{byteLength=Function.prototype.call.bind(Object.getOwnPropertyDescriptor(ArrayBuffer.prototype,"byteLength").get);byteLength(new ArrayBuffer(4))}catch(e){byteLength=(function(buffer){return buffer.byteLength})}var TOTAL_STACK=Module["TOTAL_STACK"]||5242880;var TOTAL_MEMORY=Module["TOTAL_MEMORY"]||16777216;var totalMemory=64*1024;while(totalMemory<TOTAL_MEMORY||totalMemory<2*TOTAL_STACK){if(totalMemory<16*1024*1024){totalMemory*=2}else{totalMemory+=16*1024*1024}}totalMemory=Math.max(totalMemory,16*1024*1024);if(totalMemory!==TOTAL_MEMORY){TOTAL_MEMORY=totalMemory}assert(typeof Int32Array!=="undefined"&&typeof Float64Array!=="undefined"&&!!(new Int32Array(1))["subarray"]&&!!(new Int32Array(1))["set"],"JS engine does not provide full typed array support");var buffer;if(Module["buffer"]){buffer=Module["buffer"];assert(buffer.byteLength===TOTAL_MEMORY,"provided buffer should be "+TOTAL_MEMORY+" bytes, but it is "+buffer.byteLength)}else{buffer=new ArrayBuffer(TOTAL_MEMORY)}HEAP8=new Int8Array(buffer);HEAP16=new Int16Array(buffer);HEAP32=new Int32Array(buffer);HEAPU8=new Uint8Array(buffer);HEAPU16=new Uint16Array(buffer);HEAPU32=new Uint32Array(buffer);HEAPF32=new Float32Array(buffer);HEAPF64=new Float64Array(buffer);HEAP32[0]=255;assert(HEAPU8[0]===255&&HEAPU8[3]===0,"Typed arrays 2 must be run on a little-endian system");Module["HEAP"]=HEAP;Module["buffer"]=buffer;Module["HEAP8"]=HEAP8;Module["HEAP16"]=HEAP16;Module["HEAP32"]=HEAP32;Module["HEAPU8"]=HEAPU8;Module["HEAPU16"]=HEAPU16;Module["HEAPU32"]=HEAPU32;Module["HEAPF32"]=HEAPF32;Module["HEAPF64"]=HEAPF64;function callRuntimeCallbacks(callbacks){while(callbacks.length>0){var callback=callbacks.shift();if(typeof callback=="function"){callback();continue}var func=callback.func;if(typeof func==="number"){if(callback.arg===undefined){Runtime.dynCall("v",func)}else{Runtime.dynCall("vi",func,[callback.arg])}}else{func(callback.arg===undefined?null:callback.arg)}}}var __ATPRERUN__=[];var __ATINIT__=[];var __ATMAIN__=[];var __ATEXIT__=[];var __ATPOSTRUN__=[];var runtimeInitialized=false;var runtimeExited=false;function preRun(){if(Module["preRun"]){if(typeof Module["preRun"]=="function")Module["preRun"]=[Module["preRun"]];while(Module["preRun"].length){addOnPreRun(Module["preRun"].shift())}}callRuntimeCallbacks(__ATPRERUN__)}function ensureInitRuntime(){if(runtimeInitialized)return;runtimeInitialized=true;callRuntimeCallbacks(__ATINIT__)}function preMain(){callRuntimeCallbacks(__ATMAIN__)}function exitRuntime(){callRuntimeCallbacks(__ATEXIT__);runtimeExited=true}function postRun(){if(Module["postRun"]){if(typeof Module["postRun"]=="function")Module["postRun"]=[Module["postRun"]];while(Module["postRun"].length){addOnPostRun(Module["postRun"].shift())}}callRuntimeCallbacks(__ATPOSTRUN__)}function addOnPreRun(cb){__ATPRERUN__.unshift(cb)}Module["addOnPreRun"]=addOnPreRun;function addOnInit(cb){__ATINIT__.unshift(cb)}Module["addOnInit"]=addOnInit;function addOnPreMain(cb){__ATMAIN__.unshift(cb)}Module["addOnPreMain"]=addOnPreMain;function addOnExit(cb){__ATEXIT__.unshift(cb)}Module["addOnExit"]=addOnExit;function addOnPostRun(cb){__ATPOSTRUN__.unshift(cb)}Module["addOnPostRun"]=addOnPostRun;function intArrayFromString(stringy,dontAddNull,length){var len=length>0?length:lengthBytesUTF8(stringy)+1;var u8array=new Array(len);var numBytesWritten=stringToUTF8Array(stringy,u8array,0,u8array.length);if(dontAddNull)u8array.length=numBytesWritten;return u8array}Module["intArrayFromString"]=intArrayFromString;function intArrayToString(array){var ret=[];for(var i=0;i<array.length;i++){var chr=array[i];if(chr>255){chr&=255}ret.push(String.fromCharCode(chr))}return ret.join("")}Module["intArrayToString"]=intArrayToString;function writeStringToMemory(string,buffer,dontAddNull){var array=intArrayFromString(string,dontAddNull);var i=0;while(i<array.length){var chr=array[i];HEAP8[buffer+i>>0]=chr;i=i+1}}Module["writeStringToMemory"]=writeStringToMemory;function writeArrayToMemory(array,buffer){for(var i=0;i<array.length;i++){HEAP8[buffer++>>0]=array[i]}}Module["writeArrayToMemory"]=writeArrayToMemory;function writeAsciiToMemory(str,buffer,dontAddNull){for(var i=0;i<str.length;++i){HEAP8[buffer++>>0]=str.charCodeAt(i)}if(!dontAddNull)HEAP8[buffer>>0]=0}Module["writeAsciiToMemory"]=writeAsciiToMemory;function unSign(value,bits,ignore){if(value>=0){return value}return bits<=32?2*Math.abs(1<<bits-1)+value:Math.pow(2,bits)+value}function reSign(value,bits,ignore){if(value<=0){return value}var half=bits<=32?Math.abs(1<<bits-1):Math.pow(2,bits-1);if(value>=half&&(bits<=32||value>half)){value=-2*half+value}return value}if(!Math["imul"]||Math["imul"](4294967295,5)!==-5)Math["imul"]=function imul(a,b){var ah=a>>>16;var al=a&65535;var bh=b>>>16;var bl=b&65535;return al*bl+(ah*bl+al*bh<<16)|0};Math.imul=Math["imul"];if(!Math["clz32"])Math["clz32"]=(function(x){x=x>>>0;for(var i=0;i<32;i++){if(x&1<<31-i)return i}return 32});Math.clz32=Math["clz32"];var Math_abs=Math.abs;var Math_cos=Math.cos;var Math_sin=Math.sin;var Math_tan=Math.tan;var Math_acos=Math.acos;var Math_asin=Math.asin;var Math_atan=Math.atan;var Math_atan2=Math.atan2;var Math_exp=Math.exp;var Math_log=Math.log;var Math_sqrt=Math.sqrt;var Math_ceil=Math.ceil;var Math_floor=Math.floor;var Math_pow=Math.pow;var Math_imul=Math.imul;var Math_fround=Math.fround;var Math_min=Math.min;var Math_clz32=Math.clz32;var runDependencies=0;var runDependencyWatcher=null;var dependenciesFulfilled=null;function getUniqueRunDependency(id){return id}function addRunDependency(id){runDependencies++;if(Module["monitorRunDependencies"]){Module["monitorRunDependencies"](runDependencies)}}Module["addRunDependency"]=addRunDependency;function removeRunDependency(id){runDependencies--;if(Module["monitorRunDependencies"]){Module["monitorRunDependencies"](runDependencies)}if(runDependencies==0){if(runDependencyWatcher!==null){clearInterval(runDependencyWatcher);runDependencyWatcher=null}if(dependenciesFulfilled){var callback=dependenciesFulfilled;dependenciesFulfilled=null;callback()}}}Module["removeRunDependency"]=removeRunDependency;Module["preloadedImages"]={};Module["preloadedAudios"]={};var memoryInitializer=null;var ASM_CONSTS=[];STATIC_BASE=8;STATICTOP=STATIC_BASE+150256;__ATINIT__.push({func:(function(){__GLOBAL__I_000101()})},{func:(function(){__GLOBAL__sub_I_Mesh_cpp()})},{func:(function(){__GLOBAL__sub_I_Geometry_cpp()})},{func:(function(){__GLOBAL__sub_I_iostream_cpp()})});allocate([220,163,0,0,35,209,0,0,200,99,0,0,0,0,0,0,220,163,0,0,105,209,0,0,136,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,163,0,0,56,250,0,0,152,0,0,0,0,0,0,0,220,163,0,0,75,250,0,0,168,0,0,0,0,0,0,0,180,163,0,0,83,250,0,0,220,163,0,0,234,250,0,0,168,0,0,0,0,0,0,0,220,163,0,0,1,4,1,0,208,0,0,0,0,0,0,0,220,163,0,0,20,4,1,0,168,0,0,0,0,0,0,0,220,163,0,0,27,4,1,0,208,0,0,0,0,0,0,0,220,163,0,0,16,5,1,0,0,1,0,0,0,0,0,0,220,163,0,0,53,5,1,0,168,0,0,0,0,0,0,0,220,163,0,0,118,5,1,0,0,1,0,0,0,0,0,0,220,163,0,0,107,6,1,0,48,1,0,0,0,0,0,0,180,163,0,0,123,6,1,0,220,163,0,0,70,7,1,0,0,1,0,0,0,0,0,0,220,163,0,0,97,7,1,0,208,0,0,0,0,0,0,0,220,163,0,0,115,7,1,0,104,1,0,0,0,0,0,0,220,163,0,0,74,8,1,0,168,0,0,0,0,0,0,0,220,163,0,0,172,8,1,0,136,1,0,0,0,0,0,0,220,163,0,0,19,10,1,0,168,0,0,0,0,0,0,0,220,163,0,0,3,22,1,0,168,1,0,0,0,0,0,0,180,163,0,0,21,22,1,0,220,163,0,0,237,22,1,0,168,0,0,0,0,0,0,0,220,163,0,0,148,24,1,0,168,0,0,0,0,0,0,0,4,164,0,0,60,25,1,0,0,0,0,0,2,0,0,0,168,0,0,0,2,0,0,0,240,1,0,0,2,4,0,0,180,163,0,0,202,30,1,0,4,164,0,0,187,31,1,0,0,0,0,0,3,0,0,0,208,1,0,0,2,0,0,0,32,2,0,0,2,48,0,0,40,2,0,0,2,0,0,0,180,163,0,0,53,32,1,0,180,163,0,0,45,32,1,0,4,164,0,0,117,34,1,0,0,0,0,0,3,0,0,0,208,1,0,0,2,0,0,0,32,2,0,0,2,48,0,0,88,2,0,0,2,0,0,0,180,163,0,0,184,38,1,0,4,164,0,0,188,39,1,0,0,0,0,0,3,0,0,0,208,1,0,0,2,0,0,0,32,2,0,0,2,48,0,0,136,2,0,0,2,0,0,0,180,163,0,0,104,40,1,0,4,164,0,0,163,43,1,0,0,0,0,0,3,0,0,0,208,1,0,0,2,0,0,0,32,2,0,0,2,48,0,0,184,2,0,0,2,0,0,0,180,163,0,0,58,47,1,0,4,164,0,0,124,50,1,0,0,0,0,0,2,0,0,0,104,1,0,0,2,0,0,0,40,2,0,0,2,0,0,0,4,164,0,0,142,51,1,0,0,0,0,0,2,0,0,0,104,1,0,0,2,0,0,0,88,2,0,0,2,0,0,0,220,163,0,0,104,52,1,0,104,1,0,0,0,0,0,0,220,163,0,0,182,53,1,0,104,1,0,0,0,0,0,0,220,163,0,0,9,55,1,0,104,1,0,0,0,0,0,0,220,163,0,0,14,56,1,0,104,1,0,0,0,0,0,0,220,163,0,0,253,56,1,0,104,1,0,0,0,0,0,0,220,163,0,0,193,58,1,0,96,3,0,0,0,0,0,0,4,164,0,0,62,59,1,0,0,0,0,0,2,0,0,0,168,0,0,0,2,0,0,0,240,1,0,0,2,4,0,0,220,163,0,0,148,59,1,0,96,3,0,0,0,0,0,0,220,163,0,0,152,62,1,0,136,1,0,0,0,0,0,0,220,163,0,0,96,63,1,0,136,1,0,0,0,0,0,0,220,163,0,0,105,64,1,0,192,3,0,0,0,0,0,0,4,164,0,0,197,64,1,0,0,0,0,0,2,0,0,0,168,0,0,0,2,0,0,0,240,1,0,0,2,4,0,0,220,163,0,0,29,65,1,0,192,3,0,0,0,0,0,0,220,163,0,0,7,66,1,0,192,3,0,0,0,0,0,0,220,163,0,0,240,70,1,0,208,0,0,0,0,0,0,0,220,163,0,0,219,71,1,0,208,0,0,0,0,0,0,0,220,163,0,0,219,72,1,0,208,0,0,0,0,0,0,0,220,163,0,0,251,73,1,0,208,0,0,0,0,0,0,0,220,163,0,0,194,74,1,0,208,0,0,0,0,0,0,0,220,163,0,0,27,76,1,0,208,0,0,0,0,0,0,0,220,163,0,0,234,76,1,0,208,0,0,0,0,0,0,0,220,163,0,0,224,77,1,0,208,0,0,0,0,0,0,0,220,163,0,0,210,78,1,0,208,0,0,0,0,0,0,0,220,163,0,0,18,82,1,0,208,0,0,0,0,0,0,0,220,163,0,0,71,83,1,0,208,0,0,0,0,0,0,0,220,163,0,0,54,84,1,0,208,0,0,0,0,0,0,0,220,163,0,0,23,87,1,0,208,0,0,0,0,0,0,0,220,163,0,0,32,90,1,0,168,0,0,0,0,0,0,0,220,163,0,0,92,160,1,0,120,98,0,0,0,0,0,0,220,163,0,0,3,170,1,0,0,1,0,0,0,0,0,0,220,163,0,0,83,188,1,0,104,1,0,0,0,0,0,0,220,163,0,0,137,200,1,0,192,3,0,0,0,0,0,0,4,164,0,0,153,205,1,0,0,0,0,0,2,0,0,0,168,0,0,0,2,0,0,0,64,5,0,0,2,4,0,0,180,163,0,0,168,205,1,0,4,164,0,0,56,206,1,0,0,0,0,0,2,0,0,0,168,0,0,0,2,0,0,0,64,5,0,0,2,4,0,0,4,164,0,0,207,206,1,0,0,0,0,0,2,0,0,0,168,0,0,0,2,0,0,0,64,5,0,0,2,4,0,0,4,164,0,0,148,207,1,0,0,0,0,0,2,0,0,0,168,0,0,0,2,0,0,0,64,5,0,0,2,4,0,0,4,164,0,0,69,208,1,0,0,0,0,0,2,0,0,0,168,0,0,0,2,0,0,0,64,5,0,0,2,4,0,0,4,164,0,0,71,209,1,0,0,0,0,0,2,0,0,0,104,1,0,0,2,0,0,0,184,2,0,0,2,0,0,0,4,164,0,0,168,210,1,0,0,0,0,0,2,0,0,0,104,1,0,0,2,0,0,0,136,2,0,0,2,0,0,0,220,163,0,0,140,225,1,0,24,6,0,0,0,0,0,0,180,163,0,0,10,226,1,0,220,163,0,0,20,226,1,0,24,6,0,0,0,0,0,0,220,163,0,0,148,226,1,0,24,6,0,0,0,0,0,0,220,163,0,0,62,228,1,0,24,6,0,0,0,0,0,0,220,163,0,0,193,228,1,0,24,6,0,0,0,0,0,0,220,163,0,0,82,229,1,0,24,6,0,0,0,0,0,0,220,163,0,0,202,229,1,0,24,6,0,0,0,0,0,0,220,163,0,0,56,230,1,0,24,6,0,0,0,0,0,0,220,163,0,0,168,230,1,0,24,6,0,0,0,0,0,0,220,163,0,0,155,232,1,0,24,6,0,0,0,0,0,0,220,163,0,0,135,236,1,0,24,6,0,0,0,0,0,0,220,163,0,0,45,237,1,0,24,6,0,0,0,0,0,0,220,163,0,0,186,237,1,0,24,6,0,0,0,0,0,0,220,163,0,0,22,239,1,0,24,6,0,0,0,0,0,0,220,163,0,0,241,239,1,0,24,6,0,0,0,0,0,0,220,163,0,0,95,240,1,0,24,6,0,0,0,0,0,0,220,163,0,0,207,240,1,0,24,6,0,0,0,0,0,0,220,163,0,0,75,241,1,0,24,6,0,0,0,0,0,0,220,163,0,0,236,242,1,0,24,6,0,0,0,0,0,0,220,163,0,0,90,243,1,0,24,6,0,0,0,0,0,0,220,163,0,0,184,243,1,0,24,6,0,0,0,0,0,0,220,163,0,0,129,244,1,0,24,6,0,0,0,0,0,0,220,163,0,0,26,246,1,0,24,6,0,0,0,0,0,0,220,163,0,0,136,246,1,0,24,6,0,0,0,0,0,0,220,163,0,0,244,246,1,0,24,6,0,0,0,0,0,0,220,163,0,0,98,247,1,0,24,6,0,0,0,0,0,0,220,163,0,0,206,247,1,0,24,6,0,0,0,0,0,0,220,163,0,0,54,248,1,0,24,6,0,0,0,0,0,0,220,163,0,0,247,248,1,0,24,6,0,0,0,0,0,0,220,163,0,0,83,249,1,0,24,6,0,0,0,0,0,0,220,163,0,0,252,249,1,0,24,6,0,0,0,0,0,0,220,163,0,0,55,251,1,0,24,6,0,0,0,0,0,0,220,163,0,0,207,251,1,0,24,6,0,0,0,0,0,0,220,163,0,0,103,253,1,0,24,6,0,0,0,0,0,0,220,163,0,0,114,255,1,0,208,0,0,0,0,0,0,0,220,163,0,0,66,0,2,0,24,6,0,0,0,0,0,0,220,163,0,0,239,0,2,0,96,8,0,0,0,0,0,0,180,163,0,0,140,1,2,0,220,163,0,0,147,1,2,0,96,8,0,0,0,0,0,0,220,163,0,0,35,3,2,0,96,8,0,0,0,0,0,0,220,163,0,0,54,4,2,0,96,8,0,0,0,0,0,0,220,163,0,0,174,11,2,0,192,3,0,0,0,0,0,0,220,163,0,0,121,12,2,0,192,3,0,0,0,0,0,0,0,0,0,0,0,0,240,63,0,0,0,0,0,0,208,63,0,0,0,0,0,0,208,63,0,0,0,0,0,0,208,63,0,0,0,0,0,0,208,63,154,153,153,153,153,153,233,191,205,204,204,204,204,204,220,63,205,204,204,204,204,204,220,63,205,204,204,204,204,204,220,63,205,204,204,204,204,204,220,63,238,126,249,198,76,241,138,191,29,54,173,178,122,56,127,63,29,54,173,178,122,56,127,63,29,54,173,178,122,56,127,63,29,54,173,178,122,56,127,63,139,8,63,15,121,124,153,63,139,8,63,15,121,124,153,63,139,8,63,15,121,124,153,63,139,8,63,15,121,124,153,63,139,8,63,15,121,124,153,63,139,8,63,15,121,124,153,63,133,76,30,60,175,2,159,63,66,140,175,248,138,175,120,63,66,140,175,248,138,175,120,63,66,140,175,248,138,175,120,63,66,140,175,248,138,175,120,63,104,126,211,223,118,217,135,63,104,126,211,223,118,217,135,63,104,126,211,223,118,217,135,63,104,126,211,223,118,217,135,63,102,70,103,231,128,108,134,63,102,70,103,231,128,108,134,63,102,70,103,231,128,108,134,63,102,70,103,231,128,108,134,63,102,70,103,231,128,108,134,63,102,70,103,231,128,108,134,63,76,191,153,156,1,65,123,63,76,191,153,156,1,65,123,63,76,191,153,156,1,65,123,63,76,191,153,156,1,65,123,63,86,97,201,41,123,132,91,63,86,97,201,41,123,132,91,63,86,97,201,41,123,132,91,63,86,97,201,41,123,132,91,63,1,52,112,48,47,229,130,63,1,52,112,48,47,229,130,63,1,52,112,48,47,229,130,63,1,52,112,48,47,229,130,63,172,6,117,80,7,117,128,63,172,6,117,80,7,117,128,63,172,6,117,80,7,117,128,63,172,6,117,80,7,117,128,63,172,6,117,80,7,117,128,63,172,6,117,80,7,117,128,63,172,6,117,80,7,117,128,63,172,6,117,80,7,117,128,63,172,6,117,80,7,117,128,63,172,6,117,80,7,117,128,63,172,6,117,80,7,117,128,63,172,6,117,80,7,117,128,63,0,0,0,0,0,0,208,63,37,9,32,29,109,176,193,63,37,9,32,29,109,176,193,63,37,9,32,29,109,176,193,63,54,249,39,42,174,187,226,63,0,0,0,0,0,0,208,63,97,85,85,85,85,85,197,63,97,85,85,85,85,85,197,63,97,85,85,85,85,85,197,63,0,0,0,0,0,0,224,63,0,0,0,0,0,0,208,63,115,36,73,146,36,73,178,63,115,36,73,146,36,73,178,63,115,36,73,146,36,73,178,63,76,146,36,73,146,36,233,63,136,73,117,238,175,192,185,63,136,73,117,238,175,192,185,63,158,173,98,4,212,143,217,63,136,73,117,238,175,192,185,63,158,173,98,4,212,143,217,63,158,173,98,4,212,143,217,63,0,0,0,0,0,0,208,63,79,85,85,85,85,85,213,63,79,85,85,85,85,85,213,63,79,85,85,85,85,85,213,63,0,0,0,0,0,0,0,0,76,23,93,116,209,69,183,63,76,23,93,116,209,69,183,63,76,23,93,116,209,69,183,63,67,23,93,116,209,69,231,63,150,222,182,108,164,189,219,63,150,222,182,108,164,189,219,63,168,133,36,77,110,9,177,63,150,222,182,108,164,189,219,63,168,133,36,77,110,9,177,63,168,133,36,77,110,9,177,63,220,198,215,92,27,120,203,63,220,198,215,92,27,120,203,63,220,198,215,92,27,120,203,63,200,85,188,244,214,203,214,63,163,247,104,147,55,211,164,63,163,247,104,147,55,211,164,63,163,247,104,147,55,211,164,63,136,81,92,148,101,24,236,63,200,13,27,26,47,161,212,63,200,13,27,26,47,161,212,63,200,13,27,26,47,161,212,63,201,181,118,141,149,227,160,63,225,166,50,137,210,75,227,63,115,248,15,189,79,66,209,63,225,166,50,137,210,75,227,63,115,248,15,189,79,66,209,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,225,166,50,137,210,75,227,63,115,248,15,189,79,66,209,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,0,0,0,0,0,0,208,63,37,9,32,29,109,176,193,63,37,9,32,29,109,176,193,63,54,249,39,42,174,187,226,63,37,9,32,29,109,176,193,63,0,0,0,0,0,0,208,63,97,85,85,85,85,85,197,63,97,85,85,85,85,85,197,63,0,0,0,0,0,0,224,63,97,85,85,85,85,85,197,63,0,0,0,0,0,0,208,63,115,36,73,146,36,73,178,63,115,36,73,146,36,73,178,63,76,146,36,73,146,36,233,63,115,36,73,146,36,73,178,63,136,73,117,238,175,192,185,63,158,173,98,4,212,143,217,63,136,73,117,238,175,192,185,63,158,173,98,4,212,143,217,63,136,73,117,238,175,192,185,63,158,173,98,4,212,143,217,63,0,0,0,0,0,0,208,63,79,85,85,85,85,85,213,63,79,85,85,85,85,85,213,63,0,0,0,0,0,0,0,0,79,85,85,85,85,85,213,63,76,23,93,116,209,69,183,63,76,23,93,116,209,69,183,63,67,23,93,116,209,69,231,63,76,23,93,116,209,69,183,63,150,222,182,108,164,189,219,63,168,133,36,77,110,9,177,63,150,222,182,108,164,189,219,63,168,133,36,77,110,9,177,63,150,222,182,108,164,189,219,63,168,133,36,77,110,9,177,63,220,198,215,92,27,120,203,63,220,198,215,92,27,120,203,63,200,85,188,244,214,203,214,63,220,198,215,92,27,120,203,63,163,247,104,147,55,211,164,63,163,247,104,147,55,211,164,63,136,81,92,148,101,24,236,63,163,247,104,147,55,211,164,63,200,13,27,26,47,161,212,63,200,13,27,26,47,161,212,63,201,181,118,141,149,227,160,63,200,13,27,26,47,161,212,63,115,248,15,189,79,66,209,63,225,166,50,137,210,75,227,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,225,166,50,137,210,75,227,63,115,248,15,189,79,66,209,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,225,166,50,137,210,75,227,63,115,248,15,189,79,66,209,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,0,0,0,0,0,0,208,63,37,9,32,29,109,176,193,63,54,249,39,42,174,187,226,63,37,9,32,29,109,176,193,63,37,9,32,29,109,176,193,63,0,0,0,0,0,0,208,63,97,85,85,85,85,85,197,63,0,0,0,0,0,0,224,63,97,85,85,85,85,85,197,63,97,85,85,85,85,85,197,63,0,0,0,0,0,0,208,63,115,36,73,146,36,73,178,63,76,146,36,73,146,36,233,63,115,36,73,146,36,73,178,63,115,36,73,146,36,73,178,63,158,173,98,4,212,143,217,63,136,73,117,238,175,192,185,63,136,73,117,238,175,192,185,63,158,173,98,4,212,143,217,63,158,173,98,4,212,143,217,63,136,73,117,238,175,192,185,63,0,0,0,0,0,0,208,63,79,85,85,85,85,85,213,63,0,0,0,0,0,0,0,0,79,85,85,85,85,85,213,63,79,85,85,85,85,85,213,63,76,23,93,116,209,69,183,63,67,23,93,116,209,69,231,63,76,23,93,116,209,69,183,63,76,23,93,116,209,69,183,63,168,133,36,77,110,9,177,63,150,222,182,108,164,189,219,63,150,222,182,108,164,189,219,63,168,133,36,77,110,9,177,63,168,133,36,77,110,9,177,63,150,222,182,108,164,189,219,63,220,198,215,92,27,120,203,63,200,85,188,244,214,203,214,63,220,198,215,92,27,120,203,63,220,198,215,92,27,120,203,63,163,247,104,147,55,211,164,63,136,81,92,148,101,24,236,63,163,247,104,147,55,211,164,63,163,247,104,147,55,211,164,63,200,13,27,26,47,161,212,63,201,181,118,141,149,227,160,63,200,13,27,26,47,161,212,63,200,13,27,26,47,161,212,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,115,248,15,189,79,66,209,63,225,166,50,137,210,75,227,63,115,248,15,189,79,66,209,63,225,166,50,137,210,75,227,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,225,166,50,137,210,75,227,63,115,248,15,189,79,66,209,63,0,0,0,0,0,0,208,63,54,249,39,42,174,187,226,63,37,9,32,29,109,176,193,63,37,9,32,29,109,176,193,63,37,9,32,29,109,176,193,63,0,0,0,0,0,0,208,63,0,0,0,0,0,0,224,63,97,85,85,85,85,85,197,63,97,85,85,85,85,85,197,63,97,85,85,85,85,85,197,63,0,0,0,0,0,0,208,63,76,146,36,73,146,36,233,63,115,36,73,146,36,73,178,63,115,36,73,146,36,73,178,63,115,36,73,146,36,73,178,63,158,173,98,4,212,143,217,63,158,173,98,4,212,143,217,63,158,173,98,4,212,143,217,63,136,73,117,238,175,192,185,63,136,73,117,238,175,192,185,63,136,73,117,238,175,192,185,63,0,0,0,0,0,0,208,63,0,0,0,0,0,0,0,0,79,85,85,85,85,85,213,63,79,85,85,85,85,85,213,63,79,85,85,85,85,85,213,63,67,23,93,116,209,69,231,63,76,23,93,116,209,69,183,63,76,23,93,116,209,69,183,63,76,23,93,116,209,69,183,63,168,133,36,77,110,9,177,63,168,133,36,77,110,9,177,63,168,133,36,77,110,9,177,63,150,222,182,108,164,189,219,63,150,222,182,108,164,189,219,63,150,222,182,108,164,189,219,63,200,85,188,244,214,203,214,63,220,198,215,92,27,120,203,63,220,198,215,92,27,120,203,63,220,198,215,92,27,120,203,63,136,81,92,148,101,24,236,63,163,247,104,147,55,211,164,63,163,247,104,147,55,211,164,63,163,247,104,147,55,211,164,63,201,181,118,141,149,227,160,63,200,13,27,26,47,161,212,63,200,13,27,26,47,161,212,63,200,13,27,26,47,161,212,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,184,115,21,97,22,76,176,63,115,248,15,189,79,66,209,63,225,166,50,137,210,75,227,63,115,248,15,189,79,66,209,63,225,166,50,137,210,75,227,63,115,248,15,189,79,66,209,63,225,166,50,137,210,75,227,63,169,76,88,232,122,182,251,63,21,51,144,69,167,121,226,63,21,51,144,69,167,121,226,63,21,51,144,69,167,121,226,63,59,86,99,69,74,45,239,191,218,79,177,92,21,222,236,63,218,79,177,92,21,222,236,63,218,79,177,92,21,222,236,63,30,52,209,223,26,195,216,63,30,52,209,223,26,195,216,63,30,52,209,223,26,195,216,63,61,100,158,86,103,96,200,63,61,100,158,86,103,96,200,63,61,100,158,86,103,96,200,63,95,171,130,55,8,241,216,63,26,181,42,55,36,90,205,63,26,181,42,55,36,90,205,63,26,181,42,55,36,90,205,63,154,79,20,186,200,235,203,63,154,79,20,186,200,235,203,63,154,79,20,186,200,235,203,63,93,192,194,163,77,228,201,63,93,192,194,163,77,228,201,63,93,192,194,163,77,228,201,63,168,232,251,166,125,139,182,63,168,232,251,166,125,139,182,63,168,232,251,166,125,139,182,63,247,11,128,79,72,94,194,63,247,11,128,79,72,94,194,63,247,11,128,79,72,94,194,63,247,11,128,79,72,94,194,63,247,11,128,79,72,94,194,63,247,11,128,79,72,94,194,63,180,166,73,160,124,148,208,191,247,239,121,134,152,119,211,63,247,239,121,134,152,119,211,63,247,239,121,134,152,119,211,63,255,78,92,208,136,167,183,63,255,78,92,208,136,167,183,63,255,78,92,208,136,167,183,63,164,68,253,5,168,24,193,63,164,68,253,5,168,24,193,63,164,68,253,5,168,24,193,63,164,68,253,5,168,24,193,63,164,68,253,5,168,24,193,63,164,68,253,5,168,24,193,63,130,94,20,240,192,254,207,63,199,152,13,115,1,21,197,63,199,152,13,115,1,21,197,63,199,152,13,115,1,21,197,63,155,150,199,75,48,226,198,63,155,150,199,75,48,226,198,63,155,150,199,75,48,226,198,63,168,78,124,89,214,200,172,63,168,78,124,89,214,200,172,63,168,78,124,89,214,200,172,63,37,126,11,226,234,37,168,63,37,126,11,226,234,37,168,63,37,126,11,226,234,37,168,63,37,126,11,226,234,37,168,63,37,126,11,226,234,37,168,63,37,126,11,226,234,37,168,63,107,118,124,23,6,137,197,63,203,232,135,131,181,201,171,63,203,232,135,131,181,201,171,63,203,232,135,131,181,201,171,63,247,55,96,227,42,65,193,63,247,55,96,227,42,65,193,63,247,55,96,227,42,65,193,63,204,132,253,113,121,168,193,63,204,132,253,113,121,168,193,63,204,132,253,113,121,168,193,63,81,169,122,222,186,174,166,63,81,169,122,222,186,174,166,63,81,169,122,222,186,174,166,63,220,152,227,245,47,49,179,63,220,152,227,245,47,49,179,63,220,152,227,245,47,49,179,63,220,152,227,245,47,49,179,63,220,152,227,245,47,49,179,63,220,152,227,245,47,49,179,63,243,127,148,143,115,34,196,63,229,150,34,66,211,72,176,63,229,150,34,66,211,72,176,63,229,150,34,66,211,72,176,63,106,116,143,60,120,24,180,63,106,116,143,60,120,24,180,63,106,116,143,60,120,24,180,63,32,105,216,206,111,33,192,63,32,105,216,206,111,33,192,63,32,105,216,206,111,33,192,63,32,105,216,206,111,33,192,63,32,105,216,206,111,33,192,63,32,105,216,206,111,33,192,63,158,243,248,36,242,30,169,63,158,243,248,36,242,30,169,63,158,243,248,36,242,30,169,63,158,243,248,36,242,30,169,63,158,243,248,36,242,30,169,63,158,243,248,36,242,30,169,63,26,235,35,134,224,181,144,63,26,235,35,134,224,181,144,63,26,235,35,134,224,181,144,63,26,235,35,134,224,181,144,63,26,235,35,134,224,181,144,63,26,235,35,134,224,181,144,63,113,69,99,143,119,78,90,63,113,69,99,143,119,78,90,63,113,69,99,143,119,78,90,63,237,94,74,205,175,26,193,63,237,94,74,205,175,26,193,63,237,94,74,205,175,26,193,63,200,14,97,214,217,77,186,63,200,14,97,214,217,77,186,63,200,14,97,214,217,77,186,63,246,46,50,56,94,11,176,63,246,46,50,56,94,11,176,63,246,46,50,56,94,11,176,63,99,93,149,249,39,58,152,63,99,93,149,249,39,58,152,63,99,93,149,249,39,58,152,63,163,146,243,163,223,52,183,63,163,146,243,163,223,52,183,63,163,146,243,163,223,52,183,63,163,146,243,163,223,52,183,63,163,146,243,163,223,52,183,63,163,146,243,163,223,52,183,63,238,147,207,94,31,93,162,63,238,147,207,94,31,93,162,63,238,147,207,94,31,93,162,63,238,147,207,94,31,93,162,63,238,147,207,94,31,93,162,63,238,147,207,94,31,93,162,63,25,230,176,161,141,209,166,63,25,230,176,161,141,209,166,63,25,230,176,161,141,209,166,63,195,238,210,60,157,95,179,63,195,238,210,60,157,95,179,63,195,238,210,60,157,95,179,63,143,23,176,138,36,223,187,63,143,23,176,138,36,223,187,63,143,23,176,138,36,223,187,63,34,12,233,212,135,219,174,63,34,12,233,212,135,219,174,63,34,12,233,212,135,219,174,63,69,190,84,146,138,223,133,63,69,190,84,146,138,223,133,63,69,190,84,146,138,223,133,63,237,192,231,182,164,230,177,63,237,192,231,182,164,230,177,63,237,192,231,182,164,230,177,63,237,192,231,182,164,230,177,63,237,192,231,182,164,230,177,63,237,192,231,182,164,230,177,63,227,133,106,133,130,211,163,63,227,133,106,133,130,211,163,63,227,133,106,133,130,211,163,63,227,133,106,133,130,211,163,63,227,133,106,133,130,211,163,63,227,133,106,133,130,211,163,63,75,242,89,200,96,182,158,63,75,242,89,200,96,182,158,63,75,242,89,200,96,182,158,63,75,242,89,200,96,182,158,63,75,242,89,200,96,182,158,63,75,242,89,200,96,182,158,63,36,223,233,3,189,73,183,63,226,172,231,135,182,1,148,63,226,172,231,135,182,1,148,63,226,172,231,135,182,1,148,63,136,143,202,113,223,221,171,63,136,143,202,113,223,221,171,63,136,143,202,113,223,221,171,63,40,203,114,103,71,223,180,63,40,203,114,103,71,223,180,63,40,203,114,103,71,223,180,63,116,77,33,3,82,0,181,63,116,77,33,3,82,0,181,63,116,77,33,3,82,0,181,63,19,9,188,220,193,163,171,63,19,9,188,220,193,163,171,63,19,9,188,220,193,163,171,63,254,100,114,240,190,74,140,63,254,100,114,240,190,74,140,63,254,100,114,240,190,74,140,63,206,144,184,98,185,86,176,63,206,144,184,98,185,86,176,63,206,144,184,98,185,86,176,63,206,144,184,98,185,86,176,63,206,144,184,98,185,86,176,63,206,144,184,98,185,86,176,63,228,17,46,212,19,221,158,63,228,17,46,212,19,221,158,63,228,17,46,212,19,221,158,63,228,17,46,212,19,221,158,63,228,17,46,212,19,221,158,63,228,17,46,212,19,221,158,63,224,183,151,211,157,135,155,63,224,183,151,211,157,135,155,63,224,183,151,211,157,135,155,63,224,183,151,211,157,135,155,63,224,183,151,211,157,135,155,63,224,183,151,211,157,135,155,63,154,247,50,145,21,104,163,63,154,247,50,145,21,104,163,63,154,247,50,145,21,104,163,63,210,233,226,232,184,19,173,63,210,233,226,232,184,19,173,63,210,233,226,232,184,19,173,63,194,198,78,54,247,244,182,63,194,198,78,54,247,244,182,63,194,198,78,54,247,244,182,63,108,140,55,68,242,177,178,63,108,140,55,68,242,177,178,63,108,140,55,68,242,177,178,63,160,86,179,247,146,153,153,63,160,86,179,247,146,153,153,63,160,86,179,247,146,153,153,63,6,172,226,27,233,118,129,63,6,172,226,27,233,118,129,63,6,172,226,27,233,118,129,63,79,59,18,199,179,223,165,63,79,59,18,199,179,223,165,63,79,59,18,199,179,223,165,63,79,59,18,199,179,223,165,63,79,59,18,199,179,223,165,63,79,59,18,199,179,223,165,63,22,202,150,42,81,26,177,63,22,202,150,42,81,26,177,63,22,202,150,42,81,26,177,63,22,202,150,42,81,26,177,63,22,202,150,42,81,26,177,63,22,202,150,42,81,26,177,63,166,222,156,41,194,154,153,63,166,222,156,41,194,154,153,63,166,222,156,41,194,154,153,63,166,222,156,41,194,154,153,63,166,222,156,41,194,154,153,63,166,222,156,41,194,154,153,63,64,180,227,137,193,197,129,63,64,180,227,137,193,197,129,63,64,180,227,137,193,197,129,63,64,180,227,137,193,197,129,63,64,180,227,137,193,197,129,63,64,180,227,137,193,197,129,63,172,249,140,19,207,50,107,63,172,249,140,19,207,50,107,63,172,249,140,19,207,50,107,63,49,66,102,16,200,158,179,63,49,66,102,16,200,158,179,63,49,66,102,16,200,158,179,63,34,14,30,111,69,178,182,63,34,14,30,111,69,178,182,63,34,14,30,111,69,178,182,63,25,161,96,98,171,1,165,63,25,161,96,98,171,1,165,63,25,161,96,98,171,1,165,63,40,72,253,220,45,146,151,63,40,72,253,220,45,146,151,63,40,72,253,220,45,146,151,63,133,158,25,200,117,216,128,63,133,158,25,200,117,216,128,63,133,158,25,200,117,216,128,63,251,33,188,49,226,23,177,63,251,33,188,49,226,23,177,63,251,33,188,49,226,23,177,63,251,33,188,49,226,23,177,63,251,33,188,49,226,23,177,63,251,33,188,49,226,23,177,63,238,55,70,46,160,34,168,63,238,55,70,46,160,34,168,63,238,55,70,46,160,34,168,63,238,55,70,46,160,34,168,63,238,55,70,46,160,34,168,63,238,55,70,46,160,34,168,63,57,62,136,229,30,246,110,63,57,62,136,229,30,246,110,63,57,62,136,229,30,246,110,63,57,62,136,229,30,246,110,63,57,62,136,229,30,246,110,63,57,62,136,229,30,246,110,63,254,193,111,197,53,18,163,63,254,193,111,197,53,18,163,63,254,193,111,197,53,18,163,63,254,193,111,197,53,18,163,63,254,193,111,197,53,18,163,63,254,193,111,197,53,18,163,63,220,36,157,53,168,56,139,63,220,36,157,53,168,56,139,63,220,36,157,53,168,56,139,63,220,36,157,53,168,56,139,63,220,36,157,53,168,56,139,63,220,36,157,53,168,56,139,63,19,103,126,114,240,200,180,63,129,76,79,235,34,185,134,63,129,76,79,235,34,185,134,63,129,76,79,235,34,185,134,63,193,28,27,30,155,126,178,63,193,28,27,30,155,126,178,63,193,28,27,30,155,126,178,63,2,10,153,197,254,216,167,63,2,10,153,197,254,216,167,63,2,10,153,197,254,216,167,63,104,92,95,148,136,174,178,63,104,92,95,148,136,174,178,63,104,92,95,148,136,174,178,63,19,87,141,240,194,154,170,63,19,87,141,240,194,154,170,63,19,87,141,240,194,154,170,63,100,113,7,41,130,47,153,63,100,113,7,41,130,47,153,63,100,113,7,41,130,47,153,63,224,225,7,213,107,106,121,63,224,225,7,213,107,106,121,63,224,225,7,213,107,106,121,63,197,224,175,41,69,16,173,63,197,224,175,41,69,16,173,63,197,224,175,41,69,16,173,63,197,224,175,41,69,16,173,63,197,224,175,41,69,16,173,63,197,224,175,41,69,16,173,63,133,252,86,88,37,34,155,63,133,252,86,88,37,34,155,63,133,252,86,88,37,34,155,63,133,252,86,88,37,34,155,63,133,252,86,88,37,34,155,63,133,252,86,88,37,34,155,63,119,62,224,142,221,237,112,63,119,62,224,142,221,237,112,63,119,62,224,142,221,237,112,63,119,62,224,142,221,237,112,63,119,62,224,142,221,237,112,63,119,62,224,142,221,237,112,63,87,49,164,159,177,236,160,63,87,49,164,159,177,236,160,63,87,49,164,159,177,236,160,63,87,49,164,159,177,236,160,63,87,49,164,159,177,236,160,63,87,49,164,159,177,236,160,63,239,107,114,92,125,76,136,63,239,107,114,92,125,76,136,63,239,107,114,92,125,76,136,63,239,107,114,92,125,76,136,63,239,107,114,92,125,76,136,63,239,107,114,92,125,76,136,63,119,111,183,131,6,167,173,63,67,99,139,22,76,17,130,63,67,99,139,22,76,17,130,63,67,99,139,22,76,17,130,63,165,217,145,13,66,5,154,63,165,217,145,13,66,5,154,63,165,217,145,13,66,5,154,63,10,105,244,226,55,152,165,63,10,105,244,226,55,152,165,63,10,105,244,226,55,152,165,63,55,62,60,13,36,150,171,63,55,62,60,13,36,150,171,63,55,62,60,13,36,150,171,63,58,11,128,50,24,184,171,63,58,11,128,50,24,184,171,63,58,11,128,50,24,184,171,63,177,0,126,186,187,1,166,63,177,0,126,186,187,1,166,63,177,0,126,186,187,1,166,63,7,203,47,45,29,238,152,63,7,203,47,45,29,238,152,63,7,203,47,45,29,238,152,63,101,227,209,8,33,170,118,63,101,227,209,8,33,170,118,63,101,227,209,8,33,170,118,63,167,148,71,229,103,205,140,63,167,148,71,229,103,205,140,63,167,148,71,229,103,205,140,63,167,148,71,229,103,205,140,63,167,148,71,229,103,205,140,63,167,148,71,229,103,205,140,63,55,47,50,159,135,197,167,63,55,47,50,159,135,197,167,63,55,47,50,159,135,197,167,63,55,47,50,159,135,197,167,63,55,47,50,159,135,197,167,63,55,47,50,159,135,197,167,63,217,98,21,137,217,94,160,63,217,98,21,137,217,94,160,63,217,98,21,137,217,94,160,63,217,98,21,137,217,94,160,63,217,98,21,137,217,94,160,63,217,98,21,137,217,94,160,63,231,247,106,240,201,17,142,63,231,247,106,240,201,17,142,63,231,247,106,240,201,17,142,63,231,247,106,240,201,17,142,63,231,247,106,240,201,17,142,63,231,247,106,240,201,17,142,63,127,206,120,106,228,56,160,63,127,206,120,106,228,56,160,63,127,206,120,106,228,56,160,63,127,206,120,106,228,56,160,63,127,206,120,106,228,56,160,63,127,206,120,106,228,56,160,63,150,213,62,76,4,165,135,63,150,213,62,76,4,165,135,63,150,213,62,76,4,165,135,63,150,213,62,76,4,165,135,63,150,213,62,76,4,165,135,63,150,213,62,76,4,165,135,63,19,32,87,129,147,82,171,63,42,220,232,103,79,23,144,63,42,220,232,103,79,23,144,63,42,220,232,103,79,23,144,63,252,180,187,220,65,163,160,63,252,180,187,220,65,163,160,63,252,180,187,220,65,163,160,63,12,185,58,93,149,61,161,63,12,185,58,93,149,61,161,63,12,185,58,93,149,61,161,63,9,103,144,117,203,156,168,63,9,103,144,117,203,156,168,63,9,103,144,117,203,156,168,63,244,28,137,177,234,154,172,63,244,28,137,177,234,154,172,63,244,28,137,177,234,154,172,63,28,113,253,38,100,60,166,63,28,113,253,38,100,60,166,63,28,113,253,38,100,60,166,63,24,169,159,61,43,22,155,63,24,169,159,61,43,22,155,63,24,169,159,61,43,22,155,63,61,82,23,32,132,25,136,63,61,82,23,32,132,25,136,63,61,82,23,32,132,25,136,63,168,143,80,40,32,139,111,191,168,143,80,40,32,139,111,191,168,143,80,40,32,139,111,191,134,224,192,63,248,117,134,63,134,224,192,63,248,117,134,63,134,224,192,63,248,117,134,63,134,224,192,63,248,117,134,63,134,224,192,63,248,117,134,63,134,224,192,63,248,117,134,63,0,24,98,20,25,44,168,63,0,24,98,20,25,44,168,63,0,24,98,20,25,44,168,63,0,24,98,20,25,44,168,63,0,24,98,20,25,44,168,63,0,24,98,20,25,44,168,63,216,61,194,47,22,90,159,63,216,61,194,47,22,90,159,63,216,61,194,47,22,90,159,63,216,61,194,47,22,90,159,63,216,61,194,47,22,90,159,63,216,61,194,47,22,90,159,63,5,221,46,138,146,76,160,63,5,221,46,138,146,76,160,63,5,221,46,138,146,76,160,63,5,221,46,138,146,76,160,63,5,221,46,138,146,76,160,63,5,221,46,138,146,76,160,63,218,226,176,212,218,191,140,63,218,226,176,212,218,191,140,63,218,226,176,212,218,191,140,63,218,226,176,212,218,191,140,63,218,226,176,212,218,191,140,63,218,226,176,212,218,191,140,63,51,232,168,172,128,20,139,63,51,232,168,172,128,20,139,63,51,232,168,172,128,20,139,63,51,232,168,172,128,20,139,63,51,232,168,172,128,20,139,63,51,232,168,172,128,20,139,63,218,96,228,204,170,248,20,63,218,96,228,204,170,248,20,63,218,96,228,204,170,248,20,63,218,96,228,204,170,248,20,63,218,96,228,204,170,248,20,63,218,96,228,204,170,248,20,63,191,98,58,139,129,46,173,63,168,202,120,143,162,82,146,63,168,202,120,143,162,82,146,63,168,202,120,143,162,82,146,63,192,158,174,156,109,218,163,63,192,158,174,156,109,218,163,63,192,158,174,156,109,218,163,63,46,229,40,33,30,215,170,63,46,229,40,33,30,215,170,63,46,229,40,33,30,215,170,63,197,98,28,125,41,10,171,63,197,98,28,125,41,10,171,63,197,98,28,125,41,10,171,63,6,234,10,171,180,108,165,63,6,234,10,171,180,108,165,63,6,234,10,171,180,108,165,63,236,15,224,152,208,119,156,63,236,15,224,152,208,119,156,63,236,15,224,152,208,119,156,63,66,48,249,237,141,173,140,63,66,48,249,237,141,173,140,63,66,48,249,237,141,173,140,63,199,21,148,42,5,129,109,63,199,21,148,42,5,129,109,63,199,21,148,42,5,129,109,63,30,44,89,158,169,143,123,63,30,44,89,158,169,143,123,63,30,44,89,158,169,143,123,63,30,44,89,158,169,143,123,63,30,44,89,158,169,143,123,63,30,44,89,158,169,143,123,63,135,143,230,148,238,173,166,63,135,143,230,148,238,173,166,63,135,143,230,148,238,173,166,63,135,143,230,148,238,173,166,63,135,143,230,148,238,173,166,63,135,143,230,148,238,173,166,63,245,53,249,189,176,128,143,63,245,53,249,189,176,128,143,63,245,53,249,189,176,128,143,63,245,53,249,189,176,128,143,63,245,53,249,189,176,128,143,63,245,53,249,189,176,128,143,63,151,247,191,52,76,153,156,63,151,247,191,52,76,153,156,63,151,247,191,52,76,153,156,63,151,247,191,52,76,153,156,63,151,247,191,52,76,153,156,63,151,247,191,52,76,153,156,63,106,214,186,4,213,173,113,63,106,214,186,4,213,173,113,63,106,214,186,4,213,173,113,63,106,214,186,4,213,173,113,63,106,214,186,4,213,173,113,63,106,214,186,4,213,173,113,63,14,200,68,9,141,45,160,63,14,200,68,9,141,45,160,63,14,200,68,9,141,45,160,63,14,200,68,9,141,45,160,63,14,200,68,9,141,45,160,63,14,200,68,9,141,45,160,63,180,42,135,6,222,49,146,63,180,42,135,6,222,49,146,63,180,42,135,6,222,49,146,63,180,42,135,6,222,49,146,63,180,42,135,6,222,49,146,63,180,42,135,6,222,49,146,63,189,181,160,139,97,245,122,63,189,181,160,139,97,245,122,63,189,181,160,139,97,245,122,63,189,181,160,139,97,245,122,63,189,181,160,139,97,245,122,63,189,181,160,139,97,245,122,63,211,138,202,83,185,80,173,63,135,21,116,242,172,154,88,63,135,21,116,242,172,154,88,63,135,21,116,242,172,154,88,63,187,123,10,114,53,174,148,63,187,123,10,114,53,174,148,63,187,123,10,114,53,174,148,63,66,32,126,90,153,73,164,63,66,32,126,90,153,73,164,63,66,32,126,90,153,73,164,63,202,237,215,101,161,0,171,63,202,237,215,101,161,0,171,63,202,237,215,101,161,0,171,63,247,35,114,224,144,40,171,63,247,35,114,224,144,40,171,63,247,35,114,224,144,40,171,63,219,5,106,80,30,156,165,63,219,5,106,80,30,156,165,63,219,5,106,80,30,156,165,63,97,237,52,231,148,95,156,63,97,237,52,231,148,95,156,63,97,237,52,231,148,95,156,63,125,236,50,9,199,78,139,63,125,236,50,9,199,78,139,63,125,236,50,9,199,78,139,63,176,108,115,143,194,239,81,191,176,108,115,143,194,239,81,191,176,108,115,143,194,239,81,191,104,48,239,61,198,216,104,63,104,48,239,61,198,216,104,63,104,48,239,61,198,216,104,63,234,221,229,168,66,52,157,63,234,221,229,168,66,52,157,63,234,221,229,168,66,52,157,63,234,221,229,168,66,52,157,63,234,221,229,168,66,52,157,63,234,221,229,168,66,52,157,63,84,225,51,112,75,42,129,63,84,225,51,112,75,42,129,63,84,225,51,112,75,42,129,63,84,225,51,112,75,42,129,63,84,225,51,112,75,42,129,63,84,225,51,112,75,42,129,63,228,226,156,17,81,226,166,63,228,226,156,17,81,226,166,63,228,226,156,17,81,226,166,63,228,226,156,17,81,226,166,63,228,226,156,17,81,226,166,63,228,226,156,17,81,226,166,63,171,16,87,214,138,12,142,63,171,16,87,214,138,12,142,63,171,16,87,214,138,12,142,63,171,16,87,214,138,12,142,63,171,16,87,214,138,12,142,63,171,16,87,214,138,12,142,63,59,213,209,140,254,70,160,63,59,213,209,140,254,70,160,63,59,213,209,140,254,70,160,63,59,213,209,140,254,70,160,63,59,213,209,140,254,70,160,63,59,213,209,140,254,70,160,63,64,21,169,124,82,253,83,63,64,21,169,124,82,253,83,63,64,21,169,124,82,253,83,63,64,21,169,124,82,253,83,63,64,21,169,124,82,253,83,63,64,21,169,124,82,253,83,63,26,239,108,169,161,239,145,63,26,239,108,169,161,239,145,63,26,239,108,169,161,239,145,63,26,239,108,169,161,239,145,63,26,239,108,169,161,239,145,63,26,239,108,169,161,239,145,63,75,117,252,46,227,90,121,63,75,117,252,46,227,90,121,63,75,117,252,46,227,90,121,63,75,117,252,46,227,90,121,63,75,117,252,46,227,90,121,63,75,117,252,46,227,90,121,63,79,85,85,85,85,85,213,63,97,85,85,85,85,85,197,63,97,85,85,85,85,85,197,63,88,85,85,85,85,85,229,63,79,85,85,85,85,85,213,63,154,153,153,153,153,153,201,63,154,153,153,153,153,153,201,63,51,51,51,51,51,51,227,63,220,208,11,138,107,138,220,63,220,208,11,138,107,138,220,63,32,121,161,175,163,172,187,63,219,176,29,234,137,113,183,63,219,176,29,234,137,113,183,63,210,147,120,133,157,35,234,63,79,85,85,85,85,85,213,63,191,139,115,189,206,22,222,63,191,139,115,189,206,22,222,63,16,68,199,40,20,147,174,63,119,245,122,156,233,237,185,63,119,245,122,156,233,237,185,63,153,66,225,152,133,132,233,63,145,236,234,200,160,232,207,63,145,236,234,200,160,232,207,63,175,137,138,155,175,11,224,63,196,85,250,5,154,38,176,63,196,85,250,5,154,38,176,63,143,106,129,126,89,246,235,63,196,142,49,125,58,94,228,63,87,27,219,134,208,220,211,63,196,142,49,125,58,94,228,63,87,27,219,134,208,220,211,63,9,57,14,246,211,53,171,63,9,57,14,246,211,53,171,63,79,85,85,85,85,85,213,63,193,30,125,32,130,169,208,63,193,30,125,32,130,169,208,63,125,194,5,191,251,172,222,63,225,59,243,205,93,172,176,63,225,59,243,205,93,172,176,63,8,49,131,140,232,212,235,63,158,80,199,129,34,110,228,63,240,208,88,0,253,5,212,63,158,80,199,129,34,110,228,63,240,208,88,0,253,5,212,63,162,110,196,224,239,237,168,63,162,110,196,224,239,237,168,63,79,85,85,85,85,85,213,63,200,203,128,189,12,101,221,63,200,203,128,189,12,101,221,63,194,161,249,19,154,215,180,63,174,97,63,15,55,213,197,63,174,97,63,15,55,213,197,63,41,79,96,120,100,21,229,63,6,217,12,137,83,225,169,63,6,217,12,137,83,225,169,63,223,100,222,142,213,195,236,63,141,26,100,71,207,79,231,63,34,21,156,49,215,214,208,63,141,26,100,71,207,79,231,63,34,21,156,49,215,214,208,63,142,184,118,243,71,49,129,63,142,184,118,243,71,49,129,63,79,85,85,85,85,85,213,63,9,106,88,89,245,86,223,63,9,106,88,89,245,86,223,63,5,192,242,212,84,33,149,63,38,150,56,159,70,249,219,63,38,150,56,159,70,249,219,63,142,167,29,131,229,26,192,63,55,215,60,175,13,23,200,63,55,215,60,175,13,23,200,63,109,148,97,40,121,244,227,63,68,237,99,106,201,230,166,63,68,237,99,106,201,230,166,63],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE);allocate([96,130,179,210,38,35,237,63,174,218,179,31,230,183,231,63,26,23,175,129,72,105,204,63,174,218,179,31,230,183,231,63,26,23,175,129,72,105,204,63,187,248,5,254,123,220,162,63,187,248,5,254,123,220,162,63,79,85,85,85,85,85,213,63,20,82,195,53,180,19,223,63,20,82,195,53,180,19,223,63,98,188,149,71,121,137,157,63,183,8,57,10,252,6,188,63,183,8,57,10,252,6,188,63,210,189,113,253,64,254,232,63,87,29,204,197,125,156,225,63,136,39,40,76,73,181,211,63,87,29,204,197,125,156,225,63,136,39,40,76,73,181,211,63,149,59,127,80,118,35,194,63,149,59,127,80,118,35,194,63,74,130,228,238,109,78,231,63,74,109,98,106,247,146,207,63,74,130,228,238,109,78,231,63,74,109,98,106,247,146,207,63,122,76,92,208,134,154,153,63,122,76,92,208,134,154,153,63,25,120,112,227,150,142,237,63,26,39,93,146,4,26,177,63,25,120,112,227,150,142,237,63,26,39,93,146,4,26,177,63,162,190,248,144,34,138,131,63,162,190,248,144,34,138,131,63,224,230,12,161,136,27,225,63,224,230,12,161,136,27,225,63,71,110,206,16,138,184,177,191,44,41,22,139,182,136,217,63,44,41,22,139,182,136,217,63,78,91,167,211,37,221,201,63,114,22,167,14,15,6,202,63,114,22,167,14,15,6,202,63,190,116,172,120,248,252,226,63,9,95,0,9,200,141,190,63,9,95,0,9,200,141,190,63,53,232,191,253,141,92,232,63,1,110,89,115,35,146,160,63,1,110,89,115,35,146,160,63,64,210,148,145,187,237,237,63,228,228,115,28,129,251,226,63,160,19,87,105,223,210,214,63,228,228,115,28,129,251,226,63,160,19,87,105,223,210,214,63,80,21,9,238,242,176,169,63,80,21,9,238,242,176,169,63,41,110,142,45,243,214,233,63,47,223,3,213,89,243,197,63,41,110,142,45,243,214,233,63,47,223,3,213,89,243,197,63,103,65,19,166,203,134,149,63,103,65,19,166,203,134,149,63,194,175,154,38,244,62,223,63,194,175,154,38,244,62,223,63,204,7,170,44,123,33,152,63,159,31,10,199,113,36,220,63,159,31,10,199,113,36,220,63,188,2,175,199,113,220,190,63,89,155,125,205,130,91,209,63,89,155,125,205,130,91,209,63,77,201,4,101,250,72,221,63,151,232,108,70,106,84,192,63,151,232,108,70,106,84,192,63,181,139,201,220,202,213,231,63,181,184,34,44,55,212,149,63,181,184,34,44,55,212,149,63,108,212,61,141,188,162,238,63,107,109,89,134,118,124,227,63,56,215,23,69,73,165,209,63,107,109,89,134,118,124,227,63,56,215,23,69,73,165,209,63,204,55,213,184,38,135,189,63,204,55,213,184,38,135,189,63,219,42,132,10,74,68,230,63,200,81,120,2,61,1,210,63,219,42,132,10,74,68,230,63,200,81,120,2,61,1,210,63,14,136,245,135,238,98,151,63,14,136,245,135,238,98,151,63,63,150,178,216,217,116,235,63,1,164,235,128,175,194,189,63,63,150,178,216,217,116,235,63,1,164,235,128,175,194,189,63,32,168,254,229,6,90,154,63,32,168,254,229,6,90,154,63,79,85,85,85,85,85,213,63,255,166,16,149,222,174,223,63,255,166,16,149,222,174,223,63,109,66,214,187,90,72,132,63,25,193,184,14,116,255,221,63,25,193,184,14,116,255,221,63,57,247,57,138,95,4,176,63,92,144,67,128,132,135,218,63,92,144,67,128,132,135,218,63,181,190,241,254,237,225,197,63,40,247,209,21,247,92,205,63,40,247,209,21,247,92,205,63,99,4,23,117,132,81,225,63,142,195,186,120,236,74,189,63,142,195,186,120,236,74,189,63,28,79,209,225,68,173,232,63,96,178,229,73,40,104,153,63,96,178,229,73,40,104,153,63,218,164,97,123,125,105,238,63,187,110,112,35,253,92,228,63,230,135,108,238,239,51,209,63,187,110,112,35,253,92,228,63,230,135,108,238,239,51,209,63,143,106,202,42,87,72,184,63,143,106,202,42,87,72,184,63,227,63,91,162,221,21,230,63,245,141,248,149,180,171,210,63,227,63,91,162,221,21,230,63,245,141,248,149,180,171,210,63,66,36,15,85,2,137,146,63,66,36,15,85,2,137,146,63,108,4,16,55,191,62,235,63,117,187,37,147,122,44,192,63,108,4,16,55,191,62,235,63,117,187,37,147,122,44,192,63,217,150,209,132,68,196,150,63,217,150,209,132,68,196,150,63,104,130,251,72,47,75,223,63,104,130,251,72,47,75,223,63,38,180,143,224,22,154,150,63,176,132,186,234,176,186,218,63,176,132,186,234,176,186,218,63,64,237,21,85,60,21,197,63,142,192,17,228,167,128,209,63,142,192,17,228,167,128,209,63,245,126,220,55,176,254,220,63,248,233,48,191,171,174,198,63,248,233,48,191,171,174,198,63,251,138,103,32,170,168,228,63,124,57,139,249,59,164,175,63,124,57,139,249,59,164,175,63,218,152,206,128,120,11,236,63,99,205,154,97,57,219,147,63,99,205,154,97,57,219,147,63,42,83,230,105,76,194,238,63,137,216,152,69,211,168,232,63,4,5,49,183,213,12,198,63,137,216,152,69,211,168,232,63,4,5,49,183,213,12,198,63,100,99,174,201,116,63,173,63,100,99,174,201,116,63,173,63,224,95,84,209,66,63,226,63,115,235,70,94,35,143,213,63,224,95,84,209,66,63,226,63,115,235,70,94,35,143,213,63,51,83,65,252,91,201,183,63,51,83,65,252,91,201,183,63,4,144,55,213,189,251,229,63,162,118,178,146,138,24,211,63,4,144,55,213,189,251,229,63,162,118,178,146,138,24,211,63,207,42,205,91,56,255,141,63,207,42,205,91,56,255,141,63,111,154,238,130,248,38,236,63,248,156,170,216,28,117,190,63,111,154,238,130,248,38,236,63,248,156,170,216,28,117,190,63,64,227,35,248,195,199,84,63,64,227,35,248,195,199,84,63,233,164,59,68,31,57,224,63,233,164,59,68,31,57,224,63,160,116,210,29,162,143,140,191,186,168,187,101,41,156,219,63,186,168,187,101,41,156,219,63,59,93,17,105,90,143,193,63,194,59,187,144,187,197,209,63,194,59,187,144,187,197,209,63,105,136,137,222,136,116,220,63,79,164,142,101,0,48,192,63,79,164,142,101,0,48,192,63,216,173,56,205,255,231,231,63,226,55,11,150,127,32,178,63,226,55,11,150,127,32,178,63,8,50,125,26,224,119,235,63,152,62,203,7,155,107,147,63,152,62,203,7,155,107,147,63,13,76,131,79,70,201,238,63,141,6,92,120,201,91,227,63,101,167,71,84,83,185,208,63,141,6,92,120,201,91,227,63,101,167,71,84,83,185,208,63,223,150,0,118,51,30,193,63,223,150,0,118,51,30,193,63,1,60,254,126,52,107,226,63,180,40,126,27,194,213,216,63,1,60,254,126,52,107,226,63,180,40,126,27,194,213,216,63,78,250,42,52,167,158,162,63,78,250,42,52,167,158,162,63,41,203,162,82,204,46,231,63,251,164,75,232,27,73,210,63,41,203,162,82,204,46,231,63,251,164,75,232,27,73,210,63,192,105,39,178,145,214,132,191,192,105,39,178,145,214,132,191,45,100,179,137,251,235,231,63,179,9,28,17,197,152,203,63,45,100,179,137,251,235,231,63,179,9,28,17,197,152,203,63,103,150,89,32,51,221,162,63,103,150,89,32,51,221,162,63,120,149,17,19,112,73,236,63,22,206,219,125,238,131,186,63,120,149,17,19,112,73,236,63,22,206,219,125,238,131,186,63,159,51,188,76,135,132,137,63,159,51,188,76,135,132,137,63,79,85,85,85,85,85,213,63,131,117,85,50,21,213,223,63,131,117,85,50,21,213,223,63,9,58,69,213,102,117,117,63,183,193,29,133,72,118,218,63,183,193,29,133,72,118,218,63,254,248,136,235,221,38,198,63,73,87,71,98,254,27,222,63,73,87,71,98,254,27,222,63,224,138,138,219,25,64,174,63,242,77,134,28,119,202,206,63,242,77,134,28,119,202,206,63,7,217,188,113,196,154,224,63,58,106,142,11,139,240,194,63,58,106,142,11,139,240,194,63,227,202,56,122,186,135,230,63,199,176,249,192,175,81,179,63,199,176,249,192,175,81,179,63,206,147,193,15,148,43,235,63,135,18,253,185,165,254,144,63,135,18,253,185,165,254,144,63,216,46,96,164,21,240,238,63,103,120,12,222,31,50,227,63,153,104,48,83,196,250,210,63,103,120,12,222,31,50,227,63,153,104,48,83,196,250,210,63,97,154,218,194,239,131,186,63,97,154,218,194,239,131,186,63,134,116,106,111,217,140,228,63,68,244,26,50,65,157,213,63,134,116,106,111,217,140,228,63,68,244,26,50,65,157,213,63,242,42,2,241,190,144,148,63,242,42,2,241,190,144,148,63,203,32,86,121,67,150,233,63,172,182,200,18,49,53,202,63,203,32,86,121,67,150,233,63,172,182,200,18,49,53,202,63,192,58,39,4,223,199,113,191,192,58,39,4,223,199,113,191,58,159,123,40,48,153,232,63,6,224,117,41,230,60,200,63,58,159,123,40,48,153,232,63,6,224,117,41,230,60,200,63,216,140,110,210,100,121,165,63,216,140,110,210,100,121,165,63,44,61,179,178,17,208,236,63,106,220,61,164,37,213,181,63,44,61,179,178,17,208,236,63,106,220,61,164,37,213,181,63,130,207,65,49,102,82,141,63,130,207,65,49,102,82,141,63,79,85,85,85,85,85,213,63,142,118,14,99,164,209,223,63,142,118,14,99,164,209,223,63,23,185,196,120,206,45,119,63,5,3,37,20,250,219,222,63,5,3,37,20,250,219,222,63,67,208,175,189,94,64,162,63,198,246,146,77,187,208,220,63,198,246,146,77,187,208,220,63,25,74,104,147,37,122,185,63,172,103,244,73,246,157,217,63,172,103,244,73,246,157,217,63,82,97,46,216,38,136,201,63,209,15,250,32,21,35,208,63,209,15,250,32,21,35,208,63,112,224,11,190,213,185,223,63,240,31,182,204,244,189,196,63,240,31,182,204,244,189,196,63,255,239,164,153,5,161,229,63,245,115,221,16,154,108,179,63,245,115,221,16,154,108,179,63,3,163,200,123,217,36,235,63,129,29,107,241,202,7,144,63,129,29,107,241,202,7,144,63,40,78,233,80,131,255,238,63,84,147,35,226,204,249,228,63,207,183,46,40,127,101,213,63,84,147,35,226,204,249,228,63,207,183,46,40,127,101,213,63,22,49,68,113,226,220,132,63,22,49,68,113,226,220,132,63,42,104,51,236,150,80,226,63,185,148,1,247,193,179,210,63,42,104,51,236,150,80,226,63,185,148,1,247,193,179,210,63,231,53,47,97,32,86,193,63,231,53,47,97,32,86,193,63,95,42,241,165,51,8,228,63,190,254,96,56,234,115,212,63,95,42,241,165,51,8,228,63,190,254,96,56,234,115,212,63,182,100,229,221,115,221,171,63,182,100,229,221,115,221,171,63,232,167,62,237,84,124,233,63,142,182,14,251,254,104,200,63,232,167,62,237,84,124,233,63,142,182,14,251,254,104,200,63,20,157,106,255,212,90,138,63,20,157,106,255,212,90,138,63,28,176,179,106,66,19,232,63,151,19,193,233,18,26,199,63,28,176,179,106,66,19,232,63,151,19,193,233,18,26,199,63,168,87,224,214,198,49,177,63,168,87,224,214,198,49,177,63,23,251,102,47,177,242,236,63,186,193,171,36,127,169,180,63,23,251,102,47,177,242,236,63,186,193,171,36,127,169,180,63,136,44,227,0,187,7,142,63,136,44,227,0,187,7,142,63,79,85,85,85,85,85,213,63,226,39,174,26,246,146,223,63,226,39,174,26,246,146,223,63,70,5,118,84,121,66,139,63,250,45,77,223,139,7,222,63,250,45,77,223,139,7,222,63,93,32,45,11,66,135,175,63,224,6,204,205,8,236,219,63,224,6,204,205,8,236,219,63,163,228,207,200,220,79,192,63,222,143,127,223,40,69,217,63,222,143,127,223,40,69,217,63,135,192,1,130,92,235,202,63,125,46,221,183,68,249,207,63,125,46,221,183,68,249,207,63,194,104,17,164,93,3,224,63,40,95,83,100,207,169,196,63,40,95,83,100,207,169,196,63,117,80,214,77,24,171,229,63,85,50,251,1,241,155,179,63,85,50,251,1,241,155,179,63,98,51,129,191,3,25,235,63,190,51,124,167,161,213,152,63,190,51,124,167,161,213,152,63,196,60,136,229,165,114,238,63,69,177,70,216,71,23,166,63,69,177,70,216,71,23,166,63,224,41,247,4,23,61,237,63,146,96,185,236,187,62,228,63,60,130,59,235,103,248,214,63,146,96,185,236,187,62,228,63,60,130,59,235,103,248,214,63,24,148,55,106,7,68,129,63,24,148,55,106,7,68,129,63,102,73,35,30,147,97,226,63,35,181,96,124,125,215,210,63,102,73,35,30,147,97,226,63,35,181,96,124,125,215,210,63,33,112,177,142,184,202,192,63,33,112,177,142,184,202,192,63,133,101,166,160,48,254,227,63,141,215,42,119,23,205,212,63,133,101,166,160,48,254,227,63,141,215,42,119,23,205,212,63,207,235,66,60,58,180,169,63,207,235,66,60,58,180,169,63,111,201,129,181,66,247,231,63,220,227,141,248,122,165,199,63,111,201,129,181,66,247,231,63,220,227,141,248,122,165,199,63,138,236,213,98,244,250,176,63,138,236,213,98,244,250,176,63,20,81,132,2,88,157,232,63,203,105,67,150,136,1,204,63,20,81,132,2,88,157,232,63,203,105,67,150,136,1,204,63,54,30,181,250,117,145,136,63,54,30,181,250,117,145,136,63,63,42,101,79,107,73,236,63,0,44,185,244,231,230,185,63,63,42,101,79,107,73,236,63,0,44,185,244,231,230,185,63,38,16,236,128,236,109,142,63,38,16,236,128,236,109,142,63,244,41,72,49,196,58,240,63,39,186,0,29,49,96,149,63,244,41,72,49,196,58,240,63,39,186,0,29,49,96,149,63,161,155,133,183,158,8,162,191,161,155,133,183,158,8,162,191,79,85,85,85,85,85,213,63,189,129,145,32,197,85,223,63,189,129,145,32,197,85,223,63,72,199,207,237,91,71,149,63,88,160,61,232,33,23,221,63,88,160,61,232,33,23,221,63,135,253,18,190,240,70,183,63,219,82,221,150,207,176,217,63,219,82,221,150,207,176,217,63,146,180,138,164,193,60,201,63,104,102,134,83,245,90,208,63,104,102,134,83,245,90,208,63,67,51,243,88,21,74,223,63,215,116,49,113,125,170,198,63,215,116,49,113,125,170,198,63,157,69,103,71,193,170,228,63,15,104,45,16,246,44,188,63,15,104,45,16,246,44,188,63,252,165,244,123,194,244,232,63,221,154,78,114,63,110,172,63,221,154,78,114,63,110,172,63,173,44,182,17,56,114,236,63,216,36,26,225,125,217,137,63,216,36,26,225,125,217,137,63,226,46,247,16,52,49,239,63,244,125,48,93,100,56,227,63,138,190,59,227,11,84,217,63,244,125,48,93,100,56,227,63,138,190,59,227,11,84,217,63,226,198,162,49,177,149,109,63,226,198,162,49,177,149,109,63,26,86,149,203,226,215,225,63,154,90,142,246,31,181,211,63,26,86,149,203,226,215,225,63,154,90,142,246,31,181,211,63,102,242,141,228,52,54,193,63,102,242,141,228,52,54,193,63,126,58,23,98,83,18,231,63,163,139,192,53,170,238,208,63,126,58,23,98,83,18,231,63,163,139,192,53,170,238,208,63,12,236,31,194,224,149,141,63,12,236,31,194,224,149,141,63,12,18,184,161,93,6,227,63,240,109,3,10,79,242,214,63,12,18,184,161,93,6,227,63,240,109,3,10,79,242,214,63,200,111,99,148,173,7,168,63,200,111,99,148,173,7,168,63,88,206,190,173,205,219,234,63,168,209,217,119,8,51,196,63,88,206,190,173,205,219,234,63,168,209,217,119,8,51,196,63,94,71,189,74,52,112,103,63,94,71,189,74,52,112,103,63,174,105,163,14,80,111,230,63,97,139,142,19,132,54,179,63,174,105,163,14,80,111,230,63,97,139,142,19,132,54,179,63,153,19,171,187,125,167,204,63,153,19,171,187,125,167,204,63,10,125,119,18,116,85,234,63,233,81,131,251,222,58,194,63,10,125,119,18,116,85,234,63,233,81,131,251,222,58,194,63,193,231,122,234,66,189,161,63,193,231,122,234,66,189,161,63,253,14,232,101,58,148,237,63,13,14,174,133,65,196,176,63,253,14,232,101,58,148,237,63,13,14,174,133,65,196,176,63,97,208,139,88,90,207,132,63,97,208,139,88,90,207,132,63,79,85,85,85,85,85,213,63,250,224,169,68,201,7,224,63,250,224,169,68,201,7,224,63,225,230,131,167,18,37,95,191,149,9,244,143,225,62,223,63,149,9,244,143,225,62,223,63,139,206,126,1,206,35,152,63,74,238,11,155,245,32,221,63,74,238,11,155,245,32,221,63,249,141,160,39,83,248,182,63,142,186,184,132,78,186,217,63,142,186,184,132,78,186,217,63,162,21,29,237,197,22,201,63,193,122,172,167,140,96,208,63,193,122,172,167,140,96,208,63,107,10,167,176,230,62,223,63,97,67,208,203,42,151,198,63,97,67,208,203,42,151,198,63,80,222,23,154,106,180,228,63,19,35,222,240,240,170,186,63,19,35,222,240,240,170,186,63,50,119,200,195,67,85,233,63,214,64,66,240,218,43,171,63,214,64,66,240,218,43,171,63,229,183,247,161,132,154,236,63,96,67,86,89,12,79,165,63,96,67,86,89,12,79,165,63,148,55,213,116,30,86,237,63,247,78,86,45,67,184,135,63,247,78,86,45,67,184,135,63,136,77,149,230,61,66,239,63,114,123,167,133,166,103,227,63,217,27,190,239,29,18,214,63,114,123,167,133,166,103,227,63,217,27,190,239,29,18,214,63,25,106,151,39,168,244,168,63,25,106,151,39,168,244,168,63,12,94,18,146,251,180,227,63,105,28,123,137,149,46,216,63,12,94,18,146,251,180,227,63,105,28,123,137,149,46,216,63,149,223,9,152,212,220,121,63,149,223,9,152,212,220,121,63,163,65,210,161,184,227,225,63,114,110,151,106,234,159,211,63,163,65,210,161,184,227,225,63,114,110,151,106,234,159,211,63,144,28,136,163,72,49,193,63,144,28,136,163,72,49,193,63,108,119,10,78,72,146,231,63,149,167,13,67,249,236,207,63,108,119,10,78,72,146,231,63,149,167,13,67,249,236,207,63,184,171,135,76,88,158,140,63,184,171,135,76,88,158,140,63,137,5,176,242,10,198,230,63,210,146,180,37,60,60,203,63,137,5,176,242,10,198,230,63,210,146,180,37,60,60,203,63,20,174,22,31,48,87,179,63,20,174,22,31,48,87,179,63,248,183,91,110,156,144,235,63,109,218,190,110,195,207,194,63,248,183,91,110,156,144,235,63,109,218,190,110,195,207,194,63,43,167,219,130,82,35,129,191,43,167,219,130,82,35,129,191,201,83,38,220,32,189,234,63,15,110,236,162,9,161,193,63,201,83,38,220,32,189,234,63,15,110,236,162,9,161,193,63,119,22,210,99,151,83,155,63,119,22,210,99,151,83,155,63,60,232,194,3,144,192,237,63,12,163,252,8,125,144,174,63,60,232,194,3,144,192,237,63,12,163,252,8,125,144,174,63,0,103,83,235,10,154,133,63,0,103,83,235,10,154,133,63,79,85,85,85,85,85,213,63,97,85,85,85,85,85,197,63,88,85,85,85,85,85,229,63,97,85,85,85,85,85,197,63,79,85,85,85,85,85,213,63,154,153,153,153,153,153,201,63,51,51,51,51,51,51,227,63,154,153,153,153,153,153,201,63,220,208,11,138,107,138,220,63,32,121,161,175,163,172,187,63,220,208,11,138,107,138,220,63,219,176,29,234,137,113,183,63,210,147,120,133,157,35,234,63,219,176,29,234,137,113,183,63,79,85,85,85,85,85,213,63,191,139,115,189,206,22,222,63,16,68,199,40,20,147,174,63,191,139,115,189,206,22,222,63,119,245,122,156,233,237,185,63,153,66,225,152,133,132,233,63,119,245,122,156,233,237,185,63,145,236,234,200,160,232,207,63,175,137,138,155,175,11,224,63,145,236,234,200,160,232,207,63,196,85,250,5,154,38,176,63,143,106,129,126,89,246,235,63,196,85,250,5,154,38,176,63,87,27,219,134,208,220,211,63,196,142,49,125,58,94,228,63,9,57,14,246,211,53,171,63,9,57,14,246,211,53,171,63,196,142,49,125,58,94,228,63,87,27,219,134,208,220,211,63,79,85,85,85,85,85,213,63,193,30,125,32,130,169,208,63,125,194,5,191,251,172,222,63,193,30,125,32,130,169,208,63,225,59,243,205,93,172,176,63,8,49,131,140,232,212,235,63,225,59,243,205,93,172,176,63,240,208,88,0,253,5,212,63,158,80,199,129,34,110,228,63,162,110,196,224,239,237,168,63,162,110,196,224,239,237,168,63,158,80,199,129,34,110,228,63,240,208,88,0,253,5,212,63,79,85,85,85,85,85,213,63,200,203,128,189,12,101,221,63,194,161,249,19,154,215,180,63,200,203,128,189,12,101,221,63,174,97,63,15,55,213,197,63,41,79,96,120,100,21,229,63,174,97,63,15,55,213,197,63,6,217,12,137,83,225,169,63,223,100,222,142,213,195,236,63,6,217,12,137,83,225,169,63,34,21,156,49,215,214,208,63,141,26,100,71,207,79,231,63,142,184,118,243,71,49,129,63,142,184,118,243,71,49,129,63,141,26,100,71,207,79,231,63,34,21,156,49,215,214,208,63,79,85,85,85,85,85,213,63,9,106,88,89,245,86,223,63,5,192,242,212,84,33,149,63,9,106,88,89,245,86,223,63,38,150,56,159,70,249,219,63,142,167,29,131,229,26,192,63,38,150,56,159,70,249,219,63,55,215,60,175,13,23,200,63,109,148,97,40,121,244,227,63,55,215,60,175,13,23,200,63,68,237,99,106,201,230,166,63,96,130,179,210,38,35,237,63,68,237,99,106,201,230,166,63,26,23,175,129,72,105,204,63,174,218,179,31,230,183,231,63,187,248,5,254,123,220,162,63,187,248,5,254,123,220,162,63,174,218,179,31,230,183,231,63,26,23,175,129,72,105,204,63,79,85,85,85,85,85,213,63,20,82,195,53,180,19,223,63,98,188,149,71,121,137,157,63,20,82,195,53,180,19,223,63,183,8,57,10,252,6,188,63,210,189,113,253,64,254,232,63,183,8,57,10,252,6,188,63,136,39,40,76,73,181,211,63,87,29,204,197,125,156,225,63,149,59,127,80,118,35,194,63,149,59,127,80,118,35,194,63,87,29,204,197,125,156,225,63,136,39,40,76,73,181,211,63,74,109,98,106,247,146,207,63,74,130,228,238,109,78,231,63,122,76,92,208,134,154,153,63,122,76,92,208,134,154,153,63,74,130,228,238,109,78,231,63,74,109,98,106,247,146,207,63,26,39,93,146,4,26,177,63,25,120,112,227,150,142,237,63,162,190,248,144,34,138,131,63,162,190,248,144,34,138,131,63,25,120,112,227,150,142,237,63,26,39,93,146,4,26,177,63,224,230,12,161,136,27,225,63,71,110,206,16,138,184,177,191,224,230,12,161,136,27,225,63,44,41,22,139,182,136,217,63,78,91,167,211,37,221,201,63,44,41,22,139,182,136,217,63,114,22,167,14,15,6,202,63,190,116,172,120,248,252,226,63,114,22,167,14,15,6,202,63,9,95,0,9,200,141,190,63,53,232,191,253,141,92,232,63,9,95,0,9,200,141,190,63,1,110,89,115,35,146,160,63,64,210,148,145,187,237,237,63,1,110,89,115,35,146,160,63,160,19,87,105,223,210,214,63,228,228,115,28,129,251,226,63,80,21,9,238,242,176,169,63,80,21,9,238,242,176,169,63,228,228,115,28,129,251,226,63,160,19,87,105,223,210,214,63,47,223,3,213,89,243,197,63,41,110,142,45,243,214,233,63,103,65,19,166,203,134,149,63,103,65,19,166,203,134,149,63,41,110,142,45,243,214,233,63,47,223,3,213,89,243,197,63,194,175,154,38,244,62,223,63,204,7,170,44,123,33,152,63,194,175,154,38,244,62,223,63,159,31,10,199,113,36,220,63,188,2,175,199,113,220,190,63,159,31,10,199,113,36,220,63,89,155,125,205,130,91,209,63,77,201,4,101,250,72,221,63,89,155,125,205,130,91,209,63,151,232,108,70,106,84,192,63,181,139,201,220,202,213,231,63,151,232,108,70,106,84,192,63,181,184,34,44,55,212,149,63,108,212,61,141,188,162,238,63,181,184,34,44,55,212,149,63,56,215,23,69,73,165,209,63,107,109,89,134,118,124,227,63,204,55,213,184,38,135,189,63,204,55,213,184,38,135,189,63,107,109,89,134,118,124,227,63,56,215,23,69,73,165,209,63,200,81,120,2,61,1,210,63,219,42,132,10,74,68,230,63,14,136,245,135,238,98,151,63,14,136,245,135,238,98,151,63,219,42,132,10,74,68,230,63,200,81,120,2,61,1,210,63,1,164,235,128,175,194,189,63,63,150,178,216,217,116,235,63,32,168,254,229,6,90,154,63,32,168,254,229,6,90,154,63,63,150,178,216,217,116,235,63,1,164,235,128,175,194,189,63,79,85,85,85,85,85,213,63,255,166,16,149,222,174,223,63,109,66,214,187,90,72,132,63,255,166,16,149,222,174,223,63,25,193,184,14,116,255,221,63,57,247,57,138,95,4,176,63,25,193,184,14,116,255,221,63,92,144,67,128,132,135,218,63,181,190,241,254,237,225,197,63,92,144,67,128,132,135,218,63,40,247,209,21,247,92,205,63,99,4,23,117,132,81,225,63,40,247,209,21,247,92,205,63,142,195,186,120,236,74,189,63,28,79,209,225,68,173,232,63,142,195,186,120,236,74,189,63,96,178,229,73,40,104,153,63,218,164,97,123,125,105,238,63,96,178,229,73,40,104,153,63,230,135,108,238,239,51,209,63,187,110,112,35,253,92,228,63,143,106,202,42,87,72,184,63,143,106,202,42,87,72,184,63,187,110,112,35,253,92,228,63,230,135,108,238,239,51,209,63,245,141,248,149,180,171,210,63,227,63,91,162,221,21,230,63,66,36,15,85,2,137,146,63,66,36,15,85,2,137,146,63,227,63,91,162,221,21,230,63,245,141,248,149,180,171,210,63,117,187,37,147,122,44,192,63,108,4,16,55,191,62,235,63,217,150,209,132,68,196,150,63,217,150,209,132,68,196,150,63,108,4,16,55,191,62,235,63,117,187,37,147,122,44,192,63,104,130,251,72,47,75,223,63,38,180,143,224,22,154,150,63,104,130,251,72,47,75,223,63,176,132,186,234,176,186,218,63,64,237,21,85,60,21,197,63,176,132,186,234,176,186,218,63,142,192,17,228,167,128,209,63,245,126,220,55,176,254,220,63,142,192,17,228,167,128,209,63,248,233,48,191,171,174,198,63,251,138,103,32,170,168,228,63,248,233,48,191,171,174,198,63,124,57,139,249,59,164,175,63,218,152,206,128,120,11,236,63,124,57,139,249,59,164,175,63,99,205,154,97,57,219,147,63,42,83,230,105,76,194,238,63,99,205,154,97,57,219,147,63,4,5,49,183,213,12,198,63,137,216,152,69,211,168,232,63,100,99,174,201,116,63,173,63,100,99,174,201,116,63,173,63,137,216,152,69,211,168,232,63,4,5,49,183,213,12,198,63,115,235,70,94,35,143,213,63,224,95,84,209,66,63,226,63,51,83,65,252,91,201,183,63,51,83,65,252,91,201,183,63,224,95,84,209,66,63,226,63,115,235,70,94,35,143,213,63,162,118,178,146,138,24,211,63,4,144,55,213,189,251,229,63,207,42,205,91,56,255,141,63,207,42,205,91,56,255,141,63,4,144,55,213,189,251,229,63,162,118,178,146,138,24,211,63,248,156,170,216,28,117,190,63,111,154,238,130,248,38,236,63,64,227,35,248,195,199,84,63,64,227,35,248,195,199,84,63,111,154,238,130,248,38,236,63,248,156,170,216,28,117,190,63,233,164,59,68,31,57,224,63,160,116,210,29,162,143,140,191,233,164,59,68,31,57,224,63,186,168,187,101,41,156,219,63,59,93,17,105,90,143,193,63,186,168,187,101,41,156,219,63,194,59,187,144,187,197,209,63,105,136,137,222,136,116,220,63,194,59,187,144,187,197,209,63,79,164,142,101,0,48,192,63,216,173,56,205,255,231,231,63,79,164,142,101,0,48,192,63,226,55,11,150,127,32,178,63,8,50,125,26,224,119,235,63,226,55,11,150,127,32,178,63,152,62,203,7,155,107,147,63,13,76,131,79,70,201,238,63,152,62,203,7,155,107,147,63,101,167,71,84,83,185,208,63,141,6,92,120,201,91,227,63,223,150,0,118,51,30,193,63,223,150,0,118,51,30,193,63,141,6,92,120,201,91,227,63,101,167,71,84,83,185,208,63,180,40,126,27,194,213,216,63,1,60,254,126,52,107,226,63,78,250,42,52,167,158,162,63,78,250,42,52,167,158,162,63,1,60,254,126,52,107,226,63,180,40,126,27,194,213,216,63,251,164,75,232,27,73,210,63,41,203,162,82,204,46,231,63,192,105,39,178,145,214,132,191,192,105,39,178,145,214,132,191,41,203,162,82,204,46,231,63,251,164,75,232,27,73,210,63,179,9,28,17,197,152,203,63,45,100,179,137,251,235,231,63,103,150,89,32,51,221,162,63,103,150,89,32,51,221,162,63,45,100,179,137,251,235,231,63,179,9,28,17,197,152,203,63,22,206,219,125,238,131,186,63,120,149,17,19,112,73,236,63,159,51,188,76,135,132,137,63,159,51,188,76,135,132,137,63,120,149,17,19,112,73,236,63,22,206,219,125,238,131,186,63,79,85,85,85,85,85,213,63,131,117,85,50,21,213,223,63,9,58,69,213,102,117,117,63,131,117,85,50,21,213,223,63,183,193,29,133,72,118,218,63,254,248,136,235,221,38,198,63,183,193,29,133,72,118,218,63,73,87,71,98,254,27,222,63,224,138,138,219,25,64,174,63,73,87,71,98,254,27,222,63,242,77,134,28,119,202,206,63,7,217,188,113,196,154,224,63,242,77,134,28,119,202,206,63,58,106,142,11,139,240,194,63,227,202,56,122,186,135,230,63,58,106,142,11,139,240,194,63,199,176,249,192,175,81,179,63,206,147,193,15,148,43,235,63,199,176,249,192,175,81,179,63,135,18,253,185,165,254,144,63,216,46,96,164,21,240,238,63,135,18,253,185,165,254,144,63,153,104,48,83,196,250,210,63,103,120,12,222,31,50,227,63,97,154,218,194,239,131,186,63,97,154,218,194,239,131,186,63,103,120,12,222,31,50,227,63,153,104,48,83,196,250,210,63,68,244,26,50,65,157,213,63,134,116,106,111,217,140,228,63,242,42,2,241,190,144,148,63,242,42,2,241,190,144,148,63,134,116,106,111,217,140,228,63,68,244,26,50,65,157,213,63,172,182,200,18,49,53,202,63,203,32,86,121,67,150,233,63,192,58,39,4,223,199,113,191,192,58,39,4,223,199,113,191,203,32,86,121,67,150,233,63,172,182,200,18,49,53,202,63,6,224,117,41,230,60,200,63,58,159,123,40,48,153,232,63,216,140,110,210,100,121,165,63,216,140,110,210,100,121,165,63,58,159,123,40,48,153,232,63,6,224,117,41,230,60,200,63,106,220,61,164,37,213,181,63,44,61,179,178,17,208,236,63,130,207,65,49,102,82,141,63,130,207,65,49,102,82,141,63,44,61,179,178,17,208,236,63,106,220,61,164,37,213,181,63,79,85,85,85,85,85,213,63,142,118,14,99,164,209,223,63,23,185,196,120,206,45,119,63,142,118,14,99,164,209,223,63,5,3,37,20,250,219,222,63,67,208,175,189,94,64,162,63,5,3,37,20,250,219,222,63,198,246,146,77,187,208,220,63,25,74,104,147,37,122,185,63,198,246,146,77,187,208,220,63,172,103,244,73,246,157,217,63,82,97,46,216,38,136,201,63,172,103,244,73,246,157,217,63,209,15,250,32,21,35,208,63,112,224,11,190,213,185,223,63,209,15,250,32,21,35,208,63,240,31,182,204,244,189,196,63,255,239,164,153,5,161,229,63,240,31,182,204,244,189,196,63,245,115,221,16,154,108,179,63,3,163,200,123,217,36,235,63,245,115,221,16,154,108,179,63,129,29,107,241,202,7,144,63,40,78,233,80,131,255,238,63,129,29,107,241,202,7,144,63,207,183,46,40,127,101,213,63,84,147,35,226,204,249,228,63,22,49,68,113,226,220,132,63,22,49,68,113,226,220,132,63,84,147,35,226,204,249,228,63,207,183,46,40,127,101,213,63,185,148,1,247,193,179,210,63,42,104,51,236,150,80,226,63,231,53,47,97,32,86,193,63,231,53,47,97,32,86,193,63,42,104,51,236,150,80,226,63,185,148,1,247,193,179,210,63,190,254,96,56,234,115,212,63,95,42,241,165,51,8,228,63,182,100,229,221,115,221,171,63,182,100,229,221,115,221,171,63,95,42,241,165,51,8,228,63,190,254,96,56,234,115,212,63,142,182,14,251,254,104,200,63,232,167,62,237,84,124,233,63,20,157,106,255,212,90,138,63,20,157,106,255,212,90,138,63,232,167,62,237,84,124,233,63,142,182,14,251,254,104,200,63,151,19,193,233,18,26,199,63,28,176,179,106,66,19,232,63,168,87,224,214,198,49,177,63,168,87,224,214,198,49,177,63,28,176,179,106,66,19,232,63,151,19,193,233,18,26,199,63,186,193,171,36,127,169,180,63,23,251,102,47,177,242,236,63,136,44,227,0,187,7,142,63,136,44,227,0,187,7,142,63,23,251,102,47,177,242,236,63,186,193,171,36,127,169,180,63,79,85,85,85,85,85,213,63,226,39,174,26,246,146,223,63,70,5,118,84,121,66,139,63,226,39,174,26,246,146,223,63,250,45,77,223,139,7,222,63,93,32,45,11,66,135,175,63,250,45,77,223,139,7,222,63,224,6,204,205,8,236,219,63,163,228,207,200,220,79,192,63,224,6,204,205,8,236,219,63,222,143,127,223,40,69,217,63,135,192,1,130,92,235,202,63,222,143,127,223,40,69,217,63,125,46,221,183,68,249,207,63,194,104,17,164,93,3,224,63,125,46,221,183,68,249,207,63,40,95,83,100,207,169,196,63,117,80,214,77,24,171,229,63,40,95,83,100,207,169,196,63,85,50,251,1,241,155,179,63,98,51,129,191,3,25,235,63,85,50,251,1,241,155,179,63,190,51,124,167,161,213,152,63,196,60,136,229,165,114,238,63,190,51,124,167,161,213,152,63,69,177,70,216,71,23,166,63,224,41,247,4,23,61,237,63,69,177,70,216,71,23,166,63,60,130,59,235,103,248,214,63,146,96,185,236,187,62,228,63,24,148,55,106,7,68,129,63,24,148,55,106,7,68,129,63,146,96,185,236,187,62,228,63,60,130,59,235,103,248,214,63,35,181,96,124,125,215,210,63,102,73,35,30,147,97,226,63,33,112,177,142,184,202,192,63,33,112,177,142,184,202,192,63,102,73,35,30,147,97,226,63,35,181,96,124,125,215,210,63,141,215,42,119,23,205,212,63,133,101,166,160,48,254,227,63,207,235,66,60,58,180,169,63,207,235,66,60,58,180,169,63,133,101,166,160,48,254,227,63,141,215,42,119,23,205,212,63,220,227,141,248,122,165,199,63,111,201,129,181,66,247,231,63,138,236,213,98,244,250,176,63,138,236,213,98,244,250,176,63,111,201,129,181,66,247,231,63,220,227,141,248,122,165,199,63,203,105,67,150,136,1,204,63,20,81,132,2,88,157,232,63,54,30,181,250,117,145,136,63,54,30,181,250,117,145,136,63,20,81,132,2,88,157,232,63,203,105,67,150,136,1,204,63,0,44,185,244,231,230,185,63,63,42,101,79,107,73,236,63,38,16,236,128,236,109,142,63,38,16,236,128,236,109,142,63,63,42,101,79,107,73,236,63,0,44,185,244,231,230,185,63,39,186,0,29,49,96,149,63,244,41,72,49,196,58,240,63,161,155,133,183,158,8,162,191,161,155,133,183,158,8,162,191,244,41,72,49,196,58,240,63,39,186,0,29,49,96,149,63,79,85,85,85,85,85,213,63,189,129,145,32,197,85,223,63,72,199,207,237,91,71,149,63,189,129,145,32,197,85,223,63,88,160,61,232,33,23,221,63,135,253,18,190,240,70,183,63,88,160,61,232,33,23,221,63,219,82,221,150,207,176,217,63,146,180,138,164,193,60,201,63,219,82,221,150,207,176,217,63,104,102,134,83,245,90,208,63,67,51,243,88,21,74,223,63,104,102,134,83,245,90,208,63,215,116,49,113,125,170,198,63,157,69,103,71,193,170,228,63,215,116,49,113,125,170,198,63,15,104,45,16,246,44,188,63,252,165,244,123,194,244,232,63,15,104,45,16,246,44,188,63,221,154,78,114,63,110,172,63,173,44,182,17,56,114,236,63,221,154,78,114,63,110,172,63,216,36,26,225,125,217,137,63,226,46,247,16,52,49,239,63,216,36,26,225,125,217,137,63,138,190,59,227,11,84,217,63,244,125,48,93,100,56,227,63,226,198,162,49,177,149,109,63,226,198,162,49,177,149,109,63,244,125,48,93,100,56,227,63,138,190,59,227,11,84,217,63,154,90,142,246,31,181,211,63,26,86,149,203,226,215,225,63,102,242,141,228,52,54,193,63,102,242,141,228,52,54,193,63,26,86,149,203,226,215,225,63,154,90,142,246,31,181,211,63,163,139,192,53,170,238,208,63,126,58,23,98,83,18,231,63,12,236,31,194,224,149,141,63,12,236,31,194,224,149,141,63,126,58,23,98,83,18,231,63,163,139,192,53,170,238,208,63,240,109,3,10,79,242,214,63,12,18,184,161,93,6,227,63,200,111,99,148,173,7,168,63,200,111,99,148,173,7,168,63,12,18,184,161,93,6,227,63,240,109,3,10,79,242,214,63,168,209,217,119,8,51,196,63,88,206,190,173,205,219,234,63,94,71,189,74,52,112,103,63,94,71,189,74,52,112,103,63,88,206,190,173,205,219,234,63,168,209,217,119,8,51,196,63,97,139,142,19,132,54,179,63,174,105,163,14,80,111,230,63,153,19,171,187,125,167,204,63,153,19,171,187,125,167,204,63,174,105,163,14,80,111,230,63,97,139,142,19,132,54,179,63,233,81,131,251,222,58,194,63,10,125,119,18,116,85,234,63,193,231,122,234,66,189,161,63,193,231,122,234,66,189,161,63,10,125,119,18,116,85,234,63,233,81,131,251,222,58,194,63,13,14,174,133,65,196,176,63,253,14,232,101,58,148,237,63,97,208,139,88,90,207,132,63,97,208,139,88,90,207,132,63,253,14,232,101,58,148,237,63,13,14,174,133,65,196,176,63,79,85,85,85,85,85,213,63,250,224,169,68,201,7,224,63,225,230,131,167,18,37,95,191,250,224,169,68,201,7,224,63,149,9,244,143,225,62,223,63,139,206,126,1,206,35,152,63,149,9,244,143,225,62,223,63,74,238,11,155,245,32,221,63,249,141,160,39,83,248,182,63,74,238,11,155,245,32,221,63,142,186,184,132,78,186,217,63,162,21,29,237,197,22,201,63,142,186,184,132,78,186,217,63,193,122,172,167,140,96,208,63,107,10,167,176,230,62,223,63,193,122,172,167,140,96,208,63,97,67,208,203,42,151,198,63,80,222,23,154,106,180,228,63,97,67,208,203,42,151,198,63,19,35,222,240,240,170,186,63,50,119,200,195,67,85,233,63,19,35,222,240,240,170,186,63,214,64,66,240,218,43,171,63,229,183,247,161,132,154,236,63,214,64,66,240,218,43,171,63,96,67,86,89,12,79,165,63,148,55,213,116,30,86,237,63,96,67,86,89,12,79,165,63,247,78,86,45,67,184,135,63,136,77,149,230,61,66,239,63,247,78,86,45,67,184,135,63,217,27,190,239,29,18,214,63,114,123,167,133,166,103,227,63,25,106,151,39,168,244,168,63,25,106,151,39,168,244,168,63,114,123,167,133,166,103,227,63,217,27,190,239,29,18,214,63,105,28,123,137,149,46,216,63,12,94,18,146,251,180,227,63,149,223,9,152,212,220,121,63,149,223,9,152,212,220,121,63,12,94,18,146,251,180,227,63,105,28,123,137,149,46,216,63,114,110,151,106,234,159,211,63,163,65,210,161,184,227,225,63,144,28,136,163,72,49,193,63,144,28,136,163,72,49,193,63,163,65,210,161,184,227,225,63,114,110,151,106,234,159,211,63,149,167,13,67,249,236,207,63,108,119,10,78,72,146,231,63,184,171,135,76,88,158,140,63,184,171,135,76,88,158,140,63,108,119,10,78,72,146,231,63,149,167,13,67,249,236,207,63,210,146,180,37,60,60,203,63,137,5,176,242,10,198,230,63,20,174,22,31,48,87,179,63,20,174,22,31,48,87,179,63,137,5,176,242,10,198,230,63,210,146,180,37,60,60,203,63,109,218,190,110,195,207,194,63,248,183,91,110,156,144,235,63,43,167,219,130,82,35,129,191,43,167,219,130,82,35,129,191,248,183,91,110,156,144,235,63,109,218,190,110,195,207,194,63,15,110,236,162,9,161,193,63,201,83,38,220,32,189,234,63,119,22,210,99,151,83,155,63,119,22,210,99,151,83,155,63,201,83,38,220,32,189,234,63,15,110,236,162,9,161,193,63,12,163,252,8,125,144,174,63,60,232,194,3,144,192,237,63,0,103,83,235,10,154,133,63,0,103,83,235,10,154,133,63,60,232,194,3,144,192,237,63,12,163,252,8,125,144,174,63,79,85,85,85,85,85,213,63,88,85,85,85,85,85,229,63,97,85,85,85,85,85,197,63,97,85,85,85,85,85,197,63,79,85,85,85,85,85,213,63,51,51,51,51,51,51,227,63,154,153,153,153,153,153,201,63,154,153,153,153,153,153,201,63,32,121,161,175,163,172,187,63,220,208,11,138,107,138,220,63,220,208,11,138,107,138,220,63,210,147,120,133,157,35,234,63,219,176,29,234,137,113,183,63,219,176,29,234,137,113,183,63,79,85,85,85,85,85,213,63,16,68,199,40,20,147,174,63,191,139,115,189,206,22,222,63,191,139,115,189,206,22,222,63,153,66,225,152,133,132,233,63,119,245,122,156,233,237,185,63,119,245,122,156,233,237,185,63,175,137,138,155,175,11,224,63,145,236,234,200,160,232,207,63,145,236,234,200,160,232,207,63,143,106,129,126,89,246,235,63,196,85,250,5,154,38,176,63,196,85,250,5,154,38,176,63,9,57,14,246,211,53,171,63,9,57,14,246,211,53,171,63,87,27,219,134,208,220,211,63,196,142,49,125,58,94,228,63,87,27,219,134,208,220,211,63,196,142,49,125,58,94,228,63,79,85,85,85,85,85,213,63,125,194,5,191,251,172,222,63,193,30,125,32,130,169,208,63,193,30,125,32,130,169,208,63,8,49,131,140,232,212,235,63,225,59,243,205,93,172,176,63,225,59,243,205,93,172,176,63,162,110,196,224,239,237,168,63,162,110,196,224,239,237,168,63,240,208,88,0,253,5,212,63,158,80,199,129,34,110,228,63,240,208,88,0,253,5,212,63,158,80,199,129,34,110,228,63,79,85,85,85,85,85,213,63,194,161,249,19,154,215,180,63,200,203,128,189,12,101,221,63,200,203,128,189,12,101,221,63,41,79,96,120,100,21,229,63,174,97,63,15,55,213,197,63,174,97,63,15,55,213,197,63,223,100,222,142,213,195,236,63,6,217,12,137,83,225,169,63,6,217,12,137,83,225,169,63,142,184,118,243,71,49,129,63,142,184,118,243,71,49,129,63,34,21,156,49,215,214,208,63,141,26,100,71,207,79,231,63,34,21,156,49,215,214,208,63,141,26,100,71,207,79,231,63,79,85,85,85,85,85,213,63,5,192,242,212,84,33,149,63,9,106,88,89,245,86,223,63,9,106,88,89,245,86,223,63,142,167,29,131,229,26,192,63,38,150,56,159,70,249,219,63,38,150,56,159,70,249,219,63,109,148,97,40,121,244,227,63,55,215,60,175,13,23,200,63,55,215,60,175,13,23,200,63,96,130,179,210,38,35,237,63,68,237,99,106,201,230,166,63,68,237,99,106,201,230,166,63,187,248,5,254,123,220,162,63,187,248,5,254,123,220,162,63,26,23,175,129,72,105,204,63,174,218,179,31,230,183,231,63,26,23,175,129,72,105,204,63,174,218,179,31,230,183,231,63,79,85,85,85,85,85,213,63,98,188,149,71,121,137,157,63,20,82,195,53,180,19,223,63,20,82,195,53,180,19,223,63,210,189,113,253,64,254,232,63,183,8,57,10,252,6,188,63,183,8,57,10,252,6,188,63,149,59,127,80,118,35,194,63,149,59,127,80,118,35,194,63,136,39,40,76,73,181,211,63,87,29,204,197,125,156,225,63,136,39,40,76,73,181,211,63,87,29,204,197,125,156,225,63,122,76,92,208,134,154,153,63,122,76,92,208,134,154,153,63,74,109,98,106,247,146,207,63,74,130,228,238,109,78,231,63],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+10240);allocate([74,109,98,106,247,146,207,63,74,130,228,238,109,78,231,63,162,190,248,144,34,138,131,63,162,190,248,144,34,138,131,63,26,39,93,146,4,26,177,63,25,120,112,227,150,142,237,63,26,39,93,146,4,26,177,63,25,120,112,227,150,142,237,63,71,110,206,16,138,184,177,191,224,230,12,161,136,27,225,63,224,230,12,161,136,27,225,63,78,91,167,211,37,221,201,63,44,41,22,139,182,136,217,63,44,41,22,139,182,136,217,63,190,116,172,120,248,252,226,63,114,22,167,14,15,6,202,63,114,22,167,14,15,6,202,63,53,232,191,253,141,92,232,63,9,95,0,9,200,141,190,63,9,95,0,9,200,141,190,63,64,210,148,145,187,237,237,63,1,110,89,115,35,146,160,63,1,110,89,115,35,146,160,63,80,21,9,238,242,176,169,63,80,21,9,238,242,176,169,63,160,19,87,105,223,210,214,63,228,228,115,28,129,251,226,63,160,19,87,105,223,210,214,63,228,228,115,28,129,251,226,63,103,65,19,166,203,134,149,63,103,65,19,166,203,134,149,63,47,223,3,213,89,243,197,63,41,110,142,45,243,214,233,63,47,223,3,213,89,243,197,63,41,110,142,45,243,214,233,63,204,7,170,44,123,33,152,63,194,175,154,38,244,62,223,63,194,175,154,38,244,62,223,63,188,2,175,199,113,220,190,63,159,31,10,199,113,36,220,63,159,31,10,199,113,36,220,63,77,201,4,101,250,72,221,63,89,155,125,205,130,91,209,63,89,155,125,205,130,91,209,63,181,139,201,220,202,213,231,63,151,232,108,70,106,84,192,63,151,232,108,70,106,84,192,63,108,212,61,141,188,162,238,63,181,184,34,44,55,212,149,63,181,184,34,44,55,212,149,63,204,55,213,184,38,135,189,63,204,55,213,184,38,135,189,63,56,215,23,69,73,165,209,63,107,109,89,134,118,124,227,63,56,215,23,69,73,165,209,63,107,109,89,134,118,124,227,63,14,136,245,135,238,98,151,63,14,136,245,135,238,98,151,63,200,81,120,2,61,1,210,63,219,42,132,10,74,68,230,63,200,81,120,2,61,1,210,63,219,42,132,10,74,68,230,63,32,168,254,229,6,90,154,63,32,168,254,229,6,90,154,63,1,164,235,128,175,194,189,63,63,150,178,216,217,116,235,63,1,164,235,128,175,194,189,63,63,150,178,216,217,116,235,63,79,85,85,85,85,85,213,63,109,66,214,187,90,72,132,63,255,166,16,149,222,174,223,63,255,166,16,149,222,174,223,63,57,247,57,138,95,4,176,63,25,193,184,14,116,255,221,63,25,193,184,14,116,255,221,63,181,190,241,254,237,225,197,63,92,144,67,128,132,135,218,63,92,144,67,128,132,135,218,63,99,4,23,117,132,81,225,63,40,247,209,21,247,92,205,63,40,247,209,21,247,92,205,63,28,79,209,225,68,173,232,63,142,195,186,120,236,74,189,63,142,195,186,120,236,74,189,63,218,164,97,123,125,105,238,63,96,178,229,73,40,104,153,63,96,178,229,73,40,104,153,63,143,106,202,42,87,72,184,63,143,106,202,42,87,72,184,63,230,135,108,238,239,51,209,63,187,110,112,35,253,92,228,63,230,135,108,238,239,51,209,63,187,110,112,35,253,92,228,63,66,36,15,85,2,137,146,63,66,36,15,85,2,137,146,63,245,141,248,149,180,171,210,63,227,63,91,162,221,21,230,63,245,141,248,149,180,171,210,63,227,63,91,162,221,21,230,63,217,150,209,132,68,196,150,63,217,150,209,132,68,196,150,63,117,187,37,147,122,44,192,63,108,4,16,55,191,62,235,63,117,187,37,147,122,44,192,63,108,4,16,55,191,62,235,63,38,180,143,224,22,154,150,63,104,130,251,72,47,75,223,63,104,130,251,72,47,75,223,63,64,237,21,85,60,21,197,63,176,132,186,234,176,186,218,63,176,132,186,234,176,186,218,63,245,126,220,55,176,254,220,63,142,192,17,228,167,128,209,63,142,192,17,228,167,128,209,63,251,138,103,32,170,168,228,63,248,233,48,191,171,174,198,63,248,233,48,191,171,174,198,63,218,152,206,128,120,11,236,63,124,57,139,249,59,164,175,63,124,57,139,249,59,164,175,63,42,83,230,105,76,194,238,63,99,205,154,97,57,219,147,63,99,205,154,97,57,219,147,63,100,99,174,201,116,63,173,63,100,99,174,201,116,63,173,63,4,5,49,183,213,12,198,63,137,216,152,69,211,168,232,63,4,5,49,183,213,12,198,63,137,216,152,69,211,168,232,63,51,83,65,252,91,201,183,63,51,83,65,252,91,201,183,63,115,235,70,94,35,143,213,63,224,95,84,209,66,63,226,63,115,235,70,94,35,143,213,63,224,95,84,209,66,63,226,63,207,42,205,91,56,255,141,63,207,42,205,91,56,255,141,63,162,118,178,146,138,24,211,63,4,144,55,213,189,251,229,63,162,118,178,146,138,24,211,63,4,144,55,213,189,251,229,63,64,227,35,248,195,199,84,63,64,227,35,248,195,199,84,63,248,156,170,216,28,117,190,63,111,154,238,130,248,38,236,63,248,156,170,216,28,117,190,63,111,154,238,130,248,38,236,63,160,116,210,29,162,143,140,191,233,164,59,68,31,57,224,63,233,164,59,68,31,57,224,63,59,93,17,105,90,143,193,63,186,168,187,101,41,156,219,63,186,168,187,101,41,156,219,63,105,136,137,222,136,116,220,63,194,59,187,144,187,197,209,63,194,59,187,144,187,197,209,63,216,173,56,205,255,231,231,63,79,164,142,101,0,48,192,63,79,164,142,101,0,48,192,63,8,50,125,26,224,119,235,63,226,55,11,150,127,32,178,63,226,55,11,150,127,32,178,63,13,76,131,79,70,201,238,63,152,62,203,7,155,107,147,63,152,62,203,7,155,107,147,63,223,150,0,118,51,30,193,63,223,150,0,118,51,30,193,63,101,167,71,84,83,185,208,63,141,6,92,120,201,91,227,63,101,167,71,84,83,185,208,63,141,6,92,120,201,91,227,63,78,250,42,52,167,158,162,63,78,250,42,52,167,158,162,63,180,40,126,27,194,213,216,63,1,60,254,126,52,107,226,63,180,40,126,27,194,213,216,63,1,60,254,126,52,107,226,63,192,105,39,178,145,214,132,191,192,105,39,178,145,214,132,191,251,164,75,232,27,73,210,63,41,203,162,82,204,46,231,63,251,164,75,232,27,73,210,63,41,203,162,82,204,46,231,63,103,150,89,32,51,221,162,63,103,150,89,32,51,221,162,63,179,9,28,17,197,152,203,63,45,100,179,137,251,235,231,63,179,9,28,17,197,152,203,63,45,100,179,137,251,235,231,63,159,51,188,76,135,132,137,63,159,51,188,76,135,132,137,63,22,206,219,125,238,131,186,63,120,149,17,19,112,73,236,63,22,206,219,125,238,131,186,63,120,149,17,19,112,73,236,63,79,85,85,85,85,85,213,63,9,58,69,213,102,117,117,63,131,117,85,50,21,213,223,63,131,117,85,50,21,213,223,63,254,248,136,235,221,38,198,63,183,193,29,133,72,118,218,63,183,193,29,133,72,118,218,63,224,138,138,219,25,64,174,63,73,87,71,98,254,27,222,63,73,87,71,98,254,27,222,63,7,217,188,113,196,154,224,63,242,77,134,28,119,202,206,63,242,77,134,28,119,202,206,63,227,202,56,122,186,135,230,63,58,106,142,11,139,240,194,63,58,106,142,11,139,240,194,63,206,147,193,15,148,43,235,63,199,176,249,192,175,81,179,63,199,176,249,192,175,81,179,63,216,46,96,164,21,240,238,63,135,18,253,185,165,254,144,63,135,18,253,185,165,254,144,63,97,154,218,194,239,131,186,63,97,154,218,194,239,131,186,63,153,104,48,83,196,250,210,63,103,120,12,222,31,50,227,63,153,104,48,83,196,250,210,63,103,120,12,222,31,50,227,63,242,42,2,241,190,144,148,63,242,42,2,241,190,144,148,63,68,244,26,50,65,157,213,63,134,116,106,111,217,140,228,63,68,244,26,50,65,157,213,63,134,116,106,111,217,140,228,63,192,58,39,4,223,199,113,191,192,58,39,4,223,199,113,191,172,182,200,18,49,53,202,63,203,32,86,121,67,150,233,63,172,182,200,18,49,53,202,63,203,32,86,121,67,150,233,63,216,140,110,210,100,121,165,63,216,140,110,210,100,121,165,63,6,224,117,41,230,60,200,63,58,159,123,40,48,153,232,63,6,224,117,41,230,60,200,63,58,159,123,40,48,153,232,63,130,207,65,49,102,82,141,63,130,207,65,49,102,82,141,63,106,220,61,164,37,213,181,63,44,61,179,178,17,208,236,63,106,220,61,164,37,213,181,63,44,61,179,178,17,208,236,63,79,85,85,85,85,85,213,63,23,185,196,120,206,45,119,63,142,118,14,99,164,209,223,63,142,118,14,99,164,209,223,63,67,208,175,189,94,64,162,63,5,3,37,20,250,219,222,63,5,3,37,20,250,219,222,63,25,74,104,147,37,122,185,63,198,246,146,77,187,208,220,63,198,246,146,77,187,208,220,63,82,97,46,216,38,136,201,63,172,103,244,73,246,157,217,63,172,103,244,73,246,157,217,63,112,224,11,190,213,185,223,63,209,15,250,32,21,35,208,63,209,15,250,32,21,35,208,63,255,239,164,153,5,161,229,63,240,31,182,204,244,189,196,63,240,31,182,204,244,189,196,63,3,163,200,123,217,36,235,63,245,115,221,16,154,108,179,63,245,115,221,16,154,108,179,63,40,78,233,80,131,255,238,63,129,29,107,241,202,7,144,63,129,29,107,241,202,7,144,63,22,49,68,113,226,220,132,63,22,49,68,113,226,220,132,63,207,183,46,40,127,101,213,63,84,147,35,226,204,249,228,63,207,183,46,40,127,101,213,63,84,147,35,226,204,249,228,63,231,53,47,97,32,86,193,63,231,53,47,97,32,86,193,63,185,148,1,247,193,179,210,63,42,104,51,236,150,80,226,63,185,148,1,247,193,179,210,63,42,104,51,236,150,80,226,63,182,100,229,221,115,221,171,63,182,100,229,221,115,221,171,63,190,254,96,56,234,115,212,63,95,42,241,165,51,8,228,63,190,254,96,56,234,115,212,63,95,42,241,165,51,8,228,63,20,157,106,255,212,90,138,63,20,157,106,255,212,90,138,63,142,182,14,251,254,104,200,63,232,167,62,237,84,124,233,63,142,182,14,251,254,104,200,63,232,167,62,237,84,124,233,63,168,87,224,214,198,49,177,63,168,87,224,214,198,49,177,63,151,19,193,233,18,26,199,63,28,176,179,106,66,19,232,63,151,19,193,233,18,26,199,63,28,176,179,106,66,19,232,63,136,44,227,0,187,7,142,63,136,44,227,0,187,7,142,63,186,193,171,36,127,169,180,63,23,251,102,47,177,242,236,63,186,193,171,36,127,169,180,63,23,251,102,47,177,242,236,63,79,85,85,85,85,85,213,63,70,5,118,84,121,66,139,63,226,39,174,26,246,146,223,63,226,39,174,26,246,146,223,63,93,32,45,11,66,135,175,63,250,45,77,223,139,7,222,63,250,45,77,223,139,7,222,63,163,228,207,200,220,79,192,63,224,6,204,205,8,236,219,63,224,6,204,205,8,236,219,63,135,192,1,130,92,235,202,63,222,143,127,223,40,69,217,63,222,143,127,223,40,69,217,63,194,104,17,164,93,3,224,63,125,46,221,183,68,249,207,63,125,46,221,183,68,249,207,63,117,80,214,77,24,171,229,63,40,95,83,100,207,169,196,63,40,95,83,100,207,169,196,63,98,51,129,191,3,25,235,63,85,50,251,1,241,155,179,63,85,50,251,1,241,155,179,63,196,60,136,229,165,114,238,63,190,51,124,167,161,213,152,63,190,51,124,167,161,213,152,63,224,41,247,4,23,61,237,63,69,177,70,216,71,23,166,63,69,177,70,216,71,23,166,63,24,148,55,106,7,68,129,63,24,148,55,106,7,68,129,63,60,130,59,235,103,248,214,63,146,96,185,236,187,62,228,63,60,130,59,235,103,248,214,63,146,96,185,236,187,62,228,63,33,112,177,142,184,202,192,63,33,112,177,142,184,202,192,63,35,181,96,124,125,215,210,63,102,73,35,30,147,97,226,63,35,181,96,124,125,215,210,63,102,73,35,30,147,97,226,63,207,235,66,60,58,180,169,63,207,235,66,60,58,180,169,63,141,215,42,119,23,205,212,63,133,101,166,160,48,254,227,63,141,215,42,119,23,205,212,63,133,101,166,160,48,254,227,63,138,236,213,98,244,250,176,63,138,236,213,98,244,250,176,63,220,227,141,248,122,165,199,63,111,201,129,181,66,247,231,63,220,227,141,248,122,165,199,63,111,201,129,181,66,247,231,63,54,30,181,250,117,145,136,63,54,30,181,250,117,145,136,63,203,105,67,150,136,1,204,63,20,81,132,2,88,157,232,63,203,105,67,150,136,1,204,63,20,81,132,2,88,157,232,63,38,16,236,128,236,109,142,63,38,16,236,128,236,109,142,63,0,44,185,244,231,230,185,63,63,42,101,79,107,73,236,63,0,44,185,244,231,230,185,63,63,42,101,79,107,73,236,63,161,155,133,183,158,8,162,191,161,155,133,183,158,8,162,191,39,186,0,29,49,96,149,63,244,41,72,49,196,58,240,63,39,186,0,29,49,96,149,63,244,41,72,49,196,58,240,63,79,85,85,85,85,85,213,63,72,199,207,237,91,71,149,63,189,129,145,32,197,85,223,63,189,129,145,32,197,85,223,63,135,253,18,190,240,70,183,63,88,160,61,232,33,23,221,63,88,160,61,232,33,23,221,63,146,180,138,164,193,60,201,63,219,82,221,150,207,176,217,63,219,82,221,150,207,176,217,63,67,51,243,88,21,74,223,63,104,102,134,83,245,90,208,63,104,102,134,83,245,90,208,63,157,69,103,71,193,170,228,63,215,116,49,113,125,170,198,63,215,116,49,113,125,170,198,63,252,165,244,123,194,244,232,63,15,104,45,16,246,44,188,63,15,104,45,16,246,44,188,63,173,44,182,17,56,114,236,63,221,154,78,114,63,110,172,63,221,154,78,114,63,110,172,63,226,46,247,16,52,49,239,63,216,36,26,225,125,217,137,63,216,36,26,225,125,217,137,63,226,198,162,49,177,149,109,63,226,198,162,49,177,149,109,63,138,190,59,227,11,84,217,63,244,125,48,93,100,56,227,63,138,190,59,227,11,84,217,63,244,125,48,93,100,56,227,63,102,242,141,228,52,54,193,63,102,242,141,228,52,54,193,63,154,90,142,246,31,181,211,63,26,86,149,203,226,215,225,63,154,90,142,246,31,181,211,63,26,86,149,203,226,215,225,63,12,236,31,194,224,149,141,63,12,236,31,194,224,149,141,63,163,139,192,53,170,238,208,63,126,58,23,98,83,18,231,63,163,139,192,53,170,238,208,63,126,58,23,98,83,18,231,63,200,111,99,148,173,7,168,63,200,111,99,148,173,7,168,63,240,109,3,10,79,242,214,63,12,18,184,161,93,6,227,63,240,109,3,10,79,242,214,63,12,18,184,161,93,6,227,63,94,71,189,74,52,112,103,63,94,71,189,74,52,112,103,63,168,209,217,119,8,51,196,63,88,206,190,173,205,219,234,63,168,209,217,119,8,51,196,63,88,206,190,173,205,219,234,63,153,19,171,187,125,167,204,63,153,19,171,187,125,167,204,63,97,139,142,19,132,54,179,63,174,105,163,14,80,111,230,63,97,139,142,19,132,54,179,63,174,105,163,14,80,111,230,63,193,231,122,234,66,189,161,63,193,231,122,234,66,189,161,63,233,81,131,251,222,58,194,63,10,125,119,18,116,85,234,63,233,81,131,251,222,58,194,63,10,125,119,18,116,85,234,63,97,208,139,88,90,207,132,63,97,208,139,88,90,207,132,63,13,14,174,133,65,196,176,63,253,14,232,101,58,148,237,63,13,14,174,133,65,196,176,63,253,14,232,101,58,148,237,63,79,85,85,85,85,85,213,63,225,230,131,167,18,37,95,191,250,224,169,68,201,7,224,63,250,224,169,68,201,7,224,63,139,206,126,1,206,35,152,63,149,9,244,143,225,62,223,63,149,9,244,143,225,62,223,63,249,141,160,39,83,248,182,63,74,238,11,155,245,32,221,63,74,238,11,155,245,32,221,63,162,21,29,237,197,22,201,63,142,186,184,132,78,186,217,63,142,186,184,132,78,186,217,63,107,10,167,176,230,62,223,63,193,122,172,167,140,96,208,63,193,122,172,167,140,96,208,63,80,222,23,154,106,180,228,63,97,67,208,203,42,151,198,63,97,67,208,203,42,151,198,63,50,119,200,195,67,85,233,63,19,35,222,240,240,170,186,63,19,35,222,240,240,170,186,63,229,183,247,161,132,154,236,63,214,64,66,240,218,43,171,63,214,64,66,240,218,43,171,63,148,55,213,116,30,86,237,63,96,67,86,89,12,79,165,63,96,67,86,89,12,79,165,63,136,77,149,230,61,66,239,63,247,78,86,45,67,184,135,63,247,78,86,45,67,184,135,63,25,106,151,39,168,244,168,63,25,106,151,39,168,244,168,63,217,27,190,239,29,18,214,63,114,123,167,133,166,103,227,63,217,27,190,239,29,18,214,63,114,123,167,133,166,103,227,63,149,223,9,152,212,220,121,63,149,223,9,152,212,220,121,63,105,28,123,137,149,46,216,63,12,94,18,146,251,180,227,63,105,28,123,137,149,46,216,63,12,94,18,146,251,180,227,63,144,28,136,163,72,49,193,63,144,28,136,163,72,49,193,63,114,110,151,106,234,159,211,63,163,65,210,161,184,227,225,63,114,110,151,106,234,159,211,63,163,65,210,161,184,227,225,63,184,171,135,76,88,158,140,63,184,171,135,76,88,158,140,63,149,167,13,67,249,236,207,63,108,119,10,78,72,146,231,63,149,167,13,67,249,236,207,63,108,119,10,78,72,146,231,63,20,174,22,31,48,87,179,63,20,174,22,31,48,87,179,63,210,146,180,37,60,60,203,63,137,5,176,242,10,198,230,63,210,146,180,37,60,60,203,63,137,5,176,242,10,198,230,63,43,167,219,130,82,35,129,191,43,167,219,130,82,35,129,191,109,218,190,110,195,207,194,63,248,183,91,110,156,144,235,63,109,218,190,110,195,207,194,63,248,183,91,110,156,144,235,63,119,22,210,99,151,83,155,63,119,22,210,99,151,83,155,63,15,110,236,162,9,161,193,63,201,83,38,220,32,189,234,63,15,110,236,162,9,161,193,63,201,83,38,220,32,189,234,63,0,103,83,235,10,154,133,63,0,103,83,235,10,154,133,63,12,163,252,8,125,144,174,63,60,232,194,3,144,192,237,63,12,163,252,8,125,144,174,63,60,232,194,3,144,192,237,63,0,0,0,0,0,0,0,64,0,0,0,0,0,0,240,63,0,0,0,0,0,0,240,63,118,28,199,113,28,199,225,63,29,199,113,28,199,113,236,63,118,28,199,113,28,199,225,63,110,182,231,247,64,67,214,63,201,36,12,132,95,222,228,63,201,36,12,132,95,222,228,63,110,182,231,247,64,67,214,63,223,188,154,120,86,52,226,63,92,65,174,37,218,161,222,63,92,65,174,37,218,161,222,63,141,3,110,195,158,83,206,63,141,3,110,195,158,83,206,63,28,76,121,181,183,22,215,63,28,76,121,181,183,22,215,63,231,69,149,73,77,242,221,63,231,69,149,73,77,242,221,63,250,219,226,1,246,237,197,63,250,219,226,1,246,237,197,63,166,47,60,224,215,191,218,63,61,43,227,78,231,111,216,63,61,43,227,78,231,111,216,63,67,134,61,113,177,230,209,63,67,134,61,113,177,230,209,63,87,109,130,159,246,146,192,63,87,109,130,159,246,146,192,63,0,0,0,0,0,0,0,0,30,51,144,69,167,121,226,191,30,51,144,69,167,121,226,63,68,114,63,244,126,201,232,191,0,0,0,0,0,0,0,0,68,114,63,244,126,201,232,63,137,57,246,188,109,142,235,191,247,61,221,217,63,194,213,191,247,61,221,217,63,194,213,63,137,57,246,188,109,142,235,63,0,0,0,0,0,0,0,0,5,183,153,253,35,59,225,191,5,183,153,253,35,59,225,63,105,58,83,224,108,255,236,191,105,58,83,224,108,255,236,63,94,201,85,150,160,40,229,191,94,201,85,150,160,40,229,63,197,117,54,208,18,139,206,191,197,117,54,208,18,139,206,63,30,160,128,78,202,214,237,191,30,160,128,78,202,214,237,63,0,0,0,0,0,0,0,0,98,253,25,241,93,249,217,191,98,253,25,241,93,249,217,63,214,161,227,155,159,186,231,191,214,161,227,155,159,186,231,63,41,98,124,142,23,95,238,191,41,98,124,142,23,95,238,63,220,163,0,0,229,33,2,0,168,0,0,0,0,0,0,0,220,163,0,0,180,40,2,0,120,98,0,0,0,0,0,0,180,163,0,0,193,40,2,0,220,163,0,0,206,40,2,0,120,98,0,0,0,0,0,0,180,163,0,0,224,40,2,0,220,163,0,0,237,40,2,0,120,98,0,0,0,0,0,0,220,163,0,0,249,40,2,0,144,98,0,0,0,0,0,0,220,163,0,0,26,41,2,0,168,98,0,0,0,0,0,0,220,163,0,0,96,41,2,0,168,98,0,0,0,0,0,0,220,163,0,0,60,41,2,0,200,98,0,0,0,0,0,0,220,163,0,0,130,41,2,0,184,98,0,0,0,0,0,0,220,163,0,0,167,41,2,0,184,98,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,163,0,0,62,63,2,0,144,99,0,0,0,0,0,0,220,163,0,0,125,63,2,0,144,99,0,0,0,0,0,0,220,163,0,0,149,63,2,0,136,99,0,0,0,0,0,0,220,163,0,0,174,63,2,0,136,99,0,0,0,0,0,0,220,163,0,0,198,63,2,0,8,107,0,0,0,0,0,0,180,163,0,0,224,63,2,0,220,163,0,0,242,63,2,0,96,99,0,0,0,0,0,0,220,163,0,0,28,64,2,0,96,99,0,0,0,0,0,0,180,163,0,0,70,64,2,0,180,163,0,0,119,64,2,0,4,164,0,0,168,64,2,0,0,0,0,0,1,0,0,0,104,99,0,0,3,244,255,255,4,164,0,0,215,64,2,0,0,0,0,0,1,0,0,0,120,99,0,0,3,244,255,255,4,164,0,0,6,65,2,0,0,0,0,0,1,0,0,0,104,99,0,0,3,244,255,255,4,164,0,0,53,65,2,0,0,0,0,0,1,0,0,0,120,99,0,0,3,244,255,255,220,163,0,0,100,65,2,0,32,107,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,163,0,0,224,65,2,0,0,107,0,0,0,0,0,0,4,164,0,0,246,65,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,48,106,0,0,2,0,0,0,4,164,0,0,8,66,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,56,106,0,0,2,0,0,0,4,164,0,0,42,66,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,56,106,0,0,2,0,0,0,4,164,0,0,77,66,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,56,106,0,0,2,0,0,0,220,163,0,0,112,66,2,0,128,100,0,0,0,0,0,0,220,163,0,0,146,66,2,0,128,100,0,0,0,0,0,0,4,164,0,0,181,66,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,56,106,0,0,2,0,0,0,220,163,0,0,215,66,2,0,16,100,0,0,0,0,0,0,220,163,0,0,237,66,2,0,16,100,0,0,0,0,0,0,220,163,0,0,1,67,2,0,16,100,0,0,0,0,0,0,4,164,0,0,21,67,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,48,106,0,0,2,0,0,0,220,163,0,0,39,67,2,0,16,100,0,0,0,0,0,0,220,163,0,0,60,67,2,0,16,100,0,0,0,0,0,0,4,164,0,0,81,67,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,64,106,0,0,0,0,0,0,4,164,0,0,149,67,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,88,106,0,0,0,0,0,0,4,164,0,0,217,67,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,112,106,0,0,0,0,0,0,4,164,0,0,29,68,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,136,106,0,0,0,0,0,0,4,164,0,0,97,68,2,0,0,0,0,0,3,0,0,0,16,100,0,0,2,0,0,0,160,106,0,0,2,0,0,0,168,106,0,0,0,8,0,0,4,164,0,0,166,68,2,0,0,0,0,0,3,0,0,0,16,100,0,0,2,0,0,0,160,106,0,0,2,0,0,0,176,106,0,0,0,8,0,0,4,164,0,0,235,68,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,184,106,0,0,0,8,0,0,4,164,0,0,48,69,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,184,106,0,0,0,8,0,0,4,164,0,0,117,69,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,192,106,0,0,2,0,0,0,4,164,0,0,145,69,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,192,106,0,0,2,0,0,0,4,164,0,0,173,69,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,192,106,0,0,2,0,0,0,4,164,0,0,201,69,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,192,106,0,0,2,0,0,0,4,164,0,0,229,69,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,200,106,0,0,0,0,0,0,4,164,0,0,43,70,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,208,106,0,0,0,0,0,0,4,164,0,0,113,70,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,216,106,0,0,0,0,0,0,4,164,0,0,183,70,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,224,106,0,0,0,0,0,0,4,164,0,0,253,70,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,232,106,0,0,2,0,0,0,4,164,0,0,18,71,2,0,0,0,0,0,2,0,0,0,16,100,0,0,2,0,0,0,232,106,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,180,163,0,0,110,74,2,0,180,163,0,0,87,74,2,0,4,164,0,0,65,74,2,0,0,0,0,0,1,0,0,0,248,106,0,0,0,0,0,0,4,164,0,0,18,74,2,0,0,0,0,0,1,0,0,0,248,106,0,0,0,0,0,0,4,164,0,0,252,73,2,0,0,0,0,0,1,0,0,0,240,106,0,0,0,0,0,0,4,164,0,0,205,73,2,0,0,0,0,0,1,0,0,0,240,106,0,0,0,0,0,0,180,163,0,0,186,73,2,0,180,163,0,0,152,73,2,0,180,163,0,0,118,73,2,0,180,163,0,0,97,73,2,0,180,163,0,0,76,73,2,0,180,163,0,0,51,73,2,0,180,163,0,0,26,73,2,0,180,163,0,0,1,73,2,0,180,163,0,0,232,72,2,0,180,163,0,0,208,72,2,0,180,163,0,0,227,73,2,0,180,163,0,0,40,74,2,0,180,163,0,0,131,74,2,0,220,163,0,0,156,74,2,0,128,98,0,0,0,0,0,0,180,163,0,0,179,74,2,0,220,163,0,0,204,74,2,0,24,107,0,0,0,0,0,0,0,0,0,0,0,0,0,0,56,0,0,0,0,0,0,0,8,0,0,0,1,0,0,0,2,0,0,0,200,255,255,255,200,255,255,255,8,0,0,0,3,0,0,0,4,0,0,0,68,107,0,0,124,107,0,0,144,107,0,0,88,107,0,0,56,0,0,0,0,0,0,0,200,99,0,0,5,0,0,0,6,0,0,0,200,255,255,255,200,255,255,255,200,99,0,0,7,0,0,0,8,0,0,0,0,0,0,0,24,0,0,0,9,0,0,0,10,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,1,0,0,0,3,0,0,0,2,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,176,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,2,0,0,0,0,0,0,0,136,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,3,0,0,0,2,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,0,0,0,0,72,1,0,0,19,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,4,0,0,0,18,0,0,0,3,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,5,0,0,0,1,0,0,0,6,0,0,0,2,0,0,0,3,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,4,0,0,0,5,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,2,0,0,0,0,0,0,0,56,1,0,0,23,0,0,0,24,0,0,0,25,0,0,0,26,0,0,0,19,0,0,0,20,0,0,0,21,0,0,0,9,0,0,0,7,0,0,0,22,0,0,0,23,0,0,0,1,0,0,0,24,0,0,0,0,0,0,0,16,1,0,0,27,0,0,0,28,0,0,0,29,0,0,0,30,0,0,0,25,0,0,0,26,0,0,0,27,0,0,0,10,0,0,0,8,0,0,0,28,0,0,0,29,0,0,0,2,0,0,0,30,0,0,0,0,0,0,0,240,0,0,0,31,0,0,0,32,0,0,0,33,0,0,0,34,0,0,0,31,0,0,0,32,0,0,0,33,0,0,0,11,0,0,0,9,0,0,0,34,0,0,0,35,0,0,0,3,0,0,0,36,0,0,0,0,0,0,0,224,0,0,0,19,0,0,0,35,0,0,0,36,0,0,0,37,0,0,0,37,0,0,0,38,0,0,0,39,0,0,0,12,0,0,0,40,0,0,0,17,0,0,0,18,0,0,0,10,0,0,0,13,0,0,0,19,0,0,0,2,0,0,0,20,0,0,0,11,0,0,0,12,0,0,0,14,0,0,0,3,0,0,0,21,0,0,0,22,0,0,0,23,0,0,0,24,0,0,0,25,0,0,0,26,0,0,0,2,0,0,0,27,0,0,0,13,0,0,0,14,0,0,0,15,0,0,0,15,0,0,0,16,0,0,0,28,0,0,0,29,0,0,0,30,0,0,0,4,0,0,0,0,0,0,0,192,0,0,0,19,0,0,0,38,0,0,0,39,0,0,0,40,0,0,0,41,0,0,0,42,0,0,0,43,0,0,0,17,0,0,0,44,0,0,0,31,0,0,0,32,0,0,0,16,0,0,0,18,0,0,0,33,0,0,0,3,0,0,0,34,0,0,0,17,0,0,0,18,0,0,0,19,0,0,0,5,0,0,0,35,0,0,0,36,0,0,0,37,0,0,0,38,0,0,0,39,0,0,0,40,0,0,0,3,0,0,0,41,0,0,0,19,0,0,0,20,0,0,0,20,0,0,0,21,0,0,0,21,0,0,0,42,0,0,0,43,0,0,0,44,0,0,0,6,0,0,0,0,0,0,0,32,1,0,0,41,0,0,0,42,0,0,0,43,0,0,0,44,0,0,0,7,0,0,0,1,0,0,0,22,0,0,0,45,0,0,0,46,0,0,0,45,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,46,0,0,0,5,0,0,0,47,0,0,0,1,0,0,0,6,0,0,0,2,0,0,0,23,0,0,0,0,0,0,0,88,1,0,0,45,0,0,0,46,0,0,0,47,0,0,0,48,0,0,0,47,0,0,0,48,0,0,0,49,0,0,0,22,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,50,0,0,0,24,0,0,0,1,0,0,0,23,0,0,0,24,0,0,0,51,0,0,0,25,0,0,0,25,0,0,0,1,0,0,0,52,0,0,0,53,0,0,0,26,0,0,0,7,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,8,0,0,0,9,0,0,0,6,0,0,0,1,0,0,0,54,0,0,0,55,0,0,0,27,0,0,0,1,0,0,0,4,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,51,0,0,0,52,0,0,0,53,0,0,0,26,0,0,0,27,0,0,0,0,0,0,0,120,1,0,0,49,0,0,0,50,0,0,0,51,0,0,0,52,0,0,0,54,0,0,0,55,0,0,0,56,0,0,0,28,0,0,0,53,0,0,0,28,0,0,0,6,0,0,0,8,0,0,0,0,0,0,0,152,1,0,0,54,0,0,0,55,0,0,0,56,0,0,0,57,0,0,0,3,0,0,0,29,0,0,0,30,0,0,0,31,0,0,0,57,0,0,0,58,0,0,0,1,0,0,0,56,0,0,0,5,0,0,0,0,0,0,0,176,1,0,0,58,0,0,0,59,0,0,0,60,0,0,0,61,0,0,0,59,0,0,0,60,0,0,0,61,0,0,0,29,0,0,0,0,0,0,0,192,1,0,0,62,0,0,0,63,0,0,0,64,0,0,0,65,0,0,0,62,0,0,0,63,0,0,0,64,0,0,0,30,0,0,0,0,0,0,0,208,1,0,0,66,0,0,0,67,0,0,0,68,0,0,0,69,0,0,0,65,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,32,0,0,0,33,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,70,0,0,0,57,0,0,0,34,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,252,255,255,255,208,1,0,0,1,0,0,0,2,0,0,0,35,0,0,0,36,0,0,0,1,0,0,0,0,0,0,0,248,1,0,0,71,0,0,0,72,0,0,0,68,0,0,0,69,0,0,0,65,0,0,0,66,0,0,0,67,0,0,0,31,0,0,0,1,0,0,0,32,0,0,0,33,0,0,0,32,0,0,0,33,0,0,0,9,0,0,0,73,0,0,0,70,0,0,0,57,0,0,0,34,0,0,0,9,0,0,0,58,0,0,0,74,0,0,0,75,0,0,0,76,0,0,0,2,0,0,0,34,0,0,0,35,0,0,0,37,0,0,0,36,0,0,0,68,0,0,0,10,0,0,0,38,0,0,0,69,0,0,0,70,0,0,0,10,0,0,0,4,0,0,0,37,0,0,0,38,0,0,0,2,0,0,0,2,0,0,0,7,0,0,0,8,0,0,0,71,0,0,0,9,0,0,0,72,0,0,0,39,0,0,0,73,0,0,0,3,0,0,0,59,0,0,0,60,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,3,0,0,0,40,0,0,0,41,0,0,0,7,0,0,0,42,0,0,0,43,0,0,0,74,0,0,0,75,0,0,0,10,0,0,0,76,0,0,0,77,0,0,0,11,0,0,0,39,0,0,0,40,0,0,0,41,0,0,0,42,0,0,0,43,0,0,0,5,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,1,0,0,0,8,0,0,0,78,0,0,0,12,0,0,0,2,0,0,0,1,0,0,0,1,0,0,0,13,0,0,0,3,0,0,0,14,0,0,0,44,0,0,0,44,0,0,0,45,0,0,0,46,0,0,0,45,0,0,0,46,0,0,0,47,0,0,0,47,0,0,0,48,0,0,0,49,0,0,0,4,0,0,0,50,0,0,0,51,0,0,0,52,0,0,0,79,0,0,0,80,0,0,0,61,0,0,0,81,0,0,0,53,0,0,0,77,0,0,0,78,0,0,0,48,0,0,0,4,0,0,0,62,0,0,0,82,0,0,0,83,0,0,0,1,0,0,0,79,0,0,0,80,0,0,0,81,0,0,0,15,0,0,0,84,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,19,0,0,0,5,0,0,0,82,0,0,0,83,0,0,0,5,0,0,0,11,0,0,0,49,0,0,0,85,0,0,0,15,0,0,0,54,0,0,0,3,0,0,0,50,0,0,0,12,0,0,0,1,0,0,0,2,0,0,0,9,0,0,0,20,0,0,0,21,0,0,0,55,0,0,0,51,0,0,0,252,255,255,255,248,1,0,0,52,0,0,0,2,0,0,0,35,0,0,0,36,0,0,0,56,0,0,0,0,0,0,0,48,2,0,0,84,0,0,0,85,0,0,0,68,0,0,0,69,0,0,0,65,0,0,0,86,0,0,0,87,0,0,0,53,0,0,0,1,0,0,0,32,0,0,0,33,0,0,0,54,0,0,0,55,0,0,0,13,0,0,0,86,0,0,0,87,0,0,0,63,0,0,0,57,0,0,0,22,0,0,0,64,0,0,0,88,0,0,0,89,0,0,0,90,0,0,0,4,0,0,0,56,0,0,0,57,0,0,0,58,0,0,0,58,0,0,0,88,0,0,0,23,0,0,0,59,0,0,0,89,0,0,0,90,0,0,0,14,0,0,0,10,0,0,0,59,0,0,0,60,0,0,0,3,0,0,0,6,0,0,0,16,0,0,0,17,0,0,0,91,0,0,0,18,0,0,0,92,0,0,0,60,0,0,0,93,0,0,0,7,0,0,0,65,0,0,0,66,0,0,0,24,0,0,0,25,0,0,0,26,0,0,0,27,0,0,0,4,0,0,0,61,0,0,0,62,0,0,0,8,0,0,0,63,0,0,0,64,0,0,0,94,0,0,0,95,0,0,0,19,0,0,0,96,0,0,0,97,0,0,0,20,0,0,0,61,0,0,0,62,0,0,0,63,0,0,0,64,0,0,0,65,0,0,0,11,0,0,0,12,0,0,0,2,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,98,0,0,0,21,0,0,0,6,0,0,0,2,0,0,0,2,0,0,0,22,0,0,0,7,0,0,0,23,0,0,0,65,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,66,0,0,0,67,0,0,0,69,0,0,0,68,0,0,0,69,0,0,0,70,0,0,0,8,0,0,0,71,0,0,0,72,0,0,0,73,0,0,0,99,0,0,0,100,0,0,0,67,0,0,0,101,0,0,0,74,0,0,0,91,0,0,0,92,0,0,0,70,0,0,0,8,0,0,0,68,0,0,0,102,0,0,0,103,0,0,0,2,0,0,0,93,0,0,0,94,0,0,0,95,0,0,0,28,0,0,0,104,0,0,0,29,0,0,0,30,0,0,0,31,0,0,0,32,0,0,0,9,0,0,0,96,0,0,0,97,0,0,0,9,0,0,0,15,0,0,0,71,0,0,0,105,0,0,0,24,0,0,0,75,0,0,0,5,0,0,0,72,0,0,0,16,0,0,0,3,0,0,0,4,0,0,0,15,0,0,0,33,0,0,0,34,0,0,0,76,0,0,0,73,0,0,0,252,255,255,255,48,2,0,0,74,0,0,0,2,0,0,0,35,0,0,0,36,0,0,0,77,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,96,2,0,0,98,0,0,0,99,0,0,0,68,0,0,0,69,0,0,0,65,0,0,0,106,0,0,0,107,0,0,0,75,0,0,0,1,0,0,0,32,0,0,0,33,0,0,0,76,0,0,0,77,0,0,0,17,0,0,0,100,0,0,0,70,0,0,0,57],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+20480);allocate([34,0,0,0,35,0,0,0,69,0,0,0,101,0,0,0,102,0,0,0,103,0,0,0,6,0,0,0,78,0,0,0,79,0,0,0,78,0,0,0,80,0,0,0,108,0,0,0,36,0,0,0,79,0,0,0,109,0,0,0,110,0,0,0,18,0,0,0,16,0,0,0,81,0,0,0,82,0,0,0,4,0,0,0,10,0,0,0,25,0,0,0,26,0,0,0,111,0,0,0,27,0,0,0,112,0,0,0,80,0,0,0,113,0,0,0,11,0,0,0,70,0,0,0,71,0,0,0,37,0,0,0,38,0,0,0,39,0,0,0,40,0,0,0,5,0,0,0,81,0,0,0,82,0,0,0,9,0,0,0,83,0,0,0,84,0,0,0,114,0,0,0,115,0,0,0,28,0,0,0,116,0,0,0,117,0,0,0,29,0,0,0,83,0,0,0,84,0,0,0,85,0,0,0,86,0,0,0,87,0,0,0,17,0,0,0,18,0,0,0,3,0,0,0,19,0,0,0,3,0,0,0,20,0,0,0,118,0,0,0,30,0,0,0,10,0,0,0,3,0,0,0,3,0,0,0,31,0,0,0,11,0,0,0,32,0,0,0,85,0,0,0,88,0,0,0,89,0,0,0,90,0,0,0,86,0,0,0,87,0,0,0,91,0,0,0,88,0,0,0,89,0,0,0,90,0,0,0,12,0,0,0,91,0,0,0,92,0,0,0,93,0,0,0,119,0,0,0,120,0,0,0,72,0,0,0,121,0,0,0,94,0,0,0,104,0,0,0,105,0,0,0,92,0,0,0,12,0,0,0,73,0,0,0,122,0,0,0,123,0,0,0,3,0,0,0,106,0,0,0,107,0,0,0,108,0,0,0,41,0,0,0,124,0,0,0,42,0,0,0,43,0,0,0,44,0,0,0,45,0,0,0,13,0,0,0,109,0,0,0,110,0,0,0,13,0,0,0,19,0,0,0,93,0,0,0,125,0,0,0,33,0,0,0,95,0,0,0,7,0,0,0,94,0,0,0,20,0,0,0,5,0,0,0,6,0,0,0,21,0,0,0,46,0,0,0,47,0,0,0,96,0,0,0,95,0,0,0,252,255,255,255,96,2,0,0,96,0,0,0,2,0,0,0,35,0,0,0,36,0,0,0,97,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,0,0,0,0,3,0,0,0,1,0,0,0,1,0,0,0,3,0,0,0,2,0,0,0,0,0,0,0,2,0,0,0,3,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,0,0,0,0,144,2,0,0,111,0,0,0,112,0,0,0,68,0,0,0,69,0,0,0,65,0,0,0,126,0,0,0,127,0,0,0,97,0,0,0,1,0,0,0,32,0,0,0,33,0,0,0,98,0,0,0,99,0,0,0,21,0,0,0,113,0,0,0,70,0,0,0,57,0,0,0,34,0,0,0,48,0,0,0,74,0,0,0,114,0,0,0,115,0,0,0,116,0,0,0,8,0,0,0,100,0,0,0,101,0,0,0,98,0,0,0,102,0,0,0,128,0,0,0,49,0,0,0,99,0,0,0,129,0,0,0,130,0,0,0,22,0,0,0,22,0,0,0,103,0,0,0,104,0,0,0,5,0,0,0,14,0,0,0,34,0,0,0,35,0,0,0,131,0,0,0,36,0,0,0,132,0,0,0,100,0,0,0,133,0,0,0,15,0,0,0,75,0,0,0,76,0,0,0,50,0,0,0,51,0,0,0,52,0,0,0,53,0,0,0,6,0,0,0,101,0,0,0,102,0,0,0,10,0,0,0,103,0,0,0,104,0,0,0,134,0,0,0,135,0,0,0,37,0,0,0,136,0,0,0,137,0,0,0,38,0,0,0,105,0,0,0,106,0,0,0,107,0,0,0,108,0,0,0,109,0,0,0,23,0,0,0,24,0,0,0,4,0,0,0,25,0,0,0,4,0,0,0,26,0,0,0,138,0,0,0,39,0,0,0,14,0,0,0,4,0,0,0,4,0,0,0,40,0,0,0,15,0,0,0,41,0,0,0,105,0,0,0,110,0,0,0,111,0,0,0,112,0,0,0,106,0,0,0,107,0,0,0,113,0,0,0,108,0,0,0,109,0,0,0,110,0,0,0,16,0,0,0,111,0,0,0,112,0,0,0,113,0,0,0,139,0,0,0,140,0,0,0,77,0,0,0,141,0,0,0,114,0,0,0,117,0,0,0,118,0,0,0,114,0,0,0,16,0,0,0,78,0,0,0,142,0,0,0,143,0,0,0,4,0,0,0,119,0,0,0,120,0,0,0,121,0,0,0,54,0,0,0,144,0,0,0,55,0,0,0,56,0,0,0,57,0,0,0,58,0,0,0,17,0,0,0,122,0,0,0,123,0,0,0,17,0,0,0,23,0,0,0,115,0,0,0,145,0,0,0,42,0,0,0,115,0,0,0,9,0,0,0,116,0,0,0,24,0,0,0,7,0,0,0,8,0,0,0,27,0,0,0,59,0,0,0,60,0,0,0,116,0,0,0,117,0,0,0,252,255,255,255,144,2,0,0,118,0,0,0,2,0,0,0,35,0,0,0,36,0,0,0,117,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,4,0,0,0,5,0,0,0,0,0,0,0,3,0,0,0,1,0,0,0,4,0,0,0,2,0,0,0,5,0,0,0,0,0,0,0,192,2,0,0,124,0,0,0,125,0,0,0,126,0,0,0,127,0,0,0,146,0,0,0,147,0,0,0,148,0,0,0,119,0,0,0,149,0,0,0,79,0,0,0,80,0,0,0,81,0,0,0,118,0,0,0,2,0,0,0,120,0,0,0,121,0,0,0,82,0,0,0,119,0,0,0,122,0,0,0,6,0,0,0,83,0,0,0,84,0,0,0,120,0,0,0,10,0,0,0,61,0,0,0,62,0,0,0,63,0,0,0,64,0,0,0,65,0,0,0,11,0,0,0,12,0,0,0,11,0,0,0,2,0,0,0,85,0,0,0,86,0,0,0,121,0,0,0,18,0,0,0,6,0,0,0,43,0,0,0,44,0,0,0,45,0,0,0,150,0,0,0,151,0,0,0,152,0,0,0,123,0,0,0,27,0,0,0,0,0,0,0,104,1,0,0,19,0,0,0,128,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,124,0,0,0,27,0,0,0,0,0,0,0,224,2,0,0,129,0,0,0,130,0,0,0,131,0,0,0,132,0,0,0,153,0,0,0,154,0,0,0,155,0,0,0,125,0,0,0,156,0,0,0,87,0,0,0,88,0,0,0,89,0,0,0,122,0,0,0,3,0,0,0,126,0,0,0,127,0,0,0,90,0,0,0,123,0,0,0,128,0,0,0,7,0,0,0,91,0,0,0,92,0,0,0,124,0,0,0,13,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,69,0,0,0,70,0,0,0,14,0,0,0,15,0,0,0,12,0,0,0,3,0,0,0,93,0,0,0,94,0,0,0,125,0,0,0,19,0,0,0,7,0,0,0,46,0,0,0,47,0,0,0,48,0,0,0,157,0,0,0,158,0,0,0,159,0,0,0,129,0,0,0,27,0,0,0,0,0,0,0,0,3,0,0,133,0,0,0,134,0,0,0,135,0,0,0,136,0,0,0,160,0,0,0,161,0,0,0,162,0,0,0,130,0,0,0,163,0,0,0,95,0,0,0,96,0,0,0,97,0,0,0,126,0,0,0,4,0,0,0,131,0,0,0,132,0,0,0,98,0,0,0,127,0,0,0,133,0,0,0,8,0,0,0,99,0,0,0,100,0,0,0,128,0,0,0,16,0,0,0,71,0,0,0,72,0,0,0,73,0,0,0,74,0,0,0,75,0,0,0,17,0,0,0,18,0,0,0,13,0,0,0,4,0,0,0,101,0,0,0,102,0,0,0,129,0,0,0,20,0,0,0,8,0,0,0,49,0,0,0,50,0,0,0,51,0,0,0,164,0,0,0,165,0,0,0,166,0,0,0,134,0,0,0,27,0,0,0,0,0,0,0,16,3,0,0,137,0,0,0,138,0,0,0,139,0,0,0,140,0,0,0,167,0,0,0,168,0,0,0,169,0,0,0,135,0,0,0,170,0,0,0,103,0,0,0,104,0,0,0,105,0,0,0,130,0,0,0,5,0,0,0,136,0,0,0,137,0,0,0,106,0,0,0,131,0,0,0,138,0,0,0,9,0,0,0,107,0,0,0,108,0,0,0,132,0,0,0,19,0,0,0,76,0,0,0,77,0,0,0,78,0,0,0,79,0,0,0,80,0,0,0,20,0,0,0,21,0,0,0,14,0,0,0,5,0,0,0,109,0,0,0,110,0,0,0,133,0,0,0,21,0,0,0,9,0,0,0,52,0,0,0,53,0,0,0,54,0,0,0,171,0,0,0,172,0,0,0,173,0,0,0,139,0,0,0,27,0,0,0,0,0,0,0,32,3,0,0,141,0,0,0,142,0,0,0,143,0,0,0,144,0,0,0,174,0,0,0,175,0,0,0,176,0,0,0,140,0,0,0,177,0,0,0,111,0,0,0,112,0,0,0,113,0,0,0,134,0,0,0,6,0,0,0,141,0,0,0,142,0,0,0,114,0,0,0,135,0,0,0,143,0,0,0,10,0,0,0,115,0,0,0,116,0,0,0,136,0,0,0,22,0,0,0,81,0,0,0,82,0,0,0,83,0,0,0,84,0,0,0,85,0,0,0,23,0,0,0,24,0,0,0,15,0,0,0,6,0,0,0,117,0,0,0,118,0,0,0,137,0,0,0,22,0,0,0,10,0,0,0,55,0,0,0,56,0,0,0,57,0,0,0,178,0,0,0,179,0,0,0,180,0,0,0,144,0,0,0,27,0,0,0,0,0,0,0,48,3,0,0,145,0,0,0,146,0,0,0,147,0,0,0,148,0,0,0,181,0,0,0,182,0,0,0,183,0,0,0,145,0,0,0,184,0,0,0,119,0,0,0,120,0,0,0,121,0,0,0,138,0,0,0,7,0,0,0,146,0,0,0,147,0,0,0,122,0,0,0,139,0,0,0,148,0,0,0,11,0,0,0,123,0,0,0,124,0,0,0,140,0,0,0,25,0,0,0,86,0,0,0,87,0,0,0,88,0,0,0,89,0,0,0,90,0,0,0,26,0,0,0,27,0,0,0,16,0,0,0,7,0,0,0,125,0,0,0,126,0,0,0,141,0,0,0,23,0,0,0,11,0,0,0,58,0,0,0,59,0,0,0,60,0,0,0,185,0,0,0,186,0,0,0,187,0,0,0,124,0,0,0,149,0,0,0,0,0,0,0,64,3,0,0,149,0,0,0,150,0,0,0,151,0,0,0,152,0,0,0,188,0,0,0,189,0,0,0,190,0,0,0,150,0,0,0,191,0,0,0,127,0,0,0,128,0,0,0,129,0,0,0,142,0,0,0,8,0,0,0,151,0,0,0,152,0,0,0,130,0,0,0,143,0,0,0,153,0,0,0,12,0,0,0,131,0,0,0,132,0,0,0,144,0,0,0,28,0,0,0,91,0,0,0,92,0,0,0,93,0,0,0,94,0,0,0,95,0,0,0,29,0,0,0,30,0,0,0,17,0,0,0,8,0,0,0,133,0,0,0,134,0,0,0,145,0,0,0,24,0,0,0,12,0,0,0,61,0,0,0,62,0,0,0,63,0,0,0,192,0,0,0,193,0,0,0,194,0,0,0,154,0,0,0,27,0,0,0,0,0,0,0,80,3,0,0,153,0,0,0,154,0,0,0,155,0,0,0,156,0,0,0,195,0,0,0,196,0,0,0,197,0,0,0,155,0,0,0,64,0,0,0,135,0,0,0,96,0,0,0,97,0,0,0,98,0,0,0,99,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,18,0,0,0,19,0,0,0,146,0,0,0,147,0,0,0,148,0,0,0,20,0,0,0,21,0,0,0,149,0,0,0,198,0,0,0,157,0,0,0,156,0,0,0,7,0,0,0,150,0,0,0,151,0,0,0,152,0,0,0,252,255,255,255,80,3,0,0,157,0,0,0,8,0,0,0,153,0,0,0,154,0,0,0,155,0,0,0,0,0,0,0,128,3,0,0,158,0,0,0,159,0,0,0,160,0,0,0,161,0,0,0,199,0,0,0,200,0,0,0,201,0,0,0,158,0,0,0,65,0,0,0,136,0,0,0,103,0,0,0,104,0,0,0,105,0,0,0,106,0,0,0,107,0,0,0,108,0,0,0,109,0,0,0,22,0,0,0,23,0,0,0,156,0,0,0,157,0,0,0,158,0,0,0,24,0,0,0,25,0,0,0,159,0,0,0,202,0,0,0,162,0,0,0,159,0,0,0,9,0,0,0,160,0,0,0,161,0,0,0,162,0,0,0,252,255,255,255,128,3,0,0,160,0,0,0,10,0,0,0,163,0,0,0,164,0,0,0,165,0,0,0,0,0,0,0,144,3,0,0,163,0,0,0,164,0,0,0,165,0,0,0,166,0,0,0,203,0,0,0,204,0,0,0,205,0,0,0,161,0,0,0,167,0,0,0,166,0,0,0,66,0,0,0,25,0,0,0,0,0,0,0,160,3,0,0,168,0,0,0,169,0,0,0,170,0,0,0,171,0,0,0,206,0,0,0,207,0,0,0,208,0,0,0,162,0,0,0,172,0,0,0,167,0,0,0,67,0,0,0,26,0,0,0,0,0,0,0,176,3,0,0,173,0,0,0,174,0,0,0,175,0,0,0,176,0,0,0,209,0,0,0,210,0,0,0,211,0,0,0,163,0,0,0,10,0,0,0,137,0,0,0,168,0,0,0,138,0,0,0,139,0,0,0,140,0,0,0,212,0,0,0,68,0,0,0,213,0,0,0,26,0,0,0,9,0,0,0,27,0,0,0,177,0,0,0,11,0,0,0,1,0,0,0,11,0,0,0,169,0,0,0,170,0,0,0,171,0,0,0,164,0,0,0,252,255,255,255,176,3,0,0,165,0,0,0,12,0,0,0,172,0,0,0,173,0,0,0,174,0,0,0,0,0,0,0,224,3,0,0,178,0,0,0,179,0,0,0,180,0,0,0,181,0,0,0,214,0,0,0,215,0,0,0,216,0,0,0,166,0,0,0,12,0,0,0,141,0,0,0,175,0,0,0,142,0,0,0,143,0,0,0,144,0,0,0,217,0,0,0,69,0,0,0,218,0,0,0,28,0,0,0,10,0,0,0,29,0,0,0,182,0,0,0,13,0,0,0,2,0,0,0,167,0,0,0,13,0,0,0,176,0,0,0,177,0,0,0,178,0,0,0,252,255,255,255,224,3,0,0,168,0,0,0,14,0,0,0,179,0,0,0,180,0,0,0,181,0,0,0,0,0,0,0,240,3,0,0,183,0,0,0,184,0,0,0,185,0,0,0,186,0,0,0,219,0,0,0,220,0,0,0,221,0,0,0,169,0,0,0,14,0,0,0,145,0,0,0,182,0,0,0,146,0,0,0,147,0,0,0,148,0,0,0,222,0,0,0,70,0,0,0,223,0,0,0,30,0,0,0,11,0,0,0,31,0,0,0,187,0,0,0,15,0,0,0,3,0,0,0,15,0,0,0,183,0,0,0,184,0,0,0,185,0,0,0,170,0,0,0,252,255,255,255,240,3,0,0,171,0,0,0,16,0,0,0,186,0,0,0,187,0,0,0,188,0,0,0,0,0,0,0,0,4,0,0,188,0,0,0,189,0,0,0,190,0,0,0,191,0,0,0,224,0,0,0,225,0,0,0,226,0,0,0,172,0,0,0,227,0,0,0,149,0,0,0,150,0,0,0,189,0,0,0,173,0,0,0,151,0,0,0,32,0,0,0,152,0,0,0,190,0,0,0,191,0,0,0,174,0,0,0,27,0,0,0,153,0,0,0,154,0,0,0,155,0,0,0,156,0,0,0,157,0,0,0,158,0,0,0,31,0,0,0,159,0,0,0,192,0,0,0,193,0,0,0,175,0,0,0,194,0,0,0,176,0,0,0,160,0,0,0,161,0,0,0,162,0,0,0,28,0,0,0,0,0,0,0,16,4,0,0,192,0,0,0,193,0,0,0,194,0,0,0,195,0,0,0,228,0,0,0,229,0,0,0,230,0,0,0,177,0,0,0,231,0,0,0,163,0,0,0,164,0,0,0,195,0,0,0,178,0,0,0,165,0,0,0,33,0,0,0,166,0,0,0,196,0,0,0,197,0,0,0,179,0,0,0,29,0,0,0,167,0,0,0,168,0,0,0,169,0,0,0,170,0,0,0,171,0,0,0,172,0,0,0,32,0,0,0,173,0,0,0,198,0,0,0,199,0,0,0,180,0,0,0,200,0,0,0,181,0,0,0,174,0,0,0,175,0,0,0,176,0,0,0,30,0,0,0,0,0,0,0,32,4,0,0,196,0,0,0,197,0,0,0,198,0,0,0,199,0,0,0,232,0,0,0,233,0,0,0,234,0,0,0,182,0,0,0,235,0,0,0,177,0,0,0,178,0,0,0,201,0,0,0,183,0,0,0,179,0,0,0,34,0,0,0,180,0,0,0,202,0,0,0,203,0,0,0,184,0,0,0,31,0,0,0,181,0,0,0,182,0,0,0,183,0,0,0,184,0,0,0,185,0,0,0,186,0,0,0,33,0,0,0,187,0,0,0,204,0,0,0,205,0,0,0,185,0,0,0,206,0,0,0,186,0,0,0,188,0,0,0,189,0,0,0,190,0,0,0,32,0,0,0,0,0,0,0,48,4,0,0,200,0,0,0,201,0,0,0,202,0,0,0,203,0,0,0,236,0,0,0,237,0,0,0,238,0,0,0,187,0,0,0,239,0,0,0,191,0,0,0,192,0,0,0,207,0,0,0,188,0,0,0,193,0,0,0,35,0,0,0,194,0,0,0,208,0,0,0,209,0,0,0,189,0,0,0,33,0,0,0,195,0,0,0,196,0,0,0,197,0,0,0,198,0,0,0,199,0,0,0,200,0,0,0,34,0,0,0,201,0,0,0,210,0,0,0,211,0,0,0,190,0,0,0,212,0,0,0,191,0,0,0,202,0,0,0,203,0,0,0,204,0,0,0,34,0,0,0,0,0,0,0,64,4,0,0,204,0,0,0,205,0,0,0,206,0,0,0,207,0,0,0,240,0,0,0,241,0,0,0,242,0,0,0,192,0,0,0,243,0,0,0,205,0,0,0,206,0,0,0,213,0,0,0,193,0,0,0,207,0,0,0,36,0,0,0,208,0,0,0,214,0,0,0,215,0,0,0,194,0,0,0,35,0,0,0,209,0,0,0,210,0,0,0,211,0,0,0,212,0,0,0,213,0,0,0,214,0,0,0,35,0,0,0,215,0,0,0,216,0,0,0,217,0,0,0,195,0,0,0,218,0,0,0,196,0,0,0,216,0,0,0,217,0,0,0,218,0,0,0,36,0,0,0,0,0,0,0,80,4,0,0,208,0,0,0,209,0,0,0,210,0,0,0,211,0,0,0,244,0,0,0,245,0,0,0,246,0,0,0,197,0,0,0,247,0,0,0,219,0,0,0,220,0,0,0,219,0,0,0,198,0,0,0,221,0,0,0,37,0,0,0,222,0,0,0,220,0,0,0,221,0,0,0,199,0,0,0,37,0,0,0,223,0,0,0,224,0,0,0,225,0,0,0,226,0,0,0,227,0,0,0,228,0,0,0,36,0,0,0,229,0,0,0,222,0,0,0,223,0,0,0,200,0,0,0,224,0,0,0,201,0,0,0,230,0,0,0,231,0,0,0,232,0,0,0,38,0,0,0,0,0,0,0,96,4,0,0,212,0,0,0,213,0,0,0,214,0,0,0,215,0,0,0,248,0,0,0,249,0,0,0,250,0,0,0,202,0,0,0,251,0,0,0,233,0,0,0,234,0,0,0,225,0,0,0,203,0,0,0,235,0,0,0,38,0,0,0,236,0,0,0,226,0,0,0,227,0,0,0,204,0,0,0,39,0,0,0,237,0,0,0,238,0,0,0,239,0,0,0,240,0,0,0,241,0,0,0,242,0,0,0,37,0,0,0,243,0,0,0,228,0,0,0,229,0,0,0,205,0,0,0,230,0,0,0,206,0,0,0,244,0,0,0,245,0,0,0,246,0,0,0,40,0,0,0,0,0,0,0,112,4,0,0,216,0,0,0,217,0,0,0,218,0,0,0,219,0,0,0,252,0,0,0,253,0,0,0,254,0,0,0,207,0,0,0,255,0,0,0,247,0,0,0,248,0,0,0,231,0,0,0,208,0,0,0,249,0,0,0,39,0,0,0,250,0,0,0,232,0,0,0,233,0,0,0,209,0,0,0,41,0,0,0,251,0,0,0,252,0,0,0,253,0,0,0,254,0,0,0,255,0,0,0,0,1,0,0,38,0,0,0,1,1,0,0,234,0,0,0,235,0,0,0,210,0,0,0,236,0,0,0,211,0,0,0,2,1,0,0,3,1,0,0,4,1,0,0,42,0,0,0,0,0,0,0,128,4,0,0,220,0,0,0,221,0,0,0,222,0,0,0,223,0,0,0,0,1,0,0,1,1,0,0,2,1,0,0,212,0,0,0,3,1,0,0,5,1,0,0,6,1,0,0,237,0,0,0,213,0,0,0,7,1,0,0,40,0,0,0,8,1,0,0,238,0,0,0,239,0,0,0,214,0,0,0,43,0,0,0,9,1,0,0,10,1,0,0,11,1,0,0,12,1,0,0,13,1,0,0,14,1,0,0,39,0,0,0,15,1,0,0,240,0,0,0,241,0,0,0,215,0,0,0,242,0,0,0,216,0,0,0,16,1,0,0,17,1,0,0,18,1,0,0,44,0,0,0,0,0,0,0,144,4,0,0,224,0,0,0,225,0,0,0,226,0,0,0,227,0,0,0,4,1,0,0,5,1,0,0,6,1,0,0,217,0,0,0,7,1,0,0,19,1,0,0,20,1,0,0,243,0,0,0,218,0,0,0,21,1,0,0,41,0,0,0,22,1,0,0,244,0,0,0,245,0,0,0,219,0,0,0,45,0,0,0,23,1,0,0,24,1,0,0,25,1,0,0,26,1,0,0,27,1,0,0,28,1,0,0,40,0,0,0,29,1,0,0,246,0,0,0,247,0,0,0,220,0,0,0,248,0,0,0,221,0,0,0,30,1,0,0,31,1,0,0,32,1,0,0,46,0,0,0,0,0,0,0,160,4,0,0,228,0,0,0,229,0,0,0,230,0,0,0,231,0,0,0,8,1,0,0,9,1,0,0,10,1,0,0,222,0,0,0,11,1,0,0,33,1,0,0,34,1,0,0,249,0,0,0,223,0,0,0,35,1,0,0,42,0,0,0,36,1,0,0,250,0,0,0,251,0,0,0,224,0,0,0,47,0,0,0,37,1,0,0,38,1,0,0,39,1,0,0,40,1,0,0,41,1,0,0,42,1,0,0,41,0,0,0,43,1,0,0,252,0,0,0,253,0,0,0,225,0,0,0,254,0,0,0,226,0,0,0,44,1,0,0,45,1,0,0,46,1,0,0,48,0,0,0,0,0,0,0,176,4,0,0,232,0,0,0,233,0,0,0,234,0,0,0,235,0,0,0,12,1,0,0,13,1,0,0,14,1,0,0,227,0,0,0,15,1,0,0,47,1,0,0,48,1,0,0,255,0,0,0,228,0,0,0,49,1,0,0,43,0,0,0,50,1,0,0,0,1,0,0,1,1,0,0,229,0,0,0,49,0,0,0,51,1,0,0,52,1,0,0,53,1,0,0,54,1,0,0,55,1,0,0,56,1,0,0,42,0,0,0,57,1,0,0,2,1,0,0,3,1,0,0,230,0,0,0,4,1,0,0,231,0,0,0,58,1,0,0,59,1,0,0,60,1,0,0,50,0,0,0,0,0,0,0,192,4,0,0,236,0,0,0,237,0,0,0,238,0,0,0,239,0,0,0,16,1,0,0,17,1,0,0,18,1,0,0,232,0,0,0,19,1,0,0,61,1,0,0,62,1,0,0,5,1,0,0,233,0,0,0,63,1,0,0,44,0,0,0,64,1,0,0,6,1,0,0,7,1,0,0,234,0,0,0,51,0,0,0,65,1,0,0,66,1,0,0,67,1,0,0,68,1,0,0,69,1,0,0,70,1,0,0,43,0,0,0,71,1,0,0,8,1,0,0,9,1,0,0,235,0,0,0,10,1,0,0,236,0,0,0,72,1,0,0,73,1,0,0,74,1,0,0,52,0,0,0,0,0,0,0,208,4,0,0,240,0,0,0,241,0,0,0,242,0,0,0,243,0,0,0,20,1,0,0,21,1,0,0,22,1,0,0,237,0,0,0,255,255,255,255,0,0,0,0,224,4,0,0,244,0,0,0,245,0,0,0,23,1,0,0,11,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,0,0,0,0,240,4,0,0,246,0,0,0,247,0,0,0,248,0,0,0,249,0,0,0,24,1,0,0,25,1,0,0,26,1,0,0,238,0,0,0,11,1,0,0,27,1,0,0,28,1,0,0,110,0,0,0,29,1,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,12,0,0,0,15,0,0,0,16,0,0,0,18,0,0,0,20,0,0,0,24,0,0,0,25,0,0,0,30,0,0,0,36,0,0,0,40,0,0,0,45,0,0,0,48,0,0,0,50,0,0,0,60,0,0,0,72,0,0,0,75,0,0,0,80,0,0,0,90,0,0,0,100,0,0,0,120,0,0,0,144,0,0,0,150,0,0,0,180,0,0,0,200,0,0,0,225,0,0,0,240,0,0,0,44,1,0,0,104,1,0,0,144,1,0,0,194,1,0,0,88,2,0,0,208,2,0,0,132,3,0,0,176,4,0,0,8,7,0,0,16,14,0,0,75,1,0,0,63,0,0,0,150,3,0,0,0,0,0,0,0,5,0,0,250,0,0,0,251,0,0,0,252,0,0,0,253,0,0,0,30,1,0,0,31,1,0,0,32,1,0,0,239,0,0,0,33,1,0,0,75,1,0,0,76,1,0,0,77,1,0,0,12,1,0,0,12,0,0,0,240,0,0,0,241,0,0,0,78,1,0,0,13,1,0,0,242,0,0,0,13,0,0,0,79,1,0,0,80,1,0,0,14,1,0,0,44,0,0,0,111,0,0,0,112,0,0,0,113,0,0,0,114,0,0,0,115,0,0,0,45,0,0,0,46,0,0,0,45,0,0,0,9,0,0,0,81,1,0,0,82,1,0,0,15,1,0,0,25,0,0,0,13,0,0,0,71,0,0,0,72,0,0,0,73,0,0,0,34,1,0,0,35,1,0,0,36,1,0,0,243,0,0,0,27,0,0,0,178,2,0,0,181,2,0,0,182,2,0,0,121,0,0,0,123,0,0,0,125,0,0,0,139,0,0,0,140,0,0,0,127,0,0,0,128,0,0,0,146,0,0,0,124,0,0,0,121,0,0,0,123,0,0,0,124,0,0,0,75,1,0,0,143,2,0,0,144,2,0,0,153,2,0,0,156,2,0,0,150,2,0,0,183,2,0,0,184,2,0,0,150,2,0,0,0,0,0,0,16,5,0,0,254,0,0,0,255,0,0,0,0,1,0,0,1,1,0,0,37,1,0,0,38,1,0,0,39,1,0,0,244,0,0,0,16,0,0,0,83,1,0,0,16,1,0,0,84,1,0,0,85,1,0,0,86,1,0,0,40,1,0,0,74,0,0,0,41,1,0,0,46,0,0,0,13,0,0,0,47,0,0,0,2,1,0,0,17,0,0,0,4,0,0,0,17,0,0,0,17,1,0,0,18,1,0,0,19,1,0,0,245,0,0,0,252,255,255,255,16,5,0,0,246,0,0,0,18,0,0,0,20,1,0,0,21,1,0,0,22,1,0,0,0,0,0,0,104,5,0,0,3,1,0,0,4,1,0,0,5,1,0,0,6,1,0,0,42,1,0,0,43,1,0,0,44,1,0,0,247,0,0,0,45,1,0,0,46,1,0,0,28,0,0,0,252,255,255,255,104,5,0,0,7,1,0,0,8,1,0,0,47,1,0,0,48,1,0,0,29,0,0,0,0,0,0,0,32,5,0,0,9,1,0,0,10,1,0,0,11,1,0,0,12,1,0,0,49,1,0,0,50,1,0,0,51,1,0,0,248,0,0,0,52,1,0,0,53,1,0,0,30,0,0,0,252,255,255,255,32,5,0,0,13,1,0,0,14,1,0,0,54,1,0,0,55,1,0,0,31,0,0,0,0,0,0,0,72,5,0,0,15,1,0,0,16,1,0,0,17,1,0,0,18,1,0,0,56,1,0,0,57,1,0,0,58,1,0,0,249,0,0,0,59,1,0,0,60,1,0,0,32,0,0,0,252,255,255,255,72,5,0,0,19,1,0,0,20,1,0,0,61,1,0,0,62,1,0,0,33,0,0,0,0,0,0,0,136,5,0,0,21,1,0,0,22,1,0,0,23,1,0,0,24,1,0,0,63,1,0,0,64,1,0,0,65,1,0,0,250,0,0,0,66,1,0,0,67,1,0,0,34,0,0,0,252,255,255,255,136,5,0,0,25,1,0,0,26,1,0,0,68,1,0,0,69,1,0,0,35,0,0,0,0,0,0,0,168,5,0,0,27,1,0,0,28,1,0,0,29,1,0,0,30,1,0,0,70,1,0,0,71,1,0,0,72,1,0,0,251,0,0,0,73,1,0,0,74,1,0,0,36,0,0,0,252,255,255,255,168,5,0,0,31,1,0,0,32,1,0,0,75,1,0,0,76,1,0,0,37,0,0,0,0,0,0,0,200,5,0,0,33,1,0,0,34,1,0,0,35,1,0,0,36,1,0,0,77,1,0,0,78,1,0,0,79,1,0,0,252,0,0,0,80,1,0,0,87,1,0,0,88,1,0,0,89,1,0,0,23,1,0,0,14,0,0,0,253,0,0,0,254,0,0,0,90,1,0,0,24,1,0,0,255,0,0,0,14,0,0,0,91,1,0,0,92,1,0,0,25,1,0,0,47,0,0,0,116,0,0,0,117,0,0,0,118,0,0,0,119,0,0,0,120,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,10,0,0,0,93,1,0,0,94,1,0,0,26,1,0,0,26,0,0,0,14,0,0,0,75,0,0,0,76,0,0,0,77,0,0,0,81,1,0,0,82,1,0,0,83,1,0,0,0,1,0,0,27,0,0,0,0,0,0,0,232,5,0,0,37,1,0,0,38,1,0,0,39,1,0,0,40,1,0,0,84,1,0,0,85,1,0,0,86,1,0,0,1,1,0,0,87,1,0,0,95,1,0,0,96,1,0,0,97,1,0,0,27,1,0,0,15,0,0,0,2,1,0,0,3,1,0,0,98,1,0,0,28,1,0,0,4,1,0,0,15,0,0,0,99,1,0,0,100,1,0,0,29,1,0,0,50,0,0,0,121,0,0,0,122,0,0,0,123,0,0,0,124,0,0,0,125,0,0,0,51,0,0,0,52,0,0,0,49,0,0,0,11,0,0,0,101,1,0,0,102,1,0,0,30,1,0,0,27,0,0,0,15,0,0,0,78,0,0,0,79,0,0,0,80,0,0,0,88,1,0,0,89,1,0,0,90,1,0,0,5,1,0,0,27,0,0,0,0,0,0,0,0,0,0,0,8,6,0,0,41,1,0,0,42,1,0,0,31,1,0,0,32,1,0,0,33,1,0,0,28,0,0,0,53,0,0,0,54,0,0,0,103,1,0,0,81,0,0,0,82,0,0,0,83,0,0,0,84,0,0,0,34,1,0,0,55,0,0,0,35,1,0,0,104,1,0,0,0,0,0,0,32,6,0,0,41,1,0,0,43,1,0,0,36,1,0,0,37,1,0,0,38,1,0,0,29,0,0,0,56,0,0,0,57,0,0,0,105,1,0,0,85,0,0,0,86,0,0,0,87,0,0,0,88,0,0,0,39,1,0,0,58,0,0,0,40,1,0,0,106,1,0,0,0,0,0,0,48,6,0,0,41,1,0,0,44,1,0,0,41,1,0,0,42,1,0,0,43,1,0,0,30,0,0,0,59,0,0,0,60,0,0,0,107,1,0,0,89,0,0,0,90,0,0,0,91,0,0,0,92,0,0,0,44,1,0,0,61,0,0,0,45,1,0,0,108,1,0,0,0,0,0,0,64,6,0,0,41,1,0,0,45,1,0,0,46,1,0,0,47,1,0,0,48,1,0,0,31,0,0,0,62,0,0,0,63,0,0,0,109,1,0,0,93,0,0,0,94,0,0,0,95,0,0,0,96,0,0,0,49,1,0,0,64,0,0,0,50,1,0,0,110,1,0,0,0,0,0,0,80,6,0,0,41,1,0,0,46,1,0,0,51,1,0,0,52,1,0,0,53,1,0,0,32,0,0,0,65,0,0,0,66,0,0,0,111,1,0,0,97,0,0,0,98,0,0,0,99,0,0,0,100,0,0,0,54,1,0,0,67,0,0,0,55,1,0,0,112,1,0,0,0,0,0,0,96,6,0,0,41,1,0,0,47,1,0,0,56,1,0,0,57,1,0,0,58,1,0,0,33,0,0,0,68,0,0,0,69,0,0,0,113,1,0,0,101,0,0,0,102,0,0,0,103,0,0,0,104,0,0,0,59,1,0,0,70,0,0,0,60,1,0,0,114,1,0,0,0,0,0,0,112,6,0,0,41,1,0,0,48,1,0,0,61,1,0,0,62,1,0,0,63,1,0,0,34,0,0,0,71,0,0,0,72,0,0,0,115,1,0,0,105,0,0,0,106,0,0,0,107,0,0,0,108,0,0,0,64,1,0,0,73,0,0,0,65,1,0,0,116,1,0,0,0,0,0,0,128,6,0,0,41,1,0,0,49,1,0,0,66,1,0,0,67,1,0,0,68,1,0,0,35,0,0,0,74,0,0,0,75,0,0,0,117,1,0,0,109,0,0,0,110,0,0,0,111,0,0,0,112,0,0,0,69,1,0,0,76,0,0,0,70,1,0,0,118,1,0,0,0,0,0,0,144,6,0,0,41,1,0,0,50,1,0,0,71,1,0,0,72,1,0,0,73,1,0,0,36,0,0,0,77,0,0,0,78,0,0,0,119,1,0,0,113,0,0,0,114,0,0,0,115,0,0,0,116,0,0,0,74,1,0,0,79,0,0,0,75,1,0,0,120,1,0,0,0,0,0,0,160,6,0,0,41,1,0,0,51,1,0,0,76,1,0,0,77,1,0,0,78,1,0,0,37,0,0,0,80,0,0,0,81,0,0,0,121,1,0,0,117,0,0,0,118,0,0,0,119,0,0,0,120,0,0,0,79,1,0,0,82,0,0,0,80,1,0,0,122,1,0,0,18,0,0,0,19,0,0,0,20,0,0,0,0,0,0,0,176,6,0,0,41,1,0,0,52,1,0,0,81,1,0,0,82,1,0,0,83,1,0,0,38,0,0,0,83,0,0,0,84,0,0,0,123,1,0,0,121,0,0,0,122,0,0,0,123,0,0,0,124,0,0,0,84,1,0,0,85,0,0,0,85,1,0,0,124,1,0,0,0,0,0,0,192,6,0,0,41,1,0,0,53,1,0,0,86,1,0,0,87,1,0,0,88,1,0,0,39,0,0,0,86,0,0,0,87,0,0,0,125,1,0,0,125,0,0,0,126,0,0,0,127,0,0,0,128,0,0,0,89,1,0,0,88,0,0,0,90,1,0,0,126,1,0,0,0,0,0,0,208,6,0,0,41,1,0,0,54,1,0,0,91,1,0,0,92,1,0,0,93,1,0,0,40,0,0,0,89,0,0,0,90,0,0,0,127,1,0,0,129,0,0,0,130,0,0,0,131,0,0,0,132,0,0,0,94,1,0,0,91,0,0,0,95,1,0,0,128,1,0,0,0,0,0,0,224,6,0,0,41,1,0,0,55,1,0,0,96,1,0,0,97,1,0,0,98,1,0,0,41,0,0,0,92,0,0,0,93,0,0,0,129,1,0,0,133,0,0,0,134,0,0,0,135,0,0,0,136,0,0,0,99,1,0,0,94,0,0,0,100,1,0,0,130,1,0,0,0,0,0,0,240,6,0,0,41,1,0,0,56,1,0,0,101,1,0,0,102,1,0,0,103,1,0,0,42,0,0,0,95,0,0,0,96,0,0,0,131,1,0,0,137,0,0,0,138,0,0,0,139,0,0,0,140,0,0,0,104,1,0,0,97,0,0,0,105,1,0,0,132,1,0,0,0,0,0,0,0,7,0,0,41,1,0,0,57,1,0,0,106,1,0,0,107,1,0,0,108,1,0,0,43,0,0,0,98,0,0,0,99,0,0,0,133,1,0,0,141,0,0,0,142,0,0,0,143,0,0,0,144,0,0,0,109,1,0,0,100,0,0,0,110,1,0,0,134,1,0,0,0,0,0,0,16,7,0,0,41,1,0,0,58,1,0,0,111,1,0,0,112,1,0,0,113,1,0,0,44,0,0,0,101,0,0,0,102,0,0,0,135,1,0,0,145,0,0,0,146,0,0,0,147,0,0,0,148,0,0,0,114,1,0,0,103,0,0,0,115,1,0,0,136,1,0,0,0,0,0,0,32,7,0,0,41,1,0,0,59,1,0,0,116,1,0,0,117,1,0,0,118,1,0,0,45,0,0,0,104,0,0,0,105,0,0,0,137,1,0,0,149,0,0,0,150,0,0,0,151,0,0,0,152,0,0,0,119,1,0,0,106,0,0,0,120,1,0,0,138,1,0,0,0,0,0,0,48,7,0,0,41,1,0,0,60,1,0,0,121,1,0,0,122,1,0,0,123,1,0,0,46,0,0,0,107,0,0,0,108,0,0,0,139,1,0,0,153,0,0,0,154,0,0,0,155,0,0,0,156,0,0,0,124,1,0,0,109,0,0,0,125,1,0,0,140,1,0,0,0,0,0,0,64,7,0,0,41,1,0,0,61,1,0,0,126,1,0,0,127,1,0,0,128,1,0,0,47,0,0,0,110,0,0,0,111,0,0,0,141,1,0,0,157,0,0,0,158,0,0,0,159,0,0,0,160,0,0,0,129,1,0,0,112,0,0,0,130,1,0,0,142,1,0,0,0,0,0,0,80,7,0,0,41,1,0,0,62,1,0,0,131,1,0,0,132,1,0,0,133,1,0,0,48,0,0,0,113,0,0,0,114,0,0,0,143,1,0,0,161,0,0,0,162,0,0,0,163,0,0,0,164,0,0,0,134,1,0,0,115,0,0,0,135,1,0,0,144,1,0,0,0,0,0,0,96,7,0,0,41,1,0,0,63,1,0,0,136,1,0,0,137,1,0,0,138,1,0,0,49,0,0,0,116,0,0,0,117,0,0,0,145,1,0,0,165,0,0,0,166,0,0,0,167,0,0,0,168,0,0,0,139,1,0,0,118,0,0,0,140,1,0,0,146,1,0,0,0,0,0,0,112,7,0,0,41,1,0,0,64,1,0,0,141,1,0,0,142,1,0,0,143,1,0,0,50,0,0,0,119,0,0,0,120,0,0,0,147,1,0,0,169,0,0,0,170,0,0,0,171,0,0,0,172,0,0,0,144,1,0,0,121,0,0,0,145,1,0,0,148,1,0,0,0,0,0,0,128,7,0,0,41,1,0,0,65,1,0,0,146,1,0,0,147,1,0,0,148,1,0,0,51,0,0,0,122,0,0,0,123,0,0,0,149,1,0,0,173,0,0,0,174,0,0,0,175,0,0,0,176,0,0,0,149,1,0,0,124,0,0,0,150,1,0,0,150,1,0,0,0,0,0,0,144,7,0,0,41,1,0,0,66,1,0,0,151,1,0,0,152,1,0,0,153,1,0,0,52,0,0,0,125,0,0,0,126,0,0,0,151,1,0,0,177,0,0,0,178,0,0,0,179,0,0,0,180,0,0,0,154,1,0,0,127,0,0,0,155,1,0,0,152,1,0,0,0,0,0,0,160,7,0,0,41,1,0,0,67,1,0,0,156,1,0,0,157,1,0,0,158,1,0,0,53,0,0,0,128,0,0,0,129,0,0,0,153,1,0,0,181,0,0,0,182,0,0,0,183,0,0,0,184,0,0,0,159,1,0,0,130,0,0,0,160,1,0,0,154,1,0,0,0,0,0,0,176,7,0,0,41,1,0,0,68,1,0,0,161,1,0,0,162,1,0,0,163,1,0,0,54,0,0,0,131,0,0,0,132,0,0,0,155,1,0,0,185,0,0,0,186,0,0,0,187,0,0,0,188,0,0,0,164,1,0,0,133,0,0,0,165,1,0,0,156,1,0,0,0,0,0,0,192,7,0,0,41,1,0,0,69,1,0,0,166,1,0,0,167,1,0,0,168,1,0,0,55,0,0,0,134,0,0,0,135,0,0,0,157,1,0,0,189,0,0,0,190,0,0,0,191,0,0,0,192,0,0,0,169,1,0,0,136,0,0,0,170,1,0,0,158,1,0,0,0,0,0,0,208,7,0,0,41,1,0,0,70,1,0,0,171,1,0,0,172,1,0,0,173,1,0,0,56,0,0,0,137,0,0,0,138,0,0,0,159,1,0,0,193,0,0,0,194,0,0,0,195,0,0,0,196,0,0,0,174,1,0,0,139,0,0,0,175,1,0,0,160,1,0,0,0,0,0,0,224,7,0,0,41,1,0,0,71,1,0,0,176,1,0,0,177,1,0,0,178,1,0,0,57,0,0,0,140,0,0,0,141,0,0,0,161,1,0,0,197,0,0,0,198,0,0,0,199,0,0,0,200,0,0,0,179,1,0,0,142,0,0,0,180,1,0,0,162,1,0,0,0,0,0,0,240,7,0,0,41,1,0,0,72,1,0,0,181,1,0,0,182,1,0,0,183,1,0,0,58,0,0,0,143,0,0,0,144,0,0,0,163,1,0,0,201,0,0,0,202,0,0,0,203,0,0,0,204,0,0,0,184,1,0,0,145,0,0,0,185,1,0,0,164,1,0,0,0,0,0,0,0,8,0,0,41,1,0,0,73,1,0,0,186,1,0,0,187,1,0,0,188,1,0,0,59,0,0,0,146,0,0,0,147,0,0,0,165,1,0,0,205,0,0,0,206,0,0,0,207,0,0,0,208,0,0,0,189,1,0,0,148,0,0,0,190,1,0,0,166,1,0,0,0,0,0,0,16,8,0,0,41,1,0,0,74,1,0,0,191,1,0,0,192,1,0,0,193,1,0,0,60,0,0,0,149,0,0,0,150,0,0,0,167,1,0,0,209,0,0,0,210,0,0,0,211,0,0,0,212,0,0,0,194,1,0,0,151,0,0,0,195,1,0,0,168,1,0,0,0,0,0,0,32,8,0,0,41,1,0,0,75,1,0,0,196,1,0,0,197,1,0,0,198,1,0,0,61,0,0,0,152,0,0,0,153,0,0,0,169,1,0,0,213,0,0,0,214,0,0,0,215,0,0,0,216,0,0,0,199,1,0,0,154,0,0,0,200,1,0,0,170,1,0,0,0,0,0,0,64,8,0,0,41,1,0,0,76,1,0,0,201,1,0,0,202,1,0,0,203,1,0,0,62,0,0,0,155,0,0,0,156,0,0,0,171,1,0,0,217,0,0,0,218,0,0,0,219,0,0,0,220,0,0,0,204,1,0,0,157,0,0,0,205,1,0,0,172,1,0,0,0,0,0,0,80,8,0,0,77,1,0,0,78,1,0,0,91,1,0,0,92,1,0,0,79,1,0,0,93,1,0,0,173,1,0,0,174,1,0,0,206,1,0,0,175,1,0,0,0,0,0,0,104,8,0,0,80,1,0,0,81,1,0,0,94,1,0,0,95,1,0,0,82,1,0,0,96,1,0,0,176,1,0,0,177,1,0,0,207,1,0,0,178,1,0,0,0,0,0,0,120,8,0,0,83,1,0,0,84,1,0,0,97,1,0,0,98,1,0,0,85,1,0,0,99,1,0,0,179,1,0,0,180,1,0,0,208,1,0,0,181,1,0,0,0,0,0,0,136,8,0,0,86,1,0,0,87,1,0,0,100,1,0,0,101,1,0,0,88,1,0,0,102,1,0,0,182,1,0,0,183,1,0,0,209,1,0,0,184,1,0,0,0,0,0,0,152,8,0,0,89,1,0,0,90,1,0,0,91,1,0,0,92,1,0,0,103,1,0,0,104,1,0,0,105,1,0,0,6,1,0,0,18,0,0,0,185,1,0,0,210,1,0,0,186,1,0,0,187,1,0,0,188,1,0,0,106,1,0,0,221,0,0,0,107,1,0,0,50,0,0,0,16,0,0,0,51,0,0,0,93,1,0,0,19,0,0,0,5],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+30720);allocate([7,1,0,0,19,0,0,0,211,1,0,0,212,1,0,0,213,1,0,0,252,255,255,255,152,8,0,0,8,1,0,0,20,0,0,0,214,1,0,0,215,1,0,0,216,1,0,0,0,0,0,0,168,8,0,0,94,1,0,0,95,1,0,0,96,1,0,0,97,1,0,0,108,1,0,0,109,1,0,0,110,1,0,0,9,1,0,0,20,0,0,0,189,1,0,0,217,1,0,0,190,1,0,0,191,1,0,0,192,1,0,0,111,1,0,0,222,0,0,0,112,1,0,0,52,0,0,0,17,0,0,0,53,0,0,0,98,1,0,0,21,0,0,0,6,0,0,0,21,0,0,0,218,1,0,0,219,1,0,0,220,1,0,0,10,1,0,0,252,255,255,255,168,8,0,0,11,1,0,0,22,0,0,0,221,1,0,0,222,1,0,0,223,1,0,0,120,97,0,0,128,97,0,0,144,97,0,0,168,97,0,0,200,97,0,0,240,97,0,0,32,98,0,0,152,96,0,0,160,96,0,0,176,96,0,0,200,96,0,0,232,96,0,0,16,97,0,0,64,97,0,0,1,0,0,0,3,0,0,0,4,0,0,0,6,0,0,0,7,0,0,0,12,0,0,0,13,0,0,0,16,0,0,0,19,0,0,0,25,0,0,0,27,0,0,0,33,0,0,0,37,0,0,0,42,0,0,0,48,0,0,0,52,0,0,0,61,0,0,0,70,0,0,0,73,0,0,0,79,0,0,0,248,76,0,0,0,77,0,0,24,77,0,0,56,77,0,0,104,77,0,0,160,77,0,0,0,78,0,0,104,78,0,0,232,78,0,0,128,79,0,0,72,80,0,0,32,81,0,0,40,82,0,0,80,83,0,0,160,84,0,0,32,86,0,0,192,87,0,0,168,89,0,0,216,91,0,0,32,94,0,0,88,57,0,0,96,57,0,0,120,57,0,0,152,57,0,0,200,57,0,0,0,58,0,0,96,58,0,0,200,58,0,0,72,59,0,0,224,59,0,0,168,60,0,0,128,61,0,0,136,62,0,0,176,63,0,0,0,65,0,0,128,66,0,0,32,68,0,0,8,70,0,0,56,72,0,0,128,74,0,0,184,37,0,0,192,37,0,0,216,37,0,0,248,37,0,0,40,38,0,0,96,38,0,0,192,38,0,0,40,39,0,0,168,39,0,0,64,40,0,0,8,41,0,0,224,41,0,0,232,42,0,0,16,44,0,0,96,45,0,0,224,46,0,0,128,48,0,0,104,50,0,0,152,52,0,0,224,54,0,0,24,18,0,0,32,18,0,0,56,18,0,0,88,18,0,0,136,18,0,0,192,18,0,0,32,19,0,0,136,19,0,0,8,20,0,0,160,20,0,0,104,21,0,0,64,22,0,0,72,23,0,0,112,24,0,0,192,25,0,0,64,27,0,0,224,28,0,0,200,30,0,0,248,32,0,0,64,35,0,0,1,0,0,0,4,0,0,0,5,0,0,0,11,0,0,0,15,0,0,0,24,0,0,0,56,16,0,0,64,16,0,0,96,16,0,0,136,16,0,0,224,16,0,0,88,17,0,0,88,14,0,0,96,14,0,0,128,14,0,0,168,14,0,0,0,15,0,0,120,15,0,0,120,12,0,0,128,12,0,0,160,12,0,0,200,12,0,0,32,13,0,0,152,13,0,0,152,10,0,0,160,10,0,0,192,10,0,0,232,10,0,0,64,11,0,0,184,11,0,0,184,8,0,0,192,8,0,0,224,8,0,0,8,9,0,0,96,9,0,0,216,9,0,0,117,2,0,0,118,2,0,0,116,2,0,0,0,0,0,0,0,0,0,0,88,98,0,0,19,0,0,0,99,1,0,0,100,1,0,0,101,1,0,0,113,1,0,0,114,1,0,0,115,1,0,0,12,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,104,98,0,0,102,1,0,0,103,1,0,0,116,1,0,0,0,0,0,0,0,0,0,0,128,98,0,0,104,1,0,0,105,1,0,0,117,1,0,0,0,0,0,0,152,98,0,0,106,1,0,0,107,1,0,0,118,1,0,0,0,0,0,0,184,98,0,0,108,1,0,0,109,1,0,0,110,1,0,0,111,1,0,0,16,0,0,0,18,0,0,0,158,0,0,0,13,1,0,0,0,0,0,0,232,98,0,0,108,1,0,0,112,1,0,0,110,1,0,0,111,1,0,0,16,0,0,0,19,0,0,0,159,0,0,0,14,1,0,0,0,0,0,0,248,98,0,0,108,1,0,0,113,1,0,0,110,1,0,0,111,1,0,0,16,0,0,0,20,0,0,0,160,0,0,0,15,1,0,0,0,0,0,0,0,0,0,0,133,44,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,192,3,0,0,192,4,0,0,192,5,0,0,192,6,0,0,192,7,0,0,192,8,0,0,192,9,0,0,192,10,0,0,192,11,0,0,192,12,0,0,192,13,0,0,192,14,0,0,192,15,0,0,192,16,0,0,192,17,0,0,192,18,0,0,192,19,0,0,192,20,0,0,192,21,0,0,192,22,0,0,192,23,0,0,192,24,0,0,192,25,0,0,192,26,0,0,192,27,0,0,192,28,0,0,192,29,0,0,192,30,0,0,192,31,0,0,192,0,0,0,179,1,0,0,195,2,0,0,195,3,0,0,195,4,0,0,195,5,0,0,195,6,0,0,195,7,0,0,195,8,0,0,195,9,0,0,195,10,0,0,195,11,0,0,195,12,0,0,195,13,0,0,211,14,0,0,195,15,0,0,195,0,0,12,187,1,0,12,195,2,0,12,195,3,0,12,195,4,0,12,211,228,165,0,0,84,166,0,0,196,166,0,0,196,166,0,0,34,207,0,0,52,175,0,0,52,169,0,0,0,0,0,0,10,0,0,0,100,0,0,0,232,3,0,0,16,39,0,0,160,134,1,0,64,66,15,0,128,150,152,0,0,225,245,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,119,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,18,0,0,0,19,0,0,0,39,61,2,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,119,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,0,0,0,0,0,0,0,19,0,0,0,31,57,2,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,119,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,21,0,0,0,19,0,0,0,23,53,2,0,0,4,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,19,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,23,0,0,0,24,0,0,0,25,0,0,0,26,0,0,0,27,0,0,0,28,0,0,0,29,0,0,0,30,0,0,0,31,0,0,0,32,0,0,0,33,0,0,0,34,0,0,0,35,0,0,0,36,0,0,0,37,0,0,0,38,0,0,0,39,0,0,0,40,0,0,0,41,0,0,0,42,0,0,0,43,0,0,0,44,0,0,0,45,0,0,0,46,0,0,0,47,0,0,0,48,0,0,0,49,0,0,0,50,0,0,0,51,0,0,0,52,0,0,0,53,0,0,0,54,0,0,0,55,0,0,0,56,0,0,0,57,0,0,0,58,0,0,0,59,0,0,0,60,0,0,0,61,0,0,0,62,0,0,0,63,0,0,0,64,0,0,0,65,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,69,0,0,0,70,0,0,0,71,0,0,0,72,0,0,0,73,0,0,0,74,0,0,0,75,0,0,0,76,0,0,0,77,0,0,0,78,0,0,0,79,0,0,0,80,0,0,0,81,0,0,0,82,0,0,0,83,0,0,0,84,0,0,0,85,0,0,0,86,0,0,0,87,0,0,0,88,0,0,0,89,0,0,0,90,0,0,0,91,0,0,0,92,0,0,0,93,0,0,0,94,0,0,0,95,0,0,0,96,0,0,0,65,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,69,0,0,0,70,0,0,0,71,0,0,0,72,0,0,0,73,0,0,0,74,0,0,0,75,0,0,0,76,0,0,0,77,0,0,0,78,0,0,0,79,0,0,0,80,0,0,0,81,0,0,0,82,0,0,0,83,0,0,0,84,0,0,0,85,0,0,0,86,0,0,0,87,0,0,0,88,0,0,0,89,0,0,0,90,0,0,0,123,0,0,0,124,0,0,0,125,0,0,0,126,0,0,0,127],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+40960);allocate([1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,19,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,23,0,0,0,24,0,0,0,25,0,0,0,26,0,0,0,27,0,0,0,28,0,0,0,29,0,0,0,30,0,0,0,31,0,0,0,32,0,0,0,33,0,0,0,34,0,0,0,35,0,0,0,36,0,0,0,37,0,0,0,38,0,0,0,39,0,0,0,40,0,0,0,41,0,0,0,42,0,0,0,43,0,0,0,44,0,0,0,45,0,0,0,46,0,0,0,47,0,0,0,48,0,0,0,49,0,0,0,50,0,0,0,51,0,0,0,52,0,0,0,53,0,0,0,54,0,0,0,55,0,0,0,56,0,0,0,57,0,0,0,58,0,0,0,59,0,0,0,60,0,0,0,61,0,0,0,62,0,0,0,63,0,0,0,64,0,0,0,97,0,0,0,98,0,0,0,99,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,103,0,0,0,104,0,0,0,105,0,0,0,106,0,0,0,107,0,0,0,108,0,0,0,109,0,0,0,110,0,0,0,111,0,0,0,112,0,0,0,113,0,0,0,114,0,0,0,115,0,0,0,116,0,0,0,117,0,0,0,118,0,0,0,119,0,0,0,120,0,0,0,121,0,0,0,122,0,0,0,91,0,0,0,92,0,0,0,93,0,0,0,94,0,0,0,95,0,0,0,96,0,0,0,97,0,0,0,98,0,0,0,99,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,103,0,0,0,104,0,0,0,105,0,0,0,106,0,0,0,107,0,0,0,108,0,0,0,109,0,0,0,110,0,0,0,111,0,0,0,112,0,0,0,113,0,0,0,114,0,0,0,115,0,0,0,116,0,0,0,117,0,0,0,118,0,0,0,119,0,0,0,120,0,0,0,121,0,0,0,122,0,0,0,123,0,0,0,124,0,0,0,125,0,0,0,126,0,0,0,127],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+44848);allocate([16,99,0,0,114,1,0,0,115,1,0,0,193,1,0,0,22,0,0,0,21,0,0,0,16,1,0,0,120,1,0,0,121,1,0,0,23,0,0,0,122,1,0,0,123,1,0,0,223,0,0,0,24,0,0,0,224,0,0,0,0,0,0,0,32,99,0,0,114,1,0,0,116,1,0,0,194,1,0,0,22,0,0,0,21,0,0,0,16,1,0,0,124,1,0,0,121,1,0,0,23,0,0,0,125,1,0,0,126,1,0,0,225,0,0,0,25,0,0,0,226,0,0,0,0,0,0,0,48,99,0,0,117,1,0,0,118,1,0,0,195,1,0,0,1,0,0,0,22,0,0,0,17,1,0,0,127,1,0,0,2,0,0,0,2,0,0,0,128,1,0,0,4,0,0,0,227,0,0,0,26,0,0,0,228,0,0,0,0,0,0,0,64,99,0,0,117,1,0,0,119,1,0,0,196,1,0,0,1,0,0,0,22,0,0,0,17,1,0,0,1,0,0,0,2,0,0,0,2,0,0,0,129,1,0,0,130,1,0,0,229,0,0,0,3,0,0,0,230,0,0,0,0,0,0,0,136,99,0,0,117,1,0,0,120,1,0,0,1,0,0,0,1,0,0,0,22,0,0,0,17,1,0,0,1,0,0,0,2,0,0,0,2,0,0,0,128,1,0,0,4,0,0,0,227,0,0,0,3,0,0,0,230,0,0,0,0,0,0,0,144,99,0,0,114,1,0,0,121,1,0,0,197,1,0,0,22,0,0,0,21,0,0,0,16,1,0,0,124,1,0,0,121,1,0,0,23,0,0,0,122,1,0,0,123,1,0,0,223,0,0,0,25,0,0,0,226,0,0,0,8,0,0,0,0,0,0,0,152,99,0,0,122,1,0,0,123,1,0,0,248,255,255,255,248,255,255,255,152,99,0,0,124,1,0,0,125,1,0,0,8,0,0,0,0,0,0,0,176,99,0,0,126,1,0,0,127,1,0,0,248,255,255,255,248,255,255,255,176,99,0,0,128,1,0,0,129,1,0,0,4,0,0,0,0,0,0,0,200,99,0,0,5,0,0,0,6,0,0,0,252,255,255,255,252,255,255,255,200,99,0,0,7,0,0,0,8,0,0,0,4,0,0,0,0,0,0,0,224,99,0,0,130,1,0,0,131,1,0,0,252,255,255,255,252,255,255,255,224,99,0,0,132,1,0,0,133,1,0,0,0,0,0,0,80,99,0,0,134,1,0,0,135,1,0,0,117,1,0,0,0,0,0,0,96,99,0,0,136,1,0,0,137,1,0,0,0,0,0,0,248,99,0,0,138,1,0,0,139,1,0,0,131,1,0,0,224,1,0,0,27,0,0,0,28,0,0,0,225,1,0,0,0,0,0,0,0,0,0,0,240,100,0,0,140,1,0,0,141,1,0,0,142,1,0,0,5,0,0,0,18,1,0,0,29,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,101,0,0,143,1,0,0,144,1,0,0,142,1,0,0,6,0,0,0,19,1,0,0,30,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,80,101,0,0,145,1,0,0,146,1,0,0,142,1,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,101,0,0,147,1,0,0,148,1,0,0,142,1,0,0,12,0,0,0,13,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,19,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,144,101,0,0,149,1,0,0,150,1,0,0,142,1,0,0,7,0,0,0,8,0,0,0,23,0,0,0,9,0,0,0,24,0,0,0,1,0,0,0,2,0,0,0,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,176,101,0,0,151,1,0,0,152,1,0,0,142,1,0,0,11,0,0,0,12,0,0,0,25,0,0,0,13,0,0,0,26,0,0,0,3,0,0,0,4,0,0,0,14,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,208,101,0,0,153,1,0,0,154,1,0,0,142,1,0,0,132,1,0,0,27,0,0,0,28,0,0,0,29,0,0,0,30,0,0,0,31,0,0,0,1,0,0,0,248,255,255,255,208,101,0,0,133,1,0,0,134,1,0,0,135,1,0,0,136,1,0,0,137,1,0,0,138,1,0,0,139,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,248,101,0,0,155,1,0,0,156,1,0,0,142,1,0,0,140,1,0,0,32,0,0,0,33,0,0,0,34,0,0,0,35,0,0,0,36,0,0,0,2,0,0,0,248,255,255,255,248,101,0,0,141,1,0,0,142,1,0,0,143,1,0,0,144,1,0,0,145,1,0,0,146,1,0,0,147,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,32,102,0,0,157,1,0,0,158,1,0,0,142,1,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,102,0,0,159,1,0,0,160,1,0,0,142,1,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,96,102,0,0,161,1,0,0,162,1,0,0,142,1,0,0,148,1,0,0,149,1,0,0,198,1,0,0,199,1,0,0,200,1,0,0,201,1,0,0,150,1,0,0,202,1,0,0,203,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,102,0,0,163,1,0,0,164,1,0,0,142,1,0,0,151,1,0,0,152,1,0,0,204,1,0,0,205,1,0,0,206,1,0,0,207,1,0,0,153,1,0,0,208,1,0,0,209,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,160,102,0,0,165,1,0,0,166,1,0,0,142,1,0,0,154,1,0,0,155,1,0,0,210,1,0,0,211,1,0,0,212,1,0,0,213,1,0,0,156,1,0,0,214,1,0,0,215,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,192,102,0,0,167,1,0,0,168,1,0,0,142,1,0,0,157,1,0,0,158,1,0,0,216,1,0,0,217,1,0,0,218,1,0,0,219,1,0,0,159,1,0,0,220,1,0,0,221,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,224,102,0,0,169,1,0,0,170,1,0,0,142,1,0,0,3,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,103,0,0,171,1,0,0,172,1,0,0,142,1,0,0,5,0,0,0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,32,103,0,0,173,1,0,0,174,1,0,0,142,1,0,0,1,0,0,0,37,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,103,0,0,175,1,0,0,176,1,0,0,142,1,0,0,2,0,0,0,38,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,96,103,0,0,177,1,0,0,178,1,0,0,142,1,0,0,31,0,0,0,23,0,0,0,222,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,103,0,0,179,1,0,0,180,1,0,0,142,1,0,0,32,0,0,0,24,0,0,0,223,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,224,100,0,0,181,1,0,0,182,1,0,0,142,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,101,0,0,183,1,0,0,184,1,0,0,142,1,0,0,231,0,0,0,33,0,0,0,232,0,0,0,34,0,0,0,233,0,0,0,63,0,0,0,35,0,0,0,15,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,192,100,0,0,185,1,0,0,186,1,0,0,142,1,0,0,3,0,0,0,4,0,0,0,16,0,0,0,160,1,0,0,161,1,0,0,17,0,0,0,162,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,101,0,0,187,1,0,0,188,1,0,0,142,1,0,0,163,1,0,0,164,1,0,0,224,1,0,0,225,1,0,0,226,1,0,0,0,0,0,0,64,101,0,0,189,1,0,0,190,1,0,0,142,1,0,0,165,1,0,0,166,1,0,0,227,1,0,0,228,1,0,0,229,1,0,0,0,0,0,0,16,100,0,0,191,1,0,0,192,1,0,0,142,1,0,0,0,0,0,0,32,100,0,0,191,1,0,0,193,1,0,0,142,1,0,0,36,0,0,0,64,0,0,0,65,0,0,0,66,0,0,0,234,0,0,0,37,0,0,0,235,0,0,0,38,0,0,0,236,0,0,0,67,0,0,0,39,0,0,0,18,0,0,0,0,0,0,0,64,100,0,0,191,1,0,0,194,1,0,0,142,1,0,0,5,0,0,0,6,0,0,0,19,0,0,0,167,1,0,0,168,1,0,0,20,0,0,0,169,1,0,0,0,0,0,0,96,100,0,0,191,1,0,0,195,1,0,0,142,1,0,0,7,0,0,0,8,0,0,0,21,0,0,0,170,1,0,0,171,1,0,0,22,0,0,0,172,1,0,0,0,0,0,0,128,100,0,0,191,1,0,0,196,1,0,0,142,1,0,0,9,0,0,0,10,0,0,0,23,0,0,0,173,1,0,0,174,1,0,0,24,0,0,0,175,1,0,0,0,0,0,0,160,100,0,0,191,1,0,0,197,1,0,0,142,1,0,0,9,0,0,0,10,0,0,0,23,0,0,0,173,1,0,0,174,1,0,0,24,0,0,0,175,1,0,0,0,0,0,0,176,100,0,0,191,1,0,0,198,1,0,0,142,1,0,0,9,0,0,0,10,0,0,0,23,0,0,0,173,1,0,0,174,1,0,0,24,0,0,0,175,1,0,0,0,0,0,0,37,0,0,0,72,0,0,0,58,0,0,0,37,0,0,0,77,0,0,0,58,0,0,0,37,0,0,0,83,0,0,0,37,0,0,0,109,0,0,0,47,0,0,0,37,0,0,0,100,0,0,0,47,0,0,0,37,0,0,0,121,0,0,0,37,0,0,0,89,0,0,0,45,0,0,0,37,0,0,0,109,0,0,0,45,0,0,0,37,0,0,0,100,0,0,0,37,0,0,0,73,0,0,0,58,0,0,0,37,0,0,0,77,0,0,0,58,0,0,0,37,0,0,0,83,0,0,0,32,0,0,0,37,0,0,0,112,0,0,0,37,0,0,0,72,0,0,0,58,0,0,0,37,0,0,0,77,0,0,0,37,0,0,0,72,0,0,0,58,0,0,0,37,0,0,0,77,0,0,0,58,0,0,0,37,0,0,0,83,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,116,0,0,0,114,0,0,0,117,0,0,0,101,0,0,0,0,0,0,0,102,0,0,0,97,0,0,0,108,0,0,0,115,0,0,0,101,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,83,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,97,0,0,0,121,0,0,0,0,0,0,0,77,0,0,0,111,0,0,0,110,0,0,0,100,0,0,0,97,0,0,0,121,0,0,0,0,0,0,0,84,0,0,0,117,0,0,0,101,0,0,0,115,0,0,0,100,0,0,0,97,0,0,0,121,0,0,0,0,0,0,0,87,0,0,0,101,0,0,0,100,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,100,0,0,0,97,0,0,0,121,0,0,0,0,0,0,0,84,0,0,0,104,0,0,0,117,0,0,0,114,0,0,0,115,0,0,0,100,0,0,0,97,0,0,0,121,0,0,0,0,0,0,0,70,0,0,0,114,0,0,0,105,0,0,0,100,0,0,0,97,0,0,0,121,0,0,0,0,0,0,0,83,0,0,0,97,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,100,0,0,0,97,0,0,0,121,0,0,0,0,0,0,0,83,0,0,0,117,0,0,0,110,0,0,0,0,0,0,0,77,0,0,0,111,0,0,0,110,0,0,0,0,0,0,0,84,0,0,0,117,0,0,0,101,0,0,0,0,0,0,0,87,0,0,0,101,0,0,0,100,0,0,0,0,0,0,0,84,0,0,0,104,0,0,0,117,0,0,0,0,0,0,0,70,0,0,0,114,0,0,0,105,0,0,0,0,0,0,0,83,0,0,0,97,0,0,0,116,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,74,0,0,0,97,0,0,0,110,0,0,0,117,0,0,0,97,0,0,0,114,0,0,0,121,0,0,0,0,0,0,0,70,0,0,0,101,0,0,0,98,0,0,0,114,0,0,0,117,0,0,0,97,0,0,0,114,0,0,0,121,0,0,0,0,0,0,0,77,0,0,0,97,0,0,0,114,0,0,0,99,0,0,0,104,0,0,0,0,0,0,0,65,0,0,0,112,0,0,0,114,0,0,0,105,0,0,0,108,0,0,0,0,0,0,0,77,0,0,0,97,0,0,0,121,0,0,0,0,0,0,0,74,0,0,0,117,0,0,0,110,0,0,0,101,0,0,0,0,0,0,0,74,0,0,0,117,0,0,0,108,0,0,0,121,0,0,0,0,0,0,0,65,0,0,0,117,0,0,0,103,0,0,0,117,0,0,0,115,0,0,0,116,0,0,0,0,0,0,0,83,0,0,0,101,0,0,0,112,0,0,0,116,0,0,0,101,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,0,0,0,0,79,0,0,0,99,0,0,0,116,0,0,0,111,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,0,0,0,0,78,0,0,0,111,0,0,0,118,0,0,0,101,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,0,0,0,0,68,0,0,0,101,0,0,0,99,0,0,0,101,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,0,0,0,0,74,0,0,0,97,0,0,0,110,0,0,0,0,0,0,0,70,0,0,0,101,0,0,0,98,0,0,0,0,0,0,0,77,0,0,0,97,0,0,0,114,0,0,0,0,0,0,0,65,0,0,0,112,0,0,0,114,0,0,0,0,0,0,0,74,0,0,0,117,0,0,0,110,0,0,0,0,0,0,0,74,0,0,0,117,0,0,0,108,0,0,0,0,0,0,0,65,0,0,0,117,0,0,0,103,0,0,0,0,0,0,0,83,0,0,0,101,0,0,0,112,0,0,0,0,0,0,0,79,0,0,0,99,0,0,0,116,0,0,0,0,0,0,0,78,0,0,0,111,0,0,0,118,0,0,0,0,0,0,0,68,0,0,0,101,0,0,0,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65,0,0,0,77,0,0,0,0,0,0,0,80,0,0,0,77,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,37,0,0,0,109,0,0,0,47,0,0,0,37,0,0,0,100,0,0,0,47,0,0,0,37,0,0,0,121,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,37,0,0,0,72,0,0,0,58,0,0,0,37,0,0,0,77,0,0,0,58,0,0,0,37,0,0,0,83,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,37,0,0,0,97,0,0,0,32,0,0,0,37,0,0,0,98,0,0,0,32,0,0,0,37,0,0,0,100,0,0,0,32,0,0,0,37,0,0,0,72,0,0,0,58,0,0,0,37,0,0,0,77,0,0,0,58,0,0,0,37,0,0,0,83,0,0,0,32,0,0,0,37,0,0,0,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,37,0,0,0,73,0,0,0,58,0,0,0,37,0,0,0,77,0,0,0,58,0,0,0,37,0,0,0,83,0,0,0,32,0,0,0,37,0,0,0,112,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,107,0,0,199,1,0,0,200,1,0,0,117,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,2,0,2,0,0,0,0,0,1,0,0,0,1,0,2,0,1,0,2,0,0,0,2,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,3,32,2,32,2,32,2,32,2,32,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,1,96,4,192,4,192,4,192,4,192,4,192,4,192,4,192,4,192,4,192,4,192,4,192,4,192,4,192,4,192,4,192,8,216,8,216,8,216,8,216,8,216,8,216,8,216,8,216,8,216,8,216,4,192,4,192,4,192,4,192,4,192,4,192,4,192,8,213,8,213,8,213,8,213,8,213,8,213,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,8,197,4,192,4,192,4,192,4,192,4,192,4,192,8,214,8,214,8,214,8,214,8,214,8,214,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,8,198,4,192,4,192,4,192,4,192,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,78,83,116,51,95,95,49,49,57,98,97,115,105,99,95,111,115,116,114,105,110,103,115,116,114,101,97,109,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,78,83,95,57,97,108,108,111,99,97,116,111,114,73,99,69,69,69,69,0,78,83,116,51,95,95,49,49,53,98,97,115,105,99,95,115,116,114,105,110,103,98,117,102,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,78,83,95,57,97,108,108,111,99,97,116,111,114,73,99,69,69,69,69,0,110,111,100,101,0,119,114,111,110,103,32,105,110,116,101,114,112,111,108,97,116,105,111,110,32,116,121,112,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,119,114,97,112,112,101,114,115,47,106,97,118,97,115,99,114,105,112,116,47,46,46,47,67,111,110,116,111,117,114,84,111,77,101,115,104,47,67,111,110,116,111,117,114,84,111,77,101,115,104,46,99,112,112,0,67,111,110,116,111,117,114,84,111,77,101,115,104,77,111,100,117,108,101,0,105,110,100,101,120,32,115,104,111,117,108,100,32,104,97,118,101,32,51,32,99,111,108,117,109,110,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,119,114,97,112,112,101,114,115,47,106,97,118,97,115,99,114,105,112,116,47,46,46,47,73,110,116,101,114,112,70,114,111,109,77,101,115,104,84,111,77,101,115,104,50,100,47,73,110,116,101,114,112,70,114,111,109,77,101,115,104,84,111,77,101,115,104,50,100,46,99,112,112,0,73,110,116,101,114,112,70,114,111,109,77,101,115,104,84,111,77,101,115,104,50,100,77,111,100,117,108,101,0,116,104,101,114,101,32,115,104,111,117,108,100,32,98,101,32,97,116,32,108,101,97,115,116,32,116,104,114,101,101,32,112,111,105,110,116,115,0,118,101,99,116,111,114,115,32,120,32,97,110,100,32,121,32,115,104,111,117,108,100,32,104,97,118,101,32,116,104,101,32,115,97,109,101,32,108,101,110,103,116,104,0,100,97,116,97,32,105,115,32,101,109,112,116,121,0,110,111,32,105,110,116,101,114,112,111,108,97,116,105,111,110,32,114,101,113,117,101,115,116,101,100,0,118,101,99,116,111,114,115,32,120,95,105,110,116,101,114,112,32,97,110,100,32,121,95,105,110,116,101,114,112,32,115,104,111,117,108,100,32,104,97,118,101,32,116,104,101,32,115,97,109,101,32,108,101,110,103,116,104,0,100,101,102,97,117,108,116,95,118,97,108,117,101,0,95,72,65,86,69,95,77,80,73,95,0,95,72,65,86,69,95,80,69,84,83,67,95,77,80,73,95,0,95,72,65,86,69,95,68,65,75,79,84,65,95,0,95,72,65,86,69,95,77,85,77,80,83,95,0,95,72,65,86,69,95,71,83,76,95,0,95,72,65,86,69,95,84,65,79,95,0,95,72,65,86,69,95,77,49,81,78,51,95,0,95,72,65,86,69,95,80,69,84,83,67,95,0,95,80,69,84,83,67,95,77,65,74,79,82,95,0,95,80,69,84,83,67,95,77,65,74,79,82,95,32,110,111,116,32,102,111,117,110,100,32,105,110,32,99,111,110,102,105,103,46,104,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,119,114,97,112,112,101,114,115,47,106,97,118,97,115,99,114,105,112,116,47,46,46,47,73,115,115,109,67,111,110,102,105,103,47,73,115,115,109,67,111,110,102,105,103,46,99,112,112,0,73,115,115,109,67,111,110,102,105,103,77,111,100,117,108,101,0,95,80,69,84,83,67,95,77,73,78,79,82,95,0,95,80,69,84,83,67,95,77,73,78,79,82,95,32,110,111,116,32,102,111,117,110,100,32,105,110,32,99,111,110,102,105,103,46,104,0,95,68,65,75,79,84,65,95,86,69,82,83,73,79,78,95,0,95,68,65,75,79,84,65,95,86,69,82,83,73,79,78,95,32,110,111,116,32,102,111,117,110,100,32,105,110,32,99,111,110,102,105,103,46,104,0,73,83,83,77,95,80,82,69,70,73,88,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,98,117,105,108,100,45,106,115,0,80,65,67,75,65,71,69,95,78,65,77,69,0,80,65,67,75,65,71,69,95,86,69,82,83,73,79,78,0,80,65,67,75,65,71,69,95,85,82,76,0,80,65,67,75,65,71,69,95,66,85,71,82,69,80,79,82,84,0,80,65,67,75,65,71,69,95,66,85,73,76,68,95,68,65,84,69,0,77,111,110,32,65,112,114,32,50,53,32,49,53,58,51,51,58,50,55,32,80,68,84,32,50,48,49,54,0,72,79,83,84,95,79,83,0,100,97,114,119,105,110,49,52,46,53,46,48,0,85,83,69,82,95,78,65,77,69,0,108,97,114,111,117,114,0,72,79,83,84,95,86,69,78,68,79,82,0,97,112,112,108,101,0,72,79,83,84,95,65,82,67,72,0,120,56,54,95,54,52,0,118,97,114,105,97,98,108,101,32,0,65,110,32,117,110,101,120,112,101,99,116,101,100,32,101,114,114,111,114,32,111,99,99,117,114,114,101,100,0,83,116,97,110,100,97,114,100,32,101,120,99,101,112,116,105,111,110,58,32,0,65,110,32,117,110,101,120,112,101,99,116,101,100,32,101,114,114,111,114,32,111,99,99,117,114,114,101,100,32,10,10,0,69,114,114,111,114,58,32,32,79,117,116,32,111,102,32,109,101,109,111,114,121,46,10,0,32,32,80,108,101,97,115,101,32,114,101,112,111,114,116,32,116,104,105,115,32,98,117,103,32,116,111,32,106,114,115,64,99,115,46,98,101,114,107,101,108,101,121,46,101,100,117,10,0,32,32,73,110,99,108,117,100,101,32,116,104,101,32,109,101,115,115,97,103,101,32,97,98,111,118,101,44,32,121,111,117,114,32,105,110,112,117,116,32,100,97,116,97,32,115,101,116,44,32,97,110,100,32,116,104,101,32,101,120,97,99,116,10,0,32,32,32,32,99,111,109,109,97,110,100,32,108,105,110,101,32,121,111,117,32,117,115,101,100,32,116,111,32,114,117,110,32,84,114,105,97,110,103,108,101,46,10,0,69,114,114,111,114,58,32,32,77,97,120,105,109,117,109,32,97,114,101,97,32,109,117,115,116,32,98,101,32,103,114,101,97,116,101,114,32,116,104,97,110,32,122,101,114,111,46,10,0,69,114,114,111,114,58,32,32,89,111,117,32,99,97,110,110,111,116,32,117,115,101,32,116,104,101,32,45,73,32,115,119,105,116,99,104,32,119,104,101,110,32,114,101,102,105,110,105,110,103,32,97,32,116,114,105,97,110,103,117,108,97,116,105,111,110,46,10,0,87,97,114,110,105,110,103,58,32,32,119,101,105,103,104,116,101,100,32,116,114,105,97,110,103,117,108,97,116,105,111,110,115,32,40,45,119,44,32,45,87,41,32,97,114,101,32,105,110,99,111,109,112,97,116,105,98,108,101,10,0,32,32,119,105,116,104,32,80,83,76,71,115,32,40,45,112,41,32,97,110,100,32,109,101,115,104,105,110,103,32,40,45,113,44,32,45,97,44,32,45,117,41,46,32,32,87,101,105,103,104,116,115,32,105,103,110,111,114,101,100,46,10,0,87,97,114,110,105,110,103,58,32,32,45,106,32,97,110,100,32,45,78,32,115,119,105,116,99,104,101,115,32,97,114,101,32,115,111,109,101,119,104,97,116,32,105,110,99,111,109,112,97,116,105,98,108,101,46,10,0,32,32,73,102,32,97,110,121,32,118,101,114,116,105,99,101,115,32,97,114,101,32,106,101,116,116,105,115,111,110,101,100,44,32,121,111,117,32,119,105,108,108,32,110,101,101,100,32,116,104,101,32,111,117,116,112,117,116,10,0,32,32,46,110,111,100,101,32,102,105,108,101,32,116,111,32,114,101,99,111,110,115,116,114,117,99,116,32,116,104,101,32,110,101,119,32,110,111,100,101,32,105,110,100,105,99,101,115,46,0,116,114,105,97,110,103,108,101,32,120,37,108,120,32,119,105,116,104,32,111,114,105,101,110,116,97,116,105,111,110,32,37,100,58,10,0,32,32,32,32,91,48,93,32,61,32,79,117,116,101,114,32,115,112,97,99,101,10,0,32,32,32,32,91,48,93,32,61,32,120,37,108,120,32,32,37,100,10,0,32,32,32,32,91,49,93,32,61,32,79,117,116,101,114,32,115,112,97,99,101,10,0,32,32,32,32,91,49,93,32,61,32,120,37,108,120,32,32,37,100,10,0,32,32,32,32,91,50,93,32,61,32,79,117,116,101,114,32,115,112,97,99,101,10,0,32,32,32,32,91,50,93,32,61,32,120,37,108,120,32,32,37,100,10,0,32,32,32,32,79,114,105,103,105,110,91,37,100,93,32,61,32,78,85,76,76,10,0,32,32,32,32,79,114,105,103,105,110,91,37,100,93,32,61,32,120,37,108,120,32,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,32,32,68,101,115,116,32,32,91,37,100,93,32,61,32,78,85,76,76,10,0,32,32,32,32,68,101,115,116,32,32,91,37,100,93,32,61,32,120,37,108,120,32,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,32,32,65,112,101,120,32,32,91,37,100,93,32,61,32,78,85,76,76,10,0,32,32,32,32,65,112,101,120,32,32,91,37,100,93,32,61,32,120,37,108,120,32,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,32,32,91,54,93,32,61,32,120,37,108,120,32,32,37,100,10,0,32,32,32,32,91,55,93,32,61,32,120,37,108,120,32,32,37,100,10,0,32,32,32,32,91,56,93,32,61,32,120,37,108,120,32,32,37,100,10,0,32,32,32,32,65,114,101,97,32,99,111,110,115,116,114,97,105,110,116,58,32,32,37,46,52,103,10,0,115,117,98,115,101,103,109,101,110,116,32,120,37,108,120,32,119,105,116,104,32,111,114,105,101,110,116,97,116,105,111,110,32,37,100,32,97,110,100,32,109,97,114,107,32,37,100,58,10,0,32,32,32,32,91,48,93,32,61,32,78,111,32,115,117,98,115,101,103,109,101,110,116,10,0,32,32,32,32,91,49,93,32,61,32,78,111,32,115,117,98,115,101,103,109,101,110,116,10,0,32,32,32,32,91,54,93,32,61,32,79,117,116,101,114,32,115,112,97,99,101,10,0,32,32,32,32,91,55,93,32,61,32,79,117,116,101,114,32,115,112,97,99,101,10,0,32,32,32,32,83,101,103,109,101,110,116,32,111,114,105,103,105,110,91,37,100,93,32,61,32,78,85,76,76,10,0,32,32,32,32,83,101,103,109,101,110,116,32,111,114,105,103,105,110,91,37,100,93,32,61,32,120,37,108,120,32,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,32,32,83,101,103,109,101,110,116,32,100,101,115,116,32,32,91,37,100,93,32,61,32,78,85,76,76,10,0,32,32,32,32,83,101,103,109,101,110,116,32,100,101,115,116,32,32,91,37,100,93,32,61,32,120,37,108,120,32,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,67,104,101,99,107,105,110,103,32,99,111,110,115,105,115,116,101,110,99,121,32,111,102,32,109,101,115,104,46,46,46,10,0,32,32,33,33,32,33,33,32,73,110,118,101,114,116,101,100,32,0,32,32,33,33,32,33,33,32,65,115,121,109,109,101,116,114,105,99,32,116,114,105,97,110,103,108,101,45,116,114,105,97,110,103,108,101,32,98,111,110,100,58,10,0,32,32,32,40,82,105,103,104,116,32,116,114,105,97,110,103,108,101,44,32,119,114,111,110,103,32,111,114,105,101,110,116,97,116,105,111,110,41,10,0,32,32,32,32,70,105,114,115,116,32,0,32,32,32,32,83,101,99,111,110,100,32,40,110,111,110,114,101,99,105,112,114,111,99,97,116,105,110,103,41,32,0,32,32,33,33,32,33,33,32,77,105,115,109,97,116,99,104,101,100,32,101,100,103,101,32,99,111,111,114,100,105,110,97,116,101,115,32,98,101,116,119,101,101,110,32,116,119,111,32,116,114,105,97,110,103,108,101,115,58,10,0,32,32,32,32,70,105,114,115,116,32,109,105,115,109,97,116,99,104,101,100,32,0,32,32,32,32,83,101,99,111,110,100,32,109,105,115,109,97,116,99,104,101,100,32,0,32,32,73,110,32,109,121,32,115,116,117,100,105,101,100,32,111,112,105,110,105,111,110,44,32,116,104,101,32,109,101,115,104,32,97,112,112,101,97,114,115,32,116,111,32,98,101,32,99,111,110,115,105,115,116,101,110,116,46,10,0,32,32,33,33,32,33,33,32,33,33,32,33,33,32,80,114,101,99,105,115,101,108,121,32,111,110,101,32,102,101,115,116,101,114,105,110,103,32,119,111,117,110,100,32,100,105,115,99,111,118,101,114,101,100,46,10,0,32,32,33,33,32,33,33,32,33,33,32,33,33,32,37,100,32,97,98,111,109,105,110,97,116,105,111,110,115,32,119,105,116,110,101,115,115,101,100,46,10,0,32,32,67,104,101,99,107,105,110,103,32,68,101,108,97,117,110,97,121,32,112,114,111,112,101,114,116,121,32,111,102,32,109,101,115,104,46,46,46,10,0,32,32,33,33,32,33,33,32,78,111,110,45,68,101,108,97,117,110,97,121,32,112,97,105,114,32,111,102,32,116,114,105,97,110,103,108,101,115,58,10,0,32,32,32,32,70,105,114,115,116,32,110,111,110,45,68,101,108,97,117,110,97,121,32,0,32,32,32,32,83,101,99,111,110,100,32,110,111,110,45,68,101,108,97,117,110,97,121,32,0,32,32,33,33,32,33,33,32,78,111,110,45,114,101,103,117,108,97,114,32,112,97,105,114,32,111,102,32,116,114,105,97,110,103,108,101,115,58,10,0,32,32,32,32,70,105,114,115,116,32,110,111,110,45,114,101,103,117,108,97,114,32,0,32,32,32,32,83,101,99,111,110,100,32,110,111,110,45,114,101,103,117,108,97,114,32,0,32,32,66,121,32,118,105,114,116,117,101,32,111,102,32,109,121,32,112,101,114,99,101,112,116,105,118,101,32,105,110,116,101,108,108,105,103,101,110,99,101,44,32,73,32,100,101,99,108,97,114,101,32,116,104,101,32,109,101,115,104,32,68,101,108,97,117,110,97,121,46,10,0,32,32,33,33,32,33,33,32,33,33,32,33,33,32,80,114,101,99,105,115,101,108,121,32,111,110,101,32,116,101,114,114,105,102,121,105,110,103,32,116,114,97,110,115,103,114,101,115,115,105,111,110,32,105,100,101,110,116,105,102,105,101,100,46,10,0,32,32,33,33,32,33,33,32,33,33,32,33,33,32,37,100,32,111,98,115,99,101,110,105,116,105,101,115,32,118,105,101,119,101,100,32,119,105,116,104,32,104,111,114,114,111,114,46,10,0,32,32,81,117,101,117,101,105,110,103,32,98,97,100,32,116,114,105,97,110,103,108,101,58,10,0,32,32,32,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,81,117,101,117,101,105,110,103,32,101,110,99,114,111,97,99,104,101,100,32,115,117,98,115,101,103,109,101,110,116,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,32,32,32,32,67,111,110,115,116,114,117,99,116,105,110,103,32,109,97,112,112,105,110,103,32,102,114,111,109,32,118,101,114,116,105,99,101,115,32,116,111,32,116,114,105,97,110,103,108,101,115,46,10,0,32,32,83,101,97,114,99,104,105,110,103,32,102,111,114,32,112,111,105,110,116,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,32,32,32,32,65,116,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,82,97,110,100,111,109,108,121,32,115,97,109,112,108,105,110,103,32,102,111,114,32,97,32,116,114,105,97,110,103,108,101,32,110,101,97,114,32,112,111,105,110,116,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,32,32,32,32,66,111,117,110,100,97,114,121,32,116,114,105,97,110,103,108,101,32,104,97,115,32,111,114,105,103,105,110,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,32,32,32,32,67,104,111,111,115,105,110,103,32,114,101,99,101,110,116,32,116,114,105,97,110,103,108,101,32,119,105,116,104,32,111,114,105,103,105,110,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,32,32,32,32,67,104,111,111,115,105,110,103,32,116,114,105,97,110,103,108,101,32,119,105,116,104,32,111,114,105,103,105,110,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,32,32,73,110,115,101,114,116,105,110,103,32,110,101,119,32,0,32,32,69,100,103,101,32,102,108,105,112,32,114,101,115,117,108,116,115,32,105,110,32,108,101,102,116,32,0,32,32,97,110,100,32,114,105,103,104,116,32,0,32,32,69,100,103,101],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+47400);allocate([32,117,110,102,108,105,112,32,114,101,115,117,108,116,115,32,105,110,32,108,101,102,116,32,0,32,32,73,110,115,101,114,116,105,110,103,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,32,32,85,112,100,97,116,105,110,103,32,98,111,116,116,111,109,32,108,101,102,116,32,0,32,32,85,112,100,97,116,105,110,103,32,116,111,112,32,108,101,102,116,32,0,32,32,67,114,101,97,116,105,110,103,32,116,111,112,32,114,105,103,104,116,32,0,32,32,67,114,101,97,116,105,110,103,32,98,111,116,116,111,109,32,114,105,103,104,116,32,0,32,32,85,112,100,97,116,105,110,103,32,116,111,112,32,0,32,32,67,114,101,97,116,105,110,103,32,108,101,102,116,32,0,32,32,67,114,101,97,116,105,110,103,32,114,105,103,104,116,32,0,32,32,84,114,105,97,110,103,117,108,97,116,105,110,103,32,105,110,116,101,114,105,111,114,32,112,111,108,121,103,111,110,32,97,116,32,101,100,103,101,10,0,32,32,32,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,32,32,67,111,110,110,101,99,116,105,110,103,32,101,100,103,101,32,116,111,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,68,101,108,101,116,105,110,103,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,32,32,67,114,101,97,116,105,110,103,32,98,97,115,101,32,98,111,117,110,100,105,110,103,32,0,32,32,67,114,101,97,116,105,110,103,32,116,111,112,32,98,111,117,110,100,105,110,103,32,0,32,32,67,111,110,110,101,99,116,105,110,103,32,0,32,32,84,114,105,97,110,103,117,108,97,116,105,110,103,32,37,100,32,118,101,114,116,105,99,101,115,46,10,0,32,32,67,114,101,97,116,105,110,103,32,0,32,32,74,111,105,110,105,110,103,32,116,114,105,97,110,103,117,108,97,116,105,111,110,115,32,119,105,116,104,32,37,100,32,97,110,100,32,37,100,32,118,101,114,116,105,99,101,115,46,10,0,32,32,82,101,109,111,118,105,110,103,32,103,104,111,115,116,32,116,114,105,97,110,103,108,101,115,46,10,0,32,32,83,111,114,116,105,110,103,32,118,101,114,116,105,99,101,115,46,10,0,87,97,114,110,105,110,103,58,32,32,65,32,100,117,112,108,105,99,97,116,101,32,118,101,114,116,101,120,32,97,116,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,97,112,112,101,97,114,101,100,32,97,110,100,32,119,97,115,32,105,103,110,111,114,101,100,46,10,0,32,32,70,111,114,109,105,110,103,32,116,114,105,97,110,103,117,108,97,116,105,111,110,46,10,0,32,32,67,114,101,97,116,105,110,103,32,116,114,105,97,110,103,117,108,97,114,32,98,111,117,110,100,105,110,103,32,98,111,120,46,10,0,32,32,82,101,109,111,118,105,110,103,32,116,114,105,97,110,103,117,108,97,114,32,98,111,117,110,100,105,110,103,32,98,111,120,46,10,0,32,32,73,110,99,114,101,109,101,110,116,97,108,108,121,32,105,110,115,101,114,116,105,110,103,32,118,101,114,116,105,99,101,115,46,10,0,32,32,80,108,97,99,105,110,103,32,118,101,114,116,105,99,101,115,32,105,110,32,101,118,101,110,116,32,104,101,97,112,46,10,0,69,114,114,111,114,58,32,32,73,110,112,117,116,32,118,101,114,116,105,99,101,115,32,97,114,101,32,97,108,108,32,105,100,101,110,116,105,99,97,108,46,10,0,67,111,110,115,116,114,117,99,116,105,110,103,32,68,101,108,97,117,110,97,121,32,116,114,105,97,110,103,117,108,97,116,105,111,110,32,0,98,121,32,105,110,99,114,101,109,101,110,116,97,108,32,109,101,116,104,111,100,46,10,0,98,121,32,115,119,101,101,112,108,105,110,101,32,109,101,116,104,111,100,46,10,0,98,121,32,100,105,118,105,100,101,45,97,110,100,45,99,111,110,113,117,101,114,32,109,101,116,104,111,100,46,10,0,69,114,114,111,114,58,32,32,84,114,105,97,110,103,108,101,115,32,109,117,115,116,32,104,97,118,101,32,97,116,32,108,101,97,115,116,32,51,32,118,101,114,116,105,99,101,115,46,10,0,82,101,99,111,110,115,116,114,117,99,116,105,110,103,32,109,101,115,104,46,10,0,32,32,65,115,115,101,109,98,108,105,110,103,32,116,114,105,97,110,103,108,101,115,46,10,0,69,114,114,111,114,58,32,32,84,114,105,97,110,103,108,101,32,37,108,100,32,104,97,115,32,97,110,32,105,110,118,97,108,105,100,32,118,101,114,116,101,120,32,105,110,100,101,120,46,10,0,32,32,77,97,114,107,105,110,103,32,115,101,103,109,101,110,116,115,32,105,110,32,116,114,105,97,110,103,117,108,97,116,105,111,110,46,10,0,69,114,114,111,114,58,32,32,83,101,103,109,101,110,116,32,37,108,100,32,104,97,115,32,97,110,32,105,110,118,97,108,105,100,32,118,101,114,116,101,120,32,105,110,100,101,120,46,10,0,73,110,116,101,114,110,97,108,32,101,114,114,111,114,32,105,110,32,102,105,110,100,100,105,114,101,99,116,105,111,110,40,41,58,32,32,85,110,97,98,108,101,32,116,111,32,102,105,110,100,32,97,10,0,32,32,116,114,105,97,110,103,108,101,32,108,101,97,100,105,110,103,32,102,114,111,109,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,116,111,0,32,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,73,110,116,101,114,110,97,108,32,101,114,114,111,114,32,105,110,32,115,101,103,109,101,110,116,105,110,116,101,114,115,101,99,116,105,111,110,40,41,58,0,32,32,65,116,116,101,109,112,116,32,116,111,32,102,105,110,100,32,105,110,116,101,114,115,101,99,116,105,111,110,32,111,102,32,112,97,114,97,108,108,101,108,32,115,101,103,109,101,110,116,115,46,10,0,32,32,83,112,108,105,116,116,105,110,103,32,115,117,98,115,101,103,109,101,110,116,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,97,116,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,73,110,116,101,114,110,97,108,32,101,114,114,111,114,32,105,110,32,115,101,103,109,101,110,116,105,110,116,101,114,115,101,99,116,105,111,110,40,41,58,10,0,32,32,70,97,105,108,117,114,101,32,116,111,32,115,112,108,105,116,32,97,32,115,101,103,109,101,110,116,46,10,0,32,32,84,111,112,111,108,111,103,105,99,97,108,32,105,110,99,111,110,115,105,115,116,101,110,99,121,32,97,102,116,101,114,32,115,112,108,105,116,116,105,110,103,32,97,32,115,101,103,109,101,110,116,46,10,0,70,111,114,99,105,110,103,32,115,101,103,109,101,110,116,32,105,110,116,111,32,116,114,105,97,110,103,117,108,97,116,105,111,110,32,98,121,32,114,101,99,117,114,115,105,118,101,32,115,112,108,105,116,116,105,110,103,58,10,0,32,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,83,101,103,109,101,110,116,32,105,110,116,101,114,115,101,99,116,115,32,101,120,105,115,116,105,110,103,32,118,101,114,116,101,120,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,32,32,84,119,111,32,115,101,103,109,101,110,116,115,32,105,110,116,101,114,115,101,99,116,32,97,116,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,73,110,116,101,114,110,97,108,32,101,114,114,111,114,32,105,110,32,99,111,110,102,111,114,109,105,110,103,101,100,103,101,40,41,58,10,0,32,32,67,111,110,110,101,99,116,105,110,103,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,116,111,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,73,110,116,101,114,110,97,108,32,101,114,114,111,114,32,105,110,32,105,110,115,101,114,116,115,101,103,109,101,110,116,40,41,58,32,32,85,110,97,98,108,101,32,116,111,32,108,111,99,97,116,101,32,80,83,76,71,32,118,101,114,116,101,120,10,0,32,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,105,110,32,116,114,105,97,110,103,117,108,97,116,105,111,110,46,10,0,82,101,99,111,118,101,114,105,110,103,32,115,101,103,109,101,110,116,115,32,105,110,32,68,101,108,97,117,110,97,121,32,116,114,105,97,110,103,117,108,97,116,105,111,110,46,10,0,105,110,112,117,116,0,32,32,82,101,99,111,118,101,114,105,110,103,32,80,83,76,71,32,115,101,103,109,101,110,116,115,46,10,0,87,97,114,110,105,110,103,58,32,32,73,110,118,97,108,105,100,32,102,105,114,115,116,32,101,110,100,112,111,105,110,116,32,111,102,32,115,101,103,109,101,110,116,32,37,100,32,105,110,32,37,115,46,10,0,87,97,114,110,105,110,103,58,32,32,73,110,118,97,108,105,100,32,115,101,99,111,110,100,32,101,110,100,112,111,105,110,116,32,111,102,32,115,101,103,109,101,110,116,32,37,100,32,105,110,32,37,115,46,10,0,87,97,114,110,105,110,103,58,32,32,69,110,100,112,111,105,110,116,115,32,111,102,32,115,101,103,109,101,110,116,32,37,100,32,97,114,101,32,99,111,105,110,99,105,100,101,110,116,32,105,110,32,37,115,46,10,0,32,32,69,110,99,108,111,115,105,110,103,32,99,111,110,118,101,120,32,104,117,108,108,32,119,105,116,104,32,115,101,103,109,101,110,116,115,46,10,0,32,32,77,97,114,107,105,110,103,32,99,111,110,99,97,118,105,116,105,101,115,32,40,101,120,116,101,114,110,97,108,32,116,114,105,97,110,103,108,101,115,41,32,102,111,114,32,101,108,105,109,105,110,97,116,105,111,110,46,10,0,32,32,77,97,114,107,105,110,103,32,110,101,105,103,104,98,111,114,115,32,111,102,32,109,97,114,107,101,100,32,116,114,105,97,110,103,108,101,115,46,10,0,32,32,32,32,67,104,101,99,107,105,110,103,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,32,32,77,97,114,107,105,110,103,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,68,101,108,101,116,105,110,103,32,109,97,114,107,101,100,32,116,114,105,97,110,103,108,101,115,46,10,0,32,32,32,32,68,101,108,101,116,105,110,103,32,118,101,114,116,101,120,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,10,0,32,32,85,110,109,97,114,107,105,110,103,32,109,97,114,107,101,100,32,116,114,105,97,110,103,108,101,115,46,10,0,82,101,109,111,118,105,110,103,32,117,110,119,97,110,116,101,100,32,116,114,105,97,110,103,108,101,115,46,10,0,32,32,77,97,114,107,105,110,103,32,104,111,108,101,115,32,102,111,114,32,101,108,105,109,105,110,97,116,105,111,110,46,10,0,83,112,114,101,97,100,105,110,103,32,114,101,103,105,111,110,97,108,32,97,116,116,114,105,98,117,116,101,115,32,97,110,100,32,97,114,101,97,32,99,111,110,115,116,114,97,105,110,116,115,46,10,0,83,112,114,101,97,100,105,110,103,32,114,101,103,105,111,110,97,108,32,97,116,116,114,105,98,117,116,101,115,46,10,0,83,112,114,101,97,100,105,110,103,32,114,101,103,105,111,110,97,108,32,97,114,101,97,32,99,111,110,115,116,114,97,105,110,116,115,46,10,0,84,114,121,32,105,110,99,114,101,97,115,105,110,103,32,116,104,101,32,97,114,101,97,32,99,114,105,116,101,114,105,111,110,32,97,110,100,47,111,114,32,114,101,100,117,99,105,110,103,32,116,104,101,32,109,105,110,105,109,117,109,10,0,32,32,97,108,108,111,119,97,98,108,101,32,97,110,103,108,101,32,115,111,32,116,104,97,116,32,116,105,110,121,32,116,114,105,97,110,103,108,101,115,32,97,114,101,32,110,111,116,32,99,114,101,97,116,101,100,46,10,0,69,114,114,111,114,58,32,32,82,97,110,32,111,117,116,32,111,102,32,112,114,101,99,105,115,105,111,110,32,97,116,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,73,32,97,116,116,101,109,112,116,101,100,32,116,111,32,115,112,108,105,116,32,97,32,115,101,103,109,101,110,116,32,116,111,32,97,32,115,109,97,108,108,101,114,32,115,105,122,101,32,116,104,97,110,10,0,32,32,99,97,110,32,98,101,32,97,99,99,111,109,109,111,100,97,116,101,100,32,98,121,32,116,104,101,32,102,105,110,105,116,101,32,112,114,101,99,105,115,105,111,110,32,111,102,10,0,32,32,102,108,111,97,116,105,110,103,32,112,111,105,110,116,32,97,114,105,116,104,109,101,116,105,99,46,10,0,73,110,116,101,114,110,97,108,32,101,114,114,111,114,32,105,110,32,115,112,108,105,116,101,110,99,115,101,103,115,40,41,58,10,0,32,32,77,97,107,105,110,103,32,97,32,108,105,115,116,32,111,102,32,98,97,100,32,116,114,105,97,110,103,108,101,115,46,10,0,32,32,83,112,108,105,116,116,105,110,103,32,116,104,105,115,32,116,114,105,97,110,103,108,101,32,97,116,32,105,116,115,32,99,105,114,99,117,109,99,101,110,116,101,114,58,10,0,87,97,114,110,105,110,103,58,32,32,78,101,119,32,118,101,114,116,101,120,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,32,102,97,108,108,115,32,111,110,32,101,120,105,115,116,105,110,103,32,118,101,114,116,101,120,46,10,0,32,32,82,101,106,101,99,116,105,110,103,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,32,32,84,104,101,32,110,101,119,32,118,101,114,116,101,120,32,105,115,32,97,116,32,116,104,101,32,99,105,114,99,117,109,99,101,110,116,101,114,32,111,102,32,116,114,105,97,110,103,108,101,10,0,84,104,105,115,32,112,114,111,98,97,98,108,121,32,109,101,97,110,115,32,116,104,97,116,32,73,32,97,109,32,116,114,121,105,110,103,32,116,111,32,114,101,102,105,110,101,32,116,114,105,97,110,103,108,101,115,10,0,32,32,116,111,32,97,32,115,109,97,108,108,101,114,32,115,105,122,101,32,116,104,97,110,32,99,97,110,32,98,101,32,97,99,99,111,109,109,111,100,97,116,101,100,32,98,121,32,116,104,101,32,102,105,110,105,116,101,10,0,32,32,112,114,101,99,105,115,105,111,110,32,111,102,32,102,108,111,97,116,105,110,103,32,112,111,105,110,116,32,97,114,105,116,104,109,101,116,105,99,46,32,32,40,89,111,117,32,99,97,110,32,98,101,10,0,32,32,115,117,114,101,32,111,102,32,116,104,105,115,32,105,102,32,73,32,102,97,105,108,32,116,111,32,116,101,114,109,105,110,97,116,101,46,41,10,0,65,100,100,105,110,103,32,83,116,101,105,110,101,114,32,112,111,105,110,116,115,32,116,111,32,101,110,102,111,114,99,101,32,113,117,97,108,105,116,121,46,10,0,32,32,76,111,111,107,105,110,103,32,102,111,114,32,101,110,99,114,111,97,99,104,101,100,32,115,117,98,115,101,103,109,101,110,116,115,46,10,0,32,32,83,112,108,105,116,116,105,110,103,32,101,110,99,114,111,97,99,104,101,100,32,115,117,98,115,101,103,109,101,110,116,115,46,10,0,32,32,83,112,108,105,116,116,105,110,103,32,98,97,100,32,116,114,105,97,110,103,108,101,115,46,10,0,10,87,97,114,110,105,110,103,58,32,32,73,32,114,97,110,32,111,117,116,32,111,102,32,83,116,101,105,110,101,114,32,112,111,105,110,116,115,44,32,98,117,116,32,116,104,101,32,109,101,115,104,32,104,97,115,10,0,32,32,111,110,101,32,101,110,99,114,111,97,99,104,101,100,32,115,117,98,115,101,103,109,101,110,116,44,32,97,110,100,32,116,104,101,114,101,102,111,114,101,32,109,105,103,104,116,32,110,111,116,32,98,101,32,116,114,117,108,121,10,0,32,32,37,108,100,32,101,110,99,114,111,97,99,104,101,100,32,115,117,98,115,101,103,109,101,110,116,115,44,32,97,110,100,32,116,104,101,114,101,102,111,114,101,32,109,105,103,104,116,32,110,111,116,32,98,101,32,116,114,117,108,121,10,0,32,32,68,101,108,97,117,110,97,121,46,32,32,73,102,32,116,104,101,32,68,101,108,97,117,110,97,121,32,112,114,111,112,101,114,116,121,32,105,115,32,105,109,112,111,114,116,97,110,116,32,116,111,32,121,111,117,44,10,0,32,32,116,114,121,32,105,110,99,114,101,97,115,105,110,103,32,116,104,101,32,110,117,109,98,101,114,32,111,102,32,83,116,101,105,110,101,114,32,112,111,105,110,116,115,32,40,99,111,110,116,114,111,108,108,101,100,32,98,121,10,0,32,32,116,104,101,32,45,83,32,115,119,105,116,99,104,41,32,115,108,105,103,104,116,108,121,32,97,110,100,32,116,114,121,32,97,103,97,105,110,46,10,10,0,65,100,100,105,110,103,32,118,101,114,116,105,99,101,115,32,102,111,114,32,115,101,99,111,110,100,45,111,114,100,101,114,32,116,114,105,97,110,103,108,101,115,46,10,0,32,32,67,114,101,97,116,105,110,103,32,40,37,46,49,50,103,44,32,37,46,49,50,103,41,46,10,0,69,114,114,111,114,58,32,32,73,110,112,117,116,32,109,117,115,116,32,104,97,118,101,32,97,116,32,108,101,97,115,116,32,116,104,114,101,101,32,105,110,112,117,116,32,118,101,114,116,105,99,101,115,46,10,0,87,114,105,116,105,110,103,32,118,101,114,116,105,99,101,115,46,10,0,87,114,105,116,105,110,103,32,116,114,105,97,110,103,108,101,115,46,10,0,87,114,105,116,105,110,103,32,115,101,103,109,101,110,116,115,46,10,0,87,114,105,116,105,110,103,32,101,100,103,101,115,46,10,0,87,114,105,116,105,110,103,32,86,111,114,111,110,111,105,32,118,101,114,116,105,99,101,115,46,10,0,87,114,105,116,105,110,103,32,86,111,114,111,110,111,105,32,101,100,103,101,115,46,10,0,87,114,105,116,105,110,103,32,110,101,105,103,104,98,111,114,115,46,10,0,77,101,115,104,32,113,117,97,108,105,116,121,32,115,116,97,116,105,115,116,105,99,115,58,10,10,0,32,32,83,109,97,108,108,101,115,116,32,97,114,101,97,58,32,37,49,54,46,53,103,32,32,32,124,32,32,76,97,114,103,101,115,116,32,97,114,101,97,58,32,37,49,54,46,53,103,10,0,32,32,83,104,111,114,116,101,115,116,32,101,100,103,101,58,32,37,49,54,46,53,103,32,32,32,124,32,32,76,111,110,103,101,115,116,32,101,100,103,101,58,32,37,49,54,46,53,103,10,0,32,32,83,104,111,114,116,101,115,116,32,97,108,116,105,116,117,100,101,58,32,37,49,50,46,53,103,32,32,32,124,32,32,76,97,114,103,101,115,116,32,97,115,112,101,99,116,32,114,97,116,105,111,58,32,37,56,46,53,103,10,10,0,32,32,84,114,105,97,110,103,108,101,32,97,115,112,101,99,116,32,114,97,116,105,111,32,104,105,115,116,111,103,114,97,109,58,10,0,32,32,49,46,49,53,52,55,32,45,32,37,45,54,46,54,103,32,32,32,32,58,32,32,37,56,100,32,32,32,32,124,32,37,54,46,54,103,32,45,32,37,45,54,46,54,103,32,32,32,32,32,58,32,32,37,56,100,10,0,32,32,37,54,46,54,103,32,45,32,37,45,54,46,54,103,32,32,32,32,58,32,32,37,56,100,32,32,32,32,124,32,37,54,46,54,103,32,45,32,37,45,54,46,54,103,32,32,32,32,32,58,32,32,37,56,100,10,0,32,32,37,54,46,54,103,32,45,32,37,45,54,46,54,103,32,32,32,32,58,32,32,37,56,100,32,32,32,32,124,32,37,54,46,54,103,32,45,32,32,32,32,32,32,32,32,32,32,32,32,58,32,32,37,56,100,10,0,32,32,40,65,115,112,101,99,116,32,114,97,116,105,111,32,105,115,32,108,111,110,103,101,115,116,32,101,100,103,101,32,100,105,118,105,100,101,100,32,98,121,32,115,104,111,114,116,101,115,116,32,97,108,116,105,116,117,100,101,41,10,10,0,32,32,83,109,97,108,108,101,115,116,32,97,110,103,108,101,58,32,37,49,53,46,53,103,32,32,32,124,32,32,76,97,114,103,101,115,116,32,97,110,103,108,101,58,32,37,49,53,46,53,103,10,10,0,32,32,65,110,103,108,101,32,104,105,115,116,111,103,114,97,109,58,10,0,32,32,32,32,37,51,100,32,45,32,37,51,100,32,100,101,103,114,101,101,115,58,32,32,37,56,100,32,32,32,32,124,32,32,32,32,37,51,100,32,45,32,37,51,100,32,100,101,103,114,101,101,115,58,32,32,37,56,100,10,0,10,83,116,97,116,105,115,116,105,99,115,58,10,10,0,32,32,73,110,112,117,116,32,118,101,114,116,105,99,101,115,58,32,37,100,10,0,32,32,73,110,112,117,116,32,116,114,105,97,110,103,108,101,115,58,32,37,100,10,0,32,32,73,110,112,117,116,32,115,101,103,109,101,110,116,115,58,32,37,100,10,0,32,32,73,110,112,117,116,32,104,111,108,101,115,58,32,37,100,10,0,10,32,32,77,101,115,104,32,118,101,114,116,105,99,101,115,58,32,37,108,100,10,0,32,32,77,101,115,104,32,116,114,105,97,110,103,108,101,115,58,32,37,108,100,10,0,32,32,77,101,115,104,32,101,100,103,101,115,58,32,37,108,100,10,0,32,32,77,101,115,104,32,101,120,116,101,114,105,111,114,32,98,111,117,110,100,97,114,121,32,101,100,103,101,115,58,32,37,108,100,10,0,32,32,77,101,115,104,32,105,110,116,101,114,105,111,114,32,98,111,117,110,100,97,114,121,32,101,100,103,101,115,58,32,37,108,100,10,0,32,32,77,101,115,104,32,115,117,98,115,101,103,109,101,110,116,115,32,40,99,111,110,115,116,114,97,105,110,101,100,32,101,100,103,101,115,41,58,32,37,108,100,10,0,77,101,109,111,114,121,32,97,108,108,111,99,97,116,105,111,110,32,115,116,97,116,105,115,116,105,99,115,58,10,10,0,32,32,77,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,118,101,114,116,105,99,101,115,58,32,37,108,100,10,0,32,32,77,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,116,114,105,97,110,103,108,101,115,58,32,37,108,100,10,0,32,32,77,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,115,117,98,115,101,103,109,101,110,116,115,58,32,37,108,100,10,0,32,32,77,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,118,105,114,105,58,32,37,108,100,10,0,32,32,77,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,101,110,99,114,111,97,99,104,101,100,32,115,117,98,115,101,103,109,101,110,116,115,58,32,37,108,100,10,0,32,32,77,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,98,97,100,32,116,114,105,97,110,103,108,101,115,58,32,37,108,100,10,0,32,32,77,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,115,116,97,99,107,101,100,32,116,114,105,97,110,103,108,101,32,102,108,105,112,115,58,32,37,108,100,10,0,32,32,77,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,115,112,108,97,121,32,116,114,101,101,32,110,111,100,101,115,58,32,37,108,100,10,0,32,32,65,112,112,114,111,120,105,109,97,116,101,32,104,101,97,112,32,109,101,109,111,114,121,32,117,115,101,32,40,98,121,116,101,115,41,58,32,37,108,100,10,10,0,65,108,103,111,114,105,116,104,109,105,99,32,115,116,97,116,105,115,116,105,99,115,58,10,10,0,32,32,78,117,109,98,101,114,32,111,102,32,105,110,99,105,114,99,108,101,32,116,101,115,116,115,58,32,37,108,100,10,0,32,32,78,117,109,98,101,114,32,111,102,32,51,68,32,111,114,105,101,110,116,97,116,105,111,110,32,116,101,115,116,115,58,32,37,108,100,10,0,32,32,78,117,109,98,101,114,32,111,102,32,50,68,32,111,114,105,101,110,116,97,116,105,111,110,32,116,101,115,116,115,58,32,37,108,100,10,0,32,32,78,117,109,98,101,114,32,111,102,32,114,105,103,104,116,45,111,102,45,104,121,112,101,114,98,111,108,97,32,116,101,115,116,115,58,32,37,108,100,10,0,32,32,78,117,109,98,101,114,32,111,102,32,99,105,114,99,108,101,32,116,111,112,32,99,111,109,112,117,116,97,116,105,111,110,115,58,32,37,108,100,10,0,32,32,78,117,109,98,101,114,32,111,102,32,116,114,105,97,110,103,108,101,32,99,105,114,99,117,109,99,101,110,116,101,114,32,99,111,109,112,117,116,97,116,105,111,110,115,58,32,37,108,100,10,0,78,79,84,32,119,114,105,116,105,110,103,32,118,101,114,116,105,99,101,115,46,10,0,78,79,84,32,119,114,105,116,105,110,103,32,116,114,105,97,110,103,108,101,115,46,10,0,78,79,84,32,119,114,105,116,105,110,103,32,115,101,103,109,101,110,116,115,46,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,119,114,97,112,112,101,114,115,47,106,97,118,97,115,99,114,105,112,116,47,46,47,105,111,47,46,46,47,46,46,47,46,46,47,99,47,98,97,109,103,47,46,46,47,115,104,97,114,101,100,47,46,47,77,101,109,79,112,115,47,77,101,109,79,112,115,46,104,0,32,32,32,32,32,32,32,32,32,110,97,109,101,58,32,34,0,34,10,0,32,32,32,32,32,32,32,32,32,110,117,109,101,108,58,32,0,32,32,32,32,32,32,32,32,32,110,100,105,109,115,58,32,0,32,32,32,32,32,32,32,32,32,32,115,105,122,101,58,32,0,32,32,32,32,32,32,32,32,32,32,115,105,122,101,58,32,91,101,109,112,116,121,93,10,0,32,32,32,32,32,32,32,32,32,118,97,108,117,101,58,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,119,114,97,112,112,101,114,115,47,106,97,118,97,115,99,114,105,112,116,47,46,47,105,111,47,46,46,47,46,46,47,46,46,47,99,47,99,108,97,115,115,101,115,47,46,47,79,112,116,105,111,110,115,47,79,112,116,105,111,110,46,104,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,119,114,97,112,112,101,114,115,47,106,97,118,97,115,99,114,105,112,116,47,46,47,105,111,47,46,46,47,46,46,47,46,46,47,99,47,99,108,97,115,115,101,115,47,46,47,79,112,116,105,111,110,115,47,46,47,71,101,110,101,114,105,99,79,112,116,105,111,110,46,104,0,49,51,71,101,110,101,114,105,99,79,112,116,105,111,110,73,100,69,0,54,79,112,116,105,111,110,0,54,79,98,106,101,99,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,119,114,97,112,112,101,114,115,47,106,97,118,97,115,99,114,105,112,116,47,46,47,105,111,47,46,46,47,46,46,47,46,46,47,99,47,99,108,97,115,115,101,115,47,46,47,67,111,110,116,111,117,114,46,104,0,32,67,111,110,116,111,117,114,58,32,0,32,32,32,32,110,111,100,115,58,32,0,32,32,99,108,111,115,101,100,58,32,0,32,32,32,120,32,44,32,121,58,10,0,55,67,111,110,116,111,117,114,73,100,69,0,37,115,37,108,102,0,112,81,122,68,113,51,48,105,97,0,100,97,116,97,32,112,114,111,118,105,100,101,100,32,115,104,111,117,108,100,32,104,97,118,101,32,101,105,116,104,101,114,32,0,32,111,114,32,0,32,108,105,110,101,115,32,40,110,111,116,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,73,110,116,101,114,112,70,114,111,109,77,101,115,104,84,111,77,101,115,104,50,100,120,47,73,110,116,101,114,112,70,114,111,109,77,101,115,104,84,111,77,101,115,104,50,100,120,46,99,112,112,0,73,110,116,101,114,112,70,114,111,109,77,101,115,104,84,111,77,101,115,104,50,100,120,0,100,101,102,97,117,108,116,0,84,114,105,97,110,103,108,101,32,110,117,109,98,101,114,32,0,32,110,111,116,32,105,110,32,91,48,32,0,93,44,32,114,101,112,111,114,116,32,98,117,103,32,116,111,32,100,101,118,101,108,111,112,101,114,115,32,40,105,110,116,101,114,112,111,108,97,116,105,111,110,32,112,111,105,110,116,58,32,0,111,112,116,105,111,110,32,111,102,32,110,97,109,101,32,34,0,34,32,110,111,116,32,102,111,117,110,100,44,32,97,110,100,32,110,111,32,100,101,102,97,117,108,116,32,118,97,108,117,101,32,104,97,115,32,98,101,101,110,32,112,114,111,118,105,100,101,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,73,110,116,101,114,112,70,114,111,109,77,101,115,104,84,111,77,101,115,104,50,100,120,47,46,46,47,46,46,47,99,108,97,115,115,101,115,47,46,47,79,112,116,105,111,110,115,47,79,112,116,105,111,110,115,46,104,0,71,101,116,0,101,108,101,109,101,110,116,0,101,108,101,109,101,110,116,32,97,110,100,32,110,111,100,101,0,109,97,120,32,99,111,110,110,101,99,116,105,118,105,116,121,32,119,105,100,116,104,32,114,101,97,99,104,101,100,32,40,0,41,33,32,105,110,99,114,101,97,115,101,32,119,105,100,116,104,32,111,102,32,99,111,110,110,101,99,116,105,118,105,116,121,32,116,97,98,108,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,78,111,100,101,67,111,110,110,101,99,116,105,118,105,116,121,120,47,78,111,100,101,67,111,110,110,101,99,116,105,118,105,116,121,120,46,99,112,112,0,78,111,100,101,67,111,110,110,101,99,116,105,118,105,116,121,120,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,100,97,116,97,115,116,114,117,99,116,117,114,101,115,47,68,97,116,97,83,101,116,46,99,112,112,0,99,111,117,108,100,32,110,111,116,32,114,101,99,111,103,110,105,122,101,32,101,110,117,109,32,116,121,112,101,58,32,0,68,97,116,97,83,101,116,32,101,99,104,111,58,32,0,32,111,98,106,101,99,116,115,10,0,116,114,121,105,110,103,32,116,111,32,98,105,110,97,114,121,32,115,101,97,114,99,104,32,111,110,32,97,32,110,111,110,45,115,111,114,116,101,100,32,100,97,116,97,115,101,116,33,0,71,101,116,79,98,106,101,99,116,66,121,73,100,0,99,111,117,108,100,32,110,111,116,32,102,105,110,100,32,111,98,106,101,99,116,32,119,105,116,104,32,105,100,32,0,32,105,110,32,68,97,116,97,83,101,116,32,0,114,98,0,114,0,32,32,32,80,114,111,99,101,115,115,105,110,103,32,102,105,110,105,116,101,32,101,108,101,109,101,110,116,32,109,111,100,101,108,32,111,102,32,97,110,97,108,121,115,105,115,32,0,32,32,32,32,32,32,99,111,110,102,105,103,117,114,105,110,103,32,101,108,101,109,101,110,116,32,97,110,100,32,108,111,97,100,115,10,0,32,32,32,32,32,32,99,114,101,97,116,105,110,103,32,118,101,114,116,101,120,32,80,73,68,115,10,0,32,32,32,32,32,32,114,101,115,111,108,118,105,110,103,32,110,111,100,101,32,99,111,110,115,116,114,97,105,110,116,115,10,0,32,32,32,32,32,32,99,114,101,97,116,105,110,103,32,110,111,100,97,108,32,100,101,103,114,101,101,115,32,111,102,32,102,114,101,101,100,111,109,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,70,101,109,77,111,100,101,108,46,99,112,112,0,83,111,108,117,116,105,111,110,65,110,97,108,121,115,101,115,76,105,115,116,0,70,101,109,77,111,100,101,108,32,105,110,105,116,105,97,108,105,122,97,116,105,111,110,32,101,108,97,112,115,101,100,32,116,105,109,101,58,0,67,111,114,101,32,115,111,108,117,116,105,111,110,32,101,108,97,112,115,101,100,32,116,105,109,101,58,0,32,32,32,84,111,116,97,108,32,101,108,97,112,115,101,100,32,116,105,109,101,58,32,0,32,104,114,115,32,0,32,109,105,110,32,0,32,115,101,99,0,67,111,117,108,100,32,110,111,116,32,102,105,110,100,32,97,108,105,97,115,32,102,111,114,32,97,110,97,108,121,115,105,115,95,116,121,112,101,32,0,32,105,110,32,108,105,115,116,32,111,102,32,70,101,109,77,111,100,101,108,32,97,110,97,108,121,115,101,115,0,32,32,32,32,32,32,116,111,111,108,107,105,116,115,32,79,112,116,105,111,110,115,32,115,101,116,32,102,111,114,32,97,110,97,108,121,115,105,115,32,116,121,112,101,58,32,0,67,111,114,101,32,115,111,108,117,116,105,111,110,32,101,108,97,112,115,101,100,32,116,105,109,101,32,32,32,32,58,32,0,32,83,101,99,111,110,100,115,10,0,67,111,114,101,32,115,111,108,117,116,105,111,110,32,110,117,109,98,101,114,32,111,102,32,102,108,111,112,115,32,58,32,0,32,70,108,111,112,115,10,0,67,111,114,101,32,115,111,108,117,116,105,111,110,32,109,101,109,111,114,121,32,117,115,101,100,32,32,32,32,32,58,32,0,32,66,121,116,101,115,10,0,101,114,114,111,114,32,114,101,97,100,105,110,103,32,116,104,101,32,98,117,102,102,101,114,32,102,114,111,109,32,109,97,114,115,104,97,108,108,101,100,32,102,105,108,101,33,0,82,101,115,116,97,114,116,0,32,32,32,117,112,100,97,116,105,110,103,32,118,101,114,116,105,99,101,115,32,112,111,115,105,116,105,111,110,115,10,0,32,32,32,85,112,100,97,116,105,110,103,32,99,111,110,115,116,114,97,105,110,116,115,32,97,110,100,32,97,99,116,105,118,101,32,100,111,109,97,105,110,32,111,102,32,97,110,97,108,121,115,105,115,32,0,32,102,111,114,32,116,105,109,101,58,32,0,110,111,32,109,97,115,115,32,102,108,117,120,32,115,101,103,109,101,110,116,115,32,97,118,97,105,108,97,98,108,101,33,0,77,97,115,115,70,108,117,120,120,0,73,110,116,101,114,112,111,108,97,116,105,111,110,32,0,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,120,0,83,104,111,117,108,100,32,110,111,116,32,98,101,32,114,101,113,117,101,115,116,105,110,103,32,100,101,112,101,110,100,101,110,116,115,32,119,104,101,110,32,97,110,32,65,68,32,108,105,98,114,97,114,121,32,105,115,32,110,111,116,32,97,118,97,105,108,97,98,108,101,33,0,82,101,113,117,101,115,116,101,100,68,101,112,101,110,100,101,110,116,115,120,0,109,111,114,101,32,116,104,97,110,32,51,32,99,111,110,116,114,111,108,115,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,0,79,117,116,112,117,116,67,111,110,116,114,111,108,115,120,0,13,0,32,32,32,32,32,32,99,111,110,118,111,108,117,116,105,111,110,32,112,114,111,103,114,101,115,115,58,32,0,37,0,32,32,32,78,117,109,98,101,114,32,111,102,32,97,99,116,105,118,101,32,110,111,100,101,115,32,105,110,32,69,80,76,32,108,97,121,101,114,58,32,0,32,32,32,78,117,109,98,101,114,32,111,102,32,97,99,116,105,118,101,32,110,111,100,101,115,32,76,50,32,80,114,111,106,101,99,116,105,111,110,58,32,0,70,101,109,77,111,100,101,108,58,58,73,110,105,116,70,114,111,109,66,117,102,102,101,114,115,32,101,114,114,111,114,32,109,101,115,115,97,103,101,58,32,99,111,117,108,100,32,110,111,116,32,105,110,105,116,105,97,108,105,122,101,32,111,117,116,112,117,116,32,115,116,114,101,97,109,0,73,110,105,116,70,114,111,109,66,117,102,102,101,114,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,46,47,80,97,114,97,109,115,47,71,101,110,101,114,105,99,80,97,114,97,109,46,104,0,32,32,32,87,65,82,78,73,78,71,58,32,112,97,114,97,109,101,116,101,114,32,0,32,105,115,32,97,32,71,101,110,101,114,105,99,80,97,114,97,109,32,97,110,100,32,99,97,110,110,111,116,32,98,101,32,109,97,114,115,104,97,108,108,101,100,10,0,71,101,110,101,114,105,99,80,97,114,97,109,58,10,0,32,32,32,101,110,117,109,58,32,32,0,49,50,71,101,110,101,114,105,99,80,97,114,97,109,73,80,106,69,0,53,80,97,114,97,109,0,49,50,71,101,110,101,114,105,99,80,97,114,97,109,73,80,80,99,69,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,46,47,69,120,116,101,114,110,97,108,82,101,115,117,108,116,115,47,71,101,110,101,114,105,99,69,120,116,101,114,110,97,108,82,101,115,117,108,116,46,104,0,87,114,105,116,101,68,97,116,97,0,71,101,110,101,114,105,99,69,120,116,101,114,110,97,108,82,101,115,117,108,116,32,105,110,115,116,97,110,116,105,97,116,101,100,32,102,111,114,32,116,121,112,101,32,86,101,99,116,111,114,60,73,115,115,109,68,111,117,98,108,101,62,42,32,99,97,108,108,101,100,32,0,71,101,110,101,114,105,99,69,120,116,101,114,110,97,108,82,101,115,117,108,116,60,86,101,99,116,111,114,60,73,115,115,109,80,68,111,117,98,108,101,62,42,62,58,10,0,50,49,71,101,110,101,114,105,99,69,120,116,101,114,110,97,108,82,101,115,117,108,116,73,80,54,86,101,99,116,111,114,73,100,69,69,0,49,52,69,120,116,101,114,110,97,108,82,101,115,117,108,116,0,71,101,110,101,114,105,99,69,120,116,101,114,110,97,108,82,101,115,117,108,116,60,73,115,115,109,80,68,111,117,98,108,101,42,62,58,10,0,32,32,32,91,32,0,32,93,10,0,50,49,71,101,110,101,114,105,99,69,120,116,101,114,110,97,108,82,101,115,117,108,116,73,80,100,69,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,46,46,47,116,111,111,108,107,105,116,115,47,46,47,111,98,106,101,99,116,115,47,46,46,47,105,115,115,109,47,46,47,73,115,115,109,86,101,99,46,104,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,46,46,47,116,111,111,108,107,105,116,115,47,46,47,111,98,106,101,99,116,115,47,46,46,47,105,115,115,109,47,46,47,73,115,115,109,83,101,113,86,101,99,46,104,0,73,115,115,109,83,101,113,86,101,99,32,115,105,122,101,32,0,10,32,0,49,48,73,115,115,109,83,101,113,86,101,99,73,100,69,0,49,48,73,115,115,109,65,98,115,86,101,99,73,100,69,0,99,97,110,110,111,116,32,99,114,101,97,116,101,32,112,101,116,115,99,32,118,101,99,116,111,114,32,119,105,116,104,111,117,116,32,80,69,84,83,67,32,99,111,109,112,105,108,101,100,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,46,46,47,116,111,111,108,107,105,116,115,47,46,47,111,98,106,101,99,116,115,47,46,47,86,101,99,116,111,114,46,104,0,71,101,110,101,114,105,99,77,97,114,115,104,97,108,108,0,71,101,110,101,114,105,99,69,120,116,101,114,110,97,108,82,101,115,117,108,116,60,100,111,117,98,108,101,62,58,10,0,50,49,71,101,110,101,114,105,99,69,120,116,101,114,110,97,108,82,101,115,117,108,116,73,100,69,0,49,50,71,101,110,101,114,105,99,80,97,114,97,109,73,105,69,0,49,52,84,114,97,110,115,105,101,110,116,73,110,112,117,116,0,84,114,97,110,115,105,101,110,116,73,110,112,117,116,58,10,0,32,32,32,110,117,109,116,105,109,101,115,116,101,112,115,58,32,0,45,45,45,105,110,112,117,116,115,58,32,10,0,32,32,32,116,105,109,101,58,32,0,32,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,84,114,97,110,115,105,101,110,116,73,110,112,117,116,46,99,112,112,0,116,105,109,101,115,116,101,112,32,118,97,108,117,101,115,32,109,117,115,116,32,105,110,99,114,101,97,115,101,32,115,101,113,117,101,110,116,105,97,108,108,121,0,65,100,100,84,105,109,101,73,110,112,117,116,0,53,73,110,112,117,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,47,73,110,112,117,116,115,47,84,114,97,110,115,105,101,110,116,73,110,112,117,116,46,104,0,49,50,83,112,99,84,114,97,110,115,105,101,110,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,67,111,110,115,116,114,97,105,110,116,115,47,46,46,47,46,46,47,115,104,97,114,101,100,47,46,47,77,101,109,79,112,115,47,77,101,109,79,112,115,46,104,0,83,112,99,84,114,97,110,115,105,101,110,116,58,10,0,32,32,32,110],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+57640);allocate([115,116,101,112,115,58,32,0,32,32,32,115,116,101,112,115,124,116,105,109,101,115,124,118,97,108,117,101,115,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,67,111,110,115,116,114,97,105,110,116,115,47,83,112,99,84,114,97,110,115,105,101,110,116,46,99,112,112,0,99,111,117,108,100,32,110,111,116,32,102,105,110,100,32,116,105,109,101,32,115,101,103,109,101,110,116,32,102,111,114,32,99,111,110,115,116,114,97,105,110,116,0,67,111,110,115,116,114,97,105,110,78,111,100,101,0,99,97,110,110,111,116,32,114,101,116,117,114,110,32,100,111,102,32,97,110,100,32,118,97,108,117,101,32,102,111,114,32,110,111,110,32,112,101,110,97,108,116,121,32,99,111,110,115,116,114,97,105,110,116,0,49,48,67,111,110,115,116,114,97,105,110,116,0,77,97,114,115,104,97,108,108,101,100,32,102,105,108,101,32,105,115,32,101,109,112,116,121,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,111,77,111,100,101,108,46,99,112,112,0,67,104,101,99,107,69,110,117,109,83,121,110,99,0,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,10,0,32,69,110,117,109,115,32,105,110,32,109,97,114,115,104,97,108,108,101,100,32,102,105,108,101,32,97,114,101,32,110,111,116,32,99,111,109,112,97,116,105,98,108,101,32,119,105,116,104,32,99,111,109,112,105,108,101,100,32,99,111,100,101,32,32,32,32,32,32,32,32,32,32,10,0,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,10,0,32,32,32,42,32,73,102,32,121,111,117,32,97,114,101,32,114,117,110,110,105,110,103,32,73,83,83,77,32,111,110,32,121,111,117,114,32,108,111,99,97,108,32,109,97,99,104,105,110,101,58,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,10,0,32,32,32,32,32,109,97,107,101,32,115,117,114,101,32,116,104,97,116,32,97,108,108,32,116,104,101,32,99,111,100,101,32,105,115,32,99,111,109,112,105,108,101,100,32,97,110,100,32,105,110,115,116,97,108,108,101,100,32,40,109,97,107,101,32,105,110,115,116,97,108,108,41,10,0,32,32,32,42,32,73,102,32,121,111,117,32,97,114,101,32,114,117,110,110,105,110,103,32,73,83,83,77,32,111,110,32,97,32,114,101,109,111,116,101,32,99,108,117,115,116,101,114,58,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,10,0,32,32,32,32,32,109,97,107,101,32,115,117,114,101,32,116,104,97,116,32,121,111,117,32,97,114,101,32,117,115,105,110,103,32,116,104,101,32,115,97,109,101,32,118,101,114,115,105,111,110,32,111,102,32,73,83,83,77,32,111,110,32,121,111,117,114,32,108,111,99,97,108,32,10,0,32,32,32,32,32,109,97,99,104,105,110,101,32,97,110,100,32,114,101,109,111,116,101,32,99,108,117,115,116,101,114,32,40,121,111,117,32,109,105,103,104,116,32,110,101,101,100,32,116,111,32,114,117,110,32,115,118,110,32,117,112,100,97,116,101,41,32,32,32,32,32,32,32,10,0,32,32,32,42,32,73,102,32,121,111,117,32,97,114,101,32,97,32,100,101,118,101,108,111,112,101,114,32,97,110,100,32,106,117,115,116,32,97,100,100,101,100,32,97,32,110,101,119,32,69,110,117,109,58,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,10,0,32,32,32,32,32,121,111,117,32,109,105,103,104,116,32,110,101,101,100,32,116,111,32,114,117,110,32,46,47,83,121,110,99,104,114,111,110,105,122,101,46,115,104,32,105,110,32,115,114,99,47,99,47,115,104,97,114,101,100,47,69,110,117,109,32,32,32,32,32,32,32,32,32,10,0,32,32,32,32,32,97,110,100,32,114,101,99,111,109,112,105,108,101,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,10,0,69,110,117,109,115,32,110,111,116,32,99,111,110,115,105,115,116,101,110,116,32,40,83,101,101,32,101,114,114,111,114,32,109,101,115,115,97,103,101,32,97,98,111,118,101,41,0,67,111,117,108,100,32,110,111,116,32,114,101,97,100,32,114,101,99,111,114,100,95,108,101,110,103,116,104,0,32,77,97,114,115,104,97,108,108,101,100,32,102,105,108,101,32,105,115,32,99,111,114,114,117,112,116,101,100,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,10,0,32,32,32,42,32,73,102,32,121,111,117,32,97,114,101,32,114,117,110,110,105,110,103,32,97,110,32,111,108,100,32,109,111,100,101,108,44,32,115,101,110,100,32,105,116,32,116,111,32,116,104,101,32,73,83,83,77,32,100,101,118,101,108,111,112,101,114,115,32,32,32,32,32,10,0,32,32,32,32,32,115,111,32,116,104,97,116,32,97,32,99,104,101,99,107,32,105,115,32,97,100,100,101,100,32,98,101,102,111,114,101,32,109,97,114,115,104,97,108,108,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,10,0,32,32,32,42,32,76,97,115,116,32,69,110,117,109,32,102,111,117,110,100,58,32,0,69,110,117,109,32,40,0,32,32,32,32,32,116,104,101,32,99,111,114,114,101,115,112,111,110,100,105,110,103,32,109,111,100,101,108,32,102,105,101,108,100,32,104,97,115,32,112,114,111,98,97,98,108,121,32,98,101,101,110,32,109,97,114,115,104,97,108,108,101,100,32,32,32,32,32,32,32,32,32,32,10,0,32,32,32,32,32,105,110,99,111,114,114,101,99,116,108,121,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,10,0,66,105,110,97,114,121,32,102,105,108,101,32,99,111,114,114,117,112,116,101,100,32,40,83,101,101,32,101,114,114,111,114,32,109,101,115,115,97,103,101,32,97,98,111,118,101,41,0,67,97,110,110,111,116,32,99,97,114,114,121,32,111,117,116,32,65,68,32,109,111,100,101,32,99,111,109,112,117,116,97,116,105,111,110,115,32,119,105,116,104,111,117,116,32,115,117,112,112,111,114,116,32,111,102,32,65,68,79,76,67,32,99,111,109,112,105,108,101,100,32,105,110,33,0,68,101,99,108,97,114,101,73,110,100,101,112,101,110,100,101,110,116,115,0,67,111,117,110,100,32,110,111,116,32,114,101,97,100,32,114,101,99,111,114,100,95,108,101,110,103,116,104,0,70,101,116,99,104,67,111,110,115,116,97,110,116,115,0,67,111,117,110,100,32,110,111,116,32,114,101,97,100,32,114,101,99,111,114,100,95,99,111,100,101,0,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,98,111,111,108,101,97,110,32,0,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,105,110,116,101,103,101,114,32,0,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,115,99,97,108,97,114,32,0,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,108,101,110,103,116,104,32,111,102,32,115,116,114,105,110,103,32,0,32,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,115,116,114,105,110,103,32,0,117,110,107,110,111,119,110,32,114,101,99,111,114,100,32,116,121,112,101,58,0,101,120,112,101,99,116,105,110,103,32,97,110,32,105,110,116,101,103,101,114,32,102,111,114,32,101,110,117,109,32,0,70,101,116,99,104,68,97,116,97,0,101,120,112,101,99,116,105,110,103,32,97,32,73,115,115,109,68,111,117,98,108,101,44,32,105,110,116,101,103,101,114,32,111,114,32,98,111,111,108,101,97,110,32,109,97,116,114,105,120,32,102,111,114,32,101,110,117,109,32,0,32,40,67,111,100,101,32,105,115,32,0,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,110,117,109,98,101,114,32,111,102,32,114,111,119,115,32,102,111,114,32,109,97,116,114,105,120,32,0,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,110,117,109,98,101,114,32,111,102,32,99,111,108,117,109,110,115,32,102,111,114,32,109,97,116,114,105,120,32,0,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,109,97,116,114,105,120,32,0,67,111,110,115,116,97,110,116,32,0,32,110,111,116,32,102,111,117,110,100,32,105,110,32,105,111,109,111,100,101,108,0,67,111,112,121,67,111,110,115,116,97,110,116,79,98,106,101,99,116,0,101,120,112,101,99,116,105,110,103,32,97,32,98,111,111,108,101,97,110,32,102,111,114,32,101,110,117,109,32,0,67,111,117,108,100,32,110,111,116,32,114,101,97,100,32,114,101,99,111,114,100,95,99,111,100,101,0,83,101,116,70,105,108,101,80,111,105,110,116,101,114,84,111,68,97,116,97,0,67,111,117,108,100,32,110,111,116,32,114,101,97,100,32,118,101,99,116,111,114,95,116,121,112,101,0,99,111,117,108,100,32,110,111,116,32,102,105,110,100,32,100,97,116,97,32,119,105,116,104,32,110,97,109,101,32,0,32,105,110,32,98,105,110,97,114,121,32,102,105,108,101,0,101,120,112,101,99,116,105,110,103,32,97,32,73,115,115,109,68,111,117,98,108,101,32,102,111,114,32,101,110,117,109,32,0,101,120,112,101,99,116,105,110,103,32,97,32,115,116,114,105,110,103,32,102,111,114,32,101,110,117,109,32,0,101,120,112,101,99,116,105,110,103,32,97,32,115,116,114,105,110,103,32,97,114,114,97,121,32,102,111,114,32,101,110,117,109,32,0,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,108,101,110,103,116,104,32,111,102,32,115,116,114,105,110,103,32,97,114,114,97,121,0,101,120,112,101,99,116,105,110,103,32,97,32,73,115,115,109,68,111,117,98,108,101,32,109,97,116,32,97,114,114,97,121,32,102,111,114,32,101,110,117,109,32,0,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,110,117,109,98,101,114,32,111,102,32,114,101,99,111,114,100,115,32,105,110,32,109,97,116,114,105,120,32,97,114,114,97,121,32,0,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,110,117,109,98,101,114,32,111,102,32,114,111,119,115,32,105,110,32,0,116,104,32,109,97,116,114,105,120,32,111,102,32,109,97,116,114,105,120,32,97,114,114,97,121,0,99,111,117,108,100,32,110,111,116,32,114,101,97,100,32,110,117,109,98,101,114,32,111,102,32,99,111,108,117,109,110,115,32,105,110,32,0,70,101,116,99,104,77,117,108,116,105,112,108,101,68,97,116,97,0,83,101,116,70,105,108,101,80,111,105,110,116,101,114,115,84,111,68,97,116,97,0,101,120,112,101,99,116,105,110,103,32,97,32,100,111,117,98,108,101,32,102,111,114,32,101,110,117,109,32,0,32,105,115,32,110,111,116,32,97,32,100,111,117,98,108,101,32,97,114,114,97,121,0,70,101,116,99,104,68,97,116,97,84,111,73,110,112,117,116,0,32,110,111,116,32,102,111,117,110,100,32,105,110,32,98,105,110,97,114,121,32,102,105,108,101,0,100,97,116,97,32,99,111,100,101,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,46,46,47,115,104,97,114,101,100,47,46,47,77,101,109,79,112,115,47,77,101,109,79,112,115,46,104,0,117,110,107,110,111,119,110,32,105,110,115,101,114,116,32,109,111,100,101,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,46,47,69,108,101,109,101,110,116,115,47,46,46,47,46,46,47,116,111,111,108,107,105,116,115,47,46,47,111,98,106,101,99,116,115,47,46,46,47,105,115,115,109,47,46,47,73,115,115,109,68,101,110,115,101,77,97,116,46,104,0,83,101,116,86,97,108,117,101,115,0,65,32,97,110,100,32,65,88,32,115,104,111,117,108,100,32,104,97,118,101,32,116,104,101,32,115,97,109,101,32,110,117,109,98,101,114,32,111,102,32,114,111,119,115,33,0,77,97,116,77,117,108,116,0,65,32,97,110,100,32,88,32,115,104,111,117,108,100,32,104,97,118,101,32,116,104,101,32,115,97,109,101,32,110,117,109,98,101,114,32,111,102,32,99,111,108,117,109,110,115,33,0,117,110,107,110,111,119,110,32,110,111,114,109,32,33,0,78,111,114,109,0,73,115,115,109,68,101,110,115,101,77,97,116,32,115,105,122,101,32,0,49,50,73,115,115,109,68,101,110,115,101,77,97,116,73,100,69,0,49,48,73,115,115,109,65,98,115,77,97,116,73,100,69,0,112,101,116,115,99,0,99,97,110,110,111,116,32,99,114,101,97,116,101,32,112,101,116,115,99,32,109,97,116,114,105,120,32,119,105,116,104,111,117,116,32,80,69,84,83,67,32,99,111,109,112,105,108,101,100,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,46,47,69,108,101,109,101,110,116,115,47,46,46,47,46,46,47,116,111,111,108,107,105,116,115,47,46,47,111,98,106,101,99,116,115,47,46,47,77,97,116,114,105,120,46,104,0,73,110,105,116,67,104,101,99,107,65,110,100,83,101,116,84,121,112,101,0,117,110,107,110,111,119,32,116,111,111,108,107,105,116,32,116,121,112,101,32,0,52,78,111,100,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,78,111,100,101,46,99,112,112,0,78,111,100,101,58,10,0,32,32,32,97,110,97,108,121,115,105,115,95,101,110,117,109,58,32,0,32,32,32,97,112,112,114,111,120,105,109,97,116,105,111,110,58,32,0,32,32,32,105,110,100,101,120,105,110,103,117,112,100,97,116,101,58,32,0,71,101,116,68,111,102,0,71,101,116,68,111,102,76,105,115,116,0,71,101,116,76,111,99,97,108,68,111,102,76,105,115,116,0,67,97,110,110,111,116,32,99,104,97,110,103,101,32,100,111,102,32,111,102,32,102,114,111,122,101,110,32,110,111,100,101,0,68,111,102,73,110,70,83,101,116,0,71,101,116,78,117,109,98,101,114,79,102,68,111,102,115,0,86,101,99,77,101,114,103,101,32,99,97,110,32,111,110,108,121,32,109,101,114,103,101,32,102,114,111,109,32,116,104,101,32,115,32,111,114,32,102,45,115,101,116,32,111,110,116,111,32,116,104,101,32,103,45,115,101,116,33,0,86,101,99,77,101,114,103,101,0,86,101,99,82,101,100,117,99,101,32,99,97,110,32,111,110,108,121,32,109,101,114,103,101,32,102,114,111,109,32,116,104,101,32,115,32,111,114,32,102,45,115,101,116,32,111,110,116,111,32,116,104,101,32,103,45,115,101,116,33,0,86,101,99,82,101,100,117,99,101,0,68,105,115,116,114,105,98,117,116,101,68,111,102,115,0,79,102,102,115,101,116,68,111,102,115,0,83,104,111,119,84,114,117,101,68,111,102,115,0,85,112,100,97,116,101,67,108,111,110,101,68,111,102,115,0,54,86,101,114,116,101,120,0,86,101,114,116,101,120,58,10,0,32,32,32,112,105,100,58,32,0,32,32,32,120,58,32,0,32,32,32,121,58,32,0,32,32,32,122,58,32,0,32,32,32,115,105,103,109,97,58,32,0,32,32,32,99,111,110,110,101,99,116,105,118,105,116,121,58,32,0,32,32,32,99,108,111,110,101,58,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,86,101,114,116,101,120,46,99,112,112,0,85,112,100,97,116,101,80,111,115,105,116,105,111,110,0,55,69,108,101,109,101,110,116,0,73,110,112,117,116,32,109,105,115,115,105,110,103,46,32,72,101,114,101,32,97,114,101,32,116,104,101,32,105,110,112,117,116,32,112,111,105,110,116,101,114,115,32,119,101,32,104,97,118,101,32,102,111,114,32,118,120,58,32,0,44,32,118,121,58,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,69,108,101,109,101,110,116,46,99,112,112,0,83,116,114,97,105,110,82,97,116,101,83,83,65,0,44,32,118,122,58,32,0,83,116,114,97,105,110,82,97,116,101,70,83,0,67,111,111,114,100,105,110,97,116,101,32,115,121,115,116,101,109,32,0,67,111,111,114,100,105,110,97,116,101,83,121,115,116,101,109,84,114,97,110,115,102,111,114,109,0,32,101,108,101,109,101,110,116,58,10,0,32,32,32,105,100,32,58,32,0,118,101,114,116,105,99,101,115,32,61,32,78,85,76,76,10,0,110,111,100,101,115,32,61,32,78,85,76,76,10,0,109,97,116,101,114,105,97,108,32,61,32,78,85,76,76,10,0,109,97,116,112,97,114,32,61,32,78,85,76,76,10,0,112,97,114,97,109,101,116,101,114,115,32,61,32,78,85,76,76,10,0,32,32,32,105,110,112,117,116,115,10,0,105,110,112,117,116,115,61,78,85,76,76,10,0,83,116,114,97,105,110,82,97,116,101,72,79,0,83,116,114,97,105,110,82,97,116,101,72,79,50,100,118,101,114,116,105,99,97,108,0,83,116,114,97,105,110,82,97,116,101,83,83,65,49,100,0,68,101,108,116,97,49,56,111,80,97,114,97,109,101,116,101,114,105,122,97,116,105,111,110,0,77,117,110,103,115,109,116,112,80,97,114,97,109,101,116,101,114,105,122,97,116,105,111,110,0,68,101,108,116,97,49,56,111,112,100,80,97,114,97,109,101,116,101,114,105,122,97,116,105,111,110,0,110,111,100,101,115,91,0,93,32,61,32,0,71,101,116,73,110,112,117,116,76,105,115,116,79,110,78,111,100,101,115,0,71,101,116,73,110,112,117,116,76,105,115,116,79,110,78,111,100,101,115,86,101,108,111,99,105,116,121,0,71,101,116,73,110,112,117,116,76,105,115,116,79,110,86,101,114,116,105,99,101,115,0,73,110,112,117,116,67,114,101,97,116,101,0,110,111,100,97,108,32,118,101,99,116,111,114,32,105,115,32,101,105,116,104,101,114,32,110,117,109,98,101,114,111,102,118,101,114,116,105,99,101,115,32,111,114,32,110,117,109,98,101,114,111,102,118,101,114,116,105,99,101,115,43,49,32,108,111,110,103,46,32,70,105,101,108,100,32,112,114,111,118,105,100,101,100,32,40,0,41,32,105,115,32,0,32,108,111,110,103,0,99,111,117,108,100,32,110,111,116,32,114,101,99,111,103,110,105,122,101,32,110,97,116,117,114,101,32,111,102,32,118,101,99,116,111,114,32,102,114,111,109,32,99,111,100,101,32,0,101,108,101,109,101,110,116,32,118,101,99,116,111,114,32,105,115,32,101,105,116,104,101,114,32,110,117,109,98,101,114,111,102,101,108,101,109,101,110,116,115,32,111,114,32,110,117,109,98,101,114,111,102,101,108,101,109,101,110,116,115,43,49,32,108,111,110,103,46,32,70,105,101,108,100,32,112,114,111,118,105,100,101,100,32,40,0,67,97,110,110,111,116,32,97,100,100,32,105,110,112,117,116,32,102,111,114,32,118,101,99,116,111,114,32,116,121,112,101,32,0,32,40,110,111,116,32,115,117,112,112,111,114,116,101,100,41,0,109,105,103,114,97,116,105,111,110,95,115,116,121,108,101,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,0,73,115,70,108,111,97,116,105,110,103,0,77,97,114,115,104,97,108,108,69,108,101,109,101,110,116,0,69,114,114,111,114,58,32,109,105,103,114,97,116,105,111,110,32,115,104,111,117,108,100,32,98,101,32,65,103,103,114,101,115,115,105,118,101,44,32,83,111,102,116,44,32,83,117,98,101,108,101,109,101,110,116,44,32,67,111,110,116,97,99,116,32,111,114,32,71,114,111,117,110,100,105,110,103,79,110,108,121,0,77,105,103,114,97,116,101,71,114,111,117,110,100,105,110,103,76,105,110,101,0,80,111,115,105,116,105,118,101,68,101,103,114,101,101,68,97,121,0,82,101,115,117,108,116,73,110,116,101,114,112,111,108,97,116,105,111,110,0,78,111,32,101,105,103,101,110,32,118,97,108,117,101,32,102,111,117,110,100,0,83,116,114,101,115,115,77,97,120,80,114,105,110,99,105,112,97,108,67,114,101,97,116,101,73,110,112,117,116,0,82,101,115,117,108,116,84,111,77,97,116,114,105,120,0,82,101,115,117,108,116,84,111,86,101,99,116,111,114,0,105,110,116,101,114,112,111,108,97,116,105,111,110,32,0,115,109,98,32,99,111,114,101,58,32,73,110,105,116,105,97,108,105,122,105,110,103,32,103,114,105,100,10,0,84,105,109,101,58,32,116,61,0,32,121,114,47,0,32,121,114,0,32,83,116,101,112,58,32,0,116,111,116,97,108,32,115,121,115,116,101,109,32,109,97,115,115,32,110,111,116,32,99,111,110,115,101,114,118,101,100,32,105,110,32,77,66,32,102,117,110,99,116,105,111,110,0,84,40,101,110,100,41,126,61,84,95,98,111,116,116,111,109,0,84,114,97,110,115,102,111,114,109,73,110,118,83,116,105,102,102,110,101,115,115,77,97,116,114,105,120,67,111,111,114,100,0,84,114,97,110,115,102,111,114,109,76,111,97,100,86,101,99,116,111,114,67,111,111,114,100,0,84,114,97,110,115,102,111,114,109,83,111,108,117,116,105,111,110,67,111,111,114,100,0,84,114,97,110,115,102,111,114,109,83,116,105,102,102,110,101,115,115,77,97,116,114,105,120,67,111,111,114,100,0,73,110,112,117,116,32,109,105,115,115,105,110,103,0,86,105,115,99,111,115,105,116,121,76,49,76,50,0,54,85,112,100,97,116,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,46,46,47,46,47,69,108,101,109,101,110,116,115,47,69,108,101,109,101,110,116,46,104,0,82,101,115,101,116,76,101,118,101,108,115,101,116,70,114,111,109,83,101,103,109,101,110,116,108,105,115,116,0,87,114,105,116,101,76,101,118,101,108,115,101,116,83,101,103,109,101,110,116,0,67,97,108,118,105,110,103,82,97,116,101,68,101,118,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,69,108,101,109,101,110,116,72,111,111,107,46,99,112,112,0,51,83,101,103,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,83,101,103,46,99,112,112,0,102,114,111,110,116,32,105,115,32,110,111,116,32,108,111,99,97,116,101,100,32,111,110,32,101,108,101,109,101,110,116,32,101,100,103,101,0,54,83,101,103,82,101,102,0,49,49,69,108,101,109,101,110,116,72,111,111,107,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,46,46,47,46,47,69,108,101,109,101,110,116,115,47,83,101,103,46,104,0,86,97,108,117,101,80,49,79,110,71,97,117,115,115,0,83,101,116,67,117,114,114,101,110,116,67,111,110,102,105,103,117,114,97,116,105,111,110,0,82,101,115,101,116,72,111,111,107,115,0,82,101,100,117,99,101,77,97,116,114,105,99,101,115,0,78,117,109,98,101,114,111,102,78,111,100,101,115,86,101,108,111,99,105,116,121,0,78,117,109,98,101,114,111,102,78,111,100,101,115,80,114,101,115,115,117,114,101,0,78,111,114,109,97,108,84,111,112,0,78,111,114,109,97,108,66,97,115,101,0,78,111,100,97,108,70,117,110,99,116,105,111,110,115,84,101,110,115,111,114,0,78,111,100,97,108,70,117,110,99,116,105,111,110,115,86,101,108,111,99,105,116,121,0,78,111,100,97,108,70,117,110,99,116,105,111,110,115,80,114,101,115,115,117,114,101,0,78,111,100,97,108,70,117,110,99,116,105,111,110,115,68,101,114,105,118,97,116,105,118,101,115,86,101,108,111,99,105,116,121,0,78,101,119,71,97,117,115,115,84,111,112,0,78,101,119,71,97,117,115,115,66,97,115,101,0,74,97,99,111,98,105,97,110,68,101,116,101,114,109,105,110,97,110,116,84,111,112,0,74,97,99,111,98,105,97,110,68,101,116,101,114,109,105,110,97,110,116,66,97,115,101,0,71,101,116,86,101,114,116,105,99,101,115,67,111,111,114,100,105,110,97,116,101,115,84,111,112,0,71,101,116,86,101,114,116,105,99,101,115,67,111,111,114,100,105,110,97,116,101,115,66,97,115,101,0,71,101,116,69,108,101,109,101,110,116,84,121,112,101,0,69,108,101,109,101,110,116,83,105,122,101,115,0,67,111,110,102,105,103,117,114,101,0,65,100,100,73,110,112,117,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,83,101,103,82,101,102,46,99,112,112,0,52,84,114,105,97,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,84,114,105,97,46,99,112,112,0,108,105,115,116,32,111,102,32,118,101,114,116,105,99,101,115,32,111,110,32,98,101,100,58,32,0,67,111,117,108,100,32,110,111,116,32,102,105,110,100,32,50,32,118,101,114,116,105,99,101,115,32,111,110,32,98,101,100,0,69,100,103,101,79,110,66,97,115,101,73,110,100,101,120,0,78,111,116,32,73,109,112,108,101,109,101,110,116,101,100,32,121,101,116,0,100,101,118,105,97,116,111,114,105,99,32,115,116,114,101,115,115,32,116,101,110,115,111,114,32,99,97,108,99,117,108,97,116,105,111,110,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,102,111,114,32,109,101,115,104,32,111,102,32,116,121,112,101,32,0,115,116,114,101,115,115,32,116,101,110,115,111,114,32,99,97,108,99,117,108,97,116,105,111,110,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,102,111,114,32,109,101,115,104,32,111,102,32,116,121,112,101,32,0,44,32,101,120,116,114,117,100,101,32,109,101,115,104,32,111,114,32,99,97,108,108,32,67,111,109,112,117,116,101,68,101,118,105,97,116,111,114,105,99,83,116,114,101,115,115,84,101,110,115,111,114,0,69,100,103,101,79,110,66,97,115,101,73,110,100,105,99,101,115,0,108,105,115,116,32,111,102,32,118,101,114,116,105,99,101,115,32,111,110,32,115,117,114,102,97,99,101,58,32,0,67,111,117,108,100,32,110,111,116,32,102,105,110,100,32,50,32,118,101,114,116,105,99,101,115,32,111,110,32,115,117,114,102,97,99,101,0,69,100,103,101,79,110,83,117,114,102,97,99,101,73,110,100,105,99,101,115,0,87,114,111,110,103,32,110,117,109,98,101,114,32,111,102,32,105,99,101,32,118,101,114,116,105,99,101,115,32,105,110,32,84,114,105,97,58,58,71,101,116,76,101,118,101,108,115,101,116,73,110,116,101,114,115,101,99,116,105,111,110,33,0,71,101,116,76,101,118,101,108,115,101,116,73,110,116,101,114,115,101,99,116,105,111,110,0,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,32,0,84,119,111,32,101,100,103,101,115,32,111,110,32,98,101,100,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,46,46,46,0,72,97,115,69,100,103,101,79,110,66,97,115,101,0,84,119,111,32,101,100,103,101,115,32,111,110,32,115,117,114,102,97,99,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,46,46,46,0,72,97,115,69,100,103,101,79,110,83,117,114,102,97,99,101,0,78,97,78,32,102,111,117,110,100,32,105,110,32,118,101,99,116,111,114,0,84,119,111,32,101,100,103,101,115,32,111,110,32,98,111,117,110,100,97,114,121,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,46,46,46,0,73,115,79,110,66,97,115,101,0,73,115,79,110,83,117,114,102,97,99,101,0,101,114,114,111,114,32,109,101,115,115,97,103,101,58,32,115,101,103,109,101,110,116,32,119,105,116,104,32,105,100,32,0,32,100,111,101,115,32,110,111,116,32,98,101,108,111,110,103,32,116,111,32,101,108,101,109,101,110,116,32,119,105,116,104,32,105,100,58,0,83,112,97,119,110,66,97,115,97,108,69,108,101,109,101,110,116,0,83,112,97,119,110,84,111,112,69,108,101,109,101,110,116,0,116,104,105,99,107,110,101,115,115,32,105,110,112,117,116,32,110,101,101,100,101,100,32,116,111,32,99,111,109,112,117,116,101,32,103,105,97,32,100,101,102,108,101,99,116,105,111,110,33,0,109,97,110,116,108,101,32,118,105,115,99,111,115,105,116,121,32,105,110,112,117,116,32,110,101,101,100,101,100,32,116,111,32,99,111,109,112,117,116,101,32,103,105,97,32,100,101,102,108,101,99,116,105,111,110,33,0,108,105,116,104,111,115,112,104,101,114,101,32,116,104,105,99,107,110,101,115,115,32,105,110,112,117,116,32,110,101,101,100,101,100,32,116,111,32,99,111,109,112,117,116,101,32,103,105,97,32,100,101,102,108,101,99,116,105,111,110,33,0,100,101,108,116,97,32,116,104,105,99,107,110,101,115,115,32,105,110,112,117,116,32,110,101,101,100,101,100,32,116,111,32,99,111,109,112,117,116,101,32,115,101,97,32,108,101,118,101,108,32,114,105,115,101,33,0,55,84,114,105,97,82,101,102,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,46,46,47,46,47,69,108,101,109,101,110,116,115,47,84,114,105,97,46,104,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,84,114,105,97,82,101,102,46,99,112,112,0,71,101,116,83,101,103,109,101,110,116,78,111,100,97,108,70,117,110,99,116,105,111,110,115,0,69,100,103,101,32,105,110,100,101,120,32,112,114,111,118,105,100,101,100,32,40,0,41,32,105,115,32,110,111,116,32,98,101,116,119,101,101,110,32,48,32,97,110,100,32,50,0,78,111,100,101,79,110,69,100,103,101,73,110,100,105,99,101,115,0,53,84,101,116,114,97,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,84,101,116,114,97,46,99,112,112,0,67,111,117,108,100,32,110,111,116,32,102,105,110,100,32,51,32,118,101,114,116,105,99,101,115,32,111,110,32,98,101,100,0,70,97,99,101,79,110,66,97,115,101,73,110,100,105,99,101,115,0,70,97,99,101,79,110,70,114,111,110,116,73,110,100,105,99,101,115,0,70,97,99,101,79,110,83,117,114,102,97,99,101,73,110,100,105,99,101,115,0,56,84,101,116,114,97,82,101,102,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,46,46,47,46,47,69,108,101,109,101,110,116,115,47,84,101,116,114,97,46,104,0,86,101,114,116,105,99,97,108,83,101,103,109,101,110,116,73,110,100,105,99,101,115,0,86,101,114,116,101,120,67,111,110,110,101,99,116,105,118,105,116,121,0,86,97,108,117,101,80,49,68,101,114,105,118,97,116,105,118,101,115,79,110,71,97,117,115,115,0,85,112,100,97,116,101,80,111,116,101,110,116,105,97,108,85,110,103,114,111,117,110,100,105,110,103,0,85,112,100,97,116,101,67,111,110,115,116,114,97,105,110,116,115,69,120,116,114,117,100,101,70,114,111,109,84,111,112,0,85,112,100,97,116,101,67,111,110,115,116,114,97,105,110,116,115,69,120,116,114,117,100,101,70,114,111,109,66,97,115,101,0,84,105,109,101,65,100,97,112,116,0,83,116,97,98,105,108,105,122,97,116,105,111,110,80,97,114,97,109,101,116,101,114,0,83,101,116,84,101,109,112,111,114,97,114,121,69,108,101,109,101,110,116,84,121,112,101,0,80,111,116,101,110,116,105,97,108,85,110,103,114,111,117,110,100,105,110,103,0,78,111,100,97,108,86,97,108,117,101,0,78,111,100,97,108,70,117,110,99,116,105,111,110,115,80,50,0,78,111,100,97,108,70,117,110,99,116,105,111,110,115,80,49,68,101,114,105,118,97,116,105,118,101,115,0,78,111,100,97,108,70,117,110,99,116,105,111,110,115,80,49,0,78,111,100,97,108,70,117,110,99,116,105,111,110,115,77,73,78,73,68,101,114,105,118,97,116,105,118,101,115,0,78,101,119,71,97,117,115,115,76,105,110,101,0,77,105,110,69,100,103,101,76,101,110,103,116,104,0,74,97,99,111,98,105,97,110,68,101,116,101,114,109,105,110,97,110,116,76,105,110,101,0,73,115,90,101,114,111,76,101,118,101,108,115,101,116,0,73,115,78,111,100,101,79,110,83,104,101,108,102,70,114,111,109,70,108,97,103,115,0,71,101,116,85,112,112,101,114,69,108,101,109,101,110,116,0,71,101,116,83,111,108,117,116,105,111,110,70,114,111,109,73,110,112,117,116,115,79,110,101,68,111,102,0,71,101,116,78,111,100,101,0,71,101,116,73,99,101,102,114,111,110,116,67,111,111,114,100,105,110,97,116,101,115,0,71,101,116,71,114,111,117,110,100,101,100,80,97,114,116,0,71,101,116,66,97,115,97,108,69,108,101,109,101,110,116,0,70,83,67,111,110,116,97,99,116,77,105,103,114,97,116,105,111,110,0,67,111,109,112,117,116,101,83,116,114,101,115,115,84,101,110,115,111,114,0,67,111,109,112,117,116,101,68,101,118,105,97,116,111,114,105,99,83,116,114,101,115,115,84,101,110,115,111,114,0,67,97,108,118,105,110,103,82,97,116,101,76,101,118,101,114,109,97,110,110,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,84,101,116,114,97,82,101,102,46,99,112,112,0,71,101,116,74,97,99,111,98,105,97,110,68,101,116,101,114,109,105,110,97,110,116,70,97,99,101,0,53,80,101,110,116,97,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,80,101,110,116,97,46,99,112,112,0,65,100,100,66,97,115,97,108,73,110,112,117,116,0,78,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,33,0,65,118,101,114,97,103,101,79,110,116,111,80,97,114,116,105,116,105,111,110,0,67,111,109,112,117,116,101,66,97,115,97,108,83,116,114,101,115,115,0,67,111,110,116,114,111,108,73,110,112,117,116,83,101,116,71,114,97,100,105,101,110,116,0,32,105,115,32,110,111,116,32,97,32,67,111,110,116,114,111,108,73,110,112,117,116,0,67,111,110,116,114,111,108,84,111,86,101,99,116,111,114,115,0,82,101,115,112,111,110,115,101,32,116,121,112,101,32,0,69,108,101,109,101,110,116,82,101,115,112,111,110,115,101,0,69,114,114,111,114,46,32,80,114,111,98,108,101,109,32,119,105,116,104,32,112,111,114,116,105,111,110,32,111,102,32,103,114,111,117,110,100,101,100,32,101,108,101,109,101,110,116,58,32,118,97,108,117,101,32,115,104,111,117,108,100,32,98,101,32,98,101,116,119,101,101,110,32,48,32,97,110,100,32,49,0,71,101,116,71,114,111,117,110,100,101,100,80,111,114,116,105,111,110,0,78,111,32,105,110,112,117,116,32,111,102,32,116,121,112,101,32,0,32,102,111,117,110,100,32,105,110,32,116,114,105,97,0,78,111,100,101,32,112,114,111,118,105,100,101,100,32,110,111,116,32,102,111,117,110,100,32,97,109,111,110,103,32,101,108,101,109,101,110,116,32,110,111,100,101,115,0,71,101,116,78,111,100,101,73,110,100,101,120,0,71,101,116,86,101,99,116,111,114,70,114,111,109,67,111,110,116,114,111,108,73,110,112,117,116,115,0,105,110,112,117,116,32,0,73,110,112,117,116,67,111,110,116,114,111,108,85,112,100,97,116,101,0,99,111,117,108,100,32,110,111,116,32,102,105,110,100,32,105,110,112,117,116,32,119,105,116,104,32,101,110,117,109,32,0,73,110,112,117,116,68,101,112,116,104,65,118,101,114,97,103,101,65,116,66,97,115,101,0,111,98,106,101,99,116,32,0,73,110,112,117,116,69,120,116,114,117,100,101,0,99,111,117,108,100,32,110,111,116,32,102,105,110,100,32,111,108,100,32,105,110,112,117,116,32,119,105,116,104,32,101,110,117,109,58,32,0,73,110,112,117,116,83,99,97,108,101,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,83,111,108,117,116,105,111,110,79,110,101,68,111,102,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,83,111,108,117,116,105,111,110,79,110,101,68,111,102,67,111,108,108,97,112,115,101,100,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,86,101,99,116,111,114,0,116,121,112,101,32,0,41,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,0,70,108,111,119,32,101,113,117,97,116,105,111,110,32,97,112,112,114,111,120,105,109,97,116,105,111,110,0,82,101,115,101,116,70,83,66,97,115,97,108,66,111,117,110,100,97,114,121,67,111,110,100,105,116,105,111,110,0,83,101,116,67,111,110,116,114,111,108,73,110,112,117,116,115,70,114,111,109,86,101,99,116,111,114,0,85,112,100,97,116,101,0,67,97,115,101,32,110,111,116,32,99,111,118,101,114,101,100,0,90,101,114,111,76,101,118,101,108,115,101,116,67,111,111,114,100,105,110,97,116,101,115,0,71,73,65,32,100,101,102,108,101,99,116,105,111,110,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,33,0,71,105,97,68,101,102,108,101,99,116,105,111,110,0,56,80,101,110,116,97,82,101,102,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,46,46,47,46,47,69,108,101,109,101,110,116,115,47,80,101,110,116,97,46,104,0,71,101,116,65,114,101,97,51,68,0,79,99,101,97,110,65,114,101,97,0,79,99,101,97,110,65,118,101,114,97,103,101,0,83,101,97,108,101,118,101,108,114,105,115,101,78,111,110,69,117,115,116,97,116,105,99,0,83,101,97,108,101,118,101,108,114,105,115,101,69,117,115,116,97,116,105,99,0,84,101,110,115,111,114,73,110,116,101,114,112,111,108,97,116,105,111,110,0,78,101,119,71,97,117,115,115,0,77,105,115,102,105,116,65,114,101,97,0,73,115,70,97,99,101,79,110,66,111,117,110,100,97,114,121,0,71,101,116,76,101,118,101,108,115,101,116,80,111,115,105,116,105,118,101,80,97,114,116,0,71,101,116,76,101,118,101,108,67,111,111,114,100,105,110,97,116,101,115,0,67,111,109,112,117,116,101,83,105,103,109,97,78,78,0,67,104,97,114,97,99,116,101,114,105,115,116,105,99,76,101,110,103,116,104,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,69,108,101,109,101,110,116,115,47,80,101,110,116,97,82,101,102,46,99,112,112,0,66,97,115,97,108,78,111,100,101,73,110,100,105,99,101,115,0,78,117,109,98,101,114,111,102,78,111,100,101,115,0,71,101,116,78,111,100,97,108,70,117,110,99,116,105,111,110,115,0,110,101,103,97,116,105,118,101,32,106,97,99,111,98,105,97,110,32,100,101,116,101,114,109,105,110,97,110,116,33,0,71,101,116,74,97,99,111,98,105,97,110,68,101,116,101,114,109,105,110,97,110,116,0,71,101,116,78,111,100,97,108,70,117,110],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+67880);allocate([99,116,105,111,110,115,68,101,114,105,118,97,116,105,118,101,115,82,101,102,101,114,101,110,99,101,0,71,101,116,81,117,97,100,74,97,99,111,98,105,97,110,68,101,116,101,114,109,105,110,97,110,116,0,71,101,116,83,101,103,109,101,110,116,74,97,99,111,98,105,97,110,68,101,116,101,114,109,105,110,97,110,116,0,71,101,116,84,114,105,97,74,97,99,111,98,105,97,110,68,101,116,101,114,109,105,110,97,110,116,0,80,114,101,115,115,117,114,101,73,110,116,101,114,112,111,108,97,116,105,111,110,0,83,117,114,102,97,99,101,78,111,100,101,73,110,100,105,99,101,115,0,86,101,108,111,99,105,116,121,73,110,116,101,114,112,111,108,97,116,105,111,110,0,99,111,117,108,100,32,110,111,116,32,102,105,110,100,32,105,110,112,117,116,32,119,105,116,104,32,101,110,117,109,32,116,121,112,101,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,73,110,112,117,116,115,46,99,112,112,0,99,111,117,108,100,32,110,111,116,32,102,105,110,100,32,105,110,112,117,116,32,119,105,116,104,32,101,110,117,109,58,32,0,68,117,112,108,105,99,97,116,101,73,110,112,117,116,0,56,83,101,103,73,110,112,117,116,0,32,32,32,83,101,103,73,110,112,117,116,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,83,101,103,73,110,112,117,116,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,47,73,110,112,117,116,115,47,83,101,103,73,110,112,117,116,46,104,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,73,110,112,117,116,46,104,0,57,84,114,105,97,73,110,112,117,116,0,32,32,32,84,114,105,97,73,110,112,117,116,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,84,114,105,97,73,110,112,117,116,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,47,73,110,112,117,116,115,47,84,114,105,97,73,110,112,117,116,46,104,0,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,32,40,73,110,112,117,116,32,105,115,32,0,57,66,111,111,108,73,110,112,117,116,0,32,32,32,66,111,111,108,73,110,112,117,116,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,66,111,111,108,73,110,112,117,116,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,47,73,110,112,117,116,115,47,66,111,111,108,73,110,112,117,116,46,104,0,67,111,110,115,116,114,97,105,110,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,102,111,114,32,98,111,111,108,101,97,110,115,0,77,105,110,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,102,111,114,32,98,111,111,108,101,97,110,115,0,77,97,120,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,102,111,114,32,98,111,111,108,101,97,110,115,0,73,110,102,105,110,105,116,121,78,111,114,109,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,102,111,114,32,98,111,111,108,101,97,110,115,0,56,73,110,116,73,110,112,117,116,0,32,32,32,73,110,116,73,110,112,117,116,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,73,110,116,73,110,112,117,116,46,99,112,112,0,73,110,116,73,110,112,117,116,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,73,115,115,109,68,111,117,98,108,101,32,105,110,32,112,97,114,97,108,108,101,108,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,47,73,110,112,117,116,115,47,73,110,116,73,110,112,117,116,46,104,0,77,105,110,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,102,111,114,32,105,110,116,101,103,101,114,115,0,77,97,120,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,102,111,114,32,105,110,116,101,103,101,114,115,0,73,110,102,105,110,105,116,121,78,111,114,109,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,102,111,114,32,105,110,116,101,103,101,114,115,0,49,49,68,111,117,98,108,101,73,110,112,117,116,0,32,32,32,68,111,117,98,108,101,73,110,112,117,116,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,68,111,117,98,108,101,73,110,112,117,116,46,99,112,112,0,68,111,117,98,108,101,32,105,110,112,117,116,32,111,102,32,101,110,117,109,32,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,98,111,111,108,101,97,110,0,110,111,116,32,115,117,112,112,111,114,116,101,32,121,101,116,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,47,73,110,112,117,116,115,47,68,111,117,98,108,101,73,110,112,117,116,46,104,0,49,54,68,111,117,98,108,101,65,114,114,97,121,73,110,112,117,116,0,32,32,32,68,111,117,98,108,101,65,114,114,97,121,73,110,112,117,116,32,0,32,83,105,122,101,58,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,68,111,117,98,108,101,65,114,114,97,121,73,110,112,117,116,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,47,73,110,112,117,116,115,47,68,111,117,98,108,101,65,114,114,97,121,73,110,112,117,116,46,104,0,83,112,97,119,110,84,114,105,97,73,110,112,117,116,0,49,50,68,97,116,97,115,101,116,73,110,112,117,116,0,68,97,116,97,115,101,116,73,110,112,117,116,58,10,0,32,32,32,110,117,109,105,100,115,58,0,32,32,32,105,110,112,117,116,115,58,32,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,46,47,115,104,97,114,101,100,47,46,47,77,101,109,79,112,115,47,77,101,109,79,112,115,46,104,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,68,97,116,97,115,101,116,73,110,112,117,116,46,99,112,112,0,67,111,117,108,100,32,110,111,116,32,102,105,110,100,32,105,110,112,117,116,32,111,102,32,105,100,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,47,73,110,112,117,116,115,47,68,97,116,97,115,101,116,73,110,112,117,116,46,104,0,71,101,116,82,101,115,117,108,116,65,114,114,97,121,83,105,122,101,0,71,101,116,82,101,115,117,108,116,78,117,109,98,101,114,79,102,78,111,100,101,115,0,71,101,116,82,101,115,117,108,116,73,110,116,101,114,112,111,108,97,116,105,111,110,0,77,105,110,0,71,101,116,73,110,112,117,116,65,118,101,114,97,103,101,0,71,101,116,73,110,112,117,116,68,101,114,105,118,97,116,105,118,101,86,97,108,117,101,0,54,77,97,116,105,99,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,77,97,116,101,114,105,97,108,115,47,77,97,116,105,99,101,46,99,112,112,0,77,97,116,105,99,101,58,10,0,32,32,32,101,108,101,109,101,110,116,58,10,0,78,101,103,97,116,105,118,101,32,118,105,115,99,111,115,105,116,121,0,56,77,97,116,101,114,105,97,108,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,77,97,116,101,114,105,97,108,115,47,46,47,77,97,116,105,99,101,46,104,0,54,77,97,116,112,97,114,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,77,97,116,101,114,105,97,108,115,47,77,97,116,112,97,114,46,99,112,112,0,77,97,116,101,114,105,97,108,32,0,77,97,116,112,97,114,58,10,0,32,32,32,109,105,100,58,32,0,32,32,32,114,104,111,95,105,99,101,58,32,0,32,32,32,114,104,111,95,119,97,116,101,114,58,32,0,32,32,32,114,104,111,95,102,114,101,115,104,119,97,116,101,114,58,32,0,32,32,32,109,117,95,119,97,116,101,114,58,32,0,32,32,32,104,101,97,116,99,97,112,97,99,105,116,121,58,32,0,32,32,32,116,104,101,114,109,97,108,99,111,110,100,117,99,116,105,118,105,116,121,58,32,0,32,32,32,116,101,109,112,101,114,97,116,101,105,99,101,99,111,110,100,117,99,116,105,118,105,116,121,58,32,0,32,32,32,108,97,116,101,110,116,104,101,97,116,58,32,0,32,32,32,98,101,116,97,58,32,0,32,32,32,109,101,108,116,105,110,103,112,111,105,110,116,58,32,0,32,32,32,114,101,102,101,114,101,110,99,101,116,101,109,112,101,114,97,116,117,114,101,58,32,0,32,32,32,109,105,120,101,100,95,108,97,121,101,114,95,99,97,112,97,99,105,116,121,58,32,0,32,32,32,116,104,101,114,109,97,108,95,101,120,99,104,97,110,103,101,95,118,101,108,111,99,105,116,121,58,32,0,32,32,32,103,58,32,0,32,32,32,100,101,115,102,97,99,58,32,0,32,32,32,114,108,97,112,115,58,32,0,32,32,32,114,108,97,112,115,108,103,109,58,32,0,32,32,32,100,112,101,114,109,105,108,58,32,0,71,101,116,77,97,116,101,114,105,97,108,80,97,114,97,109,101,116,101,114,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,77,97,116,101,114,105,97,108,115,47,46,46,47,46,47,77,97,116,101,114,105,97,108,115,47,77,97,116,112,97,114,46,104,0,73,115,68,97,109,97,103,101,0,71,101,116,86,105,115,99,111,115,105,116,121,50,100,68,101,114,105,118,97,116,105,118,101,69,112,115,83,113,117,97,114,101,0,71,101,116,86,105,115,99,111,115,105,116,121,95,68,0,71,101,116,86,105,115,99,111,115,105,116,121,95,66,0,71,101,116,86,105,115,99,111,115,105,116,121,68,101,114,105,118,97,116,105,118,101,69,112,115,83,113,117,97,114,101,0,71,101,116,86,105,115,99,111,115,105,116,121,68,67,111,109,112,108,101,109,101,110,116,0,71,101,116,86,105,115,99,111,115,105,116,121,67,111,109,112,108,101,109,101,110,116,0,71,101,116,86,105,115,99,111,115,105,116,121,66,97,114,0,71,101,116,86,105,115,99,111,115,105,116,121,0,71,101,116,78,0,71,101,116,68,98,97,114,0,71,101,116,68,0,71,101,116,66,98,97,114,0,71,101,116,66,0,71,101,116,65,98,97,114,0,71,101,116,65,0,99,111,112,121,50,0,57,83,112,99,83,116,97,116,105,99,0,83,112,99,83,116,97,116,105,99,58,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,67,111,110,115,116,114,97,105,110,116,115,47,83,112,99,83,116,97,116,105,99,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,67,111,110,115,116,114,97,105,110,116,115,47,46,46,47,46,47,67,111,110,115,116,114,97,105,110,116,115,47,83,112,99,83,116,97,116,105,99,46,104,0,49,48,83,112,99,68,121,110,97,109,105,99,0,83,112,99,68,121,110,97,109,105,99,58,10,0,32,32,32,115,105,100,58,32,0,32,32,32,110,111,100,101,105,100,58,32,0,32,32,32,100,111,102,58,32,0,32,32,32,105,115,115,101,116,58,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,67,111,110,115,116,114,97,105,110,116,115,47,83,112,99,68,121,110,97,109,105,99,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,67,111,110,115,116,114,97,105,110,116,115,47,46,46,47,46,47,67,111,110,115,116,114,97,105,110,116,115,47,83,112,99,68,121,110,97,109,105,99,46,104,0,80,101,110,97,108,116,121,68,111,102,65,110,100,86,97,108,117,101,0,55,80,101,110,112,97,105,114,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,80,101,110,112,97,105,114,46,99,112,112,0,80,101,110,112,97,105,114,58,10,0,52,76,111,97,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,46,46,47,46,47,76,111,97,100,115,47,80,101,110,112,97,105,114,46,104,0,55,80,101,110,103,114,105,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,80,101,110,103,114,105,100,46,99,112,112,0,80,101,110,103,114,105,100,58,10,0,32,32,32,97,99,116,105,118,101,32,0,32,32,32,122,105,103,122,97,103,95,99,111,117,110,116,101,114,32,0,97,110,97,108,121,115,105,115,58,32,0,67,111,110,115,116,114,97,105,110,116,65,99,116,105,118,97,116,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,46,46,47,46,47,76,111,97,100,115,47,80,101,110,103,114,105,100,46,104,0,49,51,78,117,109,101,114,105,99,97,108,102,108,117,120,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,78,117,109,101,114,105,99,97,108,102,108,117,120,46,99,112,112,0,78,117,109,101,114,105,99,97,108,102,108,117,120,58,10,0,32,32,32,102,108,117,120,95,116,121,112,101,58,32,0,116,121,112,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,67,114,101,97,116,101,75,77,97,116,114,105,120,77,97,115,115,116,114,97,110,115,112,111,114,116,0,67,114,101,97,116,101,75,77,97,116,114,105,120,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,0,67,114,101,97,116,101,75,77,97,116,114,105,120,65,100,106,111,105,110,116,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,0,67,114,101,97,116,101,80,86,101,99,116,111,114,77,97,115,115,116,114,97,110,115,112,111,114,116,0,67,114,101,97,116,101,80,86,101,99,116,111,114,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,0,78,117,109,101,114,105,99,97,108,102,108,117,120,32,116,121,112,101,32,0,71,101,116,78,111,100,101,115,76,105,100,76,105,115,116,0,71,101,116,78,111,100,101,115,83,105,100,76,105,115,116,0,71,101,116,78,117,109,98,101,114,79,102,78,111,100,101,115,0,67,97,110,110,111,116,32,119,101,97,107,108,121,32,97,112,112,108,121,32,99,111,110,115,116,114,97,105,110,116,32,98,101,99,97,117,115,101,32,78,97,78,32,119,97,115,32,112,114,111,118,105,100,101,100,0,67,114,101,97,116,101,80,86,101,99,116,111,114,77,97,115,115,116,114,97,110,115,112,111,114,116,66,111,117,110,100,97,114,121,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,46,46,47,46,47,76,111,97,100,115,47,78,117,109,101,114,105,99,97,108,102,108,117,120,46,104,0,73,110,112,117,116,32,69,108,101,109,101,110,116,32,77,97,116,114,105,120,32,105,115,32,97,32,78,85,76,76,32,112,111,105,110,116,101,114,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,109,97,116,114,105,120,47,69,108,101,109,101,110,116,77,97,116,114,105,120,46,99,112,112,0,69,108,101,109,101,110,116,77,97,116,114,105,120,0,109,101,114,103,105,110,103,32,50,32,110,111,110,32,100,111,102,115,121,109,109,101,116,114,105,99,97,108,32,109,97,116,114,105,99,101,115,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,0,110,111,110,32,100,111,102,115,121,109,109,101,116,114,105,99,97,108,32,109,97,116,114,105,120,32,65,100,100,84,111,71,108,111,98,97,108,32,114,111,117,116,105,110,101,32,110,111,116,32,115,117,112,112,111,114,116,32,121,101,116,33,0,65,100,100,84,111,71,108,111,98,97,108,0,65,100,100,68,105,97,103,111,110,97,108,84,111,71,108,111,98,97,108,0,76,117,109,112,0,84,114,97,110,115,112,111,115,101,0,84,119,111,32,105,110,112,117,116,32,101,108,101,109,101,110,116,32,109,97,116,114,105,99,101,115,32,97,114,101,32,78,85,76,76,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,109,97,116,114,105,120,47,69,108,101,109,101,110,116,86,101,99,116,111,114,46,99,112,112,0,69,108,101,109,101,110,116,86,101,99,116,111,114,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,109,97,116,114,105,120,47,46,46,47,46,46,47,115,104,97,114,101,100,47,46,47,77,101,109,79,112,115,47,77,101,109,79,112,115,46,104,0,99,111,117,108,100,32,110,111,116,32,102,105,110,100,32,112,97,114,97,109,101,116,101,114,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,80,97,114,97,109,101,116,101,114,115,46,99,112,112,0,70,105,110,100,80,97,114,97,109,0,99,111,117,108,100,32,102,105,110,100,32,110,101,105,116,104,101,114,32,97,32,100,101,102,97,117,108,116,32,97,110,97,108,121,115,105,115,32,110,111,114,32,97,110,97,108,121,115,105,115,32,0,79,112,116,105,111,110,115,70,114,111,109,65,110,97,108,121,115,105,115,0,57,66,111,111,108,80,97,114,97,109,0,32,32,32,66,111,111,108,80,97,114,97,109,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,66,111,111,108,80,97,114,97,109,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,66,111,111,108,80,97,114,97,109,46,104,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,110,32,73,115,115,109,80,68,111,117,98,108,101,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,110,32,105,110,116,0,56,73,110,116,80,97,114,97,109,0,32,32,32,73,110,116,80,97,114,97,109,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,73,110,116,80,97,114,97,109,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,73,110,116,80,97,114,97,109,46,104,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,110,32,105,110,116,32,97,114,114,97,121,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,110,32,73,115,115,109,68,111,117,98,108,101,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,98,111,111,108,0,49,49,73,110,116,86,101,99,80,97,114,97,109,0,73,110,116,86,101,99,80,97,114,97,109,58,10,0,32,32,32,118,101,99,116,111,114,32,115,105,122,101,58,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,73,110,116,86,101,99,80,97,114,97,109,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,73,110,116,86,101,99,80,97,114,97,109,46,104,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,73,115,115,109,68,111,117,98,108,101,32,97,114,114,97,121,32,40,109,97,121,98,101,32,105,110,32,115,101,114,105,97,108,63,41,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,109,97,116,114,105,120,0,49,49,73,110,116,77,97,116,80,97,114,97,109,0,73,110,116,77,97,116,80,97,114,97,109,58,10,0,40,0,41,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,73,110,116,77,97,116,80,97,114,97,109,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,73,110,116,77,97,116,80,97,114,97,109,46,104,0,49,49,68,111,117,98,108,101,80,97,114,97,109,0,32,32,32,68,111,117,98,108,101,80,97,114,97,109,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,68,111,117,98,108,101,80,97,114,97,109,46,99,112,112,0,68,111,117,98,108,101,32,112,97,114,97,109,32,111,102,32,101,110,117,109,32,0,41,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,110,32,105,110,116,101,103,101,114,0,41,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,110,32,98,111,111,108,0,41,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,110,32,97,114,114,97,121,32,111,102,32,105,110,116,101,103,101,114,115,0,41,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,110,32,97,114,114,97,121,32,111,102,32,73,115,115,109,68,111,117,98,108,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,68,111,117,98,108,101,80,97,114,97,109,46,104,0,57,70,105,108,101,80,97,114,97,109,0,32,32,32,70,105,108,101,80,97,114,97,109,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,70,105,108,101,80,97,114,97,109,46,99,112,112,0,70,105,108,101,32,112,97,114,97,109,32,111,102,32,101,110,117,109,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,70,105,108,101,80,97,114,97,109,46,104,0,49,54,83,116,114,105,110,103,65,114,114,97,121,80,97,114,97,109,0,32,32,32,83,116,114,105,110,103,65,114,114,97,121,80,97,114,97,109,32,0,32,123,0,125,10,0,32,39,0,39,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,83,116,114,105,110,103,65,114,114,97,121,80,97,114,97,109,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,83,116,114,105,110,103,65,114,114,97,121,80,97,114,97,109,46,104,0,86,101,99,32,112,97,114,97,109,32,111,102,32,101,110,117,109,32,0,49,52,68,111,117,98,108,101,77,97,116,80,97,114,97,109,0,68,111,117,98,108,101,77,97,116,80,97,114,97,109,58,10,0,32,32,32,109,97,116,114,105,120,32,115,105,122,101,58,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,68,111,117,98,108,101,77,97,116,80,97,114,97,109,46,99,112,112,0,68,111,117,98,108,101,77,97,116,32,111,102,32,101,110,117,109,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,68,111,117,98,108,101,77,97,116,80,97,114,97,109,46,104,0,49,57,68,111,117,98,108,101,77,97,116,65,114,114,97,121,80,97,114,97,109,0,68,111,117,98,108,101,77,97,116,65,114,114,97,121,80,97,114,97,109,58,10,0,32,32,32,97,114,114,97,121,32,115,105,122,101,58,32,0,32,32,32,97,114,114,97,121,32,112,111,105,110,116,101,114,58,32,0,32,32,32,97,114,114,97,121,32,0,120,0,41,58,10,0,32,32,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,68,111,117,98,108,101,77,97,116,65,114,114,97,121,80,97,114,97,109,46,99,112,112,0,41,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,70,73,76,69,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,68,111,117,98,108,101,77,97,116,65,114,114,97,121,80,97,114,97,109,46,104,0,99,97,110,110,111,116,32,104,111,108,100,32,97,32,77,97,116,0,99,97,110,110,111,116,32,104,111,108,100,32,97,32,86,101,99,0,99,97,110,110,111,116,32,104,111,108,100,32,97,32,105,110,116,32,109,97,116,32,97,114,114,97,121,0,99,97,110,110,111,116,32,104,111,108,100,32,97,32,105,110,116,32,118,101,99,32,97,114,114,97,121,0,99,97,110,110,111,116,32,104,111,108,100,32,97,32,73,115,115,109,68,111,117,98,108,101,32,109,97,116,32,97,114,114,97,121,0,99,97,110,110,111,116,32,104,111,108,100,32,97,32,73,115,115,109,68,111,117,98,108,101,32,118,101,99,32,97,114,114,97,121,0,99,97,110,110,111,116,32,104,111,108,100,32,97,32,115,116,114,105,110,103,32,97,114,114,97,121,0,99,97,110,110,111,116,32,104,111,108,100,32,97,32,115,116,114,105,110,103,0,99,97,110,110,111,116,32,104,111,108,100,32,97,32,115,99,97,108,97,114,0,99,97,110,110,111,116,32,104,111,108,100,32,97,110,32,105,110,116,101,103,101,114,0,99,97,110,110,111,116,32,104,111,108,100,32,97,32,98,111,111,108,101,97,110,0,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,70,73,76,69,0,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,77,97,116,0,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,86,101,99,0,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,73,115,115,109,68,111,117,98,108,101,32,97,114,114,97,121,0,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,115,116,114,105,110,103,32,97,114,114,97,121,0,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,115,116,114,105,110,103,0,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,73,115,115,109,68,111,117,98,108,101,0,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,110,32,97,114,114,97,121,32,111,102,32,105,110,116,101,103,101,114,115,0,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,110,32,105,110,116,101,103,101,114,0,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,98,111,111,108,0,49,52,68,111,117,98,108,101,86,101,99,80,97,114,97,109,0,32,32,32,68,111,117,98,108,101,86,101,99,80,97,114,97,109,32,0,32,115,105,122,101,58,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,68,111,117,98,108,101,86,101,99,80,97,114,97,109,46,99,112,112,0,68,111,117,98,108,101,86,101,99,32,112,97,114,97,109,32,111,102,32,101,110,117,109,32,0,41,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,110,32,97,114,114,97,121,32,111,102,32,105,110,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,68,111,117,98,108,101,86,101,99,80,97,114,97,109,46,104,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,73,115,115,109,68,111,117,98,108,101,32,109,97,116,32,97,114,114,97,121,0,49,49,83,116,114,105,110,103,80,97,114,97,109,0,32,32,32,83,116,114,105,110,103,80,97,114,97,109,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,83,116,114,105,110,103,80,97,114,97,109,46,99,112,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,83,116,114,105,110,103,80,97,114,97,109,46,104,0,66,111,111,108,32,112,97,114,97,109,32,111,102,32,101,110,117,109,32,0,41,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,70,73,76,69,0,49,52,84,114,97,110,115,105,101,110,116,80,97,114,97,109,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,46,47,115,104,97,114,101,100,47,46,47,77,101,109,79,112,115,47,77,101,109,79,112,115,46,104,0,84,114,97,110,115,105,101,110,116,80,97,114,97,109,58,10,0,32,32,32,101,110,117,109,58,32,0,32,32,32,115,105,122,101,58,32,0,116,105,109,101,58,32,0,32,118,97,108,117,101,58,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,84,114,97,110,115,105,101,110,116,80,97,114,97,109,46,99,112,112,0,80,97,114,97,109,101,116,101,114,32,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,110,32,97,114,114,97,121,32,111,102,32,109,97,116,114,105,99,101,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,84,114,97,110,115,105,101,110,116,80,97,114,97,109,46,104,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,70,73,76,69,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,105,110,116,32,109,97,116,32,97,114,114,97,121,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,105,110,116,32,118,101,99,32,97,114,114,97,121,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,73,115,115,109,68,111,117,98,108,101,32,118,101,99,32,97,114,114,97,121,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,115,99,97,108,97,114,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,110,32,105,110,116,101,103,101,114,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,98,111,111,108,101,97,110,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,68,97,116,97,83,101,116,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,70,73,76,69,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,109,97,116,114,105,120,32,97,114,114,97,121,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,97,114,114,97,121,32,111,102,32,105,110,116,101,103,101,114,115,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,110,32,97,114,114,97,121,32,111,102,32,105,110,116,101,103,101,114,115,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,110,32,105,110,116,101,103,101,114,0,49,50,68,97,116,97,83,101,116,80,97,114,97,109,0,32,32,32,68,97,116,97,83,101,116,80,97,114,97,109,32,0,32,45,45,45,45,45,32,98,101,103,105,110,10,0,32,45,45,45,45,45,32,101,110,100,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,68,97,116,97,83,101,116,80,97,114,97,109,46,99,112,112,0,68,97,116,97,83,101,116,32,112,97,114,97,109,32,111,102,32,101,110,117,109,32,0,41,32,99,97,110,110,111,116,32,104,111,108,100,32,97,110,32,97,114,114,97,121,32,111,102,32,109,97,116,114,105,99,101,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,97,114,97,109,115,47,46,46,47,46,47,80,97,114,97,109,115,47,68,97,116,97,83,101,116,80,97,114,97,109,46,104,0,83,101,116,86,97,108,117,101,0,80,97,114,97,109,32,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,102,105,108,101,32,112,111,105,110,116,101,114,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,77,97,116,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,86,101,99,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,105,110,116,32,97,114,114,97,121,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,73,115,115,109,68,111,117,98,108,101,32,97,114,114,97,121,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,115,116,114,105,110,103,32,97,114,114,97,121,0,32,99,97,110,110,111,116,32,104,111,108,100,32,97,32,115,116,114,105,110,103,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,102,105,108,101,32,112,111,105,110,116,101,114,0,71,101,116,80,97,114,97,109,101,116,101,114,86,97,108,117,101,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+78120);allocate([32,77,97,116,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,86,101,99,0,41,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,109,97,116,114,105,120,32,97,114,114,97,121,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,73,115,115,109,68,111,117,98,108,101,32,97,114,114,97,121,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,115,116,114,105,110,103,32,97,114,114,97,121,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,115,116,114,105,110,103,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,73,115,115,109,68,111,117,98,108,101,32,102,111,114,32,97,32,103,105,118,101,110,32,116,105,109,101,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,73,115,115,109,68,111,117,98,108,101,0,32,99,97,110,110,111,116,32,114,101,116,117,114,110,32,97,32,98,111,111,108,0,56,80,114,111,102,105,108,101,114,0,80,114,111,102,105,108,101,114,58,10,0,32,32,32,116,105,109,101,32,116,97,103,115,58,32,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,80,114,111,102,105,108,101,114,46,99,112,112,0,77,97,116,114,105,120,32,65,32,97,110,100,32,66,32,105,110,110,101,114,32,118,101,99,116,111,114,115,32,110,111,116,32,101,113,117,97,108,32,115,105,122,101,46,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,104,97,114,101,100,47,77,97,116,114,105,120,47,77,97,116,114,105,120,85,116,105,108,115,46,99,112,112,0,84,114,105,112,108,101,77,117,108,116,105,112,108,121,0,77,97,116,114,105,120,32,66,32,97,110,100,32,67,32,105,110,110,101,114,32,118,101,99,116,111,114,115,32,110,111,116,32,101,113,117,97,108,32,115,105,122,101,46,0,77,97,116,114,105,120,32,65,32,97,110,100,32,66,32,105,110,110,101,114,32,118,101,99,116,111,114,115,32,110,111,116,32,101,113,117,97,108,32,115,105,122,101,0,77,97,116,114,105,120,77,117,108,116,105,112,108,121,0,78,111,32,114,105,103,104,116,45,104,97,110,100,32,115,105,100,101,32,102,111,114,32,110,118,101,99,61,0,77,97,116,114,105,120,73,110,118,101,114,115,101,0,80,105,118,111,116,32,0,32,108,101,115,115,32,116,104,97,110,32,109,97,99,104,105,110,101,32,101,112,115,105,108,111,110,0,68,101,116,101,114,109,105,110,97,110,116,32,115,109,97,108,108,101,114,32,116,104,97,110,32,109,97,99,104,105,110,101,32,101,112,115,105,108,111,110,0,77,97,116,114,105,120,50,120,50,73,110,118,101,114,116,0,77,97,116,114,105,120,51,120,51,73,110,118,101,114,116,0,84,104,105,115,32,102,117,110,99,116,105,111,110,32,115,104,111,117,108,100,32,111,110,108,121,32,98,101,32,99,97,108,108,101,100,32,98,121,32,99,112,117,32,48,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,104,97,114,101,100,47,105,111,47,68,105,115,107,47,112,102,111,112,101,110,46,99,112,112,0,112,102,111,112,101,110,48,0,99,111,117,108,100,32,110,111,116,32,111,112,101,110,32,102,105,108,101,32,0,32,102,111,114,32,98,105,110,97,114,121,32,114,101,97,100,105,110,103,32,111,114,32,119,114,105,116,105,110,103,0,112,102,111,112,101,110,0,99,111,117,108,100,32,110,111,116,32,99,108,111,115,101,32,102,105,108,101,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,104,97,114,101,100,47,105,111,47,68,105,115,107,47,112,102,99,108,111,115,101,46,99,112,112,0,112,102,99,108,111,115,101,0,67,97,110,110,111,116,32,114,101,116,117,114,110,32,99,111,109,109,32,105,110,32,115,101,114,105,97,108,32,109,111,100,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,104,97,114,101,100,47,105,111,47,67,111,109,109,47,73,115,115,109,67,111,109,109,46,99,112,112,0,71,101,116,67,111,109,109,0,117,110,107,110,111,119,110,0,70,101,109,77,111,100,101,108,0,70,101,109,77,111,100,101,108,67,111,109,109,0,87,111,114,108,100,67,111,109,109,0,73,99,101,99,97,112,84,111,69,97,114,116,104,67,111,109,109,0,78,117,109,77,111,100,101,108,115,0,77,111,100,101,108,73,100,0,69,97,114,116,104,73,100,0,65,117,116,111,100,105,102,102,73,115,97,117,116,111,100,105,102,102,0,65,117,116,111,100,105,102,102,78,117,109,68,101,112,101,110,100,101,110,116,115,0,65,117,116,111,100,105,102,102,78,117,109,68,101,112,101,110,100,101,110,116,79,98,106,101,99,116,115,0,65,117,116,111,100,105,102,102,68,101,112,101,110,100,101,110,116,79,98,106,101,99,116,78,97,109,101,115,0,65,117,116,111,100,105,102,102,68,101,112,101,110,100,101,110,116,79,98,106,101,99,116,84,121,112,101,115,0,65,117,116,111,100,105,102,102,68,101,112,101,110,100,101,110,116,79,98,106,101,99,116,73,110,100,105,99,101,115,0,65,117,116,111,100,105,102,102,68,101,112,101,110,100,101,110,116,79,98,106,101,99,116,115,0,65,117,116,111,100,105,102,102,78,117,109,73,110,100,101,112,101,110,100,101,110,116,115,0,65,117,116,111,100,105,102,102,78,117,109,73,110,100,101,112,101,110,100,101,110,116,79,98,106,101,99,116,115,0,65,117,116,111,100,105,102,102,73,110,100,101,112,101,110,100,101,110,116,79,98,106,101,99,116,78,97,109,101,115,0,65,117,116,111,100,105,102,102,73,110,100,101,112,101,110,100,101,110,116,79,98,106,101,99,116,84,121,112,101,115,0,65,117,116,111,100,105,102,102,73,110,100,101,112,101,110,100,101,110,116,79,98,106,101,99,116,115,0,65,117,116,111,100,105,102,102,74,97,99,111,98,105,97,110,0,65,117,116,111,100,105,102,102,88,112,0,65,117,116,111,100,105,102,102,68,114,105,118,101,114,0,65,117,116,111,100,105,102,102,70,111,115,70,111,114,119,97,114,100,73,110,100,101,120,0,65,117,116,111,100,105,102,102,70,111,118,70,111,114,119,97,114,100,73,110,100,105,99,101,115,0,65,117,116,111,100,105,102,102,70,111,115,82,101,118,101,114,115,101,73,110,100,101,120,0,65,117,116,111,100,105,102,102,77,97,115,115,70,108,117,120,83,101,103,109,101,110,116,115,80,114,101,115,101,110,116,0,65,117,116,111,100,105,102,102,75,101,101,112,0,65,117,116,111,100,105,102,102,79,98,117,102,115,105,122,101,0,65,117,116,111,100,105,102,102,76,98,117,102,115,105,122,101,0,65,117,116,111,100,105,102,102,67,98,117,102,115,105,122,101,0,65,117,116,111,100,105,102,102,84,98,117,102,115,105,122,101,0,65,117,116,111,100,105,102,102,71,99,84,114,105,103,103,101,114,82,97,116,105,111,0,65,117,116,111,100,105,102,102,71,99,84,114,105,103,103,101,114,77,97,120,83,105,122,101,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,83,112,99,116,104,105,99,107,110,101,115,115,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,83,116,97,98,105,108,105,122,97,116,105,111,110,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,84,104,105,99,107,101,110,105,110,103,82,97,116,101,0,66,97,115,97,108,102,111,114,99,105,110,103,115,0,66,97,115,97,108,102,111,114,99,105,110,103,115,71,101,111,116,104,101,114,109,97,108,102,108,117,120,0,66,97,115,97,108,102,111,114,99,105,110,103,115,71,114,111,117,110,100,101,100,105,99,101,77,101,108,116,105,110,103,82,97,116,101,0,66,97,115,97,108,102,111,114,99,105,110,103,115,70,108,111,97,116,105,110,103,105,99,101,77,101,108,116,105,110,103,82,97,116,101,0,66,97,115,97,108,102,111,114,99,105,110,103,115,68,101,101,112,119,97,116,101,114,77,101,108,116,105,110,103,82,97,116,101,0,66,97,115,97,108,102,111,114,99,105,110,103,115,68,101,101,112,119,97,116,101,114,69,108,101,118,97,116,105,111,110,0,66,97,115,97,108,102,111,114,99,105,110,103,115,85,112,112,101,114,119,97,116,101,114,69,108,101,118,97,116,105,111,110,0,66,97,115,97,108,102,111,114,99,105,110,103,115,77,101,108,116,114,97,116,101,70,97,99,116,111,114,0,66,97,115,97,108,102,111,114,99,105,110,103,115,84,104,114,101,115,104,111,108,100,84,104,105,99,107,110,101,115,115,0,66,97,115,97,108,102,111,114,99,105,110,103,115,85,112,112,101,114,100,101,112,116,104,77,101,108,116,0,66,97,115,97,108,102,111,114,99,105,110,103,115,77,97,110,116,108,101,99,111,110,100,117,99,116,105,118,105,116,121,0,66,97,115,97,108,102,111,114,99,105,110,103,115,78,117,115,115,101,108,116,0,66,97,115,97,108,102,111,114,99,105,110,103,115,68,116,98,103,0,66,97,115,97,108,102,111,114,99,105,110,103,115,80,108,117,109,101,114,97,100,105,117,115,0,66,97,115,97,108,102,111,114,99,105,110,103,115,84,111,112,112,108,117,109,101,100,101,112,116,104,0,66,97,115,97,108,102,111,114,99,105,110,103,115,66,111,116,116,111,109,112,108,117,109,101,100,101,112,116,104,0,66,97,115,97,108,102,111,114,99,105,110,103,115,80,108,117,109,101,120,0,66,97,115,97,108,102,111,114,99,105,110,103,115,80,108,117,109,101,121,0,66,97,115,97,108,102,111,114,99,105,110,103,115,67,114,117,115,116,116,104,105,99,107,110,101,115,115,0,66,97,115,97,108,102,111,114,99,105,110,103,115,85,112,112,101,114,99,114,117,115,116,116,104,105,99,107,110,101,115,115,0,66,97,115,97,108,102,111,114,99,105,110,103,115,85,112,112,101,114,99,114,117,115,116,104,101,97,116,0,66,97,115,97,108,102,111,114,99,105,110,103,115,76,111,119,101,114,99,114,117,115,116,104,101,97,116,0,70,108,111,97,116,105,110,103,77,101,108,116,82,97,116,101,0,76,105,110,101,97,114,70,108,111,97,116,105,110,103,77,101,108,116,82,97,116,101,0,77,105,115,109,105,112,70,108,111,97,116,105,110,103,77,101,108,116,82,97,116,101,0,77,97,110,116,108,101,80,108,117,109,101,71,101,111,116,104,101,114,109,97,108,70,108,117,120,0,66,101,100,0,66,97,115,101,0,67,111,110,115,116,97,110,116,115,71,0,67,111,110,115,116,97,110,116,115,82,101,102,101,114,101,110,99,101,116,101,109,112,101,114,97,116,117,114,101,0,67,111,110,115,116,97,110,116,115,89,116,115,0,68,101,112,101,110,100,101,110,116,79,98,106,101,99,116,0,83,116,114,101,115,115,98,97,108,97,110,99,101,65,98,115,116,111,108,0,83,116,114,101,115,115,98,97,108,97,110,99,101,73,115,110,101,119,116,111,110,0,83,116,114,101,115,115,98,97,108,97,110,99,101,77,97,120,105,116,101,114,0,83,116,114,101,115,115,98,97,108,97,110,99,101,80,101,110,97,108,116,121,70,97,99,116,111,114,0,83,116,114,101,115,115,98,97,108,97,110,99,101,82,101,102,101,114,101,110,116,105,97,108,0,83,116,114,101,115,115,98,97,108,97,110,99,101,82,101,108,116,111,108,0,83,116,114,101,115,115,98,97,108,97,110,99,101,78,117,109,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,83,116,114,101,115,115,98,97,108,97,110,99,101,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,83,116,114,101,115,115,98,97,108,97,110,99,101,82,101,115,116,111,108,0,83,116,114,101,115,115,98,97,108,97,110,99,101,82,105,102,116,80,101,110,97,108,116,121,76,111,99,107,0,83,116,114,101,115,115,98,97,108,97,110,99,101,82,105,102,116,80,101,110,97,108,116,121,84,104,114,101,115,104,111,108,100,0,83,116,114,101,115,115,98,97,108,97,110,99,101,83,104,101,108,102,68,97,109,112,101,110,105,110,103,0,83,116,114,101,115,115,98,97,108,97,110,99,101,83,112,99,118,120,0,83,116,114,101,115,115,98,97,108,97,110,99,101,83,112,99,118,121,0,83,116,114,101,115,115,98,97,108,97,110,99,101,83,112,99,118,122,0,83,116,114,101,115,115,98,97,108,97,110,99,101,70,83,114,101,99,111,110,100,105,116,105,111,110,105,110,103,0,83,116,114,101,115,115,98,97,108,97,110,99,101,86,101,114,116,101,120,80,97,105,114,105,110,103,0,83,116,114,101,115,115,98,97,108,97,110,99,101,86,105,115,99,111,115,105,116,121,79,118,101,114,115,104,111,111,116,0,76,111,97,100,105,110,103,102,111,114,99,101,88,0,76,111,97,100,105,110,103,102,111,114,99,101,89,0,76,111,97,100,105,110,103,102,111,114,99,101,90,0,70,108,111,119,101,113,117,97,116,105,111,110,66,111,114,100,101,114,83,83,65,0,70,108,111,119,101,113,117,97,116,105,111,110,66,111,114,100,101,114,72,79,0,70,108,111,119,101,113,117,97,116,105,111,110,66,111,114,100,101,114,70,83,0,70,108,111,119,101,113,117,97,116,105,111,110,69,108,101,109,101,110,116,69,113,117,97,116,105,111,110,0,70,108,111,119,101,113,117,97,116,105,111,110,73,115,83,73,65,0,70,108,111,119,101,113,117,97,116,105,111,110,73,115,83,83,65,0,70,108,111,119,101,113,117,97,116,105,111,110,73,115,76,49,76,50,0,70,108,111,119,101,113,117,97,116,105,111,110,73,115,72,79,0,70,108,111,119,101,113,117,97,116,105,111,110,73,115,70,83,0,70,108,111,119,101,113,117,97,116,105,111,110,70,101,83,83,65,0,70,108,111,119,101,113,117,97,116,105,111,110,70,101,72,79,0,70,108,111,119,101,113,117,97,116,105,111,110,70,101,70,83,0,70,108,111,119,101,113,117,97,116,105,111,110,86,101,114,116,101,120,69,113,117,97,116,105,111,110,0,70,114,105,99,116,105,111,110,65,115,0,70,114,105,99,116,105,111,110,67,111,101,102,102,105,99,105,101,110,116,0,70,114,105,99,116,105,111,110,67,111,101,102,102,105,99,105,101,110,116,99,111,117,108,111,109,98,0,70,114,105,99,116,105,111,110,80,0,70,114,105,99,116,105,111,110,81,0,70,114,105,99,116,105,111,110,77,0,70,114,105,99,116,105,111,110,67,0,70,114,105,99,116,105,111,110,76,97,119,0,70,114,105,99,116,105,111,110,71,97,109,109,97,0,70,114,105,99,116,105,111,110,70,0,70,114,105,99,116,105,111,110,87,97,116,101,114,76,97,121,101,114,0,70,114,105,99,116,105,111,110,69,102,102,101,99,116,105,118,101,80,114,101,115,115,117,114,101,0,70,114,105,99,116,105,111,110,67,111,117,112,108,105,110,103,0,71,101,111,109,101,116,114,121,72,121,100,114,111,115,116,97,116,105,99,82,97,116,105,111,0,72,121,100,114,111,108,111,103,121,77,111,100,101,108,0,72,121,100,114,111,108,111,103,121,115,104,114,101,118,101,0,72,121,100,114,111,108,111,103,121,115,104,114,101,118,101,83,112,99,119,97,116,101,114,99,111,108,117,109,110,0,72,121,100,114,111,108,111,103,121,115,104,114,101,118,101,83,116,97,98,105,108,105,122,97,116,105,111,110,0,72,121,100,114,111,108,111,103,121,100,99,0,83,101,100,105,109,101,110,116,72,101,97,100,0,83,101,100,105,109,101,110,116,72,101,97,100,79,108,100,0,83,101,100,105,109,101,110,116,72,101,97,100,82,101,115,105,100,117,97,108,0,69,102,102,101,99,116,105,118,101,80,114,101,115,115,117,114,101,0,69,112,108,72,101,97,100,0,69,112,108,72,101,97,100,79,108,100,0,69,112,108,72,101,97,100,83,108,111,112,101,88,0,69,112,108,72,101,97,100,83,108,111,112,101,89,0,69,112,108,90,105,103,90,97,103,67,111,117,110,116,101,114,0,72,121,100,114,111,108,111,103,121,100,99,77,97,120,73,116,101,114,0,72,121,100,114,111,108,111,103,121,100,99,82,101,108,84,111,108,0,72,121,100,114,111,108,111,103,121,100,99,83,112,99,115,101,100,105,109,101,110,116,72,101,97,100,0,72,121,100,114,111,108,111,103,121,100,99,83,101,100,105,109,101,110,116,67,111,109,112,114,101,115,115,105,98,105,108,105,116,121,0,72,121,100,114,111,108,111,103,121,100,99,83,101,100,105,109,101,110,116,80,111,114,111,115,105,116,121,0,72,121,100,114,111,108,111,103,121,100,99,83,101,100,105,109,101,110,116,84,104,105,99,107,110,101,115,115,0,72,121,100,114,111,108,111,103,121,100,99,83,101,100,105,109,101,110,116,84,114,97,110,115,109,105,116,105,118,105,116,121,0,72,121,100,114,111,108,111,103,121,100,99,87,97,116,101,114,67,111,109,112,114,101,115,115,105,98,105,108,105,116,121,0,72,121,100,114,111,108,111,103,121,100,99,83,112,99,101,112,108,72,101,97,100,0,72,121,100,114,111,108,111,103,121,100,99,77,97,115,107,69,112,108,97,99,116,105,118,101,78,111,100,101,0,72,121,100,114,111,108,111,103,121,100,99,77,97,115,107,69,112,108,97,99,116,105,118,101,69,108,116,0,72,121,100,114,111,108,111,103,121,100,99,69,112,108,67,111,109,112,114,101,115,115,105,98,105,108,105,116,121,0,72,121,100,114,111,108,111,103,121,100,99,69,112,108,80,111,114,111,115,105,116,121,0,72,121,100,114,111,108,111,103,121,100,99,69,112,108,73,110,105,116,105,97,108,84,104,105,99,107,110,101,115,115,0,72,121,100,114,111,108,111,103,121,100,99,69,112,108,67,111,108,97,112,115,101,84,104,105,99,107,110,101,115,115,0,72,121,100,114,111,108,111,103,121,100,99,69,112,108,77,97,120,84,104,105,99,107,110,101,115,115,0,72,121,100,114,111,108,111,103,121,100,99,69,112,108,84,104,105,99,107,110,101,115,115,0,72,121,100,114,111,108,111,103,121,100,99,69,112,108,84,104,105,99,107,110,101,115,115,79,108,100,0,72,121,100,114,111,108,111,103,121,100,99,69,112,108,84,104,105,99,107,67,111,109,112,0,72,121,100,114,111,108,111,103,121,100,99,69,112,108,67,111,110,100,117,99,116,105,118,105,116,121,0,72,121,100,114,111,108,111,103,121,100,99,73,115,101,102,102,105,99,105,101,110,116,108,97,121,101,114,0,72,121,100,114,111,108,111,103,121,100,99,83,101,100,105,109,101,110,116,108,105,109,105,116,70,108,97,103,0,72,121,100,114,111,108,111,103,121,100,99,83,101,100,105,109,101,110,116,108,105,109,105,116,0,72,121,100,114,111,108,111,103,121,100,99,84,114,97,110,115,102,101,114,70,108,97,103,0,72,121,100,114,111,108,111,103,121,100,99,76,101,97,107,97,103,101,70,97,99,116,111,114,0,72,121,100,114,111,108,111,103,121,100,99,80,101,110,97,108,116,121,70,97,99,116,111,114,0,72,121,100,114,111,108,111,103,121,100,99,80,101,110,97,108,116,121,76,111,99,107,0,72,121,100,114,111,108,111,103,121,100,99,69,112,108,102,108,105,112,76,111,99,107,0,72,121,100,114,111,108,111,103,121,100,99,66,97,115,97,108,77,111,117,108,105,110,73,110,112,117,116,0,72,121,100,114,111,108,111,103,121,76,97,121,101,114,0,72,121,100,114,111,108,111,103,121,83,101,100,105,109,101,110,116,0,72,121,100,114,111,108,111,103,121,69,102,102,105,99,105,101,110,116,0,72,121,100,114,111,108,111,103,121,83,101,100,105,109,101,110,116,75,109,97,120,0,72,121,100,114,111,108,111,103,121,115,111,109,109,101,114,115,0,72,121,100,114,111,108,111,103,121,72,101,97,100,0,72,121,100,114,111,108,111,103,121,71,97,112,72,101,105,103,104,116,0,72,121,100,114,111,108,111,103,121,66,117,109,112,83,112,97,99,105,110,103,0,72,121,100,114,111,108,111,103,121,66,117,109,112,72,101,105,103,104,116,0,72,121,100,114,111,108,111,103,121,69,110,103,108,97,99,105,97,108,73,110,112,117,116,0,72,121,100,114,111,108,111,103,121,77,111,117,108,105,110,73,110,112,117,116,0,72,121,100,114,111,108,111,103,121,82,101,121,110,111,108,100,115,0,72,121,100,114,111,108,111,103,121,78,101,117,109,97,110,110,102,108,117,120,0,72,121,100,114,111,108,111,103,121,83,112,99,104,101,97,100,0,72,121,100,114,111,108,111,103,121,67,111,110,100,117,99,116,105,118,105,116,121,0,73,110,100,101,112,101,110,100,101,110,116,79,98,106,101,99,116,0,73,110,118,101,114,115,105,111,110,67,111,110,116,114,111,108,80,97,114,97,109,101,116,101,114,115,0,73,110,118,101,114,115,105,111,110,67,111,110,116,114,111,108,83,99,97,108,105,110,103,70,97,99,116,111,114,115,0,73,110,118,101,114,115,105,111,110,67,111,115,116,70,117,110,99,116,105,111,110,84,104,114,101,115,104,111,108,100,0,73,110,118,101,114,115,105,111,110,67,111,115,116,70,117,110,99,116,105,111,110,115,67,111,101,102,102,105,99,105,101,110,116,115,0,73,110,118,101,114,115,105,111,110,67,111,115,116,70,117,110,99,116,105,111,110,115,0,73,110,118,101,114,115,105,111,110,71,114,97,100,105,101,110,116,83,99,97,108,105,110,103,0,73,110,118,101,114,115,105,111,110,73,115,99,111,110,116,114,111,108,0,73,110,118,101,114,115,105,111,110,84,121,112,101,0,73,110,118,101,114,115,105,111,110,73,110,99,111,109,112,108,101,116,101,65,100,106,111,105,110,116,0,73,110,118,101,114,115,105,111,110,77,97,120,80,97,114,97,109,101,116,101,114,115,0,73,110,118,101,114,115,105,111,110,77,97,120,105,116,101,114,80,101,114,83,116,101,112,0,73,110,118,101,114,115,105,111,110,77,97,120,105,116,101,114,0,73,110,118,101,114,115,105,111,110,77,97,120,115,116,101,112,115,0,73,110,118,101,114,115,105,111,110,70,97,116,111,108,0,73,110,118,101,114,115,105,111,110,70,114,116,111,108,0,73,110,118,101,114,115,105,111,110,71,97,116,111,108,0,73,110,118,101,114,115,105,111,110,71,114,116,111,108,0,73,110,118,101,114,115,105,111,110,71,116,116,111,108,0,73,110,118,101,114,115,105,111,110,65,108,103,111,114,105,116,104,109,0,73,110,118,101,114,115,105,111,110,77,105,110,80,97,114,97,109,101,116,101,114,115,0,73,110,118,101,114,115,105,111,110,78,115,116,101,112,115,0,73,110,118,101,114,115,105,111,110,68,120,109,105,110,0,73,110,118,101,114,115,105,111,110,78,117,109,67,111,110,116,114,111,108,80,97,114,97,109,101,116,101,114,115,0,73,110,118,101,114,115,105,111,110,78,117,109,67,111,115,116,70,117,110,99,116,105,111,110,115,0,73,110,118,101,114,115,105,111,110,83,116,101,112,84,104,114,101,115,104,111,108,100,0,73,110,118,101,114,115,105,111,110,84,104,105,99,107,110,101,115,115,79,98,115,0,73,110,118,101,114,115,105,111,110,83,117,114,102,97,99,101,79,98,115,0,73,110,118,101,114,115,105,111,110,86,120,79,98,115,0,73,110,118,101,114,115,105,111,110,86,121,79,98,115,0,73,110,118,101,114,115,105,111,110,86,122,79,98,115,0,77,97,115,107,73,99,101,76,101,118,101,108,115,101,116,0,77,97,115,107,79,99,101,97,110,76,101,118,101,108,115,101,116,0,77,97,115,107,76,97,110,100,76,101,118,101,108,115,101,116,0,77,97,116,101,114,105,97,108,115,66,101,116,97,0,77,97,116,101,114,105,97,108,115,72,101,97,116,99,97,112,97,99,105,116,121,0,77,97,116,101,114,105,97,108,115,76,97,116,101,110,116,104,101,97,116,0,77,97,116,101,114,105,97,108,115,77,101,108,116,105,110,103,112,111,105,110,116,0,77,97,116,101,114,105,97,108,115,77,105,120,101,100,76,97,121,101,114,67,97,112,97,99,105,116,121,0,77,97,116,101,114,105,97,108,115,82,104,101,111,108,111,103,121,66,0,77,97,116,101,114,105,97,108,115,82,104,101,111,108,111,103,121,66,98,97,114,0,77,97,116,101,114,105,97,108,115,82,104,101,111,108,111,103,121,76,97,119,0,77,97,116,101,114,105,97,108,115,82,104,101,111,108,111,103,121,78,0,68,97,109,97,103,101,73,115,100,97,109,97,103,101,0,68,97,109,97,103,101,68,0,68,97,109,97,103,101,70,0,68,97,109,97,103,101,68,98,97,114,0,68,97,109,97,103,101,76,97,119,0,68,97,109,97,103,101,67,49,0,68,97,109,97,103,101,67,50,0,68,97,109,97,103,101,67,51,0,68,97,109,97,103,101,67,52,0,68,97,109,97,103,101,69,108,101,109,101,110,116,105,110,116,101,114,112,0,68,97,109,97,103,101,72,101,97,108,105,110,103,0,68,97,109,97,103,101,83,116,114,101,115,115,84,104,114,101,115,104,111,108,100,0,68,97,109,97,103,101,75,97,112,112,97,0,68,97,109,97,103,101,83,116,97,98,105,108,105,122,97,116,105,111,110,0,68,97,109,97,103,101,77,97,120,105,116,101,114,0,68,97,109,97,103,101,83,112,99,100,97,109,97,103,101,0,68,97,109,97,103,101,77,97,120,68,97,109,97,103,101,0,68,97,109,97,103,101,69,113,117,105,118,83,116,114,101,115,115,0,68,97,109,97,103,101,69,118,111,108,117,116,105,111,110,78,117,109,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,68,97,109,97,103,101,69,118,111,108,117,116,105,111,110,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,68,97,109,97,103,101,0,78,101,119,68,97,109,97,103,101,0,83,116,114,101,115,115,73,110,116,101,110,115,105,116,121,70,97,99,116,111,114,0,67,97,108,118,105,110,103,76,97,119,0,67,97,108,118,105,110,103,67,97,108,118,105,110,103,114,97,116,101,0,67,97,108,118,105,110,103,77,101,108,116,105,110,103,114,97,116,101,0,67,97,108,118,105,110,103,76,101,118,101,114,109,97,110,110,0,67,97,108,118,105,110,103,68,101,118,0,68,101,102,97,117,108,116,67,97,108,118,105,110,103,0,67,97,108,118,105,110,103,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,67,97,108,118,105,110,103,108,101,118,101,114,109,97,110,110,67,111,101,102,102,0,67,97,108,118,105,110,103,108,101,118,101,114,109,97,110,110,77,101,108,116,105,110,103,114,97,116,101,0,67,97,108,118,105,110,103,100,101,118,67,111,101,102,102,0,67,97,108,118,105,110,103,114,97,116,101,120,0,67,97,108,118,105,110,103,114,97,116,101,121,0,67,97,108,118,105,110,103,114,97,116,101,120,65,118,101,114,97,103,101,0,67,97,108,118,105,110,103,114,97,116,101,121,65,118,101,114,97,103,101,0,83,116,114,97,105,110,82,97,116,101,112,97,114,97,108,108,101,108,0,83,116,114,97,105,110,82,97,116,101,112,101,114,112,101,110,100,105,99,117,108,97,114,0,83,116,114,97,105,110,82,97,116,101,101,102,102,101,99,116,105,118,101,0,77,97,116,101,114,105,97,108,115,82,104,111,73,99,101,0,77,97,116,101,114,105,97,108,115,82,104,111,83,101,97,119,97,116,101,114,0,77,97,116,101,114,105,97,108,115,82,104,111,70,114,101,115,104,119,97,116,101,114,0,77,97,116,101,114,105,97,108,115,77,117,87,97,116,101,114,0,77,97,116,101,114,105,97,108,115,84,104,101,114,109,97,108,69,120,99,104,97,110,103,101,86,101,108,111,99,105,116,121,0,77,97,116,101,114,105,97,108,115,84,104,101,114,109,97,108,99,111,110,100,117,99,116,105,118,105,116,121,0,77,97,116,101,114,105,97,108,115,84,101,109,112,101,114,97,116,101,105,99,101,99,111,110,100,117,99,116,105,118,105,116,121,0,77,97,116,101,114,105,97,108,115,76,105,116,104,111,115,112,104,101,114,101,83,104,101,97,114,77,111,100,117,108,117,115,0,77,97,116,101,114,105,97,108,115,76,105,116,104,111,115,112,104,101,114,101,68,101,110,115,105,116,121,0,77,97,116,101,114,105,97,108,115,77,97,110,116,108,101,83,104,101,97,114,77,111,100,117,108,117,115,0,77,97,116,101,114,105,97,108,115,77,97,110,116,108,101,68,101,110,115,105,116,121,0,77,97,116,101,114,105,97,108,115,69,97,114,116,104,68,101,110,115,105,116,121,0,77,101,115,104,65,118,101,114,97,103,101,86,101,114,116,101,120,67,111,110,110,101,99,116,105,118,105,116,121,0,77,101,115,104,69,108,101,109,101,110,116,115,50,100,0,77,101,115,104,69,108,101,109,101,110,116,115,0,77,101,115,104,76,111,119,101,114,101,108,101,109,101,110,116,115,0,77,101,115,104,78,117,109,98,101,114,111,102,101,108,101,109,101,110,116,115,50,100,0,77,101,115,104,78,117,109,98,101,114,111,102,101,108,101,109,101,110,116,115,0,77,101,115,104,78,117,109,98,101,114,111,102,108,97,121,101,114,115,0,77,101,115,104,78,117,109,98,101,114,111,102,118,101,114,116,105,99,101,115,50,100,0,77,101,115,104,78,117,109,98,101,114,111,102,118,101,114,116,105,99,101,115,0,77,101,115,104,85,112,112,101,114,101,108,101,109,101,110,116,115,0,77,101,115,104,86,101,114,116,101,120,111,110,98,97,115,101,0,77,101,115,104,86,101,114,116,101,120,111,110,115,117,114,102,97,99,101,0,77,101,115,104,86,101,114,116,101,120,111,110,98,111,117,110,100,97,114,121,0,77,101,115,104,88,0,77,101,115,104,89,0,77,101,115,104,90,0,77,101,115,104,76,97,116,0,77,101,115,104,76,111,110,103,0,77,101,115,104,82,0,77,101,115,104,69,108,101,109,101,110,116,116,121,112,101,0,77,101,115,104,83,101,103,109,101,110,116,115,0,68,111,109,97,105,110,84,121,112,101,0,68,111,109,97,105,110,68,105,109,101,110,115,105,111,110,0,68,111,109,97,105,110,50,68,104,111,114,105,122,111,110,116,97,108,0,68,111,109,97,105,110,50,68,118,101,114,116,105,99,97,108,0,68,111,109,97,105,110,51,68,0,68,111,109,97,105,110,51,68,115,117,114,102,97,99,101,0,77,105,115,99,101,108,108,97,110,101,111,117,115,78,97,109,101,0,77,97,115,115,116,114,97,110,115,112,111,114,116,72,121,100,114,111,115,116,97,116,105,99,65,100,106,117,115,116,109,101,110,116,0,77,97,115,115,116,114,97,110,115,112,111,114,116,73,115,102,114,101,101,115,117,114,102,97,99,101,0,77,97,115,115,116,114,97,110,115,112,111,114,116,77,105,110,84,104,105,99,107,110,101,115,115,0,77,97,115,115,116,114,97,110,115,112,111,114,116,80,101,110,97,108,116,121,70,97,99,116,111,114,0,77,97,115,115,116,114,97,110,115,112,111,114,116,83,112,99,116,104,105,99,107,110,101,115,115,0,77,97,115,115,116,114,97,110,115,112,111,114,116,83,116,97,98,105,108,105,122,97,116,105,111,110,0,77,97,115,115,116,114,97,110,115,112,111,114,116,86,101,114,116,101,120,80,97,105,114,105,110,103,0,77,97,115,115,116,114,97,110,115,112,111,114,116,78,117,109,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,77,97,115,115,116,114,97,110,115,112,111,114,116,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,81,109,117,73,115,100,97,107,111,116,97,0,77,97,115,115,70,108,117,120,83,101,103,109,101,110,116,115,0,77,97,115,115,70,108,117,120,83,101,103,109,101,110,116,115,80,114,101,115,101,110,116,0,81,109,117,77,97,115,115,70,108,117,120,83,101,103,109,101,110,116,115,80,114,101,115,101,110,116,0,81,109,117,78,117,109,98,101,114,111,102,112,97,114,116,105,116,105,111,110,115,0,81,109,117,78,117,109,98,101,114,111,102,114,101,115,112,111,110,115,101,115,0,81,109,117,80,97,114,116,105,116,105,111,110,0,81,109,117,82,101,115,112,111,110,115,101,100,101,115,99,114,105,112,116,111,114,115,0,81,109,117,86,97,114,105,97,98,108,101,100,101,115,99,114,105,112,116,111,114,115,0,82,105,102,116,115,78,117,109,114,105,102,116,115,0,82,105,102,116,115,82,105,102,116,115,116,114,117,99,116,0,83,101,116,116,105,110,103,115,82,101,115,117,108,116,115,79,110,78,111,100,101,115,0,83,101,116,116,105,110,103,115,73,111,71,97,116,104,101,114,0,83,101,116,116,105,110,103,115,76,111,119,109,101,109,0,83,101,116,116,105,110,103,115,79,117,116,112,117,116,70,114,101,113,117,101,110,99,121,0,83,101,116,116,105,110,103,115,82,101,99,111,114,100,105,110,103,70,114,101,113,117,101,110,99,121,0,83,101,116,116,105,110,103,115,87,97,105,116,111,110,108,111,99,107,0,68,101,98,117,103,80,114,111,102,105,108,105,110,103,0,80,114,111,102,105,108,105,110,103,67,117,114,114,101,110,116,77,101,109,0,80,114,111,102,105,108,105,110,103,67,117,114,114,101,110,116,70,108,111,112,115,0,80,114,111,102,105,108,105,110,103,83,111,108,117,116,105,111,110,84,105,109,101,0,83,116,101,97,100,121,115,116,97,116,101,77,97,120,105,116,101,114,0,83,116,101,97,100,121,115,116,97,116,101,78,117,109,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,83,116,101,97,100,121,115,116,97,116,101,82,101,108,116,111,108,0,83,116,101,97,100,121,115,116,97,116,101,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,83,117,114,102,97,99,101,0,84,104,101,114,109,97,108,73,115,101,110,116,104,97,108,112,121,0,84,104,101,114,109,97,108,73,115,100,121,110,97,109,105,99,98,97,115,97,108,115,112,99,0,84,104,101,114,109,97,108,82,101,108,116,111,108,0,84,104,101,114,109,97,108,77,97,120,105,116,101,114,0,84,104,101,114,109,97,108,80,101,110,97,108,116,121,70,97,99,116,111,114,0,84,104,101,114,109,97,108,80,101,110,97,108,116,121,76,111,99,107,0,84,104,101,114,109,97,108,80,101,110,97,108,116,121,84,104,114,101,115,104,111,108,100,0,84,104,101,114,109,97,108,83,112,99,116,101,109,112,101,114,97,116,117,114,101,0,84,104,101,114,109,97,108,83,116,97,98,105,108,105,122,97,116,105,111,110,0,84,104,101,114,109,97,108,78,117,109,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,84,104,101,114,109,97,108,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,71,105,97,77,97,110,116,108,101,86,105,115,99,111,115,105,116,121,0,71,105,97,76,105,116,104,111,115,112,104,101,114,101,84,104,105,99,107,110,101,115,115,0,84,104,105,99,107,110,101,115,115,0,84,105,109,101,115,116,101,112,112,105,110,103,83,116,97,114,116,84,105,109,101,0,84,105,109,101,115,116,101,112,112,105,110,103,70,105,110,97,108,84,105,109,101,0,84,105,109,101,115,116,101,112,112,105,110,103,67,102,108,67,111,101,102,102,105,99,105,101,110,116,0,84,105,109,101,115,116,101,112,112,105,110,103,84,105,109,101,65,100,97,112,116,0,84,105,109,101,115,116,101,112,112,105,110,103,84,105,109,101,83,116,101,112,0,84,105,109,101,115,116,101,112,112,105,110,103,73,110,116,101,114,112,70,111,114,99,105,110,103,115,0,84,114,97,110,115,105,101,110,116,73,115,115,109,98,0,84,114,97,110,115,105,101,110,116,73,115,99,111,117,112,108,101,114,0,84,114,97,110,115,105,101,110,116,73,115,115,116,114,101,115,115,98,97,108,97,110,99,101,0,84,114,97,110,115,105,101,110,116,73,115,103,114,111,117,110,100,105,110,103,108,105,110,101,0,84,114,97,110,115,105,101,110,116,73,115,109,97,115,115,116,114,97,110,115,112,111,114,116,0,84,114,97,110,115,105,101,110,116,73,115,116,104,101,114,109,97,108,0,84,114,97,110,115,105,101,110,116,73,115,103,105,97,0,84,114,97,110,115,105,101,110,116,73,115,100,97,109,97,103,101,101,118,111,108,117,116,105,111,110,0,84,114,97,110,115,105,101,110,116,73,115,104,121,100,114,111,108,111,103,121,0,84,114,97,110,115,105,101,110,116,73,115,109,111,118,105,110,103,102,114,111,110,116,0,84,114,97,110,115,105,101,110,116,73,115,115,108,114,0,84,114,97,110,115,105,101,110,116,78,117,109,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,84,114,97,110,115,105,101,110,116,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,80,111,116,101,110,116,105,97,108,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,83,112,99,112,111,116,101,110,116,105,97,108,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,65,112,112,97,114,101,110,116,77,97,115,115,98,97,108,97,110,99,101,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,50,77,105,115,102,105,116,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,68,105,102,102,117,115,105,111,110,67,111,101,102,102,105,99,105,101,110,116,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,67,109,117,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,79,109,101,103,97,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,68,48,0,83,109,98,0,83,109,98,65,110,97,108,121,115,105,115,0,83,109,98,83,111,108,117,116,105,111,110,0,83,109,98,78,117,109,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,83,109,98,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,83,109,98,73,115,73,110,105,116,105,97,108,105,122,101,100,0,83,77,66,102,111,114,99,105,110,103,0,83,109,98,77,97,115,115,66,97,108,97,110,99,101,0,83,77,66,103,101,109,98,0,83,109,98,73,110,105,116,68,101,110,115,105,116,121,83,99,97,108,105,110,103,0,83,109,98,84,97,0,83,109,98,86,0,83,109,98,68,115,119,114,102,0,83,109,98,68,108,119,114,102,0,83,109,98,80,0,83,109,98,83,119,102,0,83,109,98,69,65,105,114,0,83,109,98,80,65,105,114,0,83,109,98,84,109,101,97,110,0,83,109,98,67,0,83,109,98,84,122,0,83,109,98,86,122,0,83,109,98,68,116,0,83,109,98,68,122,0,83,109,98,65,73,100,120,0,83,109,98,83,119,73,100,120,0,83,109,98,68,101,110,73,100,120,0,83,109,98,90,84,111,112,0,83,109,98,68,122,84,111,112,0,83,109,98,68,122,77,105,110,0,83,109,98,90,89,0,83,109,98,90,77,97,120,0,83,109,98,90,77,105,110,0,83,109,98,79,117,116,112,117,116,70,114,101,113,0,83,109,98,65,83,110,111,119,0,83,109,98,65,73,99,101,0,83,109,98,67,108,100,70,114,97,99,0,83,109,98,84,48,119,101,116,0,83,109,98,84,48,100,114,121,0,83,109,98,75,0,83,109,98,68,0,83,109,98,82,101,0,83,109,98,71,100,110,0,83,109,98,71,115,112,0,83,109,98,69,67,0,83,109,98,67,111,110,100,101,110,115,97,116,105,111,110,0,83,109,98,87,0,83,109,98,65,0,83,109,98,84,0,83,109,98,73,115,103,114,97,105,110,103,114,111,119,116,104,0,83,109,98,73,115,97,108,98,101,100,111,0,83,109,98,73,115,115,104,111,114,116,119,97,118,101,0,83,109,98,73,115,116,104,101,114,109,97,108,0,83,109,98,73,115,97,99,99,117,109,117,108,97,116,105,111,110,0,83,109,98,73,115,109,101,108,116,0,83,109,98,73,115,100,101,110,115,105,102,105,99,97,116,105,111,110,0,83,109,98,73,115,116,117,114,98,117,108,101,110,116,102,108,117,120,0,83,77,66,112,100,100,0,83,109,98,68,101,108,116,97,49,56,111,0,83,109,98,68,101,108,116,97,49,56,111,83,117,114,102,97,99,101,0,83,109,98,73,115,100,101,108,116,97,49,56,111,0,83,109,98,73,115,109,117,110,103,115,109,0,83,109,98,73,115,100,49,56,111,112,100,0,83,109,98,80,114,101,99,105,112,105,116,97,116,105,111,110,115,80,114,101,115,101,110,116,100,97,121,0,83,109,98,80,114,101,99,105,112,105,116,97,116,105,111,110,115,76,103,109,0,83,109,98,84,101,109,112,101,114,97,116,117,114,101,115,80,114,101,115,101,110,116,100,97,121,0,83,109,98,84,101,109,112,101,114,97,116,117,114,101,115,76,103,109,0,83,109,98,80,114,101,99,105,112,105,116,97,116,105,111,110,0,83,109,98,68,101,115,102,97,99,0,83,109,98,83,48,112,0,83,109,98,83,48,116,0,83,109,98,82,108,97,112,115,0,83,109,98,82,108,97,112,115,108,103,109,0,83,109,98,80,102,97,99,0,83,109,98,84,100,105,102,102,0,83,109,98,83,101,97,108,101,118,0,83,77,66,100,49,56,111,112,100,100,0,83,109,98,68,112,101,114,109,105,108,0,83,77,66,103,114,97,100,105,101,110,116,115,0,83,109,98,77,111,110,116,104,108,121,116,101,109,112,101,114,97,116,117,114,101,115,0,83,109,98,72,114,101,102,0,83,109,98,83,109,98,114,101,102,0,83,109,98,66,80,111,115,0,83,109,98,66,78,101,103,0,83,77,66,104,101,110,110,105,110,103,0,83,77,66,99,111,109,112,111,110,101,110,116,115,0,83,109,98,65,99,99,117,109,117,108,97,116,105,111,110,0,83,109,98,69,118,97,112,111,114,97,116,105,111,110,0,83,109,98,82,117,110,111,102,102,0,83,77,66,109,101,108,116,99,111,109,112,111,110,101,110,116,115,0,83,109,98,77,101,108,116,0,83,109,98,82,101,102,114,101,101,122,101,0,83,77,66,103,99,109,0,83,109,98,73,115,112,100,100,0,83,109,98,73,115,115,109,98,103,114,97,100,105,101,110,116,115,0,83,111,108,117,116,105,111,110,84,121,112,101,0,65,110,97,108,121,115,105,115,84,121,112,101,0,67,111,110,102,105,103,117,114,97,116,105,111,110,84,121,112,101,0,65,100,106,111,105,110,116,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,65,110,97,108,121,115,105,115,0,65,100,106,111,105,110,116,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,50,65,110,97,108,121,115,105,115,0,65,100,106,111,105,110,116,72,111,114,105,122,65,110,97,108,121,115,105,115,0,65,110,97,108,121,115,105,115,67,111,117,110,116,101,114,0,68,101,102,97,117,108,116,65,110,97,108,121,115,105,115,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,65,110,97,108,121,115,105,115,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,83,111,108,117,116,105,111,110,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,50,65,110,97,108,121,115,105,115,0,66,97,108,97,110,99,101,116,104,105],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+88360);allocate([99,107,110,101,115,115,50,83,111,108,117,116,105,111,110,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,83,111,102,116,65,110,97,108,121,115,105,115,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,83,111,102,116,83,111,108,117,116,105,111,110,0,66,97,108,97,110,99,101,118,101,108,111,99,105,116,121,65,110,97,108,121,115,105,115,0,66,97,108,97,110,99,101,118,101,108,111,99,105,116,121,83,111,108,117,116,105,111,110,0,76,50,80,114,111,106,101,99,116,105,111,110,69,80,76,65,110,97,108,121,115,105,115,0,76,50,80,114,111,106,101,99,116,105,111,110,66,97,115,101,65,110,97,108,121,115,105,115,0,66,101,100,83,108,111,112,101,83,111,108,117,116,105,111,110,0,68,97,109,97,103,101,69,118,111,108,117,116,105,111,110,83,111,108,117,116,105,111,110,0,68,97,109,97,103,101,69,118,111,108,117,116,105,111,110,65,110,97,108,121,115,105,115,0,83,116,114,101,115,115,98,97,108,97,110,99,101,65,110,97,108,121,115,105,115,0,83,116,114,101,115,115,98,97,108,97,110,99,101,83,73,65,65,110,97,108,121,115,105,115,0,83,116,114,101,115,115,98,97,108,97,110,99,101,83,111,108,117,116,105,111,110,0,83,116,114,101,115,115,98,97,108,97,110,99,101,86,101,114,116,105,99,97,108,65,110,97,108,121,115,105,115,0,69,110,116,104,97,108,112,121,65,110,97,108,121,115,105,115,0,70,108,97,105,109,65,110,97,108,121,115,105,115,0,70,108,97,105,109,83,111,108,117,116,105,111,110,0,72,121,100,114,111,108,111,103,121,83,104,114,101,118,101,65,110,97,108,121,115,105,115,0,72,121,100,114,111,108,111,103,121,68,67,73,110,101,102,102,105,99,105,101,110,116,65,110,97,108,121,115,105,115,0,72,121,100,114,111,108,111,103,121,68,67,69,102,102,105,99,105,101,110,116,65,110,97,108,121,115,105,115,0,72,121,100,114,111,108,111,103,121,83,111,109,109,101,114,115,65,110,97,108,121,115,105,115,0,72,121,100,114,111,108,111,103,121,83,111,108,117,116,105,111,110,0,77,101,108,116,105,110,103,65,110,97,108,121,115,105,115,0,77,97,115,115,116,114,97,110,115,112,111,114,116,65,110,97,108,121,115,105,115,0,77,97,115,115,116,114,97,110,115,112,111,114,116,83,111,108,117,116,105,111,110,0,70,114,101,101,83,117,114,102,97,99,101,66,97,115,101,65,110,97,108,121,115,105,115,0,70,114,101,101,83,117,114,102,97,99,101,84,111,112,65,110,97,108,121,115,105,115,0,83,117,114,102,97,99,101,78,111,114,109,97,108,86,101,108,111,99,105,116,121,0,69,120,116,114,117,100,101,70,114,111,109,66,97,115,101,65,110,97,108,121,115,105,115,0,69,120,116,114,117,100,101,70,114,111,109,84,111,112,65,110,97,108,121,115,105,115,0,68,101,112,116,104,65,118,101,114,97,103,101,65,110,97,108,121,115,105,115,0,83,116,101,97,100,121,115,116,97,116,101,83,111,108,117,116,105,111,110,0,83,117,114,102,97,99,101,83,108,111,112,101,83,111,108,117,116,105,111,110,0,83,109,111,111,116,104,65,110,97,108,121,115,105,115,0,84,104,101,114,109,97,108,65,110,97,108,121,115,105,115,0,84,104,101,114,109,97,108,83,111,108,117,116,105,111,110,0,84,114,97,110,115,105,101,110,116,83,111,108,117,116,105,111,110,0,85,122,97,119,97,80,114,101,115,115,117,114,101,65,110,97,108,121,115,105,115,0,71,105,97,83,111,108,117,116,105,111,110,0,71,105,97,65,110,97,108,121,115,105,115,0,77,101,115,104,100,101,102,111,114,109,97,116,105,111,110,83,111,108,117,116,105,111,110,0,77,101,115,104,100,101,102,111,114,109,97,116,105,111,110,65,110,97,108,121,115,105,115,0,76,101,118,101,108,115,101,116,65,110,97,108,121,115,105,115,0,76,101,118,101,108,115,101,116,83,116,97,98,105,108,105,122,97,116,105,111,110,0,69,120,116,114,97,112,111,108,97,116,105,111,110,65,110,97,108,121,115,105,115,0,76,115,102,82,101,105,110,105,116,105,97,108,105,122,97,116,105,111,110,65,110,97,108,121,115,105,115,0,65,112,112,114,111,120,105,109,97,116,105,111,110,0,78,111,110,101,65,112,112,114,111,120,105,109,97,116,105,111,110,0,83,73,65,65,112,112,114,111,120,105,109,97,116,105,111,110,0,83,83,65,65,112,112,114,111,120,105,109,97,116,105,111,110,0,83,83,65,72,79,65,112,112,114,111,120,105,109,97,116,105,111,110,0,83,83,65,70,83,65,112,112,114,111,120,105,109,97,116,105,111,110,0,76,49,76,50,65,112,112,114,111,120,105,109,97,116,105,111,110,0,72,79,65,112,112,114,111,120,105,109,97,116,105,111,110,0,72,79,70,83,65,112,112,114,111,120,105,109,97,116,105,111,110,0,70,83,65,112,112,114,111,120,105,109,97,116,105,111,110,0,70,83,118,101,108,111,99,105,116,121,0,70,83,112,114,101,115,115,117,114,101,0,68,97,116,97,83,101,116,0,67,111,110,115,116,114,97,105,110,116,115,0,76,111,97,100,115,0,77,97,116,101,114,105,97,108,115,0,78,111,100,101,115,0,67,111,110,116,111,117,114,115,0,80,97,114,97,109,101,116,101,114,115,0,86,101,114,116,105,99,101,115,0,82,101,115,117,108,116,115,0,71,101,110,101,114,105,99,80,97,114,97,109,0,65,100,111,108,99,80,97,114,97,109,0,66,111,111,108,73,110,112,117,116,0,66,111,111,108,80,97,114,97,109,0,67,111,110,116,111,117,114,0,68,97,116,97,115,101,116,73,110,112,117,116,0,68,111,117,98,108,101,73,110,112,117,116,0,68,111,117,98,108,101,65,114,114,97,121,73,110,112,117,116,0,68,97,116,97,83,101,116,80,97,114,97,109,0,68,111,117,98,108,101,77,97,116,65,114,114,97,121,80,97,114,97,109,0,68,111,117,98,108,101,77,97,116,80,97,114,97,109,0,68,111,117,98,108,101,80,97,114,97,109,0,68,111,117,98,108,101,86,101,99,80,97,114,97,109,0,69,108,101,109,101,110,116,0,69,108,101,109,101,110,116,72,111,111,107,0,72,111,111,107,0,69,120,116,101,114,110,97,108,82,101,115,117,108,116,0,70,105,108,101,80,97,114,97,109,0,73,110,112,117,116,0,73,110,116,73,110,112,117,116,0,73,110,112,117,116,84,111,69,120,116,114,117,100,101,0,73,110,112,117,116,84,111,76,50,80,114,111,106,101,99,116,0,73,110,112,117,116,84,111,68,101,112,116,104,97,118,101,114,97,103,101,0,73,110,112,117,116,84,111,83,109,111,111,116,104,0,83,109,111,111,116,104,84,104,105,99,107,110,101,115,115,77,117,108,116,105,112,108,105,101,114,0,73,110,116,80,97,114,97,109,0,73,110,116,86,101,99,80,97,114,97,109,0,84,114,97,110,115,105,101,110,116,80,97,114,97,109,0,77,97,116,105,99,101,0,77,97,116,100,97,109,97,103,101,105,99,101,0,77,97,116,112,97,114,0,78,111,100,101,0,78,117,109,101,114,105,99,97,108,102,108,117,120,0,78,117,109,101,114,105,99,97,108,102,108,117,120,84,121,112,101,0,78,101,117,109,97,110,110,102,108,117,120,0,80,97,114,97,109,0,77,111,117,108,105,110,0,80,101,110,103,114,105,100,0,80,101,110,112,97,105,114,0,80,114,111,102,105,108,101,114,0,77,97,116,114,105,120,80,97,114,97,109,0,77,97,115,115,99,111,110,0,77,97,115,115,99,111,110,78,97,109,101,0,77,97,115,115,99,111,110,68,101,102,105,110,105,116,105,111,110,101,110,117,109,0,77,97,115,115,99,111,110,76,101,118,101,108,115,101,116,0,77,97,115,115,99,111,110,97,120,112,98,121,0,77,97,115,115,99,111,110,97,120,112,98,121,78,97,109,101,0,77,97,115,115,99,111,110,97,120,112,98,121,68,101,102,105,110,105,116,105,111,110,101,110,117,109,0,77,97,115,115,99,111,110,97,120,112,98,121,78,97,109,101,120,0,77,97,115,115,99,111,110,97,120,112,98,121,78,97,109,101,121,0,77,97,115,115,99,111,110,97,120,112,98,121,65,108,112,104,97,0,77,97,115,115,99,111,110,97,120,112,98,121,66,101,116,97,0,78,111,100,101,83,73,100,0,86,101,99,116,111,114,80,97,114,97,109,0,82,105,102,116,102,114,111,110,116,0,82,105,102,116,102,114,111,110,116,84,121,112,101,0,83,101,103,109,101,110,116,0,83,101,103,109,101,110,116,82,105,102,116,102,114,111,110,116,0,83,112,99,68,121,110,97,109,105,99,0,83,112,99,83,116,97,116,105,99,0,83,112,99,84,114,97,110,115,105,101,110,116,0,83,116,114,105,110,103,65,114,114,97,121,80,97,114,97,109,0,83,116,114,105,110,103,80,97,114,97,109,0,83,101,103,0,83,101,103,73,110,112,117,116,0,84,114,105,97,0,84,114,105,97,73,110,112,117,116,0,84,101,116,114,97,0,84,101,116,114,97,73,110,112,117,116,0,80,101,110,116,97,0,80,101,110,116,97,73,110,112,117,116,0,86,101,114,116,101,120,0,86,101,114,116,101,120,80,73,100,0,86,101,114,116,101,120,83,73,100,0,65,105,114,0,73,99,101,0,77,101,108,97,110,103,101,0,87,97,116,101,114,0,67,108,111,115,101,100,0,70,114,101,101,0,79,112,101,110,0,65,100,106,111,105,110,116,112,0,65,100,106,111,105,110,116,120,0,65,100,106,111,105,110,116,121,0,65,100,106,111,105,110,116,122,0,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,77,105,115,102,105,116,0,66,101,100,83,108,111,112,101,88,0,66,101,100,83,108,111,112,101,89,0,66,111,117,110,100,97,114,121,0,67,111,110,118,101,114,103,101,100,0,70,105,108,108,0,70,114,97,99,116,105,111,110,73,110,99,114,101,109,101,110,116,0,70,114,105,99,116,105,111,110,0,73,110,116,101,114,110,97,108,0,77,97,115,115,70,108,117,120,0,77,101,108,116,105,110,103,79,102,102,115,101,116,0,77,105,115,102,105,116,0,80,114,101,115,115,117,114,101,0,80,114,101,115,115,117,114,101,80,105,99,97,114,100,0,65,110,100,114,111,105,100,70,114,105,99,116,105,111,110,67,111,101,102,102,105,99,105,101,110,116,0,82,101,115,101,116,80,101,110,97,108,116,105,101,115,0,83,101,103,109,101,110,116,79,110,73,99,101,83,104,101,108,102,0,83,117,114,102,97,99,101,65,114,101,97,0,83,117,114,102,97,99,101,83,108,111,112,101,88,0,83,117,114,102,97,99,101,83,108,111,112,101,89,0,84,101,109,112,101,114,97,116,117,114,101,0,84,101,109,112,101,114,97,116,117,114,101,80,105,99,97,114,100,0,84,101,109,112,101,114,97,116,117,114,101,80,68,68,0,84,104,105,99,107,110,101,115,115,65,98,115,77,105,115,102,105,116,0,83,117,114,102,97,99,101,65,98,115,77,105,115,102,105,116,0,86,101,108,0,86,101,108,111,99,105,116,121,0,86,120,65,118,101,114,97,103,101,0,86,120,0,86,120,80,105,99,97,114,100,0,86,121,65,118,101,114,97,103,101,0,86,121,0,86,121,80,105,99,97,114,100,0,86,122,0,86,122,83,83,65,0,86,122,72,79,0,86,122,80,105,99,97,114,100,0,86,122,70,83,0,86,120,77,101,115,104,0,86,121,77,101,115,104,0,86,122,77,101,115,104,0,69,110,116,104,97,108,112,121,0,69,110,116,104,97,108,112,121,80,105,99,97,114,100,0,84,104,105,99,107,110,101,115,115,65,98,115,71,114,97,100,105,101,110,116,0,84,104,105,99,107,110,101,115,115,65,108,111,110,103,71,114,97,100,105,101,110,116,0,84,104,105,99,107,110,101,115,115,65,99,114,111,115,115,71,114,97,100,105,101,110,116,0,84,104,105,99,107,110,101,115,115,80,111,115,105,116,105,118,101,0,73,110,116,77,97,116,80,97,114,97,109,0,84,114,97,110,115,105,101,110,116,73,110,112,117,116,0,87,97,116,101,114,102,114,97,99,116,105,111,110,0,87,97,116,101,114,99,111,108,117,109,110,0,66,97,115,97,108,70,114,105,99,116,105,111,110,0,86,105,115,99,111,117,115,72,101,97,116,105,110,103,0,72,121,100,114,111,108,111,103,121,87,97,116,101,114,86,120,0,72,121,100,114,111,108,111,103,121,87,97,116,101,114,86,121,0,68,114,105,118,105,110,103,83,116,114,101,115,115,88,0,68,114,105,118,105,110,103,83,116,114,101,115,115,89,0,83,105,103,109,97,78,78,0,83,116,114,101,115,115,84,101,110,115,111,114,0,83,116,114,101,115,115,84,101,110,115,111,114,120,120,0,83,116,114,101,115,115,84,101,110,115,111,114,120,121,0,83,116,114,101,115,115,84,101,110,115,111,114,120,122,0,83,116,114,101,115,115,84,101,110,115,111,114,121,121,0,83,116,114,101,115,115,84,101,110,115,111,114,121,122,0,83,116,114,101,115,115,84,101,110,115,111,114,122,122,0,83,116,114,101,115,115,77,97,120,80,114,105,110,99,105,112,97,108,0,68,101,118,105,97,116,111,114,105,99,83,116,114,101,115,115,0,68,101,118,105,97,116,111,114,105,99,83,116,114,101,115,115,120,120,0,68,101,118,105,97,116,111,114,105,99,83,116,114,101,115,115,120,121,0,68,101,118,105,97,116,111,114,105,99,83,116,114,101,115,115,120,122,0,68,101,118,105,97,116,111,114,105,99,83,116,114,101,115,115,121,121,0,68,101,118,105,97,116,111,114,105,99,83,116,114,101,115,115,121,122,0,68,101,118,105,97,116,111,114,105,99,83,116,114,101,115,115,122,122,0,68,101,118,105,97,116,111,114,105,99,83,116,114,101,115,115,101,102,102,101,99,116,105,118,101,0,83,116,114,97,105,110,82,97,116,101,0,83,116,114,97,105,110,82,97,116,101,120,120,0,83,116,114,97,105,110,82,97,116,101,120,121,0,83,116,114,97,105,110,82,97,116,101,120,122,0,83,116,114,97,105,110,82,97,116,101,121,121,0,83,116,114,97,105,110,82,97,116,101,121,122,0,83,116,114,97,105,110,82,97,116,101,122,122,0,68,105,118,101,114,103,101,110,99,101,0,77,97,120,68,105,118,101,114,103,101,110,99,101,0,71,105,97,67,114,111,115,115,83,101,99,116,105,111,110,83,104,97,112,101,0,71,105,97,100,87,100,116,0,71,105,97,87,0,80,48,0,80,48,65,114,114,97,121,0,80,49,0,80,49,68,71,0,80,49,98,117,98,98,108,101,0,80,49,98,117,98,98,108,101,99,111,110,100,101,110,115,101,100,0,80,50,0,80,50,98,117,98,98,108,101,0,80,50,98,117,98,98,108,101,99,111,110,100,101,110,115,101,100,0,80,50,120,80,49,0,80,49,120,80,50,0,80,49,120,80,51,0,80,50,120,80,52,0,80,49,80,49,0,80,49,80,49,71,76,83,0,77,73,78,73,0,77,73,78,73,99,111,110,100,101,110,115,101,100,0,84,97,121,108,111,114,72,111,111,100,0,76,65,84,97,121,108,111,114,72,111,111,100,0,88,84,97,121,108,111,114,72,111,111,100,0,79,110,101,76,97,121,101,114,80,52,122,0,67,114,111,117,122,101,105,120,82,97,118,105,97,114,116,0,76,65,67,114,111,117,122,101,105,120,82,97,118,105,97,114,116,0,83,97,118,101,82,101,115,117,108,116,115,0,66,111,111,108,69,120,116,101,114,110,97,108,82,101,115,117,108,116,0,68,111,117,98,108,101,69,120,116,101,114,110,97,108,82,101,115,117,108,116,0,68,111,117,98,108,101,77,97,116,69,120,116,101,114,110,97,108,82,101,115,117,108,116,0,73,110,116,69,120,116,101,114,110,97,108,82,101,115,117,108,116,0,74,0,83,116,114,105,110,103,69,120,116,101,114,110,97,108,82,101,115,117,108,116,0,83,116,101,112,0,84,105,109,101,0,87,97,116,101,114,67,111,108,117,109,110,79,108,100,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,50,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,51,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,52,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,53,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,54,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,55,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,56,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,57,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,48,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,49,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,50,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,51,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,52,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,53,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,54,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,55,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,56,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,57,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,50,48,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,50,49,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,50,50,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,50,51,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,50,52,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,50,53,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,50,54,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,50,55,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,50,56,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,50,57,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,51,48,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,51,49,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,51,50,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,51,51,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,51,52,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,51,53,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,51,54,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,51,55,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,51,56,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,51,57,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,52,48,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,52,49,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,52,50,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,52,51,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,52,52,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,52,53,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,52,54,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,52,55,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,52,56,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,52,57,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,53,48,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,53,49,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,53,50,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,53,51,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,53,52,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,53,53,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,53,54,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,53,55,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,53,56,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,53,57,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,54,48,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,54,49,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,54,50,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,54,51,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,54,52,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,54,53,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,54,54,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,54,55,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,54,56,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,54,57,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,55,48,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,55,49,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,55,50,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,55,51,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,55,52,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,55,53,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,55,54,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,55,55,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,55,56,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,55,57,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,56,48,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,56,49,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,56,50,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,56,51,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,56,52,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,56,53,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,56,54,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,56,55,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,56,56,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,56,57,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,57,48,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,57,49,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,57,50,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,57,51,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,57,52,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,57,53,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,57,54,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,57,55,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,57,56,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,57,57,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,49,48,48,0,79,117,116,112,117,116,100,101,102,105,110,105,116,105,111,110,76,105,115,116,0,77,97,115,115,102,108,117,120,97,116,103,97,116,101,0,77,97,115,115,102,108,117,120,97,116,103,97,116,101,78,97,109,101,0,77,97,115,115,102,108,117,120,97,116,103,97,116,101,68,101,102,105,110,105,116,105,111,110,101,110,117,109,0,77,97,115,115,102,108,117,120,97,116,103,97,116,101,83,101,103,109,101,110,116,115,0,78,111,100,97,108,118,97,108,117,101,0,78,111,100,97,108,118,97,108,117,101,78,97,109,101,0,78,111,100,97,108,118,97,108,117,101,68,101,102,105,110,105,116,105,111,110,101,110,117,109,0,78,111,100,97,108,118,97,108,117,101,77,111,100,101,108,69,110,117,109,0,78,111,100,97,108,118,97,108,117,101,78,111,100,101,0,77,105,115,102,105,116,78,97,109,101,0,77,105,115,102,105,116,68,101,102,105,110,105,116,105,111,110,101,110,117,109,0,77,105,115,102,105,116,77,111,100,101,108,69,110,117,109,0,77,105,115,102,105,116,79,98,115,101,114,118,97,116,105,111,110,0,77,105,115,102,105,116,79,98,115,101,114,118,97,116,105,111,110,69,110,117,109,0,77,105,115,102,105,116,76,111,99,97,108,0,77,105,115,102,105,116,84,105,109,101,105,110,116,101,114,112,111,108,97,116,105,111,110,0,77,105,115,102,105,116,87,101,105,103,104,116,115,0,77,105,115,102,105,116,87,101,105,103,104,116,115,69,110,117,109,0,83,117,114,102,97,99,101,79,98,115,101,114,118,97,116,105,111,110,0,87,101,105,103,104,116,115,83,117,114,102,97,99,101,79,98,115,101,114,118,97,116,105,111,110,0,86,120,79,98,115,0,87,101,105,103,104,116,115,86,120,79,98,115,0,86,121,79,98,115,0,87,101,105,103,104,116,115,86,121,79,98,115,0,77,105,110,86,101,108,0,77,97,120,86,101,108,0,77,105,110,86,120,0,77,97,120,86,120,0,77,97,120,65,98,115,86,120,0,77,105,110,86,121,0,77,97,120,86,121,0,77,97,120,65,98,115,86,121,0,77,105,110,86,122,0,77,97,120,86,122,0,77,97,120,65,98,115,86,122,0,70,108,111,97,116,105,110,103,65,114,101,97,0,71,114,111,117,110,100,101,100,65,114,101,97,0,73,99,101,77,97,115,115,0,73,99,101,86,111,108,117,109,101,0,73,99,101,86,111,108,117,109,101,65,98,111,118,101,70,108,111,97,116,97,116,105,111,110,0,84,111,116,97,108,70,108,111,97,116,105,110,103,66,109,98,0,84,111,116,97,108,71,114,111,117,110,100,101,100,66,109,98,0,84,111,116,97,108,83,109,98,0,65,98,115,111,108,117,116,101,0,73,110,99,114,101,109,101,110,116,97,108,0,65,117,103,109,101,110,116,101,100,76,97,103,114,97,110,103,105,97,110,82,0,65,117,103,109,101,110,116,101,100,76,97,103,114,97,110,103,105,97,110,82,104,111,112,0,65,117,103,109,101,110,116,101,100,76,97,103,114,97,110,103,105,97,110,82,108,97,109,98,100,97,0,65,117,103,109,101,110,116,101,100,76,97,103,114,97,110,103,105,97,110,82,104,111,108,97,109,98,100,97,0,65,117,103,109,101,110,116,101,100,76,97,103,114,97,110,103,105,97,110,84,104,101,116,97,0,78,111,110,101,0,65,103,103,114,101,115,115,105,118,101,77,105,103,114,97,116,105,111,110,0,83,111,102,116,77,105,103,114,97,116,105,111,110,0,83,117,98,101,108,101,109,101,110,116,77,105,103,114,97,116,105,111,110,0,83,117,98,101,108,101,109,101,110,116,77,105,103,114,97,116,105,111,110,50,0,67,111,110,116,97,99,116,0,71,114,111,117,110,100,105,110,103,79,110,108,121,0,77,97,115,107,71,114,111,117,110,100,101,100,105,99,101,76,101,118,101,108,115,101,116,0,71,97,117,115,115,83,101,103,0,70,83,83,111,108,118,101,114,0,65,100,106,111,105,110,116,0,67,111,108,105,110,101,97,114,0,67,111,110,116,114,111,108,83,116,101,97,100,121,0,70,115,101,116,0,71,114,97,100,105,101,110,116,49,0,71,114,97,100,105,101,110,116,50,0,71,114,97,100,105,101,110,116,51,0,71,114,97,100,105,101,110,116,0,71,114,111,117,110,100,105,110,103,108,105,110,101,77,105,103,114,97,116,105,111,110,0,71,115,101,116,0,73,110,100,101,120,0,73,110,100,101,120,101,100,0,73,110,116,101,114,115,101,99,116,0,78,111,100,97,108,0,79,108,100,71,114,97,100,105,101,110,116,0,79,117,116,112,117,116,66,117,102,102,101,114,80,111,105,110,116,101,114,0,79,117,116,112,117,116,66,117,102,102,101,114,83,105,122,101,80,111,105,110,116,101,114,0,79,117,116,112,117,116,70,105,108,101,80,111,105,110,116,101,114,0,84,111,111,108,107,105,116,115,70,105,108,101,78,97,109,101,0,82,111,111,116,80,97,116,104,0,79,117,116,112,117,116,70,105,108,101,78,97,109,101,0,73,110,112,117,116,70,105,108,101,78,97,109,101,0,76,111,99,107,70,105,108,101,78,97,109,101,0,82,101,115,116,97,114,116,70,105,108,101,78,97,109,101,0,84,111,111,108,107,105,116,115,79,112,116,105,111,110,115,65,110,97,108,121,115,101,115,0,84,111,111,108,107,105,116,115,79,112,116,105,111,110,115,83,116,114,105,110,103,115,0,81,109,117,69,114,114,78,97,109,101,0,81,109,117,73,110,78,97,109,101,0,81,109,117,79,117,116,78,97,109,101,0,82,101,103,117,108,97,114,0,83,99,97,108,101,100,0,83,101,112,97,114,97,116,101,0,83,115,101,116,0,86,101,114,98,111,115,101,0,84,114,105,97,110,103,108,101,73,110,116,101,114,112,0,66,105,108,105,110,101,97,114,73,110,116,101,114,112,0,78,101,97,114,101,115,116,73,110,116,101,114,112,0,88,89,0,88,89,90,0,68,101,110,115,101,0,77,112,105,68,101,110,115,101,0,77,112,105,83,112,97,114,115,101,0,83,101,113,0,77,112,105,0,77,117,109,112,115,0,71,115,108,0,79,112,116,105,111,110,0,71,101,110,101,114,105,99,79,112,116,105,111,110,0,79,112,116,105,111,110,67,101,108,108,0,79,112,116,105,111,110,83,116,114,117,99,116,0,67,117,102,102,101,121,0,80,97,116,101,114,115,111,110,0,65,114,114,104,101,110,105,117,115,0,76,108,105,98,111,117,116,114,121,68,117,118,97,108,0,83,112,99,108,101,118,101,108,115,101,116,0,69,120,116,114,97,112,111,108,97,116,105,111,110,86,97,114,105,97,98,108,101,0,73,99,101,77,97,115,107,78,111,100,101,65,99,116,105,118,97,116,105,111,110,0,76,101,118,101,108,115,101,116,102,117,110,99,116,105,111,110,83,108,111,112,101,88,0,76,101,118,101,108,115,101,116,102,117,110,99,116,105,111,110,83,108,111,112,101,89,0,76,101,118,101,108,115,101,116,102,117,110,99,116,105,111,110,80,105,99,97,114,100,0,76,101,118,101,108,115,101,116,82,101,105,110,105,116,70,114,101,113,117,101,110,99,121,0,83,101,97,108,101,118,101,108,114,105,115,101,83,111,108,117,116,105,111,110,0,83,101,97,108,101,118,101,108,114,105,115,101,65,110,97,108,121,115,105,115,0,83,101,97,108,101,118,101,108,0,83,101,97,108,101,118,101,108,69,117,115,116,97,116,105,99,0,83,101,97,108,101,118,101,108,114,105,115,101,68,101,108,116,97,116,104,105,99,107,110,101,115,115,0,83,101,97,108,101,118,101,108,114,105,115,101,77,97,120,105,116,101,114,0,83,101,97,108,101,118,101,108,114,105,115,101,82,101,108,116,111,108,0,83,101,97,108,101,118,101,108,114,105,115,101,65,98,115,116,111,108,0,83,101,97,108,101,118,101,108,114,105,115,101,76,111,118,101,72,0,83,101,97,108,101,118,101,108,114,105,115,101,76,111,118,101,75,0,83,101,97,108,101,118,101,108,114,105,115,101,84,105,100,101,76,111,118,101,72,0,83,101,97,108,101,118,101,108,114,105,115,101,84,105,100,101,76,111,118,101,75,0,83,101,97,108,101,118,101,108,114,105,115,101,82,105,103,105,100,0,83,101,97,108,101,118,101,108,114,105,115,101,69,108,97,115,116,105,99,0,83,101,97,108,101,118,101,108,114,105,115,101,82,111,116,97,116,105,111,110,0,83,101,97,108,101,118,101,108,114,105,115,101,71,69,108,97,115,116,105,99,0,83,101,97,108,101,118,101,108,114,105,115,101,68,101,103,97,99,99,0,83,101,97,108,101,118,101,108,114,105,115,101,84,114,97,110,115,105,116,105,111,110,115,0,83,101,97,108,101,118,101,108,114,105,115,101,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,83,101,97,108,101,118,101,108,114,105,115,101,78,117,109,82,101,113,117,101,115,116,101,100,79,117,116,112,117,116,115,0,77,97,120,105,109,117,109,78,117,109,98,101,114,79,102,68,101,102,105,110,105,116,105,111,110,115,0,69,110,117,109,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,104,97,114,101,100,47,69,110,117,109,47,83,116,114,105,110,103,84,111,69,110,117,109,120,46,99,112,112,0,83,116,114,105,110,103,84,111,69,110,117,109,120,0,118,101,98,111,115,105,116,121,32,108,101,118,101,108,32,115,104,111,117,108,100,32,98,101,32,97,32,112,111,115,105,116,105,118,101,32,105,110,116,101,103,101,114,32,40,117,115,101,114,32,112,114,111,118,105,100,101,100,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,104,97,114,101,100,47,78,117,109,101,114,105,99,115,47,86,101,114,98,111,115,105,116,121,46,99,112,112,0,83,101,116,86,101,114,98,111,115,105,116,121,76,101,118,101,108,0,49,52,69,114,114,111,114,69,120,99,101,112,116,105,111,110,0,69,114,114,111,114,32,109,101,115,115,97,103,101,58,32,0,10,63,63,63,32,69,114,114,111,114,32,105,110,32,61,61,62,32,0,58,0,32,101,114,114,111,114,32,109,101,115,115,97,103,101,58,32,0,10,10,0,10,91,0,93,32,63,63,63,32,69,114,114,111,114,32,117,115,105,110,103,32,61,61,62,32,0,91,0,93,32,0,10,69,114,114,111,114,32,105,110,32,61,61,62,32,0,109,97,116,95,116,121,112,101,0,109,112,105,100,101,110,115,101,0,109,112,105,115,112,97,114,115,101,0,100,101,110,115,101,0,68,101,110,115,101,32,109,97,116,114,105,120,32,116,121,112,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,102,111,114,32,112,97,114,97,108,108,101,108,32,114,117,110,115,32,119,105,116,104,32,110,117,109,95,112,114,111,99,115,62,49,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,116,111,111,108,107,105,116,115,47,105,115,115,109,47,73,115,115,109,84,111,111,108,107,105,116,85,116,105,108,115,46,99,112,112,0,73,115,115,109,77,97,116,84,121,112,101,70,114,111,109,84,111,111,108,107,105,116,79,112,116,105,111,110,115,0,118,101,99,95,116,121,112,101,0,109,112,105,0,115,101,113,0,68,101,110,115,101,32,118,101,99,116,111,114,32,116,121,112,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,102,111,114,32,112,97,114,97,108,108,101,108,32,114,117,110,115,32,119,105,116,104,32,110,117,109,95,112,114,111,99,115,62,49,0,73,115,115,109,86,101,99,84,121,112,101,70,114,111,109,84,111,111,108,107,105,116,79,112,116,105,111,110,115,0,48,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,116,111,111,108,107,105,116,115,47,109,112,105,47,105,115,115,109,109,112,105,46,99,112,112,0,115,105,122,101,72,101,108,112,101,114,0,115,101,110,100,99,111,117,110,116,61,61,114,101,99,118,99,111,117,110,116,32,124,124,32,115,101,110,100,116,121,112,101,61,61,114,101,99,118,116,121,112,101,0,73,83,83,77,95,77,80,73,95,65,108,108,103,97,116,104,101,114,0,115,101,110,100,116,121,112,101,61,61,114,101,99,118,116,121,112,101,0,73,83,83,77,95,77,80,73,95,65,108,108,103,97,116,104,101,114,118,0,115,101,110,100,99,111,117,110,116,61,61,114,101,99,118,99,111,117,110,116,115,91,48,93,0,115,101,110,100,116,121,112,101,61,61,114,101,99,118,116,121,112,101,32,38,38,32,115,101,110,100,99,110,116,61,61,114,101,99,118,99,110,116,0,73,83,83,77,95,77,80,73,95,71,97,116,104,101,114,0,116,111,111,108,107,105,116,0,116,111,107,101,110,32,108,105,115,116,32,115,104,111,117,108,100,32,115,116,97,114,116,32,119,105,116,104,32,97,110,32,111,112,116,105,111,110,44,32,110,111,116,32,97,32,118,97,108,117,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,116,111,111,108,107,105,116,115,47,84,111,111,108,107,105,116,79,112,116,105,111,110,115,46,99,112,112,0,84,111,107,101,110,86,97,108,117,101,0,32,32,32,115,116,97,114,116,105,110,103,32,109,111,100,101,108,32,112,114,111,99,101,115,115,111,114,32,10,0,32,32,32,99,114,101,97,116,105,110,103,32,100,97,116,97,115,101,116,115,32,102,111,114,32,97,110,97,108,121,115,105,115,32,0,32,32,32,100,111,110,101,32,119,105,116,104,32,109,111,100,101,108,32,112,114,111,99,101,115,115,111,114,32,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,69,108,101,109,101,110,116,115,65,110,100,86,101,114,116,105,99,101,115,80,97,114,116,105,116,105,111,110,105,110,103,46,99,112,112,0,69,108,101,109,101,110,116,115,65,110,100,86,101,114,116,105,99,101,115,80,97,114,116,105,116,105,111,110,105,110,103,0,109,101,116,105,115,32,104,97,115,32,110,111,116,32,98,101,101,100,32,105,110,115,116,97,108,108,101,100,46,32,67,97,110,110,111,116,32,114,117,110,32,119,105,116,104,32,109,111,114,101,32,116,104,97,110,32,49,32,99,112,117,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,46,46,47,77,101,115,104,80,97,114,116,105,116,105,111,110,120,47,77,101,115,104,80,97,114,116,105,116,105,111,110,120,46,104,0,77,101,115,104,80,97,114,116,105,116,105,111,110,120,0,65,116,32,108,101,97,115,116,32,111,110,101,32,112,114,111,99,101,115,115,111,114,32,105,115,32,114,101,113,117,105,114,101,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,67,114,101,97,116,101,80,97,114,97,109,101,116,101,114,115,46,99,112,112,0,67,114,101,97,116,101,80,97,114,97,109,101,116,101,114,115,0,109,97,115,115,95,102,108,117,120,95,110,117,109,95,112,114,111,102,105,108,101,115,32,105,115,32,48,44,32,119,104,101,110,32,77,97,115,115,70,108,117,120,32,99,111,109,112,117,116,97,116,105,111,110,115,32,119,101,114,101,32,114,101,113,117,101,115,116,101,100,33,0,77,101,115,104,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,67,114,101,97,116,101,69,108,101,109,101,110,116,115,86,101,114,116,105,99,101,115,65,110,100,77,97,116,101,114,105,97,108,115,46,99,112,112,0,67,114,101,97,116,101,69,108,101,109,101,110,116,115,86,101,114,116,105,99,101,115,65,110,100,77,97,116,101,114,105,97,108,115,0,77,97,116,101,114,105,97,108,115,32,0,80,69,84,83,99,32,111,112,116,105,111,110,115,32,102,111,114,32,97,110,97,108,121,115,105,115,32,0,32,104,97,118,101,32,98,101,101,110,32,100,101,99,108,97,114,101,100,32,98,117,116,32,119,101,114,101,32,110,111,116,32,102,111,117,110,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,80,97,114,115,101,84,111,111,108,107,105,116,115,79,112,116,105,111,110,115,120,47,80,97,114,115,101,84,111,111,108,107,105,116,115,79,112,116,105,111,110,115,120,46,99,112,112,0,80,97,114,115,101,84,111,111,108,107,105,116,115,79,112,116,105,111,110,115,120,0,32,32,32,82,101,110,117,109,98,101,114,105,110,103,32,100,101,103,114,101,101,115,32,111,102,32,102,114,101,101,100,111,109,10,0,99,111,117,108,100,32,110,111,116,32,102,105,110,100,32,101,108,101,109,101,110,116,32,119,105,116,104,32,118,101,114,116,101,120,32,119,105,116,104,32,105,100,32,0,32,116,111,32,99,111,109,112,117,116,101,32,110,111,100,97,108,32,118,97,108,117,101,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,78,111,100,97,108,86,97,108,117,101,120,47,78,111,100,97,108,86,97,108,117,101,120,46,99,112,112,0,78,111,100,97,108,86,97,108,117,101,120,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,79,117,116,112,117,116,82,101,115,117,108,116,115,120,47,46,46,47,46,46,47,99,108,97,115,115,101,115,47,46,46,47,115,104,97,114,101,100,47,46,47,77,101,109,79,112,115,47,77,101,109,79,112,115,46,104,0,119,98,0,37,115,46,37,105,0,110,111,116,32,115,117,112,112,111,114,116,101,100,32,102,111,114,32,116,104,105,115,32,116,121,112,101,32,111,102,32,114,101,115,117,108,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+98600);allocate([47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,79,117,116,112,117,116,82,101,115,117,108,116,115,120,47,46,46,47,46,46,47,99,108,97,115,115,101,115,47,46,47,69,120,116,101,114,110,97,108,82,101,115,117,108,116,115,47,71,101,110,101,114,105,99,69,120,116,101,114,110,97,108,82,101,115,117,108,116,46,104,0,71,101,116,86,97,108,117,101,0,71,101,110,101,114,105,99,69,120,116,101,114,110,97,108,82,101,115,117,108,116,60,99,104,97,114,42,62,58,10,0,32,32,32,118,97,108,117,101,58,32,0,32,32,32,105,100,32,32,32,32,32,32,32,32,32,32,58,32,0,32,32,32,114,101,115,117,108,116,95,110,97,109,101,32,58,32,0,32,32,32,115,116,101,112,32,32,32,32,32,32,32,32,58,32,0,32,32,32,116,105,109,101,32,32,32,32,32,32,32,32,58,32,0,50,49,71,101,110,101,114,105,99,69,120,116,101,114,110,97,108,82,101,115,117,108,116,73,80,99,69,0,73,110,112,117,116,32,0,32,110,111,116,32,102,111,117,110,100,32,105,110,32,101,108,101,109,101,110,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,83,101,116,65,99,116,105,118,101,78,111,100,101,115,76,83,77,120,47,83,101,116,65,99,116,105,118,101,78,111,100,101,115,76,83,77,120,46,99,112,112,0,83,101,116,65,99,116,105,118,101,78,111,100,101,115,76,83,77,120,0,118,101,99,116,111,114,32,116,121,112,101,58,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,71,101,116,86,101,99,116,111,114,70,114,111,109,73,110,112,117,116,115,120,47,71,101,116,86,101,99,116,111,114,70,114,111,109,73,110,112,117,116,115,120,46,99,112,112,0,71,101,116,86,101,99,116,111,114,70,114,111,109,73,110,112,117,116,115,120,0,32,32,32,32,32,32,67,111,110,102,105,103,117,114,105,110,103,32,101,108,101,109,101,110,116,115,46,46,46,10,0,32,32,32,32,32,32,67,111,110,102,105,103,117,114,105,110,103,32,108,111,97,100,115,46,46,46,10,0,32,32,32,32,32,32,67,111,110,102,105,103,117,114,105,110,103,32,109,97,116,101,114,105,97,108,115,46,46,46,10,0,116,111,112,32,103,114,105,100,32,99,101,108,108,32,115,116,114,117,99,116,117,114,101,32,108,101,110,103,116,104,32,100,111,101,115,32,110,111,116,32,103,111,32,101,118,101,110,108,121,32,105,110,116,111,32,115,112,101,99,105,102,105,101,100,32,116,111,112,32,115,116,114,117,99,116,117,114,101,32,100,101,112,116,104,44,32,97,100,106,117,115,116,32,100,122,84,111,112,32,111,114,32,122,84,111,112,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,83,117,114,102,97,99,101,77,97,115,115,66,97,108,97,110,99,101,120,47,71,101,109,98,120,46,99,112,112,0,71,101,109,98,103,114,105,100,73,110,105,116,105,97,108,105,122,101,0,105,110,105,116,105,97,108,32,116,111,112,32,103,114,105,100,32,99,101,108,108,32,108,101,110,103,116,104,32,40,100,122,84,111,112,41,32,105,115,32,60,32,48,46,48,53,32,109,0,32,32,32,103,114,97,105,110,32,103,114,111,119,116,104,32,109,111,100,117,108,101,10,0,32,32,32,97,108,98,101,100,111,32,109,111,100,117,108,101,10,0,97,108,98,101,100,111,32,109,101,116,104,111,100,32,115,119,105,116,99,104,32,115,104,111,117,108,100,32,114,97,110,103,101,32,102,114,111,109,32,49,32,116,111,32,52,33,0,97,108,98,101,100,111,0,97,108,98,101,100,111,32,62,32,49,46,48,10,0,97,108,98,101,100,111,32,105,115,32,110,101,103,97,116,105,118,101,10,0,97,108,98,101,100,111,32,61,61,32,78,65,78,10,0,32,32,32,116,104,101,114,109,97,108,32,109,111,100,117,108,101,10,0,32,32,32,115,104,111,114,116,119,97,118,101,32,109,111,100,117,108,101,10,0,32,32,32,97,99,99,117,109,117,108,97,116,105,111,110,32,109,111,100,117,108,101,10,0,109,97,115,115,32,110,111,116,32,99,111,110,115,101,114,118,101,100,32,105,110,32,97,99,99,117,109,117,108,97,116,105,111,110,32,102,117,110,99,116,105,111,110,0,97,99,99,117,109,117,108,97,116,105,111,110,0,32,32,32,109,101,108,116,32,109,111,100,117,108,101,10,0,32,32,32,32,32,32,112,111,114,101,32,119,97,116,101,114,32,114,101,102,114,101,101,122,101,10,0,110,101,103,97,116,105,118,101,32,112,111,114,101,32,119,97,116,101,114,32,103,101,110,101,114,97,116,101,100,32,105,110,32,109,101,108,116,32,101,113,117,97,116,105,111,110,115,0,109,101,108,116,0,110,101,103,97,116,105,118,101,32,112,111,114,101,32,119,97,116,101,114,32,103,101,110,101,114,97,116,101,100,32,105,110,32,109,101,108,116,32,101,113,117,97,116,105,111,110,115,10,0,109,97,115,115,32,111,114,32,101,110,101,114,103,121,32,97,114,101,32,110,111,116,32,99,111,110,115,101,114,118,101,100,32,105,110,32,109,101,108,116,32,101,113,117,97,116,105,111,110,115,10,0,100,109,58,32,0,32,100,69,58,32,0,32,32,32,100,101,110,115,105,102,105,99,97,116,105,111,110,32,109,111,100,117,108,101,10,0,32,32,32,116,117,114,98,117,108,101,110,116,70,108,117,120,32,109,111,100,117,108,101,10,0,73,83,83,77,32,119,97,115,32,110,111,116,32,99,111,109,112,105,108,101,100,32,119,105,116,104,32,100,97,107,111,116,97,32,115,117,112,112,111,114,116,44,32,99,97,110,110,111,116,32,99,97,114,114,121,32,111,117,116,32,100,97,107,111,116,97,32,97,110,97,108,121,115,105,115,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,87,114,97,112,112,101,114,67,111,114,101,80,111,105,110,116,101,114,70,114,111,109,83,111,108,117,116,105,111,110,69,110,117,109,46,99,112,112,0,87,114,97,112,112,101,114,67,111,114,101,80,111,105,110,116,101,114,70,114,111,109,83,111,108,117,116,105,111,110,69,110,117,109,0,99,111,110,116,114,111,108,32,116,121,112,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,0,115,111,108,117,116,105,111,110,32,116,121,112,101,58,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,67,111,114,101,80,111,105,110,116,101,114,70,114,111,109,83,111,108,117,116,105,111,110,69,110,117,109,46,99,112,112,0,67,111,114,101,80,111,105,110,116,101,114,70,114,111,109,83,111,108,117,116,105,111,110,69,110,117,109,0,83,104,111,117,108,100,32,110,111,116,32,98,101,32,114,101,113,117,101,115,116,105,110,103,32,65,68,32,100,114,105,118,101,114,115,32,119,104,101,110,32,97,110,32,65,68,32,108,105,98,114,97,114,121,32,105,115,32,110,111,116,32,97,118,97,105,108,97,98,108,101,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,97,100,95,99,111,114,101,46,99,112,112,0,97,100,95,99,111,114,101,0,37,115,32,118,101,114,115,105,111,110,32,32,37,115,10,0,73,99,101,32,83,104,101,101,116,32,83,121,115,116,101,109,32,77,111,100,101,108,32,40,73,83,83,77,41,0,52,46,57,0,40,119,101,98,115,105,116,101,58,32,37,115,32,99,111,110,116,97,99,116,58,32,37,115,41,10,0,104,116,116,112,58,47,47,105,115,115,109,46,106,112,108,46,110,97,115,97,46,103,111,118,0,105,115,115,109,64,106,112,108,46,110,97,115,97,46,103,111,118,0,99,108,111,115,105,110,103,32,77,80,73,0,101,110,117,109,32,112,114,111,118,105,100,101,100,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,40,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,69,110,117,109,84,111,65,110,97,108,121,115,105,115,46,99,112,112,0,69,110,117,109,84,111,65,110,97,108,121,115,105,115,0,105,110,112,117,116,32,111,112,116,105,111,110,32,104,97,115,32,97,110,32,101,109,112,116,121,32,110,97,109,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,79,112,116,105,111,110,115,47,79,112,116,105,111,110,115,46,99,112,112,0,65,100,100,79,112,116,105,111,110,0,79,112,116,105,111,110,32,34,0,34,32,104,97,115,32,97,32,112,114,111,116,101,99,116,101,100,32,99,104,97,114,97,99,116,101,114,32,34,46,34,0,34,32,104,97,115,32,97,32,112,114,111,116,101,99,116,101,100,32,99,104,97,114,97,99,116,101,114,32,34,91,34,0,34,32,104,97,115,32,97,32,112,114,111,116,101,99,116,101,100,32,99,104,97,114,97,99,116,101,114,32,34,93,34,0,79,112,116,105,111,110,115,32,34,0,34,32,102,111,117,110,100,32,109,117,108,116,105,112,108,101,32,116,105,109,101,115,0,84,104,101,32,100,121,110,97,109,105,99,95,99,97,115,116,32,102,114,111,109,32,79,98,106,101,99,116,42,32,116,111,32,79,112,116,105,111,110,42,32,105,115,32,102,97,105,108,105,110,103,46,10,0,65,32,113,117,105,99,107,32,113,111,114,107,97,114,111,117,110,100,32,99,111,110,115,105,115,116,115,32,111,102,32,117,115,105,110,103,32,97,32,67,45,108,105,107,101,32,99,97,115,116,10,0,79,112,101,110,32,79,112,116,105,111,110,115,46,99,112,112,32,97,110,100,32,99,104,97,110,103,101,32,116,104,101,32,100,121,110,97,109,105,99,95,99,97,115,116,32,105,110,32,79,112,116,105,111,110,115,58,58,71,101,116,79,112,116,105,111,110,32,98,121,32,97,32,67,45,108,105,107,101,32,99,97,115,116,10,0,77,97,107,101,32,116,104,101,32,102,105,120,32,97,98,111,118,101,32,97,110,100,32,114,101,99,111,109,112,105,108,101,32,73,83,83,77,0,71,101,116,79,112,116,105,111,110,0,67,97,110,110,111,116,32,114,101,99,111,118,101,114,32,102,105,101,108,100,32,34,0,34,32,102,111,114,32,97,110,32,111,112,116,105,111,110,32,111,102,32,116,121,112,101,32,0,37,100,93,0,37,100,120,0,105,116,101,114,97,116,105,111,110,32,0,32,32,116,105,109,101,32,91,121,114,93,58,32,0,32,40,116,105,109,101,32,115,116,101,112,58,32,0,32,32,32,99,111,109,112,117,116,105,110,103,32,116,104,101,114,109,97,108,32,114,101,103,105,109,101,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,110,101,119,32,103,114,111,117,110,100,105,110,103,32,108,105,110,101,32,112,111,115,105,116,105,111,110,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,103,108,97,99,105,97,108,32,105,115,111,115,116,97,116,105,99,32,97,100,106,117,115,116,109,101,110,116,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,114,101,113,117,101,115,116,101,100,32,111,117,116,112,117,116,115,10,0,32,32,32,99,104,101,99,107,112,111,105,110,116,105,110,103,32,109,111,100,101,108,32,10,0,10,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,118,101,108,111,99,105,116,121,32,97,110,100,32,116,101,109,112,101,114,97,116,117,114,101,32,102,111,114,32,115,116,101,112,58,32,0,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,10,0,10,32,32,32,45,45,32,99,111,109,112,117,116,105,110,103,32,110,101,119,32,118,101,108,111,99,105,116,121,32,45,45,32,10,10,0,10,32,32,32,45,45,32,99,111,109,112,117,116,105,110,103,32,110,101,119,32,116,101,109,112,101,114,97,116,117,114,101,32,45,45,10,10,0,32,32,32,99,104,101,99,107,105,110,103,32,115,116,101,97,100,121,115,116,97,116,101,32,99,111,110,118,101,114,103,101,110,99,101,10,0,32,32,32,109,97,120,105,109,117,109,32,110,117,109,98,101,114,32,115,116,101,97,100,121,115,116,97,116,101,32,105,116,101,114,97,116,105,111,110,115,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,115,116,101,97,100,121,115,116,97,116,101,95,99,111,114,101,46,99,112,112,0,115,116,101,97,100,121,115,116,97,116,101,99,111,110,118,101,114,103,101,110,99,101,0,32,32,32,84,101,109,112,101,114,97,116,117,114,101,32,99,111,110,118,101,114,103,101,110,99,101,58,32,110,111,114,109,40,100,116,41,47,110,111,114,109,40,116,41,0,32,32,32,99,111,109,112,117,116,105,110,103,32,109,97,115,115,32,116,114,97,110,115,112,111,114,116,10,0,32,32,32,99,97,108,108,32,102,114,101,101,32,115,117,114,102,97,99,101,32,99,111,109,112,117,116,97,116,105,111,110,97,108,32,99,111,114,101,10,0,32,32,32,99,97,108,108,32,99,111,109,112,117,116,97,116,105,111,110,97,108,32,99,111,114,101,10,0,101,120,116,114,117,100,105,110,103,32,115,111,108,117,116,105,111,110,32,102,114,111,109,32,98,97,115,101,46,46,46,10,0,101,120,116,114,117,100,105,110,103,32,115,111,108,117,116,105,111,110,32,102,114,111,109,32,116,111,112,46,46,46,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,116,101,109,112,101,114,97,116,117,114,101,115,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,109,101,108,116,105,110,103,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,115,109,98,32,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,67,111,110,116,114,111,108,47,67,114,101,97,116,101,80,97,114,97,109,101,116,101,114,115,67,111,110,116,114,111,108,46,99,112,112,0,67,114,101,97,116,101,80,97,114,97,109,101,116,101,114,115,67,111,110,116,114,111,108,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,67,111,110,116,114,111,108,47,85,112,100,97,116,101,69,108,101,109,101,110,116,115,65,110,100,77,97,116,101,114,105,97,108,115,67,111,110,116,114,111,108,46,99,112,112,0,85,112,100,97,116,101,69,108,101,109,101,110,116,115,65,110,100,77,97,116,101,114,105,97,108,115,67,111,110,116,114,111,108,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,83,117,114,102,97,99,101,65,98,115,86,101,108,77,105,115,102,105,116,120,47,83,117,114,102,97,99,101,65,98,115,86,101,108,77,105,115,102,105,116,120,46,99,112,112,0,83,117,114,102,97,99,101,65,98,115,86,101,108,77,105,115,102,105,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,83,117,114,102,97,99,101,82,101,108,86,101,108,77,105,115,102,105,116,120,47,83,117,114,102,97,99,101,82,101,108,86,101,108,77,105,115,102,105,116,120,46,99,112,112,0,83,117,114,102,97,99,101,82,101,108,86,101,108,77,105,115,102,105,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,83,117,114,102,97,99,101,76,111,103,86,101,108,77,105,115,102,105,116,120,47,83,117,114,102,97,99,101,76,111,103,86,101,108,77,105,115,102,105,116,120,46,99,112,112,0,83,117,114,102,97,99,101,76,111,103,86,101,108,77,105,115,102,105,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,83,117,114,102,97,99,101,76,111,103,86,120,86,121,77,105,115,102,105,116,120,47,83,117,114,102,97,99,101,76,111,103,86,120,86,121,77,105,115,102,105,116,120,46,99,112,112,0,83,117,114,102,97,99,101,76,111,103,86,120,86,121,77,105,115,102,105,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,83,117,114,102,97,99,101,65,118,101,114,97,103,101,86,101,108,77,105,115,102,105,116,120,47,83,117,114,102,97,99,101,65,118,101,114,97,103,101,86,101,108,77,105,115,102,105,116,120,46,99,112,112,0,83,117,114,102,97,99,101,65,118,101,114,97,103,101,86,101,108,77,105,115,102,105,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,68,114,97,103,67,111,101,102,102,105,99,105,101,110,116,65,98,115,71,114,97,100,105,101,110,116,120,47,68,114,97,103,67,111,101,102,102,105,99,105,101,110,116,65,98,115,71,114,97,100,105,101,110,116,120,46,99,112,112,0,68,114,97,103,67,111,101,102,102,105,99,105,101,110,116,65,98,115,71,114,97,100,105,101,110,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,82,104,101,111,108,111,103,121,66,98,97,114,65,98,115,71,114,97,100,105,101,110,116,120,47,82,104,101,111,108,111,103,121,66,98,97,114,65,98,115,71,114,97,100,105,101,110,116,120,46,99,112,112,0,82,104,101,111,108,111,103,121,66,98,97,114,65,98,115,71,114,97,100,105,101,110,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,82,104,101,111,108,111,103,121,66,65,98,115,71,114,97,100,105,101,110,116,120,47,82,104,101,111,108,111,103,121,66,65,98,115,71,114,97,100,105,101,110,116,120,46,99,112,112,0,82,104,101,111,108,111,103,121,66,65,98,115,71,114,97,100,105,101,110,116,0,49,50,67,111,110,116,114,111,108,73,110,112,117,116,0,73,110,112,117,116,32,111,102,32,69,110,117,109,32,0,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,32,98,121,32,67,111,110,116,114,111,108,73,110,112,117,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,67,111,110,116,114,111,108,73,110,112,117,116,46,99,112,112,0,67,111,110,116,114,111,108,73,110,112,117,116,0,67,111,110,116,114,111,108,73,110,112,117,116,58,10,0,32,32,32,67,111,110,116,114,111,108,73,110,112,117,116,32,0,45,45,45,118,97,108,117,101,115,58,32,10,0,45,45,45,115,97,118,101,100,118,97,108,117,101,115,58,32,10,0,45,45,45,109,105,110,118,97,108,117,101,115,58,32,10,0,45,45,45,109,97,120,118,97,108,117,101,115,58,32,10,0,45,45,45,103,114,97,100,105,101,110,116,58,32,10,0,32,110,111,116,32,102,111,117,110,100,0,109,111,114,101,32,116,104,97,110,32,51,32,99,111,110,116,114,111,108,115,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,32,40,71,114,97,100,105,101,110,116,32,0,32,119,97,115,32,114,101,113,117,101,115,116,101,100,41,46,32,69,110,117,109,68,101,102,105,110,105,116,105,111,110,115,46,104,32,110,101,101,100,115,32,116,111,32,98,101,32,117,112,100,97,116,101,100,46,0,83,101,116,71,114,97,100,105,101,110,116,0,86,97,108,117,101,115,32,111,102,32,0,83,97,118,101,86,97,108,117,101,0,103,114,97,100,105,101,110,116,0,68,97,116,97,32,0,71,101,116,86,101,99,116,111,114,70,114,111,109,73,110,112,117,116,115,0,71,114,97,100,105,101,110,116,32,111,102,32,0,85,112,100,97,116,101,86,97,108,117,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,47,73,110,112,117,116,115,47,67,111,110,116,114,111,108,73,110,112,117,116,46,104,0,82,101,115,117,108,116,84,111,80,97,116,99,104,0,83,99,97,108,101,0,77,97,120,0,77,105,110,65,98,115,0,77,97,120,65,98,115,0,73,110,102,105,110,105,116,121,78,111,114,109,0,67,111,110,115,116,114,97,105,110,77,105,110,0,83,113,117,97,114,101,77,105,110,0,67,104,97,110,103,101,69,110,117,109,0,32,32,32,112,114,101,112,97,114,105,110,103,32,105,110,105,116,105,97,108,32,115,111,108,117,116,105,111,110,10,0,32,32,32,112,114,101,112,97,114,105,110,103,32,102,105,110,97,108,32,115,111,108,117,116,105,111,110,10,0,108,111,119,101,114,98,111,117,110,100,0,117,112,112,101,114,98,111,117,110,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,46,46,47,116,111,111,108,107,105,116,115,47,46,47,111,98,106,101,99,116,115,47,46,46,47,105,115,115,109,47,46,46,47,46,46,47,115,104,97,114,101,100,47,46,47,77,101,109,79,112,115,47,77,101,109,79,112,115,46,104,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,99,111,110,116,114,111,108,95,99,111,114,101,46,99,112,112,0,70,111,114,109,70,117,110,99,116,105,111,110,0,102,40,120,41,32,61,32,0,32,32,124,32,32,0,84,65,79,32,110,111,116,32,105,110,115,116,97,108,108,101,100,32,111,114,32,80,69,84,83,99,32,118,101,114,115,105,111,110,32,110,111,116,32,115,117,112,112,111,114,116,101,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,99,111,110,116,114,111,108,116,97,111,95,99,111,114,101,46,99,112,112,0,99,111,110,116,114,111,108,116,97,111,95,99,111,114,101,0,65,68,32,97,110,100,47,111,114,32,77,49,81,78,51,32,110,111,116,32,105,110,115,116,97,108,108,101,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,99,111,110,116,114,111,108,97,100,95,99,111,114,101,46,99,112,112,0,99,111,110,116,114,111,108,97,100,95,99,111,114,101,0,77,49,81,78,51,32,110,111,116,32,105,110,115,116,97,108,108,101,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,99,111,110,116,114,111,108,109,49,113,110,51,95,99,111,114,101,46,99,112,112,0,99,111,110,116,114,111,108,109,49,113,110,51,95,99,111,114,101,0,118,97,108,117,101,0,32,32,32,67,111,109,112,117,116,101,32,73,110,105,116,105,97,108,32,115,111,108,117,116,105,111,110,10,0,32,32,32,67,111,109,112,117,116,101,32,65,100,106,111,105,110,116,10,0,32,32,32,67,111,109,112,117,116,101,32,73,110,105,116,105,97,108,32,99,111,115,116,32,102,117,110,99,116,105,111,110,10,0,73,110,105,116,105,97,108,32,99,111,115,116,32,102,117,110,99,116,105,111,110,32,74,40,120,41,32,61,32,0,32,32,32,67,111,109,112,117,116,101,32,71,114,97,100,105,101,110,116,10,0,32,32,32,32,32,32,32,97,108,112,104,97,32,32,32,32,32,32,73,97,108,112,104,97,32,10,0,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,10,0,78,111,32,97,100,106,111,105,110,116,32,104,97,115,32,98,101,101,110,32,105,109,112,108,101,109,101,110,116,101,100,32,102,111,114,32,115,111,108,117,116,105,111,110,32,0,32,121,101,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,65,100,106,111,105,110,116,67,111,114,101,80,111,105,110,116,101,114,70,114,111,109,83,111,108,117,116,105,111,110,69,110,117,109,46,99,112,112,0,65,100,106,111,105,110,116,67,111,114,101,80,111,105,110,116,101,114,70,114,111,109,83,111,108,117,116,105,111,110,69,110,117,109,0,32,32,32,99,111,109,112,117,116,105,110,103,32,119,97,116,101,114,32,104,101,97,100,115,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,119,97,116,101,114,32,99,111,108,117,109,110,10,0,32,32,32,115,97,118,105,110,103,32,114,101,115,117,108,116,115,32,10,0,32,32,32,115,97,118,105,110,103,32,116,101,109,112,111,114,97,114,121,32,114,101,115,117,108,116,115,10,0,32,32,32,117,112,100,97,116,105,110,103,32,103,97,112,32,104,101,105,103,104,116,10,0,72,121,100,114,111,108,111,103,121,32,109,111,100,101,108,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,104,121,100,114,111,108,111,103,121,95,99,111,114,101,46,99,112,112,0,104,121,100,114,111,108,111,103,121,95,99,111,114,101,0,66,117,105,108,100,105,110,103,32,83,101,100,105,109,101,110,116,32,77,97,116,114,105,120,46,46,46,10,0,32,32,32,35,32,83,101,100,105,109,101,110,116,32,117,110,115,116,97,98,108,101,32,99,111,110,115,116,114,97,105,110,116,115,32,61,32,0,32,32,32,109,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,83,101,100,105,109,101,110,116,32,105,116,101,114,97,116,105,111,110,115,32,40,0,41,32,101,120,99,101,101,100,101,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,115,47,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,95,104,121,100,114,111,95,110,111,110,108,105,110,101,97,114,46,99,112,112,0,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,95,104,121,100,114,111,95,110,111,110,108,105,110,101,97,114,0,32,32,32,73,110,110,101,114,32,83,101,100,105,109,101,110,116,32,67,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,58,0,37,44,32,97,105,109,105,110,103,32,108,111,119,101,114,32,116,104,97,110,32,0,32,32,32,35,32,73,110,110,101,114,32,115,101,100,105,109,101,110,116,32,99,111,110,118,101,114,103,101,110,99,101,32,97,99,104,105,101,118,101,32,10,0,99,111,109,112,117,116,105,110,103,32,69,80,76,32,72,101,97,100,32,115,108,111,112,101,46,46,46,10,0,66,117,105,108,100,105,110,103,32,69,80,76,32,77,97,116,114,105,120,46,46,46,10,0,32,32,32,73,110,110,101,114,32,69,80,76,32,67,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,58,0,32,32,32,109,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,69,80,76,32,105,116,101,114,97,116,105,111,110,115,32,40,0,101,112,108,99,111,110,118,101,114,103,101,100,46,46,46,10,0,83,101,100,32,99,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,32,105,115,32,78,97,78,33,0,69,80,76,32,99,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,32,105,115,32,78,97,78,33,0,32,32,32,67,111,110,118,101,114,103,101,100,32,97,102,116,101,114,44,32,0,32,105,116,101,114,97,116,105,111,110,115,32,10,0,32,32,32,83,101,100,105,109,101,110,116,32,67,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,58,0,32,32,32,69,80,76,32,67,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,58,0,32,32,32,67,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,58,0,32,32,32,109,97,120,105,109,117,109,32,110,117,109,98,101,114,32,102,111,114,32,104,121,100,114,111,108,111,103,105,99,97,108,32,103,108,111,98,97,108,32,105,116,101,114,97,116,105,111,110,115,32,40,0,32,32,32,99,111,109,112,117,116,105,110,103,32,110,101,119,32,118,101,108,111,99,105,116,121,10,0,99,111,109,112,117,116,105,110,103,32,115,109,111,111,116,104,32,115,117,114,102,97,99,101,32,115,108,111,112,101,115,58,10,0,99,111,109,112,117,116,105,110,103,32,115,109,111,111,116,104,32,100,114,105,118,105,110,103,32,115,116,114,101,115,115,58,10,0,99,97,108,108,32,99,111,109,112,117,116,97,116,105,111,110,97,108,32,99,111,114,101,58,10,0,99,111,109,112,117,116,105,110,103,32,115,108,111,112,101,46,46,46,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,115,108,111,112,101,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,100,97,109,97,103,101,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,108,101,118,101,108,115,101,116,32,116,114,97,110,115,112,111,114,116,10,0,32,32,32,77,105,103,114,97,116,105,110,103,32,103,114,111,117,110,100,105,110,103,32,108,105,110,101,10,0,71,114,111,117,110,100,105,110,103,32,108,105,110,101,32,109,105,103,114,97,116,105,111,110,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,71,114,111,117,110,100,105,110,103,108,105,110,101,77,105,103,114,97,116,105,111,110,120,47,71,114,111,117,110,100,105,110,103,108,105,110,101,77,105,103,114,97,116,105,111,110,120,46,99,112,112,0,71,114,111,117,110,100,105,110,103,108,105,110,101,77,105,103,114,97,116,105,111,110,120,0,32,32,32,65,100,100,105,116,105,111,110,97,108,32,110,117,109,98,101,114,32,111,102,32,118,101,114,116,105,99,101,115,32,97,108,108,111,119,101,100,32,116,111,32,117,110,103,114,111,117,110,100,58,32,0,57,82,105,102,116,102,114,111,110,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,82,105,102,116,102,114,111,110,116,46,99,112,112,0,82,105,102,116,102,114,111,110,116,58,10,0,32,32,32,104,110,111,100,101,115,58,32,0,32,32,32,104,101,108,101,109,101,110,116,115,58,32,0,32,32,32,104,109,97,116,112,97,114,58,32,0,32,32,32,105,110,116,101,114,110,97,108,32,112,97,114,97,109,101,116,101,114,115,58,32,10,0,32,32,32,110,111,114,109,97,108,58,32,0,124,0,32,32,32,108,101,110,103,116,104,58,32,0,32,32,32,112,101,110,97,108,116,121,95,108,111,99,107,58,32,0,32,32,32,99,111,117,110,116,101,114,58,32,0,32,32,32,112,114,101,115,116,97,98,108,101,58,32,0,32,32,32,109,97,116,101,114,105,97,108,95,99,111,110,118,101,114,103,101,100,58,32,0,32,32,32,102,105,108,108,58,32,0,32,32,32,102,114,105,99,116,105,111,110,58,32,0,32,32,32,102,114,97,99,116,105,111,110,58,32,0,32,32,32,102,114,97,99,116,105,111,110,105,110,99,114,101,109,101,110,116,58,32,0,32,32,32,115,116,97,116,101,58,32,0,32,32,32,102,114,111,122,101,110,58,32,0,80,101,110,97,108,116,121,67,114,101,97,116,101,75,77,97,116,114,105,120,0,111,110,108,121,32,84,114,105,97,32,101,108,101,109,101,110,116,32,97,108,108,111,119,101,100,32,102,111,114,32,82,105,102,116,102,114,111,110,116,32,108,111,97,100,33,0,80,101,110,97,108,116,121,67,114,101,97,116,101,75,77,97,116,114,105,120,83,116,114,101,115,115,98,97,108,97,110,99,101,72,111,114,105,122,0,100,105,102,102,101,114,101,110,116,32,116,104,105,99,107,110,101,115,115,101,115,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,102,111,114,32,114,105,102,116,32,102,114,111,110,116,115,0,80,101,110,97,108,116,121,67,114,101,97,116,101,80,86,101,99,116,111,114,0,80,101,110,97,108,116,121,67,114,101,97,116,101,80,86,101,99,116,111,114,83,116,114,101,115,115,98,97,108,97,110,99,101,72,111,114,105,122,0,100,105,102,102,101,114,101,110,116,32,98,101,100,115,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,102,111,114,32,114,105,102,116,32,102,114,111,110,116,115,0,102,105,108,108,32,116,121,112,101,32,0,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,111,110,32,105,99,101,32,115,104,101,101,116,115,32,121,101,116,46,0,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,46,0,67,111,110,115,116,114,97,105,110,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,46,46,47,46,46,47,99,108,97,115,115,101,115,47,46,47,76,111,97,100,115,47,82,105,102,116,102,114,111,110,116,46,104,0,78,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,33,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,67,111,110,115,116,97,110,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,46,46,47,46,46,47,116,111,111,108,107,105,116,115,47,46,47,111,98,106,101,99,116,115,47,46,46,47,105,115,115,109,47,46,46,47,46,46,47,115,104,97,114,101,100,47,46,47,77,101,109,79,112,115,47,77,101,109,79,112,115,46,104,0,111,117,116,112,117,116,32,100,101,102,105,110,105,116,105,111,110,32,101,110,117,109,32,0,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,67,114,101,97,116,101,79,117,116,112,117,116,68,101,102,105,110,105,116,105,111,110,115,46,99,112,112,0,67,114,101,97,116,101,79,117,116,112,117,116,68,101,102,105,110,105,116,105,111,110,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,46,46,47,46,46,47,99,108,97,115,115,101,115,47,46,47,77,97,115,115,99,111,110,97,120,112,98,121,46,104,0,32,77,97,115,115,99,111,110,97,120,112,98,121,58,32,0,32,32,32,32,110,97,109,101,120,58,32,0,32,32,32,32,110,97,109,101,121,58,32,0,32,32,32,32,97,108,112,104,97,58,32,0,32,32,32,32,98,101,116,97,58,32,0,49,50,77,97,115,115,99,111,110,97,120,112,98,121,0,49,48,68,101,102,105,110,105,116,105,111,110,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,46,46,47,46,46,47,99,108,97,115,115,101,115,47,46,47,77,97,115,115,99,111,110,46,104,0,32,77,97,115,115,99,111,110,58,32,0,32,32,32,32,108,101,118,101,108,115,101,116,58,32,0,32,32,32,32,77,58,32,0,55,77,97,115,115,99,111,110,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,46,46,47,46,46,47,99,108,97,115,115,101,115,47,46,47,78,111,100,97,108,118,97,108,117,101,46,104,0,32,78,111,100,97,108,118,97,108,117,101,58,32,0,32,32,32,32,109,111,100,101,108,95,101,110,117,109,58,32,0,32,32,32,32,110,111,100,101,58,32,0,49,48,78,111,100,97,108,118,97,108,117,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,46,46,47,46,46,47,99,108,97,115,115,101,115,47,46,47,77,105,115,102,105,116,46,104,0,32,77,105,115,102,105,116,58,32,0,32,32,32,32,111,98,115,101,114,118,97,116,105,111,110,95,101,110,117,109,58,32,0,32,32,32,32,119,101,105,103,104,116,115,95,101,110,117,109,58,32,0,32,32,32,32,116,105,109,101,105,110,116,101,114,112,111,108,97,116,105,111,110,58,32,0,32,32,32,32,108,111,99,97,108,58,32,0,54,77,105,115,102,105,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,46,46,47,46,46,47,99,108,97,115,115,101,115,47,46,47,77,97,115,115,102,108,117,120,97,116,103,97,116,101,46,104,0,32,77,97,115,115,102,108,117,120,97,116,103,97,116,101,58,32,0,32,32,32,32,110,117,109,115,101,103,109,101,110,116,115,58,32,0,32,32,32,101,108,101,109,101,110,116,58,32,120,49,44,32,121,49,44,32,120,50,44,32,121,50,58,10,0,49,52,77,97,115,115,102,108,117,120,97,116,103,97,116,101,73,100,69,0,67,111,117,108,100,32,110,111,116,32,102,105,110,100,32,116,104,101,32,114,101,115,112,111,110,115,101,32,102,111,114,32,111,117,116,112,117,116,32,100,101,102,105,110,105,116,105,111,110,32,0,32,98,101,99,97,117,115,101,32,99,111,117,108,100,32,110,111,116,32,102,105,110,100,32,116,104,101,32,100,101,102,105,110,105,116,105,111,110,32,105,116,115,101,108,102,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,79,117,116,112,117,116,68,101,102,105,110,105,116,105,111,110,115,82,101,115,112,111,110,115,101,120,47,79,117,116,112,117,116,68,101,102,105,110,105,116,105,111,110,115,82,101,115,112,111,110,115,101,120,46,99,112,112,0,79,117,116,112],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+108840);allocate([117,116,68,101,102,105,110,105,116,105,111,110,115,82,101,115,112,111,110,115,101,120,0,49,48,80,101,110,116,97,73,110,112,117,116,0,32,32,32,80,101,110,116,97,73,110,112,117,116,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,80,101,110,116,97,73,110,112,117,116,46,99,112,112,0,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,32,102,111,114,32,116,121,112,101,32,0,73,110,112,117,116,32,112,114,111,118,105,100,101,100,32,105,115,32,110,111,116,32,97,32,84,104,105,99,107,110,101,115,115,32,40,101,110,117,109,95,116,121,112,101,32,105,115,32,0,83,101,116,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,47,73,110,112,117,116,115,47,80,101,110,116,97,73,110,112,117,116,46,104,0,83,101,116,0,71,101,116,73,110,112,117,116,85,112,84,111,67,117,114,114,101,110,116,84,105,109,101,65,118,101,114,97,103,101,115,0,71,101,116,73,110,112,117,116,65,108,108,84,105,109,101,65,118,101,114,97,103,101,115,0,49,48,84,101,116,114,97,73,110,112,117,116,0,32,32,32,84,101,116,114,97,73,110,112,117,116,32,0,32,91,0,93,32,40,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,84,101,116,114,97,73,110,112,117,116,46,99,112,112,0,79,112,101,114,97,116,105,111,110,32,110,111,116,32,112,101,114,109,105,116,116,101,100,32,98,101,99,97,117,115,101,32,120,105,110,112,117,116,32,105,115,32,111,102,32,116,121,112,101,32,0,65,88,80,89,0,79,112,101,114,97,116,105,111,110,32,110,111,116,32,112,101,114,109,105,116,116,101,100,32,98,101,99,97,117,115,101,32,105,110,112,117,116,66,32,105,115,32,111,102,32,116,121,112,101,32,0,80,111,105,110,116,119,105,115,101,77,105,110,0,80,111,105,110,116,119,105,115,101,77,97,120,0,80,111,105,110,116,119,105,115,101,68,105,118,105,100,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,73,110,112,117,116,115,47,46,46,47,46,47,73,110,112,117,116,115,47,84,101,116,114,97,73,110,112,117,116,46,104,0,83,112,97,119,110,83,101,103,73,110,112,117,116,0,69,120,116,114,117,100,101,0,86,101,114,116,105,99,97,108,108,121,73,110,116,101,103,114,97,116,101,0,71,101,116,73,110,112,117,116,68,101,114,105,118,97,116,105,118,101,65,118,101,114,97,103,101,86,97,108,117,101,0,71,101,116,73,110,112,117,116,86,97,108,117,101,0,105,60,48,32,124,124,32,106,60,48,32,124,124,32,107,60,48,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,98,97,109,103,47,84,114,105,97,110,103,108,101,46,99,112,112,0,84,114,105,97,110,103,108,101,0,105,62,61,110,98,118,32,124,124,32,106,62,61,110,98,118,32,124,124,32,107,62,61,110,98,118,0,107,62,61,50,48,48,48,48,0,79,112,116,105,109,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,98,97,109,103,47,77,101,115,104,46,99,112,112,0,82,101,97,100,77,101,115,104,0,58,32,91,0,32,32,32,99,111,110,115,116,114,117,99,116,105,111,110,32,111,102,32,116,104,101,32,103,101,111,109,101,116,114,121,32,102,114,111,109,32,116,104,101,32,50,100,32,109,101,115,104,10,0,110,98,116,32,111,114,32,110,98,118,32,105,115,32,110,101,103,97,116,105,118,101,32,40,77,101,115,104,32,101,109,112,116,121,63,41,0,66,117,105,108,100,71,101,111,109,101,116,114,121,70,114,111,109,77,101,115,104,0,83,111,109,101,32,68,111,117,98,108,101,32,101,100,103,101,32,105,110,32,116,104,101,32,109,101,115,104,44,32,116,104,101,32,110,117,109,98,101,114,32,105,115,32,0,44,32,110,98,101,52,61,0,112,114,111,98,108,101,109,32,105,110,32,71,101,111,109,101,116,114,121,32,114,101,99,111,110,115,116,114,117,99,116,105,111,110,58,32,97,110,32,101,100,103,101,32,111,110,32,98,111,117,110,100,97,114,121,32,105,115,32,100,117,112,108,105,99,97,116,101,100,32,40,100,111,117,98,108,101,32,101,108,101,109,101,110,116,63,41,0,84,104,101,32,101,100,103,101,32,40,0,44,0,41,32,98,101,108,111,110,103,115,32,116,111,32,109,111,114,101,32,116,104,97,110,32,50,32,116,114,105,97,110,103,108,101,115,32,40,0,69,100,103,101,32,0,32,111,102,32,116,114,105,97,110,103,108,101,32,0,65,110,32,101,100,103,101,32,98,101,108,111,110,103,115,32,116,111,32,109,111,114,101,32,116,104,97,110,32,50,32,116,114,105,97,110,103,108,101,115,0,32,32,32,32,32,32,32,32,32,105,110,102,111,32,111,110,32,77,101,115,104,58,10,0,32,32,32,32,32,32,32,32,32,32,32,32,45,32,110,117,109,98,101,114,32,111,102,32,118,101,114,116,105,99,101,115,32,32,32,32,61,32,0,32,32,32,32,32,32,32,32,32,32,32,32,45,32,110,117,109,98,101,114,32,111,102,32,116,114,105,97,110,103,108,101,115,32,32,32,61,32,0,32,32,32,32,32,32,32,32,32,32,32,32,45,32,110,117,109,98,101,114,32,111,102,32,103,105,118,101,110,32,101,100,103,101,115,32,61,32,0,32,32,32,32,32,32,32,32,32,32,32,32,45,32,110,117,109,98,101,114,32,111,102,32,97,108,108,32,101,100,103,101,115,32,32,32,61,32,0,32,32,32,32,32,32,32,32,32,32,32,32,45,32,69,117,108,101,114,32,110,117,109,98,101,114,32,49,32,45,32,110,98,32,111,102,32,104,111,108,101,115,32,61,32,0,32,32,32,67,111,110,115,116,114,117,99,116,105,111,110,32,111,102,32,116,104,101,32,101,100,103,101,115,32,0,111,104,32,110,111,46,46,46,10,0,112,114,111,98,108,101,109,32,105,110,32,101,100,103,101,32,99,111,110,115,116,114,117,99,116,105,111,110,32,112,114,111,99,101,115,115,58,32,107,33,61,110,98,101,32,40,115,104,111,117,108,100,32,110,111,116,32,104,97,112,112,101,110,41,0,118,33,61,101,100,103,101,115,91,105,48,32,93,46,118,91,106,48,93,58,32,116,104,105,115,32,115,104,111,117,108,100,32,110,111,116,32,104,97,112,112,101,110,32,97,115,32,116,104,101,32,118,101,114,116,101,120,32,98,101,108,111,110,103,115,32,116,111,32,116,104,105,115,32,101,100,103,101,0,110,98,115,117,98,100,111,109,97,105,110,115,32,115,104,111,117,108,100,32,98,101,32,48,0,32,32,32,32,32,32,84,104,101,32,78,117,109,98,101,114,32,111,102,32,115,117,98,32,100,111,109,97,105,110,32,61,32,0,107,33,61,32,110,98,115,117,98,100,111,109,97,105,110,115,0,32,32,32,110,117,109,98,101,114,32,111,102,32,118,101,114,116,105,99,101,115,32,61,32,0,10,32,32,32,110,117,109,98,101,114,32,111,102,32,101,100,103,101,115,32,61,32,0,71,104,46,99,111,101,102,73,99,111,111,114,60,61,48,32,105,110,32,105,110,102,101,114,101,100,32,71,101,111,109,101,116,114,121,32,40,116,104,105,115,32,115,104,111,117,108,100,32,110,111,116,32,104,97,112,112,101,110,41,0,112,114,111,98,108,101,109,32,105,110,32,69,100,103,101,52,32,99,111,110,115,116,114,117,99,116,105,111,110,58,32,107,32,33,61,32,105,0,99,111,101,102,73,99,111,111,114,32,115,104,111,117,108,100,32,98,101,32,112,111,115,105,116,105,118,101,44,32,97,32,112,114,111,98,108,101,109,32,105,110,32,116,104,101,32,103,101,111,109,101,116,114,121,32,105,115,32,108,105,107,101,108,121,0,83,101,116,73,110,116,67,111,111,114,0,65,114,101,97,32,111,102,32,84,114,105,97,110,103,108,101,32,0,32,60,32,48,32,40,100,101,116,61,0,70,97,116,97,108,32,101,114,114,111,114,58,32,115,111,109,101,32,116,114,105,97,110,103,108,101,115,32,104,97,118,101,32,110,101,103,97,116,105,118,101,32,97,114,101,97,115,44,32,115,101,101,32,97,98,111,118,101,0,84,104,101,114,101,32,97,114,101,32,0,32,100,111,117,98,108,101,32,101,100,103,101,115,32,105,110,32,116,104,101,32,109,101,115,104,0,82,101,99,111,110,115,116,114,117,99,116,69,120,105,115,116,105,110,103,77,101,115,104,0,32,44,32,0,41,32,98,101,108,111,110,103,115,32,116,111,32,109,111,114,101,32,116,104,97,110,32,50,32,116,114,105,97,110,103,108,101,115,0,76,111,115,116,32,98,111,117,110,100,97,114,121,32,101,100,103,101,115,32,0,32,58,32,0,79,116,104,101,114,32,108,111,115,116,32,98,111,117,110,100,97,114,121,32,101,100,103,101,115,32,110,111,116,32,115,104,111,119,110,46,46,46,10,0,32,98,111,117,110,100,97,114,121,32,101,100,103,101,115,32,40,102,114,111,109,32,116,104,101,32,103,101,111,109,101,116,114,121,41,32,97,114,101,32,110,111,116,32,100,101,102,105,110,101,100,32,97,115,32,109,101,115,104,32,101,100,103,101,115,0,82,101,99,111,110,115,116,114,117,99,116,69,120,105,115,116,105,110,103,77,101,115,104,58,32,65,108,108,32,116,104,101,32,118,101,114,116,105,99,101,115,32,97,114,101,32,97,108,105,103,110,101,100,0,119,101,32,108,111,115,116,32,0,32,101,120,105,115,116,105,110,103,32,101,100,103,101,115,32,111,116,104,101,114,32,0,110,117,109,98,101,114,32,111,102,32,116,114,105,97,110,103,108,101,115,32,101,100,103,101,115,32,97,108,111,110,101,32,61,32,0,82,101,99,111,110,115,116,114,117,99,116,69,120,105,115,116,105,110,103,77,101,115,104,32,101,114,114,111,114,32,109,101,115,115,97,103,101,58,32,112,114,111,98,108,101,109,32,119,105,116,104,32,116,104,101,32,101,100,103,101,32,110,117,109,98,101,114,32,0,84,104,105,115,32,101,100,103,101,32,105,115,32,111,110,32,103,101,111,109,101,116,114,105,99,97,108,32,101,100,103,101,32,110,117,109,98,101,114,32,0,116,104,101,32,118,101,114,116,101,120,32,110,117,109,98,101,114,32,0,32,111,102,32,116,104,105,115,32,101,100,103,101,32,105,115,32,97,32,103,101,111,109,101,116,114,105,99,32,66,97,109,103,86,101,114,116,101,120,32,110,117,109,98,101,114,32,0,32,111,102,32,116,104,105,115,32,101,100,103,101,32,105,115,32,97,32,103,101,111,109,101,116,114,105,99,32,69,100,103,101,32,110,117,109,98,101,114,32,0,73,116,115,32,112,111,105,110,116,101,114,32,105,115,32,0,84,104,105,115,32,101,100,103,101,32,105,115,32,111,110,32,103,101,111,109,101,116,114,121,32,97,110,100,32,104,97,115,32,110,111,32,97,100,106,97,99,101,110,116,32,101,100,103,101,32,40,111,112,101,110,32,99,117,114,118,101,41,32,97,110,100,32,111,110,101,32,111,102,32,116,104,101,32,116,105,112,32,105,115,32,110,111,116,32,114,101,113,117,105,114,101,100,10,0,83,101,101,32,97,98,111,118,101,32,40,109,105,103,104,116,32,98,101,32,99,114,121,112,116,105,99,46,46,46,41,0,86,101,114,116,101,120,32,0,32,100,111,101,115,32,110,111,116,32,98,101,108,111,110,103,32,116,111,32,97,110,121,32,101,108,101,109,101,110,116,10,0,79,114,112,104,97,110,32,102,111,117,110,100,32,105,110,32,109,101,115,104,44,32,115,101,101,32,105,100,115,32,97,98,111,118,101,0,65,116,32,108,101,97,115,116,32,111,110,101,32,115,117,98,100,111,109,97,105,110,32,105,115,32,101,109,112,116,121,0,84,114,105,97,110,103,108,101,82,101,102,101,114,101,110,99,101,76,105,115,116,0,105,110,99,111,110,115,105,115,116,101,110,116,32,99,111,110,102,105,103,117,114,97,116,105,111,110,32,40,67,111,110,116,97,99,116,32,73,83,83,77,32,100,101,118,101,108,111,112,101,114,115,41,0,65,100,100,86,101,114,116,101,120,0,67,97,110,110,111,116,32,97,100,100,32,97,32,118,101,114,116,101,120,32,109,111,114,101,32,116,104,97,110,32,111,110,99,101,46,32,67,104,101,99,107,32,100,117,112,108,105,99,97,116,101,115,0,78,111,116,32,101,110,111,117,103,104,32,116,114,105,97,110,103,108,101,115,0,115,119,97,112,32,116,104,101,32,112,111,105,110,116,32,115,32,105,115,32,111,110,32,97,32,101,100,103,101,0,33,97,46,116,32,124,124,32,33,98,46,116,0,70,111,114,99,101,69,100,103,101,0,33,118,50,0,33,118,49,32,124,124,32,33,118,50,0,76,111,111,112,32,105,110,32,102,111,114,99,105,110,103,32,69,103,100,101,44,32,110,98,32,100,101,32,115,119,97,112,61,0,44,32,110,98,32,111,102,32,116,114,121,32,115,119,97,112,32,40,0,41,32,116,111,111,32,98,105,103,0,107,62,61,50,48,48,48,0,84,104,101,32,98,111,117,110,100,97,114,121,32,105,115,32,110,111,116,32,99,108,111,115,101,58,32,97,108,108,32,116,114,105,97,110,103,108,101,115,32,97,114,101,32,111,117,116,115,105,100,101,0,70,105,110,100,83,117,98,68,111,109,97,105,110,0,107,33,61,110,98,115,117,98,100,111,109,97,105,110,115,0,98,97,100,32,100,101,102,105,110,105,116,105,111,110,32,111,102,32,83,117,98,83,111,109,97,105,110,32,0,109,97,114,107,91,71,101,116,73,100,40,116,116,41,93,62,61,48,0,33,71,104,46,110,98,101,0,77,97,107,101,71,101,111,109,69,100,103,101,84,111,69,100,103,101,0,66,85,71,58,32,116,104,101,32,103,101,111,109,101,116,114,105,99,97,108,32,101,100,103,101,32,0,32,105,115,32,111,110,32,110,111,32,101,100,103,101,32,99,117,114,118,101,10,0,83,101,101,32,97,98,111,118,101,0,110,111,32,115,116,97,114,116,105,110,103,32,116,114,105,97,110,103,108,101,32,112,114,111,118,105,100,101,100,32,97,110,100,32,110,111,32,113,117,97,100,116,114,101,101,32,97,118,97,105,108,97,98,108,101,0,84,114,105,97,110,103,108,101,70,105,110,100,70,114,111,109,67,111,111,114,100,0,112,114,111,98,108,101,109,32,119,104,105,108,101,32,116,114,121,105,110,103,32,116,111,32,102,105,110,100,32,110,101,97,114,101,115,116,32,118,101,114,116,101,120,32,102,114,111,109,32,97,32,103,105,118,101,110,32,112,111,105,110,116,46,32,78,111,32,111,117,116,112,117,116,32,102,111,117,110,100,0,110,111,32,116,114,105,97,110,103,108,101,32,105,115,32,97,115,115,111,99,105,97,116,101,100,32,116,111,32,118,101,114,116,101,120,32,110,117,109,98,101,114,32,0,32,40,111,114,112,104,97,110,63,41,0,77,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,105,116,101,114,97,116,105,111,110,32,114,101,97,99,104,101,100,32,40,116,104,114,101,115,104,111,108,100,32,61,32,0,41,46,0,107,60,48,0,67,108,111,115,101,66,111,117,110,100,97,114,121,69,100,103,101,0,107,107,107,62,61,49,48,48,48,0,73,74,50,61,61,48,0,97,49,60,48,32,124,124,32,97,49,62,61,51,0,83,119,97,112,70,111,114,70,111,114,99,105,110,103,69,100,103,101,0,40,100,101,116,49,60,61,48,32,41,32,124,124,32,40,100,101,116,50,60,61,48,41,0,40,100,101,116,115,97,62,61,48,41,32,124,124,32,40,100,101,116,115,98,60,61,48,41,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,116,111,111,108,107,105,116,115,47,103,115,108,47,46,46,47,46,46,47,115,104,97,114,101,100,47,46,47,77,101,109,79,112,115,47,77,101,109,79,112,115,46,104,0,82,105,103,104,116,32,104,97,110,100,32,115,105,100,101,32,118,101,99,116,111,114,32,111,102,32,115,105,122,101,32,0,44,32,119,104,101,110,32,109,97,116,114,105,120,32,105,115,32,111,102,32,115,105,122,101,32,0,45,0,32,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,116,111,111,108,107,105,116,115,47,103,115,108,47,68,101,110,115,101,71,115,108,83,111,108,118,101,46,99,112,112,0,68,101,110,115,101,71,115,108,83,111,108,118,101,0,83,116,105,102,102,110,101,115,115,32,109,97,116,114,105,120,32,115,104,111,117,108,100,32,98,101,32,115,113,117,97,114,101,33,0,51,49,65,100,106,111,105,110,116,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,65,100,106,111,105,110,116,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,65,110,97,108,121,115,105,115,46,99,112,112,0,56,65,110,97,108,121,115,105,115,0,51,50,65,100,106,111,105,110,116,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,50,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,65,100,106,111,105,110,116,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,50,65,110,97,108,121,115,105,115,46,99,112,112,0,50,48,65,100,106,111,105,110,116,72,111,114,105,122,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,65,100,106,111,105,110,116,72,111,114,105,122,65,110,97,108,121,115,105,115,46,99,112,112,0,69,120,97,99,116,32,97,100,106,111,105,110,116,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,32,102,111,114,32,76,49,76,50,32,109,111,100,101,108,0,67,114,101,97,116,101,75,77,97,116,114,105,120,76,49,76,50,0,67,114,101,97,116,101,75,77,97,116,114,105,120,70,83,0,67,114,101,97,116,101,80,86,101,99,116,111,114,72,79,0,67,114,101,97,116,101,80,86,101,99,116,111,114,70,83,0,99,111,110,116,114,111,108,32,116,121,112,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,58,32,0,71,114,97,100,105,101,110,116,74,68,114,97,103,71,114,97,100,105,101,110,116,0,71,114,97,100,105,101,110,116,74,66,98,97,114,71,114,97,100,105,101,110,116,0,71,114,97,100,105,101,110,116,74,66,71,114,97,100,105,101,110,116,0,71,114,97,100,105,101,110,116,74,68,114,97,103,83,83,65,0,71,114,97,100,105,101,110,116,74,68,114,97,103,72,121,100,114,111,83,83,65,0,71,114,97,100,105,101,110,116,74,66,98,97,114,83,83,65,0,71,114,97,100,105,101,110,116,74,66,83,83,65,0,71,114,97,100,105,101,110,116,74,68,83,83,65,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,83,111,108,117,116,105,111,110,72,111,114,105,122,0,50,52,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,65,110,97,108,121,115,105,115,46,99,112,112,0,67,111,110,116,114,111,108,32,0,114,101,115,112,111,110,115,101,32,0,50,53,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,50,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,50,65,110,97,108,121,115,105,115,46,99,112,112,0,115,117,114,102,97,99,101,32,115,108,111,112,101,32,105,115,32,122,101,114,111,0,67,114,101,97,116,101,68,48,0,50,56,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,83,111,102,116,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,66,97,108,97,110,99,101,116,104,105,99,107,110,101,115,115,83,111,102,116,65,110,97,108,121,115,105,115,46,99,112,112,0,50,51,66,97,108,97,110,99,101,118,101,108,111,99,105,116,121,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,66,97,108,97,110,99,101,118,101,108,111,99,105,116,121,65,110,97,108,121,115,105,115,46,99,112,112,0,50,52,76,50,80,114,111,106,101,99,116,105,111,110,66,97,115,101,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,76,50,80,114,111,106,101,99,116,105,111,110,66,97,115,101,65,110,97,108,121,115,105,115,46,99,112,112,0,50,51,68,97,109,97,103,101,69,118,111,108,117,116,105,111,110,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,68,97,109,97,103,101,69,118,111,108,117,116,105,111,110,65,110,97,108,121,115,105,115,46,99,112,112,0,67,114,101,97,116,101,68,97,109,97,103,101,70,73,110,112,117,116,80,114,97,108,111,110,103,0,77,112,105,32,118,101,99,116,111,114,32,114,101,113,117,105,114,101,115,32,99,111,109,112,105,108,97,116,105,111,110,32,111,102,32,77,80,73,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,46,46,47,116,111,111,108,107,105,116,115,47,111,98,106,101,99,116,115,47,46,46,47,105,115,115,109,47,46,47,73,115,115,109,86,101,99,46,104,0,73,115,115,109,86,101,99,0,118,101,99,116,111,114,32,116,121,112,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,33,0,32,32,32,99,111,109,112,117,116,105,110,103,32,100,97,109,97,103,101,32,97,110,97,108,121,116,105,99,97,108,108,121,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,100,97,109,97,103,101,32,117,115,105,110,103,32,115,111,117,114,99,101,32,116,101,114,109,32,105,110,32,97,100,118,101,99,116,105,111,110,32,115,99,104,101,109,101,10,0,68,97,109,97,103,101,32,108,97,119,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,68,97,109,97,103,101,120,47,68,97,109,97,103,101,120,46,99,112,112,0,68,97,109,97,103,101,120,0,50,49,83,116,114,101,115,115,98,97,108,97,110,99,101,65,110,97,108,121,115,105,115,0,102,105,110,105,116,101,32,101,108,101,109,101,110,116,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,83,116,114,101,115,115,98,97,108,97,110,99,101,65,110,97,108,121,115,105,115,46,99,112,112,0,109,111,100,101,108,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,105,102,32,118,101,114,116,105,99,101,115,95,116,121,112,101,32,105,115,32,83,83,65,72,79,44,32,121,111,117,32,115,104,111,117,100,32,104,97,118,101,32,110,111,100,101,111,110,72,79,32,111,114,32,110,111,100,101,111,110,83,83,65,0,105,102,32,118,101,114,116,105,99,101,115,95,116,121,112,101,32,105,115,32,72,79,70,83,44,32,121,111,117,32,115,104,111,117,100,32,104,97,118,101,32,110,111,100,101,111,110,72,79,32,111,114,32,110,111,100,101,111,110,70,83,0,105,102,32,118,101,114,116,105,99,101,115,95,116,121,112,101,32,105,115,32,83,83,65,70,83,44,32,121,111,117,32,115,104,111,117,100,32,104,97,118,101,32,110,111,100,101,111,110,83,83,65,32,111,114,32,110,111,100,101,111,110,70,83,0,65,112,112,114,111,120,105,109,97,116,105,111,110,32,0,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,0,77,97,116,101,114,105,97,108,32,116,121,112,101,32,110,111,116,32,114,101,99,111,103,110,105,122,101,100,0,32,32,32,99,111,109,112,117,116,105,110,103,32,99,111,117,112,108,105,110,103,32,98,101,116,119,101,101,110,32,108,111,119,101,114,32,111,114,100,101,114,32,109,111,100,101,108,115,32,97,110,100,32,70,83,10,0,67,114,101,97,116,101,74,97,99,111,98,105,97,110,77,97,116,114,105,120,83,83,65,0,67,114,101,97,116,101,75,77,97,116,114,105,120,83,83,65,0,67,114,101,97,116,101,80,86,101,99,116,111,114,83,83,65,0,67,114,101,97,116,101,80,86,101,99,116,111,114,76,49,76,50,0,71,101,116,83,111,108,117,116,105,111,110,70,114,111,109,73,110,112,117,116,115,72,111,114,105,122,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,83,111,108,117,116,105,111,110,70,83,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,83,111,108,117,116,105,111,110,83,83,65,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,83,111,108,117,116,105,111,110,72,79,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,83,111,108,117,116,105,111,110,76,49,76,50,0,67,111,117,112,108,105,110,103,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,102,111,114,32,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,83,111,108,117,116,105,111,110,83,83,65,72,79,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,83,111,108,117,116,105,111,110,72,79,70,83,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,83,111,108,117,116,105,111,110,83,83,65,70,83,0,67,114,101,97,116,101,75,77,97,116,114,105,120,83,83,65,86,105,115,99,111,117,115,0,67,114,101,97,116,101,75,77,97,116,114,105,120,83,83,65,70,114,105,99,116,105,111,110,0,67,114,101,97,116,101,80,86,101,99,116,111,114,83,83,65,68,114,105,118,105,110,103,83,116,114,101,115,115,0,67,114,101,97,116,101,80,86,101,99,116,111,114,83,83,65,70,114,111,110,116,0,83,117,98,101,108,101,109,101,110,116,32,77,105,103,114,97,116,105,111,110,32,50,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,32,102,111,114,32,70,108,111,119,108,105,110,101,0,67,114,101,97,116,101,75,77,97,116,114,105,120,70,83,70,114,105,99,116,105,111,110,0,97,112,112,114,111,120,105,109,97,116,105,111,110,32,0,67,114,101,97,116,101,75,77,97,116,114,105,120,83,83,65,51,100,86,105,115,99,111,117,115,0,50,49,85,122,97,119,97,80,114,101,115,115,117,114,101,65,110,97,108,121,115,105,115,0,115,111,108,117,116,105,111,110,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,85,122,97,119,97,80,114,101,115,115,117,114,101,65,110,97,108,121,115,105,115,46,99,112,112,0,102,105,110,105,116,101,32,101,108,101,109,101,110,116,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,50,52,83,116,114,101,115,115,98,97,108,97,110,99,101,83,73,65,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,83,116,114,101,115,115,98,97,108,97,110,99,101,83,73,65,65,110,97,108,121,115,105,115,46,99,112,112,0,32,32,32,99,111,109,112,117,116,105,110,103,32,83,73,65,32,118,101,108,111,99,105,116,105,101,115,10,0,50,57,83,116,114,101,115,115,98,97,108,97,110,99,101,86,101,114,116,105,99,97,108,65,110,97,108,121,115,105,115,0,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,32,40,110,101,101,100,115,32,116,111,32,98,101,32,99,111,100,101,100,41,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,83,116,114,101,115,115,98,97,108,97,110,99,101,86,101,114,116,105,99,97,108,65,110,97,108,121,115,105,115,46,99,112,112,0,32,32,32,99,111,109,112,117,116,105,110,103,32,118,101,114,116,105,99,97,108,32,118,101,108,111,99,105,116,105,101,115,10,0,67,97,110,110,111,116,32,99,111,109,112,117,116,101,32,86,101,108,32,97,115,32,86,122,70,83,32,105,115,32,111,102,32,116,121,112,101,32,0,67,97,110,110,111,116,32,99,111,109,112,117,116,101,32,86,122,32,97,115,32,86,122,70,83,32,105,110,32,110,111,116,32,112,114,101,115,101,110,116,32,105,110,32,72,79,70,83,32,101,108,101,109,101,110,116,0,67,97,110,110,111,116,32,99,111,109,112,117,116,101,32,86,122,32,97,115,32,86,122,70,83,32,105,110,32,110,111,116,32,112,114,101,115,101,110,116,32,105,110,32,83,83,65,70,83,32,101,108,101,109,101,110,116,0,49,54,69,110,116,104,97,108,112,121,65,110,97,108,121,115,105,115,0,83,105,122,101,32,111,102,32,102,105,101,108,100,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,69,110,116,104,97,108,112,121,65,110,97,108,121,115,105,115,46,99,112,112,0,9,117,110,107,110,111,119,110,32,116,104,101,114,109,97,108,32,98,97,115,97,108,32,115,116,97,116,101,32,102,111,117,110,100,33,0,32,32,32,99,111,109,112,117,116,105,110,103,32,101,110,116,104,97,108,112,121,10,0,78,101,103,97,116,105,118,101,32,119,97,116,101,114,32,102,114,97,99,116,105,111,110,32,102,111,117,110,100,32,105,110,32,115,111,108,117,116,105,111,110,32,118,101,99,116,111,114,0,50,51,72,121,100,114,111,108,111,103,121,83,104,114,101,118,101,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,72,121,100,114,111,108,111,103,121,83,104,114,101,118,101,65,110,97,108,121,115,105,115,46,99,112,112,0,50,52,72,121,100,114,111,108,111,103,121,83,111,109,109,101,114,115,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,72,121,100,114,111,108,111,103,121,83,111,109,109,101,114,115,65,110,97,108,121,115,105,115,46,99,112,112,0,51,48,72,121,100,114,111,108,111,103,121,68,67,73,110,101,102,102,105,99,105,101,110,116,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,72,121,100,114,111,108,111,103,121,68,67,73,110,101,102,102,105,99,105,101,110,116,65,110,97,108,121,115,105,115,46,99,112,112,0,50,56,72,121,100,114,111,108,111,103,121,68,67,69,102,102,105,99,105,101,110,116,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,72,121,100,114,111,108,111,103,121,68,67,69,102,102,105,99,105,101,110,116,65,110,97,108,121,115,105,115,46,99,112,112,0,110,111,32,99,97,115,101,32,104,105,103,104,101,114,32,116,104,97,110,32,49,32,102,111,114,32,116,104,101,32,84,114,97,110,115,102,101,114,32,109,101,116,104,111,100,0,71,101,116,72,121,100,114,111,108,111,103,121,75,77,97,116,114,105,120,84,114,97,110,115,102,101,114,0,71,101,116,72,121,100,114,111,108,111,103,121,80,86,101,99,116,111,114,84,114,97,110,115,102,101,114,0,110,111,116,32,73,109,112,108,101,109,101,110,116,101,100,32,89,101,116,0,67,111,109,112,117,116,101,69,80,76,84,104,105,99,107,110,101,115,115,0,72,121,100,114,111,108,111,103,121,69,80,76,71,101,116,77,97,115,107,0,85,115,105,110,103,32,110,111,114,109,97,108,32,115,116,114,101,115,115,32,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,71,101,116,72,121,100,114,111,108,111,103,121,68,67,73,110,101,102,102,105,99,105,101,110,116,72,109,97,120,0,110,111,32,99,97,115,101,32,104,105,103,104,101,114,32,116,104,97,110,32,51,32,102,111,114,32,83,101,100,105,109,101,110,116,108,105,109,105,116,70,108,97,103,0,72,121,100,114,111,108,111,103,121,69,80,76,71,101,116,65,99,116,105,118,101,0,50,51,76,50,80,114,111,106,101,99,116,105,111,110,69,80,76,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,76,50,80,114,111,106,101,99,116,105,111,110,69,80,76,65,110,97,108,121,115,105,115,46,99,112,112,0,49,53,77,101,108,116,105,110,103,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,77,101,108,116,105,110,103,65,110,97,108,121,115,105,115,46,99,112,112,0,50,49,77,97,115,115,116,114,97,110,115,112,111,114,116,65,110,97,108,121,115,105,115,0,68,71,32,51,100,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,77,97,115,115,116,114,97,110,115,112,111,114,116,65,110,97,108,121,115,105,115,46,99,112,112,0,69,108,101,109,101,110,116,32,116,121,112,101,32,0,67,114,101,97,116,101,75,77,97,116,114,105,120,67,71,0,83,116,97,98,105,108,105,122,97,116,105,111,110,32,0,72,121,100,114,111,115,116,97,116,105,99,32,97,100,106,117,115,116,109,101,110,116,32,0,49,49,83,109,98,65,110,97,108,121,115,105,115,0,83,117,114,102,97,99,101,32,109,97,115,115,32,98,97,108,97,110,99,101,32,109,111,100,101,108,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,83,109,98,65,110,97,108,121,115,105,115,46,99,112,112,0,32,32,32,99,97,108,108,32,68,101,108,116,97,49,56,111,80,97,114,97,109,101,116,101,114,105,122,97,116,105,111,110,32,109,111,100,117,108,101,10,0,32,32,32,99,97,108,108,32,77,117,110,103,115,109,116,112,80,97,114,97,109,101,116,101,114,105,122,97,116,105,111,110,32,109,111,100,117,108,101,10,0,32,32,32,99,97,108,108,32,112,111,115,105,116,105,118,101,32,100,101,103,114,101,101,32,100,97,121,32,109,111,100,117,108,101,10,0,32,32,32,99,97,108,108,32,68,101,108,116,97,49,56,111,112,100,80,97,114,97,109,101,116,101,114,105,122,97,116,105,111,110,32,109,111,100,117,108,101,10,0,9,99,97,108,108,32,115,109,98,32,103,114,97,100,105,101,110,116,115,32,109,111,100,117,108,101,10,0,32,32,99,97,108,108,32,115,109,98,32,72,101,110,110,105,110,103,32,109,111,100,117,108,101,10,0,32,32,99,97,108,108,32,115,109,98,32,67,111,109,112,111,110,101,110,116,115,32,109,111,100,117,108,101,10,0,32,32,99,97,108,108,32,115,109,98,32,77,101,108,116,32,67,111,109,112,111,110,101,110,116,115,32,109,111,100,117,108,101,10,0,67,114,101,97,116,101,68,86,101,99,116,111,114,0,50,51,70,114,101,101,83,117,114,102,97,99,101,66,97,115,101,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,70,114,101,101,83,117,114,102,97,99,101,66,97,115,101,65,110,97,108,121,115,105,115,46,99,112,112,0,50,50,70,114,101,101,83,117,114,102,97,99,101,84,111,112,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,70,114,101,101,83,117,114,102,97,99,101,84,111,112,65,110,97,108,121,115,105,115,46,99,112,112,0,50,51,69,120,116,114,117,100,101,70,114,111,109,66,97,115,101,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,69,120,116,114,117,100,101,70,114,111,109,66,97,115,101,65,110,97,108,121,115,105,115,46,99,112,112,0,50,50,69,120,116,114,117,100,101,70,114,111,109,84,111,112,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,69,120,116,114,117,100,101,70,114,111,109,84,111,112,65,110,97,108,121,115,105,115,46,99,112,112,0,50,48,68,101,112,116,104,65,118,101,114,97,103,101,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,68,101,112,116,104,65,118,101,114,97,103,101,65,110,97,108,121,115,105,115,46,99,112,112,0,49,53,84,104,101,114,109,97,108,65,110,97,108,121,115,105,115,0,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,84,104,101,114,109,97,108,65,110,97,108,121,115,105,115,46,99,112,112,0,83,111,108,117,116,105,111,110,32,0,50,100,32,109,101,115,104,101,115,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,78,97,78,32,102,111,117,110,100,32,105,110,32,115,111,108,117,116,105,111,110,32,118,101,99,116,111,114,0,82,104,101,111,108,111,103,121,32,108,97,119,32,0,49,52,83,109,111,111,116,104,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+119080);allocate([114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,83,109,111,111,116,104,65,110,97,108,121,115,105,115,46,99,112,112,0,50,51,77,101,115,104,100,101,102,111,114,109,97,116,105,111,110,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,77,101,115,104,100,101,102,111,114,109,97,116,105,111,110,65,110,97,108,121,115,105,115,46,99,112,112,0,67,114,101,97,116,101,67,111,110,115,116,114,97,105,110,116,115,0,67,114,101,97,116,101,76,111,97,100,115,0,68,111,102,115,80,101,114,78,111,100,101,0,85,112,100,97,116,101,80,97,114,97,109,101,116,101,114,115,0,49,54,76,101,118,101,108,115,101,116,65,110,97,108,121,115,105,115,0,67,97,108,118,105,110,103,32,108,97,119,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,76,101,118,101,108,115,101,116,65,110,97,108,121,115,105,115,46,99,112,112,0,85,112,100,97,116,101,69,108,101,109,101,110,116,115,0,117,110,107,110,111,119,110,32,116,121,112,101,32,111,102,32,115,116,97,98,105,108,105,122,97,116,105,111,110,32,105,110,32,76,101,118,101,108,115,101,116,65,110,97,108,121,115,105,115,46,99,112,112,0,32,32,32,99,111,109,112,117,116,105,110,103,32,76,101,118,101,114,109,97,110,110,39,115,32,99,97,108,118,105,110,103,32,114,97,116,101,10,0,67,97,118,105,110,103,32,108,97,119,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,67,97,108,118,105,110,103,120,47,67,97,108,118,105,110,103,120,46,99,112,112,0,67,97,108,118,105,110,103,120,0,50,49,69,120,116,114,97,112,111,108,97,116,105,111,110,65,110,97,108,121,115,105,115,0,101,120,116,114,97,112,111,108,97,116,105,111,110,32,111,102,32,0,58,32,99,97,108,108,32,99,111,109,112,117,116,97,116,105,111,110,97,108,32,99,111,114,101,58,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,69,120,116,114,97,112,111,108,97,116,105,111,110,65,110,97,108,121,115,105,115,46,99,112,112,0,50,55,76,115,102,82,101,105,110,105,116,105,97,108,105,122,97,116,105,111,110,65,110,97,108,121,115,105,115,0,85,112,100,97,116,101,32,115,112,99,115,32,102,111,114,32,114,101,105,110,105,116,105,97,108,105,122,97,116,105,111,110,58,10,0,115,109,111,111,116,104,105,110,103,32,108,115,102,32,115,108,111,112,101,10,0,99,97,108,108,32,99,111,109,112,117,116,97,116,105,111,110,97,108,32,99,111,114,101,32,102,111,114,32,114,101,105,110,105,116,105,97,108,105,122,97,116,105,111,110,32,105,110,32,115,116,101,112,32,0,58,10,0,32,32,32,99,104,101,99,107,105,110,103,32,114,101,105,110,105,116,105,97,108,105,122,97,116,105,111,110,32,99,111,110,118,101,114,103,101,110,99,101,10,0,32,32,32,109,97,120,105,109,117,109,32,110,117,109,98,101,114,32,114,101,105,110,105,116,105,97,108,105,122,97,116,105,111,110,32,105,116,101,114,97,116,105,111,110,115,32,0,32,114,101,97,99,104,101,100,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,76,115,102,82,101,105,110,105,116,105,97,108,105,122,97,116,105,111,110,65,110,97,108,121,115,105,115,46,99,112,112,0,82,101,105,110,105,116,67,111,110,118,101,114,103,101,110,99,101,0,32,32,32,86,101,108,111,99,105,116,121,32,99,111,110,118,101,114,103,101,110,99,101,58,32,110,111,114,109,40,100,117,41,47,110,111,114,109,40,117,41,0,32,32,32,99,111,109,112,117,116,105,110,103,32,71,73,65,10,0,49,49,71,105,97,65,110,97,108,121,115,105,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,71,105,97,65,110,97,108,121,115,105,115,46,99,112,112,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,83,111,108,117,116,105,111,110,0,32,32,32,99,111,109,112,117,116,105,110,103,32,115,101,97,32,108,101,118,101,108,32,114,105,115,101,10,0,84,114,97,110,115,102,101,114,70,111,114,99,105,110,103,32,101,114,114,111,114,32,109,101,115,115,97,103,101,58,32,99,111,117,108,100,32,110,111,116,32,102,105,110,100,32,73,99,101,99,97,112,84,111,69,97,114,116,104,67,111,109,109,32,99,111,109,109,117,110,105,99,97,116,111,114,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,115,101,97,108,101,118,101,108,114,105,115,101,95,99,111,114,101,46,99,112,112,0,84,114,97,110,115,102,101,114,70,111,114,99,105,110,103,0,84,114,97,110,115,102,101,114,70,111,114,99,105,110,103,32,101,114,114,111,114,32,109,101,115,115,97,103,101,58,32,110,117,109,98,101,114,32,111,102,32,116,114,97,110,115,105,116,105,111,110,32,118,101,99,116,111,114,115,32,105,115,32,110,111,116,32,101,113,117,97,108,32,116,111,32,116,104,101,32,110,117,109,98,101,114,32,111,102,32,105,99,101,99,97,112,115,33,0,84,114,97,110,115,102,101,114,83,101,97,76,101,118,101,108,32,101,114,114,111,114,32,109,101,115,115,97,103,101,58,32,110,117,109,98,101,114,32,111,102,32,116,114,97,110,115,105,116,105,111,110,32,118,101,99,116,111,114,115,32,105,115,32,110,111,116,32,101,113,117,97,108,32,116,111,32,116,104,101,32,110,117,109,98,101,114,32,111,102,32,105,99,101,99,97,112,115,33,0,84,114,97,110,115,102,101,114,83,101,97,108,101,118,101,108,0,49,50,71,101,110,101,114,105,99,80,97,114,97,109,73,80,105,69,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,111,114,101,115,47,115,101,97,108,101,118,101,108,114,105,115,101,95,99,111,114,101,95,110,111,110,101,117,115,116,97,116,105,99,46,99,112,112,0,115,108,114,99,111,110,118,101,114,103,101,110,99,101,0,32,32,32,32,32,32,99,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,58,32,110,111,114,109,40,100,83,41,47,110,111,114,109,40,83,41,0,32,32,32,32,32,32,99,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,58,32,110,111,114,109,40,100,83,41,0,32,10,0,50,48,83,101,97,108,101,118,101,108,114,105,115,101,65,110,97,108,121,115,105,115,0,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,97,110,97,108,121,115,101,115,47,83,101,97,108,101,118,101,108,114,105,115,101,65,110,97,108,121,115,105,115,46,99,112,112,0,67,111,114,101,0,78,111,116,32,105,109,112,108,101,109,101,110,116,101,100,0,71,101,116,83,111,108,117,116,105,111,110,70,114,111,109,73,110,112,117,116,115,0,71,114,97,100,105,101,110,116,74,0,56,71,97,117,115,115,83,101,103,0,71,97,117,115,115,83,101,103,58,10,0,118,101,114,116,101,120,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,49,93,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,103,97,117,115,115,47,71,97,117,115,115,83,101,103,46,99,112,112,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,49,93,0,53,71,97,117,115,115,0,57,71,97,117,115,115,84,114,105,97,0,70,105,114,115,116,32,105,110,100,105,99,101,32,112,114,111,118,105,100,101,100,32,105,115,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,32,40,117,115,101,114,32,112,114,111,118,105,100,101,100,32,0,41,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,103,97,117,115,115,47,71,97,117,115,115,84,114,105,97,46,99,112,112,0,71,97,117,115,115,84,114,105,97,0,83,101,99,111,110,100,32,105,110,100,105,99,101,32,112,114,111,118,105,100,101,100,32,105,115,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,32,40,117,115,101,114,32,112,114,111,118,105,100,101,100,32,0,32,41,0,71,97,117,115,115,84,114,105,97,58,10,0,84,104,101,32,50,32,105,110,100,105,99,101,115,32,112,114,111,118,105,100,101,100,32,97,114,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,32,40,117,115,101,114,32,112,114,111,118,105,100,101,100,32,0,32,97,110,100,32,0,71,97,117,115,115,69,100,103,101,67,101,110,116,101,114,0,118,101,114,116,101,120,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,50,93,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,48,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,50,93,0,49,48,71,97,117,115,115,84,101,116,114,97,0,32,78,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,103,97,117,115,115,47,71,97,117,115,115,84,101,116,114,97,46,99,112,112,0,71,97,117,115,115,84,101,116,114,97,0,71,97,117,115,115,84,101,116,114,97,58,10,0,118,101,114,116,101,120,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,51,93,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,51,93,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,52,93,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,57,93,0,83,121,110,99,104,114,111,110,105,122,101,71,97,117,115,115,66,97,115,101,0,49,48,71,97,117,115,115,80,101,110,116,97,0,80,101,110,116,97,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,103,97,117,115,115,47,71,97,117,115,115,80,101,110,116,97,46,99,112,112,0,71,97,117,115,115,80,101,110,116,97,0,84,114,105,97,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,105,110,100,101,120,32,0,71,97,117,115,115,80,101,110,116,97,58,10,0,32,32,32,110,117,109,103,97,117,115,115,58,32,0,32,32,32,119,101,105,103,104,116,115,32,61,32,91,0,119,101,105,103,104,116,115,32,61,32,78,85,76,76,10,0,32,32,32,99,111,111,114,100,115,49,32,61,32,91,0,99,111,111,114,100,115,49,32,61,32,78,85,76,76,10,0,32,32,32,99,111,111,114,100,115,50,32,61,32,91,0,99,111,111,114,100,115,50,32,61,32,78,85,76,76,10,0,32,32,32,99,111,111,114,100,115,51,32,61,32,91,0,99,111,111,114,100,115,51,32,61,32,78,85,76,76,10,0,32,32,32,99,111,111,114,100,115,52,32,61,32,91,0,99,111,111,114,100,115,52,32,61,32,78,85,76,76,10,0,32,32,32,119,101,105,103,104,116,32,61,32,0,32,32,32,99,111,111,114,100,49,32,61,32,0,32,32,32,99,111,111,114,100,50,32,61,32,0,32,32,32,99,111,111,114,100,51,32,61,32,0,32,32,32,99,111,111,114,100,52,32,61,32,0,118,101,114,116,101,120,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,53,93,0,71,97,117,115,115,86,101,114,116,101,120,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,53,93,0,71,97,117,115,115,78,111,100,101,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,56,93,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,49,49,93,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,54,93,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,49,55,93,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,49,56,93,0,110,111,100,101,32,105,110,100,101,120,32,115,104,111,117,108,100,32,98,101,32,105,110,32,91,48,32,50,57,93,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,70,114,105,99,116,105,111,110,46,99,112,112,0,71,101,116,65,108,112,104,97,67,111,109,112,108,101,109,101,110,116,0,71,101,116,65,108,112,104,97,86,105,115,99,111,117,115,67,111,109,112,108,101,109,101,110,116,0,71,101,116,65,108,112,104,97,72,121,100,114,111,67,111,109,112,108,101,109,101,110,116,0,70,114,105,99,116,105,111,110,32,108,97,119,32,0,71,101,116,65,108,112,104,97,50,0,71,101,116,65,108,112,104,97,50,86,105,115,99,111,117,115,0,71,101,116,65,108,112,104,97,50,87,101,101,114,116,109,97,110,0,71,101,116,65,108,112,104,97,50,72,121,100,114,111,0,110,101,103,97,116,105,118,101,32,119,97,116,101,114,32,108,97,121,101,114,32,116,104,105,99,107,110,101,115,115,0,71,101,116,65,108,112,104,97,50,87,97,116,101,114,76,97,121,101,114,0,71,101,116,65,108,112,104,97,50,67,111,117,108,111,109,98,0,100,101,115,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,46,46,47,115,104,97,114,101,100,47,77,101,109,79,112,115,47,77,101,109,79,112,115,46,104,0,120,77,101,109,67,112,121,0,115,114,99,0,115,101,116,32,111,102,32,101,110,117,109,32,116,121,112,101,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,68,111,102,73,110,100,101,120,105,110,103,46,99,112,112,0,73,110,105,116,83,101,116,0,68,111,102,73,110,100,101,120,105,110,103,58,10,0,32,32,32,103,115,105,122,101,58,32,32,0,32,32,32,102,115,105,122,101,58,32,32,0,32,32,32,115,115,105,122,101,58,32,32,0,32,32,32,99,108,111,110,101,58,32,32,0,32,32,32,97,99,116,105,118,101,58,32,0,32,32,32,102,114,101,101,122,101,58,32,0,32,32,32,115,101,116,32,109,101,109,98,101,114,115,104,105,112,58,32,102,44,115,32,115,101,116,115,32,10,0,32,32,32,32,32,32,100,111,102,32,0,32,32,32,115,118,97,108,117,101,115,32,40,0,41,58,32,124,0,32,124,0,32,32,32,100,111,102,116,121,112,101,58,32,124,0,32,32,32,100,111,102,116,121,112,101,58,32,78,85,76,76,10,0,32,32,32,103,95,100,111,102,108,105,115,116,32,40,0,32,32,32,102,95,100,111,102,108,105,115,116,32,40,0,32,32,32,115,95,100,111,102,108,105,115,116,32,40,0,32,32,32,72,111,111,107,58,32,10,0,32,32,32,32,32,32,110,117,109,61,0,32,32,32,32,32,32,105,100,115,58,32,0,32,32,32,32,32,32,111,102,102,115,101,116,115,58,32,0,32,32,32,72,111,111,107,58,32,110,117,109,61,48,32,10,0,32,32,32,32,32,32,119,97,114,110,105,110,103,58,32,111,98,106,101,99,116,32,110,111,116,32,104,111,111,107,101,100,32,121,101,116,10,0,32,32,32,32,32,32,111,98,106,101,99,116,115,58,10,32,32,32,0,32,32,32,32,32,32,32,32,32,111,98,106,101,99,116,32,0,32,32,32,32,32,32,32,32,32,32,32,32,110,111,32,111,98,106,101,99,116,32,104,111,111,107,101,100,32,121,101,116,32,40,110,111,116,32,99,111,110,102,105,103,117,114,101,100,41,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,72,111,111,107,46,99,112,112,0,119,114,111,110,103,32,105,100,58,32,0,32,118,115,32,0,32,32,105,110,32,114,101,115,111,108,118,101,100,32,112,111,105,110,116,101,114,33,0,99,111,110,102,105,103,117,114,101,0,116,114,121,105,110,103,32,116,111,32,100,101,108,105,118,101,114,32,97,32,115,105,110,103,108,101,32,104,111,111,107,32,111,98,106,101,99,116,32,119,104,101,110,32,104,111,111,107,32,104,111,108,100,115,32,0,32,111,98,106,101,99,116,115,0,100,101,108,105,118,101,114,115,0,104,111,111,107,32,105,115,32,110,111,116,32,112,111,105,110,116,105,110,103,32,116,111,32,97,110,121,32,111,98,106,101,99,116,44,32,111,98,106,101,99,116,115,32,112,111,105,110,116,101,114,32,105,115,32,78,85,76,76,0,67,97,110,110,111,116,32,115,112,97,119,110,32,104,111,111,107,32,119,105,116,104,32,0,32,111,98,106,101,99,116,115,32,102,114,111,109,32,97,32,72,111,111,107,32,111,102,32,0,83,112,97,119,110,0,84,114,121,105,110,103,32,116,111,32,115,112,97,119,110,32,97,110,32,101,109,112,116,121,32,69,108,101,109,101,110,116,80,114,111,112,101,114,116,105,101,115,33,0,54,77,111,117,108,105,110,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,77,111,117,108,105,110,46,99,112,112,0,77,111,117,108,105,110,58,10,0,68,111,110,39,116,32,107,110,111,119,32,119,104,121,32,119,101,32,115,104,111,117,108,100,32,98,101,32,104,101,114,101,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,46,46,47,46,47,76,111,97,100,115,47,77,111,117,108,105,110,46,104,0,49,49,78,101,117,109,97,110,110,102,108,117,120,0,87,114,111,110,103,32,100,105,114,101,99,116,105,111,110,32,100,117,114,105,110,103,32,116,104,101,32,77,97,114,115,104,97,108,108,32,112,114,111,99,101,115,115,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,78,101,117,109,97,110,110,102,108,117,120,46,99,112,112,0,78,101,117,109,97,110,110,102,108,117,120,58,10,0,32,32,32,105,100,58,32,0,32,32,32,97,110,97,108,121,115,105,115,95,116,121,112,101,58,32,0,32,32,32,112,97,114,97,109,101,116,101,114,115,10,0,32,32,32,32,32,32,78,85,76,76,10,0,32,32,32,112,97,114,97,109,101,116,101,114,115,58,32,0,97,110,97,108,121,115,105,115,32,0,32,40,0,41,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,67,114,101,97,116,101,75,77,97,116,114,105,120,0,67,114,101,97,116,101,80,86,101,99,116,111,114,0,83,101,116,119,105,115,101,78,111,100,101,67,111,110,110,101,99,116,105,118,105,116,121,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,99,108,97,115,115,101,115,47,76,111,97,100,115,47,46,46,47,46,47,76,111,97,100,115,47,78,101,117,109,97,110,110,102,108,117,120,46,104,0,73,110,112,117,116,85,112,100,97,116,101,70,114,111,109,73,111,77,111,100,101,108,0,78,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,0,80,101,110,97,108,116,121,67,114,101,97,116,101,74,97,99,111,98,105,97,110,77,97,116,114,105,120,0,67,114,101,97,116,101,74,97,99,111,98,105,97,110,77,97,116,114,105,120,0,77,97,120,32,105,116,101,114,97,116,105,111,110,115,32,101,120,99,101,101,100,101,100,32,102,111,114,32,108,61,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,104,97,114,101,100,47,78,117,109,101,114,105,99,115,47,71,97,117,115,115,80,111,105,110,116,115,46,99,112,112,0,71,97,117,115,115,82,101,99,117,114,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,67,114,101,97,116,101,70,97,99,101,115,46,99,112,112,0,67,114,101,97,116,101,70,97,99,101,115,0,109,101,115,104,32,0,32,110,111,116,32,115,117,112,112,111,114,116,101,100,0,67,114,101,97,116,101,70,97,99,101,115,51,100,0,109,101,115,104,32,116,121,112,101,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,67,114,101,97,116,101,83,105,110,103,108,101,78,111,100,101,84,111,69,108,101,109,101,110,116,67,111,110,110,101,99,116,105,118,105,116,121,46,99,112,112,0,67,114,101,97,116,101,83,105,110,103,108,101,78,111,100,101,84,111,69,108,101,109,101,110,116,67,111,110,110,101,99,116,105,118,105,116,121,0,109,101,115,104,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,67,114,101,97,116,101,78,117,109,98,101,114,78,111,100,101,84,111,69,108,101,109,101,110,116,67,111,110,110,101,99,116,105,118,105,116,121,46,99,112,112,0,67,114,101,97,116,101,78,117,109,98,101,114,78,111,100,101,84,111,69,108,101,109,101,110,116,67,111,110,110,101,99,116,105,118,105,116,121,0,110,111,116,32,115,117,112,112,111,114,116,101,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,67,114,101,97,116,101,78,111,100,101,115,46,99,112,112,0,67,114,101,97,116,101,78,111,100,101,115,0,105,115,115,109,0,83,84,79,80,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,65,108,108,111,99,97,116,101,83,121,115,116,101,109,77,97,116,114,105,99,101,115,120,47,65,108,108,111,99,97,116,101,83,121,115,116,101,109,77,97,116,114,105,99,101,115,120,46,99,112,112,0,77,97,116,114,105,120,78,111,110,122,101,114,111,115,0,77,112,105,68,101,110,115,101,32,109,97,116,114,105,120,32,114,101,113,117,105,114,101,115,32,99,111,109,112,105,108,97,116,105,111,110,32,111,102,32,77,80,73,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,65,108,108,111,99,97,116,101,83,121,115,116,101,109,77,97,116,114,105,99,101,115,120,47,46,46,47,46,46,47,99,108,97,115,115,101,115,47,46,46,47,116,111,111,108,107,105,116,115,47,46,47,111,98,106,101,99,116,115,47,46,46,47,105,115,115,109,47,46,47,73,115,115,109,77,97,116,46,104,0,73,115,115,109,77,97,116,0,77,112,105,83,112,97,114,115,101,32,109,97,116,114,105,120,32,114,101,113,117,105,114,101,115,32,99,111,109,112,105,108,97,116,105,111,110,32,111,102,32,77,80,73,33,0,109,97,116,114,105,120,32,116,121,112,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,33,0,32,32,32,71,101,110,101,114,97,116,105,110,103,32,109,97,116,114,105,99,101,115,0,32,40,75,102,102,32,115,116,105,102,102,110,101,115,115,32,109,97,116,114,105,120,32,115,105,122,101,58,32,0,32,120,32,0,41,10,0,32,32,32,67,114,101,97,116,101,32,110,111,100,97,108,32,99,111,110,115,116,114,97,105,110,116,115,10,0,101,120,112,101,99,116,105,110,103,32,97,32,73,115,115,109,68,111,117,98,108,101,32,118,101,99,116,111,114,32,102,111,114,32,99,111,110,115,116,114,97,105,110,116,115,32,119,105,116,104,32,101,110,117,109,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,73,111,77,111,100,101,108,84,111,67,111,110,115,116,114,97,105,110,116,115,120,47,73,111,77,111,100,101,108,84,111,67,111,110,115,116,114,97,105,110,116,115,120,46,99,112,112,0,73,111,77,111,100,101,108,84,111,67,111,110,115,116,114,97,105,110,116,115,120,0,101,120,112,101,99,116,105,110,103,32,97,32,110,111,100,97,108,32,118,101,99,116,111,114,32,102,111,114,32,99,111,110,115,116,114,97,105,110,116,115,32,119,105,116,104,32,101,110,117,109,32,0,109,101,115,104,32,116,121,112,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,70,105,110,105,116,101,32,101,108,101,109,101,110,116,32,0,83,105,122,101,32,111,102,32,115,112,99,32,102,105,101,108,100,32,110,111,116,32,115,117,112,112,111,114,116,101,100,0,73,111,77,111,100,101,108,84,111,68,121,110,97,109,105,99,67,111,110,115,116,114,97,105,110,116,115,120,0,32,32,32,73,110,112,117,116,32,117,112,100,97,116,101,115,32,102,114,111,109,32,99,111,110,115,116,97,110,116,10,0,32,32,32,85,112,100,97,116,105,110,103,32,105,110,112,117,116,115,32,102,114,111,109,32,115,111,108,117,116,105,111,110,10,0,9,99,97,108,108,32,77,97,110,116,108,101,32,80,108,117,109,101,32,71,101,111,116,104,101,114,109,97,108,32,70,108,117,120,32,109,111,100,117,108,101,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,71,101,111,116,104,101,114,109,97,108,70,108,117,120,120,47,71,101,111,116,104,101,114,109,97,108,70,108,117,120,120,46,99,112,112,0,71,101,111,116,104,101,114,109,97,108,70,108,117,120,120,0,32,32,32,71,101,116,32,115,111,108,117,116,105,111,110,32,102,114,111,109,32,105,110,112,117,116,115,10,0,65,108,108,111,99,97,116,105,110,103,32,97,32,86,101,99,32,111,102,32,115,105,122,101,32,48,32,97,115,32,103,115,105,122,101,61,48,32,102,111,114,32,99,111,110,102,105,103,117,114,97,116,105,111,110,32,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,71,101,116,83,111,108,117,116,105,111,110,70,114,111,109,73,110,112,117,116,115,120,47,71,101,116,83,111,108,117,116,105,111,110,70,114,111,109,73,110,112,117,116,115,120,46,99,112,112,0,71,101,116,83,111,108,117,116,105,111,110,70,114,111,109,73,110,112,117,116,115,120,0,9,99,97,108,108,32,76,105,110,101,97,114,32,70,108,111,97,116,105,110,103,32,109,101,108,116,105,110,103,32,114,97,116,101,32,109,111,100,117,108,101,10,0,9,99,97,108,108,32,77,105,115,109,105,112,32,70,108,111,97,116,105,110,103,32,109,101,108,116,105,110,103,32,114,97,116,101,32,109,111,100,117,108,101,10,0,66,97,115,97,108,32,102,111,114,99,105,110,103,32,109,111,100,101,108,32,0,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,70,108,111,97,116,105,110,103,105,99,101,77,101,108,116,105,110,103,82,97,116,101,120,47,70,108,111,97,116,105,110,103,105,99,101,77,101,108,116,105,110,103,82,97,116,101,120,46,99,112,112,0,70,108,111,97,116,105,110,103,105,99,101,77,101,108,116,105,110,103,82,97,116,101,120,0,32,32,32,82,101,100,117,99,101,32,118,101,99,116,111,114,32,102,114,111,109,32,103,32,116,111,32,102,32,115,101,116,10,0,32,32,32,68,105,114,105,99,104,108,101,116,32,108,105,102,116,105,110,103,32,97,112,112,108,105,101,100,32,116,111,32,108,111,97,100,32,118,101,99,116,111,114,10,0,32,32,32,67,111,110,115,116,114,97,105,110,105,110,103,32,112,101,110,97,108,116,105,101,115,10,0,32,32,32,82,101,115,101,116,116,105,110,103,32,112,101,110,97,108,116,105,101,115,10,0,114,105,102,116,32,99,111,110,115,116,114,97,105,110,116,115,32,114,101,115,101,116,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,82,101,115,101,116,67,111,110,115,116,114,97,105,110,116,115,120,47,82,101,115,101,116,67,111,110,115,116,114,97,105,110,116,115,120,46,99,112,112,0,82,101,115,101,116,67,111,110,115,116,114,97,105,110,116,115,120,0,32,32,32,83,111,108,118,105,110,103,32,109,97,116,114,105,120,32,115,121,115,116,101,109,10,0,77,97,116,114,105,120,32,116,121,112,101,58,32,0,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,83,111,108,118,101,114,120,47,46,46,47,46,46,47,116,111,111,108,107,105,116,115,47,46,47,111,98,106,101,99,116,115,47,46,47,83,111,108,118,101,114,46,104,0,83,111,108,118,101,0,32,32,32,77,101,114,103,105,110,103,32,115,111,108,117,116,105,111,110,32,118,101,99,116,111,114,32,102,114,111,109,32,102,115,101,116,32,116,111,32,103,115,101,116,10,0,32,32,32,117,112,100,97,116,105,110,103,32,98,111,117,110,100,97,114,121,32,99,111,110,100,105,116,105,111,110,115,46,46,46,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,115,47,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,95,108,97,46,99,112,112,0,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,95,108,97,0,32,32,32,67,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,58,32,110,111,114,109,40,100,112,41,47,110,111,114,109,40,112,41,0,32,32,32,67,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,58,32,110,111,114,109,40,100,112,47,41,47,110,111,114,109,40,112,41,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,115,47,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,95,108,97,95,116,104,101,116,97,46,99,112,112,0,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,95,108,97,95,116,104,101,116,97,0,32,32,32,110,117,109,98,101,114,32,111,102,32,117,110,115,116,97,98,108,101,32,99,111,110,115,116,114,97,105,110,116,115,58,32,0,32,32,32,109,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,110,111,110,108,105,110,101,97,114,32,105,116,101,114,97,116,105,111,110,115,32,40,0,32,32,32,109,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,78,101,119,116,111,110,32,105,116,101,114,97,116,105,111,110,115,32,40,0,10,32,32,32,116,111,116,97,108,32,110,117,109,98,101,114,32,111,102,32,105,116,101,114,97,116,105,111,110,115,58,32,0,97,110,97,108,121,115,105,115,32,116,121,112,101,32,0,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,102,111,114,32,70,67,84,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,115,47,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,95,102,99,116,46,99,112,112,0,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,95,102,99,116,0,80,69,84,83,99,32,110,101,101,100,115,32,116,111,32,98,101,32,105,110,115,116,97,108,108,101,100,0,32,32,32,99,104,101,99,107,105,110,103,32,99,111,110,118,101,114,103,101,110,99,101,10,0,32,32,32,115,111,108,118,101,114,32,114,101,115,105,100,117,101,58,32,110,111,114,109,40,75,85,45,70,41,47,110,111,114,109,40,70,41,61,0,110,111,114,109,32,110,102,32,61,32,0,102,32,97,110,100,32,110,111,114,109,32,107,117,111,108,100,32,61,32,0,102,10,0,109,101,99,104,97,110,105,99,97,108,32,101,113,117,105,108,105,98,114,105,117,109,32,99,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,32,105,115,32,78,97,78,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,111,108,117,116,105,111,110,115,101,113,117,101,110,99,101,115,47,99,111,110,118,101,114,103,101,110,99,101,46,99,112,112,0,99,111,110,118,101,114,103,101,110,99,101,0,32,32,32,109,101,99,104,97,110,105,99,97,108,32,101,113,117,105,108,105,98,114,105,117,109,32,99,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,0,32,60,32,0,32,37,10,0,32,62,32,0,99,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,32,105,115,32,78,97,78,33,0,32,32,32,67,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,58,32,110,111,114,109,40,100,117,41,47,110,111,114,109,40,117,41,0,32,32,32,67,111,110,118,101,114,103,101,110,99,101,32,99,114,105,116,101,114,105,111,110,58,32,109,97,120,40,100,117,41,0,32,32,32,67,111,109,112,117,116,105,110,103,32,99,111,115,116,32,102,117,110,99,116,105,111,110,32,103,114,97,100,105,101,110,116,10,0,124,124,100,74,47,100,107,124,124,32,61,32,48,32,32,32,32,103,114,97,100,105,101,110,116,32,110,111,114,109,32,105,115,32,122,101,114,111,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,71,114,97,100,106,120,47,71,114,97,100,106,120,46,99,112,112,0,71,114,97,100,106,120,0,124,124,100,74,47,100,107,124,124,32,61,32,78,97,78,32,32,103,114,97,100,105,101,110,116,32,110,111,114,109,32,105,115,32,78,97,78,0,32,32,32,32,32,32,32,120,32,32,32,32,32,32,32,124,32,32,67,111,115,116,32,102,117,110,99,116,105,111,110,32,102,40,120,41,32,32,124,32,32,76,105,115,116,32,111,102,32,99,111,110,116,114,105,98,117,116,105,111,110,115,10,0,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,32,115,116,101,112,32,0,47,0,32,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,10,0,32,120,32,61,32,0,32,124,32,0,70,117,110,99,116,105,111,110,32,101,118,97,108,117,97,116,105,111,110,32,114,101,116,117,114,110,101,100,32,78,97,78,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,115,104,97,114,101,100,47,78,117,109,101,114,105,99,115,47,66,114,101,110,116,83,101,97,114,99,104,46,99,112,112,0,66,114,101,110,116,83,101,97,114,99,104,0,32,32,32,99,111,109,112,117,116,105,110,103,32,118,101,108,111,99,105,116,105,101,115,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,116,104,105,99,107,110,101,115,115,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,116,104,105,99,107,110,101,115,115,50,10,0,32,32,32,99,111,109,112,117,116,105,110,103,32,97,100,106,111,105,110,116,10,0,32,32,32,115,97,118,105,110,103,32,114,101,115,117,108,116,115,10,0,32,32,32,109,97,120,105,109,117,109,32,110,117,109,98,101,114,32,111,102,32,105,116,101,114,97,116,105,111,110,115,32,40,0,41,32,101,120,99,101,101,100,101,100,10,0,99,111,109,112,117,116,105,110,103,32,115,108,111,112,101,32,111,102,32,108,101,118,101,108,115,101,116,32,102,117,110,99,116,105,111,110,46,46,46,10,0,115,97,118,105,110,103,32,114,101,115,117,108,116,115,58,10,0,32,32,32,102,114,101,101,122,105,110,103,32,99,111,110,115,116,114,97,105,110,116,115,10,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,98,97,109,103,47,71,101,111,109,101,116,114,121,46,99,112,112,0,114,101,102,101,114,101,110,99,101,32,110,117,109,98,101,114,115,58,32,0,32,0,10,0,73,100,58,32,0,67,111,111,114,100,115,58,32,91,0,93,32,91,0,93,10,0,116,119,111,32,112,111,105,110,116,115,32,111,102,32,116,104,101,32,103,101,111,109,101,116,114,121,32,97,114,101,32,118,101,114,121,32,99,108,111,115,101,100,32,116,111,32,101,97,99,104,32,111,116,104,101,114,32,40,115,101,101,32,114,101,102,101,114,101,110,99,101,32,110,117,109,98,101,114,115,32,97,98,111,118,101,41,0,80,111,115,116,82,101,97,100,0,76,101,110,103,116,104,32,111,102,32,101,100,103,101,32,0,32,105,115,32,48,0,80,114,111,98,108,101,109,32,119,104,105,108,101,32,112,114,111,99,101,115,115,105,110,103,32,101,100,103,101,115,58,32,99,104,101,99,107,32,116,104,101,32,101,100,103,101,32,108,105,115,116,0,84,114,121,105,110,103,32,116,111,32,100,101,108,101,116],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+129320);allocate([101,32,103,101,111,109,101,116,114,121,32,97,110,100,32,78,98,82,101,102,62,48,44,32,112,114,111,98,97,98,108,121,32,100,117,101,32,116,111,32,97,110,32,101,114,114,111,114,0,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,33,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,98,97,109,103,47,46,47,66,97,109,103,81,117,97,100,116,114,101,101,46,104,0,77,97,114,115,104,97,108,108,0,99,111,112,121,0,79,98,106,101,99,116,69,110,117,109,0,73,100,0,68,101,101,112,69,99,104,111,0,69,99,104,111,0,78,52,98,97,109,103,49,50,66,97,109,103,81,117,97,100,116,114,101,101,49,53,66,97,109,103,81,117,97,100,116,114,101,101,66,111,120,69,0,83,101,116,79,102,69,100,103,101,115,52,58,58,97,100,100,32,111,118,101,114,102,108,111,119,58,32,78,98,79,102,69,100,103,101,115,61,0,32,62,32,110,98,97,120,61,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,98,97,109,103,47,83,101,116,79,102,69,52,46,99,112,112,0,97,100,100,0,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,32,121,101,116,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,78,111,100,101,115,80,97,114,116,105,116,105,111,110,105,110,103,46,99,112,112,0,68,105,115,99,111,110,116,105,110,117,111,117,115,71,97,108,101,114,107,105,110,78,111,100,101,115,80,97,114,116,105,116,105,111,110,105,110,103,0,80,114,111,98,108,101,109,32,105,110,32,102,97,99,101,115,32,99,114,101,97,116,105,111,110,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,69,100,103,101,115,80,97,114,116,105,116,105,111,110,105,110,103,46,99,112,112,0,69,100,103,101,115,80,97,114,116,105,116,105,111,110,105,110,103,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,70,97,99,101,115,80,97,114,116,105,116,105,111,110,105,110,103,46,99,112,112,0,70,97,99,101,115,80,97,114,116,105,116,105,111,110,105,110,103,0,110,111,116,32,101,110,111,117,103,104,32,101,108,101,109,101,110,116,115,32,105,110,32,109,101,115,104,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,105,115,115,109,45,116,114,117,110,107,115,47,116,114,117,110,107,45,106,112,108,45,106,115,47,115,114,99,47,99,47,46,47,109,111,100,117,108,101,115,47,77,111,100,101,108,80,114,111,99,101,115,115,111,114,120,47,67,114,101,97,116,101,69,100,103,101,115,46,99,112,112,0,67,114,101,97,116,101,69,100,103,101,115,0,109,101,115,104,32,100,105,109,101,110,115,105,111,110,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,121,101,116,0,69,100,103,101,79,110,66,111,117,110,100,97,114,121,70,108,97,103,115,0,100,105,109,101,110,115,105,111,110,32,110,111,116,32,115,117,112,112,111,114,116,101,100,0,105,110,118,97,108,105,100,32,108,101,110,103,116,104,0,98,108,97,115,46,99,0,109,97,116,114,105,120,32,109,117,115,116,32,98,101,32,115,113,117,97,114,101,0,69,82,82,79,82,0,68,101,102,97,117,108,116,32,71,83,76,32,101,114,114,111,114,32,104,97,110,100,108,101,114,32,105,110,118,111,107,101,100,46,10,0,103,115,108,58,32,37,115,58,37,100,58,32,37,115,58,32,37,115,10,0,76,85,32,100,101,99,111,109,112,111,115,105,116,105,111,110,32,114,101,113,117,105,114,101,115,32,115,113,117,97,114,101,32,109,97,116,114,105,120,0,108,117,46,99,0,112,101,114,109,117,116,97,116,105,111,110,32,108,101,110,103,116,104,32,109,117,115,116,32,109,97,116,99,104,32,109,97,116,114,105,120,32,115,105,122,101,0,76,85,32,109,97,116,114,105,120,32,109,117,115,116,32,98,101,32,115,113,117,97,114,101,0,109,97,116,114,105,120,32,115,105,122,101,32,109,117,115,116,32,109,97,116,99,104,32,98,32,115,105,122,101,0,109,97,116,114,105,120,32,115,105,122,101,32,109,117,115,116,32,109,97,116,99,104,32,115,111,108,117,116,105,111,110,32,115,105,122,101,0,109,97,116,114,105,120,32,105,115,32,115,105,110,103,117,108,97,114,0,109,97,116,114,105,120,32,115,105,122,101,32,109,117,115,116,32,109,97,116,99,104,32,115,111,108,117,116,105,111,110,47,114,104,115,32,115,105,122,101,0,102,105,114,115,116,32,114,111,119,32,105,110,100,101,120,32,105,115,32,111,117,116,32,111,102,32,114,97,110,103,101,0,46,47,115,119,97,112,95,115,111,117,114,99,101,46,99,0,115,101,99,111,110,100,32,114,111,119,32,105,110,100,101,120,32,105,115,32,111,117,116,32,111,102,32,114,97,110,103,101,0,109,97,116,114,105,120,32,100,105,109,101,110,115,105,111,110,32,110,49,32,109,117,115,116,32,98,101,32,112,111,115,105,116,105,118,101,32,105,110,116,101,103,101,114,0,109,97,116,114,105,120,32,100,105,109,101,110,115,105,111,110,32,110,50,32,109,117,115,116,32,98,101,32,112,111,115,105,116,105,118,101,32,105,110,116,101,103,101,114,0,112,101,114,109,117,116,97,116,105,111,110,32,108,101,110,103,116,104,32,110,32,109,117,115,116,32,98,101,32,112,111,115,105,116,105,118,101,32,105,110,116,101,103,101,114,0,105,110,105,116,46,99,0,102,97,105,108,101,100,32,116,111,32,97,108,108,111,99,97,116,101,32,115,112,97,99,101,32,102,111,114,32,112,101,114,109,117,116,97,116,105,111,110,32,115,116,114,117,99,116,0,102,97,105,108,101,100,32,116,111,32,97,108,108,111,99,97,116,101,32,115,112,97,99,101,32,102,111,114,32,112,101,114,109,117,116,97,116,105,111,110,32,100,97,116,97,0,102,105,114,115,116,32,105,110,100,101,120,32,105,115,32,111,117,116,32,111,102,32,114,97,110,103,101,0,112,101,114,109,117,116,97,116,105,111,110,46,99,0,115,101,99,111,110,100,32,105,110,100,101,120,32,105,115,32,111,117,116,32,111,102,32,114,97,110,103,101,0,118,101,99,116,111,114,32,97,110,100,32,112,101,114,109,117,116,97,116,105,111,110,32,109,117,115,116,32,98,101,32,116,104,101,32,115,97,109,101,32,108,101,110,103,116,104,0,46,47,112,101,114,109,117,116,101,95,115,111,117,114,99,101,46,99,0,118,101,99,116,111,114,32,108,101,110,103,116,104,115,32,97,114,101,32,110,111,116,32,101,113,117,97,108,0,46,47,99,111,112,121,95,115,111,117,114,99,101,46,99,0,118,101,99,116,111,114,32,108,101,110,103,116,104,32,110,32,109,117,115,116,32,98,101,32,112,111,115,105,116,105,118,101,32,105,110,116,101,103,101,114,0,46,47,118,105,101,119,95,115,111,117,114,99,101,46,99,0,46,47,115,111,117,114,99,101,95,116,114,115,118,95,114,46,104,0,117,110,114,101,99,111,103,110,105,122,101,100,32,111,112,101,114,97,116,105,111,110,0,80,97,114,97,109,101,116,101,114,32,37,100,32,116,111,32,114,111,117,116,105,110,101,32,37,115,32,119,97,115,32,105,110,99,111,114,114,101,99,116,10,0,83,116,57,98,97,100,95,97,108,108,111,99,0,83,116,57,101,120,99,101,112,116,105,111,110,0,83,116,49,51,114,117,110,116,105,109,101,95,101,114,114,111,114,0,83,116,57,116,121,112,101,95,105,110,102,111,0,83,116,56,98,97,100,95,99,97,115,116,0,78,49,48,95,95,99,120,120,97,98,105,118,49,49,54,95,95,115,104,105,109,95,116,121,112,101,95,105,110,102,111,69,0,78,49,48,95,95,99,120,120,97,98,105,118,49,49,55,95,95,99,108,97,115,115,95,116,121,112,101,95,105,110,102,111,69,0,78,49,48,95,95,99,120,120,97,98,105,118,49,49,57,95,95,112,111,105,110,116,101,114,95,116,121,112,101,95,105,110,102,111,69,0,78,49,48,95,95,99,120,120,97,98,105,118,49,49,55,95,95,112,98,97,115,101,95,116,121,112,101,95,105,110,102,111,69,0,78,49,48,95,95,99,120,120,97,98,105,118,49,50,48,95,95,115,105,95,99,108,97,115,115,95,116,121,112,101,95,105,110,102,111,69,0,78,49,48,95,95,99,120,120,97,98,105,118,49,50,49,95,95,118,109,105,95,99,108,97,115,115,95,116,121,112,101,95,105,110,102,111,69,0,33,34,98,97,115,105,99,95,115,116,114,105,110,103,32,108,101,110,103,116,104,95,101,114,114,111,114,34,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,101,120,116,101,114,110,97,108,112,97,99,107,97,103,101,115,47,101,109,115,99,114,105,112,116,101,110,47,101,109,115,100,107,95,112,111,114,116,97,98,108,101,47,101,109,115,99,114,105,112,116,101,110,47,116,97,103,45,49,46,51,53,46,56,47,115,121,115,116,101,109,47,105,110,99,108,117,100,101,47,108,105,98,99,120,120,47,115,116,114,105,110,103,0,95,95,116,104,114,111,119,95,108,101,110,103,116,104,95,101,114,114,111,114,0,33,34,118,101,99,116,111,114,32,108,101,110,103,116,104,95,101,114,114,111,114,34,0,47,85,115,101,114,115,47,108,97,114,111,117,114,47,101,120,116,101,114,110,97,108,112,97,99,107,97,103,101,115,47,101,109,115,99,114,105,112,116,101,110,47,101,109,115,100,107,95,112,111,114,116,97,98,108,101,47,101,109,115,99,114,105,112,116,101,110,47,116,97,103,45,49,46,51,53,46,56,47,115,121,115,116,101,109,47,105,110,99,108,117,100,101,47,108,105,98,99,120,120,47,118,101,99,116,111,114,0,112,116,104,114,101,97,100,95,111,110,99,101,32,102,97,105,108,117,114,101,32,105,110,32,95,95,99,120,97,95,103,101,116,95,103,108,111,98,97,108,115,95,102,97,115,116,40,41,0,115,116,100,58,58,98,97,100,95,97,108,108,111,99,0,116,101,114,109,105,110,97,116,101,95,104,97,110,100,108,101,114,32,117,110,101,120,112,101,99,116,101,100,108,121,32,114,101,116,117,114,110,101,100,0,116,101,114,109,105,110,97,116,101,95,104,97,110,100,108,101,114,32,117,110,101,120,112,101,99,116,101,100,108,121,32,116,104,114,101,119,32,97,110,32,101,120,99,101,112,116,105,111,110,0,115,116,100,58,58,98,97,100,95,99,97,115,116,0,99,97,110,110,111,116,32,99,114,101,97,116,101,32,112,116,104,114,101,97,100,32,107,101,121,32,102,111,114,32,95,95,99,120,97,95,103,101,116,95,103,108,111,98,97,108,115,40,41,0,99,97,110,110,111,116,32,122,101,114,111,32,111,117,116,32,116,104,114,101,97,100,32,118,97,108,117,101,32,102,111,114,32,95,95,99,120,97,95,103,101,116,95,103,108,111,98,97,108,115,40,41,0,116,101,114,109,105,110,97,116,105,110,103,32,119,105,116,104,32,37,115,32,101,120,99,101,112,116,105,111,110,32,111,102,32,116,121,112,101,32,37,115,58,32,37,115,0,116,101,114,109,105,110,97,116,105,110,103,32,119,105,116,104,32,37,115,32,101,120,99,101,112,116,105,111,110,32,111,102,32,116,121,112,101,32,37,115,0,116,101,114,109,105,110,97,116,105,110,103,32,119,105,116,104,32,37,115,32,102,111,114,101,105,103,110,32,101,120,99,101,112,116,105,111,110,0,116,101,114,109,105,110,97,116,105,110,103,0,117,110,99,97,117,103,104,116,0,84,33,34,25,13,1,2,3,17,75,28,12,16,4,11,29,18,30,39,104,110,111,112,113,98,32,5,6,15,19,20,21,26,8,22,7,40,36,23,24,9,10,14,27,31,37,35,131,130,125,38,42,43,60,61,62,63,67,71,74,77,88,89,90,91,92,93,94,95,96,97,99,100,101,102,103,105,106,107,108,114,115,116,121,122,123,124,0,73,108,108,101,103,97,108,32,98,121,116,101,32,115,101,113,117,101,110,99,101,0,68,111,109,97,105,110,32,101,114,114,111,114,0,82,101,115,117,108,116,32,110,111,116,32,114,101,112,114,101,115,101,110,116,97,98,108,101,0,78,111,116,32,97,32,116,116,121,0,80,101,114,109,105,115,115,105,111,110,32,100,101,110,105,101,100,0,79,112,101,114,97,116,105,111,110,32,110,111,116,32,112,101,114,109,105,116,116,101,100,0,78,111,32,115,117,99,104,32,102,105,108,101,32,111,114,32,100,105,114,101,99,116,111,114,121,0,78,111,32,115,117,99,104,32,112,114,111,99,101,115,115,0,70,105,108,101,32,101,120,105,115,116,115,0,86,97,108,117,101,32,116,111,111,32,108,97,114,103,101,32,102,111,114,32,100,97,116,97,32,116,121,112,101,0,78,111,32,115,112,97,99,101,32,108,101,102,116,32,111,110,32,100,101,118,105,99,101,0,79,117,116,32,111,102,32,109,101,109,111,114,121,0,82,101,115,111,117,114,99,101,32,98,117,115,121,0,73,110,116,101,114,114,117,112,116,101,100,32,115,121,115,116,101,109,32,99,97,108,108,0,82,101,115,111,117,114,99,101,32,116,101,109,112,111,114,97,114,105,108,121,32,117,110,97,118,97,105,108,97,98,108,101,0,73,110,118,97,108,105,100,32,115,101,101,107,0,67,114,111,115,115,45,100,101,118,105,99,101,32,108,105,110,107,0,82,101,97,100,45,111,110,108,121,32,102,105,108,101,32,115,121,115,116,101,109,0,68,105,114,101,99,116,111,114,121,32,110,111,116,32,101,109,112,116,121,0,67,111,110,110,101,99,116,105,111,110,32,114,101,115,101,116,32,98,121,32,112,101,101,114,0,79,112,101,114,97,116,105,111,110,32,116,105,109,101,100,32,111,117,116,0,67,111,110,110,101,99,116,105,111,110,32,114,101,102,117,115,101,100,0,72,111,115,116,32,105,115,32,100,111,119,110,0,72,111,115,116,32,105,115,32,117,110,114,101,97,99,104,97,98,108,101,0,65,100,100,114,101,115,115,32,105,110,32,117,115,101,0,66,114,111,107,101,110,32,112,105,112,101,0,73,47,79,32,101,114,114,111,114,0,78,111,32,115,117,99,104,32,100,101,118,105,99,101,32,111,114,32,97,100,100,114,101,115,115,0,66,108,111,99,107,32,100,101,118,105,99,101,32,114,101,113,117,105,114,101,100,0,78,111,32,115,117,99,104,32,100,101,118,105,99,101,0,78,111,116,32,97,32,100,105,114,101,99,116,111,114,121,0,73,115,32,97,32,100,105,114,101,99,116,111,114,121,0,84,101,120,116,32,102,105,108,101,32,98,117,115,121,0,69,120,101,99,32,102,111,114,109,97,116,32,101,114,114,111,114,0,73,110,118,97,108,105,100,32,97,114,103,117,109,101,110,116,0,65,114,103,117,109,101,110,116,32,108,105,115,116,32,116,111,111,32,108,111,110,103,0,83,121,109,98,111,108,105,99,32,108,105,110,107,32,108,111,111,112,0,70,105,108,101,110,97,109,101,32,116,111,111,32,108,111,110,103,0,84,111,111,32,109,97,110,121,32,111,112,101,110,32,102,105,108,101,115,32,105,110,32,115,121,115,116,101,109,0,78,111,32,102,105,108,101,32,100,101,115,99,114,105,112,116,111,114,115,32,97,118,97,105,108,97,98,108,101,0,66,97,100,32,102,105,108,101,32,100,101,115,99,114,105,112,116,111,114,0,78,111,32,99,104,105,108,100,32,112,114,111,99,101,115,115,0,66,97,100,32,97,100,100,114,101,115,115,0,70,105,108,101,32,116,111,111,32,108,97,114,103,101,0,84,111,111,32,109,97,110,121,32,108,105,110,107,115,0,78,111,32,108,111,99,107,115,32,97,118,97,105,108,97,98,108,101,0,82,101,115,111,117,114,99,101,32,100,101,97,100,108,111,99,107,32,119,111,117,108,100,32,111,99,99,117,114,0,83,116,97,116,101,32,110,111,116,32,114,101,99,111,118,101,114,97,98,108,101,0,80,114,101,118,105,111,117,115,32,111,119,110,101,114,32,100,105,101,100,0,79,112,101,114,97,116,105,111,110,32,99,97,110,99,101,108,101,100,0,70,117,110,99,116,105,111,110,32,110,111,116,32,105,109,112,108,101,109,101,110,116,101,100,0,78,111,32,109,101,115,115,97,103,101,32,111,102,32,100,101,115,105,114,101,100,32,116,121,112,101,0,73,100,101,110,116,105,102,105,101,114,32,114,101,109,111,118,101,100,0,68,101,118,105,99,101,32,110,111,116,32,97,32,115,116,114,101,97,109,0,78,111,32,100,97,116,97,32,97,118,97,105,108,97,98,108,101,0,68,101,118,105,99,101,32,116,105,109,101,111,117,116,0,79,117,116,32,111,102,32,115,116,114,101,97,109,115,32,114,101,115,111,117,114,99,101,115,0,76,105,110,107,32,104,97,115,32,98,101,101,110,32,115,101,118,101,114,101,100,0,80,114,111,116,111,99,111,108,32,101,114,114,111,114,0,66,97,100,32,109,101,115,115,97,103,101,0,70,105,108,101,32,100,101,115,99,114,105,112,116,111,114,32,105,110,32,98,97,100,32,115,116,97,116,101,0,78,111,116,32,97,32,115,111,99,107,101,116,0,68,101,115,116,105,110,97,116,105,111,110,32,97,100,100,114,101,115,115,32,114,101,113,117,105,114,101,100,0,77,101,115,115,97,103,101,32,116,111,111,32,108,97,114,103,101,0,80,114,111,116,111,99,111,108,32,119,114,111,110,103,32,116,121,112,101,32,102,111,114,32,115,111,99,107,101,116,0,80,114,111,116,111,99,111,108,32,110,111,116,32,97,118,97,105,108,97,98,108,101,0,80,114,111,116,111,99,111,108,32,110,111,116,32,115,117,112,112,111,114,116,101,100,0,83,111,99,107,101,116,32,116,121,112,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,0,78,111,116,32,115,117,112,112,111,114,116,101,100,0,80,114,111,116,111,99,111,108,32,102,97,109,105,108,121,32,110,111,116,32,115,117,112,112,111,114,116,101,100,0,65,100,100,114,101,115,115,32,102,97,109,105,108,121,32,110,111,116,32,115,117,112,112,111,114,116,101,100,32,98,121,32,112,114,111,116,111,99,111,108,0,65,100,100,114,101,115,115,32,110,111,116,32,97,118,97,105,108,97,98,108,101,0,78,101,116,119,111,114,107,32,105,115,32,100,111,119,110,0,78,101,116,119,111,114,107,32,117,110,114,101,97,99,104,97,98,108,101,0,67,111,110,110,101,99,116,105,111,110,32,114,101,115,101,116,32,98,121,32,110,101,116,119,111,114,107,0,67,111,110,110,101,99,116,105,111,110,32,97,98,111,114,116,101,100,0,78,111,32,98,117,102,102,101,114,32,115,112,97,99,101,32,97,118,97,105,108,97,98,108,101,0,83,111,99,107,101,116,32,105,115,32,99,111,110,110,101,99,116,101,100,0,83,111,99,107,101,116,32,110,111,116,32,99,111,110,110,101,99,116,101,100,0,67,97,110,110,111,116,32,115,101,110,100,32,97,102,116,101,114,32,115,111,99,107,101,116,32,115,104,117,116,100,111,119,110,0,79,112,101,114,97,116,105,111,110,32,97,108,114,101,97,100,121,32,105,110,32,112,114,111,103,114,101,115,115,0,79,112,101,114,97,116,105,111,110,32,105,110,32,112,114,111,103,114,101,115,115,0,83,116,97,108,101,32,102,105,108,101,32,104,97,110,100,108,101,0,82,101,109,111,116,101,32,73,47,79,32,101,114,114,111,114,0,81,117,111,116,97,32,101,120,99,101,101,100,101,100,0,78,111,32,109,101,100,105,117,109,32,102,111,117,110,100,0,87,114,111,110,103,32,109,101,100,105,117,109,32,116,121,112,101,0,78,111,32,101,114,114,111,114,32,105,110,102,111,114,109,97,116,105,111,110,0,0,105,110,102,105,110,105,116,121,0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,0,1,2,3,4,5,6,7,8,9,255,255,255,255,255,255,255,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,255,255,255,255,255,255,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,0,1,2,4,7,3,6,5,0,80,79,83,73,88,0,114,119,97],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+139560);allocate([17,0,10,0,17,17,17,0,0,0,0,5,0,0,0,0,0,0,9,0,0,0,0,11,0,0,0,0,0,0,0,0,17,0,15,10,17,17,17,3,10,7,0,1,19,9,11,11,0,0,9,6,11,0,0,11,0,6,17,0,0,0,17,17,17,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,11,0,0,0,0,0,0,0,0,17,0,10,10,17,17,17,0,10,0,0,2,0,9,11,0,0,0,9,0,11,0,0,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,12,0,0,0,0,12,0,0,0,0,9,12,0,0,0,0,0,12,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,14,0,0,0,0,0,0,0,0,0,0,0,13,0,0,0,4,13,0,0,0,0,9,14,0,0,0,0,0,14,0,0,14,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,0,0,0,0,0,0,0,0,0,0,0,15,0,0,0,0,15,0,0,0,0,9,16,0,0,0,0,0,16,0,0,16,0,0,18,0,0,0,18,18,18,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,18,0,0,0,18,18,18,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,11,0,0,0,0,0,0,0,0,0,0,0,10,0,0,0,0,10,0,0,0,0,9,11,0,0,0,0,0,11,0,0,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,12,0,0,0,0,12,0,0,0,0,9,12,0,0,0,0,0,12,0,0,12,0,0,48,49,50,51,52,53,54,55,56,57,65,66,67,68,69,70,45,43,32,32,32,48,88,48,120,0,40,110,117,108,108,41,0,45,48,88,43,48,88,32,48,88,45,48,120,43,48,120,32,48,120,0,105,110,102,0,73,78,70,0,110,97,110,0,78,65,78,0,46,0,0,78,83,116,51,95,95,49,49,49,95,95,115,116,100,111,117,116,98,117,102,73,119,69,69,0,117,110,115,117,112,112,111,114,116,101,100,32,108,111,99,97,108,101,32,102,111,114,32,115,116,97,110,100,97,114,100,32,105,110,112,117,116,0,78,83,116,51,95,95,49,49,48,95,95,115,116,100,105,110,98,117,102,73,119,69,69,0,78,83,116,51,95,95,49,49,49,95,95,115,116,100,111,117,116,98,117,102,73,99,69,69,0,78,83,116,51,95,95,49,49,48,95,95,115,116,100,105,110,98,117,102,73,99,69,69,0,78,83,116,51,95,95,49,56,105,111,115,95,98,97,115,101,55,102,97,105,108,117,114,101,69,0,78,83,116,51,95,95,49,56,105,111,115,95,98,97,115,101,69,0,78,83,116,51,95,95,49,57,98,97,115,105,99,95,105,111,115,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,69,69,0,78,83,116,51,95,95,49,57,98,97,115,105,99,95,105,111,115,73,119,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,119,69,69,69,69,0,78,83,116,51,95,95,49,49,53,98,97,115,105,99,95,115,116,114,101,97,109,98,117,102,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,69,69,0,78,83,116,51,95,95,49,49,53,98,97,115,105,99,95,115,116,114,101,97,109,98,117,102,73,119,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,119,69,69,69,69,0,78,83,116,51,95,95,49,49,51,98,97,115,105,99,95,105,115,116,114,101,97,109,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,69,69,0,78,83,116,51,95,95,49,49,51,98,97,115,105,99,95,105,115,116,114,101,97,109,73,119,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,119,69,69,69,69,0,78,83,116,51,95,95,49,49,51,98,97,115,105,99,95,111,115,116,114,101,97,109,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,69,69,0,78,83,116,51,95,95,49,49,51,98,97,115,105,99,95,111,115,116,114,101,97,109,73,119,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,119,69,69,69,69,0,78,83,116,51,95,95,49,49,57,95,95,105,111,115,116,114,101,97,109,95,99,97,116,101,103,111,114,121,69,0,105,111,115,95,98,97,115,101,58,58,99,108,101,97,114,0,105,111,115,116,114,101,97,109,0,117,110,115,112,101,99,105,102,105,101,100,32,105,111,115,116,114,101,97,109,95,99,97,116,101,103,111,114,121,32,101,114,114,111,114,0,48,49,50,51,52,53,54,55,56,57,97,98,99,100,101,102,65,66,67,68,69,70,120,88,43,45,112,80,105,73,110,78,0,78,83,116,51,95,95,49,54,108,111,99,97,108,101,53,102,97,99,101,116,69,0,78,83,116,51,95,95,49,53,99,116,121,112,101,73,119,69,69,0,78,83,116,51,95,95,49,55,99,111,100,101,99,118,116,73,99,99,49,49,95,95,109,98,115,116,97,116,101,95,116,69,69,0,78,83,116,51,95,95,49,55,99,111,100,101,99,118,116,73,68,115,99,49,49,95,95,109,98,115,116,97,116,101,95,116,69,69,0,78,83,116,51,95,95,49,55,99,111,100,101,99,118,116,73,68,105,99,49,49,95,95,109,98,115,116,97,116,101,95,116,69,69,0,78,83,116,51,95,95,49,49,54,95,95,110,97,114,114,111,119,95,116,111,95,117,116,102,56,73,76,106,51,50,69,69,69,0,78,83,116,51,95,95,49,49,55,95,95,119,105,100,101,110,95,102,114,111,109,95,117,116,102,56,73,76,106,51,50,69,69,69,0,78,83,116,51,95,95,49,55,99,111,100,101,99,118,116,73,119,99,49,49,95,95,109,98,115,116,97,116,101,95,116,69,69,0,78,83,116,51,95,95,49,54,108,111,99,97,108,101,53,95,95,105,109,112,69,0,78,83,116,51,95,95,49,55,99,111,108,108,97,116,101,73,99,69,69,0,78,83,116,51,95,95,49,55,99,111,108,108,97,116,101,73,119,69,69,0,78,83,116,51,95,95,49,53,99,116,121,112,101,73,99,69,69,0,78,83,116,51,95,95,49,56,110,117,109,112,117,110,99,116,73,99,69,69,0,78,83,116,51,95,95,49,56,110,117,109,112,117,110,99,116,73,119,69,69,0,78,83,116,51,95,95,49,55,110,117,109,95,103,101,116,73,99,78,83,95,49,57,105,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,69,69,69,69,0,78,83,116,51,95,95,49,55,110,117,109,95,103,101,116,73,119,78,83,95,49,57,105,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,119,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,119,69,69,69,69,69,69,0,78,83,116,51,95,95,49,55,110,117,109,95,112,117,116,73,99,78,83,95,49,57,111,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,69,69,69,69,0,78,83,116,51,95,95,49,55,110,117,109,95,112,117,116,73,119,78,83,95,49,57,111,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,119,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,119,69,69,69,69,69,69,0,78,83,116,51,95,95,49,56,116,105,109,101,95,103,101,116,73,99,78,83,95,49,57,105,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,69,69,69,69,0,78,83,116,51,95,95,49,56,116,105,109,101,95,103,101,116,73,119,78,83,95,49,57,105,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,119,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,119,69,69,69,69,69,69,0,78,83,116,51,95,95,49,56,116,105,109,101,95,112,117,116,73,99,78,83,95,49,57,111,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,69,69,69,69,0,78,83,116,51,95,95,49,56,116,105,109,101,95,112,117,116,73,119,78,83,95,49,57,111,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,119,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,119,69,69,69,69,69,69,0,78,83,116,51,95,95,49,49,48,109,111,110,101,121,112,117,110,99,116,73,99,76,98,48,69,69,69,0,78,83,116,51,95,95,49,49,48,109,111,110,101,121,112,117,110,99,116,73,99,76,98,49,69,69,69,0,78,83,116,51,95,95,49,49,48,109,111,110,101,121,112,117,110,99,116,73,119,76,98,48,69,69,69,0,78,83,116,51,95,95,49,49,48,109,111,110,101,121,112,117,110,99,116,73,119,76,98,49,69,69,69,0,78,83,116,51,95,95,49,57,109,111,110,101,121,95,103,101,116,73,99,78,83,95,49,57,105,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,69,69,69,69,0,78,83,116,51,95,95,49,57,109,111,110,101,121,95,103,101,116,73,119,78,83,95,49,57,105,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,119,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,119,69,69,69,69,69,69,0,78,83,116,51,95,95,49,57,109,111,110,101,121,95,112,117,116,73,99,78,83,95,49,57,111,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,99,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,99,69,69,69,69,69,69,0,78,83,116,51,95,95,49,57,109,111,110,101,121,95,112,117,116,73,119,78,83,95,49,57,111,115,116,114,101,97,109,98,117,102,95,105,116,101,114,97,116,111,114,73,119,78,83,95,49,49,99,104,97,114,95,116,114,97,105,116,115,73,119,69,69,69,69,69,69,0,78,83,116,51,95,95,49,56,109,101,115,115,97,103,101,115,73,99,69,69,0,78,83,116,51,95,95,49,56,109,101,115,115,97,103,101,115,73,119,69,69,0,37,112,0,67,0,37,0,0,0,0,0,108,0,108,108,0,0,76,0,37,112,0,0,0,0,37,72,58,37,77,58,37,83,37,109,47,37,100,47,37,121,37,89,45,37,109,45,37,100,37,73,58,37,77,58,37,83,32,37,112,37,72,58,37,77,37,72,58,37,77,58,37,83,108,111,99,97,108,101,32,110,111,116,32,115,117,112,112,111,114,116,101,100,0,48,49,50,51,52,53,54,55,56,57,0,37,76,102,0,109,111,110,101,121,95,103,101,116,32,101,114,114,111,114,0,48,49,50,51,52,53,54,55,56,57,0,37,46,48,76,102,0,116,114,117,101,0,102,97,108,115,101,0,83,117,110,100,97,121,0,77,111,110,100,97,121,0,84,117,101,115,100,97,121,0,87,101,100,110,101,115,100,97,121,0,84,104,117,114,115,100,97,121,0,70,114,105,100,97,121,0,83,97,116,117,114,100,97,121,0,83,117,110,0,77,111,110,0,84,117,101,0,87,101,100,0,84,104,117,0,70,114,105,0,83,97,116,0,74,97,110,117,97,114,121,0,70,101,98,114,117,97,114,121,0,77,97,114,99,104,0,65,112,114,105,108,0,77,97,121,0,74,117,110,101,0,74,117,108,121,0,65,117,103,117,115,116,0,83,101,112,116,101,109,98,101,114,0,79,99,116,111,98,101,114,0,78,111,118,101,109,98,101,114,0,68,101,99,101,109,98,101,114,0,74,97,110,0,70,101,98,0,77,97,114,0,65,112,114,0,74,117,110,0,74,117,108,0,65,117,103,0,83,101,112,0,79,99,116,0,78,111,118,0,68,101,99,0,65,77,0,80,77,0,37,109,47,37,100,47,37,121,0,37,72,58,37,77,58,37,83,0,37,97,32,37,98,32,37,100,32,37,72,58,37,77,58,37,83,32,37,89,0,37,73,58,37,77,58,37,83,32,37,112,0,78,83,116,51,95,95,49,49,51,109,101,115,115,97,103,101,115,95,98,97,115,101,69,0,78,83,116,51,95,95,49,49,49,95,95,109,111,110,101,121,95,112,117,116,73,119,69,69,0,78,83,116,51,95,95,49,49,49,95,95,109,111,110,101,121,95,112,117,116,73,99,69,69,0,78,83,116,51,95,95,49,49,49,95,95,109,111,110,101,121,95,103,101,116,73,119,69,69,0,78,83,116,51,95,95,49,49,49,95,95,109,111,110,101,121,95,103,101,116,73,99,69,69,0,78,83,116,51,95,95,49,49,48,109,111,110,101,121,95,98,97,115,101,69,0,78,83,116,51,95,95,49,49,48,95,95,116,105,109,101,95,112,117,116,69,0,78,83,116,51,95,95,49,50,48,95,95,116,105,109,101,95,103,101,116,95,99,95,115,116,111,114,97,103,101,73,119,69,69,0,78,83,116,51,95,95,49,50,48,95,95,116,105,109,101,95,103,101,116,95,99,95,115,116,111,114,97,103,101,73,99,69,69,0,78,83,116,51,95,95,49,57,116,105,109,101,95,98,97,115,101,69,0,78,83,116,51,95,95,49,57,95,95,110,117,109,95,112,117,116,73,119,69,69,0,78,83,116,51,95,95,49,49,52,95,95,110,117,109,95,112,117,116,95,98,97,115,101,69,0,78,83,116,51,95,95,49,57,95,95,110,117,109,95,112,117,116,73,99,69,69,0,78,83,116,51,95,95,49,57,95,95,110,117,109,95,103,101,116,73,119,69,69,0,78,83,116,51,95,95,49,49,52,95,95,110,117,109,95,103,101,116,95,98,97,115,101,69,0,78,83,116,51,95,95,49,57,95,95,110,117,109,95,103,101,116,73,99,69,69,0,78,83,116,51,95,95,49,49,50,99,111,100,101,99,118,116,95,98,97,115,101,69,0,78,83,116,51,95,95,49,49,48,99,116,121,112,101,95,98,97,115,101,69,0,78,83,116,51,95,95,49,49,52,95,95,115,104,97,114,101,100,95,99,111,117,110,116,69,0,78,83,116,51,95,95,49,49,50,115,121,115,116,101,109,95,101,114,114,111,114,69,0,78,83,116,51,95,95,49,49,52,101,114,114,111,114,95,99,97,116,101,103,111,114,121,69,0,78,83,116,51,95,95,49,49,50,95,95,100,111,95,109,101,115,115,97,103,101,69,0,58,32,0],"i8",ALLOC_NONE,Runtime.GLOBAL_BASE+146719);var tempDoublePtr=Runtime.alignMemory(allocate(12,"i8",ALLOC_STATIC),8);assert(tempDoublePtr%8==0);function copyTempFloat(ptr){HEAP8[tempDoublePtr]=HEAP8[ptr];HEAP8[tempDoublePtr+1]=HEAP8[ptr+1];HEAP8[tempDoublePtr+2]=HEAP8[ptr+2];HEAP8[tempDoublePtr+3]=HEAP8[ptr+3]}function copyTempDouble(ptr){HEAP8[tempDoublePtr]=HEAP8[ptr];HEAP8[tempDoublePtr+1]=HEAP8[ptr+1];HEAP8[tempDoublePtr+2]=HEAP8[ptr+2];HEAP8[tempDoublePtr+3]=HEAP8[ptr+3];HEAP8[tempDoublePtr+4]=HEAP8[ptr+4];HEAP8[tempDoublePtr+5]=HEAP8[ptr+5];HEAP8[tempDoublePtr+6]=HEAP8[ptr+6];HEAP8[tempDoublePtr+7]=HEAP8[ptr+7]}function _atexit(func,arg){__ATEXIT__.unshift({func:func,arg:arg})}function ___cxa_atexit(){return _atexit.apply(null,arguments)}Module["_i64Add"]=_i64Add;Module["_i64Subtract"]=_i64Subtract;function ___setErrNo(value){if(Module["___errno_location"])HEAP32[Module["___errno_location"]()>>2]=value;return value}var ERRNO_CODES={EPERM:1,ENOENT:2,ESRCH:3,EINTR:4,EIO:5,ENXIO:6,E2BIG:7,ENOEXEC:8,EBADF:9,ECHILD:10,EAGAIN:11,EWOULDBLOCK:11,ENOMEM:12,EACCES:13,EFAULT:14,ENOTBLK:15,EBUSY:16,EEXIST:17,EXDEV:18,ENODEV:19,ENOTDIR:20,EISDIR:21,EINVAL:22,ENFILE:23,EMFILE:24,ENOTTY:25,ETXTBSY:26,EFBIG:27,ENOSPC:28,ESPIPE:29,EROFS:30,EMLINK:31,EPIPE:32,EDOM:33,ERANGE:34,ENOMSG:42,EIDRM:43,ECHRNG:44,EL2NSYNC:45,EL3HLT:46,EL3RST:47,ELNRNG:48,EUNATCH:49,ENOCSI:50,EL2HLT:51,EDEADLK:35,ENOLCK:37,EBADE:52,EBADR:53,EXFULL:54,ENOANO:55,EBADRQC:56,EBADSLT:57,EDEADLOCK:35,EBFONT:59,ENOSTR:60,ENODATA:61,ETIME:62,ENOSR:63,ENONET:64,ENOPKG:65,EREMOTE:66,ENOLINK:67,EADV:68,ESRMNT:69,ECOMM:70,EPROTO:71,EMULTIHOP:72,EDOTDOT:73,EBADMSG:74,ENOTUNIQ:76,EBADFD:77,EREMCHG:78,ELIBACC:79,ELIBBAD:80,ELIBSCN:81,ELIBMAX:82,ELIBEXEC:83,ENOSYS:38,ENOTEMPTY:39,ENAMETOOLONG:36,ELOOP:40,EOPNOTSUPP:95,EPFNOSUPPORT:96,ECONNRESET:104,ENOBUFS:105,EAFNOSUPPORT:97,EPROTOTYPE:91,ENOTSOCK:88,ENOPROTOOPT:92,ESHUTDOWN:108,ECONNREFUSED:111,EADDRINUSE:98,ECONNABORTED:103,ENETUNREACH:101,ENETDOWN:100,ETIMEDOUT:110,EHOSTDOWN:112,EHOSTUNREACH:113,EINPROGRESS:115,EALREADY:114,EDESTADDRREQ:89,EMSGSIZE:90,EPROTONOSUPPORT:93,ESOCKTNOSUPPORT:94,EADDRNOTAVAIL:99,ENETRESET:102,EISCONN:106,ENOTCONN:107,ETOOMANYREFS:109,EUSERS:87,EDQUOT:122,ESTALE:116,ENOTSUP:95,ENOMEDIUM:123,EILSEQ:84,EOVERFLOW:75,ECANCELED:125,ENOTRECOVERABLE:131,EOWNERDEAD:130,ESTRPIPE:86};function _sysconf(name){switch(name){case 30:return PAGE_SIZE;case 85:return totalMemory/PAGE_SIZE;case 132:case 133:case 12:case 137:case 138:case 15:case 235:case 16:case 17:case 18:case 19:case 20:case 149:case 13:case 10:case 236:case 153:case 9:case 21:case 22:case 159:case 154:case 14:case 77:case 78:case 139:case 80:case 81:case 82:case 68:case 67:case 164:case 11:case 29:case 47:case 48:case 95:case 52:case 51:case 46:return 200809;case 79:return 0;case 27:case 246:case 127:case 128:case 23:case 24:case 160:case 161:case 181:case 182:case 242:case 183:case 184:case 243:case 244:case 245:case 165:case 178:case 179:case 49:case 50:case 168:case 169:case 175:case 170:case 171:case 172:case 97:case 76:case 32:case 173:case 35:return-1;case 176:case 177:case 7:case 155:case 8:case 157:case 125:case 126:case 92:case 93:case 129:case 130:case 131:case 94:case 91:return 1;case 74:case 60:case 69:case 70:case 4:return 1024;case 31:case 42:case 72:return 32;case 87:case 26:case 33:return 2147483647;case 34:case 1:return 47839;case 38:case 36:return 99;case 43:case 37:return 2048;case 0:return 2097152;case 3:return 65536;case 28:return 32768;case 44:return 32767;case 75:return 16384;case 39:return 1e3;case 89:return 700;case 71:return 256;case 40:return 255;case 2:return 100;case 180:return 64;case 25:return 20;case 5:return 16;case 6:return 6;case 73:return 4;case 84:{if(typeof navigator==="object")return navigator["hardwareConcurrency"]||1;return 1}}___setErrNo(ERRNO_CODES.EINVAL);return-1}function __ZSt18uncaught_exceptionv(){return!!__ZSt18uncaught_exceptionv.uncaught_exception}var EXCEPTIONS={last:0,caught:[],infos:{},deAdjust:(function(adjusted){if(!adjusted||EXCEPTIONS.infos[adjusted])return adjusted;for(var ptr in EXCEPTIONS.infos){var info=EXCEPTIONS.infos[ptr];if(info.adjusted===adjusted){return ptr}}return adjusted}),addRef:(function(ptr){if(!ptr)return;var info=EXCEPTIONS.infos[ptr];info.refcount++}),decRef:(function(ptr){if(!ptr)return;var info=EXCEPTIONS.infos[ptr];assert(info.refcount>0);info.refcount--;if(info.refcount===0){if(info.destructor){Runtime.dynCall("vi",info.destructor,[ptr])}delete EXCEPTIONS.infos[ptr];___cxa_free_exception(ptr)}}),clearRef:(function(ptr){if(!ptr)return;var info=EXCEPTIONS.infos[ptr];info.refcount=0})};function ___resumeException(ptr){if(!EXCEPTIONS.last){EXCEPTIONS.last=ptr}EXCEPTIONS.clearRef(EXCEPTIONS.deAdjust(ptr));throw ptr}function ___cxa_find_matching_catch(){var thrown=EXCEPTIONS.last;if(!thrown){return(asm["setTempRet0"](0),0)|0}var info=EXCEPTIONS.infos[thrown];var throwntype=info.type;if(!throwntype){return(asm["setTempRet0"](0),thrown)|0}var typeArray=Array.prototype.slice.call(arguments);var pointer=Module["___cxa_is_pointer_type"](throwntype);if(!___cxa_find_matching_catch.buffer)___cxa_find_matching_catch.buffer=_malloc(4);HEAP32[___cxa_find_matching_catch.buffer>>2]=thrown;thrown=___cxa_find_matching_catch.buffer;for(var i=0;i<typeArray.length;i++){if(typeArray[i]&&Module["___cxa_can_catch"](typeArray[i],throwntype,thrown)){thrown=HEAP32[thrown>>2];info.adjusted=thrown;return(asm["setTempRet0"](typeArray[i]),thrown)|0}}thrown=HEAP32[thrown>>2];return(asm["setTempRet0"](throwntype),thrown)|0}function ___cxa_throw(ptr,type,destructor){EXCEPTIONS.infos[ptr]={ptr:ptr,adjusted:ptr,type:type,destructor:destructor,refcount:0};EXCEPTIONS.last=ptr;if(!("uncaught_exception"in __ZSt18uncaught_exceptionv)){__ZSt18uncaught_exceptionv.uncaught_exception=1}else{__ZSt18uncaught_exceptionv.uncaught_exception++}throw ptr}var _emscripten_resume=true;var _emscripten_landingpad=true;function _free(){}Module["_free"]=_free;function ___cxa_free_exception(ptr){try{return _free(ptr)}catch(e){}}function ___cxa_end_catch(){if(___cxa_end_catch.rethrown){___cxa_end_catch.rethrown=false;return}asm["setThrew"](0);var ptr=EXCEPTIONS.caught.pop();if(ptr){EXCEPTIONS.decRef(EXCEPTIONS.deAdjust(ptr));EXCEPTIONS.last=0}}function _what0_(){Module["printErr"]("missing function: what0_");abort(-1)}var _log=Math_log;var _emscripten_postinvoke=true;var PTHREAD_SPECIFIC={};var PTHREAD_SPECIFIC_NEXT_KEY=1;function _pthread_key_create(key,destructor){if(key==0){return ERRNO_CODES.EINVAL}HEAP32[key>>2]=PTHREAD_SPECIFIC_NEXT_KEY;PTHREAD_SPECIFIC[PTHREAD_SPECIFIC_NEXT_KEY]=0;PTHREAD_SPECIFIC_NEXT_KEY++;return 0}var ERRNO_MESSAGES={0:"Success",1:"Not super-user",2:"No such file or directory",3:"No such process",4:"Interrupted system call",5:"I/O error",6:"No such device or address",7:"Arg list too long",8:"Exec format error",9:"Bad file number",10:"No children",11:"No more processes",12:"Not enough core",13:"Permission denied",14:"Bad address",15:"Block device required",16:"Mount device busy",17:"File exists",18:"Cross-device link",19:"No such device",20:"Not a directory",21:"Is a directory",22:"Invalid argument",23:"Too many open files in system",24:"Too many open files",25:"Not a typewriter",26:"Text file busy",27:"File too large",28:"No space left on device",29:"Illegal seek",30:"Read only file system",31:"Too many links",32:"Broken pipe",33:"Math arg out of domain of func",34:"Math result not representable",35:"File locking deadlock error",36:"File or path name too long",37:"No record locks available",38:"Function not implemented",39:"Directory not empty",40:"Too many symbolic links",42:"No message of desired type",43:"Identifier removed",44:"Channel number out of range",45:"Level 2 not synchronized",46:"Level 3 halted",47:"Level 3 reset",48:"Link number out of range",49:"Protocol driver not attached",50:"No CSI structure available",51:"Level 2 halted",52:"Invalid exchange",53:"Invalid request descriptor",54:"Exchange full",55:"No anode",56:"Invalid request code",57:"Invalid slot",59:"Bad font file fmt",60:"Device not a stream",61:"No data (for no delay io)",62:"Timer expired",63:"Out of streams resources",64:"Machine is not on the network",65:"Package not installed",66:"The object is remote",67:"The link has been severed",68:"Advertise error",69:"Srmount error",70:"Communication error on send",71:"Protocol error",72:"Multihop attempted",73:"Cross mount point (not really error)",74:"Trying to read unreadable message",75:"Value too large for defined data type",76:"Given log. name not unique",77:"f.d. invalid for this operation",78:"Remote address changed",79:"Can   access a needed shared lib",80:"Accessing a corrupted shared lib",81:".lib section in a.out corrupted",82:"Attempting to link in too many libs",83:"Attempting to exec a shared library",84:"Illegal byte sequence",86:"Streams pipe error",87:"Too many users",88:"Socket operation on non-socket",89:"Destination address required",90:"Message too long",91:"Protocol wrong type for socket",92:"Protocol not available",93:"Unknown protocol",94:"Socket type not supported",95:"Not supported",96:"Protocol family not supported",97:"Address family not supported by protocol family",98:"Address already in use",99:"Address not available",100:"Network interface is not configured",101:"Network is unreachable",102:"Connection reset by network",103:"Connection aborted",104:"Connection reset by peer",105:"No buffer space available",106:"Socket is already connected",107:"Socket is not connected",108:"Can't send after socket shutdown",109:"Too many references",110:"Connection timed out",111:"Connection refused",112:"Host is down",113:"Host is unreachable",114:"Socket already connected",115:"Connection already in progress",116:"Stale file handle",122:"Quota exceeded",123:"No medium (in tape drive)",125:"Operation canceled",130:"Previous owner died",131:"State not recoverable"};var TTY={ttys:[],init:(function(){}),shutdown:(function(){}),register:(function(dev,ops){TTY.ttys[dev]={input:[],output:[],ops:ops};FS.registerDevice(dev,TTY.stream_ops)}),stream_ops:{open:(function(stream){var tty=TTY.ttys[stream.node.rdev];if(!tty){throw new FS.ErrnoError(ERRNO_CODES.ENODEV)}stream.tty=tty;stream.seekable=false}),close:(function(stream){stream.tty.ops.flush(stream.tty)}),flush:(function(stream){stream.tty.ops.flush(stream.tty)}),read:(function(stream,buffer,offset,length,pos){if(!stream.tty||!stream.tty.ops.get_char){throw new FS.ErrnoError(ERRNO_CODES.ENXIO)}var bytesRead=0;for(var i=0;i<length;i++){var result;try{result=stream.tty.ops.get_char(stream.tty)}catch(e){throw new FS.ErrnoError(ERRNO_CODES.EIO)}if(result===undefined&&bytesRead===0){throw new FS.ErrnoError(ERRNO_CODES.EAGAIN)}if(result===null||result===undefined)break;bytesRead++;buffer[offset+i]=result}if(bytesRead){stream.node.timestamp=Date.now()}return bytesRead}),write:(function(stream,buffer,offset,length,pos){if(!stream.tty||!stream.tty.ops.put_char){throw new FS.ErrnoError(ERRNO_CODES.ENXIO)}for(var i=0;i<length;i++){try{stream.tty.ops.put_char(stream.tty,buffer[offset+i])}catch(e){throw new FS.ErrnoError(ERRNO_CODES.EIO)}}if(length){stream.node.timestamp=Date.now()}return i})},default_tty_ops:{get_char:(function(tty){if(!tty.input.length){var result=null;if(ENVIRONMENT_IS_NODE){var BUFSIZE=256;var buf=new Buffer(BUFSIZE);var bytesRead=0;var fd=process.stdin.fd;var usingDevice=false;try{fd=fs.openSync("/dev/stdin","r");usingDevice=true}catch(e){}bytesRead=fs.readSync(fd,buf,0,BUFSIZE,null);if(usingDevice){fs.closeSync(fd)}if(bytesRead>0){result=buf.slice(0,bytesRead).toString("utf-8")}else{result=null}}else if(typeof window!="undefined"&&typeof window.prompt=="function"){result=window.prompt("Input: ");if(result!==null){result+="\n"}}else if(typeof readline=="function"){result=readline();if(result!==null){result+="\n"}}if(!result){return null}tty.input=intArrayFromString(result,true)}return tty.input.shift()}),put_char:(function(tty,val){if(val===null||val===10){Module["print"](UTF8ArrayToString(tty.output,0));tty.output=[]}else{if(val!=0)tty.output.push(val)}}),flush:(function(tty){if(tty.output&&tty.output.length>0){Module["print"](UTF8ArrayToString(tty.output,0));tty.output=[]}})},default_tty1_ops:{put_char:(function(tty,val){if(val===null||val===10){Module["printErr"](UTF8ArrayToString(tty.output,0));tty.output=[]}else{if(val!=0)tty.output.push(val)}}),flush:(function(tty){if(tty.output&&tty.output.length>0){Module["printErr"](UTF8ArrayToString(tty.output,0));tty.output=[]}})}};var MEMFS={ops_table:null,mount:(function(mount){return MEMFS.createNode(null,"/",16384|511,0)}),createNode:(function(parent,name,mode,dev){if(FS.isBlkdev(mode)||FS.isFIFO(mode)){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}if(!MEMFS.ops_table){MEMFS.ops_table={dir:{node:{getattr:MEMFS.node_ops.getattr,setattr:MEMFS.node_ops.setattr,lookup:MEMFS.node_ops.lookup,mknod:MEMFS.node_ops.mknod,rename:MEMFS.node_ops.rename,unlink:MEMFS.node_ops.unlink,rmdir:MEMFS.node_ops.rmdir,readdir:MEMFS.node_ops.readdir,symlink:MEMFS.node_ops.symlink},stream:{llseek:MEMFS.stream_ops.llseek}},file:{node:{getattr:MEMFS.node_ops.getattr,setattr:MEMFS.node_ops.setattr},stream:{llseek:MEMFS.stream_ops.llseek,read:MEMFS.stream_ops.read,write:MEMFS.stream_ops.write,allocate:MEMFS.stream_ops.allocate,mmap:MEMFS.stream_ops.mmap,msync:MEMFS.stream_ops.msync}},link:{node:{getattr:MEMFS.node_ops.getattr,setattr:MEMFS.node_ops.setattr,readlink:MEMFS.node_ops.readlink},stream:{}},chrdev:{node:{getattr:MEMFS.node_ops.getattr,setattr:MEMFS.node_ops.setattr},stream:FS.chrdev_stream_ops}}}var node=FS.createNode(parent,name,mode,dev);if(FS.isDir(node.mode)){node.node_ops=MEMFS.ops_table.dir.node;node.stream_ops=MEMFS.ops_table.dir.stream;node.contents={}}else if(FS.isFile(node.mode)){node.node_ops=MEMFS.ops_table.file.node;node.stream_ops=MEMFS.ops_table.file.stream;node.usedBytes=0;node.contents=null}else if(FS.isLink(node.mode)){node.node_ops=MEMFS.ops_table.link.node;node.stream_ops=MEMFS.ops_table.link.stream}else if(FS.isChrdev(node.mode)){node.node_ops=MEMFS.ops_table.chrdev.node;node.stream_ops=MEMFS.ops_table.chrdev.stream}node.timestamp=Date.now();if(parent){parent.contents[name]=node}return node}),getFileDataAsRegularArray:(function(node){if(node.contents&&node.contents.subarray){var arr=[];for(var i=0;i<node.usedBytes;++i)arr.push(node.contents[i]);return arr}return node.contents}),getFileDataAsTypedArray:(function(node){if(!node.contents)return new Uint8Array;if(node.contents.subarray)return node.contents.subarray(0,node.usedBytes);return new Uint8Array(node.contents)}),expandFileStorage:(function(node,newCapacity){if(node.contents&&node.contents.subarray&&newCapacity>node.contents.length){node.contents=MEMFS.getFileDataAsRegularArray(node);node.usedBytes=node.contents.length}if(!node.contents||node.contents.subarray){var prevCapacity=node.contents?node.contents.buffer.byteLength:0;if(prevCapacity>=newCapacity)return;var CAPACITY_DOUBLING_MAX=1024*1024;newCapacity=Math.max(newCapacity,prevCapacity*(prevCapacity<CAPACITY_DOUBLING_MAX?2:1.125)|0);if(prevCapacity!=0)newCapacity=Math.max(newCapacity,256);var oldContents=node.contents;node.contents=new Uint8Array(newCapacity);if(node.usedBytes>0)node.contents.set(oldContents.subarray(0,node.usedBytes),0);return}if(!node.contents&&newCapacity>0)node.contents=[];while(node.contents.length<newCapacity)node.contents.push(0)}),resizeFileStorage:(function(node,newSize){if(node.usedBytes==newSize)return;if(newSize==0){node.contents=null;node.usedBytes=0;return}if(!node.contents||node.contents.subarray){var oldContents=node.contents;node.contents=new Uint8Array(new ArrayBuffer(newSize));if(oldContents){node.contents.set(oldContents.subarray(0,Math.min(newSize,node.usedBytes)))}node.usedBytes=newSize;return}if(!node.contents)node.contents=[];if(node.contents.length>newSize)node.contents.length=newSize;else while(node.contents.length<newSize)node.contents.push(0);node.usedBytes=newSize}),node_ops:{getattr:(function(node){var attr={};attr.dev=FS.isChrdev(node.mode)?node.id:1;attr.ino=node.id;attr.mode=node.mode;attr.nlink=1;attr.uid=0;attr.gid=0;attr.rdev=node.rdev;if(FS.isDir(node.mode)){attr.size=4096}else if(FS.isFile(node.mode)){attr.size=node.usedBytes}else if(FS.isLink(node.mode)){attr.size=node.link.length}else{attr.size=0}attr.atime=new Date(node.timestamp);attr.mtime=new Date(node.timestamp);attr.ctime=new Date(node.timestamp);attr.blksize=4096;attr.blocks=Math.ceil(attr.size/attr.blksize);return attr}),setattr:(function(node,attr){if(attr.mode!==undefined){node.mode=attr.mode}if(attr.timestamp!==undefined){node.timestamp=attr.timestamp}if(attr.size!==undefined){MEMFS.resizeFileStorage(node,attr.size)}}),lookup:(function(parent,name){throw FS.genericErrors[ERRNO_CODES.ENOENT]}),mknod:(function(parent,name,mode,dev){return MEMFS.createNode(parent,name,mode,dev)}),rename:(function(old_node,new_dir,new_name){if(FS.isDir(old_node.mode)){var new_node;try{new_node=FS.lookupNode(new_dir,new_name)}catch(e){}if(new_node){for(var i in new_node.contents){throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY)}}}delete old_node.parent.contents[old_node.name];old_node.name=new_name;new_dir.contents[new_name]=old_node;old_node.parent=new_dir}),unlink:(function(parent,name){delete parent.contents[name]}),rmdir:(function(parent,name){var node=FS.lookupNode(parent,name);for(var i in node.contents){throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY)}delete parent.contents[name]}),readdir:(function(node){var entries=[".",".."];for(var key in node.contents){if(!node.contents.hasOwnProperty(key)){continue}entries.push(key)}return entries}),symlink:(function(parent,newname,oldpath){var node=MEMFS.createNode(parent,newname,511|40960,0);node.link=oldpath;return node}),readlink:(function(node){if(!FS.isLink(node.mode)){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}return node.link})},stream_ops:{read:(function(stream,buffer,offset,length,position){var contents=stream.node.contents;if(position>=stream.node.usedBytes)return 0;var size=Math.min(stream.node.usedBytes-position,length);assert(size>=0);if(size>8&&contents.subarray){buffer.set(contents.subarray(position,position+size),offset)}else{for(var i=0;i<size;i++)buffer[offset+i]=contents[position+i]}return size}),write:(function(stream,buffer,offset,length,position,canOwn){if(!length)return 0;var node=stream.node;node.timestamp=Date.now();if(buffer.subarray&&(!node.contents||node.contents.subarray)){if(canOwn){node.contents=buffer.subarray(offset,offset+length);node.usedBytes=length;return length}else if(node.usedBytes===0&&position===0){node.contents=new Uint8Array(buffer.subarray(offset,offset+length));node.usedBytes=length;return length}else if(position+length<=node.usedBytes){node.contents.set(buffer.subarray(offset,offset+length),position);return length}}MEMFS.expandFileStorage(node,position+length);if(node.contents.subarray&&buffer.subarray)node.contents.set(buffer.subarray(offset,offset+length),position);else{for(var i=0;i<length;i++){node.contents[position+i]=buffer[offset+i]}}node.usedBytes=Math.max(node.usedBytes,position+length);return length}),llseek:(function(stream,offset,whence){var position=offset;if(whence===1){position+=stream.position}else if(whence===2){if(FS.isFile(stream.node.mode)){position+=stream.node.usedBytes}}if(position<0){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}return position}),allocate:(function(stream,offset,length){MEMFS.expandFileStorage(stream.node,offset+length);stream.node.usedBytes=Math.max(stream.node.usedBytes,offset+length)}),mmap:(function(stream,buffer,offset,length,position,prot,flags){if(!FS.isFile(stream.node.mode)){throw new FS.ErrnoError(ERRNO_CODES.ENODEV)}var ptr;var allocated;var contents=stream.node.contents;if(!(flags&2)&&(contents.buffer===buffer||contents.buffer===buffer.buffer)){allocated=false;ptr=contents.byteOffset}else{if(position>0||position+length<stream.node.usedBytes){if(contents.subarray){contents=contents.subarray(position,position+length)}else{contents=Array.prototype.slice.call(contents,position,position+length)}}allocated=true;ptr=_malloc(length);if(!ptr){throw new FS.ErrnoError(ERRNO_CODES.ENOMEM)}buffer.set(contents,ptr)}return{ptr:ptr,allocated:allocated}}),msync:(function(stream,buffer,offset,length,mmapFlags){if(!FS.isFile(stream.node.mode)){throw new FS.ErrnoError(ERRNO_CODES.ENODEV)}if(mmapFlags&2){return 0}var bytesWritten=MEMFS.stream_ops.write(stream,buffer,0,length,offset,false);return 0})}};var IDBFS={dbs:{},indexedDB:(function(){if(typeof indexedDB!=="undefined")return indexedDB;var ret=null;if(typeof window==="object")ret=window.indexedDB||window.mozIndexedDB||window.webkitIndexedDB||window.msIndexedDB;assert(ret,"IDBFS used, but indexedDB not supported");return ret}),DB_VERSION:21,DB_STORE_NAME:"FILE_DATA",mount:(function(mount){return MEMFS.mount.apply(null,arguments)}),syncfs:(function(mount,populate,callback){IDBFS.getLocalSet(mount,(function(err,local){if(err)return callback(err);IDBFS.getRemoteSet(mount,(function(err,remote){if(err)return callback(err);var src=populate?remote:local;var dst=populate?local:remote;IDBFS.reconcile(src,dst,callback)}))}))}),getDB:(function(name,callback){var db=IDBFS.dbs[name];if(db){return callback(null,db)}var req;try{req=IDBFS.indexedDB().open(name,IDBFS.DB_VERSION)}catch(e){return callback(e)}req.onupgradeneeded=(function(e){var db=e.target.result;var transaction=e.target.transaction;var fileStore;if(db.objectStoreNames.contains(IDBFS.DB_STORE_NAME)){fileStore=transaction.objectStore(IDBFS.DB_STORE_NAME)}else{fileStore=db.createObjectStore(IDBFS.DB_STORE_NAME)}if(!fileStore.indexNames.contains("timestamp")){fileStore.createIndex("timestamp","timestamp",{unique:false})}});req.onsuccess=(function(){db=req.result;IDBFS.dbs[name]=db;callback(null,db)});req.onerror=(function(e){callback(this.error);e.preventDefault()})}),getLocalSet:(function(mount,callback){var entries={};function isRealDir(p){return p!=="."&&p!==".."}function toAbsolute(root){return(function(p){return PATH.join2(root,p)})}var check=FS.readdir(mount.mountpoint).filter(isRealDir).map(toAbsolute(mount.mountpoint));while(check.length){var path=check.pop();var stat;try{stat=FS.stat(path)}catch(e){return callback(e)}if(FS.isDir(stat.mode)){check.push.apply(check,FS.readdir(path).filter(isRealDir).map(toAbsolute(path)))}entries[path]={timestamp:stat.mtime}}return callback(null,{type:"local",entries:entries})}),getRemoteSet:(function(mount,callback){var entries={};IDBFS.getDB(mount.mountpoint,(function(err,db){if(err)return callback(err);var transaction=db.transaction([IDBFS.DB_STORE_NAME],"readonly");transaction.onerror=(function(e){callback(this.error);e.preventDefault()});var store=transaction.objectStore(IDBFS.DB_STORE_NAME);var index=store.index("timestamp");index.openKeyCursor().onsuccess=(function(event){var cursor=event.target.result;if(!cursor){return callback(null,{type:"remote",db:db,entries:entries})}entries[cursor.primaryKey]={timestamp:cursor.key};cursor.continue()})}))}),loadLocalEntry:(function(path,callback){var stat,node;try{var lookup=FS.lookupPath(path);node=lookup.node;stat=FS.stat(path)}catch(e){return callback(e)}if(FS.isDir(stat.mode)){return callback(null,{timestamp:stat.mtime,mode:stat.mode})}else if(FS.isFile(stat.mode)){node.contents=MEMFS.getFileDataAsTypedArray(node);return callback(null,{timestamp:stat.mtime,mode:stat.mode,contents:node.contents})}else{return callback(new Error("node type not supported"))}}),storeLocalEntry:(function(path,entry,callback){try{if(FS.isDir(entry.mode)){FS.mkdir(path,entry.mode)}else if(FS.isFile(entry.mode)){FS.writeFile(path,entry.contents,{encoding:"binary",canOwn:true})}else{return callback(new Error("node type not supported"))}FS.chmod(path,entry.mode);FS.utime(path,entry.timestamp,entry.timestamp)}catch(e){return callback(e)}callback(null)}),removeLocalEntry:(function(path,callback){try{var lookup=FS.lookupPath(path);var stat=FS.stat(path);if(FS.isDir(stat.mode)){FS.rmdir(path)}else if(FS.isFile(stat.mode)){FS.unlink(path)}}catch(e){return callback(e)}callback(null)}),loadRemoteEntry:(function(store,path,callback){var req=store.get(path);req.onsuccess=(function(event){callback(null,event.target.result)});req.onerror=(function(e){callback(this.error);e.preventDefault()})}),storeRemoteEntry:(function(store,path,entry,callback){var req=store.put(entry,path);req.onsuccess=(function(){callback(null)});req.onerror=(function(e){callback(this.error);e.preventDefault()})}),removeRemoteEntry:(function(store,path,callback){var req=store.delete(path);req.onsuccess=(function(){callback(null)});req.onerror=(function(e){callback(this.error);e.preventDefault()})}),reconcile:(function(src,dst,callback){var total=0;var create=[];Object.keys(src.entries).forEach((function(key){var e=src.entries[key];var e2=dst.entries[key];if(!e2||e.timestamp>e2.timestamp){create.push(key);total++}}));var remove=[];Object.keys(dst.entries).forEach((function(key){var e=dst.entries[key];var e2=src.entries[key];if(!e2){remove.push(key);total++}}));if(!total){return callback(null)}var errored=false;var completed=0;var db=src.type==="remote"?src.db:dst.db;var transaction=db.transaction([IDBFS.DB_STORE_NAME],"readwrite");var store=transaction.objectStore(IDBFS.DB_STORE_NAME);function done(err){if(err){if(!done.errored){done.errored=true;return callback(err)}return}if(++completed>=total){return callback(null)}}transaction.onerror=(function(e){done(this.error);e.preventDefault()});create.sort().forEach((function(path){if(dst.type==="local"){IDBFS.loadRemoteEntry(store,path,(function(err,entry){if(err)return done(err);IDBFS.storeLocalEntry(path,entry,done)}))}else{IDBFS.loadLocalEntry(path,(function(err,entry){if(err)return done(err);IDBFS.storeRemoteEntry(store,path,entry,done)}))}}));remove.sort().reverse().forEach((function(path){if(dst.type==="local"){IDBFS.removeLocalEntry(path,done)}else{IDBFS.removeRemoteEntry(store,path,done)}}))})};var NODEFS={isWindows:false,staticInit:(function(){NODEFS.isWindows=!!process.platform.match(/^win/)}),mount:(function(mount){assert(ENVIRONMENT_IS_NODE);return NODEFS.createNode(null,"/",NODEFS.getMode(mount.opts.root),0)}),createNode:(function(parent,name,mode,dev){if(!FS.isDir(mode)&&!FS.isFile(mode)&&!FS.isLink(mode)){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}var node=FS.createNode(parent,name,mode);node.node_ops=NODEFS.node_ops;node.stream_ops=NODEFS.stream_ops;return node}),getMode:(function(path){var stat;try{stat=fs.lstatSync(path);if(NODEFS.isWindows){stat.mode=stat.mode|(stat.mode&146)>>1}}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}return stat.mode}),realPath:(function(node){var parts=[];while(node.parent!==node){parts.push(node.name);node=node.parent}parts.push(node.mount.opts.root);parts.reverse();return PATH.join.apply(null,parts)}),flagsToPermissionStringMap:{0:"r",1:"r+",2:"r+",64:"r",65:"r+",66:"r+",129:"rx+",193:"rx+",514:"w+",577:"w",578:"w+",705:"wx",706:"wx+",1024:"a",1025:"a",1026:"a+",1089:"a",1090:"a+",1153:"ax",1154:"ax+",1217:"ax",1218:"ax+",4096:"rs",4098:"rs+"},flagsToPermissionString:(function(flags){flags&=~32768;flags&=~524288;if(flags in NODEFS.flagsToPermissionStringMap){return NODEFS.flagsToPermissionStringMap[flags]}else{throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}}),node_ops:{getattr:(function(node){var path=NODEFS.realPath(node);var stat;try{stat=fs.lstatSync(path)}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}if(NODEFS.isWindows&&!stat.blksize){stat.blksize=4096}if(NODEFS.isWindows&&!stat.blocks){stat.blocks=(stat.size+stat.blksize-1)/stat.blksize|0}return{dev:stat.dev,ino:stat.ino,mode:stat.mode,nlink:stat.nlink,uid:stat.uid,gid:stat.gid,rdev:stat.rdev,size:stat.size,atime:stat.atime,mtime:stat.mtime,ctime:stat.ctime,blksize:stat.blksize,blocks:stat.blocks}}),setattr:(function(node,attr){var path=NODEFS.realPath(node);try{if(attr.mode!==undefined){fs.chmodSync(path,attr.mode);node.mode=attr.mode}if(attr.timestamp!==undefined){var date=new Date(attr.timestamp);fs.utimesSync(path,date,date)}if(attr.size!==undefined){fs.truncateSync(path,attr.size)}}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}}),lookup:(function(parent,name){var path=PATH.join2(NODEFS.realPath(parent),name);var mode=NODEFS.getMode(path);return NODEFS.createNode(parent,name,mode)}),mknod:(function(parent,name,mode,dev){var node=NODEFS.createNode(parent,name,mode,dev);var path=NODEFS.realPath(node);try{if(FS.isDir(node.mode)){fs.mkdirSync(path,node.mode)}else{fs.writeFileSync(path,"",{mode:node.mode})}}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}return node}),rename:(function(oldNode,newDir,newName){var oldPath=NODEFS.realPath(oldNode);var newPath=PATH.join2(NODEFS.realPath(newDir),newName);try{fs.renameSync(oldPath,newPath)}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}}),unlink:(function(parent,name){var path=PATH.join2(NODEFS.realPath(parent),name);try{fs.unlinkSync(path)}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}}),rmdir:(function(parent,name){var path=PATH.join2(NODEFS.realPath(parent),name);try{fs.rmdirSync(path)}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}}),readdir:(function(node){var path=NODEFS.realPath(node);try{return fs.readdirSync(path)}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}}),symlink:(function(parent,newName,oldPath){var newPath=PATH.join2(NODEFS.realPath(parent),newName);try{fs.symlinkSync(oldPath,newPath)}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}}),readlink:(function(node){var path=NODEFS.realPath(node);try{path=fs.readlinkSync(path);path=NODEJS_PATH.relative(NODEJS_PATH.resolve(node.mount.opts.root),path);return path}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}})},stream_ops:{open:(function(stream){var path=NODEFS.realPath(stream.node);try{if(FS.isFile(stream.node.mode)){stream.nfd=fs.openSync(path,NODEFS.flagsToPermissionString(stream.flags))}}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}}),close:(function(stream){try{if(FS.isFile(stream.node.mode)&&stream.nfd){fs.closeSync(stream.nfd)}}catch(e){if(!e.code)throw e;throw new FS.ErrnoError(ERRNO_CODES[e.code])}}),read:(function(stream,buffer,offset,length,position){if(length===0)return 0;var nbuffer=new Buffer(length);var res;try{res=fs.readSync(stream.nfd,nbuffer,0,length,position)}catch(e){throw new FS.ErrnoError(ERRNO_CODES[e.code])}if(res>0){for(var i=0;i<res;i++){buffer[offset+i]=nbuffer[i]}}return res}),write:(function(stream,buffer,offset,length,position){var nbuffer=new Buffer(buffer.subarray(offset,offset+length));var res;try{res=fs.writeSync(stream.nfd,nbuffer,0,length,position)}catch(e){throw new FS.ErrnoError(ERRNO_CODES[e.code])}return res}),llseek:(function(stream,offset,whence){var position=offset;if(whence===1){position+=stream.position}else if(whence===2){if(FS.isFile(stream.node.mode)){try{var stat=fs.fstatSync(stream.nfd);position+=stat.size}catch(e){throw new FS.ErrnoError(ERRNO_CODES[e.code])}}}if(position<0){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}return position})}};var WORKERFS={DIR_MODE:16895,FILE_MODE:33279,reader:null,mount:(function(mount){assert(ENVIRONMENT_IS_WORKER);if(!WORKERFS.reader)WORKERFS.reader=new FileReaderSync;var root=WORKERFS.createNode(null,"/",WORKERFS.DIR_MODE,0);var createdParents={};function ensureParent(path){var parts=path.split("/");var parent=root;for(var i=0;i<parts.length-1;i++){var curr=parts.slice(0,i+1).join("/");if(!createdParents[curr]){createdParents[curr]=WORKERFS.createNode(parent,curr,WORKERFS.DIR_MODE,0)}parent=createdParents[curr]}return parent}function base(path){var parts=path.split("/");return parts[parts.length-1]}Array.prototype.forEach.call(mount.opts["files"]||[],(function(file){WORKERFS.createNode(ensureParent(file.name),base(file.name),WORKERFS.FILE_MODE,0,file,file.lastModifiedDate)}));(mount.opts["blobs"]||[]).forEach((function(obj){WORKERFS.createNode(ensureParent(obj["name"]),base(obj["name"]),WORKERFS.FILE_MODE,0,obj["data"])}));(mount.opts["packages"]||[]).forEach((function(pack){pack["metadata"].files.forEach((function(file){var name=file.filename.substr(1);WORKERFS.createNode(ensureParent(name),base(name),WORKERFS.FILE_MODE,0,pack["blob"].slice(file.start,file.end))}))}));return root}),createNode:(function(parent,name,mode,dev,contents,mtime){var node=FS.createNode(parent,name,mode);node.mode=mode;node.node_ops=WORKERFS.node_ops;node.stream_ops=WORKERFS.stream_ops;node.timestamp=(mtime||new Date).getTime();assert(WORKERFS.FILE_MODE!==WORKERFS.DIR_MODE);if(mode===WORKERFS.FILE_MODE){node.size=contents.size;node.contents=contents}else{node.size=4096;node.contents={}}if(parent){parent.contents[name]=node}return node}),node_ops:{getattr:(function(node){return{dev:1,ino:undefined,mode:node.mode,nlink:1,uid:0,gid:0,rdev:undefined,size:node.size,atime:new Date(node.timestamp),mtime:new Date(node.timestamp),ctime:new Date(node.timestamp),blksize:4096,blocks:Math.ceil(node.size/4096)}}),setattr:(function(node,attr){if(attr.mode!==undefined){node.mode=attr.mode}if(attr.timestamp!==undefined){node.timestamp=attr.timestamp}}),lookup:(function(parent,name){throw new FS.ErrnoError(ERRNO_CODES.ENOENT)}),mknod:(function(parent,name,mode,dev){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}),rename:(function(oldNode,newDir,newName){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}),unlink:(function(parent,name){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}),rmdir:(function(parent,name){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}),readdir:(function(node){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}),symlink:(function(parent,newName,oldPath){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}),readlink:(function(node){throw new FS.ErrnoError(ERRNO_CODES.EPERM)})},stream_ops:{read:(function(stream,buffer,offset,length,position){if(position>=stream.node.size)return 0;var chunk=stream.node.contents.slice(position,position+length);var ab=WORKERFS.reader.readAsArrayBuffer(chunk);buffer.set(new Uint8Array(ab),offset);return chunk.size}),write:(function(stream,buffer,offset,length,position){throw new FS.ErrnoError(ERRNO_CODES.EIO)}),llseek:(function(stream,offset,whence){var position=offset;if(whence===1){position+=stream.position}else if(whence===2){if(FS.isFile(stream.node.mode)){position+=stream.node.size}}if(position<0){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}return position})}};var _stdin=allocate(1,"i32*",ALLOC_STATIC);var _stdout=allocate(1,"i32*",ALLOC_STATIC);var _stderr=allocate(1,"i32*",ALLOC_STATIC);var FS={root:null,mounts:[],devices:[null],streams:[],nextInode:1,nameTable:null,currentPath:"/",initialized:false,ignorePermissions:true,trackingDelegate:{},tracking:{openFlags:{READ:1,WRITE:2}},ErrnoError:null,genericErrors:{},filesystems:null,handleFSError:(function(e){if(!(e instanceof FS.ErrnoError))throw e+" : "+stackTrace();return ___setErrNo(e.errno)}),lookupPath:(function(path,opts){path=PATH.resolve(FS.cwd(),path);opts=opts||{};if(!path)return{path:"",node:null};var defaults={follow_mount:true,recurse_count:0};for(var key in defaults){if(opts[key]===undefined){opts[key]=defaults[key]}}if(opts.recurse_count>8){throw new FS.ErrnoError(ERRNO_CODES.ELOOP)}var parts=PATH.normalizeArray(path.split("/").filter((function(p){return!!p})),false);var current=FS.root;var current_path="/";for(var i=0;i<parts.length;i++){var islast=i===parts.length-1;if(islast&&opts.parent){break}current=FS.lookupNode(current,parts[i]);current_path=PATH.join2(current_path,parts[i]);if(FS.isMountpoint(current)){if(!islast||islast&&opts.follow_mount){current=current.mounted.root}}if(!islast||opts.follow){var count=0;while(FS.isLink(current.mode)){var link=FS.readlink(current_path);current_path=PATH.resolve(PATH.dirname(current_path),link);var lookup=FS.lookupPath(current_path,{recurse_count:opts.recurse_count});current=lookup.node;if(count++>40){throw new FS.ErrnoError(ERRNO_CODES.ELOOP)}}}}return{path:current_path,node:current}}),getPath:(function(node){var path;while(true){if(FS.isRoot(node)){var mount=node.mount.mountpoint;if(!path)return mount;return mount[mount.length-1]!=="/"?mount+"/"+path:mount+path}path=path?node.name+"/"+path:node.name;node=node.parent}}),hashName:(function(parentid,name){var hash=0;for(var i=0;i<name.length;i++){hash=(hash<<5)-hash+name.charCodeAt(i)|0}return(parentid+hash>>>0)%FS.nameTable.length}),hashAddNode:(function(node){var hash=FS.hashName(node.parent.id,node.name);node.name_next=FS.nameTable[hash];FS.nameTable[hash]=node}),hashRemoveNode:(function(node){var hash=FS.hashName(node.parent.id,node.name);if(FS.nameTable[hash]===node){FS.nameTable[hash]=node.name_next}else{var current=FS.nameTable[hash];while(current){if(current.name_next===node){current.name_next=node.name_next;break}current=current.name_next}}}),lookupNode:(function(parent,name){var err=FS.mayLookup(parent);if(err){throw new FS.ErrnoError(err,parent)}var hash=FS.hashName(parent.id,name);for(var node=FS.nameTable[hash];node;node=node.name_next){var nodeName=node.name;if(node.parent.id===parent.id&&nodeName===name){return node}}return FS.lookup(parent,name)}),createNode:(function(parent,name,mode,rdev){if(!FS.FSNode){FS.FSNode=(function(parent,name,mode,rdev){if(!parent){parent=this}this.parent=parent;this.mount=parent.mount;this.mounted=null;this.id=FS.nextInode++;this.name=name;this.mode=mode;this.node_ops={};this.stream_ops={};this.rdev=rdev});FS.FSNode.prototype={};var readMode=292|73;var writeMode=146;Object.defineProperties(FS.FSNode.prototype,{read:{get:(function(){return(this.mode&readMode)===readMode}),set:(function(val){val?this.mode|=readMode:this.mode&=~readMode})},write:{get:(function(){return(this.mode&writeMode)===writeMode}),set:(function(val){val?this.mode|=writeMode:this.mode&=~writeMode})},isFolder:{get:(function(){return FS.isDir(this.mode)})},isDevice:{get:(function(){return FS.isChrdev(this.mode)})}})}var node=new FS.FSNode(parent,name,mode,rdev);FS.hashAddNode(node);return node}),destroyNode:(function(node){FS.hashRemoveNode(node)}),isRoot:(function(node){return node===node.parent}),isMountpoint:(function(node){return!!node.mounted}),isFile:(function(mode){return(mode&61440)===32768}),isDir:(function(mode){return(mode&61440)===16384}),isLink:(function(mode){return(mode&61440)===40960}),isChrdev:(function(mode){return(mode&61440)===8192}),isBlkdev:(function(mode){return(mode&61440)===24576}),isFIFO:(function(mode){return(mode&61440)===4096}),isSocket:(function(mode){return(mode&49152)===49152}),flagModes:{"r":0,"rs":1052672,"r+":2,"w":577,"wx":705,"xw":705,"w+":578,"wx+":706,"xw+":706,"a":1089,"ax":1217,"xa":1217,"a+":1090,"ax+":1218,"xa+":1218},modeStringToFlags:(function(str){var flags=FS.flagModes[str];if(typeof flags==="undefined"){throw new Error("Unknown file open mode: "+str)}return flags}),flagsToPermissionString:(function(flag){var perms=["r","w","rw"][flag&3];if(flag&512){perms+="w"}return perms}),nodePermissions:(function(node,perms){if(FS.ignorePermissions){return 0}if(perms.indexOf("r")!==-1&&!(node.mode&292)){return ERRNO_CODES.EACCES}else if(perms.indexOf("w")!==-1&&!(node.mode&146)){return ERRNO_CODES.EACCES}else if(perms.indexOf("x")!==-1&&!(node.mode&73)){return ERRNO_CODES.EACCES}return 0}),mayLookup:(function(dir){var err=FS.nodePermissions(dir,"x");if(err)return err;if(!dir.node_ops.lookup)return ERRNO_CODES.EACCES;return 0}),mayCreate:(function(dir,name){try{var node=FS.lookupNode(dir,name);return ERRNO_CODES.EEXIST}catch(e){}return FS.nodePermissions(dir,"wx")}),mayDelete:(function(dir,name,isdir){var node;try{node=FS.lookupNode(dir,name)}catch(e){return e.errno}var err=FS.nodePermissions(dir,"wx");if(err){return err}if(isdir){if(!FS.isDir(node.mode)){return ERRNO_CODES.ENOTDIR}if(FS.isRoot(node)||FS.getPath(node)===FS.cwd()){return ERRNO_CODES.EBUSY}}else{if(FS.isDir(node.mode)){return ERRNO_CODES.EISDIR}}return 0}),mayOpen:(function(node,flags){if(!node){return ERRNO_CODES.ENOENT}if(FS.isLink(node.mode)){return ERRNO_CODES.ELOOP}else if(FS.isDir(node.mode)){if(FS.flagsToPermissionString(flags)!=="r"||flags&512){return ERRNO_CODES.EISDIR}}return FS.nodePermissions(node,FS.flagsToPermissionString(flags))}),MAX_OPEN_FDS:4096,nextfd:(function(fd_start,fd_end){fd_start=fd_start||0;fd_end=fd_end||FS.MAX_OPEN_FDS;for(var fd=fd_start;fd<=fd_end;fd++){if(!FS.streams[fd]){return fd}}throw new FS.ErrnoError(ERRNO_CODES.EMFILE)}),getStream:(function(fd){return FS.streams[fd]}),createStream:(function(stream,fd_start,fd_end){if(!FS.FSStream){FS.FSStream=(function(){});FS.FSStream.prototype={};Object.defineProperties(FS.FSStream.prototype,{object:{get:(function(){return this.node}),set:(function(val){this.node=val})},isRead:{get:(function(){return(this.flags&2097155)!==1})},isWrite:{get:(function(){return(this.flags&2097155)!==0})},isAppend:{get:(function(){return this.flags&1024})}})}var newStream=new FS.FSStream;for(var p in stream){newStream[p]=stream[p]}stream=newStream;var fd=FS.nextfd(fd_start,fd_end);stream.fd=fd;FS.streams[fd]=stream;return stream}),closeStream:(function(fd){FS.streams[fd]=null}),chrdev_stream_ops:{open:(function(stream){var device=FS.getDevice(stream.node.rdev);stream.stream_ops=device.stream_ops;if(stream.stream_ops.open){stream.stream_ops.open(stream)}}),llseek:(function(){throw new FS.ErrnoError(ERRNO_CODES.ESPIPE)})},major:(function(dev){return dev>>8}),minor:(function(dev){return dev&255}),makedev:(function(ma,mi){return ma<<8|mi}),registerDevice:(function(dev,ops){FS.devices[dev]={stream_ops:ops}}),getDevice:(function(dev){return FS.devices[dev]}),getMounts:(function(mount){var mounts=[];var check=[mount];while(check.length){var m=check.pop();mounts.push(m);check.push.apply(check,m.mounts)}return mounts}),syncfs:(function(populate,callback){if(typeof populate==="function"){callback=populate;populate=false}var mounts=FS.getMounts(FS.root.mount);var completed=0;function done(err){if(err){if(!done.errored){done.errored=true;return callback(err)}return}if(++completed>=mounts.length){callback(null)}}mounts.forEach((function(mount){if(!mount.type.syncfs){return done(null)}mount.type.syncfs(mount,populate,done)}))}),mount:(function(type,opts,mountpoint){var root=mountpoint==="/";var pseudo=!mountpoint;var node;if(root&&FS.root){throw new FS.ErrnoError(ERRNO_CODES.EBUSY)}else if(!root&&!pseudo){var lookup=FS.lookupPath(mountpoint,{follow_mount:false});mountpoint=lookup.path;node=lookup.node;if(FS.isMountpoint(node)){throw new FS.ErrnoError(ERRNO_CODES.EBUSY)}if(!FS.isDir(node.mode)){throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR)}}var mount={type:type,opts:opts,mountpoint:mountpoint,mounts:[]};var mountRoot=type.mount(mount);mountRoot.mount=mount;mount.root=mountRoot;if(root){FS.root=mountRoot}else if(node){node.mounted=mount;if(node.mount){node.mount.mounts.push(mount)}}return mountRoot}),unmount:(function(mountpoint){var lookup=FS.lookupPath(mountpoint,{follow_mount:false});if(!FS.isMountpoint(lookup.node)){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}var node=lookup.node;var mount=node.mounted;var mounts=FS.getMounts(mount);Object.keys(FS.nameTable).forEach((function(hash){var current=FS.nameTable[hash];while(current){var next=current.name_next;if(mounts.indexOf(current.mount)!==-1){FS.destroyNode(current)}current=next}}));node.mounted=null;var idx=node.mount.mounts.indexOf(mount);assert(idx!==-1);node.mount.mounts.splice(idx,1)}),lookup:(function(parent,name){return parent.node_ops.lookup(parent,name)}),mknod:(function(path,mode,dev){var lookup=FS.lookupPath(path,{parent:true});var parent=lookup.node;var name=PATH.basename(path);if(!name||name==="."||name===".."){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}var err=FS.mayCreate(parent,name);if(err){throw new FS.ErrnoError(err)}if(!parent.node_ops.mknod){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}return parent.node_ops.mknod(parent,name,mode,dev)}),create:(function(path,mode){mode=mode!==undefined?mode:438;mode&=4095;mode|=32768;return FS.mknod(path,mode,0)}),mkdir:(function(path,mode){mode=mode!==undefined?mode:511;mode&=511|512;mode|=16384;return FS.mknod(path,mode,0)}),mkdev:(function(path,mode,dev){if(typeof dev==="undefined"){dev=mode;mode=438}mode|=8192;return FS.mknod(path,mode,dev)}),symlink:(function(oldpath,newpath){if(!PATH.resolve(oldpath)){throw new FS.ErrnoError(ERRNO_CODES.ENOENT)}var lookup=FS.lookupPath(newpath,{parent:true});var parent=lookup.node;if(!parent){throw new FS.ErrnoError(ERRNO_CODES.ENOENT)}var newname=PATH.basename(newpath);var err=FS.mayCreate(parent,newname);if(err){throw new FS.ErrnoError(err)}if(!parent.node_ops.symlink){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}return parent.node_ops.symlink(parent,newname,oldpath)}),rename:(function(old_path,new_path){var old_dirname=PATH.dirname(old_path);var new_dirname=PATH.dirname(new_path);var old_name=PATH.basename(old_path);var new_name=PATH.basename(new_path);var lookup,old_dir,new_dir;try{lookup=FS.lookupPath(old_path,{parent:true});old_dir=lookup.node;lookup=FS.lookupPath(new_path,{parent:true});new_dir=lookup.node}catch(e){throw new FS.ErrnoError(ERRNO_CODES.EBUSY)}if(!old_dir||!new_dir)throw new FS.ErrnoError(ERRNO_CODES.ENOENT);if(old_dir.mount!==new_dir.mount){throw new FS.ErrnoError(ERRNO_CODES.EXDEV)}var old_node=FS.lookupNode(old_dir,old_name);var relative=PATH.relative(old_path,new_dirname);if(relative.charAt(0)!=="."){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}relative=PATH.relative(new_path,old_dirname);if(relative.charAt(0)!=="."){throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY)}var new_node;try{new_node=FS.lookupNode(new_dir,new_name)}catch(e){}if(old_node===new_node){return}var isdir=FS.isDir(old_node.mode);var err=FS.mayDelete(old_dir,old_name,isdir);if(err){throw new FS.ErrnoError(err)}err=new_node?FS.mayDelete(new_dir,new_name,isdir):FS.mayCreate(new_dir,new_name);if(err){throw new FS.ErrnoError(err)}if(!old_dir.node_ops.rename){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}if(FS.isMountpoint(old_node)||new_node&&FS.isMountpoint(new_node)){throw new FS.ErrnoError(ERRNO_CODES.EBUSY)}if(new_dir!==old_dir){err=FS.nodePermissions(old_dir,"w");if(err){throw new FS.ErrnoError(err)}}try{if(FS.trackingDelegate["willMovePath"]){FS.trackingDelegate["willMovePath"](old_path,new_path)}}catch(e){console.log("FS.trackingDelegate['willMovePath']('"+old_path+"', '"+new_path+"') threw an exception: "+e.message)}FS.hashRemoveNode(old_node);try{old_dir.node_ops.rename(old_node,new_dir,new_name)}catch(e){throw e}finally{FS.hashAddNode(old_node)}try{if(FS.trackingDelegate["onMovePath"])FS.trackingDelegate["onMovePath"](old_path,new_path)}catch(e){console.log("FS.trackingDelegate['onMovePath']('"+old_path+"', '"+new_path+"') threw an exception: "+e.message)}}),rmdir:(function(path){var lookup=FS.lookupPath(path,{parent:true});var parent=lookup.node;var name=PATH.basename(path);var node=FS.lookupNode(parent,name);var err=FS.mayDelete(parent,name,true);if(err){throw new FS.ErrnoError(err)}if(!parent.node_ops.rmdir){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}if(FS.isMountpoint(node)){throw new FS.ErrnoError(ERRNO_CODES.EBUSY)}try{if(FS.trackingDelegate["willDeletePath"]){FS.trackingDelegate["willDeletePath"](path)}}catch(e){console.log("FS.trackingDelegate['willDeletePath']('"+path+"') threw an exception: "+e.message)}parent.node_ops.rmdir(parent,name);FS.destroyNode(node);try{if(FS.trackingDelegate["onDeletePath"])FS.trackingDelegate["onDeletePath"](path)}catch(e){console.log("FS.trackingDelegate['onDeletePath']('"+path+"') threw an exception: "+e.message)}}),readdir:(function(path){var lookup=FS.lookupPath(path,{follow:true});var node=lookup.node;if(!node.node_ops.readdir){throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR)}return node.node_ops.readdir(node)}),unlink:(function(path){var lookup=FS.lookupPath(path,{parent:true});var parent=lookup.node;var name=PATH.basename(path);var node=FS.lookupNode(parent,name);var err=FS.mayDelete(parent,name,false);if(err){if(err===ERRNO_CODES.EISDIR)err=ERRNO_CODES.EPERM;throw new FS.ErrnoError(err)}if(!parent.node_ops.unlink){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}if(FS.isMountpoint(node)){throw new FS.ErrnoError(ERRNO_CODES.EBUSY)}try{if(FS.trackingDelegate["willDeletePath"]){FS.trackingDelegate["willDeletePath"](path)}}catch(e){console.log("FS.trackingDelegate['willDeletePath']('"+path+"') threw an exception: "+e.message)}parent.node_ops.unlink(parent,name);FS.destroyNode(node);try{if(FS.trackingDelegate["onDeletePath"])FS.trackingDelegate["onDeletePath"](path)}catch(e){console.log("FS.trackingDelegate['onDeletePath']('"+path+"') threw an exception: "+e.message)}}),readlink:(function(path){var lookup=FS.lookupPath(path);var link=lookup.node;if(!link){throw new FS.ErrnoError(ERRNO_CODES.ENOENT)}if(!link.node_ops.readlink){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}return PATH.resolve(FS.getPath(link.parent),link.node_ops.readlink(link))}),stat:(function(path,dontFollow){var lookup=FS.lookupPath(path,{follow:!dontFollow});var node=lookup.node;if(!node){throw new FS.ErrnoError(ERRNO_CODES.ENOENT)}if(!node.node_ops.getattr){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}return node.node_ops.getattr(node)}),lstat:(function(path){return FS.stat(path,true)}),chmod:(function(path,mode,dontFollow){var node;if(typeof path==="string"){var lookup=FS.lookupPath(path,{follow:!dontFollow});node=lookup.node}else{node=path}if(!node.node_ops.setattr){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}node.node_ops.setattr(node,{mode:mode&4095|node.mode&~4095,timestamp:Date.now()})}),lchmod:(function(path,mode){FS.chmod(path,mode,true)}),fchmod:(function(fd,mode){var stream=FS.getStream(fd);if(!stream){throw new FS.ErrnoError(ERRNO_CODES.EBADF)}FS.chmod(stream.node,mode)}),chown:(function(path,uid,gid,dontFollow){var node;if(typeof path==="string"){var lookup=FS.lookupPath(path,{follow:!dontFollow});node=lookup.node}else{node=path}if(!node.node_ops.setattr){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}node.node_ops.setattr(node,{timestamp:Date.now()})}),lchown:(function(path,uid,gid){FS.chown(path,uid,gid,true)}),fchown:(function(fd,uid,gid){var stream=FS.getStream(fd);if(!stream){throw new FS.ErrnoError(ERRNO_CODES.EBADF)}FS.chown(stream.node,uid,gid)}),truncate:(function(path,len){if(len<0){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}var node;if(typeof path==="string"){var lookup=FS.lookupPath(path,{follow:true});node=lookup.node}else{node=path}if(!node.node_ops.setattr){throw new FS.ErrnoError(ERRNO_CODES.EPERM)}if(FS.isDir(node.mode)){throw new FS.ErrnoError(ERRNO_CODES.EISDIR)}if(!FS.isFile(node.mode)){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}var err=FS.nodePermissions(node,"w");if(err){throw new FS.ErrnoError(err)}node.node_ops.setattr(node,{size:len,timestamp:Date.now()})}),ftruncate:(function(fd,len){var stream=FS.getStream(fd);if(!stream){throw new FS.ErrnoError(ERRNO_CODES.EBADF)}if((stream.flags&2097155)===0){throw new FS.ErrnoError(ERRNO_CODES.EINVAL)}FS.truncate(stream.node,len)}),utime:(function(path,atime,mtime){var lookup=FS.lookupPath(path,{follow:true});var node=lookup.node;node.node_ops.setattr(node,{timestamp:Math.max(atime,mtime)})}),open:(function(path,flags,mode,fd_start,fd_end){if(path===""){throw new FS.ErrnoError(ERRNO_CODES.ENOENT)}flags=typeof flags==="string"?FS.modeStringToFlags(flags):flags;mode=typeof mode==="undefined"?438:mode;if(flags&64){mode=mode&4095|32768}else{mode=0}var node;if(typeof path==="object"){node=path}else{path=PATH.normalize(path);try{var lookup=FS.lookupPath(path,{follow:!(flags&131072)});node=lookup.node}catch(e){}}var created=false;if(flags&64){if(node){if(flags&128){throw new FS.ErrnoError(ERRNO_CODES.EEXIST)}}else{node=FS.mknod(path,mode,0);created=true}}if(!node){throw new FS.ErrnoError(ERRNO_CODES.ENOENT)}if(FS.isChrdev(node.mode)){flags&=~512}if(flags&65536&&!FS.isDir(node.mode)){throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR)}if(!created){var err=FS.mayOpen(node,flags);if(err){throw new FS.ErrnoError(err)}}if(flags&512){FS.truncate(node,0)}flags&=~(128|512);var stream=FS.createStream({node:node,path:FS.getPath(node),flags:flags,seekable:true,position:0,stream_ops:node.stream_ops,ungotten:[],error:false},fd_start,fd_end);if(stream.stream_ops.open){stream.stream_ops.open(stream)}if(Module["logReadFi