#include <math.h>
#include "../Numerics/types.h"
#include "../Exceptions/exceptions.h"

void EarlStrainrateQuantities(IssmDouble *pepso, IssmDouble *pepsprime_norm, IssmDouble vx,IssmDouble vy,IssmDouble vz,IssmDouble* dvx,IssmDouble* dvy,IssmDouble* dvz){/*{{{*/

	/*Intermediaries*/
	IssmDouble vorticity[3],vorticity_norm;
	IssmDouble nrsp[3],nrsp_norm;
	IssmDouble eps[3][3],epso;
	IssmDouble epsprime[3],epsprime_norm;

	/*Create vorticity vector*/
	_assert_(dvx && dvy && dvz);
	vorticity[0] =  dvz[1] - dvy[2];
	vorticity[1] =  dvx[2] - dvz[0];
	vorticity[2] =  dvy[0] - dvx[1];

	/*Normalize*/
	vorticity_norm = sqrt(vorticity[0]*vorticity[0] + vorticity[1]*vorticity[1] + vorticity[2]*vorticity[2]);
	if(vorticity_norm==0){
		vorticity[0] = 0.;
		vorticity[1] = 0.;
		vorticity[2] = 1.;
	}
	else{
		vorticity[0] =vorticity[0]/vorticity_norm;
		vorticity[1] =vorticity[1]/vorticity_norm;
		vorticity[2] =vorticity[2]/vorticity_norm;
	}

	/*Non-rotating shear plane*/
	nrsp[0] =  vy*vorticity[2] - vz*vorticity[1];
	nrsp[1] =  vz*vorticity[0] - vx*vorticity[2];
	nrsp[2] =  vx*vorticity[1] - vy*vorticity[0];

	/*Normalize*/
	nrsp_norm = sqrt(nrsp[0]*nrsp[0] + nrsp[1]*nrsp[1] + nrsp[2]*nrsp[2]);
	if(nrsp_norm==0){
		nrsp[0] = 0.;
		nrsp[1] = 0.;
		nrsp[2] = 1.;
	}
	else{
		nrsp[0] =nrsp[0]/nrsp_norm;
		nrsp[1] =nrsp[1]/nrsp_norm;
		nrsp[2] =nrsp[2]/nrsp_norm;
	}

	/*Build strain rate tensor*/
	eps[0][0] = dvx[0];             eps[0][1] = .5*(dvx[1]+dvy[0]); eps[0][2] = .5*(dvx[2]+dvz[0]);
	eps[1][0] = .5*(dvx[1]+dvy[0]); eps[1][1] = dvy[1];             eps[1][2] = .5*(dvy[2]+dvz[1]);
	eps[2][0] = .5*(dvx[2]+dvz[0]); eps[2][1] = .5*(dvy[2]+dvz[1]); eps[2][2] = dvz[2];

	/*octahedral strain rate*/
	epso = 0.;
	for(int i=0;i<3;i++) for(int j=0;j<3;j++) epso += eps[i][j]*eps[i][j];
	if(epso==0.) epso=1.e-14;
	epso=sqrt(epso)/sqrt(3.);

	/*Compute the shear strain rate on the non ratating shear plane*/
	epsprime[0]=0.;
	epsprime[1]=0.;
	epsprime[2]=0.;
	/*term #1: eps'.n */
	for(int i=0;i<3;i++){
		for(int j=0;j<3;j++){
			epsprime[i] += eps[i][j]*nrsp[j];
		}
	}
	/*term #2: ((eps'.n).n)n */
	for(int i=0;i<3;i++){
		for(int j=0;j<3;j++){
			for(int k=0;k<3;k++){
				epsprime[j] += -nrsp[i]*eps[i][k]*nrsp[k]*nrsp[j];
			}
		}
	}
	/*term #3: ((eps'.n).omega)omega */
	for(int i=0;i<3;i++){
		for(int j=0;j<3;j++){
			for(int k=0;k<3;k++){
				epsprime[j] += -nrsp[i]*eps[i][k]*vorticity[k]*vorticity[j];
			}
		}
	}

	/*Get norm of epsprime*/
	epsprime_norm = sqrt(epsprime[0]*epsprime[0] + epsprime[1]*epsprime[1] + epsprime[2]*epsprime[2]);

	/*Assign output pointers*/
	*pepso=epso;
	*pepsprime_norm=epsprime_norm;
}/*}}}*/
IssmDouble EarlLambdaS(IssmDouble epso, IssmDouble epsprime_norm){/*{{{*/
   _assert_(epso>0.);
	_assert_(epsprime_norm>=0.); 
	return sqrt(2*epsprime_norm*epsprime_norm/(3*epso*epso));
}/*}}}*/
