/*!\file Matearl.c
 * \brief: implementation of the Matearl object
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./Matearl.h"
#include "./Materials.h"
#include "../Inputs/Input.h"
#include "../Inputs/Inputs.h"
#include "../Inputs/TriaInput.h"
#include "../Inputs/PentaInput.h"
#include "../Inputs/ControlInput.h"
#include "../Elements/Element.h"
#include "../Elements/Tria.h"
#include "../Elements/Penta.h"
#include "../Params/Parameters.h"
#include "../Vertex.h"
#include "../Hook.h"
#include "../Node.h"
#include "../IoModel.h"
#include "../../shared/shared.h"

/*Matearl constructors and destructor*/
Matearl::Matearl(){/*{{{*/
	this->helement=NULL;
	this->element=NULL;
	return;
}
/*}}}*/
Matearl::Matearl(int matearl_mid,int index, IoModel* iomodel){/*{{{*/

	/*Intermediaries:*/
	int    matearl_eid;

	/*Initialize id*/
	this->mid=matearl_mid;

	/*Hooks: */
	matearl_eid=index+1;
	this->helement=new Hook(&matearl_eid,1);
	this->element=NULL;

	return;
}
/*}}}*/
Matearl::~Matearl(){/*{{{*/
	delete helement;
	return;
}
/*}}}*/

/*Object virtual functions definitions:*/
Object*   Matearl::copy() {/*{{{*/

	/*Output*/
	Matearl* matearl=NULL;

	/*Initialize output*/
	matearl=new Matearl();

	/*copy fields: */
	matearl->mid=this->mid;
	matearl->helement=(Hook*)this->helement->copy();
	matearl->element =(Element*)this->helement->delivers();

	return matearl;
}
/*}}}*/
Material* Matearl::copy2(Element* element_in) {/*{{{*/

	/*Output*/
	Matearl* matearl=NULL;

	/*Initialize output*/
	matearl=new Matearl();

	/*copy fields: */
	matearl->mid=this->mid;
	matearl->helement=(Hook*)this->helement->copy();
	matearl->element =element_in;

	return matearl;
}
/*}}}*/
void      Matearl::DeepEcho(void){/*{{{*/

	_printf_("Matearl:\n");
	_printf_("   mid: " << mid << "\n");
	_printf_("   element:\n");
	helement->Echo();
}		
/*}}}*/
void      Matearl::Echo(void){/*{{{*/

	_printf_("Matearl:\n");
	_printf_("   mid: " << mid << "\n");
	_printf_("   element:\n");
	helement->Echo();
}
/*}}}*/
int       Matearl::Id(void){ return mid; }/*{{{*/
/*}}}*/
int       Matearl::ObjectEnum(void){/*{{{*/

	return MatearlEnum;

}
/*}}}*/
void      Matearl::Marshall(char** pmarshalled_data,int* pmarshalled_data_size, int marshall_direction){ /*{{{*/

	if(marshall_direction==MARSHALLING_BACKWARD)helement=new Hook(); 
	
	MARSHALLING_ENUM(MatearlEnum);
	MARSHALLING(mid);
	this->helement->Marshall(pmarshalled_data,pmarshalled_data_size,marshall_direction);
	this->element=(Element*)this->helement->delivers();

}
/*}}}*/

/*Matearl management*/
void  Matearl::Configure(Elements* elementsin){/*{{{*/

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	helement->configure((DataSet*)elementsin);
	this->element  = (Element*)helement->delivers();
}
/*}}}*/
void  Matearl::SetCurrentConfiguration(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){/*{{{*/

}
/*}}}*/
IssmDouble Matearl::GetA(){/*{{{*/
	_error_("not implemented yet");
}
/*}}}*/
IssmDouble Matearl::GetAbar(){/*{{{*/
	_error_("not implemented yet");
}
/*}}}*/
IssmDouble Matearl::GetB(){/*{{{*/
	_error_("not implemented yet");
}
/*}}}*/
IssmDouble Matearl::GetBbar(){/*{{{*/

	_error_("not implemented yet");
}
/*}}}*/
IssmDouble Matearl::GetN(){/*{{{*/
	_error_("not implemented yet");
}
/*}}}*/
IssmDouble Matearl::GetD(){/*{{{*/
	_error_("not implemented yet");
}
/*}}}*/
IssmDouble Matearl::GetDbar(){/*{{{*/

	_error_("not implemented yet");
}
/*}}}*/
bool Matearl::IsDamage(){/*{{{*/

	_error_("not implemented yet");
}
/*}}}*/
void  Matearl::GetViscosity(IssmDouble* pviscosity,IssmDouble eps_eff){/*{{{*/
	_error_("not implemented yet");
	/*From a string tensor and a material object, return viscosity, using Glen's flow law.
								(1-D) B
	  viscosity= -------------------------
						  2 eps_eff ^[(n-1)/n]

	  where viscosity is the viscotiy, B the flow law parameter , eps_eff is the effective strain rate
	  and n the flow law exponent.

	  If eps_eff = 0 , it means this is the first time SystemMatrices is being run, and we 
	  return 10^14, initial viscosity.
	  */

	/*output: */
	IssmDouble viscosity;

	/*Intermediary: */
	IssmDouble B,D=0.,n;

	/*Get B and n*/
	B=GetB(); _assert_(B>0.);
	n=GetN(); _assert_(n>0.);

	if (n==1.){
		/*Linear Viscous behavior (Newtonian fluid) viscosity=B/2: */
		viscosity=(1.-D)*B/2.;
	}
	else{

		/*if no strain rate, return maximum viscosity*/
		if(eps_eff==0.){
			viscosity = 1.e+14/2.;
			//viscosity = B;
			//viscosity=2.5*pow(10.,17);
		}

		else{
			viscosity=(1.-D)*B/(2.*pow(eps_eff,(n-1.)/n));
		}
	}

	/*Checks in debugging mode*/
	if(viscosity<=0) _error_("Negative viscosity");

	/*Return: */
	*pviscosity=viscosity;
}
/*}}}*/
void  Matearl::GetViscosity_B(IssmDouble* pdmudB,IssmDouble eps_eff){/*{{{*/
	_error_("not implemented yet");
}
/*}}}*/
void  Matearl::GetViscosity_D(IssmDouble* pdmudD,IssmDouble eps_eff){/*{{{*/
	 _error_("not implemented yet");
}
/*}}}*/
void  Matearl::GetViscosityBar(IssmDouble* pviscosity,IssmDouble eps_eff){/*{{{*/
	_error_("not implemented yet");
}
/*}}}*/
void  Matearl::GetViscosityComplement(IssmDouble* pviscosity_complement, IssmDouble* epsilon){/*{{{*/
	_error_("not implemented yet");
}
/*}}}*/
void  Matearl::GetViscosityDComplement(IssmDouble* pviscosity_complement, IssmDouble* epsilon){/*{{{*/
	_error_("not implemented yet");
}
/*}}}*/
void  Matearl::GetViscosityDerivativeEpsSquare(IssmDouble* pmu_prime, IssmDouble* epsilon){/*{{{*/
	_error_("not implemented yet");
}
/*}}}*/
void  Matearl::GetViscosity2dDerivativeEpsSquare(IssmDouble* pmu_prime, IssmDouble* epsilon){/*{{{*/
	_error_("not implemented yet");
}
/*}}}*/
void  Matearl::InputUpdateFromVector(IssmDouble* vector, int name, int type){/*{{{*/

}
/*}}}*/
void  Matearl::InputUpdateFromVectorDakota(IssmDouble* vector, int name, int type){/*{{{*/

}
/*}}}*/
void  Matearl::InputUpdateFromMatrixDakota(IssmDouble* matrix, int nrows, int ncols,int name, int type){/*{{{*/
	/*Nothing updated yet*/
}
/*}}}*/
void  Matearl::InputUpdateFromConstant(IssmDouble constant, int name){/*{{{*/
	/*Nothing updated yet*/
}
/*}}}*/
void  Matearl::InputUpdateFromConstant(int constant, int name){/*{{{*/
	/*Nothing updated yet*/
}
/*}}}*/
void  Matearl::InputUpdateFromConstant(bool constant, int name){/*{{{*/
	/*Nothing updated yet*/
}
/*}}}*/
void  Matearl::ViscosityFS(IssmDouble* pviscosity,int dim,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input,Input* vz_input){/*{{{*/

	/*Intermediaries*/
	IssmDouble vx,vy,vz;
	IssmDouble dvx[3],dvy[3],dvz[3];
	IssmDouble ko,Ec,Es;

	/*Get velocity derivatives in all directions*/
	_assert_(dim>1);
	_assert_(vx_input);
	vx_input->GetInputValue(&vx,gauss);
	vx_input->GetInputDerivativeValue(&dvx[0],xyz_list,gauss);
	_assert_(vy_input);
	vy_input->GetInputValue(&vy,gauss);
	vy_input->GetInputDerivativeValue(&dvy[0],xyz_list,gauss);
	if(dim==3){
		_assert_(vz_input);
		vz_input->GetInputValue(&vz,gauss);
		vz_input->GetInputDerivativeValue(&dvz[0],xyz_list,gauss);
	}
	else{
		vz = 0.;
		dvz[0] = 0.; dvz[1] = 0.; dvz[2] = 0.;
	}

	/*Get enhancement factors and ko*/
	Input* ec_input = element->inputs->GetInput(MaterialsRheologyEcEnum); _assert_(ec_input);
	Input* es_input = element->inputs->GetInput(MaterialsRheologyEsEnum); _assert_(es_input);
	Input* ko_input = element->inputs->GetInput(MaterialsRheologyKoEnum); _assert_(ko_input);
	ec_input->GetInputValue(&Ec,gauss);
	es_input->GetInputValue(&Es,gauss);
	ko_input->GetInputValue(&ko,gauss);

	/*Compute viscosity*/
	*pviscosity=GetViscosityGeneral(ko,Ec,Es,vx,vy,vz,&dvx[0],&dvy[0],&dvz[0]);
}
/*}}}*/
void  Matearl::ViscosityFSDerivativeEpsSquare(IssmDouble* pmu_prime,IssmDouble* epsilon){/*{{{*/
	this->GetViscosityDerivativeEpsSquare(pmu_prime,epsilon);
}/*}}}*/
void  Matearl::ViscosityHO(IssmDouble* pviscosity,int dim,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input){/*{{{*/

	/*Intermediaries*/
	IssmDouble vx,vy,vz;
	IssmDouble dvx[3],dvy[3],dvz[3];
	IssmDouble ko,Ec,Es;

	/*Get velocity derivatives in all directions*/
	_assert_(dim==2 || dim==3);
	_assert_(vx_input);
	vx_input->GetInputValue(&vx,gauss);
	vx_input->GetInputDerivativeValue(&dvx[0],xyz_list,gauss);
	if(dim==3){
		_assert_(vy_input);
		vy_input->GetInputValue(&vy,gauss);
		vy_input->GetInputDerivativeValue(&dvy[0],xyz_list,gauss);
	}
	else{
		dvx[2] = 0.;
		vy = 0.;
		dvy[0] = 0.; dvy[1] = 0.; dvy[2] = 0.;
	}
	vz = 0.;
	dvz[0] = 0.; dvz[1] = 0.; dvz[2] = -dvx[0]-dvy[1];

	/*Get enhancement factors and ko*/
	Input* ec_input = element->inputs->GetInput(MaterialsRheologyEcEnum); _assert_(ec_input);
	Input* es_input = element->inputs->GetInput(MaterialsRheologyEsEnum); _assert_(es_input);
	Input* ko_input = element->inputs->GetInput(MaterialsRheologyKoEnum); _assert_(ko_input);
	ec_input->GetInputValue(&Ec,gauss);
	es_input->GetInputValue(&Es,gauss);
	ko_input->GetInputValue(&ko,gauss);

	/*Compute viscosity*/
	*pviscosity=GetViscosityGeneral(ko,Ec,Es,vx,vy,vz,&dvx[0],&dvy[0],&dvz[0]);
}/*}}}*/
void  Matearl::ViscosityHODerivativeEpsSquare(IssmDouble* pmu_prime,IssmDouble* epsilon){/*{{{*/
	_error_("not implemented yet");
}/*}}}*/
void  Matearl::ViscosityL1L2(IssmDouble* pviscosity,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input,Input* surface_input){/*{{{*/
	_error_("not implemented yet");
}/*}}}*/
void  Matearl::ViscositySSA(IssmDouble* pviscosity,int dim,IssmDouble* xyz_list,Gauss* gauss,Input* vx_input,Input* vy_input){/*{{{*/
	/*Intermediaries*/
	IssmDouble vx,vy,vz;
	IssmDouble dvx[3],dvy[3],dvz[3];
	IssmDouble ko,Ec,Es;

	/*Get velocity derivatives in all directions*/
	_assert_(dim==1 || dim==2);
	_assert_(vx_input);
	vx_input->GetInputValue(&vx,gauss);
	vx_input->GetInputDerivativeValue(&dvx[0],xyz_list,gauss);
	if(dim==2){
		_assert_(vy_input);
		vy_input->GetInputValue(&vy,gauss);
		vy_input->GetInputDerivativeValue(&dvy[0],xyz_list,gauss);
	}
	else{
		dvx[1] = 0.;
		dvx[2] = 0.;
		vy = 0.;
		dvy[0] = 0.; dvy[1] = 0.; dvy[2] = 0.;
	}
	vz = 0.;
	dvz[0] = 0.; dvz[1] = 0.; dvz[2] = -dvx[0]-dvy[1];

	/*Get enhancement factors and ko*/
	Input* ec_input = element->inputs->GetInput(MaterialsRheologyEcbarEnum); _assert_(ec_input);
	Input* es_input = element->inputs->GetInput(MaterialsRheologyEsbarEnum); _assert_(es_input);
	Input* ko_input = element->inputs->GetInput(MaterialsRheologyKobarEnum); _assert_(ko_input);
	ec_input->GetInputValue(&Ec,gauss);
	es_input->GetInputValue(&Es,gauss);
	ko_input->GetInputValue(&ko,gauss);

	/*Compute viscosity*/
	*pviscosity=GetViscosityGeneral(ko,Ec,Es,vx,vy,vz,&dvx[0],&dvy[0],&dvz[0]);
}/*}}}*/
void  Matearl::ViscositySSADerivativeEpsSquare(IssmDouble* pmu_prime,IssmDouble* epsilon){/*{{{*/
	_error_("not implemented yet");
}/*}}}*/
void  Matearl::ResetHooks(){/*{{{*/

	this->element=NULL;

	/*Get Element type*/
	this->helement->reset();

}
/*}}}*/
IssmDouble Matearl::GetViscosityGeneral(IssmDouble ko,IssmDouble Ec, IssmDouble Es,IssmDouble vx,IssmDouble vy,IssmDouble vz,IssmDouble* dvx,IssmDouble* dvy,IssmDouble* dvz){/*{{{*/

	/*Intermediaries*/
	IssmDouble viscosity;
	IssmDouble vorticity[3],vorticity_norm;
	IssmDouble nrsp[3],nrsp_norm;
	IssmDouble eps[3][3],epso;
	IssmDouble epsprime[3],epsprime_norm;
	IssmDouble E,lambdas;

	/*Create vorticity vector*/
	_assert_(dvx && dvy && dvz);
	vorticity[0] =  dvz[1] - dvy[2];
	vorticity[1] =  dvx[2] - dvz[0];
	vorticity[2] =  dvy[0] - dvx[1];

	/*Normalize*/
	vorticity_norm = sqrt(vorticity[0]*vorticity[0] + vorticity[1]*vorticity[1] + vorticity[2]*vorticity[2]);
	if(vorticity_norm==0){
		vorticity[0] = 0.;
		vorticity[1] = 0.;
		vorticity[2] = 1.;
	}
	else{
		vorticity[0] =vorticity[0]/vorticity_norm;
		vorticity[1] =vorticity[1]/vorticity_norm;
		vorticity[2] =vorticity[2]/vorticity_norm;
	}

	/*Non-rotating shear plane*/
	nrsp[0] =  vy*vorticity[2] - vz*vorticity[1];
	nrsp[1] =  vz*vorticity[0] - vx*vorticity[2];
	nrsp[2] =  vx*vorticity[1] - vy*vorticity[0];

	/*Normalize*/
	nrsp_norm = sqrt(nrsp[0]*nrsp[0] + nrsp[1]*nrsp[1] + nrsp[2]*nrsp[2]);
	if(nrsp_norm==0){
		nrsp[0] = 0.;
		nrsp[1] = 0.;
		nrsp[2] = 1.;
	}
	else{
		nrsp[0] =nrsp[0]/nrsp_norm;
		nrsp[1] =nrsp[1]/nrsp_norm;
		nrsp[2] =nrsp[2]/nrsp_norm;
	}

	/*Build strain rate tensor*/
	eps[0][0] = dvx[0];             eps[0][1] = .5*(dvx[1]+dvy[0]); eps[0][2] = .5*(dvx[2]+dvz[0]);
	eps[1][0] = .5*(dvx[1]+dvy[0]); eps[1][1] = dvy[1];             eps[1][2] = .5*(dvy[2]+dvz[1]);
	eps[2][0] = .5*(dvx[2]+dvz[0]); eps[2][1] = .5*(dvy[2]+dvz[1]); eps[2][2] = dvz[2];

	/*octahedral strain rate*/
	epso = 0.;
	for(int i=0;i<3;i++) for(int j=0;j<3;j++) epso += eps[i][j]*eps[i][j];
	if(epso==0.) epso=1.e-14;
	epso=sqrt(epso)/sqrt(3.);

	/*Compute the shear strain rate on the non ratating shear plane*/
	epsprime[0]=0.;
	epsprime[1]=0.;
	epsprime[2]=0.;
	/*term #1: eps'.n */
	for(int i=0;i<3;i++){
		for(int j=0;j<3;j++){
			epsprime[i] += eps[i][j]*nrsp[j];
		}
	}
	/*term #2: ((eps'.n).n)n */
	for(int i=0;i<3;i++){
		for(int j=0;j<3;j++){
			for(int k=0;k<3;k++){
				epsprime[j] += -nrsp[i]*eps[i][k]*nrsp[k]*nrsp[j];
			}
		}
	}
	/*term #3: ((eps'.n).omega)omega */
	for(int i=0;i<3;i++){
		for(int j=0;j<3;j++){
			for(int k=0;k<3;k++){
				epsprime[j] += -nrsp[i]*eps[i][k]*vorticity[k]*vorticity[j];
			}
		}
	}

	/*Get norm of epsprime*/
	epsprime_norm = sqrt(epsprime[0]*epsprime[0] + epsprime[1]*epsprime[1] + epsprime[2]*epsprime[2]);

	/*Compute lambda_s*/
	_assert_(epso>0.);
	lambdas = sqrt(2*epsprime_norm*epsprime_norm/(3*epso*epso));

	/*Get total enhancement factor E(lambdas)*/
	E = Ec + (Es-Ec)*lambdas*lambdas;

	/*Compute viscosity*/
	_assert_(E>0.); 
	_assert_(ko>0.); 
	_assert_(epso>0.); 
	viscosity = 1./(2*pow(ko*E*epso*epso,1./3.));

	/*Assign output pointer*/
	return viscosity;
}
/*}}}*/
