/* \file IoModel.h
 * \brief  Header file defining the IoModel structure that will help in processing the input data coming 
 * into ISSM, from Matlab, or through a binary file opened for reading.
 * \sa IoModel.cpp
 */

#ifndef _IOMODEL_H
#define _IOMODEL_H

#include "../shared/Enum/Enum.h"
#include <vector>

class Parameters;
class Elements;
class Param;
class Option;

class IoConstant { /*holds single IssmDouble, int, bool and char from input*/
	public:
		bool   isindependent;
		int    data_enum;
		char*  name;
		Param* constant; 

		~IoConstant();
		IoConstant();
		IoConstant(bool value,int enum_in);
		IoConstant(int value,int enum_in);
		IoConstant(IssmDouble value,int enum_in);
		IoConstant(char* value,int enum_in);
};

class IoData { /*holds temporary data (array), memory intensive*/
	public:
		bool        isindependent;
		int         data_enum;
		char*       name;
		int         M,N;
		int         code;
		int         layout;
		IssmDouble* data;

		~IoData();
		IoData();
		IoData(IssmDouble* matrix,int code,int layout_in,int M,int N,int enum_in);
};

class IoModel {

	private: 
		std::vector<IoConstant*> constants; //this dataset holds all IssmDouble, int, bool and char from input
		std::vector<IoData*>     data;      //this dataset holds temporary data, memory intensive

		/*for AD mode: to keep track of our independent variables we fetch:*/
		//bool    *independents;
		//DataSet *independent_objects;

	public:
		/*pointer to input file*/
		FILE *fid;

		/*Solution*/
		int   solution_enum;

		/*Partitioning*/
		bool *my_elements;
		int  *my_vertices;

		/*Mesh properties and connectivity tables*/
		int  domaintype;
		int  domaindim;
		int  meshelementtype;
		int  numberofvertices;
		int  numberofelements;
		int  numberoffaces;
		int  numberofedges;
		int  facescols;
		int *elements;
		int *faces;
		int *edges;
		int *elementtoedgeconnectivity;
		int *elementtofaceconnectivity;
		int *singlenodetoelementconnectivity;
		int *numbernodetoelementconnectivity;

		/*Data to synchronize through low level object drivers: */
		int nodecounter;         //keep track of how many nodes are being created in each analysis
		int loadcounter;         //keep track of how many loads are being created in each analysis
		int constraintcounter;   //keep track of how many constraints are being created in each analysis

		/*Methods*/
		~IoModel();
		IoModel();
		IoModel(FILE* iomodel_handle,int solution_enum_in,bool trace,IssmPDouble* X);

		/*NEW*/
		void        AddConstant(IoConstant* constant_in);
		void        AddConstantIndependent(IoConstant* constant_in);
		void        AddData(IoData* data_in);
		void        AddDataIndependent(IoData* data_in);
		void        FindConstant(bool* pvalue,int constant_enum);
		void        FindConstant(int* pvalue,int constant_enum);
		void        FindConstant(IssmDouble* pvalue,int constant_enum);
		void        FindConstant(char **pvalue,int constant_enum);
		void        FetchIndependentConstant(int* pXcount,IssmPDouble* X,int name);
		void        FetchIndependentData(int* pXcount,IssmPDouble* X,int name);
		int         NumIndependents();
		void        FillIndependents(IssmDouble* xp);

		/*Input/Output*/
		void        CheckEnumSync(void);
		Param      *CopyConstantObject(int constant_enum);
		IssmDouble *Data(int dataenum);
		void        DeleteData(int num,...);
		void        DeleteData(IssmDouble* vector, int dataenum);
		void        DeleteData(char*** pstringarray, int numstrings, int dataenum);
		void        FetchConstants(void);
		void        FetchData(bool* pboolean,int data_enum);
		void        FetchData(int* pinteger,int data_enum);
		void        FetchData(IssmDouble* pscalar,int data_enum);
		void        FetchData(char** pstring,int data_enum);
		void        FetchData(int** pmatrix,int* pM,int* pN,int data_enum);
		void        FetchData(IssmDouble**  pscalarmatrix,int* pM,int* pN,int data_enum);
		void        FetchData(char***   pstringarray,int* pnumstrings,int data_enum);
		void        FetchData(IssmDouble*** pmatrixarray,int** pmdims,int** pndims, int* pnumrecords,int data_enum);
		void        FetchMultipleData(char***   pstringarray,int* pnumstrings,int data_enum);
		void        FetchMultipleData(IssmDouble*** pmatrixarray,int** pmdims,int** pndims, int* pnumrecords,int data_enum);
		void        FetchMultipleData(int*** pmatrices,int** pmdims,int** pndims, int* pnumrecords,int data_enum);
		void        FetchMultipleData(int** pvector, int* pnum_instances,int data_enum);
		void        FetchMultipleData(IssmDouble** pvector, int* pnum_instances,int data_enum);
		void        FetchData(Option **poption,int data_enum);
		void        FetchData(int num,...);
		void        FetchDataToInput(Elements* elements,int vector_enum);
		void        FetchDataToInput(Elements* elements,int vector_enum,IssmDouble default_value);
		void        LastIndex(int *pindex);
		FILE*       SetFilePointerToData(int* pcode,int* pvector_type, int data_enum);
		fpos_t*     SetFilePointersToData(int** pcodes,int** pvector_types, int* pnum_instances, int data_enum);
		void        DeclareIndependents(bool trace,IssmPDouble* X);
		void        StartTrace(bool trace);
		void        FetchIndependent(int dependent_enum);
};

#endif  /* _IOMODEL_H */
