/*!\file:  steadystate.cpp
 * \brief: steadystate solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "steadystate"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


int main(int argc,char* *argv){
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   numberofnodes;
	int   qmu_analysis=0;
	int   control_analysis=0;
	char* control_type=NULL;

	/*Model: */
	Model* model=NULL;
	FemModel* fem_dh=NULL;
	FemModel* fem_ds=NULL;

	/*Results: */
	DataSet* results=NULL;
	Result* result=NULL;
	
	ParameterInputs* inputs=NULL;
	int waitonlock=0;
	
	double* u_g_initial=NULL;
	double* p_g_initial=NULL;
	double* u_g_obs=NULL;
	double  dt;
	Param*  param=NULL;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	/*Initialize model structure: */
	model=new Model();

	_printf_("read and create finite element model:\n");
	
	_printf_("\n   reading diagnostic horiz model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),HorizAnalysisEnum());

	_printf_("\n   reading diagnostic vert model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),VertAnalysisEnum());
	
	_printf_("\n   reading diagnostic stokes model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),StokesAnalysisEnum());
	
	_printf_("\n   reading diagnostic hutter model data:\n");
	model->AddFormulation(fid,DiagnosticAnalysisEnum(),HutterAnalysisEnum());
	
	_printf_("\n   reading surface and bed slope computation model data:\n");
	model->AddFormulation(fid,SlopeComputeAnalysisEnum());

	_printf_("\n   read and create thermal finite element model:\n");
	model->AddFormulation(fid,ThermalAnalysisEnum());
	_printf_("\n   read and create melting finite element model:\n");
	model->AddFormulation(fid,MeltingAnalysisEnum());

	/*recover parameters: */
	model->FindParam(&waitonlock,"waitonlock");
	model->FindParam(&control_analysis,"control_analysis");
	model->FindParam(&qmu_analysis,"qmu_analysis");

	_printf_("initialize inputs:\n");
	model->FindParam(&u_g_initial,"u_g",DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	model->FindParam(&p_g_initial,"p_g",ThermalAnalysisEnum());
	model->FindParam(&dt,"dt");
	model->FindParam(&numberofnodes,"numberofnodes");
	
	inputs=new ParameterInputs;
	inputs->Add("velocity",u_g_initial,3,numberofnodes);
	inputs->Add("pressure",p_g_initial,1,numberofnodes);
	inputs->Add("dt",dt);

	if(control_analysis){
		model->FindParam(&u_g_obs,"u_g_obs",DiagnosticAnalysisEnum(),HorizAnalysisEnum());
		inputs->Add("velocity_obs",u_g_obs,2,numberofnodes);
	}
	
	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());

	/*are we running the solution sequence, or a qmu wrapper around it? : */
	if(!qmu_analysis){

		if(!control_analysis){
			/*run diagnostic analysis: */
			_printf_("call computational core:\n");
			steadystate_core(results,model,inputs);
		}
		else{
			/*change control_steady to 1 to know we are doing steadystate*/
			fem_dh=model->GetFormulation(DiagnosticAnalysisEnum(),HorizAnalysisEnum());
			fem_ds=model->GetFormulation(DiagnosticAnalysisEnum(),StokesAnalysisEnum());
			param=(Param*)fem_dh->parameters->FindParamObject("control_steady");
			param->SetInteger(1);
			param=(Param*)fem_ds->parameters->FindParamObject("control_steady");
			param->SetInteger(1);
			/*run control analysis: */
			_printf_("call computational core:\n");
			control_core(results,model,inputs);
		}

		if(!control_analysis){
			/*Add analysis_type to results: */
			result=new Result(results->Size()+1,0,1,"analysis_type","steadystate");
			results->AddObject(result);

			_printf_("process results:\n");
			ProcessResults(&results,model,SteadystateAnalysisEnum());
		}
		else{
			/*Add analysis_type and control_type to results: */
			result=new Result(results->Size()+1,0,1,"analysis_type","steadystate");
			results->AddObject(result);

			model->FindParam(&control_type,"control_type");
			result=new Result(results->Size()+1,0,1,"control_type",control_type);
			results->AddObject(result);

			_printf_("process results:\n");
			ProcessResults(&results,model,ControlAnalysisEnum());
		}

		_printf_("write results to disk:\n");
		OutputResults(results,outputfilename);
	}
	else{
		/*run qmu analysis: */
		_printf_("calling qmu analysis on steadystate core:\n");

		#ifdef _HAVE_DAKOTA_ 
		Qmux(model,inputs,SteadystateAnalysisEnum(),NoneAnalysisEnum());
	 	#else
		throw ErrorException(__FUNCT__," Dakota not present, cannot do qmu!");
		#endif
	}

	_printf_("write lock file:\n");
	if (waitonlock){
		WriteLockFile(lockname);
	}
	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 
	

	/*end module: */
	MODULEEND();
	
	/*Free ressources */
	xfree((void**)&u_g_initial);
	xfree((void**)&u_g_obs);
	xfree((void**)&p_g_initial);
	delete model;
	delete results;
	delete inputs;

	return 0; //unix success return;
}
