/*!\file:  Qmux.cpp
 * \brief: wrapper to the Dakota capabilities. qmu fires up Dakota, and registers a Dakota Pluggin
 * which will be in charge of running the solution sequences repeteadly, to garner statistics. 
 *
 * This routine deals with running ISSM and Dakota in library mode. In library mode, Dakota does not 
 * run as an execuatble. Its capabilities are linked into the ISSM software. ISSM calls dakota routines 
 * directly from the dakota library. qmu.cpp is the code that is in charge of calling those routines. 
 *
 * Dakota has its own way of running in parallel (for embarassingly parallel jobs). We do not want that, 
 * as ISSM knows exactly how to run "really parallel" jobs that use all CPUS. To bypass Dakota's parallelism, 
 * we overloaded the constructor for the parallel library (see the Dakota patch in the externalpackages/dakota
 * directory). This overloaded constructor fires up Dakota serially on CPU 0 only! We take care of broadcasting 
 * to the other CPUS, hence ISSM is running in parallel, and Dakota serially on CPU0. 
 *
 * Now, how does CPU 0 drive all other CPUS to carry out sensitivity analysese? By synchronizing its call to 
 * our ISSM cores (diagnostic_core, thermal_core, transient_core, etc ...) on CPU 0 with all other CPUS. 
 * This explains the structure of qmu.cpp, where cpu 0 runs Dakota, the Dakota pluggin fires up SpawnCore.cpp, 
 * while the other CPUS are waiting for a broadcast from CPU0, once they get it, they also fire up 
 * SpawnCore. In the end, SpawnCore is fired up on all CPUS, with CPU0 having Dakota inputs, that it will 
 * broacast to other CPUS. 
 *
 * Now, how does dakota call the SpawnCore routine? The SpawnCore is embedded into the DakotaPlugin object 
 * which is derived from the Direct Interface Dakota objct. This is the only way to run Dakota in library 
 * mode (see their developper guide for more info). Dakota registers the DakotaPlugin object into its own 
 * database, and calls on the embedded SpawnCore from CPU0. 
 *
 */ 



#ifdef HAVE_CONFIG_H
#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


#undef __FUNCT__ 
#define __FUNCT__ "Qmux"

#include "./Qmux.h"

#include "../shared/shared.h"
#include "../include/macros.h"
#include "../toolkits/toolkits.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../objects/DakotaPlugin.h"

#ifdef _HAVE_DAKOTA_ //only works if dakota library has been compiled in.
#include "ParallelLibrary.H"
#include "ProblemDescDB.H"
#include "DakotaStrategy.H"
#include "DakotaModel.H"
#include "DakotaInterface.H"

#endif

#ifdef _SERIAL_
void Qmux(mxArray* model,mxArray* inputs,int analysis_type,int sub_analysis_type,char* dakota_input_file,char* dakota_output_file,char* dakota_error_file){
#else
void Qmux(Model* model,ParameterInputs* inputs,int analysis_type,int sub_analysis_type){
#endif


	#ifdef _HAVE_DAKOTA_ //only works if dakota library has been compiled in.

	#ifdef _PARALLEL_
	char* dakota_input_file=NULL;
	char* dakota_output_file=NULL;
	char* dakota_error_file=NULL;
	extern int my_rank;
	#endif
	int status=0;
	Dakota::ModelLIter ml_iter;

	#ifdef _PARALLEL_
	/*Recover dakota_input_file, dakota_output_file and dakota_error_file, in the parameters dataset in parallel */
	model->FindParam(&dakota_input_file,"qmuinname");
	model->FindParam(&dakota_output_file,"qmuoutname");
	model->FindParam(&dakota_error_file,"qmuerrname");
	#endif

	#ifdef _PARALLEL_
	if(my_rank==0){
	#endif
	
		// Instantiate/initialize the parallel library and problem description
		// database objects.
		#ifdef _SERIAL_
			Dakota::ParallelLibrary parallel_lib; //use Dakota's standard library mode constructor
		#else
			Dakota::ParallelLibrary parallel_lib("serial"); //use our own ISSM Dakota library mode constructor, which only fires up Dakota on CPU 0. 
		#endif
		Dakota::ProblemDescDB problem_db(parallel_lib); 

		// Manage input file parsing, output redirection, and restart processing
		// without a CommandLineHandler.  This version relies on parsing of an
		// input file.
		problem_db.manage_inputs(dakota_input_file);
		// specify_outputs_restart() is only necessary if specifying non-defaults
		parallel_lib.specify_outputs_restart(dakota_output_file,dakota_error_file,NULL,NULL);

		// Instantiate the Strategy object (which instantiates all Model and
		// Iterator objects) using the parsed information in problem_db.
		Dakota::Strategy selected_strategy(problem_db);

		// convenience function for iterating over models and performing any
		// interface plug-ins
		Dakota::ModelList& models = problem_db.model_list();

		for (ml_iter = models.begin(); ml_iter != models.end(); ml_iter++) {

			Dakota::Interface& interface = ml_iter->interface();

			//set DB nodes to the existing Model specification
			problem_db.set_db_model_nodes(ml_iter->model_id());

			// Serial case: plug in derived Interface object without an analysisComm
			interface.assign_rep(new SIM::DakotaPlugin(problem_db,(void*)model,(void*)inputs,analysis_type,sub_analysis_type), false);
		}
	
		// Execute the strategy
		problem_db.lock(); // prevent run-time DB queries
		selected_strategy.run_strategy();
		
		#ifdef _PARALLEL_
		//Warn other cpus that we are done running the dakota iterator, by setting the counter to -1:
		SpawnCore(NULL,0, NULL,NULL,0,model,inputs,analysis_type,sub_analysis_type,-1);
		#endif

	#ifdef _PARALLEL_
	}
	else{

		for(;;){
			if(!SpawnCore(NULL,0, NULL,NULL,0,model,inputs,analysis_type,sub_analysis_type,0))break; //counter came in at -1 on cpu0, bail out.
		}
	}
	#endif

	/*Free ressources:*/
	#ifdef _PARALLEL_
	xfree((void**)&dakota_input_file);
	xfree((void**)&dakota_error_file);
	xfree((void**)&dakota_output_file);
	#endif


	#endif

}
