#!/bin/bash
#This bash script calls the nightlyrun.m matlab file to run our nightly test decks. 
#It then processes the results and sends an email to the Ice developpers.

echo "Cleaning up execution directory"
rm -rf $ISSM_DIR/execution/*
rm -rf $ISSM_DIR/nightlylog
mkdir  $ISSM_DIR/nightlylog

#Get configuration
#Source config file{{{
if [ $# -ne 1 ];
then
	#no config file specified: exit
	echo "no config file specified. Exiting..." >&2 # Error message to stderr.
	exit 1
fi
if [ ! -f "$1" ]
then
	echo "File $1 not found!" >&2   # Error message to stderr.
	exit 1
fi 
source $1;
#}}}
#Get Operating system (OS) name{{{
OS=$(uname -s)
if [[ $OS == CYGWIN* ]]; then 
	OS="win";
fi
#}}}

#SVN version issue requires that we set SVN and SVNVERSION explicitly in order
#to remain compatible with the version used by Jenkins to do checkouts
#{{{
if [[ -z $SVN ]]; then
	echo "SVN variable is not set!"
	echo "Defaulting to SVN found in path..."
	SVN='svn'
fi
if [[ -z $SVNVERSION ]]; then
	echo "SVNVERSION variable is not set!"
	echo "Defaulting to SVNVERSION found in path..."
	SVNVERSION='svnversion'
fi
#}}}

PYTHON_TEST=0
#testing {{{
echo "======================================================";
echo "             Determining Installation type            "
echo "======================================================";
ISSM_EXTERNALPACKAGES="none"
ISSM_RECONFIGURE="no"
ISSM_COMPILATION="no"
if [ -a $ISSM_DIR/svn_revision_old ]
then
	SVN_PREVIOUS=$(cat $ISSM_DIR/svn_revision_old)
	SVN_CURRENT=$SVN_REVISION_1
	echo "Previous revision number: $SVN_PREVIOUS"
	echo "Current  revision number: $SVN_CURRENT"

	svn diff -r $SVN_PREVIOUS:$SVN_CURRENT --summarize $ISSM_DIR > $ISSM_DIR/TEMP
	echo "List of changed files"
	cat $ISSM_DIR/TEMP

	#Do we need to reinstall externalpackages?
	if [ ! -z "$(cat $ISSM_DIR/TEMP | grep externalpackages)" ] ; then
		echo "  -- checking for changed externalpackages... yes";
		ISSM_EXTERNALPACKAGES="install"
	else
		echo "  -- checking for changed externalpackages... no";
		ISSM_EXTERNALPACKAGES="none"
	fi

	#Do we need to reconfigure
	if [ ! -z "$(cat $ISSM_DIR/TEMP | grep -e "Makefile.am" -e "m4" )" ] ; then
		echo "  -- checking for reconfiguration... yes";
		ISSM_RECONFIGURE="yes"
	else
		echo "  -- checking for reconfiguration... no";
		ISSM_RECONFIGURE="no"
	fi

	#Do we need to recompile
	if [ ! -z "$(cat $ISSM_DIR/TEMP | grep -e "\.cpp" -e "\.h" )" ] ; then
		echo "  -- checking for recompilation... yes";
		ISSM_COMPILATION="yes"
	else
		echo "  -- checking for recompilation... no";
		ISSM_COMPILATION="no"
	fi

else
	ISSM_EXTERNALPACKAGES="install"
	ISSM_RECONFIGURE="yes"
	ISSM_COMPILATION="yes"
fi
echo "Recording current svn version: $SVN_REVISION_1"
echo $SVN_REVISION_1 > $ISSM_DIR/svn_revision_old
#}}}

#install/copy/none external packages    (ISSM_EXTERNALPACKAGES){{{
if [ "$ISSM_EXTERNALPACKAGES" == "install" ]
then
	#number of packages: 
	NUMPACKAGES=$(($(echo $EXTERNALPACKAGES | wc -w )/2))
	EXTERNAL_TEST_FILE="$ISSM_DIR/nightlylog/results/external.xml"

	mkdir -p $ISSM_DIR/nightlylog/results

	echo "<testsuite tests=\"$NUMPACKAGES\">" > $EXTERNAL_TEST_FILE

	for ((i=1;i<=$NUMPACKAGES;i++))
	do
		NUM1=$((2*$i-1))
		NUM2=$((2*$i))
		PACKAGENAME=$(echo $EXTERNALPACKAGES | cut -d " " -f $NUM1-$NUM1)
		PACKAGEINST=$(echo $EXTERNALPACKAGES | cut -d " " -f $NUM2-$NUM2)

		cd $ISSM_DIR/externalpackages/$PACKAGENAME
		#tricky part here. We do not want to reinstall this package if it's already installed
		#and the svn file which does the install has not be modified! To keep track of 
		#this, we check the current svn version against the "last changed" revision number.  If they 
		#are the same, we reinstall the package. 
		current_version=`$SVNVERSION`
		last_changed_version=`$SVN info $PACKAGEINST | grep "Last Changed Rev" | awk '{printf("%s\n",$4);}'`

		if [[ $current_version == $last_changed_version ]]; then 
			#go ahead and reinstall. 
			echo "For $PACKAGENAME: svn current_version  is $current_version and last changed version is $last_changed_version"
			echo "Triggering new install of $PACKAGENAME"
			install_test=1
		else
			#ok, we want to skip, unless the package is not installed: 
			if [ -d install ]; then 
				#could be empty, signaling a failed previous install: 
				if [ "$(ls -A install)" ];then
					echo "For $PACKAGENAME: svn current_version  is $current_version and last changed version is $last_changed_version"
					echo "and install directory exists, so skipping install of $PACKAGENAME"
					install_test=0;
				else
					echo "For $PACKAGENAME: svn current_version  is $current_version and last changed version is $last_changed_version"
					echo "and install directory exists, however, it is empty, so triggering install of $PACKAGENAME"
					install_test=1;
				fi
			else
				echo "For $PACKAGENAME: svn current_version  is $current_version and last changed version is $last_changed_version"
				echo "However, install directory does not exist, so triggering install of $PACKAGENAME"
				install_test=1;
			fi
		fi

		if [[ $install_test == 1 ]]; then 
			echo "======================================================";
			echo "       Installing $PACKAGENAME                        ";
			echo "======================================================";
			./$PACKAGEINST |  tee compil.log
			if [ $? -ne 0 ]; then
				echo "======================================================";
				echo "    ERROR: installation of $PACKAGENAME failed        ";
				echo "======================================================";
				#erase install directory, so that next time, we still try and compile this!
				rm -rf install
				echo "<testcase classname=\"externalpackages\" name=\"$PACKAGENAME\">" >> $EXTERNAL_TEST_FILE
				echo '<failure message="failure">External packages did not install right. Check it.</failure>' >> $EXTERNAL_TEST_FILE
				echo '</testcase>' >> $EXTERNAL_TEST_FILE
			else
				echo "<testcase classname=\"externalpackages\" name=\"$PACKAGENAME\"/>" >> $EXTERNAL_TEST_FILE
			fi
			source $ISSM_DIR/etc/environment.sh
		else
			echo "======================================================";
			echo "       Skipping install of $PACKAGENAME               ";
			echo "======================================================";
			source $ISSM_DIR/etc/environment.sh #for good measure :)

			echo "<testcase classname=\"externalpackages\" name=\"$PACKAGENAME\"/>" >> $EXTERNAL_TEST_FILE
		fi
		cd ..
	done
	echo '</testsuite>' >> $EXTERNAL_TEST_FILE

elif [ "$ISSM_EXTERNALPACKAGES" == "copy" ]
then
	cd $ISSM_DIR
	rm -rf externalpackages
	cp -Rf $EXTERNALPACKAGESDIR ./
elif [ "$ISSM_EXTERNALPACKAGES" == "link" ]
then
	cd $ISSM_DIR
	rm -rf externalpackages
	ln -s  $EXTERNALPACKAGESDIR  .
elif [ "$ISSM_EXTERNALPACKAGES" == "none" ]
then
	echo "Skipping external packages installation"
else
	echo "ISSM_EXTERNALPACKAGES supported values are: install, copy and none. Exiting..." >&2 # Error message to stderr.
	exit 1
fi

source $ISSM_DIR/etc/environment.sh

if [ "$OS" == "win" ]
then
	echo "WINDOWS ENVIRONMENT DETECTED"
	source $ISSM_DIR/externalpackages/windows/windows_environment.sh
fi

#}}}
#ISSM compilation yes/no                (ISSM_COMPILATION) {{{
if [ "$ISSM_COMPILATION" == "yes" ]
then
	cd $ISSM_DIR
	if [ "$ISSM_RECONFIGURE" == "yes" ]
	then
		echo "======================================================";
		echo "             Cleaning up and reconfiguring            "
		echo "======================================================";
		make uninstall
		make distclean
		./scripts/automakererun.sh
		eval "./configure $ISSM_CONFIG"
	fi

	#4: compile and install ISSM
	echo "======================================================";
	echo "                    Compiling ISSM                    "
	echo "======================================================";
	if [ $NUMCPUS_INSTALL -gt 1 ]; then
		echo "Making with " $NUMCPUS_INSTALL " cpus"
		make -j $NUMCPUS_INSTALL
	else
		make
	fi
	if [ $? -ne 0 ]; then 
		echo "ISSM_COMPILATION failed!"
		exit 1
	fi
	make install
elif [ "$ISSM_COMPILATION" == "no" ]
then
	echo "Skipping ISSM compilation"
else
	echo "ISSM_COMPILATION supported values are: yes and no. Exiting..." >&2 # Error message to stderr.
	exit 1
fi
#}}}

#matlab tests
if [ $MATLAB_TEST -eq 1 ]; then
#Launch all tests on different cpus {{{
for (( i=1;i<=$NUMCPUS_RUN;i++ ))
do
	#Launch matlab and the nightly run script
	cat > $ISSM_DIR/nightlylog/matlab_run$i.m << EOF
	warning off %necessary to avoid a log of several Go for parallel runs
	try,
	$(if [ "$MATLAB_NROPTIONS" = ""  ]
	then
		echo "runme('output','nightly','rank',$i,'numprocs',$NUMCPUS_RUN);"
	else
		echo "runme($MATLAB_NROPTIONS,'output','nightly','rank',$i,'numprocs',$NUMCPUS_RUN);"
	fi
	)
	catch me,
		%An error occured, get report and exit
		message=getReport(me)
		directory=strsplit(pwd,'/');
		fid=fopen([issmdir '/nightlylog/matlaberror.log'], 'at');
		fprintf(fid,'\nMatlab error occured in: %s\n\n',directory{end});
		fprintf(fid,'%s',message);
		fclose(fid);
	end
	disp('MATLABEXITEDCORRECTLY');
	exit
EOF
	cd $ISSM_DIR/test/NightlyRun
	if [ "$OS" = "win" ]; then
		$MATLAB_PATH/bin/matlab -nojvm -nosplash -r "addpath $ISSM_DIR_WIN/src/m/dev; devpath; addpath $ISSM_DIR_WIN/nightlylog/; matlab_run$i" -logfile $ISSM_DIR_WIN/nightlylog/matlab_log$i.log &
	else
		$MATLAB_PATH/bin/matlab -nojvm -nosplash -r "addpath $ISSM_DIR/src/m/dev; devpath; addpath $ISSM_DIR/nightlylog/; matlab_run$i" -logfile $ISSM_DIR/nightlylog/matlab_log$i.log &
	fi
done

#wait until matlab closes
if [ "$OS" = "win" ]; then
	sleep 5;
	echo "Waiting for matlab on windows"
	pid=$(ps aux -W | grep MATLAB | awk '{printf("%s\n","MATLAB");}')
	echo '-----------------------------'
	echo "pid: $pid"
	echo '-----------------------------'
	while [ -n "$pid" ]
	do
		pid=$(ps aux -W | grep MATLAB | awk '{printf("%s\n","MATLAB");}')
		sleep 1;
	done
	echo "DONE!"
else
	wait
fi

#concatenate reports
cd $ISSM_DIR/nightlylog/
echo 'CHECKING NIGHTLYLOG DIRECTORY'
echo '-----------------------------'
ls -la
echo '-----------------------------'

rm matlab_log.log

for job in `jobs -p`
do
echo "Waiting on: $job"
    wait $job
done

for (( i=1;i<=$NUMCPUS_RUN;i++ ))
do
	cat matlab_log$i.log >> matlab_log.log
done

#filter out windows characters: 
cat matlab_log.log | tr -cd '\11\12\40-\176' > matlab_log.log2 && mv matlab_log.log2 matlab_log.log
#}}}
fi

#python tests
if [ $PYTHON_TEST -eq 1 ]; then
#Launch all tests on different cpus {{{
PYTHON_START_TIME=$(timer);
export PYTHONSTARTUP=$ISSM_DIR/src/m/dev/devpath.py
for (( i=1;i<=$NUMCPUS_RUN;i++ ))
do
	cd $ISSM_DIR/test/NightlyRun
	echo "--------------Running Python test for Rank $i---------------------"
	#./runme.py --output=nightly --rank=$i --numprocs=$NUMCPUS_RUN $PYTHON_NROPTIONS
	./runme.py --output=nightly --rank=$i --numprocs=$NUMCPUS_RUN $PYTHON_NROPTIONS &> $ISSM_DIR/nightlylog/python_log$i.log &
	echo "--------------Running Python test for Rank $i---------------------"
done

# concatenate reports
cd $ISSM_DIR/nightlylog/
rm python_log.log

for job in `jobs -p`
do
echo "Waiting on: $job"
    wait $job
done

for (( i=1;i<=$NUMCPUS_RUN;i++ ))
do
	echo "This is the concatenation phase for rank: python_log$i.log"
	#cat python_log$i.log
	cat python_log$i.log >> python_log.log
done
#}}}
fi

# This test will allow us to check on the status of the examples.
if [ $EXAMPLES_TEST -eq 1]; then
	cd $ISSM_DIR/examples
fi

#process logs to be junit compatible
#{{{
cd $ISSM_DIR/nightlylog/
source $ISSM_DIR/externalpackages/shell2junit/install/sh2ju.sh
juLogClean

if [ $MATLAB_TEST -eq 1 ]; then
	#number tests:
	numtests=`cat matlab_log.log  | grep "\-\-\-\-\-\-\-\-starting" | wc -l`
	testlist=`cat matlab_log.log  | grep "\-\-\-\-\-\-\-\-starting" | sed 's/----------------starting://g'  | sed 's/-//g'`

	#look through numtests:
	for i in `echo $testlist`
	do
		juLog  -test=MATLAB-$i -name=Error -error=ERROR awk "/starting:$i/{flag=1;next}/finished/{flag=0} flag{print}" matlab_log.log
		juLog  -test=MATLAB-$i -name=Failure -error=FAILURE awk "/starting:$i/{flag=1;next}/finished/{flag=0} flag{print}" matlab_log.log
	done
fi
if [ $PYTHON_TEST -eq 1 ]; then
	#number tests:
	numtests=`cat python_log.log  | grep "\-\-\-\-\-\-\-\-starting" | wc -l`
	testlist=`cat python_log.log  | grep "\-\-\-\-\-\-\-\-starting" | sed 's/----------------starting://g'  | sed 's/-//g'`

	#look through numtests:
	for i in `echo $testlist`
	do
		juLog  -test=PYTHON-$i -name=Error -error=ERROR awk "/starting:$i/{flag=1;next}/finished/{flag=0} flag{print}" python_log.log
		juLog  -test=PYTHON-$i -name=Failure -error=FAILURE awk "/starting:$i/{flag=1;next}/finished/{flag=0} flag{print}" python_log.log
	done
fi
#}}}
