function ismodelselfconsistent(md),
%ISMODELSELFCONSISTENT - check that model forms a closed form solvable problem.
%
%   Usage:
%      ismodelselfconsistent(md),

%initialize consistency as true
md.private.isconsistent=true;

%Get solution and associated analyses
solution=md.private.solution;
[analyses]=AnalysisConfiguration(solution);

%Go through a model field, check that it is a class, and call checkconsistency
fields=properties('model');
for i=1:length(fields),
	field=fields{i};

	%Some properties do not need to be checked
	if ismember(field,{'results' 'debug' 'radaroverlay'}),
		continue;
	end

	%Check that current field is an object
	if ~isobject(md.(field))
		md=checkmessage(md,['field ''' char(field) ''' is not an object']);
		continue;
	end

	%Check consistency of the object
	md=checkconsistency(md.(field),md,solution,analyses);
end

%error message if mode is not consistent
if md.private.isconsistent==false,
	error('Model not consistent, see messages above');
end
end

function [analyses]=AnalysisConfiguration(solutiontype), % {{{
%ANALYSISCONFIGURATION - return type of analyses, number of analyses 
%
%   Usage:
%      [analyses]=AnalysisConfiguration(solutiontype);

switch solutiontype,

	case StressbalanceSolutionEnum(),
		analyses=[StressbalanceAnalysisEnum();StressbalanceVerticalAnalysisEnum();StressbalanceSIAAnalysisEnum();L2ProjectionBaseAnalysisEnum()];

	case SteadystateSolutionEnum(),
		analyses=[StressbalanceAnalysisEnum();StressbalanceVerticalAnalysisEnum();StressbalanceSIAAnalysisEnum();L2ProjectionBaseAnalysisEnum();ThermalAnalysisEnum();MeltingAnalysisEnum(),EnthalpyAnalysisEnum()];

	case ThermalSolutionEnum(),
		analyses=[EnthalpyAnalysisEnum;ThermalAnalysisEnum();MeltingAnalysisEnum()];

	case MasstransportSolutionEnum(),
		analyses=[MasstransportAnalysisEnum()];

	case BalancethicknessSolutionEnum(),
		analyses=[BalancethicknessAnalysisEnum()];

	case Balancethickness2SolutionEnum(),
		analyses=[Balancethickness2AnalysisEnum()];

	case BalancethicknessSoftSolutionEnum(),
		analyses=[BalancethicknessAnalysisEnum()];

	case BalancevelocitySolutionEnum(),
		analyses=[BalancevelocityAnalysisEnum()];

	case SurfaceSlopeSolutionEnum(),
		analyses=[L2ProjectionBaseAnalysisEnum()];

	case BedSlopeSolutionEnum(),
		analyses=[L2ProjectionBaseAnalysisEnum()];

	case GiaSolutionEnum(),
		analyses=[GiaAnalysisEnum()];

	case TransientSolutionEnum(),
		analyses=[StressbalanceAnalysisEnum();StressbalanceVerticalAnalysisEnum();StressbalanceSIAAnalysisEnum();L2ProjectionBaseAnalysisEnum();ThermalAnalysisEnum();MeltingAnalysisEnum();EnthalpyAnalysisEnum();MasstransportAnalysisEnum();HydrologySommersAnalysisEnum];

	case FlaimSolutionEnum(),
		analyses=[FlaimAnalysisEnum()];

	case SealevelriseSolutionEnum(),
		analyses=[SealevelriseAnalysisEnum()];

	case HydrologySolutionEnum(),
		analyses=[L2ProjectionBaseAnalysisEnum();HydrologyShreveAnalysisEnum();HydrologyDCInefficientAnalysisEnum();HydrologyDCEfficientAnalysisEnum()];
	
	case DamageEvolutionSolutionEnum(),
		analyses=[DamageEvolutionAnalysisEnum()];

	otherwise
		error('%s%s%s',' solution type: ',EnumToString(solutiontype),' not supported yet!');

	end
end % }}}
