/*!\file:  thermal.cpp
 * \brief: thermal solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "thermal"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

int main(int argc,char* *argv){

	int i,n;
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   qmu_analysis=0;
	int   numberofnodes;

	/*Model: */
	Model* model=NULL;

	/*initial velocity and pressure: */
	double* u_g=NULL;
	double* p_g=NULL;

	/*Results: */
	DataSet* results=NULL;
	Result*  result=NULL;
	
	ParameterInputs* inputs=NULL;
	Param*           param=NULL;
	double  dt;
	double  yts;

	int    waitonlock=0;
		
	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	/*Initialize model structure: */
	model=new Model();

	_printf_("read and create thermal finite element model:\n");
	model->AddFormulation(fid,ThermalAnalysisEnum());
	_printf_("read and create melting finite element model:\n");
	model->AddFormulation(fid,MeltingAnalysisEnum());

	/*recover parameters: */
	model->FindParam(&waitonlock,"waitonlock");
	model->FindParam(&qmu_analysis,"qmu_analysis");
	
	_printf_("initialize inputs:\n");
	model->FindParam(&u_g,"u_g",ThermalAnalysisEnum());
	model->FindParam(&p_g,"p_g",ThermalAnalysisEnum());
	model->FindParam(&numberofnodes,"numberofnodes");
	model->FindParam(&dt,"dt");
	model->FindParam(&yts,"yts");

	inputs=new ParameterInputs;

	inputs->Add("velocity",u_g,3,numberofnodes);
	inputs->Add("pressure",p_g,1,numberofnodes);
	inputs->Add("dt",dt*yts); //convert in seconds 

	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());

	/*are we running the solutoin sequence, or a qmu wrapper around it? : */
	if(!qmu_analysis){

		/*run thermal analysis: */
		_printf_("call computational core:\n");
		thermal_core(results,model,inputs);

	}
	else{
		/*run qmu analysis: */
		_printf_("calling qmu analysis on thermal core:\n");
	
		#ifdef _HAVE_DAKOTA_ 
		Qmux(model,inputs,ThermalAnalysisEnum(),NoneAnalysisEnum());
	 	#else
		throw ErrorException(__FUNCT__," Dakota not present, cannot do qmu!");
		#endif
	}

	/*Add analysis_type to results: */
	result=new Result(results->Size()+1,0,1,"analysis_type","thermal");
	results->AddObject(result);
		
	_printf_("process results:\n");
	ProcessResults(&results,model,ThermalAnalysisEnum());
	
	_printf_("write results to disk:\n");
	OutputResults(results,outputfilename);

	_printf_("write lock file:\n");
	if (waitonlock){
		WriteLockFile(lockname);
	}
		
	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 

	/*end module: */
	MODULEEND();

	/*Free ressources */
	xfree((void**)&u_g);
	xfree((void**)&p_g);
	delete model;
	delete results;
	delete inputs;
	
	return 0; //unix success return;
}
