/*!\file: thermal_core.cpp
 * \brief: core of the thermal solution 
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "thermal_core"

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./parallel.h"
#include "../issm.h"

void thermal_core(DataSet* results,Model* model, ParameterInputs* inputs){

	extern int my_rank;
	int i;

	/*fem models: */
	FemModel* fem_t=NULL;
	FemModel* fem_m=NULL;

	/*output: */
	Result* result=NULL;

	/*solutions vectors: */
	Vec*    t_g=NULL;
	Vec*    m_g=NULL;
	double* time=NULL;
	double* t_g_serial=NULL;
	double* m_g_serial=NULL;

	/*intermediary*/
	Vec Vect=NULL;
	Vec Vecm=NULL;

	/*flags: */
	int    debug=0;
	int    numberofdofspernode;
	int    numberofnodes;
	int    nsteps;
	double ndt;
	double dt;

	int    sub_analysis_type;
	double melting_offset;
	
	Param* param=NULL;

	/*recover fem models: */
	fem_t=model->GetFormulation(ThermalAnalysisEnum());
	fem_m=model->GetFormulation(MeltingAnalysisEnum());

	//first recover parameters common to all solutions
	fem_t->FindParam((void*)&numberofnodes,"numberofnodes");
	fem_t->FindParam((void*)&sub_analysis_type,"sub_analysis_type");
	fem_t->FindParam((void*)&debug,"debug");
	fem_t->FindParam((void*)&ndt,"ndt");
	fem_t->FindParam((void*)&dt,"dt");

	if(dt==0){

		time=(double*)xmalloc(sizeof(double));
		time[0]=0;

		/*allocate t_g and m_g arrays: */
		t_g=(Vec*)xmalloc(sizeof(Vec));
		m_g=(Vec*)xmalloc(sizeof(Vec));

		if(debug)_printf_("computing temperatures:\n");
		thermal_core_nonlinear(&t_g[0],&melting_offset,fem_t,inputs,ThermalAnalysisEnum(),NoneAnalysisEnum());
		inputs->Add("temperature",t_g[0],1,numberofnodes);
		inputs->Add("melting_offset",melting_offset);
		
		if(debug)_printf_("computing melting:\n");
		diagnostic_core_linear(&m_g[0],fem_m,inputs,MeltingAnalysisEnum(),NoneAnalysisEnum());
	}
	else{
		
		nsteps=(int)(ndt/dt);
		time=(double*)xmalloc((nsteps+1)*sizeof(double));
		time[0]=0;

		/*allocate t_g and m_g arrays: */
		t_g=(Vec*)xmalloc((nsteps+1)*sizeof(Vec));
		m_g=(Vec*)xmalloc((nsteps+1)*sizeof(Vec));

		//initialize temperature and melting
		fem_t->FindParam((void*)&t_g_serial,"t_g");
		t_g[0]=SerialToVec(t_g_serial,numberofnodes);
		xfree((void**)&t_g_serial);
		fem_m->FindParam((void*)&m_g_serial,"m_g");
		m_g[0]=SerialToVec(m_g_serial,numberofnodes);
		xfree((void**)&m_g_serial);

		for(i=0;i<nsteps;i++){
			if(debug)_printf_("time step: %i/%i\n",i+1,nsteps);
			time[i+1]=(i+1)*dt;
			
			if(debug)_printf_("computing temperatures:\n");
			inputs->Add("temperature",t_g[i],1,numberofnodes);
			thermal_core_nonlinear(&t_g[i+1],&melting_offset,fem_t,inputs,ThermalAnalysisEnum(),NoneAnalysisEnum());
			
			if(debug)_printf_("computing melting:\n");
			inputs->Add("temperature",t_g[i+1],1,numberofnodes);
			inputs->Add("melting_offset",melting_offset);
			diagnostic_core_linear(&m_g[i+1],fem_m,inputs,MeltingAnalysisEnum(),NoneAnalysisEnum());
		}
	}
	
	/*Plug results into output dataset: */
	if(dt==0){
		result=new Result(results->Size()+1,0,1,"t_g",t_g[0]);
		results->AddObject(result);
		
		result=new Result(results->Size()+1,0,1,"m_g",m_g[0]);
		results->AddObject(result);

		/*free ressource*/
		Vect=t_g[0];    Vecm=m_g[0];
		VecFree(&Vect); VecFree(&Vecm);
	}
	else{
		for(i=0;i<nsteps+1;i++){
			result=new Result(results->Size()+1,time[i],i+1,"t_g",t_g[i]);
			results->AddObject(result);

			result=new Result(results->Size()+1,time[i],i+1,"m_g",m_g[i]);
			results->AddObject(result);

			/*free ressource*/
			Vect=t_g[i];    Vecm=m_g[i];
			VecFree(&Vect); VecFree(&Vecm);
		}
	}
	/*free ressource*/
	xfree((void**)&t_g);
}
