/*!\file: thermal_core_nonlinear.cpp
 * \brief: core of the thermal solution 
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "thermal_core_nonlinear"

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../issm.h"

void thermal_core_nonlinear(Vec* ptg,double* pmelting_offset,FemModel* fem,ParameterInputs* inputs,int analysis_type,int sub_analysis_type){

	/*solution : */
	Vec tg=NULL; 
	Vec tf=NULL; 
	double melting_offset;

	/*intermediary: */
	Mat Kgg=NULL;
	Mat Kgg_nopenalty=NULL;
	Mat Kff=NULL;
	Mat Kfs=NULL;
	Vec pg=NULL;
	Vec pg_nopenalty=NULL;
	Vec pf=NULL;

	int converged;
	int constraints_converged;
	int num_unstable_constraints;
	int count;
	int numberofnodes;
	int min_thermal_constraints;

	/*parameters:*/
	int kflag,pflag,connectivity,numberofdofspernode;
	char* solver_string=NULL;
	int debug=0;
	int lowmem=0;

	/*Recover parameters: */
	kflag=1; pflag=1;

	fem->parameters->FindParam((void*)&connectivity,"connectivity");
	fem->parameters->FindParam((void*)&numberofdofspernode,"numberofdofspernode");
	fem->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
	fem->parameters->FindParam((void*)&solver_string,"solverstring");
	fem->parameters->FindParam((void*)&debug,"debug");
	fem->parameters->FindParam((void*)&lowmem,"lowmem");
	fem->parameters->FindParam((void*)&min_thermal_constraints,"min_thermal_constraints");

	count=1;
	converged=0;

	for(;;){

		if(debug)_printf_("%s\n","starting direct shooting method");

		/*Update parameters: */
		UpdateFromInputsx(fem->elements,fem->nodes,fem->loads, fem->materials,inputs);

		//*Generate system matrices
		if (!lowmem){

			/*Compute Kgg_nopenalty and pg_nopenalty once for all: */
			if (count==1){
				SystemMatricesx(&Kgg_nopenalty, &pg_nopenalty,fem->elements,fem->nodes,fem->loads,fem->materials,kflag,pflag,connectivity,numberofdofspernode,inputs,analysis_type,sub_analysis_type); 
			}

			/*Copy K_gg_nopenalty into Kgg, same for pg: */
			Kgg=(Mat)xmalloc(sizeof(Mat));
			MatDuplicate(Kgg_nopenalty,MAT_COPY_VALUES,&Kgg);
			pg=(Vec)xmalloc(sizeof(Vec));
			VecDuplicate(pg_nopenalty,&pg);VecCopy(pg_nopenalty,pg);

			//apply penalties each time
			PenaltySystemMatricesx(Kgg, pg,&melting_offset,fem->elements,fem->nodes,fem->loads,fem->materials,kflag,pflag,inputs,analysis_type,sub_analysis_type); 
		}
		else{
			SystemMatricesx(&Kgg, &pg,fem->elements,fem->nodes,fem->loads,fem->materials,kflag,pflag,connectivity,numberofdofspernode,inputs,analysis_type,sub_analysis_type); 
			//apply penalties
			PenaltySystemMatricesx(Kgg, pg,&melting_offset,fem->elements,fem->nodes,fem->loads,fem->materials,kflag,pflag,inputs,analysis_type,sub_analysis_type); 
		}

		/*!Reduce matrix from g to f size:*/
		Reducematrixfromgtofx(&Kff,&Kfs,Kgg,fem->Gmn,fem->nodesets);

		/*Free ressources: */
		MatFree(&Kgg);
	
		if (debug) _printf_("   reducing load from g to f set\n");
		/*!Reduce load from g to f size: */
		Reduceloadfromgtofx(&pf, pg, fem->Gmn, Kfs, fem->ys, fem->nodesets);

		//no need for pg and Kfs anymore 
		VecFree(&pg); 
		MatFree(&Kfs);

		/*Solve: */
		if(debug)_printf_("%s\n","solving");
		Solverx(&tf, Kff, pf, tf, solver_string);
	
		//no need for Kff and pf anymore
		MatFree(&Kff);VecFree(&pf);

		if (debug) _printf_("   merging solution from f to g set\n");
		//Merge back to g set
		Mergesolutionfromftogx(&tg, tf,fem->Gmn,fem->ys,fem->nodesets);

		//Deal with penalty loads
		if (debug) _printf_("   penalty constraints\n");
		inputs->Add("temperature",tg,numberofdofspernode,numberofnodes);
		
		PenaltyConstraintsx(&constraints_converged, &num_unstable_constraints, fem->elements,fem->nodes,fem->loads,fem->materials,inputs,analysis_type,sub_analysis_type); 

		if (!converged){
			if(debug)_printf_("%s%i\n","   #unstable constraints = ",num_unstable_constraints);
			if (num_unstable_constraints <= min_thermal_constraints)converged=1;
		}
		count++;
		
		if(converged==1)break;
	}

	/*Free ressources: */
	MatFree(&Kgg_nopenalty);
	VecFree(&pg_nopenalty);
	VecFree(&tf);

	/*Assign output pointers: */
	*ptg=tg;
	*pmelting_offset=melting_offset;
}
