/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        OrthogonalPolynomial
//- Description:  Abstract base class for orthogonal polynomials
//-               
//- Owner:        Mike Eldred, Sandia National Laboratories

#ifndef ORTHOGONAL_POLYNOMIAL_H
#define ORTHOGONAL_POLYNOMIAL_H

#include "BasisPolynomial.H"
#include "data_types.h"

namespace Dakota {


/// Base class for the orthogonal polynomial class hierarchy.

/** The OrthogonalPolynomial class is the base class for the
    univariate orthogonal polynomial class hierarchy in DAKOTA.  One
    instance of an OrthogonalPolynomial is created for each variable
    within a multidimensional orthogonal polynomial basis function (a
    vector of OrthogonalPolynomials is contained in
    OrthogPolyApproximation, which may be mixed and matched in, e.g.,
    the Wiener-Askey scheme for polynomial chaos). */

class OrthogonalPolynomial: public BasisPolynomial
{
public:

  //
  //- Heading: Constructors, destructor, assignment operator
  //

  OrthogonalPolynomial();  /// default constructor
  ~OrthogonalPolynomial(); /// destructor

  //
  //- Heading: Virtual function redefinitions
  //

  /// destroy history of Gauss pts/wts due to change in alpha/beta stats
  void reset_gauss();

  //
  //- Heading: Member functions
  //

  /// perform unit testing on the Gauss points/weights
  void gauss_check(unsigned short order);

protected:

  //
  //- Heading: Data
  //

  /// norm-squared of the n_th order polynomial defined by the inner product
  /// <Poly_n, Poly_n> = ||Poly_n||^2 (returned by norm_squared())
  Real orthogPolyNormSq;

  /// Gauss points for one-dimensional Gaussian quadrature
  /// (x parameter values for which Poly_n(x) = 0)
  RealVector gaussPoints;
  /// Gauss weights for one-dimensional Gaussian quadrature
  RealVector gaussWeights;

private:

  //
  //- Heading: Data
  //
};


inline OrthogonalPolynomial::OrthogonalPolynomial():
  BasisPolynomial(BaseConstructor())
{ }


inline OrthogonalPolynomial::~OrthogonalPolynomial()
{ }


inline void OrthogonalPolynomial::reset_gauss()
{ gaussPoints.clear(); gaussWeights.clear(); }

} // namespace Dakota

#endif
