/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        NIDRProblemDescDB
//- Description:  Problem description database populated by new IDR parser
//- Owner:        David M. Gay

#ifndef NIDR_PROBLEM_DESC_DB_H
#define NIDR_PROBLEM_DESC_DB_H

#include "ProblemDescDB.H"
#include "nidr.h"

namespace Dakota {

class ParallelLibrary;

/// The derived input file database utilizing the new IDR parser.

/** The NIDRProblemDescDB class is derived from ProblemDescDB for use
    by the NIDR parser in processing DAKOTA input file data.
    For information on modifying the NIDR input parsing procedures,
    refer to Dakota/docs/Dev_Spec_Change.dox.  For more on the parsing
    technology, see "Specifying and Reading Program Input with NIDR"
    by David M. Gay (report SAND2008-2261P, which is available in PDF
    form as http://www.sandia.gov/~dmgay/nidr08.pdf).  Source for the
    routines declared herein is NIDRProblemDescDB.C, in which most
    routines are so short that a description seems unnecessary. */

class NIDRProblemDescDB: public ProblemDescDB
{
public:

  //
  //- Heading: Constructors and Destructor
  //

  /// constructor
  NIDRProblemDescDB(ParallelLibrary& parallel_lib, const char *parser);
  /// destructor
  ~NIDRProblemDescDB();

  //
  //- Heading: Member methods
  //

  /// parses the input file and populates the problem description
  /// database using NIDR.
  void derived_manage_inputs(const char* dakota_input_file);

private:

  //
  //- Heading: Data
  //

  /// pointer to the active object instance used within the static kwhandler
  /// functions in order to avoid the need for static data
  static NIDRProblemDescDB* pDDBInstance;
  List<void*> VIL;
  static void var_stop1(void*);

public:

  const char *parser;
  static int nerr;
  static void Var_boundchk(DataVariablesRep*);
  static void Var_boundchku(DataVariablesRep*); // Temporary until Var_bhck and Var_bchku
						// are again merged.
  static void Var_iboundchk(DataVariablesRep*);
  static void botch(const char *fmt, ...);
  static void post_process_response_data(DataResponses*);
  static void post_process_variables(List<DataVariables>*);
  static void squawk(const char *fmt, ...);
  static void warn(const char *fmt,...);

  //
  //- Heading: Keyword handlers called by nidr
  //

#define KWH(x) static void x(const char *keyname, Values *val, void **g, void *v)

  KWH(iface_Rlit);
  KWH(iface_false);
  KWH(iface_ilit);
  KWH(iface_int);
  KWH(iface_lit);
  KWH(iface_start);
  KWH(iface_stop);
  KWH(iface_str);
  KWH(iface_str2D);
  KWH(iface_strL);
  KWH(iface_true);

  KWH(method_Ii);
  KWH(method_Real);
  KWH(method_Real01);
  KWH(method_RealL);
  KWH(method_RealLlit);
  KWH(method_Realp);
  KWH(method_Realz);
  KWH(method_Ri);
  KWH(method_coliny_ea);
  KWH(method_false);
  KWH(method_ilit2);
  KWH(method_ilit2p);
  KWH(method_int);
  KWH(method_intL);
  KWH(method_intLa);
  KWH(method_lit);
  KWH(method_lit2);
  KWH(method_litc);
  KWH(method_liti);
  KWH(method_litp);
  KWH(method_litpp);
  KWH(method_litpp_final);
  KWH(method_litr);
  KWH(method_litz);
  KWH(method_meritFn);
  KWH(method_moga_begin);
  KWH(method_moga_final);
  KWH(method_nnint);
  KWH(method_nnintz);
  KWH(method_num_resplevs);
  KWH(method_pint);
  KWH(method_pintz);
  KWH(method_resplevs);
  KWH(method_resplevs01);
  KWH(method_shint);
  KWH(method_slit);
  KWH(method_slit2);
  KWH(method_soga_begin);
  KWH(method_soga_final);
  KWH(method_start);
  KWH(method_stop);
  KWH(method_str);
  KWH(method_strL);
  KWH(method_true);
  KWH(method_tr_final);
  KWH(method_ushint);
  KWH(method_ushintL);

  KWH(model_Real);
  KWH(model_RealL);
  KWH(model_intset);
  KWH(model_lit);
  KWH(model_order);
  KWH(model_shint);
  KWH(model_slit2);
  KWH(model_start);
  KWH(model_stop);
  KWH(model_str);
  KWH(model_strL);
  KWH(model_true);

  KWH(resp_RealL);
  KWH(resp_intL);
  KWH(resp_lit);
  KWH(resp_nnintz);
  KWH(resp_start);
  KWH(resp_stop);
  KWH(resp_str);
  KWH(resp_strL);

  KWH(strategy_Real);
  KWH(strategy_RealL);
  KWH(strategy_int);
  KWH(strategy_lit);
  KWH(strategy_slit);
  KWH(strategy_str);
  KWH(strategy_strL);
  KWH(strategy_true);
  KWH(strategy_start);

  KWH(var_RealL);
  KWH(var_RealLb);
  KWH(var_RealLd);
  KWH(var_RealUb);
  KWH(var_hbp);
  KWH(var_intL);
  KWH(var_intz);
  KWH(var_nbp);
  KWH(var_nhbp);
  KWH(var_start);
  KWH(var_stop);
  KWH(var_str);
  KWH(var_strL);
  KWH(var_ulbl);

#undef KWH
};

} // namespace Dakota

#endif
