/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       MergedVariables
//- Description: Container class of variables employing the "merged" approach
//-              to variables usage.
//- Owner:       Mike Eldred
//- Version: $Id: MergedVariables.H 5337 2008-10-09 18:12:05Z mseldre $

#ifndef MERGED_VARIABLES_H
#define MERGED_VARIABLES_H

#include "DakotaVariables.H"
#include "DataVariables.H"


namespace Dakota {

/// Derived class within the Variables hierarchy which employs the
/// merged data view.

/** Derived variables classes take different views of the design,
    uncertain, and state variable types and the continuous and
    discrete domain types.  The MergedVariables derived class combines
    continuous and discrete domain types but separates design,
    uncertain, and state variable types.  The result is a single
    continuous array of design variables (mergedDesignVars), a single
    continuous array of uncertain variables (uncertainVars), and a
    single continuous array of state variables (mergedStateVars).  The
    branch and bound strategy uses this approach (see
    Variables::get_variables(problem_db)). */

class MergedVariables: public Variables
{
  //
  //- Heading: Friends
  //

  /// equality operator
  friend bool operator==(const MergedVariables& vars1,
			 const MergedVariables& vars2);
#ifdef DAKOTA_BOOST
  /// hash_value
  friend std::size_t hash_value(const MergedVariables& vars);

  /// binary_equal_to (since 'operator==' is not suitable for boost/hash_set)
  friend bool binary_equal_to(const MergedVariables& vars1,
                              const MergedVariables& vars2);
#endif

public:

  //
  //- Heading: Constructors and destructor
  //

  /// default constructor
  MergedVariables();
  /// standard constructor
  MergedVariables(const ProblemDescDB& problem_db,
		  const pair<short,short>& view);
  /// destructor
  ~MergedVariables();

  //
  //- Heading: Virtual function redefinitions
  //

  size_t tv() const;
  //size_t cv() const;
  //size_t dv() const;

  const UIntArray& merged_discrete_ids() const;

  const RealVector& continuous_variables() const;
  void continuous_variable(const Real& c_var, const size_t& i);
  void continuous_variables(const RealVector& c_vars);
  //const IntVector& discrete_variables() const;
  //void discrete_variable(const int& d_var, const size_t& i);
  //void discrete_variables(const IntVector& d_vars);

  const StringArray& continuous_variable_labels() const;
  void continuous_variable_labels(const StringArray& c_v_labels);
  //const StringArray& discrete_variable_labels()   const;
  //void discrete_variable_labels(const StringArray& d_v_labels);

  const StringArray& continuous_variable_types() const;
  //const StringArray& discrete_variable_types() const;

  const UIntArray& continuous_variable_ids() const;

  //size_t icv() const;
  //size_t idv() const;

  const RealVector& inactive_continuous_variables() const;
  void inactive_continuous_variables(const RealVector& i_c_vars);
  //const IntVector& inactive_discrete_variables() const;
  //void inactive_discrete_variables(const IntVector& i_d_vars);

  const StringArray& inactive_continuous_variable_labels() const;
  void inactive_continuous_variable_labels(const StringArray& i_c_v_labels);
  //const StringArray& inactive_discrete_variable_labels() const;
  //void inactive_discrete_variable_labels(const StringArray& i_d_v_labels);

  const UIntArray& inactive_continuous_variable_ids() const;

  size_t acv() const;
  //size_t adv() const;

  RealVector all_continuous_variables() const;
  void all_continuous_variables(const RealVector& a_c_vars);
  //IntVector all_discrete_variables()   const;
  //void all_discrete_variables(const IntVector& a_d_vars);

  StringArray all_continuous_variable_labels() const;
  void all_continuous_variable_labels(const StringArray& a_c_v_labels);
  //StringArray all_discrete_variable_labels()   const;
  //void all_discrete_variable_labels(const StringArray& a_d_v_labels);
  StringArray all_variable_labels()            const;

  //const StringArray& all_discrete_variable_types() const;

  void read(istream& s);
  void write(ostream& s) const;
  void write_aprepro(ostream& s) const;

  void read_annotated(istream& s);
  void write_annotated(ostream& s) const;

  void write_tabular(ostream& s) const;

  void read(BiStream& s);
  void write(BoStream& s) const;

  void read(MPIUnpackBuffer& s);
  void write(MPIPackBuffer& s) const;

protected:

  void copy_rep(const Variables* vars_rep);

  void reshape_rep(const Sizet2DArray& vars_comps);

private:

  //
  //- Heading: Private member functions
  //

  /// construct VarTypes and VarIds arrays using variablesComponents
  void build_types_ids();

  //
  //- Heading: Private data members
  //

  /// a design variables array merging continuous and discrete
  /// domains (discrete values promoted to continuous)
  RealVector mergedDesignVars;
  /// the uncertain variables array (no discrete uncertain to merge)
  RealVector uncertainVars;
  /// a state variables array merging continuous and discrete
  /// domains (discrete values promoted to continuous)
  RealVector mergedStateVars;

  /// a label array combining continuous and discrete design labels
  StringArray mergedDesignLabels;
  /// the uncertain variables label array (no discrete uncertain to combine)
  StringArray uncertainLabels;
  /// a label array combining continuous and discrete state labels
  StringArray mergedStateLabels;

  /// array of variable types for the active continuous variables
  StringArray continuousVarTypes;

  /// array of position identifiers for the active continuous variables
  /** These identifiers define positions of the active continuous variables
      within the total variable sequence. */
  UIntArray continuousVarIds;
  /// array of position identifiers for the inactive continuous variables
  /** These identifiers define positions of the inactive continuous variables
      within the total variable sequence. */
  UIntArray inactiveContinuousVarIds;
  /// array of discrete variable identifiers for which the discrete
  /// requirement is relaxed by merging them into a continuous array
  UIntArray mergedDiscreteIds;
};


inline MergedVariables::MergedVariables()
{ }


inline MergedVariables::~MergedVariables()
{ }


inline size_t MergedVariables::tv() const
{
  return mergedDesignVars.length() + uncertainVars.length() 
    + mergedStateVars.length();
}


inline const UIntArray& MergedVariables::merged_discrete_ids() const
{ return mergedDiscreteIds; }


inline const RealVector& MergedVariables::continuous_variables() const
{
  switch (variablesView.first) {
  case MERGED_DISTINCT_DESIGN:
    return mergedDesignVars; break;
  case MERGED_DISTINCT_UNCERTAIN:
    return uncertainVars;    break;
  case MERGED_DISTINCT_STATE:
    return mergedStateVars;  break;
  }
  return emptyRealVector;
}


inline void MergedVariables::
continuous_variable(const Real& c_var, const size_t& i)
{
  switch (variablesView.first) {
  case MERGED_DISTINCT_DESIGN:
    mergedDesignVars[i] = c_var; break;
  case MERGED_DISTINCT_UNCERTAIN:
    uncertainVars[i]    = c_var; break;
  case MERGED_DISTINCT_STATE:
    mergedStateVars[i]  = c_var; break;
  }
}


inline void MergedVariables::continuous_variables(const RealVector& c_vars)
{
  switch (variablesView.first) {
  case MERGED_DISTINCT_DESIGN:
    mergedDesignVars = c_vars; break;
  case MERGED_DISTINCT_UNCERTAIN:
    uncertainVars    = c_vars; break;
  case MERGED_DISTINCT_STATE:
    mergedStateVars  = c_vars; break;
  }
}


inline const StringArray& MergedVariables::continuous_variable_labels() const
{
  switch (variablesView.first) {
  case MERGED_DISTINCT_DESIGN:
    return mergedDesignLabels; break;
  case MERGED_DISTINCT_UNCERTAIN:
    return uncertainLabels;    break;
  case MERGED_DISTINCT_STATE:
    return mergedStateLabels;  break;
  }
  return emptyStringArray;
}


inline void MergedVariables::
continuous_variable_labels(const StringArray& c_v_labels)
{
  switch (variablesView.first) {
  case MERGED_DISTINCT_DESIGN:
    mergedDesignLabels = c_v_labels; break;
  case MERGED_DISTINCT_UNCERTAIN:
    uncertainLabels    = c_v_labels; break;
  case MERGED_DISTINCT_STATE:
    mergedStateLabels  = c_v_labels; break;
  }
}


inline const StringArray& MergedVariables::continuous_variable_types() const
{ return continuousVarTypes; }


inline const UIntArray& MergedVariables::continuous_variable_ids() const
{ return continuousVarIds; }


inline const RealVector& MergedVariables::inactive_continuous_variables() const
{
  switch (variablesView.second) {
  case MERGED_DISTINCT_DESIGN:
    return mergedDesignVars; break;
  case MERGED_DISTINCT_UNCERTAIN:
    return uncertainVars;    break;
  case MERGED_DISTINCT_STATE:
    return mergedStateVars;  break;
  }
  return emptyRealVector;
}


inline void MergedVariables::
inactive_continuous_variables(const RealVector& i_c_vars)
{
  switch (variablesView.second) {
  case MERGED_DISTINCT_DESIGN:
    mergedDesignVars = i_c_vars; break;
  case MERGED_DISTINCT_UNCERTAIN:
    uncertainVars    = i_c_vars; break;
  case MERGED_DISTINCT_STATE:
    mergedStateVars  = i_c_vars; break;
  }
}


inline const StringArray& MergedVariables::
inactive_continuous_variable_labels() const
{
  switch (variablesView.second) {
  case MERGED_DISTINCT_DESIGN:
    return mergedDesignLabels; break;
  case MERGED_DISTINCT_UNCERTAIN:
    return uncertainLabels;    break;
  case MERGED_DISTINCT_STATE:
    return mergedStateLabels;  break;
  }
  return emptyStringArray;
}


inline void MergedVariables::
inactive_continuous_variable_labels(const StringArray& i_c_v_labels)
{
  switch (variablesView.second) {
  case MERGED_DISTINCT_DESIGN:
    mergedDesignLabels = i_c_v_labels; break;
  case MERGED_DISTINCT_UNCERTAIN:
    uncertainLabels    = i_c_v_labels; break;
  case MERGED_DISTINCT_STATE:
    mergedStateLabels  = i_c_v_labels; break;
  }
}


inline const UIntArray& MergedVariables::
inactive_continuous_variable_ids() const
{ return inactiveContinuousVarIds; }


inline size_t MergedVariables::acv() const
{
  return mergedDesignVars.length() + uncertainVars.length()
    + mergedStateVars.length();
}


inline RealVector MergedVariables::all_continuous_variables() const
{
  RealVector all_c_vars;
  aggregate_data(mergedDesignVars, uncertainVars, mergedStateVars, all_c_vars);
  return all_c_vars;
}


inline void MergedVariables::
all_continuous_variables(const RealVector& a_c_vars)
{ separate_data(a_c_vars, mergedDesignVars, uncertainVars, mergedStateVars); }


inline StringArray MergedVariables::all_continuous_variable_labels() const
{
  StringArray a_c_v_labels;
  aggregate_data(mergedDesignLabels, uncertainLabels, mergedStateLabels,
		 a_c_v_labels);
  return a_c_v_labels;
}


inline void MergedVariables::
all_continuous_variable_labels(const StringArray& a_c_v_labels)
{
  separate_data(a_c_v_labels, mergedDesignLabels, uncertainLabels,
		mergedStateLabels);
}


inline StringArray MergedVariables::all_variable_labels() const
{
  StringArray a_v_labels;
  aggregate_data(mergedDesignLabels, uncertainLabels, mergedStateLabels,
		 a_v_labels);
  return a_v_labels;
}


#ifdef DAKOTA_BOOST
/// binary_equal_to for MergedVariables -- NOTE: no tolerance is used!
inline bool binary_equal_to(const MergedVariables& vars1,
                            const MergedVariables& vars2)
{
  // Check for equality in array lengths
  size_t mdv_size = vars1.mergedDesignVars.size();
  size_t uv_size  = vars1.uncertainVars.size();
  size_t msv_size = vars1.mergedStateVars.size();
  if (vars2.mergedDesignVars.size() != mdv_size ||
      vars2.uncertainVars.size()    != uv_size  ||
      vars2.mergedStateVars.size()  != msv_size)
    return false;

  // Check each value (labels are ignored!)
  size_t i;
  for (i=0; i<mdv_size; ++i) {
    if ( vars2.mergedDesignVars[i] != vars1.mergedDesignVars[i] )
      return false;
  }

  for (i=0; i<uv_size; ++i) {
    if ( vars2.uncertainVars[i] != vars1.uncertainVars[i] )
      return false;
  }

  for (i=0; i<msv_size; ++i) {
    if ( vars2.mergedStateVars[i] != vars1.mergedStateVars[i] )
      return false;
  }

  return true;
}

/// hash_value for MergedVariables
inline std::size_t hash_value(const MergedVariables& vars)
{
  std::size_t seed = 0;
  boost::hash_combine(seed, vars.mergedDesignVars);
  boost::hash_combine(seed, vars.uncertainVars);
  boost::hash_combine(seed, vars.mergedStateVars);

  return seed;
}
#endif  // DAKOTA_BOOST

} // namespace Dakota

#endif
