/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        MergedVariables
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "MergedVariables.H"
#include "ProblemDescDB.H"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

/** In this class, a merged data approach is used in which continuous
    and discrete arrays are combined into a single continuous array
    (integrality is relaxed; the converse of truncating reals is not
    currently supported but could be in the future if needed).
    Iterators/strategies which use this class include:
    BranchBndOptimizer.  Extract fundamental variable types and labels
    and merge continuous and discrete domains to create mergedDesignVars,
    mergedStateVars, mergedDesignLabels, and mergedStateLabels.  */
MergedVariables::
MergedVariables(const ProblemDescDB& problem_db, const pair<short,short>& view):
  Variables(BaseConstructor(), problem_db, view),
  uncertainVars(problem_db.get_drv("variables.uncertain.initial_point")),
  uncertainLabels(problem_db.get_dsa("variables.uncertain.labels"))
{
  // Get fundamental arrays from problem_db
  const RealVector& continuous_des_vars
    = problem_db.get_drv("variables.continuous_design.initial_point");
  const IntVector& discrete_des_vars
    = problem_db.get_div("variables.discrete_design.initial_point");
  const RealVector& continuous_state_vars
    = problem_db.get_drv("variables.continuous_state.initial_state");
  const IntVector& discrete_state_vars
    = problem_db.get_div("variables.discrete_state.initial_state");
  const StringArray& cdv_labels
    = problem_db.get_dsa("variables.continuous_design.labels");
  const StringArray& ddv_labels
    = problem_db.get_dsa("variables.discrete_design.labels");
  const StringArray& csv_labels
    = problem_db.get_dsa("variables.continuous_state.labels");
  const StringArray& dsv_labels
    = problem_db.get_dsa("variables.discrete_state.labels");

  // Initialize Merged arrays
  aggregate_merged(continuous_des_vars, discrete_des_vars, mergedDesignVars);
  aggregate_merged(continuous_state_vars, discrete_state_vars, mergedStateVars);
  aggregate_data(cdv_labels, ddv_labels, mergedDesignLabels);
  aggregate_data(csv_labels, dsv_labels, mergedStateLabels);

  // construct continuousVarTypes/discreteVarTypes and
  // continuousVarIds/inactiveContinuousVarIds/allContinuousVarIds
  build_types_ids();

#ifdef REFCOUNT_DEBUG
  Cout << "Letter instantiated: variablesView active = " << variablesView.first
       << " inactive = " << variablesView.second << endl;
#endif
}


void MergedVariables::reshape_rep(const Sizet2DArray& vars_comps)
{
  size_t num_mdv = vars_comps[0][0], num_uv = vars_comps[1][0],
         num_msv = vars_comps[2][0];

  mergedDesignVars.reshape(num_mdv);
  uncertainVars.reshape(num_uv);
  mergedStateVars.reshape(num_msv);

  mergedDesignLabels.reshape(num_mdv);
  uncertainLabels.reshape(num_uv);
  mergedStateLabels.reshape(num_msv);

  build_types_ids();
}


void MergedVariables::build_types_ids()
{
  size_t i, offset, acv_cntr = 0, num_cdv = variablesComponents[0][1],
    num_ddv = variablesComponents[0][2], num_mdv  = mergedDesignVars.length(),
    num_uv  = uncertainVars.length(),    num_csv  = variablesComponents[2][1],
    num_dsv = variablesComponents[2][2], num_msv  = mergedStateVars.length(),
    num_acv = num_mdv + num_uv + num_msv;

  allContinuousVarIds.reshape(num_acv);
  allContinuousVarTypes.reshape(num_acv);
  for (i=0; i<num_cdv; i++) {
    allContinuousVarTypes[acv_cntr] = "continuous_design";
    allContinuousVarIds[acv_cntr++] = i + 1;
  }
  offset = num_cdv + 1;
  for (i=0; i<num_ddv; i++) {
    allContinuousVarTypes[acv_cntr] = "discrete_design";
    allContinuousVarIds[acv_cntr++] = i + offset;
  }
  uncertain_var_types(acv_cntr);
  offset += num_ddv;
  for (i=0; i<num_uv; i++)
    allContinuousVarIds[i + num_mdv] = i + offset;
  offset += num_uv;
  for (i=0; i<num_csv; i++) {
    allContinuousVarTypes[acv_cntr] = "continuous_state";
    allContinuousVarIds[acv_cntr++] = i + offset;
  }
  offset += num_csv;
  for (i=0; i<num_dsv; i++) {
    allContinuousVarTypes[acv_cntr] = "discrete_state";
    allContinuousVarIds[acv_cntr++] = i + offset;
  }

  // Don't bleed over any logic about supported view combinations.
  // Rather, keep this class general and encapsulated.

  // Initialize continuousVarTypes/discreteVarTypes and
  // continuousVarIds/mergedDiscreteIds
  if (variablesView.first == MIXED_DISTINCT_DESIGN) {
    copy_data_partial(allContinuousVarTypes, 0, num_mdv, continuousVarTypes);
    copy_data_partial(allContinuousVarIds,   0, num_mdv, continuousVarIds);
    mergedDiscreteIds.reshape(num_ddv);
    offset = num_cdv + 1;
    for (i=0; i<num_ddv; i++)
      mergedDiscreteIds[i] = i + offset;
  }
  else if (variablesView.first == MIXED_DISTINCT_UNCERTAIN) {
    copy_data_partial(allContinuousVarTypes, num_mdv, num_uv,
		      continuousVarTypes);
    copy_data_partial(allContinuousVarIds,   num_mdv, num_uv, continuousVarIds);
  }
  else if (variablesView.first == MIXED_DISTINCT_STATE) {
    copy_data_partial(allContinuousVarTypes, num_mdv + num_uv, num_msv,
		      continuousVarTypes);
    copy_data_partial(allContinuousVarIds, num_mdv + num_uv, num_msv,
		      continuousVarIds);
    mergedDiscreteIds.reshape(num_dsv);
    offset = num_mdv + num_uv + num_csv + 1;
    for (i=0; i<num_dsv; i++)
      mergedDiscreteIds[i] = i + offset;
  }

  // Initialize inactiveContinuousVarIds
  if (variablesView.second == MIXED_DISTINCT_DESIGN)
    copy_data_partial(allContinuousVarIds, 0, num_mdv,
		      inactiveContinuousVarIds);
  else if (variablesView.second == MIXED_DISTINCT_UNCERTAIN)
    copy_data_partial(allContinuousVarIds, num_mdv, num_uv,
		      inactiveContinuousVarIds);
  else if (variablesView.second == MIXED_DISTINCT_STATE)
    copy_data_partial(allContinuousVarIds, num_mdv + num_uv, num_msv,
		      inactiveContinuousVarIds);
}


void MergedVariables::read(istream& s)
{
  mergedDesignVars.read(s, mergedDesignLabels);
  uncertainVars.read(s, uncertainLabels);
  mergedStateVars.read(s, mergedStateLabels);
}


void MergedVariables::write(ostream& s) const
{
  mergedDesignVars.write(s, mergedDesignLabels);
  uncertainVars.write(s, uncertainLabels);
  mergedStateVars.write(s, mergedStateLabels);
}


void MergedVariables::write_aprepro(ostream& s) const
{
  mergedDesignVars.write_aprepro(s, mergedDesignLabels);
  uncertainVars.write_aprepro(s, uncertainLabels);
  mergedStateVars.write_aprepro(s, mergedStateLabels);
}


void MergedVariables::read_annotated(istream& s)
{
  mergedDesignVars.read_annotated(s, mergedDesignLabels);
  uncertainVars.read_annotated(s, uncertainLabels);
  mergedStateVars.read_annotated(s, mergedStateLabels);
  // types/ids not required
}


void MergedVariables::write_annotated(ostream& s) const
{
  mergedDesignVars.write_annotated(s, mergedDesignLabels);
  uncertainVars.write_annotated(s, uncertainLabels);
  mergedStateVars.write_annotated(s, mergedStateLabels);
  // types/ids not required
}


void MergedVariables::write_tabular(ostream& s) const
{
  mergedDesignVars.write_tabular(s);
  uncertainVars.write_tabular(s);
  mergedStateVars.write_tabular(s);
}


void MergedVariables::read(BiStream& s)
{
  mergedDesignVars.read(s, mergedDesignLabels);
  uncertainVars.read(s, uncertainLabels);
  mergedStateVars.read(s, mergedStateLabels);
  // types/ids not required
}


void MergedVariables::write(BoStream& s) const
{
  mergedDesignVars.write(s, mergedDesignLabels);
  uncertainVars.write(s, uncertainLabels);
  mergedStateVars.write(s, mergedStateLabels);
  // types/ids not required
}


void MergedVariables::read(MPIUnpackBuffer& s)
{
  mergedDesignVars.read(s, mergedDesignLabels);
  uncertainVars.read(s, uncertainLabels);
  mergedStateVars.read(s, mergedStateLabels);
  // types/ids not required
}


void MergedVariables::write(MPIPackBuffer& s) const
{
  mergedDesignVars.write(s, mergedDesignLabels);
  uncertainVars.write(s, uncertainLabels);
  mergedStateVars.write(s, mergedStateLabels);
  // types/ids not required
}


void MergedVariables::copy_rep(const Variables* vars_rep)
{
  // copy derived class attributes
  mergedDesignVars   = ((MergedVariables*)vars_rep)->mergedDesignVars;
  uncertainVars      = ((MergedVariables*)vars_rep)->uncertainVars;
  mergedStateVars    = ((MergedVariables*)vars_rep)->mergedStateVars;

  mergedDesignLabels = ((MergedVariables*)vars_rep)->mergedDesignLabels;
  uncertainLabels    = ((MergedVariables*)vars_rep)->uncertainLabels;
  mergedStateLabels  = ((MergedVariables*)vars_rep)->mergedStateLabels;

  continuousVarTypes = ((MergedVariables*)vars_rep)->continuousVarTypes;
  continuousVarIds   = ((MergedVariables*)vars_rep)->continuousVarIds;
  inactiveContinuousVarIds
    = ((MergedVariables*)vars_rep)->inactiveContinuousVarIds;
  if (variablesView.first != MERGED_DISTINCT_UNCERTAIN)
    mergedDiscreteIds = ((MergedVariables*)vars_rep)->mergedDiscreteIds;
}


/// equality operator for MergedVariables
bool operator==(const MergedVariables& vars1, const MergedVariables& vars2)
{
  return (vars2.mergedDesignVars == vars1.mergedDesignVars &&
	  vars2.uncertainVars    == vars1.uncertainVars    &&
	  vars2.mergedStateVars  == vars1.mergedStateVars);
  // labels/types/ids not required
}

} // namespace Dakota
