/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        LegendreOrthogPolynomial
//- Description:  Class for Legendre Orthogonal Polynomial
//-               
//- Owner:        Mike Eldred, Sandia National Laboratories

#ifndef LEGENDRE_ORTHOG_POLYNOMIAL_H
#define LEGENDRE_ORTHOG_POLYNOMIAL_H

#include "OrthogonalPolynomial.H"


namespace Dakota {

/// Derived orthogonal polynomial class for Legendre polynomials

/** The LegendreOrthogPolynomial class evaluates a univariate Legendre
    polynomial of a particular order.  These polynomials are
    orthogonal with respect to the weight function 1 when integrated
    over the support range of [-1,+1].  This corresponds to the
    probability density function f(x) = 1/(U-L) = 1/2 for the uniform
    distribution for [L,U]=[-1,1].  It enables (mixed)
    multidimensional orthogonal polynomial basis functions within
    OrthogPolyApproximation.  Legendre polynomials are a special case
    (alpha = beta = 0) of the more general Jacobi polynomials
    (implemented separately) which correspond to the beta distribution. */

class LegendreOrthogPolynomial: public OrthogonalPolynomial
{
public:

  //
  //- Heading: Constructor and destructor
  //

  LegendreOrthogPolynomial();  ///< default constructor
  ~LegendreOrthogPolynomial(); ///< destructor

protected:

  //
  //- Heading: Virtual function redefinitions
  //

  /// retrieve the Legendre polynomial value for a given parameter x 
  const Real& get_value(const Real& x, unsigned short order);
  /// retrieve the Legendre polynomial gradient for a given parameter x 
  const Real& get_gradient(const Real& x, unsigned short order);

  /// return the inner product <P_n,P_n> = ||P_n||^2
  const Real& norm_squared(unsigned short order);

  /// return the Gauss-Legendre quadrature points corresponding to
  /// polynomial order n
  const RealVector& gauss_points(unsigned short order);
  /// return the Gauss-Legendre quadrature weights corresponding to
  /// polynomial order n
  const RealVector& gauss_weights(unsigned short order);

private:

  //
  //- Heading: Data
  //

};


inline LegendreOrthogPolynomial::LegendreOrthogPolynomial()
{ }


inline LegendreOrthogPolynomial::~LegendreOrthogPolynomial()
{ }

} // namespace Dakota

#endif
