/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        LagrangeInterpPolynomial
//- Description:  Class for 1-D Lagrange Interpolation Polynomials
//-               
//- Owner:        Mike Eldred, Sandia National Laboratories

#ifndef LAGRANGE_INTERP_POLYNOMIAL_H
#define LAGRANGE_INTERP_POLYNOMIAL_H

#include "BasisPolynomial.H"
#include "data_types.h"


namespace Dakota {

/// Derived basis polynomial class for 1-D Lagrange interpolation polynomials

/** The LagrangeInterpPolynomial class evaluates a univariate Lagrange
    interpolation polynomial.  The order of the polynomial is dictated
    by the number of interpolation points (order = N_p - 1).  It enables
    multidimensional interpolants within InterpPolyApproximation. */

class LagrangeInterpPolynomial: public BasisPolynomial
{
public:

  //
  //- Heading: Constructor and destructor
  //

  /// default constructor
  LagrangeInterpPolynomial();
  /// standard constructor
  LagrangeInterpPolynomial(const RealVector& interpolation_pts);
  /// destructor
  ~LagrangeInterpPolynomial();

  //
  //- Heading: Virtual function redefinitions
  //

  /// retrieve the value of the i_th Lagrange polynomial for a given
  /// parameter x
  const Real& get_value(const Real& x, unsigned short i);
  /// retrieve the gradient of the i_th Lagrange polynomial for a
  /// given parameter x
  const Real& get_gradient(const Real& x, unsigned short i);

  //
  //- Heading: Set/get functions
  //

  /// set interpolationPts
  void interpolation_points(const RealVector& interpolation_pts);

private:

  //
  //- Heading: Convenience functions
  //

  /// precompute data that is reused repeatedly within Lagrange interpolation
  void precompute_data();

  //
  //- Heading: Data
  //

  /// set of 1-D interpolation points: the i_th interpolation polynomial
  /// evaluated at the j_th interpolation point produces Kronecker delta_ij
  RealVector interpolationPts;
  /// number of 1-D interpolation points
  size_t numInterpPts;

  /// set of denominator products calculated from interpolationPts in
  /// precompute_data()
  RealVector lagDenominators;
};


inline LagrangeInterpPolynomial::LagrangeInterpPolynomial():
  BasisPolynomial(BaseConstructor())
{ }


inline LagrangeInterpPolynomial::
LagrangeInterpPolynomial(const RealVector& interpolation_pts):
  BasisPolynomial(BaseConstructor()), interpolationPts(interpolation_pts),
  numInterpPts(interpolationPts.length())
{ precompute_data(); }


inline LagrangeInterpPolynomial::~LagrangeInterpPolynomial()
{ }


inline void LagrangeInterpPolynomial::
interpolation_points(const RealVector& interpolation_pts)
{
  interpolationPts = interpolation_pts;
  numInterpPts = interpolationPts.length();
  precompute_data();
}

} // namespace Dakota

#endif
