/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       DistinctVariables
//- Description: Container class of variables, employing the "distinct" 
//-              approach to variables usage.
//- Owner:       Mike Eldred
//- Version: $Id: DistinctVariables.H 5337 2008-10-09 18:12:05Z mseldre $

#ifndef DISTINCT_VARS_H
#define DISTINCT_VARS_H

#include "DakotaVariables.H"
#include "DataVariables.H"


namespace Dakota {

/// Derived class within the Variables hierarchy which employs
/// the default data view (no variable or domain type array merging).

/** Derived variables classes take different views of the design,
    uncertain, and state variable types and the continuous and
    discrete domain types.  The DistinctVariables derived class
    separates the design, uncertain, and state variable types as well
    as the continuous and discrete domain types.  The result is
    separate arrays for continuous design, discrete design, uncertain,
    continuous state, and discrete state variables.  This is the
    default approach, so all iterators and strategies not specifically
    utilizing the All or Merged views use this approach (see
    Variables::get_variables(problem_db)). */

class DistinctVariables: public Variables
{
  //
  //- Heading: Friends
  //

  /// equality operator
  friend bool operator==(const DistinctVariables& vars1,
                         const DistinctVariables& vars2);
#ifdef DAKOTA_BOOST
  /// hash_value
  friend std::size_t hash_value(const DistinctVariables& vars);

  /// binary_equal_to (since 'operator==' is not suitable for boost/hash_set)
  friend bool binary_equal_to(const DistinctVariables& vars1,
                              const DistinctVariables& vars2);
#endif

public:

  //
  //- Heading: Constructors and destructor
  //

  /// default constructor
  DistinctVariables();
  /// standard constructor
  DistinctVariables(const ProblemDescDB& problem_db,
		    const pair<short,short>& view);
  /// destructor
  ~DistinctVariables();

  //
  //- Heading: Virtual function redefinitions
  //

  size_t tv() const;

  const RealVector& continuous_variables() const;
  void continuous_variable(const Real& c_var, const size_t& i);
  void continuous_variables(const RealVector& c_vars);
  const IntVector& discrete_variables() const;
  void discrete_variable(const int& d_var, const size_t& i);
  void discrete_variables(const IntVector& d_vars);

  const StringArray& continuous_variable_labels() const;
  void continuous_variable_labels(const StringArray& c_v_labels);
  const StringArray& discrete_variable_labels()   const;
  void discrete_variable_labels(const StringArray& d_v_labels);

  const StringArray& continuous_variable_types() const;
  const StringArray& discrete_variable_types() const;

  const UIntArray& continuous_variable_ids() const;

  const RealVector& inactive_continuous_variables() const;
  void inactive_continuous_variables(const RealVector& i_c_vars);
  const IntVector& inactive_discrete_variables() const;
  void inactive_discrete_variables(const IntVector& i_d_vars);

  const StringArray& inactive_continuous_variable_labels() const;
  void inactive_continuous_variable_labels(const StringArray& i_c_v_labels);
  const StringArray& inactive_discrete_variable_labels() const;
  void inactive_discrete_variable_labels(const StringArray& i_d_v_labels);

  const UIntArray& inactive_continuous_variable_ids() const;

  size_t acv() const;
  size_t adv() const;

  RealVector all_continuous_variables() const;
  void all_continuous_variables(const RealVector& a_c_vars);
  IntVector  all_discrete_variables()   const;
  void all_discrete_variables(const IntVector& a_d_vars);

  StringArray all_continuous_variable_labels() const;
  void all_continuous_variable_labels(const StringArray& a_c_v_labels);
  StringArray all_discrete_variable_labels()   const;
  void all_discrete_variable_labels(const StringArray& a_d_v_labels);
  StringArray all_variable_labels()            const;

  const StringArray& all_discrete_variable_types() const;

  void read(istream& s);
  void write(ostream& s) const;
  void write_aprepro(ostream& s) const;

  void read_annotated(istream& s);
  void write_annotated(ostream& s) const;

  void write_tabular(ostream& s) const;

  void read(BiStream& s);
  void write(BoStream& s) const;

  void read(MPIUnpackBuffer& s);
  void write(MPIPackBuffer& s) const;

protected:

  void copy_rep(const Variables* vars_rep);

  void reshape_rep(const Sizet2DArray& vars_comps);

private:

  //
  //- Heading: Private member functions
  //

  /// construct VarTypes and VarIds arrays using variablesComponents
  void build_types_ids();

  //
  //- Heading: Private data members
  //

  /// the continuous design variables array
  RealVector continuousDesignVars;
  /// the discrete design variables array
  IntVector discreteDesignVars;
  /// the uncertain variables array
  RealVector uncertainVars;
  /// the continuous state variables array
  RealVector continuousStateVars;
  /// the discrete state variables array
  IntVector discreteStateVars;

  /// the continuous design variables label array
  StringArray continuousDesignLabels;
  /// the discrete design variables label array
  StringArray discreteDesignLabels;
  /// the uncertain variables label array
  StringArray uncertainLabels;
  /// the continuous state variables label array
  StringArray continuousStateLabels;
  /// the discrete state variables label array
  StringArray discreteStateLabels;

  /// array of variable types for the active continuous variables
  StringArray continuousVarTypes;
  /// array of variable types for the active discrete variables
  StringArray discreteVarTypes;
  /// array of variable types for all of the discrete variables
  StringArray allDiscreteVarTypes;

  /// array of position identifiers for the active continuous variables
  /** These identifiers define positions of the active continuous variables
      within the total variable sequence. */
  UIntArray continuousVarIds;
  /// array of position identifiers for the inactive continuous variables
  /** These identifiers define positions of the inactive continuous variables
      within the total variable sequence. */
  UIntArray inactiveContinuousVarIds;
};


inline DistinctVariables::DistinctVariables()
{ }


inline DistinctVariables::~DistinctVariables()
{ }


inline size_t DistinctVariables::tv() const
{
  return continuousDesignVars.length() + discreteDesignVars.length() 
    + uncertainVars.length() + continuousStateVars.length() 
    + discreteStateVars.length();
}


inline const RealVector& DistinctVariables::continuous_variables() const
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    return continuousDesignVars; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return uncertainVars;        break;
  case MIXED_DISTINCT_STATE:
    return continuousStateVars;  break;
  }
  return emptyRealVector;
}


inline void DistinctVariables::
continuous_variable(const Real& c_var, const size_t& i)
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    continuousDesignVars[i] = c_var; break;
  case MIXED_DISTINCT_UNCERTAIN:
    uncertainVars[i]        = c_var; break;
  case MIXED_DISTINCT_STATE:
    continuousStateVars[i]  = c_var; break;
  }
}


inline void DistinctVariables::continuous_variables(const RealVector& c_vars)
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    continuousDesignVars = c_vars; break;
  case MIXED_DISTINCT_UNCERTAIN:
    uncertainVars        = c_vars; break;
  case MIXED_DISTINCT_STATE:
    continuousStateVars  = c_vars; break;
  }
}


inline const IntVector& DistinctVariables::discrete_variables() const
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    return discreteDesignVars; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return emptyIntVector;     break;
  case MIXED_DISTINCT_STATE:
    return discreteStateVars;  break;
  }
  return emptyIntVector;
}


inline void DistinctVariables::
discrete_variable(const int& d_var, const size_t& i)
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    discreteDesignVars[i] = d_var; break;
  case MIXED_DISTINCT_UNCERTAIN:
    Cerr << "Error: DistinctVariables contains no active discrete variables "
	 << "in uncertain mode.\n       Setting an entry in this array is "
	 << "erroneous." << endl;
    abort_handler(-1);
    break;
  case MIXED_DISTINCT_STATE:
    discreteStateVars[i] = d_var; break;
  }
}


inline void DistinctVariables::discrete_variables(const IntVector& d_vars)
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    discreteDesignVars = d_vars; break;
  case MIXED_DISTINCT_UNCERTAIN:
    if (d_vars.length()) {
      Cerr << "Error: DistinctVariables contains no active discrete variables "
	   << "in uncertain mode.\n       Setting this array is erroneous."
	   << endl;
      abort_handler(-1);
    }
    break;
  case MIXED_DISTINCT_STATE:
    discreteStateVars = d_vars; break;
  }
}


inline const StringArray& DistinctVariables::
continuous_variable_labels() const
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    return continuousDesignLabels; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return uncertainLabels;        break;
  case MIXED_DISTINCT_STATE:
    return continuousStateLabels;  break;
  }
  return emptyStringArray;
}


inline void DistinctVariables::
continuous_variable_labels(const StringArray& c_v_labels)
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    continuousDesignLabels = c_v_labels; break;
  case MIXED_DISTINCT_UNCERTAIN:
    uncertainLabels        = c_v_labels; break;
  case MIXED_DISTINCT_STATE:
    continuousStateLabels  = c_v_labels; break;
  }
}


inline const StringArray& DistinctVariables::discrete_variable_labels() const
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    return discreteDesignLabels; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return emptyStringArray;     break;
  case MIXED_DISTINCT_STATE:
    return discreteStateLabels;  break;
  }
  return emptyStringArray;
}


inline void DistinctVariables::
discrete_variable_labels(const StringArray& d_v_labels)
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    discreteDesignLabels = d_v_labels; break;
  case MIXED_DISTINCT_UNCERTAIN:
    if (d_v_labels.length()) {
      Cerr << "Error: DistinctVariables contains no active discrete variable "
	   << "labels in uncertain mode.\n       Setting labels for this array "
	   << "is erroneous." << endl;
      abort_handler(-1);
    }
    break;
  case MIXED_DISTINCT_STATE:
    discreteStateLabels  = d_v_labels; break;
  }
}


inline const StringArray& DistinctVariables::continuous_variable_types() const
{ return continuousVarTypes; }


inline const StringArray& DistinctVariables::discrete_variable_types() const
{ return discreteVarTypes; }


inline const UIntArray& DistinctVariables::continuous_variable_ids() const
{ return continuousVarIds; }


inline const RealVector& DistinctVariables::
inactive_continuous_variables() const
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    return continuousDesignVars; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return uncertainVars;        break;
  case MIXED_DISTINCT_STATE:
    return continuousStateVars;  break;
  }
  return emptyRealVector;
}


inline void DistinctVariables::
inactive_continuous_variables(const RealVector& i_c_vars)
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    continuousDesignVars = i_c_vars; break;
  case MIXED_DISTINCT_UNCERTAIN:
    uncertainVars        = i_c_vars; break;
  case MIXED_DISTINCT_STATE:
    continuousStateVars  = i_c_vars; break;
  }
}


inline const IntVector& DistinctVariables::
inactive_discrete_variables() const
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    return discreteDesignVars; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return emptyIntVector;     break;
  case MIXED_DISTINCT_STATE:
    return discreteStateVars;  break;
  }
  return emptyIntVector;
}


inline void DistinctVariables::
inactive_discrete_variables(const IntVector& i_d_vars)
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    discreteDesignVars = i_d_vars; break;
  case MIXED_DISTINCT_UNCERTAIN:
    if (i_d_vars.length()) {
      Cerr << "Error: DistinctVariables contains no inactive discrete variables"
	   << " in design mode.\n       Setting this array is erroneous."<<endl;
      abort_handler(-1);
    }
    break;
  case MIXED_DISTINCT_STATE:
    discreteStateVars  = i_d_vars; break;
  }
}


inline const StringArray& DistinctVariables::
inactive_continuous_variable_labels() const
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    return continuousDesignLabels; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return uncertainLabels;        break;
  case MIXED_DISTINCT_STATE:
    return continuousStateLabels;  break;
  }
  return emptyStringArray;
}


inline void DistinctVariables::
inactive_continuous_variable_labels(const StringArray& i_c_v_labels)
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    continuousDesignLabels = i_c_v_labels; break;
  case MIXED_DISTINCT_UNCERTAIN:
    uncertainLabels        = i_c_v_labels; break;
  case MIXED_DISTINCT_STATE:
    continuousStateLabels  = i_c_v_labels; break;
  }
}


inline const StringArray& DistinctVariables::
inactive_discrete_variable_labels() const
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    return discreteDesignLabels; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return emptyStringArray;     break;
  case MIXED_DISTINCT_STATE:
    return discreteStateLabels;  break;
  }
  return emptyStringArray;
}


inline void DistinctVariables::
inactive_discrete_variable_labels(const StringArray& i_d_v_labels)
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    discreteDesignLabels = i_d_v_labels; break;
  case MIXED_DISTINCT_UNCERTAIN:
    if (i_d_v_labels.length()) {
      Cerr << "Error: DistinctVariables contains no active discrete variable "
	   << "labels in design mode.\n       Setting labels for this array is "
	   << "erroneous." << endl;
      abort_handler(-1);
    }
    break;
  case MIXED_DISTINCT_STATE:
    discreteStateLabels  = i_d_v_labels; break;
  }
}


inline const UIntArray& DistinctVariables::
inactive_continuous_variable_ids() const
{ return inactiveContinuousVarIds; }


inline size_t DistinctVariables::acv() const
{
  return continuousDesignVars.length() + uncertainVars.length()
       + continuousStateVars.length();
}


inline size_t DistinctVariables::adv() const
{ return discreteDesignVars.length() + discreteStateVars.length(); }


inline RealVector DistinctVariables::all_continuous_variables() const
{
  RealVector a_c_vars;
  aggregate_data(continuousDesignVars, uncertainVars, continuousStateVars,
		 a_c_vars);
  return a_c_vars;
}


inline void DistinctVariables::
all_continuous_variables(const RealVector& a_c_vars)
{
  separate_data(a_c_vars, continuousDesignVars, uncertainVars,
		continuousStateVars);
}


inline IntVector DistinctVariables::all_discrete_variables() const
{
  IntVector a_d_vars;
  aggregate_data(discreteDesignVars, discreteStateVars, a_d_vars);
  return a_d_vars;
}


inline void DistinctVariables::
all_discrete_variables(const IntVector& a_d_vars)
{ separate_data(a_d_vars, discreteDesignVars, discreteStateVars); }


inline StringArray DistinctVariables::all_continuous_variable_labels() const
{
  StringArray a_c_v_labels;
  aggregate_data(continuousDesignLabels, uncertainLabels, continuousStateLabels,
		 a_c_v_labels);
  return a_c_v_labels;
}


inline void DistinctVariables::
all_continuous_variable_labels(const StringArray& a_c_v_labels)
{
  separate_data(a_c_v_labels, continuousDesignLabels, uncertainLabels,
		continuousStateLabels);
}


inline StringArray DistinctVariables::all_discrete_variable_labels() const
{
  StringArray a_d_v_labels;
  aggregate_data(discreteDesignLabels, discreteStateLabels, a_d_v_labels);
  return a_d_v_labels;
}


inline void DistinctVariables::
all_discrete_variable_labels(const StringArray& a_d_v_labels)
{ separate_data(a_d_v_labels, discreteDesignLabels, discreteStateLabels); }


inline StringArray DistinctVariables::all_variable_labels() const
{
  StringArray m_dv_labels, m_sv_labels, a_v_labels;
  aggregate_data(continuousDesignLabels, discreteDesignLabels, m_dv_labels);
  aggregate_data(continuousStateLabels,  discreteStateLabels,  m_sv_labels);
  aggregate_data(m_dv_labels, uncertainLabels, m_sv_labels, a_v_labels);
  return a_v_labels;
}


inline const StringArray& DistinctVariables::all_discrete_variable_types() const
{ return allDiscreteVarTypes; }


#ifdef DAKOTA_BOOST
/// binary_equal_to for DistinctVariables -- NOTE: no tolerance is used!
inline bool binary_equal_to(const DistinctVariables& vars1,
                            const DistinctVariables& vars2)
{
  // Check for equality in array lengths
  size_t cdv_size = vars1.continuousDesignVars.size();
  size_t ddv_size = vars1.discreteDesignVars.size();
  size_t uv_size  = vars1.uncertainVars.size();
  size_t csv_size = vars1.continuousStateVars.size();
  size_t dsv_size = vars1.discreteStateVars.size();
  if (vars2.continuousDesignVars.size() != cdv_size ||
      vars2.discreteDesignVars.size()   != ddv_size ||
      vars2.uncertainVars.size()        != uv_size  ||
      vars2.continuousStateVars.size()  != csv_size ||
      vars2.discreteStateVars.size()    != dsv_size)
    return false;

  // Check each value (labels are ignored!)
  size_t i;
  for (i=0; i<cdv_size; ++i) {
    if ( vars2.continuousDesignVars[i] != vars1.continuousDesignVars[i] )
      return false;
  }

  for (i=0; i<ddv_size; ++i) {
    if ( vars2.discreteDesignVars[i] != vars1.discreteDesignVars[i] )
      return false;
  }

  for (i=0; i<uv_size; ++i) {
    if ( vars2.uncertainVars[i] != vars1.uncertainVars[i] )
      return false;
  }

  for (i=0; i<csv_size; ++i) {
    if ( vars2.continuousStateVars[i] != vars1.continuousStateVars[i] )
      return false;
  }

  for (i=0; i<dsv_size; ++i) {
    if ( vars2.discreteStateVars[i] != vars1.discreteStateVars[i] )
      return false;
  }

  return true;
}

/// hash_value for DistinctVariables
inline std::size_t hash_value(const DistinctVariables& vars)
{
  std::size_t seed = 0;
  boost::hash_combine(seed, vars.continuousDesignVars);
  boost::hash_combine(seed, vars.discreteDesignVars);
  boost::hash_combine(seed, vars.uncertainVars);
  boost::hash_combine(seed, vars.continuousStateVars);
  boost::hash_combine(seed, vars.discreteStateVars);

  return seed;
}
#endif  // DAKOTA_BOOST

} // namespace Dakota

#endif
