/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       DataFitSurrModel
//- Description: Implementation code for the DataFitSurrModel class
//- Owner:       Mike Eldred
//- Checked by:

#include "DataFitSurrModel.H"
#include "ApproximationInterface.H"
#include "ParamResponsePair.H"
#include "ProblemDescDB.H"
#include "PRPCache.H"

static const char rcsId[]="@(#) $Id: DataFitSurrModel.C 5091 2008-06-12 00:20:53Z mseldre $";


namespace Dakota {

// define special values for componentParallelMode
#define APPROX_INTERFACE 1
#define ACTUAL_MODEL     2


DataFitSurrModel::DataFitSurrModel(ProblemDescDB& problem_db):
  SurrogateModel(problem_db), surrModelEvals(0),
  sampleReuse(problem_db.get_string("model.surrogate.sample_reuse")),
  sampleReuseFile(problem_db.get_string("model.surrogate.sample_reuse_file"))
{
  if (sampleReuse.empty()) // assign default
    sampleReuse = (sampleReuseFile.empty()) ? "none" : "all";

  // DataFitSurrModel is allowed to set the db list nodes, so long as it 
  // restores the list nodes to their previous setting.  This removes the need
  // to continuously reset at the strategy level (which would be wasteful
  // since the type of derived model may not be known at the strategy level).
  const String& dace_method_pointer
    = problem_db.get_string("model.surrogate.dace_method_pointer");
  const String& actual_model_pointer
    = problem_db.get_string("model.surrogate.actual_model_pointer");
  if (!dace_method_pointer.empty()) { // global DACE approximations
    size_t method_index = problem_db.get_db_method_node(); // for restoration
    size_t model_index  = problem_db.get_db_model_node();  // for restoration
    problem_db.set_db_list_nodes(dace_method_pointer);
    // instantiate the actual model
    actualModel = problem_db.get_model();
    check_submodel_compatibility(actualModel);
    // if outer level output is verbose/debug and actualModel verbosity is
    // defined by the DACE method spec, request fine-grained evaluation
    // reporting for purposes of the final output summary.  This allows verbose
    // final summaries without verbose output on every dace-iterator completion.
    if (outputLevel > NORMAL_OUTPUT)
      actualModel.fine_grained_evaluation_counters();
    // instantiate the DACE iterator
    daceIterator = problem_db.get_iterator(actualModel);
    daceIterator.sub_iterator_flag(true);
    problem_db.set_db_method_node(method_index); // restore method only
    problem_db.set_db_model_nodes(model_index);  // restore all model nodes
  }
  else if (!actual_model_pointer.empty()) { // local/multipoint approximation
    size_t model_index = problem_db.get_db_model_node(); // for restoration
    problem_db.set_db_model_nodes(actual_model_pointer);
    actualModel = problem_db.get_model();
    check_submodel_compatibility(actualModel);
    problem_db.set_db_model_nodes(model_index); // restore
  }
  // else global approx. built solely from reuse_samples: daceIterator/
  // actualModel remain empty envelopes.  Verify that there is a data source:
  // this basic check is augmented with a build_global() check which enforces
  // that the total samples from both sources be >= minimum required.
  else if ( sampleReuse == "none" ) {
    Cerr << "Error: to build an data fit surrogate model, either a global "
	 << "approximation\n       must be specified with reuse_samples or "
	 << "dace_method_pointer, or a\n       local/multipoint approximation "
	 << "must be specified with an actual_model_pointer." << endl;
    abort_handler(-1);
  }

  // assign the ApproximationInterface instance which manages the
  // local/multipoint/global approximation.  By instantiating with assign_rep(),
  // Interface::get_interface() does not need special logic for approximations.
  // The number of approximation variables is defined by the active variable set
  // in the sub-model, and any conversions based on differing variable views
  // must be performed in ApproximationInterface::map().
  const Variables& vars = (actualModel.is_null()) ? currentVariables :
    actualModel.current_variables();
  approxInterface.assign_rep(new
    ApproximationInterface(problem_db, vars, numFns), false);

  // read the samples_file once and then reuse this data as appropriate
  // within build_global()
  if (!sampleReuseFile.empty()) {
    ifstream sample_stream(sampleReuseFile);
    if (!sample_stream) {
      Cerr << "Error: sample reuse file " << sampleReuseFile << " not found."
	   << endl;
      abort_handler(-1);
    }
    else
      Cout << "Retrieving samples from " << sampleReuseFile << '\n';
    const pair<short,short>& view = (actualModel.is_null()) ?
      currentVariables.view() : actualModel.current_variables().view();
    size_t num_c_vars = (actualModel.is_null()) ?
      currentVariables.cv()   : actualModel.cv();
    ActiveSet temp_set(numFns, num_c_vars); // function values only
    while (!sample_stream.eof()) {
      // reuse_file_vars/responses must already be sized for istream read
      RealVector reuse_file_c_vars(num_c_vars);
      Response   reuse_file_responses(temp_set);
      try {
	reuse_file_c_vars.read_tabular(sample_stream);// read() lacks eof checks
	reuse_file_responses.read_tabular(sample_stream);
      }
      catch(String& err_msg) {
	//Cerr << "Warning: " << err_msg << endl;
	break; // out of while loop
      }
      Variables reuse_file_vars(view); // instantiate-on-the-fly
      reuse_file_vars.continuous_variables(reuse_file_c_vars);
      if (outputLevel > NORMAL_OUTPUT)
	Cout << "Sample read:\n" << reuse_file_c_vars << reuse_file_responses;
      reuseFileVars.insert(reuse_file_vars);           // shallow copy
      reuseFileResponses.insert(reuse_file_responses); // shallow copy
    }
  }
}


DataFitSurrModel::
DataFitSurrModel(Iterator& dace_iterator,       Model& actual_model,
		 const pair<short,short>& view, const ActiveSet& set,
		 const String& approx_type,     const short& approx_order,
		 const String& corr_type,       const short& corr_order,
		 const String& sample_reuse):
  SurrogateModel(actual_model.parallel_library(), view, set, corr_type,
		 corr_order),
  daceIterator(dace_iterator), actualModel(actual_model), surrModelEvals(0),
  sampleReuse(sample_reuse)
{
  // dace_iterator may be an empty envelope (local, multipoint approx),
  // but actual_model must be defined.
  if (actualModel.is_null()) {
    Cerr << "Error: actualModel is empty envelope in alternate "
	 << "DataFitSurrModel constructor." << endl;
    abort_handler(-1);
  }

  surrogateType = approx_type;

  // currentVariables/userDefinedConstraints are minimally constructed in
  // Model(NoDBBaseConstructor) using default ctors for letter instances and
  // need to be populated.  By passing the lower level variables components,
  // views may differ between this Model and the actualModel.
  const Sizet2DArray& vars_comps
    = actualModel.current_variables().variables_components();
  currentVariables.reshape(vars_comps);
  userDefinedConstraints.reshape(actualModel.num_nonlinear_ineq_constraints(),
				 actualModel.num_nonlinear_eq_constraints(),
				 actualModel.num_linear_ineq_constraints(),
				 actualModel.num_linear_eq_constraints());
  userDefinedConstraints.reshape(vars_comps);
  update_from_actual_model();
  check_submodel_compatibility(actualModel);

  // assign the ApproximationInterface instance which manages the
  // local/multipoint/global approximation.  By instantiating with assign_rep(),
  // Interface::get_interface() does not need special logic for approximations.
  approxInterface.assign_rep(new ApproximationInterface(approx_type,
    approx_order, actualModel.current_variables(), numFns), false);

  if (!daceIterator.is_null()) // global DACE approximations
    daceIterator.sub_iterator_flag(true);
  //else { // local/multipoint approximation
  //}

  // for construction on the fly, we cannot support a separate derivative
  // specification for the surrogate, so we link to the actual_model settings.
  // TO DO: could base this of of incoming ASV content.
  if (actual_model.gradient_type() == "none")
    gradType = "none";
  else
    gradType = (approx_type == "global_polynomial" ||
		approx_type.begins("local_") ||
		approx_type.begins("multipoint_")) ? "analytic" : "numerical";
  if (actual_model.hessian_type() == "none")
    hessType = "none";
  else
    hessType = (approx_type == "global_polynomial" ||
		approx_type.begins("local_"))      ? "analytic" : "numerical";

  // Promote fdGradSS/fdHessByFnSS/fdHessByGradSS to defaults if needed.
  if (gradType == "numerical") { // mixed not supported for this Model
    methodSrc    = "dakota";
    intervalType = "central";
    fdGradSS.reshape(1);
    fdGradSS[0] = 0.001;
  }
  if (hessType == "numerical") { // mixed not supported for this Model
    if (gradType == "numerical") {
      fdHessByFnSS.reshape(1);
      fdHessByFnSS[0] = 0.002;
    }
    else {
      fdHessByGradSS.reshape(1);
      fdHessByGradSS[0] = 0.001;
    }
  }
}


/** This function constructs a new approximation, discarding any
    previous data.  It constructs any required currentPoints and does
    not define an anchorPoint. */
void DataFitSurrModel::build_approximation()
{
  Cout << "\n>>>>> Building " << surrogateType << " approximations.\n";

  // clear out previous anchor/data points, but preserve history (if multipoint)
  approxInterface.clear_current();
  // update actualModel w/ variable values/bounds/labels
  update_actual_model();

  // build a local, multipoint, or global data fit approximation.
  if (surrogateType.begins("local_") || surrogateType.begins("multipoint_")) {
    // NOTE: branch used by SBO
    update_local_multipoint();
    build_local_multipoint();
  }
  else { // global approximation.  NOTE: branch not used by SBO.
    update_global();
    build_global();
    //compute_correction(...need data...);
    // could add compute_correction() here and in
    // HierarchSurrModel::build_approximation if global approximations had
    // easy access to the truth/approx responses.  Instead, it is called
    // from SBOStrategy using data from the center of the trust region.
  }
  if (actualModel.is_null())
    approxInterface.build_approximation(
      userDefinedConstraints.continuous_lower_bounds(),
      userDefinedConstraints.continuous_upper_bounds());
  else // employ sub-model vars view, if available
    approxInterface.build_approximation(actualModel.continuous_lower_bounds(),
      actualModel.continuous_upper_bounds());
  approxBuilds++;

  Cout << "\n<<<<< " << surrogateType << " approximation builds completed.\n";
}


/** This function constructs a new approximation, discarding any
    previous data.  It uses the passed data to populate the
    anchorPoint and constructs any required currentPoints. */
bool DataFitSurrModel::
build_approximation(const Variables& vars, const Response& response)
{
  Cout << "\n>>>>> Building " << surrogateType << " approximations.\n";

  // clear out previous anchor/data points, but preserve history (if multipoint)
  approxInterface.clear_current();
  // update actualModel w/ variable values/bounds/labels
  update_actual_model();
  // populate/replace the anchor point for the approximation.  When supported by
  // the surrogate type (local, multipoint, global polynomial regression), this
  // is enforced as a hard constraint. Otherwise, it is just another data point.
  approxInterface.update_approximation(vars, response);

  // build a local, multipoint, or global data fit approximation.
  if (surrogateType.begins("local_") || surrogateType.begins("multipoint_"))
    // NOTE: branch not used by SBO
    update_local_multipoint();
  else { // global approximation.  NOTE: branch used by SBO.
    update_global();
    build_global();
    //compute_correction(...need data...);
    // could add compute_correction() here and in
    // HierarchSurrModel::build_approximation if global approximations had
    // easy access to the truth/approx responses.  Instead, it is called
    // from SBOStrategy using data from the center of the trust region.
  }
  if (actualModel.is_null())
    approxInterface.build_approximation(
      userDefinedConstraints.continuous_lower_bounds(),
      userDefinedConstraints.continuous_upper_bounds());
  else // employ sub-model vars view, if available
    approxInterface.build_approximation(actualModel.continuous_lower_bounds(),
      actualModel.continuous_upper_bounds());
  approxBuilds++;

  Cout << "\n<<<<< " << surrogateType << " approximation builds completed.\n";

  // return a bool indicating whether the incoming data defines an embedded
  // correction (hard constraint) or just another data point.  It would be
  // preferable to flow this up from the surrogate, but keep it simple for now.
#ifdef DAKOTA_SURFPACK
  return (surrogateType.begins("local_") ||
	  surrogateType.begins("multipoint_") ||
	  surrogateType == "global_polynomial");
#else
  return (surrogateType.begins("local_") ||
	  surrogateType.begins("multipoint_"));
#endif // DAKOTA_SURFPACK
}


/** This function populates/replaces Approximation::anchorPoint and
    rebuilds the approximation, if requested.  It does not clear other
    data (i.e., Approximation::currentPoints) and does not update the
    actualModel with revised bounds, labels, etc.  Thus, it updates
    data from a previous call to build_approximation(), and is not
    intended to be used in isolation. */
void DataFitSurrModel::
update_approximation(const Variables& vars, const Response& response,
		     bool rebuild_flag)
{
  Cout << "\n>>>>> Updating " << surrogateType << " approximations.\n";

  // populate/replace the anchor point for each approximation
  approxInterface.update_approximation(vars, response); // update anchorPoint

  if (rebuild_flag) { // find the coefficients for each approximation
    if (actualModel.is_null())
      approxInterface.build_approximation(
        userDefinedConstraints.continuous_lower_bounds(),
        userDefinedConstraints.continuous_upper_bounds());
    else // employ sub-model vars view, if available
      approxInterface.build_approximation(actualModel.continuous_lower_bounds(),
        actualModel.continuous_upper_bounds());
    approxBuilds++;
  }

  Cout << "\n<<<<< " << surrogateType << " approximation updates completed.\n";
}


/** This function populates/replaces Approximation::currentPoints and
    rebuilds the approximation, if requested.  It does not clear other
    data (i.e., Approximation::anchorPoint) and does not update the
    actualModel with revised bounds, labels, etc.  Thus, it updates
    data from a previous call to build_approximation(), and is not
    intended to be used in isolation. */
void DataFitSurrModel::
update_approximation(const VariablesArray& vars_array,
		     const ResponseArray& resp_array, bool rebuild_flag)
{
  Cout << "\n>>>>> Updating " << surrogateType << " approximations.\n";

  // populate/replace the current points for each approximation
  approxInterface.update_approximation(vars_array, resp_array);

  if (rebuild_flag) { // find the coefficients for each approximation
    if (actualModel.is_null())
      approxInterface.build_approximation(
        userDefinedConstraints.continuous_lower_bounds(),
        userDefinedConstraints.continuous_upper_bounds());
    else // employ sub-model vars view, if available
      approxInterface.build_approximation(actualModel.continuous_lower_bounds(),
        actualModel.continuous_upper_bounds());
    approxBuilds++;
  }

  Cout << "\n<<<<< " << surrogateType << " approximation updates completed.\n";
}


/** This function appends one point to Approximation::currentPoints
    and rebuilds the approximation, if requested.  It does not modify
    other data (i.e., Approximation::anchorPoint) and does not update
    the actualModel with revised bounds, labels, etc.  Thus, it
    appends to data from a previous call to build_approximation(), and
    is not intended to be used in isolation. */
void DataFitSurrModel::
append_approximation(const Variables& vars, const Response& response,
		     bool rebuild_flag)
{
  Cout << "\n>>>>> Appending to " << surrogateType << " approximations.\n";

  // append to the current points for each approximation
  approxInterface.append_approximation(vars, response);

  if (rebuild_flag) { // find the coefficients for each approximation
    if (actualModel.is_null())
      approxInterface.build_approximation(
        userDefinedConstraints.continuous_lower_bounds(),
        userDefinedConstraints.continuous_upper_bounds());
    else // employ sub-model vars view, if available
      approxInterface.build_approximation(actualModel.continuous_lower_bounds(),
        actualModel.continuous_upper_bounds());
    approxBuilds++;
  }

  Cout << "\n<<<<< " << surrogateType << " approximation updates completed.\n";
}


/** This function appends multiple points to Approximation::currentPoints
    and rebuilds the approximation, if requested.  It does not modify
    other data (i.e., Approximation::anchorPoint) and does not update the
    actualModel with revised bounds, labels, etc.  Thus, it appends to
    data from a previous call to build_approximation(), and is not
    intended to be used in isolation. */
void DataFitSurrModel::
append_approximation(const VariablesArray& vars_array,
		     const ResponseArray&  resp_array, bool rebuild_flag)
{
  Cout << "\n>>>>> Appending to " << surrogateType << " approximations.\n";

  // append to the current points for each approximation
  approxInterface.append_approximation(vars_array, resp_array);

  if (rebuild_flag) { // find the coefficients for each approximation
    if (actualModel.is_null())
      approxInterface.build_approximation(
        userDefinedConstraints.continuous_lower_bounds(),
        userDefinedConstraints.continuous_upper_bounds());
    else // employ sub-model vars view, if available
      approxInterface.build_approximation(actualModel.continuous_lower_bounds(),
        actualModel.continuous_upper_bounds());
    approxBuilds++;
  }

  Cout << "\n<<<<< " << surrogateType << " approximation updates completed.\n";
}


void DataFitSurrModel::update_local_multipoint()
{
  // Store the actualModel inactive variable values for use in force_rebuild()
  // for determining whether an automatic approximation rebuild is required.

  // the actualModel data has been updated by update_actual_model(), which
  // precedes update_local_multipoint()

  const Variables& actual_model_vars = actualModel.current_variables();
  if (actual_model_vars.view().first >= MERGED_DISTINCT_DESIGN) {//Distinct view
    fitInactCVars = actual_model_vars.inactive_continuous_variables();
    fitInactDVars = actual_model_vars.inactive_discrete_variables();
  }
}


void DataFitSurrModel::update_global()
{
  // Store the actualModel active variable bounds and inactive variable values
  // for use in force_rebuild() to determine whether an automatic approximation
  // rebuild is required.

  // the actualModel data has been updated by update_actual_model(),
  // which precedes update_global().

  const Constraints& cons = (actualModel.is_null()) ? userDefinedConstraints :
    actualModel.user_defined_constraints();
  fitCLBnds = cons.continuous_lower_bounds();
  fitCUBnds = cons.continuous_upper_bounds();
  fitDLBnds = cons.discrete_lower_bounds();
  fitDUBnds = cons.discrete_upper_bounds();

  const Variables& vars = (actualModel.is_null()) ? currentVariables :
    actualModel.current_variables();
  if (vars.view().first >= MERGED_DISTINCT_DESIGN) {//Distinct view
    fitInactCVars = vars.inactive_continuous_variables();
    fitInactDVars = vars.inactive_discrete_variables();
  }
}


/** Evaluate the value, gradient, and possibly Hessian needed for a
    local or multipoint approximation using actualModel. */
void DataFitSurrModel::build_local_multipoint()
{
  // set DataFitSurrModel parallelism mode to actualModel
  component_parallel_mode(ACTUAL_MODEL);

  // Define the data requests
  short asv_value = 3;
  if (surrogateType.begins("local_") && actualModel.hessian_type() != "none")
    asv_value += 4;
  ShortArray orig_asv(numFns, asv_value), actual_asv, approx_asv;
  asv_mapping(orig_asv, actual_asv, approx_asv, true);

  // Evaluate value and derivatives using actualModel
  ActiveSet set = actualModel.current_response().active_set(); // copy
  set.request_vector(actual_asv);
  set.derivative_vector(actualModel.continuous_variable_ids());
  actualModel.compute_response(set);

  approxInterface.update_approximation(actualModel.current_variables(),
    actualModel.current_response());
}


/** Determine sample points to use in building the approximation and
    then evaluate them on actualModel using daceIterator.  Any changes
    to the bounds should be performed by setting them at a higher
    level (e.g., SurrBasedOptStrategy). */
void DataFitSurrModel::build_global()
{
  // build_global() follows update_actual_model() so we may use
  // actualModel.continuous_(lower/upper)_bounds() to avoid view
  // conversions and allow pass-by-reference.
  size_t i, j, num_c_vars, num_d_vars;
  if (actualModel.is_null()) {
    num_c_vars = currentVariables.cv();
    num_d_vars = currentVariables.dv();
  }
  else {
    num_c_vars = actualModel.cv();
    num_d_vars = actualModel.dv();
  }

  // **************************************************************************
  // Check data_pairs and sampleReuseFile for any existing evaluations to reuse
  // **************************************************************************
  VariablesList reuse_vars;
  ResponseList  reuse_responses;
  if (sampleReuse == "all" || sampleReuse == "region") {

    // since SurrBasedLocalMinimizer currently evaluates the trust region center
    // first, we must take care to not include this point in the sample reuse,
    // since this would cause it to be used twice.
    const SurrogateDataPoint& anchor_pt = approxInterface.anchor_point();

    // Process the PRPCache
    extern PRPCache data_pairs;
    PRPCacheOrderedConstIter prp_end_iter = prpCacheEnd(data_pairs);
    for (PRPCacheOrderedConstIter prp_iter = prpCacheBegin(data_pairs);
         prp_iter != prp_end_iter; ++prp_iter) {
      const Variables&  db_vars   = prp_iter->prp_parameters();
      const RealVector& db_c_vars = db_vars.continuous_variables();
      const IntVector&  db_d_vars = db_vars.discrete_variables();
      if ( db_c_vars.length() == num_c_vars &&
	   db_d_vars.length() == num_d_vars &&
	   prp_iter->interface_id() == actualModel.interface_id() &&
	   db_c_vars != anchor_pt.continuous_variables() &&
	   inside(db_c_vars, db_d_vars) ) {
	reuse_vars.insert(db_vars);
	reuse_responses.insert(prp_iter->prp_response());
      }
    }

    // Process the samples_file
    if (!sampleReuseFile.empty()) {
      VarsLIter v_it; RespLIter r_it;
      for (v_it  = reuseFileVars.begin(), r_it = reuseFileResponses.begin();
	   v_it != reuseFileVars.end();   v_it++, r_it++) {
	if (inside(v_it->continuous_variables(), v_it->discrete_variables())) {
	  reuse_vars.insert(*v_it);
	  reuse_responses.insert(*r_it);
	}
      }
    }
  }
  size_t reuse_samples = reuse_vars.entries();

  // *******************************************
  // Evaluate new data points using daceIterator
  // *******************************************
  VariablesArray all_vars;
  ResponseArray  all_responses;
  int min_samples = approxInterface.minimum_samples(true);// include constraints
  int rec_samples = approxInterface.recommended_samples(true);
  if (!daceIterator.is_null()) { // else use rst info only (no new data)

    // set DataFitSurrModel parallelism mode to actualModel
    component_parallel_mode(ACTUAL_MODEL);

    // daceIterator must generate at least min_samples-reuse_samples samples,
    // should populate allData lists (allDataFlag = true), and should bypass 
    // statistics computation (statsFlag = false).
    daceIterator.sampling_reset(min_samples-reuse_samples,
				rec_samples-reuse_samples, true, false);

    // Define the data requests
    ActiveSet set = daceIterator.active_set(); // copy
    ShortArray actual_asv, approx_asv;
    asv_mapping(set.request_vector(), actual_asv, approx_asv, true);
    set.request_vector(actual_asv);
    daceIterator.active_set(set);

    daceIterator.run_iterator(); // quiet mode: no annotation or summary
    all_vars      = daceIterator.all_variables();
    all_responses = daceIterator.all_responses();
  }

  // ************************************************************************
  // Append reuse_vars and reuse_responses to end of all_vars & all_responses
  // ************************************************************************
  size_t dace_samples  = all_vars.length(),
         total_samples = reuse_samples + dace_samples;
  if (total_samples < min_samples) {
    Cerr << "Error: a minimum of " << min_samples << " samples is required by "
	 << "DataFitSurrModel::build_global.\n" << total_samples
	 << " were provided."<< endl;
    abort_handler(-1);
  }
  if (total_samples != dace_samples) {
    all_vars.reshape(total_samples);
    all_responses.reshape(total_samples);
    for (i=dace_samples; i<total_samples; i++) {
      all_vars[i]      = reuse_vars.get();
      all_responses[i] = reuse_responses.get();
    }
  }

  // ***************************************
  // Build the global approximation surfaces
  // ***************************************
  String anchor = (approxInterface.anchor()) ? "one" : "no";
  Cout << "Constructing global approximations with " << anchor << " anchor, "
       << dace_samples << " DACE samples, and " << reuse_samples
       << " reused samples.\n";
  approxInterface.update_approximation(all_vars, all_responses);
}


/** Compute the response synchronously using actualModel, approxInterface,
    or both (mixed case).  For the approxInterface portion, build the
    approximation if needed, evaluate the approximate response, and apply 
    correction (if active) to the results. */
void DataFitSurrModel::derived_compute_response(const ActiveSet& set)
{
  surrModelEvals++;

  ShortArray actual_asv, approx_asv;
  asv_mapping(set.request_vector(), actual_asv, approx_asv, false);
  bool actual_eval = !actual_asv.empty(), approx_eval = !approx_asv.empty(),
    mixed_eval = (actual_eval && approx_eval);
  Response actual_response, approx_response; // empty handles

  if (actual_eval) { // rare case: use actualModel i/o approxInterface
    component_parallel_mode(ACTUAL_MODEL);
    update_actual_model(); // update variables/bounds/labels in actualModel
    ActiveSet actual_set = set;
    actual_set.request_vector(actual_asv);
    actualModel.compute_response(actual_set);
    if (mixed_eval)
      actual_response = actualModel.current_response(); // shared rep
    else {
      currentResponse.active_set(actual_set);
      currentResponse.copy_results(actualModel.current_response());
    }
  }

  if (approx_eval) { // normal case: evaluation of approxInterface

    // if build_approximation has not yet been called, call it now
    if (!approxBuilds || force_rebuild()) {
      //if (!approxBuilds && !correctionType.empty())
      //  autoCorrection = true; // default if stand-alone use
      build_approximation();
    }

    //component_parallel_mode(APPROX_INTERFACE); // does not use parallelism
    ActiveSet approx_set = set;
    approx_set.request_vector(approx_asv);
    approx_response = (mixed_eval) ? currentResponse.copy() : currentResponse;
    approxInterface.map(currentVariables, approx_set, approx_response);

    if (autoCorrection && !correctionType.empty()) {
      //if (!correctionComputed)
      //  compute_correction(centerResponse, approx_response,
      //                     currentVariables.continuous_variables());
      apply_correction(approx_response,currentVariables.continuous_variables());
    }
  }

  if (mixed_eval) {
    currentResponse.active_set(set);
    response_mapping(actual_response, approx_response, currentResponse);
  }
}


/** Compute the response asynchronously using actualModel,
    approxInterface, or both (mixed case).  For the approxInterface
    portion, build the approximation if needed and evaluate the
    approximate response in a quasi-asynchronous approach
    (ApproximationInterface::map() performs the map synchronously and
    bookkeeps the results for return in derived_synchronize() below). */
void DataFitSurrModel::derived_asynch_compute_response(const ActiveSet& set)
{
  surrModelEvals++;

  ShortArray actual_asv, approx_asv;
  asv_mapping(set.request_vector(), actual_asv, approx_asv, false);

  if (!actual_asv.empty()) { // rare case: use actualModel i/o approxInterface
    // don't need to set component parallel mode since this only queues the job
    update_actual_model(); // update variables/bounds/labels in actualModel
    ActiveSet actual_set = set;
    actual_set.request_vector(actual_asv);
    actualModel.asynch_compute_response(actual_set);
    // store mapping from actualModel eval id to DataFitSurrModel id
    truthIdMap[actualModel.evaluation_id()] = surrModelEvals;
  }

  if (!approx_asv.empty()) { // normal case: evaluation of approxInterface

    // if build_approximation has not yet been called, call it now
    if (!approxBuilds || force_rebuild()) {
      //if (!approxBuilds && !correctionType.empty())
      //  autoCorrection = true; // default if stand-alone use
      build_approximation();
    }

    // don't need to set component parallel mode since this only queues the job
    ActiveSet approx_set = set;
    approx_set.request_vector(approx_asv);
    approxInterface.map(currentVariables, approx_set, currentResponse, true);

    if (autoCorrection && !correctionType.empty())
      rawCVarsMap[surrModelEvals] = currentVariables.continuous_variables();
    // store map from approxInterface eval id to DataFitSurrModel id
    surrIdMap[approxInterface.evaluation_id()] = surrModelEvals;
  }
}


/** Blocking retrieval of asynchronous evaluations from actualModel,
    approxInterface, or both (mixed case).  For the approxInterface
    portion, apply correction (if active) to each response in the array.
    derived_synchronize() is designed for the general case where
    derived_asynch_compute_response() may be inconsistent in its use
    of actual evaluations, approximate evaluations, or both. */
const ResponseArray& DataFitSurrModel::derived_synchronize()
{
  surrResponseArray.clear();
  bool actual_evals = !truthIdMap.empty(), approx_evals = !surrIdMap.empty();

  // synchronize actualModel evals
  if (actual_evals)
    component_parallel_mode(ACTUAL_MODEL);
  // surrResponseArray serves as a dummy reference initializer since
  // actual_resp_array is not used if !actual_evals
  const ResponseArray& actual_resp_array = (actual_evals) ?
    actualModel.synchronize() : surrResponseArray;
  if (actual_evals && !approx_evals) {
    truthIdMap.clear();
    return actual_resp_array;
  }

  // synchronize approxInterface evals
  ResponseArray approx_resp_array_copy;
  if (approx_evals) {
    //component_parallel_mode(APPROX_INTERFACE); // does not use parallelism
    const ResponseArray& approx_resp_array = approxInterface.synch();

    bool correct = (autoCorrection && !correctionType.empty());
    if (correct) {
      // Interface::rawResponseArray can be corrected directly in the case
      // of an ApproximationInterface since data_pairs is not used (see also
      // HierarchSurrModel::derived_synchronize()).
      approx_resp_array_copy  = approx_resp_array; // shared reps
      size_t i, num_responses = approx_resp_array_copy.length();
      //if (!correctionComputed && num_responses)
      //  compute_correction(..., approx_resp_array_copy[0],
      //                     rawCVarsMap.begin()->second);
      IntRVMIter v_it;
      for (i=0, v_it = rawCVarsMap.begin(); i<num_responses; i++, v_it++)
        apply_correction(approx_resp_array_copy[i], v_it->second);
      rawCVarsMap.clear();
    }

    // In order to support mixed usage of blocking/nonblocking synchronize,
    // need to insert (in order) any cached approx evals carried over from
    // synchronize_nowait().  Do not correct them a second time.
    bool cached  = !cachedApproxRespMap.empty();
    if (cached) {
      const ResponseArray& approx_resp_array_ref = (correct) ?
	approx_resp_array_copy : approx_resp_array;
      cached_mapping(approx_resp_array_ref, cachedApproxRespMap, surrIdMap,
		     approx_resp_array_copy);
    }

    if (actual_evals && !correct && !cached)
      approx_resp_array_copy  = approx_resp_array; // shared reps
    if (!actual_evals) {
      surrIdMap.clear();
      if (correct || cached) { // return ref to non-temporary
	surrResponseArray = approx_resp_array_copy; // shared reps
	return surrResponseArray;
      }
      else
	return approx_resp_array;
    }
  }

  // mixed actualModel and approxInterface evals
  Response empty_resp;
  size_t actual_cntr = 0, approx_cntr = 0;
  IntIntMIter truth_it = truthIdMap.begin(), surr_it = surrIdMap.begin();
  int start = min(truth_it->second, surr_it->second),
    num_evals = surrModelEvals - start + 1;
  surrResponseArray.reshape(num_evals);
  for (size_t i=0; i<num_evals; i++) {
    int act_dfs_model_eval = truth_it->second,
        app_dfs_model_eval =  surr_it->second;
    const Response& actual_resp = (act_dfs_model_eval <= app_dfs_model_eval)
      ? actual_resp_array[actual_cntr++] : empty_resp;
    const Response& approx_resp = (app_dfs_model_eval <= act_dfs_model_eval)
      ? approx_resp_array_copy[approx_cntr++] : empty_resp;
    response_mapping(actual_resp, approx_resp, surrResponseArray[i]);
    if (act_dfs_model_eval <= app_dfs_model_eval)
      truth_it++;
    if (app_dfs_model_eval <= act_dfs_model_eval)
      surr_it++;
  }
  truthIdMap.clear();
  surrIdMap.clear();
  return surrResponseArray;
}


/** Nonblocking retrieval of asynchronous evaluations from
    actualModel, approxInterface, or both (mixed case).  For the
    approxInterface portion, apply correction (if active) to each
    response in the map.  derived_synchronize_nowait() is designed for
    the general case where derived_asynch_compute_response() may be
    inconsistent in its use of actual evals, approx evals, or both. */
const IntResponseMap& DataFitSurrModel::derived_synchronize_nowait()
{
  surrResponseMap.clear();
  bool actual_evals = !truthIdMap.empty(), approx_evals = !surrIdMap.empty();

  // synchronize actualModel evals
  IntResponseMap actual_resp_map_copy;
  if (actual_evals) {
    component_parallel_mode(ACTUAL_MODEL);
    const IntResponseMap& actual_resp_map = actualModel.synchronize_nowait();
    // renumber to use surrModelEvals identifiers
    for (IntRespMCIter r_cit = actual_resp_map.begin();
	 r_cit != actual_resp_map.end(); r_cit++) {
      int am_eval_id = r_cit->first;
      if (approx_evals)
	actual_resp_map_copy[truthIdMap[am_eval_id]] = r_cit->second;
      else {
	surrResponseMap[truthIdMap[am_eval_id]] = r_cit->second;
	truthIdMap.erase(am_eval_id); // erase now prior to return below
      }
    }
    if (!approx_evals)
      return surrResponseMap;
  }

  // synchronize approxInterface evals
  IntResponseMap approx_resp_map_copy;
  if (approx_evals) {
    //component_parallel_mode(APPROX_INTERFACE); // does not use parallelism
    const IntResponseMap& approx_resp_map = approxInterface.synch_nowait();
    // renumber to use surrModelEvals identifiers
    for (IntRespMCIter r_cit = approx_resp_map.begin();
	 r_cit != approx_resp_map.end(); r_cit++)
      approx_resp_map_copy[surrIdMap[r_cit->first]] = r_cit->second;

    if (autoCorrection && !correctionType.empty()) {
      // Interface::rawResponseMap can be corrected directly in the case of an
      // ApproximationInterface since data_pairs is not used (see also
      // HierarchSurrModel::derived_synchronize_nowait()).  The response map
      // from ApproximationInterface's quasi-asynch mode is complete & in order.

      //if (!correctionComputed && !approx_resp_map_copy.empty())
      //  compute_correction(..., approx_resp_map_copy.begin()->second,
      //                     rawCVarsMap.begin()->second);
      for (IntRespMIter r_it = approx_resp_map_copy.begin();
	   r_it != approx_resp_map_copy.end(); r_it++)
        apply_correction(r_it->second, rawCVarsMap[r_it->first]);
      rawCVarsMap.clear();
    }

    // add cached evals (synchronized approx evals that could not be returned
    // since truth eval portions were still pending) for processing.  Do not
    // correct them a second time.
    for (IntRespMCIter r_cit = cachedApproxRespMap.begin();
	 r_cit != cachedApproxRespMap.end(); r_cit++)
      approx_resp_map_copy[r_cit->first] = r_cit->second;
    cachedApproxRespMap.clear();

    if (!actual_evals) { // return ref to non-temporary
      surrIdMap.clear(); // approx_resp_map_copy is complete
      surrResponseMap = approx_resp_map_copy;
      return surrResponseMap;
    }
  }

  // mixed actualModel and approxInterface evals:
  // > actual_resp_map_copy may be a partial set of evals
  // > approx_resp_map_copy is a complete set of evals
  Response empty_resp;
  IntRespMIter actual_it = actual_resp_map_copy.begin(),
               approx_it = approx_resp_map_copy.begin();
  bool actual_complete = false, approx_complete = false;
  // invert truthIdMap and surrIdMap
  IntIntMap inverse_truth_id_map, inverse_surr_id_map;
  for (IntIntMIter tim_it=truthIdMap.begin();
       tim_it!=truthIdMap.end(); tim_it++)
    inverse_truth_id_map[tim_it->second] = tim_it->first;
  for (IntIntMIter sim_it=surrIdMap.begin();
       sim_it!=surrIdMap.end(); sim_it++)
    inverse_surr_id_map[sim_it->second] = sim_it->first;
  // process any combination of actual and approx completions
  while (!actual_complete || !approx_complete) {
    if (actual_it == actual_resp_map_copy.end())
      actual_complete = true;
    if (approx_it == approx_resp_map_copy.end())
      approx_complete = true;

    int act_dfs_model_eval = (actual_complete) ? INT_MAX : actual_it->first;
    int app_dfs_model_eval = (approx_complete) ? INT_MAX : approx_it->first;

    if (act_dfs_model_eval < app_dfs_model_eval) { // only actual
      // there is no approx component to this response
      response_mapping(actual_it->second, empty_resp,
		       surrResponseMap[act_dfs_model_eval]);
      truthIdMap.erase(inverse_truth_id_map[act_dfs_model_eval]);
      actual_it++;
    }
    else if (app_dfs_model_eval < act_dfs_model_eval) { // only approx
      if (inverse_truth_id_map.count(app_dfs_model_eval))
	// response not complete: actual contribution not yet available
	cachedApproxRespMap[app_dfs_model_eval] = approx_it->second;
      else { // response complete: there is no actual contribution
	response_mapping(empty_resp, approx_it->second, 
			 surrResponseMap[app_dfs_model_eval]);
	surrIdMap.erase(inverse_surr_id_map[app_dfs_model_eval]);
      }
      approx_it++;
    }
    else if (!actual_complete && !approx_complete) { // both actual and approx
      response_mapping(actual_it->second, approx_it->second,
		       surrResponseMap[act_dfs_model_eval]);
      truthIdMap.erase(inverse_truth_id_map[act_dfs_model_eval]);
      surrIdMap.erase(inverse_surr_id_map[app_dfs_model_eval]);
      actual_it++;
      approx_it++;
    }
  }

  return surrResponseMap;
}


void DataFitSurrModel::component_parallel_mode(short mode)
{
  // mode may be correct, but can't guarantee active parallel configuration is
  // in synch
  //if (componentParallelMode == mode)
  //  return; // already in correct parallel mode

  if (mode == ACTUAL_MODEL)
    parallelLib.parallel_configuration_iterator(
      actualModel.parallel_configuration_iterator());
  else if (mode == APPROX_INTERFACE)
    parallelLib.parallel_configuration_iterator(modelPCIter);
  //else if (mode == 0)

  componentParallelMode = mode;
}


/** Update variables and constraints data within actualModel using
    values and labels from currentVariables and bound/linear/nonlinear
    constraints from userDefinedConstraints. */
void DataFitSurrModel::update_actual_model()
{
  if (actualModel.is_null())
    return;

  // linear constraints

  if (userDefinedConstraints.num_linear_ineq_constraints()) {
    // the views don't necessarily have to be the same, but the number of
    // active continuous and active discrete variables have to be consistent.
    if (currentVariables.cv() == actualModel.cv() &&
	currentVariables.dv() == actualModel.dv()) {
      actualModel.linear_ineq_constraint_coeffs(
        userDefinedConstraints.linear_ineq_constraint_coeffs());
      actualModel.linear_ineq_constraint_lower_bounds(
        userDefinedConstraints.linear_ineq_constraint_lower_bounds());
      actualModel.linear_ineq_constraint_upper_bounds(
        userDefinedConstraints.linear_ineq_constraint_upper_bounds());
    }
    else {
      Cerr << "Error: cannot update linear inequality constraints in "
	   << "DataFitSurrModel::update_actual_model() due to inconsistent "
	   << "active variables." << endl;
      abort_handler(-1);
    }
  }
  if (userDefinedConstraints.num_linear_eq_constraints()) {
    // the views don't necessarily have to be the same, but the number of
    // active continuous and active discrete variables have to be consistent.
    if (currentVariables.cv() == actualModel.cv() &&
	currentVariables.dv() == actualModel.dv()) {
      actualModel.linear_eq_constraint_coeffs(
        userDefinedConstraints.linear_eq_constraint_coeffs());
      actualModel.linear_eq_constraint_targets(
        userDefinedConstraints.linear_eq_constraint_targets());
    }
    else {
      Cerr << "Error: cannot update linear equality constraints in "
	   << "DataFitSurrModel::update_actual_model() due to inconsistent "
	   << "active variables." << endl;
      abort_handler(-1);
    }
  }

  // nonlinear constraints

  if (userDefinedConstraints.num_nonlinear_ineq_constraints()) {
    actualModel.nonlinear_ineq_constraint_lower_bounds(
      userDefinedConstraints.nonlinear_ineq_constraint_lower_bounds());
    actualModel.nonlinear_ineq_constraint_upper_bounds(
      userDefinedConstraints.nonlinear_ineq_constraint_upper_bounds());
  }
  if (userDefinedConstraints.num_nonlinear_eq_constraints())
    actualModel.nonlinear_eq_constraint_targets(
      userDefinedConstraints.nonlinear_eq_constraint_targets());

  // vars/bounds/labels

  const short& approx_active_view = currentVariables.view().first;
  const short& actual_active_view
    = actualModel.current_variables().view().first;
  // Update actualModel variables, bounds, and labels in all view cases.
  // Note 1: bounds updating isn't strictly required for local/multipoint, but
  // is needed for global and could be relevant in cases where actualModel
  // involves additional surrogates/nestings.
  // Note 2: label updating eliminates the need to replicate variable
  // descriptors, e.g., in SBOUU input files.  It only needs to be performed
  // once (as opposed to the update of vars and bounds).  However, performing
  // this updating in the constructor does not propagate properly for multiple
  // surrogates/nestings since the sub-model construction (and therefore any
  // sub-sub-model constructions) must finish before calling any set functions
  // on it.  That is, after-the-fact updating in constructors only propagates
  // one level, whereas before-the-fact updating in compute/build functions
  // propagates multiple levels.
  if (approx_active_view == actual_active_view) {
    // update active actualModel vars/cons with active currentVariables data
    actualModel.continuous_variables(currentVariables.continuous_variables());
    actualModel.discrete_variables(currentVariables.discrete_variables());
    actualModel.continuous_lower_bounds(
      userDefinedConstraints.continuous_lower_bounds());
    actualModel.continuous_upper_bounds(
      userDefinedConstraints.continuous_upper_bounds());
    actualModel.discrete_lower_bounds(
      userDefinedConstraints.discrete_lower_bounds());
    actualModel.discrete_upper_bounds(
      userDefinedConstraints.discrete_upper_bounds());

    // update actualModel variable descriptors with currentVariables descriptors
    if (!approxBuilds) {
      // active not currently necessary for local/multipt, but needed for global
      actualModel.continuous_variable_labels(
        currentVariables.continuous_variable_labels());
      actualModel.discrete_variable_labels(
        currentVariables.discrete_variable_labels());
      if (approx_active_view >= MERGED_DISTINCT_DESIGN) {
	// inactive needed for Nested/Surrogate propagation
	actualModel.inactive_continuous_variable_labels(
          currentVariables.inactive_continuous_variable_labels());
	actualModel.inactive_discrete_variable_labels(
          currentVariables.inactive_discrete_variable_labels());
      }
    }
  }
  else if ( approx_active_view >= MERGED_DISTINCT_DESIGN &&
	    ( actual_active_view == MERGED_ALL ||
	      actual_active_view == MIXED_ALL ) ) {
    // update active actualModel vars/cons using "All" view of
    // currentVariables/userDefinedConstraints data.
    actualModel.continuous_variables(
      currentVariables.all_continuous_variables());
    actualModel.discrete_variables(currentVariables.all_discrete_variables());
    actualModel.continuous_lower_bounds(
      userDefinedConstraints.all_continuous_lower_bounds());
    actualModel.continuous_upper_bounds(
      userDefinedConstraints.all_continuous_upper_bounds());
    actualModel.discrete_lower_bounds(
      userDefinedConstraints.all_discrete_lower_bounds());
    actualModel.discrete_upper_bounds(
      userDefinedConstraints.all_discrete_upper_bounds());
    if (!approxBuilds) { // only performed once
      actualModel.continuous_variable_labels(
        currentVariables.all_continuous_variable_labels());
      actualModel.discrete_variable_labels(
        currentVariables.all_discrete_variable_labels());
    }
  }
  else if ( actual_active_view >= MERGED_DISTINCT_DESIGN &&
	    ( approx_active_view == MERGED_ALL ||
	      approx_active_view == MIXED_ALL ) ) {
    // update "All" view of actualModel vars/cons using active
    // currentVariables/userDefinedConstraints data.
    actualModel.all_continuous_variables(
      currentVariables.continuous_variables());
    actualModel.all_discrete_variables(currentVariables.discrete_variables());
    actualModel.all_continuous_lower_bounds(
      userDefinedConstraints.continuous_lower_bounds());
    actualModel.all_continuous_upper_bounds(
      userDefinedConstraints.continuous_upper_bounds());
    actualModel.all_discrete_lower_bounds(
      userDefinedConstraints.discrete_lower_bounds());
    actualModel.all_discrete_upper_bounds(
      userDefinedConstraints.discrete_upper_bounds());
    if (!approxBuilds) { // only performed once
      actualModel.all_continuous_variable_labels(
        currentVariables.continuous_variable_labels());
      actualModel.all_discrete_variable_labels(
        currentVariables.discrete_variable_labels());
    }
  }
  if (!approxBuilds)
    actualModel.response_labels(currentResponse.function_labels());
  // uncertain variable distribution data
  if (!normalMeans.empty()) {
    actualModel.normal_means(normalMeans);
    actualModel.normal_std_deviations(normalStdDevs);
    actualModel.normal_lower_bounds(normalLowerBnds);
    actualModel.normal_upper_bounds(normalUpperBnds);
  }
  if (!lognormalMeans.empty()) {
    actualModel.lognormal_means(lognormalMeans);
    actualModel.lognormal_std_deviations(lognormalStdDevs);
    actualModel.lognormal_error_factors(lognormalErrFacts);
    actualModel.lognormal_lower_bounds(lognormalLowerBnds);
    actualModel.lognormal_upper_bounds(lognormalUpperBnds);
  }
  if (!uniformLowerBnds.empty()) {
    actualModel.uniform_lower_bounds(uniformLowerBnds);
    actualModel.uniform_upper_bounds(uniformUpperBnds);
  }
  if (!loguniformLowerBnds.empty()) {
    actualModel.loguniform_lower_bounds(loguniformLowerBnds);
    actualModel.loguniform_upper_bounds(loguniformUpperBnds);
  }
  if (!triangularModes.empty()) {
    actualModel.triangular_modes(triangularModes);
    actualModel.triangular_lower_bounds(triangularLowerBnds);
    actualModel.triangular_upper_bounds(triangularUpperBnds);
  }
  if (!exponentialBetas.empty())
    actualModel.exponential_betas(exponentialBetas);
  if (!betaAlphas.empty()) {
    actualModel.beta_alphas(betaAlphas);
    actualModel.beta_betas(betaBetas);
    actualModel.beta_lower_bounds(betaLowerBnds);
    actualModel.beta_upper_bounds(betaUpperBnds);
  }
  if (!gammaAlphas.empty()) {
    actualModel.gamma_alphas(gammaAlphas);
    actualModel.gamma_betas(gammaBetas);
  }
  if (!gumbelAlphas.empty()) {
    actualModel.gumbel_alphas(gumbelAlphas);
    actualModel.gumbel_betas(gumbelBetas);
  }
  if (!frechetAlphas.empty()) {
    actualModel.frechet_alphas(frechetAlphas);
    actualModel.frechet_betas(frechetBetas);
  }
  if (!weibullAlphas.empty()) {
    actualModel.weibull_alphas(weibullAlphas);
    actualModel.weibull_betas(weibullBetas);
  }
  if (!histogramBinPairs.empty())
    actualModel.histogram_bin_pairs(histogramBinPairs);
  if (!histogramPointPairs.empty())
    actualModel.histogram_point_pairs(histogramPointPairs);
  if (!intervalBasicProbs.empty()) {
    actualModel.interval_probabilities(intervalBasicProbs);
    actualModel.interval_bounds(intervalBounds);
  }
  if (!uncertainCorrelations.empty())
    actualModel.uncertain_correlations(uncertainCorrelations);
}


/** Update values and labels in currentVariables and
    bound/linear/nonlinear constraints in userDefinedConstraints from
    variables and constraints data within actualModel. */
void DataFitSurrModel::update_from_actual_model()
{
  // vars/bounds/labels

  // update vars/bounds/labels with actualModel data using All view for both
  // (since approx arrays are sized but otherwise uninitialized)
  currentVariables.all_continuous_variables(
    actualModel.all_continuous_variables());
  currentVariables.all_discrete_variables(
    actualModel.all_discrete_variables());
  userDefinedConstraints.all_continuous_lower_bounds(
    actualModel.all_continuous_lower_bounds());
  userDefinedConstraints.all_continuous_upper_bounds(
    actualModel.all_continuous_upper_bounds());
  userDefinedConstraints.all_discrete_lower_bounds(
    actualModel.all_discrete_lower_bounds());
  userDefinedConstraints.all_discrete_upper_bounds(
    actualModel.all_discrete_upper_bounds());
  if (!approxBuilds) {
    currentVariables.all_continuous_variable_labels(
      actualModel.all_continuous_variable_labels());
    currentVariables.all_discrete_variable_labels(
      actualModel.all_discrete_variable_labels());
    currentResponse.function_labels(actualModel.response_labels());
  }
  // uncertain variable distribution data
  if (!actualModel.normal_means().empty()) {
    normalMeans     = actualModel.normal_means();
    normalStdDevs   = actualModel.normal_std_deviations();
    normalLowerBnds = actualModel.normal_lower_bounds();
    normalUpperBnds = actualModel.normal_upper_bounds();
  }
  if (!actualModel.lognormal_means().empty()) {
    lognormalMeans     = actualModel.lognormal_means();
    lognormalStdDevs   = actualModel.lognormal_std_deviations();
    lognormalErrFacts  = actualModel.lognormal_error_factors();
    lognormalLowerBnds = actualModel.lognormal_lower_bounds();
    lognormalUpperBnds = actualModel.lognormal_upper_bounds();
  }
  if (!actualModel.uniform_lower_bounds().empty()) {
    uniformLowerBnds = actualModel.uniform_lower_bounds();
    uniformUpperBnds = actualModel.uniform_upper_bounds();
  }
  if (!actualModel.loguniform_lower_bounds().empty()) {
    loguniformLowerBnds = actualModel.loguniform_lower_bounds();
    loguniformUpperBnds = actualModel.loguniform_upper_bounds();
  }
  if (!actualModel.triangular_modes().empty()) {
    triangularModes     = actualModel.triangular_modes();
    triangularLowerBnds = actualModel.triangular_lower_bounds();
    triangularUpperBnds = actualModel.triangular_upper_bounds();
  }
  if (!actualModel.exponential_betas().empty())
    exponentialBetas = actualModel.exponential_betas();
  if (!actualModel.beta_alphas().empty()) {
    betaAlphas    = actualModel.beta_alphas();
    betaBetas     = actualModel.beta_betas();
    betaLowerBnds = actualModel.beta_lower_bounds();
    betaUpperBnds = actualModel.beta_upper_bounds();
  }
  if (!actualModel.gamma_alphas().empty()) {
    gammaAlphas = actualModel.gamma_alphas();
    gammaBetas  = actualModel.gamma_betas();
  }
  if (!actualModel.gumbel_alphas().empty()) {
    gumbelAlphas = actualModel.gumbel_alphas();
    gumbelBetas  = actualModel.gumbel_betas();
  }
  if (!actualModel.frechet_alphas().empty()) {
    frechetAlphas = actualModel.frechet_alphas();
    frechetBetas  = actualModel.frechet_betas();
  }
  if (!actualModel.weibull_alphas().empty()) {
    weibullAlphas = actualModel.weibull_alphas();
    weibullBetas  = actualModel.weibull_betas();
  }
  if (!actualModel.histogram_bin_pairs().empty())
    histogramBinPairs = actualModel.histogram_bin_pairs();
  if (!actualModel.histogram_point_pairs().empty())
    histogramPointPairs = actualModel.histogram_point_pairs();
  if (!actualModel.interval_probabilities().empty()) {
    intervalBasicProbs = actualModel.interval_probabilities();
    intervalBounds     = actualModel.interval_bounds();
  }
  if (!actualModel.uncertain_correlations().empty())
    uncertainCorrelations = actualModel.uncertain_correlations();

  // linear constraints

  if (actualModel.num_linear_ineq_constraints()) {
    // the views don't necessarily have to be the same, but the number of
    // active continuous and active discrete variables have to be consistent.
    if (actualModel.cv() == currentVariables.cv() &&
	actualModel.dv() == currentVariables.dv()) {
      userDefinedConstraints.linear_ineq_constraint_coeffs(
        actualModel.linear_ineq_constraint_coeffs());
      userDefinedConstraints.linear_ineq_constraint_lower_bounds(
        actualModel.linear_ineq_constraint_lower_bounds());
      userDefinedConstraints.linear_ineq_constraint_upper_bounds(
        actualModel.linear_ineq_constraint_upper_bounds());
    }
    else {
      Cerr << "Error: cannot update linear inequality constraints in "
	   << "DataFitSurrModel::update_from_actual_model() due to "
	   << "inconsistent active variables." << endl;
      abort_handler(-1);
    }
  }
  if (actualModel.num_linear_eq_constraints()) {
    // the views don't necessarily have to be the same, but the number of
    // active continuous and active discrete variables have to be consistent.
    if (actualModel.cv() == currentVariables.cv() &&
	actualModel.dv() == currentVariables.dv()) {
      userDefinedConstraints.linear_eq_constraint_coeffs(
        actualModel.linear_eq_constraint_coeffs());
      userDefinedConstraints.linear_eq_constraint_targets(
        actualModel.linear_eq_constraint_targets());
    }
    else {
      Cerr << "Error: cannot update linear equality constraints in "
	   << "DataFitSurrModel::update_from_actual_model() due to "
	   << "inconsistent active variables." << endl;
      abort_handler(-1);
    }
  }

  // nonlinear constraints

  if (actualModel.num_nonlinear_ineq_constraints()) {
    userDefinedConstraints.nonlinear_ineq_constraint_lower_bounds(
      actualModel.nonlinear_ineq_constraint_lower_bounds());
    userDefinedConstraints.nonlinear_ineq_constraint_upper_bounds(
      actualModel.nonlinear_ineq_constraint_upper_bounds());
  }
  if (actualModel.num_nonlinear_eq_constraints())
    userDefinedConstraints.nonlinear_eq_constraint_targets(
      actualModel.nonlinear_eq_constraint_targets());
}

} // namespace Dakota
