/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       PStudyDACE
//- Description: Implementation code for the PStudyDACE class
//- Owner:       Mike Eldred

#include "system_defs.h"
#include "DakotaPStudyDACE.H"
#include "ParamResponsePair.H"
#include "PRPCache.H"
#include "ProblemDescDB.H"

static const char rcsId[]="@(#) $Id: DakotaPStudyDACE.C 5364 2008-10-16 03:16:39Z mseldre $";


namespace Dakota {

PStudyDACE::PStudyDACE(Model& model): Analyzer(model),
  bestVariables(model.current_variables().copy()),
  bestResponse(model.current_response().copy()),
  numObjFns(probDescDB.get_sizet("responses.num_objective_functions")),
  numLSqTerms(probDescDB.get_sizet("responses.num_least_squares_terms"))
{
  // Initialize bestResponse with an ASV of 1's as the default data set
  ShortArray asv(numFunctions, 1);
  bestResponse.active_set_request_vector(asv);

  // Check for vendor numerical gradients (manage_asv will not work properly)
  if (gradientType == "numerical" && methodSource == "vendor") {
    Cerr << "\nError: ParamStudy/DACE do not contain a vendor algorithm for "
         << "numerical derivatives;\n       please select dakota as the finite "
	 << "difference method_source." << endl;
    abort_handler(-1);
  }
}


PStudyDACE::PStudyDACE(NoDBBaseConstructor, Model& model):
  Analyzer(NoDBBaseConstructor(), model),
  bestVariables(model.current_variables().copy()),
  bestResponse(model.current_response().copy()),
  numObjFns(0), numLSqTerms(0) // generic data set: no best data tracking
{
  // Initialize bestResponse with an ASV of 1's as the default data set
  ShortArray asv(numFunctions, 1);
  bestResponse.active_set_request_vector(asv);

  // Check for vendor numerical gradients (manage_asv will not work properly)
  if (gradientType == "numerical" && methodSource == "vendor") {
    Cerr << "\nError: ParamStudy/DACE do not contain a vendor algorithm for "
         << "numerical derivatives;\n       please select dakota as the finite "
	 << "difference method_source." << endl;
    abort_handler(-1);
  }
}


PStudyDACE::~PStudyDACE() { }


void PStudyDACE::print_results(ostream& s)
{
  if (qualityFlag)
    s << "\nThe following lists volumetric uniformity measures\n"
      << "For all of these measures, smaller values are better"
      << "\nChi measure is: " << chiMeas << "\nD measure is: " << dMeas
      << "\nH measure is: " << hMeas << "\nTau measure is: " << tauMeas
      << "\n" <<  endl;

  if (!numObjFns && !numLSqTerms) {
    s << "<<<<< Best data metrics not defined for generic response functions\n";
    return;
  }

  // Print best parameters
  s << "<<<<< Best parameters          =\n" << bestVariables;

  // Print best response functions.  Since update_best assumes an
  // optimization data set, then print_results will as well.
  const RealVector& fn_vals_star = bestResponse.function_values();
  size_t offset, num_fns = fn_vals_star.length();
  if (numObjFns) {
    if (numObjFns > 1)
      s << "<<<<< Best objective functions =\n";
    else
      s << "<<<<< Best objective function  =\n";
    fn_vals_star.write_partial(s, 0, numObjFns);
    offset = numObjFns;
  }
  else if (numLSqTerms) {
    s << "<<<<< Best residual terms      =\n";
    fn_vals_star.write_partial(s, 0, numLSqTerms);
    offset = numLSqTerms;
  }
  if (num_fns > offset) {
    s << "<<<<< Best constraint values   =\n";
    fn_vals_star.write_partial(s, offset, num_fns - offset);
  }

  extern PRPCache data_pairs; // global container
  int eval_id;
  activeSet.request_values(1);
  if (lookup_by_val(data_pairs, iteratedModel.interface_id(), bestVariables,
		    activeSet, eval_id))
    s << "<<<<< Best data captured at function evaluation " << eval_id << endl;
  else
    s << "<<<<< Best data not found in evaluation cache" << endl;
}


void PStudyDACE::
update_best(const RealVector& vars, const Response& response, 
	    const int eval_num)
{
  // Updating best in the context of multilevel optimization.  Therefore,
  // if constraints are violated in the current best, then only a reduction in
  // sum of squares constraint violation (more feasible design) is accepted as
  // an improvement.  Likewise, if constraints are not violated in the current 
  // best, then only a design without constraint violation and lower objective
  // is accepted as an improvement.  If eval_num == 0, then this is the first
  // evaluation and the best variables need to be set.

  size_t i, constr_offset;
  const RealVector& fn_vals = response.function_values();
  const RealVector& primary_wts = iteratedModel.primary_response_fn_weights();
  Real obj_fn = 0.0;
  if (numObjFns) {
    constr_offset = numObjFns;
    if (primary_wts.empty()) {
      for (i=0; i<numObjFns; i++)
	obj_fn += fn_vals[i];
      if (numObjFns > 1)
	obj_fn /= (Real)numObjFns;
    }
    else
      for (i=0; i<numObjFns; i++)
	obj_fn += primary_wts[i] * fn_vals[i];
  }
  else if (numLSqTerms) {
    constr_offset = numLSqTerms;
    if (primary_wts.empty())
      for (i=0; i<numLSqTerms; i++)
	obj_fn += pow(fn_vals[i], 2);
    else
      for (i=0; i<numLSqTerms; i++)
	obj_fn += pow(primary_wts[i]*fn_vals[i], 2);
  }
  else // no "best" metric currently defined for generic response fns
    return;
  Real constr_viol = 0.0;
  size_t num_nln_ineq = iteratedModel.num_nonlinear_ineq_constraints(),
         num_nln_eq   = iteratedModel.num_nonlinear_eq_constraints();
  const RealVector& nln_ineq_lwr_bnds
    = iteratedModel.nonlinear_ineq_constraint_lower_bounds();
  const RealVector& nln_ineq_upr_bnds
    = iteratedModel.nonlinear_ineq_constraint_upper_bounds();
  const RealVector& nln_eq_targets
    = iteratedModel.nonlinear_eq_constraint_targets();
  for (i=0; i<num_nln_ineq; i++) { // ineq constraint violation
    size_t index = i + constr_offset;
    Real ineq_con = fn_vals[index];
    if (ineq_con > nln_ineq_upr_bnds[i])
      constr_viol += pow(ineq_con - nln_ineq_upr_bnds[i],2);
    else if (ineq_con < nln_ineq_lwr_bnds[i])
      constr_viol += pow(nln_ineq_lwr_bnds[i] - ineq_con,2);
  }
  for (i=0; i<num_nln_eq; i++) { // eq constraint violation
    size_t index = i + constr_offset + num_nln_ineq;
    Real eq_con = fn_vals[index];
    if (fabs(eq_con - nln_eq_targets[i]) > 0.)
      constr_viol += pow(eq_con - nln_eq_targets[i], 2);
  }

  // Set best values if initial evaluation or if there is improvement.
  size_t num_nln_con = num_nln_ineq + num_nln_eq;
  if (eval_num == 0 || (!num_nln_con && obj_fn < bestObjFn) ||
       (num_nln_con && (constr_viol < bestConViol || 
       (bestConViol == constr_viol && obj_fn < bestObjFn) ) ) ) {
    //Cout << "Improvement detected: setting best vals to current vals.\n";
    bestVariables.continuous_variables(vars);
    bestResponse.copy_results(response);
    bestObjFn   = obj_fn;
    bestConViol = constr_viol;
  }
}

} // namespace Dakota
