/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        Optimizer
//- Description:  Abstract base class to logically represent a variety
//-               of DAKOTA optimizer objects in a generic fashion.
//- Owner:        Mike Eldred
//- Version: $Id: DakotaOptimizer.H 5373 2008-10-17 02:36:46Z mseldre $

#ifndef DAKOTA_OPTIMIZER_H
#define DAKOTA_OPTIMIZER_H

#include "DakotaMinimizer.H"

namespace Dakota {


/// Base class for the optimizer branch of the iterator hierarchy.

/** The Optimizer class provides common data and functionality for
    DOTOptimizer, CONMINOptimizer, NPSOLOptimizer, SNLLOptimizer,
    NLPQLPOptimizer, COLINOptimizer, and JEGAOptimizer. */

class Optimizer: public Minimizer
{
public:

protected:

  //
  //- Heading: Constructors and destructor
  //

  Optimizer();             ///< default constructor
  Optimizer(Model& model); ///< standard constructor

  /// alternate constructor for "on the fly" instantiations
  Optimizer(NoDBBaseConstructor, Model& model);
  /// alternate constructor for "on the fly" instantiations
  Optimizer(NoDBBaseConstructor, size_t num_cv, size_t num_dv,
	    size_t num_lin_ineq, size_t num_lin_eq, size_t num_nln_ineq,
	    size_t num_nln_eq);

  ~Optimizer();            ///< destructor

  //
  //- Heading: Virtual member function redefinitions
  //

  void derived_pre_run();
  void run();
  void derived_post_run();
  void print_results(ostream& s);

  //
  //- Heading: New virtual member functions
  //

  /// Used within the optimizer branch for computing the optimal solution.
  /// Redefines the run virtual function for the optimizer branch.
  virtual void find_optimum() = 0;

  //
  //- Heading: Data
  //
  
  /// number of objective functions (iterator view)
  size_t numObjectiveFns;
  /// number of objective functions (user's model view)
  size_t numUserObjectiveFns;

  /// flag indicating whether multi-objective transformations are necessary
  bool multiObjFlag;

  /// pointer to Optimizer instance used in static member functions
  static Optimizer* optimizerInstance;
  /// pointer containing previous value of optimizerInstance
  Optimizer* prevOptInstance;

private:

  //
  //- Heading: Convenience/Helper functions
  //

  /// primary response conversion map for RecastModel used in scaling and 
  /// multiobjective:  transform objectives (fns, grads, Hessians)
  /// from native (user) to iterator space
  static void primary_resp_recast(const Variables& native_vars,
				  const Variables& scaled_vars,
				  const Response& native_response,
				  Response& scaled_response);

  /// forward mapping: maps multiple objective functions to a single
  /// weighted objective for single-objective optimizers
  void weighted_sum(const Response& full_response, Response& reduced_response,
		    const RealVector& wts) const;

  /// inverse mapping: retrieves values for multiple objective functions
  /// from the solution of a single-objective optimizer
  void multi_objective_retrieve(const Variables& vars,
				Response& response) const;

  //
  //- Heading: Data
  //
};


inline Optimizer::Optimizer(): multiObjFlag(false)
{ }


inline Optimizer::~Optimizer()
{ 
  if (scaleFlag || multiObjFlag)
    iteratedModel.free_communicators(maxConcurrency);
}

inline void Optimizer::run()
{ find_optimum(); }


//inline const VariablesArray& Optimizer::variables_array_results() const
//{ return bestVariablesArray; }


//inline const ResponseArray& Optimizer::response_array_results() const
//{ return bestResponseArray; }

} // namespace Dakota

#endif
