/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonD
//- Description: Base class for NonDeterministic methods
//- Owner:       Mike Eldred
//- Checked by:
//- Version:

#include "NestedModel.H"
#include "DakotaResponse.H"
#include "DakotaNonD.H"
#include "ProblemDescDB.H"

static const char rcsId[]="@(#) $Id: DakotaNonD.C 5337 2008-10-09 18:12:05Z mseldre $";

namespace Dakota {

// initialization of statics
NonD* NonD::nondInstance(NULL);
const Real NonD::Pi(3.1415926535897932385);


NonD::NonD(Model& model): Analyzer(model),
#ifdef DAKOTA_PECOS
  natafTransform("nataf"),
#endif
  numDesignVars(0), numStateVars(0),
  numNormalVars(probDescDB.get_sizet("variables.normal_uncertain")),
  numLognormalVars(probDescDB.get_sizet("variables.lognormal_uncertain")),
  numUniformVars(probDescDB.get_sizet("variables.uniform_uncertain")),
  numLoguniformVars(probDescDB.get_sizet("variables.loguniform_uncertain")),
  numTriangularVars(probDescDB.get_sizet("variables.triangular_uncertain")),
  numExponentialVars(probDescDB.get_sizet("variables.exponential_uncertain")),
  numBetaVars(probDescDB.get_sizet("variables.beta_uncertain")),
  numGammaVars(probDescDB.get_sizet("variables.gamma_uncertain")),
  numGumbelVars(probDescDB.get_sizet("variables.gumbel_uncertain")),
  numFrechetVars(probDescDB.get_sizet("variables.frechet_uncertain")),
  numWeibullVars(probDescDB.get_sizet("variables.weibull_uncertain")),
  numHistogramVars(probDescDB.get_sizet("variables.histogram_uncertain")),
  numIntervalVars(probDescDB.get_sizet("variables.interval_uncertain")),
  numUncertainVars(numNormalVars     + numLognormalVars  + numUniformVars     +
                   numLoguniformVars + numTriangularVars + numExponentialVars +
		   numBetaVars       + numGammaVars      + numGumbelVars      +
		   numFrechetVars    + numWeibullVars    + numHistogramVars   +
		   numIntervalVars),
  numResponseFunctions(
    probDescDB.get_sizet("responses.num_response_functions")),
  requestedRespLevels(probDescDB.get_drva("method.nond.response_levels")),
  requestedProbLevels(probDescDB.get_drva("method.nond.probability_levels")),
  requestedRelLevels(probDescDB.get_drva("method.nond.reliability_levels")),
  requestedGenRelLevels(
    probDescDB.get_drva("method.nond.gen_reliability_levels")),
  totalLevelRequests(0), distParamDerivs(false)
{
  // initialize convenience flags & enums
  cdfFlag = (probDescDB.get_string("method.nond.distribution") ==
             "complementary") ? false : true;
  const String& resp_mapping
    = probDescDB.get_string("method.nond.response_level_mapping_type");
  if (resp_mapping == "probabilities")
    respLevelTarget = PROBABILITIES;
  else if (resp_mapping == "reliabilities")
    respLevelTarget = RELIABILITIES;
  else if (resp_mapping == "gen_reliabilities")
    respLevelTarget = GEN_RELIABILITIES;
  else {
    Cerr << "Error: bad response level mapping type in DakotaNonD constructor."
	 << endl;
    abort_handler(-1);
  }

  const RealSymDenseMatrix& uncertain_corr = model.uncertain_correlations();
  if (!uncertain_corr.empty())
    natafTransform.initialize_random_variable_correlations(uncertain_corr);

  // When specifying z/p/beta/beta* levels, a spec with an index key (number of
  // levels = list of ints) will result in multiple vectors of levels, one for
  // each response fn.  A spec without an index key will result in a single
  // vector of levels.  This is as much logic as the parser can support since it
  // cannot access the number of response fns.  Here, support the shorthand spec
  // where there are multiple response fns but only one vector of levels (no
  // index key provided), which are to be evenly distributed among the response
  // fns.  This provides some measure of backwards compatibility.
  distribute_levels(requestedRespLevels);
  distribute_levels(requestedProbLevels);
  distribute_levels(requestedRelLevels);
  distribute_levels(requestedGenRelLevels);

  for (size_t i=0; i<numFunctions; i++)
    totalLevelRequests += requestedRespLevels[i].length() +
      requestedProbLevels[i].length() + requestedRelLevels[i].length() +
      requestedGenRelLevels[i].length();
}


NonD::NonD(NoDBBaseConstructor, Model& model):
  Analyzer(NoDBBaseConstructor(), model),
#ifdef DAKOTA_PECOS
  natafTransform("nataf"),
#endif
  numDesignVars(0), numStateVars(0),
  numNormalVars(model.normal_means().length()),
  numLognormalVars(model.lognormal_means().length()),
  numUniformVars(model.uniform_lower_bounds().length()),
  numLoguniformVars(model.loguniform_lower_bounds().length()),
  numTriangularVars(model.triangular_modes().length()),
  numExponentialVars(model.exponential_betas().length()),
  numBetaVars(model.beta_alphas().length()),
  numGammaVars(model.gamma_alphas().length()),
  numGumbelVars(model.gumbel_alphas().length()),
  numFrechetVars(model.frechet_alphas().length()),
  numWeibullVars(model.weibull_alphas().length()),
  numHistogramVars(model.histogram_bin_pairs().length() +
		   model.histogram_point_pairs().length()),
  numIntervalVars(model.interval_probabilities().length()),
  numUncertainVars(numNormalVars     + numLognormalVars  + numUniformVars     +
                   numLoguniformVars + numTriangularVars + numExponentialVars +
		   numBetaVars       + numGammaVars      + numGumbelVars      +
		   numFrechetVars    + numWeibullVars    + numHistogramVars   +
		   numIntervalVars),
  numResponseFunctions(numFunctions), totalLevelRequests(0), cdfFlag(true),
  distParamDerivs(false)
{
  // NonDEvidence and NonDAdaptImpSampling use this ctor

  const RealSymDenseMatrix& uncertain_corr = model.uncertain_correlations();
  if (!uncertain_corr.empty())
    natafTransform.initialize_random_variable_correlations(uncertain_corr);

  // current set of statistics is mean, standard deviation, and
  // probability of failure for each response function
  //ShortArray asv(3*numFunctions, 1);
  //finalStatistics = Response(numUncertainVars, asv);
}


NonD::NonD(NoDBBaseConstructor, const RealVector& lower_bnds,
	   const RealVector& upper_bnds):
  Analyzer(NoDBBaseConstructor()),
#ifdef DAKOTA_PECOS
  natafTransform("nataf"),
#endif
  numDesignVars(0),
  numStateVars(0), numNormalVars(0), numLognormalVars(0),
  numUniformVars(lower_bnds.length()), numLoguniformVars(0),
  numTriangularVars(0), numExponentialVars(0), numBetaVars(0), numGammaVars(0),
  numGumbelVars(0), numFrechetVars(0), numWeibullVars(0), numHistogramVars(0),
  numIntervalVars(0), numUncertainVars(numUniformVars), numResponseFunctions(0),
  totalLevelRequests(0), cdfFlag(true), distParamDerivs(false)
{
  // ConcurrentStrategy uses this ctor for design opt, either for multi-start
  // initial points or multibjective weight sets.

  numContinuousVars = numUniformVars;
  numDiscreteVars   = 0;
}


void NonD::
requested_levels(const RealVectorArray& req_resp_levels,
		 const RealVectorArray& req_prob_levels,
		 const RealVectorArray& req_rel_levels,
		 const RealVectorArray& req_gen_rel_levels,
		 short resp_lev_target, bool cdf_flag)
{
  requestedRespLevels   = req_resp_levels;
  requestedProbLevels   = req_prob_levels;
  requestedRelLevels    = req_rel_levels;
  requestedGenRelLevels = req_gen_rel_levels;
  respLevelTarget       = resp_lev_target;
  cdfFlag               = cdf_flag;

  // In current usage, incoming levels are already distributed.
  // But if they're not, distribute them now.
  distribute_levels(requestedRespLevels);
  distribute_levels(requestedProbLevels);
  distribute_levels(requestedRelLevels);
  distribute_levels(requestedGenRelLevels);

  for (size_t i=0; i<numFunctions; i++)
    totalLevelRequests += requestedRespLevels[i].length() +
      requestedProbLevels[i].length() + requestedRelLevels[i].length() +
      requestedGenRelLevels[i].length();

  initialize_final_statistics();
}


void NonD::distribute_levels(RealVectorArray& levels)
{
  size_t i, j, num_level_arrays = levels.length();
  if (num_level_arrays != numFunctions) {
    if (num_level_arrays == 0) // create array of empty vectors
      levels.reshape(numFunctions);
      // NOTE: old response_thresholds default was a single 0 per resp fn.  New
      // default is no output level calculations if no requested input levels.
    else if (num_level_arrays == 1) { // evenly distribute among all resp fns
      RealVector level_array0 = levels[0];
      size_t total_len = level_array0.length();
      // check for divisibility
      if (total_len%numFunctions) {
        Cerr << "\nError: number of levels not evenly divisible by the number "
             << "of response functions." << endl;
        abort_handler(-1);
      }
      levels.reshape(numFunctions);
      size_t new_len = total_len/numFunctions;
      RealVector new_vec(new_len);
      for (i=0; i<numFunctions; i++) {
        for (j=0; j<new_len; j++)
          new_vec[j] = level_array0[i*new_len+j];
        levels[i] = new_vec;
      }
    }
    else {
      Cerr << "\nError: num_levels specification differs from the number of "
           << "response functions." << endl;
      abort_handler(-1);
    }
  }
}


/** Default definition of virtual function (used by sampling,
    reliability, and polynomial chaos) defines the set of statistical
    results to include means, standard deviations, and level mappings. */
void NonD::initialize_final_statistics()
{
  size_t num_final_stats = 2*numFunctions, num_active_vars = iteratedModel.cv();
  if (!numIntervalVars) // aleatory UQ
    num_final_stats += totalLevelRequests;
  ActiveSet stats_set(num_final_stats, num_active_vars);
  finalStatistics = Response(stats_set);

  // Assign meaningful labels to finalStatistics (appear in NestedModel output)
  size_t i, j, num_levels, cntr = 0;
  char tag_string[10], lev_string[10];
  StringArray stats_labels(num_final_stats);
  for (i=0; i<numFunctions; i++) {
    sprintf(tag_string, "_r%i", i+1);
    if (numIntervalVars) { // epistemic & mixed aleatory/epistemic
      stats_labels[cntr++] = String("z_lo") + String(tag_string);
      stats_labels[cntr++] = String("z_up") + String(tag_string);
    }
    else {                  // aleatory
      stats_labels[cntr++] = String("mean")    + String(tag_string);
      stats_labels[cntr++] = String("std_dev") + String(tag_string);
      num_levels = requestedRespLevels[i].length();
      for (j=0; j<num_levels; j++, cntr++) {
	stats_labels[cntr] = (cdfFlag) ? String("cdf") : String("ccdf");
	switch (respLevelTarget) {
	case PROBABILITIES:
	  sprintf(lev_string, "_plev%i",  j+1); break;
	case RELIABILITIES:
	  sprintf(lev_string, "_blev%i",  j+1); break;
	case GEN_RELIABILITIES:
	  sprintf(lev_string, "_b*lev%i", j+1); break;
	}
	stats_labels[cntr] += String(lev_string) + String(tag_string);
      }
      num_levels = requestedProbLevels[i].length() +
	requestedRelLevels[i].length() + requestedGenRelLevels[i].length();
      for (j=0; j<num_levels; j++, cntr++) {
	stats_labels[cntr] = (cdfFlag) ? String("cdf") : String("ccdf");
	sprintf(lev_string, "_zlev%i", j+1);
	stats_labels[cntr] += String(lev_string) + String(tag_string);
      }
    }
  }
  finalStatistics.function_labels(stats_labels);
}


/** Map the variables from iterator space (u) to simulation space (x). */
void NonD::vars_u_to_x_mapping(const Variables& u_vars, Variables& x_vars)
{
  RealVector x_rv;
  nondInstance->trans_U_to_X(u_vars.continuous_variables(), x_rv);
  x_vars.continuous_variables(x_rv);
}


/** Define the DVV for x-space derivative evaluations by augmenting
    the iterator requests to account for correlations. */
void NonD::set_u_to_x_mapping(const ActiveSet& u_set, ActiveSet& x_set)
{
  //if (nondInstance->distParamDerivs) {
  //}
  //else
  if (nondInstance->natafTransform.x_correlation()) {
    const UIntArray& u_dvv = u_set.derivative_vector();
    const UIntArray& cv_ids
      = nondInstance->iteratedModel.continuous_variable_ids();
    const UIntArray& icv_ids
      = nondInstance->iteratedModel.inactive_continuous_variable_ids();
    bool std_dvv = (u_dvv == cv_ids || u_dvv == icv_ids);
    if (!std_dvv) { // partial random variable derivatives: check correlations
      const UIntArray& acv_ids
	= nondInstance->iteratedModel.all_continuous_variable_ids();
      size_t i, j, num_cv = cv_ids.length(), num_acv = acv_ids.length();
      UIntArray x_dvv;
      const RealSymDenseMatrix& corr_x
	= nondInstance->natafTransform.x_correlation_matrix();
      for (i=0; i<num_acv; i++) { // insert in ascending order
	unsigned int acv_id = acv_ids[i];
	if (u_dvv.contains(acv_id))
	  x_dvv.push_back(acv_id);
	else {
	  size_t cv_index = cv_ids.index(acv_id);
	  if (cv_index != _NPOS) { // random var: check correlation
	    for (j=0; j<num_cv; j++) {
	      if (cv_index != j && fabs(corr_x(cv_index, j)) > 1.e-25 &&
		  u_dvv.contains(cv_ids[j])) {
		x_dvv.push_back(acv_id);
		break;
	      }
	    }
	  }
	}
      }
      x_set.derivative_vector(x_dvv);
    }
  }
}


void NonD::
resp_x_to_u_mapping(const Variables& x_vars,     const Variables& u_vars,
		    const Response&  x_response, Response&        u_response)
{
  const RealVector&      x_rv       = x_vars.continuous_variables();
  const UIntArray&       cv_ids     = x_vars.continuous_variable_ids();
  const UIntArray&       acv_ids    = x_vars.all_continuous_variable_ids();
  const RealVector&      x_fns      = x_response.function_values();
  const RealMatrix&      x_grads    = x_response.function_gradients();
  const RealMatrixArray& x_hessians = x_response.function_hessians();

  // In this recasting, the inputs and outputs are mapped one-to-one, with no
  // reordering.  However, the x-space ASV may be augmented from the original
  // u-space ASV due to nonlinear mapping logic in RecastModel::asv_mapping().
  const ShortArray& u_asv = u_response.active_set_request_vector();
  const UIntArray&  u_dvv = u_response.active_set_derivative_vector();
  const ShortArray& x_asv = x_response.active_set_request_vector();
  const UIntArray&  x_dvv = x_response.active_set_derivative_vector();
  size_t i, j, num_fns = x_asv.length(), num_deriv_vars = x_dvv.length();
  if (u_asv.length() != num_fns) {
    Cerr << "Error: inconsistent response function definition in NonD::"
	 << "resp_x_to_u_mapping()" << endl;
    abort_handler(-1);
  }
  if (!nondInstance->natafTransform.x_correlation() && u_dvv != x_dvv) {
    Cerr << "Error: inconsistent derivative component definition in NonD::"
	 << "resp_x_to_u_mapping()" << endl;
    abort_handler(-1);
  }
  bool u_grad_flag = false, u_hess_flag = false;
  for (i=0; i<num_fns; i++) {
    if (u_asv[i] & 2)
      u_grad_flag = true;
    if (u_asv[i] & 4)
      u_hess_flag = true;
  }

  const Pecos::ShortArray& x_types = nondInstance->natafTransform.x_types();
  bool map_derivs = ( (u_grad_flag || u_hess_flag) &&
		      u_dvv != u_vars.inactive_continuous_variable_ids() );
  bool nonlinear_vars_map = ( ( !nondInstance->extendedUSpace &&
	 nondInstance->numUncertainVars !=
	 std::count(x_types.begin(), x_types.end(), (short)NORMAL) )
    || ( nondInstance->extendedUSpace && 
	 nondInstance->numUncertainVars !=
	 std::count(x_types.begin(), x_types.end(), (short)NORMAL)      +
	 std::count(x_types.begin(), x_types.end(), (short)UNIFORM)     +
	 std::count(x_types.begin(), x_types.end(), (short)EXPONENTIAL) +
	 std::count(x_types.begin(), x_types.end(), (short)BETA)        +
	 std::count(x_types.begin(), x_types.end(), (short)GAMMA) ) );

  RealDenseVector         fn_grad_x_rdv, fn_grad_us_rdv;
  RealSymDenseMatrix      fn_hess_x_rsdm, fn_hess_u_rsdm;
  RealBaseVector          fn_grad_us_rbv;
  RealMatrix              fn_hess_u_rm;
  RealDenseMatrix         jacobian_xu, jacobian_xs;
  RealSymDenseMatrixArray hessian_xu;

  if (map_derivs) {
    // The following transformation data is invariant w.r.t. the response fns
    // and is computed outside of the num_fns loop
    RealDenseVector x_rdv;
    copy_data(x_rv, x_rdv);
    if (nondInstance->distParamDerivs)
      nondInstance->natafTransform.jacobian_dX_dS(x_rdv, jacobian_xs,
	cv_ids, acv_ids, nondInstance->primaryACVarMapIndices,
	nondInstance->secondaryACVarMapTargets);
    else {
      if (u_grad_flag || u_hess_flag)
	nondInstance->natafTransform.jacobian_dX_dU(x_rdv, jacobian_xu);
      if (u_hess_flag && nonlinear_vars_map)
	nondInstance->natafTransform.hessian_d2X_dU2(x_rdv, hessian_xu);
    }
  }

  for (i=0; i<num_fns; i++) {
    short u_asv_val = u_asv[i]; // original request from iterator
    short x_asv_val = x_asv[i]; // mapped request for sub-model

    // map value g(x) to G(u)
    if (u_asv_val & 1) {
      if ( !(x_asv_val & 1) ) {
	Cerr << "Error: missing required sub-model data in NonD::"
	     << "resp_x_to_u_mapping()" << endl;
	abort_handler(-1);
      }
      // no transformation: g(x) = G(u) by definition
      u_response.function_value(x_fns[i], i);
    }

    // manage data requirements for derivative transformations: if fn_grad_x_rdv
    // is needed for Hessian x-form (nonlinear I/O mapping), then x_asv has been
    // augmented to include the gradient in RecastModel::asv_mapping().
    if (map_derivs && (x_asv_val & 2))
      copy_data(x_grads[i], fn_grad_x_rdv);

    // map gradient dg/dx to dG/du
    if (u_asv_val & 2) {
      if ( !(x_asv_val & 2) ) {
	Cerr << "Error: missing required sub-model data in NonD::"
	     << "resp_x_to_u_mapping()" << endl;
	abort_handler(-1);
      }
      if (map_derivs) { // perform transformation

	if (nondInstance->distParamDerivs) // transform subset of components
	  nondInstance->natafTransform.trans_grad_X_to_S(fn_grad_x_rdv,
	    fn_grad_us_rdv, jacobian_xs, x_dvv, cv_ids, acv_ids,
	    nondInstance->primaryACVarMapIndices,
	    nondInstance->secondaryACVarMapTargets);
	else   // transform subset of components
	  nondInstance->natafTransform.trans_grad_X_to_U(fn_grad_x_rdv,
	    fn_grad_us_rdv, jacobian_xu, x_dvv, cv_ids);
	copy_data(fn_grad_us_rdv, fn_grad_us_rbv);
	u_response.function_gradient(fn_grad_us_rbv, i);
      }
      else // no transformation: dg/dx = dG/du
	u_response.function_gradient(x_grads[i], i);
    }

    // map Hessian d^2g/dx^2 to d^2G/du^2
    if (u_asv_val & 4) {
      if ( !(x_asv_val & 4) ||
	   ( map_derivs && nonlinear_vars_map && !(x_asv_val & 2) ) ) {
	Cerr << "Error: missing required sub-model data in NonD::"
	     << "resp_x_to_u_mapping()" << endl;
	abort_handler(-1);
      }
      if (map_derivs) { // perform transformation
	copy_data(x_hessians[i], fn_hess_x_rsdm);
	if (nondInstance->distParamDerivs) { // transform subset of components
	  Cerr << "Error: Hessians with respect to inserted variables not yet "
	       << "supported." << endl;
	  abort_handler(-1);;
	  //nondInstance->natafTransform.trans_hess_X_to_S(fn_hess_x_rsdm,
	  //  fn_hess_s_rsdm, jacobian_xs, hessian_xs, fn_grad_s_rdv, x_dvv,
	  //  cv_ids, x_vars.all_continuous_variable_ids(),
	  //  nondInstance->primaryACVarMapIndices,
	  //  nondInstance->secondaryACVarMapTargets);
	}
	else // transform subset of components
	  nondInstance->natafTransform.trans_hess_X_to_U(fn_hess_x_rsdm,
            fn_hess_u_rsdm, jacobian_xu, hessian_xu, fn_grad_x_rdv, x_dvv,
	    cv_ids);
	copy_data(fn_hess_u_rsdm, fn_hess_u_rm);
	u_response.function_hessian(fn_hess_u_rm, i);
      }
      else // no transformation: d^2g/dx^2 = d^2G/du^2
	u_response.function_hessian(x_hessians[i], i);
    }
  }

#ifdef DEBUG
  Cout << "\nx_response:\n" << x_response
       << "\nu_response:\n" << u_response << endl;
#endif
}


/** Build ProbabilityTransformation::ranVar arrays containing the
    uncertain variable distribution types and their corresponding
    means/standard deviations.  This function is used when the Model
    variables are in x-space. */
void NonD::initialize_random_variables()
{
  initialize_random_variable_types();
  initialize_random_variable_parameters();
}


/** Build ProbabilityTransformation::ranVar arrays containing the
    uncertain variable distribution types and their corresponding
    means/standard deviations.  This function is used when the Model
    variables are in x-space. */
void NonD::initialize_random_variable_types()
{
  size_t i, av_cntr = 0, num_active_vars = iteratedModel.cv();
  ShortArray x_types(num_active_vars), u_types(num_active_vars);

  for (i=0; i<numDesignVars; i++, av_cntr++) {
    x_types[av_cntr] = DESIGN;
    u_types[av_cntr] = UNIFORM;
  }
  const RealDenseVector& n_l_bnds = iteratedModel.normal_lower_bounds();
  const RealDenseVector& n_u_bnds = iteratedModel.normal_upper_bounds();
  for (i=0; i<numNormalVars; i++, av_cntr++) {
    x_types[av_cntr] = (n_l_bnds[i] > -DBL_MAX || n_u_bnds[i] < DBL_MAX) ?
      BOUNDED_NORMAL : NORMAL;
    u_types[av_cntr] = NORMAL;
  }
  const RealDenseVector& ln_l_bnds = iteratedModel.lognormal_lower_bounds();
  const RealDenseVector& ln_u_bnds = iteratedModel.lognormal_upper_bounds();
  for (i=0; i<numLognormalVars; i++, av_cntr++) {
    x_types[av_cntr] = (ln_l_bnds[i] > 0. || ln_u_bnds[i] < DBL_MAX) ?
      BOUNDED_LOGNORMAL : LOGNORMAL;
    u_types[av_cntr] = NORMAL;
  }
  for (i=0; i<numUniformVars; i++, av_cntr++) {
    x_types[av_cntr] = UNIFORM;
    u_types[av_cntr] = (extendedUSpace) ? UNIFORM : NORMAL;
  }
  for (i=0; i<numLoguniformVars; i++, av_cntr++) {
    x_types[av_cntr] = LOGUNIFORM;
    u_types[av_cntr] = (extendedUSpace) ? UNIFORM : NORMAL;
  }
  for (i=0; i<numTriangularVars; i++, av_cntr++) {
    x_types[av_cntr] = TRIANGULAR;
    u_types[av_cntr] = (extendedUSpace) ? UNIFORM : NORMAL;// possibly BETA
  }
  for (i=0; i<numExponentialVars; i++, av_cntr++) {
    x_types[av_cntr] = EXPONENTIAL;
    u_types[av_cntr] = (extendedUSpace) ? EXPONENTIAL : NORMAL;
  }
  for (i=0; i<numBetaVars; i++, av_cntr++) {
    x_types[av_cntr] = BETA;
    u_types[av_cntr] = (extendedUSpace) ? BETA : NORMAL;
  }
  for (i=0; i<numGammaVars; i++, av_cntr++) {
    x_types[av_cntr] = GAMMA;
    u_types[av_cntr] = (extendedUSpace) ? GAMMA : NORMAL;
  }
  for (i=0; i<numGumbelVars; i++, av_cntr++) {
    x_types[av_cntr] = GUMBEL;
    u_types[av_cntr] = NORMAL; // 2-sided
  }
  for (i=0; i<numFrechetVars; i++, av_cntr++) {
    x_types[av_cntr] = FRECHET;
    u_types[av_cntr] = NORMAL; // 1-sided: possibly EXPONENTIAL or GAMMA
  }
  for (i=0; i<numWeibullVars; i++, av_cntr++) {
    x_types[av_cntr] = WEIBULL;
    u_types[av_cntr] = NORMAL; // 1-sided: possibly EXPONENTIAL or GAMMA
  }
  for (i=0; i<numStateVars; i++, av_cntr++) {
    x_types[av_cntr] = STATE;
    u_types[av_cntr] = UNIFORM;
  }

  natafTransform.initialize_random_variable_types(x_types, u_types);
}


/** Build ProbabilityTransformation::ranVar arrays containing the
    uncertain variable distribution types and their corresponding
    means/standard deviations.  This function is used when the Model
    variables are in x-space. */
void NonD::initialize_random_variable_parameters()
{
  size_t num_active_vars = iteratedModel.cv();
  RealDenseVector x_means(num_active_vars, false),
    x_std_devs(num_active_vars, false), x_l_bnds(num_active_vars, false),
    x_u_bnds(num_active_vars, false);
  RealDenseVectorArray x_addtl(num_active_vars);

  const RealVector& cv_l_bnds = iteratedModel.continuous_lower_bounds();
  const RealVector& cv_u_bnds = iteratedModel.continuous_upper_bounds();

  size_t i, av_cntr = 0;
  for (i=0; i<numDesignVars; i++, av_cntr++) {
    const Real& lwr = x_l_bnds[av_cntr] = cv_l_bnds[av_cntr];
    const Real& upr = x_u_bnds[av_cntr] = cv_u_bnds[av_cntr];
    natafTransform.moments_from_uniform_params(lwr, upr, x_means[av_cntr],
					       x_std_devs[av_cntr]);
  }
  if (numNormalVars) {
    const RealDenseVector& n_means    = iteratedModel.normal_means();
    const RealDenseVector& n_std_devs = iteratedModel.normal_std_deviations();
    const RealDenseVector& n_l_bnds   = iteratedModel.normal_lower_bounds();
    const RealDenseVector& n_u_bnds   = iteratedModel.normal_upper_bounds();
    for (i=0; i<numNormalVars; i++, av_cntr++) {
      x_means[av_cntr]    = n_means[i];
      x_std_devs[av_cntr] = n_std_devs[i];
      x_l_bnds[av_cntr]   = n_l_bnds[i];
      x_u_bnds[av_cntr]   = n_u_bnds[i];
    }
  }
  if (numLognormalVars) {
    const RealDenseVector& ln_means  = iteratedModel.lognormal_means();
    const RealDenseVector& ln_std_devs
      = iteratedModel.lognormal_std_deviations();
    const RealDenseVector& ln_err_facts
      = iteratedModel.lognormal_error_factors();
    const RealDenseVector& ln_l_bnds = iteratedModel.lognormal_lower_bounds();
    const RealDenseVector& ln_u_bnds = iteratedModel.lognormal_upper_bounds();
    bool ln_s_d = !ln_std_devs.empty();
    for (i=0; i<numLognormalVars; i++, av_cntr++) {
      // DAKOTA/UQ standardizes on use of the mean/std dev or mean/error factor 
      // of the actual lognormal distribution.  It does not use the mean/std dev
      // of the underlying normal distribution (see also NonDSampling.C for
      // mappings for LHS lognormal inputs).
      x_means[av_cntr]  = ln_means[i];
      x_l_bnds[av_cntr] = ln_l_bnds[i];
      x_u_bnds[av_cntr] = ln_u_bnds[i];
      if (ln_s_d)
	x_std_devs[av_cntr] = ln_std_devs[i];
      else {
	// Pre-process the error factors to compute the std dev of the lognormal
	// distribution.  See LHS SAND Report #98-0210, pp. 39-41.
	const Real& err_fact = ln_err_facts[i];
	natafTransform.moments_from_lognormal_params(ln_means[i], err_fact,
						     x_std_devs[av_cntr]);
	x_addtl[av_cntr].sizeUninitialized(1);
	x_addtl[av_cntr][0] = err_fact;
      }
    }
  }
  if (numUniformVars) {
    const RealDenseVector& u_l_bnds = iteratedModel.uniform_lower_bounds();
    const RealDenseVector& u_u_bnds = iteratedModel.uniform_upper_bounds();
    for (i=0; i<numUniformVars; i++, av_cntr++) {
      const Real& lwr = x_l_bnds[av_cntr] = u_l_bnds[i];
      const Real& upr = x_u_bnds[av_cntr] = u_u_bnds[i];
      natafTransform.moments_from_uniform_params(lwr, upr, x_means[av_cntr],
						 x_std_devs[av_cntr]);
    }
  }
  if (numLoguniformVars) {
    // see SAND98-0210 LHS manual, pp. 43-44
    const RealDenseVector& lu_l_bnds = iteratedModel.loguniform_lower_bounds();
    const RealDenseVector& lu_u_bnds = iteratedModel.loguniform_upper_bounds();
    for (i=0; i<numLoguniformVars; i++, av_cntr++) {
      const Real& lwr = x_l_bnds[av_cntr] = lu_l_bnds[i];
      const Real& upr = x_u_bnds[av_cntr] = lu_u_bnds[i];
      natafTransform.moments_from_loguniform_params(lwr, upr, x_means[av_cntr],
						    x_std_devs[av_cntr]);
    }
  }
  if (numTriangularVars) {
    // See Haldar and Mahadevan, p. 99
    const RealDenseVector& t_modes  = iteratedModel.triangular_modes();
    const RealDenseVector& t_l_bnds = iteratedModel.triangular_lower_bounds();
    const RealDenseVector& t_u_bnds = iteratedModel.triangular_upper_bounds();
    for (i=0; i<numTriangularVars; i++, av_cntr++) {
      const Real& mode = t_modes[i];
      const Real& lwr  = x_l_bnds[av_cntr] = t_l_bnds[i];
      const Real& upr  = x_u_bnds[av_cntr] = t_u_bnds[i];
      natafTransform.moments_from_triangular_params(lwr, upr, mode,
	x_means[av_cntr], x_std_devs[av_cntr]);
      x_addtl[av_cntr].sizeUninitialized(1);
      x_addtl[av_cntr][0] = mode;
    }
  }
  if (numExponentialVars) {
    // DAKOTA employs the 1/beta exp(-x/beta) definition, which differs from
    // the lambda exp(-lambda x) LHS definition (lambda_LHS = 1/beta_DAKOTA).
    const RealDenseVector& e_betas = iteratedModel.exponential_betas();
    for (i=0; i<numExponentialVars; i++, av_cntr++) {
      const Real& beta = e_betas[i];
      natafTransform.moments_from_exponential_params(beta, x_means[av_cntr],
						     x_std_devs[av_cntr]);
      // use default distribution bounds, not default global DACE bounds
      x_l_bnds[av_cntr] = 0.;
      x_u_bnds[av_cntr] = DBL_MAX;
      x_addtl[av_cntr].sizeUninitialized(1);
      x_addtl[av_cntr][0] = beta;
    }
  }
  if (numBetaVars) {
    // See Haldar and Mahadevan, p. 72
    const RealDenseVector& b_alphas = iteratedModel.beta_alphas();
    const RealDenseVector& b_betas  = iteratedModel.beta_betas();
    const RealDenseVector& b_l_bnds = iteratedModel.beta_lower_bounds();
    const RealDenseVector& b_u_bnds = iteratedModel.beta_upper_bounds();
    for (i=0; i<numBetaVars; i++, av_cntr++) {
      const Real& alpha = b_alphas[i];
      const Real& beta  = b_betas[i];
      const Real& lwr   = x_l_bnds[av_cntr] = b_l_bnds[i];
      const Real& upr   = x_u_bnds[av_cntr] = b_u_bnds[i];
      natafTransform.moments_from_beta_params(lwr, upr, alpha, beta,
        x_means[av_cntr], x_std_devs[av_cntr]);
      x_addtl[av_cntr].sizeUninitialized(2);
      x_addtl[av_cntr][0] = alpha;
      x_addtl[av_cntr][1] = beta;
    }
  }
  if (numGammaVars) {
    // This follows the Gamma(alpha,beta) definition in Law & Kelton, which
    // differs from the LHS definition (beta_LK = 1/beta_LHS).
    const RealDenseVector& ga_alphas = iteratedModel.gamma_alphas();
    const RealDenseVector& ga_betas  = iteratedModel.gamma_betas();
    for (i=0; i<numGammaVars; i++, av_cntr++) {
      const Real& alpha = ga_alphas[i]; const Real& beta = ga_betas[i];
      natafTransform.moments_from_gamma_params(alpha, beta, x_means[av_cntr],
					       x_std_devs[av_cntr]);
      // use default distribution bounds, not default global DACE bounds
      x_l_bnds[av_cntr] = 0.;
      x_u_bnds[av_cntr] = DBL_MAX;
      x_addtl[av_cntr].sizeUninitialized(2);
      x_addtl[av_cntr][0] = alpha;
      x_addtl[av_cntr][1] = beta;
    }
  }
  if (numGumbelVars) {
    // See Haldar and Mahadevan, p. 90
    const RealDenseVector& gu_alphas = iteratedModel.gumbel_alphas();
    const RealDenseVector& gu_betas  = iteratedModel.gumbel_betas();
    for (i=0; i<numGumbelVars; i++, av_cntr++) {
      const Real& alpha = gu_alphas[i]; const Real& beta = gu_betas[i];
      natafTransform.moments_from_gumbel_params(alpha, beta, x_means[av_cntr],
						x_std_devs[av_cntr]);
      // use default distribution bounds, not default global DACE bounds
      x_l_bnds[av_cntr] = -DBL_MAX;
      x_u_bnds[av_cntr] =  DBL_MAX;
      x_addtl[av_cntr].sizeUninitialized(2);
      x_addtl[av_cntr][0] = alpha;
      x_addtl[av_cntr][1] = beta;
    }
  }
  if (numFrechetVars) {
    const RealDenseVector& f_alphas = iteratedModel.frechet_alphas();
    const RealDenseVector& f_betas  = iteratedModel.frechet_betas();
    for (i=0; i<numFrechetVars; i++, av_cntr++) {
      const Real& alpha = f_alphas[i]; const Real& beta = f_betas[i];
      natafTransform.moments_from_frechet_params(alpha, beta, x_means[av_cntr],
						 x_std_devs[av_cntr]);
      // use default distribution bounds, not default global DACE bounds
      x_l_bnds[av_cntr] = 0.;
      x_u_bnds[av_cntr] = DBL_MAX;
      x_addtl[av_cntr].sizeUninitialized(2);
      x_addtl[av_cntr][0] = alpha;
      x_addtl[av_cntr][1] = beta;
    }
  }
  if (numWeibullVars) {
    const RealDenseVector& w_alphas = iteratedModel.weibull_alphas();
    const RealDenseVector& w_betas  = iteratedModel.weibull_betas();
    for (i=0; i<numWeibullVars; i++, av_cntr++) {
      const Real& alpha = w_alphas[i]; const Real& beta = w_betas[i];
      natafTransform.moments_from_weibull_params(alpha, beta, x_means[av_cntr],
						 x_std_devs[av_cntr]);
      // use default distribution bounds, not default global DACE bounds
      x_l_bnds[av_cntr] = 0.;
      x_u_bnds[av_cntr] = DBL_MAX;
      x_addtl[av_cntr].sizeUninitialized(2);
      x_addtl[av_cntr][0] = alpha;
      x_addtl[av_cntr][1] = beta;
    }
  }
  for (i=0; i<numStateVars; i++, av_cntr++) {
    const Real& lwr = x_l_bnds[av_cntr] = cv_l_bnds[av_cntr];
    const Real& upr = x_u_bnds[av_cntr] = cv_u_bnds[av_cntr];
    natafTransform.moments_from_uniform_params(lwr, upr, x_means[av_cntr],
					       x_std_devs[av_cntr]);
  }

  natafTransform.initialize_random_variable_parameters(x_means, x_std_devs,
    x_l_bnds, x_u_bnds, x_addtl);
}


/** This function is commonly used to publish tranformation data when
    the Model variables are in a transformed space (e.g., u-space) and
    ProbabilityTransformation::ranVarTypes et al. may not be generated
    directly.  This allows for the use of inverse transformations to
    return the transformed space variables to their original states. */
void NonD::
initialize_random_variables(const Pecos::ProbabilityTransformation& transform)
{
  natafTransform.initialize_random_variables(transform); // or could use copy()

  const Pecos::ShortArray& x_types = transform.x_types();
  numDesignVars = std::count(x_types.begin(), x_types.end(), (short)DESIGN);
  numStateVars  = std::count(x_types.begin(), x_types.end(), (short)STATE);
  // TO DO: need to pass extendedUspace
}

} // namespace Dakota
