/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        Iterator
//- Description:  Base class for iterator objects and envelope for 
//-               letter/envelope idiom.
//- Owner:        Mike Eldred
//- Version: $Id: DakotaIterator.H 5234 2008-09-04 19:19:28Z mseldre $

#ifndef DAKOTA_ITERATOR_H
#define DAKOTA_ITERATOR_H

#include "data_types.h"
#include "DakotaModel.H"
#ifdef HAVE_GSL
#include "gsl/gsl_randist.h"
#include "gsl/gsl_cdf.h"
#endif

namespace Dakota {

class ProblemDescDB;
class Variables;
class Response;


/// Base class for the iterator class hierarchy.

/** The Iterator class is the base class for one of the primary class
    hierarchies in DAKOTA.  The iterator hierarchy contains all of the
    iterative algorithms which use repeated execution of simulations
    as function evaluations.  For memory efficiency and enhanced
    polymorphism, the iterator hierarchy employs the "letter/envelope
    idiom" (see Coplien "Advanced C++", p. 133), for which the base
    class (Iterator) serves as the envelope and one of the derived
    classes (selected in Iterator::get_iterator()) serves as the letter. */

class Iterator
{
public:

  //
  //- Heading: Constructors, destructor, assignment operator
  //

  /// default constructor
  Iterator();
  /// standard envelope constructor
  Iterator(Model& model);
  /// alternate envelope constructor for instantiations by name
  Iterator(const String& method_name, Model& model);
  /// copy constructor
  Iterator(const Iterator& iterator);

  /// destructor
  virtual ~Iterator();

  /// assignment operator
  Iterator operator=(const Iterator& iterator);

  //
  //- Heading: Virtual functions
  //

  /// run the iterator; portion of run_iterator()
  virtual void run();

  /// return a single final iterator solution (variables)
  virtual const Variables& variables_results() const;
  /// return a single final iterator solution (response)
  virtual const Response&  response_results() const;

  /// indicates if this iterator accepts multiple initial points.  Default
  /// return is false.  Override to return true if appropriate.
  virtual bool accepts_multiple_points() const;
  /// indicates if this iterator returns multiple final points.  Default
  /// return is false.  Override to return true if appropriate.
  virtual bool returns_multiple_points() const;

  /// return multiple final iterator solutions (variables).  This should
  /// only be used if returns_multiple_points() returns true.
  virtual const VariablesArray& variables_array_results() const;
  /// return multiple final iterator solutions (response).  This should
  /// only be used if returns_multiple_points() returns true.
  virtual const ResponseArray&  response_array_results()  const;

  /// sets the multiple initial points for this iterator.  This should
  /// only be used if accepts_multiple_points() returns true.
  virtual void initial_points(const VariablesArray& pts);

  /// set the requested data for the final iterator response results
  virtual void response_results_active_set(const ActiveSet& set);

  /// initialize the 2D graphics window and the tabular graphics data
  virtual void initialize_graphics(bool graph_2d, bool tabular_data,
				   const String& tabular_file);

  /// print the final iterator results
  virtual void print_results(ostream& s);

  /// reset sampling iterator
  virtual void sampling_reset(int min_samples, int rec_samples, 
			      bool all_data_flag, bool stats_flag);

  /// return sampling name
  virtual const String& sampling_scheme() const;

  /// return name of any enabling iterator used by this iterator
  virtual String uses_method() const;

  /// perform a method switch, if possible, due to a detected conflict
  virtual void method_recourse();

  /// return the complete set of evaluated variables
  virtual const VariablesArray& all_variables() const;
  /// return the complete set of computed responses
  virtual const ResponseArray&  all_responses() const;

  //
  //- Heading: Member functions
  //

  /// utility function to verbosely perform common operations prior to run()
  void pre_run(ostream& s);
  /// utility function to quietly perform common operations prior to run()
  void pre_run();
  /// utility function to automate pre_run()/run()/post_run() verbosely
  void run_iterator(ostream& s);
  /// utility function to automate pre_run()/run()/post_run() quietly
  void run_iterator();
  /// utility function to verbosely perform common operations following run()
  void post_run(ostream& s);
  /// utility function to quietly perform common operations following run()
  void post_run();

  /// replaces existing letter with a new one
  void assign_rep(Iterator* iterator_rep, bool ref_count_incr = true);

  // set the model
  //void iterated_model(const Model& the_model);
  // return the model
  //Model& iterated_model() const;

  /// return the problem description database (probDescDB)
  ProblemDescDB& problem_description_db() const;

  /// return the method name
  const String& method_name() const;
  /// return the method identifier (idMethod)
  const String& method_id() const;

  /// return the method output level (outputLevel)
  short output_level() const;

  /// return the maximum concurrency supported by the iterator
  int maximum_concurrency() const;
  /// set the maximum concurrency supported by the iterator
  void maximum_concurrency(int max_conc);

  /// set the default active set vector (for use with iterators that
  /// employ evaluate_parameter_sets())
  void active_set(const ActiveSet& set);
  /// return the default active set vector (used by iterators that
  /// employ evaluate_parameter_sets())
  const ActiveSet& active_set() const;

  /// set subIteratorFlag
  void sub_iterator_flag(bool si_flag);
  /// set primaryACVMapIndices, primaryADVMapIndices, secondaryACVMapTargets
  void variable_mappings(const SizetArray& c_index1, const SizetArray& d_index1,
			 const ShortArray& c_target2,
			 const ShortArray& d_target2);

  /// function to check iteratorRep (does this envelope contain a letter?)
  bool is_null() const;

  /// returns iteratorRep for access to derived class member functions
  /// that are not mapped to the top Iterator level
  Iterator* iterator_rep() const;

  //
  //- Heading: Operator overloaded functions
  //

  // Possible future implementation for enhanced granularity in
  // Iterator virtual function.  Could be very useful for Strategy
  // level control!
  //virtual void operator++();  // increment operator
  //virtual void operator--();  // decrement operator

protected:

  //
  //- Heading: Constructors
  //

  /// constructor initializes the base class part of letter classes
  /// (BaseConstructor overloading avoids infinite recursion in the
  /// derived class constructors - Coplien, p. 139)
  Iterator(BaseConstructor, Model& model);

  /// alternate constructor for base iterator classes constructed on the fly
  Iterator(NoDBBaseConstructor, Model& model);

  /// alternate constructor for base iterator classes constructed on the fly
  Iterator(NoDBBaseConstructor);

  //
  //- Heading: Virtual functions
  //

  /// portions of pre_run specific to derived iterators
  virtual void derived_pre_run();
  /// portions of post_run specific to derived iterators
  virtual void derived_post_run();

  /// gets the multiple initial points for this iterator.  This will only
  /// be meaningful after a call to initial_points mutator.
  virtual const VariablesArray& initial_points() const;

  //
  //- Heading: Utility routines
  //

  /// Standard normal density function
  Real phi(const Real& beta);
  /// Standard normal cumulative distribution function
  Real Phi(const Real& beta);

  //
  //- Heading: Data
  //

  /// shallow copy of the model passed into the constructor
  /// or a thin RecastModel wrapped around it
  Model iteratedModel;

  /// class member reference to the problem description database
  ProblemDescDB& probDescDB;

  String methodName; ///< name of the iterator (the user's method spec)

  Real convergenceTol;  ///< iteration convergence tolerance
  int maxIterations;    ///< maximum number of iterations for the iterator
  int maxFunctionEvals; ///< maximum number of fn evaluations for the iterator
  int maxConcurrency;   ///< maximum coarse-grained concurrency

  // Isolate complexity by letting Model::currentVariables/currentResponse
  // manage details.  Then Iterator only needs the following:
  size_t numFunctions;      ///< number of response functions
  size_t numContinuousVars; ///< number of active continuous vars.
  size_t numDiscreteVars;   ///< number of active discrete vars.

  /// tracks the response data requirements on each function evaluation
  ActiveSet activeSet;

  /// flag indicating if this Iterator is a sub-iterator
  /// (NestedModel::subIterator or DataFitSurrModel::daceIterator)
  bool subIteratorFlag;
  /// "primary" all continuous variable mapping indices flowed down
  /// from higher level iteration
  SizetArray primaryACVarMapIndices;
  /// "primary" all discrete variable mapping indices flowed down from
  /// higher level iteration
  SizetArray primaryADVarMapIndices;
  /// "secondary" all continuous variable mapping targets flowed down
  /// from higher level iteration
  ShortArray secondaryACVarMapTargets;
  /// "secondary" all discrete variable mapping targets flowed down
  /// from higher level iteration
  ShortArray secondaryADVarMapTargets;

  /// type of gradient data: analytic, numerical, mixed, or none
  String gradientType;
  /// source of numerical gradient routine: dakota or vendor
  String methodSource;
  /// type of numerical gradient interval: central or forward
  String intervalType;
  /// type of Hessian data: analytic, numerical, quasi, mixed, or none
  String hessianType;

  /// relative finite difference step size for numerical gradients
  /** A scalar value (instead of the vector fd_gradient_step_size spec) is
      used within the iterator hierarchy since this attribute is only used
      to publish a step size to vendor numerical gradient algorithms. */
  Real fdGradStepSize;
  /// relative finite difference step size for numerical Hessians estimated 
  /// using first-order differences of gradients
  /** A scalar value (instead of the vector fd_hessian_step_size spec) is
      used within the iterator hierarchy since this attribute is only used
      to publish a step size to vendor numerical Hessian algorithms. */
  Real fdHessByGradStepSize;
  /// relative finite difference step size for numerical Hessians estimated 
  /// using second-order differences of function values
  /** A scalar value (instead of the vector fd_hessian_step_size spec) is
      used within the iterator hierarchy since this attribute is only used
      to publish a step size to vendor numerical Hessian algorithms. */
  Real fdHessByFnStepSize;

  /// output verbosity level: {SILENT,QUIET,NORMAL,VERBOSE,DEBUG}_OUTPUT
  short outputLevel;

  /// copy of the model's asynchronous evaluation flag
  bool asynchFlag;

private:

  //
  //- Heading: Member functions
  //

  /// Used by the envelope to instantiate the correct letter class
  Iterator* get_iterator(Model& model);
  /// Used by the envelope to instantiate the correct letter class
  Iterator* get_iterator(const String& method_name, Model& model);

  //
  //- Heading: Data
  //

  /// method identifier string from the input file
  String idMethod;
  /// pointer to the letter (initialized only for the envelope)
  Iterator* iteratorRep;
  /// number of objects sharing iteratorRep
  int referenceCount;
};


// nonvirtual functions can access letter attributes directly (only need to fwd
// member function call when the function could be redefined).
//inline Model& Iterator::iterated_model() const
//{ return (iteratorRep) ? iteratorRep->iteratedModel : iteratedModel; }


//inline void Iterator::iterated_model(const Model& the_model)
//{
//  if (iteratorRep)
//    iteratorRep->iteratedModel = the_model;
//  else
//    iteratedModel = the_model;
//}


inline ProblemDescDB& Iterator::problem_description_db() const
{ return (iteratorRep) ? iteratorRep->probDescDB : probDescDB; }


inline const String& Iterator::method_name() const
{ return (iteratorRep) ? iteratorRep->methodName : methodName; }


inline const String& Iterator::method_id() const
{ return (iteratorRep) ? iteratorRep->idMethod : idMethod; }


inline short Iterator::output_level() const
{ return (iteratorRep) ? iteratorRep->outputLevel : outputLevel; }


inline int Iterator::maximum_concurrency() const
{ return (iteratorRep) ? iteratorRep->maxConcurrency : maxConcurrency; }


inline void Iterator::maximum_concurrency(int max_conc)
{
  if (iteratorRep)
    iteratorRep->maxConcurrency = max_conc;
  else
    maxConcurrency = max_conc;
}


inline void Iterator::active_set(const ActiveSet& set)
{
  if (iteratorRep)
    iteratorRep->activeSet = set;
  else
    activeSet = set;
}


inline const ActiveSet& Iterator::active_set() const
{ return (iteratorRep) ? iteratorRep->activeSet : activeSet; }


inline void Iterator::sub_iterator_flag(bool si_flag)
{
  if (iteratorRep)
    iteratorRep->subIteratorFlag = si_flag;
  else
    subIteratorFlag = si_flag;
}


inline void Iterator::
variable_mappings(const SizetArray& c_index1,  const SizetArray& d_index1,
		  const ShortArray& c_target2, const ShortArray& d_target2)
{
  if (iteratorRep) {
    iteratorRep->primaryACVarMapIndices   = c_index1;
    iteratorRep->primaryADVarMapIndices   = d_index1;
    iteratorRep->secondaryACVarMapTargets = c_target2;
    iteratorRep->secondaryADVarMapTargets = d_target2;
  }
  else {
    primaryACVarMapIndices   = c_index1;
    primaryADVarMapIndices   = d_index1;
    secondaryACVarMapTargets = c_target2;
    secondaryADVarMapTargets = d_target2;
  }
}


inline bool Iterator::is_null() const
{ return (iteratorRep) ? false : true; }


inline Iterator* Iterator::iterator_rep() const
{ return iteratorRep; }


inline Real Iterator::phi(const Real& beta)
{
#ifdef HAVE_GSL
  return gsl_ran_ugaussian_pdf(beta);
#else
  const Real Pi = 3.1415926535897932385;
  return exp(-beta*beta/2.)/sqrt(2.*Pi);
#endif // HAVE_GSL
}


/** returns a probability < 0.5 for negative beta and a probability > 0.5
    for positive beta. */
inline Real Iterator::Phi(const Real& beta)
{
#ifdef HAVE_GSL
  return gsl_cdf_ugaussian_P(beta);
#else
  return (1. + erf(beta/sqrt(2.)))/2.;
#endif // HAVE_GSL
}


/// global comparison function for Iterator
inline bool method_id_compare(const Iterator& iterator, const void* id)
{ return ( *(const String*)id == iterator.method_id() ); }

} // namespace Dakota

#endif
