/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        AllVariables
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "AllVariables.H"
#include "ProblemDescDB.H"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

/** In this class, the all data approach (design, uncertain, and state
    types are combined) is used.  Iterators/strategies which use this
    class include: parameter studies, DACE, and the all_variables mode
    of nond_sampling.  Extract fundamental variable types and labels
    and combine them into allContinuousVars, allDiscreteVars,
    allContinuousLabels, and allDiscreteLabels. */
AllVariables::
AllVariables(const ProblemDescDB& problem_db, const pair<short,short>& view):
  Variables(BaseConstructor(), problem_db, view)
{
  // Get fundamental arrays from problem_db (most are stored only as locals to
  // avoid replicated data in data_pairs list)
  const RealVector& continuous_des_vars
    = problem_db.get_drv("variables.continuous_design.initial_point");
  const IntVector& discrete_des_vars
    = problem_db.get_div("variables.discrete_design.initial_point");
  const RealVector& uncertain_vars
    = problem_db.get_drv("variables.uncertain.initial_point");
  const RealVector& continuous_state_vars
    = problem_db.get_drv("variables.continuous_state.initial_state");
  const IntVector& discrete_state_vars
    = problem_db.get_div("variables.discrete_state.initial_state");
  const StringArray& cdv_labels
    = problem_db.get_dsa("variables.continuous_design.labels");
  const StringArray& ddv_labels
    = problem_db.get_dsa("variables.discrete_design.labels");
  const StringArray& uv_labels
    = problem_db.get_dsa("variables.uncertain.labels");
  const StringArray& csv_labels
    = problem_db.get_dsa("variables.continuous_state.labels");
  const StringArray& dsv_labels
    = problem_db.get_dsa("variables.discrete_state.labels");

  // Initialize All arrays
  aggregate_data(continuous_des_vars, uncertain_vars, continuous_state_vars,
		 allContinuousVars);
  aggregate_data(discrete_des_vars, discrete_state_vars, allDiscreteVars);
  aggregate_data(cdv_labels, uv_labels, csv_labels, allContinuousLabels);
  aggregate_data(ddv_labels, dsv_labels, allDiscreteLabels);

  // construct continuousVarTypes/discreteVarTypes and
  // continuousVarIds/inactiveContinuousVarIds/allContinuousVarIds
  build_types_ids();

#ifdef REFCOUNT_DEBUG
  Cout << "Letter instantiated: variablesView active = " << variablesView.first
       << " inactive = " << variablesView.second << endl;
#endif
}


void AllVariables::reshape_rep(const Sizet2DArray& vars_comps)
{
  size_t num_acv = vars_comps[0][1] + vars_comps[1][0] + vars_comps[2][1],
         num_adv = vars_comps[0][2] + vars_comps[2][2];

  allContinuousVars.reshape(num_acv);
  allDiscreteVars.reshape(num_adv);
  allContinuousLabels.reshape(num_acv);
  allDiscreteLabels.reshape(num_adv);

  build_types_ids();
}


void AllVariables::build_types_ids()
{
  size_t i, offset, num_uv = variablesComponents[1][0],
    num_cdv  = variablesComponents[0][1],  num_ddv = variablesComponents[0][2],
    num_csv  = variablesComponents[2][1],  num_dsv = variablesComponents[2][2],
    num_acv  = num_cdv + num_uv + num_csv, num_adv = num_ddv + num_dsv,
    acv_cntr = 0, adv_cntr = 0;

  allContinuousVarIds.reshape(num_acv);
  allContinuousVarTypes.reshape(num_acv);
  allDiscreteVarTypes.reshape(num_adv);
  for (i=0; i<num_cdv; i++) {
    allContinuousVarTypes[acv_cntr] = "continuous_design";
    allContinuousVarIds[acv_cntr++] = i + 1;
  }
  for (i=0; i<num_ddv; i++)
    allDiscreteVarTypes[adv_cntr++] = "discrete_design";
  uncertain_var_types(acv_cntr);
  offset = num_cdv + num_ddv + 1;
  for (i=0; i<num_uv; i++)
    allContinuousVarIds[i + num_cdv] = i + offset;
  offset += num_uv;
  for (i=0; i<num_csv; i++) {
    allContinuousVarTypes[acv_cntr] = "continuous_state";
    allContinuousVarIds[acv_cntr++] = i + offset;
  }
  for (i=0; i<num_dsv; i++)
    allDiscreteVarTypes[adv_cntr++] = "discrete_state";
}


// Reordering is required in all read/write cases that will be visible to the
// user since all derived vars classes should use the same CDV/DDV/UV/CSV/DSV
// ordering for clarity.  Neutral file I/O, binary streams, and packed buffers
// do not need to reorder (so long as read/write are consistent) since this data
// is not intended for public consumption.
void AllVariables::read(istream& s)
{
  size_t num_cdv    = variablesComponents[0][1],
         num_ddv    = variablesComponents[0][2],
         num_uv_csv = allContinuousVars.length() - num_cdv,
         num_dsv    = allDiscreteVars.length()   - num_ddv;
  // start & number to read passed in addition to stream and label array
  allContinuousVars.read_partial(s, 0, num_cdv, allContinuousLabels);
  allDiscreteVars.read_partial(s, 0, num_ddv, allDiscreteLabels);
  allContinuousVars.read_partial(s, num_cdv, num_uv_csv, allContinuousLabels);
  allDiscreteVars.read_partial(s, num_ddv, num_dsv, allDiscreteLabels);
}


void AllVariables::write(ostream& s) const
{
  size_t num_cdv    = variablesComponents[0][1],
         num_ddv    = variablesComponents[0][2],
         num_uv_csv = allContinuousVars.length() - num_cdv,
         num_dsv    = allDiscreteVars.length()   - num_ddv;
  // start & number to write passed in addition to stream and label array
  allContinuousVars.write_partial(s, 0, num_cdv, allContinuousLabels);
  allDiscreteVars.write_partial(s, 0, num_ddv, allDiscreteLabels);
  allContinuousVars.write_partial(s, num_cdv, num_uv_csv, allContinuousLabels);
  allDiscreteVars.write_partial(s, num_ddv, num_dsv, allDiscreteLabels);
}


void AllVariables::write_aprepro(ostream& s) const
{
  size_t num_cdv    = variablesComponents[0][1],
         num_ddv    = variablesComponents[0][2],
         num_uv_csv = allContinuousVars.length() - num_cdv,
         num_dsv    = allDiscreteVars.length()   - num_ddv;
  // start & number to write passed in addition to stream and label array
  allContinuousVars.write_partial_aprepro(s, 0, num_cdv, allContinuousLabels);
  allDiscreteVars.write_partial_aprepro(s, 0, num_ddv, allDiscreteLabels);
  allContinuousVars.write_partial_aprepro(s, num_cdv, num_uv_csv,
                                          allContinuousLabels);
  allDiscreteVars.write_partial_aprepro(s, num_ddv, num_dsv, allDiscreteLabels);
}


// AllVariables is the only class that currently requires I/O of
// variablesComponents.  If additional derived types develop a need
// for this data, its I/O should be elevated to Variables::read/write.
void AllVariables::read_annotated(istream& s)
{
  if (variablesComponents.empty()) {
    variablesComponents.reshape(1);//(3);
    variablesComponents[0].reshape(3);
    //variablesComponents[1].reshape(13);
    //variablesComponents[2].reshape(3);
  }
  s >> variablesComponents[0][1] >> variablesComponents[0][2];
  allContinuousVars.read_annotated(s, allContinuousLabels);
  allDiscreteVars.read_annotated(s, allDiscreteLabels);
  // types/ids not required
}


void AllVariables::write_annotated(ostream& s) const
{
  s << variablesComponents[0][1] << ' ' << variablesComponents[0][2] << ' ';
  allContinuousVars.write_annotated(s, allContinuousLabels);
  allDiscreteVars.write_annotated(s, allDiscreteLabels);
  // types/ids not required
}


void AllVariables::write_tabular(ostream& s) const
{
  size_t num_cdv    = variablesComponents[0][1],
         num_ddv    = variablesComponents[0][2],
         num_uv_csv = allContinuousVars.length() - num_cdv,
         num_dsv    = allDiscreteVars.length()   - num_ddv;
  // pass stream, start position, & number to write
  allContinuousVars.write_partial_tabular(s, 0, num_cdv);
  allDiscreteVars.write_partial_tabular(s, 0, num_ddv);
  allContinuousVars.write_partial_tabular(s, num_cdv, num_uv_csv);
  allDiscreteVars.write_partial_tabular(s, num_ddv, num_dsv);
}


void AllVariables::read(BiStream& s)
{
  if (variablesComponents.empty()) {
    variablesComponents.reshape(1);//(3);
    variablesComponents[0].reshape(3);
    //variablesComponents[1].reshape(13);
    //variablesComponents[2].reshape(3);
  }
  s >> variablesComponents[0][1] >> variablesComponents[0][2];
  allContinuousVars.read(s, allContinuousLabels);
  allDiscreteVars.read(s, allDiscreteLabels);
  // types/ids not required
}


void AllVariables::write(BoStream& s) const
{
  s << variablesComponents[0][1] << variablesComponents[0][2];
  allContinuousVars.write(s, allContinuousLabels);
  allDiscreteVars.write(s, allDiscreteLabels);
  // types/ids not required
}


void AllVariables::read(MPIUnpackBuffer& s)
{
  if (variablesComponents.empty()) {
    variablesComponents.reshape(1);//(3);
    variablesComponents[0].reshape(3);
    //variablesComponents[1].reshape(13);
    //variablesComponents[2].reshape(3);
  }
  s >> variablesComponents[0][1] >> variablesComponents[0][2];
  allContinuousVars.read(s, allContinuousLabels);
  allDiscreteVars.read(s, allDiscreteLabels);
  // types/ids not required
}


void AllVariables::write(MPIPackBuffer& s) const
{
  s << variablesComponents[0][1] << variablesComponents[0][2];
  allContinuousVars.write(s, allContinuousLabels);
  allDiscreteVars.write(s, allDiscreteLabels);
  // types/ids not required
}


void AllVariables::copy_rep(const Variables* vars_rep)
{
  // copy derived class attributes
  allContinuousVars   = ((AllVariables*)vars_rep)->allContinuousVars;
  allDiscreteVars     = ((AllVariables*)vars_rep)->allDiscreteVars;

  allContinuousLabels = ((AllVariables*)vars_rep)->allContinuousLabels;
  allDiscreteLabels   = ((AllVariables*)vars_rep)->allDiscreteLabels;

  allDiscreteVarTypes = ((AllVariables*)vars_rep)->allDiscreteVarTypes;
}


/// equality operator for AllVariables
bool operator==(const AllVariables& vars1, const AllVariables& vars2)
{
  return (vars2.allContinuousVars == vars1.allContinuousVars && 
	  vars2.allDiscreteVars   == vars1.allDiscreteVars);
  // sizes/labels/types/ids not required
}

} // namespace Dakota
