# include <cstdlib>
# include <iostream>
# include <iomanip>
# include <fstream>
# include <ctime>
# include <cmath>
# include <cstring> // for strlen

using namespace std;

# include "sparse_grid_mixed.H"

namespace webbur {

//****************************************************************************80

double cc_point ( int order, int index )

//****************************************************************************80
//
//  Purpose:
//
//    CC_POINT returns an abscissa of a Clenshaw Curtis rule.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    This rule is defined on [-1,1].
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 August 20080
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, int INDEX, the index of the abscissa.  1 <= INDEX <= ORDER.
//
//    Output, double CC_POINT, the abscissa.
//
{
  double pi = 3.141592653589793;
  double value;

  if ( order < 1 )
  {
    value = - r8_huge ( );
  }
  else if ( index < 1 || order < index )
  {
    value = - r8_huge ( );
  }
  else if ( order == 1 )
  {
    value = 0.0;
  }
  else if ( 2 *  index == order + 1 )
  {
    value = 0.0;
  }
  else
  {
    value = cos ( ( double ) ( order - index ) * pi 
                / ( double ) ( order - 1 ) );
  }
  return value;
}
//****************************************************************************80

void cc_weights ( int n, double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    CC_WEIGHTS computes Clenshaw Curtis weights.
//
//  Discussion:
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Charles Clenshaw, Alan Curtis,
//    A Method for Numerical Integration on an Automatic Computer,
//    Numerische Mathematik,
//    Volume 2, Number 1, December 1960, pages 197-205.
//
//  Parameters:
//
//    Input, int N, the order of the rule.
//
//    Output, double W[N], the weights of the rule.
//
{
  double b;
  int i;
  int j;
  double pi = 3.141592653589793;
  double theta;

  if ( n == 1 )
  {
    w[0] = 2.0;
    return;
  }

  for ( i = 1; i <= n; i++ )
  {
    theta = ( double ) ( i - 1 ) * pi / ( double ) ( n - 1 );

    w[i-1] = 1.0;

    for ( j = 1; j <= ( n - 1 ) / 2; j++ )
    {
      if ( 2 * j == ( n - 1 ) )
      {
        b = 1.0;
      }
      else
      {
        b = 2.0;
      }

      w[i-1] = w[i-1] - b * cos ( 2.0 * ( double ) ( j ) * theta ) 
           / ( double ) ( 4 * j * j - 1 );
    }
  }

  w[0] = w[0] / ( double ) ( n - 1 );
  for ( i = 1; i < n-1; i++ )
  {
    w[i] = 2.0 * w[i] / ( double ) ( n - 1 );
  }
  w[n-1] = w[n-1] / ( double ) ( n - 1 );

  return;
}
//****************************************************************************80

void comp_next ( int n, int k, int a[], bool *more, int *h, int *t )

//****************************************************************************80
//
//  Purpose:
//
//    COMP_NEXT computes the compositions of the integer N into K parts.
//
//  Discussion:
//
//    A composition of the integer N into K parts is an ordered sequence
//    of K nonnegative integers which sum to N.  The compositions (1,2,1)
//    and (1,1,2) are considered to be distinct.
//
//    The routine computes one composition on each call until there are no more.
//    For instance, one composition of 6 into 3 parts is
//    3+2+1, another would be 6+0+0.
//
//    On the first call to this routine, set MORE = FALSE.  The routine
//    will compute the first element in the sequence of compositions, and
//    return it, as well as setting MORE = TRUE.  If more compositions
//    are desired, call again, and again.  Each time, the routine will
//    return with a new composition.
//
//    However, when the LAST composition in the sequence is computed 
//    and returned, the routine will reset MORE to FALSE, signaling that
//    the end of the sequence has been reached.
//
//    This routine originally used a SAVE statement to maintain the
//    variables H and T.  I have decided that it is safer
//    to pass these variables as arguments, even though the user should
//    never alter them.  This allows this routine to safely shuffle
//    between several ongoing calculations.
//
//
//    There are 28 compositions of 6 into three parts.  This routine will
//    produce those compositions in the following order:
//
//     I         A
//     -     ---------
//     1     6   0   0
//     2     5   1   0
//     3     4   2   0
//     4     3   3   0
//     5     2   4   0
//     6     1   5   0
//     7     0   6   0
//     8     5   0   1
//     9     4   1   1
//    10     3   2   1
//    11     2   3   1
//    12     1   4   1
//    13     0   5   1
//    14     4   0   2
//    15     3   1   2
//    16     2   2   2
//    17     1   3   2
//    18     0   4   2
//    19     3   0   3
//    20     2   1   3
//    21     1   2   3
//    22     0   3   3
//    23     2   0   4
//    24     1   1   4
//    25     0   2   4
//    26     1   0   5
//    27     0   1   5
//    28     0   0   6
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    02 July 2008
//
//  Author:
//
//    FORTRAN77 original version by Albert Nijenhuis, Herbert Wilf.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Albert Nijenhuis, Herbert Wilf,
//    Combinatorial Algorithms for Computers and Calculators,
//    Second Edition,
//    Academic Press, 1978,
//    ISBN: 0-12-519260-6,
//    LC: QA164.N54.
//
//  Parameters:
//
//    Input, int N, the integer whose compositions are desired.
//
//    Input, int K, the number of parts in the composition.
//
//    Input/output, int A[K], the parts of the composition.
//
//    Input/output, bool *MORE.
//    Set MORE = FALSE on first call.  It will be reset to TRUE on return
//    with a new composition.  Each new call returns another composition until
//    MORE is set to FALSE when the last composition has been computed
//    and returned.
//
//    Input/output, int *H, *T, two internal parameters needed for the
//    computation.  The user should allocate space for these in the calling
//    program, include them in the calling sequence, but never alter them!
//
{
  int i;

  if ( !( *more ) )
  {
    *t = n;
    *h = 0;
    a[0] = n;
    for ( i = 1; i < k; i++ )
    {
       a[i] = 0;
    }
  }
  else
  {
    if ( 1 < *t )
    {
      *h = 0;
    }
    *h = *h + 1;
    *t = a[*h-1];
    a[*h-1] = 0;
    a[0] = *t - 1;
    a[*h] = a[*h] + 1;
  }

  *more = ( a[k-1] != n );

  return;
}
//****************************************************************************80

void dtable_close_write ( ofstream &output )

//****************************************************************************80
//
//  Purpose:
//
//    DTABLE_CLOSE_WRITE closes a file to which a DTABLE was to be written.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    27 September 2006
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, ofstream output, the output file stream.
//
{
  output.close ( );

  return;
}
//****************************************************************************80

void dtable_data_write ( ofstream &output, int m, int n, double table[] )

//****************************************************************************80
//
//  Purpose:
//
//    DTABLE_DATA_WRITE writes data to a DTABLE file.
//
//  Discussion:
//
//    The file should already be open.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    11 December 2003
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, ofstream &OUTPUT, a pointer to the output stream.
//
//    Input, int M, the spatial dimension.
//
//    Input, int N, the number of points.
//
//    Input, double TABLE[M*N], the table data.
//
{
  int i;
  int j;
  char *s;

  for ( j = 0; j < n; j++ )
  {
    for ( i = 0; i < m; i++ )
    {
      output << setprecision(16) << setw(24) << table[i+j*m] << "  ";
    }
    output << "\n";
  }

  return;
}
//****************************************************************************80
 
void dtable_header_write ( char *output_filename, ofstream &output, int m, 
  int n )
 
//****************************************************************************80
//
//  Purpose:
//
//    DTABLE_HEADER_WRITE writes the header of a DTABLE file.
//
//  Discussion:
//
//    The file should already be open.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    22 March 2006
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, char *OUTPUT_FILENAME, the output filename.
//
//    Input, ofstream &OUTPUT, the output stream.
//
//    Input, int M, the spatial dimension.
//
//    Input, int N, the number of points.
//
{
  char *s;

  s = timestring ( );

  output << "#  " << output_filename << "\n";
  output << "#  created by dtable_write in table_io.C" << "\n";
  output << "#  at " << s << "\n";
  output << "#\n";
  output << "#  Spatial dimension M = " << m << "\n";
  output << "#  Number of points N =  " << n << "\n";
  output << "#  EPSILON (unit roundoff) = " << r8_epsilon ( ) << "\n";
  output << "#\n";

  delete [] s;

  return;
}
//****************************************************************************80

void dtable_write ( char *output_filename, int m, int n, double table[], 
  bool header )

//****************************************************************************80
//
//  Purpose:
//
//    DTABLE_WRITE writes information to a DTABLE file.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    20 July 2005
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, char *OUTPUT_FILENAME, the output filename.
//
//    Input, int M, the spatial dimension.
//
//    Input, int N, the number of points.
//
//    Input, double TABLE[M*N], the table data.
//
//    Input, bool HEADER, is TRUE if the header is to be included.
//
{
  ofstream output;

  output.open ( output_filename );

  if ( !output )
  {
    cerr << "\n";
    cerr << "DTABLE_WRITE - Fatal error!\n";
    cerr << "  Could not open the output file.\n";
    return;
  }

  if ( header )
  {
    dtable_header_write ( output_filename, output, m, n );
  }

  dtable_data_write ( output, m, n, table );

  dtable_close_write ( output );

  return;
}
//****************************************************************************80

double f2_point ( int order, int index )

//****************************************************************************80
//
//  Purpose:
//
//    F2_POINT returns an abscissa of a Fejer type 2 rule.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    The rule is defined on [-1,1].
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 August 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the Fejer type 2 rule.
//    1 <= ORDER.
//
//    Input, int INDEX, the index of the abscissa.  1 <= INDEX <= ORDER.
//
//    Output, double F2_POINT, the abscissa.
//
{
  double pi = 3.141592653589793;
  double value;

  if ( order < 1 )
  {
    value = - r8_huge ( );
  }
  else if ( index < 1 || order < index )
  {
    value = - r8_huge ( );
  }
  else if ( order == 1 )
  {
    value = 0.0;
  }
  else if ( 2 * index == order + 1 )
  {
    value = 0.0;
  }
  else
  {
    value = cos ( ( double ) ( order + 1 - index ) * pi 
                / ( double ) ( order + 1 ) );
  }
  return value;
}
//****************************************************************************80

void f2_weights ( int order, double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    F2_WEIGHTS computes weights for a Fejer type 2 rule.
//
//  Discussion:
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Philip Davis, Philip Rabinowitz,
//    Methods of Numerical Integration,
//    Second Edition,
//    Dover, 2007,
//    ISBN: 0486453391,
//    LC: QA299.3.D28.
//
//    Walter Gautschi,
//    Numerical Quadrature in the Presence of a Singularity,
//    SIAM Journal on Numerical Analysis,
//    Volume 4, Number 3, 1967, pages 357-362.
//
//    Joerg Waldvogel,
//    Fast Construction of the Fejer and Clenshaw-Curtis Quadrature Rules,
//    BIT Numerical Mathematics,
//    Volume 43, Number 1, 2003, pages 1-18.
//
//  Parameters:
//
//    Input, int ORDER, the order.
//
//    Output, double W[ORDER], the weights.
//
{
  int i;
  int j;
  double p;
  double pi = 3.141592653589793;
  double theta;

  if ( order < 1 )
  {
    cerr << "\n";
    cerr << "F2_WEIGHTS - Fatal error!\n";
    cerr << "  ORDER < 1.\n";
    exit ( 1 );
  }

  if ( order == 1 )
  {
    w[0] = 2.0;
  }
  else if ( order == 2 )
  {
    w[0] = 1.0;
    w[1] = 1.0;
  }
  else
  {
    for ( i = 1; i <= order; i++ )
    {
      theta = ( double ) ( order + 1 - i ) * pi 
            / ( double ) ( order + 1 );

      w[i-1] = 1.0;

      for ( j = 1; j <= ( ( order - 1 ) / 2 ); j++ )
      {
        w[i-1] = w[i-1] - 2.0 * cos ( 2.0 * ( double ) ( j ) * theta ) 
          / ( double ) ( 4 * j * j - 1 );
      }
      p = 2.0 * ( double ) ( ( ( order + 1 ) / 2 ) ) - 1.0;
      w[i-1] = w[i-1] - cos ( ( p + 1.0 ) * theta ) / p;
    }
    for ( i = 0; i < order; i++ )
    {
      w[i] = 2.0 * w[i] / ( double ) ( order + 1 );
    }
  }
  return;
}
//****************************************************************************80

double gh_point ( int order, int index )

//****************************************************************************80
//
//  Purpose:
//
//    GH_POINT returns an abscissa of a Gauss Hermite rule.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    The rule is defined on (-oo,+oo).
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 August 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    Legal values are 1, 3, 7, 15, 31, 63 and 127.
//
//    Input, int INDEX, the index of the abscissa.  1 <= INDEX <= ORDER.
//
//    Output, double GH_POINT, the abscissa.
//
{
  double value;
  static double x_001[1] =
  {
    0.0E+00
  };
  static double x_003[3] =
  {
   -0.122474487139158904909864203735E+01, 
    0.0E+00, 
    0.122474487139158904909864203735E+01
  };
  static double x_007[7] =
  {
   -0.265196135683523349244708200652E+01, 
   -0.167355162876747144503180139830E+01, 
   -0.816287882858964663038710959027E+00, 
    0.0E+00, 
    0.816287882858964663038710959027E+00, 
    0.167355162876747144503180139830E+01, 
    0.265196135683523349244708200652E+01
  };
  static double x_015[15] =
  {
   -0.449999070730939155366438053053E+01, 
   -0.366995037340445253472922383312E+01, 
   -0.296716692790560324848896036355E+01, 
   -0.232573248617385774545404479449E+01, 
   -0.171999257518648893241583152515E+01, 
   -0.113611558521092066631913490556E+01, 
   -0.565069583255575748526020337198E+00, 
    0.0E+00, 
    0.565069583255575748526020337198E+00, 
    0.113611558521092066631913490556E+01, 
    0.171999257518648893241583152515E+01, 
    0.232573248617385774545404479449E+01, 
    0.296716692790560324848896036355E+01, 
    0.366995037340445253472922383312E+01, 
    0.449999070730939155366438053053E+01
  };
  static double x_031[31] =
  {
   -6.9956801237185402753248521473232E+00, 
   -6.2750787049428601427036567812530E+00, 
   -5.6739614446185883296332558789276E+00, 
   -5.1335955771123807045862968913996E+00, 
   -4.6315595063128599420667997654336E+00, 
   -4.1562717558181451724831352315314E+00, 
   -3.7007434032314694224497164589673E+00, 
   -3.2603207323135408104645401509648E+00, 
   -2.8316804533902054557015640151425E+00, 
   -2.4123177054804201051740184582119E+00, 
   -2.0002585489356389657975562598571E+00, 
   -1.5938858604721398261388419455550E+00, 
   -1.1918269983500464260821358649242E+00, 
   -0.79287697691530893968593032998830E+00, 
   -0.39594273647142311094670041663436E+00, 
    0.0000000000000000000000000000000E+00, 
    0.39594273647142311094670041663436E+00, 
    0.79287697691530893968593032998830E+00, 
    1.1918269983500464260821358649242E+00, 
    1.5938858604721398261388419455550E+00, 
    2.0002585489356389657975562598571E+00, 
    2.4123177054804201051740184582119E+00, 
    2.8316804533902054557015640151425E+00, 
    3.2603207323135408104645401509648E+00, 
    3.7007434032314694224497164589673E+00, 
    4.1562717558181451724831352315314E+00, 
    4.6315595063128599420667997654336E+00, 
    5.1335955771123807045862968913996E+00, 
    5.6739614446185883296332558789276E+00, 
    6.2750787049428601427036567812530E+00, 
    6.9956801237185402753248521473232E+00
  };
  static double x_063[63] =
  {
   -10.435499877854168053468115427285E+00, 
   -9.8028759912974963635223935286507E+00, 
   -9.2792019543050391319404745506496E+00, 
   -8.8118581437284546442526628275570E+00, 
   -8.3807683451863219343010651043788E+00, 
   -7.9755950801420373181541806298501E+00, 
   -7.5901395198641066762479783194468E+00, 
   -7.2203167078889678461161324222529E+00, 
   -6.8632544331795368527353285876066E+00, 
   -6.5168348106821160605273395854042E+00, 
   -6.1794379922705969862418461787263E+00, 
   -5.8497884000810673462526582961482E+00, 
   -5.5268572526403031425047575122840E+00, 
   -5.2097979830408354861575136416263E+00, 
   -4.8979018644975742350745099214868E+00, 
   -4.5905665744435190229271294569091E+00, 
   -4.2872733352824404031727616199454E+00, 
   -3.9875699104197157485227052068068E+00, 
   -3.6910577000963465117322810559754E+00, 
   -3.3973817713303911852755941806287E+00, 
   -3.1062230279282566329138616746036E+00, 
   -2.8172919672837977750747135657355E+00, 
   -2.5303236304712010926855221718499E+00, 
   -2.2450734604812066298995918179330E+00, 
   -1.9613138583081485293922008411321E+00, 
   -1.6788312791720137520802800622638E+00, 
   -1.3974237486049625107570752063702E+00, 
   -1.1168987050996462690510970277840E+00, 
   -0.83707109558947615977737795461293E+00, 
   -0.55776166427908221668763665253822E+00, 
   -0.27879538567115223986687628627202E+00, 
    0.00000000000000000000000000000000E+00, 
    0.27879538567115223986687628627202E+00, 
    0.55776166427908221668763665253822E+00, 
    0.83707109558947615977737795461293E+00, 
    1.1168987050996462690510970277840E+00, 
    1.3974237486049625107570752063702E+00, 
    1.6788312791720137520802800622638E+00, 
    1.9613138583081485293922008411321E+00, 
    2.2450734604812066298995918179330E+00, 
    2.5303236304712010926855221718499E+00, 
    2.8172919672837977750747135657355E+00, 
    3.1062230279282566329138616746036E+00, 
    3.3973817713303911852755941806287E+00, 
    3.6910577000963465117322810559754E+00, 
    3.9875699104197157485227052068068E+00, 
    4.2872733352824404031727616199454E+00, 
    4.5905665744435190229271294569091E+00, 
    4.8979018644975742350745099214868E+00, 
    5.2097979830408354861575136416263E+00, 
    5.5268572526403031425047575122840E+00, 
    5.8497884000810673462526582961482E+00, 
    6.1794379922705969862418461787263E+00, 
    6.5168348106821160605273395854042E+00, 
    6.8632544331795368527353285876066E+00, 
    7.2203167078889678461161324222529E+00, 
    7.5901395198641066762479783194468E+00, 
    7.9755950801420373181541806298501E+00, 
    8.3807683451863219343010651043788E+00, 
    8.8118581437284546442526628275570E+00, 
    9.2792019543050391319404745506496E+00, 
    9.8028759912974963635223935286507E+00, 
    10.435499877854168053468115427285E+00
  };
  static double x_127[127] =
  {
   -15.228338148167350978246954433464E+00, 
   -14.669595158833972632746354112896E+00, 
   -14.209085995284870755168244250887E+00, 
   -13.799722290211676634645246746673E+00, 
   -13.423518590070950062438258321855E+00, 
   -13.071208660474601901583995439649E+00, 
   -12.737235652415686338138003924072E+00, 
   -12.417939378869715805445879624069E+00, 
   -12.110749020947747600132123508132E+00, 
   -11.813772198267727195134584136191E+00, 
   -11.525565112572696599167888588564E+00, 
   -11.244994583785543445194384194300E+00, 
   -10.971150569840247423423040263881E+00, 
   -10.703288201027481347670940744690E+00, 
   -10.440787957772772867742591798027E+00, 
   -10.183127473450343888624126450357E+00, 
   -9.9298610495114250736847004273684E+00, 
   -9.6806044412474728038150712732737E+00, 
   -9.4350233389881650135019598506287E+00, 
   -9.1928244988460305715774195052527E+00, 
   -8.9537488108565404323807890169970E+00, 
   -8.7175658087076307363833999548548E+00, 
   -8.4840692689832473326097180339984E+00, 
   -8.2530736454457156579694124243888E+00, 
   -8.0244111514703375578594739796798E+00, 
   -7.7979293513870105420829120455591E+00, 
   -7.5734891556083454022834960763301E+00, 
   -7.3509631392269052701961258043733E+00, 
   -7.1302341220350710668064025713431E+00, 
   -6.9111939615465713197465633109366E+00, 
   -6.6937425208758294190074417381666E+00, 
   -6.4777867811645365448144903821487E+00, 
   -6.2632400742737354345609723857092E+00, 
   -6.0500214161419845694465474482388E+00, 
   -5.8380549248774187386601690807757E+00, 
   -5.6272693105464816659423455794909E+00, 
   -5.4175974259243240722848425872924E+00, 
   -5.2089758693153983587570258372239E+00, 
   -5.0013446320386360038520809107373E+00, 
   -4.7946467843764925009748509930857E+00, 
   -4.5888281947698372951606485031212E+00, 
   -4.3838372778464736294253744407459E+00, 
   -4.1796247675352031349421189892408E+00, 
   -3.9761435120673355916035814195920E+00, 
   -3.7733482881250526721004678400057E+00, 
   -3.5711956317782180447199756485249E+00, 
   -3.3696436841717397896643629240035E+00, 
   -3.1686520501953630191857798261495E+00, 
   -2.9681816685955910267761649521505E+00, 
   -2.7681946921824058801226545958892E+00, 
   -2.5686543769473501723144013022363E+00, 
   -2.3695249790490401080012474645702E+00, 
   -2.1707716587411506879498498083695E+00, 
   -1.9723603904195020079324743227565E+00, 
   -1.7742578780516791584676442103681E+00, 
   -1.5764314753267801315519597621879E+00, 
   -1.3788491099261778091441557053728E+00, 
   -1.1814792113700685848678583598423E+00, 
   -0.98429064194027277726568984213773E+00, 
   -0.78725263021825034151596831878971E+00, 
   -0.59033470680942102142230439346102E+00, 
   -0.39350664185130136568037826200185E+00, 
   -0.19673838392423251964272239737078E+00, 
    0.0000000000000000000000000000000E+00, 
    0.19673838392423251964272239737078E+00, 
    0.39350664185130136568037826200185E+00, 
    0.59033470680942102142230439346102E+00, 
    0.78725263021825034151596831878971E+00, 
    0.98429064194027277726568984213773E+00, 
    1.1814792113700685848678583598423E+00, 
    1.3788491099261778091441557053728E+00, 
    1.5764314753267801315519597621879E+00, 
    1.7742578780516791584676442103681E+00, 
    1.9723603904195020079324743227565E+00, 
    2.1707716587411506879498498083695E+00, 
    2.3695249790490401080012474645702E+00, 
    2.5686543769473501723144013022363E+00, 
    2.7681946921824058801226545958892E+00, 
    2.9681816685955910267761649521505E+00, 
    3.1686520501953630191857798261495E+00, 
    3.3696436841717397896643629240035E+00, 
    3.5711956317782180447199756485249E+00, 
    3.7733482881250526721004678400057E+00, 
    3.9761435120673355916035814195920E+00, 
    4.1796247675352031349421189892408E+00, 
    4.3838372778464736294253744407459E+00, 
    4.5888281947698372951606485031212E+00, 
    4.7946467843764925009748509930857E+00, 
    5.0013446320386360038520809107373E+00, 
    5.2089758693153983587570258372239E+00, 
    5.4175974259243240722848425872924E+00, 
    5.6272693105464816659423455794909E+00, 
    5.8380549248774187386601690807757E+00, 
    6.0500214161419845694465474482388E+00, 
    6.2632400742737354345609723857092E+00, 
    6.4777867811645365448144903821487E+00, 
    6.6937425208758294190074417381666E+00, 
    6.9111939615465713197465633109366E+00, 
    7.1302341220350710668064025713431E+00, 
    7.3509631392269052701961258043733E+00, 
    7.5734891556083454022834960763301E+00, 
    7.7979293513870105420829120455591E+00, 
    8.0244111514703375578594739796798E+00, 
    8.2530736454457156579694124243888E+00, 
    8.4840692689832473326097180339984E+00, 
    8.7175658087076307363833999548548E+00, 
    8.9537488108565404323807890169970E+00, 
    9.1928244988460305715774195052527E+00, 
    9.4350233389881650135019598506287E+00, 
    9.6806044412474728038150712732737E+00, 
    9.9298610495114250736847004273684E+00, 
    10.183127473450343888624126450357E+00, 
    10.440787957772772867742591798027E+00, 
    10.703288201027481347670940744690E+00, 
    10.971150569840247423423040263881E+00, 
    11.244994583785543445194384194300E+00, 
    11.525565112572696599167888588564E+00, 
    11.813772198267727195134584136191E+00, 
    12.110749020947747600132123508132E+00, 
    12.417939378869715805445879624069E+00, 
    12.737235652415686338138003924072E+00, 
    13.071208660474601901583995439649E+00, 
    13.423518590070950062438258321855E+00, 
    13.799722290211676634645246746673E+00, 
    14.209085995284870755168244250887E+00, 
    14.669595158833972632746354112896E+00, 
    15.228338148167350978246954433464E+00
  };
 
  if ( order < 1 )
  {
    value = - r8_huge ( );
  }
  else if ( index < 1 || order < index )
  {
    value = - r8_huge ( );
  }
  else if ( order == 1 )
  {
    value = x_001[index-1];
  }
  else if ( order == 3 )
  {
    value = x_003[index-1];
  }
  else if ( order == 7 )
  {
    value = x_007[index-1];
  }
  else if ( order == 15 )
  {
    value = x_015[index-1];
  }
  else if ( order == 31 )
  {
    value = x_031[index-1];
  }
  else if ( order == 63 )
  {
    value = x_063[index-1];
  }
  else if ( order == 127 )
  {
    value = x_127[index-1];
  }
  else
  {
    value = - r8_huge ( );
  }
  return value;
}
//****************************************************************************80

void gh_weights ( int order, double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    GH_WEIGHTS returns weights for certain Gauss-Hermite quadrature rules.
//
//  Discussion:
//
//    The allowed orders are 1, 3, 7, 15, 31, 63 and 127.
//
//    The weights are positive, symmetric and should sum to SQRT(PI).
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Milton Abramowitz, Irene Stegun,
//    Handbook of Mathematical Functions,
//    National Bureau of Standards, 1964,
//    ISBN: 0-486-61272-4,
//    LC: QA47.A34.
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    ORDER must be 1, 3, 7, 15, 31, 63 or 127.
//
//    Output, double W[ORDER], the weights.
//
{
  static double w_001[1] =
  {
    1.77245385090551602729816748334E+00
  };
  static double w_003[31] =
  {
    0.295408975150919337883027913890E+00,
    0.118163590060367735153211165556E+01,
    0.295408975150919337883027913890E+00
  };
  static double w_007[7] = 
  {
    0.971781245099519154149424255939E-03,
    0.545155828191270305921785688417E-01,
    0.425607252610127800520317466666E+00,
    0.810264617556807326764876563813E+00,
    0.425607252610127800520317466666E+00,
    0.545155828191270305921785688417E-01,
    0.971781245099519154149424255939E-03
  };
  static double w_015[15] =
  {
    0.152247580425351702016062666965E-08,
    0.105911554771106663577520791055E-05,
    0.100004441232499868127296736177E-03,
    0.277806884291277589607887049229E-02,
    0.307800338725460822286814158758E-01,
    0.158488915795935746883839384960E+00,
    0.412028687498898627025891079568E+00,
    0.564100308726417532852625797340E+00,
    0.412028687498898627025891079568E+00,
    0.158488915795935746883839384960E+00,
    0.307800338725460822286814158758E-01,
    0.277806884291277589607887049229E-02,
    0.100004441232499868127296736177E-03,
    0.105911554771106663577520791055E-05,
    0.152247580425351702016062666965E-08
  };
  static double w_031[31] = 
  {
    0.46189683944498305857470556847735E-21,
    0.51106090079112519643027197715274E-17,
    0.58995564987355133075257722133966E-14,
    0.18603735214463569590294465062239E-11,
    0.23524920032013205739850619940094E-09,
    0.14611988344865057576066495091513E-07,
    0.50437125589241034841778074689627E-06,
    0.10498602757642934202945441341697E-04,
    0.13952090395003623854995664958146E-03,
    0.12336833073030489880608311394968E-02,
    0.74827999140119116765002499116934E-02,
    0.31847230731201222775249585776902E-01,
    0.96717948160569462991143316029341E-01,
    0.21213278866810461318136114862419E+00,
    0.33877265789305344906000174083214E+00,
    0.39577855609737786462923720809676E+00,
    0.33877265789305344906000174083214E+00,
    0.21213278866810461318136114862419E+00,
    0.96717948160569462991143316029341E-01,
    0.31847230731201222775249585776902E-01,
    0.74827999140119116765002499116934E-02,
    0.12336833073030489880608311394968E-02,
    0.13952090395003623854995664958146E-03,
    0.10498602757642934202945441341697E-04,
    0.50437125589241034841778074689627E-06,
    0.14611988344865057576066495091513E-07,
    0.23524920032013205739850619940094E-09,
    0.18603735214463569590294465062239E-11,
    0.58995564987355133075257722133966E-14,
    0.51106090079112519643027197715274E-17,
    0.46189683944498305857470556847735E-21
  };
  static double w_063[63] = 
  {
    0.37099206434787551197827130470031E-47,
    0.10400778615192299534481914814892E-41,
    0.19796804708258311251124226474396E-37,
    0.84687478191640015120141181138947E-34,
    0.13071305930779945903630127634063E-30,
    0.93437837175367456929765381518998E-28,
    0.36027426635173044862245783257252E-25,
    0.82963863115951789374753323156164E-23,
    0.12266629909105281472971700203949E-20,
    0.12288435628797061539461585325494E-18,
    0.86925536958188009075932426691516E-17,
    0.44857058689176221240330804981619E-15,
    0.17335817955735154599902643794700E-13,
    0.51265062385038307838565047455223E-12,
    0.11808921844532942490513037158404E-10,
    0.21508698297808025739828859845140E-09,
    0.31371929535285447801497640621672E-08,
    0.37041625984781705796752840204084E-07,
    0.35734732949879669663960738150956E-06,
    0.28393114498380927832990899215541E-05,
    0.18709113003730498008961134765721E-04,
    0.10284880800653635546698378640623E-03,
    0.47411702610173128107201781718693E-03,
    0.18409222622384813438539657470055E-02,
    0.60436044551187631655712178246467E-02,
    0.16829299199599730926458559757600E-01,
    0.39858264027692992170237391875317E-01,
    0.80467087993950415219587554532823E-01,
    0.13871950817615293377792092082674E+00,
    0.20448695346833761570957197160475E+00,
    0.25799889943058042204920467417642E+00,
    0.27876694884838411919175686949858E+00,
    0.25799889943058042204920467417642E+00,
    0.20448695346833761570957197160475E+00,
    0.13871950817615293377792092082674E+00,
    0.80467087993950415219587554532823E-01,
    0.39858264027692992170237391875317E-01,
    0.16829299199599730926458559757600E-01,
    0.60436044551187631655712178246467E-02,
    0.18409222622384813438539657470055E-02,
    0.47411702610173128107201781718693E-03,
    0.10284880800653635546698378640623E-03,
    0.18709113003730498008961134765721E-04,
    0.28393114498380927832990899215541E-05,
    0.35734732949879669663960738150956E-06,
    0.37041625984781705796752840204084E-07,
    0.31371929535285447801497640621672E-08,
    0.21508698297808025739828859845140E-09,
    0.11808921844532942490513037158404E-10,
    0.51265062385038307838565047455223E-12,
    0.17335817955735154599902643794700E-13,
    0.44857058689176221240330804981619E-15,
    0.86925536958188009075932426691516E-17,
    0.12288435628797061539461585325494E-18,
    0.12266629909105281472971700203949E-20,
    0.82963863115951789374753323156164E-23,
    0.36027426635173044862245783257252E-25,
    0.93437837175367456929765381518998E-28,
    0.13071305930779945903630127634063E-30,
    0.84687478191640015120141181138947E-34,
    0.19796804708258311251124226474396E-37,
    0.10400778615192299534481914814892E-41,
    0.37099206434787551197827130470031E-47
  };
  static double w_127[127] = 
  {
    0.12504497577050595552677230002883E-10,
    0.17272798059419131415318615789672E-93,
    0.89321681571986548608031150791499E-88,
    0.77306185240893578449625186483810E-83,
    0.20143957652648255497735460506196E-78,
    0.21503714733610239701351039429345E-74,
    0.11341924208594594813715533569504E-70,
    0.33489139011795051950683388483136E-67,
    0.60486548964016681064424451668405E-64,
    0.71375092946352177824971347343892E-61,
    0.57884563374885556636801095624030E-58,
    0.33581166223858230300409326551248E-55,
    0.14394641949253923568603163698953E-52,
    0.46821808383216117724080263903889E-50,
    0.11817054440684264071348471955361E-47,
    0.23581659156008927203181682045005E-45,
    0.37814427940797540210712758405540E-43,
    0.49411031115771638145610738414006E-41,
    0.53255303775425059266087298458297E-39,
    0.47854390680131484999315199332765E-37,
    0.36191883445952356128627543209554E-35,
    0.23232083386343554805352497446119E-33,
    0.12753331411008716683688974281454E-31,
    0.60277753850758742112436095241270E-30,
    0.24679773241777200207460855084439E-28,
    0.88019567691698482573264198727415E-27,
    0.27482489212040561315005725890593E-25,
    0.75468218903085486125222816438456E-24,
    0.18303134636280466270545996891835E-22,
    0.39355990860860813085582448449811E-21,
    0.75293161638581191068419292570042E-20,
    0.12857997786722855037584105682618E-18,
    0.19659326888445857792541925311450E-17,
    0.26986511907214101894995783364250E-16,
    0.33344414303198856330118301113874E-15,
    0.37173303125150639885726463109574E-14,
    0.37473954472839737091885387788983E-13,
    0.34230094493397259538669512076007E-12,
    0.28385303724993373166810860630552E-11,
    0.21406920290454669208938772802828E-10,
    0.14706331273431716244229273183839E-09,
    0.92173940967434659264335883218167E-09,
    0.52781663936972714041837056042506E-08,
    0.27650497044951117835905283127679E-07,
    0.13267855842539464770913063113371E-06,
    0.58380944276113062188573331195042E-06,
    0.23581561724775629112332165335800E-05,
    0.87524468034280444703919485644809E-05,
    0.29876790535909012274846532159647E-04,
    0.93874435720072545206729594267039E-04,
    0.27170762627931172053444716883938E-03,
    0.72493929742498358979684249380921E-03,
    0.17841208326763432884316727108264E-02,
    0.40524855186046131499765636276283E-02,
    0.85000263041544110385806705526917E-02,
    0.16471142241609687824005585301760E-01,
    0.29499296248213632269675010319119E-01,
    0.48847387114300011006959603975676E-01,
    0.74807989768583731416517226905270E-01,
    0.10598520508090929403834368934301E+00,
    0.13893945309051540832066283010510E+00,
    0.16856236074207929740526975049765E+00,
    0.18927849580120432177170145550076E+00,
    0.19673340688823289786163676995151E+00,
    0.18927849580120432177170145550076E+00,
    0.16856236074207929740526975049765E+00,
    0.13893945309051540832066283010510E+00,
    0.10598520508090929403834368934301E+00,
    0.74807989768583731416517226905270E-01,
    0.48847387114300011006959603975676E-01,
    0.29499296248213632269675010319119E-01,
    0.16471142241609687824005585301760E-01,
    0.85000263041544110385806705526917E-02,
    0.40524855186046131499765636276283E-02,
    0.17841208326763432884316727108264E-02,
    0.72493929742498358979684249380921E-03,
    0.27170762627931172053444716883938E-03,
    0.93874435720072545206729594267039E-04,
    0.29876790535909012274846532159647E-04,
    0.87524468034280444703919485644809E-05,
    0.23581561724775629112332165335800E-05,
    0.58380944276113062188573331195042E-06,
    0.13267855842539464770913063113371E-06,
    0.27650497044951117835905283127679E-07,
    0.52781663936972714041837056042506E-08,
    0.92173940967434659264335883218167E-09,
    0.14706331273431716244229273183839E-09,
    0.21406920290454669208938772802828E-10,
    0.28385303724993373166810860630552E-11,
    0.34230094493397259538669512076007E-12,
    0.37473954472839737091885387788983E-13,
    0.37173303125150639885726463109574E-14,
    0.33344414303198856330118301113874E-15,
    0.26986511907214101894995783364250E-16,
    0.19659326888445857792541925311450E-17,
    0.12857997786722855037584105682618E-18,
    0.75293161638581191068419292570042E-20,
    0.39355990860860813085582448449811E-21,
    0.18303134636280466270545996891835E-22,
    0.75468218903085486125222816438456E-24,
    0.27482489212040561315005725890593E-25,
    0.88019567691698482573264198727415E-27,
    0.24679773241777200207460855084439E-28,
    0.60277753850758742112436095241270E-30,
    0.12753331411008716683688974281454E-31,
    0.23232083386343554805352497446119E-33,
    0.36191883445952356128627543209554E-35,
    0.47854390680131484999315199332765E-37,
    0.53255303775425059266087298458297E-39,
    0.49411031115771638145610738414006E-41,
    0.37814427940797540210712758405540E-43,
    0.23581659156008927203181682045005E-45,
    0.11817054440684264071348471955361E-47,
    0.46821808383216117724080263903889E-50,
    0.14394641949253923568603163698953E-52,
    0.33581166223858230300409326551248E-55,
    0.57884563374885556636801095624030E-58,
    0.71375092946352177824971347343892E-61,
    0.60486548964016681064424451668405E-64,
    0.33489139011795051950683388483136E-67,
    0.11341924208594594813715533569504E-70,
    0.21503714733610239701351039429345E-74,
    0.20143957652648255497735460506196E-78,
    0.77306185240893578449625186483810E-83,
    0.89321681571986548608031150791499E-88,
    0.17272798059419131415318615789672E-93,
    0.12504497577050595552677230002883E-100
  };
  if ( order == 1 )
  {
    r8vec_copy ( order, w_001, w );
  }
  else if ( order == 3 )
  {
    r8vec_copy ( order, w_003, w );
  }
  else if ( order == 7 )
  {
    r8vec_copy ( order, w_007, w );
  }
  else if ( order == 15 )
  {
    r8vec_copy ( order, w_015, w );
  }
  else if ( order == 31 )
  {
    r8vec_copy ( order, w_031, w );
  }
  else if ( order == 63 )
  {
    r8vec_copy ( order, w_063, w );
  }
  else if ( order == 127 )
  {
    r8vec_copy ( order, w_127, w );
  }
  else
  {
    cerr << "\n";
    cerr << "GH_WEIGHTS - Fatal error!\n";
    cerr << "  Unexpected value of ORDER = " << order << ".\n";
    exit ( 1 );
  }
  return;
}
//****************************************************************************80

double gl_point ( int order, int index )

//****************************************************************************80
//
//  Purpose:
//
//    GL_POINT returns an abscissa of a Gauss Legendre rule.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    The rule is defined on [-1,1].
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 August 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    Legal values are 1, 3, 7, 15, 31, 63 and 127.
//
//    Input, int INDEX, the index of the abscissa.  1 <= INDEX <= ORDER.
//
//    Output, double GL_POINT, the abscissa.
//
{
  double value;
  static double x_001[1] =
  {
       0.0E+00
  };
  static double x_003[3] =
  {
     - 0.774596669241483377035853079956E+00, 
       0.0E+00, 
       0.774596669241483377035853079956E+00
  };
  static double x_007[7] =
  {
     - 0.949107912342758524526189684048E+00, 
     - 0.741531185599394439863864773281E+00, 
     - 0.405845151377397166906606412077E+00, 
       0.0E+00, 
       0.405845151377397166906606412077E+00, 
       0.741531185599394439863864773281E+00, 
       0.949107912342758524526189684048E+00
  };
  static double x_015[15] =
  {
      - 0.987992518020485428489565718587E+00, 
      - 0.937273392400705904307758947710E+00, 
      - 0.848206583410427216200648320774E+00, 
      - 0.724417731360170047416186054614E+00, 
      - 0.570972172608538847537226737254E+00, 
      - 0.394151347077563369897207370981E+00, 
      - 0.201194093997434522300628303395E+00, 
        0.0E+00, 
        0.201194093997434522300628303395E+00, 
       0.394151347077563369897207370981E+00, 
       0.570972172608538847537226737254E+00, 
       0.724417731360170047416186054614E+00, 
       0.848206583410427216200648320774E+00, 
       0.937273392400705904307758947710E+00, 
       0.987992518020485428489565718587E+00
  };
  static double x_031[31] =
  {
      -0.99708748181947707454263838179654,    
      -0.98468590966515248400211329970113,    
      -0.96250392509294966178905249675943,    
      -0.93075699789664816495694576311725,    
      -0.88976002994827104337419200908023,    
      -0.83992032014626734008690453594388,    
      -0.78173314841662494040636002019484,    
      -0.71577678458685328390597086536649,    
      -0.64270672292426034618441820323250,    
      -0.56324916140714926272094492359516,    
      -0.47819378204490248044059403935649,    
      -0.38838590160823294306135146128752,    
      -0.29471806998170161661790389767170,    
      -0.19812119933557062877241299603283,    
      -0.99555312152341520325174790118941E-01,
       0.00000000000000000000000000000000,
       0.99555312152341520325174790118941E-01,
       0.19812119933557062877241299603283,    
       0.29471806998170161661790389767170,    
       0.38838590160823294306135146128752,    
       0.47819378204490248044059403935649,    
       0.56324916140714926272094492359516,
       0.64270672292426034618441820323250,    
       0.71577678458685328390597086536649,    
       0.78173314841662494040636002019484,    
       0.83992032014626734008690453594388,    
       0.88976002994827104337419200908023,    
       0.93075699789664816495694576311725,    
       0.96250392509294966178905249675943,    
       0.98468590966515248400211329970113,   
       0.99708748181947707454263838179654
  };
  static double x_063[63] = 
  {
      -0.99928298402912378050701628988630E+00,     
      -0.99622401277797010860209018267357E+00,     
      -0.99072854689218946681089469460884E+00,     
      -0.98280881059372723486251140727639E+00,     
      -0.97248403469757002280196067864927E+00,     
      -0.95977944975894192707035416626398E+00,     
      -0.94472613404100980296637531962798E+00,     
      -0.92736092062184320544703138132518E+00,     
      -0.90772630277853155803695313291596E+00,     
      -0.88587032850785342629029845731337E+00,     
      -0.86184648236412371953961183943106E+00,     
      -0.83571355431950284347180776961571E+00,     
      -0.80753549577345676005146598636324E+00,     
      -0.77738126299037233556333018991104E+00,     
      -0.74532464831784741782932166103759E+00,     
      -0.71144409958484580785143153770401E+00,     
      -0.67582252811498609013110331596954E+00,     
      -0.63854710582136538500030695387338E+00,     
      -0.59970905187762523573900892686880E+00,     
      -0.55940340948628501326769780007005E+00,     
      -0.51772881329003324812447758452632E+00,     
      -0.47478724799480439992221230985149E+00,     
      -0.43068379879511160066208893391863E+00,     
      -0.38552639421224789247761502227440E+00,     
      -0.33942554197458440246883443159432E+00,     
      -0.29249405858625144003615715555067E+00,     
      -0.24484679324595336274840459392483E+00,     
      -0.19660034679150668455762745706572E+00,     
      -0.14787278635787196856983909655297E+00,     
      -0.98783356446945279529703669453922E-01, 
      -0.49452187116159627234233818051808E-01, 
       0.00000000000000000000000000000000E+00,     
       0.49452187116159627234233818051808E-01, 
       0.98783356446945279529703669453922E-01, 
       0.14787278635787196856983909655297E+00,     
       0.19660034679150668455762745706572E+00,     
       0.24484679324595336274840459392483E+00,     
       0.29249405858625144003615715555067E+00,     
       0.33942554197458440246883443159432E+00,     
       0.38552639421224789247761502227440E+00,     
       0.43068379879511160066208893391863E+00,     
       0.47478724799480439992221230985149E+00,     
       0.51772881329003324812447758452632E+00,     
       0.55940340948628501326769780007005E+00,     
       0.59970905187762523573900892686880E+00,     
       0.63854710582136538500030695387338E+00,     
       0.67582252811498609013110331596954E+00,     
       0.71144409958484580785143153770401E+00,     
       0.74532464831784741782932166103759E+00,     
       0.77738126299037233556333018991104E+00,     
       0.80753549577345676005146598636324E+00,     
       0.83571355431950284347180776961571E+00,     
       0.86184648236412371953961183943106E+00,     
       0.88587032850785342629029845731337E+00,     
       0.90772630277853155803695313291596E+00,     
       0.92736092062184320544703138132518E+00,     
       0.94472613404100980296637531962798E+00,     
       0.95977944975894192707035416626398E+00,     
       0.97248403469757002280196067864927E+00,     
       0.98280881059372723486251140727639E+00,     
       0.99072854689218946681089469460884E+00,     
       0.99622401277797010860209018267357E+00,  
       0.99928298402912378050701628988630E+00
  };
  static double x_127[127] = 
  {
       -0.99982213041530614629963254927125E+00,     
       -0.99906293435531189513828920479421E+00,     
       -0.99769756618980462107441703193392E+00,     
       -0.99572655135202722663543337085008E+00,     
       -0.99315104925451714736113079489080E+00,     
       -0.98997261459148415760778669967548E+00,     
       -0.98619317401693166671043833175407E+00,     
       -0.98181502080381411003346312451200E+00,     
       -0.97684081234307032681744391886221E+00,     
       -0.97127356816152919228894689830512E+00,     
       -0.96511666794529212109082507703391E+00,     
       -0.95837384942523877114910286998060E+00,     
       -0.95104920607788031054790764659636E+00,     
       -0.94314718462481482734544963026201E+00,     
       -0.93467258232473796857363487794906E+00,     
       -0.92563054405623384912746466814259E+00,     
       -0.91602655919146580931308861741716E+00,     
       -0.90586645826182138280246131760282E+00,     
       -0.89515640941708370896904382642451E+00,     
       -0.88390291468002656994525794802849E+00,     
       -0.87211280599856071141963753428864E+00,     
       -0.85979324109774080981203134414483E+00,     
       -0.84695169913409759845333931085437E+00,     
       -0.83359597615489951437955716480123E+00,     
       -0.81973418036507867415511910167470E+00,     
       -0.80537472720468021466656079404644E+00,     
       -0.79052633423981379994544995252740E+00,     
       -0.77519801587020238244496276354566E+00,     
       -0.75939907785653667155666366659810E+00,     
       -0.74313911167095451292056688997595E+00,    
       -0.72642798867407268553569290153270E+00,     
       -0.70927585412210456099944463906757E+00,     
       -0.69169312100770067015644143286666E+00,     
       -0.67369046373825048534668253831602E+00,     
       -0.65527881165548263027676505156852E+00,     
       -0.63646934240029724134760815684175E+00,     
       -0.61727347512685828385763916340822E+00,     
       -0.59770286357006522938441201887478E+00,     
       -0.57776938897061258000325165713764E+00,     
       -0.55748515286193223292186190687872E+00,     
       -0.53686246972339756745816636353452E+00,     
       -0.51591385950424935727727729906662E+00,     
       -0.49465204002278211739494017368636E+00,     
       -0.47308991924540524164509989939699E+00,     
       -0.45124058745026622733189858020729E+00,     
       -0.42911730928019337626254405355418E+00,     
       -0.40673351568978256340867288124339E+00,     
       -0.38410279579151693577907781452239E+00,     
       -0.36123888860586970607092484346723E+00,     
       -0.33815567472039850137600027657095E+00,     
       -0.31486716786289498148601475374890E+00,     
       -0.29138750639370562079451875284568E+00,     
       -0.26773094472238862088834352027938E+00,     
       -0.24391184465391785797071324453138E+00,     
       -0.21994466666968754245452337866940E+00,     
       -0.19584396114861085150428162519610E+00,     
       -0.17162435953364216500834492248954E+00,     
       -0.14730056544908566938932929319807E+00,     
       -0.12288734577408297172603365288567E+00,     
       -0.98399521677698970751091751509101E-01, 
       -0.73851959621048545273440409360569E-01, 
       -0.49259562331926630315379321821927E-01, 
       -0.24637259757420944614897071846088E-01, 
        0.00000000000000000000000000000000E+00,     
        0.24637259757420944614897071846088E-01, 
        0.49259562331926630315379321821927E-01, 
        0.73851959621048545273440409360569E-01, 
        0.98399521677698970751091751509101E-01, 
        0.12288734577408297172603365288567E+00,     
        0.14730056544908566938932929319807E+00,     
        0.17162435953364216500834492248954E+00,     
        0.19584396114861085150428162519610E+00,     
        0.21994466666968754245452337866940E+00,     
        0.24391184465391785797071324453138E+00,     
        0.26773094472238862088834352027938E+00,     
        0.29138750639370562079451875284568E+00,     
        0.31486716786289498148601475374890E+00,     
        0.33815567472039850137600027657095E+00,     
        0.36123888860586970607092484346723E+00,     
        0.38410279579151693577907781452239E+00,     
        0.40673351568978256340867288124339E+00,     
        0.42911730928019337626254405355418E+00,     
        0.45124058745026622733189858020729E+00,     
        0.47308991924540524164509989939699E+00,     
        0.49465204002278211739494017368636E+00,     
        0.51591385950424935727727729906662E+00,     
        0.53686246972339756745816636353452E+00,     
        0.55748515286193223292186190687872E+00,     
        0.57776938897061258000325165713764E+00,     
        0.59770286357006522938441201887478E+00,     
        0.61727347512685828385763916340822E+00,     
        0.63646934240029724134760815684175E+00,     
        0.65527881165548263027676505156852E+00,     
        0.67369046373825048534668253831602E+00,     
        0.69169312100770067015644143286666E+00,    
        0.70927585412210456099944463906757E+00,     
        0.72642798867407268553569290153270E+00,     
        0.74313911167095451292056688997595E+00,     
        0.75939907785653667155666366659810E+00,     
        0.77519801587020238244496276354566E+00,     
        0.79052633423981379994544995252740E+00,     
        0.80537472720468021466656079404644E+00,     
        0.81973418036507867415511910167470E+00,     
        0.83359597615489951437955716480123E+00,     
        0.84695169913409759845333931085437E+00,     
        0.85979324109774080981203134414483E+00,     
        0.87211280599856071141963753428864E+00,     
        0.88390291468002656994525794802849E+00,     
        0.89515640941708370896904382642451E+00,     
        0.90586645826182138280246131760282E+00,     
        0.91602655919146580931308861741716E+00,     
        0.92563054405623384912746466814259E+00,     
        0.93467258232473796857363487794906E+00,     
        0.94314718462481482734544963026201E+00,     
        0.95104920607788031054790764659636E+00,     
        0.95837384942523877114910286998060E+00,     
        0.96511666794529212109082507703391E+00,     
        0.97127356816152919228894689830512E+00,     
        0.97684081234307032681744391886221E+00,     
        0.98181502080381411003346312451200E+00,     
        0.98619317401693166671043833175407E+00,     
        0.98997261459148415760778669967548E+00,     
        0.99315104925451714736113079489080E+00,     
        0.99572655135202722663543337085008E+00,     
        0.99769756618980462107441703193392E+00,     
        0.99906293435531189513828920479421E+00,     
        0.99982213041530614629963254927125E+00 
  };

  if ( order < 1 )
  {
    value = - r8_huge ( );
  }
  else if ( index < 1 || order < index )
  {
    value = - r8_huge ( );
  }
  else if ( order == 1 )
  {
    value = x_001[index-1];
  }
  else if ( order == 3 )
  {
    value = x_003[index-1];
  }
  else if ( order == 7 )
  {
    value = x_007[index-1];
  }
  else if ( order == 15 )
  {
    value = x_015[index-1];
  }
  else if ( order == 31 )
  {
    value = x_031[index-1];
  }
  else if ( order == 63 )
  {
    value = x_063[index-1];
  }
  else if ( order == 127 )
  {
    value = x_127[index-1];
  }
  else
  {
    value = - r8_huge ( );
  }
  return value;
}
//****************************************************************************80

void gl_weights ( int order, double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    GL_WEIGHTS returns weights for certain Gauss-Legendre quadrature rules.
//
//  Discussion:
//
//    The allowed orders are 1, 3, 7, 15, 31, 63 and 127.
//
//    The weights are positive, symmetric and should sum to 2.
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    01 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Milton Abramowitz, Irene Stegun,
//    Handbook of Mathematical Functions,
//    National Bureau of Standards, 1964,
//    ISBN: 0-486-61272-4,
//    LC: QA47.A34.
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    ORDER must be 1, 3, 7, 15, 31, 63 or 127.
//
//    Output, double W[ORDER], the weights.
//
{
  static double w_001[1] =
  {
    2.0E+00
  };
  static double w_003[3] = 
  {
    0.555555555555555555555555555555E+00,
    0.888888888888888888888888888888E+00,
    0.555555555555555555555555555555E+00
  };
  static double w_007[7] =
  {
    0.129484966168869693270611432679E+00,
    0.279705391489276667901467771424E+00,
    0.381830050505118944950369775489E+00,
    0.417959183673469387755102040816E+00,
    0.381830050505118944950369775489E+00,
    0.279705391489276667901467771424E+00,
    0.129484966168869693270611432679E+00
  };
  static double w_015[15] =
  {
    0.307532419961172683546283935772E-01,
    0.703660474881081247092674164507E-01,
    0.107159220467171935011869546686E+00,
    0.139570677926154314447804794511E+00,
    0.166269205816993933553200860481E+00,
    0.186161000015562211026800561866E+00,
    0.198431485327111576456118326444E+00,
    0.202578241925561272880620199968E+00,
    0.198431485327111576456118326444E+00,
    0.186161000015562211026800561866E+00,
    0.166269205816993933553200860481E+00,
    0.139570677926154314447804794511E+00,
    0.107159220467171935011869546686E+00,
    0.703660474881081247092674164507E-01,
    0.307532419961172683546283935772E-01
  };
  static double w_031[31] =
  {
    0.74708315792487746093913218970494E-02,
    0.17318620790310582463552990782414E-01,
    0.27009019184979421800608642617676E-01,
    0.36432273912385464024392008749009E-01,
    0.45493707527201102902315857856518E-01,
    0.54103082424916853711666259085477E-01,
    0.62174786561028426910343543686657E-01,
    0.69628583235410366167756126255124E-01,
    0.76390386598776616426357674901331E-01,
    0.82392991761589263903823367431962E-01,
    0.87576740608477876126198069695333E-01,
    0.91890113893641478215362871607150E-01,
    0.95290242912319512807204197487597E-01,
    0.97743335386328725093474010978997E-01,
    0.99225011226672307874875514428615E-01,
    0.99720544793426451427533833734349E-01,
    0.99225011226672307874875514428615E-01,
    0.97743335386328725093474010978997E-01,
    0.95290242912319512807204197487597E-01,
    0.91890113893641478215362871607150E-01,
    0.87576740608477876126198069695333E-01,
    0.82392991761589263903823367431962E-01,
    0.76390386598776616426357674901331E-01,
    0.69628583235410366167756126255124E-01,
    0.62174786561028426910343543686657E-01,
    0.54103082424916853711666259085477E-01,
    0.45493707527201102902315857856518E-01,
    0.36432273912385464024392008749009E-01,
    0.27009019184979421800608642617676E-01,
    0.17318620790310582463552990782414E-01,
    0.74708315792487746093913218970494E-02
  };
  static double w_063[63] =
  {
    0.18398745955770837880499331680577E-02,
    0.42785083468637618661951422543371E-02,
    0.67102917659601362519069109850892E-02,
    0.91259686763266563540586445877022E-02,
    0.11519376076880041750750606118707E-01,
    0.13884612616115610824866086365937E-01,
    0.16215878410338338882283672974995E-01,
    0.18507464460161270409260545805144E-01,
    0.20753761258039090775341953421471E-01,
    0.22949271004889933148942319561770E-01,
    0.25088620553344986618630138068443E-01,
    0.27166574359097933225189839439413E-01,
    0.29178047208280526945551502154029E-01,
    0.31118116622219817508215988557189E-01,
    0.32982034883779341765683179672459E-01,
    0.34765240645355877697180504642788E-01,
    0.36463370085457289630452409787542E-01,
    0.38072267584349556763638324927889E-01,
    0.39587995891544093984807928149202E-01,
    0.41006845759666398635110037009072E-01,
    0.42325345020815822982505485403028E-01,
    0.43540267083027590798964315704401E-01,
    0.44648638825941395370332669516813E-01,
    0.45647747876292608685885992608542E-01,
    0.46535149245383696510395418746953E-01,
    0.47308671312268919080604988338844E-01,
    0.47966421137995131411052756195132E-01,
    0.48506789097883847864090099145802E-01,
    0.48928452820511989944709361549215E-01,
    0.49230380423747560785043116988145E-01,
    0.49411833039918178967039646116705E-01,
    0.49472366623931020888669360420926E-01,
    0.49411833039918178967039646116705E-01,
    0.49230380423747560785043116988145E-01,
    0.48928452820511989944709361549215E-01,
    0.48506789097883847864090099145802E-01,
    0.47966421137995131411052756195132E-01,
    0.47308671312268919080604988338844E-01,
    0.46535149245383696510395418746953E-01,
    0.45647747876292608685885992608542E-01,
    0.44648638825941395370332669516813E-01,
    0.43540267083027590798964315704401E-01,
    0.42325345020815822982505485403028E-01,
    0.41006845759666398635110037009072E-01,
    0.39587995891544093984807928149202E-01,
    0.38072267584349556763638324927889E-01,
    0.36463370085457289630452409787542E-01,
    0.34765240645355877697180504642788E-01,
    0.32982034883779341765683179672459E-01,
    0.31118116622219817508215988557189E-01,
    0.29178047208280526945551502154029E-01,
    0.27166574359097933225189839439413E-01,
    0.25088620553344986618630138068443E-01,
    0.22949271004889933148942319561770E-01,
    0.20753761258039090775341953421471E-01,
    0.18507464460161270409260545805144E-01,
    0.16215878410338338882283672974995E-01,
    0.13884612616115610824866086365937E-01,
    0.11519376076880041750750606118707E-01,
    0.91259686763266563540586445877022E-02,
    0.67102917659601362519069109850892E-02,
    0.42785083468637618661951422543371E-02,
    0.18398745955770837880499331680577E-02
  };
  static double w_127[127] =
  {
    0.45645726109586654495731936146574E-03,
    0.10622766869538486959954760554099E-02,
    0.16683488125171936761028811985672E-02,
    0.22734860707492547802810838362671E-02,
    0.28772587656289004082883197417581E-02,
    0.34792893810051465908910894094105E-02,
    0.40792095178254605327114733456293E-02,
    0.46766539777779034772638165662478E-02,
    0.52712596565634400891303815906251E-02,
    0.58626653903523901033648343751367E-02,
    0.64505120486899171845442463868748E-02,
    0.70344427036681608755685893032552E-02,
    0.76141028256526859356393930849227E-02,
    0.81891404887415730817235884718726E-02,
    0.87592065795403145773316804234385E-02,
    0.93239550065309714787536985834029E-02,
    0.98830429087554914716648010899606E-02,
    0.10436130863141005225673171997668E-01,
    0.10982883090068975788799657376065E-01,
    0.11522967656921087154811609734510E-01,
    0.12056056679400848183529562144697E-01,
    0.12581826520465013101514365424172E-01,
    0.13099957986718627426172681912499E-01,
    0.13610136522139249906034237533759E-01,
    0.14112052399003395774044161633613E-01,
    0.14605400905893418351737288078952E-01,
    0.15089882532666922992635733981431E-01,
    0.15565203152273955098532590262975E-01,
    0.16031074199309941802254151842763E-01,
    0.16487212845194879399346060358146E-01,
    0.16933342169871654545878815295200E-01,
    0.17369191329918731922164721250350E-01,
    0.17794495722974774231027912900351E-01,
    0.18208997148375106468721469154479E-01,
    0.18612443963902310429440419898958E-01,
    0.19004591238555646611148901044533E-01,
    0.19385200901246454628112623489471E-01,
    0.19754041885329183081815217323169E-01,
    0.20110890268880247225644623956287E-01,
    0.20455529410639508279497065713301E-01,
    0.20787750081531811812652137291250E-01,
    0.21107350591688713643523847921658E-01,
    0.21414136912893259295449693233545E-01,
    0.21707922796373466052301324695331E-01,
    0.21988529885872983756478409758807E-01,
    0.22255787825930280235631416460158E-01,
    0.22509534365300608085694429903050E-01,
    0.22749615455457959852242553240982E-01,
    0.22975885344117206754377437838947E-01,
    0.23188206663719640249922582981729E-01,
    0.23386450514828194170722043496950E-01,
    0.23570496544381716050033676844306E-01,
    0.23740233018760777777714726703424E-01,
    0.23895556891620665983864481754172E-01,
    0.24036373866450369675132086026456E-01,
    0.24162598453819584716522917710986E-01,
    0.24274154023278979833195063936748E-01,
    0.24370972849882214952813561907241E-01,
    0.24452996155301467956140198471529E-01,
    0.24520174143511508275183033290175E-01,
    0.24572466031020653286354137335186E-01,
    0.24609840071630254092545634003360E-01,
    0.24632273575707679066033370218017E-01,
    0.24639752923961094419579417477503E-01,
    0.24632273575707679066033370218017E-01,
    0.24609840071630254092545634003360E-01,
    0.24572466031020653286354137335186E-01,
    0.24520174143511508275183033290175E-01,
    0.24452996155301467956140198471529E-01,
    0.24370972849882214952813561907241E-01,
    0.24274154023278979833195063936748E-01,
    0.24162598453819584716522917710986E-01,
    0.24036373866450369675132086026456E-01,
    0.23895556891620665983864481754172E-01,
    0.23740233018760777777714726703424E-01,
    0.23570496544381716050033676844306E-01,
    0.23386450514828194170722043496950E-01,
    0.23188206663719640249922582981729E-01,
    0.22975885344117206754377437838947E-01,
    0.22749615455457959852242553240982E-01,
    0.22509534365300608085694429903050E-01,
    0.22255787825930280235631416460158E-01,
    0.21988529885872983756478409758807E-01,
    0.21707922796373466052301324695331E-01,
    0.21414136912893259295449693233545E-01,
    0.21107350591688713643523847921658E-01,
    0.20787750081531811812652137291250E-01,
    0.20455529410639508279497065713301E-01,
    0.20110890268880247225644623956287E-01,
    0.19754041885329183081815217323169E-01,
    0.19385200901246454628112623489471E-01,
    0.19004591238555646611148901044533E-01,
    0.18612443963902310429440419898958E-01,
    0.18208997148375106468721469154479E-01,
    0.17794495722974774231027912900351E-01,
    0.17369191329918731922164721250350E-01,
    0.16933342169871654545878815295200E-01,
    0.16487212845194879399346060358146E-01,
    0.16031074199309941802254151842763E-01,
    0.15565203152273955098532590262975E-01,
    0.15089882532666922992635733981431E-01,
    0.14605400905893418351737288078952E-01,
    0.14112052399003395774044161633613E-01,
    0.13610136522139249906034237533759E-01,
    0.13099957986718627426172681912499E-01,
    0.12581826520465013101514365424172E-01,
    0.12056056679400848183529562144697E-01,
    0.11522967656921087154811609734510E-01,
    0.10982883090068975788799657376065E-01,
    0.10436130863141005225673171997668E-01,
    0.98830429087554914716648010899606E-02,
    0.93239550065309714787536985834029E-02,
    0.87592065795403145773316804234385E-02,
    0.81891404887415730817235884718726E-02,
    0.76141028256526859356393930849227E-02,
    0.70344427036681608755685893032552E-02,
    0.64505120486899171845442463868748E-02,
    0.58626653903523901033648343751367E-02,
    0.52712596565634400891303815906251E-02,
    0.46766539777779034772638165662478E-02,
    0.40792095178254605327114733456293E-02,
    0.34792893810051465908910894094105E-02,
    0.28772587656289004082883197417581E-02,
    0.22734860707492547802810838362671E-02,
    0.16683488125171936761028811985672E-02,
    0.10622766869538486959954760554099E-02,
    0.45645726109586654495731936146574E-03
  };
  if ( order == 1 )
  {
    r8vec_copy ( order, w_001, w );
  }
  else if ( order == 3 )
  {
    r8vec_copy ( order, w_003, w );
  }
  else if ( order == 7 )
  {
    r8vec_copy ( order, w_007, w );
  }
  else if ( order == 15 )
  {
    r8vec_copy ( order, w_015, w );
  }
  else if ( order == 31 )
  {
    r8vec_copy ( order, w_031, w );
  }
  else if ( order == 63 )
  {
    r8vec_copy ( order, w_063, w );
  }
  else if ( order == 127 )
  {
    r8vec_copy ( order, w_127, w );
  }
  else
  {
    cerr << "\n";
    cerr << "GL_WEIGHTS - Fatal error!\n";
    cerr << "  Unexpected value of ORDER = " << order << ".\n";
    exit ( 1 );
  }
  return;
}
//****************************************************************************80

double gp_point ( int order, int index )

//****************************************************************************80
//
//  Purpose:
//
//    GP_POINT returns an abscissa of a Gauss Patterson rule.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    The rule is defined on [-1,1],
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 August 2008
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Prem Kythe, Michael Schaeferkotter,
//    Handbook of Computational Methods for Integration,
//    Chapman and Hall, 2004,
//    ISBN: 1-58488-428-2,
//    LC: QA299.3.K98.
//
//    Thomas Patterson,
//    The Optimal Addition of Points to Quadrature Formulae,
//    Mathematics of Computation,
//    Volume 22, Number 104, October 1968, pages 847-856.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    Legal values are 1, 3, 7, 15, 31, 63 and 127.
//
//    Input, int INDEX, the index of the abscissa.  1 <= INDEX <= ORDER.
//
//    Output, double GP_POINT, the abscissa.
//
{
  double value;
  static double x_001[1] = 
  {
     0.0 
  };
  static double x_003[3] =
  {
    -0.77459666924148337704, 
     0.0, 
     0.77459666924148337704
  };
  static double x_007[7] =
  {
    -0.96049126870802028342, 
    -0.77459666924148337704, 
    -0.43424374934680255800, 
     0.0, 
     0.43424374934680255800, 
     0.77459666924148337704, 
     0.96049126870802028342 
  };
  static double x_015[15] =
  {
    -0.99383196321275502221, 
    -0.96049126870802028342, 
    -0.88845923287225699889, 
    -0.77459666924148337704, 
    -0.62110294673722640294, 
    -0.43424374934680255800, 
    -0.22338668642896688163, 
     0.0, 
     0.22338668642896688163, 
     0.43424374934680255800, 
     0.62110294673722640294, 
     0.77459666924148337704, 
     0.88845923287225699889, 
     0.96049126870802028342, 
     0.99383196321275502221
  };
  static double x_031[31] =
  {
    -0.99909812496766759766, 
    -0.99383196321275502221, 
    -0.98153114955374010687, 
    -0.96049126870802028342, 
    -0.92965485742974005667, 
    -0.88845923287225699889, 
    -0.83672593816886873550, 
    -0.77459666924148337704, 
    -0.70249620649152707861, 
    -0.62110294673722640294, 
    -0.53131974364437562397, 
    -0.43424374934680255800, 
    -0.33113539325797683309, 
    -0.22338668642896688163, 
    -0.11248894313318662575, 
     0.0, 
     0.11248894313318662575, 
     0.22338668642896688163, 
     0.33113539325797683309, 
     0.43424374934680255800, 
     0.53131974364437562397, 
     0.62110294673722640294, 
     0.70249620649152707861, 
     0.77459666924148337704, 
     0.83672593816886873550, 
     0.88845923287225699889, 
     0.92965485742974005667, 
     0.96049126870802028342, 
     0.98153114955374010687, 
     0.99383196321275502221, 
     0.99909812496766759766
  };
  static double x_063[63] =
  {
    -0.99987288812035761194, 
    -0.99909812496766759766, 
    -0.99720625937222195908, 
    -0.99383196321275502221, 
    -0.98868475754742947994, 
    -0.98153114955374010687, 
    -0.97218287474858179658, 
    -0.96049126870802028342, 
    -0.94634285837340290515, 
    -0.92965485742974005667, 
    -0.91037115695700429250, 
    -0.88845923287225699889, 
    -0.86390793819369047715, 
    -0.83672593816886873550, 
    -0.80694053195021761186, 
    -0.77459666924148337704, 
    -0.73975604435269475868, 
    -0.70249620649152707861, 
    -0.66290966002478059546, 
    -0.62110294673722640294, 
    -0.57719571005204581484, 
    -0.53131974364437562397, 
    -0.48361802694584102756, 
    -0.43424374934680255800, 
    -0.38335932419873034692, 
    -0.33113539325797683309, 
    -0.27774982202182431507, 
    -0.22338668642896688163, 
    -0.16823525155220746498, 
    -0.11248894313318662575, 
    -0.056344313046592789972, 
     0.0, 
     0.056344313046592789972, 
     0.11248894313318662575, 
     0.16823525155220746498, 
     0.22338668642896688163, 
     0.27774982202182431507, 
     0.33113539325797683309, 
     0.38335932419873034692, 
     0.43424374934680255800, 
     0.48361802694584102756, 
     0.53131974364437562397, 
     0.57719571005204581484, 
     0.62110294673722640294, 
     0.66290966002478059546, 
     0.70249620649152707861, 
     0.73975604435269475868, 
     0.77459666924148337704, 
     0.80694053195021761186, 
     0.83672593816886873550, 
     0.86390793819369047715, 
     0.88845923287225699889, 
     0.91037115695700429250, 
     0.92965485742974005667, 
     0.94634285837340290515, 
     0.96049126870802028342, 
     0.97218287474858179658, 
     0.98153114955374010687, 
     0.98868475754742947994, 
     0.99383196321275502221, 
     0.99720625937222195908, 
     0.99909812496766759766, 
     0.99987288812035761194
  };
  static double x_127[127] =
  {
    -0.99998243035489159858, 
    -0.99987288812035761194, 
    -0.99959879967191068325, 
    -0.99909812496766759766, 
    -0.99831663531840739253, 
    -0.99720625937222195908, 
    -0.99572410469840718851, 
    -0.99383196321275502221, 
    -0.99149572117810613240, 
    -0.98868475754742947994, 
    -0.98537149959852037111, 
    -0.98153114955374010687, 
    -0.97714151463970571416, 
    -0.97218287474858179658, 
    -0.96663785155841656709, 
    -0.96049126870802028342, 
    -0.95373000642576113641, 
    -0.94634285837340290515, 
    -0.93832039777959288365, 
    -0.92965485742974005667, 
    -0.92034002547001242073, 
    -0.91037115695700429250, 
    -0.89974489977694003664, 
    -0.88845923287225699889, 
    -0.87651341448470526974, 
    -0.86390793819369047715, 
    -0.85064449476835027976, 
    -0.83672593816886873550, 
    -0.82215625436498040737, 
    -0.80694053195021761186, 
    -0.79108493379984836143, 
    -0.77459666924148337704, 
    -0.75748396638051363793, 
    -0.73975604435269475868, 
    -0.72142308537009891548, 
    -0.70249620649152707861, 
    -0.68298743109107922809, 
    -0.66290966002478059546, 
    -0.64227664250975951377, 
    -0.62110294673722640294, 
    -0.59940393024224289297, 
    -0.57719571005204581484, 
    -0.55449513263193254887, 
    -0.53131974364437562397, 
    -0.50768775753371660215, 
    -0.48361802694584102756, 
    -0.45913001198983233287, 
    -0.43424374934680255800, 
    -0.40897982122988867241, 
    -0.38335932419873034692, 
    -0.35740383783153215238, 
    -0.33113539325797683309, 
    -0.30457644155671404334, 
    -0.27774982202182431507, 
    -0.25067873030348317661, 
    -0.22338668642896688163, 
    -0.19589750271110015392, 
    -0.16823525155220746498, 
    -0.14042423315256017459, 
    -0.11248894313318662575, 
    -0.084454040083710883710, 
    -0.056344313046592789972, 
    -0.028184648949745694339, 
     0.0, 
     0.028184648949745694339, 
     0.056344313046592789972, 
     0.084454040083710883710, 
     0.11248894313318662575, 
     0.14042423315256017459, 
     0.16823525155220746498, 
     0.19589750271110015392, 
     0.22338668642896688163, 
     0.25067873030348317661, 
     0.27774982202182431507, 
     0.30457644155671404334, 
     0.33113539325797683309, 
     0.35740383783153215238, 
     0.38335932419873034692, 
     0.40897982122988867241, 
     0.43424374934680255800, 
     0.45913001198983233287, 
     0.48361802694584102756, 
     0.50768775753371660215, 
     0.53131974364437562397, 
     0.55449513263193254887, 
     0.57719571005204581484, 
     0.59940393024224289297, 
     0.62110294673722640294, 
     0.64227664250975951377, 
     0.66290966002478059546, 
     0.68298743109107922809, 
     0.70249620649152707861, 
     0.72142308537009891548, 
     0.73975604435269475868, 
     0.75748396638051363793, 
     0.77459666924148337704, 
     0.79108493379984836143, 
     0.80694053195021761186, 
     0.82215625436498040737, 
     0.83672593816886873550, 
     0.85064449476835027976, 
     0.86390793819369047715, 
     0.87651341448470526974, 
     0.88845923287225699889, 
     0.89974489977694003664, 
     0.91037115695700429250, 
     0.92034002547001242073, 
     0.92965485742974005667, 
     0.93832039777959288365, 
     0.94634285837340290515, 
     0.95373000642576113641, 
     0.96049126870802028342, 
     0.96663785155841656709, 
     0.97218287474858179658, 
     0.97714151463970571416, 
     0.98153114955374010687, 
     0.98537149959852037111, 
     0.98868475754742947994, 
     0.99149572117810613240, 
     0.99383196321275502221, 
     0.99572410469840718851, 
     0.99720625937222195908, 
     0.99831663531840739253, 
     0.99909812496766759766, 
     0.99959879967191068325, 
     0.99987288812035761194, 
     0.99998243035489159858
  };

  if ( order < 1 )
  {
    value = - r8_huge ( );
  }
  else if ( index < 1 || order < index )
  {
    value = - r8_huge ( );
  }
  else if ( order == 1 )
  {
    value = x_001[index-1];
  }
  else if ( order == 3 )
  {
    value = x_003[index-1];
  }
  else if ( order == 7 )
  {
    value = x_007[index-1];
  }
  else if ( order == 15 )
  {
    value = x_015[index-1];
  }
  else if ( order == 31 )
  {
    value = x_031[index-1];
  }
  else if ( order == 63 )
  {
    value = x_063[index-1];
  }
  else if ( order == 127 )
  {
    value = x_127[index-1];
  }
  else
  {
    value = - r8_huge ( );
  }
  return value;
}
//****************************************************************************80

void gp_weights ( int order, double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    GP_WEIGHTS returns weights for certain Gauss-Patterson quadrature rules.
//
//  Discussion:
//
//    The allowed orders are 1, 3, 7, 15, 31, 63 and 127.
//
//    The weights are positive, symmetric and should sum to 2.
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Milton Abramowitz, Irene Stegun,
//    Handbook of Mathematical Functions,
//    National Bureau of Standards, 1964,
//    ISBN: 0-486-61272-4,
//    LC: QA47.A34.
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    ORDER must be 1, 3, 7, 15, 31, 63 or 127.
//
//    Output, double W[ORDER], the weights.
//
{
  static double w_001[1] =
  {
    2.0
  };
  static double w_003[3] = 
  {
    0.555555555555555555556,
    0.888888888888888888889,
    0.555555555555555555556
  };
  static double w_007[7] =
  {
    0.104656226026467265194,
    0.268488089868333440729,
    0.401397414775962222905,
    0.450916538658474142345,
    0.401397414775962222905,
    0.268488089868333440729,
    0.104656226026467265194
  };
  static double w_015[15] =
  {
    0.0170017196299402603390,
    0.0516032829970797396969,
    0.0929271953151245376859,
    0.134415255243784220360,
    0.171511909136391380787,
    0.200628529376989021034,
    0.219156858401587496404,
    0.225510499798206687386,
    0.219156858401587496404,
    0.200628529376989021034,
    0.171511909136391380787,
    0.134415255243784220360,
    0.0929271953151245376859,
    0.0516032829970797396969,
    0.0170017196299402603390
  };
  static double w_031[31] =
  {
    0.00254478079156187441540,
    0.00843456573932110624631,
    0.0164460498543878109338,
    0.0258075980961766535646,
    0.0359571033071293220968,
    0.0464628932617579865414,
    0.0569795094941233574122,
    0.0672077542959907035404,
    0.0768796204990035310427,
    0.0857559200499903511542,
    0.0936271099812644736167,
    0.100314278611795578771,
    0.105669893580234809744,
    0.109578421055924638237,
    0.111956873020953456880,
    0.112755256720768691607,
    0.111956873020953456880,
    0.109578421055924638237,
    0.105669893580234809744,
    0.100314278611795578771,
    0.0936271099812644736167,
    0.0857559200499903511542,
    0.0768796204990035310427,
    0.0672077542959907035404,
    0.0569795094941233574122,
    0.0464628932617579865414,
    0.0359571033071293220968,
    0.0258075980961766535646,
    0.0164460498543878109338,
    0.00843456573932110624631,
    0.00254478079156187441540
  };
  static double w_063[63] =
  {
    0.000363221481845530659694,
    0.00126515655623006801137,
    0.00257904979468568827243,
    0.00421763044155885483908,
    0.00611550682211724633968,
    0.00822300795723592966926,
    0.0104982469096213218983,
    0.0129038001003512656260,
    0.0154067504665594978021,
    0.0179785515681282703329,
    0.0205942339159127111492,
    0.0232314466399102694433,
    0.0258696793272147469108,
    0.0284897547458335486125,
    0.0310735511116879648799,
    0.0336038771482077305417,
    0.0360644327807825726401,
    0.0384398102494555320386,
    0.0407155101169443189339,
    0.0428779600250077344929,
    0.0449145316536321974143,
    0.0468135549906280124026,
    0.0485643304066731987159,
    0.0501571393058995374137,
    0.0515832539520484587768,
    0.0528349467901165198621,
    0.0539054993352660639269,
    0.0547892105279628650322,
    0.0554814043565593639878,
    0.0559784365104763194076,
    0.0562776998312543012726,
    0.0563776283603847173877,
    0.0562776998312543012726,
    0.0559784365104763194076,
    0.0554814043565593639878,
    0.0547892105279628650322,
    0.0539054993352660639269,
    0.0528349467901165198621,
    0.0515832539520484587768,
    0.0501571393058995374137,
    0.0485643304066731987159,
    0.0468135549906280124026,
    0.0449145316536321974143,
    0.0428779600250077344929,
    0.0407155101169443189339,
    0.0384398102494555320386,
    0.0360644327807825726401,
    0.0336038771482077305417,
    0.0310735511116879648799,
    0.0284897547458335486125,
    0.0258696793272147469108,
    0.0232314466399102694433,
    0.0205942339159127111492,
    0.0179785515681282703329,
    0.0154067504665594978021,
    0.0129038001003512656260,
    0.0104982469096213218983,
    0.00822300795723592966926,
    0.00611550682211724633968,
    0.00421763044155885483908,
    0.00257904979468568827243,
    0.00126515655623006801137,
    0.000363221481845530659694
  };
  static double w_127[127] =
  {
    0.0000505360952078625176247,
    0.000180739564445388357820,
    0.000377746646326984660274,
    0.000632607319362633544219,
    0.000938369848542381500794,
    0.00128952408261041739210,
    0.00168114286542146990631,
    0.00210881524572663287933,
    0.00256876494379402037313,
    0.00305775341017553113613,
    0.00357289278351729964938,
    0.00411150397865469304717,
    0.00467105037211432174741,
    0.00524912345480885912513,
    0.00584344987583563950756,
    0.00645190005017573692280,
    0.00707248999543355546805,
    0.00770337523327974184817,
    0.00834283875396815770558,
    0.00898927578406413572328,
    0.00964117772970253669530,
    0.0102971169579563555237,
    0.0109557333878379016480,
    0.0116157233199551347270,
    0.0122758305600827700870,
    0.0129348396636073734547,
    0.0135915710097655467896,
    0.0142448773729167743063,
    0.0148936416648151820348,
    0.0155367755558439824399,
    0.0161732187295777199419,
    0.0168019385741038652709,
    0.0174219301594641737472,
    0.0180322163903912863201,
    0.0186318482561387901863,
    0.0192199051247277660193,
    0.0197954950480974994880,
    0.0203577550584721594669,
    0.0209058514458120238522,
    0.0214389800125038672465,
    0.0219563663053178249393,
    0.0224572658268160987071,
    0.0229409642293877487608,
    0.0234067774953140062013,
    0.0238540521060385400804,
    0.0242821652033365993580,
    0.0246905247444876769091,
    0.0250785696529497687068,
    0.0254457699654647658126,
    0.0257916269760242293884,
    0.0261156733767060976805,
    0.0264174733950582599310,
    0.0266966229274503599062,
    0.0269527496676330319634,
    0.0271855132296247918192,
    0.0273946052639814325161,
    0.0275797495664818730349,
    0.0277407021782796819939,
    0.0278772514766137016085,
    0.0279892182552381597038,
    0.0280764557938172466068,
    0.0281388499156271506363,
    0.0281763190330166021307,
    0.0281888141801923586938,
    0.0281763190330166021307,
    0.0281388499156271506363,
    0.0280764557938172466068,
    0.0279892182552381597038,
    0.0278772514766137016085,
    0.0277407021782796819939,
    0.0275797495664818730349,
    0.0273946052639814325161,
    0.0271855132296247918192,
    0.0269527496676330319634,
    0.0266966229274503599062,
    0.0264174733950582599310,
    0.0261156733767060976805,
    0.0257916269760242293884,
    0.0254457699654647658126,
    0.0250785696529497687068,
    0.0246905247444876769091,
    0.0242821652033365993580,
    0.0238540521060385400804,
    0.0234067774953140062013,
    0.0229409642293877487608,
    0.0224572658268160987071,
    0.0219563663053178249393,
    0.0214389800125038672465,
    0.0209058514458120238522,
    0.0203577550584721594669,
    0.0197954950480974994880,
    0.0192199051247277660193,
    0.0186318482561387901863,
    0.0180322163903912863201,
    0.0174219301594641737472,
    0.0168019385741038652709,
    0.0161732187295777199419,
    0.0155367755558439824399,
    0.0148936416648151820348,
    0.0142448773729167743063,
    0.0135915710097655467896,
    0.0129348396636073734547,
    0.0122758305600827700870,
    0.0116157233199551347270,
    0.0109557333878379016480,
    0.0102971169579563555237,
    0.00964117772970253669530,
    0.00898927578406413572328,
    0.00834283875396815770558,
    0.00770337523327974184817,
    0.00707248999543355546805,
    0.00645190005017573692280,
    0.00584344987583563950756,
    0.00524912345480885912513,
    0.00467105037211432174741,
    0.00411150397865469304717,
    0.00357289278351729964938,
    0.00305775341017553113613,
    0.00256876494379402037313,
    0.00210881524572663287933,
    0.00168114286542146990631,
    0.00128952408261041739210,
    0.000938369848542381500794,
    0.000632607319362633544219,
    0.000377746646326984660274,
    0.000180739564445388357820,
    0.0000505360952078625176247
  };
  if ( order == 1 )
  {
    r8vec_copy ( order, w_001, w );
  }
  else if ( order == 3 )
  {
    r8vec_copy ( order, w_003, w );
  }
  else if ( order == 7 )
  {
    r8vec_copy ( order, w_007, w );
  }
  else if ( order == 15 )
  {
    r8vec_copy ( order, w_015, w );
  }
  else if ( order == 31 )
  {
    r8vec_copy ( order, w_031, w );
  }
  else if ( order == 63 )
  {
    r8vec_copy ( order, w_063, w );
  }
  else if ( order == 127 )
  {
    r8vec_copy ( order, w_127, w );
  }
  else
  {
    cerr << "\n";
    cerr << "GP_WEIGHTS - Fatal error!\n";
    cerr << "  Unexpected value of ORDER = " << order << ".\n";
    exit ( 1 );
  }
  return;
}
//****************************************************************************80

int i4_max ( int i1, int i2 )

//****************************************************************************80
//
//  Purpose:
//
//    I4_MAX returns the maximum of two I4's.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    13 October 1998
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int I1, I2, are two integers to be compared.
//
//    Output, int I4_MAX, the larger of I1 and I2.
//
{
  int value;

  if ( i2 < i1 )
  {
    value = i1;
  }
  else
  {
    value = i2;
  }
  return value;
}
//****************************************************************************80

int i4_min ( int i1, int i2 )

//****************************************************************************80
//
//  Purpose:
//
//    I4_MIN returns the minimum of two I4's.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    13 October 1998
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int I1, I2, two integers to be compared.
//
//    Output, int I4_MIN, the smaller of I1 and I2.
//
{
  int value;

  if ( i1 < i2 )
  {
    value = i1;
  }
  else
  {
    value = i2;
  }
  return value;
}
//****************************************************************************80

int i4_power ( int i, int j )

//****************************************************************************80
//
//  Purpose:
//
//    I4_POWER returns the value of I^J.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    01 April 2004
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int I, J, the base and the power.  J should be nonnegative.
//
//    Output, int I4_POWER, the value of I^J.
//
{
  int k;
  int value;

  if ( j < 0 )
  {
    if ( i == 1 )
    {
      value = 1;
    }
    else if ( i == 0 )
    {
      cerr << "\n";
      cerr << "I4_POWER - Fatal error!\n";
      cerr << "  I^J requested, with I = 0 and J negative.\n";
      exit ( 1 );
    }
    else
    {
      value = 0;
    }
  }
  else if ( j == 0 )
  {
    if ( i == 0 )
    {
      cerr << "\n";
      cerr << "I4_POWER - Fatal error!\n";
      cerr << "  I^J requested, with I = 0 and J = 0.\n";
      exit ( 1 );
    }
    else
    {
      value = 1;
    }
  }
  else if ( j == 1 )
  {
    value = i;
  }
  else
  {
    value = 1;
    for ( k = 1; k <= j; k++ )
    {
      value = value * i;
    }
  }
  return value;
}
//****************************************************************************80

int i4vec_product ( int n, int a[] )

//****************************************************************************80
//
//  Purpose:
//
//    I4VEC_PRODUCT multiplies the entries of an I4VEC.
//
//  Discussion:
//
//    An I4VEC is a vector of integer values.
//
//  Example:
//
//    Input:
//
//      A = ( 1, 2, 3, 4 )
//
//    Output:
//
//      I4VEC_PRODUCT = 24
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    17 May 2003
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int N, the number of entries in the vector.
//
//    Input, int A[N], the vector
//
//    Output, int I4VEC_PRODUCT, the product of the entries of A.
//
{
  int i;
  int product;

  product = 1;
  for ( i = 0; i < n; i++ )
  {
    product = product * a[i];
  }

  return product;
}
//****************************************************************************80

void i4vec_transpose_print ( int n, int a[], char *title )

//****************************************************************************80
//
//  Purpose:
//
//    I4VEC_TRANSPOSE_PRINT prints an I4VEC "transposed".
//
//  Discussion:
//
//    An I4VEC is a vector of integer values.
//
//  Example:
//
//    A = { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 }
//    TITLE = "My vector:  "
//
//    My vector:      1    2    3    4    5
//                    6    7    8    9   10
//                   11
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    03 July 2004
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int N, the number of components of the vector.
//
//    Input, int A[N], the vector to be printed.
//
//    Input, char *TITLE, a title to be printed first.
//    TITLE may be blank or NULL.
//
{
  int i;
  int ihi;
  int ilo;
  int title_len;

  if ( 0 < s_len_trim ( title ) )
  {
    title_len = strlen ( title );

    for ( ilo = 1; ilo <= n; ilo = ilo + 5 )
    {
      ihi = i4_min ( ilo + 5 - 1, n );
      if ( ilo == 1 )
      {
        cout << title;
      }
      else
      {
        for ( i = 1; i <= title_len; i++ )
        {
          cout << " ";
        }
      }
      for ( i = ilo; i <= ihi; i++ )
      {
        cout << setw(12) << a[i-1];
      }
      cout << "\n";
    }
  }
  else
  {
    for ( ilo = 1; ilo <= n; ilo = ilo + 5 )
    {
      ihi = i4_min ( ilo + 5 - 1, n );
      for ( i = ilo; i <= ihi; i++ )
      {
        cout << setw(12) << a[i-1];
      }
      cout << "\n";
    }
  }
  return;
}
//****************************************************************************80

void level_to_order ( int dim_num, int level[], int rule[], int order[] )

//****************************************************************************80
//
//  Purpose:
//
//    LEVEL_TO_ORDER converts 1D levels to 1D orders based on rule types.
//
//  Discussion:
//
//    The user must preallocate space for the output array ORDER.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL[DIM_NUM], the 1D levels.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Output, int ORDER[DIM_NUM], the 1D orders (number of points).
//
{
  int dim;

  for ( dim = 0; dim < dim_num; dim++ )
  {
    if ( level[dim] < 0 )
    {
      cerr << "\n";
      cerr << "LEVEL_TO_ORDER - Fatal error!\n";
      cerr << "  LEVEL(DIM) < 0.\n";
      exit ( 1 );
    }
    else if ( level[dim] == 0 )
    {
      order[dim] = 1;
    }
    else if ( rule[dim] == 1 )
    {
      order[dim] = i4_power ( 2, level[dim] ) + 1;
    }
    else if ( 2 <= rule[dim] && rule[dim] <= 5 )
    {
      order[dim] = i4_power ( 2, level[dim] + 1 ) - 1;
    }
    else if ( rule[dim] <= 7 )
    {
      order[dim] = i4_power ( 2, level[dim] + 1 ) - 1;
    }
    else
    {
      cerr << "\n";
      cerr << "LEVEL_TO_ORDER - Fatal error!\n";
      cerr << "  Unexpected value of RULE[" << dim << "] = " << rule[dim] << ".\n";
      exit ( 1 );
    }
  }
  return;
}
//****************************************************************************80

double lg_point ( int order, int index )

//****************************************************************************80
//
//  Purpose:
//
//    LG_POINT returns an abscissa of a Gauss Laguerre rule.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    The rule is defined on [0,+oo).
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 August 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.  Legal values 
//    are 1, 3, 7, 15, 31, 63 and 127.
//
//    Input, int INDEX, the index of the abscissa.  1 <= INDEX <= ORDER.
//
//    Output, double LG_POINT, the abscissa.
//
{
  double value;
  static double x_001[1] =
  {
    1.0E+00
  };
  static double x_003[3] =
  {
    0.415774556783479083311533873128E+00, 
    0.229428036027904171982205036136E+01, 
    0.628994508293747919686641576551E+01
  };
  static double x_007[7] = 
  {
    0.193043676560362413838247885004E+00, 
    0.102666489533919195034519944317E+01, 
    0.256787674495074620690778622666E+01, 
    0.490035308452648456810171437810E+01, 
    0.818215344456286079108182755123E+01, 
    0.127341802917978137580126424582E+02, 
    0.193957278622625403117125820576E+02
  };
  static double x_015[15] =
  {
    0.933078120172818047629030383672E-01, 
    0.492691740301883908960101791412E+00, 
    0.121559541207094946372992716488E+01, 
    0.226994952620374320247421741375E+01, 
    0.366762272175143727724905959436E+01, 
    0.542533662741355316534358132596E+01, 
    0.756591622661306786049739555812E+01, 
    0.101202285680191127347927394568E+02, 
    0.131302824821757235640991204176E+02, 
    0.166544077083299578225202408430E+02, 
    0.207764788994487667729157175676E+02, 
    0.256238942267287801445868285977E+02, 
    0.314075191697539385152432196202E+02, 
    0.385306833064860094162515167595E+02, 
    0.480260855726857943465734308508E+02
  };
  static double x_031[31] =
  {
    0.45901947621108290743496080275224E-01, 
    0.24198016382477204890408974151714E+00, 
    0.59525389422235073707330165005414E+00, 
    1.1066894995329987162111308789792E+00, 
    1.7775956928747727211593727482675E+00, 
    2.6097034152566806503893375925315E+00, 
    3.6051968023400442698805817554243E+00, 
    4.7667470844717611313629127271123E+00, 
    6.0975545671817409269925429328463E+00, 
    7.6014009492331374229360106942867E+00, 
    9.2827143134708894182536695297710E+00, 
    11.146649755619291358993815629587E+00, 
    13.199189576244998522464925028637E+00, 
    15.447268315549310075809325891801E+00, 
    17.898929826644757646725793817752E+00, 
    20.563526336715822170743048968779E+00, 
    23.451973482011858591050255575933E+00, 
    26.577081352118260459975876986478E+00, 
    29.953990872346445506951917840024E+00, 
    33.600759532902202735410313885784E+00, 
    37.539164407330440882887902558001E+00, 
    41.795830870182219981347945853330E+00, 
    46.403866806411123136029227604386E+00, 
    51.405314476797755161861461088395E+00, 
    56.854992868715843620511922055660E+00, 
    62.826855908786321453677523304806E+00, 
    69.425277191080345623322251656443E+00, 
    76.807047763862732837609972285484E+00, 
    85.230358607545669169387065607043E+00, 
    95.188939891525629981308606853957E+00, 
    107.95224382757871475002440117666E+00
  };
  static double x_063[63] =
  {
    0.22768893732576153785994330248562E-01, 
    0.11998325242727824715771416426383E+00, 
    0.29494185444770149577427738517405E+00, 
    0.54779087896237725363865073775856E+00, 
    0.87869061179931901673895567052285E+00, 
    1.2878464335919706302309207788611E+00, 
    1.7755123815388553763979463268728E+00, 
    2.3419925567085989256055628337716E+00, 
    2.9876423223246473939976731053629E+00, 
    3.7128695992018000346299637413422E+00, 
    4.5181363349503584391105568561550E+00, 
    5.4039601781825946286902599782736E+00, 
    6.3709163787865330220392250891777E+00, 
    7.4196399339311711154888493199004E+00, 
    8.5508280008403328312589048722235E+00, 
    9.7652425999245366807004592977996E+00, 
    11.063713635140661736220550410604E+00, 
    12.447142262356492749798687569289E+00, 
    13.916504641057818562912967008183E+00, 
    15.472856110036296424777143607779E+00, 
    17.117335833863588753116900303886E+00, 
    18.851171974154856850873483787506E+00, 
    20.675687448056515660377265667433E+00, 
    22.592306346311528381292277759986E+00, 
    24.602561094972638883700642760037E+00, 
    26.708100458737343969779087998829E+00, 
    28.910698500451382640177718103234E+00, 
    31.212264631175912885477773820802E+00, 
    33.614854909101154836598842888345E+00, 
    36.120684774484823056306328740825E+00, 
    38.732143442933582145626041607663E+00, 
    41.451810222318741191114726181363E+00, 
    44.282473071479233839358857134636E+00, 
    47.227149784295686898935095231536E+00, 
    50.289112264240695761749021839419E+00, 
    53.471914456788652808348280619542E+00, 
    56.779424636342062213099781057119E+00, 
    60.215862909019862886417550114424E+00, 
    63.785845004235974631701139601836E+00, 
    67.494433702293885830374325695045E+00, 
    71.347199604295266286654803376075E+00, 
    75.350293425653234254290504744279E+00, 
    79.510532629986309149555391354778E+00, 
    83.835506080872257843339817658508E+00, 
    88.333701570354369086112766326498E+00, 
    93.014662728558547405303399037100E+00, 
    97.889184147578140043386727677112E+00, 
    102.96955690741381650783952746778E+00, 
    108.26988161961595392226350967206E+00, 
    113.80647350287462738934485955901E+00, 
    119.59839538830458666962452963285E+00, 
    125.66817255856119431291196303280E+00, 
    132.04277272091165746585590583045E+00, 
    138.75498418103789078167590567526E+00, 
    145.84541318313540358283994248439E+00, 
    153.36548459497863623710815962660E+00, 
    161.38215194813761243562172669592E+00, 
    169.98570600665839438795175301156E+00, 
    179.30366247401580910251827858515E+00, 
    189.52789596532475473668721332981E+00, 
    200.97521159924656741628671841018E+00, 
    214.25368536638788642698056296400E+00, 
    230.93465747089703971246562985079E+00
  };
  static double x_127[127] =
  {
    0.11339635298518611691893169631306E-01, 
    0.59749753435726620281348237057387E-01, 
    0.14685098690746167612388223687431E+00, 
    0.27267590735859553131378008278900E+00, 
    0.43724600644192665554577035869932E+00, 
    0.64058688222566929533576416399983E+00, 
    0.88272968639058364481487653650042E+00, 
    1.1637114160166537661560584700951E+00, 
    1.4835750152834613891313584861012E+00, 
    1.8423694351613565380686320809853E+00, 
    2.2401496839579024244513315656522E+00, 
    2.6769768780141303692167869961238E+00, 
    3.1529182957082825565771508308846E+00, 
    3.6680474360304752540226339926515E+00, 
    4.2224440823301888455977876667425E+00, 
    4.8161943715870502475665535087286E+00, 
    5.4493908694559416755862178908416E+00, 
    6.1221326512997254193944584763155E+00, 
    6.8345253894122668112237994973336E+00, 
    7.5866814466367472174205986836847E+00, 
    8.3787199765932725254842120659452E+00, 
    9.2107670307426558777922506102445E+00, 
    10.082955672528643809166439353647E+00, 
    10.995426098858125429803147358780E+00, 
    11.948325769197725997610605127857E+00, 
    12.941809542585531053723381098192E+00, 
    13.976039822878506520014405668679E+00, 
    15.051186712579523631574796365435E+00, 
    16.167428175612852922977395051768E+00, 
    17.324950209443673446561163712616E+00, 
    18.523947026965688560811711309349E+00, 
    19.764621248611504104071669386884E+00, 
    21.047184105173183606877044020054E+00, 
    22.371855651855542817648123918101E+00, 
    23.738864994122497183652313788712E+00, 
    25.148450525937368234077278385644E+00, 
    26.600860181041749607253384279755E+00, 
    28.096351697964619201753961292129E+00, 
    29.635192899504178910610227138642E+00, 
    31.217661987479759144214467152615E+00, 
    32.844047853610430460522951341338E+00, 
    34.514650407441149149105635947422E+00, 
    36.229780922306804019615388508885E+00, 
    37.989762400399956435968780140278E+00, 
    39.794929958089961778396437141707E+00, 
    41.645631232730180705153990897484E+00, 
    43.542226812286859549950892993822E+00, 
    45.485090689228791137996151336673E+00, 
    47.474610740231964719468766599146E+00, 
    49.511189233379087716728884584381E+00, 
    51.595243364671244443182771266934E+00, 
    53.727205825819316758288140069145E+00, 
    55.907525405447553305830605991732E+00, 
    58.136667626022439197077526025660E+00, 
    60.415115419018590295707192053805E+00, 
    62.743369841051809700207126742685E+00, 
    65.121950833949996311956025417139E+00, 
    67.551398031997886314411872443149E+00, 
    70.032271619884584511229871192030E+00, 
    72.565153245206849090888669416801E+00, 
    75.150646989739935299354362325096E+00, 
    77.789380404085816000647405462136E+00, 
    80.482005610750729205803962926758E+00, 
    83.229200481195914886796120019048E+00, 
    86.031669892953582966798238732643E+00, 
    88.890147073512051099652518544282E+00, 
    91.805395038358177994971250170499E+00, 
    94.778208131331583205387031034825E+00, 
    97.809413676305116411054110115424E+00, 
    100.89987375017285940371939762172E+00, 
    104.05048708821598934704076845022E+00, 
    107.26219113414600428423116401414E+00, 
    110.53596424851500530602771351277E+00, 
    113.87282809075839485348376187652E+00, 
    117.27385019192517774095477886379E+00, 
    120.74014673718880106173978002719E+00, 
    124.27288557955698354259506446928E+00, 
    127.87328950885942645093841745425E+00, 
    131.54263980314366921809377742137E+00, 
    135.28228009311836970132738106369E+00, 
    139.09362057432970013964422086977E+00, 
    142.97814260643601776808227753574E+00, 
    146.93740374437366549441080969072E+00, 
    150.97304325252187127492511437460E+00, 
    155.08678816034612572229641420609E+00, 
    159.28045992663288235401956989889E+00, 
    163.55598178957571104015967182053E+00, 
    167.91538689194360134245547184721E+00, 
    172.36082728473812536838156191681E+00, 
    176.89458392960192176311674993508E+00, 
    181.51907784036813069227528834025E+00, 
    186.23688252828112373861202530357E+00, 
    191.05073794450929196790836610789E+00, 
    195.96356614879879837839002542988E+00, 
    200.97848897600025153696475526130E+00, 
    206.09884802468871112127283042753E+00, 
    211.32822735671655260572377256981E+00, 
    216.67047937658230323477089465777E+00, 
    222.12975445929687246267304963754E+00, 
    227.71053502072232419089132431317E+00, 
    233.41767488282602453367775322563E+00, 
    239.25644498830308620018749667089E+00, 
    245.23258677871567172531254018984E+00, 
    251.35237488718128030005500991754E+00, 
    257.62269123792061413076191882313E+00, 
    264.05111322908240551754377241831E+00, 
    270.64601945722796749299111718606E+00, 
    277.41671750163651071798388218104E+00, 
    284.37359974220870326674402873120E+00, 
    291.52833521346495719581282021650E+00, 
    298.89410837028248600878895615414E+00, 
    306.48591978262611320418112423947E+00, 
    314.32096986471177487400007507615E+00, 
    322.41915589128679683349440361344E+00, 
    330.80372663802405651933847334878E+00, 
    339.50216127832433747735367595958E+00, 
    348.54737559472697355480761787441E+00, 
    357.97942028029845454049007443090E+00, 
    367.84794520076004578858341422871E+00, 
    378.21590623135532818332979188889E+00, 
    389.16539141251004101579475325153E+00, 
    400.80729331451702589996361286427E+00, 
    413.29853681779384418008260081859E+00, 
    426.87579153663675538288509017051E+00, 
    441.93085485310841412460309271842E+00, 
    459.21804639888429981971267313224E+00, 
    480.69378263388373859704269229304E+00
  };

  if ( order < 1 )
  {
    value = - r8_huge ( );
  }
  else if ( index < 1 || order < index )
  {
    value = - r8_huge ( );
  }
  else if ( order == 1 )
  {
    value = x_001[index-1];
  }
  else if ( order == 3 )
  {
    value = x_003[index-1];
  }
  else if ( order == 7 )
  {
    value = x_007[index-1];
  }
  else if ( order == 15 )
  {
    value = x_015[index-1];
  }
  else if ( order == 31 )
  {
    value = x_031[index-1];
  }
  else if ( order == 63 )
  {
    value = x_063[index-1];
  }
  else if ( order == 127 )
  {
    value = x_127[index-1];
  }
  else
  {
    value = - r8_huge ( );
  }
  return value;
}
//****************************************************************************80

void lg_weights ( int order, double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    LG_WEIGHTS returns weights for certain Gauss-Laguerre quadrature rules.
//
//  Discussion:
//
//    The allowed orders are 1, 3, 7, 15, 31, 63 and 127.
//
//    The weights are positive, symmetric and should sum to 2.
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Milton Abramowitz, Irene Stegun,
//    Handbook of Mathematical Functions,
//    National Bureau of Standards, 1964,
//    ISBN: 0-486-61272-4,
//    LC: QA47.A34.
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    ORDER must be 1, 3, 7, 15, 31, 63 or 127.
//
//    Output, double W[ORDER], the weights.
//
{
  static double w_001[1] =
  {
    1.0E+00
  };
  static double w_003[3] = 
  {
    0.711093009929173015449590191143E+00,
    0.278517733569240848801444888457E+00,
    0.103892565015861357489649204007E-01
  };
  static double w_007[7] =
  {
    0.409318951701273902130432880018E+00,
    0.421831277861719779929281005417E+00,
    0.147126348657505278395374184637E+00,
    0.206335144687169398657056149642E-01,
    0.107401014328074552213195962843E-02,
    0.158654643485642012687326223234E-04,
    0.317031547899558056227132215385E-07
  };
  static double w_015[15] =
  {
    0.218234885940086889856413236448E+00,
    0.342210177922883329638948956807E+00,
    0.263027577941680097414812275022E+00,
    0.126425818105930535843030549378E+00,
    0.402068649210009148415854789871E-01,
    0.856387780361183836391575987649E-02,
    0.121243614721425207621920522467E-02,
    0.111674392344251941992578595518E-03,
    0.645992676202290092465319025312E-05,
    0.222631690709627263033182809179E-06,
    0.422743038497936500735127949331E-08,
    0.392189726704108929038460981949E-10,
    0.145651526407312640633273963455E-12,
    0.148302705111330133546164737187E-15,
    0.160059490621113323104997812370E-19
  };
  static double w_031[31] =
  {
     0.11252789550372583820847728082801E+00,
     0.21552760818089123795222505285045E+00,
     0.23830825164569654731905788089234E+00,
     0.19538830929790229249915303390711E+00,
     0.12698283289306190143635272904602E+00,
     0.67186168923899300670929441993508E-01,
     0.29303224993879487404888669311974E-01,
     0.10597569915295736089529380314433E-01,
     0.31851272582386980320974842433019E-02,
     0.79549548307940382922092149012477E-03,
     0.16480052126636687317862967116412E-03,
     0.28229237864310816393860971468993E-04,
     0.39802902551008580387116174900106E-05,
     0.45931839841801061673729694510289E-06,
     0.43075545187731100930131457465897E-07,
     0.32551249938271570855175749257884E-08,
     0.19620246675410594996247151593142E-09,
     0.93190499086617587129534716431331E-11,
     0.34377541819411620520312597898311E-12,
     0.96795247130446716997405035776206E-14,
     0.20368066110115247398010624219291E-15,
     0.31212687280713526831765358632585E-17,
     0.33729581704161052453395678308350E-19,
     0.24672796386616696011038363242541E-21,
     0.11582201904525643634834564576593E-23,
     0.32472922591425422434798022809020E-26,
     0.49143017308057432740820076259666E-29,
     0.34500071104808394132223135953806E-32,
     0.87663710117162041472932760732881E-36,
     0.50363643921161490411297172316582E-40,
     0.19909984582531456482439549080330E-45
  };
  static double w_063[63] =
  {
     0.57118633213868979811587283390476E-01,
     0.12067476090640395283319932036351E+00,
     0.15925001096581873723870561096472E+00,
     0.16875178327560799234596192963585E+00,
     0.15366641977668956696193711310131E+00,
     0.12368770614716481641086652261948E+00,
     0.89275098854848671545279150057422E-01,
     0.58258485446105944957571825725160E-01,
     0.34546657545992580874717085812508E-01,
     0.18675685985714656798286552591203E-01,
     0.92233449044093536528490075241649E-02,
     0.41671250684839592762582663470209E-02,
     0.17238120299900582715386728541955E-02,
     0.65320845029716311169340559359043E-03,
     0.22677644670909586952405173207471E-03,
     0.72127674154810668410750270234861E-04,
     0.21011261180466484598811536851241E-04,
     0.56035500893357212749181536071292E-05,
     0.13673642785604888017836641282292E-05,
     0.30507263930195817240736097189550E-06,
     0.62180061839309763559981775409241E-07,
     0.11566529551931711260022448996296E-07,
     0.19614588267565478081534781863335E-08,
     0.30286171195709411244334756404054E-09,
     0.42521344539400686769012963452599E-10,
     0.54202220578073819334698791381873E-11,
     0.62627306838597672554166850420603E-12,
     0.65474443156573322992307089591924E-13,
     0.61815575808729181846302500000047E-14,
     0.52592721363507381404263991342633E-15,
     0.40230920092646484015391506025408E-16,
     0.27600740511819536505013824207729E-17,
     0.16936946756968296053322009855265E-18,
     0.92689146872177087314963772462726E-20,
     0.45093739060365632939780140603959E-21,
     0.19435162876132376573629962695374E-22,
     0.73926270895169207037999639194513E-24,
     0.24714364154434632615980126000066E-25,
     0.72288649446741597655145390616476E-27,
     0.18407617292614039362985209905608E-28,
     0.40583498566841960105759537058880E-30,
     0.77000496416438368114463925286343E-32,
     0.12488505764999334328843314866038E-33,
     0.17185000226767010697663950619912E-35,
     0.19896372636672396938013975755522E-37,
     0.19199671378804058267713164416870E-39,
     0.15278588285522166920459714708240E-41,
     0.99054752688842142955854138884590E-44,
     0.51597523673029211884228858692990E-46,
     0.21249846664084111245693912887783E-48,
     0.67903852766852910591172042494884E-51,
     0.16466654148296177467908300517887E-53,
     0.29509065402691055027053659375033E-56,
     0.37838420647571051984882241014675E-59,
     0.33358130068542431878174667995217E-62,
     0.19223461022273880981363303073329E-65,
     0.67812696961083016872779388922288E-69,
     0.13404752802440604607620468935693E-72,
     0.13109745101805029757648048223928E-76,
     0.52624863881401787388694579143866E-81,
     0.63780013856587414257760666006511E-86,
     0.12997078942372924566347473916943E-91,
     0.10008511496968754063443740168421E-98
  };
  static double w_127[127] =
  {
     0.28773246692000124355770010301506E-01,
     0.63817468175134649363480949265236E-01,
     0.91919669721570571389864194652717E-01,
     0.11054167914413766381245463002967E+00,
     0.11879771633375850188328329422643E+00,
     0.11737818530052695148804451630074E+00,
     0.10819305984180551488335145581193E+00,
     0.93827075290489628080377261401107E-01,
     0.76966450960588843995822485928431E-01,
     0.59934903912939714332570730063476E-01,
     0.44417742073889001371708316272923E-01,
     0.31385080966252320983009372215062E-01,
     0.21172316041924506411370709025015E-01,
     0.13650145364230541652171185564626E-01,
     0.84172852710599172279366657385445E-02,
     0.49674990059882760515912858620175E-02,
     0.28069903895001884631961957446400E-02,
     0.15192951003941952460445341057817E-02,
     0.78789028751796084086217287140548E-03,
     0.39156751064868450584507324648999E-03,
     0.18652434268825860550093566260060E-03,
     0.85173160415576621908809828160247E-04,
     0.37285639197853037712145321577724E-04,
     0.15648416791712993947447805296768E-04,
     0.62964340695224829035692735524979E-05,
     0.24288929711328724574541379938222E-05,
     0.89824607890051007201922871545035E-06,
     0.31844174740760353710742966328091E-06,
     0.10821272905566839211861807542741E-06,
     0.35245076750635536015902779085340E-07,
     0.11001224365719347407063839761738E-07,
     0.32904079616717932125329343003261E-08,
     0.94289145237889976419772700772988E-09,
     0.25882578904668318184050195309296E-09,
     0.68047437103370762630942259017560E-10,
     0.17131398805120837835399564475632E-10,
     0.41291744524052865469443922304935E-11,
     0.95264189718807273220707664873469E-12,
     0.21032604432442425932962942047474E-12,
     0.44427151938729352860940434285789E-13,
     0.89760500362833703323319846405449E-14,
     0.17341511407769287074627948346848E-14,
     0.32028099548988356631494379835210E-15,
     0.56531388950793682022660742095189E-16,
     0.95329672799026591234588044025896E-17,
     0.15353453477310142565288509437552E-17,
     0.23608962179467365686057842132176E-18,
     0.34648742794456611332193876653230E-19,
     0.48515241897086461320126957663545E-20,
     0.64786228633519813428137373790678E-21,
     0.82476020965403242936448553126316E-22,
     0.10005361880214719793491658282977E-22,
     0.11561395116207304954233181263632E-23,
     0.12719342731167922655612134264961E-24,
     0.13316584714165372967340004160814E-25,
     0.13261218454678944033646108509198E-26,
     0.12554995447643949807286074138324E-27,
     0.11294412178579462703240913107219E-28,
     0.96491020279562119228500608131696E-30,
     0.78241846768302099396733076955632E-31,
     0.60181503542219626658249939076636E-32,
     0.43882482704961741551510518054138E-33,
     0.30314137647517256304035802501863E-34,
     0.19826016543944539545224676057020E-35,
     0.12267623373665926559013654872402E-36,
     0.71763931692508888943812834967620E-38,
     0.39659378833836963584113716149270E-39,
     0.20688970553868040099581951696677E-40,
     0.10179587017979517245268418427523E-41,
     0.47200827745986374625714293679649E-43,
     0.20606828985553374825744353490744E-44,
     0.84627575907305987245899032156188E-46,
     0.32661123687088798658026998931647E-47,
     0.11833939207883162380564134612682E-48,
     0.40211209123895013807243250164050E-50,
     0.12799824394111125389430292847476E-51,
     0.38123877747548846504399051365162E-53,
     0.10612057542701156767898551949650E-54,
     0.27571446947200403594113572720812E-56,
     0.66772544240928492881306904862856E-58,
     0.15052438383868234954068178600268E-59,
     0.31538986800113758526689068500772E-61,
     0.61326614299483180785237418887960E-63,
     0.11048510030324810567549119229368E-64,
     0.18410563538091348076979665543900E-66,
     0.28323926570052832195543883237652E-68,
     0.40154409843763655508670978777418E-70,
     0.52351530215683708779772201956106E-72,
     0.62634476665005100555787696642851E-74,
     0.68612210535666530365348093803922E-76,
     0.68651298840956019297134099761855E-78,
     0.62581388433728084867318704240915E-80,
     0.51833271237514904046803469968027E-82,
     0.38893621571918443533108973497673E-84,
     0.26357711379476932781525533730623E-86,
     0.16078851293917979699005509638883E-88,
     0.87978042070968939637972577886624E-91,
     0.43013405077495109903408697802188E-93,
     0.18713435881342838527144321803729E-95,
     0.72125744708060471675805761366523E-98,
     0.24508746062177874383231742333023E-100,
     0.73042094619470875777647865078327E-103,
     0.18983290818383463537886818579820E-105,
     0.42757400244246684123093264825902E-108,
     0.82894681420515755691423485228897E-111,
     0.13729432219324400013067050156048E-113,
     0.19265464126404973222043166489406E-116,
     0.22693344503301354826140809941334E-119,
     0.22209290603717355061909071271535E-122,
     0.17851087685544512662856555121755E-125,
     0.11630931990387164467431190485525E-128,
     0.60524443584652392290952805077893E-132,
     0.24729569115063528647628375096400E-135,
     0.77789065006489410364997205809045E-139,
     0.18409738662712607039570678274636E-142,
     0.31900921131079114970179071968597E-146,
     0.39179487139174199737617666077555E-150,
     0.32782158394188697053774429820559E-154,
     0.17793590713138888062819640128739E-158,
     0.58882353408932623157467835381214E-163,
     0.10957236509071169877747203273886E-167,
     0.10281621114867000898285076975760E-172,
     0.41704725557697758145816510853967E-178,
     0.58002877720316101774638319601971E-184,
     0.18873507745825517106171619101120E-190,
     0.69106601826730911682786705950895E-198,
     0.43506813201105855628383313334402E-207
  };
  if ( order == 1 )
  {
    r8vec_copy ( order, w_001, w );
  }
  else if ( order == 3 )
  {
    r8vec_copy ( order, w_003, w );
  }
  else if ( order == 7 )
  {
    r8vec_copy ( order, w_007, w );
  }
  else if ( order == 15 )
  {
    r8vec_copy ( order, w_015, w );
  }
  else if ( order == 31 )
  {
    r8vec_copy ( order, w_031, w );
  }
  else if ( order == 63 )
  {
    r8vec_copy ( order, w_063, w );
  }
  else if ( order == 127 )
  {
    r8vec_copy ( order, w_127, w );
  }
  else
  {
    cerr << "\n";
    cerr << "LG_WEIGHTS - Fatal error!\n";
    cerr << "  Unexpected value of ORDER = " << order << ".\n";
    exit ( 1 );
  }
  return;
}
//****************************************************************************80

void product_mixed_weight ( int dim_num, int order_1d[], int order_nd, 
  int rule[], double weight_nd[] )

//****************************************************************************80
//
//  Purpose:
//
//    PRODUCT_MIXED_WEIGHT computes the weights of a mixed product rule.
//
//  Discussion:
//
//    This routine computes the weights for a quadrature rule which is
//    a product of 1D open rules of varying order and kind.
//
//    The user must preallocate space for the output array WEIGHT_ND.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int ORDER_1D[DIM_NUM], the order of the 1D rules.
//
//    Input, int ORDER_ND, the order of the product rule.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Output, double WEIGHT_ND[ORDER_ND], the product rule weights.
//
{
  int dim;
  int i;
  double *weight_1d;

  for ( i = 0; i < order_nd; i++ )
  {
    weight_nd[i] = 1.0;
  }

  for ( dim = 0; dim < dim_num; dim++ )
  {
    weight_1d = new double[order_1d[dim]];

    if ( rule[dim] == 1 )
    {
      cc_weights ( order_1d[dim], weight_1d );
    }
    else if ( rule[dim] == 2 )
    {
      f2_weights ( order_1d[dim], weight_1d );
    }
    else if ( rule[dim] == 3 )
    {
      gp_weights ( order_1d[dim], weight_1d );
    }
    else if ( rule[dim] == 4 )
    {
      gl_weights ( order_1d[dim], weight_1d );
    }
    else if ( rule[dim] == 5 )
    {
      gh_weights ( order_1d[dim], weight_1d );
    }
    else if ( rule[dim] == 7 )
    {
      lg_weights ( order_1d[dim], weight_1d );
    }
    else
    {
      cerr << "\n";
      cerr << "PRODUCT_MIXED_WEIGHT - Fatal error!\n";
      cerr << "  Unexpected value of RULE[" << dim << "] = " << rule[dim] << ".\n";
      exit ( 1 );
    }

    r8vec_direct_product2 ( dim, order_1d[dim], weight_1d, 
      dim_num, order_nd, weight_nd );

    delete [] weight_1d;
  }
  return;
}
//****************************************************************************80

double r8_abs ( double x )

//****************************************************************************80
//
//  Purpose:
//
//    R8_ABS returns the absolute value of an R8.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    14 November 2006
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, double X, the quantity whose absolute value is desired.
//
//    Output, double R8_ABS, the absolute value of X.
//
{
  double value;

  if ( 0.0 <= x )
  {
    value = x;
  } 
  else
  {
    value = -x;
  }
  return value;
}
//****************************************************************************80

double r8_choose ( int n, int k )

//****************************************************************************80
//
//  Purpose:
//
//    R8_CHOOSE computes the binomial coefficient C(N,K) as an R8.
//
//  Discussion:
//
//    The value is calculated in such a way as to avoid overflow and
//    roundoff.  The calculation is done in R8 arithmetic.
//
//    The formula used is:
//
//      C(N,K) = N// / ( K// * (N-K)// )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 March 2008
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    ML Wolfson, HV Wright,
//    Algorithm 160:
//    Combinatorial of M Things Taken N at a Time,
//    Communications of the ACM,
//    Volume 6, Number 4, April 1963, page 161.
//
//  Parameters:
//
//    Input, int N, K, the values of N and K.
//
//    Output, double R8_CHOOSE, the number of combinations of N
//    things taken K at a time.
//
{
  int i;
  int mn;
  int mx;
  int value;

  mn = i4_min ( k, n - k );

  if ( mn < 0 )
  {
    value = 0.0;
  }
  else if ( mn == 0 )
  {
    value = 1.0;
  }
  else
  {
    mx = i4_max ( k, n - k );
    value = ( double ) ( mx + 1 );

    for ( i = 2; i <= mn; i++ )
    {
      value = ( value * ( double ) ( mx + i ) ) / ( double ) i;
    }
  }
  return value;
}
//****************************************************************************80

double r8_epsilon ( void )

//****************************************************************************80
//
//  Purpose:
//
//    R8_EPSILON returns the R8 roundoff unit.
//
//  Discussion:
//
//    The roundoff unit is a number R which is a power of 2 with the 
//    property that, to the precision of the computer's arithmetic,
//      1 < 1 + R
//    but 
//      1 = ( 1 + R / 2 )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    01 July 2004
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Output, double R8_EPSILON, the R8 round-off unit.
//
{
  double value;

  value = 1.0;

  while ( 1.0 < ( double ) ( 1.0 + value )  )
  {
    value = value / 2.0;
  }

  value = 2.0 * value;

  return value;
}
//****************************************************************************80

double r8_huge ( void )

//****************************************************************************80
//
//  Purpose:
//
//    R8_HUGE returns a "huge" R8.
//
//  Discussion:
//
//    The value returned by this function is NOT required to be the
//    maximum representable R8.  This value varies from machine to machine,
//    from compiler to compiler, and may cause problems when being printed.
//    We simply want a "very large" but non-infinite number.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    06 October 2007
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Output, double R8_HUGE, a "huge" R8 value.
//
{
  double value;

  value = 1.0E+30;

  return value;
}
//****************************************************************************80

double r8_mop ( int i )

//****************************************************************************80
//
//  Purpose:
//
//    R8_MOP returns the I-th power of -1 as an R8 value.
//
//  Discussion:
//
//    An R8 is an double value.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    16 November 2007
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int I, the power of -1.
//
//    Output, double R8_MOP, the I-th power of -1.
//
{
  double value;

  if ( ( i % 2 ) == 0 )
  {
    value = 1.0;
  }
  else
  {
    value = -1.0;
  }

  return value;
}
//****************************************************************************80

void r8col_unique_index ( int m, int n, double a[], double tol, 
  int unique_index[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8COL_UNIQUE_INDEX indexes the first occurrence of values in an R8COL.
//
//  Discussion:
//
//    An R8COL is an M by N array of R8 values.
//    It is regarded as an array of N columns of length M.
//
//    For element A(1:M,J) of the matrix, UNIQUE_INDEX(J) is the uniqueness index
//    of A(1:M,J).  That is, if A_UNIQUE contains the unique elements of A, 
//    gathered in order, then 
//
//      A_UNIQUE ( 1:M, UNIQUE_INDEX(J) ) = A(1:M,J)
//
//    The user must preallocate space for the output array UNIQUE_INDEX.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int M, N, the number of rows and columns of A.
//    The length of an "element" of A, and the number of "elements".
//
//    Input, double A[M*N], the array.
//
//    Input, double TOL, a tolerance for equality.
//
//    Output, int UNIQUE_INDEX[N], the unique index.
//
{
  double diff;
  int i;
  int j1;
  int j2;
  int unique_num;

  for ( j1 = 0; j1 < n; j1++ )
  {
    unique_index[j1] = -1;
  }
  unique_num = 0;

  for ( j1 = 0; j1 < n; j1++ )
  {
    if ( unique_index[j1] == -1 )
    {
      unique_index[j1] = unique_num;

      for ( j2 = j1 + 1; j2 < n; j2++ )
      {
        diff = 0.0;
        for ( i = 0; i < m; i++ )
        {
          diff = diff + r8_abs ( a[i+j1*m] - a[i+j2*m] );
        }
        if ( diff <= tol )
        {
          unique_index[j2] = unique_num;
        }
      }
      unique_num = unique_num + 1;
    }
  }
  return;
}
//****************************************************************************80

void r8vec_copy ( int n, double a1[], double a2[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8VEC_COPY copies an R8VEC.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    03 July 2005
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int N, the number of entries in the vectors.
//
//    Input, double A1[N], the vector to be copied.
//
//    Output, double A2[N], the copy of A1.
//
{
  int i;

  for ( i = 0; i < n; i++ )
  {
    a2[i] = a1[i];
  }
  return;
}
//****************************************************************************80

void r8vec_direct_product2 ( int factor_index, int factor_order, 
  double factor_value[], int factor_num, int point_num, double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8VEC_DIRECT_PRODUCT2 creates a direct product of R8VEC's.
//
//  Discussion:
//
//    An R8VEC is a vector of double precision values.
//
//    To explain what is going on here, suppose we had to construct
//    a multidimensional quadrature rule as the product of K rules
//    for 1D quadrature.
//
//    The product rule will be represented as a list of points and weights.
//
//    The J-th item in the product rule will be associated with
//      item J1 of 1D rule 1,
//      item J2 of 1D rule 2, 
//      ..., 
//      item JK of 1D rule K.
//
//    In particular, 
//      X(J) = ( X(1,J1), X(2,J2), ..., X(K,JK))
//    and
//      W(J) = W(1,J1) * W(2,J2) * ... * W(K,JK)
//
//    So we can construct the quadrature rule if we can properly
//    distribute the information in the 1D quadrature rules.
//
//    This routine carries out that task for the weights W.
//
//    Another way to do this would be to compute, one by one, the
//    set of all possible indices (J1,J2,...,JK), and then index
//    the appropriate information.  An advantage of the method shown
//    here is that you can process the K-th set of information and
//    then discard it.
//
//  Example:
//
//    Rule 1: 
//      Order = 4
//      W(1:4) = ( 2, 3, 5, 7 )
//
//    Rule 2:
//      Order = 3
//      W(1:3) = ( 11, 13, 17 )
//
//    Rule 3:
//      Order = 2
//      W(1:2) = ( 19, 23 )
//
//    Product Rule:
//      Order = 24
//      W(1:24) =
//        ( 2 * 11 * 19 )
//        ( 3 * 11 * 19 )
//        ( 4 * 11 * 19 )
//        ( 7 * 11 * 19 )
//        ( 2 * 13 * 19 )
//        ( 3 * 13 * 19 )
//        ( 5 * 13 * 19 )
//        ( 7 * 13 * 19 )
//        ( 2 * 17 * 19 )
//        ( 3 * 17 * 19 )
//        ( 5 * 17 * 19 )
//        ( 7 * 17 * 19 )
//        ( 2 * 11 * 23 )
//        ( 3 * 11 * 23 )
//        ( 5 * 11 * 23 )
//        ( 7 * 11 * 23 )
//        ( 2 * 13 * 23 )
//        ( 3 * 13 * 23 )
//        ( 5 * 13 * 23 )
//        ( 7 * 13 * 23 )
//        ( 2 * 17 * 23 )
//        ( 3 * 17 * 23 )
//        ( 5 * 17 * 23 )
//        ( 7 * 17 * 23 )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    10 May 2007
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int FACTOR_INDEX, the index of the factor being processed.
//    The first factor processed must be factor 0.
//
//    Input, int FACTOR_ORDER, the order of the factor.
//
//    Input, double FACTOR_VALUE[FACTOR_ORDER], the factor values for
//    factor FACTOR_INDEX.
//
//    Input, int FACTOR_NUM, the number of factors.
//
//    Input, int POINT_NUM, the number of elements in the direct product.
//
//    Input/output, double W[POINT_NUM], the elements of the
//    direct product, which are built up gradually.  Before the first call,
//    W must be set to 1.
//
//  Local Parameters:
//
//    Local, integer START, the first location of a block of values to set.
//
//    Local, integer CONTIG, the number of consecutive values to set.
//
//    Local, integer SKIP, the distance from the current value of START
//    to the next location of a block of values to set.
//
//    Local, integer REP, the number of blocks of values to set.
//
{
  static int contig = 0;
  int i;
  int j;
  int k;
  static int rep = 0;
  static int skip = 0;
  int start;

  if ( factor_index == 0 )
  {
    contig = 1;
    skip = 1;
    rep = point_num;
  }

  rep = rep / factor_order;
  skip = skip * factor_order;

  for ( j = 0; j < factor_order; j++ )
  {
    start = 0 + j * contig;

    for ( k = 1; k <= rep; k++ )
    {
      for ( i = start; i < start + contig; i++ )
      {
        w[i] = w[i] * factor_value[j];
      }
      start = start + skip;
    }
  }

  contig = contig * factor_order;

  return;
}
//****************************************************************************80

double r8vec_sum ( int n, double a[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8VEC_SUM returns the sum of an R8VEC.
//
//  Discussion:
//
//    An R8VEC is a double precision vector.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    15 October 2004
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int N, the number of entries in the vector.
//
//    Input, double A[N], the vector.
//
//    Output, double R8VEC_SUM, the sum of the vector.
//
{
  int i;
  double value;

  value = 0.0;
  for ( i = 0; i < n; i++ )
  {
    value = value + a[i];
  }

  return value;
}
//****************************************************************************80

int s_len_trim ( char *s )

//****************************************************************************80
//
//  Purpose:
//
//    S_LEN_TRIM returns the length of a string to the last nonblank.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 April 2003
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, char *S, a pointer to a string.
//
//    Output, int S_LEN_TRIM, the length of the string to the last nonblank.
//    If S_LEN_TRIM is 0, then the string is entirely blank.
//
{
  int n;
  char *t;

  n = strlen ( s );
  t = s + strlen ( s ) - 1;

  while ( 0 < n ) 
  {
    if ( *t != ' ' )
    {
      return n;
    }
    t--;
    n--;
  }

  return n;
}
//****************************************************************************80

void sparse_grid_mixed_index ( int dim_num, int level_max, int rule[], 
  int point_num, int sparse_order[], int sparse_index[] )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_INDEX indexes a sparse grid made from mixed 1D rules.
//
//  Discussion:
//
//    For each "unique" point in the sparse grid, we return its INDEX and  and ORDER.
//
//    That is, for the I-th unique point P, we determine the product grid which
//    first generated this point, and  and we return in SPARSE_ORDER the orders of 
//    the 1D rules in that grid, and  and in SPARSE_INDEX the component indexes in 
//    those rules that generated this specific point.
//
//    For instance, say P was first generated by a rule which was a 3D product
//    of a 9th order CC rule and  and a 15th order GL rule, and  and that to generate P,
//    we used the 7-th point of the CC rule and  and the 3rh point of the GL rule.
//    Then the SPARSE_ORDER information would be (9,15) and  and the SPARSE_INDEX
//    information would be (7,3).  This, combined with the information in RULE,
//    is enough to regenerate the value of P.
//
//    The user must preallocate space for the output arrays SPARSE_ORDER and
//    SPARSE_INDEX.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL_MAX, the maximum value of LEVEL.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, int POINT_NUM, the number of unique points 
//    in the grid. 
//
//    Output, int SPARSE_ORDER[DIM_NUM*POINT_NUM], lists, 
//    for each point, the order of the 1D rules used in the grid that 
//    generated it.
//
//    Output, int SPARSE_INDEX[DIM_NUM*POINT_NUM)] lists, for 
//    each point, its index in each of the 1D rules in the grid that generated 
//    it.  The indices are 1-based.
//
{
  int d1;
  int d2;
  int dim;
  int dim1;
  int dim2;
  int h;
  int level;
  int *level_1d;
  int level_min;
  bool more_grids;
  bool more_points;
  bool new_point;
  int *order_1d;
  int *point_index;
  int point_unique_num;
  int t;
//
//  Special case.
//
  if ( level_max == 0 )
  {
    for ( dim = 0; dim < dim_num; dim++ )
    {
      sparse_order[dim+0*dim_num] = 1;
      sparse_index[dim+0*dim_num] = 1;
    }
    return;
  }

  point_unique_num = 0;

  level_1d = new int[dim_num];
  point_index = new int[dim_num];
//
//  The outer loop generates values of LEVEL.
//
  level_min = i4_max ( 0, level_max + 1 - dim_num );

  for ( level = level_min; level <= level_max; level++ )
  {
//
//  The middle loop generates a GRID, 
//  based on the next partition that adds up to LEVEL.
//
    more_grids = false;
    h = 0;
    t = 0;

    for ( ; ; )
    {
      comp_next ( level, dim_num, level_1d, &more_grids, &h, &t );

      order_1d = new int[dim_num];

      level_to_order ( dim_num, level_1d, rule, order_1d );
//
//  The inner loop generates a POINT of the GRID of the LEVEL.
//
      more_points = false;

      for ( ; ; )
      {
        vec_colex_next3 ( dim_num, order_1d, point_index, &more_points );

        if ( !more_points )
        {
          break;
        }
//
//  We examine each point and  and see if it is "new".
//
        new_point = true;
//
//  For a point to be old at a level equal to LEVEL_MIN:
//
//  There is a dimension D1 and  and another dimension D2 so that
//  the point also occurs in the grid whose value of LEVEL_1D(D1) is
//  decreased by one, and  and LEVEL_1D(D2) is increased by 1.
//
//  Moreover, D2 < D1 (so that we only count this once).
//
        if ( level_min == level )
        {
          for ( dim1 = 1; dim1 < dim_num; dim1++ )
          {
            d1 = -1;

            if ( 0 < level_1d[dim1] )
            {
              if ( rule[dim1] == 1 )
              {
                if ( 1 == level_1d[dim1] )
                {
                  if ( ( point_index[dim1] % 2 ) == 0 )
                  {
                    d1 = dim1;
                  }
                }
                else if ( 1 < level_1d[dim1] )
                {
                  if ( ( point_index[dim1] % 2 ) == 1 )
                  {
                    d1 = dim1;
                  }
                }
              }
              else if ( 2 <= rule[dim1] && rule[dim1] <= 3 )
              {
                if ( ( point_index[dim1] % 2 ) == 0 )
                {
                  d1 = dim1;
                }
              }
              else if ( 4 <= rule[dim1] && rule[dim1] <= 5 )
              {
                if ( 2 * point_index[dim1] - 1 == order_1d[dim1] )
                {
                  d1 = dim1;
                }
              }
              else if ( rule[dim1] == 7 )
              {
              }
              else 
              {
                cerr << "\n";
                cerr << "SPARSE_GRID_MIXED_INDEX - Fatal error!\n";
                cerr << "  Unexpected value of RULE[" << dim1 << "] = " << rule[dim1] << ".\n";
                exit ( 1 );
              }
//
//  A candidate for D1 found.  Does it have a D2?
//
              if ( d1 != -1 )
              {
                d2 = -1;

                for ( dim2 = 0; dim2 < dim1; dim2++ )
                {
                  if ( rule[dim2] == 1 )
                  {
                    d2 = dim2;
                    break;
                  }
                  else if ( 2 <= rule[dim2] && rule[dim2] <= 3 )
                  {
                    d2 = dim2;
                    break;
                  }
                  else if ( 4 <= rule[dim2] && rule[dim2] <= 5 )
                  {
                    if ( 2 * point_index[dim2] - 1 == order_1d[dim2] )
                    {
                      d2 = dim2;
                      break;
                    }
                  }
                  else if ( rule[dim2] == 7 )
                  {
                  }
                  else 
                  {
                    cerr << "\n";
                    cerr << "SPARSE_GRID_MIXED_INDEX- Fatal error!\n";
                    cerr << "  Unexpected value of RULE[" << dim2 << "] = " << rule[dim2] << ".\n";
                    exit ( 1 );
                  }
                }
//
//  If we found a D2, we know the point is OLD and  and are done.
//  Otherwise, consider next possible D1.
//
                if ( d2 != -1 )
                {
                  new_point = false;
                  break;
                }
                else
                {
                  d1 = -1;
                }
              }
            }
          }
        }
//
//  For a point to be old at a level greater than LEVEL_MIN:
//  If in component DIM, the point is a "nest" point that also occurs on the
//  previous 1D Level, and  and we're not at Level LEVEL_MIN, then for sure we've
//  already generated the point when we generated the grid:
//
//    ( Level_1d(1), ..., Level_1d(DIM-1), Level_1d(DIM)-1, Level_1d(DIM+1), ..., Level_1d(DIM_NUM) )
//
        else if ( level_min < level )
        {
          for ( dim = 0; dim < dim_num; dim++ )
          {
            if ( rule[dim] == 1 )
            {
              if ( 3 == order_1d[dim] && point_index[dim] == 2 )
              {
                new_point = false;
                break;
              }
              else if ( 3 < order_1d[dim] && ( point_index[dim] % 2 ) == 1 )
              {
                new_point = false;
                break;
              }
            }
            else if ( 2 <= rule[dim] && rule[dim] <= 3 )
            {
              if ( ( point_index[dim] % 2 ) == 0 )
              {
                new_point = false;
                break;
              }
            }
            else if ( 4 <= rule[dim] && rule[dim] <= 5 )
            {
              if ( 1 < order_1d[dim] && 2 * point_index[dim] - 1 == order_1d[dim] )
              {
                new_point = false;
                break;
              }
            }
            else if ( rule[dim] == 7 )
            {
            }
            else 
            {
              cerr << "\n";
              cerr << "SPARSE_GRID_MIXED_INDEX - Fatal error!\n";
              cerr << "  Unexpected value of RULE[" << dim << "] = " << rule[dim] << ".\n";
              exit ( 1 );
            }
          }
        }
//
//  New points are added to the count.
// 
        if ( new_point )
        {
          if ( point_num <= point_unique_num )
          {
            cerr << "\n";
            cerr << "SPARSE_GRID_MIXED_INDEX - Fatal error!\n";
            cerr << "  Number of points exceeds expectations.\n";
            exit ( 1 );
          }
          for ( dim = 0; dim < dim_num; dim++ )
          {
            sparse_order[dim+point_unique_num*dim_num] = order_1d[dim];
          }
          for ( dim = 0; dim < dim_num; dim++ )
          {
            sparse_index[dim+point_unique_num*dim_num] = point_index[dim];
          }
          point_unique_num = point_unique_num + 1;
        }
      }
      delete [] order_1d;

      if ( !more_grids )
      {
        break;
      }
    }
  }

  delete [] level_1d;
  delete [] point_index;

  return;
}
//****************************************************************************80

void sparse_grid_mixed_point ( int dim_num, int level_max, int rule[], 
  int point_num, double sparse_point[] )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_POINT computes the points of a sparse grid rule.
//
//  Discussion:
//
//    The sparse grid is the logical sum of low degree product rules.
//
//    Each product rule is the product of 1D factor rules.
//
//    The user specifies:
//    * the spatial dimension of the quadrature region,
//    * the level that defines the Smolyak grid.
//    * the quadrature rules.
//    * the number of points.
//
//    The user must preallocate space for the output array SPARSE_POINT.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL_MAX, controls the size of the final
//    sparse grid.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, int POINT_NUM, the number of points in the grid,
//    as determined by SPARSE_GRID_MIXED_SIZE.
//
//    Output, double SPARSE_POINT[DIM_NUM*POINT_NUM], the points.
//
{
  int dim;
  int point;
  int *sparse_index;
  int *sparse_order;
//
//  Get the index information.
//
  sparse_order = new int[dim_num*point_num];
  sparse_index = new int[dim_num*point_num];

  sparse_grid_mixed_index ( dim_num, level_max, rule, point_num, 
    sparse_order, sparse_index );
//
//  Compute the point coordinates.
//
  for ( dim = 0; dim < dim_num; dim++ )
  {
    for ( point = 0; point < point_num; point++ )
    {
      if ( rule[dim] == 1 )
      {
        sparse_point[dim+point*dim_num] = cc_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else if ( rule[dim] == 2 )
      {
        sparse_point[dim+point*dim_num] = f2_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else if ( rule[dim] == 3 )
      {
        sparse_point[dim+point*dim_num] = gp_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else if ( rule[dim] == 4 )
      {
        sparse_point[dim+point*dim_num] = gl_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else if ( rule[dim] == 5 )
      {
        sparse_point[dim+point*dim_num] = gh_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else if ( rule[dim] == 7 )
      {
        sparse_point[dim+point*dim_num] = lg_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else
      {
        cerr << "\n";
        cerr << "SPARSE_GRID_MIXED_POINT - Fatal error!\n";
        cerr << "  Unexpected value of RULE[" << dim << "] = " << rule[dim] << ".\n";
        exit ( 1 );
      }
    }
  }

  delete [] sparse_index;
  delete [] sparse_order;

  return;
}
//****************************************************************************80

int sparse_grid_mixed_size ( int dim_num, int level_max, int rule[] )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_SIZE sizes a sparse grid, discounting duplicate points.
//
//  Discussion:
//
//    The sparse grid is the logical sum of product grids with total LEVEL
//    between LEVEL_MIN and LEVEL_MAX.
//
//    In some cases, the same point may occur in different product grids
//    used to form the sparse grid.
//
//    This routine will only count one occurrence of a given point.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL_MAX, the maximum value of LEVEL.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Output, int SPARSE_GRID_MIXED_SIZE, the number of unique points.
//
{
  int d1;
  int d2;
  int dim;
  int dim1;
  int dim2;
  int h;
  int level;
  int *level_1d;
  int level_min;
  bool more_grids;
  bool more_points;
  bool new_point;
  int *order_1d;
  int *point_index;
  int point_num;
  int t;
//
//  Special case.
//
  if ( level_max == 0 )
  {
    point_num = 1;
    return point_num;
  }

  point_num = 0;

  level_1d = new int[dim_num];
  point_index = new int[dim_num];
//
//  The outer loop generates values of LEVEL.
//
  level_min = i4_max ( 0, level_max + 1 - dim_num );

  for ( level = level_min; level <= level_max; level++ )
  {
//
//  The middle loop generates a GRID, 
//  based on the next partition that adds up to LEVEL.
//
    more_grids = false;
    h = 0;
    t = 0;

    for ( ; ; )
    {
      comp_next ( level, dim_num, level_1d, &more_grids, &h, &t );

      order_1d = new int[dim_num];

      level_to_order ( dim_num, level_1d, rule, order_1d );
//
//  The inner loop generates a POINT of the GRID of the LEVEL.
//
      more_points = false;

      for ( ; ; )
      {
        vec_colex_next3 ( dim_num, order_1d, point_index, &more_points );

        if ( !more_points )
        {
          break;
        }
//
//  We examine each point and see if it is "new".
//
        new_point = true;
//
//  For a point to be old at a level equal to LEVEL_MIN:
//
//  There is a dimension D1 and another dimension D2 so that
//  the point also occurs in the grid whose value of LEVEL_1D(D1) is
//  decreased by one, and LEVEL_1D(D2) is increased by 1.
//
//  Moreover, D2 < D1 (so that we only count this once).
//
        if ( level_min == level )
        {
          for ( dim1 = 1; dim1 < dim_num; dim1++ )
          {
            d1 = -1;

            if ( 0 < level_1d[dim1] )
            {
              if ( rule[dim1] == 1 )
              {
                if ( 1 == level_1d[dim1] )
                {
                  if ( ( point_index[dim1] % 2 ) == 0 )
                  {
                    d1 = dim1;
                  }
                }
                else if ( 1 < level_1d[dim1] )
                {
                  if ( ( point_index[dim1] % 2 ) == 1 )
                  {
                    d1 = dim1;
                  }
                }
              }
              else if ( 2 <= rule[dim1] && rule[dim1] <= 3 )
              {
                if ( ( point_index[dim1] % 2 ) == 0 )
                {
                  d1 = dim1;
                }
              }
              else if ( 4 <= rule[dim1] && rule[dim1] <= 5 )
              {
                if ( 2 * point_index[dim1] - 1 == order_1d[dim1] )
                {
                  d1 = dim1;
                }
              }
              else if ( rule[dim1] == 7 )
              {
              }
              else 
              {
                cerr << "\n";
                cerr << "SPARSE_GRID_MIXED_SIZE - Fatal error!\n";
                cerr << "  Unexpected value of RULE[" << dim1 << "] = " << rule[dim1] << ".\n";
                exit ( 1 );
              }
//
//  A candidate for D1 found.  Does it have a D2?
//
              if ( d1 != -1 )
              {
                d2 = -1;
                for ( dim2 = 0; dim2 < dim1; dim2++ )
                {
                  if ( rule[dim2] == 1 )
                  {
                    d2 = dim2;
                    break;
                  }
                  else if ( 2 <= rule[dim2] && rule[dim2] <= 3 )
                  {
                    d2 = dim2;
                    break;
                  }
                  else if ( 4 <= rule[dim2] && rule[dim2] <= 5 )
                  {
                    if ( 2 * point_index[dim2] - 1 == order_1d[dim2] )
                    {
                      d2 = dim2;
                      break;
                    }
                  }
                  else if ( rule[dim2] == 7 )
                  {
                  }
                  else 
                  {
                    cerr << "\n";
                    cerr << "SPARSE_GRID_MIXED_SIZE - Fatal error!\n";
                    cerr << "  Unexpected value of RULE[" << dim2 << "] = " << rule[dim2] << ".\n";
                    exit ( 1 );
                  }
                }
//
//  If we found a D2, we know the point is OLD and are done.
//  Otherwise, consider next possible D1.
//
                if ( d2 != -1 )
                {
                  new_point = false;
                  break;
                }
                else
                {
                  d1 = -1;
                }
              }
            }
          }
        }
//
//  For a point to be old at a level greater than LEVEL_MIN:
//  If in component DIM, the point is a "nest" point that also occurs on the
//  previous 1D Level, and we're not at Level LEVEL_MIN, then for sure we've
//  already generated the point when we generated the grid:
//
//    ( Level_1d(1), ..., Level_1d(DIM-1), Level_1d(DIM)-1, Level_1d(DIM+1), ..., Level_1d(DIM_NUM) )
//
        else if ( level_min < level )
        {
          for ( dim = 0; dim < dim_num; dim++ )
          {
            if ( rule[dim] == 1 )
            {
              if ( 3 == order_1d[dim] && point_index[dim] == 2 )
              {
                new_point = false;
                break;
              }
              else if ( 3 < order_1d[dim] && ( point_index[dim] % 2 ) == 1 )
              {
                new_point = false;
                break;
              }
            }
            else if ( 2 <= rule[dim] && rule[dim] <= 3 )
            {
              if ( ( point_index[dim] % 2 ) == 0 )
              {
                new_point = false;
                break;
              }
            }
            else if ( 4 <= rule[dim] && rule[dim] <= 5 )
            {
              if ( 1 < order_1d[dim] && 2 * point_index[dim] - 1 == order_1d[dim] )
              {
                new_point = false;
                break;
              }
            }
            else if ( rule[dim] == 7 )
            {
            }
            else 
            {
              cerr << "\n";
              cerr << "SPARSE_GRID_MIXED_SIZE - Fatal error!\n";
              cerr << "  Unexpected value of RULE[" << dim << "] = " << rule[dim] << ".\n";
              exit ( 1 );
            }
          }
        }
//
//  New points are added to the count.
// 
        if ( new_point )
        {
          point_num = point_num + 1;
        }
      }

      delete [] order_1d;

      if ( !more_grids )
      {
        break;
      }
    }
  }
  delete [] level_1d;
  delete [] point_index;

  return point_num;
}
//****************************************************************************80

int sparse_grid_mixed_size_total ( int dim_num, int level_max, int rule[] )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_SIZE_TOTAL sizes a sparse grid, counting duplicate points.
//
//  Discussion:
//
//    The sparse grid is the logical sum of product grids with total LEVEL
//    between LEVEL_MIN and LEVEL_MAX.
//
//    In some cases, the same point may occur in different product grids
//    used to form the sparse grid.
//
//    This routine counts the total number of points used to construct the sparse
//    grid; if the same point occurs several times, each occurrence is added
//    to the sum.
//
//    This computation is useful in order to be able to allocate enough
//    space for the full set of points, before they are compressed by removing
//    duplicates.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL_MAX, the maximum value of LEVEL.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Output, int SPARSE_GRID_MIXED_SIZE_TOTAL, the number of points including repetitions.
//
{
  int h;
  int level;
  int *level_1d;
  int level_min;
  bool more_grids;
  int *order_1d;
  int point_total_num;
  int t;
//
//  Special case.
//
  if ( level_max == 0 )
  {
    point_total_num = 1;
    return point_total_num;
  }

  point_total_num = 0;

  level_1d = new int[dim_num];
//
//  The outer loop generates values of LEVEL.
//
  level_min = i4_max ( 0, level_max + 1 - dim_num );

  for ( level = level_min; level <= level_max; level++ )
  {
//
//  The middle loop generates a GRID, 
//  based on the next partition that adds up to LEVEL.
//
    more_grids = false;
    h = 0;
    t = 0;

    for ( ; ; )
    {
      comp_next ( level, dim_num, level_1d, &more_grids, &h, &t );

      order_1d = new int[dim_num];

      level_to_order ( dim_num, level_1d, rule, order_1d );

      point_total_num = point_total_num + i4vec_product ( dim_num, order_1d );

      delete [] order_1d;

      if ( !more_grids )
      {
        break;
      }
    }
  }
  delete [] level_1d;

  return point_total_num;
}
//****************************************************************************80

void sparse_grid_mixed_unique_index ( int dim_num, int level_max, int rule[], 
  int point_total_num, int sparse_unique_index[] )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_UNIQUE_INDEX maps nonunique points to unique points.
//
//  Discussion:
//
//    The sparse grid usually contains many points that occur in more
//    than one product grid.
//
//    When generating the point locations, it is easy to realize that a point
//    has already been generated.
//
//    But when it's time to compute the weights of the sparse grids, it is
//    necessary to handle situations in which weights corresponding to 
//    the same point generated in multiple grids must be collected together.
//
//    This routine generates ALL the points, including their multiplicities,
//    and figures out a mapping from them to the collapsed set of unique points.
//
//    This mapping can then be used during the weight calculation so that
//    a contribution to the weight gets to the right place.
//
//    The user must preallocate space for the output array SPARSE_UNIQUE_INDEX.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL_MAX, the maximum value of LEVEL.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, int POINT_TOTAL_NUM, the total number of points 
//    in the grid. 
//
//    Output, int SPARSE UNIQUE_INDEX[POINT_TOTAL_NUM], lists, 
//    for each (nonunique) point, the corresponding index of the same point in 
//    the unique listing.
//
{
  int dim;
  int h;
  int level;
  int *level_1d;
  int level_min;
  bool more_grids;
  bool more_points;
  int *order_1d;
  int point;
  int *point_index;
  int point_total_num2;
  int *sparse_index;
  int *sparse_order;
  double *sparse_point;
  int t;
  double tol;
//
//  Special case.
//
  if ( level_max == 0 )
  {
    sparse_unique_index[0] = 0;
    return;
  }

  level_1d = new int[dim_num];
  point_index = new int[dim_num];
  sparse_index = new int[dim_num*point_total_num];
  sparse_order = new int[dim_num*point_total_num];
  sparse_point = new double[dim_num*point_total_num];
//
//  Generate SPARSE_ORDER and SPARSE_INDEX arrays for the TOTAL set of points.
//
  point_total_num2 = 0;
//
//  The outer loop generates values of LEVEL.
//
  level_min = i4_max ( 0, level_max + 1 - dim_num );

  for ( level = level_min; level <= level_max; level++ )
  {
//
//  The middle loop generates a GRID, 
//  based on the next partition that adds up to LEVEL.
//
    more_grids = false;
    h = 0;
    t = 0;

    for ( ; ; )
    {
      comp_next ( level, dim_num, level_1d, &more_grids, &h, &t );

      order_1d = new int[dim_num];

      level_to_order ( dim_num, level_1d, rule, order_1d );
//
//  The inner loop generates a POINT of the GRID of the LEVEL.
//
      more_points = false;

      for ( ; ; )
      {
        vec_colex_next3 ( dim_num, order_1d, point_index, &more_points );

        if ( !more_points )
        {
          break;
        }
        for ( dim = 0; dim < dim_num; dim++ )
        {
          sparse_order[dim+point_total_num2*dim_num] = order_1d[dim];
          sparse_index[dim+point_total_num2*dim_num] = point_index[dim];
        }
        point_total_num2 = point_total_num2 + 1;
      }

      delete [] order_1d;

      if ( !more_grids )
      {
        break;
      }
    }
  }

  if ( point_total_num != point_total_num2 )
  {
    cerr << "\n";
    cerr << "SPARSE_GRID_MIXED_UNIQUE_INDEX - Fatal error!\n";
    cerr << "  POINT_TOTAL_NUM  = " << point_total_num  << "\n";
    cerr << "  POINT_TOTAL_NUM2 = " << point_total_num2 << "\n";
    exit ( 1 );
  }
//
//  Now compute the coordinates of the TOTAL set of points.
//
  for ( dim = 0; dim < dim_num; dim++ )
  {
    for ( point = 0; point < point_total_num; point++ )
    {
      if ( rule[dim] == 1 )
      {
        sparse_point[dim+point*dim_num] = cc_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else if ( rule[dim] == 2 )
      {
        sparse_point[dim+point*dim_num] = f2_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else if ( rule[dim] == 3 )
      {
        sparse_point[dim+point*dim_num] = gp_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else if ( rule[dim] == 4 )
      {
        sparse_point[dim+point*dim_num] = gl_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else if ( rule[dim] == 5 )
      {
        sparse_point[dim+point*dim_num] = gh_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else if ( rule[dim] == 7 )
      {
        sparse_point[dim+point*dim_num] = lg_point ( sparse_order[dim+point*dim_num],
          sparse_index[dim+point*dim_num] );
      }
      else
      {
        cerr << "\n";
        cerr << "SPARSE_GRID_MIXED_UNIQUE_INDEX - Fatal error!\n";
        cerr << "  Unexpected value of RULE[" << dim << "] = " << rule[dim] << ".\n";
        exit ( 1 );
      }
    }
  }
//
//  Now determine the mapping from nonunique points to unique points.
//
  tol = sqrt ( r8_epsilon ( ) );

  r8col_unique_index ( dim_num, point_total_num, sparse_point, tol, 
    sparse_unique_index );

  delete [] level_1d;
  delete [] point_index;
  delete [] sparse_index;
  delete [] sparse_order;
  delete [] sparse_point;

  return;
}
//****************************************************************************80

void sparse_grid_mixed_weight ( int dim_num, int level_max, int rule[], 
  int point_num, double sparse_weight[] )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_WEIGHT computes sparse grid weights based on a mix of 1D rules.
//
//  Discussion:
//
//    The user must preallocate space for the output array SPARSE_WEIGHT.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    02 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL_MAX, the maximum value of LEVEL.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, int POINT_NUM, the number of unique points 
//    in the grid. 
//
//    Output, double SPARSE_WEIGHT[POINT_NUM], the weights
//    associated with the sparse grid points.
//
{
  double coeff;
  double *grid_weight;
  int h;
  int level;
  int *level_1d;
  int level_min;
  bool more_grids;
  int order;
  int *order_1d;
  int order_nd;
  int point;
  int point_total;
  int point_total_num;
  int point_unique;
  int *sparse_unique_index;
  int t;

  point_total_num = sparse_grid_mixed_size_total ( dim_num, level_max, rule );

  sparse_unique_index = new int[point_total_num];

  sparse_grid_mixed_unique_index ( dim_num, level_max, rule, point_total_num, 
    sparse_unique_index );

  level_1d = new int[dim_num];

  for ( point = 0; point < point_num; point++ )
  {
    sparse_weight[point] = 0.0;
  }

  point_total = 0;

  level_min = i4_max ( 0, level_max + 1 - dim_num );

  for ( level = level_min; level <= level_max; level++ )
  {
//
//  The middle loop generates the next partition LEVEL_1D(1:DIM_NUM)
//  that adds up to LEVEL.
//
    more_grids = false;
    h = 0;
    t = 0;

    for ( ; ; )
    {
      comp_next ( level, dim_num, level_1d, &more_grids, &h, &t );
//
//  Transform each 1D level to a corresponding 1D order.
//
      order_1d = new int[dim_num];

      level_to_order ( dim_num, level_1d, rule, order_1d );
//
//  The product of the 1D orders gives us the number of points in this grid.
//
      order_nd = i4vec_product ( dim_num, order_1d );
//
//  Compute the weights for this grid.
//
//  The correct transfer of data from the product grid to the sparse grid
//  depends on the fact that the product rule weights are stored under colex
//  order of the points, and this is the same ordering implicitly used in
//  generating the SPARSE_UNIQUE_INDEX array.
//
      grid_weight = new double[order_nd];

      product_mixed_weight ( dim_num, order_1d, order_nd, rule, grid_weight );
//
//  Compute Smolyak's binomial coefficient for this grid.
//
      coeff = r8_mop ( level_max - level ) 
        * r8_choose ( dim_num - 1, level_max - level );
//
//  Add these weights to the rule.
//
      for ( order = 0; order < order_nd; order++ )
      {
        point_unique = sparse_unique_index[point_total];

        point_total = point_total + 1;

        sparse_weight[point_unique] = sparse_weight[point_unique] 
          + coeff * grid_weight[order];
      }

      delete [] grid_weight;
      delete [] order_1d;

      if ( !more_grids )
      {
        break;
      }
    }
  }

  delete [] level_1d;
  delete [] sparse_unique_index;

  return;
}
//****************************************************************************80

void sparse_grid_mixed_write ( int dim_num, int rule[], int point_num, 
  double sparse_weight[], double sparse_point[], char *file_name )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_WRITE writes a sparse grid rule to three files.
//
//  Discussion:
//
//    The files are:
//    * the "X" file stores the abscissas as a DIM_NUM x POINT_NUM list;
//    * the "W" file stores the weights as a POINT_NUM list;
//    * the "R" file stores the region, as a DIM_NUM x 2 list.
//
//    The entries in the "R" file are the two corners of the DIM_NUM dimensional
//    rectangle that constitutes the integration region.  Coordinates that
//    should be infinite are set to 1.0E+30.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    25 September 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, int POINT_NUM, the number of unique points 
//    in the grid. 
//
//    Input, double SPARSE_WEIGHT[POINT_NUM], the weights.
//
//    Input, double SPARSE_POINT[DIM_NUM*POINT_NUM], the points.
//
//    Input, char *FILE_NAME, the main part of the file name.
//    The three files will be called "FILE_NAME_x.txt", "FILE_NAME_w.txt" and
//    "FILE_NAME_r.txt".
//
{
  int dim;
  char file_name_r[80];
  char file_name_w[80];
  char file_name_x[80];
  bool header;
  double *sparse_region;

  sparse_region = new double[dim_num*2];

  for ( dim = 0; dim < dim_num; dim++ )
  {
    if ( 1 <= rule[dim] && rule[dim] <= 4 )
    {
      sparse_region[dim+0*dim_num] = -1.0;
      sparse_region[dim+1*dim_num] = +1.0;
    }
    else if ( 5 <= rule[dim] && rule[dim] <= 6 )
    {
      sparse_region[dim+0*dim_num] = - r8_huge ( );
      sparse_region[dim+1*dim_num] = + r8_huge ( );
    }
    else if ( 7 <= rule[dim] && rule[dim] <= 8 )
    {
      sparse_region[dim+0*dim_num] = 0.0;
      sparse_region[dim+1*dim_num] = r8_huge ( );
    }
    else if ( rule[dim] == 9 )
    {
      sparse_region[dim+0*dim_num] = -1.0;
      sparse_region[dim+1*dim_num] = +1.0;
    }
    else 
    {
      cerr << "\n";
      cerr << "SPARSE_GRID_MIXED_WRITE - Fatal error!\n";
      cerr << "  Unexpected value of RULE[" << dim << "] = " << rule[dim] << ".\n";
      exit ( 1 );
    }
  }

  header = false;
  strcpy ( file_name_r, file_name );
  strcat ( file_name_r, "_r.txt" );
  dtable_write ( file_name_r, dim_num, 2, sparse_region, header );

  cout << "\n";
  cout << "SPARSE_GRID_MIXED_WRITE:\n";
  cout << "  Wrote the R file = \"" << file_name_r << "\".\n";

  header = false;
  strcpy ( file_name_w, file_name );
  strcat ( file_name_w, "_w.txt" );
  dtable_write ( file_name_w, 1, point_num, sparse_weight, header );
 
  cout << "  Wrote the W file = \"" << file_name_w << "\".\n";

  header = false;
  strcpy ( file_name_x, file_name );
  strcat ( file_name_x, "_x.txt" );
  dtable_write ( file_name_x, dim_num, point_num, sparse_point, header );

  cout << "  Wrote the X file = \"" << file_name_x << "\".\n";

  delete [] sparse_region;

  return;
}
//****************************************************************************80

void timestamp ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TIMESTAMP prints the current YMDHMS date as a time stamp.
//
//  Example:
//
//    31 May 2001 09:45:54 AM
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 September 2003
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    None
//
{
# define TIME_SIZE 40

  static char time_buffer[TIME_SIZE];
  const struct tm *tm;
  size_t len;
  time_t now;

  now = time ( NULL );
  tm = localtime ( &now );

  len = strftime ( time_buffer, TIME_SIZE, "%d %B %Y %I:%M:%S %p", tm );

  cout << time_buffer << "\n";

  return;
# undef TIME_SIZE
}
//****************************************************************************80

char *timestring ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TIMESTRING returns the current YMDHMS date as a string.
//
//  Example:
//
//    May 31 2001 09:45:54 AM
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    13 June 2003
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Output, char *TIMESTRING, a string containing the current YMDHMS date.
//
{
# define TIME_SIZE 40

  const struct tm *tm;
  size_t len;
  time_t now;
  char *s;

  now = time ( NULL );
  tm = localtime ( &now );

  s = new char[TIME_SIZE];

  len = strftime ( s, TIME_SIZE, "%d %B %Y %I:%M:%S %p", tm );

  return s;
# undef TIME_SIZE
}
//****************************************************************************80

void vec_colex_next3 ( int dim_num, int base[], int a[], bool *more )

//****************************************************************************80
//
//  Purpose:
//
//    VEC_COLEX_NEXT3 generates vectors in colex order.
//
//  Discussion:
//
//    The vectors are produced in colexical order, starting with
//
//    (1,        1,        ...,1),
//    (2,        1,        ...,1),
//     ...
//    (BASE(1),  1,        ...,1)
//
//    (1,        2,        ...,1)
//    (2,        2,        ...,1)
//    ...
//    (BASE(1),  2,        ...,1)
//
//    (1,        3,        ...,1)
//    (2,        3,        ...,1)
//    ...
//    (BASE(1),  BASE(2), ...,BASE(DIM_NUM)).
//
//  Example:
//
//    DIM_NUM = 2,
//    BASE = { 3, 3 }
//
//    1   1
//    2   1
//    3   1
//    1   2
//    2   2
//    3   2
//    1   3
//    2   3
//    3   3
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 August 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int BASE[DIM_NUM], the bases to be used in each dimension.
//    In dimension I, entries will range from 1 to BASE[I].
//
//    Output, int A[DIM_NUM], the next vector.
//
//    Input/output, bool *MORE.  Set this variable false before
//    the first call.  On return, MORE is TRUE if another vector has
//    been computed.  If MORE is returned FALSE, ignore the output 
//    vector and stop calling the routine.
//
{
  int i;

  if ( !( *more ) )
  {
    for ( i = 0; i < dim_num; i++ )
    {
      a[i] = 1;
    }
    *more = true;
  }
  else
  {
    for ( i = 0; i < dim_num; i++ )
    {
      a[i] = a[i] + 1;

      if ( a[i] <= base[i] )
      {
        return;
      }
      a[i] = 1;
    }
    *more = false;
  }

  return;
}

}
