/*  _________________________________________________________________________
 *
 *  UTILIB: A utility library for developing portable C++ codes.
 *  Copyright (c) 2007, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README file in the top UTILIB directory.
 *  _________________________________________________________________________
 */

/**
 * \file Any_Shallow.h
 *
 * Defines the utilib::Any class.  
 * 
 * This is a variant of the Boost any class (along with the UTILIB
 * AnyValue and AnyReference classes).  Unlike boost::any, utilib::Any
 * implements a Smart Pointer scheme to facilitate shallow copying of
 * the Any structure and reduce the number of times a contained value
 * must be copied to 0 - 2 (depending on the use of Any::extract() or
 * Any::expose(), and if the Any was initialized with a reference to the
 * original variable).
 *
 * Eventually, this should be incorporated into the Any.h file.
 */

#ifndef utilib_Any_Shallow_h
#define utilib_Any_Shallow_h

//#include <utilib/PackObject.h>
#include <utilib/Any.h>

#include<cstddef>
#include<cassert>



namespace utilib {

namespace error {
  const int Any_Empty            = -100;
  const int Any_IncompatibleType = -101;
};

class Any;
class AnyRef;
class AnyFixedRef;


/// An exception class used by Immutable Any classes
class bad_any_typeid : public std::bad_typeid
  {
  public:
    /// Constructor
    explicit bad_any_typeid(const char* msg_) : msg(msg_) {}
    /// Return the exception description
    virtual const char * what() const throw()
      { return msg; }
    
  protected:
    /// The exception description
    const char* msg;
};



/** This class stores any object by value or reference.
 *
 *  This class was adapted from the boost::any class, for UTILIB by John
 *  Siirola.  
 *
 *  Extensions from the standard boost::any class:
 *    - stores either a type value or a reference to an external value
 *    - copying is a shallow copy operation (fast & memory efficient)
 *    - can be declared "immutable" (fixes type and memory reference)
 *
 *  The storage mechanism used by utilib::Any leverages a smart pointer
 *  to improve memory efficiency and reduce the number of times the
 *  contained data must be copied.  Passing Anys by value and by
 *  reference uses the same amount of memory.  Pass-by-value incurs
 *  only a slight additional overhead for reference counting.
 *
 *  Immutable Anys have their stored data type fixed and the location of
 *  the referred data (either external reference or contained object)
 *  fixed.  The data itself is not const -- this is similar to the
 *  difference between const int* and int* const.  Assignment and "set"
 *  operations must match the current data type (non-matching data types
 *  will throw a bad_any_typeid exception) and will perform DEEP copies
 *  into the Immutable Any's contained data.  Making a copy of the
 *  Immutable Any will perform a shallow copy (as usual), resulting in a
 *  second Immutable Any that refers to the same data.
 */ 
class Any : public PackObject
  {
  public:
    /// Empty constructor
    Any()
      : PackObject(),
        m_data(NULL)
      {}

    /// Templated constructor that takes (and copies / references) a value
    /** Note, that unlike boost::any, the utilib::Any class does not
     *  force explicit construction.  This allows for implicit type
     *  conversion (containment) in constructors and function arguments.
     *  The latter is rather nice, as a function that takes an
     *  utilib::Any can actually be passed *any* variable as an
     *  argument.
     *
     *  If you call this constructor explicitly and provide asReference
     *  = true, then the Any class will refer directly to the value you
     *  provide (no copy will occur).  You will be responsible for
     *  making sure that the original value persists longer than this
     *  Any, AND any other Any that is copied from this one.
     */
    template<typename TYPE>
    Any( const TYPE & value, 
         const bool asReference = false, 
         const bool immutable = false )
      : PackObject()
      { 
      if ( asReference )
        { m_data = new ReferenceContainer<TYPE>(value); }
      else
        { m_data = new Container<TYPE>(value); }
      m_data->immutable = immutable;
      }

    /// Copy constructor (creates a "shallow copy" of the contained cargo data)
    Any(const Any & rhs)
      : PackObject(rhs),
        m_data(rhs.m_data)
      {
      if ( m_data != NULL )
        { ++(m_data->refCount); }
      }

    /// Copy constructor to silently convert AnyRef back to generic Any
    Any(const AnyRef & rhs)
      : PackObject(reinterpret_cast<const Any&>(rhs)),
        m_data(reinterpret_cast<const Any&>(rhs).m_data)
      {
      if ( m_data != NULL )
        { ++(m_data->refCount); }
      }

    /// Copy constructor to silently convert AnyFixedRef back to generic Any
    Any(const AnyFixedRef & rhs)
      : PackObject(reinterpret_cast<const Any&>(rhs)),
        m_data(reinterpret_cast<const Any&>(rhs).m_data)
      {
      if ( m_data != NULL )
        { ++(m_data->refCount); }
      }

    /// Destructor
    virtual ~Any()
      { 
      if ( m_data != NULL )
        {
        if ( --(m_data->refCount) == 0 )
          { delete m_data; }
        }
      }

    /// Copy one Any object to another (USUALLY performs a shallow copy)
    Any& operator=(const Any & rhs) 
      {
      if ( m_data != NULL )
        {
        if ( m_data->immutable )
          {
          if (( rhs.m_data != NULL ) && ( is_type(rhs.m_data->type()) ))
            { 
            m_data->copyData(rhs.m_data); 
            return *this;
            }
          else
            { EXCEPTION_MNGR( utilib::bad_any_typeid, 
                              "Any::operator=(): assignment to immutable Any "
                              "from invalid type." ); }
          }
        else
          { 
          if ( --(m_data->refCount) == 0 )
            { delete m_data; }
          }
        }

      m_data = rhs.m_data;
      if ( m_data != NULL )
        { ++(m_data->refCount); }
      return *this;
      }

    /// Copy a derived AnyRef object into an Any (USUALLY a shallow copy)
    Any& operator=(const AnyRef & rhs)
      { return operator=(reinterpret_cast<const Any&>(rhs)); }
    /// Copy a derived AnyFixedRef object into an Any (USUALLY a shallow copy)
    Any& operator=(const AnyFixedRef & rhs)
      { return operator=(reinterpret_cast<const Any&>(rhs)); }


    /// Set the "cargo" data for this Any object (creates a copy of value)
    /** This stores the provided value into the Any object.  If
     *  asReference == true, then only a reference to the provided value
     *  is stored.  You will be responsible for making sure that the
     *  original value persists longer than this Any AND any other
     *  future copy of this Any.
     *
     *  This returns a non-constant reference to the contained data.
     *  This is one of the few cases where we provide a non-const
     *  reference to the contained data.  The intent is that you can
     *  initialize the container data and obtain a reference to the data
     *  without the need to go through type-checking.  The reference can
     *  then be used to populate the data structure.  This avoids the
     *  case where you create the fully-populated data structure, then
     *  must copy the whole thing when you call Any::set() (i.e., this
     *  allows you to prevent a copy).
     */
    template<typename TYPE>
    TYPE& set( const TYPE & value, 
               bool asReference = false, 
               bool immutable = false )
      {
      if ( m_data != NULL )
        { 
        if ( m_data->immutable )
          {
          if ( immutable )
            { EXCEPTION_MNGR( utilib::bad_any_typeid, 
                              "Any::set<>(): assigning immutable to an "
                              "already immutable Any." ); }
          if ( asReference )
            { EXCEPTION_MNGR( utilib::bad_any_typeid, 
                              "Any::set<>(): assigning reference to an "
                              "immutable Any." ); }

          if ( is_type(typeid(TYPE)) )
            { 
            if ( m_data->isReference() )
              {
              TYPE& tmp = const_cast<TYPE&>
                  (static_cast<ReferenceContainer<TYPE>*>(m_data)->data);
              tmp = value; 
              return tmp;
              }
            else
              { 
              TYPE& tmp = static_cast<Container<TYPE>*>(m_data)->data;
              tmp = value;
              return tmp;
              }
            }
          else
            { EXCEPTION_MNGR( utilib::bad_any_typeid, 
                              "Any::set(value): assignment to immutable Any "
                              "from invalid type." ); }
          }

        if ( --(m_data->refCount) == 0 )
          { delete m_data; }
        }

      if ( asReference )
        { 
        m_data = new ReferenceContainer<TYPE>(value);
        m_data->immutable = immutable;
        return const_cast<TYPE&>
            (static_cast<ReferenceContainer<TYPE>*>(m_data)->data);
        }
      else
        { 
        m_data = new Container<TYPE>(value); 
        m_data->immutable = immutable;
        return static_cast<Container<TYPE>*>(m_data)->data;
        }
      }



    /** \brief Convenience function for creating a new object within an Any
     *  without calling the object's copy-constructor. 
     *
     *  This method can be used to avoid a call to the object Copy Constructor,
     *  and thus is a slightly more efficient form of:
     *  <DIR><TT>
     *     TYPE value;
     *     Any myAny(value);
     *  </TT></DIR>
     *
     *  As with Any::set<TYPE>(const TYPE&, bool, bool), this returns a
     *  non-constant reference to the contained data.
     */
    template<typename TYPE>
    TYPE& set()
      {
      if ( m_data != NULL )
        { 
        if ( m_data->immutable )
          {
          if ( is_type(typeid(TYPE)) )
            { 
            if ( m_data->isReference() )
              { 
              TYPE& tmp = const_cast<TYPE&>
                  ( static_cast<ReferenceContainer<TYPE>*>(m_data)->data );
              tmp = TYPE();
              return tmp;
              }
            else
              { 
              TYPE& tmp = static_cast<Container<TYPE>*>(m_data)->data;
              tmp = TYPE(); 
              return tmp;
              }
            }
          else
            { EXCEPTION_MNGR( utilib::bad_any_typeid, 
                              "Any::set<>(): assignment to immutable Any "
                              "from invalid type." ); }
          }

        if ( --(m_data->refCount) == 0 )
          { delete m_data; }
        }

      m_data = new Container<TYPE>();
      return static_cast<Container<TYPE>*>(m_data)->data;
      }


    /// Shortcut using assignment in lieu of Any::set()
    /** I'm not sure if I like supporting this syntax.  It has a
     *  different behavior when called with an Any RHS (creates a 
     *  duplicate reference to the RHS Any) than for all other types 
     *  (causes this Any to "contain" the RHS as the data cargo).
     *
     *  It is included solely for compatibility / consistency with the
     *  boost::any class -- I would prefer only supporting Any::set()
     */
    template<typename TYPE>
    inline Any& operator=(const TYPE & value)
      { 
      set(value); 
      return *this;
      }


    /// Shortcut for reference assignment in lieu of Any::set(value, true)
    /** This is an extension to the standard assignment operator for
     *  assigning a reference to the RHS value into the Any (instead of
     *  making a deep copy of the RHS value).
     *
     *  I'm not sure if I like supporting this syntax.  It is a
     *  non-standard overload of a standard assignment operator.  
     */
    template<typename TYPE>
    inline Any& operator&=(const TYPE & value)
      { 
      set(value, true); 
      return *this;
      }


    /// Extract the "cargo" data from this Any object (return <0 on error)
    /** This will extract the stored value out of the Any object into
     *  the value reference, provided that the two types match EXACTLY.
     *
     *  This compares the mangled type name in addition to the type_info
     *  reference because each time a template is emitted into a
     *  compilation unit, it gets its own type_info structure.  If
     *  libraries are dynamically linked, this can cause the same type
     *  to return different type_info structures depending on which
     *  library instantiated it (so a direct reference comparison would
     *  fail).  The mangled type name, however, will be the same.  Note
     *  that this is only a problem with dynamic-link libraries.  Most
     *  static linkers will remove the redundant template emission at
     *  link time.
     *
     *  Returns 0 on success, <0 on error (value left unchanged).  If
     *  Any::throwCastExceptions() has been set to true, error conditions
     *  will also throw a bad_any_cast exception.
     */
    template<typename TYPE>
    int extract(TYPE & value) const
      { 
      if ( m_data == NULL )
        { 
        if ( throwCastExceptions() )
          { EXCEPTION_MNGR(utilib::bad_any_cast, 
                           "Any::extract() - NULL data"); }
        return error::Any_Empty; 
        }

      if ( is_type(typeid(TYPE)) )
        {
        if ( m_data->isReference() )
          { value = static_cast<ReferenceContainer<TYPE>*>(m_data)->data; }
        else
          { value = static_cast<Container<TYPE>*>(m_data)->data; }
        return 0;
        }

      if ( throwCastExceptions() )
        {
        ostringstream buf;
        buf << "Any::extract() - failed conversion from '" 
            << m_data->type().name() << "' to '" << typeid(TYPE).name() << "'";
        EXCEPTION_MNGR( utilib::bad_any_cast, buf.str() );
        }

      return error::Any_IncompatibleType; 
      }


    /// Exposes a reference to the "cargo" data from this Any object
    /** This will expose a reference to the stored value in the Any object,
     *  provided that the two types match EXACTLY.  If they do not
     *  match, it will throw an exception (bad_any_cast).
     *
     *  Note: you must explicitly specify the destination type when you
     *  call this method (e.g. <tt>Any foo(5); std::cerr <<
     *  foo.expose<int>();</tt>).
     *
     *  WARNING: Abusing this function may be hazardous to your health.
     *  Any is a reference class (a la C#), so if you store a new value
     *  or edit the value exposed by expose(), that change will be
     *  apparent to ANYONE who still maintains a reference to this Any
     *  object.  I try and save you by storing the data as 'const', but
     *  if you attempt any funny-business with const_cast, well,
     *  consider yourself warned.
     *
     *  See Any::extract() for details on type resolution.
     */
    template<typename TYPE>
    const TYPE& expose() const
      {
      if ( m_data == NULL )
        { EXCEPTION_MNGR(utilib::bad_any_cast, "Any::expose() - NULL data"); }

      if ( ! is_type(typeid(TYPE)) )
        {
        ostringstream buf;
        buf << "Any::expose() - failed conversion from '" 
            << m_data->type().name() << "' to '" << typeid(TYPE).name() << "'";
        EXCEPTION_MNGR( utilib::bad_any_cast, buf.str() );
        }

      if ( m_data->isReference() )
        { return static_cast<ReferenceContainer<TYPE>*>(m_data)->data; }
      else
        { return static_cast<Container<TYPE>*>(m_data)->data; }
      }


    /// Verifies the type of the stored value.
    /** This will return true if the stored type (EXACTLY) matches the
     *  calling type (i.e. extract / expose would succeed).  
     *
     *  See Any::extract() for details on type resolution.
     */
    bool is_type(const std::type_info &targetType) const
      {
      if ( m_data == NULL )
        { return false; }

      // 1) the quick test: do the type_info references match?
      const std::type_info &myType = m_data->type();
      if ( myType == targetType )
        { return true; }
      
      // 2) the not-so-quick test: do the mangled names match?
      return ( 0 == strcmp(myType.name(), targetType.name()) );
      }


    /// Clear the "cargo" data for this Any object (empty() will become true)
    void clear()
      {
      if ( m_data != NULL )
        { 
        if ( --(m_data->refCount) == 0 )
          { delete m_data; }
        m_data = NULL;
        }
      }

    /// Returns true if the rhs Any refers to the same Container
    bool reference_same_data(const Any& rhs) const
      { return (( m_data != NULL ) && ( m_data == rhs.m_data )); }

    /// Returns true if the data stored in the rhs Any is equal to this data 
    bool operator==(const Any& rhs) const
      {
      if ( m_data == NULL )
        { return rhs.m_data == NULL; }
      if ( ! rhs.is_type(m_data->type()) )
        { return false; }
      return m_data->isEqual(rhs.m_data);
      }
    
    /// Returns true if the data stored in the rhs Any is equal to this data 
    bool operator<(const Any& rhs) const
      {
      if ( m_data == NULL )
        { return rhs.m_data != NULL; }
      if ( rhs.m_data == NULL )
        { return false; }
      if ( ! rhs.is_type(m_data->type()) )
        { return m_data->type().before(rhs.m_data->type()); }
      else
        { return m_data->isLessThan(rhs.m_data); }
      }
    
    /// Returns true if the object value has not been set
    inline bool empty() const
      { return m_data == NULL; }

    /// Returns if the stored type is actually a reference
    inline bool is_reference() const
      { return m_data ? m_data->isReference() : false; }

    /// Remove reference/immutable flags (performs a copy if necessary)
    void to_basic_any()
      { 
      if ( m_data == NULL )
        { return; }
      if (( m_data->isReference() ) || m_data->immutable )
        {
        ContainerBase *tmp = m_data;
        m_data = m_data->valueData();
        if ( --(tmp->refCount) == 0 )
          { delete tmp; }
        }
      }

    /// Returns if this Any is immutable (data is fixed in type and address)
    /** An "immutable" Any is an Any where the data type and the memory
     *  address (either reference or value location) is fixed.
     *
     *  Any.set() on an immutable Any will reset the value of the stored
     *  data if the data types match and throw an exception if they do
     *  not.  
     *
     *  Assigning an immutable Any to another Any will perform a shallow
     *  copy as usual (the new Any will also be immutable).  However,
     *  assigning another Any to the immutable Any will perform a DEEP
     *  copy of the contained data if the data types match and throw an
     *  exception if they do not.
     */
    inline bool is_immutable() const
      { return m_data ? m_data->immutable : false; }

    /// Returns the number of other Any instances that refer to this data
    inline unsigned int anyCount() const
      { return m_data ? (m_data->refCount - 1) : 0; }

    /// Returns the type information of the object contained in this Any object
    inline const std::type_info & type() const
      { return m_data ? m_data->type() : typeid(void); }

    /// Indicates whether extract() should generate exceptions for bad casts
    /** This both indicates whether Any::extract() should throw
     *  exceptions for invalid casts, and allows the client to set the
     *  result of future throwCastExceptions() calls [as the bool value is
     *  returned by non-constant reference).
     *
     *  By default, throwCastExceptions() == true and Any::extract()
     *  will generate cast exceptions.
     */
    static bool& throwCastExceptions() 
      { 
      static bool value = true; 
      return value;
      }

    /// Describe the contents of this Any
    void describe(std::ostream& os, std::string name = "(Any)") const
      {
      os << name << ": ";
      if ( empty() )
        { os << "[empty]"; }
      else
        { 
        os << m_data->type().name();
        if ( is_immutable() )
          { os << " [immutable]"; }
        if ( is_reference() )
          { os << " [reference]"; }
        os << std::endl;
        }
      }



    /// Write the referenced object
    virtual void write(std::ostream& ostr) const
      {
      static_cast<void>(ostr);
      assert(0&&"Please implement Any::write(std::ostream&)");
      // Notes: I would probably implement the read/write functionality as: 
      //
      // A) char indicating if the contained type is:
      //    0 => m_data == NULL
      //    1 => registered with the TypeRegistry system
      //    2 => not registered with the TypeRegistry.
      // B) switch(A)
      //    case 1:
      //      - LONG indicating TypeRegistry type
      //      - the results of m_data->write()
      //    case 2:
      //      - CHAR* (NULL-terminated) mangled type name (from typeid())
      //      - the results of m_data->write()
      }

    /// Read the referenced object
    virtual void read(std::istream& istr)
      {
      static_cast<void>(istr);
      assert(0&&"Please implement Any::read(std::istream&)");
      }

    /// Write the referenced object to a PackBuffer
    virtual void write(PackBuffer& ostr) const
      {
      static_cast<void>(ostr);
      assert(0&&"Please implement Any::write(PackBuffer&)");
      }

    /// Read the referenced object from an UnPackBuffer
    virtual void read(UnPackBuffer& istr)
      {
      static_cast<void>(istr);
      assert(0&&"Please implement Any::read(UnPackBuffer&)");
      }


  private:
    struct ContainerBase
      {
        ContainerBase()
          : refCount(1),
            immutable(false)
          {}
        virtual ~ContainerBase()
          {}

        /// Return the type of the stored object
        virtual const std::type_info& type() const = 0;
        /// Return whether the derived container holds a reference
        virtual bool isReference() const = 0;
        /// Return whether the rhs container data is equal to this data
        virtual bool isEqual(const ContainerBase* rhs) const = 0;
        /// Return whether this data is less than the rhs container data
        virtual bool isLessThan(const ContainerBase* rhs) const = 0;
        /// Write the referenced object
        virtual void write(std::ostream& ostr) const = 0;
        /// Write the referenced object to a PackBuffer
        virtual void write(PackBuffer& ostr) const = 0;
        /// Copy the data from another (hopefully compatible) ContainerBase*
        virtual void copyData(ContainerBase *rhs) = 0;
        /// Create a new ReferenceContainer to this container's data 
        virtual ContainerBase* referenceData() const = 0;
        /// Create a new Container containing this container's data 
        virtual ContainerBase* valueData() const = 0;

        unsigned int   refCount;
        bool           immutable;
      };

    template<typename TYPE>
    struct ReferenceContainer : public ContainerBase
      {
      public:
        ReferenceContainer(const TYPE & value)
          : ContainerBase(),
            data(value)
          {}
        virtual ~ReferenceContainer()
          {}

        /// Return the type of the stored object
        virtual const std::type_info& type() const 
          { return typeid(TYPE); }
        /// Return whether the derived container holds a reference
        virtual bool isReference() const 
          { return true; }
        /// Return whether the rhs container data is equal to this data
        virtual bool isEqual(const ContainerBase* rhs) const
          { 
          return data == ( rhs->isReference() 
                   ? static_cast<const ReferenceContainer<TYPE>*>(rhs)->data
                   : static_cast<const Container<TYPE>*>(rhs)->data );
          }
        /// Return whether this data is less than the rhs container data
        virtual bool isLessThan(const ContainerBase* rhs) const
          { 
          return data < ( rhs->isReference() 
                   ? static_cast<const ReferenceContainer<TYPE>*>(rhs)->data
                   : static_cast<const Container<TYPE>*>(rhs)->data );
          }
        /// Write the referenced object
        virtual void write(std::ostream& ostr) const
          { static_cast<void>(ostr); }
        /// Write the referenced object to a PackBuffer
        virtual void write(PackBuffer& ostr) const 
          { static_cast<void>(ostr); }
        /// Copy the data from another (hopefully compatible) ContainerBase*
        virtual void copyData(ContainerBase *rhs);
        /// Create a new ReferenceContainer to this container's data 
        virtual ContainerBase* referenceData() const
          { return new ReferenceContainer<TYPE>(data); }
        /// Create a new Container containing this container's data 
        virtual ContainerBase* valueData() const;

        const TYPE & data;
      };

    template<typename TYPE>
    struct Container : public ContainerBase
      {
      public:
        /// default constructor only used by Any::set<TYPE>()
        Container()
          : ContainerBase(),
            data()
          {}
        /// Standard constructor used 99\% of the time
        Container(const TYPE & value)
          : ContainerBase(),
            data(value)
          {}
        virtual ~Container()
          {}

        /// Return the type of the stored object
        virtual const std::type_info& type() const 
          { return typeid(TYPE); }
        /// Return whether the derived container holds a reference
        virtual bool isReference() const 
          { return false; }
        /// Return whether the rhs container data is equal to this data
        virtual bool isEqual(const ContainerBase* rhs) const
          { 
          return data == ( rhs->isReference() 
                   ? static_cast<const ReferenceContainer<TYPE>*>(rhs)->data
                   : static_cast<const Container<TYPE>*>(rhs)->data );
          }
        /// Return whether this data is less than the rhs container data
        virtual bool isLessThan(const ContainerBase* rhs) const
          { 
          return data < ( rhs->isReference() 
                   ? static_cast<const ReferenceContainer<TYPE>*>(rhs)->data
                   : static_cast<const Container<TYPE>*>(rhs)->data );
          }
        /// Write the referenced object
        virtual void write(std::ostream& ostr) const
          { static_cast<void>(ostr); }
        /// Write the referenced object to a PackBuffer
        virtual void write(PackBuffer& ostr) const 
          { static_cast<void>(ostr); }
        /// Copy the data from another (hopefully compatible) ContainerBase*
        virtual void copyData(ContainerBase *rhs);
        /// Create a new ReferenceContainer to this container's data 
        virtual ContainerBase* referenceData() const
          { return new ReferenceContainer<TYPE>(data); }
        /// Create a new Container containing this container's data 
        virtual ContainerBase* valueData() const
          { return new Container<TYPE>(data); }

        TYPE data;
      };

  protected:
    ContainerBase *m_data;
  };


template<typename TYPE>
Any::ContainerBase* Any::ReferenceContainer<TYPE>::valueData() const
  { return new Any::Container<TYPE>(data); }

template<typename TYPE>
void Any::Container<TYPE>::copyData(Any::ContainerBase *rhs)
  {
  if ( rhs->isReference() )
    { data = static_cast<Any::ReferenceContainer<TYPE>*>(rhs)->data; }
  else
    { data = static_cast<Any::Container<TYPE>*>(rhs)->data; }
  }

template<typename TYPE>
void Any::ReferenceContainer<TYPE>::copyData(Any::ContainerBase *rhs)
  {
  TYPE& tmp = const_cast<TYPE&>(data);
  if ( rhs->isReference() )
    { tmp = static_cast<Any::ReferenceContainer<TYPE>*>(rhs)->data; }
  else
    { tmp = static_cast<Any::Container<TYPE>*>(rhs)->data; }
  }


/// Cast an Any reference into its value
template <typename TYPE>
inline TYPE 
anyval_cast(const Any & operand)
  { return operand.template expose<TYPE>(); }


/// Cast a const Any and expose a const reference to its value
template <typename TYPE>
inline const TYPE & 
anyref_cast(const Any & operand)
  { return operand.template expose<TYPE>(); }


/** \brief Cast an Any and expose a non-const reference to its value
 *  (WARNING: may cause unintended side effects!  See Any::expose())
 */
template <typename TYPE>
inline TYPE & 
anyref_cast(Any & operand)
  { return const_cast<TYPE&>(operand.template expose<TYPE>()); }


/// A shadow class to cause implicit casts to create reference (not value) Anys
/** This is a "shadow class" (no new members, no new methods) that
 *  causes the compiler to use a different rule for implicitly casting
 *  variables into Anys.  

 *  Normally, implicitly casting a non-Any into an Any causes the source
 *  varialbe to be <i>copied</i> into the new Any (that is, it returns a
 *  "value" Any that is independent of the original data).  Implicit
 *  casts to AnyRef variables will create a reference Any that refers
 *  directly back to the source variable.
 *
 *  This class is designed primarily to be used in function definitions
 *  to control the nature of the Anys that are passed into the function.
 *  AnyRef can be implicitly converted back to a general Any.
 */
class AnyRef : public Any
  {
  public:
    /// Empty constructor
    AnyRef()
      : Any()
      {}

    /// Standard copy constructor
    AnyRef(const AnyRef & rhs)
      : Any(reinterpret_cast<const Any&>(rhs))
      {}

    /// Templated constructor that takes (and references) a value
    /** Note, that unlike boost::any, the utilib::Any class does not
     *  force explicit construction.  This allows for implicit type
     *  conversion (containment) in constructors and function arguments.
     *  The latter is rather nice, as a function that takes an
     *  utilib::Any can actually be passed *any* variable as an
     *  argument.
     */
    template<typename TYPE>
    AnyRef( const TYPE & value, 
            const bool immutable = false )
      : Any(value, true, immutable)
      {}

    /// Allow implicit conversion of Anys to AnyRefs
    AnyRef(const Any & rhs)
      : Any(rhs)
      {}

    /* NB: we *MUST* use reinterpret_cast here as using static_cast will
     * cause a NEW Any to be constructed using Any's templated implicit
     * cast constructor (because AnyFixedRef has not yet been defined as
     * derived from Any).
     */
    /// Allow implicit conversion of AnyFixedRefs to AnyRefs
    AnyRef(const AnyFixedRef & rhs)
      : Any(reinterpret_cast<const Any&>(rhs))
      {}
  };


/** \brief A shadow class to cause implicit casts to create immutable reference
 *  (not value) Anys.
 *
 *  This is a "shadow class" (no new members, no new methods) that
 *  causes the compiler to use a different rule for implicitly casting
 *  variables into Anys.

 *  Normally, implicitly casting a non-Any into an Any causes the source
 *  varialbe to be <i>copied</i> into the new Any (that is, it returns a
 *  "value" Any that is independent of the original data).  Implicit
 *  casts to AnyFixedRef variables will create an immutable reference
 *  Any that refers directly back to the source variable.
 *
 *  This class is designed primarily to be used in function definitions
 *  to control the nature of the Anys that are passed into the function.
 *  AnyFixedRef can be implicitly converted back to a general Any.
 */
class AnyFixedRef : public Any
  {
  public:
    /// Empty constructor
    AnyFixedRef()
      : Any()
      {}

    /// Standard copy constructor
    AnyFixedRef(const AnyFixedRef & rhs)
      : Any(reinterpret_cast<const Any&>(rhs))
      {}

    /// Templated constructor that takes (and references) a value
    /** Note, that unlike boost::any, the utilib::Any class does not
     *  force explicit construction.  This allows for implicit type
     *  conversion (containment) in constructors and function arguments.
     *  The latter is rather nice, as a function that takes an
     *  utilib::Any can actually be passed *any* variable as an
     *  argument.
     */
    template<typename TYPE>
    AnyFixedRef( const TYPE & value )
      : Any(value, true, true)
      {}

    /// Allow implicit conversion of Anys to AnyFixedRefs
    AnyFixedRef(const Any & rhs)
      : Any(rhs)
      {
      if (( m_data == NULL ) || m_data->immutable )
        { return; }

      // As we are copying the data from another Any, this will not
      // result in the refCount going to 0
      --(m_data->refCount);
      
      m_data = m_data->referenceData();
      m_data->immutable = true;
      }

    /// Allow implicit conversion of AnyRefs to AnyFixedRefs
    AnyFixedRef(const AnyRef & rhs)
      : Any(reinterpret_cast<const Any&>(rhs))
      {
      if (( m_data == NULL ) || m_data->immutable )
        { return; }

      // As we are copying the data from another Any, this will not
      // result in the refCount going to 0
      --(m_data->refCount);
      
      m_data = m_data->referenceData();
      m_data->immutable = true;
      }
  };


/* Not sure why AnyReference implemented special handlers for bool
   values, but just in case it really was necessary, here is an
   efficient re-implementation
*/
/*
template<>
void Any::Container<bool>::write(std::ostream& ostr) const
  { ostr << (data ? "true" : "false"); }

template<>
void Any::Container<bool>::read(std::istream& istr) 
  {
  std::string tmp;
  istr >> tmp;
  if ( tmp.empty() )
    {
    EXCEPTION_MNGR( runtime_error,
                    "Any::Container<bool>::read() - empty data string" );
    return;
    }
  
  char firstChar = toupper(tmp[0]);
  if (( firstChar=='F' ) || ( firstChar=='0' ))
    {
    int len = tmp.size();
    if ( len == 1 )
      { 
      data = false;
      return;
      }
    else if ( len == 5 )
      {
      if (( toupper(tmp[0]) == 'F' ) && ( toupper(tmp[1]) == 'A' ) &&
          ( toupper(tmp[2]) == 'L' ) && ( toupper(tmp[3]) == 'S' ) &&
          ( toupper(tmp[4]) == 'E' ))
        {
        data = false;
        return;
        }
      }
    }
  else if (( firstChar=='T' ) || ( firstChar=='1' ))
    {
    int len = tmp.size();
    if ( len == 1 )
      { 
      data = true;
      return;
      }
    else if ( len == 4 )
      {
      if (( toupper(tmp[0]) == 'T' ) && ( toupper(tmp[1]) == 'R' ) &&
          ( toupper(tmp[2]) == 'U' ) && ( toupper(tmp[3]) == 'E' ))
        {
        data = true;
        return;
        }
      }
    }

  EXCEPTION_MNGR( runtime_error,
                  "Any::Container<bool>::read() - nonboolean value for "
                  "read method: " << tmp );
  }
*/
}

#endif // utilib_Any_Shallow_h
