/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2007, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

//
// Coliny_MultiStatePS.cpp
//
// A "multi-state" pattern search algorithm built on COLIN's AsyncEvaluator.
//

#include <acro_config.h>
#include <utilib/_math.h>
#include <utilib/stl_auxillary.h>
#include <coliny/MultiStatePS.h>

#include <cmath>

#define IMPROVING_SOLVER 1
#define SECONDARY_SOLVER 2

using std::runtime_error;
using std::cerr;
using std::endl;


namespace coliny {


MultiStatePS::MultiStatePS()
  : async_evaluator_t(&problem),
    m_contraction_factor(0.5),
    m_delta_init(1),
    m_delta_thresh(1e-5),
    m_expansion_factor(2),
    m_sufficient_decrease_coef(0.01),
    m_max_success(5)
  { 
  opt_name = "MultiState Pattern Search";
  constraint_penalty=1.0;
  ParameterSet::set_parameter_default("constraint_penalty","1.0");

  ParameterSet::create_categorized_parameter
      ("initial_step", m_delta_init, "<double>", "1.0",
       "Initial step length",
       "Step Length Control");
  ParameterSet::alias_parameter("initial_step","initial_stepsize");
  ParameterSet::alias_parameter("initial_step","initial_steplength");
  
  ParameterSet::create_categorized_parameter
      ("step_tolerance", m_delta_thresh, "<double>", "1e-5",
       "Convergence tolerance step length",
       "Termination");

  ParameterSet::create_categorized_parameter
      ("contraction_factor", m_contraction_factor, "<double>", "0.5",
       "Contraction factor",
       "Step Length Control");

  ParameterSet::create_categorized_parameter
      ("max_success", m_max_success, "<int>", "5",
       "Number of successful iterations before step length is expanded",
       "Step Length Control");

  ParameterSet::create_categorized_parameter
      ("step_scales", m_sigma, "<Array<double>>", "All 1.0",
       "The scale factors for each dimension",
       "Step Length Control");

  ParameterSet::create_categorized_parameter
      ("alpha", m_sufficient_decrease_coef, "<double>", "0.01",
       "Sufficient decrease parameter",
       "Step Length Control");

  ParameterSet::create_categorized_parameter
      ("expansion_factor", m_expansion_factor, "<double>", "2.0",
       "Expansion factor",
       "Step Length Control");

  }


void MultiStatePS::reset()
  {
  if (!problem) 
    { return; }

  colin::StdOptSolver<BasicArray<double>,response_t>::reset();

  unsigned int nvars = problem.num_real_params();
  if ( nvars == 0 )
    { return; }

  m_sigma.resize(nvars);
  m_sigma << 1.0;

  async_evaluator_t::reset();
  colin::AppResponseAnalysis::initialize(problem.numNonlinearIneqConstraints(),
                                         constraint_tolerance);
  }



void MultiStatePS::minimize()
  {
  if ( DEBUG_MULTISTATE_PS > 1 )
    { cerr << "entering minimize(); " << endl; }

  //
  // Misc initialization of the optimizer
  //
  opt_init();
  if ( DEBUG_MULTISTATE_PS > 1 )
    { cerr << "minimize(): opt_init done" << endl; }

  if ( ! this->initial_point_flag )
    {
    EXCEPTION_MNGR
        (runtime_error,"MultiStatePS::minimize - no initial point specified.");
    }

  if ( problem.num_real_params() != best().point.size() )
    {
    EXCEPTION_MNGR
        ( runtime_error, "MultiStatePS::minimize - problem has " <<
          problem.num_real_params() << " real params, but initial point has "
          << best().point.size() );
    }
  if ( best().point.size() == 0 ) 
    {
    best().termination_info = "No-Real-Params";
    return;
    }

  if (( max_neval == 0 ) && ( max_neval_curr == 0 ) && ( max_iters == 0 ) && 
      ( std::fabs(max_time) < 0.001 ))
    {
    cerr << "(INFO): none of max_time, max_iters, max_neval, or "
         << "max_neval_curr set." << endl
         << "        MultiStatePS currently has no termination criteria; "
         << endl
         << "        setting max_neval = 100,000." << endl;
    max_neval = 100000;
    }

  if ( DEBUG_MULTISTATE_PS )
    {
    write(cerr);
    cerr << "----------------" << endl;
    }

  /*
  //
  //
  unsigned int num_iters;
  if ( max_iters <= 0 )
    { num_iters = MAXINT; }
  else
    { num_iters = curr_iter + max_iters; }

  //
  // This is a hack to make the default number of iterations be small.
  // In general, this will not be necessary...
  //
  num_iters = std::min(static_cast<unsigned int>(10), num_iters);
  */

  if ( ! problem.test_feasibility(best().point) )
    {
    EXCEPTION_MNGR
        ( runtime_error,
          "MultiStatePS::minimize - initial point is not bound-feasible" );
    }

  std::list<PSState>::iterator newState;
  response_t response;
  response_t baseResponse;
  responseSet_t responseSet;

  if ( DEBUG_MULTISTATE_PS > 1 )
    {
    cerr << "minimize(): evaluating initial point" << endl; 
    cerr << "  initial point: " << best().point << endl;
    }

  // NB: the point won't be in the cache, but this will make sure it is added
  responseSet_t start = checkCache(best().point);
  perform_evaluation(start);

  start.stat(response, responseSet_t::NEWEST);
  best().response << response;
  best().value() = response.augmented_function_value();
  best().constraint_violation = response.l2_constraint_violation();

  if ( DEBUG_MULTISTATE_PS > 1 )
    { cerr << "minimize(): exploring initial point" << endl; }
  newState = explore(start, m_delta_init, true);
  newState->improving_count = 0;

  // set the target allocation
  std::map<solverID_t, double> targetCompAlloc;
  targetCompAlloc[IMPROVING_SOLVER] = 1.0;
  targetCompAlloc[SECONDARY_SOLVER] = 0.0;
  set_target_allocation(targetCompAlloc);

  if ( DEBUG_MULTISTATE_PS > 1 )
    { cerr << "entering minimize()::loop; " << endl; }
  std::map<solverID_t, evalID_t> result;
  std::map<solverID_t, evalID_t>::iterator r_it;
  std::map<solverID_t, evalID_t>::iterator r_itEnd;
  while ( return_evaluation(responseSet, result) == 0 )
    {
    // I don't really have a concept of an "iteration", but to make this
    // somewhat consistent with other Colin solvers, I will make every
    // processed evaluation an iteration.
    ++curr_iter;

    if ( DEBUG_MULTISTATE_PS == 1 )
      { 
      cerr << "minimize(): best = " << best().value() 
           << ", total eval = " << neval() << endl;
      }
    else if ( DEBUG_MULTISTATE_PS > 1 )
      { debug_io(cerr); }

    if ( check_convergence(best().value()) )
      { break; }

    //
    // TODO: check feasibility here...
    //

    // pull the newest response from the ResponseSet
    responseSet.stat(response, responseSet_t::NEWEST);

    // process the 'callbacks'
    r_itEnd = result.end();
    for ( r_it = result.begin(); r_it != r_itEnd; ++r_it )
      {
      std::map<evalID_t, std::list<std::list<PSState>::iterator> >::iterator 
          eval_it = m_pending[r_it->first].find(r_it->second);
      if ( eval_it == m_pending[r_it->first].end() )
        {
        EXCEPTION_MNGR
            ( runtime_error,
              "MultiStatePS::minimize - invalid evalID returned" );
        }


      while ( ! eval_it->second.empty() )
        {
        std::list<PSState>::iterator s_it = eval_it->second.front();
        eval_it->second.pop_front();

        --(s_it->pending);
        s_it->base.stat(baseResponse, responseSet_t::NEWEST);

        if ( DEBUG_MULTISTATE_PS )
          {
          cerr << "minimize(): processing evalid " << eval_it->first
               << ", responseSet id = " << s_it->base.id()
               << ", state = " << s_it->id
               << ", pending = " << s_it->pending << endl;
          }
        
        if ( response.augmented_function_value() < 
             baseResponse.augmented_function_value() )
          {
          // found an improving point 
          s_it->improving_found = true;
          
          // should we expand the step size?
          if ( s_it->improving_count >= m_max_success )
            { 
            if ( DEBUG_MULTISTATE_PS )
              {
              cerr << "minimize(): expanding around responseSet "
                   << s_it->base.id() << ", new step = " << 
                  s_it->step*m_expansion_factor << endl;
              }

            newState = explore(responseSet,s_it->step*m_expansion_factor,true);
            newState->improving_count = 0;
            }
          else
            { 
            newState = explore(responseSet, s_it->step, true); 
            ++(newState->improving_count);
            }
          
          if ( best().value() > response.augmented_function_value() )
            {
            best().point << responseSet.point();
            best().response << response;
            best().value() = response.augmented_function_value();
            best().constraint_violation = response.l2_constraint_violation();
            
            DEBUGPR(100, cerr << "New best point: id = " << responseSet.id()
                    << ", value = " << best().value() << endl);
            }
          }
        else
          {
          // oh well...
          newState = explore(responseSet, s_it->step, false);
          newState->improving_count = 0;
          
          // check if we should contract
          if (( s_it->pending == 0 ) && ( ! s_it->improving_found ))
            { 
            double newStep = m_contraction_factor * s_it->step;
            if ( newStep >= m_delta_thresh )
              { 
              if ( DEBUG_MULTISTATE_PS )
                {
                cerr << "minimize(): contracting around responseSet "
                     << s_it->base.id() << ", new step = " << newStep << endl;
                }

              newState = explore(s_it->base, newStep, true); 
              newState->improving_count = 0;
              }
            else
              {
              if ( DEBUG_MULTISTATE_PS )
                {
                cerr << "minimize(): minimum step size reached." << endl;

                // re-prioritize the search to balance with the
                // secondary exploration
                targetCompAlloc[IMPROVING_SOLVER] = 0.5;
                targetCompAlloc[SECONDARY_SOLVER] = 0.5;
                set_target_allocation(targetCompAlloc);
                }

              DEBUGPR(100, cerr << "Minimum delta reached.  No further "
                      "contraction allowed for point " << s_it->base.id() 
                      << ", value = " << 
                      baseResponse.augmented_function_value() << endl);
              }
            }
          }
        }

      m_pending[r_it->first].erase(eval_it);
      }
    }

  if ( DEBUG_MULTISTATE_PS )
    { cerr << "minimize(): complete" << endl; }
  debug_io(cerr, true);
  }


void MultiStatePS::write(std::ostream& os) const
  {
  colin::StdOptSolver<BasicArray<double>, response_t> :: write(os);

  os << "##\n## MultiState Pattern Search Controls\n##\n";

  os << "m_max_success         " << m_max_success << endl;
  os << "m_delta_init          " << m_delta_init << endl;
  os << "m_delta_thresh        " << m_delta_thresh << endl;
  os << "m_contraction_factor  " << m_contraction_factor << endl;
  os << "m_expansion_factor    " << m_expansion_factor << endl;
  os << "m_sigma:              " << m_sigma << endl;
  }



MultiStatePS::responseSet_t 
MultiStatePS::checkCache(MultiStatePS::domain_t &point)
  {
  if ( DEBUG_MULTISTATE_PS > 2 )
    { cerr << "entering checkCache()" << endl; }
  // OK - this is going to be slow, but it will work and I don't feel
  // like being too clever here.  What we should really do is design a
  // "fuzzy map" that keys off a double and matches (in log time) any
  // double within a specified range.  Chalk that up to "future work"

  double del = m_delta_thresh * m_sigma[0] / 2.0;
  double thresh = point[0] - del;

  std::map<double, std::list<MultiStatePS::responseSet_t> >::iterator c_it
      = m_cache.begin();
  std::map<double, std::list<MultiStatePS::responseSet_t> >::iterator c_itEnd
      = m_cache.end();
  while (( c_it != c_itEnd ) && ( c_it->first < thresh ))
    { ++c_it; }

  if (( c_it == c_itEnd ) || ( c_it->first > point[0] + del ))
    {
    // new entry
    MultiStatePS::responseSet_t ans(point);
    m_cache[point[0]].push_back(ans);

    if ( DEBUG_MULTISTATE_PS > 2 )
      { cerr << "checkCache(): new(1) responseSet id = " << ans.id() << endl; }
    return ans;
    }

  // OK - search through this entry to see if we have a matching entry
  std::list<MultiStatePS::responseSet_t>::iterator r_it 
      = c_it->second.begin();
  std::list<MultiStatePS::responseSet_t>::iterator r_itEnd 
      = c_it->second.end();
  bool notFound = true;
  unsigned int nvars = problem.num_real_params();
  unsigned int i = 0;
  while (( r_it != r_itEnd ) && ( notFound ))
    {
    notFound = false;
    const MultiStatePS::domain_t &ref = r_it->point();
    if ( DEBUG_MULTISTATE_PS > 3 )
      { cerr << "checkCache(): checking " << ref << endl << "checkCache(): "; }
    for(i=0; i<nvars; ++i)
      {
      if ( std::fabs(ref[i] - point[i]) > del )
        {
        if ( DEBUG_MULTISTATE_PS > 3 ) 
          { cerr << "x"; }

        notFound = true;
        ++r_it;
        break;
        }

      if ( DEBUG_MULTISTATE_PS > 3 ) 
        { cerr << "+"; }
      }
    if ( DEBUG_MULTISTATE_PS > 3 )
      { cerr << endl; }
    }
  
  if ( r_it == r_itEnd )
    {
    // new point in entry
    MultiStatePS::responseSet_t ans(point);
    m_cache[point[0]].push_back(ans);

    if ( DEBUG_MULTISTATE_PS > 2 )
      { cerr << "checkCache(): new(2) responseSet id = " << ans.id() << endl; }
    return ans;
    }
  else
    { 
    if ( DEBUG_MULTISTATE_PS > 2 )
      {
      cerr << "checkCache(): found old responseSet id = " 
           << r_it->id() << endl;
      }
    return *r_it; 
    }
  }



std::list<MultiStatePS::PSState>::iterator 
MultiStatePS::explore( responseSet_t  newCenter, 
                       double         step, 
                       bool           improving )
  {
  if ( DEBUG_MULTISTATE_PS > 1 )
    { cerr << "entering explore()" << endl; }

  m_states.push_back(PSState());
  std::list<PSState>::iterator state = m_states.end();
  --state;
  state->base = newCenter;
  state->step = step;

  unsigned int nvars = problem.num_real_params();
  response_t centerResponse;
  newCenter.stat(centerResponse, responseSet_t::NEWEST);

  async_evaluator_t::priority_t priority
      = improving 
      ? centerResponse.augmented_function_value() 
      : utilib::Ereal<double>(-1.0*step);
  async_evaluator_t::solverID_t solver
      = improving ? IMPROVING_SOLVER : SECONDARY_SOLVER;

  domain_t      trial(newCenter.point());
  responseSet_t rs;

  if ( DEBUG_MULTISTATE_PS > 1 )
    {
    cerr << "explore(): center = " 
         << centerResponse.augmented_function_value()
         << ", pri = " << priority << ", solver = " << solver 
         << ", step = " << step << endl;
    cerr << "explore(): center point = " << trial << endl;
    }
  
  if ( DEBUG_MULTISTATE_PS > 1 )
    { cerr << "explore(): entering point generation" << endl; }

  std::list<std::list<PSState>::iterator>::iterator state_it;
  std::list<std::list<PSState>::iterator>::iterator state_itEnd;
  std::map< async_evaluator_t::evalID_t, 
      std::list<std::list<PSState>::iterator> >::iterator eval_it;
  for(unsigned int i = 0; i < nvars; ++i)
    {
    double origVal = trial[i];
    bool positive = false;

    do 
      {
      // positive direction
      if ( positive )
        { trial[i] += step * (m_sigma[i]); }
      else
        { trial[i] -= step * (m_sigma[i]); }
      rs = checkCache(trial);

      if ( DEBUG_MULTISTATE_PS > 1 )
        { 
        cerr << "  trial(" << (positive ? "+" : "-") << "): " << trial
             << "[" << rs.invalid() << "," << rs.response_count() << "]"; 
        }

      if ( rs.response_count() == 0 )
        { 
        state->children.push_back(rs);
        evalID_t evalid = queue_evaluation(rs, priority, solver);
        eval_it = m_pending[solver].find(evalid);
            
        if ( eval_it != m_pending[solver].end() )
          { 
          state_it = eval_it->second.begin();
          state_itEnd = eval_it->second.end();
          for ( ; state_it != state_itEnd; ++state_it )
            { 
            // break out if we find n equivalent state
            if ( state->step == (*state_it)->step )
              { break; }
            }
          }

        if (( eval_it == m_pending[solver].end() ) 
            || ( state_it == state_itEnd ))
          {
          m_pending[solver][evalid].push_back(state);
          ++(state->pending);
          if ( DEBUG_MULTISTATE_PS > 1 )
            { cerr << "  evalID " << evalid; }
          }
        }
      trial[i] = origVal;
      if ( DEBUG_MULTISTATE_PS > 1 )
        { cerr << endl; }

      positive = !positive;
      } while ( positive );
    }

  if ( DEBUG_MULTISTATE_PS > 1 )
    { cerr << "explore(): done" << endl; }
  return state;
  }


} // namespace coliny
