/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

//
// Coliny_DynamicGSS.cpp
//
// Local search without gradients.
//

#include <acro_config.h>
#include <utilib/_math.h>
#include <utilib/stl_auxillary.h>
#include <coliny/DynamicGSS.h>

using namespace std;

namespace coliny {

DynamicGSS::DynamicGSS()
 : batch_evaluator_t(&problem)
{
}


void DynamicGSS::reset()
{
if (!problem) return;

colin::StdOptSolver<BasicArray<double>,response_t>::reset();

if (!rng)
   EXCEPTION_MNGR(runtime_error,"DynamicGSS::reset - undefined random number generator");

nrnd.generator(&rng);
batch_evaluator_t::reset();
colin::AppResponseAnalysis::initialize(problem.numNonlinearIneqConstraints(),constraint_tolerance);
}


void DynamicGSS::minimize()
{
//
// Misc initialization of the optimizer
//
opt_init();
if (!(this->initial_point_flag))
   EXCEPTION_MNGR(runtime_error,"DynamicGSS::minimize - no initial point specified.");
if (problem.num_real_params() != best().point.size())
   EXCEPTION_MNGR(runtime_error,"DynamicGSS::minimize - problem has " <<
	problem.num_real_params() << " real params, but initial point has " << best().point.size() );
if (best().point.size() == 0) {
   best().termination_info = "No-Real-Params";
   return;
   }
//
//
//
unsigned int num_iters;
if (max_iters <= 0)
   num_iters = MAXINT;
else
   num_iters = curr_iter + max_iters;
//
// This is a hack to make the default number of iterations be small.
// In general, this will not be necessary...
//
num_iters = std::min(static_cast<unsigned int>(10),num_iters);

NumArray<double> best_pt(problem.num_real_params());
NumArray<double> tmp_pt(problem.num_real_params());
real fp;
best_pt << best().point;

bool bound_feasible = problem.test_feasibility(best_pt);
real constraint_value=0.0;

best().value() = real :: positive_infinity;
perform_evaluation(best_pt, best().response, best().value(), best().constraint_violation);
if (!bound_feasible)
   EXCEPTION_MNGR(runtime_error,"DynamicGSS::minimize - initial point is not bound-feasible");
debug_io(ucout);

response_t tmp_response;
for (curr_iter++; curr_iter <= num_iters;  curr_iter++) {

  if (check_convergence(best().value()))
     break;

  tmp_pt << best_pt;
  for (unsigned int i=0; i<tmp_pt.size(); i++)
    tmp_pt[i] += nrnd();
  //
  // TODO - should check if the point is feasible
  //
  perform_evaluation(tmp_pt, tmp_response, fp, constraint_value);

  if (fp < best().value()) {
     DEBUGPR(2, ucout << "BETTER POINT: " << best().value() << " (" << fp << ")\n";);
     best().value() = fp;
     best().constraint_violation = constraint_value;
     best().response << tmp_response;
     best_pt << tmp_pt;
     }

  best().point << best_pt;
  debug_io(ucout);
  }

best().point << best_pt;
debug_io(ucout,true);
}


} // namespace coliny
