/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

/**
 * \file MultiStart.h
 *
 * Defines the \c MultiStartBase and \c MultiStart classes
 */

#ifndef coliny_Multistart_h
#define coliny_Multistart_h

#include <acro_config.h>
#include <utilib/SampleGenerators.h>
#include <utilib/DUniform.h>
#include <utilib/CharString.h>
#include <colin/StdOptSolver.h>
#include <colin/AppResponseAnalysis.h>

namespace coliny {

using utilib::CharString;
using utilib::MixedIntVars;



/** \class MultiStartBase
  *
  * A generic engine that manages a set of independent local search
  * runs.
  */
template <class DomainT, class GeneratorT>
class MultiStartBase : 
	public colin::StdOptSolver<DomainT, colin::AppResponse_Utilib>, 
	public colin::AppResponseAnalysis
{
  typedef colin::StdOptSolver<DomainT,colin::AppResponse_Utilib> base_t;
  #if !defined(SOLARIS)
  using base_t::max_time;
  using base_t::problem;
  using base_t::max_iters;
  using base_t::curr_iter;
  using base_t::constraint_tolerance;
  using base_t::max_neval;
  using base_t::accuracy;
  using base_t::opt_name;
  #endif

public:

  /// Constructor
  MultiStartBase();

  /// Destructor
  virtual ~MultiStartBase() {}
	
  ///
  void reset();

  ///
  void minimize();

  ///
  void write_parameter_values(std::ostream& os, const char* opt_label="") const
	{
	if (strcmp(opt_label,"")==0)
	   ParameterSet::write_parameter_values(os,"Multistart-Master");
	else
	   ParameterSet::write_parameter_values(os,opt_label);
	for (unsigned int i=0; i<this->colin_solver.size(); i++) {
	  CharString str = "Solver-";
	  str += i;
	  this->colin_solver[i]->base().write_parameter_values(os,str.data());
	  }
	}

  ///
  void write_parameter_values(utilib::PackBuffer& os) const
                {base_t::write_parameter_values(os);}
                                                                                
protected:

  ///
  unsigned int batch_size;

  ///
  double ls_frequency;

  ///
  utilib::DUniform<size_type> drand;

  ///
  utilib::Uniform urand;

  ///
  utilib::BasicArray< colin::AppResponse_Utilib > ans;

  ///
  utilib::BasicArray<colin::real > cval;

  /// The sum of squared constraint violations
  utilib::BasicArray<colin::real > sq_violation;

  ///
  utilib::BasicArray<DomainT> point;

  ///
  GeneratorT generator;

  ///
  void generate_point(DomainT& pt)
	{ generator.new_point(pt); }

  ///
  virtual void setup()
	{
	generate_point(this->best().point);
	evaluate_point(this->best().point,this->best().response);
	synchronize();
	compute_response_info(this->best().response,
			problem.state->constraint_lower_bounds,
			problem.state->constraint_upper_bounds,
			this->best().value(),
			this->best().constraint_violation);
	}

  ///
  void do_ls(DomainT& pt, colin::AppResponse_Utilib& response)
	{
	try {
	  size_type ndx = drand();
	  this->generic_solver[ndx]->set_initial_point(pt);
	  this->generic_solver[ndx]->reset();
	  this->generic_solver[ndx]->minimize();
	  response << this->generic_solver[ndx]->opt_response().response;
	  pt << this->generic_solver[ndx]->opt_response().point;
          //ucout << "BUG: " << response.function_value() << " " << this->generic_solver[ndx]->opt_response().response.function_value() << std::endl;
	  }
	catch (std::exception& err) {
	  EXCEPTION_MNGR(std::runtime_error, "MultiStartBase::do_ls - Problem applying local search\n\tError: " << err.what());
	  }
	}

  ///
  void evaluate_point(DomainT& pt, colin::AppResponse_Utilib& response)
	{
        int priority=1;
	if (problem.numNonlinearConstraints() == 0)
	   problem.AsyncEval(pt, priority, &response, colin::mode_f);
	else
	   problem.AsyncEval(pt, priority, &response, colin::mode_f | colin::mode_cf);
	}

  ///
  void synchronize()
	{ problem.synchronize(); }

};


template <class DomainT, class GeneratorT>
MultiStartBase<DomainT,GeneratorT>
	::MultiStartBase()
  : batch_size(1),
    ls_frequency(0.0)
{
opt_name = "MultiStart";
//
//probe();

ParameterSet::create_parameter("batch_size", batch_size,
	"<unsigned int>","1",
	"The number of points generated in each iteration");
ParameterSet::create_parameter("ls_frequency",ls_frequency,
	"<double>","0.0",
	"The probability of applying local search to each point generated");
}


template <class DomainT, class GeneratorT>
void MultiStartBase<DomainT,GeneratorT>
		::reset()
{
if (!problem) return;

if ((ls_frequency < 0.0) || (ls_frequency > 1.0))
   EXCEPTION_MNGR(std::runtime_error, "Bad local search frequency: " << ls_frequency);

if (!(problem.has_all_bounds()))
   EXCEPTION_MNGR(std::runtime_error,"MultiStartBase::reset - not enforcing all bounds.");

urand.generator(&(this->rng));
drand.generator(&(this->rng));
generator.set_rng((this->rng));

if ((this->generic_solver.size() > 0) && (ls_frequency == 0.0))
   ls_frequency=1.0;
for (unsigned int i=0; i<this->colin_solver.size(); i++) {
  this->colin_solver[i]->base().set_parameter("accuracy",accuracy);
  this->colin_solver[i]->base().set_parameter("max_neval",max_neval);
  this->colin_solver[i]->base().set_parameter("max_time",max_time);
  }

base_t::reset();

ans.resize(batch_size);
for (unsigned int i=0; i<batch_size; i++)
  ans[i].resize(problem.numObjectives(),
                                problem.numNonlinearConstraints(),
                                problem.num_real_params());
cval.resize(batch_size);
sq_violation.resize(batch_size);
point.resize(batch_size);
drand.low(0);
drand.high(this->generic_solver.size()-1);

colin::AppResponseAnalysis::initialize(problem.numNonlinearIneqConstraints(),constraint_tolerance);
}


template <class DomainT, class GeneratorT>
void MultiStartBase<DomainT,GeneratorT>
	::minimize()
{
//
// Misc initialization of the optimizer
//
this->opt_init();
//
// Generate initial point and setup data structures
//
setup();
//
// Setup iteration counters and debugging IO
//
unsigned int num_iters;
if (max_iters <= 0)
   num_iters = MAXINT;
else
   num_iters = curr_iter + max_iters;
this->debug_io(ucout);
//
// Main Loop
//
for (curr_iter++; curr_iter <= num_iters;  curr_iter++) {
  //
  // Check termination rules
  //
  if (this->check_convergence(this->best().value()))
     break;
  //
  // Generate all of the points.
  //
  for (unsigned int b=0; b<batch_size; b++) {
    generate_point(point[b]);
    if ((ls_frequency >= 1.0) || ((ls_frequency > 0.0) && 
        (urand() < ls_frequency)))
       do_ls(point[b],ans[b]);
    else
       evaluate_point(point[b],ans[b]);
    }
  //
  // Synchronize and compute the penalized objective
  //
  synchronize();
  for (unsigned int i=0; i<batch_size; i++) {
    compute_response_info(ans[i],
		problem.state->constraint_lower_bounds,
		problem.state->constraint_upper_bounds,
		cval[i],
		sq_violation[i]);
    DEBUGPR(100, ucout << "Trial point " << i << std::endl;
		 ucout << "Final Point: " << point[i] << std::endl;
		 ucout << "Value: " << cval[i] << " " << 
			  "Fval:  " << ans[i].function_value() << " " <<
			  "CViol: " << sq_violation[i] << std::endl << utilib::Flush;);
    }

  //
  // Find the best point
  //
  int bp=0;
  for (unsigned int i=1; i<cval.size(); i++)
    if (cval[i] < cval[bp]) bp=i;
  if (cval[bp] < this->best().value()) {
     this->best().value() = cval[bp];
     this->best().response << ans[bp];
     this->best().constraint_violation = sq_violation[bp];
     this->best().point << point[bp];
     }
 
  this->debug_io(ucout);
  }

this->debug_io(ucout,true);
}


template <class DomainT, class GeneratorT=utilib::UniformSampleGenerator<DomainT> >
class MultiStart : public MultiStartBase<DomainT,GeneratorT>
{
public:

  /// Constructor
  MultiStart() {}

};


template <>
class MultiStart<MixedIntVars> 
	: public MultiStartBase<MixedIntVars, utilib::UniformSampleGenerator<MixedIntVars> >
{
  typedef MultiStartBase<MixedIntVars, utilib::UniformSampleGenerator<MixedIntVars> > base_t;

public:


  /// Constructor
  MultiStart() {}

  ///
  void reset()
	{
	if (!get_problem()) return;
	get_problem().get_real_bounds(generator.d_l_bound,generator.d_u_bound);
	get_problem().get_int_bounds(generator.i_l_bound,generator.i_u_bound);
	base_t::reset();
	}

};


} // namespace coliny

#endif
