/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file SysCallApplication.h
 *
 * Defines the colin::SysCallApplication class.
 */

#ifndef colin_SysCallApplication_h
#define colin_SysCallApplication_h

#ifdef __MINGW32__
#include <_mingw.h>
#include <windows.h>
#endif
#include <acro_config.h>
#include <utilib/std_headers.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <colin/OptApplication.h>
#include <colin/SysCallAnalysisCode.h>
#include <colin/CommandShell.h>

namespace colin {


/** The problem provides an interface for functions that are called 
via a system call.
*/
template <class DomainT, class ResponseT, class StringT=std::string>
class SysCallApplication : public OptApplication<DomainT,ResponseT> {
 
public:
 
  /**@name General Routines */
  //@{
  /// Constructor.
  SysCallApplication() : OptApplication<DomainT,ResponseT>() {}

  /// Destructor
  virtual ~SysCallApplication() {}
  //@}

  /// Setup this application
  void setup(const StringT& program, const StringT& input_filename,
				const StringT& output_filename,
				int mode,
				bool ctr_suffix=true,
				bool remove_files=true)
		{this->app_mode=mode;
		 simulator.setup(program,input_filename,output_filename,
					ctr_suffix,!remove_files);}

  /// Set the simulator
  void set_simulator(SysCallAnalysisCode<DomainT,ResponseT,StringT>& simulator_)
		{simulator = simulator_;}

  /// Get the simulator
  SysCallAnalysisCode<DomainT,ResponseT,StringT>& get_simulator() const
		{return simulator;}

  ///
  void synchronize();

protected:

  /// TODO
  void process(DomainT& point, ResponseT& response, int counter);

  /// The simulator object
  SysCallAnalysisCode<DomainT,ResponseT,StringT> simulator;

  ///
  void DoEval(DomainT& point, int& priority,
				ResponseT* response, bool synch_flag);
};



//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
void SysCallApplication<DomainT,ResponseT,StringT>::synchronize()
{
bool flag=true;
while (this->response_list.size() > 0) {
  //
  // Sleep if we didn't find any completed files last time
  //
  if (!flag)
     #if defined(__MINGW32__)
     Sleep(1000);
     #else
     sleep(1);
     #endif
  //
  // Q: do we need to keep a list of points that are associated with async
  // evals?  If not, how do we update the database?
  //
  flag=false;
  DomainT point;
#if defined(COUGAR) || defined(TFLOPS_SERVICE)
  typename utilib::LinkedList<ResponseT*>::iterator item = this->response_list.begin();
  typename utilib::LinkedList<ResponseT*>::iterator end = this->response_list.end();
#else
  typename std::list<ResponseT*>::iterator item = this->response_list.begin();
  typename std::list<ResponseT*>::iterator end = this->response_list.end();
#endif
  while (item != end) {
    std::string file_to_test = simulator.results_fname((*item)->info->id);
    struct stat buf;
    if (stat( (char*)file_to_test.data(),&buf ) != -1) {
        process(point,**item, (*item)->info->id);
	CachedAllocator<ResponseT> :: deallocate(*item);
        item = this->response_list.erase(item);
	flag=true;
        }
    else
       item++;
    }
  }
}


//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
void SysCallApplication<DomainT,ResponseT,StringT>::
	process(DomainT& point, ResponseT& response, int counter)
{
simulator.read_results_file(response, counter);
update_response(point,response);   
}


//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
void SysCallApplication<DomainT,ResponseT,StringT>
	::DoEval(DomainT& point, int& /* priority */,
				ResponseT* response, bool synch_flag)
{
verify(response->info->mode);

if (response_exists(point,*response)) {
   if (!synch_flag)
      CachedAllocator<ResponseT> :: deallocate(response);
   return;
   }

//
// We assume that constraints and function evaluations are both being
// computed here.
//
this->nprob_ctr++;
this->neval_ctr++;
if ((this->num_eq_constr + this->num_ineq_constr)>0)
   this->nconstr_ctr++;

simulator.define_filenames(response->info->id);
simulator.write_parameters_file(point,response->request_vector(),response->info->id);
CommandShell spawned_shell;
simulator.spawn_evaluation(synch_flag);
if (!synch_flag) 
   this->response_list.push_back(response);
else
   process(point,*response,response->info->id);
}


}

#endif
