/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file StatusCodes.h
 *
 * Defines enumeration types for model/solver status codes.
 */

//
// TODO: Re-assess whether all of these status codes are meaningful for 
//		Acro solvers.

#ifndef __colin_StatusCodes_h
#define __colin_StatusCodes_h

#include <acro_config.h>

namespace colin {

/// Status of final solution.
enum model_status_enum 	{
			model_optimal=1,
			model_locally_optimal=2,
			model_feasible=3,
			model_infeasible=4,
			model_locally_infeasible=5,
			model_intermediate_infeasible=6,
			model_intermediate_nonoptimal=7,
			model_integer_solution=8,
			model_intermediate_integer_solution=9,
			model_integer_infeasible=10,
			model_licensing_problems=11,
			model_error_unknown=12,
			model_error_no_solution=13,
			model_no_solution_returned=14,
			model_solved_unique=15,
			model_solved=16,
			model_solved_singular=17,
			model_unbounded_no_solution=18,
			model_infeasible_no_solution=19
			};

/// Final solver state.
enum solver_status_enum	{
			solver_normal_completion=1,
			solver_iteration_interrupt=2,
			solver_resource_interrupt=3,
			solver_terminated_by_solver=4,
			solver_evaluation_error_limit=5,
			solver_capability_problems=6,
			solver_licensing_problems=7,
			solver_user_interrupt=8,
			solver_error_setup_failure=9,
			solver_error_solver_failure=10,
			solver_error_internal_solver_error=11,
			solver_solve_processing_skipped=12,
			solver_error_system_failure=13
			};

}

/// Pack a colin::model_status_enum data.
inline utilib::PackBuffer& operator<< (utilib::PackBuffer& buff, 
					const colin::model_status_enum& data)
{buff.pack( static_cast<int>(data) ); return buff;}

/// Unpack a colin::model_status_enum data.
inline utilib::UnPackBuffer& operator>> (utilib::UnPackBuffer& buff,
					colin::model_status_enum& data)
{
int tmp;
buff.unpack(tmp);
data = static_cast< colin::model_status_enum >(tmp);
return buff;
}

/// Pack a colin::solver_status_enum data.
inline utilib::PackBuffer& operator<< (utilib::PackBuffer& buff, 
					const colin::solver_status_enum& data)
{buff.pack( static_cast<int>(data) ); return buff;}

/// Unpack a colin::solver_status_enum data.
inline utilib::UnPackBuffer& operator>> (utilib::UnPackBuffer& buff,
					colin::solver_status_enum& data)
{
int tmp;
buff.unpack(tmp);
data = static_cast< colin::solver_status_enum >(tmp);
return buff;
}

/// Write colin::model_status_enum data.
inline std::ostream& operator<< (std::ostream& ostr,
					const colin::model_status_enum data)
{
switch (data) {
	case colin::model_optimal:
			ostr << "Optimal"; break;
	case colin::model_locally_optimal:
			ostr << "LocallyOptimal"; break;
	case colin::model_feasible:
			ostr << "Feasible"; break;
	case colin::model_infeasible:
			ostr << "Infeasible"; break;
	case colin::model_locally_infeasible:
			ostr << "LocallyInfeasible"; break;
	case colin::model_intermediate_infeasible:
			ostr << "Intermediate_Infeasible"; break;
	case colin::model_intermediate_nonoptimal:
			ostr << "Intermediate_Nonoptimal"; break;
	case colin::model_integer_solution:
			ostr << "IntegerSolution"; break;
	case colin::model_intermediate_integer_solution:
			ostr << "Intermediate_IntegerSolution"; break;
	case colin::model_integer_infeasible:
			ostr << "IntegerInfeasible"; break;
	case colin::model_licensing_problems:
			ostr << "LicensingProblems"; break;
	case colin::model_error_unknown:
			ostr << "Error_Unknown"; break;
	case colin::model_error_no_solution:
			ostr << "Error_NoSolution"; break;
	case colin::model_no_solution_returned:
			ostr << "Error_NoSolution"; break;
	case colin::model_solved_unique:
			ostr << "SolvedUnique"; break;
	case colin::model_solved:
			ostr << "Solved"; break;
	case colin::model_solved_singular:
			ostr << "SolvedSingular"; break;
	case colin::model_unbounded_no_solution:
			ostr << "Unbounded_NoSolution"; break;
	case colin::model_infeasible_no_solution:
			ostr << "Infeasible_NoSolution"; break;
	default:
			ostr << "BadModelStatus";
	}

return ostr;
}

/// Read colin::model_status_enum data.
inline std::istream& operator>> (std::istream& istr,
					colin::model_status_enum& data)
{
std::string tmp;
istr >> tmp;
if (tmp == "Optimal")
   data = colin::model_optimal;
else if (tmp == "LocallyOptimal")
   data = colin::model_locally_optimal;
else if (tmp == "Feasible")
   data = colin::model_feasible;
else if (tmp == "Infeasible")
   data = colin::model_infeasible;
else if (tmp == "LocallyInfeasible")
   data = colin::model_locally_infeasible;
else if (tmp == "Intermediate_Infeasible")
   data = colin::model_intermediate_infeasible;
else if (tmp == "Intermediate_Nonoptimal")
   data = colin::model_intermediate_nonoptimal;
else if (tmp == "IntegerSolution")
   data = colin::model_integer_solution;
else if (tmp == "Intermediate_IntegerSolution")
   data = colin::model_intermediate_integer_solution;
else if (tmp == "IntegerInfeasible")
   data = colin::model_integer_infeasible;
else if (tmp == "LicensingProblems")
   data = colin::model_licensing_problems;
else if (tmp == "Error_Unknown")
   data = colin::model_error_unknown;
else if (tmp == "Error_NoSolution")
   data = colin::model_error_no_solution;
else if (tmp == "SolvedUnique")
   data = colin::model_solved_unique;
else if (tmp == "Solved")
   data = colin::model_solved;
else if (tmp == "SolvedSingular")
   data = colin::model_solved_singular;
else if (tmp == "Unbounded_NoSolution")
   data = colin::model_unbounded_no_solution;
else if (tmp == "Infeasible_NoSolution")
   data = colin::model_infeasible_no_solution;
else
   data = colin::model_error_unknown;
return istr;
}

/// Write colin::solver_status_enum data.
inline std::ostream& operator<< (std::ostream& ostr,
					const colin::solver_status_enum data)
{
switch (data) {
	case colin::solver_normal_completion:
			ostr << "NormalCompletion"; break;
	case colin::solver_iteration_interrupt:
			ostr << "IterationInterrupt"; break;
	case colin::solver_resource_interrupt:
			ostr << "ResourceInterrupt"; break;
	case colin::solver_terminated_by_solver:
			ostr << "TerminatedBySolver"; break;
	case colin::solver_evaluation_error_limit:
			ostr << "EvaluationErrorLimit"; break;
	case colin::solver_capability_problems:
			ostr << "CapabilityProblems"; break;
	case colin::solver_licensing_problems:
			ostr << "LicensingProblems"; break;
	case colin::solver_user_interrupt:
			ostr << "UserInterrupt"; break;
	case colin::solver_error_setup_failure:
			ostr << "Error_SetupFailure"; break;
	case colin::solver_error_solver_failure:
			ostr << "Error_SolverFailure"; break;
	case colin::solver_error_internal_solver_error:
			ostr << "Error_InternalSolverError"; break;
	case colin::solver_solve_processing_skipped:
			ostr << "SolveProcessingSkipped"; break;
	case colin::solver_error_system_failure:
			ostr << "Error_SystemFailure"; break;
	default:
			ostr << "BadSolverStatus";
	}

return ostr;
}

/// Read colin::solver_status_enum data.
inline std::istream& operator>> (std::istream& istr,
					colin::solver_status_enum& data)
{
std::string tmp;
istr >> tmp;
if (tmp == "NormalCompletion")
   data = colin::solver_normal_completion;
else if (tmp == "IterationInterrupt")
   data = colin::solver_iteration_interrupt;
else if (tmp == "ResourceInterrupt")
   data = colin::solver_resource_interrupt;
else if (tmp == "TerminatedBySolver")
   data = colin::solver_terminated_by_solver;
else if (tmp == "EvaluationErrorLimit")
   data = colin::solver_evaluation_error_limit;
else if (tmp == "CapabilityProblems")
   data = colin::solver_capability_problems;
else if (tmp == "LicensingProblems")
   data = colin::solver_licensing_problems;
else if (tmp == "UserInterrupt")
   data = colin::solver_user_interrupt;
else if (tmp == "Error_SetupFailure")
   data = colin::solver_error_setup_failure;
else if (tmp == "Error_SolverFailure")
   data = colin::solver_error_solver_failure;
else if (tmp == "Error_InternalSolverError")
   data = colin::solver_error_internal_solver_error;
else if (tmp == "SolveProcessingSkipped")
   data = colin::solver_solve_processing_skipped;
else if (tmp == "Error_SystemFailure")
   data = colin::solver_error_system_failure;
else
   data = colin::solver_normal_completion;

return istr;
}


#endif
