/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptSetup.h
 *
 * Defines a variety of colin::OptSetup functions, that initialize 
 * colin::OptProblem objects.
 */

#if !defined(DOXYGEN)
#ifndef colin_OptSetup_h
#define colin_OptSetup_h

#include <acro_config.h>
#include <utilib/std_headers.h>
#include <utilib/BasicArray.h>
#include <colin/OptProblem.h>
#include <colin/DirectFuncApplication.h>
#include <colin/DirectSimpleApplication.h>
#include <colin/DirectUserResponseApplication.h>
#include <colin/SysCallApplication.h>


namespace colin {


typedef void (*init_fn_type)(int*, char***);
typedef void (*fini_fn_type)();

//============================================================================
//
// OptSetup
//   <OptProblem>
//   <program-name>
//   <num-inequality-constraints>=0
//   <num-equality-constraints>=0
//   <remove-files-flag>=true
//   <use-counters-in-names>=true
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT> & opt,
        const char* program,
	const char* domain=0,
	int num_ineqc=0,
	int num_eqc=0,
	bool remove_files=true,
	bool ctr_suffix=true)
{
SysCallApplication<DomainT,ResponseT,std::string> *app 
	= new SysCallApplication<DomainT,ResponseT,std::string>();
int mode=mode_f;
if (num_ineqc+num_eqc > 0)
   mode |= mode_cf;
std::string program_ = program;
std::string outname = program_;
outname += ".out";
std::string inname = program_;
inname += ".in";
app->setup(program_,inname,outname,mode,ctr_suffix,remove_files);
  
app->num_ineq_constr = num_ineqc;
app->num_eq_constr = num_eqc;

opt.set_application(app);
if (domain) opt.set_bounds(domain);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   <program-name>
//   <num-inequality-constraints>=0
//   <num-equality-constraints>=0
//   <remove-files-flag>=true
//   <use-counters-in-names>=true
//
template <class DomainT, class ResponseT, class StringT>
void OptSetup(
	OptProblem<DomainT,ResponseT> & opt,
        const StringT& program,
	const StringT& input_filename,
	const StringT& output_filename,
	const char* domain=0,
	int num_ineqc=0,
	int num_eqc=0,
	bool remove_files=true,
	bool ctr_suffix=true)
{
SysCallApplication<DomainT,ResponseT,StringT> *app 
	= new SysCallApplication<DomainT,ResponseT,StringT>();
int mode=mode_f;
if (num_ineqc+num_eqc > 0)
   mode |= mode_cf;
app->setup(program,input_filename,output_filename,mode,ctr_suffix,remove_files);
  
app->num_ineq_constr = num_ineqc;
app->num_eq_constr = num_eqc;

opt.set_application(app);
if (domain) opt.set_bounds(domain);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, ResponseT&)
//   <domain-string>
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT> & opt,
	void (*eval_fn)(DomainT&, ResponseT&),
	const char* domain,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void(*)(DomainT&, ResponseT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void(*)(DomainT&, ResponseT&)>( mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
opt.set_bounds(domain);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, ResponseT&)
//   <domain-string>
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, ResponseT&),
	const char* domain,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void(*)(const DomainT&, ResponseT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void(*)(const DomainT&, ResponseT&)>( mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
opt.set_bounds(domain);
}

//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, ResponseT&)
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, ResponseT&),
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void(*)(DomainT&, ResponseT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void(*)(DomainT&, ResponseT&)>( mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, ResponseT&)
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, ResponseT&),
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void(*)(const DomainT&, ResponseT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void(*)(const DomainT&, ResponseT&)>( mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
}




//============================================================================
//
// OptSetup
//   <OptProblem>
//   double func(ParamT*, int)
//   <domain-string>=0
//   init_fn=0
//   fini_fn=0
//
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
template < class ParamT, template <class T> class ArrayT, class ResponseT>
void OptSetup(
	OptProblem<ArrayT<ParamT>,ResponseT>& opt,

#else
template <class ParamT, class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
#endif
	double (*eval_fn)(ParamT*, int),
        const char* domain=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectSimpleApplication<
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  ParamT,
  ArrayT,
#else
  DomainT,
#endif
  ResponseT,
  double (*)(ParamT*, int)>* app
  = new DirectSimpleApplication<
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
	ParamT,
	ArrayT,
#else
	DomainT,
#endif
	ResponseT,
	double(*)(ParamT*, int)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}



#if !defined(ACRO_SOLARIS_CC)
//============================================================================
//
// OptSetup
//   <OptProblem>
//   double func(const ParamT*, int)
//   <domain-string>=0
//   init_fn=0
//   fini_fn=0
//
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
template < class ParamT, template <class T> class ArrayT, class ResponseT>
void OptSetup(
	OptProblem<ArrayT<ParamT>,ResponseT>& opt,
#else
template <class ParamT, class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
#endif
	double (*eval_fn)(const ParamT*, int),
        const char* domain=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectSimpleApplication<
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  ParamT,
  ArrayT,
#else
  DomainT,
#endif
  ResponseT,
  double (*)(const ParamT*, int)>* app
  = new DirectSimpleApplication<
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
	ParamT,
	ArrayT,
#else
	DomainT,
#endif
	ResponseT,
	double(*)(const ParamT*, int)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}
#endif

template <class DomainT, class ResponseT>
void exec_DirectApplication_EvalFn(DomainT& point, 
	ResponseT& response,
	void (*eval_fn)(const DomainT&, 
			const std::vector<int>&, 
			ResponseT&))
{ (*eval_fn)(point, response.response_vector(), response); }


template <class DomainT, class ResponseT>
void exec_DirectApplication_EvalFn(DomainT& point, 
	ResponseT& response,
	void (*eval_fn)(DomainT&, 
			std::vector<int>&, 
			ResponseT&))
{ (*eval_fn)(point, response.response_vector(), response); }


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, const vector<int>& asv, ResponseT& response)
//   num_ineqc=0,
//   num_eqc=0,
//   <domain-string>=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, const std::vector<int>& , ResponseT&),
	int num_ineqc=0,
	int num_eqc=0,
        const char* domain=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(const DomainT&, const std::vector<int>&, ResponseT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(const DomainT&, const std::vector<int>&, ResponseT&)>(mode_f,eval_fn);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}

//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, vector<int>& asv, ResponseT& response)
//   num_ineqc=0,
//   num_eqc=0,
//   <domain-string>=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, std::vector<int>& , ResponseT&),
	int num_ineqc=0,
	int num_eqc=0,
        const char* domain=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(DomainT&, std::vector<int>&, ResponseT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(DomainT&, std::vector<int>&, ResponseT&)>(mode_f,eval_fn);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}

//============================================================================
//
// OptSetup
//   <OptProblem>
//   double func(const DomainT&)
//   <domain-string>=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	double (*eval_fn)(const DomainT&),
        const char* domain=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  double (*)(const DomainT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	double (*)(const DomainT&)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}

//============================================================================
//
// OptSetup
//   <OptProblem>
//   double func(DomainT&)
//   <domain-string>=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	double (*eval_fn)(DomainT&),
        const char* domain=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  double (*)(DomainT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	double (*)(DomainT&)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, double&)
//   <domain-string>=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, double&),
        const char* domain=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(DomainT&, double&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(DomainT&, double&)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, double&)
//   <domain-string>=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, double&),
        const char* domain=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(DomainT&, double&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(const DomainT&, double&)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, real&)
//   <domain-string>=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, real&),
        const char* domain=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(DomainT&, real&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(DomainT&, real&)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, real&)
//   <domain-string>=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, real&),
        const char* domain=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(DomainT&, real&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(const DomainT&, real&)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, double& function, VectorT& constraints)
//   <domain-string>
//   num_ineqc=0
//   num_eqc=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, double&, CArrayT&),
        const char* domain,
	int num_ineqc=0,
	int num_eqc=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(DomainT&, double&, CArrayT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(DomainT&, double&, CArrayT&)>(mode_f|mode_cf,eval_fn);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}

//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, double&, VectorT&)
//   num_ineqc
//   num_eqc=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, real&, CArrayT&),
	int num_ineqc,
	int num_eqc=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(const DomainT&, real&, CArrayT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(const DomainT&, real&, CArrayT&)>(mode_f|mode_cf,eval_fn);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, double&, VectorT&)
//   <domain-string>=0
//   num_ineqc=0
//   num_eqc=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, double&, CArrayT&),
        const char* domain=0,
	int num_ineqc=0,
	int num_eqc=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(DomainT&, double&, CArrayT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(const DomainT&, double&, CArrayT&)>(mode_f|mode_cf,eval_fn);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, UserResponse<VectorT>&)
//   <domain-string>
//   num_ineqc=0
//   num_eqc=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, UserResponse<CArrayT>&),
        const char* domain,
	int num_ineqc=0,
	int num_eqc=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectUserResponseApplication<
  DomainT,
  ResponseT,
  void (*)(DomainT&, UserResponse<CArrayT>&)>* app
  = new DirectUserResponseApplication<
	DomainT,
	ResponseT,
	void (*)(DomainT&, UserResponse<CArrayT>&)>(mode_f|mode_g|mode_cf|mode_cg,eval_fn);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, UserResponse<VectorT>&)
//   <domain-string>
//   num_ineqc=0
//   num_eqc=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, UserResponse<CArrayT>&),
        const char* domain,
	int num_ineqc=0,
	int num_eqc=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectUserResponseApplication<
  DomainT,
  ResponseT,
  void (*)(const DomainT&, UserResponse<CArrayT>&)>* app
  = new DirectUserResponseApplication<
	DomainT,
	ResponseT,
	void (*)(const DomainT&, UserResponse<CArrayT>&)>(mode_f|mode_g|mode_cf|mode_cg,eval_fn);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, UserResponse<VectorT>&)
//   num_ineqc=0
//   num_eqc=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, UserResponse<CArrayT>&),
	int num_ineqc=0,
	int num_eqc=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectUserResponseApplication<
  DomainT,
  ResponseT,
  void (*)(DomainT&, UserResponse<CArrayT>&)>* app
  = new DirectUserResponseApplication<
	DomainT,
	ResponseT,
	void (*)(DomainT&, UserResponse<CArrayT>&)>(mode_f|mode_g|mode_cf|mode_cg,eval_fn);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, UserResponse<VectorT>&)
//   num_ineqc=0
//   num_eqc=0
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, UserResponse<CArrayT>&),
	int num_ineqc=0,
	int num_eqc=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectUserResponseApplication<
  DomainT,
  ResponseT,
  void (*)(const DomainT&, UserResponse<CArrayT>&)>* app
  = new DirectUserResponseApplication<
	DomainT,
	ResponseT,
	void (*)(const DomainT&, UserResponse<CArrayT>&)>(mode_f|mode_g|mode_cf|mode_cg,eval_fn);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, double& val)
//   void cfunc(DomainT&, CArrayT& cvals)
//   <domain-string>
//   num_ineqc
//   num_eqc
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, double&),
	void (*eval_cfn)(DomainT&, CArrayT&),
	const char* domain,
	int num_ineqc,
	int num_eqc,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
FunctionContainer1<DomainT,
		ResponseT,
		void (*)(DomainT&, double&),
  		void (*)(DomainT&, CArrayT&)> container(eval_fn,eval_cfn);
DirectFuncApplication<
  DomainT,
  ResponseT,
  FunctionContainer1<DomainT,
		ResponseT,
		void (*)(DomainT&, double&),
  		void (*)(DomainT&, CArrayT&)> >* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	FunctionContainer1<DomainT,ResponseT,void (*)(DomainT&, double&), void (*)(DomainT&, CArrayT&)> >(mode_f|mode_g|mode_cf|mode_cg,container);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, double& val)
//   void cfunc(const DomainT&, CArrayT& cvals)
//   <domain-string>
//   num_ineqc
//   num_eqc
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, double&),
	void (*eval_cfn)(const DomainT&, CArrayT&),
	const char* domain,
	int num_ineqc,
	int num_eqc,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
FunctionContainer1<DomainT,
		ResponseT,
		void (*)(const DomainT&, double&),
  		void (*)(const DomainT&, CArrayT&)> container(eval_fn,eval_cfn);
DirectFuncApplication<
  DomainT,
  ResponseT,
  FunctionContainer1<DomainT,
		ResponseT,
		void (*)(const DomainT&, double&),
  		void (*)(const DomainT&, CArrayT&)> >* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
  	FunctionContainer1<DomainT,
		ResponseT,
		void (*)(const DomainT&, double&),
  		void (*)(const DomainT&, CArrayT&)> >
	(mode_f|mode_g|mode_cf|mode_cg,container);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}

//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, real& val)
//   void cfunc(DomainT&, CArrayT& cvals)
//   <domain-string>
//   num_ineqc
//   num_eqc
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, real&),
	void (*eval_cfn)(DomainT&, CArrayT&),
	const char* domain,
	int num_ineqc,
	int num_eqc,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
FunctionContainer1<DomainT,
		ResponseT,
		void (*)(DomainT&, real&),
  		void (*)(DomainT&, CArrayT&)> container(eval_fn,eval_cfn);
DirectFuncApplication<
  DomainT,
  ResponseT,
  FunctionContainer1<DomainT,
		ResponseT,
		void (*)(DomainT&, real&),
  		void (*)(DomainT&, CArrayT&)> >* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	FunctionContainer1<DomainT,ResponseT,void (*)(DomainT&, real&), void (*)(DomainT&, CArrayT&)> >(mode_f|mode_g|mode_cf|mode_cg,container);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}


#ifdef ACRO_AIX_CC
//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, real& val)
//   void cfunc(DomainT&, utilib::BasicArray<real>& cvals)
//   <domain-string>
//   num_ineqc
//   num_eqc
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, real&),
	void (*eval_cfn)(DomainT&, utilib::BasicArray<real>&),
	const char* domain,
	int num_ineqc,
	int num_eqc,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
FunctionContainer1<DomainT,
		ResponseT,
		void (*)(DomainT&, real&),
  		void (*)(DomainT&, utilib::BasicArray<real>&)> container(eval_fn,eval_cfn);
DirectFuncApplication<
  DomainT,
  ResponseT,
  FunctionContainer1<DomainT,
		ResponseT,
		void (*)(DomainT&, real&),
  		void (*)(DomainT&, utilib::BasicArray<real>&)> >* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	FunctionContainer1<DomainT,ResponseT,void (*)(DomainT&, real&), void (*)(DomainT&, utilib::BasicArray<real>&)> >(mode_f|mode_g|mode_cf|mode_cg,container);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}
#endif


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, real& val)
//   void cfunc(const DomainT&, CArrayT& cvals)
//   <domain-string>
//   num_ineqc
//   num_eqc
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, real&),
	void (*eval_cfn)(const DomainT&, CArrayT&),
	const char* domain,
	int num_ineqc,
	int num_eqc,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
FunctionContainer1<DomainT,
		ResponseT,
		void (*)(const DomainT&, real&),
  		void (*)(const DomainT&, CArrayT&)> container(eval_fn,eval_cfn);
DirectFuncApplication<
  DomainT,
  ResponseT,
  FunctionContainer1<DomainT,
		ResponseT,
		void (*)(const DomainT&, real&),
  		void (*)(const DomainT&, CArrayT&)> >* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
  	FunctionContainer1<DomainT,
		ResponseT,
		void (*)(const DomainT&, real&),
  		void (*)(const DomainT&, CArrayT&)> >
	(mode_f|mode_g|mode_cf|mode_cg,container);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
if (domain) opt.set_bounds(domain);
}

//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, double& val)
//   void cfunc(DomainT&, CArrayT& cvals)
//   num_ineqc
//   num_eqc
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, real&),
	void (*eval_cfn)(DomainT&, CArrayT&),
	int num_ineqc,
	int num_eqc=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
FunctionContainer1<DomainT,
		ResponseT,
		void (*)(DomainT&, real&),
  		void (*)(DomainT&, CArrayT&)> container(eval_fn,eval_cfn);
DirectFuncApplication<
  DomainT,
  ResponseT,
  FunctionContainer1<DomainT,
		ResponseT,
		void (*)(DomainT&, real&),
  		void (*)(DomainT&, CArrayT&)> >* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	FunctionContainer1<DomainT,ResponseT,void (*)(DomainT&, real&), void (*)(DomainT&, CArrayT&)> >(mode_f|mode_g|mode_cf|mode_cg,container);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, double& val)
//   void cfunc(const DomainT&, CArrayT& cvals)
//   num_ineqc
//   num_eqc
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT, class CArrayT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, real&),
	void (*eval_cfn)(const DomainT&, CArrayT&),
	int num_ineqc,
	int num_eqc=0,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
FunctionContainer1<DomainT,
		ResponseT,
		void (*)(const DomainT&, real&),
  		void (*)(const DomainT&, CArrayT&)> container(eval_fn,eval_cfn);
DirectFuncApplication<
  DomainT,
  ResponseT,
  FunctionContainer1<DomainT,
		ResponseT,
		void (*)(const DomainT&, real&),
  		void (*)(const DomainT&, CArrayT&)> >* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
  	FunctionContainer1<DomainT,
		ResponseT,
		void (*)(const DomainT&, real&),
  		void (*)(const DomainT&, CArrayT&)> >
	(mode_f|mode_g|mode_cf|mode_cg,container);
app->set_problem_info(num_ineqc,num_eqc,1,init_fn,fini_fn);
opt.set_application(app);
}

//============================================================================
//
// OptSetup
//   <OptProblem>
//   double func(DomainT&)
//   Vector lower
//   Vector upper
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	double (*eval_fn)(DomainT&),
        utilib::pvector<double>& lower,
        utilib::pvector<double>& upper,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  double (*)(DomainT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	double (*)(DomainT&)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
opt.set_real_bounds(lower,upper);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   double func(const DomainT&)
//   Vector lower
//   Vector upper
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	double (*eval_fn)(const DomainT&),
        utilib::pvector<double>& lower,
        utilib::pvector<double>& upper,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  double (*)(const DomainT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	double (*)(const DomainT&)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
opt.set_real_bounds(lower,upper);
}



//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, colin::real&)
//   Vector lower
//   Vector upper
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, colin::real&),
        utilib::pvector<double>& lower,
        utilib::pvector<double>& upper,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(DomainT&, colin::real&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(DomainT&, colin::real&)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
opt.set_real_bounds(lower,upper);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, colin::real&)
//   Vector lower
//   Vector upper
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, colin::real&),
        utilib::pvector<double>& lower,
        utilib::pvector<double>& upper,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(const DomainT&, colin::real&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(const DomainT&, colin::real&)>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
opt.set_real_bounds(lower,upper);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, colin::real&, utilib::BasicArray<colin::real>& )
//   Vector lower
//   Vector upper
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(DomainT&, colin::real&, utilib::BasicArray<colin::real>& ),
        utilib::pvector<double>& lower,
        utilib::pvector<double>& upper,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(DomainT&, colin::real&, utilib::BasicArray<colin::real>& )>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(DomainT&, colin::real&, utilib::BasicArray<colin::real>& )>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
opt.set_real_bounds(lower,upper);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, colin::real&, utilib::BasicArray<colin::real>& )
//   Vector lower
//   Vector upper
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, colin::real&, utilib::BasicArray<colin::real>& ),
        utilib::pvector<double>& lower,
        utilib::pvector<double>& upper,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void (*)(const DomainT&, colin::real&, utilib::BasicArray<colin::real>& )>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void (*)(const DomainT&, colin::real&, utilib::BasicArray<colin::real>& )>(mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
opt.set_real_bounds(lower,upper);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, ResponseT&)
//   var_lower
//   var_upper
//   const_lower
//   const_upper
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& prob,
	void (*eval_fn)(DomainT&, ResponseT&),
	utilib::pvector<real>& vlower,
	utilib::pvector<real>& vupper,
	utilib::pvector<real>& clower,
	utilib::pvector<real>& cupper,
	int n_objectives=1,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
int mode;
if (clower.size() > 0)
   mode = mode_f | mode_cf;
else
   mode = mode_f;
DirectFuncApplication<
  DomainT,
  ResponseT,
  void(*)(DomainT&, ResponseT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void(*)(DomainT&, ResponseT&)>( mode,eval_fn);
app->set_problem_info(clower.size(),0,n_objectives,init_fn,fini_fn);
prob.set_application(app);
prob.set_real_bounds(vlower,vupper);
prob.set_constraint_bounds(clower,cupper);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(DomainT&, ResponseT&)
//   var_lower
//   var_upper
//   const_lower
//   const_upper
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& prob,
	void (*eval_fn)(DomainT&, ResponseT&),
	utilib::BasicArray<real>& vlower,
	utilib::BasicArray<real>& vupper,
	utilib::BasicArray<real>& clower,
	utilib::BasicArray<real>& cupper,
	int n_objectives=1,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
int mode;
if (clower.size() > 0)
   mode = mode_f | mode_cf;
else
   mode = mode_f;
DirectFuncApplication<
  DomainT,
  ResponseT,
  void(*)(DomainT&, ResponseT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void(*)(DomainT&, ResponseT&)>( mode,eval_fn);
app->set_problem_info(clower.size(),0,n_objectives,init_fn,fini_fn);
prob.set_application(app);
prob.set_real_bounds(vlower,vupper);
prob.set_constraint_bounds(clower,cupper);
}


//============================================================================
//
// OptSetup
//   <OptProblem>
//   void func(const DomainT&, ResponseT&)
//   lower
//   upper
//   init_fn=0
//   fini_fn=0
//
template <class DomainT, class ResponseT>
void OptSetup(
	OptProblem<DomainT,ResponseT>& opt,
	void (*eval_fn)(const DomainT&, ResponseT&),
	utilib::pvector<double>& lower,
	utilib::pvector<double>& upper,
	init_fn_type init_fn=0,
	fini_fn_type fini_fn=0)
{
DirectFuncApplication<
  DomainT,
  ResponseT,
  void(*)(const DomainT&, ResponseT&)>* app
  = new DirectFuncApplication<
	DomainT,
	ResponseT,
	void(*)(const DomainT&, ResponseT&)>( mode_f,eval_fn);
app->set_problem_info(0,0,1,init_fn,fini_fn);
opt.set_application(app);
opt.set_real_bounds(lower,upper);
}

} // namespace colin

#endif
#endif
