/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptDomainTraits.h
 *
 * The colin::OptDomainTraits template class defines whether certain standard
 * types are part of the domain type: real, general integer and 
 * binary variables.
 */

#ifndef colin_OptDomainTraits_h
#define colin_OptDomainTraits_h

#include <acro_config.h>
#include <utilib/std_headers.h>
#include <utilib/exception_mngr.h>

namespace colin {


#if defined(ACRO_SOLARIS_CC)

//
// This version of map_domain declarations is needed for SOLARIS to work around
// a bug in their C++ template overloading rules!
//

/** A default definition for map_domain, which simply throws an exception
  * warning the user that map_domain has not been defined!  This seems
  * more friendly than forcing a user to dig through obtuse template
  * error codes.
  */
template <class DomainT, class ParamT>
void map_domain(DomainT& target_domain, const ParamT& input_params)
{ target_domain = input_params; }

#else

//
// This is the version of map_domain declarations that should work with modern
// C++ template overloading rules!
//

/** A default definition for map_domain, when the domains are equal.
  * This calls the operator= method to copy the data.  This is not the most
  * reasonable default, but this operator is always defined...
  */
template <class DomainT>
void map_domain(DomainT& target_domain, const DomainT& input_params)
{ target_domain = input_params; }


/** A default definition for map_domain, which simply throws an exception
  * warning the user that map_domain has not been defined!  This seems
  * more friendly than forcing a user to dig through obtuse template
  * error codes.
  */
template <class DomainT, class ParamT>
void map_domain(DomainT& target_domain, const ParamT& input_params)
{ EXCEPTION_MNGR(std::runtime_error, "map_domain needs to be defined for these domains!"); }

#endif


/** A default definition for map_jacobian, which simply generates an exception
  * warning the user that map_jacobian has not been defined!  This seems
  * more friendly than forcing a user to dig through obtuse template
  * error codes.
  */
template <class DomainT, class ParamT>
void map_jacobian(DomainT& /*target_jacobian*/, const ParamT& /*input_params*/)
{ EXCEPTION_MNGR(std::runtime_error, "map_jacobian needs to be defined for these domains!"); }


/** A default definition for map_matrix, which simply generates an exception
  * warning the user that map_matrix has not been defined!  This seems
  * more friendly than forcing a user to dig through obtuse template
  * error codes.
  */
template <class DomainT, class ParamT>
void map_matrix(DomainT& , const ParamT& )
{ EXCEPTION_MNGR(std::runtime_error, "map_matrix needs to be defined for these domains!"); }


/** Macro used to declare a domain trait.
  */
#define DeclareOptDomainTrait(trait,default)\
template <class DomainT>\
struct OptDomainTrait_ ## trait {\
public:\
static const bool value = default;\
};


/** Macro used to set a domain trait.
  */
#define SetOptDomainTrait(DomainT,trait,newvalue)\
namespace colin { template<> struct OptDomainTrait_ ## trait < DomainT >{\
public:\
static const bool value = newvalue;\
}; }


#if !defined(DOXYGEN)
DeclareOptDomainTrait(reals,false)
DeclareOptDomainTrait(integers,false)
DeclareOptDomainTrait(bools,false)
DeclareOptDomainTrait(linearconstraints,false)
DeclareOptDomainTrait(boundconstraints,false)
#endif

} // namespace colin


#if !defined(DOXYGEN)
//============================================================================
//
// Standard DomainTrait definitions
//
SetOptDomainTrait(std::vector<double>,reals,true)
SetOptDomainTrait(std::vector<double>,linearconstraints,true)
SetOptDomainTrait(std::vector<double>,boundconstraints,true)
SetOptDomainTrait(std::vector<int>,integers,true)
SetOptDomainTrait(std::vector<int>,linearconstraints,true)
SetOptDomainTrait(std::vector<int>,boundconstraints,true)
#endif


#endif
