/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file GenericOptSolver.h
 *
 * Defines the colin::GenericOptSolver class.
 */

#ifndef __colin_GenericOptSolver_h
#define __colin_GenericOptSolver_h

#include <acro_config.h>
#include <colin/AppResponse.h>
#include <colin/OptResponse.h>

namespace colin {

/// An abstract specification of an optimization solver.
template <class DomainT, class ResponseT=AppResponse<> >
class GenericOptSolver
{
public:

  /// Constructor.
  GenericOptSolver()
	: initial_point_flag(false)
	{}

  /// Destructor.
  virtual ~GenericOptSolver() {}

  /// Reset the state of the solver to enable it to be run again.
  virtual void reset()
	{
	if (initial_point_flag)
           opt_response().point = initial_point;
	}

  /// Perform minimization.
  virtual void minimize() = 0;

  /// Set the initial point for minimization.
  virtual void set_initial_point(DomainT& pt)
	{
	initial_point = pt;
	initial_point_flag=true;
	}

  /// Return the response of the best point seen so far
  virtual OptResponse<DomainT,ResponseT>& opt_response()
	{ return _opt_response; }

  /// Return the response of the best point seen so far
  virtual const OptResponse<DomainT,ResponseT>& opt_response() const
	{ return _opt_response; }

  /// Print statistics about the solver.
  virtual void print_stats(std::ostream& os) const
	{
	if (opt_response().values.size() == 1)
	   os << "Final-Value: " << opt_response().value() << std::endl;
	else
	   os << "Final-Value: " << opt_response().values << std::endl;
        if (opt_response().response.get_num_functions() == 1)
	   os << "True-Value:  " << opt_response().response.function_value() << std::endl;
	else {
	   typename ResponseT::realarray_t tmp;
	   opt_response().response.function_values(tmp);
	   os << "True-Value:  " << tmp << std::endl;
	   }
	}

  /** Set to true by \c set_initial_point, for optimizers that require an
    * initial point.
    */
  bool initial_point_flag;

  /// The initial point.
  DomainT initial_point;

  /// The 'response' of optimization.
  OptResponse<DomainT,ResponseT> _opt_response;

};

}

#endif
