/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file DirectApplicationBase.h
 *
 * Defines the colin::DirectApplicationBase class.
 */

#ifndef colin_DirectApplicationBase_h
#define colin_DirectApplicationBase_h

#include <acro_config.h>
#include <colin/OptApplication.h>

namespace colin {

#if !defined(DOXYGEN)
typedef void (*init_fn_type)(int*, char***);
typedef void (*fini_fn_type)();
#endif


/**
  * This class maintains a generic interface for problem classes that 
  * compute problem functions directly from subroutines.
  */
template <class DomainT, class ResponseT, class FuncT>
class DirectApplicationBase : public OptApplication<DomainT,ResponseT>
{
public:
 
  /// Generic constructor.
  DirectApplicationBase(int app_mode_, FuncT fn) 
	: OptApplication<DomainT,ResponseT>(),
	  init_fn(0),
	  fini_fn(0)
	{setup_func(app_mode_,fn);}

  /// Setup problem information.
  void set_problem_info(int num_ineqc, int num_eqc, int num_objectives,	
		init_fn_type init_fn=0,
		fini_fn_type fini_fn=0);

protected:

  /// Set the function that will be evaluated.
  void setup_func(int app_mode_, FuncT fn)
	{
	this->app_mode = app_mode_;
	eval_fn = fn;
	}

  /// Evaluate the function and return a response.
  virtual void EvalFn(DomainT& point, ResponseT& response) = 0;
		
  /// 
  void DoEval(DomainT& point, int& priority, 
			ResponseT* response, bool synch_flag);

  /// Initialize using argc/argv arguments for the initialization function.
  void initialize(int* argc, char*** argv)
	{
	if (init_fn)
   	   (*init_fn)(argc, argv);
	}

  /// Finalize the evaluation.
  void finalize()
	{
	if (fini_fn)
   	   (*fini_fn)();
	}

  /// The function object.
  FuncT eval_fn;

  /// The initialization function pointer.
  init_fn_type init_fn;

  /// The finalization function pointer.
  fini_fn_type fini_fn;

};


//
// If synch_flag == false, this does nothing.  However, it could
// conceivably spawn a thread that computes the response.
//
template <class DomainT, class ResponseT, class FuncT>
void DirectApplicationBase<DomainT,ResponseT,FuncT>::DoEval(
				DomainT& point, int& /*priority*/,
				ResponseT* response, 
				bool synch_flag)
{
verify(response->info->mode);

if (!response_exists(point,*response)) {
   //
   // We assume that constraints and function evaluations are both being
   // computed here.
   //
   this->nprob_ctr++;
   this->neval_ctr++;
   if ((this->num_eq_constr+this->num_ineq_constr)>0)
      this->nconstr_ctr++;
   
   EvalFn(point,*response);
   response->init();
   update_response(point,*response);   
   }

if (!synch_flag) {
   //
   // Since this is a direct evaluation, the asynchronous computation has
   // already completed at this point.
   //
   this->async_completed_ids.push_back(response->info->id);
   // This doesn't really work!
   utilib::CachedAllocator<ResponseT> :: deallocate(response);
   }
}


template <class DomainT, class ResponseT, class FuncT>
void DirectApplicationBase<DomainT,ResponseT,FuncT>::set_problem_info(
		int num_ineqc, int num_eqc, 
		int num_objectives,
		init_fn_type init_fn_, fini_fn_type fini_fn_)
{
init_fn = init_fn_;
fini_fn = fini_fn_;

this->num_ineq_constr = num_ineqc;
this->num_objectives = num_objectives;
this->num_eq_constr = num_eqc;
}


}

#endif
