/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file BatchPoint.h
 *
 * Defines the colin::BatchPoint class.
 **/
 
#ifndef colin_BatchPoint_h
#define colin_BatchPoint_h
 
#include <acro_config.h>
#include <utilib/std_headers.h>
#include <colin/real.h>

namespace colin {

/// A temporary class used by colin::BatchEvaluator.
template<class DomainT, class ResponseT>
class BatchPoint {
public:

   /// Constructor
   BatchPoint() : own_response(false), response(0)
		{reset();}

   /// Constructor with a given domain point
   explicit BatchPoint(DomainT& pt_) : ValuePtr(0), CValuePtr(0)
		{reset(); point =pt_; }

   /// Destructor
   ~BatchPoint()
		{
		if (own_response)
		   utilib::CachedAllocator<ResponseT>::deallocate(response);
		}

   /// Copy operator=
   BatchPoint& operator=(const BatchPoint& pt_)
		{
		point = pt_.point; 
		Value = pt_.Value; 
        	if (response && pt_.response)
           	   *response = *(pt_.response); 
		CValue = pt_.CValue;
		ValuePtr = pt_.ValuePtr;
		CValuePtr = pt_.CValuePtr;
       		convergence_factor = pt_.convergence_factor; 
     		id = pt_.id; evaluated = pt_.evaluated; spawned=pt_.spawned; 
		ignore = pt_.ignore;
     		which_iter = pt_.which_iter;
		spec = pt_.spec;
		priority = pt_.priority;
     		return *this;
		}

   /// Set a response object
   void set_response(ResponseT* response_)
		{
		if (own_response)
	   	   utilib::CachedAllocator<ResponseT>::deallocate(response);
		response=response_;
		own_response=false;
		}

   /// Prepare for work
   void reset()
     		{
		CValuePtr=0; ValuePtr=0;
      		if (!response) {
	 	   response = utilib::CachedAllocator<ResponseT>::allocate();
	 	   own_response=true;
	 	   }
      		evaluated=false;
      		ignore=false;
		spawned=false;
		id=-1;
		convergence_factor=1.0;
      		which_iter = 0;
		spec = false;
		priority = -1;
		}

   /// Write point info
   void write(std::ostream& os) const
		{ os << point << " " << value() << " " << evaluated << " " << ignore; }

   /// Read point info
   void read(std::istream& is)
		{ is >> point >> value() >> evaluated >> ignore; }
   
   /// Returns the value of the current point.
   real& value()
		{
		if (ValuePtr)
		   return *ValuePtr;
		else
		   return Value;
		}

   /// Returns the constraint violation of the current point.
   real& cvalue()
	{ if (CValuePtr) return *CValuePtr; else return CValue; }

   /// The current point.
   DomainT point;

   /// If true, then the response object needs to be deleted.
   bool own_response;

   /// A pointer to the response.
   ResponseT* response;

   /// A convergence factor.
   double convergence_factor; 

   /// If true, point is speculative, so not necessary to wait for an answer when making
   /// subsequent decisions.
   bool spec;

   /// If true, then this point is evaluated.
   bool evaluated;

   /// If true, then ignore this point.
   bool ignore;

   /// If true, then this point has been spawned.
   bool spawned;

   /// The ID for the evaluation of this point.
   int id;  

   /// Iteration information for the point
   int which_iter;

   /// The priority of this point
   int priority;
   
   /// The value of this point.
   real Value;

   /// The constraint violation for this point.
   real CValue;

   /// A pointer to the value of this point.
   real* ValuePtr;

   /// A pointer to the constraint violation of this point.
   real* CValuePtr;
 };

} // namespace colin

/// Write the value of a colin::BatchPoint object.
template <class DomainT, class ResponseT>
std::ostream& operator<<(std::ostream& os, const typename colin::BatchPoint<DomainT, ResponseT> & obj)
{ obj.write(os); return os; }

/// Read the value of a colin::BatchPoint object.
template <class DomainT, class ResponseT>
std::istream& operator>>(std::istream& is, typename colin::BatchPoint<DomainT, ResponseT> & obj)
{ obj.read(is); return is; }

#endif
