/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file AppResponseInfo.h
 *
 * Defines the colin::AppResponseInfo class.
 */

#ifndef colin_AppResponseInfo_h
#define colin_AppResponseInfo_h

#include <acro_config.h>
#include <utilib/traits.h>
#include <utilib/stl_auxillary.h>
#include <utilib/CachedAllocator.h>

namespace colin {

class AppResponseInfo;

}

///
/// Setting is_cache_allocated traits for AppResponseInfo.
/// This needs to be setup _outside_ of the colin namespace
///
SetTrait(utilib,is_cache_allocated, colin::AppResponseInfo ,true)


namespace colin {

/**
  * Enums that can be used by a user to define response modes in 
  * an active-set vector.
  */
enum ResponseMode { mode_f=1, mode_g=2, mode_h=4, mode_cf=8, mode_cg=16, 
				mode_ch=32 };

/**
  * This contains the core data that defines an OptProblem calculation.  
  * This class does not contain the results of the calculation, but the
  * specification of the response information.
  */
class AppResponseInfo : public utilib::CachedAllocatorObject<AppResponseInfo>
{
public:

  /// Constructor.
  explicit AppResponseInfo(unsigned int num_functions=1, 
		  		unsigned int num_constraints=0, 
				unsigned int num_real_params=0,
				bool using_gradients=false,
				bool using_hessians=false);

  /// Destructor.
  virtual ~AppResponseInfo() {}

  /// Copy from another AppResponseInfo object.
  void copy(const AppResponseInfo& info)
	{
	mode = info.mode;
	id = info.id;
	id_generate = info.id_generate;
	request_flag = info.request_flag;
	responseASV << info.responseASV;
	num_real_params = info.num_real_params;
	num_functions = info.num_functions;
	num_constraints = info.num_constraints;
	requestASV << info.requestASV;
	using_gradients = info.using_gradients;
	using_hessians = info.using_hessians;
	}
  
  /// Copy operator
  AppResponseInfo& operator=(const AppResponseInfo& info)
	{
	copy(info);
	return *this;
	}

  /// The evaluation mode.
  int mode;

  /// Integer used to identify this problem response.
  int id;

  /// If true, automatically generate the id used to tag the evaluation
  bool id_generate;

  ///
  bool request_flag;

  /** The response-set vector that denotes which of the data members
    * in this class have been filled.
    * TODO.
    */
  std::vector<int> responseASV;

  /** The number of continuous parameters.  This is used to define
    * the size of the gradients and Hessians (when appropriate).
    */
  unsigned int num_real_params;

  /// The number of function values that this object can record
  unsigned int num_functions;

  /// The number of constraints that this object records
  unsigned int num_constraints;

  /** The active-set vector.
   *  The active-set vector determines the ... TODO.
   */
  std::vector<int> requestASV;

  /// Get summary statistics about the ASV
  void get_requests(bool& f_flag, bool& g_flag, bool& h_flag,
			bool& cf_flag, bool& cg_flag, bool& ch_flag) const;

  /// If true, then gradients can be calculated.
  bool using_gradients;

  /// If true, then hessians can be calculated.
  bool using_hessians;

  ///
  void deallocate_derived()
        { utilib::CachedAllocator<AppResponseInfo> :: deallocate_derived(this); }

};



inline AppResponseInfo::AppResponseInfo(unsigned int num_functions_, 
			unsigned int , 
			unsigned int ,
			bool ,
			bool )
 : mode(mode_f|mode_g|mode_h|mode_cf|mode_cg|mode_ch),
   id(-1),
   id_generate(true),
   request_flag(false),
   num_real_params(0),
   num_functions(0),
   num_constraints(0),
   using_gradients(false),
   using_hessians(false)
{
if (num_functions_ > 1)
   EXCEPTION_MNGR(std::runtime_error,"AppResponseInfo::AppResponseInfo - num_functions > 1");
}


inline void AppResponseInfo::get_requests(bool& f_flag, bool& g_flag, bool& h_flag, 
			bool& cf_flag, bool& cg_flag, bool& ch_flag) const
{
const std::vector<int>& asv = requestASV;
unsigned int j=0;

if (f_flag || g_flag || h_flag) {
   f_flag = g_flag = h_flag = false;
   while (((f_flag == false) || (g_flag == false) || (h_flag == false)) 
		&& (j < num_functions)) {
     if (asv[j] & mode_f)
        f_flag = true;
     if (asv[j] & mode_g)
        g_flag = true;
     if (asv[j] & mode_h)
        h_flag = true;
     j++;
     }
   }

if (cf_flag || cg_flag || ch_flag) {
   cf_flag = cg_flag = ch_flag = false;
   j=num_functions;
   while (((cf_flag == false) || (cg_flag == false) || (ch_flag == false))
                && (j < (num_constraints+num_functions))) {
     if (asv[j] & mode_f)
        cf_flag = true;
     if (asv[j] & mode_g)
        cg_flag = true;
     if (asv[j] & mode_h)
        ch_flag = true;
     j++;
     }
   }
}

} // namespace colin


/// Stream operator to copy AppResponseInfo objects.
inline colin::AppResponseInfo& operator<<(colin::AppResponseInfo& x, const colin::AppResponseInfo& info)
{ x.copy(info); return x; }

#endif
