/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class GeneticAlgorithmSelector

    NOTES:

        See notes of GeneticAlgorithmSelector.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Wed May 21 15:26:36 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the GeneticAlgorithmSelector class.
 */





/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <cfloat>
#include <FitnessRecord.hpp>
#include <GeneticAlgorithmSelector.hpp>
#include <../Utilities/include/Logging.hpp>
#include <../Utilities/include/DesignGroup.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/DesignGroupVector.hpp>






/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA;
using namespace JEGA::Utilities;
using namespace JEGA::Logging;








/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Algorithms {







/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/








/*
================================================================================
Subclass Visible Methods
================================================================================
*/

void
GeneticAlgorithmSelector::RemoveFromGroups(
    const Design* design,
    const DesignGroupVector& groups
    )
{
    EDDY_FUNC_DEBUGSCOPE
    for(DesignGroupVector::const_iterator it(groups.begin());
        it!=groups.end(); ++it)
            (*it)->Erase(design);
}


void
GeneticAlgorithmSelector::SelectNBest(
    JEGA::Utilities::DesignGroupVector& from,
    JEGA::Utilities::DesignGroup& into,
    std::size_t n,
    const FitnessRecord& fitnesses
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // if n is zero, we don't need to do anything at all.
    if(n == 0) return;

    // to make this easy, we will create a multimap of fitnesses to designs
    // and then select the top n while erasing them from from.
    typedef multimap<double, Design*, greater<double> > TopMap;

    TopMap theMap;

    for(DesignGroupVector::const_iterator git(from.begin());
        git!=from.end(); ++git)
    {
        const DesignOFSortSet::const_iterator de((*git)->EndOF());
        for(DesignOFSortSet::const_iterator dit((*git)->BeginOF());
            dit!=de; ++dit)
        {
            double currFit = fitnesses.GetFitness(**dit);
            if(currFit == -DBL_MAX)
            {
                JEGALOG_II(GetLogger(), lquiet(), this,
                    text_entry(lquiet(), GetName() + ": Design encountered "
                        "with no fitness on record.  Ignoring.")
                        )
                continue;
            }

            theMap.insert(TopMap::value_type(currFit, *dit));
        }
    }

    // now go through and take the required number of Designs subject to the
    // size of the map.
    std::size_t numSels = 0;
    for(TopMap::const_iterator it(theMap.begin()); it!=theMap.end(); ++it)
    {
        Design* des = (*it).second;
        into.Insert(des);
        RemoveFromGroups(des, from);
        if(++numSels >= n) return;
    }
}





/*
================================================================================
Subclass Overridable Methods
================================================================================
*/
bool
GeneticAlgorithmSelector::PollForParameters(
    const JEGA::Utilities::ParameterDatabase&
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return true;
}

string
GeneticAlgorithmSelector::GetType(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return "Selector";
}

DesignOFSortSet
GeneticAlgorithmSelector::SelectNBest(
    DesignGroupVector& from,
    size_t n,
    const FitnessRecord& ftns
    )
{
    EDDY_FUNC_DEBUGSCOPE
        
    JEGALOG_II(this->GetLogger(), ldebug(), this,
        ostream_entry(ldebug(), this->GetName() + ": identifying the ") <<
        n << " best designs."
        )

    return this->GeneticAlgorithmSelector::SelectNBest<FitnessPred>(
        from, n, FitnessPred(ftns)
        );
}






/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/
GeneticAlgorithmSelector::GeneticAlgorithmSelector(
    GeneticAlgorithm& algorithm
    ) :
        GeneticAlgorithmOperator(algorithm)
{
    EDDY_FUNC_DEBUGSCOPE
}

GeneticAlgorithmSelector::GeneticAlgorithmSelector(
    const GeneticAlgorithmSelector& copy
    ) :
        GeneticAlgorithmOperator(copy)
{
    EDDY_FUNC_DEBUGSCOPE
}

GeneticAlgorithmSelector::GeneticAlgorithmSelector(
    const GeneticAlgorithmSelector& copy,
    GeneticAlgorithm& algorithm
    ) :
        GeneticAlgorithmOperator(copy, algorithm)
{
    EDDY_FUNC_DEBUGSCOPE
}








/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Algorithms
} // namespace JEGA
