/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Inline methods of class Design.

    NOTES:

        See notes of Design.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Thu May 15 11:04:05 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the inline methods of the Design class.
 */




/*
================================================================================
Includes
================================================================================
*/








/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {


/*
================================================================================
Nested Utility Class Implementations
================================================================================
*/

inline
bool
Design::AttributePred::operator ()(
    const Design& des
    ) const
{
    return operator ()(&des);
}

inline
bool
Design::AttributePred::operator ()(
    const Design* des
    ) const
{
    return des->HasAttributes(_attributes) == _hasOrNot;
}

inline
Design::AttributePred::AttributePred(
    const AttributeMask& attrs,
    bool hasOrNot
    ) :
        _attributes(attrs),
        _hasOrNot(hasOrNot)
{
}

inline
Design::AttributePred::AttributePred(
    std::size_t attrIndex,
    bool hasOrNot
    ) :
        _attributes(1 << attrIndex),
        _hasOrNot(hasOrNot)
{
}

inline
bool
Design::BoolPred::operator ()(
    const Design& des
    ) const
{
    return (des.*_func)() == _evaluatesTo;
}

inline
bool
Design::BoolPred::operator ()(
    const Design* des
    ) const
{
    return operator ()(*des);
}

inline
Design::BoolPred::BoolPred(
    BoolFunc func,
    bool evaluatesTo
    ) :
        _func(func),
        _evaluatesTo(evaluatesTo)
{
}




/*
================================================================================
Inline Mutators
================================================================================
*/
inline
void
Design::SetObjective(
    std::size_t num,
    double val
    )
{
    _objectives[num] = val;
}

inline
void
Design::SetConstraint(
    std::size_t num,
    double val
    )
{
    _constraints[num] = val;
}

inline
void
Design::SetNextClone(
    Design* clone
    )
{
    _nClone = clone;
}

inline
void
Design::SetPreviousClone(
    Design* clone
    )
{
    _pClone = clone;
}

inline
void
Design::SetRawAttributes(
    const AttributeMask& attr
    )
{
    _attributes = attr;
}




/*
================================================================================
Inline Accessors
================================================================================
*/

inline
double
Design::GetVariableRep(
    std::size_t num
    ) const
{
    return _variables[num];
}

inline
double
Design::GetObjective(
    std::size_t num
    ) const
{
    return _objectives[num];
}

inline
double
Design::GetConstraint(
    std::size_t num
    ) const
{
    return _constraints[num];
}

inline
const Design::AttributeMask&
Design::GetRawAttributes(
    ) const
{
    return _attributes;
}

inline
DesignTarget&
Design::GetDesignTarget(
    ) const
{
    return _target;
}

inline
Design*
Design::GetNextClone(
    ) const
{
    return _nClone;
}

inline
Design*
Design::GetPreviousClone(
    ) const
{
    return _pClone;
}


/*
================================================================================
Inline Public Methods
================================================================================
*/
inline
bool
Design::HasAttributes(
    const AttributeMask& attributes
    ) const
{
    return _attributes.contains(attributes);
}

inline
bool
Design::HasAttribute(
    std::size_t attribute_index
    ) const
{
    return _attributes.get_bit(attribute_index);
}

inline
bool
Design::IsEvaluated(
    ) const
{
    return HasAttribute(Evaluated);
}

inline
bool
Design::IsFeasible(
    ) const
{
    return SatisfiesConstraints() && SatisfiesBounds();
}

inline
bool
Design::SatisfiesConstraints(
    ) const
{
    return HasAttribute(FeasibleConstraints);
}

inline
bool
Design::SatisfiesBounds(
    ) const
{
    return HasAttribute(FeasibleBounds);
}

inline
bool
Design::IsIllconditioned(
    ) const
{
    return HasAttribute(Illconditioned);
}

inline
void
Design::SetVariableRep(
    std::size_t num,
    double rep
    )
{
    _variables[num] = rep;
}

inline
bool
Design::IsCloned(
    ) const
{
    return (_pClone != 0x0 || _nClone != 0x0);
}

inline
void
Design::ModifyAttributes(
    const AttributeMask& attributes,
    bool on
    )
{
    (on) ? (_attributes |= attributes) : (_attributes &= ~attributes);
}

inline
void
Design::ModifyAttribute(
    std::size_t attribute_index,
    bool on
    )
{
    _attributes.set_bit(attribute_index, on);
}

inline
void
Design::SetEvaluated(
    bool lval
    )
{
    ModifyAttribute(Evaluated, lval);
}

inline
void
Design::SetSatisfiesBounds(
    bool lval
    )
{
    ModifyAttribute(FeasibleBounds, lval);
}

inline
void
Design::SetSatisfiesConstraints(
    bool lval
    )
{
    ModifyAttribute(FeasibleConstraints, lval);
}

inline
void
Design::SetIllconditioned(
    bool lval
    )
{
    ModifyAttribute(Illconditioned, lval);
}

inline
void
Design::ResetAttributes(
    )
{
    SetRawAttributes(0);
}





/*
================================================================================
Inline Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Inline Private Methods
================================================================================
*/








/*
================================================================================
Inline Structors
================================================================================
*/








/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA
