/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

#include <cstdlib>
#include <iostream>
#include <fstream>
#include <vector>
#include <string>
using namespace std;


int main(int argc, char** argv)
{
  ifstream fin(argv[1]);
  if (!fin) {
    cerr << "\nError: failure opening " << argv[1] << endl;
    exit(-1);
  }
  size_t i, j, num_vars, num_fns, num_deriv_vars;
  string vars_text, fns_text, dvv_text;

  // Get the parameter vector and ignore the labels
  fin >> num_vars >> vars_text;
  vector<double> x(num_vars);
  for (i=0; i<num_vars; i++) {
    fin >> x[i];
    fin.ignore(256, '\n');
  }

  // Get the ASV vector and ignore the labels
  fin >> num_fns >> fns_text;
  vector<int> ASV(num_fns);
  for (i=0; i<num_fns; i++) {
    fin >> ASV[i];
    fin.ignore(256, '\n');
  }

  // Get the DVV vector and ignore the labels
  fin >> num_deriv_vars >> dvv_text;
  vector<int> DVV(num_deriv_vars);
  for (i=0; i<num_deriv_vars; i++) {
    fin >> DVV[i];
    fin.ignore(256, '\n');
  }

  if (num_vars != 5 || num_fns != 2) {
    cerr << "Error: wrong number of inputs/outputs in short_column." << endl;
    exit(-1);
  }

  // Compute the results and output them directly to argv[2] (the NO_FILTER
  // option is used).  Response tags are optional; output them for ease
  // of results readability.
  ofstream fout(argv[2]);
  if (!fout) {
    cerr << "\nError: failure creating " << argv[2] << endl;
    exit(-1);
  }
  fout.precision(15); // 16 total digits
  fout.setf(ios::scientific);
  fout.setf(ios::right);

  // b = x[0] = column base   (design var.)
  // h = x[1] = column height (design var.)
  // P = x[2] (normal uncertain var.)
  // M = x[3] (normal uncertain var.)
  // Y = x[4] (lognormal uncertain var.)
  double b = x[0], h = x[1], P = x[2], M = x[3], Y = x[4],
         b_sq = b*b, h_sq = h*h, P_sq = P*P, Y_sq = Y*Y;

  // **** f (objective = bh = cross sectional area):
  if (ASV[0] & 1)
    fout << "                     " << b*h << " f\n";

  // **** g (limit state = short column response):
  if (ASV[1] & 1)
    fout << "                     "
	 << 1. - 4.*M/(b*h_sq*Y) - P_sq/(b_sq*h_sq*Y_sq) << " g\n";

  // **** df/dx (w.r.t. active/uncertain variables):
  if (ASV[0] & 2) {
    fout << "[ ";
    for (i=0; i<num_deriv_vars; i++) {
      int var_index = DVV[i] - 1;
      switch (var_index) {
      case 0: // design variable derivative
	fout << h << ' ';
	break;
      case 1: // design variable derivative
	fout << b << ' ';
	break;
      default: // uncertain variable derivative
	fout << "0. ";
	break;
      }
    }
    fout << "]\n";
  }

  // **** dg/dx (w.r.t. active/uncertain variables):
  if (ASV[1] & 2) {
    fout << "[ ";
    for (i=0; i<num_deriv_vars; i++) {
      int var_index = DVV[i] - 1;
      switch (var_index) {
      case 0: // design variable derivative
	fout << 4.*M/(b_sq*h_sq*Y) + 2.*P_sq/(b_sq*b*h_sq*Y_sq) << ' ';
	break;
      case 1: // design variable derivative
	fout << 8.*M/(b*h_sq*h*Y)  + 2.*P_sq/(b_sq*h_sq*h*Y_sq) << ' ';
	break;
      case 2: // uncertain variable derivative
	fout << -2.*P/(b_sq*h_sq*Y_sq) << ' ';
	break;
      case 3: // uncertain variable derivative
	fout << -4./(b*h_sq*Y) << ' ';
	break;
      case 4: // uncertain variable derivative
	fout << 4.*M/(b*h_sq*Y_sq) + 2.*P_sq/(b_sq*h_sq*Y_sq*Y) << ' ';
	break;
      }
    }
    fout << "]\n";
  }

  // **** d^2f/dx^2: (SORM)
  if (ASV[0] & 4) {
    fout << "[[ ";
    for (i=0; i<num_deriv_vars; i++) {
      int var_index_i = DVV[i] - 1;
      for (j=0; j<num_deriv_vars; j++) {
	int var_index_j = DVV[j] - 1;
	if ( (var_index_i == 0 && var_index_j == 1) ||
	     (var_index_i == 1 && var_index_j == 0) )
	  fout << "1. ";
	else
	  fout << "0. ";
      }
    }
    fout << "]]\n";
  }

  // **** d^2g/dx^2: (SORM)
  if (ASV[1] & 4) {
    fout << "[[ ";
    for (i=0; i<num_deriv_vars; i++) {
      int var_index_i = DVV[i] - 1;
      for (j=0; j<num_deriv_vars; j++) {
	int var_index_j = DVV[j] - 1;
	if (var_index_i == 0 && var_index_j == 0)          // d^2g/db^2
	  fout << -8.*M/(b_sq*b*h_sq*Y) - 6.*P_sq/(b_sq*b_sq*h_sq*Y_sq) << ' ';
	else if ( (var_index_i == 0 && var_index_j == 1) ||
		  (var_index_i == 1 && var_index_j == 0) ) // d^2g/dbdh
	  fout << -8.*M/(b_sq*h_sq*h*Y) - 4.*P_sq/(b_sq*b*h_sq*h*Y_sq) << ' ';
	else if (var_index_i == 1 && var_index_j == 1)     // d^2g/dh^2
	  fout << -24.*M/(b*h_sq*h_sq*Y) - 6.*P_sq/(b_sq*h_sq*h_sq*Y_sq) << ' ';
	else if (var_index_i == 2 && var_index_j == 2)     // d^2g/dP^2
	  fout << -2./(b_sq*h_sq*Y_sq) << ' ';
	else if ( (var_index_i == 2 && var_index_j == 3) ||
		  (var_index_i == 3 && var_index_j == 2) ) // d^2g/dPdM
	  fout << "0. ";
	else if ( (var_index_i == 2 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 2) ) // d^2g/dPdY
	  fout << 4.*P/(b_sq*h_sq*Y_sq*Y) << ' ';
	else if (var_index_i == 3 && var_index_j == 3)     // d^2g/dM^2
	  fout << "0. ";
	else if ( (var_index_i == 3 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 3) ) // d^2g/dMdY
	  fout << 4./(b*h_sq*Y_sq) << ' ';
	else if (var_index_i == 4 && var_index_j == 4)     // d^2g/dY^2
	  fout << -8.*M/(b*h_sq*Y_sq*Y) - 6.*P_sq/(b_sq*h_sq*Y_sq*Y_sq) << ' ';
	else { // unsupported cross-derivative
	  cerr << "Error: unsupported Hessian cross term in short_column."
	       << endl;
	  exit(-1);
	}
      }
    }
    fout << "]]\n";
  }

  fout.flush();
  fout.close();
  return 0;
}
