/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Description: The main DAKOTA program.
//-              The ProblemDescDB and Strategy are instantiated and
//-              the virtual function run_strategy is invoked.
//- Owner:       Mike Eldred
//- Checked by:
//- Version: $Id: main.C 5396 2008-10-21 19:53:43Z mseldre $

/** \file main.C
    \brief file containing the main program for DAKOTA */

#if defined(_MSC_VER) || defined(__MINGW32__)
#include <windows.h>
#endif
#include "system_defs.h"
#include "ParallelLibrary.H"
#include "CommandLineHandler.H"
#include "ProblemDescDB.H"
#include "DakotaStrategy.H"
using namespace Dakota;


/// The main DAKOTA program.

/** Manage command line inputs, input files, restart file(s), output
    streams, and top level parallel iterator communicators.  Instantiate
    the Strategy and invoke its run_strategy() virtual function. */

int main(int argc, char* argv[])
{
#ifdef __MINGW32__
  signal(WM_QUIT, abort_handler);
  signal(WM_CHAR, abort_handler);
#else
  signal(SIGKILL, abort_handler);
  signal(SIGTERM, abort_handler);
#endif
  signal(SIGINT,  abort_handler);

#ifdef MPI_DEBUG
  // hold parallel job prior to MPI_Init() in order to attach debugger to
  // master process.  Then step past ParallelLibrary instantiation and attach
  // debugger to other processes.
  int test;
  cin >> test;
#endif

  // Instantiate/initialize the parallel library, command line handler, and
  // problem description database objects.  The ParallelLibrary constructor
  // calls MPI_Init() if a parallel launch is detected.  This must precede
  // CommandLineHandler initialization/parsing so that MPI may extract its
  // command line arguments first, prior to DAKOTA command line extractions.
  ParallelLibrary parallel_lib(argc, argv);
  CommandLineHandler cmd_line_handler(argc, argv);
  ProblemDescDB problem_db(parallel_lib, cmd_line_handler);

  // Manage input file parsing, output redirection, and restart processing.
  // Since all processors need the database, manage_inputs() does not require
  // iterator partitions and it can precede init_iterator_communicators()
  // (a simple world bcast is sufficient).  Output/restart management does
  // utilize iterator partitions, so manage_outputs_restart() must follow
  // init_iterator_communicators() within the Strategy constructor
  // (output/restart options may only be specified at this time).
  problem_db.manage_inputs(cmd_line_handler);
  parallel_lib.specify_outputs_restart(cmd_line_handler);

  // Instantiate the Strategy object (which instantiates all Model and Iterator
  // objects) using the parsed information in problem_db.  All MPI communicator
  // partitions are created during strategy construction.
  Strategy selected_strategy(problem_db);

  // Run the strategy
  if (cmd_line_handler.retrieve("check"))
    Cout << "\nDry run completed: input parsed and objects instantiated.\n"
	 << endl;
  else {
    problem_db.lock(); // prevent run-time DB queries
    selected_strategy.run_strategy();
  }

  return 0;
}
