/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Description: An API for launching DAKOTA from a DLL.
//- Owner:       Bill Hart
//- Checked by:
//- Version: $Id$

/** \file dll_api.C
    \brief This file contains a DakotaRunner class, which launches DAKOTA. */

#if defined(_MSC_VER) || defined(__MINGW32__)
#define BUILDING_DAKOTA_DLL
#include <windows.h>
#endif
#include "system_defs.h"
#include "ParallelLibrary.H"
#include "CommandLineHandler.H"
#include "ProblemDescDB.H"
#include "DakotaStrategy.H"
#ifdef DAKOTA_PLUGIN
#include "DakotaModel.H"
#include "DakotaInterface.H"
#include "PluginDirectFnApplicInterface.H"
#endif
#include "global_defs.h"
#include "dll_api.h"

using namespace Dakota;

namespace {

void signal_init()
{
#ifdef __MINGW32__
  signal(WM_QUIT, abort_handler);
  signal(WM_CHAR, abort_handler);
#else
  signal(SIGKILL, abort_handler);
  signal(SIGTERM, abort_handler);
#endif
  signal(SIGINT,  abort_handler);
}




class DakotaRunner
{
public:

  ///
  DakotaRunner(int _argc, char** _argv)
	: parallel_lib(0)
	{
	signal_init();
	argc=_argc+2;
	argv = new char* [argc];
	for (int i=0; i<argc-2; i++) {
	  argv[i] = new char [strlen(_argv[i])+1];
	  strcpy(_argv[i],argv[i]);
	  }
	parallel_lib=0;
	}

  ///
  ~DakotaRunner()
	{
	if (parallel_lib) {
		delete parallel_lib;
		delete cmd_line_handler;
		delete problem_db;
		}
	for (int i=0; i<argc-2; i++)
		delete argv[i];
	delete argv;
	}

  ///
  void read_input(char* dakotaInput)
	{
	argv[argc-2] = "-input";
	argv[argc-1] = dakotaInput;

  	// problem description database objects.  The ParallelLibrary constructor
  	// calls MPI_Init() if a parallel launch is detected.  This must precede
  	// CommandLineHandler initialization/parsing so that MPI may extract its
  	// command line arguments first, prior to DAKOTA command line extractions.
	if (parallel_lib) {
		delete parallel_lib;
		delete cmd_line_handler;
		delete problem_db;
		}
	parallel_lib = new ParallelLibrary(argc,argv);
	cmd_line_handler = new CommandLineHandler(argc,argv);
  	problem_db = new ProblemDescDB(*parallel_lib, *cmd_line_handler);
  	// Manage input file parsing, output redirection, and restart processing.
  	// Since all processors need the database, manage_inputs() does not require
  	// iterator partitions and it can precede init_iterator_communicators()
  	// (a simple world bcast is sufficient).  Output/restart management does
  	// utilize iterator partitions, so manage_outputs_restart() must follow
  	// init_iterator_communicators() within the Strategy constructor
  	// (output/restart options may only be specified at this time).
  	problem_db->manage_inputs(*cmd_line_handler);
  	parallel_lib->specify_outputs_restart(*cmd_line_handler);
	}

  ///
  void start();

  ParallelLibrary* parallel_lib;
  CommandLineHandler* cmd_line_handler;
  ProblemDescDB* problem_db;

  int argc;
  char** argv;
};


void DakotaRunner::start()
{
  // Instantiate the Strategy object (which instantiates all Model and Iterator
  // objects) using the parsed information in problem_db.  All MPI communicator
  // partitions are created during strategy construction.
  Strategy selected_strategy(*problem_db);

  // Any library mode plug-ins would go here.
  // Refer to the library mode documentation in the Developers Manual.
#ifdef DAKOTA_PLUGIN
  ModelList& models = problem_db->model_list();
  for (ModelLIter ml_iter = models.begin(); ml_iter != models.end(); ml_iter++){
    Interface& interface = ml_iter->interface();
    if ( interface.interface_type() == "direct" &&
	 interface.analysis_drivers().contains("plugin_rosenbrock") ) {
      // set the DB nodes to that of the existing Model specification
      problem_db->set_db_model_nodes(ml_iter->model_id());
      // plug in the new derived Interface object
      interface.assign_rep(new SIM::DirectFnApplicInterface(*problem_db), false);
    }
  }
#endif

  // Run the strategy
  if (cmd_line_handler->retrieve("check"))
    Cout << "\nDry run completed: input parsed and objects instantiated.\n"
	 << endl;
  else {
    problem_db->lock(); // prevent run-time DB queries
    selected_strategy.run_strategy();
  }
}



map<int,DakotaRunner*> runners;


}

extern "C" void DAKOTA_DLL_FN dakota_create(int* dakota_ptr_int)
{ 
DakotaRunner* pDakota = new DakotaRunner(0, NULL);
int id = static_cast<int>(runners.size());
runners[id] = pDakota;
*dakota_ptr_int = id;
}

extern "C" void DAKOTA_DLL_FN dakota_readInput(int id, char* dakotaInput)
{ 
runners[id]->read_input(dakotaInput);
}

extern "C" void DAKOTA_DLL_FN dakota_start(int id)
{
runners[id]->start();
}

extern "C" void DAKOTA_DLL_FN dakota_destroy (int id)
{ 
delete runners[id];
runners.erase(id);
}

extern "C" void DAKOTA_DLL_FN dakota_stop(int* id)
{
/** TODO: trick application to quit through the syscall interface. **/
}

extern "C" const char* DAKOTA_DLL_FN dakota_getStatus(int id)
{
static std::string tmp;
tmp = "<DakotaOutput>None</DakotaOutput>";
return tmp.c_str();
}

extern "C" int get_mc_ptr_int()
{
#ifdef DAKOTA_MODELCENTER
return Dakota::mc_ptr_int;
#else
return 0;
#endif
}

extern "C" void set_mc_ptr_int(int ptr_int)
{
#ifdef DAKOTA_MODELCENTER
Dakota::mc_ptr_int = ptr_int;
#endif
}

