/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

#ifndef DATA_TYPES_H
#define DATA_TYPES_H

#include "DakotaString.H"
#include "DakotaList.H"
#include "DakotaArray.H"
#include "DakotaVector.H"
#include "DakotaMatrix.H"

#include "Teuchos_SerialDenseVector.hpp"
#ifndef DLSfuncs_H
#include "Teuchos_SerialDenseSolver.hpp"
#include "Teuchos_SerialSpdDenseSolver.hpp"
#endif

#ifdef DAKOTA_UTILIB
#include <utilib/NumArray.h>
// fwd declarations cause duplicate definition errors
//class RealVector;
//class IntVector;
#endif // DAKOTA_UTILIB

#ifdef DAKOTA_NEWMAT
namespace NEWMAT {
class ColumnVector;
class SymmetricMatrix;
class Matrix;
}
#endif // DAKOTA_NEWMAT

#ifdef DAKOTA_TNT
#include "tnt.h"
#include "vec.h"
#include "cmat.h"
#endif // DAKOTA_TNT

#ifdef DAKOTA_DDACE
class DDaceSamplePoint;
#endif // DAKOTA_DDACE


namespace Dakota {

// avoid problems with circular dependencies by using fwd declarations
class Iterator;
class Model;
class Variables;
class Interface;
class Response;
class ActiveSet;
class ParamResponsePair;
class ParallelLevel;
class ParallelConfiguration;


// -----------------------------------
// Aliases for fundamental data types:
// -----------------------------------
typedef double Real;


// -----------------------------------
// Numerical arrays (serial/parallel):
// -----------------------------------
// numerical with STL: (deprecated)
typedef Vector<Real>               RealVector;
typedef Vector<int>                IntVector;
typedef BaseVector<Real>           RealBaseVector;
typedef Matrix<Real>               RealMatrix;
typedef Matrix<int>                IntMatrix;

// numerical with TNT:
//typedef TNT::Vector<Real>                         RealVector;
//typedef TNT::Matrix<Real>                         RealMatrix;

// numerical with Teuchos
typedef Teuchos::SerialDenseVector<int, Real>    RealDenseVector;
typedef Teuchos::SerialDenseVector<int, int>     IntDenseVector;
typedef Teuchos::SerialDenseMatrix<int, Real>    RealDenseMatrix;
typedef Teuchos::SerialSymDenseMatrix<int, Real> RealSymDenseMatrix;
//typedef Teuchos::Vector                           RealParallelVector
//typedef Teuchos::MultiVector                      RealParallelMatrix


// ---------------------------------
// Numerical solvers (serial dense):
// ---------------------------------
#ifndef DLSfuncs_H
typedef Teuchos::SerialDenseSolver<int, Real>    RealDenseSolver;
typedef Teuchos::SerialSpdDenseSolver<int, Real> RealSpdDenseSolver;
#endif


// ---------------------------------------
// Admin/bookkeeping arrays (serial only):
// ---------------------------------------
typedef std::deque<bool>           BoolDeque; // See Effective STL (Meyers), #18
typedef Array<BoolDeque>           BoolDequeArray;
typedef Array<Real>                RealArray;
//typedef Array<RealArray>         Real2DArray;
//typedef Array<Real2DArray>       Real3DArray;
typedef Array<int>                 IntArray;
typedef Array<IntArray>            Int2DArray;
typedef Array<unsigned int>        UIntArray;
typedef Array<short>               ShortArray;
typedef Array<unsigned short>      UShortArray;
typedef Array<UShortArray>         UShort2DArray;
typedef Array<UShort2DArray>       UShort3DArray;
typedef Array<size_t>              SizetArray;
typedef Array<SizetArray>          Sizet2DArray;
typedef Array<String>              StringArray;
typedef Array<StringArray>         String2DArray;
typedef Array<RealVector>          RealVectorArray;
typedef Array<RealVectorArray>     RealVector2DArray;
typedef Array<RealBaseVector>      RealBaseVectorArray;
typedef Array<RealMatrix>          RealMatrixArray;
typedef Array<RealDenseVector>     RealDenseVectorArray;
typedef Array<RealDenseMatrix>     RealDenseMatrixArray;
typedef Array<RealSymDenseMatrix>  RealSymDenseMatrixArray;
typedef Array<Variables>           VariablesArray;
typedef Array<Response>            ResponseArray;
typedef Array<ParamResponsePair>   PRPArray;
typedef Array<Model>               ModelArray;
typedef Array<Iterator>            IteratorArray;

typedef List<bool>                 BoolList;
typedef List<int>                  IntList;
typedef List<size_t>               SizetList;
typedef List<Real>                 RealList;
typedef List<String>               StringList;
typedef List<RealVector>           RealVectorList;
typedef List<Variables>            VariablesList;
typedef List<Interface>            InterfaceList;
typedef List<Response>             ResponseList;
typedef List<ParamResponsePair>    PRPList;
typedef List<Model>                ModelList;
typedef List<Iterator>             IteratorList;
//typedef List<ParallelLevel>         ParLevList;
//typedef List<ParallelConfiguration> ParConfigList;

typedef std::set<int>              IntSet;
typedef std::set<Real>             RealSet;
typedef std::map<int, short>       IntShortMap;
typedef std::map<int, int>         IntIntMap;
typedef std::map<int, RealVector>  IntRealVectorMap;
typedef std::map<int, ActiveSet>   IntActiveSetMap;
typedef std::map<int, Variables>   IntVariablesMap;
typedef std::map<int, Response>    IntResponseMap;
typedef std::map<IntArray, size_t> IntArraySizetMap;


// ---------
// Iterators
// ---------
typedef IntList::iterator                     ILIter;
typedef IntList::const_iterator               ILCIter;
typedef SizetList::iterator                   StLIter;
typedef SizetList::const_iterator             StLCIter;
typedef RealList::iterator                    RLIter;
typedef RealList::const_iterator              RLCIter;
typedef StringList::iterator                  StringLIter;
typedef StringList::const_iterator            StringLCIter;
typedef RealVectorList::iterator              RVLIter;
typedef RealVectorList::const_iterator        RVLCIter;
typedef VariablesList::iterator               VarsLIter;
typedef InterfaceList::iterator               InterfLIter;
typedef ResponseList::iterator                RespLIter;
typedef PRPList::iterator                     PRPLIter;
typedef PRPList::const_iterator               PRPLCIter;
typedef ModelList::iterator                   ModelLIter;
typedef IteratorList::iterator                IterLIter;
typedef List<ParallelLevel>::iterator         ParLevLIter;
typedef List<ParallelConfiguration>::iterator ParConfigLIter;

typedef IntSet::iterator                      ISIter;
typedef IntSet::const_iterator                ISCIter;
typedef IntShortMap::iterator                 IntShMIter;
typedef IntIntMap::iterator                   IntIntMIter;
typedef IntIntMap::const_iterator             IntIntMCIter;
typedef IntRealVectorMap::iterator            IntRVMIter;
typedef IntActiveSetMap::iterator             IntASMIter;
typedef IntVariablesMap::iterator             IntVarsMIter;
typedef IntVariablesMap::const_iterator       IntVarsMCIter;
typedef IntResponseMap::iterator              IntRespMIter;
typedef IntResponseMap::const_iterator        IntRespMCIter;


// ---------------------------------
// Templated typedefs (serial only):
// ---------------------------------
//typedef Array<Array<T> >          2DArray<T>;
//typedef Array<Vector<T> >         VectorArray<T>;


// ----------------------------------
// Global data function declarations:
// ----------------------------------

// Equality operator definitions for data types

/// equality operator for RealVector
bool operator==(const RealVector& drv1, const RealVector& drv2);
/// equality operator for IntVector
bool operator==(const IntVector& div1,  const IntVector& div2);
/// equality operator for IntArray
bool operator==(const IntArray& dia1,   const IntArray& dia2);
/// equality operator for ShortArray
bool operator==(const ShortArray& dsa1, const ShortArray& dsa2);
/// equality operator for RealMatrix
bool operator==(const RealMatrix& drm1, const RealMatrix& drm2);
/// equality operator for RealMatrixArray
bool operator==(const RealMatrixArray& drma1, const RealMatrixArray& drma2);
/// equality operator for StringArray
bool operator==(const StringArray& dsa1, const StringArray& dsa2);

// Non-templated conversion fns between DAKOTA/NEWMAT/TEUCHOS/DDACE data types

#ifdef DAKOTA_NEWMAT
// NOTE: newmat vector/matrix types hold Reals and are not templates
/// copy NEWMAT::ColumnVector to RealBaseVector
void copy_data(const NEWMAT::ColumnVector& cv,   RealBaseVector& drbv);
/// copy RealBaseVector to NEWMAT::ColumnVector
void copy_data(const RealBaseVector& drbv, NEWMAT::ColumnVector& cv);
/// copy RealArray to NEWMAT::ColumnVector
void copy_data(const RealArray&  dra,      NEWMAT::ColumnVector& cv);
/// copy RealMatrix to NEWMAT::SymmetricMatrix
void copy_data(const RealMatrix& drm,      NEWMAT::SymmetricMatrix& sm);
/// copy RealMatrix to NEWMAT::Matrix
void copy_data(const RealMatrix& drm,      NEWMAT::Matrix& m);
/// copy NEWMAT::ColumnVector to RealDenseVector
void copy_data(const NEWMAT::ColumnVector& cv, RealDenseVector& rdv);
#endif // DAKOTA_NEWMAT

#ifdef DAKOTA_DDACE
/// copy DDACE point to RealVector
void copy_data(const DDaceSamplePoint& dsp, RealVector& drva);
/// copy DDACE point array to RealVectorArray
void copy_data(const std::vector<DDaceSamplePoint>& dspa,
	       RealVectorArray& drva);
/// copy DDACE point array to Real*
void copy_data(const std::vector<DDaceSamplePoint>& dspa, Real* ptr,
	       const int ptr_len);
#endif // DAKOTA_DDACE


// -------------------------
// inlined utility functions
// -------------------------

// Inequality operator definitions for data types

/// inequality operator for RealVector
inline bool operator!=(const RealVector& drv1, const RealVector& drv2)
{ return !(drv1 == drv2); }

/// inequality operator for IntVector
inline bool operator!=(const IntVector& div1, const IntVector& div2)
{ return !(div1 == div2); }

/// inequality operator for IntArray
inline bool operator!=(const IntArray& dia1, const IntArray& dia2)
{ return !(dia1 == dia2); }

/// inequality operator for ShortArray
inline bool operator!=(const ShortArray& dsa1, const ShortArray& dsa2)
{ return !(dsa1 == dsa2); }

/// inequality operator for RealMatrix
inline bool operator!=(const RealMatrix& drm1, const RealMatrix& drm2)
{ return !(drm1 == drm2); }

/// inequality operator for RealMatrixArray
inline bool operator!=(const RealMatrixArray& drma1,
		       const RealMatrixArray& drma2)
{ return !(drma1 == drma2); }

/// inequality operator for StringArray
inline bool operator!=(const StringArray& dsa1, const StringArray& dsa2)
{ return !(dsa1 == dsa2); }

// Utility functions for creating string arrays

/// create a label by appending a numerical tag to the root_label
inline void build_label(String& label, const String& root_label, size_t tag)
{
  char string[10];
  sprintf(string, "%i", (int)tag);
  label = root_label + String(string); // append tag to root label
}

/// create an array of labels by tagging root_label for each entry in
/// label_array.  Uses build_label().
inline void build_labels(StringArray& label_array, const String& root_label)
{
  size_t len = label_array.length();
  for (size_t i=0; i<len; i++)
    build_label(label_array[i], root_label, i+1);
}

/// create a partial array of labels by tagging root_label for a subset
/// of entries in label_array.  Uses build_label().
inline void build_labels_partial(StringArray& label_array,
				 const String& root_label, size_t start_index,
				 size_t num_items)
{
  for (size_t i=0; i<num_items; i++)
    build_label(label_array[start_index+i], root_label, i+1);
}

// Utility functions for creating merged vectors

/// aggregate continuous and discrete arrays into a single merged array
inline void aggregate_merged(const RealVector& c_array,
			     const IntVector&  d_array, RealVector& m_array)
{
  size_t i, num_cv = c_array.length(), num_dv = d_array.length(), 
    num_mv = num_cv + num_dv;
  if (m_array.length() != num_mv)
    m_array.reshape(num_mv);
  for (i=0; i<num_cv; i++)
    m_array[i] = c_array[i];
  for (i=0; i<num_dv; i++)
    m_array[num_cv+i] = (Real)d_array[i];
}

/// separate a merged array into continuous and discrete arrays
inline void separate_merged(const RealVector& m_array, RealVector& c_array,
			    IntVector& d_array)
{
  size_t i, num_cv = c_array.length(), num_dv = d_array.length();
  if ( num_cv + num_dv != m_array.length() ) {
    Cerr << "Error: Length of merged array passed to separate_merged() does "
	 << "not equal\n       sum of lengths of continuous and discrete "
	 << "arrays to be updated.\n       Resizing is ambiguous." << endl;
    abort_handler(-1);
  }
  for (i=0; i<num_cv; i++)
    c_array[i] = m_array[i];
  for (i=0; i<num_dv; i++)
    d_array[i] = (int)m_array[num_cv+i];
}


// -----------------------
// templated I/O functions
// -----------------------

/// global ostream insertion operator for std::set
template <class T>
inline ostream& operator<<(ostream& s, const std::set<T>& data)
{
  for (typename std::set<T>::const_iterator cit = data.begin();
       cit != data.end(); cit++)
    s << "                     " << *cit << '\n';
  return s;
}

/// global MPIUnpackBuffer extraction operator for std::set
template <class T>
inline MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s, std::set<T>& data)
{
  data.clear();
  size_t len;
  s >> len;
  T val;
  for (size_t i=0; i<len; i++){
    s >> val; 
    data.insert(val);
  }
  return s;
}

/// global MPIPackBuffer insertion operator for std::set
template <class T>
inline MPIPackBuffer& operator<<(MPIPackBuffer& s, const std::set<T>& data)
{
  size_t len = data.size();
  s << len;
  for (typename std::set<T>::const_iterator cit = data.begin();
       cit != data.end(); cit++)
    s << *cit;
  return s;
}

/// global MPIPackBuffer insertion operator for Teuchos::SerialDenseVector
template <typename OrdinalType, typename ScalarType> 
MPIPackBuffer& operator<<(MPIPackBuffer& s,
  const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& data)
{
  OrdinalType i, n = data.length();
  s << n;
  for (i=0; i<n; i++)
    s << data[i];
  return s;
}

/// global MPIPackBuffer insertion operator for Teuchos::SerialDenseMatrix
template <typename OrdinalType, typename ScalarType> 
MPIPackBuffer& operator<<(MPIPackBuffer& s,
  const Teuchos::SerialDenseMatrix<OrdinalType, ScalarType>& data)
{
  OrdinalType i, j, n = data.numRows(), m = data.numCols();
  s << n << m;
  for (i=0; i<n; i++)
    for (j=0; j<m; j++)
      s << data(i,j);
  return s;
}

/// global MPIPackBuffer insertion operator for Teuchos::SerialSymDenseMatrix
template <typename OrdinalType, typename ScalarType> 
MPIPackBuffer& operator<<(MPIPackBuffer& s,
  const Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>& data)
{
  OrdinalType i, j, n = data.numRows();
  s << n;
  for (i=0; i<n; i++)
    for (j=0; j<=i; j++)
      s << data(i,j);
  return s;
}

/// global MPIUnpackBuffer extraction operator for Teuchos::SerialDenseVector
template <typename OrdinalType, typename ScalarType> 
MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s,
  Teuchos::SerialDenseVector<OrdinalType, ScalarType>& data)
{
  OrdinalType i, n;
  s >> n;
  data.sizeUninitialized(n);
  for(i=0; i<n; i++)
    s >> data[i];
  return s;
}

/// global MPIUnpackBuffer extraction operator for Teuchos::SerialDenseMatrix
template <typename OrdinalType, typename ScalarType> 
MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s,
  Teuchos::SerialDenseMatrix<OrdinalType, ScalarType>& data)
{
  OrdinalType i, j, n, m;
  s >> n >> m;
  data.shapeUninitialized(n, m);
  for (i=0; i<n; i++)
    for (j=0; j<m; j++)
      s >> data(i,j);
  return s;
}

/// global MPIUnpackBuffer extraction operator for Teuchos::SerialSymDenseMatrix
template <typename OrdinalType, typename ScalarType> 
MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s,
  Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>& data)
{
  OrdinalType i, j, n;
  s >> n;
  data.shapeUninitialized(n);
  for (i=0; i<n; i++)
    for (j=0; j<=i; j++)
      s >> data(i,j);
  return s;
}

// ------------------------------
// templated conversion functions
// ------------------------------

// Templated conversion functions between/among DAKOTA data types and
// built in/pointer data types

/// copy T* to Vector<T>
template <class T>
void copy_data(const T* ptr, const int ptr_len, Vector<T>& dv)
{
  if (dv.length() != ptr_len)
    dv.reshape(ptr_len);
  for (size_t i=0; i<ptr_len; i++)
    dv[i] = ptr[i];
}

/// copy T* to BaseVector<T>
template <class T>
void copy_data(const T* ptr, const int ptr_len, BaseVector<T>& dbv)
{
  if (dbv.length() != ptr_len)
    dbv.reshape(ptr_len);
  for (size_t i=0; i<ptr_len; i++)
    dbv[i] = ptr[i];
}

/// copy T* to Matrix<T>
template <class T> 
void copy_data(const T* ptr, const int ptr_len, const String& ptr_type,
	       Matrix<T>& dm, size_t nr, size_t nc)
{
  if (nr && nc) { // both specified
    if (ptr_len != nr*nc) {
      Cerr << "Error: pointer allocation (" << ptr_len << ") does not equal "
	   << "nr*nc (" << nr << '*' << nc << ") in copy_data(T*, "
	   << "Dakota::Matrix<T>)." << endl;
      abort_handler(-1);
    }
  }
  else if (nr) { // only nr is non-zero
    if (ptr_len%nr) {
      Cerr << "Error: pointer allocation (" << ptr_len << ") not evenly "
	   << "divisible by number of rows (" << nr << ") in copy_data(T*, "
	   << "Dakota::Matrix<T>)." << endl;
      abort_handler(-1);
    }
    nc = ptr_len/nr;
  }
  else if (nc) { // only nc is non-zero
    if (ptr_len%nc) {
      Cerr << "Error: pointer allocation (" << ptr_len << ") not evenly "
	   << "divisible by number of columns (" << nc << ") in copy_data(T*, "
	   << "Dakota::Matrix<T>)." << endl;
      abort_handler(-1);
    }
    nr = ptr_len/nc;
  }
  else { // neither specified
    Cerr << "Error: either nr or nc must be specified in copy_data(T*, "
	 << "Dakota::Matrix<T>)." << endl;
    abort_handler(-1);
  }
  if (dm.num_rows() != nr || dm.num_columns() != nc)
    dm.reshape_2d(nr, nc);
  size_t i, j;
  int cntr = 0;
  if (toLower(ptr_type) == "c") { // case insensitive
    for (i=0; i<nr; i++)   // loop over rows
      for (j=0; j<nc; j++) // loop over columns
        dm[i][j] = ptr[cntr++];
  }
  else if (toLower(ptr_type) == "fortran") { // case insensitive
    for (j=0; j<nc; j++)   // loop over columns
      for (i=0; i<nr; i++) // loop over rows
        dm[i][j] = ptr[cntr++];
  }
  else {
    Cerr << "Error: invalid ptr_type in copy_data(T* ptr, Dakota::Matrix<T>)"
	 << endl;
    abort_handler(-1);
  }
}

/// copy T* to Array<Vector<T> >
template <class T>
void copy_data(const T* ptr, const int ptr_len, const String& ptr_type,
	       Array<Vector<T> >& dva, size_t num_vec, size_t vec_len)
{
  if (num_vec && vec_len) { // both specified
    if (ptr_len != num_vec*vec_len) {
      Cerr << "Error: pointer allocation (" << ptr_len << ") does not equal "
	   << "num_vec*vec_len (" << num_vec << '*' << vec_len << ") in "
	   << "copy_data(T*, Dakota::Array<Vector<T> >)." << endl;
      abort_handler(-1);
    }
  }
  else if (num_vec) { // only num_vec is non-zero
    if (ptr_len%num_vec) {
      Cerr << "Error: pointer allocation (" << ptr_len << ") not evenly "
	   << "divisible by number of vectors (" << num_vec << ") in "
	   << "copy_data(T*, Dakota::Array<Vector<T> >)." << endl;
      abort_handler(-1);
    }
    vec_len = ptr_len/num_vec;
  }
  else if (vec_len) { // only vec_len is non-zero
    if (ptr_len%vec_len) {
      Cerr << "Error: pointer allocation (" << ptr_len << ") not evenly "
	   << "divisible by vector length (" << vec_len << ") in copy_data(T*, "
	   << "Dakota::Array<Vector<T> >)." << endl;
      abort_handler(-1);
    }
    num_vec = ptr_len/vec_len;
  }
  else { // neither specified
    Cerr << "Error: either num_vec or vec_len must be specified in "
	 << "copy_data(T*, Dakota::Array<Vector<T> >)." << endl;
    abort_handler(-1);
  }
  size_t i, j;
  if (dva.length() != num_vec)
    dva.reshape(num_vec);
  for (i=0; i<num_vec; i++)
    if (dva[i].length() != vec_len)
      dva[i].reshape(vec_len);
  int cntr = 0;
  if (toLower(ptr_type) == "c") { // case insensitive
    for (i=0; i<num_vec; i++)   // loop over rows
      for (j=0; j<vec_len; j++) // loop over columns
        dva[i][j] = ptr[cntr++];
  }
  else if (toLower(ptr_type) == "fortran") { // case insensitive
    for (j=0; j<vec_len; j++)   // loop over columns
      for (i=0; i<num_vec; i++) // loop over rows
        dva[i][j] = ptr[cntr++];
  }
  else {
    Cerr << "Error: invalid ptr_type in copy_data(T* ptr, "
         << "Dakota::Array<Vector<T> >)" << endl;
    abort_handler(-1);
  }
}

/// copy Vector<T> to T*
template <class T>
void copy_data(const Vector<T>& dv, T* ptr, const int ptr_len)
{
  if (ptr_len != dv.length()) { // could use <, but enforce exact match
    Cerr << "Error: bad ptr_len in copy_data(Dakota::Vector<T>, T* ptr)."<<endl;
    abort_handler(-1);
  }
  for (size_t i=0; i<ptr_len; i++)
    ptr[i] = dv[i];
}

/// copy BaseVector<T> to T*
template <class T>
void copy_data(const BaseVector<T>& dbv, T* ptr, const int ptr_len)
{
  if (ptr_len != dbv.length()) { // could use <, but enforce exact match
    Cerr << "Error: bad ptr_len in copy_data(Dakota::BaseVector<T>, T* ptr)."
	 << endl;
    abort_handler(-1);
  }
  for (size_t i=0; i<ptr_len; i++)
    ptr[i] = dbv[i];
}

/// copy Matrix<T> to T*
template <class T>
void copy_data(const Matrix<T>& dm, T* ptr, const int ptr_len,
	       const String& ptr_type)
{
  size_t i, j, nr = dm.num_rows(), nc = dm.num_columns();
  int cntr = 0;
  if (ptr_len != nr*nc) {
    Cerr << "Error: bad ptr_len in copy_data(Dakota::Matrix<T>, T* ptr)."<<endl;
    abort_handler(-1);
  }
  if (toLower(ptr_type) == "c") { // case insensitive
    for (i=0; i<nr; i++)   // loop over rows
      for (j=0; j<nc; j++) // loop over columns
        ptr[cntr++] = dm[i][j];
  }
  else if (toLower(ptr_type) == "fortran") { // case insensitive
    for (j=0; j<nc; j++)   // loop over columns
      for (i=0; i<nr; i++) // loop over rows
        ptr[cntr++] = dm[i][j];
  }
  else {
    Cerr << "Error: invalid ptr_type in copy_data(Dakota::Matrix<T>, T* ptr)"
	 << endl;
    abort_handler(-1);
  }
}

/// copy Array<Vector<T> > to T*
template <class T>
void copy_data(const Array<Vector<T> >& dva, T* ptr, const int ptr_len,
	       const String& ptr_type)
{
  bool c_type;
  if (toLower(ptr_type) == "c") // case insensitive
    c_type = true;
  else if (toLower(ptr_type) == "fortran") // case insensitive
    c_type = false;
  else {
    Cerr << "Error: invalid ptr_type in copy_data(Dakota::Array<Vector<T> >, "
	 << "T* ptr)" << endl;
    abort_handler(-1);
  }
  size_t i, j, num_vec = dva.length(), total_len = 0, max_vec_len = 0;
  for (i=0; i<num_vec; i++) { // loop over vectors in array
    size_t vec_len = dva[i].length();
    total_len += vec_len;
    if (!c_type && vec_len > max_vec_len)
      max_vec_len = vec_len;
  }
  if (ptr_len != total_len) {
    Cerr << "Error: bad ptr_len in copy_data(Dakota::Array<Vector<T> >, T* "
	 << "ptr)." << endl;
    abort_handler(-1);
  }
  int cntr = 0;
  if (c_type) {
    for (i=0; i<num_vec; i++) { // loop over rows
      size_t vec_len = dva[i].length(); // allowed to vary
      for (j=0; j<vec_len; j++) // loop over columns
        ptr[cntr++] = dva[i][j];
    }
  }
  else {
    for (j=0; j<max_vec_len; j++) // loop over longest column
      for (i=0; i<num_vec; i++) // loop over rows
	if (j < dva[i].length())
	  ptr[cntr++] = dva[i][j];
  }
}

/// copy Vector<T> to Matrix<T>
template <class T>
void copy_data(const Vector<T>& dv, Matrix<T>& dm, size_t nr, size_t nc)
{
  size_t size_dv = dv.length();

  // This function is set up to do the transformation with either nr or nc or
  // both specified.  To omit nr or nc specification, a 0 is passed.
  if (nr && nc) { // both specified
    if (size_dv != nr*nc) {
      Cerr << "Error: dv length (" << size_dv << ") does not equal nr*nc ("
	   << nr << '*' << nc << ") in copy_data(Dakota::Vector<T>, "
	   << "Dakota::Matrix<T>)." << endl;
      abort_handler(-1);
    }
  }
  else if (nr) { // only nr is non-zero
    if (size_dv%nr) {
      Cerr << "Error: dv length (" << size_dv << ") not evenly divisible by "
	   << "number of rows (" << nr << ") in copy_data(Dakota::Vector<T>, "
	   << "Dakota::Matrix<T>)." << endl;
      abort_handler(-1);
    }
    nc = size_dv/nr;
  }
  else if (nc) { // only nc is non-zero
    if (size_dv%nc) {
      Cerr << "Error: dv length (" << size_dv << ") not evenly divisible by "
	   << "number of columns (" << nc << ") in copy_data(Dakota::Vector<T>"
	   << ", Dakota::Matrix<T>)." << endl;
      abort_handler(-1);
    }
    nr = size_dv/nc;
  }
  else { // neither specified
    Cerr << "Error: either nr or nc must be specified in "
	 << "copy_data(Dakota::Vector<T>, Dakota::Matrix<T>)." << endl;
    abort_handler(-1);
  }

  if (dm.num_rows() != nr || dm.num_columns() != nc)
    dm.reshape_2d(nr, nc);
  // dv is head to tail by rows, which matches the visual matrix format that 
  // a user would employ in specifying a matrix as a <LISTof><REAL>
  size_t counter = 0;
  for (size_t i=0; i<nr; i++)
    for (size_t j=0; j<nc; j++)
      dm[i][j] = dv[counter++];
}

/// copy Vector<T> to Array<Vector<T> >
template <class T>
void copy_data(const Vector<T>& dv, Array<Vector<T> >& dva, size_t num_vec,
	       size_t vec_len)
{
  size_t size_dv = dv.length();

  // This function is set up to do the transformation with either num_vec or
  // vec_len or both specified.  To omit num_vec or vec_len specification, a 0
  // is passed.
  if (num_vec && vec_len) { // both specified
    if (size_dv != num_vec*vec_len) {
      Cerr << "Error: dv length (" << size_dv << ") does not equal num_vec*"
	   << "vec_len (" << num_vec << '*' << vec_len << ") in copy_data("
	   << "Dakota::Vector<T>, Dakota::Array<Vector<T> >)." << endl;
      abort_handler(-1);
    }
  }
  else if (num_vec) { // only num_vec is non-zero
    if (size_dv%num_vec) {
      Cerr << "Error: dv length (" << size_dv << ") not evenly divisible by "
	   << "number of vectors (" << num_vec << ") in copy_data("
	   << "Dakota::Vector<T>, Dakota::Array<Vector<T> >)." << endl;
      abort_handler(-1);
    }
    vec_len = size_dv/num_vec;
  }
  else if (vec_len) { // only vec_len is non-zero
    if (size_dv%vec_len) {
      Cerr << "Error: dv length (" << size_dv << ") not evenly divisible by "
	   << "vector length (" << vec_len << ") in copy_data(Dakota::Vector<T>"
	   << ", Dakota::Array<Vector<T> >)." << endl;
      abort_handler(-1);
    }
    num_vec = size_dv/vec_len;
  }
  else { // neither specified
    Cerr << "Error: either num_vec or vec_len must be specified in "
	 << "copy_data(Dakota::Vector<T>, Dakota::Array<Vector<T> >)." << endl;
    abort_handler(-1);
  }

  if (dva.length() != num_vec)
    dva.reshape(num_vec);
  // dv is head to tail by rows, which matches the visual matrix format that 
  // a user would employ in specifying a matrix as a <LISTof><REAL>
  size_t counter = 0;
  for (size_t i=0; i<num_vec; i++) {
    if (dva[i].length() != vec_len)
      dva[i].reshape(vec_len);
    for (size_t j=0; j<vec_len; j++)
      dva[i][j] = dv[counter++];
  }
}

/// copy Array<T> to Vector<T>
template <class T> void copy_data(const Array<T>& da, Vector<T>& dv)
{
  size_t size_da = da.length();
  if (size_da != dv.length())
    dv.reshape(size_da);
  for (size_t i=0; i<size_da; i++)
    dv[i] = da[i];
}

/// copy BaseVector<T> to Vector<T>
template <class T> void copy_data(const BaseVector<T>& dbv, Vector<T>& dv)
{
  size_t size_dbv = dbv.length();
  if (size_dbv != dv.length())
    dv.reshape(size_dbv);
  for (size_t i=0; i<size_dbv; i++)
    dv[i] = dbv[i];
}

/// copy List<T> to Array<T>
template <class T> void copy_data(const List<T>& dl, Array<T>& da)
{
  size_t size_dl = dl.entries();
  if (size_dl != da.length())
    da.reshape(size_dl);
  typename List<T>::const_iterator dl_cit = dl.begin();
  for (size_t i=0; i<size_dl; i++, dl_cit++)
    da[i] = *dl_cit;
}

/// copy List<T> to Array<Array<T> >
template <class T> void 
copy_data(const List<T>& dl, Array<Array<T> >& d2a, size_t num_a, size_t a_len)
{
  size_t i, j, size_dl = dl.entries();

  // This function is set up to do the copy with either num_a or a_len or both
  // specified.  To omit num_a or a_len specification, a 0 is passed.
  if (num_a && a_len) { // both specified
    if (size_dl != num_a*a_len) {
      Cerr << "Error: dl length (" << size_dl <<") does not equal num_a*a_len ("
	   << num_a << '*' << a_len << ") in copy_data(Dakota::List<T>, "
	   << "Dakota::Array<Array<T> >)." << endl;
      abort_handler(-1);
    }
  }
  else if (num_a) { // only num_a is non-zero
    if (size_dl%num_a) {
      Cerr << "Error: dl length (" << size_dl << ") not evenly divisible by "
	   << "number of arrays (" << num_a << ") in copy_data(Dakota::List<T>"
	   << ", Dakota::Array<Array<T> >)." << endl;
      abort_handler(-1);
    }
    a_len = size_dl/num_a;
  }
  else if (a_len) { // only a_len is non-zero
    if (size_dl%a_len) {
      Cerr << "Error: dl length (" << size_dl << ") not evenly divisible by "
	   << "array length (" << a_len << ") in copy_data(Dakota::List<T>, "
	   << "Dakota::Array<Array<T> >)." << endl;
      abort_handler(-1);
    }
    num_a = size_dl/a_len;
  }
  else { // neither specified
    Cerr << "Error: either num_a or a_len must be specified in "
	 << "copy_data(Dakota::List<T>, Dakota::Array<Array<T> >)." << endl;
    abort_handler(-1);
  }

  if (d2a.length() != num_a)
    d2a.reshape(num_a);
  typename List<T>::const_iterator dl_cit = dl.begin();
  for (i=0; i<num_a; i++) {
    if (d2a[i].length() != a_len)
      d2a[i].reshape(a_len);
    for (j=0; j<a_len; j++, dl_cit++)
      d2a[i][j] = *dl_cit;
  }
}

/// copy Array<Array<T> > to Array<T> (unroll 2D array into 1D array)
template <class T> void copy_data(const Array<Array<T> >& d2a, Array<T>& da)
{
  size_t i, j, size_d2a = 0, num_arrays = d2a.length(), cntr = 0;
  for (i=0; i<num_arrays; i++)
    size_d2a += d2a[i].length();
  if (size_d2a != da.length())
    da.reshape(size_d2a);
  for (i=0; i<num_arrays; i++) {
    size_t array_len = d2a[i].length();
    for (j=0; j<array_len; j++)
      da[cntr++] = d2a[i][j];
  }
}

/// copy Teuchos::SerialDenseVector<OrdinalType, ScalarType> to
/// Vector<ScalarType>
template <typename OrdinalType, typename ScalarType> 
void copy_data(const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
	       Vector<ScalarType>& dv)
{
  OrdinalType size_sdv = sdv.length();
  if (size_sdv != dv.length())
    dv.reshape(size_sdv);
  for (OrdinalType i=0; i<size_sdv; i++)
    dv[i] = sdv[i];
}

/// copy Teuchos::SerialDenseVector<OrdinalType, ScalarType> to
/// BaseVector<ScalarType>
template <typename OrdinalType, typename ScalarType> 
void copy_data(const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
	       BaseVector<ScalarType>& dbv)
{
  OrdinalType size_sdv = sdv.length();
  if (size_sdv != dbv.length())
    dbv.reshape(size_sdv);
  for (OrdinalType i=0; i<size_sdv; i++)
    dbv[i] = sdv[i];
}

/// copy Teuchos::SerialDenseMatrix<OrdinalType, ScalarType> to
/// Matrix<ScalarType>
template <typename OrdinalType, typename ScalarType> 
void copy_data(const Teuchos::SerialDenseMatrix<OrdinalType, ScalarType>& sdm,
	       Matrix<ScalarType>& dm)
{
  // Dakota::Matrix and Teuchos::SerialDenseMatrix<OrdinalType, ScalarType>
  // can both be m x n.
  OrdinalType nr = sdm.numRows(), nc = sdm.numCols();
  if (dm.num_rows() != nr || dm.num_columns() != nc)
    dm.reshape_2d(nr, nc);
  for (OrdinalType i=0; i<nr; i++)
    for (OrdinalType j=0; j<nc; j++)
      dm[i][j] = sdm(i,j);
}

/// copy Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType> to
/// Matrix<ScalarType>
template <typename OrdinalType, typename ScalarType> 
void copy_data(const Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>&
	       ssdm, Matrix<ScalarType>& dm)
{
  // Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType> must be n x n.
  OrdinalType n = ssdm.numRows();
  if (dm.num_rows() != n || dm.num_columns() != n)
    dm.reshape_2d(n, n);
  for (OrdinalType i=0; i<n; i++)
    for (OrdinalType j=0; j<n; j++)
      dm[i][j] = ssdm(i,j); // Teuchos only returns active matrix values
}

/// copy Vector<ScalarType> to
/// Teuchos::SerialDenseVector<OrdinalType, ScalarType>
template <typename OrdinalType, typename ScalarType> 
void copy_data(const Vector<ScalarType>& dv,
	       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  size_t size_dv = dv.length();
  if (sdv.length() != size_dv)
    sdv.sizeUninitialized(size_dv);
  for (OrdinalType i=0; i<size_dv; i++)
    sdv[i] = dv[i];
}

/// copy Array<ScalarType> to
/// Teuchos::SerialDenseVector<OrdinalType, ScalarType>
template <typename OrdinalType, typename ScalarType> 
void copy_data(const Array<ScalarType>& da,
	       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  size_t size_da = da.length();
  if (sdv.length() != size_da)
    sdv.sizeUninitialized(size_da);
  for (OrdinalType i=0; i<size_da; i++)
    sdv[i] = da[i];
}

/// copy BaseVector<ScalarType> to
/// Teuchos::SerialDenseVector<OrdinalType, ScalarType>
template <typename OrdinalType, typename ScalarType> 
void copy_data(const BaseVector<ScalarType>& dbv,
	       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  size_t size_dbv = dbv.length();
  if (sdv.length() != size_dbv)
    sdv.sizeUninitialized(size_dbv);
  for (OrdinalType i=0; i<size_dbv; i++)
    sdv[i] = dbv[i];
}

/// copy ScalarType* to Teuchos::SerialDenseVector<OrdinalType, ScalarType>
template <typename OrdinalType, typename ScalarType> 
void copy_data(const ScalarType* ptr, const OrdinalType ptr_len,
	       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  if (sdv.length() != ptr_len)
    sdv.sizeUninitialized(ptr_len);
  for (OrdinalType i=0; i<ptr_len; i++)
    sdv[i] = ptr[i];

  //sdv = RealDenseVector(Copy, ptr, ptr_len);
  // not advisable since relying on default operator=.  Would be a good 
  // approach if a reference counting scheme was used for operator=.
}

/// copy Matrix<ScalarType> to
/// Teuchos::SerialDenseMatrix<OrdinalType, ScalarType>
template <typename OrdinalType, typename ScalarType>
void copy_data(const Matrix<ScalarType>& dm,
	       Teuchos::SerialDenseMatrix<OrdinalType, ScalarType>& sdm)
{
  // Dakota::Matrix and Teuchos::SerialDenseMatrix<OrdinalType, ScalarType>
  // can both be m x n.
  size_t nr = dm.num_rows(), nc = dm.num_columns();
  if (sdm.numRows() != nr || sdm.numCols() != nc)
    sdm.shapeUninitialized(nr, nc);
  for (OrdinalType i=0; i<nr; i++)
    for (OrdinalType j=0; j<nc; j++)
      sdm(i,j) = dm[i][j];
}

/// copy Matrix<ScalarType> to
/// Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>
template <typename OrdinalType, typename ScalarType> 
void copy_data(const Matrix<ScalarType>& dm,
	       Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>& ssdm)
{
  // Dakota::Matrix can be m x n,
  // Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType> must be n x n.
  size_t nr = dm.num_rows(), nc = dm.num_columns();
  if (nr != nc) {
    Cerr << "Error: cannot copy from a rectangular matrix to a "
	 << "Teuchos::SerialSymDenseMatrix<OT, ST>" << endl;
    abort_handler(-1);
  }
  if (ssdm.numRows() != nr)
    ssdm.shapeUninitialized(nr);
  for (OrdinalType i=0; i<nr; i++)
    for (OrdinalType j=0; j<=i; j++) //j<nr; j++)
      ssdm(i,j) = dm[i][j]; // only lower triangle is active
}

/// copy Array<Vector<ScalarType> > to
/// Array<Teuchos::SerialDenseVector<OrdinalType, ScalarType> >
template <typename OrdinalType, typename ScalarType> 
void copy_data(const Array<Vector<ScalarType> >& dva,
  Array<Teuchos::SerialDenseVector<OrdinalType, ScalarType> >& sdva)
{
  size_t size_dva = dva.length();
  if (sdva.length() != size_dva)
    sdva.reshape(size_dva);
  for (size_t i=0; i<size_dva; i++)
    copy_data(dva[i], sdva[i]);
}

/// copy Array<Matrix<ScalarType> > to
/// Array<Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType> >
template <typename OrdinalType, typename ScalarType> 
void copy_data(const Array<Matrix<ScalarType> >& dma,
  Array<Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType> >& ssdma)
{
  size_t size_dma = dma.length();
  if (ssdma.length() != size_dma)
    ssdma.reshape(size_dma);
  for (size_t i=0; i<size_dma; i++)
    copy_data(dma[i], ssdma[i]);
}

// Partial copy functions

/// copy portion of first Vector<T> to all of second Vector<T>
template <class T>
void copy_data_partial(const Vector<T>& dv1, size_t start_index,
		       size_t num_items, Vector<T>& dv2)
{
  // dv1 will be indexed from start_index to start_index+num_items-1
  if (start_index + num_items > dv1.length()) {
    Cerr << "Error: indexing out of bounds in copy_data_partial("
	 << "Dakota::Vector<T>, size_t, size_t, Dakota::Vector<T>)." << endl;
    abort_handler(-1);
  }
  if (num_items != dv2.length())
    dv2.reshape(num_items);
  for (size_t i=0; i<num_items; i++)
    dv2[i] = dv1[start_index+i];
}

/// copy portion of Vector<ScalarType> to all of
/// Teuchos::SerialDenseVector<OrdinalType, ScalarType>
template <typename OrdinalType, typename ScalarType>
void copy_data_partial(const Vector<ScalarType>& dv,
		       size_t start_index, size_t num_items,
		       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  // dv will be indexed from start_index to start_index+num_items-1
  if (start_index + num_items > dv.length()) {
    Cerr << "Error: indexing out of bounds in copy_data_partial(Dakota::Vector"
	 << "<ST>, OT, OT, Teuchos::SerialDenseVector<OT, ST>)." << endl;
    abort_handler(-1);
  }
  if (num_items != sdv.length())
    sdv.sizeUninitialized(num_items);
  for (OrdinalType i=0; i<num_items; i++)
    sdv[i] = dv[start_index+i];
}

/// copy all of first Vector<T> to portion of second Vector<T>
template <class T>
void copy_data_partial(const Vector<T>& dv1, Vector<T>& dv2, size_t start_index)
{
  size_t num_items = dv1.length();
  // In this case, incoming dv2 must already be sized and will be
  // indexed from start_index to start_index+num_items-1
  if (start_index + num_items > dv2.length()) {
    Cerr << "Error: indexing out of bounds in copy_data_partial("
	 << "Dakota::Vector<T>, Dakota::Vector<T>, size_t)." << endl;
    abort_handler(-1);
  }
  for (size_t i=0; i<num_items; i++)
    dv2[start_index+i] = dv1[i];
}

/// copy portion of first Vector<T> to portion of second Vector<T>
template <class T>
void copy_data_partial(const Vector<T>& dv1, size_t start_index1,
		       size_t num_items, Vector<T>& dv2, size_t start_index2)
{
  // In this case, incoming dv2 must already be sized and will be
  // indexed from start_index2 to start_index2+num_items-1.  dv1 will
  // be indexed from start_index1 to start_index1+num_items-1
  if (start_index1 + num_items > dv1.length() ||
      start_index2 + num_items > dv2.length()) {
    Cerr << "Error: indexing out of bounds in copy_data_partial(Dakota::"
	 << "Vector<T>, size_t, size_t, Dakota::Vector<T>, size_t)." << endl;
    abort_handler(-1);
  }
  for (size_t i=0; i<num_items; i++)
    dv2[start_index2+i] = dv1[start_index1+i];
}

/// copy portion of first Array<T> to all of second Array<T>
template <class T>
void copy_data_partial(const Array<T>& da1, size_t start_index,
		       size_t num_items, Array<T>& da2)
{
  // da1 will be indexed from start_index to start_index+num_items-1
  if (start_index + num_items > da1.length()) {
    Cerr << "Error: indexing out of bounds in copy_data_partial("
	 << "Dakota::Array<T>, size_t, size_t, Dakota::Array<T>)." << endl;
    abort_handler(-1);
  }
  if (num_items != da2.length())
    da2.reshape(num_items);
  for (size_t i=0; i<num_items; i++)
    da2[i] = da1[start_index+i];
}

/// copy all of first Array<T> to portion of second Array<T>
template <class T>
void copy_data_partial(const Array<T>& da1, Array<T>& da2, size_t start_index)
{
  size_t num_items = da1.length();
  // In this case, incoming da2 must already be sized and will be
  // indexed from start_index to start_index+num_items-1
  if (start_index + num_items > da2.length()) {
    Cerr << "Error: indexing out of bounds in copy_data_partial("
	 << "Dakota::Array<T>, Dakota::Array<T>, size_t)." << endl;
    abort_handler(-1);
  }
  for (size_t i=0; i<num_items; i++)
    da2[start_index+i] = da1[i];
}

/// copy portion of first Array<T> to portion of second Array<T>
template <class T>
void copy_data_partial(const Array<T>& da1, size_t start_index1,
		       size_t num_items, Array<T>& da2, size_t start_index2)
{
  // In this case, incoming da2 must already be sized and will be
  // indexed from start_index2 to start_index2+num_items-1.  da1 will
  // be indexed from start_index1 to start_index1+num_items-1
  if (start_index1 + num_items > da1.length() ||
      start_index2 + num_items > da2.length()) {
    Cerr << "Error: indexing out of bounds in copy_data_partial(Dakota::"
	 << "Array<T>, size_t, size_t, Dakota::Array<T>, size_t)." << endl;
    abort_handler(-1);
  }
  for (size_t i=0; i<num_items; i++)
    da2[start_index2+i] = da1[start_index1+i];
}

/// aggregate two Array<T>'s into a single Array<T>
template <class T>
void aggregate_data(const Array<T>& da1, const Array<T>& da2, Array<T>& all_da)
{
  size_t i, num_da1 = da1.length(), num_da2 = da2.length(),
    num_all = num_da1 + num_da2;
  if (all_da.length() != num_all)
    all_da.reshape(num_all);
  // could reuse copy_data_partial, but avoid the extra overhead
  //copy_data_partial(da1, all_da, 0);
  //copy_data_partial(da2, all_da, num_da1);
  for (i=0; i<num_da1; i++)
    all_da[i] = da1[i];
  for (i=0; i<num_da2; i++)
    all_da[num_da1+i] = da2[i];
}

/// aggregate three Array<T>'s into a single Array<T>
template <class T>
void aggregate_data(const Array<T>& da1, const Array<T>& da2,
		    const Array<T>& da3, Array<T>& all_da)
{
  size_t i, num_da1 = da1.length(), num_da2 = da2.length(),
    num_da3 = da3.length(), num_da1da2 = num_da1 + num_da2,
    num_all = num_da1da2 + num_da3;
  if (all_da.length() != num_all)
    all_da.reshape(num_all);
  // could reuse copy_data_partial, but avoid the extra overhead
  //copy_data_partial(da1, all_da, 0);
  //copy_data_partial(da2, all_da, num_da1);
  //copy_data_partial(da3, all_da, num_da1da2);
  for (i=0; i<num_da1; i++)
    all_da[i] = da1[i];
  for (i=0; i<num_da2; i++)
    all_da[num_da1+i] = da2[i];
  for (i=0; i<num_da3; i++)
    all_da[num_da1da2+i] = da3[i];
}

/// aggregate two Vector<T>'s into a single Vector<T>
template <class T>
void aggregate_data(const Vector<T>& dv1, const Vector<T>& dv2,
		    Vector<T>& all_dv)
{
  size_t i, num_dv1 = dv1.length(), num_dv2 = dv2.length(),
    num_all = num_dv1 + num_dv2;
  if (all_dv.length() != num_all)
    all_dv.reshape(num_all);
  // could reuse copy_data_partial, but avoid the extra overhead
  //copy_data_partial(dv1, all_dv, 0);
  //copy_data_partial(dv2, all_dv, num_dv1);
  for (i=0; i<num_dv1; i++)
    all_dv[i] = dv1[i];
  for (i=0; i<num_dv2; i++)
    all_dv[num_dv1+i] = dv2[i];
}

/// aggregate three Vector<T>'s into a single Vector<T>
template <class T>
void aggregate_data(const Vector<T>& dv1, const Vector<T>& dv2,
		    const Vector<T>& dv3, Vector<T>& all_dv)
{
  size_t i, num_dv1 = dv1.length(), num_dv2 = dv2.length(),
    num_dv3 = dv3.length(), num_dv1dv2 = num_dv1 + num_dv2,
    num_all = num_dv1dv2 + num_dv3;
  if (all_dv.length() != num_all)
    all_dv.reshape(num_all);
  // could reuse copy_data_partial, but avoid the extra overhead
  //copy_data_partial(dv1, all_dv, 0);
  //copy_data_partial(dv2, all_dv, num_dv1);
  //copy_data_partial(dv3, all_dv, num_dv1dv2);
  for (i=0; i<num_dv1; i++)
    all_dv[i] = dv1[i];
  for (i=0; i<num_dv2; i++)
    all_dv[num_dv1+i] = dv2[i];
  for (i=0; i<num_dv3; i++)
    all_dv[num_dv1dv2+i] = dv3[i];
}

/// separate an Array<T> into two Array<T>'s
template <class T>
void separate_data(const Array<T>& all_da, Array<T>& da1, Array<T>& da2)
{
  size_t i, num_da1 = da1.length(), num_da2 = da2.length();
  if ( num_da1 + num_da2 != all_da.length() ) {
    Cerr << "Error: Length of all array passed to separate_data()\n       "
	 << "does not equal sum of lengths of arrays to be updated.\n       "
	 << "Resizing is ambiguous." << endl;
    abort_handler(-1);
  }
  // could reuse copy_data_partial, but avoid the extra overhead
  //copy_data_partial(all_da,       0, num_da1, da1);
  //copy_data_partial(all_da, num_da1, num_da2, da2);
  for (i=0; i<num_da1; i++)
    da1[i] = all_da[i];
  for (i=0; i<num_da2; i++)
    da2[i] = all_da[num_da1+i];
}

/// separate an Array<T> into three Array<T>'s
template <class T>
void separate_data(const Array<T>& all_da, Array<T>& da1, Array<T>& da2,
		   Array<T>& da3)
{
  size_t i, num_da1 = da1.length(), num_da2 = da2.length(),
    num_da3 = da3.length(), num_da1da2 = num_da1 + num_da2;
  if ( num_da1da2 + num_da3 != all_da.length() ) {
    Cerr << "Error: Length of all array passed to separate_data()\n       "
	 << "does not equal sum of lengths of arrays to be updated.\n       "
	 << "Resizing is ambiguous." << endl;
    abort_handler(-1);
  }
  // could reuse copy_data_partial, but avoid the extra overhead
  //copy_data_partial(all_da,          0, num_da1, da1);
  //copy_data_partial(all_da,    num_da1, num_da2, da2);
  //copy_data_partial(all_da, num_da1da2, num_da3, da3);
  for (i=0; i<num_da1; i++)
    da1[i] = all_da[i];
  for (i=0; i<num_da2; i++)
    da2[i] = all_da[num_da1+i];
  for (i=0; i<num_da3; i++)
    da3[i] = all_da[num_da1da2+i];
}

/// separate an Vector<T> into two Vector<T>'s
template <class T>
void separate_data(const Vector<T>& all_dv, Vector<T>& dv1, Vector<T>& dv2)
{
  size_t i, num_dv1 = dv1.length(), num_dv2 = dv2.length();
  if ( num_dv1 + num_dv2 != all_dv.length() ) {
    Cerr << "Error: Length of all array passed to separate_data()\n       "
	 << "does not equal sum of lengths of arrays to be updated.\n       "
	 << "Resizing is ambiguous." << endl;
    abort_handler(-1);
  }
  // could reuse copy_data_partial, but avoid the extra overhead
  //copy_data_partial(all_dv,       0, num_dv1, dv1);
  //copy_data_partial(all_dv, num_dv1, num_dv2, dv2);
  for (i=0; i<num_dv1; i++)
    dv1[i] = all_dv[i];
  for (i=0; i<num_dv2; i++)
    dv2[i] = all_dv[num_dv1+i];
}

/// separate an Vector<T> into three Vector<T>'s
template <class T>
void separate_data(const Vector<T>& all_dv, Vector<T>& dv1, Vector<T>& dv2,
		   Vector<T>& dv3)
{
  size_t i, num_dv1 = dv1.length(), num_dv2 = dv2.length(),
    num_dv3 = dv3.length(), num_dv1dv2 = num_dv1 + num_dv2;
  if ( num_dv1dv2 + num_dv3 != all_dv.length() ) {
    Cerr << "Error: Length of all array passed to separate_data()\n       "
	 << "does not equal sum of lengths of arrays to be updated.\n       "
	 << "Resizing is ambiguous." << endl;
    abort_handler(-1);
  }
  // could reuse copy_data_partial, but avoid the extra overhead
  //copy_data_partial(all_dv,          0, num_dv1, dv1);
  //copy_data_partial(all_dv,    num_dv1, num_dv2, dv2);
  //copy_data_partial(all_dv, num_dv1dv2, num_dv3, dv3);
  for (i=0; i<num_dv1; i++)
    dv1[i] = all_dv[i];
  for (i=0; i<num_dv2; i++)
    dv2[i] = all_dv[num_dv1+i];
  for (i=0; i<num_dv3; i++)
    dv3[i] = all_dv[num_dv1dv2+i];
}

// Templated conversion functions between UTILIB/TNT data types and
// DAKOTA/built in data types

#ifdef DAKOTA_UTILIB
/// copy utilib::NumArray<T> to Vector<T>
template <class T> void copy_data(const utilib::NumArray<T>& na, Vector<T>& dv)
{
  size_t size_na = na.size();
  if (size_na != dv.length())
    dv.reshape(size_na);
  for (size_t i=0; i<size_na; i++)
    dv[i] = na[i];
}

/// copy Vector<T> to utilib::NumArray<T>
template <class T> void copy_data(const Vector<T>& dv, utilib::NumArray<T>& na)
{
  size_t size_dv = dv.length();
  if (na.size() != size_dv)
    na.resize(size_dv);
  for (size_t i=0; i<size_dv; i++)
    na[i] = dv[i];
}

/// copy utilib::NumArray<T> to Array<T>
template <class T> void copy_data(const utilib::NumArray<T>& na, Array<T>& da)
{
  size_t size_na = na.size();
  if (size_na != da.length())
    da.reshape(size_na);
  for (size_t i=0; i<size_na; i++)
    da[i] = na[i];
}

/// copy List<T> to utilib::NumArray<T>
template <class T> void copy_data(const List<T>& dl, utilib::NumArray<T>& na)
{
  size_t size_dl = dl.entries();
  if (na.size() != size_dl)
    na.resize(size_dl);
  typename List<T>::const_iterator dl_cit = dl.begin();
  for (size_t i=0; i<size_dl; i++, dl_cit++)
    na[i] = *dl_cit;
}
#endif // DAKOTA_UTILIB

#ifdef DAKOTA_TNT
/// copy TNT::Vector<T> to Vector<T>
template <class T> void copy_data(const TNT::Vector<T>& tntv, Vector<T>& dv)
{
  int size_tntv = tntv.size();
  if (size_tntv != dv.length())
    dv.reshape(size_tntv);
  for (size_t i=0; i<size_tntv; i++)
    dv[i] = tntv[i];
}

/// copy Vector<T> to TNT::Vector<T>
template <class T> void copy_data(const Vector<T>& dv, TNT::Vector<T>& tntv)
{
  size_t size_dv = dv.length();
  if (tntv.size() != size_dv)
    tntv.newsize(size_dv);
  for (size_t i=0; i<size_dv; i++)
    tntv[i] = dv[i];
}

/// copy T* to TNT::Vector<T>
template <class T> void 
copy_data(const T* ptr, const int ptr_len, TNT::Vector<T>& tntv)
{
  if (tntv.size() != ptr_len)
    tntv.newsize(ptr_len);
  for (size_t i=0; i<ptr_len; i++)
    tntv[i] = ptr[i];
}

/// copy Matrix<T> to TNT::Matrix<T>
template <class T> void copy_data(const Matrix<T>& dm, TNT::Matrix<T>& tntm)
{
  // Dakota::Matrix and TNT::Matrix can both be general m x n matrices.
  size_t nr = dm.num_rows(), nc = dm.num_columns();
  if (tntm.num_rows() != nr || tntm.num_cols() != nc)
    tntm.newsize(nr, nc);
  for (size_t i=0; i<nr; i++)
    for (size_t j=0; j<nc; j++)
      tntm[i][j] = dm[i][j];
}
#endif // DAKOTA_TNT

} // namespace Dakota

#endif // DATA_TYPES_H
