/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        TANA3Approximation
//- Description:  Class for TANA-3 two-point exponential approximation.
//-               
//- Owner:        Mike Eldred, Sandia National Laboratories
 
#ifndef TANA3_APPROXIMATION_H
#define TANA3_APPROXIMATION_H

#include "DakotaApproximation.H"


namespace Dakota {

/// Derived approximation class for TANA-3 two-point exponential
/// approximation (a multipoint approximation).

/** The TANA3Approximation class provides a multipoint approximation
    based on matching value and gradient data from two points (typically
    the current and previous iterates) in parameter space.  It forms an
    exponential approximation in terms of intervening variables. */

class TANA3Approximation: public Approximation
{
public:

  //
  //- Heading: Constructor and destructor
  //

  /// default constructor
  TANA3Approximation();
  /// standard constructor
  TANA3Approximation(const ProblemDescDB& problem_db, const size_t& num_vars);
  /// destructor
  ~TANA3Approximation();

protected:

  //
  //- Heading: Virtual function redefinitions
  //

  int min_coefficients() const;

  int num_constraints()  const;

  void find_coefficients();

  const Real& get_value(const RealVector& x);

  const RealBaseVector& get_gradient(const RealVector& x);

  //const RealMatrix& get_hessian(const RealVector& x);

  void clear_current();

private:

  //
  //- Heading: Convenience functions
  //

  /// compute TANA coefficients based on scaled inputs
  void find_scaled_coefficients();

  /// based on minX, apply offset scaling to x to define s
  void offset(const RealVector& x, RealVector& s);

  //
  //- Heading: Data
  //

  RealVector pExp; ///< the vector of exponent values
  RealVector minX; ///< the vector of minimum parameter values used in scaling
  RealVector scX1; ///< the vector of scaled x1 values
  RealVector scX2; ///< the vector of scaled x2 values
  Real H; ///< the scalar Hessian value in the TANA-3 approximation
};


inline TANA3Approximation::TANA3Approximation()
{ }


inline TANA3Approximation::~TANA3Approximation()
{ }


/** Redefine default implementation to support history mechanism. */
inline void TANA3Approximation::clear_current()
{
  currentPoints.clear();
  if (!anchorPoint.is_null()) {
    currentPoints.insert(anchorPoint);  // store 1 previous expansion point
    anchorPoint = SurrogateDataPoint(); // clear
  }
}

} // namespace Dakota

#endif
