/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       ParamStudy
//- Description: Parameter study driver program.  This class iterates a
//-              Model object using simple rules, i.e. evaluating
//-              a variety of specified points in the design space.   
//- Owner:       Mike Eldred
//- Version: $Id: ParamStudy.H 4461 2007-08-28 17:40:08Z mseldre $

#ifndef PARAM_STUDY_H
#define PARAM_STUDY_H

#include "DakotaPStudyDACE.H"


namespace Dakota {

/// Class for vector, list, centered, and multidimensional parameter studies.

/** The ParamStudy class contains several algorithms for performing
    parameter studies of different types.  It is not a wrapper for an
    external library, rather its algorithms are self-contained.  The
    vector parameter study steps along an n-dimensional vector from an
    arbitrary initial point to an arbitrary final point in a specified
    number of steps.  The centered parameter study performs a number
    of plus and minus offsets in each coordinate direction around a
    center point.  A multidimensional parameter study fills an
    n-dimensional hypercube based on a specified number of intervals
    for each dimension.  It is a nested study in that it utilizes the
    vector parameter study internally as it recurses through the
    variables.  And the list parameter study provides for a user
    specification of a list of points to evaluate, which allows
    general parameter investigations not fitting the structure of
    vector, centered, or multidim parameter studies. */

class ParamStudy: public PStudyDACE
{
public:

  //
  //- Heading: Constructors and destructors
  //

  ParamStudy(Model& model); ///< constructor
  ~ParamStudy();            ///< destructor
    
  //
  //- Heading: Virtual member function redefinitions
  //

  void extract_trends();

private:

  //
  //- Heading: Convenience/internal member functions
  //

  /// computes stepVector and numSteps from initialPoint, finalPoint,
  /// and either numSteps or stepLength (pStudyType is 1 or 2)
  void compute_vector_steps();

  /// performs the parameter study by sampling from a list of points
  void sample(const RealVectorArray& list_of_points);

  /// performs the parameter study by looping from start in num_steps 
  /// increments of step_vect.  Total number of evaluations is num_steps + 1.
  void vector_loop(const RealVector& start, const RealVector& step_vect,
		   const int& num_steps);

  /// performs a number of plus and minus offsets for each parameter
  /// centered about start
  void centered_loop(const RealVector& start, const Real& percent_delta,
		     const int& deltas_per_variable);

  /// performs vector_loops recursively in multiple dimensions
  void multidim_loop(const IntArray& var_partitions);

  //
  //- Heading: Data
  //

  /// array of evaluation points for the list_parameter_study
  RealVectorArray listOfPoints;
  /// the starting point for vector and centered parameter studies
  RealVector initialPoint;
  /// the ending point for vector_parameter_study (a specification option)
  RealVector finalPoint;
  /// the n-dimensional increment in vector_parameter_study
  RealVector stepVector;
  /// the number of times stepVector is applied in vector_parameter_study
  int numSteps;
  /// internal code for parameter study type: -1 (list), 1,2,3
  /// (different vector specifications), 4 (centered), or 5 (multidim)
  short pStudyType;
  /// number of offsets in the plus and the minus direction for each
  /// variable in a centered_parameter_study
  int deltasPerVariable;
  /// the Cartesian length of multidimensional steps in vector_parameter_study
  /// (a specification option)
  Real stepLength;
  /// size of relative offsets in percent for each variable in a
  /// centered_parameter_study
  Real percentDelta;
  /// number of partitions for each variable in a multidim_parameter_study
  IntArray variablePartitions;
};

} // namespace Dakota

#endif
